unit Markov;

{
   COPYRIGHT 2016 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.Math, System.Classes, System.SysUtils,

  KnobsUtils;


const

  MARKOV_VERSION_1  = 'markov 1';
  MARKOV_VERSION    = MARKOV_VERSION_1;
  MAX_MARKOV_LENGTH = 256;
  MAX_MARKOV_ORDER  =  16;


type

  EMarkov          = class( Exception);
  EMarkovReadFail  = class( EMarkov);
  EMarkovWriteFail = class( EMarkov);


  PMarkovSignalSet = ^TMarkovSignalSet;
  TMarkovSignalSet = record
  private
    Next     : PMarkovSignalSet;
    Index    : Integer;
    DataSize : Integer;
    Data     : TSignalArray;
  private
    procedure   Initialize( aSize: Integer);
    function    Hash: Byte;
    procedure   Append( aSignalSet: PMarkovSignalSet);
    function    Equals( aValue: PMarkovSignalSet): Boolean;
    function    Find  ( aValue: PMarkovSignalSet): Integer;
    procedure   Forget( aValue: PMarkovSignalSet);
//    procedure   SaveToStrings  ( anIndent: Integer; const aStrings: TStringList);
  end;


  TMarkovSignalSets = array of TMarkovSignalSet;
  TMarkovSignalHash = array[ Byte] of PMarkovSignalSet;


  TMarkovData = class
  private
    FSize        : Integer;
    FOrder       : Integer;
    FSignals     : TMarkovSignalSets;
    FHashIndex   : TMarkovSignalHash;
    FScratchData : TMarkovSignalSet;
    FInPtr       : Integer;
    FOutPtr      : Integer;
    FFreeCount   : Integer;
    FFillCount   : Integer;
  private
    constructor Create( aSize, anOrder: Integer);
    procedure   Clear;
    function    FindSignalSet( aValue: PMarkovSignalSet): Integer;
    function    FindSignals( aData: TSignalArray): Integer;
    function    AddSignalSet( aValue: PMarkovSignalSet): Integer;
    function    AddSignals( aData: TSignalArray): Integer;
    procedure   RemoveSignalSet;
//    procedure   SaveToStrings  ( anIndent: Integer; const aStrings: TStringList);
//    procedure   LoadSignalSet  ( var anIndex: Integer; const aStrings: TStringList);
//    procedure   LoadFromStrings( var anIndex: Integer; const aStrings: TStringList);
  end;


  TMarkovHistory = record
  private
    DataSize  : Integer;
    Data      : TSignalArray;
    InPtr     : Integer;
    OutPtr    : Integer;
    FreeCount : Integer;
    FillCount : Integer;
  private
    procedure   Initialize( aSize: Integer);
    procedure   AddValue( aValue: TSignal);
    procedure   Forget;
    procedure   FillData( aData: TSignalArray);
//    procedure   SaveToStrings  ( anIndent: Integer; const aStrings: TStringList);
//    procedure   LoadFromStrings( var anIndex: Integer; const aStrings: TStringList);
  end;


  TMarkovChain = class
  private
    FMaxSize     : Integer;
    FOrder       : Integer;
    FCyclic      : Boolean;
    FFillCount   : Integer;
    FValues      : TMarkovData;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FHistory     : TMarkovHistory;
    FScratch     : TSignalArray;
    FLastInValue : Integer;
    FLocked      : Boolean;
    FState       : Integer;
    FOverflow    : Boolean;
  private
    function    SearchPosition( aValue: TSignal): Integer;
    procedure   Forget;
    procedure   FailRead ( const aReason: string);
    procedure   FailWrite( const aReason: string);
  public
    constructor Create( DoCyclic: Boolean; aMaxSize, anOrder : Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Learn( aNewValue, aStrength: TSignal);
    function    NextValue: TSignal;
    procedure   SaveToStrings  ( const aStrings : TStringList);
    procedure   LoadFromStrings( const aStrings : TStringList);
    procedure   SaveToFile     ( const aFilename: string     );
    procedure   LoadFromFile   ( const aFileName: string     );
  public
    property    Cyclic   : Boolean read FCyclic write FCyclic;
    property    Overflow : Boolean read FOverflow;
  end;



implementation


  function   CheckStringPrefix( const S, aPrefix: string; anIndex: Integer): string;
  var
    p : Integer;
  begin
    p := Pos( UpperCase( aPrefix), UpperCase( S));

    if p > 0
    then Result := Copy( S, p + Length( aPrefix), Length( S))
    else raise EReadError.CreateFmt( 'Expected ''%s'' in line %d', [ aPrefix, anIndex]);
  end;


{ ========
  PMarkovSignalSet = ^TMarkovSignalSet;
  TMarkovSignalSet = record
  public
    Next     : PMarkovSignalSet;
    Index    : Integer;
    DataSize : Integer;
    Data     : TSignalArray;
  public
}

    procedure   TMarkovSignalSet.Initialize( aSize: Integer);
    begin
      DataSize := aSize;
      SetLength( Data, DataSize);
    end;


    function    TMarkovSignalSet.Hash: Byte;
    type
      TByteArray = array[ 0 .. SizeOf( TSignal) - 1] of Byte;
      PByteArray = ^TByteArray;
    var
      i : Integer;
      j : Integer;
    begin
      Result := $55;

      for i := 0 to DataSize - 1
      do begin
        for j := 0 to SizeOf( Data[ 0]) - 1
        do Result := Result xor PByteArray( @ Data[ i])^[ j];
      end;
    end;


    procedure   TMarkovSignalSet.Append( aSignalSet: PMarkovSignalSet);
    var
      aCurrent : PMarkovSignalSet;
      aNext    : PMarkovSignalSet;
      aCount   : Integer;
    begin
      if Assigned( aSignalSet)
      then begin
        aCurrent := @ Self;
        aNext    := Next;
        aCount   := 0;

        while Assigned( aNext)
        do begin
          aCurrent := aNext;
          aNext    := aNext^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;

        aCurrent^  .Next := aSignalSet;
        aSignalSet^.Next := nil;
      end;
    end;


    function    TMarkovSignalSet.Equals( aValue: PMarkovSignalSet): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      if Assigned( aValue)
      then begin
        Result := True;

        for i := 0 to DataSize - 1
        do begin
          if aValue^.Data[ i] <> Data[ i]
          then begin
            Result := False;
            Break;
          end;
        end;
      end;
    end;


    function    TMarkovSignalSet.Find( aValue: PMarkovSignalSet): Integer;
    var
      aCurrent : PMarkovSignalSet;
      aCount   : Integer;
    begin
      Result   := -1;
      aCurrent := @ Self;
      aCount   := 0;

      while Assigned( aCurrent)
      do begin
        if aCurrent^.Equals( aValue)
        then begin
          Result := aCurrent^.Index;
          Break;
        end
        else begin
          aCurrent := aCurrent^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;
      end;
    end;


    procedure   TMarkovSignalSet.Forget( aValue: PMarkovSignalSet);
    var
      aCurrent : PMarkovSignalSet;
      aPrev    : PMarkovSignalSet;
      aCount   : Integer;
    begin
      aCurrent := @ Self;
      aPrev    := nil;
      aCount   := 0;

      while Assigned( aCurrent)
      do begin
        if Equals( aValue)
        then Break
        else begin
          aPrev    := aCurrent;
          aCurrent := aCurrent^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;
      end;

      if Assigned( aPrev)
      then begin
        if Assigned( aCurrent)
        then aPrev.Next := aCurrent.Next
        else aPrev.Next := nil;
      end;
    end;


//    procedure   TMarkovSignalSet.SaveToStrings( anIndent: Integer; const aStrings: TStringList);
//    var
//      S : string;
//      i : Integer;
//    begin
//      if Assigned( aStrings)
//      then begin
//        S := '';
//
//        for i := 0 to DataSize - 1
//        do S := S + Format( ' %g', [ Data[ i]], AppLocale);
//
//        aStrings.Add( Format( '%sSignalSet(%s)', [ Indent( anIndent), S], AppLocale));
//      end;
//    end;



{ ========
  TMarkovSignalSets = array of TMarkovSignalSet;
  TMarkovSignalHash = array[ Byte] of PMarkovSignalSet;


  TMarkovData = class
  private
    FSize        : Integer;
    FOrder       : Integer;
    FSignals     : TMarkovSignalSets;
    FHashIndex   : TMarkovSignalHash;
    FScratchData : TMarkovSignalSet;
    FInPtr       : Integer;
    FOutPtr      : Integer;
    FFreeCount   : Integer;
    FFillCount   : Integer;
  public
}

    constructor TMarkovData.Create( aSize, anOrder: Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FOrder := anOrder;
      FSize  := aSize;
      SetLength( FSignals, FSize);

      for i := 0 to FSize - 1
      do FSignals[ i].Initialize( anOrder);

      FScratchData.Initialize( FOrder);
      Clear;
    end;


    procedure   TMarkovData.Clear;
    var
      i : Integer;
    begin
      FFreeCount := 0;
      FFillCount := 0;
      FInPtr     := 0;
      FOutPtr    := 0;

      for i := Low( FHashIndex) to High( FHashIndex)
      do FHashIndex[ i] := nil;

      FFreeCount := FSize;
    end;


    function    TMarkovData.FindSignalSet( aValue: PMarkovSignalSet): Integer;
    var
      aHash: Byte;
    begin
      Result := -1;
      aHash  := aValue.Hash;

      if Assigned( FHashIndex[ aHash])
      then Result := FHashIndex[ aHash]^.Find( aValue);
    end;


    function    TMarkovData.FindSignals( aData: TSignalArray): Integer;
    var
      i : Integer;
    begin
      for i := 0 to FScratchData.DataSize - 1
      do FScratchData.Data[ i] := aData[ i];

      Result := FindSignalSet( @ FScratchData);
    end;


    function    TMarkovData.AddSignalSet( aValue: PMarkovSignalSet): Integer;
    var
      aHash : Byte;
      i     : Integer;
    begin
      Result := - 1;

      if Assigned( aValue) and ( FFreeCount > 0)
      then begin
        Result := FInPtr;

        for i := 0 to aValue^.DataSize - 1
        do FSignals[ FInPtr].Data[ i] := aValue^.Data[ i];

        FSignals[ FInPtr].Index  := FInPtr;
        aHash := aValue^.Hash;

        if Assigned( FHashIndex[ aHash])
        then FHashIndex[ aHash]^.Append( @ FSignals[ FInPtr])
        else FHashIndex[ aHash] := @ FSignals[ FInPtr];

        FInPtr := ( FInPtr + 1) mod FSize;
        Inc( FFillCount);
        Dec( FFreeCount)
      end;
    end;


    function    TMarkovData.AddSignals( aData: TSignalArray): Integer;
    var
      i : Integer;
    begin
      for i := 0 to FScratchData.DataSize - 1
      do FScratchData.Data[ i] := aData[ i];

      Result := AddSignalSet( @ FScratchData);
    end;


    procedure   TMarkovData.RemoveSignalSet;
    var
      anItem : PMarkovSignalSet;
      aHash  : Byte;
    begin
      if FFillCount > 0
      then begin
        anItem := @ FSignals[ FOutPtr];
        aHash  := anItem^.Hash;

        if Assigned( FHashIndex[ aHash])
        then begin
          if anItem.Equals( FHashIndex[ aHash])
          then FHashIndex[ aHash] := FHashIndex[ aHash].Next
          else FHashIndex[ aHash]^.Forget( anItem);
        end;

        anItem^.Next := nil;
        FOutPtr := FOutPtr mod FSize;
        Dec( FFillCount);
        Inc( FFreeCount);
      end;
    end;


//    procedure   TMarkovData.SaveToStrings( anIndent: Integer; const aStrings: TStringList);
//    var
//      i : Integer;
//    begin
//      if Assigned( aStrings)
//      then begin
//        aStrings.Add( Format( '%sData('  , [ Indent( anIndent    )               ], AppLocale));
//        aStrings.Add( Format(   '%s%d %d', [ Indent( anIndent + 1), FSize, FOrder], AppLocale));
//
//        for i := 0 to FSize - 1
//        do FSignals[ i].SaveToStrings( anIndent + 1, aStrings);
//
//        aStrings.Add( Format( '%s)'      , [ Indent( anIndent    )               ], AppLocale));
//      end;
//    end;


//    procedure   TMarkovData.LoadSignalSet( var anIndex: Integer; const aStrings: TStringList);
//    var
//      S      : string;
//      i      : Integer;
//      aParts : TStringList;
//      aVal   : TSignal;
//      aSet   : TMarkovSignalSet;
//    begin
//      if Assigned( aStrings)
//      then begin
//        S       := CheckStringPrefix( Trim( aStrings[ anIndex]), 'SignalSet', anIndex + 1);
//        anIndex := anIndex + 1;
//        aSet.Initialize( FSize);
//
//        if ( Length( S) > 0) and ( S[ 1] = '(') and ( S[ Length( S)] = ')')
//        then begin
//          aParts := Explode( Trim( Copy( S, 2, Length( S) - 2)), ' ');
//
//          try
//            for i := 0 to FSize - 1
//            do begin
//              aVal := StrToFloatDef( Trim( aParts[ i]), NaN);
//
//              if IsNan( aVal)
//              then raise EReadError.CreateFmt( 'Invalid float (%s) in line %d', [ aParts[ i], anIndex]);
//
//              aSet.Data[ i] := aVal;
//            end;
//          finally
//            aParts.DisposeOf;
//          end;
//
//          AddSignalSet( @ aSet);
//        end
//        else raise EReadError.CreateFmt( 'Expected ''SignalSet(..)'' in line %d', [ anIndex]);
//      end;
//    end;


//    procedure   TMarkovData.LoadFromStrings( var anIndex: Integer; const aStrings: TStringList);
//    var
//      S      : string;
//      i      : Integer;
//      aParts : TStringList;
//      aVal   : Integer;
//    begin
//      if Assigned( aStrings)
//      then begin
//        S       := CheckStringPrefix( Trim( aStrings[ anIndex]), 'Data', anIndex + 1);
//        anIndex := anIndex + 1;
//
//        if SameText( S, 'Data(')
//        then begin
//          S       := Trim( aStrings[ anIndex]);
//          anIndex := anIndex + 1;
//          aParts  := Explode( S, ' ');
//
//          try
//            if aParts.Count = 2
//            then begin
//              S    := Trim( aParts[ 0]);
//              aVal := StrToIntDef( Trim( S), -1);
//
//              if aVal > 0
//              then begin
//                FSize := aVal;
//                S     := Trim( aParts[ 1]);
//                aVal  := StrToIntDef( Trim( S), -1);
//
//                if aVal > 0
//                then begin
//                  FOrder := aVal;
//
//                  SetLength( FSignals, FSize);
//                  FScratchData.Initialize( FOrder);
//                  Clear;
//
//                  for i := 0 to FSize - 1
//                  do LoadSignalSet( anIndex, aStrings);
//                end
//                else raise EReadError.CreateFmt( 'Expected a positive integer in line %d, but saw ''%s''', [ anIndex, S]);
//              end
//              else raise EReadError.CreateFmt( 'Expected a positive integer in line %d, but saw ''%s''', [ anIndex, S]);
//            end
//            else raise EReadError.CreateFmt( 'Expected two integer values in line %d, but saw ''%s''', [ anIndex, S]);
//          finally
//            aParts.DisposeOf;
//          end;
//        end
//        else raise EReadError.CreateFmt( 'Expected ''Data('' in line %d', [ anIndex]);
//
//      end;
//    end;


{ ========
  TMarkovHistory = record
  public
    DataSize  : Integer;
    Data      : TSignalArray;
    InPtr     : Integer;
    OutPtr    : Integer;
    FreeCount : Integer;
    FillCount : Integer;
  public
}

    procedure   TMarkovHistory.Initialize( aSize: Integer);
    begin
      DataSize := aSize;
      SetLength( Data, DataSize);
      InPtr     := 0;
      OutPtr    := 0;
      FillCount := 0;
      FreeCount := DataSize;
    end;


    procedure   TMarkovHistory.AddValue( aValue: TSignal);
    begin
      Data[ InPtr] := aValue;
      InPtr := ( InPtr + 1) mod DataSize;

      if FreeCount = 0
      then OutPtr := OutPtr + 1 mod DataSize
      else begin
        FillCount := FillCount + 1;
        FreeCount := FreeCount - 1;
      end;
    end;


    procedure   TMarkovHistory.Forget;
    begin
      Dec( FillCount);
      OutPtr := ( OutPtr + 1) mod DataSize;
      Inc( FreeCount);
    end;


    procedure   TMarkovHistory.FillData( aData: TSignalArray);
    var
      p : Integer;
      i : Integer;
    begin
      p := OutPtr;

      for i := 0 to DataSize - 1
      do begin
        aData[ i] := Data[ p];
        p := ( p + 1) mod DataSize;
      end;
    end;


//    procedure   TMarkovHistory.SaveToStrings( anIndent: Integer; const aStrings: TStringList);
//    var
//      S : string;
//      i : Integer;
//    begin
//      if Assigned( aStrings)
//      then begin
//        S := '';
//
//        for i := 0 to DataSize - 1
//        do S := S + Format( ' %g', [ Data[ i]], AppLocale);
//
//        aStrings.Add( Format( '%sHistory(%s)', [ Indent( anIndent), S], AppLocale));
//      end;
//    end;


//    procedure   TMarkovHistory.LoadFromStrings( var anIndex: Integer; const aStrings: TStringList);
//    var
//      S      : string;
//      i      : Integer;
//      aParts : TStringList;
//      aVal   : TSignal;
//    begin
//      if Assigned( aStrings)
//      then begin
//        S       := CheckStringPrefix( Trim( aStrings[ anIndex]), 'History', anIndex + 1);
//        anIndex := anIndex + 1;
//
//        if ( Length( S) > 0) and ( S[ 1] = '(') and ( S[ Length( S)] = ')')
//        then begin
//          aParts := Explode( Trim( Copy( S, 2, Length( S) - 2)), ' ');
//          Initialize( aParts.Count);
//
//          try
//            for i := 0 to DataSize - 1
//            do begin
//              aVal := StrToFloatDef( Trim( aParts[ i]), NaN);
//
//              if IsNan( aVal)
//              then raise EReadError.CreateFmt( 'Invalid float (%s) in line %d', [ aParts[ i], anIndex]);
//
//              Data[ i] := aVal;
//            end;
//          finally
//            aParts.DisposeOf;
//          end;
//        end
//        else raise EReadError.CreateFmt( 'Expected ''History(..)'' in line %d', [ anIndex]);
//      end;
//    end;


{ ========
  TMarkovChain = class
  private
    FMaxSize     : Integer;
    FOrder       : Integer;
    FCyclic      : Boolean;
    FFillCount   : Integer;
    FValues      : TMarkovData;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FHistory     : TMarkovHistory;
    FScratch     : TSignalArray;
    FLastInValue : Integer;
    FLocked      : Boolean;
    FState       : Integer;
    FOverflow    : Boolean;
  private
}

    function    TMarkovChain.SearchPosition( aValue: TSignal): Integer; // override;
    var
      P : Integer;
      L : Integer;
      H : Integer;
    begin
      L      := 0;
      H      := FFillCount - 1;
      Result := 0;

      while L <= H
      do begin
        P := ( L + H) div 2;

        if aValue < FLookups[ P]
        then H := P - 1
        else if ( P = H) or ( aValue < FLookups[ P + 1])
        then begin
          Result := P;
          Break;
        end
        else L := P + 1;
      end;
    end;


    procedure   TMarkovChain.Forget; // override;
    begin
      if FMaxSize > 0
      then begin
        FValues.RemoveSignalSet;

        if FLastInValue >= 0
        then Dec( FLastInValue);

        if FLastInValue < 0
        then FHistory.Forget;
      end;
    end;


    procedure   TMarkovChain.FailRead( const aReason: string);
    begin
      if not FLocked
      then FLocked := True;

      Clear;
      FLocked := False;

      raise EMarkovReadFail.CreateFmt( 'file read error: %s', [ aReason]);
    end;


    procedure   TMarkovChain.FailWrite( const aReason: string);
    begin
      FLocked := False;
      raise EMarkovWriteFail.CreateFmt( 'file write error: %d', [ aReason]);
    end;


//  public

    constructor TMarkovChain.Create( DoCyclic: Boolean; aMaxSize, anOrder : Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FLocked  := True;
      FMaxSize := Clip( aMaxSize, 1, MAX_MARKOV_LENGTH);
      FOrder   := Clip( anOrder , 1, MAX_MARKOV_ORDER );
      FCyclic  := DoCyclic;
      FValues  := TMarkovData.Create( FMaxSize, FOrder);
      SetLength( FLookups, FMaxSize);
      SetLength( FChances, FMaxSize);
      SetLength( FScratch, FOrder);
      FHistory.Initialize( FOrder);

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], FMaxSize);

      Clear;
      FLocked := False;
    end;


    destructor  TMarkovChain.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;
      FValues.DisposeOf;

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], 0);

      SetLength( FChances, 0);
      inherited;
    end;


    procedure   TMarkovChain.Clear; // override;
    var
      i : Integer;
      j : Integer;
    begin
      FLocked := True;
      FValues.Clear;

      for i := 0 to FMaxSize - 1
      do begin
        for j := 0 to FMaxSize - 1
        do FChances[ i, j] := 0;
      end;

      FFillCount   := 0;
      FLastInValue := -1;
      FOverflow    := False;

      FHistory.Initialize( FOrder);
      FLocked := False;
    end;


    procedure   TMarkovChain.Learn( aNewValue, aStrength: TSignal); // override;
    var
      p : Integer;
    begin
      if FLocked
      then Exit;

      if FHistory.FreeCount = 0
      then begin
        FHistory.AddValue( aNewValue);
        FHistory.FillData( FScratch);
        p := FValues.FindSignals( FScratch);

        if p < 0
        then begin
          if FFillCount >= FMaxSize
          then begin
            if Cyclic
            then begin
              Forget;

              if FHistory.FreeCount = 0
              then begin
                p := FValues.AddSignals( FScratch);
                FFillCount := FMaxSize;
              end;
            end
            else FOverflow := True;
          end
          else begin
            p := FValues.AddSignals( FScratch);
            Inc( FFillCount);
          end;
        end;

        if p >= 0
        then begin
          if FLastInValue >= 0
          then FChances[ FLastInValue, p] := Max( 0.0, FChances[ FLastInValue, p] + Clip( aStrength, -1.0, 1.0));

          FLastInValue := p;
        end;
      end
      else FHistory.AddValue( aNewValue);
    end;


    function    TMarkovChain.NextValue: TSignal; // override;
    var
      i   : Integer;
      Sum : TSignal;
    begin
      Result := 0;

      try
        if FFillCount > 0
        then begin
          if ( FState >= 0) and ( FState < FFillCount)
          then begin
            Sum := 0;

            for i := 0 to FFillCount - 1
            do begin
              FLookups[ i] := Sum;
              Sum          := Sum + FChances[ FState, i];
            end;

            if Sum > 0
            then FState := SearchPosition( Sum * Random)
            else FState := Random( FFillCount);
          end
          else FState := Random( FFillCount);

          Result := FValues.FSignals[ FState].Data[ FOrder - 1];
        end;
      except
        Nop;
      end;
    end;


    procedure   TMarkovChain.SaveToStrings( const aStrings : TStringList); // override;
    var
      WasLocked : Boolean;
    begin
      WasLocked := FLocked;
      FLocked   := True;

      try
        FailWrite( 'not implemented yet');
      finally
        FLocked := WasLocked;
      end;
    end;


    procedure   TMarkovChain.LoadFromStrings( const aStrings : TStringList); // override;
    var
      WasLocked : Boolean;
    begin
      WasLocked := FLocked;
      FLocked   := True;

      try
        FailRead( 'not implemented yet');
      finally
        FLocked := WasLocked;
      end;
    end;


    procedure   TMarkovChain.SaveToFile( const aFilename: string); // override;
    var
      aStrings: TStringList;
    begin
      aStrings := TStringList.Create;

      try
        SaveToStrings( aStrings);
        aStrings.SaveToFile( aFilename);
      finally
        aStrings.DisposeOf;
      end;
    end;


    procedure   TMarkovChain.LoadFromFile( const aFileName: string); // override;
    var
      aStrings: TStringList;
    begin
      aStrings := TStringList.Create;

      try
        aStrings.LoadFromFile( aFileName);
        LoadFromStrings( aStrings);
      finally
        aStrings.DisposeOf;
      end;
    end;


end.

