{ *********************************************************************** }
{ Unit de calcul matriciel rel }
{ }
{ Version   : 1.0 }
{ Cre le   : 14/05/2006 }
{ Objectifs : -Cration d'une class TMatrix et dfinitions de diffrentes )
  { fonctions et oprateurs rels agissant sur ces matrices. }
{ -Dcompositions de matrices LU, QR, QL, Hessenberg. }
{ -Diagonalisation d'une matrice symtrique relle. }
{ *********************************************************************** }

(*
Un petit logiciel de
Morphing, bas sur des
transformations
progressives de
quadrilateres.

Soyez indulgents, dure
de codage: 1 journe :))

Nicoo
<bigbezus@free.fr>

Vos commentaires sont
les bienvenus !!!!
 *)


unit Matrix;

interface

uses

  Classes, SysUtils, Graphics;

const

  coul : array[ 1 .. 10] of Tcolor = ( clblue, clred, clyellow, clgreen, claqua, clpurple, clfuchsia, clLime, clNavy, clolive);

type
  Tfnc = function( v : extended) : extended;

type
  TMatrix = class
  private
    FSquare  : boolean;
    Fsym     : boolean;
    FLo      : boolean;
    FUp      : boolean;
    FDiag    : boolean;
    FHess    : boolean;
    FTridiag : boolean;
  public
    Cells : array of array of extended;
  private
    function    GetRowCount : integer;
    procedure   SetRowCount( RC : integer);
    function    GetColCount : integer;
    procedure   SetColCount( CC : integer);
  public
    constructor create( RC, CC : integer);
  public
    procedure   define;
    procedure   Transpose;
    procedure   Add( M : TMatrix);
    procedure   Sub( M : TMatrix);
    procedure   Mul( M : TMatrix);
    procedure   LMul( M : TMatrix);
    procedure   SMul( s : extended);
    procedure   ArMul( M : TMatrix);
    procedure   Inv;
    procedure   func( f : Tfnc);
  public
    procedure   Zeros( RC, CC : integer);
    procedure   Ones( RC, CC : integer);
    procedure   Eye( RC, CC : integer);
    procedure   Rand( RC, CC : integer);
    procedure   LinSpace( a, b : extended; n : integer);
  public
    procedure   SpDiags( M : TMatrix; d : integer);
    procedure   Diags( M : TMatrix);
    procedure   ExtractMat( M : TMatrix; R1, Rn, C1, Cn : integer);
  public
    property    RowCount : integer read GetRowCount write SetRowCount;
    property    ColCount : integer read GetColCount write SetColCount;
    property    Square   : boolean read FSquare;
    property    Sym      : boolean read Fsym;
    property    Lo       : boolean read FLo;
    property    Up       : boolean read FUp;
    property    Diag     : boolean read FDiag;
    property    Tridiag  : boolean read FTridiag;
    property    Hess     : boolean read FHess;
  end;


type

  TDoubleMat = array [ 1 .. 2] of TMatrix; // sert aux dcompositionx des matrices

  { ***************Oprations lmentaires sur les matrices**************** }

  function  Transpose( M : TMatrix) : TMatrix;
  function  Add( M1, M2 : TMatrix) : TMatrix;
  function  Sub( M1, M2 : TMatrix) : TMatrix;
  function  Mul( M1, M2 : TMatrix) : TMatrix;
  function  SMul( s : extended; M : TMatrix) : TMatrix;
  function  ArMul( M1, M2 : TMatrix) : TMatrix;
  function  Inv( M : TMatrix) : TMatrix;
  function  det( M : TMatrix) : extended;
  function  func( M : TMatrix; f : Tfnc) : TMatrix;

  { *******Cration de diffrents types de matrices et vecteurs************ }
  function  Copy( M : TMatrix) : TMatrix;
  function  Rand( RC, CC : integer) : TMatrix;
  function  Zeros( RC, CC : integer) : TMatrix;
  function  Eye( RC, CC : integer) : TMatrix;
  function  Ones( RC, CC : integer) : TMatrix;
  function  LinSpace( a, b : extended; n : integer) : TMatrix;
  function  SpDiags( M : TMatrix; d : integer) : TMatrix;
  function  Diags( M : TMatrix) : TMatrix;
  function  ExtractMat( M : TMatrix; R1, Rn, C1, Cn : integer) : TMatrix;

  { *********Dfinition des fonctions agissant sur des matrices************ }
  { ********************et gnrant des scalaires************************** }
  function  NormR( M : TMatrix; R : integer) : extended;
  function  NormC( M : TMatrix; C : integer) : extended;
  function  ScalProdR( M1, M2 : TMatrix; R1, R2 : integer) : extended;
  function  ScalProdC( M1, M2 : TMatrix; C1, C2 : integer) : extended;
  function  ScalProdRC( M1, M2 : TMatrix; R1, C2 : integer) : extended;
  function  Max( M : TMatrix) : extended;
  function  Min( M : TMatrix) : extended;
  function  SumR( M : TMatrix; R : integer) : extended;
  function  SumC( M : TMatrix; C : integer) : extended;
  function  sqrM( v : extended) : extended;
  function  sqrtM( v : extended) : extended;
  function  sinM( v : extended) : extended;
  function  cosM( v : extended) : extended;
  function  expM( v : extended) : extended;

  { ********************Dcompositions de matrices************************* }
  function  LU( M : TMatrix) : TDoubleMat;
  function  QR( M : TMatrix) : TDoubleMat;
  function  QL( M : TMatrix) : TDoubleMat;
  function  Hess( M : TMatrix) : TDoubleMat;

  { ******************Diagonalisation de matrices************************** }
  function  eigvals( M : TMatrix) : TMatrix;
  function  eig( M : TMatrix) : TDoubleMat;
  function  Schur( M : TMatrix) : TDoubleMat;

  { ******************Reprsentations des matrices************************* }
  function  MatToStr( M : TMatrix) : TStringList;
  procedure MatToBmp( M : TMatrix; Width, Height : integer; bmp : Tbitmap; mode : integer);
  function  Col( v : extended) : integer;



implementation



    { ***********************Gestion de la matrice*************************** }

    constructor TMatrix.create( RC, CC : integer);
    begin
      RowCount := RC;
      ColCount := CC;
    end;

    { destructor TMatrix.destroy;
      var n:integer;
      begin
      inherited;
      end; }

    function TMatrix.GetRowCount : integer;
    begin
      GetRowCount := length( Cells) - 1;
    end;

    procedure TMatrix.SetRowCount( RC : integer);
    begin
      SetLength( Cells, RC + 1);
    end;

    function TMatrix.GetColCount : integer;
    begin
      GetColCount := length( Cells[ 1]) - 1;
    end;

    procedure TMatrix.SetColCount( CC : integer);
    var
      n : integer;
    begin
      for n := 1 to RowCount do
      begin
        SetLength( Cells[ n], CC + 1);
      end;
    end;

    function Copy( M : TMatrix) : TMatrix;
    var
      i, j : integer;
    begin
      // Copy renvoit une copie de M
      RESULT          := TMatrix.create( M.RowCount, M.ColCount);
      RESULT.FSquare  := M.FSquare;
      RESULT.Fsym     := M.Fsym;
      RESULT.FLo      := M.FLo;
      RESULT.FUp      := M.FUp;
      RESULT.FDiag    := M.FDiag;
      RESULT.FHess    := M.FHess;
      RESULT.FTridiag := M.FTridiag;
      for i           := 1 to M.RowCount do
      begin
        for j := 1 to M.ColCount do
        begin
          RESULT.Cells[ i, j] := M.Cells[ i, j];
        end;
      end;
    end;

    procedure TMatrix.define;
    var
      i, j, n : integer;
      a, b    : extended;
    begin
      // dfinit les proprites de la matrice, ces proprits sont utilises par les
      // diffrentes fonctions pour choisir l'algorithme adapt au type de matrice
      n := RowCount;
      if n = ColCount
      then
      begin
        FSquare  := True;
        Fsym     := True;
        FLo      := True;
        FUp      := True;
        FDiag    := True;
        FHess    := True;
        FTridiag := True;
        for i    := 1 to n do
        begin
          for j := 1 to n do
          begin
            if i < j
            then
            begin
              a := Cells[ i, j];
              b := Cells[ j, i];
              if a <> b
              then
                Fsym := False;
              if a <> 0
              then
              begin
                FDiag := False;
                FLo   := False;
                if j > i + 1
                then
                  FTridiag := False;
              end;
            end;
            if i > j
            then
            begin
              a := Cells[ i, j];
              b := Cells[ j, i];
              if a <> b
              then
                Fsym := False;
              if a <> 0
              then
              begin
                FDiag := False;
                FUp   := False;
                if i > j + 1
                then
                begin
                  FTridiag := False;
                  FHess    := False;
                end;
              end;
            end;
          end;
        end;
      end
      else
      begin
        FSquare  := False;
        Fsym     := False;
        FLo      := False;
        FUp      := False;
        FDiag    := False;
        FHess    := False;
        FTridiag := False;
      end;
    end;

    { ***************Oprations lmentaires sur les matrices**************** }

    procedure TMatrix.Transpose;
    var
      i, j, RC, CC : integer;
      M1           : TMatrix;
    begin
      // calcul de la transpose
      RC       := RowCount;
      CC       := ColCount;
      M1       := Copy( Self);
      RowCount := CC;
      ColCount := RC;
      for i    := 1 to RowCount do
      begin
        for j := 1 to ColCount do
        begin
          Cells[ i, j] := M1.Cells[ j, i];
        end;
      end;
      M1.Destroy;
    end;

    procedure TMatrix.Add( M : TMatrix);
    var
      i, j, RC, CC : integer;
    begin
      // addition, Self:=Self+M
      RC    := M.RowCount;
      CC    := M.ColCount;
      for i := 1 to RC do
      begin
        for j := 1 to CC do
        begin
          Cells[ i, j] := Cells[ i, j] + M.Cells[ i, j];
        end;
      end;
    end;

    procedure TMatrix.Sub( M : TMatrix);
    var
      i, j, RC, CC : integer;
    begin
      // soustraction  Self:=Self-M
      RC := M.RowCount;
      CC := M.ColCount;
      if ( RC = RowCount) and ( CC = ColCount)
      then
      begin
        for i := 1 to RC do
        begin
          for j := 1 to CC do
          begin
            Cells[ i, j] := Cells[ i, j] - M.Cells[ i, j];
          end;
        end;
      end;
    end;

    procedure TMatrix.Mul( M : TMatrix);
    var
      i, j, k, RC1, CC1, RC2, CC2 : integer;
      M1ik                        : extended;
      M1                          : TMatrix;
    begin
      // multiplication  droite Self:=Self*M
      RC1 := RowCount;
      CC1 := ColCount;
      RC2 := M.RowCount;
      CC2 := M.ColCount;
      if ( CC1 = RC2)
      then
      begin
        M1       := Copy( Self);
        RowCount := RC1;
        ColCount := CC2;
        for i    := 1 to RC1 do
        begin
          for j := 1 to CC2 do
          begin
            Cells[ i, j] := 0;
          end;
        end;
        for i := 1 to RC1 do
        begin
          for k := 1 to RC2 do
          begin
            M1ik  := M1.Cells[ i, k];
            for j := 1 to CC2 do
            begin
              Cells[ i, j] := Cells[ i, j] + M1ik * M.Cells[ k, j];
            end;
          end;
        end;
        M1.Destroy;
      end;
    end;

    procedure TMatrix.LMul( M : TMatrix);
    var
      i, j, k, RC1, CC1, RC2, CC2 : integer;
      M1ik                        : extended;
      M1                          : TMatrix;
    begin
      // multiplication  gauche Self:=M*Self
      RC1 := RowCount;
      CC1 := ColCount;
      RC2 := M.RowCount;
      CC2 := M.ColCount;
      if ( CC1 = RC2)
      then
      begin
        M1       := Copy( Self);
        RowCount := RC1;
        ColCount := CC2;
        for i    := 1 to RC1 do
        begin
          for j := 1 to CC2 do
          begin
            Cells[ i, j] := 0;
          end;
        end;
        for i := 1 to RC1 do
        begin
          for k := 1 to RC2 do
          begin
            M1ik  := M.Cells[ i, k];
            for j := 1 to CC2 do
            begin
              Cells[ i, j] := Cells[ i, j] + M1ik * M1.Cells[ k, j];
            end;
          end;
        end;
        M1.Destroy;
      end;
    end;

    procedure TMatrix.SMul( s : extended);
    var
      i, j : integer;
    begin
      // multiplication scalaire Self:=s*Self
      for i := 1 to RowCount do
      begin
        for j := 1 to ColCount do
        begin
          Cells[ i, j] := s * Cells[ i, j];
        end;
      end;
    end;

    procedure TMatrix.ArMul( M : TMatrix);
    var
      i, j, RC1, CC1, RC2, CC2 : integer;
    begin
      // multiplication des lments des 2 matrices: aij:=aij*bij
      RC1 := RowCount;
      CC1 := ColCount;
      RC2 := M.RowCount;
      CC2 := M.ColCount;
      if ( CC1 = CC2) and ( RC1 = RC2)
      then
      begin
        for i := 1 to RC1 do
        begin
          for j := 1 to CC1 do
          begin
            Cells[ i, j] := Cells[ i, j] * M.Cells[ i, j];
          end;
        end;
      end;
    end;

    procedure TMatrix.Inv;
    var
      i, j, l, k, n : integer;
      U             : TMatrix;
      v             : array of extended;
      Vsup          : extended;
    label
      L0, L1;
    begin
      { Calcul de l'inverse de la matrice Self:=1/Self
        L'algorithme effectue la factorisation Q*U*L
        Ou Q est une matrice de rotation, U une matrice triangulaire sup
        de determinant 1 et L est une matrice triang inf
        Si la matrice n'est pas inversible on renvoit nil }

      if RowCount = ColCount
      then
      begin
        n := RowCount;
        SetLength( v, n + 1);
        U := Copy( Self);
        Self.Eye( n, n); // M=Id

        for l := n downto 2 do
        begin
          Vsup  := 1e-16;
          k     := 1;
          for j := 1 to l do
          begin
            if abs( U.Cells[ l, j]) > Vsup
            then
            begin
              Vsup := abs( U.Cells[ l, j]);
              k    := j; // On cherche le plus grand lment de la l-ieme ligne pour en faire le pivot
            end;
          end;
          if Vsup = 1e-16
          then
            goto L0; // matrice non inversible

          for i := 1 to n do
          begin
            v[ i] := U.Cells[ i, l];
          end;
          for i := 1 to n do
          begin // echange de colonnes  Cl<->Ck
            U.Cells[ i, l] := U.Cells[ i, k];
            U.Cells[ i, k] := v[ i];
          end;

          for i := 1 to n do
          begin
            v[ i] := Cells[ i, l]; // On rpercute l'change sur M
          end;
          for i := 1 to n do
          begin
            Cells[ i, l] := Cells[ i, k];
            Cells[ i, k] := v[ i];
          end;

          for j := 1 to l - 1 do
          begin
            v[ j] := - U.Cells[ l, j] / U.Cells[ l, l]; // dfinition des pivots
          end;

          for i := 1 to n do
          begin
            for j := 1 to l - 1 do
            begin
              Cells[ i, j] := Cells[ i, j] + v[ j] * Cells[ i, l];
            end;
          end;
          for i := 1 to l - 1 do
          begin
            for j := 1 to l - 1 do
            begin
              U.Cells[ i, j] := U.Cells[ i, j] + v[ j] * U.Cells[ i, l]; // ligne l de l-1 zeros
            end;
          end;
        end;
        // U est triang sup
        // M=Q*U*L*Q'
        for l := n - 1 downto 1 do
        begin
          for j := n - l + 1 to n do
          begin
            if abs( U.Cells[ n - l, n - l]) < 1e-8
            then
              goto L0;
            v[ j] := - U.Cells[ n - l, j] / U.Cells[ n - l, n - l];
          end;

          for i := 1 to n do
          begin
            for j := n - l + 1 to n do
            begin
              Cells[ i, j] := Cells[ i, j] + v[ j] * Cells[ i, n - l];
            end;
          end;
        end;

        for i := 1 to n do
        begin
          v[ i] := 1 / U.Cells[ i, i];
        end;
        for i := 1 to n do
        begin
          for j := 1 to n do
          begin
            Cells[ i, j] := v[ j] * Cells[ i, j];
          end;
        end;
        goto L1;
      L0 :
        // Self:=nil;
      L1 :
        U.Destroy;
        SetLength( v, 0);
      end; // end else Self:=nil;
    end;

    function det( M : TMatrix) : extended;
    var
      i, j, l, k, n : integer;
      U             : TMatrix;
      v             : array of extended;
      Vsup          : extended;
    label
      L0, L1;
    begin
      { Calcul du determinant de M
        La mthode utilise est galement M=Q*U*L ou seul U est calcul explicitement
        les autres matrices ayant des dterminants unitaires }
      if M.RowCount = M.ColCount
      then
      begin
        n := M.RowCount;
        SetLength( v, n + 1);
        U      := Copy( M);
        RESULT := 1;
        for l  := n downto 2 do
        begin
          Vsup  := 1e-16;
          k     := 1;
          for j := 1 to l do
          begin
            if abs( U.Cells[ l, j]) > Vsup
            then
            begin
              Vsup := abs( U.Cells[ l, j]);
              k    := j;
            end;
          end;
          if l <> k
          then
            RESULT := - RESULT; // signature de la permutation=det( Q)=-1 ou +1

          if Vsup = 1e-16
          then
            goto L0; // matrice non inversible

          for i := 1 to n do
          begin
            v[ i] := U.Cells[ i, l];
          end;
          for i := 1 to n do
          begin // echange de colonnes  Cl<->Ck
            U.Cells[ i, l] := U.Cells[ i, k];
            U.Cells[ i, k] := v[ i];
          end;
          for j := 1 to l - 1 do
          begin
            v[ j] := - U.Cells[ l, j] / U.Cells[ l, l];
          end;

          for i := 1 to l - 1 do
          begin
            for j := 1 to l - 1 do
            begin
              U.Cells[ i, j] := U.Cells[ i, j] + v[ j] * U.Cells[ i, l];
            end;
          end;
          for j := 1 to l - 1 do
          begin
            U.Cells[ l, j] := 0; // ligne l de l-1 zeros
          end;
        end;
        // U est triang sup
        for i := 1 to n do
        begin
          RESULT := RESULT * U.Cells[ i, i];
        end;
        goto L1;
      L0 :
        det := 0;
      L1 :
        U.Destroy;
        SetLength( v, 0);
      end
      else
        det := 0;
    end;

    procedure TMatrix.func( f : Tfnc);
    var
      i, j : integer;
    begin
      // On applique une fonction  chaque lment de la matrice
      for i := 1 to RowCount do
      begin
        for j := 1 to ColCount do
        begin
          Cells[ i, j] := f( Cells[ i, j]);
        end;
      end;
    end;

    { *******Cration de diffrents types de matrices et vecteurs************ }

    procedure TMatrix.Zeros( RC, CC : integer);
    var
      i, j : integer;
    begin
      // forme une matrice nulle de RC lignes et CC colonne
      RowCount := RC;
      ColCount := CC;
      for i    := 1 to RC do
      begin
        for j := 1 to CC do
        begin
          Cells[ i, j] := 0;
        end;
      end;
    end;

    procedure TMatrix.Ones( RC, CC : integer);
    var
      i, j : integer;
    begin
      // forme une matrice forme de 1 de RC lignes et CC colonne
      RowCount := RC;
      ColCount := CC;
      for i    := 1 to RC do
      begin
        for j := 1 to CC do
        begin
          Cells[ i, j] := 1;
        end;
      end;
    end;

    procedure TMatrix.Eye( RC, CC : integer);
    var
      i, j : integer;
    begin
      // forme une matrice identite de RC lignes et CC colonne
      RowCount := RC;
      ColCount := CC;
      for i    := 1 to RC do
      begin
        for j := 1 to CC do
        begin
          if i = j
          then
            Cells[ i, j] := 1
          else
            Cells[ i, j] := 0;
        end;
      end;
    end;

    procedure TMatrix.Rand( RC : integer; CC : integer);
    var
      i, j : integer;
      Norm : extended;
    begin
      // Forme une matrice alatoire de RC lignes et CC colonne
      // les lments de matrice sont compris entre -1 et +1
      RowCount := RC;
      ColCount := CC;
      // randomize;
      Norm  := 1 / 10000;
      for i := 1 to RC do
      begin
        for j := 1 to CC do
        begin
          Cells[ i, j] := Norm * ( random( 20000) - 10000);
        end;
      end;
    end;

    procedure TMatrix.LinSpace( a, b : extended; n : integer);
    var
      j : integer;
    begin
      // Forme un vecteur ligne de dimension n
      // dont les valeurs forment une progression linaire
      // entre a et b
      RowCount := 1;
      ColCount := n;
      for j    := 1 to n do
      begin
        Cells[ 1, j] := a + ( j - 1) * ( b - a) / ( n - 1);
      end;
    end;

    procedure TMatrix.SpDiags( M : TMatrix; d : integer);
    var
      i, k, e, RC, CC : integer;
    begin
      { Forme une matrice carre de taille gale au nombre de colonne de M
        SpDiags place les lignes de M dans les diagonales de la matrice
        d est le numro de la premire diagonale  remplir
        Les autres diagonales sont mises  0 }
      CC       := M.ColCount;
      RC       := M.RowCount;
      RowCount := CC;
      ColCount := CC;
      for i    := 1 to CC do
      begin
        for k := 1 to CC do
        begin
          Cells[ i, k] := 0;
        end;
      end;
      for k := 1 to RC do
      begin
        e := d - k + 2;
        if e > 0
        then
          for i := e to CC do
          begin
            Cells[ i - e + 1, i] := M.Cells[ k, i - e + 1];
          end
        else
          for i := 1 to CC + e - 1 do
          begin
            Cells[ i - e + 1, i] := M.Cells[ k, i];
          end;
      end;
    end;

    procedure TMatrix.Diags( M : TMatrix);
    var
      j, RC : integer;
      M2    : TMatrix;
    begin
      // Forme un vecteur ligne de la diagonale de M
      M2       := Copy( M);
      RC       := M.RowCount;
      RowCount := 1;
      ColCount := RC;
      for j    := 1 to RC do
      begin
        Cells[ 1, j] := M2.Cells[ j, j];
      end;
      M2.Destroy;
    end;

    procedure TMatrix.ExtractMat( M : TMatrix; R1, Rn, C1, Cn : integer);
    var
      i, j : integer;
    begin
      // Extrait la matrice comprise entre les lignes R1 et Rn
      // et les colonnes C1 et Cn
      for i := R1 to Rn do
      begin
        for j := C1 to Cn do
        begin
          Cells[ i - R1 + 1, j - C1 + 1] := M.Cells[ i, j];
        end;
      end;
      RowCount := Rn - R1 + 1;
      ColCount := Cn - C1 + 1;
    end;

  { **********Dfinition des fonctions agissant sur des matrices*********** }
  { ****************et gnrant de nouvelles matrices********************** }

  function Transpose( M : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M);
    RESULT.Transpose;
  end;

  function Add( M1, M2 : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M1);
    RESULT.Add( M2);
  end;

  function Sub( M1, M2 : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M1);
    RESULT.Sub( M2);
  end;

  function Mul( M1, M2 : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M1);
    RESULT.Mul( M2);
  end;

  function Inv( M : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M);
    RESULT.Inv;
  end;

  function SMul( s : extended; M : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M);
    RESULT.SMul( s);
  end;

  function ArMul( M1, M2 : TMatrix) : TMatrix;
  begin
    RESULT := Copy( M1);
    RESULT.ArMul( M2);
  end;

  function func( M : TMatrix; f : Tfnc) : TMatrix;
  begin
    RESULT := Copy( M);
    RESULT.func( f);
  end;

  function Rand( RC, CC : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( RC, CC);
    RESULT.Rand( RC, CC);
  end;

  function Zeros( RC, CC : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( RC, CC);
    RESULT.Zeros( RC, CC);
  end;

  function Eye( RC, CC : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( RC, CC);
    RESULT.Eye( RC, CC);
  end;

  function Ones( RC, CC : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( RC, CC);
    RESULT.Ones( RC, CC);
  end;

  function LinSpace( a, b : extended; n : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( 1, n);
    RESULT.LinSpace( a, b, n);
  end;

  function SpDiags( M : TMatrix; d : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( M.ColCount, M.ColCount);
    RESULT.SpDiags( M, d);
  end;

  function Diags( M : TMatrix) : TMatrix;
  begin
    RESULT := TMatrix.create( 1, M.RowCount);
    RESULT.Diags( M);
  end;

  function ExtractMat( M : TMatrix; R1, Rn, C1, Cn : integer) : TMatrix;
  begin
    RESULT := TMatrix.create( Rn - R1 + 1, Cn - C1 + 1);
    RESULT.ExtractMat( M, R1, Rn, C1, Cn);
  end;

  { *********Dfinition des fonctions agissant sur des matrices************ }
  { ********************et gnrant des scalaires************************** }

  { function ln( v:extended):extended; overload;
    begin
    ln:=sqr( v);
    end; }

  // On dfinit quelques fonctions qui seront utilises par la procedure func
  function sqrM( v : extended) : extended;
  begin
    sqrM := sqr( v);
  end;

  function sqrtM( v : extended) : extended;
  begin
    sqrtM := sqrt( v);
  end;

  function sinM( v : extended) : extended;
  begin
    sinM := sin( v);
  end;

  function cosM( v : extended) : extended;
  begin
    cosM := cos( v);
  end;

  function expM( v : extended) : extended;
  begin
    expM := exp( v);
  end;

  function NormR( M : TMatrix; R : integer) : extended;
  var
    j : integer;
  begin
    // calcul de la norme euclidienne de la ligne R de M
    RESULT := 0;
    for j  := 1 to M.ColCount do
    begin
      RESULT := RESULT + sqr( M.Cells[ R, j]);
    end;
    RESULT := sqrt( RESULT);
  end;

  function NormC( M : TMatrix; C : integer) : extended;
  var
    i : integer;
  begin
    // Calcul de la norme euclidienne de la colonne C de M
    RESULT := 0;
    for i  := 1 to M.ColCount do
    begin
      RESULT := RESULT + sqr( M.Cells[ i, C]);
    end;
    RESULT := sqrt( RESULT);
  end;

  function ScalProdR( M1, M2 : TMatrix; R1, R2 : integer) : extended;
  var
    j : integer;
  begin
    // Calcul du produit scalaire de la ligne R1 de M1 avec la ligne R2 de M2
    RESULT := 0;
    for j  := 1 to M1.ColCount do
    begin
      RESULT := RESULT + M1.Cells[ R1, j] * M2.Cells[ R2, j];
    end;
  end;

  function ScalProdC( M1, M2 : TMatrix; C1, C2 : integer) : extended;
  var
    i : integer;
  begin
    // Calcul du produit scalaire de la colonne C1 de M1 avec la colonne C2 de M2
    RESULT := 0;
    for i  := 1 to M1.RowCount do
    begin
      RESULT := RESULT + M1.Cells[ i, C1] * M2.Cells[ i, C2];
    end;
  end;

  function ScalProdRC( M1, M2 : TMatrix; R1, C2 : integer) : extended;
  var
    i : integer;
  begin
    // Calcul du produit scalaire de la ligne R1 de M1 avec la colonne C2 de M2
    RESULT := 0;
    for i  := 1 to M1.ColCount do
    begin
      RESULT := RESULT + M1.Cells[ R1, i] * M2.Cells[ i, C2];
    end;
  end;

  function Min( M : TMatrix) : extended;
  var
    i, j : integer;
    emin : extended;
  begin
    // recherche de l'lment minimal de M
    emin  := M.Cells[ 1, 1];
    for i := 1 to M.RowCount do
    begin
      for j := 1 to M.ColCount do
      begin
        if M.Cells[ i, j] < emin
        then
          emin := M.Cells[ i, j];
      end;
    end;
    RESULT := emin;
  end;

  function Max( M : TMatrix) : extended;
  var
    i, j : integer;
    emax : extended;
  begin
    // recherche de l'lment maximal de M
    emax  := M.Cells[ 1, 1];
    for i := 1 to M.RowCount do
    begin
      for j := 1 to M.ColCount do
      begin
        if M.Cells[ i, j] > emax
        then
          emax := M.Cells[ i, j];
      end;
    end;
    RESULT := emax;
  end;

  function SumR( M : TMatrix; R : integer) : extended;
  var
    j : integer;
  begin
    // Somme des lments prsents sur la ligne R de M
    RESULT := 0;
    for j  := 1 to M.ColCount do
    begin
      RESULT := RESULT + M.Cells[ R, j];
    end;
  end;

  function SumC( M : TMatrix; C : integer) : extended;
  var
    i : integer;
  begin
    // Somme des lments prsents sur la colonne C de M
    RESULT := 0;
    for i  := 1 to M.RowCount do
    begin
      RESULT := RESULT + M.Cells[ i, C];
    end;
  end;

  { ******************Reprsentations des matrices************************* }

  function MatToStr( M : TMatrix) : TStringList;
  var
    RC, CC, i, j : integer;
    Ri           : string;
  begin
    // Renvoit le contenu de M sous forme d'un TStringList
    RC     := M.RowCount;
    CC     := M.ColCount;
    RESULT := TStringList.create;
    for i  := 1 to RC do
    begin
      Ri    := '';
      for j := 1 to CC do
      begin
        Ri := Ri + '   ' + floattostr( M.Cells[ i, j]);
      end;
      RESULT.Add( Ri);
    end;
  end;

  procedure MatToBmp( M : TMatrix; Width, Height : integer; bmp : Tbitmap; mode : integer);
  var
    RC, CC, i, j        : integer;
    Vmin, Vmax, v, Vmoy : extended;
    H, dx, dy           : extended;
  begin
    { Reprsentation graphique de la matrice M selon diffrents modes sur bmp
      Le dessin aura une hauteur height et une largeur width
      mode dsigne le mode de rprsentation choisi:

      mode 0 et mode 1: affiche les lments de la matrices suivant x et y
      la couleur reprsente alors l'amplitude de chaque lment suivant un dgrad
      de l'amplitude la plus faible ( bleu)  l'amplitude la plus forte ( rouge)
      Dans le mode 1, les 0 sont affichs en noir et les 1 en blanc

      mode 2:  trace les courbes correspondant aux diffrentes lignes, chaque ligne
      ayant une couleur diffrente

      mode 3: trace un histogramme avec les lments de la premire ligne }

    RC         := M.RowCount;
    CC         := M.ColCount;
    bmp.Width  := Width;
    bmp.Height := Height;
    Vmin       := M.Cells[ 1, 1];
    Vmax       := Vmin;
    Vmoy       := 0;
    for i      := 1 to RC do
    begin
      for j := 1 to CC do
      begin
        v    := M.Cells[ i, j];
        Vmoy := Vmoy + abs( v);
        if v < Vmin
        then
          Vmin := v;
        if v > Vmax
        then
          Vmax := v;
      end;
    end;
    Vmoy := Vmoy / ( RC * CC);
    if mode < 2
    then
    begin
      if Vmin <> Vmax
      then
      begin
        H     := 1 / ( Vmax - Vmin);
        dx    := ( CC) / Width;
        dy    := ( RC) / Height;
        for i := 0 to Width - 1 do
        begin
          for j := 0 to Height - 1 do
          begin
            v := M.Cells[ 1 + trunc( dy * j), 1 + trunc( dx * i)];
            if mode = 1
            then
            begin
              if abs( v / Vmoy) < 1e-12
              then
                bmp.Canvas.Pixels[ i, j] := clblack // 0
              else if abs( v - 1) < 1e-12
              then
                bmp.Canvas.Pixels[ i, j] := clwhite // 1
              else
                bmp.Canvas.Pixels[ i, j] := Col( H * ( v - Vmin));
            end
            else
              bmp.Canvas.Pixels[ i, j] := Col( H * ( v - Vmin));
          end;
        end;
      end
      else
      begin
        bmp.Canvas.Rectangle( 0, 0, Width, Height);
      end;
    end;
    if mode = 2
    then
    begin
      if Vmin <> Vmax
      then
      begin
        bmp.Canvas.Pen.Color   := clwhite;
        bmp.Canvas.Brush.Color := clwhite;
        bmp.Canvas.Rectangle( 0, 0, Width, Height);
        H     := 1 / ( Vmax - Vmin);
        dx    := Width / CC;
        dy    := Height * H;
        for i := 1 to RC do
        begin
          bmp.Canvas.Pen.Color := coul[ i mod 10];
          v                    := M.Cells[ i, 1] - Vmin;
          bmp.Canvas.Moveto( 1, Height - round( dy * v) - 1);
          for j := 1 to CC do
          begin
            v := M.Cells[ i, j] - Vmin;
            bmp.Canvas.LineTo( trunc( dx * j), Height - round( dy * v) - 1);
          end;
        end;
      end;
    end;
    if mode = 3
    then
    begin
      bmp.Canvas.Pen.Color   := clwhite;
      bmp.Canvas.Brush.Color := clwhite;
      bmp.Canvas.Rectangle( 0, 0, Width, Height);
      Vmin  := M.Cells[ 1, 1];
      Vmax  := Vmin;
      for j := 1 to CC do
      begin
        v := M.Cells[ 1, j];
        if v < Vmin
        then
          Vmin := v;
        if v > Vmax
        then
          Vmax := v;
      end;
      if Vmin > 0
      then
        Vmin := 0;
      if Vmin <> Vmax
      then
      begin
        H                      := 1 / ( Vmax - Vmin);
        dx                     := Width / ( CC);
        dy                     := Height * H;
        bmp.Canvas.Pen.Color   := clblack;
        bmp.Canvas.Brush.Color := clblue;
        for j                  := 1 to CC do
        begin
          v := M.Cells[ 1, j] - Vmin;
          bmp.Canvas.Rectangle( 1 + trunc( dx * ( j - 1)), Height - round( dy * v) - 1, 1 + trunc( dx * j), Height + round( dy * Vmin) - 1);
        end;
        bmp.Canvas.Moveto( 1, Height + round( dy * Vmin) - 1);
        bmp.Canvas.LineTo( Width, Height + round( dy * Vmin) - 1);
      end;
    end;
  end;

  function Col( v : extended) : integer;
  var
    C1, C2 : integer;
  begin
    { renvoit une couleur  partir de la valeur de V
      V doit tre compris entre 0 et 1
      le bleu fonc correspond au 0 et le 1 au rouge
      le dgrad passe par les teintes bleu->vert->jaune->orange->rouge }
    Col := 0;
    if v <= 1 / 6
    then
    begin
      C1  := round( 255 * v * 3);
      Col := ( 127 + C1) * $10000;
    end;

    if ( v > 1 / 6) and ( v <= 3 / 8)
    then
    begin
      C1  := 255;
      C2  := round( 255 * ( v - 1 / 6) * 24 / 5);
      Col := C1 * $10000 + C2 * $100;
    end;

    if ( v > 3 / 8) and ( v <= 1 / 2)
    then
    begin
      C1  := round( 255 * ( v - 3 / 8) * 8);
      C2  := 255;
      Col := ( 255 - C1) * $10000 + C2 * $100;
    end;

    if ( v >= 1 / 2) and ( v <= 5 / 8)
    then
    begin
      C1  := round( 255 * ( v - 1 / 2) * 8);
      C2  := 255;
      Col := C1 + C2 * $100;
    end;

    if ( v >= 5 / 8) and ( v <= 5 / 6)
    then
    begin
      C1  := 255;
      C2  := round( 255 * ( v - 5 / 8) * 24 / 5);
      Col := C1 + ( 255 - C2) * $100;
    end;

    if v >= 5 / 6
    then
    begin
      C1  := round( 255 * ( v - 5 / 6) * 3);
      Col := 255 - C1;
    end;
  end;

  { ********************Dcompositions de matrices************************* }

  function LU( M : TMatrix) : TDoubleMat;
  var
    i, j, l, n : integer;
    v          : array of extended;
  label
    L0, L1;
  begin
    // dcomposition M=L*U ou L et U sont respectivement
    // triangulaires infrieures et triangulaire suprieures
    if M.RowCount = M.ColCount
    then
    begin
      n := M.RowCount;
      SetLength( v, n + 1);
      RESULT[ 1] := Copy( M);
      RESULT[ 2] := Eye( n, n);

      for l := 1 to n - 1 do
      begin
        for j := l + 1 to n do
        begin
          if RESULT[ 1].Cells[ l, l] = 0
          then
            goto L0; // La dcomposition n'existe pas
          v[ j] := - RESULT[ 1].Cells[ l, j] / RESULT[ 1].Cells[ l, l];
        end;

        for i := 1 to n do
        begin
          for j := l + 1 to n do
          begin
            RESULT[ 1].Cells[ i, j] := RESULT[ 1].Cells[ i, j] + v[ j] * RESULT[ 1].Cells[ i, l];
          end;
        end;
        for i := 1 to n do
        begin
          for j := l + 1 to n do
          begin
            RESULT[ 2].Cells[ i, l] := RESULT[ 2].Cells[ i, l] - v[ j] * RESULT[ 2].Cells[ i, j];
          end;
        end;
      end;
      RESULT[ 2] := Transpose( RESULT[ 2]);
      goto L1;

    L0 :
      LU[ 1] := nil;
      LU[ 2] := nil;
    L1 :
      // U.destroy;
      SetLength( v, 0);
    end
    else
    begin
      LU[ 1] := nil;
      LU[ 2] := nil;
    end;
  end;

  function QR( M : TMatrix) : TDoubleMat;
  var
    n, k, j2, j : integer;
    Q, R, M2    : TMatrix;
    e, a, C, s  : extended;
    b           : array of extended;
  begin
    // Dcomposition M=Q*R ou Q est unitaire et R est triangulaire sup
    // La mthode de calcul dpend du type de matrice ( Householder ou Givens)
    M.define;
    n := M.RowCount;
    if M.Square
    then
    begin
      if M.Hess
      then
      begin
        // Optimisation pour les matrices de Hessenberg sup
        // On applique des matrices de rotations successivement pour liminer les
        // lments sous-diagonaux ( Mthode de Givens)
        // Ncessite n^2 oprations
        R     := Copy( M);
        Q     := Eye( n, n);
        for k := 1 to n - 1 do
        begin
          a := R.Cells[ k + 1, k] / R.Cells[ k, k];
          C := 1 / sqrt( 1 + sqr( a)); // cos( theta)
          s := a * C;                // sin( theta)
          // Ce choix de coefficients assure que la matrice Q est unitaire
          for j := k to n do
          begin
            e                 := R.Cells[ k, j];
            R.Cells[ k, j]     := C * e + s * R.Cells[ k + 1, j];
            R.Cells[ k + 1, j] := - s * e + C * R.Cells[ k + 1, j]; // On annule l'lment R[ k+1,k]
          end;
          for j := 1 to k + 1 do
          begin
            e                 := Q.Cells[ k, j];
            Q.Cells[ k, j]     := C * e + s * Q.Cells[ k + 1, j];
            Q.Cells[ k + 1, j] := - s * e + C * Q.Cells[ k + 1, j]; // On reporte l'opration sur la matrice Q
          end;

        end;
        RESULT[ 1] := Transpose( Q);
        RESULT[ 2] := R;
      end
      else
      begin
        // Cas gnral, on applique la mthode de Householder
        // Ncssite n^3 oprations
        R  := Zeros( n, n);
        Q  := TMatrix.create( n, n);
        M2 := Transpose( M);
        SetLength( b, n + 1);
        e             := 1 / NormR( M2, 1);
        R.Cells[ 1, 1] := 1 / e;
        for j         := 1 to n do
        begin
          Q.Cells[ 1, j] := e * M2.Cells[ 1, j];
        end;

        for k := 2 to n do
        begin
          for j := 1 to n do
          begin
            b[ j] := M2.Cells[ k, j];
          end;
          for j := 1 to k - 1 do
          begin
            e      := 0;
            for j2 := 1 to n do
            begin
              e := e + Q.Cells[ j, j2] * M2.Cells[ k, j2]; // calcul du produit scalaire
            end;
            R.Cells[ j, k] := e;
            for j2        := 1 to n do
            begin
              b[ j2] := b[ j2] - e * Q.Cells[ j, j2]; // orthogonalisation
            end;
          end;
          e      := 0;
          for j2 := 1 to n do
          begin
            e := e + sqr( b[ j2]);
          end;
          e             := 1 / sqrt( e);
          R.Cells[ k, k] := 1 / e;
          for j2        := 1 to n do
          begin
            Q.Cells[ k, j2] := e * b[ j2]; // normalisation
          end;
        end;
        M2.Destroy;
        SetLength( b, 0);
        RESULT[ 1] := Transpose( Q);
        RESULT[ 2] := R;
      end;
    end;
  end;

  function QL( M : TMatrix) : TDoubleMat;
  var
    n, k, j2, j : integer;
    Q, l, M2    : TMatrix;
    e           : extended;
    b           : array of extended;
  begin
    // Dcomposition M=Q*L ou Q est unitaire et L est triangulaire inf
    // La mthode de calcul est celle de Householder
    M.define;
    n := M.RowCount;
    if M.Square
    then
    begin
      // Ncssite n^3 oprations
      l  := Zeros( n, n);
      Q  := TMatrix.create( n, n);
      M2 := Transpose( M);
      SetLength( b, n + 1);
      e             := 1 / NormR( M2, n);
      l.Cells[ n, n] := 1 / e;
      for j         := 1 to n do
      begin
        Q.Cells[ n, j] := e * M2.Cells[ n, j];
      end;

      for k := n - 1 downto 1 do
      begin
        for j := 1 to n do
        begin
          b[ j] := M2.Cells[ k, j];
        end;
        for j := n downto k + 1 do
        begin
          e      := 0;
          for j2 := 1 to n do
          begin
            e := e + Q.Cells[ j, j2] * M2.Cells[ k, j2]; // calcul du produit scalaire
          end;
          l.Cells[ j, k] := e;
          for j2        := 1 to n do
          begin
            b[ j2] := b[ j2] - e * Q.Cells[ j, j2]; // orthogonalisation
          end;
        end;
        e      := 0;
        for j2 := 1 to n do
        begin
          e := e + sqr( b[ j2]);
        end;
        e             := 1 / sqrt( e);
        l.Cells[ k, k] := 1 / e;
        for j2        := 1 to n do
        begin
          Q.Cells[ k, j2] := e * b[ j2]; // normalisation
        end;
      end;
      M2.Destroy;
      SetLength( b, 0);
      RESULT[ 1] := Transpose( Q);
      RESULT[ 2] := l;
    end;
  end;

  function Hess( M : TMatrix) : TDoubleMat;
  var
    i, j, i2, j2, n, p : integer;
    e, Max, C, s, a    : extended;
    sv, cv             : array of extended;
    Q, H               : TMatrix;
  begin
    { Mthode de rduction de Hessenberg
      On obtient M=QHQ* avec Q unitaire ( Q*Q'=I) et H matrice de Hessenberg suprieure
      On utilise des matrices de rotations pour former Q }
    if M.RowCount = M.ColCount
    then
    begin
      n := M.RowCount;
      SetLength( sv, n + 1);
      SetLength( cv, n + 1);
      n     := M.RowCount;
      Q     := Eye( n, n);
      H     := Copy( M);
      for j := 1 to n - 1 do
      begin // la matrice forme des j premires lignes et colonnes est une matrice de Hessenberg
        Max   := ( H.Cells[ j + 1, j]);
        p     := j + 1;
        for i := j + 2 to n do
        begin // On cherche le plus grand lment max=H[ p,j] dans les elements au dessous de la diagonale de la j colonne
          if abs( H.Cells[ i, j]) > abs( Max)
          then
          begin
            Max := H.Cells[ i, j];
            p   := i;
          end;
        end;
        if Max <> 0
        then
        begin // si max=0, on passe  la suite
          if p <> j + 1
          then
          begin
            for i := 1 to n do
            begin // On change les lignes p et j+1
              e                 := H.Cells[ p, i];
              H.Cells[ p, i]     := H.Cells[ j + 1, i];
              H.Cells[ j + 1, i] := e;
              e                 := Q.Cells[ p, i];
              Q.Cells[ p, i]     := Q.Cells[ j + 1, i];
              Q.Cells[ j + 1, i] := e;
            end;
            for i := 1 to n do
            begin // On change les colonnes p et j+1
              e                 := H.Cells[ i, p];
              H.Cells[ i, p]     := H.Cells[ i, j + 1];
              H.Cells[ i, j + 1] := e;
            end;
          end;

          for i2 := j + 2 to n do
          begin
            a      := H.Cells[ i2, j] / H.Cells[ j + 1, j];
            C      := 1 / sqrt( 1 + sqr( a)); // cos( theta)
            s      := a * C;                // sin( theta)
            cv[ i2] := C;
            sv[ i2] := s;
            for j2 := 1 to n do
            begin
              e                  := H.Cells[ j + 1, j2];
              H.Cells[ j + 1, j2] := C * e + s * H.Cells[ i2, j2];
              H.Cells[ i2, j2]    := - s * e + C * H.Cells[ i2, j2]; // On annule l'lment H[ i2,j]
            end;
            H.Cells[ i2, j] := 0;
            for j2         := 1 to n do
            begin
              e                  := Q.Cells[ j + 1, j2];
              Q.Cells[ j + 1, j2] := C * e + s * Q.Cells[ i2, j2];
              Q.Cells[ i2, j2]    := - s * e + C * Q.Cells[ i2, j2]; // On reporte les rotations sur Q
            end;
          end;
          for j2 := j + 2 to n do
          begin
            C      := cv[ j2];
            s      := sv[ j2];
            for i2 := 1 to n do
            begin
              e                  := H.Cells[ i2, j + 1];
              H.Cells[ i2, j + 1] := C * e + s * H.Cells[ i2, j2]; // memes rotations sur les colonnes
              H.Cells[ i2, j2]    := - s * e + C * H.Cells[ i2, j2];
            end;
          end;
          // H.Cells[ i2,j]:=0;
        end;
      end;
      // M=Q*H*Q'
      RESULT[ 1] := Transpose( Q);
      RESULT[ 2] := H;
      // RESULT[ 2]._Hess:=True;
    end
    else
    begin
      RESULT[ 1] := nil;
      RESULT[ 2] := nil;
    end;
  end;

  { ******************Diagonalisation de matrices************************** }

  function eigvals( M : TMatrix) : TMatrix;
  var
    i, imin, jmax, j, k, n, its, itsmax, itot, nd : integer;
    QH                                            : TDoubleMat;
    Ak                                            : TMatrix;
    a, C, s, d, e, mu, evk, Vmin, Vmax, Norm      : extended;
    sv, cv                                        : array of extended;
    pv                                            : array of integer;
  begin
    M.define;
    if M.Sym
    then
    begin
      { Mthode itrative de recherche des valeurs propres d'une matrice symtrique
        La matrice renvoy par eigvals est une matrice diagonale contenant les valeurs
        propres de M tris par amplitude
        Algorithme:On utilise la factorisation QR avec dcalages
        On pose A0=A et on effectue la liste d'oprations suivante
        Qk*Rk=Ak-u*I
        A( k+1)=Rk*Qk+u*I
        Qk n'est pas calcul explicitement
        u est le coefficient de dcalage et I la matrice identit
        Ak tend vers une matrice dans laquelle les valeurs propres sont sur la diagonale
        l'algorithme ncessite en moyenne 3 itrations par valeur propre }

      n := M.RowCount;
      if M.Tridiag
      then
      begin
        Ak    := Copy( M);
        QH[ 1] := TMatrix.create( n, n); // Si M est tridiagonale on ne fait rien
      end
      else
      begin
        QH := Hess( M);
        Ak := QH[ 2]; // Sinon on tridiagonalise M, Ak a alors les mme valeurs propres que M
      end;
      Norm   := 0;
      for i  := 1 to n do
        Norm := Norm + sqr( Ak.Cells[ i, i]);
      for i  := 1 to n - 1 do
        Norm := Norm + 2 * sqr( Ak.Cells[ i, i + 1]);
      Norm   := 1 / sqrt( Norm / 3);
      for i  := 1 to n do
      begin
        for j := 1 to n do
        begin
          Ak.Cells[ i, j] := Ak.Cells[ i, j] * Norm;
        end;
      end;
      SetLength( sv, n + 1);
      SetLength( cv, n + 1);
      itsmax := 0;
      itot   := 0;
      nd     := 0;
      for k  := n downto 2 do
      begin       // recherche de la k valeur propre
        its := 0; // numro de l'itration
        while its <= 50 do
        begin // On commence les itrations
          evk := Ak.Cells[ k, k];
          inc( its);
          // Le coefficient de dcalage mu est celui de Wilkinson
          d  := 0.5 * ( Ak.Cells[ k - 1, k - 1] - Ak.Cells[ k, k]);
          mu := Ak.Cells[ k, k];
          if d > 0
          then
            mu := Ak.Cells[ k, k] - sqr( Ak.Cells[ k, k - 1]) / ( d + sqrt( sqr( d) + sqr( Ak.Cells[ k, k - 1])));
          if d < 0
          then
            mu := Ak.Cells[ k, k] + sqr( Ak.Cells[ k, k - 1]) / ( - d + sqrt( sqr( d) + sqr( Ak.Cells[ k, k - 1])));
          if d = 0
          then
            mu  := Ak.Cells[ k, k];
          evk   := Ak.Cells[ k, k];
          for i := 1 to k do
          begin
            Ak.Cells[ i, i] := Ak.Cells[ i, i] - mu; // Ak=Ak-u*I
          end;

          for i := 1 to k - 1 do
          begin // Recherche de la k-ime valeur propre
            a     := Ak.Cells[ i + 1, i] / Ak.Cells[ i, i];
            C     := 1 / sqrt( 1 + sqr( a)); // cos( theta)
            s     := a * C;                // sin( theta)
            cv[ i] := C;
            sv[ i] := s;

            if i + 2 <= k
            then
              jmax := i + 2
            else
              jmax := k;
            for j  := i to jmax do
            begin
              e                  := Ak.Cells[ i, j];
              Ak.Cells[ i, j]     := C * e + s * Ak.Cells[ i + 1, j];
              Ak.Cells[ i + 1, j] := - s * e + C * Ak.Cells[ i + 1, j]; // On annule l'lment R[ k+1,k]
            end;

          end;
          // A( k+1)=Q*( Ak-mu*I)   triang sup
          for j := 1 to k - 1 do
          begin // factorisation A( k+1)=Qk'*AkQk
            C := cv[ j];
            s := sv[ j];
            if j > 1
            then
              imin := j - 1
            else
              imin := 1;
            for i  := imin to j + 1 do
            begin
              e                  := Ak.Cells[ i, j];
              Ak.Cells[ i, j]     := C * e + s * Ak.Cells[ i, j + 1];
              Ak.Cells[ i, j + 1] := - s * e + C * Ak.Cells[ i, j + 1]; // On effectue les mmes rotations sur les colonnes
            end;
          end;
          for i := 1 to k do
          begin
            Ak.Cells[ i, i] := Ak.Cells[ i, i] + mu; // Ak=Ak-u*I
          end;
          // A( k+1)=Q'*( Ak-mu*I)*Q+mu*I=Q*Ak*Q'  tridiagonale
          if its > itsmax
          then
            itsmax := its;
          inc( itot);
          if Ak.Cells[ k, k] = evk
          then
            break; // convergence, la valeur propre k vaut evk
        end;
        Ak.Cells[ k, k - 1] := 0;
        Ak.Cells[ k - 1, k] := 0;
        if its = 51
        then
          inc( nd);
      end;
      for i := 1 to n do
      begin
        sv[ i] := Ak.Cells[ i, i];
        cv[ i] := abs( sv[ i]); // on enregistre les amplitudes des valeurs propres
      end;

      SetLength( pv, n + 1);
      Vmin  := cv[ 1];
      Vmax  := Vmin;
      for i := 1 to n do
      begin
        if cv[ i] < Vmin
        then
          Vmin := cv[ i];
        if cv[ i] > Vmax
        then
          Vmax := cv[ i];
      end;

      k     := 1;
      for j := 1 to n do
      begin // on tri les valeurs propres par ordre croissant
        Vmin  := Vmax;
        for i := 1 to n do
        begin
          if cv[ i] <= Vmin
          then
          begin
            k    := i;
            Vmin := cv[ i];
          end;
        end;
        cv[ k] := Vmax + 1;
        pv[ j] := k;
      end;

      for k := 1 to n do
      begin // on classe les valeurs propres
        if pv[ k] <> k
        then
          Ak.Cells[ k, k] := sv[ pv[ k]];
      end;

      Norm  := 1 / Norm;
      for i := 1 to n do
      begin
        for j := 1 to n do
        begin
          if i <> j
          then
            Ak.Cells[ i, j] := 0
          else
            Ak.Cells[ i, j] := Ak.Cells[ i, j] * Norm;
        end;
      end;

      Ak.Cells[ 1, 0] := nd;     // nd indique le nombre de valeurs propres pour lesquelles il n'y a pas eu convergence, doit valoir 0
      Ak.Cells[ 2, 0] := itot;   // nombre total d'itrations ( pour les tests)
      Ak.Cells[ 3, 0] := itsmax; // nombre max d'itrations par valeur propre ( pour les tests)
      RESULT         := Ak;
      QH[ 1].Destroy;
      SetLength( cv, 0);
      SetLength( sv, 0);
    end;
  end;

  function eig( M : TMatrix) : TDoubleMat;
  var
    i, imin, jmax, j, k, n, its, itsmax, itot, nd : integer;
    QH                                            : TDoubleMat;
    Ak, Q                                         : TMatrix;
    a, C, s, d, e, mu, evk, Vmin, Vmax, Norm      : extended;
    sv, cv                                        : array of extended;
    pv                                            : array of integer;
  begin
    M.define;
    if M.Sym
    then
    begin
      { Mthode itrative de recherche des valeurs propres et des vecteurs propres
        d'une matrice symetrique.Les valeurs propres sont tries et les vecteurs propres sont normes
        On obtient M=Q*D*Q' o Q est unitaire et D est diagonale
        Algorithme:On utilise la factorisation QR avec dcalages
        On pose A0=A et on effectue la liste d'oprations suivante
        QkRk=Ak-u*I
        A( k+1)=RkQk+u*I
        ou u est le coefficient de dcalage et I la matrice identit
        Ak tend vers une matrice dans laquelle les valeurs propres sont sur la diagonale
        l'algorithme ncessite en moyenne 3 itrations par valeur propre }
      n := M.RowCount;

      if M.Tridiag
      then
      begin
        Ak    := Copy( M);
        QH[ 1] := TMatrix.create( n, n); // Si M est tridiagonale on ne fait rien
      end
      else
      begin
        QH := Hess( M);
        Ak := QH[ 2]; // sinon on tridiagonalise M, Ak a alors les mme valeurs propres que M
      end;
      Norm   := 0;
      for i  := 1 to n do
        Norm := Norm + sqr( Ak.Cells[ i, i]);
      for i  := 1 to n - 1 do
        Norm := Norm + 2 * sqr( Ak.Cells[ i, i + 1]);
      Norm   := 1 / sqrt( Norm / 3);
      for i  := 1 to n do
      begin
        for j := 1 to n do
        begin
          Ak.Cells[ i, j] := Ak.Cells[ i, j] * Norm;
        end;
      end;
      SetLength( sv, n + 1);
      SetLength( cv, n + 1);
      Q      := Eye( n, n);
      itsmax := 0;
      itot   := 0;
      nd     := 0;
      for k  := n downto 2 do
      begin       // recherche de la k valeur propre
        its := 0; // numro de l'itration
        while its <= 50 do
        begin // On commence les itrations
          evk := Ak.Cells[ k, k];
          inc( its);
          d  := 0.5 * ( Ak.Cells[ k - 1, k - 1] - Ak.Cells[ k, k]);
          mu := Ak.Cells[ k, k];
          if d > 0
          then
            mu := Ak.Cells[ k, k] - sqr( Ak.Cells[ k, k - 1]) / ( d + sqrt( sqr( d) + sqr( Ak.Cells[ k, k - 1])));
          if d < 0
          then
            mu := Ak.Cells[ k, k] + sqr( Ak.Cells[ k, k - 1]) / ( - d + sqrt( sqr( d) + sqr( Ak.Cells[ k, k - 1])));
          if d = 0
          then
            mu := Ak.Cells[ k, k];
          // mu est le coefficient de dcalage de Wilkinson
          evk   := Ak.Cells[ k, k];
          for i := 1 to k do
          begin
            Ak.Cells[ i, i] := Ak.Cells[ i, i] - mu; // Ak=Ak-u*I
          end;

          for i := 1 to k - 1 do
          begin // factorisation A( k+1)=Qk'*Ak*Qk
            a     := Ak.Cells[ i + 1, i] / Ak.Cells[ i, i];
            C     := 1 / sqrt( 1 + sqr( a)); // cos( theta)
            s     := a * C;                // sin( theta)
            cv[ i] := C;
            sv[ i] := s; // On effectue une rotation lmentaire d'angle theta
            // Ce choix de coefficients assure que la matrice Q est unitaire
            if i + 2 <= k
            then
              jmax := i + 2
            else
              jmax := k;
            for j  := i to jmax do
            begin
              e                  := Ak.Cells[ i, j];
              Ak.Cells[ i, j]     := C * e + s * Ak.Cells[ i + 1, j];
              Ak.Cells[ i + 1, j] := - s * e + C * Ak.Cells[ i + 1, j]; // On annule l'lment Ak[ k+1,k]
            end;

            for j := 1 to n do
            begin
              e                 := Q.Cells[ i, j];
              Q.Cells[ i, j]     := C * e + s * Q.Cells[ i + 1, j];
              Q.Cells[ i + 1, j] := - s * e + C * Q.Cells[ i + 1, j]; // On reporte l'opration sur la matrice Q
            end;

          end;

          for j := 1 to k - 1 do
          begin // factorisation A( k+1)=Qk*AkQk
            C := cv[ j];
            s := sv[ j];
            if j > 1
            then
              imin := j - 1
            else
              imin := 1;
            for i  := imin to j + 1 do
            begin
              e                  := Ak.Cells[ i, j];
              Ak.Cells[ i, j]     := C * e + s * Ak.Cells[ i, j + 1];
              Ak.Cells[ i, j + 1] := - s * e + C * Ak.Cells[ i, j + 1]; // On effectue les mmes rotations sur les colonnes
            end;
          end;
          for i := 1 to k do
          begin
            Ak.Cells[ i, i] := Ak.Cells[ i, i] + mu; // Ak=Ak-u*I
          end;
          // A( k+1)=Q'*( Ak-mu*I)*Q+mu*I=Q*Ak*Q'  tridiagonale
          if its > itsmax
          then
            itsmax := its;
          inc( itot);
          // if ( abs( Ak.Cells[ k-1,k]/Ak.Cells[ k,k])<1E-13) and ( abs( Ak.Cells[ k,k-1]/Ak.Cells[ k,k])<1E-13) then break;
          if evk = Ak.Cells[ k, k]
          then
            break; // on a trouv la k-ime valeur propre
        end;
        Ak.Cells[ k, k - 1] := 0;
        Ak.Cells[ k - 1, k] := 0;
        if its = 51
        then
          inc( nd); // l'algorithme ne converge pas
      end;

      if M.Tridiag
      then
        Q.Transpose
      else
      begin
        Q.Transpose;
        Q.LMul( QH[ 1]);
      end;
      // les colonnes de Q contiennent les vecteurs propres de Ak
      for i := 1 to n do
      begin
        sv[ i] := Ak.Cells[ i, i];
        cv[ i] := abs( sv[ i]); // on enregistre les amplitudes des valeurs propres
      end;

      SetLength( pv, n + 1);
      Vmin  := cv[ 1];
      Vmax  := Vmin;
      for i := 1 to n do
      begin
        if cv[ i] < Vmin
        then
          Vmin := cv[ i];
        if cv[ i] > Vmax
        then
          Vmax := cv[ i];
      end;
      k     := 1;
      for j := 1 to n do
      begin // on tri les valeurs propres par ordre croissant
        Vmin  := Vmax;
        for i := 1 to n do
        begin
          if cv[ i] <= Vmin
          then
          begin
            k    := i;
            Vmin := cv[ i];
          end;
        end;
        cv[ k] := Vmax + 1;
        pv[ j] := k;
      end;

      QH[ 1].Destroy;
      QH[ 1] := Copy( Q);

      for k := 1 to n do
      begin // on classe les valeurs propres et les vecteurs propres norms
        if pv[ k] <> k
        then
        begin
          for i := 1 to n do
          begin
            Q.Cells[ i, k] := QH[ 1].Cells[ i, pv[ k]];
          end;
          Ak.Cells[ k, k] := sv[ pv[ k]];
        end;
      end;

      Norm  := 1 / Norm;
      for i := 1 to n do
      begin
        for j := 1 to n do
        begin
          if i <> j
          then
            Ak.Cells[ i, j] := 0
          else
            Ak.Cells[ i, j] := Ak.Cells[ i, j] * Norm;
        end;
      end;

      Ak.Cells[ 1, 0] := nd;     // nd indique le nombre de valeurs propres pour lesquelles il n'y a pas eu convergence, doit valoir 0
      Ak.Cells[ 2, 0] := itot;   // nombre total d'itrations ( pour les tests)
      Ak.Cells[ 3, 0] := itsmax; // nombre max d'itrations par valeur propre ( pour les tests)
      RESULT[ 1]      := Q;
      RESULT[ 2]      := Ak;
      QH[ 1].Destroy;
      SetLength( cv, 0);
      SetLength( sv, 0);
    end;
  end;

  function Schur( M : TMatrix) : TDoubleMat;
  begin
    // Dcomposition de Schur
    M.define;
    if M.Square
    then
    begin
      if M.Sym
      then
      begin
        RESULT := eig( M);
      end;
    end;
  end;

end.
