(*

  For an example of usage see : https://github.com/vvvv/DelphiOSCUtils/blob/master/demo/Main.pas

  Original unit comments follow :

  --------------------------------------------------------------------------------------------------------------------
  /// ///project name
  //
  // OSCUtils

  /// ///description
  //
  // Utility library to encode/decode osc-packets
  // inspired by original OSC reference implementation ( OSC-Kit)
  // and OSC.Net library as shipped with the TUIO-CSharp sample
  // from http://reactable.iua.upf.edu/?software

  /// ///licence
  //
  // GNU Lesser General Public License ( LGPL)
  // english: http://www.gnu.org/licenses/lgpl.html
  // german: http://www.gnu.de/lgpl-ger.html

  /// ///language/ide
  //
  // delphi

  /// ///initial author
  //
  // joreg -> joreg@vvvv.org

  // additions for FreePascal
  //
  // simon -> simonmoscrop@googlemail.com

  /// ///instructions
  //
  /// /for use with FreePascal
  // define: FPC

  /// /encoding a single message:
  //
  // first create a message: msg := TOSCMessage.Create( address)
  // then call msg.AddFloat( value)... to add any number of arguments
  // with msg.ToOSCBytes you get the TBytes you can send via an indy10 TidUDPClient.SendBuffer

  /// /encoding a bundle:
  //
  // first create a bundle: bundle := TOSCBundle.Create( nil)
  // then add any number of packets ( i.e. message, bundle) via bundle.Add( packet)
  // with bundle.ToOSCBytes you get the TBytes you can send via an indy10 TidUDPClient.SendBuffer

  /// /decoding a string
  //
  // use TOSCPacket.Unpack( bytes, Length( bytes)) to create
  // TOSCPackets of your osc-bytes ( those can be either bundles or single
  // messages. if you want to decode several packets at once you can create
  // a container bundle first and add the packets you create like this.
  // then use msg := FPacket.MatchAddress( address) to find a message with the
  // according address in your packet-structure.
  // before you now can access the arguments and typetags of a message you have
  // to call msg.Decode
  // voila.
  --------------------------------------------------------------------------------------------------------------------
*)



// changes for Wren
//
// 2015-07-21: Changed by Blue Hell for use in Wren
//             Worked out the dependency on Indy stuff
//             Worked out the FPC stuff
//             Uglified code to meeet my own uglyness rules


unit OSCif;

interface

uses

  Classes, Contnrs, SysUtils, System.Generics.Collections;

type

  TOSCPacket  = class;
  TOSCMessage = class;

  TOSCPacket = class( TObject)
  private
  protected
    FBytes : TBytes;
  protected
    function    MatchBrackets( aMessage, anAddress: PChar): Boolean;
    function    MatchList    ( aMessage, anAddress: PChar): Boolean;
    function    MatchPattern ( aMessage, anAddress: PChar): Boolean;
    function    GetMatched: Boolean;                                                                  virtual; abstract;
    procedure   SetMatched( aValue: Boolean);                                                         virtual; abstract;
  public
    constructor Create( const aBytes: TBytes);
    function    MatchAddress( const anAddress: string): TOSCMessage;                                  virtual; abstract;
    function    ToOSCBytes: TBytes;                                                                   virtual; abstract;
    function    AsString( DoReverse: Boolean): string;                                                virtual; abstract;
    procedure   Unmatch;                                                                              virtual; abstract;
  public
    class function Unpack( const aBytes: TBytes; aCount: Integer): TOSCPacket;                                       overload;
    class function Unpack( const aBytes: TBytes; anOffset, Count: Integer; aTimeTag: Extended = 0): TOSCPacket;      overload; virtual;
  public
    property    Matched                            : Boolean  read GetMatched           write SetMatched;
  end;


  TOSCMessage = class( TOSCPacket)
  private
    FAddress       : string;
    FArguments     : TList<TBytes>;
    FIsDecoded     : Boolean;
    FMatched       : Boolean;
    FTimeTag       : Extended;
    FTypeTagOffset : Integer;
    FTypeTags      : string;
  private
    function    GetArgument        ( anIndex: Integer): TBytes;
    function    GetArgumentAsFloat ( anIndex: Integer): Single;
    function    GetArgumentAsInt   ( anIndex: Integer): Integer;
    function    GetArgumentAsString( anIndex: Integer): string;
    function    GetArgumentCount: Integer;
    function    GetTypeTag( anIndex: Integer): string;
  protected
    function    GetMatched: Boolean;                                                                           override;
    procedure   SetMatched( aValue: Boolean);                                                                  override;
  public
    constructor Create( const anAddress: string);                                                              overload;
    constructor Create( const aBytes: TBytes);                                                                 overload;
    destructor  Destroy;                                                                                       override;
    function    AddAsBytes( const aTypeTag: Char; const aValue: string; const aFormatSettings: TFormatSettings): HResult;
    procedure   AddFloat  ( aValue: Single);
    procedure   AddInteger( aValue: Integer);
    procedure   AddMIDI   ( aPort, aCh, aMsg: Byte; aV1: Byte = $ff; aV2: Byte = $ff);
    procedure   AddString ( const aValue: string);
    procedure   Decode;
    function    MatchAddress( const anAddress: string): TOSCMessage;                                           override;
    function    ToOSCBytes: TBytes;                                                                            override;
    function    AsString( IsNwOrder: Boolean): string;                                                         override;
    procedure   Unmatch;                                                                                       override;
  public
    class function Unpack( const aBytes: TBytes; PacketOffset, aCount: Integer; aTimeTag: Extended = 0): TOSCPacket; overload; override;
  public
    property    ArgumentCount                      : Integer  read GetArgumentCount;
    property    Address                            : string   read FAddress           write FAddress;
    property    IsDecoded                          : Boolean  read FIsDecoded         write FIsDecoded;
    property    TimeTag                            : Extended read FTimeTag           write FTimeTag;
    property    TypeTagOffset                      : Integer  read FTypeTagOffset     write FTypeTagOffset;
    property    TypeTag         [ index : Integer] : string   read GetTypeTag;
    property    Argument        [ index : Integer] : TBytes   read GetArgument;
    property    ArgumentAsFloat [ index : Integer] : Single   read GetArgumentAsFloat;
    property    ArgumentAsInt   [ index : Integer] : Integer  read GetArgumentAsInt;
    property    ArgumentAsString[ index : Integer] : string   read GetArgumentAsString;
  end;


  TOSCBundle = class( TOSCPacket)
  private
    FPackets : TObjectList;
  protected
    function    GetMatched: Boolean;                                                                           override;
    procedure   SetMatched( aValue: Boolean);                                                                  override;
  public
    constructor Create( const aBytes: TBytes);
    destructor  Destroy;                                                                                       override;
    procedure   Add( const aPacket : TOSCPacket);
    function    MatchAddress( const anAddress: string): TOSCMessage;                                           override;
    function    ToOSCBytes: TBytes;                                                                            override;
    function    AsString( IsNwOrder: Boolean): string;                                                         override;
    procedure   Unmatch;                                                                                       override;
  public
    class function Unpack( const aBytes: TBytes; aPacketOffset, aCount: Integer; aTimeTag: Extended = 0) : TOSCPacket; overload; override;
  end;


  function  ReverseInt  ( aValue: Integer): Integer;
  function  ReverseFloat( aValue: Single ): Single;

  function  MakeOSCFloat ( aValue: Single                  ): TBytes;
  function  MakeOSCInt   ( aValue: Integer                 ): TBytes;
  function  MakeOSCString( const aValue: string            ): TBytes;
  function  MakeOSCMIDI  ( aPort, aCh, aMsg, aV1, aV2: Byte): TBytes;

  function  UnpackInt   ( const aBytes: TBytes; var anOffset: Integer): TBytes;
  function  UnpackString( const aBytes: TBytes; var anOffset: Integer): TBytes;
  function  UnpackFloat ( const aBytes: TBytes; var anOffset: Integer): TBytes;
  function  UnpackMIDI  ( const aBytes: TBytes; var anOffset: Integer): TBytes;

  function  UnpackAndReturnInt  ( const aBytes: TBytes; var anOffset: Integer): Integer;
  function  UnpackAndReturnFloat( const aBytes: TBytes; var anOffset: Integer): Single;


const

  OSC_OK                   = 0;
  OSC_UNRECOGNIZED_TYPETAG = 1;
  OSC_CONVERT_ERROR        = 2;

  DO_REVERSE = True;
  NO_REVERSE = not DO_REVERSE;



implementation



uses

  Math, WinSock;


  function  DataToBytes( const aSrc; const aCount: Integer): TBytes;
  // Copied from Indy IdGlobal.pas
  begin
    SetLength( Result, aCount);
    if aCount > 0
    then Move( aSrc, Result[ 0], aCount);
  end;


  procedure CopyBytes( const aSrc: TBytes; const aSrcIndex: Integer; var aDst: TBytes; const aDstIndex: Integer; const aCount: Integer);
  // Copied from Indy IdGlobal.pas
  begin
    if ( aSrcIndex >= 0) and (( aSrcIndex + aCount) <= Length( aSrc))
    then Move( aSrc[ aSrcIndex], aDst[ aDstIndex], aCount);
  end;


  function ReverseInt( aValue: Integer): Integer;
  begin
    Result := ntohl( aValue);
  end;


  function ReverseFloat( aValue: Single): Single;
  begin
    Result := Single( Pointer( ntohl( PInteger( @ aValue)^)));
  end;


  function MakeOSCFloat( aValue:  Single): TBytes;
  var
    intg : Integer;
  begin
    intg := PInteger( @ aValue)^;

    intg := htonl( intg);
    Result := DataToBytes( intg, SizeOf( intg));
  end;


  function MakeOSCInt( aValue: Integer): TBytes;
  begin
    aValue := htonl( aValue);
    Result := DataToBytes( aValue, SizeOf( aValue));
  end;


  function MakeOSCString( const aValue: string): TBytes;
  var
    i, ln : Integer;
  begin
    ln := TEncoding.UTF8.GetByteCount( aValue);
    ln := ln + ( 4 - ln mod 4);
    SetLength( Result, ln);
    ln := TEncoding.UTF8.GetBytes( aValue, 1, Length( aValue), Result, 0);

    for i := ln to high( Result)
    do Result[ i] := 0;
  end;


  function  MakeOSCMIDI( aPort, aCh, aMsg, aV1, aV2: Byte): TBytes;
  // [jan] aV1 and aV2 are optional, when not used for the particular
  // type of aMsg they should be set to $ff.
  var
    aStatus : Byte;
  begin
    aStatus := (( aMsg and $0f) shl 4) or ( aCh and $0f);
    SetLength( Result, 4);
    Result[ 0] := aPort;
    Result[ 1] := aStatus;
    Result[ 2] := aV1;
    Result[ 3] := aV2;
  end;


  function UnpackInt( const aBytes: TBytes; var anOffset: Integer): TBytes;
  var
    i : Integer;
  begin
    SetLength( Result, SizeOf( Integer));

    // Copy bytes and change byte order

    for i := 0 to high( Result)
    do Result[ i] := aBytes[ anOffset + high( Result) - i];

    Inc( anOffset, SizeOf( Integer));
  end;


  function UnpackString( const aBytes: TBytes; var anOffset: Integer): TBytes;
  var
    off : Integer;
  begin
    // Strings are null terminated. Find position of null.

    off := anOffset;

    while ( off < Length( aBytes)) and ( aBytes[ off] <> 0)
    do Inc( off);

    // Retrieve the string.

    SetLength( Result, off - anOffset);
    CopyBytes( aBytes, anOffset, Result, 0, Length( Result));

    // Increase the offset by a multiple of 4.

    anOffset := off + ( 4 - off mod 4);
  end;


  function UnpackFloat( const aBytes: TBytes; var anOffset: Integer): TBytes;
  var
    // value: Integer;
    i : Integer;
  begin
    SetLength( Result, SizeOf( Single));

    // Copy bytes and change byte order

    for i := 0 to high( Result)
    do Result[ i] := aBytes[ anOffset + high( Result) - i];

    Inc( anOffset, SizeOf( Single));
  end;


  function   UnpackMIDI( const aBytes: TBytes; var anOffset: Integer): TBytes;
  // [ jan]
  var
    aPort   : Byte;
    aStatus : Byte;
    aV1     : Byte;
    aV2     : Byte;
    i       : Integer;
    aCh     : Byte;
    aMsg    : Byte;
  begin
    SetLength( Result, 5);

    for i := 0 to Length( Result) - 1
    do Result[ i] := 0;

    if Assigned( aBytes)
    then begin
      aPort   := aBytes[ anOffset + 0];
      aStatus := aBytes[ anOffset + 1];
      aV1     := aBytes[ anOffset + 2];
      aV2     := aBytes[ anOffset + 3];

      aCh     := aStatus and $0f;
      aMsg    := ( aStatus shr 4) and $0f;

      Result[ 0] := aPort;
      Result[ 1] := aCh;
      Result[ 2] := aMsg;
      Result[ 3] := aV1;
      Result[ 4] := aV2;

      Inc( anOffset, 4);
    end
  end;


  function UnpackAndReturnInt( const aBytes: TBytes; var anOffset: Integer): Integer;
  var
    resultBytes: TBytes;
  begin
    resultBytes := UnpackInt( aBytes, anOffset);
    Result      := PInteger( Pointer( resultBytes))^;
  end;


  function UnpackAndReturnFloat( const aBytes: TBytes; var anOffset: Integer): Single;
  var
    resultBytes: TBytes;
  begin
    resultBytes := UnpackFloat( aBytes, anOffset);
    Result      := PSingle( Pointer( resultBytes))^;
  end;


{ ========
  TOSCPacket = class( TObject)
  private
  protected
    FBytes : TBytes;
  protected
}

    // we know that pattern[ 0] == '[ ' and test[ 0] != 0
    function TOSCPacket.MatchBrackets( aMessage, anAddress: PChar): Boolean;
    var
      Negated : Boolean;
      p       : PChar;
      p1      : PChar;
      p2      : PChar;
    begin
      p       := aMessage;
      Result  := False;
      Negated := False;

      Inc( aMessage);

      if aMessage^ = #0
      then Exit;

      if aMessage^ = '!'
      then begin
        Negated := True;
        Inc( p);
      end;

      Result := Negated;

      while p^ <> ']'
      do begin
        if p^ = #0
        then Exit;

        p1 := p  + 1; // sizeOf( PChar);
        p2 := p1 + 1; // sizeOf( PChar);

        if ( p1^ = '-') and ( p2^ <> #0)
        then begin
          if ( Ord( anAddress^) >= Ord( p^)) and ( Ord( anAddress^) <= Ord( p2^))
          then begin
            Result := not Negated;
            Break;
          end;
        end;

        if p^ = anAddress^
        then begin
          Result := not Negated;
          Break;
        end;

        Inc( p);
      end;

      if not Result
      then Exit;

      while p^ <> ']'
      do begin
        if p^ = #0
        then Exit;

        Inc( p);
      end;

      Inc( p);
      Inc( anAddress);
      Result := MatchPattern( p, anAddress);
    end;

    function    TOSCPacket.MatchList( aMessage, anAddress: PChar): Boolean;
    var
      p, tp : PChar;
    begin
      Result := False;

      p  := aMessage;
      tp := anAddress;

      while p^ <> '}'
      do begin
        if p^ = #0
        then Exit;

        Inc( p);
      end;

      Inc( p);        // skip close curly brace
      Inc( aMessage); // skip open curly brace

      while True
      do begin
        if aMessage^ = ','
        then begin
          if MatchPattern( p, tp)
          then begin
            Result := True;
            Exit;
          end
          else begin
            tp := anAddress;
            Inc( aMessage);
          end;
        end
        else if aMessage^ = '}'
        then begin
          Result := MatchPattern( p, tp);
          Exit;
        end
        else if aMessage^ = tp^
        then begin
          Inc( aMessage);
          Inc( tp);
        end
        else begin
          tp := anAddress;

          while ( aMessage^ <> ',') and ( aMessage^ <> '}')
          do Inc( aMessage);

          if aMessage^ = ','
          then Inc( aMessage);
        end;
      end;
    end;


    function    TOSCPacket.MatchPattern( aMessage, anAddress: PChar): Boolean;
    begin
      // BH : added this, I want a receiver to be able to match any packet,
      //      this is handy to have for packet logging.
      if anAddress^ = '*'
      then begin
        Result := True;
        Exit;
      end;
      // BH - end

      if ( aMessage = nil) or ( aMessage^ = #0)
      then begin
        Result := anAddress^ = #0;
        Exit;
      end;

      if anAddress^ = #0
      then begin
        if aMessage^ = '*'
        then begin
          Result := MatchPattern( aMessage + 1, anAddress);
          Exit;
        end
        else begin
          Result := False;
          Exit;
        end;
      end;

      case aMessage^ of
        #0  : Result := anAddress^ = #0;


        '?' : Result := MatchPattern( aMessage + 1, anAddress + 1);

        '*' : begin
            if MatchPattern( aMessage + 1, anAddress)
            then Result := True
            else Result := MatchPattern( aMessage, anAddress + 1);
          end;

        ']',
        '}' : Result := False;

        '[' : Result := MatchBrackets( aMessage, anAddress);

        '{' : Result := MatchList( aMessage, anAddress);
      else
        if aMessage^ = anAddress^
        then Result := MatchPattern( aMessage + 1, anAddress + 1)
        else Result := False;
      end;
    end;


// public

    constructor TOSCPacket.Create( const aBytes: TBytes);
    begin
      FBytes := aBytes;
    end;


    class function TOSCPacket.Unpack( const aBytes: TBytes; aCount: Integer): TOSCPacket;
    begin
      Result := Unpack( aBytes, 0, aCount);
    end;


    class function TOSCPacket.Unpack( const aBytes: TBytes; anOffset, Count: Integer; aTimeTag: Extended = 0): TOSCPacket;
    begin
      if Char( aBytes[ anOffset]) = '#'
      then Result := TOSCBundle .Unpack( aBytes, anOffset, Count)
      else Result := TOSCMessage.Unpack( aBytes, anOffset, Count, aTimeTag);
    end;


{ ========
  TOSCMessage = class( TOSCPacket)
  private
    FAddress       : string;
    FArguments     : TList<TBytes>;
    FIsDecoded     : Boolean;
    FMatched       : Boolean;
    FTimeTag       : Extended;
    FTypeTagOffset : Integer;
    FTypeTags      : string;
  public
    property    ArgumentCount                      : Integer  read GetArgumentCount;
    property    Address                            : string   read FAddress           write FAddress;
    property    IsDecoded                          : Boolean  read FIsDecoded         write FIsDecoded;
    property    Matched                            : Boolean  read FMatched           write FMatched;
    property    TimeTag                            : Extended read FTimeTag           write FTimeTag;
    property    TypeTagOffset                      : Integer  read FTypeTagOffset     write FTypeTagOffset;
    property    TypeTag         [ index : Integer] : string   read GetTypeTag;
    property    Argument        [ index : Integer] : TBytes   read GetArgument;
    property    ArgumentAsFloat [ index : Integer] : Single   read GetArgumentAsFloat;
    property    ArgumentAsInt   [ index : Integer] : Integer  read GetArgumentAsInt;
    property    ArgumentAsString[ index : Integer] : string   read GetArgumentAsString;
  private
}

    function    TOSCMessage.GetArgument( anIndex: Integer): TBytes;
    begin
      Result := FArguments[ anIndex];
    end;


    function    TOSCMessage.GetArgumentAsFloat( anIndex: Integer): Single;
    begin
      Result := PSingle( Pointer( FArguments[ anIndex]))^;
    end;


    function    TOSCMessage.GetArgumentAsInt( anIndex: Integer): Integer;
    begin
      Result := PInteger( Pointer( FArguments[ anIndex]))^;
    end;


    function    TOSCMessage.GetArgumentAsString( anIndex: Integer): string;
    begin
      Result := Trim( TEncoding.UTF8.GetString( FArguments[ anIndex]));
    end;


    function    TOSCMessage.GetArgumentCount: Integer;
    begin
      Result := FArguments.Count;
    end;


    function    TOSCMessage.GetTypeTag( anIndex: Integer): string;
    begin
      Result := FTypeTags[ anIndex + 2];
    end;


//  protected

    function    TOSCMessage.GetMatched: Boolean; // override;
    begin
      Result := FMatched;
    end;


    procedure   TOSCMessage.SetMatched( aValue: Boolean); // override;
    begin
      FMAtched := aValue;
    end;


//  public

    constructor TOSCMessage.Create( const anAddress: string);
    begin
      FAddress := anAddress;
      Create( nil);
    end;


    constructor TOSCMessage.Create( const aBytes: TBytes);
    begin
      inherited;

      FTypeTags  := ',';
      FArguments := TList<TBytes>.Create;
      FIsDecoded := False;
    end;


    destructor  TOSCMessage.Destroy;
    begin
      FArguments.Free;
      inherited;
    end;


    function    TOSCMessage.AddAsBytes( const aTypeTag: Char; const aValue: string; const aFormatSettings: TFormatSettings): HResult;
    begin
      Result := OSC_OK;
      try
        if      aTypeTag = 'f' then AddFloat  ( StrToFloat( aValue, aFormatSettings))
        else if aTypeTag = 'i' then AddInteger( StrToInt  ( aValue                 ))
        else if aTypeTag = 'm' then AddInteger( StrToInt  ( aValue                 ))
        else if aTypeTag = 's' then AddString (             aValue                  )
        else Result := OSC_UNRECOGNIZED_TYPETAG;
      except
        on EConvertError
        do Result := OSC_CONVERT_ERROR;
      end;
    end;


    procedure   TOSCMessage.AddFloat( aValue: Single);
    begin
      FTypeTags := FTypeTags + 'f';
      FArguments.Add( MakeOSCFloat( aValue));
    end;


    procedure   TOSCMessage.AddInteger( aValue: Integer);
    begin
      FTypeTags := FTypeTags + 'i';
      FArguments.Add( MakeOSCInt( aValue));
    end;


    procedure   TOSCMessage.AddMIDI( aPort, aCh, aMsg: Byte; aV1: Byte = $ff; aV2: Byte = $ff);
    begin
      FTypeTags := FTypeTags + 'm';
      FArguments.Add( MakeOSCMIDI( aPort, aCh, aMsg, aV1, aV2));
    end;


    procedure   TOSCMessage.AddString( const aValue: string);
    begin
      FTypeTags := FTypeTags + 's';
      FArguments.Add( MakeOSCString( aValue));
    end;


    procedure   TOSCMessage.Decode;
    var
      i      : Integer;
      Offset : Integer;
    begin
      if FIsDecoded
      then Exit;

      Offset    := FTypeTagOffset;
      FTypeTags := TEncoding.ASCII.GetString( UnpackString( FBytes, Offset));

      for i := 1 to Length( FTypeTags) - 1
      do begin
        if      FTypeTags[ i + 1] = 'f' then FArguments.Add( UnpackFloat ( FBytes, Offset))
        else if FTypeTags[ i + 1] = 'i' then FArguments.Add( UnpackInt   ( FBytes, Offset))
        else if FTypeTags[ i + 1] = 'm' then FArguments.Add( UnpackMIDI  ( FBytes, Offset))
        else if FTypeTags[ i + 1] = 's' then FArguments.Add( UnpackString( FBytes, Offset));
      end;

      FIsDecoded := True;
    end;


    function    TOSCMessage.MatchAddress( const anAddress: string): TOSCMessage;
    begin
      if not FMatched and MatchPattern( PChar( FAddress), PChar( anAddress))
      then begin
        FMatched := True;
        Result   := Self
      end
      else Result := nil;
    end;


    function    TOSCMessage.ToOSCBytes: TBytes;
    var
      i          : Integer;
      resultList : TList<Byte>;
    begin
      resultList := TList<Byte>.Create;
      resultList.AddRange( MakeOSCString( FAddress));
      resultList.AddRange( MakeOSCString( FTypeTags));

      for i := 0 to FArguments.Count - 1
      do resultList.AddRange( FArguments[ i]);

      Result := resultList.ToArray( );
      resultList.Free;
    end;

    function    TOSCMessage.AsString( IsNwOrder: Boolean): string;
    var
      i        : Integer;
      aTypeTag : string;
      aValue   : string;
    begin
      Result := FAddress;

      for i := 0 to FArguments.Count - 1
      do begin
        Result := Result + ' ';

        aTypeTag  := TypeTag [ i];
        aValue    := '<undefined>';

        if IsNwOrder
        then begin
          if      aTypeTag = 'f' then aValue := FloatToStr( ReverseFloat( ArgumentAsFloat [ i]))
          else if aTypeTag = 'i' then aValue := IntToStr  ( ReverseInt  ( ArgumentAsInt   [ i]))
          else if aTypeTag = 'm' then aValue := IntToStr  ( ReverseInt  ( ArgumentAsInt   [ i]))
          else if aTypeTag = 's' then aValue :=                           ArgumentAsString[ i] ;
        end
        else begin
          if      aTypeTag = 'f' then aValue := FloatToStr           ( ArgumentAsFloat [ i])
          else if aTypeTag = 'i' then aValue := IntToStr             ( ArgumentAsInt   [ i])
          else if aTypeTag = 'm' then aValue := IntToStr             ( ArgumentAsInt   [ i])
          else if aTypeTag = 's' then aValue :=                        ArgumentAsString[ i] ;
        end;

        Result := Result + format( '%s( %s)', [ aTypeTag, aValue])
      end;
    end;


    procedure   TOSCMessage.Unmatch;
    begin
      FMatched := False;
    end;


    class function TOSCMessage.Unpack( const aBytes: TBytes; PacketOffset, aCount: Integer; aTimeTag: Extended = 0): TOSCPacket;
    begin
      Result := TOSCMessage.Create( aBytes);

      // for now decode address only

      ( Result as TOSCMessage).Address := TEncoding.ASCII.GetString( UnpackString( aBytes, PacketOffset));
      ( Result as TOSCMessage).TimeTag := aTimeTag;

      // save offset for later decoding on demand

      ( Result as TOSCMessage).TypeTagOffset := PacketOffset;
      ( Result as TOSCMessage).IsDecoded     := False;
    end;




{ ========
  TOSCBundle = class( TOSCPacket)
  private
    FPackets : TObjectList;
  protected
}

    function    TOSCBundle.GetMatched: Boolean; // override;
    var
      i : Integer;
    begin
      Result := True;
      for i := 0 to FPackets.Count - 1
      do begin
        if not ( FPackets[ i] as TOSCPacket).Matched
        then begin
          Result := False;
          Break;
        end;
      end;
    end;


    procedure   TOSCBundle.SetMatched( aValue: Boolean); // override;
    var
      i : Integer;
    begin
      for i := 0 to FPackets.Count - 1
      do ( FPackets[ i] as TOSCPacket).Matched := aValue;
    end;


//  public

    constructor TOSCBundle.Create( const aBytes: TBytes);
    begin
      inherited;
      FPackets             := TObjectList.Create;
      FPackets.OwnsObjects := True;
    end;


    destructor  TOSCBundle.Destroy;
    begin
      FPackets.Free;
      inherited;
    end;


    procedure   TOSCBundle.Add( const aPacket: TOSCPacket);
    begin
      FPackets.Add( aPacket);
    end;


    function    TOSCBundle.MatchAddress( const anAddress: string): TOSCMessage;
    var
      i : Integer;
    begin
      Result := nil;

      for i := 0 to FPackets.Count - 1
      do begin
        Result := ( FPackets[ i] as TOSCPacket).MatchAddress( anAddress);

        if Assigned( Result)
        then Break;
      end;
    end;


    function    TOSCBundle.ToOSCBytes: TBytes;
    var
      i          : Integer;
      Packet     : TBytes;
      resultList : TList<Byte>;
    begin
      resultList := TList<Byte>.Create;
      resultList.AddRange( MakeOSCString( '#bundle'));
      resultList.AddRange( TEncoding.UTF8.GetBytes( #0#0#0#0#0#0#0#1)); // immediately

      for i := 0 to FPackets.Count - 1
      do begin
        Packet := ( FPackets[ i] as TOSCPacket).ToOSCBytes;
        resultList.AddRange( MakeOSCInt( Length( Packet)));
        resultList.AddRange( Packet);
      end;

      Result := resultList.ToArray( );
      resultList.Free;
    end;


    function    TOSCBundle.AsString( IsNwOrder: Boolean): string;
    begin
      Result := 'To be implemented: TOSCBundle.AsString'
    end;


    procedure   TOSCBundle.Unmatch;
    var
      i : Integer;
    begin
      for i := 0 to FPackets.Count - 1
      do ( FPackets[ i] as TOSCPacket).Unmatch;
    end;


    class function TOSCBundle.Unpack( const aBytes: TBytes; aPacketOffset, aCount: Integer; aTimeTag: Extended = 0): TOSCPacket;
    var
      packetLength : Integer;
      tt1, tt2     : Cardinal;
    begin
      Result := TOSCBundle.Create( aBytes);

      // advance the '#bundle' string

      UnpackString( aBytes, aPacketOffset);

      // advance the timestamp

      tt1 := Cardinal( UnpackAndReturnInt( aBytes, aPacketOffset));
      tt2 := Cardinal( UnpackAndReturnInt( aBytes, aPacketOffset));

      aTimeTag := tt1 + tt2 / Power( 2, 32);

      while aPacketOffset < aCount
      do begin
        packetLength := UnpackAndReturnInt( aBytes, aPacketOffset);

        // note: PacketOffset is always from the very beginning of Bytes!
        // not the beginning of the current packet.

        ( Result as TOSCBundle).Add( TOSCPacket.Unpack( aBytes, aPacketOffset, aPacketOffset + packetLength, aTimeTag));
        Inc( aPacketOffset, packetLength);
      end;
    end;

end.

