unit Talkie;

{

   COPYRIGHT 2015 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl

  original code :

    https://github.com/going-digital/Talkie
    https://github.com/going-digital/Talkie/tree/master/Talkie

  With copyright notice:

    Talkie library
    Copyright 2011 Peter Knight
    This code is released under GPLv2 license.

  Note that:

    With this software comes a large set of words that are available with the
    WREN translation as well trough the supllied file words.lpc; the words in
    this file were all found on the Talkie website as well and modified a bit
    to make it easier to parse the file.
}


interface

uses

 System.SysUtils, System.Classes, System.StrUtils,

 KnobsUtils, KnobsConversions;

type

  ETalkie             = class( Exception);
  ETalkieParseError   = class( ETalkie  );

  TTalkieOnSayStarted = procedure( aSender: TObject) of object;
  TTalkieOnSayDone    = procedure( aSender: TObject) of object;

  TTalkiePhrase = class
  // Holds and parses the phrase text with it's LPC data
  private
    FPhrase : string;
    FData   : TBytes;
  private
    function    GetCount: Integer;
    procedure   Error( const aMsg, aLine: string; anOffset: Integer);
    procedure   AddData( const aValue: string);
  public
    constructor Create( const aLine: string);
  public
    property    Phrase : string  read FPhrase;
    property    Data   : TBytes  read FData;
    property    Count  : Integer read GetCount;
  end;

  TTalkiePhraseSet = class
  // Holds and parses a set of Phrases
  private
    FName    : string;
    FPhrases : array of TTalkiePhrase;
  private
    function    GetCount: Integer;
    function    GetPhrase( anIndex: Integer): string;
    function    GetData( anIndex: Integer): TBytes;
    procedure   Error( const aMsg, aLine: string; aLineNr: Integer);
    procedure   AddPhrase( const aLine: string);
  public
    constructor Create( const aLines: TStringList; var anIndex: Integer);
    destructor  Destroy;                                                     override;
  public
    property    Name                      : string  read FName;
    property    Count                     : Integer read GetCount;
    property    Phrase[ anIndex: Integer] : string  read GetPhrase;
    property    Data  [ anIndex: Integer] : TBytes  read GetData;
  end;

  TTalkiePhraseSets = class
  // Holds and parses all the sets of phrases from an lpc file
  private
    FPhraseSets : array of TTalkiePhraseSet;
  private
    function    GetCount: Integer;
    function    GetName( anIndex: Integer): string;
    function    GetPhraseSet( anIndex: Integer): TTalkiePhraseSet;
  private
    procedure   AddPhraseSet( const aLines: TStringList; var anIndex: Integer);
  public
    constructor Create( const aFileName: string);
    destructor  Destroy;                                                     override;
    function    CreateBankNames: TStringList;
  public
    property    Count                        : Integer              read GetCount;
    property    Name     [ anIndex: Integer] : string               read GetName;
    property    PhraseSet[ anIndex: Integer] : TTalkiePhraseSet read GetPhraseSet;
  end;


  TTalkie = class
  private
    FOnStarted      : TTalkieOnSayStarted;
    FOnDone         : TTalkieOnSayDone;
  private
    FFrameRate      : TSignal;
    FSampleRate     : TSignal;
    FData           : TBytes;
    FIndexAddress   : Integer;
    FIndexBit       : Byte;
    FRateDelta      : TSignal;
    FRateAccu       : TSignal;
    FFrameDelta     : TSignal;
    FFrameAccu      : TSignal;
    FActive         : Boolean;
    FPrevValue      : TSignal;
    FNextValue      : TSignal;
  private
    FPeriodCounter  : Byte;
    FX0             : SmallInt;
    FX1             : SmallInt;
    FX2             : SmallInt;
    FX3             : SmallInt;
    FX4             : SmallInt;
    FX5             : SmallInt;
    FX6             : SmallInt;
    FX7             : SmallInt;
    FX8             : SmallInt;
    FX9             : SmallInt;
  private
    FSynthPeriod    : Byte;
    FSynthEnergy    : Byte;
    FSynthK1        : SmallInt;
    FSynthK2        : SmallInt;
    FSynthK3        : ShortInt;
    FSynthK4        : ShortInt;
    FSynthK5        : ShortInt;
    FSynthK6        : ShortInt;
    FSynthK7        : ShortInt;
    FSynthK8        : ShortInt;
    FSynthK9        : ShortInt;
    FSynthK10       : ShortInt;
  private
    procedure   SetFrameRate ( aValue: TSignal);
    procedure   SetSampleRate( aValue: TSignal);
  private
    procedure   SetIndex( aValue: Integer);
    function    ReverseBits( aByte: Byte): Byte;
    function    GetBits( anAmount: Byte): Byte;
    procedure   Calculate;
    function    NextValue: TSignal;
  public
    constructor Create;
    destructor  Destroy;                                                     override;
    procedure   Say( aFrameRate, aSampleRate: TSignal; const aData: TBytes);
    procedure   Reset;
    function    Tick: TSignal;
    procedure   Start;
    procedure   Stop;
  public
    property    OnStarted  : TTalkieOnSayStarted read FOnStarted  write FOnStarted;
    property    OnDone     : TTalkieOnSayDone    read FOnDone     write FOnDone;
    property    FrameRate  : TSignal             read FFrameRate  write SetFrameRate;
    property    SampleRate : TSignal             read FSampleRate write SetSampleRate;
  end;


  procedure TalkieCreatePhrases( const aFileName: string);        // Create a global pool of phrases
  procedure TalkieDestroyPhrases;                                 // Desstroy the global phraseds pool
  function  TalkiePhraseSets: TTalkiePhraseSets;                  // Get a link to the globsal phrases pool


implementation


const

  WhiteSpace : set of AnsiChar = [ ' ', ^I];

  // Note: the actual types of some of the arrays below may be signed, there are some typecasts in the code using them.

  tmsEnergy : array[ $00 .. $0f] of Byte = ( $00, $02, $03, $04, $05, $07, $0a, $0f, $14, $20, $29, $39, $51, $72, $a1, $ff);
  tmsPeriod : array[ $00 .. $3f] of Byte = ( $00, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $1A, $1B, $1C, $1D, $1E, $1F, $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $2A, $2B, $2D, $2F, $31, $33, $35, $36, $39, $3B, $3D, $3F, $42, $45, $47, $49, $4D, $4F, $51, $55, $57, $5C, $5F, $63, $66, $6A, $6E, $73, $77, $7B, $80, $85, $8A, $8F, $95, $9A, $A0);
  tmsK1     : array[ $00 .. $1f] of Word = ( $82C0, $8380, $83C0, $8440, $84C0, $8540, $8600, $8780, $8880, $8980, $8AC0, $8C00, $8D40, $8F00, $90C0, $92C0, $9900, $A140, $AB80, $B840, $C740, $D8C0, $EBC0, $0000, $1440, $2740, $38C0, $47C0, $5480, $5EC0, $6700, $6D40);
  tmsK2     : array[ $00 .. $1f] of Word = ( $AE00, $B480, $BB80, $C340, $CB80, $D440, $DDC0, $E780, $F180, $FBC0, $0600, $1040, $1A40, $2400, $2D40, $3600, $3E40, $45C0, $4CC0, $5300, $5880, $5DC0, $6240, $6640, $69C0, $6CC0, $6F80, $71C0, $73C0, $7580, $7700, $7E80);
  tmsK3     : array[ $00 .. $0f] of Byte = ( $92, $9F, $AD, $BA, $C8, $D5, $E3, $F0, $FE, $0B, $19, $26, $34, $41, $4F, $5C);
  tmsK4     : array[ $00 .. $0f] of Byte = ( $AE, $BC, $CA, $D8, $E6, $F4, $01, $0F, $1D, $2B, $39, $47, $55, $63, $71, $7E);
  tmsK5     : array[ $00 .. $0f] of Byte = ( $AE, $BA, $C5, $D1, $DD, $E8, $F4, $FF, $0B, $17, $22, $2E, $39, $45, $51, $5C);
  tmsK6     : array[ $00 .. $0f] of Byte = ( $C0, $CB, $D6, $E1, $EC, $F7, $03, $0E, $19, $24, $2F, $3A, $45, $50, $5B, $66);
  tmsK7     : array[ $00 .. $0f] of Byte = ( $B3, $BF, $CB, $D7, $E3, $EF, $FB, $07, $13, $1F, $2B, $37, $43, $4F, $5A, $66);
  tmsK8     : array[ $00 .. $07] of Byte = ( $C0, $D8, $F0, $07, $1F, $37, $4F, $66);
  tmsK9     : array[ $00 .. $07] of Byte = ( $C0, $D4, $E8, $FC, $10, $25, $39, $4D);
  tmsK10    : array[ $00 .. $07] of Byte = ( $CD, $DF, $F1, $04, $16, $20, $3B, $4D);
  chirp     : array[ 0   ..  40] of Byte = ( $00, $2a, $d4, $32, $b2, $12, $25, $14, $02, $e1, $c5, $02, $5f, $5a, $05, $0f, $26, $fc, $a5, $a5, $d6, $dd, $dc, $fc, $25, $2b, $22, $21, $0f, $ff, $f8, $ee, $ed, $ef, $f7, $f6, $fa, $00, $03, $02, $01);


var

  GPhrases : TTalkiePhraseSets = nil;

  procedure TalkieCreatePhrases( const aFileName: string);
  begin
    if FileExists( aFileName)
    then begin
      try
        if Assigned( GPhrases)
        then FreeAndNil( GPhrases);

        GPhrases := TTalkiePhraseSets.Create( aFileName);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;
  end;


  procedure TalkieDestroyPhrases;
  begin
    FreeAndNil( GPhrases);
  end;


  function  TalkiePhraseSets: TTalkiePhraseSets;
  begin
    Result := GPhrases;
  end;


{ ========
  TWrenTalkiePhrase = class
  private
    FPhrase : string;
    FData   : TBytes;
  public
    property    Phrase : string  read FPhrase;
    property    Data   : TBytes  read FData;
    property    Count  : Integer read GetCount;
  private
}

    function    TTalkiePhrase.GetCount: Integer;
    begin
      Result := Length( FData);
    end;


    procedure   TTalkiePhrase.Error( const aMsg, aLine: string; anOffset: Integer);
    begin
      raise ETalkieParseError.CreateFmt( 'Parse error: ''%s'' in line ''%s'' detected at position %d', [ aMsg, aLine, anOffset]);
    end;


    procedure   TTalkiePhrase.AddData( const aValue: string);
    var
      V : Integer;
    begin
      V := StrToIntDef( '$' + Trim( aValue), -1);
      if ( V < 0) or ( V > $ff)
      then Error( 'Invalid number', aValue, 1);
      SetLength( FData, Count + 1);
      FData[ Count - 1] := V;
    end;


//  public

    constructor TTalkiePhrase.Create( const aLine: string);
    var
      p     : Integer;
      q     : Integer;
      Len   : Integer;
      S     : string;
      aList : TStringList;
    begin
      Len := Length( aLine);
      p   := 1;

      while ( p <= Len) and CharInSet( aLine[ p], WhiteSpace)
      do Inc( p);

      if p > Len
      then Error( 'no data', aLine, p);

      q := p;
      while ( q <= Len) and ( aLine[ q] <> '=')
      do Inc( q);

      if q > Len
      then Error( 'expected =', aLine, q);

      if q > p + 1
      then begin
        FPhrase := Trim( Copy( aLine, p, q - p - 1));
        if FPhrase <> ''
        then begin
          p := q + 1;
          while ( q <= Len) and ( aLine[ q] <> ';')
          do Inc( q);

          if q > Len
          then Error( 'expected ;', aLine, q);

          S := AnsiReplaceText( Trim( Copy( aLine, p, q - p - 1)), ' ', '');
          if S = ''
          then Error( 'no data', aLine, p);

          aList := Explode( S, ',');
          try
            if aList.Count > 0
            then begin
              for p := 0 to aList.Count - 1
              do AddData( aList[ p]);
            end
            else Error( 'no data', aLine, p);
          finally
            aList.DisposeOf;
          end;

        end
        else error( 'empty phrase', aLine, q);
      end
      else Error( 'empty phrase', aLine, q);
    end;


{ ========
  TWrenTalkiePhraseSet = class
  private
    FName    : string;
    FPhrases : array of TWrenTalkiePhrase;
  public
    property    Name                      : string  read FName;
    property    Count                     : Integer read GetCount;
    property    Phrase[ anIndex: Integer] : string  read GetPhrase;
    property    Data  [ anIndex: Integer] : TBytes  read GetData;
  private
}

    function    TTalkiePhraseSet.GetCount: Integer;
    begin
      Result := Length( FPhrases);
    end;


    function    TTalkiePhraseSet.GetPhrase( anIndex: Integer): string;
    begin
      Result := FPhrases[ anIndex].FPhrase;
    end;


    function    TTalkiePhraseSet.GetData( anIndex: Integer): TBytes;
    begin
      Result := FPhrases[ anIndex].Data;
    end;


    procedure   TTalkiePhraseSet.Error( const aMsg, aLine: string; aLineNr: Integer);
    begin
      raise ETalkieParseError.CreateFmt( 'Parse error: ''%s'' in line %d ''%s''', [ aMsg, aLineNr, aLine]);
    end;


    procedure   TTalkiePhraseSet.AddPhrase( const aLine: string);
    var
      aPhrase : TTalkiePhrase;
    begin
      aPhrase := TTalkiePhrase.Create( aLine);
      SetLength( FPhrases, Count + 1);
      FPhrases[ Count - 1] := aPhrase;
    end;


//  public

    constructor TTalkiePhraseSet.Create( const aLines: TStringList; var anIndex: Integer);
    var
      S           : string;
      aStartIndex : Integer;
      DidWork     : Boolean;
    begin
      if Assigned( aLines)
      then begin
        DidWork     := False;
        aStartIndex := anIndex;

        if aStartIndex >= aLines.Count
        then Error( 'no lines', '', aStartIndex);

        while anIndex < aLines.Count
        do begin
          DidWork := True;
          S       := Trim( aLines[ anIndex]);
          if S <> ''
          then begin
            FName := S;
            Break;
          end;
          Inc( anIndex);
        end;

        if ( anIndex >= aLines.Count) or not DidWork
        then Exit;

        Inc( anIndex);
        while anIndex < aLines.Count
        do begin
          S := Trim( aLines[ anIndex]);
          if S = '{'
          then Break;
          Inc( anIndex)
        end;
        if ( anIndex >= aLines.Count)
        then Error( 'Expected {', '', 1);

        Inc( anIndex);
        while anIndex < aLines.Count
        do begin
          S := Trim( aLines[ anIndex]);
          if S = '}'
          then Break
          else begin
            if S <> ''
            then AddPhrase( aLines[ anIndex]);
          end;
          Inc( anIndex);
        end;
        if ( anIndex >= aLines.Count)
        then Error( 'Expected }', '', anIndex);

        Inc( anIndex);
      end
      else Error( 'no lines', '', anIndex);
    end;


    destructor  TTalkiePhraseSet.Destroy; // override;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FPhrases[ i].DisposeOf;
      inherited;
    end;


{ ========
  TWrenTalkiePhraseSets = class
  private
    FPhraseSets : array of TWrenTalkiePhraseSet;
  public
    property    Count                        : Integer              read GetCount;
    property    Name     [ anIndex: Integer] : string               read GetName;
    property    PhraseSet[ anIndex: Integer] : TWrenTalkiePhraseSet read GetPhraseSet;
  private
}

    function    TTalkiePhraseSets.GetCount: Integer;
    begin
      Result := Length( FPhraseSets);
    end;


    function    TTalkiePhraseSets.GetName( anIndex: Integer): string;
    begin
      Result := FPhraseSets[ anIndex].Name;
    end;


    function    TTalkiePhraseSets.GetPhraseSet( anIndex: Integer): TTalkiePhraseSet;
    begin
      Result := FPhraseSets[ anIndex];
    end;


//  private

    procedure   TTalkiePhraseSets.AddPhraseSet( const aLines: TStringList; var anIndex: Integer);
    var
      aPhraseSet : TTalkiePhraseSet;
    begin
      aPhraseSet := TTalkiePhraseSet.Create( aLines, anIndex);
      if aPhraseSet.Count = 0
      then aPhraseSet.DisposeOf
      else begin
        SetLength( FPhraseSets, Count + 1);
        FPhraseSets[ Count - 1] := aPhraseSet;
      end;
    end;


//  public

    constructor TTalkiePhraseSets.Create( const aFileName: string);
    var
      aList   : TStringList;
      anIndex : Integer;
    begin
      inherited Create;
      if FileExists( aFileName)
      then begin
        aList := TStringList.Create;
        try
          aList.LoadFromFile( aFileName);
          anIndex := 0;
          while anIndex < aList.Count
          do AddPhraseSet( aList, anIndex);
        finally
          aList.DisposeOf;
        end;
      end
      else raise ETalkieParseError.CreateFmt( 'file ''%s'' not found', [ aFileName]);
    end;


    destructor  TTalkiePhraseSets.Destroy; // override;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FPhraseSets[ i].DisposeOf;
      inherited;
    end;


    function    TTalkiePhraseSets.CreateBankNames: TStringList;
    var
      i : Integer;
    begin
      Result := TStringList.Create;
      for i := 0 to Count - 1
      do Result.Add( Name[ i]);
    end;


{ ========
  TWrenTalkie = class
  private
    FOnStarted      : TTalkieOnSayStarted;
    FOnDone         : TTalkieOnSayDone;
  private
    FFrameRate      : TSignal;
    FSampleRate     : TSignal;
    FData           : TBytes;
    FIndexAddress   : Integer;
    FIndexBit       : Byte;
    FRateDelta      : TSignal;
    FRateAccu       : TSignal;
    FFrameDelta     : TSignal;
    FFrameAccu      : TSignal;
    FActive         : Boolean;
    FPrevValue      : TSignal;
    FNextValue      : TSignal;
  private
    FPeriodCounter  : Byte;
    FX0             : SmallInt;
    FX1             : SmallInt;
    FX2             : SmallInt;
    FX3             : SmallInt;
    FX4             : SmallInt;
    FX5             : SmallInt;
    FX6             : SmallInt;
    FX7             : SmallInt;
    FX8             : SmallInt;
    FX9             : SmallInt;
  private
    FSynthPeriod    : Byte;
    FSynthEnergy    : Byte;
    FSynthK1        : SmallInt;
    FSynthK2        : SmallInt;
    FSynthK3        : ShortInt;
    FSynthK4        : ShortInt;
    FSynthK5        : ShortInt;
    FSynthK6        : ShortInt;
    FSynthK7        : ShortInt;
    FSynthK8        : ShortInt;
    FSynthK9        : ShortInt;
    FSynthK10       : ShortInt;
  public
    property    OnStarted  : TTalkieOnSayStarted read FOnStarted  write FOnStarted;
    property    OnDone     : TTalkieOnSayDone    read FOnDone     write FOnDone;
    property    FrameRate  : TSignal             read FFrameRate  write SetFrameRate;
    property    SampleRate : TSignal             read FSampleRate write SetSampleRate;
  private
}

    procedure   TTalkie.SetFrameRate( aValue: TSignal);
    begin
      if aValue <> FFrameRate
      then begin
        FFrameRate  := aValue;
        FFrameDelta := FFrameRate * System_Rate_Rec;
      end;
    end;


    procedure   TTalkie.SetSampleRate( aValue: TSignal);
    begin
      if aValue <> FSampleRate
      then begin
        FSampleRate := aValue;
        FRateDelta  := FSampleRate * System_Rate_Rec;
      end;
    end;


//  private

    procedure   TTalkie.SetIndex( aValue: Integer);
    begin
      FIndexAddress := aValue;
      FIndexBit     := 0;
    end;


    function    TTalkie.ReverseBits( aByte: Byte): Byte;
    // The ROMs used with the TI speech were serial, not byte wide.
    // Here's a handy routine to flip ROM data which is usually reversed.
    begin
      Result :=  ( aByte           shr 4) or (  aByte           shl 4);
      Result := (( Result and $cc) shr 2) or (( Result and $33) shl 2);
      Result := (( Result and $aa) shr 1) or (( Result and $55) shl 1);
    end;


    function    TTalkie.GetBits( anAmount: Byte): Byte;
    var
      aData       : Word;
      NewIndexBit : Byte;
    begin
      NewIndexBit := FIndexBit + anAmount;
      aData       := Word( ReverseBits( FData[ FIndexAddress])) shl 8;

      if NewIndexBit > 8
      then aData := aData or ReverseBits( FData[ FIndexAddress + 1]);

      aData     := aData shl FIndexBit;
      Result    := aData shr ( 16 - anAmount);
      FIndexBit := NewIndexBit;

      if FIndexBit >= 8
      then begin
        Dec( FIndexBit, 8);
        Inc( FIndexAddress);
      end;
    end;


    procedure   TTalkie.Calculate;
    // Read speech data, processing the variable size frames
    // This function should be called at FrameRate - currently 40 frames / s.
    var
      Energy : Byte;
      DoRep  : Boolean;
    begin
      Energy := GetBits( 4);

      case Energy of

        $00 : FSynthEnergy := 0;            // Energy = $00: rest frame

        $0f : begin                         // Energy = $0f: stop frame. Silence the synthesiser.
            FSynthEnergy := 0;
            FSynthK1     := 0;
            FSynthK2     := 0;
            FSynthK3     := 0;
            FSynthK4     := 0;
            FSynthK5     := 0;
            FSynthK6     := 0;
            FSynthK7     := 0;
            FSynthK8     := 0;
            FSynthK9     := 0;
            FSynthK10    := 0;
            Stop;
          end;

        else begin
          FSynthEnergy := tmsEnergy[ Energy];
          DoRep        := GetBits( 1) = 1;
          FsynthPeriod := tmsPeriod[ getBits( 6)];

          if not DoRep                      // A repeat frame uses the last coefficients
          then begin                        // All frames use the first 4 coefficients
            FSynthK1 := SmallInt( tmsK1[ GetBits( 5)]);
            FSynthK2 := SmallInt( tmsK2[ GetBits( 5)]);
            FSynthK3 := ShortInt( tmsK3[ GetBits( 4)]);
            FSynthK4 := ShortInt( tmsK4[ GetBits( 4)]);

            if FSynthPeriod <> 0
            then begin                      // Voiced frames use 6 extra coefficients.
              FSynthK5  := ShortInt( tmsK5 [ GetBits( 4)]);
              FSynthK6  := ShortInt( tmsK6 [ GetBits( 4)]);
              FSynthK7  := ShortInt( tmsK7 [ GetBits( 4)]);
              FSynthK8  := ShortInt( tmsK8 [ GetBits( 3)]);
              FSynthK9  := ShortInt( tmsK9 [ GetBits( 3)]);
              FSynthK10 := ShortInt( tmsK10[ GetBits( 3)]);
            end;
          end;
        end;

      end;
    end;


    function    TTalkie.NextValue: TSignal;
    // This function should be called at the speech sample rate
    var
      u0  : SmallInt;
      u1  : SmallInt;
      u2  : SmallInt;
      u3  : SmallInt;
      u4  : SmallInt;
      u5  : SmallInt;
      u6  : SmallInt;
      u7  : SmallInt;
      u8  : SmallInt;
      u9  : SmallInt;
      u10 : SmallInt;
    begin
      if FSynthPeriod <> 0
      then begin                                                  // Voiced source
        if FPeriodCounter < FSynthPeriod
        then Inc( FPeriodCounter)
        else FPeriodCounter := 0;

        if FPeriodCounter < Length( chirp)
        then u10 := SmallInt( chirp[ FPeriodCounter])
        else u10 := 0;
      end
      else u10 := FSynthEnergy * ( 2 * Random( 2) - 1);           // Unvoiced source

       // Lattice filter forward path

      u9 := u10 - (( SmallInt( FSynthK10) * FX9) shr  7);
      u8 := u9  - (( SmallInt( FSynthK9 ) * FX8) shr  7);
      u7 := u8  - (( SmallInt( FSynthK8 ) * FX7) shr  7);
      u6 := u7  - (( SmallInt( FSynthK7 ) * FX6) shr  7);
      u5 := u6  - (( SmallInt( FSynthK6 ) * FX5) shr  7);
      u4 := u5  - (( SmallInt( FSynthK5 ) * FX4) shr  7);
      u3 := u4  - (( SmallInt( FSynthK4 ) * FX3) shr  7);
      u2 := u3  - (( SmallInt( FSynthK3 ) * FX2) shr  7);
      u1 := u2  - (( Integer ( FSynthK2 ) * FX1) shr 15);
      u0 := u1  - (( Integer ( FSynthK1 ) * FX0) shr 15);

      // Output clamp

      u0 := Clip( u0, -512, 511);

      // Lattice filter reverse path

      FX9 := FX8 + (( SmallInt( FSynthK9) * u8) shr  7);
      FX8 := FX7 + (( SmallInt( FSynthK8) * u7) shr  7);
      FX7 := FX6 + (( SmallInt( FSynthK7) * u6) shr  7);
      FX6 := FX5 + (( SmallInt( FSynthK6) * u5) shr  7);
      FX5 := FX4 + (( SmallInt( FSynthK5) * u4) shr  7);
      FX4 := FX3 + (( SmallInt( FSynthK4) * u3) shr  7);
      FX3 := FX2 + (( SmallInt( FSynthK3) * u2) shr  7);
      FX2 := FX1 + (( Integer ( FSynthK2) * u1) shr 15);
      FX1 := FX0 + (( Integer ( FSynthK1) * u0) shr 15);
      FX0 := u0;

      Result := u0 / 512;
    end;


//  public

    constructor TTalkie.Create;
    begin
      inherited;
      FRateAccu   :=    0.0;
      FFrameAccu  :=    0.0;
      SampleRate  := 8000.0;
      FrameRate   :=   40.0;
    end;


    destructor  TTalkie.Destroy; // override;
    begin
      Stop;
      FOnStarted := nil;
      FOnDone    := nil;
      inherited;
    end;


    procedure   TTalkie.Say( aFrameRate, aSampleRate: TSignal; const aData: TBytes);
    begin
      Stop;
      FFrameRate  := aFrameRate;
      FSampleRate := aSampleRate;
      FData       := aData;
      Reset;
    end;


    procedure   TTalkie.Reset;
    begin
      Stop;
      SetIndex( 0);           // Jump back to start of current Phrase
      FPrevValue := 0;
      FNextValue := 0;
      Start;
    end;


    function    TTalkie.Tick: TSignal;
    // This procedure is to be called at System_Rate
    begin
      Result := 0;
      if FActive
      then begin
        // Frame rate
        FFrameAccu := FFrameAccu + FFrameDelta;

        if FFrameAccu > 1
        then begin
          FFrameAccu := FFrameAccu - 1;
          Calculate;
        end;

        // Sample calculation and rate conversion, linearly interpolate between previous and next values
        FRateAccu := FRateAccu + FRateDelta;

        if FRateAccu > 1
        then begin
          FRateAccu  := FRateAccu - 1;
          FPrevValue := FNextValue;
          FNextValue := NextValue;
        end;

        Result := FPrevValue + FRateAccu * ( FNextValue - FPrevValue);
      end;
    end;


    procedure   TTalkie.Start;
    begin
      FActive := True;
      if Assigned( FOnStarted)
      then FOnStarted( Self);
    end;


    procedure   TTalkie.Stop;
    begin
      FActive := False;
      if Assigned( FOnDone)
      then FOnDone( Self);
    end;


end.

