unit freereverb;
{
  Unit: Reverb for either KOL or VCL Delphi and Freepascal 1.9.x,
  probably Kylix.
  purpose: Based on the Freeverb (C++) design by Jezar[at]dreampoint.co.uk
  Author: Thaddy de Koning, thaddy[at]thaddy.com
  Copyright: Original in C++ 2000, Jezar
  Delphi and Freepascal version 2003,
  Basm  added 2004, Thaddy de Koning
  Use as you like, copyrighted freeware.

  Remarks: Removed original skip for interleave.
  Comb and Allpass filter processing rewritten in BASM.
  It is as faithfull a translation as I could manage.
  Sound is exactly the same as FreeVerb 3.
  I have writen it to be as portable as my Pascals go
  Although not tested, it should work in BCB6+ and Kylix too.
  It works with Delphi 6/7 and Freepascal 1.9.X under both linux
  and windows.
  There are no dependencies on the windows unit.

  2014-05-16 : Changed to be used in WREN by Blue Hell to work on a single
  sample pair instead of a buffer. This was downloaded from :
  http://members.chello.nl/t.koning8/kolindex.htm .
  It will not work on all platforms anymore.
}

interface


uses

  Classes, KnobsUtils;


//
// Reverb model tuning values, taken from original algoritm by Jezar
//

const

  Kdenorm : single = 1.0e-23;

  numcombs       = 8;
  numallpasses   = 4;
  muted          = 0;
  fixedgain      = 0.015;
  scalewet       = 3;
  scaledry       = 2;
  scaledamp      = 0.9;        // was 0.4 :: 0.9 provides more damping
  scaleroom      = 0.28;       // was 0.28
  offsetroom     = 0.7;
  initialroom    = 0.5;
  initialdamp    = 0.5;
  initialwet     = 1 / scalewet;
  initialdry     = 1 / scaledry;
  initialwidth   = 1;
  initialmode    = 0;
  freezemode     = 0.5;
  stereospread_l = 0;          // Original value
  stereospread_r = +23;        // Original value


type

  TAllpass = class( TObject)
  private
    feedback : single;
    buffer   : pointer;
    bufsize  : integer;
    bufidx   : integer;
  public
    constructor Create( Buffersize : integer);                                                                  virtual;
    destructor  Destroy;                                                                                       override;
    function    process( const input : single) : single;                                                       register;
    procedure   mute;
    procedure   setfeedback( Value : single);
    function    getfeedback: single;
  end;


  Tcomb = class( TObject)
  private
    feedback    : single;
    filterstore : single;
    damp1       : single;
    damp2       : single;
    buffer      : pointer;
    bufsize     : integer;
    bufidx      : integer;
  public
    constructor Create( Buffersize: Integer);                                                                   virtual;
    destructor  Destroy;                                                                                       override;
    function    process( const input: single): single;                                                         register;
    procedure   mute;
    procedure   setdamp    ( Value : single);
    function    getdamp    : single;
    procedure   setfeedback( Value : single);
    function    getfeedback: single;
  end;


  TReverb = class( TObject)
  private
    gain      : single;
    roomsize  : single;
    roomsize1 : single;
    damp      : single;
    damp1     : single;
    wet       : single;
    wet1      : single;
    wet2      : single;
    dry       : single;
    Width     : single;
    mode      : single;
    FSize     : single;
    combL     : array[ 0 .. numcombs     - 1] of Tcomb;
    combR     : array[ 0 .. numcombs     - 1] of Tcomb;
    allpassL  : array[ 0 .. numallpasses - 1] of TAllpass;
    allpassR  : array[ 0 .. numallpasses - 1] of TAllpass;
  protected
    procedure   Update;
  public
    constructor Create( aSampleRate: Integer; aSize: TSignal);
    destructor  Destroy;                                                                                       override;
    procedure   Mute;
    procedure   Process( const anInL, anInR : single; var anOutL, anOutR : single);
    procedure   SeRoomSize( Value : single);
    function    GetRoomSize: single;
    procedure   SetDamp    ( Value : single);
    function    GetDamp    : single;
    procedure   SetWet     ( Value : single);
    function    GetWet     : single;
    procedure   SetDry     ( Value : single);
    function    GetDry     : single;
    procedure   SetWidth   ( Value : single);
    function    GetWidth   : single;
    procedure   SetMode    ( Value : single);
    function    GetMode    : single;
  end;


implementation


{ ========
  TAllpass = class( TObject)
  private
    feedback : single;
    buffer   : pointer;
    bufsize  : integer;
    bufidx   : integer;
  public
}

    constructor TAllpass.Create( Buffersize: integer);
    begin
      inherited Create;
      bufsize := Buffersize * 4;
      buffer  := AllocMem(bufsize);
      bufidx  := 0;
    end;


    destructor TAllpass.Destroy;
    begin
      freemem( buffer);
      inherited;
    end;


    function TAllpass.getfeedback: single;
    begin
      Result := feedback;
    end;


    procedure TAllpass.mute;
    begin
      Fillchar( buffer^, bufsize, 0);
    end;


    { I really don't know if this is all as fast as can be,
      but it beats Delphi's compiler generated code hands down,
      Thaddy }
    function TAllpass.process( const input: single): single;
    {$ifdef win32}
    asm
      mov  ecx, [ eax].buffer                 // buffer start in ecx
      mov  edx, [ eax].Bufidx                 // buffer index in edx
      fld  input

      // This checks for very small values that can cause a processor
      // to switch in extra precision mode, which is expensive.
      // Since such small values are irrelevant to audio, avoid this.
      // The code is equivalent to the C inline macro by Jezar
      // This is the same spot where the original C macro appears
      test dword ptr [ ecx + edx], $7F800000  // test if denormal
      jnz @Normal
      mov dword ptr [ ecx + edx], 0           // if so, zero out
    @normal:

      fld  [ ecx + edx].Single                // load current sample from buffer
      fsub st( 0), st( 1)                     // subtract input sample
      // NOT fsub, because delphi 7 translates that into fsubp!
      fxch                                    // this is a zero cycle operant,
      // just renames the stack internally
      fmul [ eax].feedback                    // multiply stored sample with feedback
      fadd input                              // and add the input
      fstp [ ecx + edx].Single;               // store at the current sample pos
      add  edx, 4                             // increment sample position
      cmp  edx, [ eax].BufSize;               // are we at end of buffer?
      jb   @OK
      xor  edx, edx                           // if so, reset buffer index
    @OK:
      mov  [ eax].bufidx, edx                 // and store new index,
      // result already in st(0),
      // hence the fxch
    end;
    {$else}
    begin
      Result := 0;
    end;
    {$endif}


    procedure TAllpass.setfeedback( Value : single);
    begin
      feedback := Value;
    end;


{ ========
  Tcomb = class( TObject)
  private
    feedback    : single;
    filterstore : single;
    damp1       : single;
    damp2       : single;
    buffer      : pointer;
    bufsize     : integer;
    bufidx      : integer;
  public
}

    constructor Tcomb.Create( Buffersize : integer);
    begin
      inherited Create;
      bufsize     := Buffersize * 4;
      buffer      := AllocMem( bufsize);
      filterstore := 0;
      bufidx      := 0;
    end;


    destructor Tcomb.Destroy;
    begin
      freemem( buffer);
      inherited;
    end;


    function Tcomb.getdamp : single;
    begin
      Result := damp1;
    end;

    function Tcomb.getfeedback : single;
    begin
      Result := feedback;
    end;


    procedure Tcomb.mute;
    begin
      Fillchar( buffer^, bufsize, 0);
    end;


    { I really don't know if this is all as fast as can be,
      but it beats Delphi's compiler generated code hands down,
      Thaddy }

    function Tcomb.process( const input : single) : single;
    {$ifdef win32}
    asm
      mov   ecx, [ eax].Buffer                       // buffer start in ecx
      mov   edx, [ eax].Bufidx                       // buffer index in edx

      // This checks for very small values that can cause a processor
      // to switch in extra precision mode, which is expensive.
      // Since such small values are irrelevant to audio, avoid this.
      // This is the same spot where the original C macro appears
      test  dword ptr [ ecx + edx], $7F800000        // test if denormal
      jnz   @Normal
      mov   dword ptr [ ecx + edx], 0                // if so, zero out
    @normal:

      fld   [ ecx + edx].Single;                     // load sample from buffer
      fld   st( 0)                                   // duplicate on the stack
      fmul  [ eax].damp2                             // multiply with damp2
      fld   [ eax].filterstore;                      // load stored filtered sample
      fmul  [ eax].damp1                             // multiply with damp1
      faddp
      // fadd  Kdenorm
      fst   [ eax].filterstore                       // store it back

      // This checks for very small values that can cause a processor
      // to switch in extra precision mode, which is expensive.
      // Since such small values are irrelevant to audio, avoid this.
      // This is the same spot where the original C macro appears
      test  dword ptr [ eax].filterstore, $7F800000  // test if denormal
      jnz   @Normal2
      mov   dword ptr [ eax].filterstore, 0          // if so, zero out
    @normal2:

      fmul  [ eax].feedback                          // multiply with feedback
      fadd  input                                    // and add to input sample

      // fadd  Kdenorm
      fstp  [ ecx + edx].Single                      // store at current buffer pos
      add   edx, 4                                   // update buffer index
      cmp   edx, [ eax].BufSize;                     // end of buffer reached?
      jb    @OK

      xor   edx, edx                                 // if so, reset buffer index
    @OK:
      mov  [ eax].bufidx, edx                        // and store new index.
      // result already in st(0),
      // hence duplicate
    end;
    {$else}
    begin
      Result := 0;
    end;
    {$endif}


    procedure Tcomb.setdamp(Value : single);
    begin
      damp1 := Value;
      damp2 := 1 - Value;
    end;


    procedure Tcomb.setfeedback(Value : single);
    begin
      feedback := Value;
    end;


{ ========
  TReverb = class( TObject)
  private
    gain      : single;
    roomsize  : single;
    roomsize1 : single;
    damp      : single;
    damp1     : single;
    wet       : single;
    wet1      : single;
    wet2      : single;
    dry       : single;
    Width     : single;
    mode      : single;
    FSize     : single;
    combL     : array[ 0 .. numcombs     - 1] of Tcomb;
    combR     : array[ 0 .. numcombs     - 1] of Tcomb;
    allpassL  : array[ 0 .. numallpasses - 1] of TAllpass;
    allpassR  : array[ 0 .. numallpasses - 1] of TAllpass;
  protected
}

    procedure TReverb.Update;
    var
      i : integer;
    begin
      // Recalculate internal values after parameter change
      wet1 := wet * ( Width / 2 + 0.5);
      wet2 := wet * (( 1 - Width) / 2);

      if mode >= freezemode
      then begin
        roomsize1 := 1;
        damp1     := 0;
        gain      := muted;
      end
      else begin
        roomsize1 := roomsize;
        damp1     := damp;
        gain      := fixedgain;
      end;

      for i := 0 to numcombs - 1
      do begin
        combL[ i].setfeedback( roomsize1);
        combR[ i].setfeedback( roomsize1);
        combL[ i].setdamp( damp1);
        combR[ i].setdamp( damp1);
      end;
    end;


//  public

    constructor TReverb.Create( aSampleRate: Integer; aSize: TSignal);
    begin
      inherited Create;

      // These values assume 44.1KHz sample rate
      // they will probably be OK for 48KHz sample rate
      // but would need scaling for 96KHz (or other) sample rates.
      // The values were obtained by listening tests.
      // Blue Hell:
      // The values are now recalculated for different sample rates,
      // maybe some prime checking would be in order though

      if aSize < 0.2
      then aSize := 0.2;

      FSize := aSize;

      combL   [ 0]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1116 / 44100) + stereospread_l)));
      combR   [ 0]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1116 / 44100) + stereospread_r)));
      combL   [ 1]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1188 / 44100) + stereospread_l)));
      combR   [ 1]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1188 / 44100) + stereospread_r)));
      combL   [ 2]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1277 / 44100) + stereospread_l)));
      combR   [ 2]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1277 / 44100) + stereospread_r)));
      combL   [ 3]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1356 / 44100) + stereospread_l)));
      combR   [ 3]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1356 / 44100) + stereospread_r)));
      combL   [ 4]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1422 / 44100) + stereospread_l)));
      combR   [ 4]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1422 / 44100) + stereospread_r)));
      combL   [ 5]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1491 / 44100) + stereospread_l)));
      combR   [ 5]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1491 / 44100) + stereospread_r)));
      combL   [ 6]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1557 / 44100) + stereospread_l)));
      combR   [ 6]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1557 / 44100) + stereospread_r)));
      combL   [ 7]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1617 / 44100) + stereospread_l)));
      combR   [ 7]    := Tcomb   .Create( NextPrime( Round(( aSize * aSampleRate * 1617 / 44100) + stereospread_r)));

      allpassL[ 0]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  556 / 44100) + stereospread_l)));
      allpassR[ 0]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  556 / 44100) + stereospread_r)));
      allpassL[ 1]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  441 / 44100) + stereospread_l)));
      allpassR[ 1]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  441 / 44100) + stereospread_r)));
      allpassL[ 2]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  341 / 44100) + stereospread_l)));
      allpassR[ 2]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  341 / 44100) + stereospread_r)));
      allpassL[ 3]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  225 / 44100) + stereospread_l)));
      allpassR[ 3]    := TAllpass.Create( NextPrime( Round(( aSize * aSampleRate *  225 / 44100) + stereospread_r)));

      // Set default values

      allpassL[ 0].setfeedback( 0.5);
      allpassR[ 0].setfeedback( 0.5);
      allpassL[ 1].setfeedback( 0.5);
      allpassR[ 1].setfeedback( 0.5);
      allpassL[ 2].setfeedback( 0.5);
      allpassR[ 2].setfeedback( 0.5);
      allpassL[ 3].setfeedback( 0.5);
      allpassR[ 3].setfeedback( 0.5);

      SetWet    ( initialwet  );
      SeRoomSize( initialroom );
      SetDry    ( initialdry  );
      SetDamp   ( initialdamp );
      SetWidth  ( initialwidth);
      SetMode   ( initialmode );
      Mute;
    end;


    destructor TReverb.Destroy;
    var
      i : integer;
    begin
      for i := Low( allPassL) to High( allPassL) do allpassL[ i].DisposeOf;
      for i := Low( allPassR) to High( allPassR) do allpassR[ i].DisposeOf;
      for i := Low( combL   ) to High( combL   ) do combL   [ i].DisposeOf;
      for i := Low( combR   ) to High( combR   ) do combR   [ i].DisposeOf;

      inherited;
    end;


    function TReverb.GetDamp: single;
    begin
      Result := damp / scaledamp;
    end;


    function TReverb.GetDry: single;
    begin
      Result := dry / scaledry;
    end;


    function TReverb.GetMode: single;
    begin
      if mode >= freezemode
      then Result := 1
      else Result := 0;
    end;


    function TReverb.GetRoomSize : single;
    begin
      Result := (roomsize - offsetroom) / scaleroom;
    end;


    function TReverb.GetWet : single;
    begin
      Result := wet / scalewet;
    end;


    function TReverb.GetWidth : single;
    begin
      Result := Width;
    end;


    procedure TReverb.Mute;
    var
      i : integer;
    begin
      if GetMode >= freezemode
      then Exit;

      for i := 0 to numcombs - 1
      do begin
        combL[ i].mute;
        combR[ i].mute;
      end;
      for i := 0 to numallpasses - 1
      do begin
        allpassL[ i].mute;
        allpassR[ i].mute;
      end;
    end;


    procedure TReverb.Process(const anInL, anInR : single; var anOutL, anOutR : single);
    var
      outL, outR, input : single;
      j                 : integer;
    begin
      outL  := 0;
      outR  := 0;
      input := ( anInL + anInR) * gain;
      // Accumulate comb filters in parallel

      for j := 0 to numcombs - 1
      do begin
        outL := outL + combL[ j].process( input);
        outR := outR + combR[ j].process( input);
      end;

      // Feed through allpasses in series
      for j := 0 to numallpasses - 1
      do begin
        outL := allpassL[ j].process( outL);
        outR := allpassR[ j].process( outR);
      end;

      // Calculate output REPLACING anything already there
      anOutL := outL * wet1 + outR * wet2 + anInL * dry;
      anOutR := outR * wet1 + outL * wet2 + anInR * dry;
    end;


    procedure TReverb.SetDamp( Value : single);
    var
      aNewValue: Single;
    begin
      aNewValue := Value * scaledamp;

      if aNewValue <> damp
      then begin
        damp := aNewValue;
        Update;
      end;
    end;


    procedure TReverb.SetDry( Value : single);
    var
      aNewValue: Single;
    begin
      aNewValue := Value * scaledry;

      if aNewValue <> dry
      then begin
        dry := aNewValue;
        Update;
      end;
    end;


    procedure TReverb.SetMode( Value : single);
    begin
      if Value <> mode
      then begin
        mode := Value;
        Update;
      end;
    end;


    procedure TReverb.SeRoomSize( Value : single);
    var
      aNewValue: Single;
    begin
      aNewValue := ( Value * scaleroom) + offsetroom;

      if aNewValue <> roomsize
      then begin
        roomsize := aNewValue;
        Update;
      end;
    end;


    procedure TReverb.SetWet( Value : single);
    var
      aNewValue: Single;
    begin
      aNewValue := Value * scalewet;

      if aNewValue <> wet
      then begin
        wet := aNewValue;
        Update;
      end;
    end;


    procedure TReverb.SetWidth( Value : single);
    begin
      if Value <> width
      then begin
        Width := Value;
        Update;
      end;
    end;


end.

