unit grains;

{

   COPYRIGHT 2015 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

 System.Classes, KnobsUtils;

type

  TBaseGrain = class                            // Basic state for a grain
  public
    Prev      : TBaseGrain;                     // Pointer to prev grain when kept on a linked list      , dynamic
    Next      : TBaseGrain;                     // Pointer to next grain when kept on a linked list      , dynamic
    StartTime : TSignal;                        // Where in the storage array to start, timewise         , static
    Duration  : TSignal;                        // How long the grain should sound                       , static
    RelTime   : TSignal;                        // The current relative position in the grain development, calculated
    Delta     : TSignal;                        // The time step between two samples                     , calculated
  end;
  TbaseGrains = array of TBaseGrain;            // A collection of grains
  TGrainType  = class of TBaseGrain;


  TSimpleGrain = class( TBaseGrain)             // Complete state for one simple grain
  public
    EnvType  : Integer;                         // The Envelope table to apply                           , static
    DelayMod : TSignal;                         // The modulation to apply to playback speed             , static
  end;


  TGrain = class( TSimpleGrain)                 // Complete state for one grain
  public
    Volume   : TSignal;                         // Overall volume for the grain                          , static
    Pan      : TSignal;                         // Left / right stereo positioning                       , static
  end;


  TBaseGranulator = class
  private
    FDestroying        : Boolean;               // True when destructor executes.
    FGrainType         : TGrainType;            // Runtime type of the grains used
    FSampleRate        : Integer;               // The current sample rate
    FStorageTime       : TSignal;               // Amount of time present in the delay line
    FGrainCount        : Integer;               // Grain count
    FStorage           : TSignalArray;          // The delay line to store samples in, set fixed at creation time
    FGrains            : TbaseGrains;           // All grains in the system          , set fixed at object creation time
    FFirstFreeGrain    : TBaseGrain;            // Pointer to first free    grain    , driven by scheduler
    FFirstWorkingGrain : TBaseGrain;            // Pointer to first working grain    , driven by scheduler
    FTime              : TSignal;               // Current time
    FNextGrainTime     : TSignal;               // The time a next grain should be started;
    FTimeDelta         : TSignal;               // Time / sample (between Tick calls)
    FInPointer         : Integer;               // Current insertion point in FStorage
  private
    function    GetStorageCount: Integer;
  private
    procedure   MoveToWorkingGrains( aGrain: TBaseGrain);
    procedure   MoveToFreeGrains   ( aGrain: TBaseGrain);
    function    InterpolateSample  ( const anIndex: TSignal): TSignal;
  public
    constructor Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainType: TGrainType; const aGrainCount: Integer);
    destructor  Destroy;                                                                                       override;
  public
    property    GrainCount   : Integer read FGrainCount;
    property    StorageCount : Integer read GetStorageCount;
  end;


  TSimpleGranulator = class( TBaseGranulator)
  private
    FGrainDuration     : TSignal;               // User real-time parameter - in seconds
    FGrainOffset       : TSignal;               // User real-time parameter - as a fraction of the loop time
    FCurrentEnvelope   : Integer;               // User real-time parameter
    FMute              : TSignal;               // User real-time parameter
    FRecording         : TSignal;               // User real-time parameter
    FDelayModulation   : TSignal;               // User real-time parameter
  private
    function    ProcessGrain( aGrain: TBaseGrain; var anOutput: TSignal): TBaseGrain;
  public
    constructor Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainCount: Integer);
    procedure   Tick( const anInput: TSignal; out anOutput: TSignal);
    procedure   StartNewGrain;
  public
    property    GrainDuration   : TSignal read FGrainDuration   write FGrainDuration;
    property    GrainOffset     : TSignal read FGrainOffset     write FGrainOffset;
    property    CurrentEnvelope : Integer read FCurrentEnvelope write FCurrentEnvelope;
    property    Mute            : TSignal read FMute            write FMute;
    property    Recording       : TSignal read FRecording       write FRecording;
    property    DelayModulation : TSignal read FDelayModulation write FDelayModulation;
  end;


  TGranulator = class( TBaseGranulator)
  private
    FInterOnsetTime    : TSignal;               // User real-time parameter - in seconds
    FGrainDuration     : TSignal;               // User real-time parameter - in seconds
    FGrainOffset       : TSignal;               // User real-time parameter - as a fraction of the loop time
    FCurrentEnvelope   : Integer;               // User real-time parameter
    FGrainVolume       : TSignal;               // User real-time parameter
    FPan               : TSignal;               // User real-time parameter
    FVolume            : TSignal;               // User real-time parameter
    FMute              : TSignal;               // User real-time parameter
    FRecording         : TSignal;               // User real-time parameter
    FDelayModulation   : TSignal;               // User real-time parameter
  private
    function    StartNewGrain( aStartTime: TSignal): TSignal;
    function    ProcessGrain( aGrain: TBaseGrain; var aLeft, aRight: TSignal): TBaseGrain;
  public
    constructor Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainCount: Integer);
    procedure   Tick( const anInput: TSignal; out aLeft, aRight: TSignal);
  public
    property    InterOnsetTime  : TSignal read FInterOnsetTime  write FInterOnsetTime;
    property    GrainDuration   : TSignal read FGrainDuration   write FGrainDuration;
    property    GrainOffset     : TSignal read FGrainOffset     write FGrainOffset;
    property    CurrentEnvelope : Integer read FCurrentEnvelope write FCurrentEnvelope;
    property    GrainVolume     : TSignal read FGrainVolume     write FGrainVolume;
    property    Pan             : TSignal read FPan             write FPan;
    property    Volume          : TSignal read FVolume          write FVolume;
    property    Mute            : TSignal read FMute            write FMute;
    property    Recording       : TSignal read FRecording       write FRecording;
    property    DelayModulation : TSignal read FDelayModulation write FDelayModulation;
  end;


  TRndGranulator = class( TBaseGranulator)
  private
    FMinInterOnsetTime : TSignal;               // User real-time parameter - in seconds
    FMaxInterOnsetTime : TSignal;               // User real-time parameter - in seconds
    FMinGrainDuration  : TSignal;               // User real-time parameter - in seconds
    FMaxGrainDuration  : TSignal;               // User real-time parameter - in seconds
    FGrainOffset       : TSignal;               // User real-time parameter - as a fraction of the loop time
    FCurrentEnvelope   : Integer;               // User real-time parameter
    FMinVolume         : TSignal;               // User real-time parameter
    FMaxVolume         : TSignal;               // User real-time parameter
    FMinPan            : TSignal;               // User real-time parameter
    FMaxPan            : TSignal;               // User real-time parameter
    FVolume            : TSignal;               // User real-time parameter
    FMute              : TSignal;               // User real-time parameter
    FRecording         : TSignal;               // User real-time parameter
    FDelayModulation   : TSignal;               // User real-time parameter
  private
    function    StartNewGrain( aStartTime: TSignal): TSignal;
    function    ProcessGrain( aGrain: TBaseGrain; var aLeft, aRight: TSignal): TBaseGrain;
  public
    constructor Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainCount: Integer); virtual;
    procedure   Tick( const anInput: TSignal; out aLeft, aRight: TSignal);
  public
    property    MinInterOnsetTime : TSignal read FMinInterOnsetTime write FMinInterOnsetTime;
    property    MaxInterOnsetTime : TSignal read FMaxInterOnsetTime write FMaxInterOnsetTime;
    property    MinGrainDuration  : TSignal read FMinGrainDuration  write FMinGrainDuration;
    property    MaxGrainDuration  : TSignal read FMaxGrainDuration  write FMaxGrainDuration;
    property    GrainOffset       : TSignal read FGrainOffset       write FGrainOffset;
    property    CurrentEnvelope   : Integer read FCurrentEnvelope   write FCurrentEnvelope;
    property    MinVolume         : TSignal read FMinVolume         write FMinVolume;
    property    MaxVolume         : TSignal read FMaxVolume         write FMaxVolume;
    property    MinPan            : TSignal read FMinPan            write FMinPan;
    property    MaxPan            : TSignal read FMaxPan            write FMaxPan;
    property    Volume            : TSignal read FVolume            write FVolume;
    property    Mute              : TSignal read FMute              write FMute;
    property    Recording         : TSignal read FRecording         write FRecording;
    property    DelayModulation   : TSignal read FDelayModulation   write FDelayModulation;
  end;


  function  CreateEnvelopeTypeNames: TStringList;


implementation



uses

  System.Math, System.SysUtils, KnobsConversions;

const

  ENVELOPE_TABLE_SIZE = 512;                    // Nr of points in an envelope table

type

  TGrainEnvTable  = TSignalArray;               // An envelope table
  TGrainEnvTables = array of TGrainEnvTable;    // A set of envelope tables
  TGrainEnvNames  = array of string;            // A set of envelope names


  TGEnvs = class                                // Global envelopes wrapper, not to be instantiated, needing the class only.
  private
  class var
    FEnvelopes     : TGrainEnvTables;           // A table of envelope tables, computed at program startup
    FEnvelopeNames : TGrainEnvNames;            // A table of envelope names , computed at program startup
  private
    class constructor Create;                   // Called at program startup (automatrick) - computes the envelope tables.
    class procedure   CreateEnvelopes;
    class function    GetEnvelopeTableCount: Integer;                                  static;
    class function    GetEnvelopeTable( anIndex: Integer): TGrainEnvTable;             static;
    class function    GetEnvelopeName ( anIndex: Integer): string;                     static;
    class procedure   AddEnvelope( const aName: string; const aTable: TGrainEnvTable);
    class procedure   CreateBellEnvelope;
    class procedure   CreateExpEnvelope;
    class procedure   CreateRevExpEnvelope;
    class procedure   CreatePulseEnvelope;
    class procedure   CreateRndBellEnvelope;
    class procedure   CreateRndExpEnvelope;
    class function    Interpolate( const aTableIndex: Integer; const anIndex: TSignal): TSignal;
  public
    class property    EnvelopeTableCount              : Integer        read GetEnvelopeTableCount;
    class property    EnvelopeTable[ anIndex: Integer]: TGrainEnvTable read GetEnvelopeTable;
    class property    EnvelopeName [ anIndex: Integer]: string         read GetEnvelopeName;
  end;


    class constructor TGEnvs.Create;
    begin
      inherited;
      CreateEnvelopes;
    end;


    class procedure   TGEnvs.CreateEnvelopes;
    begin
      CreateBellEnvelope;
      CreateExpEnvelope;
      CreateRevExpEnvelope;
      CreatePulseEnvelope;
      CreateRndBellEnvelope;
      CreateRndExpEnvelope;
    end;


    class function    TGEnvs.GetEnvelopeTableCount: Integer; // static;
    begin
      Result := Length( FEnvelopes);
    end;


    class function    TGEnvs.GetEnvelopeTable( anIndex: Integer): TGrainEnvTable; // static;
    begin
      Result := FEnvelopes[ anIndex];
    end;


    class function    TGEnvs.GetEnvelopeName ( anIndex: Integer): string; // static;
    begin
      Result := FEnvelopeNames[ anIndex];
    end;


    class procedure   TGEnvs.AddEnvelope( const aName: string; const aTable: TGrainEnvTable);
    begin
      SetLength( FEnvelopes    , EnvelopeTableCount + 1);
      SetLength( FEnvelopeNames, EnvelopeTableCount + 1);
      FEnvelopes    [ EnvelopeTableCount - 1] := aTable;
      FEnvelopeNames[ EnvelopeTableCount - 1] := aName;
    end;


    class procedure   TGEnvs.CreateBellEnvelope;
    var
      aData : TSignalArray;
      i     : Integer;
    begin
      SetLength( aData, ENVELOPE_TABLE_SIZE);

      for i := 0 to Length( aData) - 1
      do aData[ i] := sqr( Sin( Pi * ( i / ( Length( aData) - 1))));

      AddEnvelope( 'Bell', aData);
    end;


    class procedure   TGEnvs.CreateExpEnvelope;
    var
      aData : TSignalArray;
      ATime : Integer;
      DTime : Integer;
      TTime : Integer;
      i     : Integer;
      p     : Integer;
    begin
      SetLength( aData, ENVELOPE_TABLE_SIZE);

      TTime    := Length( aData);
      ATime    := Round( 0.05 * TTime);       //  5% attack time
      DTime    := Round( 0.95 * TTime);       // 95% decay  time

      i := 0;

      while i < ATime
      do begin
        aData[ i] := LinAttack( i, ATime, 0, 1);
        Inc( i);
      end;

      p := 0;

      while i < TTime
      do begin
        aData[ i] := ExpDecay( p, DTime, 1, 0);
        Inc( i);
        Inc( p);
      end;

      AddEnvelope( 'Exp', aData);
    end;


    class procedure   TGEnvs.CreateRevExpEnvelope;
    var
      aData : TSignalArray;
      ATime : Integer;
      DTime : Integer;
      TTime : Integer;
      i     : Integer;
      p     : Integer;
    begin
      SetLength( aData, ENVELOPE_TABLE_SIZE);

      TTime    := Length( aData);
      ATime    := Round( 0.95 * TTime);       // 95% attack time
      DTime    := Round( 0.05 * TTime);       //  5% decay  time

      i := 0;

      while i < ATime
      do begin
        aData[ i] := ExpAttack( i, ATime, 0, 1);
        Inc( i);
      end;

      p := 0;

      while i < TTime
      do begin
        aData[ i] := LinDecay( p, DTime, 1, 0);
        Inc( i);
        Inc( p);
      end;

      AddEnvelope( 'RevExp', aData);
    end;


    class procedure   TGEnvs.CreatePulseEnvelope;
    var
      aData  : TSignalArray;
      i      : Integer;
      aScale : TSignal;
    begin
      SetLength( aData, ENVELOPE_TABLE_SIZE);
      aScale := ( 1 / Length( aData)) - 1;

      for i := 0 to Length( aData) - 1
      do aData[ i] := Sin( 6 * Pi * aScale * i) * Sqr( Sin( Pi * aScale * i));

      AddEnvelope( 'Pulse', aData);
    end;

    class procedure   TGEnvs.CreateRndBellEnvelope;
    var
      aData  : TSignalArray;
      i      : Integer;
      aScale : TSignal;
    begin
      SetLength( aData, ENVELOPE_TABLE_SIZE);
      aScale := ( 1 / Length( aData)) - 1;

      for i := 0 to Length( aData) - 1
      do aData[ i] := Random * Sqr( Sin( Pi * aScale * i));

      AddEnvelope( 'RndBell', aData);
    end;


    class procedure   TGEnvs.CreateRndExpEnvelope;
    var
      aData : TSignalArray;
      ATime : Integer;
      DTime : Integer;
      TTime : Integer;
      i     : Integer;
      p     : Integer;
    begin
      SetLength( aData, ENVELOPE_TABLE_SIZE);

      TTime    := Length( aData);
      ATime    := Round( 0.05 * TTime);       //  5% attack time
      DTime    := Round( 0.95 * TTime);       // 95% decay  time

      i := 0;

      while i < ATime
      do begin
        aData[ i] := Random * LinAttack( i, ATime, 0, 1);
        Inc( i);
      end;

      p := 0;

      while i < TTime
      do begin
        aData[ i] := Random * ExpDecay( p, DTime, 1, 0);
        Inc( i);
        Inc( p);
      end;

      AddEnvelope( 'RndExp', aData);
    end;


    class function    TGEnvs.Interpolate( const aTableIndex: Integer; const anIndex: TSignal): TSignal;
    var
      p : Integer;
      q : Integer;
      d : TSignal;
    begin
      p := Trunc( Clip( anIndex, 0.0, ENVELOPE_TABLE_SIZE - 1));
      d := anIndex - p;
      q := ( p + 1);

      if q > ENVELOPE_TABLE_SIZE - 1
      then q := ENVELOPE_TABLE_SIZE - 1;

      Result := Normalize( TGEnvs.FEnvelopes[ aTableIndex][ p] + d * ( TGEnvs.FEnvelopes[ aTableIndex][ q] - TGEnvs.FEnvelopes[ aTableIndex][ p]));
    end;



  function CreateEnvelopeTypeNames: TStringList;
  var
    i : Integer;
  begin
    Result := TStringList.Create;

    for i := 0 to TGEnvs.EnvelopeTableCount - 1
    do Result.Add( TGEnvs.EnvelopeName[ i]);
  end;


{ ========
  TBaseGranulator = class
}

//  private

    function    TBaseGranulator.GetStorageCount: Integer;
    begin
      Result := Length( FStorage);
    end;

//  private

    procedure   TBaseGranulator.MoveToWorkingGrains( aGrain: TBaseGrain);
    // Remove aGrain from the free list and prepend it to the working list
    begin
      if Assigned( aGrain)
      then begin
        if FFirstFreeGrain = aGrain
        then FFirstFreeGrain := aGrain.Next;

        if Assigned( aGrain.Next)
        then aGrain.Next.Prev := aGrain.Prev;

        if Assigned( aGrain.Prev)
        then aGrain.Prev.Next := aGrain.Next;

        aGrain.Prev := nil;
        aGrain.Next := FFirstWorkingGrain;

        if Assigned( aGrain.Next)
        then aGrain.Next.Prev := aGrain;

        FFirstWorkingGrain := aGrain;
      end;
    end;

    procedure   TBaseGranulator.MoveToFreeGrains( aGrain: TBaseGrain);
    // Remove aGrain from the working list and prepend it to the free list
    begin
      if Assigned( aGrain)
      then begin
        if FFirstWorkingGrain = aGrain
        then FFirstWorkingGrain := aGrain.Next;

        if Assigned( aGrain.Next)
        then aGrain.Next.Prev := aGrain.Prev;

        if Assigned( aGrain.Prev)
        then aGrain.Prev.Next := aGrain.Next;

        aGrain.Prev := nil;
        aGrain.Next := FFirstFreeGrain;

        if Assigned( aGrain.Next)
        then aGrain.Next.Prev := aGrain;

        FFirstFreeGrain := aGrain;
      end;
    end;

    function    TBaseGranulator.InterpolateSample( const anIndex: TSignal): TSignal;
    var
      p : Integer;
      q : Integer;
      d : TSignal;
    begin
      p := Trunc( anIndex);
      d := anIndex - p;
      p := p mod StorageCount;
      q := ( p + 1);

      if q >= StorageCount
      then q := 0;

      Result := Normalize( FStorage[ p] + d * ( FStorage[ q] - FStorage[ p]));
    end;

//  public

    constructor TBaseGranulator.Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainType: TGrainType;  const aGrainCount: Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FGrainType         := aGrainType;
      FSampleRate        := aSampleRate;
      FStorageTime       := aStorageDuration;
      FGrainCount        := aGrainCount;
      FFirstFreeGrain    := nil;
      FFirstWorkingGrain := nil;
      FTime              := 0;
      FTimeDelta         := 1 / FSampleRate;         // Let time proceed at one sample / tick

      SetLength( FStorage, Ceil( FStorageTime * System_Rate));
      FillChar( FStorage[ 0], StorageCount * SizeOf( FStorage[ 0]), 0);

      SetLength ( FGrains , FGrainCount);
      for i := 0 to FGrainCount - 1          // Append all grains to the free list
      do begin
        FGrains[ i] := FGrainType.Create;
        MoveToFreeGrains( FGrains[ i]);
      end;
      FDestroying := False;
    end;

    destructor  TBaseGranulator.Destroy; // override;
    var
      i : Integer;
    begin
      FDestroying        := True;
      FFirstFreeGrain    := nil;
      FFirstWorkingGrain := nil;

      for i := 0 to GrainCount - 1
      do FreeAndNil( FGrains[ i]);

      inherited;
    end;


{ ========
  TSimpleGranulator
}

  // private

    procedure   TSimpleGranulator.StartNewGrain;
    begin
      // Get next idle grain and set it up. There may be none, in that case just skip new grain setup.
      if assigned( FFirstFreeGrain) and ( DelayModulation > 0.01)                                      // Zero speed grains will not terminate.
      then begin
        FFirstFreeGrain.StartTime := MathFloatMod( FTime + FGrainOffset * FStorageTime, FStorageTime); // Starting point on the delay line
        FFirstFreeGrain.Duration  := FGrainDuration;                                                   // How long the grain should sound
        FFirstFreeGrain.RelTime   := 0;                                                                // The current relative position in the grain development
        FFirstFreeGrain.Delta     := DelayModulation * FTimeDelta / FFirstFreeGrain.Duration;          // The time step between two samples

        TSimpleGrain( FFirstFreeGrain).EnvType  := FCurrentEnvelope;                                   // The Envelope table to apply
        TSimpleGrain( FFirstFreeGrain).DelayMod := DelayModulation;                                    // The modulation to apply to playback speed

        MoveToWorkingGrains( FFirstFreeGrain);                                                         // Make it a working grain
      end;
    end;

    function    TSimpleGranulator.ProcessGrain( aGrain: TBaseGrain; var anOutput: TSignal): TBaseGrain;
    var
      aPos : TSignal;
    begin
      if FDestroying
      then Result := nil
      else begin
        with TSimpleGrain( aGrain)
        do begin
          Result := Next;                 // Next grain to process

          if RelTime < 0
          then RelTime := 0;

          if RelTime >= 1                 // past life  time?
          then MoveToFreeGrains( aGrain)  // yes, unschedule it.
          else begin
            aPos := RelTime * Duration + StartTime;

            if aPos > FStorageTime
            then aPos := aPos - FStorageTime;

            anOutput := InterpolateSample( aPos * FSampleRate);
            anOutput := Normalize( anOutput * TGEnvs.Interpolate( EnvType, RelTime * ( ENVELOPE_TABLE_SIZE - 1)));
            RelTime  := RelTime + Delta;
          end;
        end;
      end;
    end;

  // public

    constructor TSimpleGranulator.Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainCount: Integer);
    begin
      inherited Create( aSampleRate, aStorageDuration, TSimpleGrain, aGrainCount);
      Recording       := 1.0;                // Enable recording
      DelayModulation := 1.0;                // Run at nominal speed.
    end;

    procedure   TSimpleGranulator.Tick( const anInput: TSignal; out anOutput: TSignal);
    var
      aGrain : TBaseGrain;
    begin
      if ( StorageCount > 0) and not FDestroying
      then begin
        FTime := FTime + FTimeDelta;

        if FTime > FStorageTime
        then begin
          FTime          := FTime          - FStorageTime;
          FNextGrainTime := FNextGrainTime - FStorageTime;
        end;

        anOutput := 0.0;
        aGrain := FFirstWorkingGrain;

        while Assigned( aGrain)
        do aGrain := ProcessGrain( aGrain, anOutput);

        anOutput := Normalize( anOutput * SignalToMute( Mute));
        FStorage[ FInPointer] := Normalize( FRecording * ( anInput - FStorage[ FInPointer]) + FStorage[ FInPointer]);
        FInPointer := ( FInPointer + 1) mod StorageCount;
      end;
    end;


{ ========
  TGranulator
}

  //  private

    function    TGranulator.StartNewGrain( aStartTime: TSignal): TSignal;
    // aStartTime is the time relative to now on which the new grain should start
    // this can be negative if it should have started earlier.
    // The return value is the time a next grain will have to be started, relative to aStartTime.
    begin
      Result := aStartTime + InterOnsetTime;

      // Get next idle grain and set it up. There may be none, in that case just skip new grain setup.
      if assigned( FFirstFreeGrain)
      then begin
        FFirstFreeGrain.StartTime := MathFloatMod( FTime + FGrainOffset * FStorageTime, FStorageTime); // Starting point on the delay line
        FFirstFreeGrain.Duration  := FGrainDuration;                                                   // How long the grain should sound
        FFirstFreeGrain.RelTime   := 0;                                                                // The current relative position in the grain development
        FFirstFreeGrain.Delta     := DelayModulation * FTimeDelta / FFirstFreeGrain.Duration;          // The time step between two samples

        TGrain( FFirstFreeGrain).EnvType  := FCurrentEnvelope;                                         // The Envelope table to apply
        TGrain( FFirstFreeGrain).Volume   := FGrainVolume;                                             // Overall volume for the grain
        TGrain( FFirstFreeGrain).Pan      := FPan;                                                     // Left / right stereo positioning
        TGrain( FFirstFreeGrain).DelayMod := DelayModulation;                                          // The modulation to apply to playback speed

        MoveToWorkingGrains( FFirstFreeGrain);                                                         // Make it a working grain
      end;
    end;

    function    TGranulator.ProcessGrain( aGrain: TBaseGrain; var aLeft, aRight: TSignal): TBaseGrain;
    var
      aSample : TSignal;
      aPos    : TSignal;
    begin
      if FDestroying
      then Result := nil
      else begin
        with TGrain( aGrain)
        do begin
          Result := Next;                 // Next grain to process

          if RelTime < 0
          then RelTime := 0;

          if RelTime >= 1                 // past life  time?
          then MoveToFreeGrains( aGrain)  // yes, unschedule it.
          else begin
            aPos := RelTime * Duration + StartTime;

            if aPos > FStorageTime
            then aPos := aPos - FStorageTime;

            aSample := InterpolateSample( aPos * FSampleRate);
            aSample := Normalize( aSample * TGEnvs.Interpolate( EnvType, RelTime * ( ENVELOPE_TABLE_SIZE - 1)));
            aSample := Normalize( 0.5 * aSample * Volume);
            aLeft   := Normalize( aLeft  + ( 1 - Pan) * aSample);
            aRight  := Normalize( aRight + ( 1 + Pan) * aSample);
            RelTime := RelTime + Delta;
          end;
        end;
      end;
    end;

  // public

    constructor TGranulator.Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainCount: Integer);
    begin
      inherited Create( aSampleRate, aStorageDuration, TGrain, aGrainCount);
      FVolume         := 1.0;                // Set a default volume for when the user does not set it
      Recording       := 1.0;                // Enable recording
      DelayModulation := 1.0;                // Run at nominal speed.
    end;

    procedure   TGranulator.Tick( const anInput: TSignal; out aLeft, aRight: TSignal);
    var
      aGrain : TBaseGrain;
      aLevel : TSignal;
    begin
      if ( StorageCount > 0) and not FDestroying
      then begin
        FTime := FTime + FTimeDelta;

        if ( FTime > FNextGrainTime) and ( DelayModulation > 0.01)  // Grains would not proceed in time when DelayModulation <= 0 ... keep some margin
        then FNextGrainTime := StartNewGrain( FNextGrainTime);

        if FTime > FStorageTime
        then begin
          FTime          := FTime          - FStorageTime;
          FNextGrainTime := FNextGrainTime - FStorageTime;
        end;

        aLeft  := 0.0;
        aRight := 0.0;

        aGrain := FFirstWorkingGrain;

        while Assigned( aGrain)
        do aGrain := ProcessGrain( aGrain, aLeft, aRight);

        aLevel := Normalize( Volume * SignalToMute( Mute));
        aLeft  := Normalize( aLeft  * aLevel);
        aRight := Normalize( aRight * aLevel);

        FStorage[ FInPointer] := Normalize( FRecording * ( anInput - FStorage[ FInPointer]) + FStorage[ FInPointer]);

        FInPointer := ( FInPointer + 1) mod StorageCount;
      end;
    end;



{ ========
  TRndGranulator
}

  //  private

    function    TRndGranulator.StartNewGrain( aStartTime: TSignal): TSignal;
    // aStartTime is the time relative to now on which the new grain should start
    // this can be negative if it should have started earlier.
    // The return value is the time a next grain will have to be started, relative to aStartTime.
    begin
      Result := aStartTime + MinInterOnsetTime + Random * ( MaxInterOnsetTime - MinInterOnsetTime);

      // Get next idle grain and set it up. There may be none, in that case just skip new grain setup.
      if assigned( FFirstFreeGrain)
      then begin
        FFirstFreeGrain.StartTime := MathFloatMod( FTime + FGrainOffset * FStorageTime, FStorageTime);       // Starting point on the delay line
        FFirstFreeGrain.Duration  := FMinGrainDuration + Random * ( FMaxGrainDuration - FMinGrainDuration);  // How long the grain should sound
        FFirstFreeGrain.RelTime   := 0;                                                                      // The current relative position in the grain development
        FFirstFreeGrain.Delta     := DelayModulation * FTimeDelta / FFirstFreeGrain.Duration;                // The time step between two samples

        TGrain( FFirstFreeGrain).EnvType  := FCurrentEnvelope;                                               // The Envelope table to apply
        TGrain( FFirstFreeGrain).Volume   := FMinVolume + Random * ( FMaxVolume - FMinVolume);               // Overall volume for the grain
        TGrain( FFirstFreeGrain).Pan      := FMinPan    + Random * ( FMaxPan    - FMinPan   );               // Left / right stereo positioning
        TGrain( FFirstFreeGrain).DelayMod := DelayModulation;                                                // The modulation to apply to playback speed

        MoveToWorkingGrains( FFirstFreeGrain);                                                               // Make it a working grain
      end
      else Nop;
    end;

    function    TRndGranulator.ProcessGrain( aGrain: TBaseGrain; var aLeft, aRight: TSignal): TBaseGrain;
    var
      aSample : TSignal;
      aPos    : TSignal;
    begin
      if FDestroying
      then Result := nil
      else begin
        with TGrain( aGrain)
        do begin
          Result := Next;                 // Next grain to process

          if RelTime < 0
          then RelTime := 0;

          if RelTime >= 1                 // past life time?
          then MoveToFreeGrains( aGrain)  // yes, unschedule it.
          else begin
            aPos := RelTime * Duration + StartTime;

            if aPos > FStorageTime
            then aPos := aPos - FStorageTime;

            aSample  := InterpolateSample( aPos * FSampleRate);
            aSample  := Normalize( aSample * TGEnvs.Interpolate( EnvType, RelTime * ( ENVELOPE_TABLE_SIZE - 1)));
            aSample  := Normalize( 0.5 * aSample * Volume);
            aLeft    := Normalize( aLeft  + ( 1 - Pan) * aSample);
            aRight   := Normalize( aRight + ( 1 + Pan) * aSample);
            RelTime := RelTime + Delta;
          end;
        end;
      end;
    end;

  // public

    constructor TRndGranulator.Create( const aSampleRate: Integer; const aStorageDuration: TSignal; const aGrainCount: Integer); // virtual;
    begin
      inherited Create( aSampleRate, aStorageDuration, TGrain, aGrainCount);
      FVolume         := 1.0;                // Set a default volume for when the user does not set it
      Recording       := 1.0;                // Enable recording
      DelayModulation := 1.0;                // Run at nominal speed.
    end;

    procedure   TRndGranulator.Tick( const anInput: TSignal; out aLeft, aRight: TSignal);
    var
      aGrain : TBaseGrain;
      aLevel : TSignal;
    begin
      if ( StorageCount > 0) and not FDestroying
      then begin
        FTime := FTime + FTimeDelta;

        if ( FTime > FNextGrainTime) and ( DelayModulation > 0.01)  // Grains would not proceed in time when DelayModulation <= 0 ... with 0.01 we keep some margin
        then FNextGrainTime := StartNewGrain( FNextGrainTime);

        if FTime > FStorageTime
        then begin
          FTime          := FTime          - FStorageTime;
          FNextGrainTime := FNextGrainTime - FStorageTime;
        end;

        aLeft  := 0.0;
        aRight := 0.0;

        aGrain := FFirstWorkingGrain;

        while Assigned( aGrain)
        do aGrain := ProcessGrain( aGrain, aLeft, aRight);

        aLevel := Normalize( Volume * SignalToMute( Mute));
        aLeft  := Normalize( aLeft  * aLevel);
        aRight := Normalize( aRight * aLevel);

        FStorage[ FInPointer] := Normalize( FRecording * ( anInput - FStorage[ FInPointer]) + FStorage[ FInPointer]);

        FInPointer := ( FInPointer + 1) mod StorageCount;
      end;
    end;


end.

