unit gverb;

{
  -------------------------------------------------------------------------------------------------------------------
  Original copyright message:

  Copyright (C) 1999 Juhana Sadeharju
  kouhia at nic.funet.fi
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
}

{
  -------------------------------------------------------------------------------------------------------------------
  2015-04-23 : Translated to Pascal for use in WREN by Blue Hell.

  Some of the changes I made to the algorithm are marked with @BH

  Added a mode parameter as an integer bit map

    bit 0 : when 0 round the delay lengths, when 1 use delays that have relatively prime lengths
    bit 1 : when 0 use the original FDN matrix, when 1 use an alternate matrix
 }

{
 -------------------------------------------------------------------------------------------------------------------

  * This FDN reverb can be made smoother by setting matrix elements at the
  * diagonal and near of it to zero or nearly zero. By setting diagonals to zero
  * means we remove the effect of the parallel comb structure from the
  * reverberation. A comb generates uniform impulse stream to the reverberation
  * impulse response, and thus it is not good. By setting near diagonal elements
  * to zero means we remove delay sequences having consequtive delays of the
  * similar lenths, when the delays are in sorted in length with respect to
  * matrix element index. The matrix described here could be generated by
  * differencing Rocchesso's circulant matrix at max diffuse value and at low
  * diffuse value (approaching parallel combs).
  *
  * Example 1:
  * Set a(k,k), for all k, equal to 0.
  *
  * Example 2:
  * Set a(k,k), a(k,k-1) and a(k,k+1) equal to 0.
  *
  * Example 3: The transition to zero gains could be smooth as well.
  * a(k,k-1) and a(k,k+1) could be 0.3, and a(k,k-2) and a(k,k+2) could
  * be 0.5, say.
}

{
  Some default settings as recommended by Audacity

    The Quick Fix Roomsize:                    40 m
    Reverb time: 	                        4 s
    Damping: 	                                0.9
    Input bandwidth: 	                        0.75
    Dry signal level: 	                        0 dB
    Early reflection level:                   -22 dB
    Tail level:                               -28 dB


    Bright, small hall Roomsize: 	       50 m
    Reverb time: 	                        1.5 s
    Damping: 	                                0.1
    Input bandwidth: 	                        0.75
    Dry signal level: 	                       -1.5 dB
    Early reflection level: 	              -10 dB
    Tail level: 	                      -20 dB


    Nice hall effect Roomsize: 	               40 m
    Reverb time: 	                       20 s
    Damping: 	                                0.50
    Input bandwidth: 	                        0.75
    Dry signal level: 	                        0 dB
    Early reflection level: 	              -10 dB
    Tail level: 	                      -30 dB


    Singing in the Sewer Roomsize:              6 m
    Reverb time: 	                       15 s
    Damping: 	                                0.9
    Input bandwidth: 	                        0.1
    Dry signal level: 	                      -10 dB
    Early reflection level: 	              -10 dB
    Tail level: 	                      -10 dB


    Last row of the church Roomsize:          200 m
    Reverb time: 	                        9 s
    Damping: 	                                0.7
    Input bandwidth: 	                        0.8
    Dry signal level: 	                      -20 dB
    Early reflection level: 	              -15 dB
    Tail level: 	                       -8 dB


    Electric guitar and electric bass Roomsize: 1 m
    Reverb time: 	                        one beat
    Damping: 	                                1
    Input bandwidth: 	                        0.7
    Dry signal level: 	                        0 dB
    Early reflection level: 	              -15 dB
    Tail level: 	                        0 dB

  Some more from : http://doc.sccode.org/Classes/GVerb.html

    in              mono input.
    roomsize	    in squared meters.
    revtime	    in seconds.
    damping	    0 to 1, high frequency rolloff, 0 damps the reverb signal completely, 1 not at all.
    inputbw	    0 to 1, same as damping control, but on the input signal.
    spread	    a control on the stereo spread and diffusion of the reverb signal.
    drylevel	    amount of dry signal.
    earlyreflevel   amount of early reflection level.
    taillevel	    amount of tail level.
    maxroomsize	    to set the size of the delay lines. Defaults to roomsize + 1.

    // bathroom
    a = Synth(\test, [\roomsize, 5, \revtime, 0.6, \damping, 0.62, \inputbw, 0.48, \drylevel -6, \earlylevel, -11, \taillevel, -13]);
    a.free;

    //living room
    a = Synth(\test, [\roomsize, 16, \revtime, 1.24, \damping, 0.10, \inputbw, 0.95, \drylevel -3, \earlylevel, -15, \taillevel, -17]);
    a.free;

    //church
    a = Synth(\test, [\roomsize, 80, \revtime, 4.85, \damping, 0.41, \inputbw, 0.19, \drylevel -3, \earlylevel, -9, \taillevel, -11]);
    a.free;

    // cathedral
    a = Synth(\test, [\roomsize, 243, \revtime, 1, \damping, 0.1, \inputbw, 0.34, \drylevel -3, \earlylevel, -11, \taillevel, -9]);
    a.free

    // canyon
    a = Synth(\test, [\roomsize, 300, \revtime, 103, \damping, 0.43, \inputbw, 0.51, \drylevel -5, \earlylevel, -26, \taillevel, -20]);
    a.free;
}

interface

uses

  System.SysUtils, KnobsUtils, Math;


const

  FDNORDER        = 4;
  DELTA           = 1e-6;       // Small value for equality comparisons
  FDN_MULTIPLIER1 = 0.43;       // @BH 0.5 for the original code, lowered it a bit to make room size modulations stable.
                                // 0..475 was not stable. ... maybe this should not be linear, but decrease with
                                // absolute levels. see aLevel variable in TGVerb.Tick method (this is not currently
                                // used).
  FDN_MULTIPLIER2 = 0.73;       // @BH added seperate multiplier for the 2nd FDN matrix

type

  TFixedDelay = class
  private
    FSize : Integer;
    FIdx  : Integer;
    FBuf  : array of Double;
  protected
    function    Read( anIndex: Integer): Double;
    procedure   Write( aValue: Double);
  public
    constructor Create( aSize: Integer);
    procedure   Flush;
  end;


  TDiffuser = class
  private
    FSize   : Integer;
    FCoeff  : Double;
    FIdx    : Integer;
    FBuffer : array of Double;
  public
    constructor Create( aSize: Integer; aCoeff: Double);
    procedure   Flush;
    function    Tick( aValue: Double): Double;
  end;


  TDamper = class
    FDamping : Double;
    FDelay   : Double;
  public
    constructor Create( aDamping: Double);
    procedure   Flush;
    procedure   SetDamping( aValue: Double);
    function    Tick( aValue: Double): Double;
  end;


  TGVerb = class
  private
    FInputDamper  : TDamper;
    FTapDelay     : TFixedDelay;
    FFdnDels      : array of TFixedDelay;
    FLDifs        : array of TDiffuser;
    FRDifs        : array of TDiffuser;
    FRate         : Integer;
    FMaxRoomSize  : Double;
    FRoomSize     : Double;
    FRevTime      : Double;
    FDamping      : Double;
    FSpread       : Double;
    FInputBW      : Double;
    FEarlyLevel   : Double;
    FTailLevel    : Double;
    FDryLevel     : Double;            // @BH : added dry level
    FMaxDelay     : Double;
    FLargestDelay : Double;
    FFdnGains     : array of Double;
    FFdnLens      : array of Integer;
    FFdnDamps     : array of TDamper;
    FTaps         : array of Integer;
    FTapGains     : array of Double;
    FD            : array of Double;
    FU            : array of Double;
    FF            : array of Double;
    FAlpha        : Double;
    FMode         : Integer;           // @BH : added mode
  public
    constructor Create(
      aSampleRate      : Integer;
      aMaxRoomSize     : Double;
      aRoomSize        : Double;
      aRevTime         : Double;
      aDamping         : Double;
      aSpread          : Double;
      anInputBandWidth : Double;
      anEarlyLevel     : Double;
      aTailLevel       : Double;
      aDryLevel        : Double;
      aMode            : Integer
    );
    destructor  Destroy;                                                                                       override;
    procedure   Flush;
    procedure   Tick( anInvalueL, anInValueR: Double; out aLeft, aRight: Double);
    function    GetMaxRoomSize   : Double;
    procedure   SetRoomSize      ( const aValue: Double);
    procedure   SetRevTime       ( const aValue: Double);
    procedure   SetDamping       ( const aValue: Double);
    procedure   SetInputBandWidth( const aValue: Double);
    procedure   SetEarlyLevel    ( const aValue: Double);
    procedure   SetTailLevel     ( const aValue: Double);
    procedure   SetDryLevel      ( const aValue: Double);
  end;



implementation



  procedure CalculateFdnMatrix1( const anInVals: array of Double; out anOutVals: array of double; aMultiplier: Double);
  // @BH : made the multiplier a parameter .. instead of fixed 0.5
  var
    dl0  : Double;
    dl1  : Double;
    dl2  : Double;
    dl3  : Double;
  begin
    dl0 := anInVals[ 0];
    dl1 := anInVals[ 1];
    dl2 := anInVals[ 2];
    dl3 := anInVals[ 3];

    anOutVals[ 0] := aMultiplier * ( + dl0 + dl1 - dl2 - dl3);    //  1  1 -1 -1
    anOutVals[ 1] := aMultiplier * ( + dl0 - dl1 - dl2 + dl3);    //  1 -1 -1  1
    anOutVals[ 2] := aMultiplier * ( - dl0 + dl1 - dl2 + dl3);    // -1  1 -1  1
    anOutVals[ 3] := aMultiplier * ( + dl0 + dl1 + dl2 + dl3);    //  1  1  1  1
  end;


  procedure CalculateFdnMatrix2( const anInVals: array of Double; out anOutVals: array of double; aMultiplier: Double);
  // @BH : made the multiplier a parameter .. instead of fixed 0.5
  var
    dl0  : Double;
    dl1  : Double;
    dl2  : Double;
    dl3  : Double;
  begin
    dl0 := anInVals[ 0];
    dl1 := anInVals[ 1];
    dl2 := anInVals[ 2];
    dl3 := anInVals[ 3];

    anOutVals[ 0] := aMultiplier * (             + 0.5 * dl1 -       dl2 - 0.5 * dl3);    //  0  .5  -1 -.5
    anOutVals[ 1] := aMultiplier * ( + 0.5 * dl0             - 0.5 * dl2 +       dl3);    // .5   0 -.5   1
    anOutVals[ 2] := aMultiplier * (       - dl0 + 0.5 * dl1             + 0.5 * dl3);    // -1  .5   0  .5
    anOutVals[ 3] := aMultiplier * ( + 0.5 * dl0       + dl1 + 0.5 * dl2            );    // .5   1  .5   0
  end;


  function NearestPrime( aValue: Cardinal; aRelativeError: Double): Integer;
  var
    Bound : Integer;
    k     : Cardinal;
  begin
    Result := -1;

    if IsPrime( aValue)
    then Result := aValue
    else begin
      Bound := Round( aValue * aRelativeError);

      for k := 1 to Bound
      do begin
        if IsPrime( aValue + k)
        then begin
          Result := aValue + k;
          Break;
        end
        else if IsPrime( aValue - k)
        then begin
          Result := aValue - k;
          Break;
        end;
      end;
    end;
  end;


  function  Normalize( const aValue: TSignal): TSignal; inline;
  // @BH implemented my own denormals avoider
  // This actually checks if aValue is zero, a denormal, inf or Nan, or it's absulote value being < 1e-20
  // and it treats all those cases as if they were zero. Changed to also filter NaN's and INF's now.
  const
    NORMAL_LIMIT = 1e-20;
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
//  if (( PWordArray( @ aValue)^[ 3] and $7ff0) = 0) or ( Abs( aValue) < NORMAL_LIMIT)
    if ((( PWordArray( @ aValue)^[ 3] and $7ff0) - $10) >= $7fe0) or ( Abs( aValue) < NORMAL_LIMIT)
    then Result := 0
    else Result := aValue;
  end;


{ ========
  TFixedDelay = class
  private
    FSize : Integer;
    FIdx  : Integer;
    FBuf  : array of Double;
  protected
    function    Read( anIndex: Integer): Double;
    procedure   Write( aValue: Float);
  public
}

    function    TFixedDelay.Read( anIndex: Integer): Double;
    begin
      Result := FBuf[ ( FIdx - anIndex + 1024 * FSize) mod FSize];  // @BH added * 1024 as the modulo for negative nummbers is .. negative :s
    end;


    procedure   TFixedDelay.Write( aValue: Double);
    begin
      FBuf[ FIdx] := aValue;
      FIdx := ( FIdx + 1) mod FSize;
    end;


//  public

    constructor TFixedDelay.Create( aSize: Integer);
    begin
      inherited Create;
      SetLength( FBuf, aSize);
      FSize := aSize;
      Flush;
    end;


    procedure   TFixedDelay.Flush;
    var
      i : Integer;
    begin
      for i := Low( FBuf) to High( FBuf)
      do FBuf[ i] := 0.0;
    end;



{ ========
  TDiffuser = class
  private
    FSize   : Integer;
    FCoeff  : Double;
    FIdx    : Integer;
    FBuffer : array of Double;
  public
}

    constructor TDiffuser.Create( aSize: Integer; aCoeff: Double);
    begin
      inherited Create;
      SetLength( FBuffer, aSize);
      FSize  := aSize;
      FCoeff := aCoeff;
      FIdx   := 0;
      Flush;
    end;


    procedure   TDiffuser.Flush;
    var
      i : Integer;
    begin
      for i := Low( FBuffer) to High( FBuffer)
      do FBuffer[ i] := 0.0;
    end;


    function    TDiffuser.Tick( aValue: Double): Double;
    var
      W : Double;
    begin
      if FSize = 0
      then Result := Normalize( aValue)
      else begin
        FIdx   := FIdx mod FSize;
        W      := Normalize( aValue - FBuffer[ FIdx] * FCoeff);   // @BH : may have added a Normalize call here
        Result := Normalize( FBuffer[ FIdx] + W * FCoeff);        // @BH : added this in case the length changes
        FBuffer[ FIdx] := W;
        FIdx := ( FIdx + 1) mod FSize;
      end;
    end;


{ ========
  TDamper = class
    FDamping : Double;
    FDelay   : Double;
  public
}

    constructor TDamper.Create( aDamping: Double);
    begin
      inherited Create;
      FDamping := aDamping;
      Flush;
    end;


    procedure   TDamper.Flush;
    begin
      FDelay := 0.0;
    end;


    procedure   TDamper.SetDamping( aValue: Double);
    begin
      FDamping := aValue;
    end;


    function    TDamper.Tick( aValue: Double): Double;
    begin
      Result := Normalize( aValue + FDamping * ( FDelay - aValue));   // @BH : may have added a Normalize call here
      FDelay := Result;
    end;


{ ========
  TGVerb = class
  private
    FInputDamper  : TDamper;
    FTapDelay     : TFixedDelay;
    FFdnDels      : array of TFixedDelay;
    FLDifs        : array of TDiffuser;
    FRDifs        : array of TDiffuser;
    FRate         : Integer;
    FMaxRoomSize  : Double;
    FRoomSize     : Double;
    FRevTime      : Double;
    FDamping      : Double;
    FSpread       : Double;
    FInputBW      : Double;
    FEarlyLevel   : Double;
    FTailLevel    : Double;
    FDryLevel     : Double;
    FMaxDelay     : Double;
    FLargestDelay : Double;
    FFdnGains     : array of Double;
    FFdnLens      : array of Integer;
    FFdnDamps     : array of TDamper;
    FTaps         : array of Integer;
    FTapGains     : array of Double;
    FD            : array of Double;
    FU            : array of Double;
    FF            : array of Double;
    FAlpha        : Double;
  public
}

    constructor TGVerb.Create(
      aSampleRate      : Integer;
      aMaxRoomSize     : Double;
      aRoomSize        : Double;
      aRevTime         : Double;
      aDamping         : Double;
      aSpread          : Double;
      anInputBandWidth : Double;
      anEarlyLevel     : Double;
      aTailLevel       : Double;
      aDryLevel        : Double;
      aMode            : Integer
    );
    var
      i        : Integer;
      DifScale : Double;
      Spread1  : Double;
      Spread2  : Double;
      a        : Double;
      b        : Double;
      c        : Double;
      cc       : Double;
      d        : Double;
      dd       : Double;
      e        : Double;
      r        : Double;
    begin
      // @BH changed the order of stuff a bit .. so I'd better understand ..
      FRate        := aSampleRate;
      FMaxRoomSize := aMaxRoomSize;
      FDamping     := aDamping;
      FSpread      := aSpread;
      FInputBW     := anInputBandWidth;
      FEarlyLevel  := anEarlyLevel;
      FTailLevel   := aTailLevel;
      FDryLevel    := aDryLevel;
      FMode        := aMode;

      SetLength( FFdnDels , FDNORDER);
      SetLength( FFdnGains, FDNORDER);
      SetLength( FFdnLens , FDNORDER);
      SetLength( FFdnDamps, FDNORDER);
      SetLength( FD       , FDNORDER);
      SetLength( FU       , FDNORDER);
      SetLength( FF       , FDNORDER);
      SetLength( FLDifs   , FDNORDER);
      SetLength( FRDifs   , FDNORDER);
      SetLength( FTaps    , FDNORDER);
      SetLength( FTapGains, FDNORDER);

      FMaxDelay    := FRate * FMaxRoomSize / 340.0;
      FInputDamper := TDamper.Create( 1.0 - FInputBW);
      FTapDelay    := TFixedDelay.Create( 44000);

      for i := 0 to FDNORDER - 1
      do begin
        FFdnDels [ i] := TFixedDelay.Create( Round( FMaxDelay) + 1000);
        FFdnDamps[ i] := TDamper.Create( FDamping);;
      end;

      // NOTE : MUST set RevTime before RoomSize
      SetRevTime ( aRevTime );
      // NOTE : MUST set RoomSize before Spread, that is, spread stuff needs recalculations after RoomSize was changed
      SetRoomSize( aRoomSize);

      DifScale := FFdnLens[ 3] / ( 210 + 159 + 562 + 410); // adds up to 1341
      Spread1  := 1.0 * FSpread;
      Spread2  := 3.0 * FSpread;       // FSpread < 480 / 3 -> 160

      b  := 210;
      r  := 0.125541;
      a  := Spread1 * r;
      c  := 210 + 159 + a;
      cc := c - b;
      r  := 0.854046;
      a  := Spread2 * r;               // spread2 * r < 410 -> spread2 < 480
      d  := 210 + 159 + 562 + a;       // a < 410
      dd := d - c;
      e  := 1341 - d;                  // d < 1341

      FLDifs[ 0] := TDiffuser.Create( Round( DifScale * b ), 0.750);
      FLDifs[ 1] := TDiffuser.Create( Round( DifScale * cc), 0.750);
      FLDifs[ 2] := TDiffuser.Create( Round( DifScale * dd), 0.625);
      FLDifs[ 3] := TDiffuser.Create( Round( DifScale * e ), 0.625);

      b  := 210;
      r  := -0.568366;
      a  := Spread1 * r;
      c  := 210 + 159 + a;
      cc := c - b;
      r  := -0.126815;
      a  := Spread2 * r;
      d  := 210 + 159 + 562 + a;
      dd := d - c;
      e := 1341 - d;

      FRDifs[ 0] := TDiffuser.Create( Round( DifScale * b ), 0.750);
      FRDifs[ 1] := TDiffuser.Create( Round( DifScale * cc), 0.750);
      FRDifs[ 2] := TDiffuser.Create( Round( DifScale * dd), 0.625);
      FRDifs[ 3] := TDiffuser.Create( Round( DifScale * e ), 0.625);
    end;


    destructor  TGVerb.Destroy; // override;
    // @BH : changed destruction order
    var
      i : Integer;
    begin
      FreeAndNil( FInputDamper);
      FreeAndNil( FTapDelay);

      for i := 0 to FDNORDER - 1
      do begin
        FreeAndNil( FFdnDels [ i]);
        FreeAndNil( FFdnDamps[ i]);
        FreeAndNil( FLDifs   [ i]);
        FreeAndNil( FRDifs   [ i]);
      end;

      inherited;
    end;


    procedure   TGVerb.Flush;
    // @BH : changed flush order
    var
      i : Integer;
    begin
      FInputDamper.Flush;

      for i := 0 to FDNORDER - 1
      do begin
        FFdnDels [ i].Flush;
        FFdnDamps[ i].Flush;
        FLDifs   [ i].Flush;
        FRDifs   [ i].Flush;

        FD[ i] := 0.0;
        FU[ i] := 0.0;
        FF[ i] := 0.0;
      end;

      FTapDelay.Flush;
    end;


    procedure   TGVerb.Tick( anInvalueL, anInValueR: Double; out aLeft, aRight: Double);
    var
      i         : Integer;
      Z         : Double;
      LSum      : Double;
      RSum      : Double;
      Sum       : Double;
      Sign      : Double;
      anInValue : Double;
   // aLevel    : Double;
    begin
      if IsNan( anInValueL) or (( Abs( anInValueL)) > 100.0)      // @BH changed input limit to 100, it was like 10e5 or so
      then anInValueL := 0.0;

      if IsNan( anInValueR) or (( Abs( anInValueR)) > 100.0)
      then anInValueR := 0.0;

      anInValue := anInvalueL + anInValueR;

      Z := FInputDamper.Tick( anInvalue);
      Z := FLDifs[ 0].Tick( Z);

      for i := 0 to FDNORDER - 1
      do FU[ i] := Normalize( FTapGains[ i] * FTapDelay.Read( FTaps[ i]));

      FTapDelay.Write( Z);

      for i := 0 to FDNORDER - 1
      do FD[ i] := FFdnDamps[ i].Tick( Normalize( FFDnGains[ i] * FFdnDels[ i].Read( FFdnLens[ i])));   // @BH : may have added a Normalize call here

      Sum  := 0.0;
      Sign := 1.0;

      for i := 0 to FDNORDER - 1
      do begin
        Sum  := Sum + Sign * ( FTailLevel * FD[ i] + FEarlyLevel * FU[ i]);
        Sign := - Sign;
      end;

      Sum  := Sum + anInValue * FEarlyLevel;
      LSum := Sum;
      RSum := Sum;

   // aLevel := Sqrt( FD[ 0] * FD[ 0] + FD[ 1] * FD[ 1] + FD[ 2] * FD[ 2] + FD[ 3] * FD[ 3]);  // @BH - not used

      if FMode and 2 = 0
      then CalculateFdnMatrix1( FD, FF, FDN_MULTIPLIER1)
      else CalculateFdnMatrix2( FD, FF, FDN_MULTIPLIER2);

      for i := 0 to FDNORDER - 1
      do FFdnDels[ i].Write( FU[ i] + FF[ i]);

      LSum := FLDifs[ 1].Tick( LSum);
      LSum := FLDifs[ 2].Tick( LSum);
      LSum := FLDifs[ 3].Tick( LSum);

      RSum := FRDifs[ 1].Tick( RSum);
      RSum := FRDifs[ 2].Tick( RSum);
      RSum := FRDifs[ 3].Tick( RSum);

      aLeft  := LSum  + FDryLevel * anInvalueL;
      aRight := RSum  + FDryLevel * anInvalueR;

      {$IFDEF DEBUG}
      if ( Abs( aLeft) > 1.5) or ( Abs( aRight) > 1.5)   // @BH : debug code
      then nop;
      {$ENDIF}
    end;


    function    TGVerb.GetMaxRoomSize: Double;
    begin
      Result := FMaxRoomSize;
    end;


    procedure   TGVerb.SetRoomSize( const aValue: Double);
    // NOTE : RevTime MUST be set before RoomSize
    var
      i  : Integer;
      gb : Double;
    begin
      if aValue <> FRoomSize
      then begin

        if IsNan( aValue) or ( aValue < 0.01)                   // @BH changed limit from 1 to 0.01
        then FRoomSize := 0.01
        else FRoomSize := aValue;

        FLargestDelay := FRate * FRoomSize / 340.0;
        gb            := 0.0;

        for i := 0 to FDNORDER - 1
        do begin
          if i = 0 then gb := 1.000000 * FLargestDelay;
          if i = 1 then gb := 0.816490 * FLargestDelay;
          if i = 2 then gb := 0.707100 * FLargestDelay;
          if i = 3 then gb := 0.632450 * FLargestDelay;

          if FMode and 1 = 0
          then FFdnLens [ i] := Round( gb)
          else FFdnLens [ i] := NearestPrime( Trunc( gb), 0.5);
        end;

        for i := 0 to FDNORDER - 1
        do FFdnGains[ i] := Normalize( - Power( FAlpha, FFdnLens[ i]));   // @BH : may have added a Normalize call here

        FTaps[ 0] := Round( 5 + 0.410 * FLargestDelay);
        FTaps[ 1] := Round( 5 + 0.300 * FLargestDelay);
        FTaps[ 2] := Round( 5 + 0.155 * FLargestDelay);
        FTaps[ 3] := Round( 5 + 0.000 * FLargestDelay);

        for i := 0 to FDNORDER - 1
        do FTapGains[ i] := Power( FAlpha, FTaps[ i]);

        Nop;
      end;
    end;


    procedure   TGVerb.SetRevTime( const aValue: Double);
    // NOTE : RoomSize MUST be set after RevTime
    var
      ga : Double;
      gt : Double;
      n  : Double;
      i  : Cardinal;
    begin
      if aValue <> FRevTime
      then begin
        if IsNan( aValue) or ( aValue < 1e-4)     // @BH : guard on reverb time changed
        then FRevTime := 1e-4
        else FRevTime := aValue;

        ga     := 60.0;
        gt     := FRevTime;
        ga     := Power( 10.0, -ga / 20.0);
        n      := FRate * gt;
        FAlpha := Power( ga, 1.0 / n);

        for i := 0 to FDNORDER - 1
        do FFdnGains[ i] := Normalize( - Power( FAlpha, FFdnLens[ i]));
      end;
    end;


    procedure   TGVerb.SetDamping( const aValue: Double);
    var
      i : Integer;
    begin
      if aValue <> FDamping
      then begin
        FDamping := aValue;

        for i := 0 to FDNORDER - 1
        do FFdnDamps[ i].SetDamping( FDamping);
      end;
    end;


    procedure   TGVerb.SetInputBandWidth( const aValue: Double);
    begin
      if aValue <> FInputBW
      then begin
        FInputBW := aValue;
        FInputDamper.SetDamping( 1.0 - FInputBW);
      end;
    end;


    procedure   TGVerb.SetEarlyLevel( const aValue: Double);
    begin
      FEarlyLevel := aValue;
    end;


    procedure   TGVerb.SetTailLevel( const aValue: Double);
    begin
      FTailLevel := aValue;
    end;


    procedure   TGVerb.SetDryLevel( const aValue: Double);
    begin
      FDryLevel := aValue;
    end;


end.

