unit midi_defs;

interface

uses

  System.SysUtils,

  KnobsUtils;

const

  MIDI_CH_OMNI = 16;             // MIID channels are zero relative, 0 -> ch1, 15 -> ch 16 and 16 -> any channel / omni


  // Midi command definitions

  CNoteOff              = $80;   // Length = 3;
  CNoteOn               = $90;   // Length = 3;
  CKeyPressure          = $a0;   // Length = 3;
  CControlChange        = $b0;   // Length = 3;
  CProgramChange        = $c0;   // Length = 2;
  CChannelPressure      = $d0;   // Length = 2;
  CPitchBend            = $e0;   // Length = 2;
  CSysEx                = $f0;   // Length = n;


  // When aCmd And $f0 = CSysEx low nibble of 1st byte is :

  SSysEx                = $00;   // Length = 1, followed by arbitrary amt of data, finalized by SEox;
  SSongPosition         = $02;   // Length = 3;
  SSongSelect           = $03;   // Length = 2;
  STuneRequest          = $06;   // Length = 1;
  SEox                  = $07;   // Length = 1;
  STimingClock          = $08;   // Length = 1;
  SStart                = $0a;   // Length = 1;
  SContinue             = $0b;   // Length = 1;
  SStop                 = $0c;   // Length = 1;
  SActiveSense          = $0e;   // Length = 1;
  SReset                = $0f;   // Length = 1;


  // CCs

  ccBankSelect          =   0;
  ccModulation          =   1;
  ccBreathControl       =   2;
  ccUndef3              =   3;
  ccFoutController      =   4;
  ccPortamentoTime      =   5;
  ccDataEntryMSB        =   6;   // Length = 3 - Parameter value                 High seven bits
  ccMinVolume           =   7;
  ccBlance              =   8;
  ccUndef9              =   9;
  ccPan                 =  10;
  ccExpression          =  11;
  ccFxController1       =  12;
  ccFxController2       =  13;
  ccUndefined14         =  14;
  ccUndefined15         =  15;
  ccGpControl1          =  16;
  ccGpControl2          =  17;
  ccGpControl3          =  18;
  ccGpControl4          =  19;
  ccUndefined20         =  20;
  ccUndefined21         =  21;
  ccUndefined22         =  22;
  ccUndefined23         =  23;
  ccUndefined24         =  24;
  ccUndefined25         =  25;
  ccUndefined26         =  26;
  ccUndefined27         =  27;
  ccUndefined28         =  28;
  ccUndefined29         =  29;
  ccUndefined30         =  30;
  ccUndefined31         =  31;
  ccBankSelectLSB       =  32;
  ccModulationLSB       =  33;
  ccBreathControlLSB    =  34;
  ccUndef3LSB           =  35;
  ccFoutControllerLSB   =  36;
  ccPortamentoTimeLSB   =  37;
  ccDataEntryLSB        =  38;   // Length = 3 - Parameter value                 Low  seven bits
  ccMinVolumeLSB        =  39;
  ccBlanceLSB           =  40;
  ccUndef9LSB           =  41;
  ccPanLSB              =  42;
  ccExpressionLSB       =  43;
  ccFxController1LSB    =  44;
  ccFxController2LSB    =  45;
  ccUndefined14LSB      =  46;
  ccUndefined15LSB      =  47;
  ccGpControl1LSB       =  48;
  ccGpControl2LSB       =  49;
  ccGpControl3LSB       =  50;
  ccGpControl4LSB       =  51;
  ccUndefined20LSB      =  52;
  ccUndefined21LSB      =  53;
  ccUndefined22LSB      =  54;
  ccUndefined23LSB      =  55;
  ccUndefined24LSB      =  56;
  ccUndefined25LSB      =  57;
  ccUndefined26LSB      =  58;
  ccUndefined27LSB      =  59;
  ccUndefined28LSB      =  60;
  ccUndefined29LSB      =  61;
  ccUndefined30LSB      =  62;
  ccUndefined31LSB      =  63;
  ccDamper              =  64;
  ccPortamento          =  65;
  ccSostemuto           =  66;
  ccSoftPedal           =  67;
  ccLegatoFootSwitch    =  68;
  ccHold2               =  69;
  ccSoundController1    =  70;
  ccSoundController2    =  71;
  ccSoundController3    =  72;
  ccSoundController4    =  73;
  ccSoundController5    =  74;
  ccSoundController6    =  75;
  ccSoundController7    =  76;
  ccSoundController8    =  77;
  ccSoundController9    =  78;
  ccSoundController10   =  79;
  ccGeneralPurpose5     =  80;
  ccGeneralPurpose6     =  81;
  ccGeneralPurpose7     =  82;
  ccGeneralPurpose8     =  83;
  ccPortamentoControl   =  84;
  ccUndefined85         =  85;
  ccUndefined86         =  86;
  ccUndefined87         =  87;
  ccUndefined88         =  88;
  ccUndefined89         =  89;
  ccUndefined90         =  90;
  ccFx1Depth            =  91;
  ccFx2Depth            =  92;
  ccFx3Depth            =  93;
  ccFx4Depth            =  94;
  ccFx5Depth            =  95;
  ccDataIncrement       =  96;
  ccDataDecrement       =  97;
  ccNRPNL               =  98;   // Length = 3 - Non Registered Parameter Number Low  seven bits
  ccNRPNH               =  99;   // Length = 3 - Non Registered Parameter Number High seven bits
  ccRPNL                = 100;   // Length = 3 - Registered     Parameter Number Low  seven bits
  ccRPNH                = 101;   // Length = 3 - Registered     Parameter Number High seven bits
  ccUndefined102        = 102;
  ccUndefined103        = 103;
  ccUndefined104        = 104;
  ccUndefined105        = 105;
  ccUndefined106        = 106;
  ccUndefined107        = 107;
  ccUndefined108        = 108;
  ccUndefined109        = 109;
  ccUndefined110        = 110;
  ccUndefined111        = 111;
  ccUndefined112        = 112;
  ccUndefined113        = 113;
  ccUndefined114        = 114;
  ccUndefined115        = 115;
  ccUndefined116        = 116;
  ccUndefined117        = 117;
  ccUndefined118        = 118;
  ccUndefined119        = 119;


  // Pseudo CC's (Channel mode messages)

  CCAllSoundOff         = 120;   // Pseudo CC - value should be 0
  CCResetAllControllers = 121;   // Pseudo CC - value should be 0
  CCLoccalControl       = 122;   // Pseudo CC - value 0 = off, value 127 = on
  CCAllNotesOff         = 123;   // Pseudo CC - value should be 0
  CCOmniModeOff         = 124;   // Pseudo CC - value should be 0
  CCOmniModeOn          = 125;   // Pseudo CC - value should be 0
  CCMonoModeOn          = 126;   // Pseudo CC - value 0 = omni on, other values M = omni off with M channels used
  CCPolyModeOn          = 127;   // Pseudo CC - value should be 0


  // RPN stuffs

  rpnPitchBendSens      = $0000;
  rpnChanFineTune       = $0001;
  rpnChanCoarseTune     = $0002;
  rpnTuningProgChance   = $0003;
  rpnTuningBankSelect   = $0004;
  rpnModDepthRange      = $0005;
  rpnAzimuthAngle       = $3d00;
  rpnElevationAngle     = $3d01;
  rpnGain               = $3d02;
  rpnDistanceRatio      = $3d03;
  rpnMaxDistance        = $3d04;
  rpnMaxDistanceGain    = $3d05;
  rpnRefDistanceGain    = $3d06;
  rpnPanSpreadAngle     = $3d07;
  rpnRollRange          = $3d08;


  // Meta types - used in MIDI files

  CMSequenceNumber      = $00;
  CMTextEvent           = $01;
  CMCopyRightNotice     = $02;
  CMSequenceTrackName   = $03;
  CMInstrumentName      = $04;
  CMLyric               = $05;
  CMMarker              = $06;
  CMCuePoint            = $07;
  CMMidiChannelPrefix   = $20;
  CMEndOfTrack          = $2f;
  CMSetTempo            = $51;
  CMSMTPOffset          = $54;
  CMTimeSignature       = $58;
  CMKeySignature        = $59;
  CMSequencerSpecific   = $7F;


type

  TInts = array of Integer;


  TMidiMessage = record
    Command : Byte;
    Channel : Byte;
    Data    : TInts;
  end;


  TShortMidiMessage = record
    DataLen : Byte;
    Command : Byte;
    Channel : Byte;
    Data1   : Byte;
    Data2   : Byte;
  end;


  TMidiRxState = (
    mrxNormal,                                 // No command received yet, or runnning status cancelled
    mrxRunningStatus                           // Valid running status
  );


  TRpnState = (
    mrsRPN ,                                   // Working on RPN
    mrsNRPN                                    // Working on NRPN
  );


  TRpnMode = (
    mrm7 ,                                     // Expecting  7 bit data for RPN and NRPN
    mrm14                                      // Expecting 14 bit data for RPN and NRPN
  );


  TOnMidiMessage      = procedure( const aSender: TObject; const aMsg: TMidiMessage          ) of object;
  TOnShortMidiMessage = procedure( const aSender: TObject; const aMsg: TShortMidiMessage     ) of object;
  TOnRpnMessage       = procedure( const aSender: TObject; aCh, aController, aValue: Integer ) of object;
  TOnNrpnMessage      = procedure( const aSender: TObject; aCh, aController, aValue: Integer ) of object;
  TOnSendMidiByte     = procedure( const aSender: TObject; aByte: Byte)                        of object;


  TMidiTransmitter = class
  private
    FRunningStatus  : Byte;
    FOnSendMidiByte : TOnSendMidiByte;
  private
    procedure   SendByte( aByte: Byte);
  public
    procedure   Accept( aByte      : Byte             ); overload;
    procedure   Accept( const aMsg : TBytes           ); overload;
    procedure   Accept( const aMsg : TShortMidiMessage); overload;
  public
    property    OnSendMidiByte : TOnSendMidiByte read FOnSendMidiByte write FOnSendMidiByte;
  end;


  TMidiReceiver = class
  private
    FOnMessage     : TOnMidiMessage;
    FOnRpnMessage  : TOnRpnMessage;
    FOnNrpnMessage : TOnNrpnMessage;
    FRxState       : TMidiRxState;
    FRpnMode       : TRpnMode;
    FRpnState      : TRpnState;
    FlastRPN       : Integer;
    FlastNRPN      : Integer;
    FlastRPNVal    : Integer;
    FlastNRPNVal   : Integer;
    FDataCount     : Integer;
    FRunningStatus : Byte;
    FChannel       : Byte;
    FData          : Byte;
    FSysEx         : TBytes;
  private
    procedure   SetRpnMode( aValue: TRpnMode);
  private
    procedure   AcceptCommand( aByte: Byte);
    procedure   AcceptData   ( aByte: Byte);
    procedure   AcceptSysex  ( aByte: Byte);
    procedure   ProcessRpn            ( aValue, aController, aChannel: Byte);
    procedure   ProcessNoteOff        ( aVelocity, aNote, aChannel: Byte);
    procedure   ProcessNoteOn         ( aVelocity, aNote, aChannel: Byte);
    procedure   ProcessKeyPressure    ( aPressure, aNote, aChannel: Byte);
    procedure   ProcessControlChange  ( aValue, aController, aChannel: Byte);
    procedure   ProcessProgramChange  ( aProgram, aChannel: Byte);
    procedure   ProcessChannelPressure( aValue, aChannel: Byte);
    procedure   ProcessPitchBend      ( aData: Word; aChannel: Byte);
    procedure   ProcessStartSysEx;
    procedure   ProcessEndSysEx;
    procedure   ProcessSongPosition   ( aData: Word);
    procedure   ProcessSongSelect     ( aSong: Byte);
    procedure   ProcessTuneRequest;
    procedure   ProcessTimingClock;
    procedure   ProcessStart;
    procedure   ProcessContinue;
    procedure   ProcessStop;
    procedure   ProcessActiveSense;
    procedure   ProcessReset;
    procedure   ProcessSysExData      ( aData: Byte);
    procedure   ProcessMessage( const aMsg: TMidiMessage);
  public
    constructor Create;
    procedure   Reset;
    procedure   AcceptByte( aByte: Byte);
    procedure   AcceptBytes( aBytes: TBytes);
  public
    property    OnMessage     : TOnMidiMessage read FOnMessage     write FOnMessage;
    property    OnRpnMessage  : TOnRpnMessage  read FOnRpnMessage  write FOnRpnMessage;
    property    OnNrpnMessage : TOnNrpnMessage read FOnNrpnMessage write FOnNrpnMessage;
    property    RpnMode       : TRpnMode       read FRpnMode       write SetRpnMode;
  end;


  procedure MidiMessageToBytes     ( const aMsg: TMidiMessage     ; var aData: TBytes);
  procedure ShortMidiMessageToBytes( const aMsg: TShortMidiMessage; var aData: TBytes);
  procedure RpnToBytes             ( const aCh: Byte; aController, aData: Integer; var aResult: TBytes);
  procedure NrpnToBytes            ( const aCh: Byte; aController, aData: Integer; var aResult: TBytes);
  function  MidiMessageToLog       ( const aMsg: TMidiMessage     ): string;
  function  ShortMidiMessageToLog  ( const aMsg: TShortMidiMessage): string;
  function  RpnToLog               ( aCh: Byte; aController, aValue: Integer) : string;
  function  NrpnToLog              ( aCh: Byte; aController, aValue: Integer) : string;



implementation


const

  MaskCmd   =   $80;               // Command has bit 7 set
  MaskCh    =   $0f;               // Channel is in low nibble of command
  MaskEvent =   $f0;               // Event type is in high nibble of command
  MaskOffLo = $3f80;               // To keep the high septet of 14 bit data
  MaskOffHi = $007f;               // To keep the low  septet of 14 bit data
  RPNNull   = $3fff;               // The NULL RPN
  MAX14Bits = $3fff;               // Max value for 14 bit data
  MaxSysex  = 1024;                // Maximum length of Sysex messages .. well .. they can be longer but they will be
                                   // broken up into chuncks of MaxSysesx bytes before passing to the callback.


  function  MakeMidiNoteOff( aVelocity, aNote, aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CNoteOff;
    Result.Channel := aChannel;
    SetLength( Result.Data, 2);
    Result.Data[ 0] := aNote;
    Result.Data[ 1] := aVelocity;
  end;


  function  MakeMidiNoteOn( aVelocity, aNote, aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CNoteOn;
    Result.Channel := aChannel;
    SetLength( Result.Data, 2);
    Result.Data[ 0] := aNote;
    Result.Data[ 1] := aVelocity;
  end;


  function  MakeMidiKeyPressure( aPressure, aNote, aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CKeyPressure;
    Result.Channel := aChannel;
    SetLength( Result.Data, 2);
    Result.Data[ 0] := aNote;
    Result.Data[ 1] := aPressure;
  end;


  function  MakeMidiControlChange( aValue, aController, aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CControlChange;
    Result.Channel := aChannel;
    SetLength( Result.Data, 2);
    Result.Data[ 0] := aController;
    Result.Data[ 1] := aValue;
  end;


  function  MakeMidiProgramChange( aProgram, aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CProgramChange;
    Result.Channel := aChannel;
    SetLength( Result.Data, 1);
    Result.Data[ 0] := aProgram;
  end;


  function  MakeMidiChannelPressure( aValue, aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CChannelPressure;
    Result.Channel := aChannel;
    SetLength( Result.Data, 1);
    Result.Data[ 0] := aValue;
  end;


  function  MakeMidiPitchBend( aData: Word; aChannel: Byte): TMidiMessage;
  begin
    Result.Command := CPitchBend;
    Result.Channel := aChannel;
    SetLength( Result.Data, 1);
    Result.Data[ 0] := aData;
  end;


  function  MakeMidiSongPosition( aData: Word): TMidiMessage;
  begin
    Result.Command := SSongPosition;
    Result.Channel := 0;
    SetLength( Result.Data, 1);
    Result.Data[ 0] := aData;
  end;


  function  MakeMidiSongSelect( aSong: Byte): TMidiMessage;
  begin
    Result.Command := SSongSelect;
    Result.Channel := 0;
    SetLength( Result.Data, 1);
    Result.Data[ 0] := aSong;
  end;


  function  MakeMidiTuneRequest: TMidiMessage;
  begin
    Result.Command := STuneRequest;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiTimingClock: TMidiMessage;
  begin
    Result.Command := STimingClock;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiStart: TMidiMessage;
  begin
    Result.Command := SStart;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiContinue: TMidiMessage;
  begin
    Result.Command := SContinue;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiStop: TMidiMessage;
  begin
    Result.Command := SStop;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiActiveSense: TMidiMessage;
  begin
    Result.Command := SActiveSense;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiReset: TMidiMessage;
  begin
    Result.Command := SReset;
    Result.Channel := 0;
    SetLength( Result.Data, 0);
  end;


  function  MakeMidiSysExData( const aData: TBytes): TMidiMessage;
  var
    i : Integer;
  begin
    Result.Command := SSysEx;
    Result.Channel := 0;
    SetLength( Result.Data, Length( aData));

    for i := 0 to Length( Result.Data) - 1
    do Result.Data[ i] := aData[ i];
  end;


  procedure MidiMessageToBytes( const aMsg: TMidiMessage; var aData: TBytes);
  begin
    SetLength( aData, 0);

    case aMsg.Command of

      CNoteOff,          // = $80; // Length = 3;
      CNoteOn,           // = $90; // Length = 3;
      CKeyPressure,      // = $a0; // Length = 3;
      CControlChange :   // = $b0; // Length = 3;
        begin
          if Length( aMsg.Data) = 2
          then begin
            SetLength( aData, 3);
            aData[ 0] := ( aMsg.Command and MaskEvent) or ( aMsg.Channel and MaskCh);
            aData[ 1] := aMsg.Data[ 0];
            aData[ 2] := aMsg.Data[ 1];
          end;
        end;

      CProgramChange,    // = $c0; // Length = 2;
      CChannelPressure : // = $d0; // Length = 2;
        begin
          if Length( aMsg.Data) = 2
          then begin
            SetLength( aData, 2);
            aData[ 0] := ( aMsg.Command and MaskEvent) or ( aMsg.Channel and MaskCh);
            aData[ 1] := aMsg.Data[ 0];
          end;
        end;

      CPitchBend :       // = $e0; // Length = 3;
        begin
          if Length( aMsg.Data) = 2
          then begin
            SetLength( aData, 3);
            aData[ 0] := ( aMsg.Command  and $00f0) or ( aMsg.Channel and $000f);
            aData[ 1] :=   aMsg.Data[ 0] and MaskOffHi;
            aData[ 2] := ( aMsg.Data[ 0] and MaskOffLo) shr 7;
          end;
        end;

//      CSysEx :          // = $f0; // Length = n;
//        begin
//        end;

    end;
  end;


  procedure ShortMidiMessageToBytes( const aMsg: TShortMidiMessage; var aData: TBytes);
  begin
    SetLength( aData, 0);

    case aMsg.Command of

      CNoteOff,          // = $80; // Length = 3;
      CNoteOn,           // = $90; // Length = 3;
      CKeyPressure,      // = $a0; // Length = 3;
      CControlChange :   // = $b0; // Length = 3;
        begin
          if aMsg.DataLen = 2
          then begin
            SetLength( aData, 3);
            aData[ 0] := ( aMsg.Command and MaskEvent) or ( aMsg.Channel and MaskCh);
            aData[ 1] := aMsg.Data1;
            aData[ 2] := aMsg.Data2;
          end;
        end;

      CProgramChange,    // = $c0; // Length = 2;
      CChannelPressure : // = $d0; // Length = 2;
        begin
          if aMsg.DataLen = 1
          then begin
            SetLength( aData, 2);
            aData[ 0] := ( aMsg.Command and MaskEvent) or ( aMsg.Channel and MaskCh);
            aData[ 1] := aMsg.Data1;
          end;
        end;

      CPitchBend :       // = $e0; // Length = 3;
        begin
          if aMsg.DataLen = 2
          then begin
            SetLength( aData, 3);
            aData[ 0] := ( aMsg.Command  and $00f0) or ( aMsg.Channel and $000f);
            aData[ 1] :=   aMsg.Data1 and MaskOffHi;
            aData[ 2] := ( aMsg.Data2 and MaskOffLo) shr 7;
          end;
        end;

//      CSysEx :          // = $f0; // Length = n;
//        begin
//        end;

    end;
  end;


  procedure RpnToBytes( const aCh: Byte; aController, aData: Integer; var aResult: TBytes);
  begin
    if Length( aResult) = 13
    then begin
      aResult[  0] := CControlChange or ( aCh and MaskCh);
      aResult[  1] := ccRPNL;
      aResult[  2] := ( aController and MaskOffLo) shr 7;
      aResult[  3] := ccRPNH;
      aResult[  4] := aController and MaskOffHi;
      aResult[  5] := ccDataEntryMSB;
      aResult[  6] := ( aData and MaskOffLo) shr 7;
      aResult[  7] := ccDataEntryLSB;
      aResult[  8] := aData and MaskOffHi;
      aResult[  9] := ccRPNL;
      aResult[ 10] := $7f;
      aResult[ 11] := ccRPNH;
      aResult[ 12] := $7f;
    end;
  end;


  procedure NrpnToBytes( const aCh: Byte; aController, aData: Integer; var aResult: TBytes);
  begin
    if Length( aResult) = 13
    then begin
      aResult[  0] := CControlChange or ( aCh and MaskCh);
      aResult[  1] := ccNRPNL;
      aResult[  2] := ( aController and MaskOffLo) shr 7;
      aResult[  3] := ccNRPNH;
      aResult[  4] := aController and MaskOffHi;
      aResult[  5] := ccDataEntryMSB;
      aResult[  6] := ( aData and MaskOffLo) shr 7;
      aResult[  7] := ccDataEntryLSB;
      aResult[  8] := aData and MaskOffHi;
      aResult[  9] := ccNRPNL;
      aResult[ 10] := $7f;
      aResult[ 11] := ccNRPNH;
      aResult[ 12] := $7f;
    end;
  end;


  function MidiCmdToStr( aCmd: Byte): string;
  begin
    case aCmd of
      CNoteOff         : Result := 'note off';         // = $80; // Length = 3;
      CNoteOn          : Result := 'note on';          // = $90; // Length = 3;
      CKeyPressure     : Result := 'key pressure';     // = $a0; // Length = 3;
      CControlChange   : Result := 'control change';   // = $b0; // Length = 3;
      CProgramChange   : Result := 'program change';   // = $c0; // Length = 2;
      CChannelPressure : Result := 'channel pressure'; // = $d0; // Length = 2;
      CPitchBend       : Result := 'pitch bend';       // = $e0; // Length = 2;
      CSysEx           : Result := 'sysex';            // = $f0; // Length = n;
    end;
  end;


  function  MidiMessageToLog( const aMsg: TMidiMessage): string;
  var
    i : Integer;
  begin
    Result := Format( 'ch %d cmd %.2x [%s] ', [ aMsg.Channel + 1, aMsg.Command, MidiCmdToStr( aMsg.Command)], AppLocale);

    for i := 0 to Length( aMsg.Data) - 1
    do Result := Result + Format( '%.2x ', [ aMSg.Data[ i]], AppLocale);

    SetLength( Result, Length( Result) - 1);
  end;


  function  ShortMidiMessageToLog( const aMsg: TShortMidiMessage): string;
  begin
    Result := Format( 'ch %d cmd %.2x [%s]', [ aMsg.Channel + 1, aMsg.Command, MidiCmdToStr( aMsg.Command)], AppLocale);

    if aMsg.DataLen >= 1
    then Result := Result + Format( ' %.2x', [ aMSg.Data1], AppLocale);

    if aMsg.DataLen = 2
    then Result := Result + Format( ' %.2x', [ aMSg.Data2], AppLocale);
  end;


  function  RpnToLog( aCh: Byte; aController, aValue: Integer) : string;
  begin
    Result := Format( 'ch %d contr %d vlue %d', [ aCh + 1, aController, aValue], AppLocale);
  end;


  function  NrpnToLog( aCh: Byte; aController, aValue: Integer) : string;
  begin
    Result := Format( 'ch %d contr %d vlue %d', [ aCh + 1, aController, aValue], AppLocale);
  end;


{ ========
  TMidiTransmitter = class
  private
    FRunningStatus  : Byte;
    FOnSendMidiByte : TOnSendMidiByte;
  public
    property    OnSendMidiByte : TOnSendMidiByte read FOnSendMidiByte write FOnSendMidiByte;
  private
}

    procedure   TMidiTransmitter.SendByte( aByte: Byte);
    begin
      if Assigned( FOnSendMidiByte)
      then FOnSendMidiByte( Self, aByte);
    end;


//  public

    procedure   TMidiTransmitter.Accept( aByte : Byte); // overload;
    // todo : this does not correctly handle running status - commands will always cancel RS and will always be sent
    begin
      if aByte and MaskCmd = MaskCmd
      then begin                         // Command byte
        if aByte in [ $f0 .. $f7]        // Command possibly changes running status
        then begin
          FRunningStatus := 0;           // Cancel running status
          SendByte( aByte);              // And send the command
        end
        else if aByte in [ $f8 .. $ff]
        then begin                       // Leave running status as is
          SendByte( aByte);              // And send the comand
        end
        else begin
          if FRunningStatus <> aByte     // Otherwise if FRunningStatus = aByte do not transmit it <- this will never happen
          then begin
            FRunningStatus := aByte;
            SendByte( aByte);
          end;
        end;
      end
      else SendByte( aByte);             // Data byte
    end;


    procedure   TMidiTransmitter.Accept( const aMsg : TBytes); // overload;
    var
      i : Integer;
    begin
      for i := Low( aMsg) to High( aMsg)
      do Accept( aMsg[ i]);
    end;


    procedure   TMidiTransmitter.Accept( const aMsg : TShortMidiMessage); // overload;
    begin
      Accept(( aMsg.Channel and MaskCh) or ( aMsg.Command and  MaskEvent));

      if aMsg.DataLen > 0
      then Accept( aMsg.Data1);

      if aMsg.DataLen > 1
      then Accept( aMsg.Data2);
    end;


{ ========
  TMidiReceiver = class
  private
    FOnMessage     : TOnMidiMessage;
    FOnRpnMessage  : TOnRpnMessage;
    FOnNrpnMessage : TOnNrpnMessage;
    FRxState       : TMidiRxState;
    FRpnMode       : TRpnMode;
    FRpnState      : TRpnState;
    FlastRPN       : Integer;
    FlastNRPN      : Integer;
    FlastRPNVal    : Integer;
    FlastNRPNVal   : Integer;
    FDataCount     : Integer;
    FRunningStatus : Byte;
    FChannel       : Byte;
    FData          : Byte;
    FSysEx         : TBytes;
  public
    property    OnMessage     : TOnMidiMessage read FOnMessage     write FOnMessage;
    property    OnRpnMessage  : TOnRpnMessage  read FOnRpnMessage  write FOnRpnMessage;
    property    OnNrpnMessage : TOnNrpnMessage read FOnNrpnMessage write FOnNrpnMessage;
    property    RpnMode       : TRpnMode       read FRpnMode       write SetRpnMode;
  private
}

    procedure   TMidiReceiver.SetRpnMode( aValue: TRpnMode);
    begin
      if aValue <> FRpnMode
      then begin
        FRpnMode := aValue;
        Reset;
      end;
    end;


//  private

    procedure   TMidiReceiver.AcceptCommand( aByte: Byte);
    begin
      FDataCount := 0;

      case aByte and MaskEvent of

        CSysEx : AcceptSysex( aByte);

        else begin
          FRunningStatus := aByte;
          FRxState       := mrxRunningStatus;
          FChannel       := aByte and MaskCh;
        end;

      end;
    end;


    procedure   TMidiReceiver.AcceptData( aByte: Byte);
    begin
      case FRunningStatus and MaskEvent of

        CNoteOff :         // = $80; // Length = 3;
          begin
            if FDataCount = 1
            then begin
              ProcessNoteOff( aByte, FData, FChannel);
              FDataCount := 0;
            end
            else begin
              FData := aByte;
              Inc( FDataCount);
            end;
          end;

        CNoteOn :          // = $90; // Length = 3;
          begin
            if FDataCount = 1
            then begin
              ProcessNoteOn( aByte, FData, FChannel);
              FDataCount := 0;
            end
            else begin
              FData := aByte;
              Inc( FDataCount);
            end;
          end;

        CKeyPressure :     // = $a0; // Length = 3;
          begin
            if FDataCount = 1
            then begin
              ProcessKeyPressure( aByte, FData, FChannel);
              FDataCount := 0;
            end
            else begin
              FData := aByte;
              Inc( FDataCount);
            end;
          end;

        CControlChange :   // = $b0; // Length = 3;
          begin
            if FDataCount = 1
            then begin
              ProcessControlChange( aByte, FData, FChannel);
              FDataCount := 0;
            end
            else begin
              FData := aByte;
              Inc( FDataCount);
            end;
          end;

        CProgramChange :   // = $c0; // Length = 2;
          begin
            ProcessProgramChange( aByte,FChannel);
          end;

        CChannelPressure : // = $d0; // Length = 2;
          begin
            ProcessChannelPressure( aByte,FChannel);
          end;

        CPitchBend :       // = $e0; // Length = 2;
          begin
            if FDataCount = 1
            then begin
              ProcessPitchBend(( aByte and $7f) + 128 * ( FData and $7f), FChannel);
              FDataCount := 0;
            end
            else begin
              FData := aByte;
              Inc( FDataCount);
            end;
          end;

        CSysEx :           // = $f0; // Length = n;
          begin
            ProcessSysExData( aByte);
          end;

        SSongPosition :
          begin
            if FDataCount = 1
            then begin
              ProcessSongPosition(( aByte and $7f) + 128 * ( FData and $7f));
              FDataCount := 0;
            end
            else begin
              FData := aByte;
              Inc( FDataCount);
            end;
          end;

        SSongSelect :
          begin
            ProcessSongSelect( aByte);
          end;

        else Reset;

      end;
    end;


    procedure   TMidiReceiver.AcceptSysex( aByte: Byte);
    begin
      case aByte and $0f of

        SSysEx :          // = $00; // Length = 1, followed by arbitrary amt of data, finalized by SEox;
          begin
            FRunningStatus := SSysEx;
            FRxState       := mrxRunningStatus;
            ProcessStartSysEx;
          end;

        SSongPosition :   // = $02; // Length = 3;
          begin
            FRunningStatus := SSongPosition;
            FRxState       := mrxRunningStatus;
          end;

        SSongSelect :     // = $03; // Length = 2;
          begin
            FRunningStatus := SSongPosition;
            FRxState       := mrxRunningStatus;
          end;

        STuneRequest :    // = $06; // Length = 1;
          begin
            FRunningStatus := 0;
            FRxState       := mrxNormal;
            ProcessTuneRequest;
          end;

        SEox :            // = $07; // Length = 1;
          begin
            FRunningStatus := 0;
            FRxState       := mrxNormal;
            ProcessEndSysEx;
          end;

        STimingClock :    // = $08; // Length = 1;
          begin
            ProcessTimingClock;
          end;

        SStart :          // = $0a; // Length = 1;
          begin
            ProcessStart;
          end;

        SContinue :       // = $0b; // Length = 1;
          begin
            ProcessContinue;
          end;

        SStop :           // = $0c; // Length = 1;
          begin
            ProcessStop;
          end;

        SActiveSense :    // = $0e; // Length = 1;
          begin
            ProcessActiveSense;
          end;

        SReset :          // = $0f; // Length = 1;
          begin
            ProcessReset;
          end;

      end;
    end;


    procedure   TMidiReceiver.ProcessRpn( aValue, aController, aChannel: Byte);
    begin
      case aController of
        ccRPNL  : begin FlastRPN  := ( FLastRPN  and MaskOffLo) or   aValue;        FRpnState := mrsRPN;  end;
        ccRPNH  : begin FlastRPN  := ( FlastRPN  and MaskOffHi) or ( aValue shl 7); FRpnState := mrsRPN;  end;
        ccNRPNL : begin FlastNRPN := ( FLastNRPN and MaskOffLo) or   aValue;        FRpnState := mrsNRPN; end;
        ccNRPNH : begin FlastRPN  := ( FlastNRPN and MaskOffHi) or ( aValue shl 7); FRpnState := mrsNRPN; end;

        ccDataEntryMSB :

          begin
            case FRpnState of
              mrsRPN :

                begin
                  FlastRPNVal := aValue shl 7;

                  if Assigned( FOnRpnMessage) and ( FlastRPN < RPNNull) and ( RpnMode = mrm7)
                  then FOnRpnMessage( Self, FlastRPN, FlastRPNVal, aChannel);
                end;

              mrsNRPN :

                begin
                  FlastNRPN := aValue shl 7;

                  if Assigned( FOnNrpnMessage) and ( FlastNRPN < RPNNull) and ( RpnMode = mrm7)
                  then FOnNrpnMessage( Self, FlastNRPN, FlastNRPNVal, aChannel);
                end;

            end;
          end;

        ccDataEntryLSB :

          begin
            case FRpnState of

              mrsRPN :

                begin
                  FlastRPNVal := ( FlastRPNVal and MaskOffLo) + aValue;

                  if Assigned( FOnRpnMessage) and ( FlastRPN < RPNNull) and ( RpnMode = mrm14)
                  then FOnRpnMessage( Self, FlastRPN, FlastRPNVal, aChannel);
                end;

              mrsNRPN :

                begin
                  FlastNRPNVal := ( FlastNRPNVal and MaskOffLo) + aValue;

                  if Assigned( FOnNrpnMessage) and ( FlastNRPN < RPNNull) and ( RpnMode = mrm14)
                  then FOnNrpnMessage( Self, FlastNRPN, FlastNRPNVal, aChannel);
                end;

            end;
          end;

        ccDataIncrement :

          begin
            case FRpnState of

              mrsRPN :

                begin
                  if RpnMode = mrm7
                  then FlastRPNVal := Clip( FlastRPNVal + $80 * aValue, 0, MAX14Bits)
                  else FlastRPNVal := Clip( FlastRPNVal +       aValue, 0, MAX14Bits);

                  if Assigned( FOnRpnMessage) and ( FlastRPN < RPNNull)
                  then FOnRpnMessage( Self, FlastRPN, FlastRPNVal, aChannel);
                end;

              mrsNRPN :

                begin
                  if RpnMode = mrm7
                  then FlastRPNVal := Clip( FlastNRPNVal + $80 * aValue, 0, MAX14Bits)
                  else FlastRPNVal := Clip( FlastNRPNVal +       aValue, 0, MAX14Bits);

                  if Assigned( FOnNrpnMessage) and ( FlastNRPN < RPNNull)
                  then FOnNrpnMessage( Self, FlastNRPN, FlastNRPNVal, aChannel);
                end;

            end;
          end;

        ccDataDecrement :

          begin
            case FRpnState of

              mrsRPN :

                begin
                  FlastRPNVal := Clip( FlastRPNVal - aValue, 0, MAX14Bits);

                  if Assigned( FOnRpnMessage) and ( FlastRPN < RPNNull)
                  then FOnRpnMessage( Self, FlastRPN, FlastRPNVal, aChannel);
                end;

              mrsNRPN :

                begin
                  FlastRPNVal := Clip( FlastNRPNVal - aValue, 0, MAX14Bits);

                  if Assigned( FOnNrpnMessage) and ( FlastNRPN < RPNNull)
                  then FOnNrpnMessage( Self, FlastNRPN, FlastNRPNVal, aChannel);
                end;

            end;
          end;

      end;
    end;


    procedure   TMidiReceiver.ProcessNoteOff( aVelocity, aNote, aChannel: Byte);
    begin
      ProcessMessage( MakeMidiNoteOff( aVelocity, aNote, aChannel));
    end;


    procedure   TMidiReceiver.ProcessNoteOn( aVelocity, aNote, aChannel: Byte);
    begin
      if aVelocity = 0
      then ProcessMessage( MakeMidiNoteOff( 64       , aNote, aChannel))
      else ProcessMessage( MakeMidiNoteOn ( aVelocity, aNote, aChannel));
    end;


    procedure   TMidiReceiver.ProcessKeyPressure( aPressure, aNote, aChannel: Byte);
    begin
      ProcessMessage( MakeMidiKeyPressure( aPressure, aNote, aChannel));
    end;


    procedure   TMidiReceiver.ProcessControlChange( aValue, aController, aChannel: Byte);
    begin
      if aController in [
        ccRPNL,
        ccRPNH,
        ccNRPNL,
        ccNRPNH,
        ccDataEntryMSB,
        ccDataEntryLSB,
        ccDataIncrement,
        ccDataDecrement
      ]
      then ProcessRpn( aValue, aController, aChannel);

      ProcessMessage( MakeMidiControlChange( aValue, aController, aChannel));
    end;


    procedure   TMidiReceiver.ProcessProgramChange( aProgram, aChannel: Byte);
    begin
      ProcessMessage( MakeMidiProgramChange( aProgram, aChannel));
    end;


    procedure   TMidiReceiver.ProcessChannelPressure( aValue, aChannel: Byte);
    begin
      ProcessMessage( MakeMidiChannelPressure( aValue, aChannel));
    end;


    procedure   TMidiReceiver.ProcessPitchBend( aData: Word; aChannel: Byte);
    begin
      ProcessMessage( MakeMidiPitchBend( aData, aChannel));
    end;


    procedure   TMidiReceiver.ProcessStartSysEx;
    begin
      SetLength( FSysex, 0);
    end;


    procedure   TMidiReceiver.ProcessEndSysEx;
    begin
      // Sysex message completed and stored in FMidiSysex
      ProcessMessage( MakeMidiSysExData( FSysex));
    end;


    procedure   TMidiReceiver.ProcessSongPosition( aData: Word);
    begin
      ProcessMessage( MakeMidiSongPosition( aData));
    end;


    procedure   TMidiReceiver.ProcessSongSelect( aSong: Byte);
    begin
      ProcessMessage( MakeMidiSongSelect( aSong));
    end;


    procedure   TMidiReceiver.ProcessTuneRequest;
    begin
      ProcessMessage( MakeMidiTuneRequest);
    end;


    procedure   TMidiReceiver.ProcessTimingClock;
    begin
      ProcessMessage( MakeMidiTimingClock);
    end;


    procedure   TMidiReceiver.ProcessStart;
    begin
      ProcessMessage( MakeMidiStart);
    end;


    procedure   TMidiReceiver.ProcessContinue;
    begin
      ProcessMessage( MakeMidiContinue);
    end;


    procedure   TMidiReceiver.ProcessStop;
    begin
      ProcessMessage( MakeMidiStop);
    end;


    procedure   TMidiReceiver.ProcessActiveSense;
    begin
      ProcessMessage( MakeMidiActiveSense);
    end;


    procedure   TMidiReceiver.ProcessReset;
    begin
      ProcessMessage( MakeMidiReset);
    end;


    procedure   TMidiReceiver.ProcessSysExData( aData: Byte);
    begin
      if Length( FSysex) >= MaxSysex // Prevent buffer from getting arbitrary large
      then begin
        ProcessEndSysEx;             // process the partial message
        ProcessStartSysEx;           // and start a new one
      end;

      SetLength( FSysex, Length( FSysex) + 1);
      FSysex[ Length( FSysex) - 1] := aData;
    end;


    procedure   TMidiReceiver.ProcessMessage( const aMsg: TMidiMessage);
    begin
      if Assigned( FOnMessage)
      then FOnMessage( Self, aMsg);
    end;


//  public

    constructor TMidiReceiver.Create;
    begin
      FRpnMode := mrm14;
      Reset;
    end;


    procedure   TMidiReceiver.Reset;
    begin
      FRxState       := mrxNormal;
      FRPNState      := mrsRPN;
      FlastRPN       := RPNNull;
      FlastNRPN      := RPNNull;
      FlastRPNVal    := 0;
      FlastNRPNVal   := 0;
      FDataCount     := 0;
      FRunningStatus := 0;
      FChannel       := 0;
      FData          := 0;
      SetLength( FSysEx, 0);
    end;


    procedure   TMidiReceiver.AcceptByte( aByte: Byte);
    begin
      case FRxState of

        mrxNormal :
          begin
            if aByte and MaskCmd = MaskCmd
            then AcceptCommand( aByte);
          end;

        mrxRunningStatus :
          begin
            if aByte and MaskCmd = MaskCmd
            then AcceptCommand( aByte)
            else AcceptData   ( aByte);
          end;

      end;
    end;


    procedure   TMidiReceiver.AcceptBytes( aBytes: TBytes);
    var
      i : Integer;
    begin
      for i := 0 to Length( aBytes) - 1
      do AcceptByte( aBytes[ i]);
    end;



end.

