unit module_implementations;

{

   COPYRIGHT 2013 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  WinApi.MMSystem,

  System.SysUtils, System.UITypes, System.Classes, System.Math,

  Vcl.Dialogs, Vcl.ExtCtrls,

  Globals, module_defs, ComplexMath, KnobsUtils, KnobsConversions, KnobsMaze, midi_defs, midi_reader, OSCif, windowing,
  WaveUtils, WaveStorage, scala, freereverb, gverb, fftwgen, fftw_, Talkie, module_scales, loro_sc, poetry_generator,
  grains, SVFUnit, FormantControl, tss, Speech, BSplines, Noise, Markov, IntSequences, Moog, Rationals, SpeechLib_TLB,
  ESpeakInterface, TaggedAudio;


const

  DELAY_BUF_SIZE  = 65536;   // main delay buffer size - 65536 samples on 44100 Hz means 1.49 s.
  MAX_XTAB        = 20480;   // crossfade lookup table size for pitch change


type

  TARState         = ( arIdle, arA, arH, arR, arWait);
  TAHDState        = ( ahdIdle, ahdA, ahdH, ahdD, ahdWait);
  TADSRState       = ( adsrIdle, adsrA, adsrD, adsrS, adsrR, adsrWait);
  TEnvShape        = ( esLog, esLin, esExp, esS);
  TLfoRange        = ( lsSlow, lsMedium, lsFast, lsBPM, lsLin, lsSlow2, lsMedium2, lsFast2);
  TDelayRange      = ( dsShort, dsMedium, dsLong);
  TRWState         = ( trwIdle, trwFired);
  TTrigOScState    = ( toIdle, toRunning, toWaiting, toDone);
  TSeqInState      = ( seqClkLow, seqClkHigh);
  TSeqStepState    = ( sssIdle, sssActive, sssWaiting);
  TPulsesState     = ( psIdle, psPulsing, psWaiting);
  TMidiPlayerState = ( msActive, msDone);


  TLongWord = record
    Lo, Hi : Word;
  end;


  TModMConst = class( TMod)
  strict private
  const
    i_value         =  0;
  const
    o_out           =  0;
  const
    c_1             =  0;
    c_2             =  1;
    c_e             =  2;
    c_pi            =  3;
    c_sqrt2         =  4;
    c_sqrt3         =  5;
    c_sqrt5         =  6;
    c_12r2          =  7;
    c_semi          =  8;
    c_r2            =  9;
    c_re            = 10;
    c_rpi           = 11;
    c_rsqrt2        = 12;
    c_rsqrt3        = 13;
    c_rsqrt5        = 14;
    c_r12r2         = 15;
    c_uni_0         = 16;
    c_mi_sec_1      = 17;
    c_second_2      = 18;
    c_mi_third_3    = 19;
    c_ma_third_4    = 20;
    c_fourth_5      = 21;
    c_tritone_6     = 22;
    c_fifth_7       = 23;
    c_mi_sixth_8    = 24;
    c_ma_sixth_9    = 25;
    c_mi_seventh_10 = 26;
    c_ma_seventh_11 = 27;
    c_octave_12     = 28;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModConstant = class( TMod)
  strict private
  const
    i_value = 0;
    i_mute  = 1;
    i_mode  = 2;
  const
    o_out   = 0;
  const
    m_lin44 = 0;
    m_db4   = 1;
    m_lin11 = 2;
    m_db0   = 3;
    m_lin04 = 4;
    m_lin01 = 5;
    m_note  = 6;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModKnobs4 = class( TMod)
  strict private
  const
    i_value1 =  0;
    i_mode1  =  1;
    i_value2 =  2;
    i_mode2  =  3;
    i_value3 =  4;
    i_mode3  =  5;
    i_value4 =  6;
    i_mode4  =  7;
    i_in1    =  8;
    i_in2    =  9;
    i_in3    = 10;
    i_in4    = 11;
    i_mute1  = 12;
    i_mute2  = 13;
    i_mute3  = 14;
    i_mute4  = 15;
  const
    o_out1   =  0;
    o_out2   =  1;
    o_out3   =  2;
    o_out4   =  3;
  const
    m_lin44  =  0;
    m_db4    =  1;
    m_lin11  =  2;
    m_db0    =  3;
    m_lin04  =  4;
    m_lin01  =  5;
    m_note   =  6;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModSwitches4 = class( TMod)
  strict private
  const
    i_sw1  = 0;
    i_sw2  = 1;
    i_sw3  = 2;
    i_sw4  = 3;
    i_in1  = 4;
    i_in2  = 5;
    i_in3  = 6;
    i_in4  = 7;
  const
    o_out1 = 0;
    o_out2 = 1;
    o_out3 = 2;
    o_out4 = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModButtons4 = class( TMod)
  strict private
  const
    i_sw1  = 0;
    i_sw2  = 1;
    i_sw3  = 2;
    i_sw4  = 3;
  const
    o_out1 = 0;
    o_out2 = 1;
    o_out3 = 2;
    o_out4 = 3;
  private
    FCount1    : Integer;
    FCount2    : Integer;
    FCount3    : Integer;
    FCount4    : Integer;
    FLedCount1 : Integer;
    FLedCount2 : Integer;
    FLedCount3 : Integer;
    FLedCount4 : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModTapper = class( TMod)
  strict private
  const
    i_sw       = 0;
    i_bpm      = 1;
    i_sync     = 2;
    i_rate     = 3;
    i_mode     = 4;
    i_inslave  = 5;
  const
    o_outpulse = 0;
    o_outspeed = 1;
    o_outslave = 2;
  private
    FCount     : Integer;
    FLedCount  : Integer;
    FTapCount  : Integer;
    FTrigCount : Integer;
    FTime      : Cardinal;
    FBPM       : TSignal;
    FChanged   : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights ( const aPrefix: string; const aCallback: TLightsHandler);                        override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
  end;


  TModPad = class( TMod)
  strict private
  const
    i_padx       = 0;
    i_pady       = 1;
    i_padr       = 2;
    i_padg       = 3;
    i_padb       = 4;
    i_outputtype = 5;
  const
    o_x          = 0;
    o_y          = 1;
    o_r          = 2;
    o_g          = 3;
    o_b          = 4;
  private
    FFileName        : string;
    FFileNameChanged : Boolean;
  private
    procedure   SetFileName( const aValue: string);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  public
    property    FileName: string read FFileName write SetFileName;
  end;


  TModSpiral = class( TMod)
  strict private
  const
    i_padx    = 0;
    i_pady    = 1;
    i_steps   = 2;
  const
    o_outnote = 0;
    o_outphi  = 1;
    o_outnorm = 2;
    o_outlen  = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModPolar = class( TMod)
  strict private
  const
    i_inx  = 0;
    i_iny  = 1;
  const
    o_outa = 0;
    o_outd = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRectangular = class( TMod)
  strict private
  const
    i_ina  = 0;
    i_ind  = 1;
  const
    o_outx = 0;
    o_outy = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModInverter = class( TMod)
  strict private
  const
    i_in1  = 0; // Keep these consecutive
    i_in2  = 1;
    i_in3  = 2;
    i_in4  = 3;
  const
    o_out1 = 0; // Keep these consecutive
    o_out2 = 1;
    o_out3 = 2;
    o_out4 = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModAdder = class( TMod)
  strict private
  const
    i_mute = 0;
    i_in1  = 1;
    i_in2  = 2;
    i_in3  = 3;
    i_in4  = 4;
    i_in5  = 5;
    i_in6  = 6;
    i_in7  = 7;
  const
    o_out  = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMultiplier = class( TMod)
  strict private
  const
    i_mute = 0;
    i_in1  = 1;
    i_in2  = 2;
    i_in3  = 3;
    i_in4  = 4;
    i_in5  = 5;
    i_in6  = 6;
    i_in7  = 7;
  const
    o_out  = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModReciprocal = class( TMod)
  strict private
  const
    i_x   = 0;
  const
    o_out = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModDivide = class( TMod)
  strict private
  const
    i_x   = 0;
    i_y   = 1;
  const
    o_out = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMonadic = class( TMod)
  strict private
  const
    i_function = 0;
    i_x        = 1;
  const
    o_out      = 0;
  const
    f_squared    =  0;        //    x^2
    f_cubed      =  1;        //    x^3
    f_squareroot =  2;        //    sqrt(x)
    f_cubicroot  =  3;        //    x^1/3
    f_sine       =  4;        //    sin(x)
    f_cosine     =  5;        //    cos(x)
    f_tangent    =  6;        //    tan(x)
    f_arcsine    =  7;        //    asin(x)
    f_arccosine  =  8;        //    acos(x)
    f_arctangent =  9;        //    atan(x)
    f_2powerx    = 10;        //    2^x
    f_epowerx    = 11;        //    e^x
    f_10powerx   = 12;        //    10^x
    f_2logx      = 13;        //    2log(x)
    f_elogx      = 14;        //    ln(x)
    f_10logx     = 15;        //    10log(x)
    f_2pwxdiv12  = 16;        //    2^(x/12)
    f_12t2logx   = 17;        //    12*2log(x)
    f_ttimespi   = 18;        //    *PI
    f_timestwopi = 19;        //    *2PI
    f_divpi      = 20;        //    /PI
    f_divtwopi   = 21;        //    /2PI
    f_floor      = 22;        //    floor(x)
    f_round      = 23;        //    round(x)
    f_ceil       = 24;        //    ceil(x)
    f_abs        = 25;        //    abs(x)
    f_negate     = 26;        //    - x
    f_flip       = 27;        //    1 - x
    f_tanh       = 28;        //    tanh( x)
    f_absflip    = 29;        //    1 - abs( x)
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModDyadic = class( TMod)
  strict private
  const
    i_function = 0;
    i_x        = 1;
    i_y        = 2;
  const
    o_out      = 0;
  const
    f_modulo   = 0;    // x mod y
    f_power    = 1;    // x ^ y
    flog       = 2;    // xlog( y)
    f_Atan2    = 3;    // atan( y / x)
    f_xminy    = 4;    // x - y
    f_yminx    = 5;    // y - x
    f_xplusy   = 6;    // x + y
    f_invpower = 7;    // x ^ ( 1 / y)
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMultiMult = class( TMod)
  strict private
  const
    i_mult = 0;
    i_in1  = 1;
    i_in2  = 2;
    i_in3  = 3;
    i_in4  = 4;
    i_in5  = 5;
    i_in6  = 6;
    i_in7  = 7;
    i_in8  = 8;
  const
    o_out1 = 0;
    o_out2 = 1;
    o_out3 = 2;
    o_out4 = 3;
    o_out5 = 4;
    o_out6 = 5;
    o_out7 = 6;
    o_out8 = 7;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMultipleMult = class( TMod)
  strict private
  const
    i_in11 =  0;
    i_in12 =  1;
    i_in21 =  2;
    i_in22 =  3;
    i_in31 =  4;
    i_in32 =  5;
    i_in41 =  6;
    i_in42 =  7;
    i_in51 =  8;
    i_in52 =  9;
    i_in61 = 10;
    i_in62 = 11;
    i_in71 = 12;
    i_in72 = 13;
    i_in81 = 14;
    i_in82 = 15;
  const
    o_out1 =  0;
    o_out2 =  1;
    o_out3 =  2;
    o_out4 =  3;
    o_out5 =  4;
    o_out6 =  5;
    o_out7 =  6;
    o_out8 =  7;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModNoteToDelay = class( TMod)
  strict private
  const
    i_in  = 0;
  const
    o_out = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModZeroCounter = class( TMod)
  strict private
  const
    i_in        = 0;
    i_threshold = 1;
  const
    o_period    = 0;
    o_gate      = 1;
    o_note      = 2;
  private
    FOldSign : Boolean;
    FTime    : Integer;
    FFreq    : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMedian = class( TMod)
  strict private
  const
    i_in   = 0;
    i_trig = 1;
  const
    o_out  = 0;
    o_high = 1;
    o_low  = 2;
    o_avg  = 3;
  private
    FData      : TMedianCalculator;
    FOldTrig   : Boolean;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModCompare = class( TMod)
  strict private
  const
    i_in1      = 0;
    i_in2      = 1;
    i_range    = 2;
  const
    o_outless  = 0;
    o_outequal = 1;
    o_outmore  = 2;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModGate = class( TMod)
  strict private
  const
    i_function   = 0;
    i_width      = 1;
    i_in1        = 2;
    i_in2        = 3;
    i_in3        = 4;
    i_in4        = 5;
    i_outputtype = 6;
    i_func       = 7;
  const
    o_out        = 0;
  const
    m_and        = 0;
    m_nand       = 1;
    m_or         = 2;
    m_nor        = 3;
    m_xor        = 4; // Even parity, true when an odd  number of inputs is active
    m_xnor       = 5; // Odd  pariry, true when an even number of inputs is active
    m_one        = 6; // true  when exactly one input is active
    m_none       = 7; // false when exactly one input is active
  private
    FFunction : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler);                     override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModMultiGate = class( TMod)
  strict private
  const
    i_in11       =  0;
    i_in12       =  1;
    i_in21       =  2;
    i_in22       =  3;
    i_in31       =  4;
    i_in32       =  5;
    i_in41       =  6;
    i_in42       =  7;
    i_outputtype =  8;
    i_inputtype  =  9;
    i_func       = 10;
  const
    o_out1       =  0;
    o_out2       =  1;
    o_out3       =  2;
    o_out4       =  3;
  const
    m_and        =  0;
    m_nand       =  1;
    m_or         =  2;
    m_nor        =  3;
    m_xor        =  4; // Even parity, true when an odd  number of inputs is active
    m_xnor       =  5; // Odd  pariry, true when an even number of inputs is active
    m_one        =  6; // true  when exactly one input is active
    m_none       =  7; // false when exactly one input is active
  private
    FFunction : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDivider = class( TMod)
  strict private
  const
    i_count      = 0;
    i_trig       = 1;
    i_reset      = 2;
    i_mode       = 3;
    i_outputtype = 4;
  const
    o_out        = 0;
  const
    m_states : array[ Boolean] of TSignal = ( -1, 1);
  private
    FPrevInput : Boolean;
    FCounter   : Integer;
    FOldCount  : Integer;
    FState     : Boolean;
    FOldReset  : Boolean;
    FOutValue  : TSignal;
    FActive    : Boolean;
  private
    procedure   PerformReset;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPulseDelay = class( TMod)
  strict private
  const
    i_trig        = 0;
    i_reset       = 1;
    i_delaymode   = 2;
    i_delaymod    = 3;
    i_delaymodamt = 4;
    i_range       = 5;
    i_delay       = 6;
    i_outputtype  = 7;
  const
    o_out         = 0;
  const
    m_delay       = 0;
    m_stretch     = 1;
    m_oscillate   = 2;
    m_oscdel      = 3;
  private
    FPrevTrig  : Boolean;
    FPrevReset : Boolean;
    FOutValue  : Boolean;
    FDuration  : Int64;
    FNegCount  : Int64;
    FPosCount  : Int64;
    FPrevMode  : Integer;
    FActive    : Boolean;
    FInDelay   : Boolean;
    FNegTrig   : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPulseSkip = class( TMod)
  strict private
  const
    i_trig       = 0;
    i_reset      = 1;
    i_in         = 2;
    i_inmod      = 3;
    i_inmodamt   = 4;
    i_cp         = 5;
    i_cpmod      = 6;
    i_cpmodamt   = 7;
    i_outputtype = 8;
  const
    o_out        = 0;
  private
    FPrevTrig  : Boolean;
    FPrevReset : Boolean;
    FOutActive : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModProgDivider = class( TMod)
  strict private
  const
    i_trig       = 0;
    i_reset      = 1;
    i_select     = 2;
    i_selectlev  = 3;
    i_steps      = 4;
    i_lowcount   = 5;
    i_highcount  = 6;
    i_mode       = 7;
    i_outputtype = 8;
  const
    o_out        = 0;
  const
    m_states : array[ Boolean] of TSignal = ( -1, 1);
  private
    FPrevInput : Boolean;
    FCounter   : Integer;
    FCount     : Integer;
    FState     : Boolean;
    FOldReset  : Boolean;
    FOutValue  : TSignal;
  private
    procedure   PerformReset;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDFlipFlop = class( TMod)
  strict private
  const
    i_d          = 0;
    i_clk        = 1;
    i_outputtype = 2;
  const
    o_q    = 0;
    o_notq = 1;
  private
    FClockState : Boolean;
    FQ          : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModRSFlipFlop = class( TMod)
  strict private
  const
    i_s          = 0;
    i_r          = 1;
    i_trigmode   = 2;
    i_outputtype = 3;
  const
    o_q        = 0;
    o_notq     = 1;
  const
    m_edge     = 0;
    m_level    = 1;
  private
    FSetState   : Boolean;
    FResetState : Boolean;
    FQ          : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPulseSync = class( TMod)
  strict private
  const
    i_pulse      = 0;
    i_sync       = 1;
    i_outputtype = 2;
  const
    o_q          = 0;
  private
    FState      : Boolean;
    FOldPulse   : Boolean;
    FOldSync    : Boolean;
    FQ          : Boolean;
    FCounter    : Integer;
    FLedCounter : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModCounter = class( TMod)
  strict private
  const
    i_trig  =  0;
    i_reset =  1;
    i_dir   =  2;
    i_mode  =  3;
  const
    o_out0  =  0;
    o_out1  =  1;
    o_out2  =  2;
    o_out3  =  3;
    o_out4  =  4;
    o_out5  =  5;
    o_out6  =  6;
    o_out7  =  7;
    o_out8  =  8;
    o_out9  =  9;
    o_out10 = 10;
    o_out11 = 11;
    o_out12 = 12;
    o_out13 = 13;
    o_out14 = 14;
    o_out15 = 15;
  const
    m_bin   =  0;
    m_seq   =  1;
  private
    FState   : Word;
    FMode    : Integer;
    FOldRes  : Boolean;
    FOldTrig : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModFixedDiv = class( TMod)
  strict private
  const
    i_trig   =  0;
    i_res    =  1;
    i_square =  2;
  const
    o_out2   =  0;
    o_out3   =  1;
    o_out4   =  2;
    o_out5   =  3;
    o_out6   =  4;
    o_out7   =  5;
    o_out8   =  6;
    o_out9   =  7;
    o_out10  =  8;
    o_out11  =  9;
    o_out12  = 10;
    o_out13  = 11;
    o_out14  = 12;
    o_out16  = 13;
    o_out32  = 14;
    o_out64  = 15;
  private
    FOldRes   : Boolean;
    FOldTrig  : Boolean;
    FCounter  : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPrimeDiv = class( TMod)
  strict private
  const
    i_trig   =  0;
    i_res    =  1;
    i_square =  2;
  const
    o_out2   =  0;
    o_out3   =  1;
    o_out5   =  2;
    o_out7   =  3;
    o_out11  =  4;
    o_out13  =  5;
    o_out17  =  6;
    o_out19  =  7;
    o_out23  =  8;
    o_out29  =  9;
    o_out31  = 10;
    o_out37  = 11;
    o_out41  = 12;
    o_out43  = 13;
    o_out47  = 14;
    o_out53  = 15;
  private
    FOldRes   : Boolean;
    FOldTrig  : Boolean;
    FCounter  : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModAmuse = class( TMod)
  strict private
  const
    i_reset      =  0;
    i_trig       =  1;
    i_sela       =  2;
    i_selb       =  3;
    i_selc       =  4;
    i_seld       =  5;
    i_sele       =  6;
    i_selw       =  7;
    i_selx       =  8;
    i_sely       =  9;
    i_selz       = 10;
    i_selmoda    = 11;
    i_selmodb    = 12;
    i_selmodc    = 13;
    i_selmodd    = 14;
    i_selmode    = 15;
    i_selmodw    = 16;
    i_selmodx    = 17;
    i_selmody    = 18;
    i_selmodz    = 19;
    i_selmodamta = 20;
    i_selmodamtb = 21;
    i_selmodamtc = 22;
    i_selmodamtd = 23;
    i_selmodamte = 24;
    i_selmodamtw = 25;
    i_selmodamtx = 26;
    i_selmodamty = 27;
    i_selmodamtz = 28;
    i_clocked    = 29;
    i_ext1       = 30;
    i_ext2       = 31;
    i_reverse    = 32;
    i_evenodd    = 33;
    i_paritymod  = 34;
    i_gatemode   = 35;
  const
    o_out        =  0;
    o_inv        =  1;
    o_bit        =  2;
    o_data       =  3;
  private
    FOldRes    : Boolean;
    FResetReq  : Boolean;
    FOldTrig   : Boolean;
    FShifter   : Cardinal;
    FCountC1   : Integer;
    FCountC2   : Integer;
    FCountC3   : Integer;
    FCountC4   : Integer;
    FCountC5   : Integer;
    FCountC6   : Integer;
    FCountC7   : Integer;
    FCountC8   : Integer;
    FMemoryC1  : Boolean;
    FMemoryC2  : Boolean;
    FMemoryC3  : Boolean;
    FMemoryC4  : Boolean;
    FMemoryC5  : Boolean;
    FMemoryC6  : Boolean;
    FMemoryC7  : Boolean;
    FMemoryC8  : Boolean;
    FSwa       : Boolean;
    FSwb       : Boolean;
    FSwc       : Boolean;
    FSwd       : Boolean;
    FSwe       : Boolean;
    FExt1      : Boolean;
    FExt2      : Boolean;
    FCarry0    : Boolean;
    FCarry31   : Boolean;
    FLights    : string;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModChladniControl = class( TMod)
  strict private
  const
    i_c       =  0;
    i_cmod    =  1;
    i_cmodamt =  2;
    i_p       =  3;
    i_pmod    =  4;
    i_pmodamt =  5;
    i_n       =  6;
    i_nmod    =  7;
    i_nmodamt =  8;
    i_m       =  9;
    i_mmod    = 10;
    i_mmodamt = 11;
  const
    o_out = 0;
  private
    FOldC : TSignal;
    FOldP : TSignal;
    FOldN : Integer;
    FOldM : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModShifter = class( TMod)
  strict private
  const
    i_reset      =  0;
    i_trig       =  1;
    i_extlow     =  2;
    i_exthigh    =  3;
    i_reverse    =  4;
    i_gatemode   =  5;
    i_clocked    =  6;
    i_in         =  7;
  const
    o_out0       =  0;
    o_out1       =  1;
    o_out2       =  2;
    o_out3       =  3;
    o_out4       =  4;
    o_out5       =  5;
    o_out6       =  6;
    o_out7       =  7;
    o_out8       =  8;
    o_out9       =  9;
    o_out10      = 10;
    o_out11      = 11;
    o_out12      = 12;
    o_out13      = 13;
    o_out14      = 14;
    o_out15      = 15;
    o_out16      = 16;
    o_out17      = 17;
    o_out18      = 18;
    o_out19      = 19;
    o_out20      = 20;
    o_out21      = 21;
    o_out22      = 22;
    o_out23      = 23;
    o_out24      = 24;
    o_out25      = 25;
    o_out26      = 26;
    o_out27      = 27;
    o_out28      = 28;
    o_out29      = 29;
    o_out30      = 30;
    o_out31      = 31;
    o_carrylow   = 32;
    o_carryhigh  = 33;
  private
    FOldRes    : Boolean;
    FResetReq  : Boolean;
    FOldTrig   : Boolean;
    FShifter   : Cardinal;
    FCOutHi    : Boolean;
    FCOutLo    : Boolean;
    FLights    : string;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                 override;
  end;


  TModLFSR = class( TMod)
  strict private
  const
    i_reset     =  0;
    i_trig      =  1;
    i_reverse   =  2;
    i_gatemode  =  3;
    i_clocked   =  4;
    i_in        =  5;
    i_evenodd   =  6;
    i_paritymod =  7;
    i_tap0      =  8;
    i_tap1      =  9;
    i_tap2      = 10;
    i_tap3      = 11;
    i_tap4      = 12;
    i_tap5      = 13;
    i_tap6      = 14;
    i_tap7      = 15;
    i_tap8      = 16;
    i_tap9      = 17;
    i_tap10     = 18;
    i_tap11     = 19;
    i_tap12     = 20;
    i_tap13     = 21;
    i_tap14     = 22;
    i_tap15     = 23;
    i_tap16     = 24;
    i_tap17     = 25;
    i_tap18     = 26;
    i_tap19     = 27;
    i_tap20     = 28;
    i_tap21     = 29;
    i_tap22     = 30;
    i_tap23     = 31;
    i_tap24     = 32;
    i_tap25     = 33;
    i_tap26     = 34;
    i_tap27     = 35;
    i_tap28     = 36;
    i_tap29     = 37;
    i_tap30     = 38;
    i_tap31     = 39;
    i_tapmod0   = 40;
    i_tapmod1   = 41;
    i_tapmod2   = 42;
    i_tapmod3   = 43;
    i_tapmod4   = 44;
    i_tapmod5   = 45;
    i_tapmod6   = 46;
    i_tapmod7   = 47;
    i_tapmod8   = 48;
    i_tapmod9   = 49;
    i_tapmod10  = 50;
    i_tapmod11  = 51;
    i_tapmod12  = 52;
    i_tapmod13  = 53;
    i_tapmod14  = 54;
    i_tapmod15  = 55;
    i_tapmod16  = 56;
    i_tapmod17  = 57;
    i_tapmod18  = 58;
    i_tapmod19  = 59;
    i_tapmod20  = 60;
    i_tapmod21  = 61;
    i_tapmod22  = 62;
    i_tapmod23  = 63;
    i_tapmod24  = 64;
    i_tapmod25  = 65;
    i_tapmod26  = 66;
    i_tapmod27  = 67;
    i_tapmod28  = 68;
    i_tapmod29  = 69;
    i_tapmod30  = 70;
    i_tapmod31  = 71;
  const
    o_out       =  0;
  private
    FOldRes   : Boolean;
    FResetReq : Boolean;
    FOldTrig  : Boolean;
    FShifter  : Cardinal;
    FFeedback : Boolean;
    FLights   : string;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                 override;
  end;


  TModStack = class( TMod)
  strict private
  const
    i_in0   =  0;
    i_in1   =  1;
    i_in2   =  2;
    i_in3   =  3;
    i_in4   =  4;
    i_in5   =  5;
    i_in6   =  6;
    i_in7   =  7;
    i_in8   =  8;
    i_in9   =  9;
    i_in10  = 10;
    i_in11  = 11;
    i_in12  = 12;
    i_in13  = 13;
    i_in14  = 14;
    i_in15  = 15;
    i_push  = 16;
    i_pop   = 17;
  const
    o_out0  =  0;
    o_out1  =  1;
    o_out2  =  2;
    o_out3  =  3;
    o_out4  =  4;
    o_out5  =  5;
    o_out6  =  6;
    o_out7  =  7;
    o_out8  =  8;
    o_out9  =  9;
    o_out10 = 10;
    o_out11 = 11;
    o_out12 = 12;
    o_out13 = 13;
    o_out14 = 14;
    o_out15 = 15;
    o_full  = 16;
    o_empty = 17;
  private
    FLights  : string;
    FStack   : array of Word;
    FSize    : Integer;
    FSP      : Integer;
    FOldPop  : Boolean;
    FOldPush : Boolean;
  private
    procedure   SetSize( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Size : Integer read FSize write SetSize;
  end;


  TModAnalogStack = class( TMod)
  strict private
  const
    i_in    =  0;
    i_push  =  1;
    i_pop   =  2;
  const
    o_out   =  0;
    o_full  =  1;
    o_empty =  2;
  private
    FLights  : string;
    FStack   : array of TSignal;
    FSize    : Integer;
    FSP      : Integer;
    FOldPop  : Boolean;
    FOldPush : Boolean;
  private
    procedure   SetSize( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Size : Integer read FSize write SetSize;
  end;


  TModSwanSong = class( TMod)
  strict private
  const
    i_swanx         =  0;
    i_swany         =  1;
    i_hunter1clock  =  2;
    i_hunter2clock  =  3;
    i_hunter3clock  =  4;
    i_hunter4clock  =  5;
    i_hunter5clock  =  6;
    i_hunter6clock  =  7;
    i_intype        =  8;
    i_outtype       =  9;
    i_mazetype      = 10;
    i_hunterrule    = 11;
    i_mazemod       = 12;
    i_scaling       = 13;
  const
    o_hunter1angle  =  0;
    o_hunter2angle  =  1;
    o_hunter3angle  =  2;
    o_hunter4angle  =  3;
    o_hunter5angle  =  4;
    o_hunter6angle  =  5;
    o_hunter1stride =  6;
    o_hunter2stride =  7;
    o_hunter3stride =  8;
    o_hunter4stride =  9;
    o_hunter5stride = 10;
    o_hunter6stride = 11;
    o_hunter1x      = 12;
    o_hunter2x      = 13;
    o_hunter3x      = 14;
    o_hunter4x      = 15;
    o_hunter5x      = 16;
    o_hunter6x      = 17;
    o_hunter1y      = 18;
    o_hunter2y      = 19;
    o_hunter3y      = 20;
    o_hunter4y      = 21;
    o_hunter5y      = 22;
    o_hunter6y      = 23;
  const
    HUNTER_COUNT = 6;
  private
    FMazeGraph       : TMazeGraph;
    FMazeForth       : TMazeForth;
    FMazeList        : TStringList;
    FMazeType        : Integer;
    FOldHunterClocks : array[ 0 .. HUNTER_COUNT - 1] of Boolean;
    FAngles          : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FStrides         : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FXValues         : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FYValues         : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FSwan            : TMazePoint;
    FHunterLocations : TSignalArray;
    FResetReq        : Boolean;
    FMazeTypeChanged : Boolean;
    FMazeExtent      : TMazeRect;
    FSwanOffSet      : TMazePoint;
    FSwanScale       : TMazePoint;
  private
    procedure   SetMazeType( aValue: Integer);
    function    ScaledSwan: TmazePoint;
    procedure   DoHunterMoved( const aSender: TObject; anId, aLocation: Integer; aDistance, anAngle, anX, anY: TSignal);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler);                     override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
    procedure   GatherData      ( const aPrefix: string; const aCallback: TDataHandler);                       override;
  public
    property    MazeType : Integer read FMazeType write SetMazeType;
  end;


  TModAdc = class( TMod)
  strict private
  const
    i_in    =  0;
  const
    o_out0  =  0;
    o_out1  =  1;
    o_out2  =  2;
    o_out3  =  3;
    o_out4  =  4;
    o_out5  =  5;
    o_out6  =  6;
    o_out7  =  7;
    o_out8  =  8;
    o_out9  =  9;
    o_out10 = 10;
    o_out11 = 11;
    o_out12 = 12;
    o_out13 = 13;
    o_out14 = 14;
    o_out15 = 15;
  const
    n_bits  = 16;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDac = class( TMod)
  strict private
  const
    i_in0  =  0;
    i_in1  =  1;
    i_in2  =  2;
    i_in3  =  3;
    i_in4  =  4;
    i_in5  =  5;
    i_in6  =  6;
    i_in7  =  7;
    i_in8  =  8;
    i_in9  =  9;
    i_in10 = 10;
    i_in11 = 11;
    i_in12 = 12;
    i_in13 = 13;
    i_in14 = 14;
    i_in15 = 15;
  const
    o_out  =  0;
  const
    n_bits = 16;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPhaseDetect = class( TMod)
  strict private
  type
    t_state = 1 .. 12;
  const
    i_in1         = 0;
    i_in2         = 1;
    i_alpha       = 2;
    i_alphamod    = 3;
    i_alphamodamt = 4;
    i_type        = 5;
    i_typesel     = 6;
  const
    o_out         = 0;
    o_lock        = 1;
  private
    FOldIn1 : Boolean;
    FOldIn2 : Boolean;
    FState  : t_state;
    FAlpha  : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModNot = class( TMod)
  strict private
  const
    i_in1  = 0;
    i_in2  = 1;
    i_in3  = 2;
    i_in4  = 3;
  const
    o_out1 = 0;
    o_out2 = 1;
    o_out3 = 2;
    o_out4 = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModLogicSelector = class( TMod)
  strict private
  const
    i_sel  = 0;
    i_ina  = 1;
    i_inb  = 2;
  const
    o_outa = 0;
    o_outb = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModAmplifier = class( TMod)
  strict private
  const
    i_gain = 0;
    i_mute = 1;
    i_in1  = 2; // keep in1 .. in4 consecutive
    i_in2  = 3;
    i_in3  = 4;
    i_in4  = 5;
  const
    o_out1 = 0; // keep out1 .. out4 consecutive
    o_out2 = 1;
    o_out3 = 2;
    o_out4 = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModScaler = class( TMod)
  strict private
  const
    i_gain   = 0;
    i_offset = 1;
    i_mode   = 2;
    i_mute   = 3;
    i_in1    = 4;
    i_in2    = 5;
    i_in3    = 6;
    i_in4    = 7;
  const
    o_out1   = 0;
    o_out2   = 1;
    o_out3   = 2;
    o_out4   = 3;
    o_sum    = 4;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModPropScaler = class( TMod)
  strict private
  const
    i_p       = 0;
    i_pmod    = 1;
    i_pmodamt = 2;
    i_in1     = 3;
    i_in2     = 4;
    i_in3     = 5;
    i_in4     = 6;
  const
    o_out1   = 0;
    o_out2   = 1;
    o_out3   = 2;
    o_out4   = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixingScaler = class( TMod)
  strict private
  const
    i_gain     =  0;
    i_offset   =  1;
    i_mode     =  2;
    i_mute     =  3;
    i_in1      =  4;
    i_gain2    =  5;
    i_offset2  =  6;
    i_mode2    =  7;
    i_mute2    =  8;
    i_in2      =  9;
    i_gain3    = 10;
    i_offset3  = 11;
    i_mode3    = 12;
    i_mute3    = 13;
    i_in3      = 14;
    i_gain4    = 15;
    i_offset4  = 16;
    i_mode4    = 17;
    i_mute4    = 18;
    i_in4      = 19;
    i_outmute  = 20;
    i_outlevel = 21;
    i_chain    = 22;
  const
    o_out1     =  0;
    o_out2     =  1;
    o_out3     =  2;
    o_out4     =  3;
    o_sum      =  4;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRatio = class( TMod)
  strict private
  const
    i_multiplier = 0;
    i_divider    = 1;
    i_mute       = 2;
    i_in1        = 3;
    i_in2        = 4;
    i_in3        = 5;
    i_in4        = 6;
  const
    o_out1       = 0;
    o_out2       = 1;
    o_out3       = 2;
    o_out4       = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRangeConverter = class( TMod)
  strict private
  const
    i_lowvalue   = 0;
    i_highvalue  = 1;
    i_mode       = 2;
    i_in1        = 3;
    i_in2        = 4;
    i_in3        = 5;
    i_in4        = 6;
  const
    o_out1       = 0;
    o_out2       = 1;
    o_out3       = 2;
    o_out4       = 3;
  const
    s_normal            = 0;
    s_inverted          = 1;
    s_positive          = 2;
    s_positive_inverted = 3;
    s_negative          = 4;
    s_negative_inverted = 5;
  private
    FInLow  : TSignal;
    FInHigh : TSignal;
    FMode   : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModLevelConverter = class( TMod)
  strict private
  const
    i_inmode     = 0;
    i_outmode    = 1;
    i_in1        = 2;
    i_in2        = 3;
    i_in3        = 4;
    i_in4        = 5;
    i_in5        = 6;
    i_in6        = 7;
    i_in7        = 8;
  const
    o_out1       = 0;
    o_out2       = 1;
    o_out3       = 2;
    o_out4       = 3;
    o_out5       = 4;
    o_out6       = 5;
    o_out7       = 6;
  const
    s_normal            = 0;
    s_inverted          = 1;
    s_positive          = 2;
    s_positive_inverted = 3;
    s_negative          = 4;
    s_negative_inverted = 5;
  private
    FInMode  : Integer;
    FOutMode : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModTypeFlip = class( TMod)
  strict private
  const
    i_in         = 0;
    i_inmode     = 1;
    i_outmodel   = 2;
    i_outmoder   = 3;
    i_select     = 4;
    i_slew       = 5;
  const
    o_outl       = 0;
    o_outr       = 1;
  const
    s_normal            = 0;
    s_inverted          = 1;
    s_positive          = 2;
    s_positive_inverted = 3;
    s_negative          = 4;
    s_negative_inverted = 5;
  private
    FInMode   : Integer;
    FOutModeL : Integer;
    FOutModeR : Integer;
    FSelect   : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModFastToSlow = class( TMod)
  strict private
  const
    i_in  = 0;
  const
    o_out = 0;
  private
    FValue : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModSlowToFast = class( TMod)
  strict private
  const
    i_in  = 0;
  const
    o_out = 0;
  private
    FOldVal : TSignal;
    FNewVal : TSignal;
    FCount  : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModArpeggiator = class( TMod)
  strict private
  const
    i_in         =  0;
    i_newinput   =  1;
    i_newoutput  =  2;
    i_inmode     =  3;
    i_sorting    =  4;
    i_length     =  5;
    i_toggle     =  6;
    i_reset      =  7;
    i_clr        =  8;
    i_reverse    =  9;
    i_trans      = 10;
    i_lengthmod  = 11;
  const
    o_out        =  0;
  const
    so_unsorted  =  0;
    so_lotohi    =  1;
    so_hitolo    =  2;
    so_random    =  3;
    so_reversed  =  4;
    so_pingpong  = 5;
  const
    if_all       =  0;
    if_changed   =  1;
    if_unique    =  2;
  private
    FValues         : TSignalArray;
    FSorted         : TSignalArray;
    FToggled        : Boolean;
    FLength         : Integer;
    FSortOrder      : Integer;
    FMustSort       : Boolean;
    FInFilter       : Integer;
    FInPtr          : Integer;
    FOutPtr         : Integer;
    FOldInTrig      : Boolean;
    FOldOutTrig     : Boolean;
    FOldReset       : Boolean;
    FOldToggled     : Boolean;
    FOldClr         : Boolean;
    FMustClr        : Boolean;
    FInTrigCounter  : Integer;
    FOutTrigCounter : Integer;
    FResetCounter   : Integer;
    FClrCounter     : Integer;
    FPrevInValue    : TSignal;
    FTransparant    : Boolean;
    FPingPongState  : Boolean;
  private
    function    CompareLoFirst( anItem1, anItem2: TSignal): Integer;
    function    CompareHiFirst( anItem1, anItem2: TSignal): Integer;
    function    Comparer      ( aSortMode: Integer): TKnobsValueCompare;
    procedure   SwapItems     ( anIndex1, anIndex2: Integer);
    procedure   Sort( aLowBound, aHighBound: Integer; aComparer: TKnobsValueCompare);                          overload;
    procedure   Sort;                                                                                          overload;
    procedure   AppendValue   ( aValue: TSignal);
    procedure   AcceptNewValue( aValue: TSignal);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   SetDefaults;                                                                                   override;
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMarkov = class( TMod)
  strict private
  const
    i_in             = 0;
    i_strength       = 1;
    i_strengthmod    = 2;
    i_strengthmodamt = 3;
    i_mode           = 4;
    i_learn          = 5;
    i_newoutput      = 6;
    i_clear          = 7;
  const
    o_out            = 0;
    o_full           = 1;
  private
    FMarkovChain      : TMarkovChain;
    FOrder            : Integer;
    FMaxSize          : Integer;
    FCyclic           : Boolean;
    FOverflow         : Boolean;
    FOldLearn         : Boolean;
    FOldNewOutput     : Boolean;
    FOldClear         : Boolean;
    FOldFull          : Boolean;
    FLearnCounter     : Integer;
    FNewOutputCounter : Integer;
    FClearCounter     : Integer;
    FFullCounter      : Integer;
  private
    procedure   Recreate;
    procedure   SetMaxSize( aValue: Integer);
    procedure   SetOrder  ( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    MaxSize : Integer read FMaxSize write SetMaxSize;
    property    Order   : Integer read FOrder   write SetOrder;
  end;


  TModAmMod = class( TMod)
  strict private
  const
    i_in       = 0;
    i_mod      = 1;
    i_modamt   = 2;
    i_offset   = 3;
    i_modtype  = 4;
    i_amrm     = 5;
    i_fxmod    = 6;
    i_fxmodamt = 7;
  const
    o_out      = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModModControl = class( TMod)
  strict private
  const
    i_in      = 0;
    i_mod     = 1;
    i_intype  = 2;
    i_modtype = 3;
    i_outtype = 4;
  const
    o_out     = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModShaper = class( TMod)
  strict private
  const
    i_in      = 0;
    i_intype  = 1;
    i_shape   = 2;
    i_outtype = 3;
  const
    o_out     = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModAnalogXor = class( TMod)
  strict private
  const
    i_ina       = 0;
    i_inb       = 1;
    i_intypea   = 2;
    i_intypeb   = 3;
    i_outtype   = 4;
    i_threshold = 5;
    i_tmod      = 6;
    i_tmodamt   = 7;
  const
    o_out       = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModLogisticMap = class( TMod)
  strict private
  const
    i_clock   =  0;
    i_start   =  1;
    i_smod    =  2;
    i_smodamt =  3;
    i_svalue  =  4;
    i_rmod    =  5;
    i_rmodamt =  6;
    i_lowr    =  7;
    i_highr   =  8;
    i_xfade   =  9;
    i_outtype = 10;
    i_mute    = 11;
  const
    o_out     =  0;
  private
    FOldStart   : Boolean;
    FOldClock   : Boolean;
    FTicks      : Integer;
    FPeriodTime : Integer;
    FOutValue   : TSignal;
    FNextValue  : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRotator = class( TMod)
  strict private
  const
    i_angle  = 0;
    i_in1    = 1;
    i_in2    = 2;
    i_sctype = 3;
    i_mode   = 4;
  const
    o_out1   = 0;
    o_out2   = 1;
    o_sin    = 2;
    o_cos    = 3;
  private
    FAngle: TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRotator2 = class( TMod)
  strict private
  const
    i_angle  = 0;
    i_in1    = 1;
    i_in2    = 2;
  const
    o_out1   = 0;
    o_out2   = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMatrixMult = class( TMod)
  strict private
  const
    i_in1    = 0;
    i_in2    = 1;
    i_ina    = 2;
    i_inb    = 3;
    i_inc    = 4;
    i_ind    = 5;
  const
    o_out1   = 0;
    o_out2   = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRectifier = class( TMod)
  strict private
  const
    i_mode   = 0;
    i_in     = 1;
  const
    o_out    = 0;
    o_outinv = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMinMax = class( TMod)
  strict private
  const
    i_in1 = 0;
    i_in2 = 1;
  const
    o_min = 0;
    o_max = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModWavolver = class( TMod)
  // after an idea by Ian Fritz, found at http://ijfritz.byethost4.com/sy_cir6.htm (Wavolver)
  strict private
  const
    i_in             =  0;
    i_inlevel        =  1;
    i_shape          =  2;
    i_signal         =  3;
    i_shapelevel     =  4;
    i_shapeoffset    =  5;
    i_signallevel    =  6;
    i_signaloffset   =  7;
    i_pulseamp       =  8;
    i_pulseampmod    =  9;
    i_pulseampmodlev = 10;
    i_mute           = 11;
  const
    o_out            =  0;
    o_inactive       =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModWaveWiper = class( TMod)
  strict private
  const
    i_break  = 0;
    i_mute   = 1;
    i_in1    = 2;
    i_in2    = 3;
  const
    o_out1   = 0;
    o_out2   = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModWaveWrapper = class( TMod)
  strict private
  const
    i_in        = 0;
    i_mute      = 1;
    i_lowlevel  = 2;
    i_highlevel = 3;
    i_mod       = 4;
    i_modamt    = 5;
    i_type      = 6;
  const
    o_out       = 0;
  const
    m_type1     = 0;
    m_type2     = 1;
    m_type3     = 2;
  private
    FPrevValue  : TSignal;
    FOutValue   : TSignal;
    FStateValue : TSignal;
    FGoingUp    : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModIntDif = class( TMod)
  strict private
  const
    i_in      = 0;
    i_mode    = 1;
    i_gain    = 2;
    i_modemod = 3;
  const
    o_int     = 0;
    o_dif     = 1;
  private
    FSum        : TSignal;
    FPrevSample : TSignal;
    FMode       : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModDif = class( TMod)
  strict private
  const
    i_in      = 0;
    i_mode    = 1;
  const
    o_out     = 0;
  private
    FPrevSample : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModChangeDetector = class( TMod)
  strict private
  const
    i_in      = 0;
    i_mode    = 1;
  const
    o_out     = 0;
    o_trig    = 1;
  private
    FPrevSample   : TSignal;
    FTrigCount    : Integer;
    FTrigLedCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPhasor = class( TMod)
  strict private
  const
    i_frequency  = 0;
    i_fm         = 1;
    i_phase      = 2;
    i_sync       = 3;
    i_pmlevel    = 4;
    i_fmlevel    = 5;
    i_cents      = 6;
    i_freq       = 7;
  const
    o_out        = 0;
  private
    FPhase      : TSignal;
    FPhaseDelta : TSignal;
    FPosition   : TSignal;
    FOldSync    : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModBaseOsc = class( TMod)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_sync       =  3;
    i_mute       =  4;
    i_shape      =  5;
    i_pmlevel    =  6;
    i_fmlevel    =  7;
    i_cents      =  8;
    i_bandlimit  =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_warp       = 12;
    i_warpmod    = 13;
    i_warpmodamt = 14;
  const
    o_out        =  0;
  private
    FSpeed      : TSignal;
    FBreakPoint : TSignal;
    FPhase      : TSignal;
    FPhaseDelta : TSignal;
    FPosition   : TSignal;
    FOldSync    : Boolean;
    FShape      : Integer;
    FBandLimit  : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModModOsc = class( TMod)
  strict private
  const
    i_sync        =  0;
    i_ammod       =  1;
    i_ammodlev    =  2;
    i_amplitude   =  3;
    i_centsmod    =  4;
    i_centsmodlev =  5;
    i_cents       =  6;
    i_pmmod       =  7;
    i_pmmodlev    =  8;
    i_phase       =  9;
    i_freq        = 10;
    i_fmmod       = 11;
    i_fmmodlev    = 12;
    i_frequency   = 13;
    i_speed       = 14;
    i_lfmmod      = 15;
    i_lfmmodlev   = 16;
    i_lfm         = 17;
    i_mute        = 18;
  const
    o_out         =  0;
  private
    FPhase   : TSignal;
    FOldSync : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModPdOsc = class( TMod)
  strict private
  const
    i_mute        =  0;
    i_sync        =  1;
    i_cents       =  2;
    i_centsmod    =  3;
    i_centsmodlev =  4;
    i_freq        =  5;
    i_fmmod       =  6;
    i_fmmodlev    =  7;
    i_frequency   =  8;
    i_speed       =  9;
    i_ratio       = 10;
    i_ratiomod    = 11;
    i_ratiomodlev = 12;
    i_lfmmod      = 13;
    i_lfmmodlev   = 14;
  const
    o_out         =  0;
  private
    FPhaseMain   : TSignal;
    FPhaseFilter : TSignal;
    FOldSync     : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModMasterOsc = class( TMod)
  strict private
  const
    i_centsmod    =  0;
    i_centsmodamt =  1;
    i_cents       =  2;
    i_freq        =  3;
    i_fmmod       =  4;
    i_fmmodlev    =  5;
    i_frequency   =  6;
    i_speed       =  7;
    i_lfmmod      =  8;
    i_lfmmodlev   =  9;
    i_lfm         = 10;
  const
    o_out         =  0;
    o_outexp      =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMultiSine = class( TMod)
  strict private
  const
    i_sync        =  0;
    i_chain       =  1;
    i_freq        =  2;
    i_frequency   =  3;
    i_fmmod       =  4;
    i_fmmodlev    =  5;
    i_cents       =  6;
    i_mute        =  7;
    i_fmult1      =  8;
    i_cents1      =  9;
    i_lmmod1      = 10;
    i_fmmod1      = 11;
    i_ammod1      = 12;
    i_ammodlev1   = 13;
    i_amplitude1  = 14;
    i_mute1       = 15;
    i_fmult2      = 16;
    i_cents2      = 17;
    i_lmmod2      = 18;
    i_fmmod2      = 19;
    i_ammod2      = 20;
    i_ammodlev2   = 21;
    i_amplitude2  = 22;
    i_mute2       = 23;
    i_fmult3      = 24;
    i_cents3      = 25;
    i_lmmod3      = 26;
    i_fmmod3      = 27;
    i_ammod3      = 28;
    i_ammodlev3   = 29;
    i_amplitude3  = 30;
    i_mute3       = 31;
    i_fmult4      = 32;
    i_cents4      = 33;
    i_lmmod4      = 34;
    i_fmmod4      = 35;
    i_ammod4      = 36;
    i_ammodlev4   = 37;
    i_amplitude4  = 38;
    i_mute4       = 39;
    i_fmult5      = 40;
    i_cents5      = 41;
    i_lmmod5      = 42;
    i_fmmod5      = 43;
    i_ammod5      = 44;
    i_ammodlev5   = 45;
    i_amplitude5  = 46;
    i_mute5       = 47;
    i_fmult6      = 48;
    i_cents6      = 49;
    i_lmmod6      = 50;
    i_fmmod6      = 51;
    i_ammod6      = 52;
    i_ammodlev6   = 53;
    i_amplitude6  = 54;
    i_mute6       = 55;
  const
    o_out         =  0;
  const
    INSTRIDE      =  8;
  private
    FPhases  : array[ 0 .. 5] of TSignal;
    FOldSync : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModChladnicOsc = class( TMod)
  strict private
  const
    i_freq        =  0;
    i_frequency   =  1;
    i_fm          =  2;
    i_fmlevel     =  3;
    i_cents       =  4;
    i_mute        =  5;
    i_pm          =  6;
    i_pmlevel     =  7;
    i_p           =  8;
    i_pmod        =  9;
    i_pmodamt     = 10;
    i_speed       = 11;
    i_level1      = 12;
    i_level2      = 13;
    i_level3      = 14;
    i_level4      = 15;
    i_level5      = 16;
    i_level6      = 17;
    i_level7      = 18;
    i_level8      = 19;
    i_level9      = 20;
    i_level10     = 21;
    i_level11     = 22;
  const
    o_out         =  0;
  const
    OSC_COUNT     = 11;
  private
    FOldP    : TSignal;
    FPhases  : array[ 0 .. OSC_COUNT - 1] of TSignal;
    FAddends : array[ 0 .. OSC_COUNT - 1] of TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModBaseTrigOsc = class( TModBaseOsc)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_trig       =  2;
    i_count      =  3;
    i_mute       =  4;
    i_shape      =  5;
    i_pmlevel    =  6;
    i_fmlevel    =  7;
    i_cents      =  8;
    i_bandlimit  =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_warp       = 12;
    i_warpmod    = 13;
    i_warpmodamt = 14;
    i_countmod   = 15;
  const
    o_out        =  0;
  private
    FState   : TTrigOScState;
    FCounter : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModOsc = class( TModBaseOsc)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_sync       =  3;
    i_mute       =  4;
    i_shape      =  5;
    i_pmlevel    =  6;
    i_fmlevel    =  7;
    i_cents      =  8;
    i_bandlimit  =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_warp       = 12;
    i_warpmod    = 13;
    i_warpmodamt = 14;
  const
    o_out        =  0;
  public
    procedure   DoTick;                                                                                        override;
  end;


  TModMultiPhaseOsc = class( TModBaseOsc)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_sync       =  3;
    i_mute       =  4;
    i_shape      =  5;
    i_pmlevel    =  6;
    i_fmlevel    =  7;
    i_cents      =  8;
    i_bandlimit  =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_warp       = 12;
    i_warpmod    = 13;
    i_warpmodamt = 14;
    i_phaseshift = 15;
  const
    o_out        =  0;
    o_out2       =  1;
    o_out3       =  2;
    o_out4       =  3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoTick;                                                                                        override;
  end;


  TModMultiOsc = class( TModBaseOsc)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_sync       =  3;
    i_mute       =  4;
    i_shape      =  5;
    i_pmlevel    =  6;
    i_fmlevel    =  7;
    i_cents      =  8;
    i_bandlimit  =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_warp       = 12;
    i_warpmod    = 13;
    i_warpmodamt = 14;
    i_pwm        = 15;
    i_pwmlevel   = 16;
  const
    o_outsine    =  0;
    o_outtri     =  1;
    o_outsaw     =  2;
    o_outsquare  =  3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModOscTrig = class( TModBaseTrigOsc)
  strict private
  const
    i_frequency   =  0;
    i_fm          =  1;
    i_trig        =  2;
    i_count       =  3;
    i_mute        =  4;
    i_shape       =  5;
    i_pmlevel     =  6;
    i_fmlevel     =  7;
    i_cents       =  8;
    i_bandlimit   =  9;
    i_freq        = 10;
    i_speed       = 11;
    i_countmod    = 12;
  const
    o_out         =  0;
  public
    procedure   DoTick;                                                                                        override;
  end;


  TModSquare = class( TModBaseOsc)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_sync       =  3;
    i_mute       =  4;
    i_shape      =  5;
    i_pmlevel    =  6;
    i_fmlevel    =  7;
    i_cents      =  8;
    i_bandlimit  =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_warp       = 12;
    i_warpmod    = 13;
    i_warpmodamt = 14;
    i_pwm        = 15;
    i_pwmlevel   = 16;
    i_pwmoff     = 17;
    i_pwmplm     = 18;
  const
    o_out        =  0;
  private
    FOldPosition  : TSignal;
    FPulseCount   : Integer;
    FPulseCounter : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModSquareTrig = class( TModBaseTrigOsc)
  strict private
  const
    i_frequency   =  0;
    i_fm          =  1;
    i_phase       =  2;
    i_sync        =  3;
    i_mute        =  4;
    i_shape       =  5;
    i_pmlevel     =  6;
    i_fmlevel     =  7;
    i_cents       =  8;
    i_bandlimit   =  9;
    i_freq        = 10;
    i_speed       = 11;
    i_countmod    = 12;
    i_pwm         = 13;
    i_pwmlevel    = 14;
  const
    o_out         =  0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModVosim = class( TMod)
  strict private
  const
    i_mute        =  0;
    i_trig        =  1;
    i_frequency   =  2;
    i_freq        =  3;
    i_fm          =  4;
    i_fmamt       =  5;
    i_count       =  6;
    i_cm          =  7;
    i_whisper     =  8;
    i_wm          =  9;
    i_wmamt       = 10;
    i_decay       = 11;
    i_dm          = 12;
    i_dmamt       = 13;
    i_amplitude   = 14;
    i_am          = 15;
    i_amamt       = 16;
    i_speed       = 17;
  const
    o_out         =  0;
  private
    FPhaseDelta : TSignal;
    FPhase      : TSignal;
    FState      : TTrigOScState;
    FCounter    : Integer;
    FLevel      : TSignal;
    FWhisper    : TSignal;
    FDecay      : TSignal;
    FMaxCount   : TSignal;
    FSpeed      : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModBaseLfo = class( TMod)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_res        =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_freq       = 10;
    i_speed      = 11;
  const
    o_out        =  0;
    o_sync       =  1;
  private
    FSpeed       : TSignal;
    FPhase       : TSignal;
    FPosition    : TSignal;
    FOldRes      : Boolean;
    FShape       : Integer;
    FWrapped     : Boolean;
    FRandomValue : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModBaseTrigLfo = class( TModBaseLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_count      =  2;
    i_trig       =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_mode       = 10;
    i_speed      = 11;
    i_countmod   = 12;
    i_autoready  = 13;
  const
    o_out        =  0;
    o_ready      =  1;
    o_active     =  2;
  private
    FState    : TTrigOScState;
    FCounter  : Integer;
    FDidRun   : Boolean;
    FPrevTrig : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModCounterLfo = class( TMod)
  strict private
  const
    i_frequency  = 0;
    i_fm         = 1;
    i_fmlevel    = 2;
    i_range      = 3;
    i_mute       = 4;
    i_outputtype = 5;
    i_freq       = 6;
    i_speed      = 7;
  private
    FPhase : TSignal;
    FSpeed : TSignal;
  protected
    procedure   Wrapped;                                                                              virtual; abstract;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModClockGen = class( TMod)
  strict private
  const
    i_reset   = 0;
    i_tempo   = 1;
    i_tempoex = 2;
    i_swing   = 3;
    i_mute    = 4;
    i_div     = 5;
    i_speed   = 6;
  const
    o_16      = 0;
    o_96      = 1;
    o_sync    = 2;
    o_phase16 = 3;
    o_phase96 = 4;
  private
    FPhase16     : TSignal; // Phase acuumulator for * 16 output
    FPhase96     : TSignal; // Phase acuumulator for * 96 output
    FOldRes      : Boolean; // Old reset value
    FOldMute     : Boolean; // Old mute state
    FEven        : Boolean; // Even / Odd cycle for * 16 output
    FState       : Boolean; // * 96 state
    FSwing       : TSignal; // Current Swing value
    FDividerSync : Integer; // Divider for sync output
    FResCount    : Integer; // Reset LED counter
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModTod = class( TMod)
  strict private
  const
    o_out   = 0;
    o_outh  = 1;
    o_outm  = 2;
    o_outs  = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModLfo = class( TModBaseLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_res        =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_freq       = 10;
    i_speed      = 11;
  const
    o_out        =  0;
    o_sync       =  1;
  public
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModLfoMultiPhase = class( TModBaseLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_res        =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_phaseshift = 12;
  const
    o_out        =  0;
    o_sync       =  1;
    o_out2       =  2;
    o_out3       =  3;
    o_out4       =  4;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModLfoTrig = class( TModBaseTrigLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_count      =  2;
    i_trig       =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_mode       = 10;
    i_speed      = 11;
    i_countmod   = 12;
    i_autoready  = 13;
  const
    o_out        =  0;
    o_ready      =  1;
  public
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModSquareLfo = class( TModBaseLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_res        =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_pwm        = 12;
    i_pwmlevel   = 13;
  const
    o_out        =  0;
    o_sync       =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModSquareSineLfo = class( TModSquareLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_phase      =  2;
    i_res        =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_freq       = 10;
    i_speed      = 11;
    i_pwm        = 12;
    i_pwmlevel   = 13;
  const
    o_out        =  0;
    o_sync       =  1;
    o_sine       =  2;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModSquareLfoTrig = class( TModBaseTrigLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_count      =  2;
    i_trig       =  3;
    i_mute       =  4;
    i_fmlevel    =  5;
    i_pmlevel    =  6;
    i_shape      =  7;
    i_range      =  8;
    i_outputtype =  9;
    i_mode       = 10;
    i_speed      = 11;
    i_countmod   = 12;
    i_autoready  = 13;
    i_pwm        = 14;
    i_pwmlevel   = 15;
  const
    o_out        =  0;
    o_ready      =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRandSig = class( TModCounterLfo)
  strict private
  const
    i_frequency  =  0;
    i_fm         =  1;
    i_fmlevel    =  2;
    i_range      =  3;
    i_mute       =  4;
    i_outputtype =  5;
    i_freq       =  6;
    i_speed      =  7;
    i_dist       =  8;
    i_distmod    =  9;
    i_distmodamt = 10;
    i_lambda     = 11;
    i_mode       = 12;
  const
    o_out        =  0;
  const
    m_lin        =  0;
    m_exp        =  1;
  private
    FStart  : TSignal;
    FTarget : TSignal;
  protected
    procedure   Wrapped;                                                                                       override;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRandSigs = class( TMod)
  strict private
  const
    i_frequency    =  0;
    i_fm           =  1;
    i_fmlevel      =  2;
    i_range        =  3;
    i_mute         =  4;
    i_outputtype   =  5;
    i_freq         =  6;
    i_speed        =  7;
    i_dist         =  8;
    i_distmod      =  9;
    i_distmodamt   = 10;
    i_lambda       = 11;
    i_mode         = 12;
    i_distfb       = 13;
    i_distfbmod    = 14;
    i_distfbmodamt = 15;
    i_fmfb         = 16;
    i_fmfbmod      = 17;
    i_fmfbmodamt   = 18;
  const
    o_out1        =  0;
    o_out2        =  1;
    o_out3        =  2;
    o_out4        =  3;
    o_out5        =  4;
    o_out6        =  5;
    o_out7        =  6;
    o_out8        =  7;
    o_outinv1     =  8;
    o_outinv2     =  9;
    o_outinv3     = 10;
    o_outinv4     = 11;
    o_outinv5     = 12;
    o_outinv6     = 13;
    o_outinv7     = 14;
    o_outinv8     = 15;
  const
    m_lin         =  0;
    m_exp         =  1;
  const
    OUT_COUNT     =  8;
  private
    FPhases  : array[ 0 .. OUT_COUNT - 1] of TSignal;
    FStarts  : array[ 0 .. OUT_COUNT - 1] of TSignal;
    FTargets : array[ 0 .. OUT_COUNT - 1] of TSignal;
  protected
    procedure   Wrapped( anIndex: Integer);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPulses = class( TMod)
  strict private
  const
    i_frequency      = 0;
    i_range          = 1;
    i_dist           = 2;
    i_distmod        = 3;
    i_distmodamt     = 4;
    i_lambda         = 5;
    i_rndmode        = 6;
    i_syncmode       = 7;
    i_outputtype     = 8;
  const
    o_out            = 0;
  const
    m_rnd_lin        = 0;
    m_rnd_exp        = 1;
  const
    m_sync_async     = 0;
    m_sync_sync      = 1;
  private
    FPhase         : TSignal;
    FState         : TPulsesState;
    FPulseCounter  : Integer;
    FPeriodCounter : Integer;
    FLedCounter    : Integer;
    FOldFrequency  : TSignal;
    FOldLambda     : TSignal;
    FOldDist       : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModRandomWalkLfo = class( TMod)
  strict private
  const
    i_range       =  0;
    i_frequency   =  1;
    i_fm          =  2;
    i_fmlevel     =  3;
    i_distance    =  4;
    i_mute        =  5;
    i_color       =  6;
    i_colormod    =  7;
    i_colormodamt =  8;
    i_outputtype  =  9;
    i_lambda      = 10;
    i_speed       = 11;
    i_mode        = 12;
    i_smode       = 13;
    i_distancemod = 14;
  const
    o_outx        =  0;
    o_outy        =  1;
    o_outz        =  2;
    o_sync        =  3;
  const
    m_lin         =  0;
    m_exp         =  0;
  private
    FPosition : TSignal;
    FState    : TRWState;
    FX        : TSignal;
    FY        : TSignal;
    FZ        : TSignal;
    FPrevX    : TSignal;
    FPrevY    : TSignal;
    FPrevZ    : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModAttractorLfo = class( TMod)
  strict private
  const
    i_range       =  0;
    i_frequency   =  1;
    i_fm          =  2;
    i_fmlevel     =  3;
    i_mute        =  4;
    i_outputtype  =  5;
    i_mode        =  6;
    i_drive       =  7;
    i_driveamt    =  8;
  const
    o_outx        =  0;
    o_outy        =  1;
    o_outz        =  2;
  const
    m_lorenz      = 0;
    m_rossler     = 1;
  private
    FLorenz    : TLorenzOsc;
    FRossler   : TRosslerOsc;
    FX         : TSignal;
    FY         : TSignal;
    FZ         : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModVanDerPolLfo = class( TMod)
  strict private
  const
    i_range       =  0;
    i_frequency   =  1;
    i_fm          =  2;
    i_fmlevel     =  3;
    i_mute        =  4;
    i_outputtype  =  5;
    i_mu          =  6;
    i_mumodamt    =  7;
    i_mumod       =  8;
    i_in          =  9;
    i_inamt       = 10;
  const
    o_outx        =  0;
    o_outy        =  1;
  private
    FVanDerPol : TVanDerPolOsc;
    FX         : TSignal;
    FY         : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModNoise = class( TMod)
  strict private
  const
    i_color       = 0;
    i_colormod    = 1;
    i_colormodamt = 2;
    i_mute        = 3;
    i_type        = 4;
  const
    o_out         = 0;
  const
    nt_lin        = 0;
    nt_gauss      = 1;
    nt_exp        = 2;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModNoiseTrig = class( TMod)
  strict private
  const
    i_color       = 0;
    i_colormod    = 1;
    i_colormodamt = 2;
    i_trig        = 3;
    i_mute        = 4;
    i_type        = 5;
  const
    o_out         = 0;
  const
    nt_lin        = 0;
    nt_gauss      = 1;
    nt_exp        = 2;
  private
    FPrevTrig: Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModNoiseLfo = class( TMod)
  strict private
  const
    i_color       = 0;
    i_colormod    = 1;
    i_colormodamt = 2;
    i_mute        = 3;
    i_outputtype  = 4;
    i_type        = 5;
  const
    o_out         = 0;
  const
    nt_lin        = 0;
    nt_gauss      = 1;
    nt_exp        = 2;
  private
    FOutValue : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModNoiseLfoTrig = class( TMod)
  strict private
  const
    i_color       = 0;
    i_colormod    = 1;
    i_colormodamt = 2;
    i_trig        = 3;
    i_mute        = 4;
    i_outputtype  = 5;
    i_type        = 6;
  const
    o_out         = 0;
  const
    nt_lin        = 0;
    nt_gauss      = 1;
    nt_exp        = 2;
  private
    FPrevTrig : Boolean;
    FOutValue : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPerlinTone = class( TMod)
  strict private
  const
    i_roughness    =  0;
    i_new          =  1;
    i_newinv       =  2;
    i_mute         =  3;
    i_roughmod     =  4;
    i_length       =  5;
    i_lengthmod    =  6;
    i_lengthmodamt =  7;
    i_frequency    =  8;
    i_freq         =  9;
    i_fm           = 10;
    i_fmlevel      = 11;
    i_cents        = 12;
  const
    o_out          =  0;
  private
    FNoise      : TPerlinNoise;
    FLayerCount : Integer;
    FPhase      : TSignal;
  private
    procedure   Recreate;
    procedure   SetLayerCount( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  public
    property    LayerCount : Integer read FLayerCount write SetLayerCount;
  end;


  TModPerlinNoise = class( TMod)
  strict private
  const
    i_roughness    = 0;
    i_new          = 1;
    i_newinv       = 2;
    i_mute         = 3;
    i_roughmod     = 4;
    i_length       = 5;
    i_lengthmod    = 6;
    i_lengthmodamt = 7;
  const
    o_out          = 0;
  private
    FNoise      : TPerlinNoise;
    FLayerCount : Integer;
  private
    procedure   Recreate;
    procedure   SetLayerCount( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  public
    property    LayerCount : Integer read FLayerCount write SetLayerCount;
  end;


  TModPerlinNoiseTrig = class( TMod)
  strict private
  const
    i_roughness    = 0;
    i_new          = 1;
    i_newinv       = 2;
    i_mute         = 3;
    i_roughmod     = 4;
    i_length       = 5;
    i_lengthmod    = 6;
    i_lengthmodamt = 7;
    i_trig         = 8;
  const
    o_out          = 0;
  private
    FNoise      : TPerlinNoise;
    FOldTrig    : Boolean;
    FLayerCount : Integer;
  private
    procedure   Recreate;
    procedure   SetLayerCount( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  public
    property    LayerCount : Integer read FLayerCount write SetLayerCount;
  end;


  TModAverage = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  strict private
  const
    i_in          = 0;
    i_alpha       = 1;
    i_alphamod    = 2;
    i_alphamodamt = 3;
  const
    o_lp          = 0;
    o_hp          = 1;
  private
    FAlpha : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModUnclick = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha))) - with a fixed alpha
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  strict private
  const
    i_in  = 0;
  const
    o_out = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMovingAverage = class( TMod)
  strict private
  const
    i_in   = 0;
    i_size = 1;
  const
    o_out  = 0;
  const
    MAX_SIZE = 1024;
  private
    FSize    : Integer;
    FPointer : Integer;
    FHistory : array[ 0 .. MAX_SIZE - 1] of TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModDCBlock = class( TMod)
  strict private
  const
    i_in    = 0;
    i_alpha = 1;
  const
    o_out   = 0;
  private
    FPrevIn : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModFilter6dB = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  strict private
  const
    i_in          = 0;
    i_freq        = 1;
    i_freqmod     = 2;
    i_freqmodamt  = 3;
  const
    o_lp          = 0;
    o_hp          = 1;
  private
    FFreq  : TSignal;
    FAlpha : TSignal;
  private
    procedure   CalcAlpha;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModFilter6dBBP = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  strict private
  const
    i_in          = 0;
    i_freqhp      = 1;
    i_freqlp      = 2;
  const
    o_out         = 0;
  private
    FFreqHP  : TSignal;
    FFreqLP  : TSignal;
    FAlphaHP : TSignal;
    FAlphaLP : TSignal;
    FInp     : TSignal;
    FMid     : TSignal;
  private
    procedure   CalcAlphaHP;
    procedure   CalcAlphaLP;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModFilter6dBBPS = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  strict private
  const
    i_inl         = 0;
    i_inr         = 1;
    i_freqhp      = 2;
    i_freqlp      = 3;
  const
    o_outl        = 0;
    o_outr        = 1;
  private
    FFreqHP  : TSignal;
    FFreqLP  : TSignal;
    FAlphaHP : TSignal;
    FAlphaLP : TSignal;
    FInpl    : TSignal;
    FMidl    : TSignal;
    FInpr    : TSignal;
    FMidr    : TSignal;
  private
    procedure   CalcAlphaHP;
    procedure   CalcAlphaLP;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModFilter6dBBPSM = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  strict private
  const
    i_inl         = 0;
    i_inr         = 1;
    i_freqhp      = 2;
    i_freqlp      = 3;
    i_freqmod     = 4;
    i_freqmodamt  = 5;
  const
    o_outl        = 0;
    o_outr        = 1;
  private
    FFreqHP  : TSignal;
    FFreqLP  : TSignal;
    FAlphaHP : TSignal;
    FAlphaLP : TSignal;
    FInpl    : TSignal;
    FMidl    : TSignal;
    FInpr    : TSignal;
    FMidr    : TSignal;
  private
    procedure   CalcAlphaHP;
    procedure   CalcAlphaLP;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModSVF = class( TMod)
  strict private
  const
    i_in         =  0;
    i_frequency  =  1;
    i_q          =  2;
    i_distortion =  3;
    i_fm         =  4;
    i_fmlevel    =  5;
    i_inlevel    =  6;
    i_freq       =  7;
    i_qm         =  8;
    i_qmamtp     =  9;
    i_qmamtn     = 10;
  const
    o_lp         =  0;
    o_bp         =  1;
    o_hp         =  2;
    o_br         =  3;
  private
    FFreq  : TSignal;
    FDamp  : TSignal;
    FDrive : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModMoogFilter = class( TMod)
  strict private
  const
    i_in         =  0;
    i_frequency  =  1;
    i_q          =  2;
    i_fm         =  3;
    i_fmlevel    =  4;
    i_inlevel    =  5;
    i_freq       =  6;
    i_qm         =  7;
    i_qmamtp     =  8;
    i_qmamtn     =  9;
    i_linfm      = 10;
    i_linfmamt   = 11;
  const
    o_out        =  0;
  private
    FFilterClass : TMoogFilterClass;
    FFilter      : TMoogFilter;
  private
    procedure   Recreate;
    procedure   SetFilterClass( aValue: TMoogFilterClass);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    FilterClass : TMoogFilterClass read FFilterClass write SetFilterClass;
  end;


  TModPinkFilter = class( TMod)
  strict private
  const
    i_in   = 0;
    i_mode = 1;
    i_gain = 2;
  const
    o_out  = 0;
  private
    FB0 : TSignal;
    FB1 : TSignal;
    FB2 : TSignal;
    FB3 : TSignal;
    FB4 : TSignal;
    FB5 : TSignal;
    FB6 : TSignal;
    FG  : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModResonator = class( TMod)
  // http://www.katjaas.nl/complexintegrator/complexresonator.html
  strict private
  const
    i_freq      =  0;
    i_decay     =  1;
    i_in        =  2;
    i_fm        =  3;
    i_fmlevel   =  4;
    i_frequency =  5;
    i_cents     =  6;
    i_pm        =  7;
    i_pmlevel   =  8;
    i_dm        =  9;
    i_dmlevel   = 10;
    i_click     = 11;
    i_inlevel   = 12;
  const
    o_out       =  0;
  private
    FOldRe  : TSignal;
    FOldIm  : TSignal;
    FRadius : TSignal;
    FSin    : TSignal;
    FCos    : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModFreqShifter = class( TMod)
  // https://github.com/swh/ladspa/blob/master/bode_shifter_1431.xml
  strict private
  const
    i_in            = 0;
    i_shift         = 1;
    i_shiftmod      = 2;
    i_shiftdepth    = 3;
    i_shiftdepthamt = 4;
  const
    o_up            = 0;
    o_down          = 1;
  private
    FDelay : array[ 0 .. 255] of TSignal;
    FDPtr  : Integer;
    FPhase : TSignal;
    FDelta : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModTiltFilter = class( TMod)
  // source: http://www.musicdsp.org/showone.php?id=267
  strict private
  const
    AMP     = LOG_TWO / 6;
    GFACTOR = 5.0;
  const
    i_in         = 0;
    i_inlevel    = 1;
    i_tilt       = 2;
    i_tiltmod    = 3;
    i_tiltmodamt = 4;
    i_freq       = 5;
    i_fmmod      = 6;
    i_fmmodamt   = 7;
  const
    o_out        = 0;
  private
    FLGain   : TSignal;
    FHGain   : TSignal;
    FLpOut   : TSignal;
    FA0      : TSignal;
    FB1      : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModTiltFilterStereo = class( TMod)
  strict private
  const
    AMP     = LOG_TWO / 6;
    GFACTOR = 5.0;
  const
    i_in         = 0;
    i_in2        = 1;
    i_inlevel    = 2;
    i_tilt       = 3;
    i_tiltmod    = 4;
    i_tiltmodamt = 5;
    i_freq       = 6;
    i_fmmod      = 7;
    i_fmmodamt   = 8;
  const
    o_out        = 0;
    o_out2       = 1;
  private
    FLGain   : TSignal;
    FHGain   : TSignal;
    FLpOut   : TSignal;
    FLpOut2  : TSignal;
    FA0      : TSignal;
    FB1      : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModEnvAR = class( TMod)
  strict private
  const
    i_trig    = 0;
    i_attack  = 1;
    i_release = 2;
    i_gain    = 3;
    i_sigin   = 4;
    i_mute    = 5;
    i_hold    = 6;
    i_amod    = 7;
    i_rmod    = 8;
  const
    o_out     = 0;
    o_inv     = 1;
    o_sigout  = 2;
  private
    FState     : TARState;
    FValue     : TSignal;
    FEnvValue  : TSignal;
    FTrigCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModEnvARRetrig = class( TMod)
  strict private
  const
    i_trig    = 0;
    i_attack  = 1;
    i_release = 2;
    i_gain    = 3;
    i_sigin   = 4;
    i_mute    = 5;
    i_hold    = 6;
    i_amod    = 7;
    i_rmod    = 8;
  const
    o_out     = 0;
    o_inv     = 1;
    o_sigout  = 2;
  private
    FState     : TARState;
    FValue     : TSignal;
    FArmed     : Boolean;
    FHoldMode  : Boolean;
    FEnvValue  : TSignal;
    FTrigCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModEnvAHD = class( TMod)
  strict private
  const
    i_trig          =  0;
    i_mode          =  1;
    i_ashape        =  2;
    i_dshape        =  3;
    i_attack        =  4;
    i_hold          =  5;
    i_decay         =  6;
    i_gain          =  7;
    i_sigin         =  8;
    i_mute          =  9;
    i_arange        = 10;
    i_hrange        = 11;
    i_drange        = 12;
    i_amod          = 13;
    i_hmod          = 14;
    i_dmod          = 15;
    i_amodneg       = 16;
    i_hmodneg       = 17;
    i_dmodneg       = 18;
    i_amodlevel     = 19;
    i_hmodlevel     = 20;
    i_dmodlevel     = 21;
    i_outputtype    = 22;
    i_holdmode      = 23;
    i_swell         = 24;
    i_swellmod      = 25;
    i_swellmodneg   = 26;
    i_swellmodlevel = 27;
    i_swellshape    = 28;
  const
    o_out           =  0;
    o_sigout        =  1;
    o_eoc           =  2;
    o_inv           =  3;
  private
    FData      : TSignalArray;
    FState     : TAHDState;
    FTime      : Integer;
    FDuration  : Integer;
    FArmed     : Boolean;
    FPrevLevel : TSignal;
    FValue     : TSignal;
    FEnvValue  : TSignal;
    FTrigFlag  : Boolean;
    FTrigCount : Integer;
    FEocCount  : Integer;
    FEoCLed    : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherData  ( const aPrefix: string; const aCallback: TDataHandler);                           override;
  end;


  TModEnvADSR = class( TMod)
  strict private
  const
    i_trig          =  0;
    i_mode          =  1;
    i_ashape        =  2;
    i_dshape        =  3;
    i_rshape        =  4;
    i_attack        =  5;
    i_decay         =  6;
    i_sustain       =  7;
    i_release       =  8;
    i_gain          =  9;
    i_sigin         = 10;
    i_mute          = 11;
    i_arange        = 12;
    i_drange        = 13;
    i_rrange        = 14;
    i_amod          = 15;
    i_dmod          = 16;
    i_smod          = 17;
    i_rmod          = 18;
    i_amodneg       = 19;
    i_dmodneg       = 20;
    i_smodneg       = 21;
    i_rmodneg       = 22;
    i_amodlevel     = 23;
    i_dmodlevel     = 24;
    i_smodlevel     = 25;
    i_rmodlevel     = 26;
    i_outputtype    = 27;
    i_swell         = 28;
    i_swellmod      = 29;
    i_swellmodneg   = 30;
    i_swellmodlevel = 31;
    i_swellshape    = 32;
  const
    o_out           =  0;
    o_sigout        =  1;
    o_eoc           =  2;
    o_inv           =  3;
  private
    FData      : TSignalArray;
    FState     : TADSRState;
    FTime      : Integer;
    FDuration  : Integer;
    FArmed     : Boolean;
    FPrevLevel : TSignal;
    FValue     : TSignal;
    FEnvValue  : TSignal;
    FTrigFlag  : Boolean;
    FTrigCount : Integer;
    FEocCount  : Integer;
    FEoCLed    : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherData  ( const aPrefix: string; const aCallback: TDataHandler);                           override;
  end;


  TModEnvControl = class( TMod)
  strict private
  const
    i_trig    = 0;
    i_attack  = 1;
    i_release = 2;
    i_level   = 3;
    i_mute    = 4;
    i_hold    = 5;
  const
    o_out     = 0;
    o_eoc     = 1;
  private
    FLevel        : TSignal;
    FTrig         : Boolean;
    FState        : TARState;
    FValue        : TSignal;
    FEnvValue     : TSignal;
    FTrigCount    : Integer;
    FArmed        : Boolean;
    FCoeffAttack  : TSignal;
    FCoeffRelease : TSignal;
    FEocCount     : Integer;
    FEoCLed       : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSequencer = class( TMod)
  strict private
  const
    i_trig   =  0;
    i_reset  =  1;
    i_dir    =  2;
    i_chain  =  3;
    i_rnd    =  4;
    i_active =  5;
    i_step1  =  6;   // Keep steps consecutive
    i_step2  =  7;
    i_step3  =  8;
    i_step4  =  9;
    i_step5  = 10;
    i_step6  = 11;
    i_step7  = 12;
    i_step8  = 13;
    i_step9  = 14;
    i_step10 = 15;
    i_step11 = 16;
    i_inv    = 17;
  const
    o_out    =  0;   // Keep trigs comsecutive
    o_trig1  =  1;
    o_trig2  =  2;
    o_trig3  =  3;
    o_trig4  =  4;
    o_trig5  =  5;
    o_trig6  =  6;
    o_trig7  =  7;
    o_trig8  =  8;
    o_trig9  =  9;
    o_trig10 = 10;
    o_trig11 = 11;
  private
    FOutValue : TSignal;
    FSteps    : Integer;
    FCounter  : Integer;
    FInState  : TSeqInState;
    FOldRes   : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSeqSeq = class( TMod)
  strict private
  const
    i_res      =  0;
    i_trig     =  1;
    i_dir      =  2;
    i_resmod   =  3;
  const
    o_outres   =  0;
    o_outtrig  =  1;
    o_outdir   =  2;
    o_out1     =  3;
    o_out2     =  4;
    o_out3     =  5;
    o_out4     =  6;
    o_out5     =  7;
    o_out6     =  8;
    o_out7     =  9;
    o_out8     = 10;
    o_out9     = 11;
    o_out10    = 12;
    o_out11    = 13;
    o_out12    = 14;
    o_out13    = 15;
    o_out14    = 16;
    o_out15    = 17;
    o_out16    = 18;
  const
    c_clkdelay = 17;
  const
    m_res1     =  0;
    m_res2     =  1;
    m_res3     =  2;
  private
    FState           : Word;
    FSteps           : Integer;
    FSequenceCounter : Integer;
    FPeriodCounter   : Integer;
    FOldRes          : Boolean;
    FOldTrig         : Boolean;
    FClkCnt          : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSeq16 = class( TMod)
  strict private
  const
    i_trig       =  0;
    i_res        =  1;
    i_dir        =  2;
    i_chain      =  3;
    i_rnd        =  4;
    i_active     =  5;
    i_mode       =  6;
    i_curveup    =  7;
    i_curvedown  =  8;
    i_xfade      =  9;
    i_step1      = 10;   // Keep steps consecutive
    i_step2      = 11;
    i_step3      = 12;
    i_step4      = 13;
    i_step5      = 14;
    i_step6      = 15;
    i_step7      = 16;
    i_step8      = 17;
    i_step9      = 18;
    i_step10     = 19;
    i_step11     = 20;
    i_step12     = 21;
    i_step13     = 22;
    i_step14     = 23;
    i_step15     = 24;
    i_step16     = 25;
    i_xfademod   = 26;
    i_inv        = 27;
    i_rnd2       = 28;
    i_hold       = 29;
    i_skip1      = 30;   // Keep skips consecutive
    i_skip2      = 31;
    i_skip3      = 32;
    i_skip4      = 33;
    i_skip5      = 34;
    i_skip6      = 35;
    i_skip7      = 36;
    i_skip8      = 37;
    i_skip9      = 38;
    i_skip10     = 39;
    i_skip11     = 40;
    i_skip12     = 41;
    i_skip13     = 42;
    i_skip14     = 43;
    i_skip15     = 44;
    i_skip16     = 45;
    i_clockskip  = 46;
    i_gatemode   = 47;
    i_rnd3       = 48;
  const
    o_out        =  0;
    o_outres     =  1;
    o_outtrig    =  2;
    o_outdir     =  3;
    o_out2       =  4;
    o_out3       =  5;
    o_out4       =  6;
    o_stepout1   =  7;
    o_stepout2   =  8;
    o_stepout3   =  9;
    o_stepout4   = 10;
    o_stepout5   = 11;
    o_stepout6   = 12;
    o_stepout7   = 13;
    o_stepout8   = 14;
    o_stepout9   = 15;
    o_stepout10  = 16;
    o_stepout11  = 17;
    o_stepout12  = 18;
    o_stepout13  = 19;
    o_stepout14  = 20;
    o_stepout15  = 21;
    o_stepout16  = 22;
    o_timewarp   = 23;
    o_outraw     = 24;
    o_outraw2    = 25;
    o_outraw3    = 26;
    o_outraw4    = 27;
  const
    m_lin44      =  0;   // linear [-4, 4]
    m_db4        =  1;   // exp    [-MAX_DB_VAL, 12] dB
    m_lin11      =  2;   // linear [-1, 1]
    m_db0        =  3;   // exp    [-MAX_DB_VAL, 0] dB
    m_lin04      =  4;   // linear [0, 4]
    m_lin01      =  5;   // linear [0, 1]
    m_note       =  6;   // note values ([0,127] nn - MiddleNote) / NOTE_SCALING
  private
    FOutValue   : TSignal;
    FOutValue2  : TSignal;
    FOutValue3  : TSignal;
    FOutValue4  : TSignal;
    FReset      : Boolean;
    FNextValue  : TSignal;
    FNextValue2 : TSignal;
    FNextValue3 : TSignal;
    FNextValue4 : TSignal;
    FForwards   : Boolean;
    FActive     : Boolean;
    FChainValue : TSignal;
    FPreSteps   : Integer;
    FSteps      : Integer;
    FCounter    : Integer;
    FInState    : TSeqInState;
    FOldRes     : Boolean;
    FTicks      : Integer;
    FPeriodTime : Integer;
    FInvert     : Boolean;
    FRandom     : Boolean;
    FOldRandom  : Boolean;
    FRndReq     : Boolean;
    FRandom3    : Boolean;
    FOldRandom3 : Boolean;
    FRndReq3    : Boolean;
    FFreeze     : Boolean;
    FOldTrig    : Boolean;
    FGateMode   : Boolean;
    FSkips      : set of 0 .. 15;
    FSkipCount  : Integer;
    FTimeWarp   : TSignal;
  private
    procedure   SampleInputs;
    procedure   MakeNextValue;
    procedure   FixSkips;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights ( const aPrefix: string; const aCallback: TLightsHandler);                        override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
  end;


  TModVcps = class( TMod)
  // Voltage controlled pattern sequence, after : http://electro-music.com/forum/topic-37829.html
  strict private
  const
    i_trig       =  0;
    i_res        =  1;
    i_dir        =  2;
    i_inv        =  3;
    i_pattern    =  4;
    i_steps      =  5;
    i_stepsmod   =  6;
    i_mode       =  7;
    i_length     =  8;
    i_start      =  9;
    i_rnd2       = 10;
    i_rnd3       = 11;
    i_clr2       = 12;
    i_step1      = 13;   // Keep steps consecutive
    i_step2      = 14;
    i_step3      = 15;
    i_step4      = 16;
    i_step5      = 17;
    i_step6      = 18;
    i_step7      = 19;
    i_step8      = 20;
    i_step9      = 21;
    i_step10     = 22;
    i_step11     = 23;
    i_step12     = 24;
    i_step13     = 25;
    i_step14     = 26;
    i_step15     = 27;
    i_step16     = 28;
    i_stepmode1  = 29;   // Keep stepmodes consecutive
    i_stepmode2  = 30;
    i_stepmode3  = 31;
    i_stepmode4  = 32;
    i_stepmode5  = 33;
    i_stepmode6  = 34;
    i_stepmode7  = 35;
    i_stepmode8  = 36;
    i_stepmode9  = 37;
    i_stepmode10 = 38;
    i_stepmode11 = 39;
    i_stepmode12 = 40;
    i_stepmode13 = 41;
    i_stepmode14 = 42;
    i_stepmode15 = 43;
    i_stepmode16 = 44;
  const
    o_out        =  0;
    o_outgate    =  1;
    o_outtrig    =  2;
  const
    m_lin44      =  0;   // linear [-4, 4]
    m_db4        =  1;   // exp    [-MAX_DB_VAL, 12] dB
    m_lin11      =  2;   // linear [-1, 1]
    m_db0        =  3;   // exp    [-MAX_DB_VAL, 0] dB
    m_lin04      =  4;   // linear [0, 4]
    m_lin01      =  5;   // linear [0, 1]
    m_note       =  6;   // note values ([0,127] nn - MiddleNote) / NOTE_SCALING
  private
    FOutValue    : TSignal;
    FPattern     : TSignal;
    FLength      : Integer;
    FStart       : Integer;
    FReset       : Boolean;
    FForwards    : Boolean;
    FSteps       : Integer;
    FCounter     : Integer;
    FInState     : TSeqInState;
    FOldRes      : Boolean;
    FInvert      : Boolean;
    FClear       : Boolean;
    FOldClear    : Boolean;
    FClrReq      : Boolean;
    FRandom      : Boolean;
    FOldRandom   : Boolean;
    FRndReq      : Boolean;
    FRandom3     : Boolean;
    FOldRandom3  : Boolean;
    FRndReq3     : Boolean;
    FOldTrig     : Boolean;
    FActiveSteps : Word;
  private
    procedure   SampleAdc;
    procedure   SampleInputs;
    procedure   MakeNextValue;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights ( const aPrefix: string; const aCallback: TLightsHandler);                        override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
  end;


  TModSeq32 = class( TMod)
  strict private
  const
    i_note11      =   0;  i_note12      =   1;  i_note13      =   2;  i_note14      =   3;
    i_note15      =   4;  i_note16      =   5;  i_note17      =   6;  i_note18      =   7;
    i_note19      =   8;  i_note110     =   9;  i_note111     =  10;  i_note112     =  11;
    i_note113     =  12;  i_note114     =  13;  i_note115     =  14;  i_note116     =  15;
    i_note117     =  16;  i_note118     =  17;  i_note119     =  18;  i_note120     =  19;
    i_note121     =  20;  i_note122     =  21;  i_note123     =  22;  i_note124     =  23;
    i_note125     =  24;  i_note126     =  25;  i_note127     =  26;  i_note128     =  27;
    i_note129     =  28;  i_note130     =  29;  i_note131     =  30;  i_note132     =  31;
    i_note21      =  32;  i_note22      =  33;  i_note23      =  34;  i_note24      =  35;
    i_note25      =  36;  i_note26      =  37;  i_note27      =  38;  i_note28      =  39;
    i_note29      =  40;  i_note210     =  41;  i_note211     =  42;  i_note212     =  43;
    i_note213     =  44;  i_note214     =  45;  i_note215     =  46;  i_note216     =  47;
    i_note217     =  48;  i_note218     =  49;  i_note219     =  50;  i_note220     =  51;
    i_note221     =  52;  i_note222     =  53;  i_note223     =  54;  i_note224     =  55;
    i_note225     =  56;  i_note226     =  57;  i_note227     =  58;  i_note228     =  59;
    i_note229     =  60;  i_note230     =  61;  i_note231     =  62;  i_note232     =  63;
    i_note31      =  64;  i_note32      =  65;  i_note33      =  66;  i_note34      =  67;
    i_note35      =  68;  i_note36      =  69;  i_note37      =  70;  i_note38      =  71;
    i_note39      =  72;  i_note310     =  73;  i_note311     =  74;  i_note312     =  75;
    i_note313     =  76;  i_note314     =  77;  i_note315     =  78;  i_note316     =  79;
    i_note317     =  80;  i_note318     =  81;  i_note319     =  82;  i_note320     =  83;
    i_note321     =  84;  i_note322     =  85;  i_note323     =  86;  i_note324     =  87;
    i_note325     =  88;  i_note326     =  89;  i_note327     =  90;  i_note328     =  91;
    i_note329     =  92;  i_note330     =  93;  i_note331     =  94;  i_note332     =  95;
    i_octave11    =  96;  i_octave12    =  97;  i_octave13    =  98;  i_octave14    =  99;
    i_octave15    = 100;  i_octave16    = 101;  i_octave17    = 102;  i_octave18    = 103;
    i_octave19    = 104;  i_octave110   = 105;  i_octave111   = 106;  i_octave112   = 107;
    i_octave113   = 108;  i_octave114   = 109;  i_octave115   = 110;  i_octave116   = 111;
    i_octave117   = 112;  i_octave118   = 113;  i_octave119   = 114;  i_octave120   = 115;
    i_octave121   = 116;  i_octave122   = 117;  i_octave123   = 118;  i_octave124   = 119;
    i_octave125   = 120;  i_octave126   = 121;  i_octave127   = 122;  i_octave128   = 123;
    i_octave129   = 124;  i_octave130   = 125;  i_octaveb131   = 126;  i_octave132   = 127;
    i_octave21    = 128;  i_octave22    = 129;  i_octave23    = 130;  i_octave24    = 131;
    i_octave25    = 132;  i_octave26    = 133;  i_octave27    = 134;  i_octave28    = 135;
    i_octave29    = 136;  i_octave210   = 137;  i_octave211   = 138;  i_octave212   = 139;
    i_octave213   = 140;  i_octave214   = 141;  i_octave215   = 142;  i_octave216   = 143;
    i_octave217   = 144;  i_octave218   = 145;  i_octave219   = 146;  i_octave220   = 147;
    i_octave221   = 148;  i_octave222   = 149;  i_octave223   = 150;  i_octave224   = 151;
    i_octave225   = 152;  i_octave226   = 153;  i_octave227   = 154;  i_octave228   = 155;
    i_octave229   = 156;  i_octave230   = 157;  i_octave231   = 158;  i_octave232   = 159;
    i_octave31    = 160;  i_octave32    = 161;  i_octave33    = 162;  i_octave34    = 163;
    i_octave35    = 164;  i_octave36    = 165;  i_octave37    = 166;  i_octave38    = 167;
    i_octave39    = 168;  i_octave310   = 169;  i_octave311   = 170;  i_octave312   = 171;
    i_octave313   = 172;  i_octave314   = 173;  i_octave315   = 174;  i_octave316   = 175;
    i_octave317   = 176;  i_octave318   = 177;  i_octave319   = 178;  i_octave320   = 179;
    i_octave321   = 180;  i_octave322   = 181;  i_octave323   = 182;  i_octave324   = 183;
    i_octave325   = 184;  i_octave326   = 185;  i_octave327   = 186;  i_octave328   = 187;
    i_octave329   = 188;  i_octave330   = 189;  i_octave331   = 190;  i_octave332   = 191;
    i_duration11  = 192;  i_duration12  = 193;  i_duration13  = 194;  i_duration14  = 195;
    i_duration15  = 196;  i_duration16  = 197;  i_duration17  = 198;  i_duration18  = 199;
    i_duration19  = 200;  i_duration110 = 201;  i_duration111 = 202;  i_duration112 = 203;
    i_duration113 = 204;  i_duration114 = 205;  i_duration115 = 206;  i_duration116 = 207;
    i_duration117 = 208;  i_duration118 = 209;  i_duration119 = 210;  i_duration120 = 211;
    i_duration121 = 212;  i_duration122 = 213;  i_duration123 = 214;  i_duration124 = 215;
    i_duration125 = 216;  i_duration126 = 217;  i_duration127 = 218;  i_duration128 = 219;
    i_duration129 = 220;  i_duration130 = 221;  i_duration131 = 222;  i_duration132 = 223;
    i_duration21  = 224;  i_duration22  = 225;  i_duration23  = 226;  i_duration24  = 227;
    i_duration25  = 228;  i_duration26  = 229;  i_duration27  = 230;  i_duration28  = 231;
    i_duration29  = 232;  i_duration210 = 233;  i_duration211 = 234;  i_duration212 = 235;
    i_duration213 = 236;  i_duration214 = 237;  i_duration215 = 238;  i_duration216 = 239;
    i_duration217 = 240;  i_duration218 = 241;  i_duration219 = 242;  i_duration220 = 243;
    i_duration221 = 244;  i_duration222 = 245;  i_duration223 = 246;  i_duration224 = 247;
    i_duration225 = 248;  i_duration226 = 249;  i_duration227 = 250;  i_duration228 = 251;
    i_duration229 = 252;  i_duration230 = 253;  i_duration231 = 254;  i_duration232 = 255;
    i_duration31  = 256;  i_duration32  = 257;  i_duration33  = 258;  i_duration34  = 259;
    i_duration35  = 260;  i_duration36  = 261;  i_duration37  = 262;  i_duration38  = 263;
    i_duration39  = 264;  i_duration310 = 265;  i_duration311 = 266;  i_duration312 = 267;
    i_duration313 = 268;  i_duration314 = 269;  i_duration315 = 270;  i_duration316 = 271;
    i_duration317 = 272;  i_duration318 = 273;  i_duration319 = 274;  i_duration320 = 275;
    i_duration321 = 276;  i_duration322 = 277;  i_duration323 = 278;  i_duration324 = 279;
    i_duration325 = 280;  i_duration326 = 281;  i_duration327 = 282;  i_duration328 = 283;
    i_duration329 = 284;  i_duration330 = 285;  i_duration331 = 286;  i_duration332 = 287;
    i_clk1        = 288;  i_clk2        = 289;  i_clk3        = 290;  i_res         = 291;
    i_active      = 292;  i_playmode    = 293;  i_linkmode    = 294;  i_gatemode    = 295;
    i_playmodemod = 296;  i_linkmodemod = 297;  i_gatemodemod = 298;  i_reverse     = 299;
    i_inverse     = 300;
  const
    o_out1        =   0;  o_out2        =   1;  o_out3        =   2;  o_trigout1    =   3;
    o_trigout2    =   4;  o_trigout3    =   5;  o_div161      =   6;  o_div162      =   7;
    o_div163      =   8;
  const
    SIZE      = 32;
    CHANNELS  =  3;
    FULL_SIZE = CHANNELS * SIZE;
    MAX_COUNT = 16;
  type
    TS32PlayMode = ( s32_3x32, s32_1x96);
    TS32LinkMode = ( s32_sync, s32_free);
  private
    FNotes        : array[ 0 .. FULL_SIZE - 1] of TSignal;
    FDurations    : array[ 0 .. FULL_SIZE - 1] of Integer;
    FOldClock1    : Boolean;
    FOldClock2    : Boolean;
    FOldClock3    : Boolean;
    FOldReset     : Boolean;
    FCounter1     : Integer;
    FCounter2     : Integer;
    FCounter3     : Integer;
    FCounterDiv1  : Integer;
    FCounterDiv2  : Integer;
    FCounterDiv3  : Integer;
    FStep1        : Integer;
    FStep2        : Integer;
    FStep3        : Integer;
    FDisplayStep1 : Integer;
    FDisplayStep2 : Integer;
    FDisplayStep3 : Integer;
    FPlayMode     : TS32PlayMode;
    FLinkMode     : TS32LinkMode;
    FClkCount1    : Integer;
    FClkCount2    : Integer;
    FClkCount3    : Integer;
    FResCount     : Integer;
    FReverse      : Boolean;
    FInverse      : Boolean;
  private
    procedure   SetPlayMode( aValue: TS32PlayMode);
    procedure   SetLinkMode( aValue: TS32LinkMode);
    procedure   NextStep ( Which: Integer);
    procedure   NextCount( Which: Integer);
    procedure   ResetState;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights ( const aPrefix: string; const aCallback: TLightsHandler);                        override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  private
    property    PlayMode : TS32PlayMode read FPlayMode write SetPlayMode;
    property    LinkMode : TS32LinkMode read FLinkMode write SetLinkMode;
  end;


  TModTextSequencer = class( TMod)
  strict private
  const
    i_trig    = 0;
    i_reset   = 1;
    i_dir     = 2;
    i_mode    = 3;
    i_chain   = 4;
  const
    o_out     = 0;
    o_gateout = 1;
  private
    FOutValue   : TSignal;
    FStepCount  : Integer;
    FStepValues : TSignalArray;
    FCounter    : Integer;
    FInState    : TSeqInState;
    FOldRes     : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   SetStepValues( const aValues: TSignalArray);
  end;


  TModSeqStep = class( TMod)
  strict private
  const
    i_triginfwd  = 0;
    i_triginbwd  = 1;
    i_chain      = 2;
    i_reset      = 3;
    i_dir        = 4;
    i_note       = 5;
    i_time       = 6;
  const
    o_trigoutfwd = 0;
    o_trigoutbwd = 1;
    o_out        = 2;
  private
    FTime    : TSignal;
    FState   : TSeqStepState;
    FOldRes  : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSeqClockStep = class( TMod)
  strict private
  const
    i_triginfwd  = 0;
    i_triginbwd  = 1;
    i_chain      = 2;
    i_reset      = 3;
    i_dir        = 4;
    i_clock      = 5;
    i_note       = 6;
    i_counts     = 7;
  const
    o_trigoutfwd = 0;
    o_trigoutbwd = 1;
    o_out        = 2;
  private
    FCount      : Integer;
    FState      : TSeqStepState;
    FOldRes     : Boolean;
    FOldClock   : Boolean;
    FTrigInFwd  : Boolean;
    FTrigInBwd  : Boolean;
    FDir        : Boolean;
    FTrigOutFwd : Boolean;
    FTrigOutBwd : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPattern = class( TMod)
  strict private
  const
    i_res     =  0;
    i_trig    =  1;
    i_step1   =  2;
    i_step2   =  3;
    i_step3   =  4;
    i_step4   =  5;
    i_step5   =  6;
    i_step6   =  7;
    i_step7   =  8;
    i_step8   =  9;
    i_mode1   = 10;
    i_mode2   = 11;
    i_mode3   = 12;
    i_mode4   = 13;
    i_mode5   = 14;
    i_mode6   = 15;
    i_mode7   = 16;
    i_mode8   = 17;
    i_active  = 18;
  const
    i_last    =  9;
  const
    o_out     =  0;
    o_out1    =  1;
    o_out2    =  2;
    o_out3    =  3;
    o_out4    =  4;
    o_out5    =  5;
    o_out6    =  6;
    o_out7    =  7;
    o_out8    =  8;
  const
    sm_normal = 0;
    sm_rnd1   = 1;
    sm_rnd2   = 2;
    sm_rnd3   = 3;
    sm_skip   = 4;
  private
    FPosition : Integer;
    FCounter  : Integer;
    FOldTrig  : Boolean;
    FOldRes   : Boolean;
    FOutVal   : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSeqChord = class( TMod)
  strict private
  const
    i_active0   =  0;
    i_active1   =  1;
    i_active2   =  2;
    i_active3   =  3;
    i_gatein0   =  4;
    i_gatein1   =  5;
    i_gatein2   =  6;
    i_gatein3   =  7;
    i_notein0   =  8;
    i_notein1   =  9;
    i_notein2   = 10;
    i_notein3   = 11;
    i_degree0   = 12;
    i_degree1   = 13;
    i_degree2   = 14;
    i_degree3   = 15;
    i_type0     = 16;
    i_type1     = 17;
    i_type2     = 18;
    i_type3     = 19;
    i_inver0    = 20;
    i_inver1    = 21;
    i_inver2    = 22;
    i_inver3    = 23;
    i_octave0   = 24;
    i_octave1   = 25;
    i_octave2   = 26;
    i_octave3   = 27;
    i_transpose = 28;
    i_gate      = 29;
    i_slew      = 30;
    i_reduce    = 31;
  const
    o_gateout0  =  0;
    o_gateout1  =  1;
    o_gateout2  =  2;
    o_gateout3  =  3;
    o_noteout0  =  4;
    o_noteout1  =  5;
    o_noteout2  =  6;
    o_noteout3  =  7;
  private
    function    CalcChord( aDegree, aType, anInversion, aMode: Integer): TTetrad;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModProbSequencer = class( TMod)
  strict private
  const
    i_note0        =  0;
    i_note1        =  1;
    i_note2        =  2;
    i_note3        =  3;
    i_note4        =  4;
    i_note5        =  5;
    i_note6        =  6;
    i_note7        =  7;
    i_note8        =  8;
    i_note9        =  9;
    i_note10       = 10;
    i_note11       = 11;
    i_prob0        = 12;
    i_prob1        = 13;
    i_prob2        = 14;
    i_prob3        = 15;
    i_prob4        = 16;
    i_prob5        = 17;
    i_prob6        = 18;
    i_prob7        = 19;
    i_prob8        = 20;
    i_prob9        = 21;
    i_prob10       = 22;
    i_prob11       = 23;
    i_ctrl0        = 24;
    i_ctrl1        = 25;
    i_ctrl2        = 26;
    i_ctrl3        = 27;
    i_ctrl4        = 28;
    i_ctrl5        = 29;
    i_ctrl6        = 30;
    i_ctrl7        = 31;
    i_ctrl8        = 32;
    i_ctrl9        = 33;
    i_ctrl10       = 34;
    i_ctrl11       = 35;
    i_scale        = 36;
    i_chain        = 37;
    i_trig         = 38;
    i_rnd          = 39;
    i_clr          = 40;
    i_tilt         = 41;
    i_tiltmod      = 42;
    i_tiltmodamt   = 43;
    i_basenote     = 44;
    i_lownote      = 45;
    i_highnote     = 46;
    i_smooth       = 47;
    i_smoothmod    = 48;
    i_smoothmodamt = 49;
    i_active       = 50;
    i_scalemod     = 51;
  const
    o_out          =  0;
  private
    FCurrentNote  : TSignal;
    FCurrNoteNr   : Integer;
    FOldTrig      : Boolean;
    FScale        : Integer;
    FScaleChanged : Boolean;
    FNotes        : TScaleNoteSet;
    FPrevProb     : TSIgnal;
    FProb         : array[ -1 .. 127] of TSignal;
  private
    function    FixProb( aProb: TSignal): TSignal;
    function    Search( aValue: TSignal): Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModSeqPattern = class( TMod)
  strict private
  const
    i_bits1   =  0;
    i_bits2   =  1;
    i_bits3   =  2;
    i_bits4   =  3;
    i_bits5   =  4;
    i_bits6   =  5;
    i_bits7   =  6;
    i_bits8   =  7;
    i_bits9   =  8;
    i_bits10  =  9;
    i_bits11  = 10;
    i_bits12  = 11;
    i_bits13  = 12;
    i_bits14  = 13;
    i_bits15  = 14;
    i_bits16  = 15;
    i_bits17  = 16;
    i_bits18  = 17;
    i_bits19  = 18;
    i_bits20  = 19;
    i_bits21  = 20;
    i_bits22  = 21;
    i_bits23  = 22;
    i_bits24  = 23;
    i_loop    = 24;
    i_mode    = 25;
    i_clk     = 26;
    i_act     = 27;
    i_res     = 28;
    i_p1      = 29;
    i_p2      = 30;
    i_p3      = 31;
    i_p4      = 32;
    i_p5      = 33;
    i_p6      = 34;
    i_p7      = 35;
    i_p8      = 36;
    i_s1      = 37;
    i_s2      = 38;
    i_s3      = 39;
    i_s4      = 40;
    i_s5      = 41;
    i_s6      = 42;
    i_s7      = 43;
    i_s8      = 44;
    i_e1      = 45;
    i_e2      = 46;
    i_e3      = 47;
    i_e4      = 48;
    i_e5      = 49;
    i_e6      = 50;
    i_e7      = 51;
    i_e8      = 52;
    i_copye   = 53;
    i_copys   = 54;
    i_copyp   = 55;
    i_clre    = 56;
    i_clrs    = 57;
    i_clrp    = 58;
    i_reset   = 59;
  const
    o_out1    =  0;
    o_out2    =  1;
    o_out3    =  2;
    o_out4    =  3;
    o_out5    =  4;
    o_out6    =  5;
    o_out7    =  6;
    o_out8    =  7;
    o_out9    =  8;
    o_out10   =  9;
    o_out11   = 10;
    o_out12   = 11;
    o_out13   = 12;
    o_out14   = 13;
    o_out15   = 14;
    o_out16   = 15;
    o_out17   = 16;
    o_out18   = 17;
    o_out19   = 18;
    o_out20   = 19;
    o_out21   = 20;
    o_out22   = 21;
    o_out23   = 22;
    o_out24   = 23;
    o_done    = 24;
  const
    BIT_COUNT     = 24;
    STATE_COUNT   = 8 * 8 * 8;
    TOT_BIT_COUNT = STATE_COUNT * BIT_COUNT;
  type
    TMode = ( mspRun, mspEdit);
  private
    FBits1       : TBitArray;           // Bit inactive (0) or active (1)
    FBits2       : TBitArray;           // Bit is short (0) or long   (1)
    FSkips       : TBitArray;
    FResets      : TBitArray;
    FMode        : TMode;
    FCyclic      : Boolean;
    FPart        : Integer;
    FSection     : Integer;
    FElement     : Integer;
    FEditPart    : Integer;
    FEditSection : Integer;
    FEditElement : Integer;
    FOldClk      : Boolean;
    FClk         : Boolean;
    FNotClk      : Boolean;
    FOldRes      : Boolean;
    FRes         : Boolean;
    FESP         : array[ 0 .. 7, 0 .. 7, 0 .. 7 ] of Integer; // order: E, S, P
    FBitsChanged : Integer;
  private
    function    GetBitsChanged : Boolean;
    procedure   SetBitsChanged( aValue: Boolean);
    procedure   CreateBitArray( var anArray: TBitArray; aSize: Integer);
    procedure   CopyBits ( aSrcIndex, aDstIndex, aCount: Integer);
    procedure   ClearBits( aDstIndex, aCount: Integer);
    function    ResetSequence: Boolean;
    procedure   IncStep;
    function    ESPIndex( E, S, P: Integer): Integer;                                                          overload;
    function    ESPIndex                   : Integer;                                                          overload;
    function    BitStartIndex              : Integer;
    function    ESPEditIndex               : Integer;
    function    BitStartEditIndex          : Integer;
    function    StepIsSkip                 : Boolean;
    function    StepIsReset                : Boolean;
    procedure   SetOutputs;
    procedure   ClearOutputs;
    procedure   NextStep;
    function    BitDataValue: TSignalArray;
    function    EspDataValue: TSignalArray;
    procedure   FixSkipsAndResets( anE, anS, anP, aNewValue: Integer);                                         overload;
    procedure   FixSkipsAndResets( anIndex, aNewValue: Integer);                                               overload;
    procedure   FixSkipsAndResets( aNewValue: Integer);                                                        overload;
    procedure   IncBitValue( anIndex: Integer);
    function    BitsToValue( anIndex: Integer): Integer;
    procedure   ValueToBits( anIndex, aValue: Integer);
  private
    property    BitsChanged : Boolean read GetBitsChanged write SetBitsChanged;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherData  ( const aPrefix: string; const aCallback: TDataHandler  );                         override;
  end;


  TModSeqValues = class( TMod)
  strict private
  const
    i_value1  =  0;
    i_value2  =  1;
    i_value3  =  2;
    i_value4  =  3;
    i_value5  =  4;
    i_value6  =  5;
    i_value7  =  6;
    i_value8  =  7;
    i_loop    =  8;
    i_mode    =  9;
    i_clk     = 10;
    i_act     = 11;
    i_res     = 12;
    i_p1      = 13;
    i_p2      = 14;
    i_p3      = 15;
    i_p4      = 16;
    i_p5      = 17;
    i_p6      = 18;
    i_p7      = 19;
    i_p8      = 20;
    i_s1      = 21;
    i_s2      = 22;
    i_s3      = 23;
    i_s4      = 24;
    i_s5      = 25;
    i_s6      = 26;
    i_s7      = 27;
    i_s8      = 28;
    i_e1      = 29;
    i_e2      = 30;
    i_e3      = 31;
    i_e4      = 32;
    i_e5      = 33;
    i_e6      = 34;
    i_e7      = 35;
    i_e8      = 36;
    i_copye   = 37;
    i_copys   = 38;
    i_copyp   = 39;
    i_clre    = 40;
    i_clrs    = 41;
    i_clrp    = 42;
    i_reset   = 43;
    i_valmode = 44;
  const
    o_out1    =  0;
    o_out2    =  1;
    o_out3    =  2;
    o_out4    =  3;
    o_out5    =  4;
    o_out6    =  5;
    o_out7    =  6;
    o_out8    =  7;
    o_done    =  8;
  const
    VALUE_COUNT     = 8;
    STATE_COUNT     = 8 * 8 * 8;
    TOT_VALUE_COUNT = STATE_COUNT * VALUE_COUNT;
  type
    TMode = ( msvRun, msvEdit);
  private
    FValues        : TSignalArray;
    FSkips         : TBitArray;
    FResets        : TBitArray;
    FMode          : TMode;
    FCyclic        : Boolean;
    FPart          : Integer;
    FSection       : Integer;
    FElement       : Integer;
    FEditPart      : Integer;
    FEditSection   : Integer;
    FEditElement   : Integer;
    FOldClk        : Boolean;
    FClk           : Boolean;
    FNotClk        : Boolean;
    FOldRes        : Boolean;
    FRes           : Boolean;
    FESP           : array[ 0 .. 7, 0 .. 7, 0 .. 7 ] of Integer; // order: E, S, P
    FValMode       : Integer;
    FValuesChanged : Integer;
  private
    procedure   CreateBitArray( var anArray: TBitArray; aSize: Integer);
    procedure   CopyValues ( aSrcIndex, aDstIndex, aCount: Integer);
    procedure   ClearValues( aDstIndex, aCount: Integer);
    function    ResetSequence: Boolean;
    procedure   IncStep;
    function    ESPIndex( E, S, P: Integer): Integer;                                                          overload;
    function    ESPIndex                   : Integer;                                                          overload;
    function    ValueStartIndex            : Integer;
    function    ESPEditIndex               : Integer;
    function    ValueStartEditIndex        : Integer;
    function    StepIsSkip                 : Boolean;
    function    StepIsReset                : Boolean;
    procedure   SetOutputs;
    procedure   ClearOutputs;
    procedure   NextStep;
    function    EspDataValue  : TSignalArray;
    procedure   FixSkipsAndResets( anE, anS, anP, aNewValue: Integer);                                         overload;
    procedure   FixSkipsAndResets( anIndex, aNewValue: Integer);                                               overload;
    procedure   FixSkipsAndResets( aNewValue: Integer);                                                        overload;
    function    GetValuesChanged: Boolean;
    procedure   SetValuesChanged( aValue: Boolean);
  private
    property    ValuesChanged: Boolean read GetValuesChanged write SetValuesChanged;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
    procedure   GatherData  ( const aPrefix: string; const aCallback: TDataHandler  );                         override;
  end;


  TModSlew = class( TMod)
  strict private
  const
    i_in   = 0;
    i_slew = 1;
  const
    o_out  = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRMS = class( TMod)
  strict private
  const
    i_in   = 0;
    i_time = 1;
  const
    o_out  = 0;
  private
    FRMSCalculator : TRMSCalculator;
    FPrevTime      : TSignal;
    FWasSpedUp     : Boolean;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModCsvData = class( TMod)
  strict private
  const
    i_reset =  0;
    i_trig  =  1;
    i_dir   =  2;
    i_mode  =  3;
  const
    o_out1  =  0;
    o_out2  =  1;
    o_out3  =  2;
    o_out4  =  3;
    o_out5  =  4;
    o_out6  =  5;
    o_out7  =  6;
    o_out8  =  7;
    o_out9  =  8;
    o_out10 =  9;
    o_out11 = 10;
    o_out12 = 11;
    o_out13 = 12;
    o_out14 = 13;
    o_out15 = 14;
    o_out16 = 15;
  private
    FMemory   : array[ 0 .. 15] of TSignalArray;
    FComments : TStringList;
    FFileName : string;
    FChannels : Integer;
    FCounter  : Integer;
    FSteps    : Integer;
    FInState  : TSeqInState;
    FOldRes   : Boolean;
    FMode     : Integer;
    FStopped  : Boolean;
  private
    procedure   LoadFile;
    procedure   SetFileName( const aValue: string);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler);                     override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  public
    property    FileName: string read FFileName write SetFileName;
  end;


  TModKleeBit = class( TMod)
  strict private
  const
    i_instep       =  0;
    i_inload       =  1;
    i_loopmode     =  2;
    i_inanalog     =  3;
    i_inbus1       =  4;
    i_inbus2       =  5;
    i_inbus3       =  6;
    i_pre1         =  7;
    i_pre2         =  8;
    i_pre3         =  9;
    i_pre4         = 10;
    i_pre5         = 11;
    i_pre6         = 12;
    i_pre7         = 13;
    i_pre8         = 14;
    i_level1       = 15;
    i_level2       = 16;
    i_level3       = 17;
    i_level4       = 18;
    i_level5       = 19;
    i_level6       = 20;
    i_level7       = 21;
    i_level8       = 22;
    i_bussel1      = 23;
    i_bussel2      = 24;
    i_bussel3      = 25;
    i_bussel4      = 26;
    i_bussel5      = 27;
    i_bussel6      = 28;
    i_bussel7      = 29;
    i_bussel8      = 30;
    i_inbit        = 31;
    i_invertmode   = 32;
    i_mainlevel    = 33;
    i_loopinvert   = 34;
    i_invertinvert = 35;
    i_mainlevelmod = 36;
  const
    o_outanalog    =  0;
    o_outbus1      =  1;
    o_outbus2      =  2;
    o_outbus3      =  3;
    o_outbit       =  4;
    o_outloop      =  5;
    o_outinvert    =  6;
  private
    FPattern    : Byte;
    FOldStep    : Boolean;
    FOldLoad    : Boolean;
    FBus1       : Boolean;
    FBus2       : Boolean;
    FBus3       : Boolean;
    FAnalog     : TSignal;
    FLoopMode   : Boolean;
    FInvertMode : Boolean;
  private
    procedure   LoadPreset;
    procedure   Step;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSeqRandom = class( TMod)
  strict private
  const
    i_trig       =  0;
    i_changeone  =  1;
    i_scale      =  2;
    i_basenote   =  3;
    i_lownote    =  4;
    i_highnote   =  5;
    i_distmod    =  6;
    i_distmodamt =  7;
    i_dist       =  8;
    i_chain      =  9;
    i_active     = 10;
    i_scalemod   = 11;
    i_changeall  = 12;
  const
    o_out        =  0;
  const
    s_count      =  0;
  private
    FValues         : array[ Byte] of TScaleNote;
    FPattern        : array[ Byte] of TScaleNote;
    FCount          : Integer;
    FScale          : Integer;
    FBaseNote       : TScaleNote;
    FLowNote        : TScaleNote;
    FHighNote       : TScaleNote;
    FNoteCount      : Integer;
    FScaleNotes     : array[ TScaleNote] of TScaleNote;
    FOldTrig        : Boolean;
    FOldOne         : Boolean;
    FOldAll         : Boolean;
    FPosition       : Byte;
    FDistribution   : TSignal;
    FTrigCount      : Integer;
    FChangeOneCount : Integer;
    FChangeAllCount : Integer;
  private
    procedure   MakeScale;
    function    RandomNote( aPosition: Byte; aDistribution: TSignal): TScaleNote;
    procedure   ChangeScale( aScale: Integer);
    procedure   ChangeOneNote( aPosition: Byte);
    procedure   ChangeAllNotes;
    procedure   Step;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModSeqRandVal = class( TMod)
  strict private
  const
    i_trig       =  0;
    i_changeone  =  1;
    i_basevalue  =  2;
    i_lowvalue   =  3;
    i_highvalue  =  4;
    i_distmod    =  5;
    i_distmodamt =  6;
    i_dist       =  7;
    i_chain      =  8;
    i_quantize   =  9;
    i_qsteps     = 10;
    i_slew       = 11;
    i_slewmode   = 12;
    i_active     = 13;
  const
    o_out        =  0;
  private
    FValues       : array[ Byte] of TSignal;
    FCount        : Integer;
    FLowValue     : TSignal;
    FHighValue    : TSignal;
    FOldTrig      : Boolean;
    FOldOne       : Boolean;
    FPosition     : Byte;
    FDistribution : TSignal;
    FOldValue     : TSignal;
    FTrigCount    : Integer;
    FChangeCount  : Integer;
  private
    function    RandomValue( aPosition: Byte; aDistribution: TSignal): TSignal;
    function    ScaledValue( aPosition: Byte): TSignal;
    procedure   ChangeOneValue( aPosition: Byte);
    procedure   Step;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModRewriter = class( TMod)
  strict private
  const
    i_step     = 0;
    i_steps    = 1;
    i_stepsmod = 2;
    i_mode     = 3;
  const
    o_a        = 0;
    o_b        = 1;
    o_c        = 2;
    o_d        = 3;
    o_e        = 4;
  private
    FSteps     : Integer;
    FRuleA     : string;
    FRuleB     : string;
    FRuleC     : string;
    FRuleD     : string;
    FRuleE     : string;
    FPatterns  : array[ 0 .. 8] of string;
    FStep      : Integer;
    FSymbol    : Char;
    FOldStep   : Boolean;
    FTrigCount : Integer;
  private
    procedure   MakePatterns;
    function    StripString( const aValue: string): string;
  private
    procedure   SetSteps( aValue: Integer);
    procedure   SetRuleA( const aValue: string);
    procedure   SetRuleB( const aValue: string);
    procedure   SetRuleC( const aValue: string);
    procedure   SetRuleD( const aValue: string);
    procedure   SetRuleE( const aValue: string);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Steps: Integer read FSteps write SetSteps;
    property    RuleA: string  read FRuleA write SetRuleA;
    property    RuleB: string  read FRuleB write SetRuleB;
    property    RuleC: string  read FRuleC write SetRuleC;
    property    RuleD: string  read FRuleD write SetRuleD;
    property    RuleE: string  read FRuleE write SetRuleE;
  end;


  TModRewriterNote = class( TMod)
  strict private
  const
    i_step     =  0;
    i_steps    =  1;
    i_stepsmod =  2;
    i_notea    =  3;
    i_noteb    =  4;
    i_notec    =  5;
    i_noted    =  6;
    i_notee    =  7;
    i_notef    =  8;
    i_noteg    =  9;
    i_noteh    = 10;
    i_chain    = 11;
    i_mode     = 12;
    i_chaina   = 13;
    i_chainb   = 14;
    i_chainc   = 15;
    i_chaind   = 16;
    i_chaine   = 17;
    i_chainf   = 18;
    i_chaing   = 19;
    i_chainh   = 20;
  const
    o_out      =  0;
  const
    m_lin44    = 0;
    m_db4      = 1;
    m_lin11    = 2;
    m_db0      = 3;
    m_lin04    = 4;
    m_lin01    = 5;
    m_note     = 6;
  private
    FSteps     : Integer;
    FRuleA     : string;
    FRuleB     : string;
    FRuleC     : string;
    FRuleD     : string;
    FRuleE     : string;
    FRuleF     : string;
    FRuleG     : string;
    FRuleH     : string;
    FPatterns  : array[ 0 .. 8] of string;
    FStep      : Integer;
    FSymbol    : Char;
    FOldStep   : Boolean;
    FTrigCount : Integer;
  private
    procedure   MakePatterns;
    function    StripString( const aValue: string): string;
  private
    procedure   SetSteps( aValue: Integer);
    procedure   SetRuleA( const aValue: string);
    procedure   SetRuleB( const aValue: string);
    procedure   SetRuleC( const aValue: string);
    procedure   SetRuleD( const aValue: string);
    procedure   SetRuleE( const aValue: string);
    procedure   SetRuleF( const aValue: string);
    procedure   SetRuleG( const aValue: string);
    procedure   SetRuleH( const aValue: string);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Steps: Integer read FSteps write SetSteps;
    property    RuleA: string  read FRuleA write SetRuleA;
    property    RuleB: string  read FRuleB write SetRuleB;
    property    RuleC: string  read FRuleC write SetRuleC;
    property    RuleD: string  read FRuleD write SetRuleD;
    property    RuleE: string  read FRuleE write SetRuleE;
    property    RuleF: string  read FRuleF write SetRuleF;
    property    RuleG: string  read FRuleG write SetRuleG;
    property    RuleH: string  read FRuleH write SetRuleH;
  end;


  TModTuneSmithy = class( TMod)
  strict private
  const
    i_step  = 0;
    i_steps = 1;
    i_chain = 2;
  const
    o_out   = 0;
  private
    FSeed       : string;
    FTuneSmithy : TTuneSmithy;
    FStep       : Integer;
    FOldStep    : Boolean;
    FTrigCount  : Integer;
  private
    procedure   Recreate;
    procedure   SetSeed( const aValue: string);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Seed: string read FSeed write SetSeed;
  end;


  TModNeuralPulser = class( TMod)
  strict private
  const
    i_clock       =  1;
    i_reset       =  2;
    i_outmode     =  3;
    i_exc1a       =  4;
    i_exc1b       =  5;
    i_inh1        =  6;
    i_mode1       =  7;
    i_exc2a       =  8;
    i_exc2b       =  9;
    i_inh2        = 10;
    i_mode2       = 11;
    i_exc3a       = 12;
    i_exc3b       = 13;
    i_inh3        = 14;
    i_mode3       = 15;
    i_exc4a       = 16;
    i_exc4b       = 17;
    i_inh4        = 18;
    i_mode4       = 19;
    i_exc5a       = 20;
    i_exc5b       = 21;
    i_inh5        = 22;
    i_mode5       = 23;
    i_exc6a       = 24;
    i_exc6b       = 25;
    i_inh6        = 26;
    i_mode6       = 27;
    i_exc7a       = 28;
    i_exc7b       = 29;
    i_inh7        = 30;
    i_mode7       = 31;
    i_exc8a       = 32;
    i_exc8b       = 33;
    i_inh8        = 34;
    i_mode8       = 35;
  const
    o_out1        =  0;
    o_out2        =  1;
    o_out3        =  2;
    o_out4        =  3;
    o_out5        =  4;
    o_out6        =  5;
    o_out7        =  6;
    o_out8        =  7;
  private
    FOutValues : array[ 0 .. 7] of Boolean;
    FCounters  : array[ 0 .. 7] of Integer;
    FOldClock  : Boolean;
    FOldReset  : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModCellAut = class( TMod)
  strict private
  const
    i_step   = 0;
    i_reset  = 1;
    i_dir    = 2;
    i_sumcnt = 3;
  const
    o_out    = 0;
    o_outsum = 1;
  private
    FEngine     : TElementaryCA;
    FRule       : Integer;
    FSteps      : Integer;
    FFaults     : Integer;
    FSumCount   : Integer;
    FSumMult    : TSignal;
    FStep       : Integer;
    FStepCount  : Integer;
    FOldStep    : Boolean;
    FOldReset   : Boolean;
    FTrigCount  : Integer;
    FResetCount : Integer;
  private
    procedure   Recreate;
    procedure   SetSteps   ( aValue: Integer);
    procedure   SetRule    ( aValue: Integer);
    procedure   SetFaults  ( aValue: Integer);
    procedure   SetSumCount( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Rule     : Integer read FRule     write SetRule;
    property    Steps    : Integer read FSteps    write SetSteps;
    property    Faults   : Integer read FFaults   write setFaults;
    property    SumCount : Integer read FSumCount write SetSumCount;
  end;


  TModCellAut2 = class( TMod)
  strict private
  const
    i_step      =  0;
    i_reset     =  1;
    i_rule      =  2;
    i_rulemod   =  3;
    i_faults    =  4;
    i_faultsmod =  5;
    i_mode      =  6;
  const
    o_out1      =  0;
    o_out2      =  1;
    o_out3      =  2;
    o_out4      =  3;
    o_out5      =  4;
    o_out6      =  5;
    o_out7      =  6;
    o_out8      =  7;
    o_out9      =  8;
    o_out10     =  9;
    o_out11     = 10;
    o_out12     = 11;
    o_out13     = 12;
    o_out14     = 13;
    o_out15     = 14;
  const
    WIDTH       = 15;
  private
    FEngine     : TLimitedCA;
    FRule       : Integer;
    FFaults     : Integer;
    FSteps      : Integer;
    FOldStep    : Boolean;
    FOldReset   : Boolean;
    FTrigCount  : Integer;
    FResetCount : Integer;
  private
    procedure   Recreate;
    procedure   SetRule    ( aValue: Integer);
    procedure   SetFaults  ( aValue: Integer);
    procedure   SetSteps   ( aValue: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Rule   : Integer read FRule   write SetRule;
    property    Faults : Integer read FFaults write setFaults;
    property    Steps  : Integer read FSteps  write SetSteps;
  end;


  TModLifeSeq = class( TMod)
  strict private
  const
    i_step         =  0;
    i_change       =  1;
    i_changemute   =  2;
    i_res          =  3;
    i_mode         =  4;
    i_wrapmode     =  5;
    i_summode      =  6;
    i_faultrate    =  7;
    i_lowcount     =  8;
    i_randomamount =  9;
    i_rnd          = 10;
    i_clr          = 11;
    i_chain        = 12;
    i_rev          = 13;
  const
    o_sum          =  0;
    o_out1         =  1;
    o_out2         =  2;
    o_out3         =  3;
    o_out4         =  4;
    o_out5         =  5;
    o_out6         =  6;
    o_out7         =  7;
    o_out8         =  8;
    o_out9         =  9;
    o_out10        = 10;
    o_out11        = 11;
    o_out12        = 12;
    o_out13        = 13;
    o_out14        = 14;
    o_out15        = 15;
    o_out16        = 16;
    o_out17        = 17;
    o_out18        = 18;
    o_out19        = 19;
    o_out20        = 20;
    o_out21        = 21;
    o_out22        = 22;
    o_out23        = 23;
    o_out24        = 24;
    o_out25        = 25;
    o_out26        = 26;
    o_out27        = 27;
    o_out28        = 28;
    o_out29        = 29;
    o_out30        = 30;
    o_out31        = 31;
    o_out32        = 32;
    o_sum1         = 33;
    o_sum2         = 34;
    o_sum3         = 35;
    o_sum4         = 36;
    o_sum5         = 37;
    o_sum6         = 38;
    o_sum7         = 39;
    o_sum8         = 40;
    o_sum9         = 41;
    o_sum10        = 42;
    o_sum11        = 43;
    o_sum12        = 44;
    o_sum13        = 45;
    o_sum14        = 46;
    o_sum15        = 47;
    o_sum16        = 48;
    o_sum17        = 49;
    o_sum18        = 50;
    o_sum19        = 51;
    o_sum20        = 52;
    o_sum21        = 53;
    o_sum22        = 54;
    o_sum23        = 55;
    o_sum24        = 56;
    o_sum25        = 57;
    o_sum26        = 58;
    o_sum27        = 59;
    o_sum28        = 60;
    o_sum29        = 61;
    o_sum30        = 62;
    o_sum31        = 63;
    o_sum32        = 64;
  const
    m_size         = 32;
  private
    FEngine      : TGameOfLife;
    FStops       : TGameOfLife;
    FExecCounter : Integer;
    FOldStep     : Boolean;
    FOldChange   : Boolean;
    FOldRes      : Boolean;
    FTrigCount   : Integer;
    FChangeCount : Integer;
    FResCount    : Integer;
    FPosition    : Integer;
    FRandomFlag  : Boolean;
    FClearFlag   : Boolean;
    FReversed    : Boolean;
    FLeftStop    : Integer;
    FRightStop   : Integer;
    FChanged     : Boolean;
  private
    function    GetGridValue : string;
    procedure   SetGridValue ( const aValue: string);
    function    GetStopsValue: string;
    procedure   SetStopsValue( const aValue: string);
    procedure   CellChanged( aSender: TObject; anX, anY: Integer; IsAlive: Boolean);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                 override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  public
    property    GridValue  : string read GetGridValue  write SetGridValue;
    property    StopsValue : string read GetStopsValue write SetStopsValue;
  end;


  TModSmallLifeSeq = class( TMod)
  strict private
  const
    i_xstep        =  0;
    i_ystep        =  1;
    i_change       =  2;
    i_res          =  3;
    i_mode         =  4;
    i_wrapmode     =  5;
    i_faultrate    =  6;
    i_lowcount     =  7;
    i_randomamount =  8;
    i_rnd          =  9;
    i_clr          = 10;
    i_rev          = 11;
  const
    o_out          =  0;
    o_sum          =  1;
    o_rowout1      =  2;
    o_rowout2      =  3;
    o_rowout3      =  4;
    o_rowout4      =  5;
    o_rowout5      =  6;
    o_rowout6      =  7;
    o_rowout7      =  8;
    o_rowout8      =  9;
    o_rowout9      = 10;
    o_rowout10     = 11;
    o_rowout11     = 12;
    o_rowout12     = 13;
    o_rowsum1      = 14;
    o_rowsum2      = 15;
    o_rowsum3      = 16;
    o_rowsum4      = 17;
    o_rowsum5      = 18;
    o_rowsum6      = 19;
    o_rowsum7      = 20;
    o_rowsum8      = 21;
    o_rowsum9      = 22;
    o_rowsum10     = 23;
    o_rowsum11     = 24;
    o_rowsum12     = 25;
    o_colout1      = 26;
    o_colout2      = 27;
    o_colout3      = 28;
    o_colout4      = 29;
    o_colout5      = 30;
    o_colout6      = 31;
    o_colout7      = 32;
    o_colout8      = 33;
    o_colout9      = 34;
    o_colout10     = 35;
    o_colout11     = 36;
    o_colout12     = 37;
    o_colsum1      = 38;
    o_colsum2      = 39;
    o_colsum3      = 40;
    o_colsum4      = 41;
    o_colsum5      = 42;
    o_colsum6      = 43;
    o_colsum7      = 44;
    o_colsum8      = 45;
    o_colsum9      = 46;
    o_colsum10     = 47;
    o_colsum11     = 48;
    o_colsum12     = 49;
  const
    m_size         = 12;
  private
    FEngine      : TGameOfLife;
    FExecCounter : Integer;
    FOldXStep    : Boolean;
    FOldYStep    : Boolean;
    FOldChange   : Boolean;
    FOldRes      : Boolean;
    FTrigXCount  : Integer;
    FTrigYCount  : Integer;
    FChangeCount : Integer;
    FResCount    : Integer;
    FXPosition   : Integer;
    FYPosition   : Integer;
    FRandomFlag  : Boolean;
    FClearFlag   : Boolean;
    FReversed    : Boolean;
    FChanged     : Boolean;
  private
    function    GetGridValue : string;
    procedure   SetGridValue ( const aValue: string);
    procedure   CellChanged( aSender: TObject; anX, anY: Integer; IsAlive: Boolean);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                 override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  public
    property    GridValue  : string read GetGridValue  write SetGridValue;
  end;


  TModVitaPHoBium = class( TMod)
  strict private
  const
    i_clock        =  0;
    i_in           =  1;
    i_res          =  2;
    i_rev          =  3;
    i_change       =  4;
    i_step         =  5;
    i_x            =  6;
    i_y            =  7;
    i_mode         =  8;
    i_wrapmode     =  9;
    i_faultrate    = 10;
    i_randomamount = 11;
    i_rnd          = 12;
    i_clr          = 13;
    i_skipmode     = 14;
    i_skipctrl     = 15;
  const
    o_out          =  0;
    o_gate         =  1;
    o_trig         =  2;
    o_xout         =  3;
    o_yout         =  4;
  const
    m_xsize        = 16;
    m_ysize        =  8;
    m_size         = m_xsize * m_ysize;
  private
    FEngine       : TGameOfLife;
    FFwdIndex     : array of Integer;
    FRevIndex     : array of Integer;
    FLiveCount    : Integer;
    FExecCounter  : Integer;
    FOldClock     : Boolean;
    FOldStep      : Boolean;
    FOldChange    : Boolean;
    FOldRes       : Boolean;
    FResCount     : Integer;
    FStepCount    : Integer;
    FChangeCount  : Integer;
    FClockCount   : Integer;
    FTrigCount    : Integer;
    FTrigLedCount : Integer;
    FXPosition    : Integer;
    FYPosition    : Integer;
    FRandomFlag   : Boolean;
    FClearFlag    : Boolean;
    FReversed     : Boolean;
    FChanged      : Boolean;
    FOldX         : Integer;
    FOldY         : Integer;
    FSkipDead     : Boolean;
  private
    function    GetGridValue : string;
    procedure   SetGridValue ( const aValue: string);
    procedure   CellChanged( aSender: TObject; anX, anY: Integer; IsAlive: Boolean);
    procedure   FindNextAlive( var XPos, YPos: Integer);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    destructor  Destroy;                                                                                       override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                 override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  public
    property    GridValue  : string read GetGridValue  write SetGridValue;
  end;


  TModLut = class( TMod)
  strict private
  const
    i_select     =  0;
    i_mode       =  1;
    i_xfade      =  2;
    i_chain      =  3;
    i_step1      =  4;   // Keep steps consecutive
    i_step2      =  5;
    i_step3      =  6;
    i_step4      =  7;
    i_step5      =  8;
    i_step6      =  9;
    i_step7      = 10;
    i_step8      = 11;
    i_step9      = 12;
    i_step10     = 13;
    i_step11     = 14;
    i_step12     = 15;
    i_step13     = 16;
    i_step14     = 17;
    i_step15     = 18;
    i_step16     = 19;
    i_steps      = 20;
    i_stepsmod   = 21;
    i_rnd2       = 22;
    i_xfademod   = 23;
    i_rnd3       = 24;
  const
    o_out        =  0;
    o_out2       =  1;
    o_out3       =  2;
    o_out4       =  3;
    o_stepout1   =  4;   // Keep steps consecutive
    o_stepout2   =  5;
    o_stepout3   =  6;
    o_stepout4   =  7;
    o_stepout5   =  8;
    o_stepout6   =  9;
    o_stepout7   = 10;
    o_stepout8   = 11;
    o_stepout9   = 12;
    o_stepout10  = 13;
    o_stepout11  = 14;
    o_stepout12  = 15;
    o_stepout13  = 16;
    o_stepout14  = 17;
    o_stepout15  = 18;
    o_stepout16  = 19;
    o_outraw     = 20;
    o_outraw2    = 21;
    o_outraw3    = 22;
    o_outraw4    = 23;
  const
    m_lin44      =  0;   // linear [-4, 4]
    m_db4        =  1;   // exp    [-MAX_DB_VAL, 12] dB
    m_lin11      =  2;   // linear [-1, 1]
    m_db0        =  3;   // exp    [-MAX_DB_VAL, 0] dB
    m_lin04      =  4;   // linear [0, 4]
    m_lin01      =  5;   // linear [0, 1]
    m_note       =  6;   // note values ([0,127] nn - MiddleNote) / NOTE_SCALING
  private
    FStepCount  : Integer;
    FPosition   : TSignal;
    FPosition2  : TSignal;
    FPosition3  : TSignal;
    FPosition4  : TSignal;
    FRandom     : Boolean;
    FOldRandom  : Boolean;
    FRndReq     : Boolean;
    FRandom3    : Boolean;
    FOldRandom3 : Boolean;
    FRndReq3    : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
  end;


  TModMultiCompare = class( TMod)
  strict private
  const
    i_in          =  0;
    i_inamt       =  1;
    i_inamtmod    =  2;
    i_level1      =  3;   // Keep levels consecutive
    i_level2      =  4;
    i_level3      =  5;
    i_level4      =  6;
    i_level5      =  7;
    i_level6      =  8;
    i_level7      =  9;
    i_level8      = 10;
    i_bussel1     = 11;   // Keep bus selects consecutive
    i_bussel2     = 12;
    i_bussel3     = 13;
    i_bussel4     = 14;
    i_bussel5     = 15;
    i_bussel6     = 16;
    i_bussel7     = 17;
    i_bussel8     = 18;
  const
    o_out1        =  0;
    o_out2        =  1;
    o_out3        =  2;
    o_pos1        =  3;   // Keep pos outputs consecutive
    o_pos2        =  4;
    o_pos3        =  5;
    o_pos4        =  6;
    o_pos5        =  7;
    o_pos6        =  8;
    o_pos7        =  9;
    o_pos8        = 10;
  private
    FPosition : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMorse = class( TMod)
  strict private
  const
    i_reset = 0;
    i_clock = 1;
  const
    o_out   = 0;
  private
    FMessage     : string;
    FSymbols     : string;
    FSymbolIndex : Integer;
    FOldReset    : Boolean;
    FOldClock    : Boolean;
  private
    function    CharToSymbols( aChar: Char): string;
    procedure   MessageToSymbols;
    procedure   SetMessage( const aValue: string);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Message: string read FMessage write SetMessage;
  end;


  TModGateSeq = class( TMod)
  strict private
  const
    i_trig    =  0;
    i_step1   =  1;
    i_step2   =  2;
    i_step3   =  3;
    i_step4   =  4;
    i_step5   =  5;
    i_step6   =  6;
    i_step7   =  7;
    i_step8   =  8;
    i_step9   =  9;
    i_step10  = 10;
    i_step11  = 11;
    i_step12  = 12;
    i_step13  = 13;
    i_step14  = 14;
    i_step15  = 15;
    i_step16  = 16;
    i_chain   = 17;
    i_res     = 18;
    i_active  = 19;
    i_dir     = 20;
  const
    o_out     =  0;
    o_outres  =  1;
    o_outtrig =  2;
    o_outdir  =  3;
  private
    FOutValue   : TSignal;
    FSteps      : Integer;
    FOldRes     : Boolean;
    FCounter    : Integer;
    FInState    : TSeqInState;
    FForwards   : Boolean;
    FReset      : Boolean;
  private
    procedure   SampleInputs;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModLeakyGateSeq = class( TMod)
  strict private
  const
    i_trig    =  0;
    i_step1   =  1;
    i_step2   =  2;
    i_step3   =  3;
    i_step4   =  4;
    i_step5   =  5;
    i_step6   =  6;
    i_step7   =  7;
    i_step8   =  8;
    i_step9   =  9;
    i_step10  = 10;
    i_step11  = 11;
    i_step12  = 12;
    i_step13  = 13;
    i_step14  = 14;
    i_step15  = 15;
    i_step16  = 16;
    i_chain   = 17;
    i_res     = 18;
    i_set     = 19;
    i_active  = 20;
    i_dir     = 21;
    i_rnd2    = 22;
    i_leak    = 23;
    i_leakmod = 24;
    i_dist    = 25;
    i_distmod = 26;
  const
    o_out     =  0;
    o_outres  =  1;
    o_outset  =  2;
    o_outtrig =  3;
    o_outdir  =  4;
  private
    FOutValue   : TSignal;
    FSteps      : Integer;
    FOldRes     : Boolean;
    FOldSet     : Boolean;
    FCounter    : Integer;
    FInState    : TSeqInState;
    FForwards   : Boolean;
    FReset      : Boolean;
    FSet        : Boolean;
    FOldRandom  : Boolean;
    FRndReq     : Boolean;
    FMemory     : array[ i_step1 .. i_step16] of TSignal;
  private
    procedure   SampleInputs;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
  end;


  TModLooper = class( TMod)
  strict private
  const
    i_in             =  0;
    i_rec            =  1;
    i_record         =  2;
    i_speed          =  3;
    i_speedmod       =  4;
    i_speedmodamt    =  5;
    i_start          =  6;
    i_startmod       =  7;
    i_startmodamt    =  8;
    i_duration       =  9;
    i_durationmod    = 10;
    i_durationmodamt = 11;
    i_mode           = 12;
    i_overlap        = 13;
    i_fm             = 14;
    i_fmamt          = 15;
  const
    o_out            =  0;
    o_collission     =  1;
  private
    FMemory      : TSignalArrayWrapper;
    FMemLength   : TSignal;
    FInPointer   : Integer;
    FLoopPointer : TSignal;
    FMode        : Integer;
    FRecording   : TSignal;
    FStart       : TSignal;
    FDuration    : TSignal;
    FIncrement   : TSignal;
    FOverlap     : TSignal;
  private
    FCollisionCounter : Integer;
    FColCounter       : Integer;
  private
    procedure   Recreate;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModStereoLooper = class( TMod)
  strict private
  const
    i_inl            =  0;
    i_inr            =  1;
    i_rec            =  2;
    i_record         =  3;
    i_speed          =  4;
    i_speedmod       =  5;
    i_speedmodamt    =  6;
    i_start          =  7;
    i_startmod       =  8;
    i_startmodamt    =  9;
    i_duration       = 10;
    i_durationmod    = 11;
    i_durationmodamt = 12;
    i_mode           = 13;
    i_overlap        = 14;
    i_fm             = 15;
    i_fmamt          = 16;
  const
    o_outl           =  0;
    o_outr           =  1;
    o_collission     =  2;
  private
    FMemoryL     : TSignalArrayWrapper;
    FMemoryR     : TSignalArrayWrapper;
    FMemLength   : TSignal;
    FInPointer   : Integer;
    FLoopPointer : TSignal;
    FMode        : Integer;
    FRecording   : TSignal;
    FStart       : TSignal;
    FDuration    : TSignal;
    FIncrement   : TSignal;
    FOverlap     : TSignal;
  private
    FCollisionCounter : Integer;
    FColCounter       : Integer;
  private
    procedure   Recreate;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModMemory = class( TMod)
  private
    FMemoryL : TSignalArrayWrapper;
    FMemoryR : TSignalArrayWrapper;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    destructor  Destroy;                                                                                       override;
  end;


  TModPitchShift = class( TMod)
  strict private
  const
    i_in          = 0;
    i_shift       = 1;
    i_shiftfine   = 2;
    i_shiftmod    = 3;
    i_shiftmodamt = 4;
    i_mute        = 5;
    i_delaymod    = 6;
    i_delaymodamt = 7;
  const
    o_out         = 0;
  const
    s_delay       = 0;
  private
    FDelayLine : TSignalArrayWrapper;
    FDelay     : TSignal;
    FLength    : Integer;
    FEffLength : Integer;
    FInPointer : Integer;
    FDelta     : TSignal;
    FPhase     : TSignal;
  private
    procedure   SetDelay( aValue: TSignal);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModDelay = class( TModMemory)
  strict private
  const
    i_in       = 0;
    i_delay    = 1;
  const
    o_out      = 0;
    o_outfixed = 1;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModDelayMix = class( TModMemory)
  strict private
  const
    i_in       = 0;
    i_delay    = 1;
    i_mix      = 2;
    i_mixmod   = 3;
  const
    o_out      = 0;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModDelayStereo = class( TModMemory)
  strict private
  const
    i_in        = 0;
    i_in2       = 1;
    i_delay     = 2;
  const
    o_out       = 0;
    o_out2      = 1;
    o_outfixed  = 2;
    o_outfixed2 = 3;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModDelayMod = class( TModMemory)
  strict private
  const
    i_in          = 0;
    i_delayoff    = 1;
    i_delaymod    = 2;
    i_delaymodamt = 3;
  const
    o_out         = 0;
    o_outfixed    = 1;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModDelay8 = class( TModMemory)
  strict private
  const
    i_in       = 0;
    i_delay1   = 1;
    i_delay2   = 2;
    i_delay3   = 3;
    i_delay4   = 4;
    i_delay5   = 5;
    i_delay6   = 6;
    i_delay7   = 7;
    i_delay8   = 8;
  const
    o_outfixed = 0;
    o_out1     = 1;
    o_out2     = 2;
    o_out3     = 3;
    o_out4     = 4;
    o_out5     = 5;
    o_out6     = 6;
    o_out7     = 7;
    o_out8     = 8;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModClockedDelay = class( TModMemory)
  strict private
  const
    i_in       = 0;
    i_clock    = 1;
  const
    o_out      = 0;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
    FOldClock  : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModClockedDelay16 = class( TMod) // Acts like a 16 stage shift regiseter
  strict private
  const
    i_in       =  0;
    i_clock    =  1;
    i_rate     =  2;
  const
    o_out0     =  0;
    o_out1     =  1;
    o_out2     =  2;
    o_out3     =  3;
    o_out4     =  4;
    o_out5     =  5;
    o_out6     =  6;
    o_out7     =  7;
    o_out8     =  8;
    o_out9     =  9;
    o_out10    = 10;
    o_out11    = 11;
    o_out12    = 12;
    o_out13    = 13;
    o_out14    = 14;
    o_out15    = 15;
  private
    FOldClock  : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModStereoWiden = class( TModMemory)
  strict private
  const
    i_inl         = 0;
    i_inr         = 1;
    i_delay       = 2;
    i_delayamt    = 3;
    i_delayamtmod = 4;
    i_widenamt    = 5;
    i_widenamtmod = 6;
    i_mute        = 7;
  const
    o_outl        = 0;
    o_outr        = 1;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModPhaseDistort = class( TModMemory)
  strict private
  const
    i_inl             =  0;
    i_inr             =  1;
    i_drywet          =  2;
    i_mute            =  3;
    i_mode            =  4;
    i_q               =  5;
    i_fm              =  6;
    i_fmlevel         =  7;
    i_freq            =  8;
    i_frequency       =  9;
    i_intensitymod    = 10;
    i_intensitymodlev = 11;
    i_intensity       = 12;
    i_delay           = 13;
  const
    o_outl            =  0;
    o_outr            =  1;
  private
    FMemLength : TSignal;
    FInPointer : Integer;
    FFreq      : TSignal;
    FDamp      : TSignal;
    FLP        : TSignal;
    FBP        : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModWave = class( TModMemory)
  strict private
  const
    o_outl = 0;
    o_outr = 1;
  private
    FFileName    : string;
    FDuration    : string;
    FIsStereo    : Boolean;
  private
    procedure   SetDuration( aValue: Cardinal);
    procedure   SwapMemoryFor( aMemory: TSignalArray);
    procedure   LoadFile;
    procedure   SetFileName( const aValue: string);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  public
    property    FileName: string read FFileName write SetFileName;
  end;


  TModWavePlayer = class( TModWave)
  strict private
  const
    i_speed     = 0;
    i_position  = 1;
    i_res       = 2;
    i_mode      = 3;
    i_fm        = 4;
  const
    o_outl      = 0;
    o_outr      = 1;
  private
    FPlayPointer : TSignal;
    FOldRes      : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModRndGranulator = class( TMod)
  strict private
  const
    i_in                  =  0;
    i_mute                =  1;
    i_envtype             =  2;
    i_interonset          =  3;
    i_interonsetmod       =  4;
    i_interonsetmodamt    =  5;
    i_grainduration       =  6;
    i_graindurationmod    =  7;
    i_graindurationmodamt =  8;
    i_volume              =  9;
    i_volumemod           = 10;
    i_volumemodamt        = 11;
    i_pan                 = 12;
    i_panmod              = 13;
    i_panmodamt           = 14;
    i_record              = 15;
    i_rec                 = 16;
    i_delay               = 17;
    i_grainoffset         = 18;
  const
    o_outl                =  0;
    o_outr                =  1;
  const
    s_delaytime           =  0;
    s_graincount          =  1;
  private
    FGranulator       : TRndGranulator;
    FDelayTime        : TSignal;
    FGrainCount       : Integer;
    FInterOnset       : TSignal;
    FInterOnsetMod    : TSignal;
    FGrainDuration    : TSignal;
    FGrainDurationMod : TSignal;
    FVolume           : TSignal;
    FVolumeMod        : TSignal;
    FPan              : TSignal;
    FPanMod           : TSignal;
    FRecording        : TSignal;
  private
    procedure   Recreate;
    procedure   SetDelayTime ( aValue: TSignal);
    procedure   SetGrainCount( aValue: Integer);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    DelayTime  : TSignal read FDelayTime  write SetDelayTime;
    property    GrainCount : Integer read FGrainCount write SetGrainCount;
  end;


  TModGranulator = class( TMod)
  strict private
  const
    i_in                  =  0;
    i_mute                =  1;
    i_envtype             =  2;
    i_interonset          =  3;
    i_interonsetmod       =  4;
    i_interonsetmodamt    =  5;
    i_grainduration       =  6;
    i_graindurationmod    =  7;
    i_graindurationmodamt =  8;
    i_volume              =  9;
    i_volumemod           = 10;
    i_volumemodamt        = 11;
    i_pan                 = 12;
    i_panmod              = 13;
    i_panmodamt           = 14;
    i_record              = 15;
    i_rec                 = 16;
    i_delay               = 17;
    i_grainoffset         = 18;
  const
    o_outl                =  0;
    o_outr                =  1;
  const
    s_delaytime           =  0;
    s_graincount          =  1;
  private
    FGranulator : TGranulator;
    FDelayTime  : TSignal;
    FGrainCount : Integer;
  private
    procedure   Recreate;
    procedure   SetDelayTime ( aValue: TSignal);
    procedure   SetGrainCount( aValue: Integer);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    DelayTime  : TSignal read FDelayTime  write SetDelayTime;
    property    GrainCount : Integer read FGrainCount write SetGrainCount;
  end;


  TModSimpleGranulator = class( TMod)
  strict private
  const
    i_in                  =  0;
    i_mute                =  1;
    i_envtype             =  2;
    i_grainduration       =  3;
    i_graindurationmod    =  4;
    i_graindurationmodamt =  5;
    i_record              =  6;
    i_rec                 =  7;
    i_delay               =  8;
    i_grainoffset         =  9;
    i_trig                = 10;
  const
    o_out                 =  0;
  const
    s_delaytime           =  0;
    s_graincount          =  1;
  private
    FGranulator : TSimpleGranulator;
    FDelayTime  : TSignal;
    FGrainCount : Integer;
    FOldTrig    : Boolean;
  private
    procedure   Recreate;
    procedure   SetDelayTime ( aValue: TSignal);
    procedure   SetGrainCount( aValue: Integer);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    DelayTime  : TSignal read FDelayTime  write SetDelayTime;
    property    GrainCount : Integer read FGrainCount write SetGrainCount;
  end;


  TModSampleAndHold = class( TMod)
  strict private
  const
    i_trig = 0;
    i_in   = 1;
  const
    o_out  = 0;
  private
    FPrevTrig  : Boolean;
    FTrigFlag  : Boolean;
    FTrigCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMultiSAndH = class( TMod)
  strict private
  const
    i_trig  =  0;
    i_in1   =  1;
    i_in2   =  2;
    i_in3   =  3;
    i_in4   =  4;
    i_in5   =  5;
    i_in6   =  6;
    i_in7   =  7;
    i_in8   =  8;
    i_in9   =  9;
    i_in10  = 10;
    i_in11  = 11;
    i_in12  = 12;
  const
    o_out1  =  0;
    o_out2  =  1;
    o_out3  =  2;
    o_out4  =  3;
    o_out5  =  4;
    o_out6  =  5;
    o_out7  =  6;
    o_out8  =  7;
    o_out9  =  8;
    o_out10 =  9;
    o_out11 = 10;
    o_out12 = 11;
  private
    FPrevTrig  : Boolean;
    FTrigFlag  : Boolean;
    FTrigCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSerPar = class( TMod)
  strict private
  const
    i_in     =  0;
    i_clkin  =  1;
    i_clkout =  2;
  const
    o_out1   =  0;
    o_out2   =  1;
    o_out3   =  2;
    o_out4   =  3;
    o_out5   =  4;
    o_out6   =  5;
    o_out7   =  6;
    o_out8   =  7;
    o_out9   =  8;
    o_out10  =  9;
    o_out11  = 10;
    o_cout   = 11;
  const
    OUT_COUNT = 11;
  private
    FState       : array[ 0 .. OUT_COUNT] of TSignal;
    FoldClkin    : Boolean;
    FOldClkOut   : Boolean;
    FClkInCount  : Integer;
    FClkOutCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMux = class( TMod)
  strict private
  const
    i_in0     =  0;
    i_in1     =  1;
    i_in2     =  2;
    i_in3     =  3;
    i_in4     =  4;
    i_in5     =  5;
    i_in6     =  6;
    i_in7     =  7;
    i_in8     =  8;
    i_in9     =  9;
    i_in10    = 10;
    i_in11    = 11;
    i_control = 12;
    i_xfade   = 13;
    i_steps   = 14;
  const
    o_out     =  0;
  private
    FState : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDemux = class( TMod)
  strict private
  const
    i_in      =  0;
    i_control =  1;
    i_xfade   =  2;
    i_hold    =  3;
    i_steps   =  4;
  const
    o_out0    =  0;
    o_out1    =  1;
    o_out2    =  2;
    o_out3    =  3;
    o_out4    =  4;
    o_out5    =  5;
    o_out6    =  6;
    o_out7    =  7;
    o_out8    =  8;
    o_out9    =  9;
    o_out10   = 10;
    o_out11   = 11;
  private
    FState : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDigiMux = class( TMod)
  strict private
  const
    i_in0     =  0;
    i_in1     =  1;
    i_in2     =  2;
    i_in3     =  3;
    i_in4     =  4;
    i_in5     =  5;
    i_in6     =  6;
    i_in7     =  7;
    i_sel0    =  8;
    i_sel1    =  9;
    i_sel2    = 10;
  const
    o_out     =  0;
  private
    FState : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDigiDemux = class( TMod)
  strict private
  const
    i_in   = 0;
    i_sel0 = 1;
    i_sel1 = 2;
    i_sel2 = 3;
    i_hold = 4;
  const
    o_out0 = 0;
    o_out1 = 1;
    o_out2 = 2;
    o_out3 = 3;
    o_out4 = 4;
    o_out5 = 5;
    o_out6 = 6;
    o_out7 = 7;
  private
    FState : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModTrackAndHold = class( TMod)
  strict private
  const
    i_trig = 0;
    i_in   = 1;
  const
    o_out  = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSwitch2to1 = class( TMod)
  strict private
  const
    i_control = 0;
    i_in1     = 1;
    i_in2     = 2;
  const
    o_out     = 0;
    o_out2    = 1;
  private
    FOn : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModXFade = class( TMod)
  strict private
  const
    i_control   = 0;
    i_in1       = 1;
    i_in2       = 2;
    i_position  = 3;
    i_posmod    = 4;
    i_intype    = 5;
    i_fadertype = 6;
  const
    o_out       = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModPan = class( TMod)
  strict private
  const
    i_control   = 0;
    i_in        = 1;
    i_position  = 2;
    i_posmod    = 3;
    i_intype    = 4;
    i_fadertype = 5;
  const
    o_out1      = 0;
    o_out2      = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixM3to1 = class( TMod)
  strict private
  const
    i_in      =  0;
    i_in1     =  1;
    i_in2     =  2;
    i_in3     =  3;
    i_mute1   =  4;
    i_mute2   =  5;
    i_mute3   =  6;
    i_level1  =  7;
    i_level2  =  8;
    i_level3  =  9;
    i_levmod1 = 10;
    i_levmod2 = 11;
    i_levmod3 = 12;
  const
    o_out     =  0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixM4to1 = class( TMod)
  strict private
  const
    i_in      =  0;
    i_in1     =  1;
    i_in2     =  2;
    i_in3     =  3;
    i_in4     =  4;
    i_mute1   =  5;
    i_mute2   =  6;
    i_mute3   =  7;
    i_mute4   =  8;
    i_level1  =  9;
    i_level2  = 10;
    i_level3  = 11;
    i_level4  = 12;
    i_levmod1 = 13;
    i_levmod2 = 14;
    i_levmod3 = 15;
    i_levmod4 = 16;
  const
    o_out     =  0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixM16to1 = class( TMod)
  strict private
  const
    i_in      =  0;
    i_in1     =  1;
    i_in2     =  2;
    i_in3     =  3;
    i_in4     =  4;
    i_in5     =  5;
    i_in6     =  6;
    i_in7     =  7;
    i_in8     =  8;
    i_in9     =  9;
    i_in10    = 10;
    i_in11    = 11;
    i_in12    = 12;
    i_in13    = 13;
    i_in14    = 14;
    i_in15    = 15;
    i_in16    = 16;
    i_lev1    = 17;
    i_lev2    = 18;
    i_lev3    = 19;
    i_lev4    = 20;
    i_lev5    = 21;
    i_lev6    = 22;
    i_lev7    = 23;
    i_lev8    = 24;
    i_lev9    = 25;
    i_lev10   = 26;
    i_lev11   = 27;
    i_lev12   = 28;
    i_lev13   = 29;
    i_lev14   = 30;
    i_lev15   = 31;
    i_lev16   = 32;
    i_mute    = 33;
  const
    o_out     =  0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixMS8to1 = class( TMod)
  strict private
  const
    i_inl     =  0;
    i_inr     =  1;
    i_inl1    =  2;
    i_inr1    =  3;
    i_lev1    =  4;
    i_levmod1 =  5;
    i_mute1   =  6;
    i_inl2    =  7;
    i_inr2    =  8;
    i_lev2    =  9;
    i_levmod2 = 10;
    i_mute2   = 11;
    i_inl3    = 12;
    i_inr3    = 13;
    i_lev3    = 14;
    i_levmod3 = 15;
    i_mute3   = 16;
    i_inl4    = 17;
    i_inr4    = 18;
    i_lev4    = 19;
    i_levmod4 = 20;
    i_mute4   = 21;
    i_inl5    = 22;
    i_inr5    = 23;
    i_lev5    = 24;
    i_levmod5 = 25;
    i_mute5   = 26;
    i_inl6    = 27;
    i_inr6    = 28;
    i_lev6    = 29;
    i_levmod6 = 30;
    i_mute6   = 31;
    i_inl7    = 32;
    i_inr7    = 33;
    i_lev7    = 34;
    i_levmod7 = 35;
    i_mute7   = 36;
    i_inl8    = 37;
    i_inr8    = 38;
    i_lev8    = 39;
    i_levmod8 = 40;
    i_mute8   = 41;
  const
    o_outl    =  0;
    o_outr    =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixMM16to1 = class( TMod)
  strict private
  const
    i_in       =  0;
    i_in1      =  1;
    i_in2      =  2;
    i_in3      =  3;
    i_in4      =  4;
    i_in5      =  5;
    i_in6      =  6;
    i_in7      =  7;
    i_in8      =  8;
    i_in9      =  9;
    i_in10     = 10;
    i_in11     = 11;
    i_in12     = 12;
    i_in13     = 13;
    i_in14     = 14;
    i_in15     = 15;
    i_in16     = 16;
    i_lev1     = 17;
    i_lev2     = 18;
    i_lev3     = 19;
    i_lev4     = 20;
    i_lev5     = 21;
    i_lev6     = 22;
    i_lev7     = 23;
    i_lev8     = 24;
    i_lev9     = 25;
    i_lev10    = 26;
    i_lev11    = 27;
    i_lev12    = 28;
    i_lev13    = 29;
    i_lev14    = 30;
    i_lev15    = 31;
    i_lev16    = 32;
    i_levmod1  = 33;
    i_levmod2  = 34;
    i_levmod3  = 35;
    i_levmod4  = 36;
    i_levmod5  = 37;
    i_levmod6  = 38;
    i_levmod7  = 39;
    i_levmod8  = 40;
    i_levmod9  = 41;
    i_levmod10 = 42;
    i_levmod11 = 43;
    i_levmod12 = 44;
    i_levmod13 = 45;
    i_levmod14 = 46;
    i_levmod15 = 47;
    i_levmod16 = 48;
    i_mute     = 49;
  const
    o_out      =  0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModVolume = class( TMod)
  strict private
  const
    i_inl      = 0;
    i_inr      = 1;
    i_level1   = 2;
    i_level2   = 3;
    i_levelmod = 4;
    i_mute     = 5;
  const
    o_outl     = 0;
    o_outr     = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixS2to1 = class( TMod)
  strict private
  const
    i_inl     =  0;
    i_inr     =  1;
    i_inl1    =  2;
    i_inr1    =  3;
    i_inl2    =  4;
    i_inr2    =  5;
    i_mute1   =  6;
    i_mute2   =  7;
    i_level1  =  8;
    i_level2  =  9;
    i_levmod1 = 10;
    i_levmod2 = 11;
  const
    o_outl    =  0;
    o_outr    =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixS5to1 = class( TMod)
  strict private
  const
    i_inl     =  0;
    i_inr     =  1;
    i_inl1    =  2;
    i_inr1    =  3;
    i_inl2    =  4;
    i_inr2    =  5;
    i_inl3    =  6;
    i_inr3    =  7;
    i_inl4    =  8;
    i_inr4    =  9;
    i_inl5    = 10;
    i_inr5    = 11;
    i_mute1   = 12;
    i_mute2   = 13;
    i_mute3   = 14;
    i_mute4   = 15;
    i_mute5   = 16;
    i_level1  = 17;
    i_level2  = 18;
    i_level3  = 19;
    i_level4  = 20;
    i_level5  = 21;
    i_levmod1 = 22;
    i_levmod2 = 23;
    i_levmod3 = 24;
    i_levmod4 = 25;
    i_levmod5 = 26;
  const
    o_outl    =  0;
    o_outr    =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixT1to1 = class( TMod)
  strict private
  const
    i_inl     = 0;
    i_inm     = 1;
    i_inr     = 2;
    i_inl1    = 3;
    i_inm1    = 4;
    i_inr1    = 5;
    i_mute1   = 6;
    i_level1  = 7;
    i_levmod1 = 8;
  const
    o_outl    = 0;
    o_outm    = 1;
    o_outr    = 2;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixQ1to1 = class( TMod)
  strict private
  const
    i_inll    =  0;
    i_inlr    =  1;
    i_inrl    =  2;
    i_inrr    =  3;
    i_inll1   =  4;
    i_inlr1   =  5;
    i_inrl1   =  6;
    i_inrr1   =  7;
    i_mute1   =  8;
    i_level1  =  9;
    i_levmod1 = 10;
  const
    o_outll    = 0;
    o_outlr    = 1;
    o_outrl    = 2;
    o_outrr    = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixM6to6 = class( TMod)
  strict private
  const
    i_in1       =  0;
    i_in2       =  1;
    i_in3       =  2;
    i_in4       =  3;
    i_in5       =  4;
    i_in6       =  5;
    i_mute1     =  6;
    i_mute2     =  7;
    i_mute3     =  8;
    i_mute4     =  9;
    i_mute5     = 10;
    i_mute6     = 11;
    i_inchcol1  = 12;
    i_inchcol2  = 13;
    i_inchcol3  = 14;
    i_inchcol4  = 15;
    i_inchcol5  = 16;
    i_inchcol6  = 17;
    i_levelr1c1 = 18;
    i_levelr1c2 = 19;
    i_levelr1c3 = 20;
    i_levelr1c4 = 21;
    i_levelr1c5 = 22;
    i_levelr1c6 = 23;
    i_levelr2c1 = 24;
    i_levelr2c2 = 25;
    i_levelr2c3 = 26;
    i_levelr2c4 = 27;
    i_levelr2c5 = 28;
    i_levelr2c6 = 29;
    i_levelr3c1 = 30;
    i_levelr3c2 = 31;
    i_levelr3c3 = 32;
    i_levelr3c4 = 33;
    i_levelr3c5 = 34;
    i_levelr3c6 = 35;
    i_levelr4c1 = 36;
    i_levelr4c2 = 37;
    i_levelr4c3 = 38;
    i_levelr4c4 = 39;
    i_levelr4c5 = 40;
    i_levelr4c6 = 41;
    i_levelr5c1 = 42;
    i_levelr5c2 = 43;
    i_levelr5c3 = 44;
    i_levelr5c4 = 45;
    i_levelr5c5 = 46;
    i_levelr5c6 = 47;
    i_levelr6c1 = 48;
    i_levelr6c2 = 49;
    i_levelr6c3 = 50;
    i_levelr6c4 = 51;
    i_levelr6c5 = 52;
    i_levelr6c6 = 53;
    i_inr1c1    = 54;
    i_inr1c2    = 55;
    i_inr1c3    = 56;
    i_inr1c4    = 57;
    i_inr1c5    = 58;
    i_inr1c6    = 59;
    i_inr2c1    = 60;
    i_inr2c2    = 61;
    i_inr2c3    = 62;
    i_inr2c4    = 63;
    i_inr2c5    = 64;
    i_inr2c6    = 65;
    i_inr3c1    = 66;
    i_inr3c2    = 67;
    i_inr3c3    = 68;
    i_inr3c4    = 69;
    i_inr3c5    = 70;
    i_inr3c6    = 71;
    i_inr4c1    = 72;
    i_inr4c2    = 73;
    i_inr4c3    = 74;
    i_inr4c4    = 75;
    i_inr4c5    = 76;
    i_inr4c6    = 77;
    i_inr5c1    = 78;
    i_inr5c2    = 79;
    i_inr5c3    = 80;
    i_inr5c4    = 81;
    i_inr5c5    = 82;
    i_inr5c6    = 83;
    i_inr6c1    = 84;
    i_inr6c2    = 85;
    i_inr6c3    = 86;
    i_inr6c4    = 87;
    i_inr6c5    = 88;
    i_inr6c6    = 89;
  const
    o_out1      =  0;
    o_out2      =  1;
    o_out3      =  2;
    o_out4      =  3;
    o_out5      =  4;
    o_out6      =  5;
    o_outchrow1 =  6;
    o_outchrow2 =  7;
    o_outchrow3 =  8;
    o_outchrow4 =  9;
    o_outchrow5 = 10;
    o_outchrow6 = 11;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixAudio = class( TMod)
  strict private
  const
    i_insolo    =  0;
    i_inchain1l =  1;
    i_inchain1r =  2;
    i_inchain2l =  3;
    i_inchain2r =  4;
    i_in1       =  5;
    i_in2       =  6;
    i_in3       =  7;
    i_in4       =  8;
    i_in5       =  9;
    i_in6       = 10;
    i_in7       = 11;
    i_in8       = 12;
    i_pan1      = 13;
    i_pan2      = 14;
    i_pan3      = 15;
    i_pan4      = 16;
    i_pan5      = 17;
    i_pan6      = 18;
    i_pan7      = 19;
    i_pan8      = 20;
    i_vol11     = 21;
    i_vol12     = 22;
    i_vol13     = 23;
    i_vol14     = 24;
    i_vol15     = 25;
    i_vol16     = 26;
    i_vol17     = 27;
    i_vol18     = 28;
    i_vol21     = 29;
    i_vol22     = 30;
    i_vol23     = 31;
    i_vol24     = 32;
    i_vol25     = 33;
    i_vol26     = 34;
    i_vol27     = 35;
    i_vol28     = 36;
    i_mute1     = 37;
    i_mute2     = 38;
    i_mute3     = 39;
    i_mute4     = 40;
    i_mute5     = 41;
    i_mute6     = 42;
    i_mute7     = 43;
    i_mute8     = 44;
    i_solo1     = 45;
    i_solo2     = 46;
    i_solo3     = 47;
    i_solo4     = 48;
    i_solo5     = 49;
    i_solo6     = 50;
    i_solo7     = 51;
    i_solo8     = 52;
    i_levmod11  = 53;
    i_levmod12  = 54;
    i_levmod13  = 55;
    i_levmod14  = 56;
    i_levmod15  = 57;
    i_levmod16  = 58;
    i_levmod17  = 59;
    i_levmod18  = 60;
    i_levmod21  = 61;
    i_levmod22  = 62;
    i_levmod23  = 63;
    i_levmod24  = 64;
    i_levmod25  = 65;
    i_levmod26  = 66;
    i_levmod27  = 67;
    i_levmod28  = 68;
  const
    o_outsolo   =  0;
    o_out1l     =  1;
    o_out1r     =  2;
    o_out2l     =  3;
    o_out2r     =  4;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMixQS2to1 = class( TMod)
  strict private
  const
    i_in1l1    =  0;
    i_in1r1    =  1;
    i_level11  =  2;
    i_mute1in1 =  3;
    i_in1l2    =  4;
    i_in1r2    =  5;
    i_level12  =  6;
    i_mute1in2 =  7;
    i_in1l3    =  8;
    i_in1r3    =  9;
    i_level13  = 10;
    i_mute1in3 = 11;
    i_in1l4    = 12;
    i_in1r4    = 13;
    i_level14  = 14;
    i_mute1in4 = 15;
    i_in2l1    = 16;
    i_in2r1    = 17;
    i_level21  = 18;
    i_mute2in1 = 19;
    i_in2l2    = 20;
    i_in2r2    = 21;
    i_level22  = 22;
    i_mute2in2 = 23;
    i_in2l3    = 24;
    i_in2r3    = 25;
    i_level23  = 26;
    i_mute2in3 = 27;
    i_in2l4    = 28;
    i_in2r4    = 29;
    i_level24  = 30;
    i_mute2in4 = 31;
    i_muteout1 = 32;
    i_muteout2 = 33;
    i_muteout3 = 34;
    i_muteout4 = 35;
  const
    o_outl1    =  0;
    o_outr1    =  1;
    o_outl2    =  2;
    o_outr2    =  3;
    o_outl3    =  4;
    o_outr3    =  5;
    o_outl4    =  6;
    o_outr4    =  7;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMixQS1to1 = class( TMod)
  strict private
  const
    i_inl1     =  0;
    i_inr1     =  1;
    i_inl2     =  2;
    i_inr2     =  3;
    i_inl3     =  4;
    i_inr3     =  5;
    i_inl4     =  6;
    i_inr4     =  7;
    i_in1l1    =  8;
    i_in1r1    =  9;
    i_level1   = 10;
    i_mutein1  = 11;
    i_in1l2    = 12;
    i_in1r2    = 13;
    i_level2   = 14;
    i_mutein2  = 15;
    i_in1l3    = 16;
    i_in1r3    = 17;
    i_level3   = 18;
    i_mutein3  = 19;
    i_in1l4    = 20;
    i_in1r4    = 21;
    i_level4   = 22;
    i_mutein4  = 23;
    i_muteout1 = 24;
    i_muteout2 = 25;
    i_muteout3 = 26;
    i_muteout4 = 27;
  const
    o_outl1    =  0;
    o_outr1    =  1;
    o_outl2    =  2;
    o_outr2    =  3;
    o_outl3    =  4;
    o_outr3    =  5;
    o_outl4    =  6;
    o_outr4    =  7;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModQuadPan = class( TMod)
  strict private
  const
    i_in1    = 0;
    i_level1 = 1;
    i_in2    = 2;
    i_level2 = 3;
    i_in3    = 4;
    i_level3 = 5;
    i_in4    = 6;
    i_level4 = 7;
  const
    o_outl1  = 0;
    o_outr1  = 1;
    o_outl2  = 2;
    o_outr2  = 3;
    o_outl3  = 4;
    o_outr3  = 5;
    o_outl4  = 6;
    o_outr4  = 7;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModQuantize = class( TMod)
  strict private
  const
    i_qsteps   = 0;
    i_in       = 1;
    i_mute     = 2;
    i_level    = 3;
    i_levelmod = 4;
    i_mode     = 5;
    i_rtype    = 6;
    i_nozero   = 7;
  const
    o_out      = 0;
    o_outinv   =  1;
  private
    FLevel : TSignal;
    FMode  : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModNoteQuantize = class( TMod)
  strict private
  const
    i_in     =  0;
    i_note0  =  1;
    i_note1  =  2;
    i_note2  =  3;
    i_note3  =  4;
    i_note4  =  5;
    i_note5  =  6;
    i_note6  =  7;
    i_note7  =  8;
    i_note8  =  9;
    i_note9  = 10;
    i_note10 = 11;
    i_note11 = 12;
    i_level  = 13;
    i_levmod = 14;
  const
    o_out    =  0;
  private
    FFilter : TByteSet;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModScaleQuantizer = class( TMod)
  strict private
  const
    i_in       =  0;
    i_scale    =  1;
    i_basenote =  2;
    i_note0    =  3;
    i_note1    =  4;
    i_note2    =  5;
    i_note3    =  6;
    i_note4    =  7;
    i_note5    =  8;
    i_note6    =  9;
    i_note7    = 10;
    i_note8    = 11;
    i_note9    = 12;
    i_note10   = 13;
    i_note11   = 14;
    i_level    = 15;
    i_chain    = 16;
    i_levmod   = 17;
    i_scalemod = 18;
  const
    o_out      =  0;
    o_changed  =  1;
  private
    FFilter         : TByteSet;
    FScale          : Integer;
    FScaleChanged   : Boolean;
    FNotes          : TScaleNoteSet;
    FPrevOutVal     : TSignal;
    FChangedCounter : Integer;
    FLedCounter     : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModModQuantizer = class( TMod)
  strict private
  const
    i_in       =  0;
    i_note0    =  1;
    i_note1    =  2;
    i_note2    =  3;
    i_note3    =  4;
    i_note4    =  5;
    i_note5    =  6;
    i_note6    =  7;
    i_note7    =  8;
    i_note8    =  9;
    i_note9    = 10;
    i_note10   = 11;
    i_note11   = 12;
    i_ctrl0    = 13;
    i_ctrl1    = 14;
    i_ctrl2    = 15;
    i_ctrl3    = 16;
    i_ctrl4    = 17;
    i_ctrl5    = 18;
    i_ctrl6    = 19;
    i_ctrl7    = 20;
    i_ctrl8    = 21;
    i_ctrl9    = 22;
    i_ctrl10   = 23;
    i_ctrl11   = 24;
    i_level    = 25;
    i_basenote = 26;
    i_chain    = 27;
    i_levmod   = 28;
  const
    o_out      =  0;
  private
    FFilter : TByteSet;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModAutoQuantizer = class( TMod)
  strict private
  const
    i_learn      =  0;
    i_clock      =  1;
    i_history    =  2;
    i_historymod =  3;
    i_range      =  4;
    i_rangemod   =  5;
    i_in         =  6;
  const
    o_out        =  0;
    o_changed    =  1;
  const
    m_max_size   = 64;
  private
    FFilter       : array[ 0 .. m_max_size - 1] of TSignal;
    FInp          : Integer;
    FOutp         : Integer;
    FFillCount    : Integer;
    FFreeCount    : Integer;
    FSize         : Integer;
    FClockCount   : Integer;
    FPulseCount   : Integer;
    FChangedCount : Integer;
    FOldClock     : Boolean;
    FOldNote      : TSignal;
  private
    procedure   ClearHistory;
    function    FindNote( aValue: TSignal): Integer;
    function    FindNearestNote( aValue: TSignal): TSignal;
    procedure   DropNote;
    procedure   AddNote( aValue: TSignal);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModFindNearest = class( TMod)
  strict private
  const
    i_learn      =  0;
    i_clock      =  1;
    i_history    =  2;
    i_historymod =  3;
    i_range      =  4;
    i_rangemod   =  5;
    i_in         =  6;
  const
    o_out        =  0;
    o_changed    =  1;
  const
    m_max_size   = 64;
  private
    FHistory      : array[ 0 .. m_max_size - 1] of TSignal;
    FInp          : Integer;
    FOutp         : Integer;
    FFillCount    : Integer;
    FFreeCount    : Integer;
    FSize         : Integer;
    FClockCount   : Integer;
    FPulseCount   : Integer;
    FChangedCount : Integer;
    FOldClock     : Boolean;
    FOldValue     : TSignal;
  private
    procedure   ClearHistory;
    function    FindValue( aValue: TSignal): Integer;
    function    FindNearestValue( aValue: TSignal): TSignal;
    procedure   DropValue;
    procedure   AddValue( aValue: TSignal);
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModQueue = class( TMod)
  strict private
  const
    i_in           =  0;
    i_clock        =  1;
    i_history      =  2;
    i_historymod   =  3;
    i_select       =  4;
    i_selectmod    =  5;
    i_selectmodamt =  6;
    i_xfade        =  7;
    i_xfademod     =  8;
  const
    o_out          =  0;
  const
    m_max_size     = 64;
  private
    FHistory      : array[ 0 .. m_max_size - 1] of TSignal;
    FInp          : Integer;
    FOutp         : Integer;
    FFillCount    : Integer;
    FFreeCount    : Integer;
    FSize         : Integer;
    FClockCount   : Integer;
    FOldClock     : Boolean;
  private
    procedure   ClearHistory;
    procedure   DropValue;
    procedure   AddValue( aValue: TSignal);
    function    Interpolate( const anIndex, anAmount: TSignal; aSize: Integer): TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModNoteSplit = class( TMod)
  strict private
  const
    i_in     = 0;
  const
    o_octave = 0;
    o_note   = 1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModNoteMerge = class( TMod)
  strict private
  const
    i_octave = 0;
    i_note   = 1;
  const
    o_out    = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModHardClip = class( TMod)
  strict private
  const
    i_in        = 0;
    i_mute      = 1;
    i_lowlevel  = 2;
    i_highlevel = 3;
    i_outamp    = 4;
  const
    o_out       = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModCubicClip = class( TMod)
  strict private
  const
    i_in            = 0;
    i_mute          = 1;
    i_alpha         = 2;
    i_hysteresis    = 3;
    i_alphamod      = 4;
    i_hysteresismod = 5;
  const
    o_out           = 0;
  private
    FAlpha    : TSignal;
    FAlphaRec : TSignal;
    FOutValue : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModCuClipStereo = class( TMod)
  strict private
  const
    i_inl        = 0;
    i_inr        = 1;
    i_mute       = 2;
    i_alpha      = 3;
    i_hysteresis = 4;
  const
    o_outl       = 0;
    o_outr       = 1;
  private
    FAlpha     : TSignal;
    FAlphaRec  : TSignal;
    FOutValueL : TSignal;
    FOutValueR : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModTransInv = class( TMod)
  strict private
  const
    i_in1       =  0;
    i_in2       =  1;
    i_in3       =  2;
    i_in4       =  3;
    i_amtpivot  =  4;
    i_modulo    =  5;
    i_mode      =  6;
    i_chain1    =  7;
    i_chain2    =  8;
    i_chain3    =  9;
    i_chain4    = 10;
  const
    o_out1      =  0;
    o_out2      =  1;
    o_out3      =  2;
    o_out4      =  3;
  const
    m_transpose =  0;
    m_invert    =  1;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModTranspose = class( TMod)
  strict private
  const
    i_in      = 0;
    i_amount1 = 1;
    i_amount2 = 2;
    i_amount3 = 3;
    i_amount4 = 4;
    i_active1 = 5;
    i_active2 = 6;
    i_active3 = 7;
    i_active4 = 8;
  const
    o_out     = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModScala = class( TMod)
  strict private
  const
    i_in  = 0;
  const
    i_out = 0;
  private
    FMappingFileName : string;
    FScaleFileName   : string;
    FScale           : TScalaScale;
    FPrevNoteNumber  : Integer;
    FPrevOut         : TSignal;
  private
    procedure   SetMappingFileName( const aValue: string);
    procedure   SetScaleFileName  ( const aValue: string);
    function    Lookup( aValue: TSignal): TSignal;
    procedure   RecalculateTuning;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   TuningChanged;                                                                                 override;
  public
    property    MappingFileName : string read FMappingFileName write SetMappingFileName;
    property    ScaleFileName   : string read FScaleFileName   write SetScaleFileName;
  end;


  TModTerraGen = class( TMod)
  strict private
  const
    i_x              =  0;
    i_y              =  1;
    i_mute           =  2;
    i_algo           =  3;
  const
    o_out            =  0;
  const
    m_off            =  0; // Output is zero
    m_inx            =  1; // Just x input
    m_iny            =  2; // Just y input
    m_xy             =  3; // Just y input
    m_curtis_roads   =  4; // Curtis roads standdard example
    m_bass           =  5; // all others from http://mymbs.mobeard.org/~pfisher/FOV2-0010016C/FOV2-0010016E/FOV2-001001A3/chapters/26mikelson/index%20.html
    m_lead1          =  6;
    m_hollow_lead    =  7;
    m_pulsar         =  8;
    m_flat_growl     =  9;
    m_atan           = 10; // except this one, standard arc tangent function
    m_softclip       = 11; // except these, these are from http://music.columbia.edu/cmc/musicandcomputers/chapter4/04_06.php
    m_third          = 12;
    m_cheb2          = 13;
    m_cheb3          = 14;
    m_cheb4          = 15;
    m_cheb5          = 16;
    m_cheb6          = 17;
    m_cheb7          = 18;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModChebGen = class( TMod)
  strict private
  const
    i_in       =  0;
    i_wmode    =  1;
    i_shapemod =  2;
  const
    o_out2     =  0;
    o_out3     =  1;
    o_out4     =  2;
    o_out5     =  3;
    o_out6     =  4;
    o_out7     =  5;
    o_out8     =  6;
    o_out9     =  7;
    o_out10    =  8;
    o_out11    =  9;
    o_out12    = 10;
    o_out13    = 11;
  const
    m_sine     =  0;
    m_tri      =  1;
    m_saw      =  2;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModSawMill = class( TMod)
  strict private
  const
    i_in         = 0;
    i_mult       = 1;
    i_multmod    = 2;
    i_multmodamt = 3;
    i_inputtype  = 4;
    i_outputtype = 5;
  const
    o_out        = 0;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRungler = class( TMod)
  // After an idea from Rob Hordijk - for more info :: http://hordijk-synths.info
  strict private
  const
    i_clock  = 0;
    i_data   = 1;
    i_level  = 2;
    i_length = 3;
    i_taps   = 4;
  const
    o_out    = 0;
    o_bit    = 1;
  private
    FShiftReg : UInt64;   // The shift register
    FPrevTap  : Integer;  // Bit index of Prev shift register tap to use [ FTapCount, 64]
    FTapCount : Integer;  // The number of taps to be actually used      [ 1, FPrevTap]    - for now set fixed to 3
    FPrevBit  : Boolean;
    FOldClock : Boolean;
    FOutValue : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPoetry = class( TMod)
  strict private
  const
    i_in         = 0;
    i_newinput   = 1;
    i_newoutput  = 2;
    i_clr        = 3;
    i_trans      = 4;
  const
    o_out        = 0;
    o_changed    = 1;
  const
    s_history    = 0;
    s_seedlen    = 1;
  private
    FGenerator         : TPoetryGenerator;
    FHistory           : Integer;
    FSeedLen           : Integer;
    FEpsilon           : TSignal;
    FOldNewInput       : Boolean;
    FOldNewOutput      : Boolean;
    FTrigCount1        : Integer;
    FTrigCount2        : Integer;
    FOldClr            : Boolean;
    FMustClr           : Boolean;
    FClrCounter        : Integer;
    FPrevInValue       : TSignal;
    FOldOutValue       : TSignal;
    FTransparant       : Boolean;
    FChangedCounter    : Integer;
    FChangedLedCounter : Integer;
  private
    procedure   Recreate;
    procedure   SetHistory( aValue: Integer);
    procedure   SetSeedLen( aValue: Integer);
  private
    property    History : Integer read FHistory write SetHistory;
    property    SeedLen : Integer read FSeedLen write SetSeedLen;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModPluck = class( TMod)
  strict private
  const
    i_pluck       =  0;
    i_plucklevel  =  1;
    i_feedback    =  2;
    i_brightness  =  3;
    i_brighctrl   =  4;
    i_bclevel     =  5;
    i_fm          =  6;
    i_fmlevel     =  7;
    i_frequency   =  8;
    i_cents       =  9;
    i_mute        = 10;
    i_noise       = 11;
    i_noisemod    = 12;
    i_noisedepth  = 13;
    i_freq        = 14;
    i_delaymod    = 15;
    i_delaymodamt = 16;
  const
    o_out         =  0;
  private
    FMemory      : TSignalArrayWrapper;
    FDelayTime   : TSignal;
    FInPointer   : Integer;
    FOutValue    : TSignal;
    FOldFeedback : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;


  TModKS = class( TMod)
  strict private
  const
    i_pluck       =  0;
    i_plucklevel  =  1;
    i_feedback    =  2;
    i_brightness  =  3;
    i_fm          =  4;
    i_fmlevel     =  5;
    i_frequency   =  6;
    i_cents       =  7;
    i_mute        =  8;
    i_freq        =  9;
    i_mode        = 10;
  const
    o_out         =  0;
  private
    FMemory      : TSignalArrayWrapper;
    FDelayTime   : TSignal;
    FInPointer   : Integer;
    FOutValue    : TSignal;
    FOldFeedback : TSignal;
    FDelay       : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModAllPass = class( TMod)
  strict private
  const
    i_in      = 0;
    i_k       = 1;
    i_kmod    = 2;
    i_kmodamt = 3;
  const
    o_out     = 0;
  private
    FK        : TSignal;
    FLength   : Integer;
    FMemory   : TSignalArrayWrapper;
    FPos      : Integer;
    FLedCount : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModValues = class( TMod)
  strict private
  const
    i_in      = 0;
    i_intype  = 1;
    i_outtype = 2;
  const
    o_out     = 0;
    o_changed = 1;
  private
    FValue      : string;
    FNewValue   : string;
    FLowValue   : TSignal;
    FHighValue  : TSignal;
    FValueCount : Integer;
    FValues     : TSignalArray;
    FShowValue  : TSignal;
    FOutValue   : TSignal;
    FError      : Boolean;
    FChanged    : Integer;
    FLedCount   : Integer;
  private
    procedure   ParseValue;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModDisplay = class( TMod)
  strict private
  const
    i_in1   = 0;
    i_in2   = 1;
    i_mode  = 2;
    i_speed = 3;
  private
    FMax1 : TSignal;
    FMax2 : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModValue = class( TMod)
  strict private
  const
    i_in    = 0;
    i_mode  = 1;
    i_reset = 2;
  private
    FValue    : TSignal;
    FMaxValue : TSignal;
    FMode     : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModCount = class( TMod)
  strict private
  const
    i_count = 0;
    i_res   = 1;
    i_reset = 2;
  private
    FCount    : UInt64;
    FOldCount : Boolean;
    FOldRes   : Boolean;
    FNewReset : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModFreqCount = class( TMod)
  strict private
  const
    i_count    = 0;
    i_res      = 1;
    i_reset    = 2;
    i_timebase = 3;
  private
    FCount     : UInt64;
    FPrevCount : UInt64;
    FOldCount  : Boolean;
    FOldRes    : Boolean;
    FNewReset  : Boolean;
    FTimer     : Int64;
    FOldRate   : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
  end;


  TModLiveMorph = class( TMod)
  strict private
  const
    i_in        = 0;
    i_inputtype = 1;
    i_active    = 2;
    i_fractions = 3;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModMorphControl = class( TMod)
  strict private
  const
    i_in1       =  0;
    i_in2       =  1;
    i_in3       =  2;
    i_in4       =  3;
    i_intype1   =  4;
    i_intype2   =  5;
    i_intype3   =  6;
    i_intype4   =  7;
    i_mod1      =  8;
    i_mod2      =  9;
    i_mod3      = 10;
    i_mod4      = 11;
    i_modamt1   = 12;
    i_modamt2   = 13;
    i_modamt3   = 14;
    i_modamt4   = 15;
    i_level1    = 16;
    i_level2    = 17;
    i_level3    = 18;
    i_level4    = 19;
    i_active1   = 20;
    i_active2   = 21;
    i_active3   = 22;
    i_active4   = 23;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModAuto = class( TMod)
  strict private
  const
    i_clock  = 0;
    i_active = 1;
    i_type   = 2;
  private
    FOldClock  : Boolean;
    FLedCounter : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModSync = class( TMod)
  strict private
  const
    i_sync = 0;
  const
    o_sync = 0;
  private
    FOscBytes    : TBytes;
    FSync        : Boolean;
    FOldSync     : Boolean;
    FLedCounter  : Integer;
    FSyncCounter : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   AcceptOSC( const aSynthName: string; const aMsg: TOSCPacket);                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModOscMessage = class( TMod)
  strict private
  const
    i_trigger = 0;
    i_input   = 1;
  const
    o_out     = 0;
  private
    FAddress    : string;
    FValue      : Single;
    FValuePtr   : PSingle;
    FOscBytes   : TBytes;
    FOldTrigger : Boolean;
    FLedCounter : Integer;
  private
    procedure   Recreate;
    procedure   SetAddress( const aValue: string);
    procedure   SetValue( const aValue: Single);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   AcceptOSC( const aSynthName: string; const aMsg: TOSCPacket);                                  override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    Address : string read FAddress write SetAddress;
    property    Value   : Single read FValue   write SetValue;
  end;


  TModMidiNoteIn = class( TMod)
  strict private
  const
    i_ch     = 0;
  const
    o_gate   = 0;
    o_note   = 1;
    o_velon  = 2;
    o_veloff = 3;
  private
    FChannel     : Byte;
    FPrevNote    : Byte;
    FPrevChannel : Byte;
    FOldChannel  : Byte;
    FGate        : Boolean;
    FNote        : TSignal;
    FVelocityOn  : TSignal;
    FVelocityOff : TSignal;
  public
    procedure   NotesOff;
    procedure   CreateIO;                                                                                      override;
    procedure   AcceptMidi( const aMsg: TMidiMessage);                                                         override;
    procedure   Panic;                                                                                         override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TMidiNoteData = record
    NoteNr      : TSignal;         // a note number identifying this record
    Channel     : Byte;            // The channel the message was received on
    GateValue   : Boolean;         // The current state of the Gate signal, true signaling the record to be in use
    OutValue    : TSignal;         // The actual value output for this note
    VelocityOn  : TSignal;         // The note on velocity
    VelocityOff : TSignal;         // the note off velocity
    AfterTouch  : TSignal;         // the note aftertouch
    Counter     : UInt64;          // counts how long the gate has been on in control rate cycles
    Level       : TSignal;         // The signal level for this note
  end;


  TModMidiMultiNoteIn = class( TMod)
  strict private
  const
    i_ch      =  0;
    i_level   =  1;
    i_level2  =  2;
    i_level3  =  3;
    i_level4  =  4;
    i_mode    =  5;
  const
    o_gate    =  0;
    o_note    =  1;
    o_velon   =  2;
    o_veloff  =  3;
    o_after   =  4;
    o_gate2   =  5;
    o_note2   =  6;
    o_velon2  =  7;
    o_veloff2 =  9;
    o_after2  = 10;
    o_gate3   = 11;
    o_note3   = 12;
    o_velon3  = 13;
    o_veloff3 = 14;
    o_after3  = 15;
    o_gate4   = 16;
    o_note4   = 17;
    o_velon4  = 18;
    o_veloff4 = 19;
    o_after4  = 20;
    o_bend    = 21;
  const
    m_none    =  0; // no note stealing, skip new notes till room
    m_cyclic  =  1; // round robin
    m_softest =  2; // replace softest note
    m_oldest  =  3; // replace oldest  note
    m_lowest  =  4; // replace lowest  note
    m_highest =  5; // replace highest note
    m_newest  =  6; // replace newest  note
  private
    FChannel      : Byte;
    FOldChannel   : Byte;
    FNoteData     : array[ 0 .. 3] of TMidiNoteData;
    FCurrentOut   : Integer;
    FBend         : TSignal;
    FMode         : Integer;
    FOldMode      : Integer;
  public
    procedure   NotesOff;
    procedure   CreateIO;                                                                                      override;
    procedure   AcceptMidi( const aMsg: TMidiMessage);                                                         override;
    procedure   Panic;                                                                                         override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiCCIn = class( TMod)
  strict private
  const
    i_ch    = 0;
    i_cc    = 1;
  const
    o_gate  = 0;
    o_value = 1;
  private
    FChannel     : Byte;
    FCC          : Byte;
    FGate        : Boolean;
    FValue       : TSignal;
    FCounter     : Integer;
    FLedCounter  : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   AcceptMidi( const aMsg: TMidiMessage);                                                         override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiNoteOut = class( TMod)
  strict private
  const
    i_ch     = 0;
    i_gate   = 1;
    i_note   = 2;
    i_velon  = 3;
    i_veloff = 4;
  private
    FPrevNote    : Byte;
    FPrevChannel : Byte;
    FGate        : Boolean;
    FDidSend     : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiCCOut = class( TMod)
  strict private
  const
    i_ch    = 0;
    i_cc    = 1;
    i_gate  = 2;
    i_value = 3;
  const
    o_sent  = 0;
  private
    FGate  : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiPNOut = class( TMod)
  strict private
  const
    i_ch    = 0;
    i_pl    = 1;
    i_ph    = 2;
    i_gate  = 3;
    i_value = 4;
    i_usel  = 5;
    i_reg   = 6;
  const
    o_sent  = 0;
  private
    FGate  : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiSysexOut = class( TMod)
  strict private
  const
    i_gate = 0;
    i_1    = 1;
    i_2    = 2;
    i_3    = 3;
    i_4    = 4;
    i_5    = 5;
    i_6    = 6;
    i_7    = 7;
    i_8    = 8;
  private
    FGate       : Boolean;
    FMessage    : string;
    FNewMessage : string;
    FValues     : TBytes;
    F1          : Byte;
    F2          : Byte;
    F3          : Byte;
    F4          : Byte;
    F5          : Byte;
    F6          : Byte;
    F7          : Byte;
    F8          : Byte;
  private
    procedure   ParseMessage;
    procedure   FixValues;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiPCOut = class( TMod)
  strict private
  const
    i_ch      = 0;
    i_gate    = 1;
    i_program = 2;
    i_prg     = 3;
  const
    o_sent    = 0;
  private
    FGate    : Boolean;
    FPrevPrg : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModMidiPlayer = class( TMod)
  strict private
  const
    i_trig      = 0;
    i_ch        = 1;
    i_reset     = 2;
    i_mode      = 3;
  const
    o_note      = 0;
    o_velon     = 1;
    o_veloff    = 2;
    o_gate      = 3;
    o_done      = 4;
  const
    m_once      = 0;
    m_loop      = 1;
  private
    FState           : TMidiPlayerState;
    FReader          : TWrenMidiReader;
    FEvent           : TWrenMidiTimedEvent;
    FFileName        : string;
    FReading         : Boolean;
    FCounter         : Integer;
    FChannel         : Byte;
    FOldTrig         : Boolean;
    FOldReset        : Boolean;
    FRestart         : Boolean;
    FGateCount       : Integer;
    FNote            : TSignal;
    FVelocityOn      : TSignal;
    FVelocityOff     : TSignal;
    FDoneCounter     : Integer;
    FDoneLedCounter  : Integer;
    FResetLedCounter : Integer;
    FChOffset        : Integer;
  private
    procedure   SetFileName( const aValue: string);
    function    GetNextEvent: TWrenMidiTimedEvent;
  public
    procedure   NotesOff;
    procedure   Panic;                                                                                         override;
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   DoTick;                                                                                        override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  public
    property    FileName: string read FFileName write SetFileName;
  end;


  TModReverb = class( TMod)
  // http://members.chello.nl/t.koning8/kolindex.htm
  strict private
  const
    i_inl      =  0;
    i_inr      =  1;
    i_roomsize =  2;
    i_damp     =  3;
    i_wet      =  4;
    i_dry      =  5;
    i_width    =  6;
    i_mode     =  7;
    i_mute     =  8;
    i_freezein =  9;
  const
    o_outl     =  0;
    o_outr     =  1;
  private
    FReverb : TReverb;
    FScale  : TSignal;
    FFrozen : Boolean;
  private
    procedure   Recreate;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModGVerb = class( TMod)
  // https://github.com/swh/lv2/tree/master/gverb
  strict private
  const
    i_inl             =  0;
    i_inr             =  1;
    i_roomsize        =  2;
    i_roomsizemod     =  3;
    i_roomsizemodlev  =  4;
    i_revtime         =  5;
    i_damping         =  6;
    i_inputbw         =  7;
    i_earlylevel      =  8;
    i_taillevel       =  9;
    i_drylevel        = 10;
    i_inputlevel      = 11;
    i_mute            = 12;
    i_revtimemod      = 13;
    i_dampingmod      = 14;
    i_inputbwmod      = 15;
    i_earlylevelmod   = 16;
    i_taillevelmod    = 17;
    i_drylevelmod     = 18;
  const
    o_outl            =  0;
    o_outr            =  1;
  const
    gverb_maxroomsize =  0;
    gverb_spread      =  1;
    gverb_mode        =  2;
  private
    FRecreateTimer : TTimer;
    FReverb        : TGVerb;
    FMaxRoomSize   : TSignal;
    FRoomSize      : TSignal;
    FRoomSizeMod   : TSignal;
    FRevTime       : TSignal;
    FDamping       : TSignal;
    FSpread        : TSignal;
    FInputBW       : TSignal;
    FEarlyLevel    : TSignal;
    FTailLevel     : TSignal;
    FDryLevel      : TSignal;
    FInputLevel    : TSignal;
    FMode          : Integer;
  private
    procedure   ReCreateReverb;
    procedure   StartTimer( ams: Cardinal);
    procedure   StopTimer;
    procedure   DoTimer( aSender: TObject);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModEnvelope = class( TMod)
  // http://www.musicdsp.org/showone.php?id=97
  // Modified to have an inverted output
  strict private
  const
    i_in          = 0;
    i_attack      = 1;
    i_release     = 2;
    i_peakdecay   = 3;
    i_peakdecayon = 4;
    i_mode        = 5;
  const
    o_out         = 0;
    o_invout      = 1;
  const
    m_peak        = 0; // peak mode
    m_rms         = 1; // RMS mode
  private
    FRMSCalculator : TRMSCalculator;
    FValue         : TSignal;
    FAttack        : TSignal;
    FRelease       : TSignal;
    FPeakValue     : TSignal;
    FPeakDecay     : TSignal;
    FOldAttack     : TSignal;
    FOldRelease    : TSignal;
    FOldPeakDecay  : TSignal;
    FMode          : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModMultiEnvelope = class( TMod)
  strict private
  const
    i_in1         =  0;
    i_in2         =  1;
    i_in3         =  2;
    i_in4         =  3;
    i_in5         =  4;
    i_in6         =  5;
    i_in7         =  6;
    i_in8         =  7;
    i_in9         =  8;
    i_in10        =  9;
    i_in11        = 10;
    i_in12        = 11;
    i_in13        = 12;
    i_in14        = 13;
    i_in15        = 14;
    i_in16        = 15;
    i_in17        = 16;
    i_mult1       = 17;
    i_mult2       = 18;
    i_mult3       = 19;
    i_mult4       = 20;
    i_mult5       = 21;
    i_mult6       = 22;
    i_mult7       = 23;
    i_mult8       = 24;
    i_mult9       = 25;
    i_mult10      = 26;
    i_mult11      = 27;
    i_mult12      = 28;
    i_mult13      = 29;
    i_mult14      = 30;
    i_mult15      = 31;
    i_mult16      = 32;
    i_mult17      = 33;
    i_attack      = 34;
    i_release     = 35;
  const
    o_out1        =  0;
    o_out2        =  1;
    o_out3        =  2;
    o_out4        =  3;
    o_out5        =  4;
    o_out6        =  5;
    o_out7        =  6;
    o_out8        =  7;
    o_out9        =  8;
    o_out10       =  9;
    o_out11       = 10;
    o_out12       = 11;
    o_out13       = 12;
    o_out14       = 13;
    o_out15       = 14;
    o_out16       = 15;
    o_out17       = 16;
  private
    FValues        : array[ 0 .. 16] of TSignal;
    FAttack        : TSignal;
    FRelease       : TSignal;
    FOldAttack     : TSignal;
    FOldRelease    : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModCompressor = class( TMod)
  strict private
  const
    i_inl        =  0;
    i_inr        =  1;
    i_reflevel   =  2;
    i_gain       =  3;
    i_attack     =  4;
    i_release    =  5;
    i_bypass     =  6;
    i_preamp     =  7;
    i_mode       =  8;
    i_side       =  9;
    i_sidesel    = 10;
    i_acdc       = 11;
  const
    o_outl       =  0;
    o_outr       =  1;
    o_env        =  2;
    o_outs       =  3;
  const
    m_bypass_off =  0; // normal working mode
    m_bypass_byp =  1; // bypass, outputs are same as inputs
  const
    m_peak       =  0; // peak mode
    m_rms        =  1; // RMS mode
  const
    COMP_MAX_GAIN =  64.0;
    COMP_MIN_GAIN = -32.0;
    ALPHA         = 0.999;     // -3dB point at approximately 20 ~ 40 Hz, depending on sample rate
  private
    FRMSCalculator : TRMSCalculator;
    FEnvelope      : TSignal;
    FAttack        : TSignal;
    FRelease       : TSignal;
    FRefLevel      : TSignal;
    FPreGain       : TSignal;
    FGain          : TSignal;
    FBypass        : Integer;
    FOutL          : TSignal;
    FOutR          : TSignal;
    FOutS          : TSignal;
    FanL           : TSignal;
    FanR           : TSignal;
    FanS           : TSignal;
    FPrevL         : TSignal;
    FPrevR         : TSignal;
    FPrevS         : Tsignal;
    FLoopAmp       : TSignal;
    FMaxLoopAmp    : TSignal;
    FOldAttack     : TSignal;
    FOldRelease    : TSignal;
    FMinDbVal      : TSignal;
    FMaxDbVal      : TSignal;
    FMode          : Integer;
    FSideSel       : Boolean;
    FACMode        : Boolean;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModFormant = class( TMod)
  // http://www.musicdsp.org/showone.php?id=110
  strict private
  const
    i_in         = 0;
    i_vowel      = 1;
    i_vowelin    = 2;
    i_controlmod = 3;
    i_level      = 4;
    i_levelmod   = 5;
  const
    o_out        = 0;
  private
    FCoeffStore    : array[ 0 .. 10] of TSignalArrayWrapper;
    FCoeffiecients : array[ 0 .. 10] of TSignal;
    FMemory        : TSignalArray;
    FVowel         : TSignal;
    FUpdateCounter : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModFormant2 = class( TMod)
  strict private
  const
    i_in         = 0;
    i_vowel      = 1;
    i_vowelin    = 2;
    i_controlmod = 3;
    i_level      = 4;
    i_levelmod   = 5;
  const
    o_out        = 0;
  private
    FFilter   : TSvfFormantFilter;
    FVowel    : TSignal;
    FVowelSet : Integer;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModFlangeChorus = class( TMod)
  // This is a copy from some archaic older FX code I had (originally Turbo Pascal era)
  strict private
  const
    i_in       = 0;
    i_mute     = 1;
    i_freeze   = 2;
    i_dry      = 3;
    i_wet      = 4;
    i_feedback = 5;
    i_rate     = 6;
    i_sweep    = 7;
    i_delay    = 8;
    i_freezein = 9;
  const
    o_out      = 0;
  private
    FDryMix   : TSignal;
    FWetMix   : TSignal;
    FFeedBack : TSignal;
    FRate     : TSignal;
    FSweep    : TSignal;
    FDelay    : TSignal;
    FFrozen   : Boolean;
  private
    fp        : Integer;
    ep1       : Integer;
    ep2       : Integer;
    step      : Integer;
    depth     : Integer;
    idelay    : Integer;
    min_sweep : Integer;
    max_sweep : Integer;
    wlSweep   : TLongWord;
    Buf       : array[ 0 .. DELAY_BUF_SIZE] of TSignal;
  private
    procedure   Setup;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TFadeTable = array[ 0 .. MAX_XTAB - 1] of TSignal;
  PFadeTable = ^TFadeTable;


  TModPitchChange = class( TMod)
  // This is a copy from some archaic older FX code I had (originally Turbo Pascal era)
  strict private
  const
    i_in       = 0;
    i_mute     = 1;
    i_freeze   = 2;
    i_dry      = 3;
    i_wet      = 4;
    i_feedback = 5;
    i_rate     = 6;
    i_depth    = 7;
    i_delay    = 8;
    i_freezein = 9;
  const
    o_out      = 0;
  private
    FDryMix   : TSignal;
    FWetMix   : TSignal;
    FFeedback : TSignal;
    FRate     : TSignal;
    FDepth    : TSignal;
    FDelay    : TSignal;
    FFrozen   : Boolean;
  private
    fp         : Integer;
    ep1        : Integer;
    ep2        : Integer;
    ep3        : Integer;
    ep4        : Integer;
    idepth     : Integer;
    idelay     : Integer;
    min_sweep  : Integer;
    max_sweep  : Integer;
    step       : Word;
    xfade      : Integer;
    xfade_cnt  : Integer;
    active     : Integer;
    active_cnt : Integer;
    sweep_up   : Boolean;
    chanA      : Boolean;
    blendA     : TSignal;
    blendB     : TSignal;
    fadeA      : PFadeTable;
    fadeB      : PFadeTable;
    wlSweep    : TLongWord;
    Buf        : array[ 0 .. DELAY_BUF_SIZE] of TSignal;
    Fade_Out   : TFadeTable;
    Fade_In    : TFadeTable;
  private
    procedure   Setup;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModEuclideanRhythm = class( TMod)
  strict private
  const
    i_trig       = 0;
    i_pattern    = 1;
    i_patternmod = 2;
    i_pulses     = 3;
    i_steps      = 4;
    i_active     = 5;
  const
    o_out        = 0;
    o_outoff     = 1;
  private
    FBits        : TScaleNoteSet;
    FSteps       : Integer;
    FPattern     : Integer;
    FOldTrig     : Boolean;
    FPosition    : Integer;
    FLedTrgCount : Integer;
    FLedOutCount : Integer;
    FLedOffCount : Integer;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModEuclideanScale = class( TMod)
  strict private
  const
    i_in            = 0;
    i_pattern       = 1;
    i_patternmod    = 2;
    i_patternmodamt = 3;
    i_steps         = 4;
  const
    o_out           = 0;
  private
    FBits       : TScaleNoteSet;
    FCount      : Integer;
    FSteps      : Integer;
    FPattern    : Integer;
    FScaleNotes : array[ Byte] of Byte;
    FNoteCount  : Integer;
  private
    procedure   MakeScale;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
  end;


  TModRandomScale = class( TMod)
  strict private
  const
    i_in            =  0;
    i_c             =  1;   //   \
    i_cis           =  2;   //    |
    i_d             =  3;   //    |
    i_dis           =  4;   //    |
    i_e             =  5;   //    |
    i_f             =  6;   //    \   permutation state,
    i_fis           =  7;   //    /   as hidden inputs.
    i_g             =  8;   //    |
    i_gis           =  9;   //    |
    i_a             = 10;   //    |
    i_ais           = 11;   //    |
    i_b             = 12;   //   /
    i_permute       = 13;   //        permutation input
    i_perm          = 14;   //        ppermutation button
  const
    o_out           =  0;
  private
    FOldPerm    : Boolean;
    FScaleNotes : array[ Byte] of Integer;
    FLedCount   : Integer;
  private
    procedure   FillNotesFor( anIndex: Integer);
    procedure   Permute;
  public
    procedure   SetDefaults;                                                                                   override;
    procedure   CreateIO;                                                                                      override;
    procedure   Pulse( anInput: Integer);                                                                      override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModXYScope = class( TMod)
  strict private
  const
    i_inx     = 0;
    i_iny     = 1;
    i_levx    = 2;
    i_levy    = 3;
    i_lockxy  = 5;
  const
    s_history = 0;
  private
    FHistory   : Integer;
    FDataQueue : TSignalPairFifo;
  private
    procedure   SetHistory( aValue: Integer);
    procedure   AddPoint( anX, anY: TSignal);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherXYData( const aPrefix: string; const aCallback: TXYDataHandler);                         override;
  public
    property    History: Integer read FHistory write SetHistory;
  end;


  TModDataGraph = class( TMod)
  strict private
  const
    i_in      = 0;
    i_inmode  = 1;
    i_outmode = 2;
    i_splines = 3;
    i_mode    = 4;
  const
    o_out     = 0;
  private
    FData     : TSignalPairArray;
    FSpline   : TBSpline;
    FPosition : TSignal;
  private
    function    GetData: string;
    procedure   SetData( const aValue: string);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
    procedure   GatherSignals   ( const aPrefix: string; const aCallback: TSignalHandler);                     override;
  public
    property    Data: string read GetData write SetData;
  end;


  TModAudioGraph = class( TMod)
  strict private
  const
    TAB_LEN = 2048;
  const
    i_in        = 0;
    i_rnd       = 1;
    i_autoscale = 2;
  const
    o_out       = 0;
  private
    FData       : TSignalPairArray;
    FIntegrated : TSignalArray;
    FPrevIn     : TSignal;
    FPrevOut    : TSignal;
    FInvTabLen  : TSignal;
    FPrevRnd    : Boolean;
    FRandomFlag : Boolean;
    FRandVal    : TSignal;
    FAutoScale  : Boolean;
    FScaleFlag  : Boolean;
  private
    function    GetData: string;
    procedure   SetData( const aValue: string);
    function    Interpolate( const anIndex: TSignal): TSignal;
    procedure   SetAutoScale( aValue: Boolean);
    procedure   Randomize;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);                        override;
  public
    property    Data      : string  read GetData    write SetData;
    property    AutoScale : Boolean read FAutoScale write SetAutoScale;
  end;


  TFFTData = record
  private
    FWindow       : TFFTWindow;
    FBinCount     : Integer;
    FIndex        : Integer;
    FFTWInBuffer  : Pfftw_real;
    FFTWOutBuffer : Pfftw_complex;
    FFTWPlan      : fftw_plan;
  public
    procedure   Initialize( aBinCount, aStartIndex: Integer; aWindow: TFFTWindow);
    procedure   Finalize;
    procedure   Tick( anInValue: TSignal; var anOutValue: TComplex);
  end;
  TFFTDataArray = array of TFFTData;


  TIFFTData = record
  private
    FWindow       : TFFTWindow;
    FBinCount     : Integer;
    FIndex        : Integer;
    FFTWInBuffer  : Pfftw_complex;
    FFTWOutBuffer : Pfftw_real;
    FFTWPlan      : fftw_plan;
  public
    procedure   Initialize( aBinCount, aStartIndex: Integer; aWindow: TFFTWindow);
    procedure   Finalize;
    procedure   Tick( anInValue: TComplex; var anOutValue: TSignal);
  end;
  TIFFTDataArray = array of TIFFTData;


  TFFTHandler = class
  private
    FDestroying : Boolean;
    FBinCount   : Integer;
    FWindow     : TFFTWindow;
    FData       : TFFTDataArray;
  private
    function    GetWindowCount: Integer;
  public
    constructor Create( aWindowCount, aBinCount: Integer; aWindowType: TFFTWindowClass);
    destructor  Destroy;                                                                                       override;
    procedure   Tick( anInValue: TSignal; var anOutValue: TComplex);
  public
    property    WindowCount: Integer read GetWindowCount;
  end;


  TIFFTHandler = class
  private
    FDestroying : Boolean;
    FBinCount   : Integer;
    FWindow     : TFFTWindow;
    FData       : TIFFTDataArray;
  private
    function    GetWindowCount: Integer;
  public
    constructor Create( aWindowCount, aBinCount: Integer; aWindowType: TFFTWindowClass);
    destructor  Destroy;                                                                                       override;
    procedure   Tick( anInValue: TComplex; var anOutValue: TSignal);
  public
    property    WindowCount: Integer read GetWindowCount;
  end;


  TModVocoder = class( TMod)
  strict private
  const
    i_modulator = 0;
    i_carrier   = 1;
    i_mute      = 2;
  const
    o_out       = 0;
  private
    FBinCount         : Integer;
    FWindowCount      : Integer;
    FModulatorHandler : TFFTHandler;
    FCarrierHandler   : TFFTHandler;
    FOutHandler       : TIFFTHandler;
  private
    procedure   SetNewHandlers( aWindowCount, aBinCount: Integer; aWindowType: TFFTWindowClass);
    procedure   SetBinCount( aValue: Integer);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  public
    property    BinCount: Integer read FBinCount write SetBinCount;
  end;


  TModConvoder = class( TMod)
  strict private
  const
    i_modulator    = 0;
    i_carrier      = 1;
    i_mute         = 2;
    i_modulatorlev = 3;
    i_carrierlev   = 4;
  const
    o_out          = 0;
  const
    s_size         = 0;
  const
    c_sizes : array[ 0 .. 11] of Integer = (
      64    ,
      128   ,
      256   ,
      512   ,
      1024  ,
      2048  ,
      4096  ,
      8192  ,
      16384 ,
      32768 ,
      65536 ,
      131072
    );
  private
    FSize             : Integer;
    FBinCount         : Integer;
    FWindowCount      : Integer;
    FAttenuation      : TSignal;
    FModulatorHandler : TFFTHandler;
    FCarrierHandler   : TFFTHandler;
    FOutHandler       : TIFFTHandler;
  private
    procedure   SetNewHandlers( aWindowCount, aBinCount: Integer; aWindowType: TFFTWindowClass);
    procedure   SetBinCount( aValue: Integer);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoTick;                                                                                        override;
  public
    property    BinCount: Integer read FBinCount write SetBinCount;
  end;


  TModFilterSet = class( TMod)
  strict protected
  const
    FILTER_COUNT = 17;
  private
    FSteepness    : Integer;
    FModeLow      : Integer;
    FModeHigh     : Integer;
    FFilter       : TSvfChamberlinBank;
    FRefFrequency : TSignal;
  private
    procedure   Recreate;                                                                             virtual; abstract;
    procedure   SetSteepness   ( aValue: Integer);
    procedure   SetModeLow     ( aValue: Integer);
    procedure   SetModeHigh    ( aValue: Integer);
    procedure   SetRefFrequency( const aValue: TSignal);                                              virtual; abstract;
    function    ScaledQ( const aValue: TSignal): TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    destructor  Destroy;                                                                                       override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
  public
    property    Steepness    : Integer read FSteepness    write SetSteepness;
    property    ModeLow      : Integer read FModeLow      write SetModeLow;
    property    ModeHigh     : Integer read FModeHigh     write SetModeHigh;
    property    RefFrequency : TSignal read FRefFrequency write SetRefFrequency;
  end;


  TModFilterBankProto = class( TModFilterSet)
  strict private
  const
    i_in      =  0;
    i_fader1  =  1;
    i_fader2  =  2;
    i_fader3  =  3;
    i_fader4  =  4;
    i_fader5  =  5;
    i_fader6  =  6;
    i_fader7  =  7;
    i_fader8  =  8;
    i_fader9  =  9;
    i_fader10 = 10;
    i_fader11 = 11;
    i_fader12 = 12;
    i_fader13 = 13;
    i_fader14 = 14;
    i_fader15 = 15;
    i_fader16 = 16;
    i_fader17 = 17;
    i_level   = 18;
    i_q       = 19;
    i_fdmod1  = 20;
    i_fdmod2  = 21;
    i_fdmod3  = 22;
    i_fdmod4  = 23;
    i_fdmod5  = 24;
    i_fdmod6  = 25;
    i_fdmod7  = 26;
    i_fdmod8  = 27;
    i_fdmod9  = 28;
    i_fdmod10 = 29;
    i_fdmod11 = 30;
    i_fdmod12 = 31;
    i_fdmod13 = 32;
    i_fdmod14 = 33;
    i_fdmod15 = 34;
    i_fdmod16 = 35;
    i_fdmod17 = 36;
    i_fm      = 37;
    i_fmamt   = 38;
    i_fi     =  39;
  const
    o_out    =  0;
  private
    procedure   Recreate;                                                                                      override;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
  end;


  TModSplitterBankProto = class( TModFilterSet)
  strict private
  const
    i_in    =  0;
    i_level =  1;
    i_q     =  2;
    i_fm    =  3;
    i_fmamt =  4;
    i_fi    =  5;
  const
    o_out1  =  0;
    o_out2  =  1;
    o_out3  =  2;
    o_out4  =  3;
    o_out5  =  4;
    o_out6  =  5;
    o_out7  =  6;
    o_out8  =  7;
    o_out9  =  8;
    o_out10 =  9;
    o_out11 = 10;
    o_out12 = 11;
    o_out13 = 12;
    o_out14 = 13;
    o_out15 = 14;
    o_out16 = 15;
    o_out17 = 16;
  private
    procedure   Recreate;                                                                                      override;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModCombinerBankProto = class( TModFilterSet)
  strict private
  const
    i_in    =  0;
    i_level =  1;
    i_q     =  2;
    i_fm    =  3;
    i_fmamt =  4;
    i_fi    =  5;
    i_in1   =  6;
    i_in2   =  7;
    i_in3   =  8;
    i_in4   =  9;
    i_in5   = 10;
    i_in6   = 11;
    i_in7   = 12;
    i_in8   = 13;
    i_in9   = 14;
    i_in10  = 15;
    i_in11  = 16;
    i_in12  = 17;
    i_in13  = 18;
    i_in14  = 19;
    i_in15  = 20;
    i_in16  = 21;
    i_in17  = 22;
  const
    o_out   =  0;
  private
    procedure   Recreate;                                                                                      override;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModTritoneBank = class( TModFilterBankProto)
  private
    procedure   SetRefFrequency( const aValue: TSignal);                                                       override;
  end;


  TModThirdBank = class( TModFilterBankProto)
  private
    procedure   SetRefFrequency( const aValue: TSignal);                                                       override;
  end;


  TModTritoneSplitter = class( TModSplitterBankProto)
  private
    procedure   SetRefFrequency( const aValue: TSignal);                                                       override;
  end;


  TModThirdSplitter = class( TModSplitterBankProto)
  private
    procedure   SetRefFrequency( const aValue: TSignal);                                                       override;
  end;


  TModTritoneCombiner = class( TModCombinerBankProto)
  private
    procedure   SetRefFrequency( const aValue: TSignal);                                                       override;
  end;


  TModThirdCombiner = class( TModCombinerBankProto)
  private
    procedure   SetRefFrequency( const aValue: TSignal);                                                       override;
  end;


  TModModal = class( TMod)
  strict private
  const
    FILTER_COUNT = 11;
  const
    i_in      =  0;
    i_level   =  1;
    i_fm      =  2;
    i_fmamt   =  3;
    i_freq1   =  4;
    i_freq2   =  5;
    i_freq3   =  6;
    i_freq4   =  7;
    i_freq5   =  8;
    i_freq6   =  9;
    i_freq7   = 10;
    i_freq8   = 11;
    i_freq9   = 12;
    i_freq10  = 13;
    i_freq11  = 14;
    i_q1      = 15;
    i_q2      = 16;
    i_q3      = 17;
    i_q4      = 18;
    i_q5      = 19;
    i_q6      = 20;
    i_q7      = 21;
    i_q8      = 22;
    i_q9      = 23;
    i_q10     = 24;
    i_q11     = 25;
    i_level1  = 26;
    i_level2  = 27;
    i_level3  = 28;
    i_level4  = 29;
    i_level5  = 30;
    i_level6  = 31;
    i_level7  = 32;
    i_level8  = 33;
    i_level9  = 34;
    i_level10 = 35;
    i_level11 = 36;
    i_freq    = 37;
    i_fi      = 38;
  const
    o_out     =  0;
  private
    FFilter : TSvfChamberlinBank;
  private
    procedure   Recreate;
    function    ScaledQ( const aValue: TSignal): TSignal;
  public
    procedure   SetDefaults;                                                                                   override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModModal2 = class( TMod)
  strict private
  const
    MAX_VOICES = 11;
  const
    i_in        =  0;
    i_level     =  1;
    i_fm        =  2;
    i_fmamt     =  3;
    i_q1        =  4;
    i_q2        =  5;
    i_q3        =  6;
    i_q4        =  7;
    i_q5        =  8;
    i_q6        =  9;
    i_q7        = 10;
    i_q8        = 11;
    i_q9        = 12;
    i_q10       = 13;
    i_q11       = 14;
    i_level1    = 15;
    i_level2    = 16;
    i_level3    = 17;
    i_level4    = 18;
    i_level5    = 19;
    i_level6    = 20;
    i_level7    = 21;
    i_level8    = 22;
    i_level9    = 23;
    i_level10   = 24;
    i_level11   = 25;
    i_freq      = 26;
    i_fi        = 27;
    i_q         = 28;
    i_qm        = 29;
    i_qmamt     = 30;
    i_presetmod = 31;
    i_lfm       = 32;
    i_lfmamt    = 33;
    i_am        = 34;
    i_frequency = 35;
  const
    o_out       =  0;
  private
    FFilter    : TSvfModalFilter;
    FPreset    : Integer;
    FPresetMod : Integer;
  private
    procedure   SetPreset   ( aValue: Integer);
    procedure   SetPresetMod( aValue: Integer);
    procedure   Recreate;
    function    ScaledQ( const aValue: TSignal): TSignal;
  public
    procedure   SetInternal( const aName: string; aValue: TSignal);                                            override;
    function    GetInternal( const aName: string): TSignal;                                                    override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    destructor  Destroy;                                                                                       override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    Preset    : Integer read FPreset write SetPreset;
    property    PresetMod : Integer read FPreset write SetPresetMod;
  end;


  TModChladni = class( TMod)
  strict private
  const
    NUM_VOICES = 11;
  const
    i_in        =  0;
    i_level     =  1;
    i_fm        =  2;
    i_fmamt     =  3;
    i_q1        =  4;
    i_q2        =  5;
    i_q3        =  6;
    i_q4        =  7;
    i_q5        =  8;
    i_q6        =  9;
    i_q7        = 10;
    i_q8        = 11;
    i_q9        = 12;
    i_q10       = 13;
    i_q11       = 14;
    i_level1    = 15;
    i_level2    = 16;
    i_level3    = 17;
    i_level4    = 18;
    i_level5    = 19;
    i_level6    = 20;
    i_level7    = 21;
    i_level8    = 22;
    i_level9    = 23;
    i_level10   = 24;
    i_level11   = 25;
    i_freq      = 26;
    i_fi        = 27;
    i_q         = 28;
    i_qm        = 29;
    i_qmamt     = 30;
    i_frequency = 31;
    i_lfm       = 32;
    i_lfmamt    = 33;
    i_am        = 34;
    i_p         = 35;
    i_pmod      = 36;
    i_pmodamt   = 37;
  const
    o_out       =  0;
  private
    FFilter  : TSvfChladniFilter;
    FOldP    : TSignal;
    FAddends : array[ 0 .. NUM_VOICES - 1] of TSignal;
  private
    procedure   Recreate;
    function    ScaledQ( const aValue: TSignal): TSignal;
    procedure   CalcAddends( aP : TSignal);
  public
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    destructor  Destroy;                                                                                       override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModTalkie = class( TMod)
  strict private
  const
    i_start    =  0;
    i_phrase   =  1;
    i_mute     =  2;
    i_fr       =  3;
    i_frmod    =  4;
    i_frmodamt =  5;
    i_sr       =  6;
    i_srmod    =  7;
    i_srmodamt =  8;
    i_bank     =  9;
    i_banksel  = 10;
    i_fm       = 11;
  const
    o_out      =  0;
    o_rdy      =  1;
  private
    FTalkie     : TTalkie;
    FDestroying : Boolean;
    FStart      : Boolean;
    FOldBankSel : TSignal;
    FBank       : TSignal;
    FPhrase     : TSignal;
    FSampleRate : TSignal;
    FFrameRate  : TSignal;
    FSpeaking   : Boolean;
    FDidRun     : Boolean;
  private
    procedure   TalkStarted( aSender: TObject);
    procedure   TalkDone   ( aSender: TObject);
    procedure   Say;
    procedure   NewFramRate  ( aValue: TSignal);
    procedure   NewSampleRate( aValue: TSignal);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModAttractor = class( TMod)
  strict private
  const
    i_frequency   = 0;
    i_fm          = 1;
    i_fmlevel     = 2;
    i_cents       = 3;
    i_freq        = 4;
    i_mute        = 5;
    i_mode        = 6;
    i_drive       = 7;
    i_driveamt    = 8;
  const
    o_outx        = 0;
    o_outy        = 1;
    o_outz        = 2;
  const
    m_lorenz      = 0;
    m_rossler     = 1;
  private
    FLorenz  : TLorenzOsc;
    FRossler : TRosslerOsc;
    FX       : TSignal;
    FY       : TSignal;
    FZ       : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModVanDerPol = class( TMod)
  strict private
  const
    i_frequency   =  0;
    i_fm          =  1;
    i_fmlevel     =  2;
    i_cents       =  3;
    i_freq        =  4;
    i_mute        =  5;
    i_mu          =  6;
    i_mumod       =  7;
    i_mumodamt    =  8;
    i_in          =  9;
    i_inamt       = 10;
  const
    o_outx        =  0;
    o_outy        =  1;
  private
    FVanDerPol : TVanDerPolOsc;
    FX         : TSignal;
    FY         : TSignal;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   SampleRateChanged;                                                                             override;
  end;


  TModTss = class( TMod)
  strict private
  const
    i_res   = 0;
    i_pos   = 1;
    i_speed = 2;
    i_fm    = 3;
    i_mute  = 4;
  const
    o_out   = 0;
  private
    FMemory       : TSignalArrayWrapper;
    FMessage      : string;
    FDuration     : string;
    FLength       : Integer;
    FPlayPointer  : TSignal;
    FResetCounter : Integer;
    FOldRes       : Boolean;
  private
    procedure   Recreate;
    procedure   SwapMemoryFor( const aMemory: TSignalArray);
    procedure   SetMessage( const aValue: string);
    procedure   SetDuration( aValue: TSignal);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData  ( const aPrefix: string; const aCallback: TStringDataHandler  );             override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    Message: string read FMessage write SetMessage;
  end;


  TModSapiVoice = class( TMod)
  strict private
  const
    i_res   = 0;
    i_pos   = 1;
    i_speed = 2;
    i_fm    = 3;
    i_mute  = 4;
    i_voice = 5;
    i_rate  = 6;
  const
    o_out   = 0;
  private
    FMemory        : TSignalArrayWrapper;
    FVoice         : TSpVoice;
    FStream        : TSpMemoryStream;
    FFormat        : TSpAudioFormat;
    FVoices        : TStringList;
    FSelectedVoice : Integer;
    FNewVoice      : Integer;
    FVoiceChanged  : Boolean;
    FSelectedRate  : Integer;
    FNewRate       : Integer;
    FRateChanged   : Boolean;
    FMessage       : string;
    FDuration      : string;
    FLength        : Integer;
    FPlayPointer   : TSignal;
    FResetCounter  : Integer;
    FOldRes        : Boolean;
    FLoading       : Boolean;
  private
    function    RateIndexToFormat( aRateIndex: Integer): Integer;
    function    FormattedAudioRate: Integer;
    function    SpeedScaling: TSignal;
    procedure   PopulateVoices;
    procedure   Recreate;
    procedure   DoStreamCompleted( aSender: TObject; aStreamNumber: Integer; aStreamPosition: OleVariant);
    procedure   SwapMemoryFor( const aMemory: TSignalArray);
    procedure   SetMessage( const aValue: string);
    procedure   FixDuration;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData  ( const aPrefix: string; const aCallback: TStringDataHandler  );             override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    Message : string read FMessage write SetMessage;
  end;


  TModESpeakVoice = class( TMod)
  strict private
  const
    i_res          =  0;
    i_pos          =  1;
    i_mute         =  2;
    i_voice        =  3;
    i_language     =  4;
    i_variation    =  5;
    i_rate         =  6;
    i_pitch        =  7;
    i_range        =  8;
    i_punctuation  =  9;
    i_wordgap      = 10;
    i_speed        = 11;
    i_speedmod     = 12;
    i_speedmodamt  = 13;
    i_freq         = 14;
    i_frequency    = 15;
    i_fmmod        = 16;
    i_fmmodamt     = 17;
    i_posmod       = 18;
    i_posmodamt    = 19;
    i_cents        = 20;
  const
    o_out          =  0;
    o_textstart    =  1;
    o_sentence     =  2;
    o_word         =  3;
    o_phoneme      =  4;
    o_silence      =  5;
    o_vowel        =  6;
    o_mark         =  7;
    o_usermark     =  8;
    o_textdone     =  9;
  private
    FSpeaker            : TESpeakSpeaker;
    FAudio              : TTaggedAudio;
    FNewSpeaker         : TESpeakSpeaker;
    FNewAudio           : TTaggedAudio;
    FSampleRate         : TSignal;
    FMessage            : string;
    FDuration           : string;
    FSelectedVoice      : Integer;
    FNewVoice           : Integer;
    FVoiceChanged       : Boolean;
    FSelectedLanguage   : Integer;
    FNewLanguage        : Integer;
    FLanguageChanged    : Boolean;
    FSelectedVariation  : Integer;
    FNewVariation       : Integer;
    FVariationChanged   : Boolean;
    FPunctuation        : Integer;
    FRate               : Integer;
    FPitch              : Integer;
    FRange              : Integer;
    FWordGap            : Integer;
    FNewPunctuation     : Integer;
    FNewRate            : Integer;
    FNewPitch           : Integer;
    FNewRange           : Integer;
    FNewWordGap         : Integer;
    FPunctuationChanged : Boolean;
    FRateChanged        : Boolean;
    FPitchChanged       : Boolean;
    FRangeChanged       : Boolean;
    FWordGapChanged     : Boolean;
    FResetCounter       : Integer;
    FOldRes             : Boolean;
    FLoading            : Boolean;
    FPlayPointer        : TSignal;
    FTextStarted        : Integer;
    FTextStartedLed     : Integer;
    FSentenceSeen       : Integer;
    FWordSeen           : Integer;
    FPhonemeSeen        : Integer;
    FVowelSeen          : Integer;
    FSilenceSeen        : Integer;
    FMarkSeen           : Integer;
    FUSerMarkSeen       : TSignal;
    FTextDone           : Integer;
    FSentenceLed        : Integer;
    FWordLed            : Integer;
    FPhonemeLed         : Integer;
    FSilenceLed         : Integer;
    FVowelLed           : Integer;
    FMarkLed            : Integer;
    FUSerMarkLed        : Integer;
    FTextDoneLed        : Integer;
  private
    procedure   HandleTextStart    ( const aSender: TObject                                );
    procedure   HandleSentence     ( const aSender: TObject; anId: Integer                 );
    procedure   HandleWord         ( const aSender: TObject; anId: Integer                 );
    procedure   HandlePhoneme      ( const aSender: TObject; const aPhoneme: string        );
    procedure   HandleSilence      ( const aSender: TObject                                );
    procedure   HandleMark         ( const aSender: TObject; const aMark   : string        );
    procedure   HandleUserMark     ( const aSender: TObject; anId: Integer; aValue: TSignal);
    procedure   HandleTextDone     ( const aSender: TObject                                );
    procedure   HandleRenderingDone( const aSender: TObject);
  private
    function    SpeedScaling: TSignal;
    procedure   SwapInNewVoice;
    procedure   Recreate;
    procedure   SetMessage( const aValue: string);
    procedure   FixDuration;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    Message : string read FMessage write SetMessage;
  end;


  TModSong = class( TMod)
  strict private
  const
    i_res            =  0;
    i_loop           =  1;
    i_freq           =  2;
    i_fmmod          =  3;
    i_fmmodamt       =  4;
    i_speed          =  5;
    i_speedmod       =  6;
    i_speedmodamt    =  7;
    i_whisper        =  8;
    i_whispermod     =  9;
    i_whispermodamt  = 10;
    i_mute           = 11;
    i_filtinv        = 12;
    i_shape          = 13;
    i_pm             = 14;
    i_pmamt          = 15;
    i_frequency      = 16;
  const
    o_phstart  = 0;
    o_txtstart = 1;
    o_txtend   = 2;
    o_silence  = 3;
    o_vowel    = 4;
    o_out      = 5;
  private
    FMessage          : string;
    FSeries           : TPhonemeSeries;
    FOldRes           : Boolean;
    FResetCounter     : Integer;
    FTextStartCounter : Integer;
    FTextEndCounter   : Integer;
    FPhonemeCounter   : Integer;
    FSilenceCounter   : Integer;
    FVowelCounter     : Integer;
  private
    procedure   Recreate;
    procedure   SetMessage( const aValue: string);
    procedure   DoTextStarted   ( const aSender: TObject);
    procedure   DoTextEnded     ( const aSender: TObject);
    procedure   DoPhonemeStarted( const aSender: TObject; const aPhoneme: char);
  public
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   SampleRateChanged;                                                                             override;
  public
    property    Message : string  read FMessage write SetMessage;
  end;


  TModTextWriter = class( TMod)
  strict private
  const
    i_trig   = 0;
    i_reset  = 1;
    i_select = 2;
  private
    FTextFileName : string;
    FTextMessage  : string;
    FSelection    : TSignal;
    FText         : TStringList;
    FCurrentLine  : Integer;
    FOldTrig      : Boolean;
    FOldReset     : Boolean;
    FTrigCounter  : Integer;
    FResetCounter : Integer;
    FTextChanged  : Boolean;
  private
    procedure   Recreate;
    procedure   SetTextMessage( const aValue: string);
    procedure   SetTextFileName( const aValue: string);
    procedure   NextLine;
    procedure   SelectLine( aValue: TSignal);
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
    procedure   GatherStringData  ( const aPrefix: string; const aCallback: TStringDataHandler);               override;
  public
    property    TextFileName : string read FTextFileName write SetTextFileName;
    property    TextMessage  : string read FTextMessage  write SetTextMessage;
  end;


  function  SignalToEnvRange  ( aValue: TSignal): TEnvRange;
  function  SignalToEnvShape  ( aValue: TSignal): TEnvShape;
  function  SignalToLfoRange  ( aValue: TSignal): TLfoRange;
  function  SignalToDelayRange( aValue: TSignal): TDelayRange;



implementation


var

  IssuedFftwWarning   : Boolean = False;
  IssuedSapiWarning   : Boolean = False;
  IssuedESpeakWarning : Boolean = False;

  procedure FftwWarning( const aModuleType: string);
  begin
    if not IssuedFftwWarning
    then begin
      IssuedFftwWarning := True;
      MessageDlg(
        Format(
          'Could not initialize the %s module, the required FFTW DLL may not be present or can not be found.'^M^M +
          'This is a one time warning for all FFT based module types, they will all not work.'^M^M +
          'Please check the debug log (F5) for FFTW issues after closing this window',
          [ aModuleType],
          AppLocale
        ),
        mtWarning,
        [ mbOK],
        0
      );
    end;
  end;


  procedure SapiWarning( const aModuleType: string);
  begin
    if not IssuedSapiWarning
    then begin
      IssuedSapiWarning := True;
      MessageDlg(
        Format(
          'Could not initialize the %s module, the required Soft Speech Platform may not be present or can not be found.'^M^M +
          'This is a one time warning for all SAPI based module types, they will all not work.'^M^M +
          'Please check the debug log (F5) for SAPI issues after closing this window'^M^M +
          'You may try to install the speech platform (https://www.microsoft.com/en-us/download/details.aspx?id=24974)',
          [ aModuleType],
          AppLocale
        ),
        mtWarning,
        [ mbOK],
        0
      );
    end;
  end;


  procedure ESpaekWarning( const aModuleType: string);
  begin
    if not IssuedFftwWarning
    then begin
      IssuedFftwWarning := True;
      MessageDlg(
        Format(
          'Could not initialize the %s module, the required aSpeak DLL may not be present or can not be found.'^M^M +
          'This is a one time warning for all eSpeak based module types, they will all not work.'^M^M +
          'Please check the debug log (F5) for eSpeak issues after closing this window',
          [ aModuleType],
          AppLocale
        ),
        mtWarning,
        [ mbOK],
        0
      );
    end;
  end;


  function SignalToEnvRange( aValue: TSignal): TEnvRange;
  begin
    if AlmostEqual( aValue, 0, 0.45)
    then Result := erFast
    else if AlmostEqual( aValue, 1, 0.45)
    then Result := erMedium
    else Result := erSlow;
  end;


  function SignalToEnvShape( aValue: TSignal): TEnvShape;
  begin
    if AlmostEqual( aValue, 0, 0.45)
    then Result := esLog
    else if AlmostEqual( aValue, 1, 0.45)
    then Result := esLin
    else if AlmostEqual( aValue, 2, 0.45)
    then Result := esExp
    else Result := esS;
  end;


  function SignalToLfoRange( aValue: TSignal): TLfoRange;
  begin
    if      AlmostEqual( aValue, 0, 0.45)
    then Result := lsFast
    else if AlmostEqual( aValue, 1, 0.45)
    then Result := lsMedium
    else if AlmostEqual( aValue, 2, 0.45)
    then Result := lsSlow
    else if AlmostEqual( aValue, 3, 0.45)
    then Result := lsBPM
    else if AlmostEqual( aValue, 4, 0.45)
    then Result := lsLin
    else if AlmostEqual( aValue, 5, 0.45)
    then Result := lsFast2
    else if AlmostEqual( aValue, 6, 0.45)
    then Result := lsMedium2
    else Result := lsSlow2
  end;


  function SignalToDelayRange( aValue: TSignal): TDelayRange;
  begin
    if AlmostEqual( aValue, 0, 0.45)
    then Result := dsShort
    else if AlmostEqual( aValue, 1, 0.45)
    then Result := dsMedium
    else Result := dsLong;
  end;


  function EuclideanPattern( aPattern, aSteps: Integer; out aCount: Integer): TScaleNoteSet;
  //
  // Idea from the wiki Bresenham and Eclidean Rhythm pages,
  // Changed to let the pattern start with a one always,
  // the original is:
  //
  // function EuclideanRhythm( aPulses, aSteps: Integer): string;
  // var
  //   Error : Single;
  //   Slope : Single;
  //   i     : Integer;
  // begin
  //   Result := '';
  //   if aSteps > 0
  //   then begin
  //     Error := 0;
  //     Slope := aPulses / aSteps;
  //     for i := 0 to aSteps - 1
  //     do begin
  //       Error := Error + Slope;
  //       if Error >= 0.5
  //       then begin
  //         Result := Result + '1';
  //         Error  := Error  -  1;
  //       end
  //       else Result := Result + '0';
  //     end;
  //   end;
  // end;
  //
  // Then the integer version became:
  //
  // function EuclideanRhythm2( aPulses, aSteps: Integer): string;
  // var
  //   Error : Integer;
  //   i     : Integer;
  // begin
  //   Result := '';
  //   if aSteps > 0
  //   then begin
  //     Error := 0;
  //     for i := 0 to aSteps - 1
  //     do begin
  //       Error := Error + aPulses;
  //       if Error >= aSteps div 2
  //       then begin
  //         Result := Result + '1';
  //         Error  := Error - aSteps;
  //       end
  //       else Result := Result + '0';
  //     end;
  //   end;
  // end;
  //
  // Then finally the modification to make the pattern start with a one
  // always (and uses a result 'set of Byte' instead of a 'string' to avoid memory allocation issues):
   //
  // tested a more compact one from : http://electro-music.com/forum/viewtopic.php?p=406480#406480
  // also see : http://www.pdpatchrepo.info/hurleur/euclid.pd (do not test for now)
  //
  /// * generates euclideian patterns
  //    c: current step number
  //    k: hits per bar
  //    n: bar length
  //    r: rotation
  //    returns 1 or 0 according to the euclidean pattern*/
  //
  // fun int euclide( int c, int k, int n, int r ) {
  //   return (((c + r) * k) % n) < k;
  //}
  //
  // 0 => int i;
  // while(i < 12) {
  //   euclide(i, 7, 12, 4) => int e;
  //   //declaring b just to print e
  //   <<<e>>> => int b;
  //   1 +=> i;
  //   100::ms => now;
  //}
  var
    Error : Integer;
    i     : Integer;
  begin
    Result := [];
    aCount := 0;

    if aSteps > 0
    then begin
      Error  := 0;

      for i := 0 to aSteps - 1
      do begin
        Error := Error + aPattern;

        if Error > 0
        then begin
          Result := Result + [ i];
          Error  := Error - aSteps;
          Inc( aCount);
        end;
      end;
    end;
  end;


{ ========
  TModMConst = class( TMod)
  public
}

    procedure   TModMConst.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_value, 'value');
      AddOutput( o_out  , 'out'  );
    end;


    procedure   TModMConst.DoSlowTick; // override;
    var
      FOutVal : TSignal;
    begin
      case Round( FInputs[ i_value]) of
        c_1             : FOutVal := 1.0;
        c_2             : FOutVal := 2.0;
        c_e             : FOutVal := EULER;
        c_pi            : FOutVal := Pi;
        c_sqrt2         : FOutVal := SQRT2;
        c_sqrt3         : FOutVal := SQRT3;
        c_sqrt5         : FOutVal := SQRT5;
        c_12r2          : FOutVal := TWO_PTWELVE;
        c_semi          : FOutVal := NOTE_SCALING_REC;
        c_r2            : FOutVal := 0.5;
        c_re            : FOutVal := EULER_REC;
        c_rpi           : FOutVal := PI_REC;
        c_rsqrt2        : FOutVal := SQRT2_REC;
        c_rsqrt3        : FOutVal := SQRT3_REC;
        c_rsqrt5        : FOutVal := SQRT5_REC;
        c_r12r2         : FOutVal := TWO_PTWELVE_REC;
        c_uni_0         : FOutVal := 0;
        c_mi_sec_1      : FOutVal :=      NOTE_SCALING_REC;
        c_second_2      : FOutVal :=  2 * NOTE_SCALING_REC;
        c_mi_third_3    : FOutVal :=  3 * NOTE_SCALING_REC;
        c_ma_third_4    : FOutVal :=  4 * NOTE_SCALING_REC;
        c_fourth_5      : FOutVal :=  5 * NOTE_SCALING_REC;
        c_tritone_6     : FOutVal :=  6 * NOTE_SCALING_REC;
        c_fifth_7       : FOutVal :=  7 * NOTE_SCALING_REC;
        c_mi_sixth_8    : FOutVal :=  8 * NOTE_SCALING_REC;
        c_ma_sixth_9    : FOutVal :=  9 * NOTE_SCALING_REC;
        c_mi_seventh_10 : FOutVal := 10 * NOTE_SCALING_REC;
        c_ma_seventh_11 : FOutVal := 11 * NOTE_SCALING_REC;
        c_octave_12     : FOutVal := 12 * NOTE_SCALING_REC;
        else              FOutVal := 0;
      end;

      FOutputs[ o_out] := FOutVal;
    end;


{ ========
  TModConstant = class( TMod)
  public
}

    procedure   TModConstant.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_value, 'value');
      AddInput ( i_mute , 'mute' );
      AddInput ( i_mode , 'mode' );
      AddOutput( o_out  , 'out'  );
    end;


    procedure   TModConstant.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_value
      ];
    end;


    procedure   TModConstant.DoSlowTick; // override;
    begin
      if Round( FInputs[ i_mode]) = m_note
      then FOutputs[ o_out] := ( FInputs[ i_value]) - NoteNumberToUnits( MiddleNote)
      else FOutputs[ o_out] :=   FInputs[ i_value] * SignalToMute( FInputs[ i_mute]);
    end;


{ ========
  TModKnobs = class( TMod)
  public
}

    procedure   TModKnobs4.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_value1, 'value1');
      AddInput ( i_mode1 , 'mode1' );
      AddInput ( i_value2, 'value2');
      AddInput ( i_mode2 , 'mode2' );
      AddInput ( i_value3, 'value3');
      AddInput ( i_mode3 , 'mode3' );
      AddInput ( i_value4, 'value4');
      AddInput ( i_mode4 , 'mode4' );
      AddInput ( i_in1   , 'in1'   );
      AddInput ( i_in2   , 'in2'   );
      AddInput ( i_in3   , 'in3'   );
      AddInput ( i_in4   , 'in4'   );
      AddInput ( i_mute1 , 'mute1' );
      AddInput ( i_mute2 , 'mute2' );
      AddInput ( i_mute3 , 'mute3' );
      AddInput ( i_mute4 , 'mute4' );
      AddOutput( o_out1  , 'out1'  );
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_out3  , 'out3'  );
      AddOutput( o_out4  , 'out4'  );
    end;


    procedure   TModKnobs4.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_value1,
        i_value2,
        i_value3,
        i_value4,
        i_mute1 , // Leave these four mutes in the zipper map
        i_mute2 , // as this will be used for playing a patch
        i_mute3 ,
        i_mute4
      ];

      FInputs[ i_in1] := 1.0;
      FInputs[ i_in2] := 1.0;
      FInputs[ i_in3] := 1.0;
      FInputs[ i_in4] := 1.0;
    end;


    procedure   TModKnobs4.DoSlowTick; // override;
    begin
      if Round( FInputs[ i_mode1]) = m_note
      then FOutputs[ o_out1] := ( FInputs[ i_value1] * FInputs[ i_in1] * SignalToMute( FInputs[ i_mute1])) - NoteNumberToUnits( MiddleNote)
      else FOutputs[ o_out1] :=   FInputs[ i_value1] * FInputs[ i_in1] * SignalToMute( FInputs[ i_mute1]);

      if Round( FInputs[ i_mode2]) = m_note
      then FOutputs[ o_out2] := ( FInputs[ i_value2] * FInputs[ i_in2] * SignalToMute( FInputs[ i_mute2])) - NoteNumberToUnits( MiddleNote)
      else FOutputs[ o_out2] :=   FInputs[ i_value2] * FInputs[ i_in2] * SignalToMute( FInputs[ i_mute2]);

      if Round( FInputs[ i_mode3]) = m_note
      then FOutputs[ o_out3] := ( FInputs[ i_value3] * FInputs[ i_in3] * SignalToMute( FInputs[ i_mute3])) - NoteNumberToUnits( MiddleNote)
      else FOutputs[ o_out3] :=   FInputs[ i_value3] * FInputs[ i_in3] * SignalToMute( FInputs[ i_mute3]);

      if Round( FInputs[ i_mode4]) = m_note
      then FOutputs[ o_out4] := ( FInputs[ i_value4] * FInputs[ i_in4] * SignalToMute( FInputs[ i_mute4])) - NoteNumberToUnits( MiddleNote)
      else FOutputs[ o_out4] :=   FInputs[ i_value4] * FInputs[ i_in4] * SignalToMute( FInputs[ i_mute4]);
    end;



{ ========
  TModSwitches4 = class( TMod)
  public
}

    procedure   TModSwitches4.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_sw1 , 'sw1' );
      AddInput ( i_sw2 , 'sw2' );
      AddInput ( i_sw3 , 'sw3' );
      AddInput ( i_sw4 , 'sw4' );
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
    end;


    procedure   TModSwitches4.SetDefaults; // override;
    begin
      FInputs[ i_in1] := 1;
      FInputs[ i_in2] := 1;
      FInputs[ i_in3] := 1;
      FInputs[ i_in4] := 1;
    end;


    procedure   TModSwitches4.DoSlowTick; // override;
    begin
      FOutputs[ o_out1] := FInputs[ i_sw1] * FInputs[ i_in1];
      FOutputs[ o_out2] := FInputs[ i_sw2] * FInputs[ i_in2];
      FOutputs[ o_out3] := FInputs[ i_sw3] * FInputs[ i_in3];
      FOutputs[ o_out4] := FInputs[ i_sw4] * FInputs[ i_in4];
    end;


{ ========
  TModButtons4 = class( TMod)
  private
    FCount1    : Integer;
    FCount2    : Integer;
    FCount3    : Integer;
    FCount4    : Integer;
    FLedCount1 : Integer;
    FLedCount2 : Integer;
    FLedCount3 : Integer;
    FLedCount4 : Integer;
  public
}

    procedure   TModButtons4.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_sw1 , 'sw1' );
      AddInput ( i_sw2 , 'sw2' );
      AddInput ( i_sw3 , 'sw3' );
      AddInput ( i_sw4 , 'sw4' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
    end;


    procedure   TModButtons4.Pulse( anInput: Integer); // override;
    begin
      case anInput of
        i_sw1 : begin FCount1 := TrigPulseTime( IsSpedUp); FLedCount1 := LedFlashTime( IsSpedUp); end;
        i_sw2 : begin FCount2 := TrigPulseTime( IsSpedUp); FLedCount2 := LedFlashTime( IsSpedUp); end;
        i_sw3 : begin FCount3 := TrigPulseTime( IsSpedUp); FLedCount3 := LedFlashTime( IsSpedUp); end;
        i_sw4 : begin FCount4 := TrigPulseTime( IsSpedUp); FLedCount4 := LedFlashTime( IsSpedUp); end;
      end;
    end;


    procedure   TModButtons4.DoSlowTick; // override;
    begin
      FOutputs[ o_out1] := LogicToSignal( FCOunt1 > 0);
      FOutputs[ o_out2] := LogicToSignal( FCOunt2 > 0);
      FOutputs[ o_out3] := LogicToSignal( FCOunt3 > 0);
      FOutputs[ o_out4] := LogicToSignal( FCOunt4 > 0);

      DecToZero( FCount1); DecToZero( FLedCount1);
      DecToZero( FCount2); DecToZero( FLedCount2);
      DecToZero( FCount3); DecToZero( FLedCount3);
      DecToZero( FCount4); DecToZero( FLedCount4);
    end;


    procedure   TModButtons4.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, '1', LogicToSignal( FLedCount1 > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, '2', LogicToSignal( FLedCount2 > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, '3', LogicToSignal( FLedCount3 > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, '4', LogicToSignal( FLedCount4 > 0)));
    end;


{ ========
  TModTapper = class( TMod)
  private
    FCount     : Integer;
    FLedCount  : Integer;
    FTapCount  : Integer;
    FTrigCount : Integer;
    FTime      : Cardinal;
    FBPM       : TSignal;
  public
}

    procedure   TModTapper.CreateIO; // override;
    begin
      FIsSlow  := True;
      AddInput ( i_sw      , 'sw'      );
      AddInput ( i_bpm     , 'bpm'     );
      AddInput ( i_sync    , 'sync'    );
      AddInput ( i_rate    , 'rate'    );
      AddInput ( i_mode    , 'mode'    );
      AddInput ( i_inslave , 'inslave' );
      AddOutput( o_outpulse, 'outpulse');
      AddOutput( o_outspeed, 'outspeed');
      AddOutput( o_outslave, 'outslave');
    end;


    procedure   TModTapper.SetDefaults; // override;
    begin
      FChanged := False;
      FInputs[ i_inslave] := -1;
    end;


    procedure   TModTapper.Pulse( anInput: Integer); // override;
    var
      aTime : TSignal;
    begin
      case anInput of

        i_sw :

          begin
            FCount    := TrigPulseTime( IsSpedUp);
            FLedCount := LedFlashTime( IsSpedUp);
            FTapCount := Round( SlowTimeToSampleCount( 12));       //  12  s = 5 BPM
            Inc( FTrigCount);

            if FTrigCount > 1
            then begin
              aTime := 1e-3 * ( TimeGetTime - FTime);

              if aTime > 0
              then begin
                FBPM     := Round( 60.0 / aTime);
                FChanged := True;
              end;
            end;

            FTime := TimeGetTime;
          end;

        i_sync :

          begin
            FCount    := TrigPulseTime( IsSpedUp);
            FLedCount := LedFlashTime ( IsSpedUp);
          end;

      end;
    end;


    procedure   TModTapper.DoSlowTick; // override;
    const
      OS  = 1.0 / 60.0;
    var
      aMode  : Integer;
      aRate  : TSignal;
      aSlave : TSignal;
    begin
      aSlave := 2000 *               FInputs[ i_inslave];
      aMode  := Round              ( FInputs[ i_mode   ]);
      aRate  := RateMultiplierValue( FInputs[ i_rate   ]);

      if not FChanged
      then begin
        if ( aSlave >= 0) and ( aSlave <> FBPM)
        then begin
          FBPM     := aSlave;
          FChanged := True;
        end
        else if ( FBPM <> FInputs[ i_bpm])
        then FBPM := FInputs[ i_bpm];
      end;

      FOutputs[ o_outpulse] := LogicToSignal( FCOunt > 0  );
      FOutputs[ o_outslave] := FBPM / 2000.0;

      case aMode of
        1 :  FOutputs[ o_outspeed] := 60.0 / (    3.0 * Clip( aRate * FBPM, 0.001, 2000));   // DELAY med 3.0 s
        2 :  FOutputs[ o_outspeed] := 60.0 / (  300.0 * Clip( aRate * FBPM, 0.001, 2000));   // DELAY long 5:00
        3 :  FOutputs[ o_outspeed] := LfoBPMFrequencyToUnits( aRate * FBPM / 60);            // BPM ClockGen
        else FOutputs[ o_outspeed] := LfoMediumFrequencyToUnits(  aRate * FBPM / 60.0 );     // LFO med 2:05
      end;

      DecToZero( FCount   );
      DecToZero( FLedCount);

      if FTapCount = 0
      then begin
        FTrigCount := 0;
        FTime      := TimeGetTime;
      end
      else if FTapCount > 0
      then Dec( FTapCount);
    end;


    procedure   TModTapper.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'pulse', LogicToSignal( FLedCount > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'busy' , LogicToSignal( FTapCount > 0)));
    end;


    procedure   TModTapper.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // override;
    begin
      if FChanged
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'bpm', FBPM));
        FChanged := False;
      end;
    end;


{ ========
  TModPad = class( TMod)
  private
    FFileName        : string;
    FFileNameChanged : Boolean;
  private
}

    procedure   TModPad.SetFileName( const aValue: string);
    begin
      if aValue <> FFileName
      then begin
        FFileName        := aValue;
        FFileNameChanged := True;
      end;
    end;


//  public

    procedure   TModPad.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_padx      , 'padx'      );
      AddInput ( i_pady      , 'pady'      );
      AddInput ( i_padr      , 'padr'      );
      AddInput ( i_padg      , 'padg'      );
      AddInput ( i_padb      , 'padb'      );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_x         , 'x'         );
      AddOutput( o_y         , 'y'         );
      AddOutput( o_r         , 'r'         );
      AddOutput( o_g         , 'g'         );
      AddOutput( o_b         , 'b'         );
    end;


    procedure   TModPad.SetStringValue( const aName, aValue: string); // override;
    begin
      if SameText( aName, 'filename')
      then FileName := aValue
      else inherited;
    end;


    function    TModPad.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'filename')
      then Result := FileName
      else Result := inherited;
    end;


    procedure   TModPad.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_padx,
        i_pady,
        i_padr,
        i_padg,
        i_padb
      ];
    end;


    procedure   TModPad.DoSlowTick; // override;
    var
      aType : Integer;
    begin
      aType := Round( FInputs[ i_outputtype]);
      FOutputs[ o_x] := SignalForOutputType( FInputs[ i_padx], aType);
      FOutputs[ o_y] := SignalForOutputType( FInputs[ i_pady], aType);
      FOutputs[ o_r] := SignalForOutputType( FInputs[ i_padr], aType);
      FOutputs[ o_g] := SignalForOutputType( FInputs[ i_padg], aType);
      FOutputs[ o_b] := SignalForOutputType( FInputs[ i_padb], aType);
    end;


    procedure   TModPad.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      if FFileNameChanged
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'filename', FileName));
        FFileNameChanged := False;
      end;
    end;


{ ========
  TModSpiral = class( TMod)
  public
}

    procedure   TModSpiral.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_padx   , 'padx'   );
      AddInput ( i_pady   , 'pady'   );
      AddInput ( i_steps  , 'steps'  );
      AddOutput( o_outnote, 'outnote');
      AddOutput( o_outphi , 'outphi' );
      AddOutput( o_outnorm, 'outnorm');
      AddOutput( o_outlen , 'outlen' );
    end;


    procedure   TModSpiral.DoSlowTick; // override;
    var
      aPoint    : TComplex;
      aPolar    : TPolar;
      aNorm     : TSignal;
      aPhi      : TSignal;
      anArcLen  : TSignal;
      aSemiTone : TSignal;
      aSteps    : TSignal;
    begin
      aPoint.Real := FInputs[ i_padx];
      aPoint.Imag := FInputs[ i_pady];
      aSteps      := FInputs[ i_steps];
      aPolar      := aPoint.ToPolar;
      aPolar.NormalizePhi( 0, TWO_PI);
      aNorm       := aPolar.Norm;
      aPhi        := aPolar.Phi;
      anArcLen    := 7.1508378 * aNorm;
      aSemiTone   := NoteNumberToUnits( anArcLen * aSteps);   // aSteps semitones per revolution
      FOutputs[ o_outnote] := aSemiTone;
      FOutputs[ o_outphi ] := aPhi;
      FOutputs[ o_outnorm] := aNorm;
      FOutputs[ o_outlen ] := anArcLen;
    end;


{ ========
  TModPolar = class( TMod)
    o_outa = 0;
    o_outd = 1;
  public
}

    procedure   TModPolar.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_inx , 'inx' );
      AddInput ( i_iny , 'iny' );
      AddOutput( o_outa, 'outa');
      AddOutput( o_outd, 'outd');
    end;


    procedure   TModPolar.DoSlowTick; // override;
    var
      aPoint : TComplex;
      aPolar : TPolar;
    begin
      aPoint.Real := FInputs[ i_inx];
      aPoint.Imag := FInputs[ i_iny];
      aPolar      := aPoint.ToPolar;
      aPolar.NormalizePhi( 0, TWO_PI);
      FOutputs[ o_outa] := aPolar.Phi;
      FOutputs[ o_outd] := aPolar.Norm;
    end;


{ ========
  TModRectangular = class( TMod)
  public
}

    procedure   TModRectangular.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_ina , 'ina' );
      AddInput ( i_ind , 'ind' );
      AddOutput( o_outx, 'outx');
      AddOutput( o_outy, 'outy');
    end;


    procedure   TModRectangular.DoSlowTick; // override;
    var
      aPoint : TComplex;
      aPolar : TPolar;
    begin
      aPolar.Phi  := FInputs[ i_ina];
      aPolar.Norm := FInputs[ i_ind];
      aPolar.NormalizePhi( 0, TWO_PI);
      aPoint      := aPolar.ToComplex;
      FOutputs[ o_outx] := aPoint.Real;
      FOutputs[ o_outy] := aPoint.Imag;
    end;


{ ========
  TModInverter = class( TMod)
  public
}

    procedure   TModInverter.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
    end;


    procedure   TModInverter.DoSlowTick; // override;
    var
      i : Integer;
    begin
      for i := i_in1 to i_in4
      do FOutputs[ i - i_in1 + o_out1] := - FInputs[ i];
    end;


{ ========
  TModAdder = class( TMod)
  public
}

    procedure   TModAdder.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_mute, 'mute');
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddInput ( i_in5 , 'in5' );
      AddInput ( i_in6 , 'in6' );
      AddInput ( i_in7 , 'in7' );
      AddOutput( o_out , 'out' );
    end;


    procedure   TModAdder.SetDefaults; // override;
    var
      i : Integer;
    begin
      for i := i_in1 to i_in7          // Make sure unused inputs are set to 0, so it'll 'work'
      do FInputs[ i] := 0;

      FDezipperMap := [
      ];
    end;


    procedure   TModAdder.DoSlowTick; // override;
    var
      i : Integer;
      R : TSignal;
    begin
      R := 0.0;

      for i := i_in1 to i_in7
      do R := R + FInputs[ i];

      FOutputs[ o_out] := R * SignalToMute( FInputs[ i_mute]);
    end;


{ ========
  TModMultiplier = class( TMod)
  public
}

    procedure   TModMultiplier.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_mute, 'mute');
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddInput ( i_in5 , 'in5' );
      AddInput ( i_in6 , 'in6' );
      AddInput ( i_in7 , 'in7' );
      AddOutput( o_out , 'out' );
    end;


    procedure   TModMultiplier.SetDefaults; // override;
    var
      i : Integer;
    begin
      for i := i_in1 to i_in7          // Make sure unused inputs are set to 1, so it'll 'work'
      do FInputs[ i] := 1;

      FDezipperMap := [
      ];
    end;


    procedure   TModMultiplier.DoSlowTick; // override;
    var
      i : Integer;
      R : TSignal;
    begin
      R := SignalToMute( FInputs[ i_mute]);

      for i := i_in1 to i_in7
      do R := Normalize( R * FInputs[ i]);

      FOutputs[ o_out] := R;
    end;


{ ========
  TModReciprocal = class( TMod)
  public
}

    procedure   TModReciprocal.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_x  , 'x'  );
      AddOutput( o_out, 'out');
    end;


    procedure   TModReciprocal.DoSlowTick; // override;
    var
      aValue : TSignal;
    begin
      aValue := FInputs[ i_x];

      if aValue = 0.0
      then FOutputs[ o_out] := 0.0
      else FOutputs[ o_out] := Clip( 1 / aValue, -1e3, 1e3);
    end;


{ ========
  TModDivide = class( TMod)
  public
}

    procedure   TModDivide.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_x  , 'x'  );
      AddInput ( i_y  , 'y'  );
      AddOutput( o_out, 'out');
    end;


    procedure   TModDivide.DoSlowTick; // override;
    var
      aValue : TSignal;
    begin
      aValue := FInputs[ i_y];

      if aValue = 0.0
      then FOutputs[ o_out] := FInputs[ i_x]
      else FOutputs[ o_out] := Clip( FInputs[ i_x] / aValue, -1e3, 1e3);
    end;


{ ========
  TModMonadic = class( TMod)
  public
}

    procedure   TModMonadic.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_function, 'function');
      AddInput ( i_x       , 'x'       );
      AddOutput( o_out     , 'out'     );
    end;


    procedure   TModMonadic.DoSlowTick; // override;
    var
      X         : TSignal;
      aFunction : Integer;
      OutVal    : TSignal;
    begin
      X         := FInputs[ i_x];
      aFunction := Round( FInputs[ i_function]);
      OutVal    := FOutputs[ o_out];                // When math fails keep output value frozen

      case aFunction of

        f_squared    :                   //    y = x ^ 2
          OutVal := X * X;

        f_cubed      :                   //    y = x ^ 3
          OutVal := X * X * X;

        f_squareroot :                   //    y = sqrt( x)
          if X >= 0
          then OutVal := Sqrt( X);

        f_cubicroot  :                   //    y = x ^ 1 / 3
          OutVal := Power( 1 / 3, X);

        f_sine       :                   //    y = sin( x)
          OutVal := Sin( X);

        f_cosine     :                   //    y = cos( x)
          OutVal := Cos( X);

        f_tangent    :                   //    y = tan( x)
          if Abs( X) <> HALF_PI
          then OutVal := Tan( X);

        f_arcsine    :                   //    y = asin( x)
          if Abs( X) <= 1.0
          then OutVal := ArcSin( X);

        f_arccosine  :                   //    y = acos( x)
          if Abs( X) <= 1.0
          then OutVal := ArcCos( X);

        f_arctangent :                   //    y = atan( x)
          OutVal := ArcTan( X);

        f_2powerx    :                   //    y = 2^x
          OutVal := Power( 2, X);

        f_epowerx    :                   //    y = e^x
          OutVal := Power( EULER, X);

        f_10powerx   :                   //    y = 10^x
          OutVal := Power( 10, X);

        f_2logx      :                   //    y = 2log(x)
          if X > 0
          then OutVal := Log2( X);

        f_elogx      :                   //    y = ln(x)
          if X > 0
          then OutVal := Ln( X);

        f_10logx     :                   //    y = 10log(x)
          if X > 0
          then OutVal := Log10( X);

        f_2pwxdiv12  :                   //    y = 2^(x/12)
          OutVal := Power( 2, X / 12);

        f_12t2logx :                     //    y = 12*2log(x)
          if X > 0
          then OutVal := 12 * Log2( X);

        f_ttimespi :                     //    y = x * Pi
          OutVal := X * Pi;

        f_timestwopi :                   //    y = x * 2 * Pi
          OutVal := X * TWO_PI;

        f_divpi :                        //    y = x / Pi
          OutVal := X * PI_REC;

        f_divtwopi :                     //    y = x / ( 2 * Pi)
          OutVal := X * TWO_PI_REC;

        f_floor :                        //    y = floor( x)
          OutVal := Floor( X);

        f_round :                        //    y = round( x)
          OutVal := Round( X);

        f_ceil :                         //    y = ceil( x)
          OutVal := Ceil( X);

        f_abs :                          //    y = abs( x)
          OutVal := Abs( X);

        f_negate :                       //    y = - x
          OutVal := - X;

        f_flip :                         //    y = 1 - x
          OutVal := 1.0 - X;

        f_tanh :                         //   y = tanh( x)
          OutVal := tanh( X);

        f_absflip :                      //   1 - abs( x)
          OutVal := 1.0 - Abs( X);

      end;

      FOutputs[ o_out] := Clip( OutVal, -1e3, 1e3);
    end;


{ ========
  TModDyadic = class( TMod)
  public
}

    procedure   TModDyadic.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_function, 'function');
      AddInput ( i_x       , 'x'       );
      AddInput ( i_y       , 'y'       );
      AddOutput( o_out     , 'out'     );
    end;


    procedure   TModDyadic.DoSlowTick; // override;
    var
      X         : TSignal;
      Y         : TSignal;
      aFunction : Integer;
      OutVal    : TSignal;
    begin
      X         := FInputs[ i_x];
      Y         := FInputs[ i_y];
      aFunction := Round( FInputs[ i_function]);
      OutVal    := FOutputs[ o_out];              // When math fails keep old out value frozen

      case aFunction of

        f_modulo :                                // x mod y
          if Y <> 0
          then OutVal := MathFloatMod( X, Y);

        f_power  :                                // x ^ y
          OutVal := Power( X, Y);

        flog     :                                // xlog( y)
          if ( Y > 0) and ( X > 0)
          then OutVal := LogN( X, Y);

        f_Atan2  :                                // atan2( y / x)
          if X <> 0
          then OutVal := ArcTan2( Y, X);

        f_xminy  :                                // x - y
          OutVal := X - Y;

        f_yminx   :                               // y - x
          OutVal := Y - X;

        f_xplusy  :                               // x + y
          OutVal := X + Y;

        f_invpower :                              // x ^ ( 1 / y)
          if Y <> 0
          then OutVal := Power( X, 1.0 / Y);

      end;

      FOutputs[ o_out] := Clip( OutVal, -1e3, 1e3);
    end;


{ ========
  TModMultiMult = class( TMod)
  public
}

    procedure   TModMultiMult.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_mult, 'mult');
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddInput ( i_in5 , 'in5' );
      AddInput ( i_in6 , 'in6' );
      AddInput ( i_in7 , 'in7' );
      AddInput ( i_in8 , 'in8' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
      AddOutput( o_out5, 'out5');
      AddOutput( o_out6, 'out6');
      AddOutput( o_out7, 'out7');
      AddOutput( o_out8, 'out8');
    end;


    procedure   TModMultiMult.SetDefaults; // override;
    begin
      FInputs[ i_mult] := 1; // set module transparent with unconnected multiplier
    end;


    procedure   TModMultiMult.DoSlowTick; // override;
    var
      Mult : TSignal;
    begin
      Mult              :=        FInputs[ i_mult];
      FOutputs[ o_out1] := Mult * FInputs[ i_in1 ];
      FOutputs[ o_out2] := Mult * FInputs[ i_in2 ];
      FOutputs[ o_out3] := Mult * FInputs[ i_in3 ];
      FOutputs[ o_out4] := Mult * FInputs[ i_in4 ];
      FOutputs[ o_out5] := Mult * FInputs[ i_in5 ];
      FOutputs[ o_out6] := Mult * FInputs[ i_in6 ];
      FOutputs[ o_out7] := Mult * FInputs[ i_in7 ];
      FOutputs[ o_out8] := Mult * FInputs[ i_in8 ];
    end;


{ ========
  TModMultipleMult = class( TMod)
  public
}

    procedure   TModMultipleMult.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in11, 'in11');
      AddInput ( i_in12, 'in12');
      AddInput ( i_in21, 'in21');
      AddInput ( i_in22, 'in22');
      AddInput ( i_in31, 'in31');
      AddInput ( i_in32, 'in32');
      AddInput ( i_in41, 'in41');
      AddInput ( i_in42, 'in42');
      AddInput ( i_in51, 'in51');
      AddInput ( i_in52, 'in52');
      AddInput ( i_in61, 'in61');
      AddInput ( i_in62, 'in62');
      AddInput ( i_in71, 'in71');
      AddInput ( i_in72, 'in72');
      AddInput ( i_in81, 'in81');
      AddInput ( i_in82, 'in82');
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
      AddOutput( o_out5, 'out5');
      AddOutput( o_out6, 'out6');
      AddOutput( o_out7, 'out7');
      AddOutput( o_out8, 'out8');
    end;


    procedure   TModMultipleMult.SetDefaults; // override;
    begin
      FInputs[ i_in11] := 1; // set unused inputs to one
      FInputs[ i_in12] := 1;
      FInputs[ i_in21] := 1;
      FInputs[ i_in32] := 1;
      FInputs[ i_in31] := 1;
      FInputs[ i_in32] := 1;
      FInputs[ i_in41] := 1;
      FInputs[ i_in42] := 1;
      FInputs[ i_in51] := 1; //
      FInputs[ i_in52] := 1;
      FInputs[ i_in61] := 1;
      FInputs[ i_in62] := 1;
      FInputs[ i_in71] := 1;
      FInputs[ i_in72] := 1;
      FInputs[ i_in81] := 1;
      FInputs[ i_in82] := 1;
    end;


    procedure   TModMultipleMult.DoSlowTick; // override;
    begin
      FOutputs[ o_out1] := FInputs[ i_in11] * FInputs[ i_in12];
      FOutputs[ o_out2] := FInputs[ i_in21] * FInputs[ i_in22];
      FOutputs[ o_out3] := FInputs[ i_in31] * FInputs[ i_in32];
      FOutputs[ o_out4] := FInputs[ i_in41] * FInputs[ i_in42];
      FOutputs[ o_out5] := FInputs[ i_in51] * FInputs[ i_in52];
      FOutputs[ o_out6] := FInputs[ i_in61] * FInputs[ i_in62];
      FOutputs[ o_out7] := FInputs[ i_in71] * FInputs[ i_in72];
      FOutputs[ o_out8] := FInputs[ i_in81] * FInputs[ i_in82];
    end;


{ ========
  TModNoteToDelay = class( TMod)
  public
}

    procedure   TModNoteToDelay.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in , 'in' );
      AddOutput( o_out, 'out');
    end;


    procedure   TModNoteToDelay.DoSlowTick; // override;
    begin
      FOutputs[ o_out] := 1 / LookupFrequency( FInputs[ i_in]);
    end;


{ ========
  TModZeroCounter = class( TMod)
  private
    FOldSign : Boolean;
    FTime    : Integer;
    FFreq    : TSignal;
  public
}

    procedure   TModZeroCounter.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in       , 'in'       );
      AddInput ( i_threshold, 'threshold');
      AddOutput( o_period   , 'period'   );
      AddOutput( o_gate     , 'gate'     );
      AddOutput( o_note     , 'note'     );
    end;


    procedure   TModZeroCounter.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_threshold
      ];
    end;


    procedure   TModZeroCounter.DoTick; // override;
    var
      aValue : TSignal;
      aLevel : TSignal;
      aSign  : Boolean;
      aTime  : TSignal;
    begin
      Inc( FTime);
      aValue := FInputs[ i_in];
      aLevel := Abs( aValue);

      if aLevel > FInputs[ i_threshold]
      then begin
        FOutputs[ o_gate] := 1;
        aSign := SignalToLogic( aValue);

        if aSign <> FOldSign
        then begin
          FOldSign := aSign;
          FOutputs[ o_period] := LogicToSignal( aSign);

          if not aSign
          then begin
            aTime := FTime * System_Rate_Rec;

            if aTime > 0
            then FFreq := 1 / aTime;

            FTime := 0;
          end;
        end;
      end
      else FOutputs[ o_gate] := -1;
    end;


    procedure   TModZeroCounter.DoSlowTick; // override;
    begin
      FOutputs[ o_note] := FrequencyToUnits( FFreq) - NoteNumberToUnits( MiddleNote);
    end;


    procedure   TModZeroCounter.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'gate', FOutputs[ o_gate]));
    end;


{ ========
  TModMedian = class( TMod)
  private
    FData      : TMedianCalculator;
    FOldTrig   : Boolean;
  public
}

    constructor TModMedian.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FData := TMedianCalculator.Create( 0);
    end;


    procedure   TModMedian.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in  , 'in'  );
      AddInput ( i_trig, 'trig');
      AddOutput( o_out , 'out' );
      AddOutput( o_high, 'high');
      AddOutput( o_low , 'low' );
      AddOutput( o_avg , 'avg' );
    end;


    destructor  TModMedian.Destroy; // override;
    var
      OldData  : TMedianCalculator;
    begin
      Locked  := True;
      OldData := AtomicExchange( Pointer( FData), nil);
      FreeAndNil( OldData);
      inherited;
    end;


    procedure   TModMedian.SetInternal( const aName: string; aValue: TSignal); // override;
    var
      aBinSize : Integer;
      NewData  : TMedianCalculator;
      OldData  : TMedianCalculator;
      i        : Integer;
    begin
      if SameText( aName, 'binsize')
      then begin
        aBinSize := Round( aValue);

        if not Assigned( FData) or ( aBinSize <> FData.Count)
        then begin
          NewData := TMedianCalculator.Create( aBinSize);

          for i := 0 to FData.Count - 1
          do NewData.Append( FData.GetData);

          Locked := True;

          try
            OldData := AtomicExchange( Pointer( FData), Pointer( NewData));
            FreeAndnil( OldData);
          finally
            Locked := False;
          end;
        end;
      end
      else inherited;
    end;


    function    TModMedian.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'binsize')
      then begin
        if Assigned( FData)
        then Result := FData.Count
        else Result := 0;
      end
      else Result := inherited;
    end;


    procedure   TModMedian.DoSlowTick; // override;
    var
      aTrig : Boolean;
    begin
      aTrig := SignalToLogic( FInputs[ i_trig]);

      if ( aTrig <> FOldTrig) and aTrig and Assigned( FData) and ( FData.Count > 0)
      then begin
        FData.Append( FInputs[ i_in]);
        FOutputs[ o_out ] := FData.Median;
        FOutputs[ o_high] := FData.Highest;
        FOutputs[ o_low ] := FData.Lowest;
        FOutputs[ o_avg ] := FData.Average;
      end;

      FOldTrig := aTrig;
    end;


    procedure   TModMedian.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig', FInputs[ i_trig]));
    end;


{ ========
  TModCompare = class( TMod)
  public
}

    procedure   TModCompare.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in1     , 'in1'     );
      AddInput ( i_in2     , 'in2'     );
      AddInput ( i_range   , 'range'   );
      AddOutput( o_outless , 'outless' );
      AddOutput( o_outequal, 'outequal');
      AddOutput( o_outmore , 'outmore' );
    end;


    procedure   TModCompare.DoSlowTick; // override;
    var
      in1      : TSignal;
      in2      : TSignal;
      outless  : TSignal;
      outequal : TSignal;
      outmore  : TSignal;
    begin
      in1      := FInputs[ i_in1];
      in2      := FInputs[ i_in2];

      outless  := LogicToSignal( False);
      outequal := LogicToSignal( False);
      outmore  := LogicToSignal( False);

      if AlmostEqual( in1, in2, FInputs[ i_range])
      then outequal := LogicToSignal( True)
      else if in1 < in2
      then outless := LogicToSignal( True)
      else outmore := LogicToSignal( True);

      FOutputs[ o_outless ] := outless;
      FOutputs[ o_outequal] := outequal;
      FOutputs[ o_outmore ] := outmore;
    end;


    procedure   TModCompare.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'less' , FOutputs[ o_outless ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'equal', FOutputs[ o_outequal]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'more' , FOutputs[ o_outmore ]));
    end;


{ ========
  TModGate = class( TMod)
  public
}

    procedure   TModGate.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_function  , 'function'  );
      AddInput ( i_width     , 'width'     );
      AddInput ( i_in1       , 'in1'       );
      AddInput ( i_in2       , 'in2'       );
      AddInput ( i_in3       , 'in3'       );
      AddInput ( i_in4       , 'in4'       );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_func      , 'func'      );
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModGate.SetDefaults; // override;
    var
      i : Integer;
    begin
      case Round( FInputs[ i_function]) of

        m_and :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 1;
          end;

        m_nand :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 1;
          end;

        m_or :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 0, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_nor :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 0, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_xor :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 0, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_xnor :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 0, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_one :

          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 0, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        else // none
          begin
            for i := i_in1 to i_in4         // Make sure unused inputs are set to 0, so it'll 'work'
            do FInputs[ i] := 0;
          end;
      end;
    end;


    procedure   TModGate.DoSlowTick; // override;
    var
      i         : Integer;
      R         : TBinary;
      OneCounts : Integer;
      BitCount  : Integer;
      aFunction : Integer;
      anOutType : Integer;
    begin
      aFunction := Clip( Round( FInputs[ i_function  ] + ( m_none - m_and + 0.999999) * FInputs[ i_func]) mod ( m_none - m_and + 1), 0, 7);
      anOutType := Round( FInputs[ i_outputtype]);

      if aFunction <> FFunction
      then begin
        FFunction := aFunction;
        SetDefaults;
      end;

      BitCount := Round( FInputs[ i_width]);

      case aFunction of

        m_and :

          begin
            if BitCount = 1
            then begin
              FOutputs[ o_out] :=
                SignalForOutputType(
                  LogicToSignal(
                    SignalToLogic( FInputs[ i_in1]) and
                    SignalToLogic( FInputs[ i_in2]) and
                    SignalToLogic( FInputs[ i_in3]) and
                    SignalToLogic( FInputs[ i_in4])
                  )
                  , anOutType
                );
            end
            else begin
              R := -1;

              for i := i_in1 to i_in4
              do  R := R and SignalToBinary( FInputs[ i], BitCount);

              FOutputs[ o_out] := SignalForOutputType( BinaryToSignal( R, BitCount), anOutType);
            end;
          end;

        m_nand :

          begin
            if BitCount = 1
            then begin
              FOutputs[ o_out] :=
                SignalForOutputType(
                  LogicToSignal(
                    not
                    (
                      SignalToLogic( FInputs[ i_in1]) and
                      SignalToLogic( FInputs[ i_in2]) and
                      SignalToLogic( FInputs[ i_in3]) and
                      SignalToLogic( FInputs[ i_in4])
                    )
                  )
                  , anOutType
                );
            end
            else begin
              R := -1;

              for i := i_in1 to i_in4
              do  R := R and SignalToBinary( FInputs[ i], BitCount);

              FOutputs[ o_out] := SignalForOutputType( BinaryToSignal( not R, BitCount), anOutType);
            end;
          end;

        m_or :

          begin
            if BitCount = 1
            then begin
              FOutputs[ o_out] :=
                SignalForOutputType(
                  LogicToSignal(
                    SignalToLogic( FInputs[ i_in1]) or
                    SignalToLogic( FInputs[ i_in2]) or
                    SignalToLogic( FInputs[ i_in3]) or
                    SignalToLogic( FInputs[ i_in4])
                  )
                  , anOutType
                );
            end
            else begin
              R := 0;

              for i := i_in1 to i_in4
              do  R := R or SignalToBinary( FInputs[ i], BitCount);

              FOutputs[ o_out] := SignalForOutputType( BinaryToSignal( R, BitCount), anOutType);
            end;
          end;

        m_nor :

          begin
            if BitCount = 1
            then begin
              FOutputs[ o_out] :=
                SignalForOutputType(
                  LogicToSignal(
                    not
                    (
                      SignalToLogic( FInputs[ i_in1]) or
                      SignalToLogic( FInputs[ i_in2]) or
                      SignalToLogic( FInputs[ i_in3]) or
                      SignalToLogic( FInputs[ i_in4])
                    )
                  )
                  , anOutType
                );
            end
            else begin
              R := 0;

              for i := i_in1 to i_in4
              do  R := R or SignalToBinary( FInputs[ i], BitCount);

              FOutputs[ o_out] := SignalForOutputType( BinaryToSignal( not R, BitCount), anOutType);
            end;
          end;

        m_xor :

          begin
            if BitCount = 1
            then begin
              FOutputs[ o_out] :=
                SignalForOutputType(
                  LogicToSignal(
                    SignalToLogic( FInputs[ i_in1]) xor
                    SignalToLogic( FInputs[ i_in2]) xor
                    SignalToLogic( FInputs[ i_in3]) xor
                    SignalToLogic( FInputs[ i_in4])
                  )
                  , anOutType
                );
            end
            else begin
              R := 0;

              for i := i_in1 to i_in4
              do  R := R xor SignalToBinary( FInputs[ i], BitCount);

              FOutputs[ o_out] := SignalForOutputType( BinaryToSignal( R, BitCount), anOutType);
            end;
          end;

        m_xnor :

          begin
            if BitCount = 1
            then begin
              FOutputs[ o_out] :=
                SignalForOutputType(
                  LogicToSignal(
                    not
                    (
                      SignalToLogic( FInputs[ i_in1]) xor
                      SignalToLogic( FInputs[ i_in2]) xor
                      SignalToLogic( FInputs[ i_in3]) xor
                      SignalToLogic( FInputs[ i_in4])
                    )
                  )
                  , anOutType
                );
            end
            else begin
              R := 0;

              for i := i_in1 to i_in4
              do  R := R xor SignalToBinary( FInputs[ i], BitCount);

              FOutputs[ o_out] := SignalForOutputType( BinaryToSignal( not R, BitCount), anOutType);
            end;
          end;

        m_one :

          begin
            OneCounts := 0;

            for i := i_in1 to i_in4
            do begin
              if SignalToLogic( FInputs[ i])
              then Inc( OneCounts);
            end;

            FOutputs[ o_out] := SignalForOutputType( LogicToSignal( OneCounts = 1), anOutType);
          end;

        else // none, which is really is 'not one' .. the 'true' 'none' would be the same as 'nor'
          begin
            OneCounts := 0;

            for i := i_in1 to i_in4
            do begin
              if SignalToLogic( FInputs[ i])
              then Inc( OneCounts);
            end;

            FOutputs[ o_out] := SignalForOutputType( LogicToSignal( OneCounts <> 1), anOutType);
          end;
      end;
    end;


    procedure   TModGate.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out', FOutputs[ o_out]));
    end;


    procedure   TModGate.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    const
      Funcs : array[ 0 .. 7] of string = (
        'and'  ,
        'nand' ,
        'or'   ,
        'nor'  ,
        'xor'  ,
        'xnor' ,
        'one'  ,
        'none'
      );
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'func', Format( 'fun: %s', [ Funcs[ FFunction]], AppLocale)));
    end;


{ ========
  TModMultiGate = class( TMod)
  public
}

    procedure   TModMultiGate.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in11      , 'in11'      );
      AddInput ( i_in12      , 'in12'      );
      AddInput ( i_in21      , 'in21'      );
      AddInput ( i_in22      , 'in22'      );
      AddInput ( i_in31      , 'in31'      );
      AddInput ( i_in32      , 'in32'      );
      AddInput ( i_in41      , 'in41'      );
      AddInput ( i_in42      , 'in42'      );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_inputtype , 'inputtype' );
      AddInput ( i_func      , 'func      ');
      AddOutput( o_out1      , 'out1'      );
      AddOutput( o_out2      , 'out2'      );
      AddOutput( o_out3      , 'out3'      );
      AddOutput( o_out4      , 'out4'      );
    end;


    procedure   TModMultiGate.SetDefaults; // override;
    var
      i : Integer;
    begin
      case Round( FInputs[ i_func]) of

        m_and :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 1;
          end;

        m_nand :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 1;
          end;

        m_or :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_nor :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_xor :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_xnor :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        m_one :

          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 0;
          end;

        else // none
          begin
            for i := i_in11 to i_in42        // Make sure unused inputs are set to 1, so it'll 'work'
            do FInputs[ i] := 0;
          end;
      end;
    end;


    procedure   TModMultiGate.DoSlowTick; // override;
    var
      aFunction : Integer;
      anOutType : Integer;
      anInType  : Integer;
      In11      : Boolean;
      In12      : Boolean;
      In21      : Boolean;
      In22      : Boolean;
      In31      : Boolean;
      In32      : Boolean;
      In41      : Boolean;
      In42      : Boolean;
    begin
      aFunction := Round( FInputs[ i_func      ]);
      anOutType := Round( FInputs[ i_outputtype]);
      anInType  := Round( FInputs[ i_inputtype ]);

      if aFunction <> FFunction
      then begin
        FFunction := aFunction;
        SetDefaults;
      end;

      In11      := SignalToLogic( SignalForInputType( FInputs[ i_in11], anInType));
      In12      := SignalToLogic( SignalForInputType( FInputs[ i_in12], anInType));
      In21      := SignalToLogic( SignalForInputType( FInputs[ i_in21], anInType));
      In22      := SignalToLogic( SignalForInputType( FInputs[ i_in22], anInType));
      In31      := SignalToLogic( SignalForInputType( FInputs[ i_in31], anInType));
      In32      := SignalToLogic( SignalForInputType( FInputs[ i_in32], anInType));
      In41      := SignalToLogic( SignalForInputType( FInputs[ i_in41], anInType));
      In42      := SignalToLogic( SignalForInputType( FInputs[ i_in42], anInType));

      case aFunction of

        m_and :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( In11 and In12), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( In21 and In22), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( In31 and In32), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( In41 and In42), anOutType);
          end;

        m_nand :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( not( In11 and In12)), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( not( In21 and In22)), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( not( In31 and In32)), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( not( In41 and In42)), anOutType);
          end;

        m_or :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( In11 or In12), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( In21 or In22), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( In31 or In32), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( In41 or In42), anOutType);
          end;

        m_nor :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( not( In11 or In12)), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( not( In21 or In22)), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( not( In31 or In32)), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( not( In41 or In42)), anOutType);
          end;

        m_xor :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( In11 xor In12), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( In21 xor In22), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( In31 xor In32), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( In41 xor In42), anOutType);
          end;

        m_xnor :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( not( In11 xor In12)), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( not( In21 xor In22)), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( not( In31 xor In32)), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( not( In41 xor In42)), anOutType);
          end;

        m_one :

          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( In11 xor In12), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( In21 xor In22), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( In31 xor In32), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( In41 xor In42), anOutType);
          end;

        else // none, which is really is 'not one' .. (or just xnor for two inputs)
          begin
            FOutputs[ o_out1] := SignalForOutputType( LogicToSignal( not( In11 xor In12)), anOutType);
            FOutputs[ o_out2] := SignalForOutputType( LogicToSignal( not( In21 xor In22)), anOutType);
            FOutputs[ o_out3] := SignalForOutputType( LogicToSignal( not( In31 xor In32)), anOutType);
            FOutputs[ o_out4] := SignalForOutputType( LogicToSignal( not( In41 xor In42)), anOutType);
          end;
      end;
    end;


    procedure   TModMultiGate.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out1', FOutputs[ o_out1]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out2', FOutputs[ o_out2]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out3', FOutputs[ o_out3]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out4', FOutputs[ o_out4]));
    end;


{ ========
  TModDivider = class( TMod)
  private
    FPrevInput : Boolean;
    FCounter   : Integer;
    FOldCount  : Integer;
    FState     : Boolean;
    FOldReset  : Boolean;
    FOutValue  : TSignal;
    FActive    : Boolean;
  private
}

    procedure   TModDivider.PerformReset;
    var
      OutType : Integer;
    begin
      FCounter         := Max( 0, FOldCount - 1);
      FState           := True;
      FPrevInput       := True;
      FOutValue        := m_states[ FState];
      OutType          := Round( FInputs[ i_outputtype]);
      FOutputs[ o_out] := SignalForOutputType( FOutValue, OutType);
    end;


//  public

    procedure   TModDivider.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_count     , 'count'     );
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_reset     , 'reset'     );
      AddInput ( i_mode      , 'mode'      );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModDivider.DoSlowTick; // override;
    var
      Inp      : Boolean;
      Res      : Boolean;
      NewCount : Integer;
      OutType  : Integer;
    begin
      Res      := SignalToLogic( FInputs[ i_reset]);
      NewCount := Round        ( FInputs[ i_count]);
      Inp      := SignalToLogic( FInputs[ i_trig ]);
      OutType  := Round( FInputs[ i_outputtype]);

      if ResetFlag or ( Res and not FOldReset) or ( FOldCount <> NewCount)
      then begin
        FOldCount := NewCount;
        PerformReset
      end
      else begin
        if FPrevInput xor Inp
        then begin
          if FCounter <= 0
          then begin
            FCounter := Max( 0, FOldCount - 1);
            FState   := not FState;
            FActive  := Inp;
          end
          else begin
            Dec( FCounter);
            FActive := False;
          end;
        end;

        FPrevInput := Inp;
      end;

      case Round( FInputs[ i_mode ]) of
        1 : FOutValue := LogicToSignal( FState and FActive);  // pulse  mode
        2 : begin
            if FState
            then FOutValue := FInputs[ i_trig]                // signal mode
            else FOutValue := LogicToSignal( False);
          end;
        else FOutValue := LogicToSignal( FState);             // square mode
      end;

      FOutputs[ o_out] := SignalForOutputType( FOutValue, OutType);
      ResetFlag        := False;
      FOldReset        := Res;
    end;


    procedure   TModDivider.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out', FOutValue));
    end;


{ ========
  TModPulseDelay = class( TMod)
  private
    FPrevTrig  : Boolean;
    FPrevReset : Boolean;
    FOutValue  : Boolean;
    FDuration  : Int64;
    FNegCount  : Int64;
    FPosCount  : Int64;
    FPrevMode  : Integer;
    FActive    : Boolean;
    FInDelay   : Boolean;
    FNegTrig   : Boolean;
  public
}

    procedure   TModPulseDelay.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig       , 'trig'       );
      AddInput ( i_reset      , 'reset'      );
      AddInput ( i_delaymode  , 'delaymode'  );
      AddInput ( i_delaymod   , 'delaymod'   );
      AddInput ( i_delaymodamt, 'delaymodamt');
      AddInput ( i_range      , 'range'      );
      AddInput ( i_delay      , 'delay'      );
      AddInput ( i_outputtype , 'outputtype' );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModPulseDelay.DoSlowTick; // override;
    var
      Trig       : Boolean;
      Reset      : Boolean;
      Modulation : TSignal;
      Mode       : Integer;
    begin
      Mode       := Round( FInputs[ i_delaymode]);
      Trig       := SignalToLogic( FInputs[ i_trig ]);
      Reset      := SignalToLogic( FInputs[ i_reset]);
      Modulation := FInputs[ i_delaymod] * FInputs[ i_delaymodamt];
      FDuration  := Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_range]), FInputs[ i_delay] + Modulation)));

      if
        (( Reset xor FPrevReset) and Reset)                            or // Reset input got active
        ResetFlag                                                      or // Global reset flag
        (( FNegCount < 0) and not ( Mode in [ m_oscillate, m_oscdel])) or // Overflow on neg count
        (( FPosCount < 0) and not ( Mode in [ m_oscillate, m_oscdel])) or // Overflow on pos count
        ( Mode <> FPrevMode)                                              // Mode change
      then begin                                                          // are all causes for a reset
        FNegCount := 0;
        FPosCount := 0;
        FPrevTrig := False;
        FOutValue := False;
        FActive   := False;
        FInDelay  := False;
      end;

      if ( Trig xor FPrevTrig) and not ( Mode in [ m_oscillate, m_oscdel])
      then begin                                         // Trig input changed
        if Trig
        then begin                                       // Rising edge on trig input
          if not FActive
          then begin
            FActive   := True;
            FInDelay  := False;
            FPosCount := 0;
            FNegCount := 0;
            FNegTrig  := False;

            if Mode in [ m_stretch]
            then FOutValue := True;
          end;
        end
        else begin                                       // Falling edge on trig input
          if FActive
          then begin
            FNegCount := 0;
            FNegTrig  := True;
          end;
        end;
      end;

      case Mode of

        m_stretch :                                      // stretch mode

          begin
            if FActive and ( FPosCount > FDuration)
            then begin
              FOutValue := False;
              FActive   := False;
            end;
          end;

        m_oscillate :                                    // Gated oscillator mode

          begin
            if Trig
            then begin
              FActive := True;

              if FPosCount > FDuration
              then begin
                FOutValue := False;
                FNegCount :=  0;
                FPosCount := -1;
              end
              else if FNegCount > FDuration
              then begin
                FOutValue := True;
                FNegCount := -1;
                FPosCount :=  0;
              end;
            end
            else begin
              FOutValue := False;
              FActive   := False;
            end;
          end;

        m_oscdel :                                       // goschillator mode

          begin
            if Trig
            then begin
              FActive := True;

              if FPosCount > FDuration
              then begin
                FOutValue := False;
                FNegCount :=  0;
                FPosCount := -1;
              end
              else if FNegCount > FDuration
              then begin
                FOutValue := True;
                FNegCount := -1;
                FPosCount :=  0;
              end;
            end
            else begin
              FOutValue := False;
              FActive   := False;

              if Mode = m_oscdel
              then begin
                if FPosCount > - ( FDuration div 2)
                then Dec( FPosCount)
                else Inc( FPosCount);

                if FNegCount > - ( FDuration div 2)
                then Dec( FNegCount)
                else Inc( FNegCount);
              end;
            end;
          end;

        else  { m_delay }                                // delay   mode
          begin
            if FActive
            then begin
              if not FInDelay and ( FPosCount > FDuration)
              then begin
                FOutValue := True;
                FInDelay  := True;
              end
              else if FNegTrig and FInDelay and ( FNegCount > FDuration)
              then begin
                FOutValue := False;
                FActive   := False;
                FInDelay  := False;
                FNegTrig  := False;
              end;
            end;
          end;
      end;

      if FActive
      then begin
        {$Q-R-}
        Inc( FNegCount);
        Inc( FPosCount);
        {$Q+R+}
      end;

      FPrevTrig  := Trig;
      FPrevReset := Reset;
      FPrevMode  := Mode;
      ResetFlag  := False;

      FOutputs[ o_out] := SignalForOutputType( LogicToSignal( FOutValue), Round( FInputs[ i_outputtype]));
    end;


    procedure   TModPulseDelay.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'   , LogicToSignal( FOutValue)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active', LogicToSignal( FActive  )));
    end;


{ ========
  TModPulseSkip = class( TMod)
    o_out         = 0;
  private
    FPrevTrig  : Boolean;
    FPrevReset : Boolean;
    FOutActive : Boolean;
  public
}

    procedure   TModPulseSkip.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_reset     , 'reset'     );
      AddInput ( i_in        , 'in'        );
      AddInput ( i_inmod     , 'inmod'     );
      AddInput ( i_inmodamt  , 'inmodamt'  );
      AddInput ( i_cp        , 'cp'        );
      AddInput ( i_cpmod     , 'cpmod'     );
      AddInput ( i_cpmodamt  , 'cpmodamt'  );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModPulseSkip.DoSlowTick; // override;
    var
      NewTrig  : Boolean;
      NewReset : Boolean;
      InLevel  : TSignal;
      CpLevel  : TSignal;
    begin
      NewTrig   := SignalToLogic( FInputs[ i_trig ]);
      NewReset  := SignalToLogic( FInputs[ i_reset]);
      InLevel   := FInputs[ i_in] + FInputs[ i_inmod] * FInputs[ i_inmodamt];
      CpLevel   := FInputs[ i_cp] + FInputs[ i_cpmod] * FInputs[ i_cpmodamt];

      if ( NewReset and not FPrevReset) or ResetFlag
      then FOutActive := False;

      if NewTrig xor FPrevTrig
      then begin
        if NewTrig
        then begin
          if InLevel > CpLevel          // Skip pulses when InLevel <= CpLevel
          then FOutActive := True;
        end
        else FOutActive := False;
      end;

      ResetFlag        := False;
      FPrevTrig        := NewTrig;
      FPrevReset       := NewReset;
      FOutputs[ o_out] := SignalForOutputType( LogicToSignal( FOutActive), Round( FInputs[ i_outputtype]));
    end;


    procedure   TModPulseSkip.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'  , LogicToSignal( FOutActive)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig' , LogicToSignal( FPrevTrig )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset', LogicToSignal( FPrevReset)));
    end;


{ ========
  TModProgDivider = class( TMod)
  private
    FPrevInput : Boolean;
    FCounter   : Integer;
    FCount     : Integer;
    FState     : Boolean;
    FOldReset  : Boolean;
    FOutValue  : TSignal;
  private
}

    procedure   TModProgDivider.PerformReset;
    var
      OutType : Integer;
    begin
      FCounter         := Max( 0, FCount - 1);
      FState           := True;
      FPrevInput       := True;
      OutType          := Round( FInputs[ i_outputtype]);
      FOutValue        := m_states[ FState];
      FOutputs[ o_out] := SignalForOutputType( FOutValue, OutType);
    end;


//  public

    procedure   TModProgDivider.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_reset     , 'reset'     );
      AddInput ( i_select    , 'select'    );
      AddInput ( i_selectlev , 'selectlev' );
      AddInput ( i_steps     , 'steps'     );
      AddInput ( i_lowcount  , 'lowcount'  );
      AddInput ( i_highcount , 'highcount' );
      AddInput ( i_mode      , 'mode'      );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModProgDivider.DoSlowTick; // override;
    var
      Inp       : Boolean;
      Res       : Boolean;
      Steps     : Integer;
      LowCount  : Integer;
      HighCount : Integer;
      Selection : TSignal;
      OutType   : Integer;
    begin
      // What to do when highcount < lowcount ... ok then it is just treated as equal to lowcount
      Res       := SignalToLogic( FInputs[ i_reset    ]);
      LowCount  := Round        ( FInputs[ i_lowcount ]);
      HighCount := Round        ( FInputs[ i_highcount]);
      Inp       := SignalToLogic( FInputs[ i_trig     ]);
      Selection := FInputs[ i_select] * FInputs[ i_selectlev];
      Steps     := Max( 1, Round( FInputs[ i_steps ]));
      FCount    := (( Round(( LowCount + Max( 0, HighCount - LowCount)) * Selection) div Steps) + 1) * Steps;
      OutType   := Round( FInputs[ i_outputtype]);

      if ResetFlag or ( Res and not FOldReset)
      then PerformReset
      else begin
        if FPrevInput xor Inp
        then begin
          if FCounter <= 0
          then begin
            FCounter := Max( 0, FCount - 1);
            FState   := not FState;
          end
          else Dec( FCounter);
        end;

        FPrevInput := Inp;
      end;

      case Round( FInputs[ i_mode ]) of
        1 : FOutValue := LogicToSignal( FState and Inp);   // pulse  mode
        2 : begin
            if FState
            then FOutValue := FInputs[ i_trig]             // signal mode
            else FOutValue := LogicToSignal( False);
          end;
        else FOutValue := LogicToSignal( FState);          // square mode
      end;

      FOutputs[ o_out] := SignalForOutputType( FOutValue, OutType);
      ResetFlag        := False;
      FOldReset        := Res;
    end;


    procedure   TModProgDivider.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'  , FOutValue));
      aCallback( Self, MakeInfo( aPrefix, Name, 'count', FCount   ));
    end;


{ ========
  TModDFlipFlop = class( TMod)
  private
    FClockState : Boolean;
  public
}

    procedure   TModDFlipFlop.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_d         , 'd'         );
      AddInput ( i_clk       , 'clk'       );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_q         , 'q'         );
      AddOutput( o_notq      , 'notq'      );
    end;


    procedure   TModDFlipFlop.DoSlowTick; // override;
    var
      CLK   : Boolean;
      aType : Integer;
    begin
      aType := Round( FInputs[ i_outputtype]);

      if ResetFlag
      then begin
        ResetFlag := False;
        FQ        := False;
      end
      else begin
        CLK := SignalToLogic( FInputs[ i_clk]);

        if not FClockState and CLK
        then FQ := SignalToLogic( FInputs[ i_d]);

        FClockState := CLK;
      end;

      FOutputs[ o_q   ] := SignalForOutputType( LogicToSignal(     FQ), aType);
      FOutputs[ o_notq] := SignalForOutputType( LogicToSignal( not FQ), aType);
    end;


    procedure   TModDFlipFlop.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'q'   , LogicToSignal(     FQ)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'notq', LogicToSignal( not FQ)));
    end;


{ ========
  TModRSFlipFlop = class( TMod)
  private
    FSetState   : Boolean;
    FResetState : Boolean;
  public
}

    procedure   TModRSFlipFlop.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_s         , 's'         );
      AddInput ( i_r         , 'r'         );
      AddInput ( i_trigmode  , 'trigmode'  );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_q         , 'q'         );
      AddOutput( o_notq      , 'notq'      );
    end;


    procedure   TModRSFlipFlop.DoSlowTick; // override;
    var
      S     : Boolean;
      R     : Boolean;
      Mode  : Integer;
      aType : Integer;
    begin
      Mode  := Round( FInputs[ i_trigmode]);
      S     := SignalToLogic( FInputs [ i_s]);
      R     := SignalToLogic( FInputs [ i_r]) or ResetFlag;
      aType := Round( FInputs[ i_outputtype]);

      if      (( not FResetState) and R) or (( Mode = m_level) and R)
      then FQ := False
      else if (( not FSetState  ) and S) or (( Mode = m_level) and S)
      then FQ := True;

      FOutputs[ o_q   ] := SignalForOutputType( LogicToSignal(     FQ), aType);
      FOutputs[ o_notq] := SignalForOutputType( LogicToSignal( not FQ), aType);
      FSetState         := S;
      FResetState       := R;
      ResetFlag         := False;
    end;


    procedure   TModRSFlipFlop.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'q'   , LogicToSignal(     FQ)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'notq', LogicToSignal( not FQ)));
    end;


{ ========
  TModPulseSync = class( TMod)
  private
    FState      : Boolean;
    FOldSync    : Boolean;
    FQ          : Boolean;
    FCounter    : Integer;
    FLedCounter : Integer;
  public
}

    procedure   TModPulseSync.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_pulse     , 'pulse'     );
      AddInput ( i_sync      , 'sync'      );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_q         , 'q'         );
    end;


    procedure   TModPulseSync.DoSlowTick; // override;
    var
      Pulse   : Boolean;
      Sync    : Boolean;
      OutType : Integer;
    begin
      Pulse   := SignalToLogic( FInputs[ i_pulse]);
      Sync    := SignalToLogic( FInputs[ i_sync ]);
      OutType := Round( FInputs[ i_outputtype]);

      if FCounter > 0
      then begin
        Dec( FCounter);

        if FCounter <= 0
        then begin
          FState := False;
          FQ     := False;
        end;
      end;

      if not FOldPulse and Pulse
      then FState := True;

      if not FOldSync and Sync and FState
      then begin
        FQ := True;

        if IsSpedUp
        then begin
          FCounter    := Control_Decimation + 1;
          FLedCounter := LedFlashTime( IsSpedUp);
        end
        else begin
          FCounter    := 2;
          FLedCounter := LedFlashTime( IsSpedUp);
        end;
      end;

      DecToZero( FLedCounter);
      FOutputs[ o_q] := SignalForOutputType( LogicToSignal( FQ), OutType);
      FOldPulse      := Pulse;
      FOldSync       := Sync;
    end;


    procedure   TModPulseSync.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'state', LogicToSignal( FState         )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'q'    , LogicToSignal( FLedCounter > 0)));
    end;


{ ========
  TModCounter = class( TMod)
  private
    FState   : Word;
    FMode    : Integer;
    FOldRes  : Boolean;
    FOldTrig : Boolean;
  public
}

    procedure   TModCounter.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig , 'trig' );
      AddInput ( i_reset, 'reset');
      AddInput ( i_dir  , 'dir'  );
      AddInput ( i_mode , 'mode' );
      AddOutput( o_out0 , 'out0' );
      AddOutput( o_out1 , 'out1' );
      AddOutput( o_out2 , 'out2' );
      AddOutput( o_out3 , 'out3' );
      AddOutput( o_out4 , 'out4' );
      AddOutput( o_out5 , 'out5' );
      AddOutput( o_out6 , 'out6' );
      AddOutput( o_out7 , 'out7' );
      AddOutput( o_out8 , 'out8' );
      AddOutput( o_out9 , 'out9' );
      AddOutput( o_out10, 'out10');
      AddOutput( o_out11, 'out11');
      AddOutput( o_out12, 'out12');
      AddOutput( o_out13, 'out13');
      AddOutput( o_out14, 'out14');
      AddOutput( o_out15, 'out15');
    end;


    procedure   TModCounter.SetDefaults; // override;
    begin
      FInputs[ i_dir] := 1; // Default direction is up
    end;


    procedure   TModCounter.DoSlowTick; // override;
    const
      MaxCount = 16;
    var
      Direction : Boolean;
      Trig      : Boolean;
      Res       : Boolean;
      Mode      : Integer;
      i         : Integer;
      p         : Integer;
    begin
      Direction := SignalToLogic( FInputs[ i_dir  ]);
      Mode      := Round        ( FInputs[ i_mode ]);
      Trig      := SignalToLogic( FInputs[ i_trig ]);
      Res       := SignalToLogic( FInputs[ i_reset]);

      if (( not FOldRes) and Res) or ResetFlag or ( Mode <> FMode)
      then begin
        FMode    := Mode;
        FOldTrig := True;

        if Direction
        then FState := 0               // Forwards, reset to first state
        else begin
          if FMode = m_bin       // Backwards, reset to Prev state
          then FState := $ffff         // binary
          else FState := MaxCount - 1; // sequential (or ring)
        end;
      end
      else begin
        if Trig and not FOldTrig
        then begin
          if Direction
          then begin                     // Forwards, go to nexxt state
          {$Q-R-}
            Inc( FState);                // binary counter will oveflow properly
          {$Q+R+}
            if FMode = m_seq             // ring counter needs a check
            then FState := FState mod MaxCount;
          end
          else begin                     // Backwards, go to previous state
            if FState > 0
            then Dec( FState)
            else begin
              if FMode = m_bin
              then FState := $ffff
              else FState := MaxCount - 1;
            end;
          end;
        end;

        FOldTrig := Trig;
      end;

      FOldRes   := Res;
      ResetFlag := False;

      if FMode = m_bin
      then begin
        p := 1;

        for i := 0 to MaxCount - 1
        do begin
          if ( FState and p) <> 0
          then FOutputs[ o_out0 + i] := LogicToSignal( True )
          else FOutputs[ o_out0 + i] := LogicToSignal( False);

          p := p shl 1;
        end;
      end
      else begin
        for i := 0 to MaxCount - 1
        do begin
          if i = FState
          then FOutputs[ o_out0 + i] := LogicToSignal( True )
          else FOutputs[ o_out0 + i] := LogicToSignal( False);
        end;
      end;
    end;


    procedure   TModCounter.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := o_out0 to o_out15
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'out%d', [ 1 + i - o_out1], AppLocale), FOutputs[ i]));

      aCallback( Self, MakeInfo( aPrefix, Name, 'trig' , FInputs[ i_trig ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset', FInputs[ i_reset]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'  , FInputs[ i_dir  ]));
    end;


{ ========
  TModFixedDiv = class( TMod)
  private
    FOldRes   : Boolean;
    FOldTrig  : Boolean;
    FCount2   : Integer;
    FCount3   : Integer;
    FCount4   : Integer;
    FCount5   : Integer;
    FCount6   : Integer;
    FCount7   : Integer;
    FCount8   : Integer;
    FCount9   : Integer;
    FCount10  : Integer;
    FCount11  : Integer;
    FCount12  : Integer;
    FCount13  : Integer;
    FCount14  : Integer;
    FCount16  : Integer;
    FCount32  : Integer;
    FCount64  : Integer;
    FMemory2  : Boolean;
    FMemory3  : Boolean;
    FMemory4  : Boolean;
    FMemory5  : Boolean;
    FMemory6  : Boolean;
    FMemory7  : Boolean;
    FMemory8  : Boolean;
    FMemory9  : Boolean;
    FMemory10 : Boolean;
    FMemory11 : Boolean;
    FMemory12 : Boolean;
    FMemory13 : Boolean;
    FMemory14 : Boolean;
    FMemory16 : Boolean;
    FMemory32 : Boolean;
    FMemory64 : Boolean;
  public
}

    procedure   TModFixedDiv.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig  , 'trig'  );
      AddInput ( i_res   , 'res'   );
      AddInput ( i_square, 'square');
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_out3  , 'out3'  );
      AddOutput( o_out4  , 'out4'  );
      AddOutput( o_out5  , 'out5'  );
      AddOutput( o_out6  , 'out6'  );
      AddOutput( o_out7  , 'out7'  );
      AddOutput( o_out8  , 'out8'  );
      AddOutput( o_out9  , 'out9'  );
      AddOutput( o_out10 , 'out10' );
      AddOutput( o_out11 , 'out11' );
      AddOutput( o_out12 , 'out12' );
      AddOutput( o_out13 , 'out13' );
      AddOutput( o_out14 , 'out14' );
      AddOutput( o_out16 , 'out16' );
      AddOutput( o_out32 , 'out32' );
      AddOutput( o_out64 , 'out64' );
    end;


    procedure   TModFixedDiv.DoSlowTick; // override;
    var
      Trig : Boolean;
      Res  : Boolean;
      Pls  : Boolean;
    begin
      Trig := SignalToLogic( FInputs[ i_trig]);
      Res  := SignalToLogic( FInputs[ i_res ]);
      Pls  := SignalToLogic( FInputs[ i_square]) or Trig;

      if ( not FOldRes and Res) or ResetFlag
      then begin
        ResetFlag := False;
        FCounter  := 0;
      end;

      if not FOldTrig and Trig
      then {$Q-R-}Inc( FCounter);{$Q+R+}

      FOutputs[ o_out2 ] := LogicToSignal(( FCounter mod  2 = 0) and Pls);
      FOutputs[ o_out3 ] := LogicToSignal(( FCounter mod  3 = 0) and Pls);
      FOutputs[ o_out4 ] := LogicToSignal(( FCounter mod  4 = 0) and Pls);
      FOutputs[ o_out5 ] := LogicToSignal(( FCounter mod  5 = 0) and Pls);
      FOutputs[ o_out6 ] := LogicToSignal(( FCounter mod  6 = 0) and Pls);
      FOutputs[ o_out7 ] := LogicToSignal(( FCounter mod  7 = 0) and Pls);
      FOutputs[ o_out8 ] := LogicToSignal(( FCounter mod  8 = 0) and Pls);
      FOutputs[ o_out9 ] := LogicToSignal(( FCounter mod  9 = 0) and Pls);
      FOutputs[ o_out10] := LogicToSignal(( FCounter mod 10 = 0) and Pls);
      FOutputs[ o_out11] := LogicToSignal(( FCounter mod 11 = 0) and Pls);
      FOutputs[ o_out12] := LogicToSignal(( FCounter mod 12 = 0) and Pls);
      FOutputs[ o_out13] := LogicToSignal(( FCounter mod 13 = 0) and Pls);
      FOutputs[ o_out14] := LogicToSignal(( FCounter mod 14 = 0) and Pls);
      FOutputs[ o_out16] := LogicToSignal(( FCounter mod 16 = 0) and Pls);
      FOutputs[ o_out32] := LogicToSignal(( FCounter mod 32 = 0) and Pls);
      FOutputs[ o_out64] := LogicToSignal(( FCounter mod 64 = 0) and Pls);

      FOldRes  := Res;
      FOldTrig := Trig;
    end;


    procedure   TModFixedDiv.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out2' , FOutputs[ o_out2 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out3' , FOutputs[ o_out3 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out4' , FOutputs[ o_out4 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out5' , FOutputs[ o_out5 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out6' , FOutputs[ o_out6 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out7' , FOutputs[ o_out7 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out8' , FOutputs[ o_out8 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out9' , FOutputs[ o_out9 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out10', FOutputs[ o_out10]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out11', FOutputs[ o_out11]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out12', FOutputs[ o_out12]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out13', FOutputs[ o_out13]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out14', FOutputs[ o_out14]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out16', FOutputs[ o_out16]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out32', FOutputs[ o_out32]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out64', FOutputs[ o_out64]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig' , FInputs[ i_trig  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'  , FInputs[ i_res   ]));
    end;


{ ========
  TModPrimeDiv = class( TMod)
  private
    FOldRes   : Boolean;
    FOldTrig  : Boolean;
    FCount2   : Integer;
    FCount3   : Integer;
    FCount5   : Integer;
    FCount7   : Integer;
    FCount11  : Integer;
    FCount13  : Integer;
    FCount17  : Integer;
    FCount19  : Integer;
    FCount23  : Integer;
    FCount29  : Integer;
    FCount31  : Integer;
    FCount37  : Integer;
    FCount41  : Integer;
    FCount43  : Integer;
    FCount47  : Integer;
    FCount53  : Integer;
    FMemory2  : Boolean;
    FMemory3  : Boolean;
    FMemory5  : Boolean;
    FMemory7  : Boolean;
    FMemory11 : Boolean;
    FMemory13 : Boolean;
    FMemory17 : Boolean;
    FMemory19 : Boolean;
    FMemory23 : Boolean;
    FMemory29 : Boolean;
    FMemory31 : Boolean;
    FMemory37 : Boolean;
    FMemory41 : Boolean;
    FMemory43 : Boolean;
    FMemory47 : Boolean;
    FMemory53 : Boolean;
  public
}

    procedure   TModPrimeDiv.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig  , 'trig'  );
      AddInput ( i_res   , 'res'   );
      AddInput ( i_square, 'square');
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_out3  , 'out3'  );
      AddOutput( o_out5  , 'out5'  );
      AddOutput( o_out7  , 'out7'  );
      AddOutput( o_out11 , 'out11' );
      AddOutput( o_out13 , 'out13' );
      AddOutput( o_out17 , 'out17' );
      AddOutput( o_out19 , 'out19' );
      AddOutput( o_out23 , 'out23' );
      AddOutput( o_out29 , 'out29' );
      AddOutput( o_out31 , 'out31' );
      AddOutput( o_out37 , 'out37' );
      AddOutput( o_out41 , 'out41' );
      AddOutput( o_out43 , 'out43' );
      AddOutput( o_out47 , 'out47' );
      AddOutput( o_out53 , 'out53' );
    end;


    procedure   TModPrimeDiv.DoSlowTick; // override;
    var
      Trig : Boolean;
      Res  : Boolean;
      Pls  : Boolean;
    begin
      Trig := SignalToLogic( FInputs[ i_trig]);
      Res  := SignalToLogic( FInputs[ i_res ]);
      Pls  := SignalToLogic( FInputs[ i_square]) or Trig;

      if ( not FOldRes and Res) or ResetFlag
      then begin
        ResetFlag := False;
        FCounter  := -1;
      end;

      if not FOldTrig and Trig
      then {$Q-R-}Inc( FCounter);{$Q+R+}

      FOutputs[ o_out2 ] := LogicToSignal(( FCounter mod  2 = 0) and Pls);
      FOutputs[ o_out3 ] := LogicToSignal(( FCounter mod  3 = 0) and Pls);
      FOutputs[ o_out5 ] := LogicToSignal(( FCounter mod  5 = 0) and Pls);
      FOutputs[ o_out7 ] := LogicToSignal(( FCounter mod  7 = 0) and Pls);
      FOutputs[ o_out11] := LogicToSignal(( FCounter mod 11 = 0) and Pls);
      FOutputs[ o_out13] := LogicToSignal(( FCounter mod 13 = 0) and Pls);
      FOutputs[ o_out17] := LogicToSignal(( FCounter mod 17 = 0) and Pls);
      FOutputs[ o_out19] := LogicToSignal(( FCounter mod 19 = 0) and Pls);
      FOutputs[ o_out23] := LogicToSignal(( FCounter mod 23 = 0) and Pls);
      FOutputs[ o_out29] := LogicToSignal(( FCounter mod 29 = 0) and Pls);
      FOutputs[ o_out31] := LogicToSignal(( FCounter mod 31 = 0) and Pls);
      FOutputs[ o_out37] := LogicToSignal(( FCounter mod 37 = 0) and Pls);
      FOutputs[ o_out41] := LogicToSignal(( FCounter mod 41 = 0) and Pls);
      FOutputs[ o_out43] := LogicToSignal(( FCounter mod 43 = 0) and Pls);
      FOutputs[ o_out47] := LogicToSignal(( FCounter mod 47 = 0) and Pls);
      FOutputs[ o_out53] := LogicToSignal(( FCounter mod 53 = 0) and Pls);

      FOldRes  := Res;
      FOldTrig := Trig;
    end;


    procedure   TModPrimeDiv.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out2' , FOutputs[ o_out2 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out3' , FOutputs[ o_out3 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out5' , FOutputs[ o_out5 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out7' , FOutputs[ o_out7 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out11', FOutputs[ o_out11]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out13', FOutputs[ o_out13]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out17', FOutputs[ o_out17]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out19', FOutputs[ o_out19]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out23', FOutputs[ o_out23]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out29', FOutputs[ o_out29]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out31', FOutputs[ o_out31]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out37', FOutputs[ o_out37]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out41', FOutputs[ o_out41]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out43', FOutputs[ o_out43]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out47', FOutputs[ o_out47]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out53', FOutputs[ o_out53]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig' , FInputs[ i_trig  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'  , FInputs[ i_res   ]));
    end;


{ ========
  TModAmuse = class( TMod)
  private
    FOldRes    : Boolean;
    FResetReq  : Boolean;
    FOldTrig   : Boolean;
    FShifter   : Cardinal;
    FCountC1   : Integer;
    FCountC2   : Integer;
    FCountC3   : Integer;
    FCountC4   : Integer;
    FCountC5   : Integer;
    FCountC6   : Integer;
    FCountC7   : Integer;
    FCountC8   : Integer;
    FMemoryC1  : Boolean;
    FMemoryC2  : Boolean;
    FMemoryC3  : Boolean;
    FMemoryC4  : Boolean;
    FMemoryC5  : Boolean;
    FMemoryC6  : Boolean;
    FMemoryC7  : Boolean;
    FMemoryC8  : Boolean;
    FSwa       : Boolean;
    FSwb       : Boolean;
    FSwc       : Boolean;
    FSwd       : Boolean;
    FSwe       : Boolean;
    FExt1      : Boolean;
    FExt2      : Boolean;
    FCarry0    : Boolean;
    FCarry31   : Boolean;
    FLights    : string;
  public
}

    procedure   TModAmuse.CreateIO; // override;
    begin
      FIsSlow := True;
      SetLength( FLights, 32);
      AddInput ( i_reset     , 'reset'     );
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_sela      , 'sela'      );
      AddInput ( i_selb      , 'selb'      );
      AddInput ( i_selc      , 'selc'      );
      AddInput ( i_seld      , 'seld'      );
      AddInput ( i_sele      , 'sele'      );
      AddInput ( i_selw      , 'selw'      );
      AddInput ( i_selx      , 'selx'      );
      AddInput ( i_sely      , 'sely'      );
      AddInput ( i_selz      , 'selz'      );
      AddInput ( i_selmoda   , 'selmoda'   );
      AddInput ( i_selmodb   , 'selmodb'   );
      AddInput ( i_selmodc   , 'selmodc'   );
      AddInput ( i_selmodd   , 'selmodd'   );
      AddInput ( i_selmode   , 'selmode'   );
      AddInput ( i_selmodw   , 'selmodw'   );
      AddInput ( i_selmodx   , 'selmodx'   );
      AddInput ( i_selmody   , 'selmody'   );
      AddInput ( i_selmodz   , 'selmodz'   );
      AddInput ( i_selmodamta, 'selmodamta');
      AddInput ( i_selmodamtb, 'selmodamtb');
      AddInput ( i_selmodamtc, 'selmodamtc');
      AddInput ( i_selmodamtd, 'selmodamtd');
      AddInput ( i_selmodamte, 'selmodamte');
      AddInput ( i_selmodamtw, 'selmodamtw');
      AddInput ( i_selmodamtx, 'selmodamtx');
      AddInput ( i_selmodamty, 'selmodamty');
      AddInput ( i_selmodamtz, 'selmodamtz');
      AddInput ( i_clocked   , 'clocked'   );
      AddInput ( i_ext1      , 'ext1'      );
      AddInput ( i_ext2      , 'ext2'      );
      AddInput ( i_reverse   , 'reverse'   );
      AddInput ( i_evenodd   , 'evenodd'   );
      AddInput ( i_paritymod , 'paritymod' );
      AddInput ( i_gatemode  , 'gatemode'  );
      AddOutput( o_out       , 'out'       );
      AddOutput( o_inv       , 'inv'       );
      AddOutput( o_bit       , 'bit'       );
      AddOutput( o_data      , 'data'      );
    end;


    procedure   TModAmuse.DoSlowTick; // override;

      function Select( aPos: Integer): Boolean;
      begin
        case aPos of
           1 : Result := True;
           2 : Result := FOldTrig;
           3 : Result := FMemoryC1;
           4 : Result := FMemoryC2;
           5 : Result := FMemoryC3;
           6 : Result := FMemoryC4;
           7 : Result := FMemoryC5;
           8 : Result := FMemoryC6;
           9 : Result := FMemoryC7;
          10 : Result := FMemoryC8;
          11 : Result := ( FShifter and ( 1 shl  0)) <> 0;
          12 : Result := ( FShifter and ( 1 shl  1)) <> 0;
          13 : Result := ( FShifter and ( 1 shl  2)) <> 0;
          14 : Result := ( FShifter and ( 1 shl  3)) <> 0;
          15 : Result := ( FShifter and ( 1 shl  4)) <> 0;
          16 : Result := ( FShifter and ( 1 shl  5)) <> 0;
          17 : Result := ( FShifter and ( 1 shl  6)) <> 0;
          18 : Result := ( FShifter and ( 1 shl  7)) <> 0;
          19 : Result := ( FShifter and ( 1 shl  8)) <> 0;
          20 : Result := ( FShifter and ( 1 shl  9)) <> 0;
          21 : Result := ( FShifter and ( 1 shl 10)) <> 0;
          22 : Result := ( FShifter and ( 1 shl 11)) <> 0;
          23 : Result := ( FShifter and ( 1 shl 12)) <> 0;
          24 : Result := ( FShifter and ( 1 shl 13)) <> 0;
          25 : Result := ( FShifter and ( 1 shl 14)) <> 0;
          26 : Result := ( FShifter and ( 1 shl 15)) <> 0;
          27 : Result := ( FShifter and ( 1 shl 16)) <> 0;
          28 : Result := ( FShifter and ( 1 shl 17)) <> 0;
          29 : Result := ( FShifter and ( 1 shl 18)) <> 0;
          30 : Result := ( FShifter and ( 1 shl 19)) <> 0;
          31 : Result := ( FShifter and ( 1 shl 20)) <> 0;
          32 : Result := ( FShifter and ( 1 shl 21)) <> 0;
          33 : Result := ( FShifter and ( 1 shl 22)) <> 0;
          34 : Result := ( FShifter and ( 1 shl 23)) <> 0;
          35 : Result := ( FShifter and ( 1 shl 24)) <> 0;
          36 : Result := ( FShifter and ( 1 shl 25)) <> 0;
          37 : Result := ( FShifter and ( 1 shl 26)) <> 0;
          38 : Result := ( FShifter and ( 1 shl 27)) <> 0;
          39 : Result := ( FShifter and ( 1 shl 28)) <> 0;
          40 : Result := ( FShifter and ( 1 shl 29)) <> 0;
          41 : Result := ( FShifter and ( 1 shl 30)) <> 0;
          42 : Result := FExt1;
          43 : Result := FExt2;
          else Result := False;
        end;
      end;

    var
      Data     : Boolean;
      Trig     : Boolean;
      Reset    : Boolean;
      Clocked  : Boolean;
      Reversed : Boolean;
      GateMode : Boolean;
    begin
      Reset    := SignalToLogic( FInputs[ i_reset   ]);
      Trig     := SignalToLogic( FInputs[ i_trig    ]);
      Clocked  := SignalToLogic( FInputs[ i_clocked ]);
      FExt1    := SignalToLogic( FInputs[ i_ext1    ]);
      FExt2    := SignalToLogic( FInputs[ i_ext2    ]);
      Reversed := SignalToLogic( FInputs[ i_reverse ]);
      GateMode := SignalToLogic( FInputs[ i_gatemode]);

      if   ( Reset and not FOldRes) or ResetFlag
      then begin
        ResetFlag := False;
        FResetReq := True;
      end;

      if   Trig xor FOldTrig
      then begin
        if   Trig
        then begin
          Data  :=
            Select( Round( FInputs[ i_selmodw] * FInputs[ i_selmodamtw] + FInputs[ i_selw])) xor
            Select( Round( FInputs[ i_selmodx] * FInputs[ i_selmodamtx] + FInputs[ i_selx])) xor
            Select( Round( FInputs[ i_selmody] * FInputs[ i_selmodamty] + FInputs[ i_sely])) xor
            Select( Round( FInputs[ i_selmodz] * FInputs[ i_selmodamtz] + FInputs[ i_selz])) xor
            SignalToLogic( FInputs[ i_evenodd  ])                                            xor
            SignalToLogic( FInputs[ i_paritymod]);

          DecToZeroWithResetValueToggle( FCountC1, FMemoryC1, 1);
          DecToZeroWithResetValueToggle( FCountC2, FMemoryC2, 2);
          DecToZeroWithResetValueToggle( FCountC3, FMemoryC3, 3);
          DecToZeroWithResetValueToggle( FCountC4, FMemoryC4, 4);
          DecToZeroWithResetValueToggle( FCountC5, FMemoryC5, 5);
          DecToZeroWithResetValueToggle( FCountC6, FMemoryC6, 6);
          DecToZeroWithResetValueToggle( FCountC7, FMemoryC7, 7);
          DecToZeroWithResetValueToggle( FCountC8, FMemoryC8, 8);

          if   Reversed
          then FShifter := ( FShifter shr 1) or ( Cardinal( Ord( Data) shl 31))
          else FShifter := ( FShifter shl 1) or   Cardinal( Ord( Data));
        end
        else begin
          if   Reversed
          then begin
            FCarry0  := ( FShifter and ( 1 shl 31)) <> 0;
            FCarry31 := ( FShifter and ( 1 shl  0)) <> 0;
          end
          else begin
            FCarry0  := ( FShifter and ( 1 shl  0)) <> 0;
            FCarry31 := ( FShifter and ( 1 shl 31)) <> 0;
          end;
        end;
      end;

      if   ( not Clocked) or ( Clocked and ( Trig and not FOldTrig))
      then begin
        if FResetReq
        then begin
          FResetReq := False;
          FCountC1  := 1; FMemoryC1 := False;
          FCountC2  := 2; FMemoryC2 := False;
          FCountC3  := 3; FMemoryC3 := False;
          FCountC4  := 4; FMemoryC4 := False;
          FCountC5  := 5; FMemoryC5 := False;
          FCountC6  := 6; FMemoryC6 := False;
          FCountC7  := 5; FMemoryC7 := False;
          FCountC8  := 8; FMemoryC8 := False;
          FShifter  := 0;
        end;

        FSwa := Select( Round( FInputs[ i_selmoda] * FInputs[ i_selmodamta] + FInputs[ i_sela]));
        FSwb := Select( Round( FInputs[ i_selmodb] * FInputs[ i_selmodamtb] + FInputs[ i_selb]));
        FSwc := Select( Round( FInputs[ i_selmodc] * FInputs[ i_selmodamtc] + FInputs[ i_selc]));
        FSwd := Select( Round( FInputs[ i_selmodd] * FInputs[ i_selmodamtd] + FInputs[ i_seld]));
        FSwe := Select( Round( FInputs[ i_selmode] * FInputs[ i_selmodamte] + FInputs[ i_sele]));
      end;

      FOutputs[ o_out ] := ( Ord( FSwa) + 2 * Ord( FSwb) + 4 * Ord( FSwc) + 8 * Ord( FSwd) + 16 * Ord( FSwe)) / 32.0;
      FOutputs[ o_inv ] := 1.0 - FOutputs[ o_out];
      FOutputs[ o_bit ] := LogicToSignal( FCarry31 and ( Trig or GateMode));
      FOutputs[ o_data] := LogicToSignal( FCarry0  and ( Trig or GateMode));

      FOldRes  := Reset;
      FOldTrig := Trig;
    end;


    procedure   TModAmuse.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      i : Integer;
      M : Cardinal;
    begin
      M := 1;

      for i := 0 to 31
      do begin
        if ( FShifter and M) = 0
        then FLights[ High( FLights) - i] := ' '
        else FLights[ High( FLights) - i] := 'x';

        M := M shl 1;
      end;

      aCallback( Self, MakeInfo( aPrefix, Name, 'lights', FLights));
    end;


{ ========
  TModChladniControl = class( TMod)
  private
    FOldC : TSignal;
    FOldP : TSignal;
    FOldN : Integer;
    FOldM : Integer;
  public
}

    procedure   TModChladniControl.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_c      , 'c'      );
      AddInput ( i_cmod   , 'cmod'   );
      AddInput ( i_cmodamt, 'cmodamt');
      AddInput ( i_p      , 'p'      );
      AddInput ( i_pmod   , 'pmod'   );
      AddInput ( i_pmodamt, 'pmodamt');
      AddInput ( i_n      , 'n'      );
      AddInput ( i_nmod   , 'nmod'   );
      AddInput ( i_nmodamt, 'nmodamt');
      AddInput ( i_m      , 'm'      );
      AddInput ( i_mmod   , 'mmod'   );
      AddInput ( i_mmodamt, 'mmodamt');
      AddOutput( o_out    , 'out'    );
    end;


    procedure   TModChladniControl.DoSlowTick; // override;
    var
      c : TSignal;
      p : TSignal;
      n : Integer;
      m : Integer;
    begin
      c :=        Abs( Normalize( FInputs[ i_c] +      FInputs[ i_cmod] * FInputs[ i_cmodamt])) ;
      p :=        Abs( Normalize( FInputs[ i_p] +      FInputs[ i_pmod] * FInputs[ i_pmodamt])) ;
      n := Round( Abs( Normalize( FInputs[ i_n] + 16 * FInputs[ i_nmod] * FInputs[ i_nmodamt])));
      m := Round( Abs( Normalize( FInputs[ i_m] + 16 * FInputs[ i_mmod] * FInputs[ i_mmodamt])));

      if  ( c <> FOldC)
      or  ( p <> FOldP)
      or  ( n <> FOldN)
      or  ( m <> FOldM)
      then FOutputs[ o_out] := FrequencyToUnits( c * Power( m + 2 * n, p));
    end;


{ ========
  TModShifter = class( TMod)
  private
    FOldRes    : Boolean;
    FResetReq  : Boolean;
    FOldTrig   : Boolean;
    FShifter   : Cardinal;
    FLights    : string;
  public
}

    procedure   TModShifter.CreateIO; // override;
    begin
      FIsSlow := True;
      SetLength( FLights, 32);
      AddInput ( i_reset    , 'reset'    );
      AddInput ( i_trig     , 'trig'     );
      AddInput ( i_extlow   , 'extlow'   );
      AddInput ( i_exthigh  , 'exthigh'  );
      AddInput ( i_reverse  , 'reverse'  );
      AddInput ( i_gatemode , 'gatemode' );
      AddInput ( i_clocked  , 'clocked'  );
      AddInput ( i_in       , 'in'       );
      AddOutput( o_out0     , 'out0'     );
      AddOutput( o_out1     , 'out1'     );
      AddOutput( o_out2     , 'out2'     );
      AddOutput( o_out3     , 'out3'     );
      AddOutput( o_out4     , 'out4'     );
      AddOutput( o_out5     , 'out5'     );
      AddOutput( o_out6     , 'out6'     );
      AddOutput( o_out7     , 'out7'     );
      AddOutput( o_out8     , 'out8'     );
      AddOutput( o_out9     , 'out9'     );
      AddOutput( o_out10    , 'out10'    );
      AddOutput( o_out11    , 'out11'    );
      AddOutput( o_out12    , 'out12'    );
      AddOutput( o_out13    , 'out13'    );
      AddOutput( o_out14    , 'out14'    );
      AddOutput( o_out15    , 'out15'    );
      AddOutput( o_out16    , 'out16'    );
      AddOutput( o_out17    , 'out17'    );
      AddOutput( o_out18    , 'out18'    );
      AddOutput( o_out19    , 'out19'    );
      AddOutput( o_out20    , 'out20'    );
      AddOutput( o_out21    , 'out21'    );
      AddOutput( o_out22    , 'out22'    );
      AddOutput( o_out23    , 'out23'    );
      AddOutput( o_out24    , 'out24'    );
      AddOutput( o_out25    , 'out25'    );
      AddOutput( o_out26    , 'out26'    );
      AddOutput( o_out27    , 'out27'    );
      AddOutput( o_out28    , 'out28'    );
      AddOutput( o_out29    , 'out29'    );
      AddOutput( o_out30    , 'out30'    );
      AddOutput( o_out31    , 'out31'    );
      AddOutput( o_carrylow , 'carrylow' );
      AddOutput( o_carryhigh, 'carryhigh');
    end;


    procedure   TModShifter.DoSlowTick; // override;
    var
      Trig     : Boolean;
      Reset    : Boolean;
      Clocked  : Boolean;
      Reversed : Boolean;
      GateMode : Boolean;
      i        : Integer;
      M        : Cardinal;
      Act      : Boolean;
    begin
      Reset    := SignalToLogic( FInputs[ i_reset   ]);
      Trig     := SignalToLogic( FInputs[ i_trig    ]);
      Clocked  := SignalToLogic( FInputs[ i_clocked ]);
      Reversed := SignalToLogic( FInputs[ i_reverse ]);
      GateMode := SignalToLogic( FInputs[ i_gatemode]);

      if   ( Reset and not FOldRes) or ResetFlag
      then begin
        ResetFlag := False;
        FResetReq := True;
      end;

      if   Trig xor FOldTrig
      then begin
        if   Trig
        then begin
          if   Reversed
          then FShifter := ( FShifter shr 1) or ( Cardinal(( SignalToLogic( FInputs[ i_in]) xor SignalToLogic( FInputs[ i_exthigh]))) shl 31)
          else FShifter := ( FShifter shl 1) or   Cardinal(( SignalToLogic( FInputs[ i_in]) xor SignalToLogic( FInputs[ i_extlow ]))        );
        end
        else begin
          if   Reversed
          then begin
            FCOutHi := ( FShifter and ( 1 shl  0)) <> 0;
            FCOutLo := ( FShifter and ( 1 shl 31)) <> 0;
          end
          else begin
            FCOutHi := ( FShifter and ( 1 shl 31)) <> 0;
            FCOutLo := ( FShifter and ( 1 shl  0)) <> 0;
          end;
        end;
      end;

      if   ( not Clocked) or ( Clocked and ( Trig and not FOldTrig))
      then begin
        if FResetReq
        then begin
          FResetReq := False;
          FShifter  := 0;
        end;
      end;

      FOutputs[ o_carryhigh] := LogicToSignal( FCOutHi);
      FOutputs[ o_carrylow ] := LogicToSignal( FCOutLo);
      M   := 1;
      Act := Trig or GateMode;

      for i := 0 to 31
      do begin
        FOutputs[ o_out0 + i] := LogicToSignal((( FShifter and M) <> 0) and Act);
        M := M shl 1;
      end;

      FOldRes  := Reset;
      FOldTrig := Trig;
    end;


    procedure   TModShifter.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      i : Integer;
      M : Cardinal;
    begin
      M := 1;

      for i := 0 to 31
      do begin
        if ( FShifter and M) = 0
        then FLights[ 1 + i] := ' '
        else FLights[ 1 + i] := 'x';

        M := M shl 1;
      end;

      aCallback( Self, MakeInfo( aPrefix, Name, 'lights', FLights));
    end;


    procedure   TModShifter.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'    , FInputs [ i_reset    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'     , FInputs [ i_trig     ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'extlow'   , FInputs [ i_extlow   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'exthigh'  , FInputs [ i_exthigh  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reverse'  , FInputs [ i_reverse  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'in'       , FInputs [ i_in       ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'carrylow' , FOutputs[ o_carrylow ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'carryhigh', FOutputs[ o_carryhigh]));
    end;


{ ========
  TModLFSR = class( TMod)
  private
    FOldRes   : Boolean;
    FResetReq : Boolean;
    FOldTrig  : Boolean;
    FShifter  : Cardinal;
    FFeedback : Boolean;
    FLights   : string;
  public
}

    procedure   TModLFSR.CreateIO; // override;
    begin
      FIsSlow := True;
      SetLength( FLights, 32);
      AddInput ( i_reset    , 'reset'    );
      AddInput ( i_trig     , 'trig'     );
      AddInput ( i_reverse  , 'reverse'  );
      AddInput ( i_gatemode , 'gatemode' );
      AddInput ( i_clocked  , 'clocked'  );
      AddInput ( i_in       , 'in'       );
      AddInput ( i_evenodd  , 'evenodd'  );
      AddInput ( i_paritymod, 'paritymod');
      AddInput ( i_tap0     , 'tap0'     );
      AddInput ( i_tap1     , 'tap1'     );
      AddInput ( i_tap2     , 'tap2'     );
      AddInput ( i_tap3     , 'tap3'     );
      AddInput ( i_tap4     , 'tap4'     );
      AddInput ( i_tap5     , 'tap5'     );
      AddInput ( i_tap6     , 'tap6'     );
      AddInput ( i_tap7     , 'tap7'     );
      AddInput ( i_tap8     , 'tap8'     );
      AddInput ( i_tap9     , 'tap9'     );
      AddInput ( i_tap10    , 'tap10'    );
      AddInput ( i_tap11    , 'tap11'    );
      AddInput ( i_tap12    , 'tap12'    );
      AddInput ( i_tap13    , 'tap13'    );
      AddInput ( i_tap14    , 'tap14'    );
      AddInput ( i_tap15    , 'tap15'    );
      AddInput ( i_tap16    , 'tap16'    );
      AddInput ( i_tap17    , 'tap17'    );
      AddInput ( i_tap18    , 'tap18'    );
      AddInput ( i_tap19    , 'tap19'    );
      AddInput ( i_tap20    , 'tap20'    );
      AddInput ( i_tap21    , 'tap21'    );
      AddInput ( i_tap22    , 'tap22'    );
      AddInput ( i_tap23    , 'tap23'    );
      AddInput ( i_tap24    , 'tap24'    );
      AddInput ( i_tap25    , 'tap25'    );
      AddInput ( i_tap26    , 'tap26'    );
      AddInput ( i_tap27    , 'tap27'    );
      AddInput ( i_tap28    , 'tap28'    );
      AddInput ( i_tap29    , 'tap29'    );
      AddInput ( i_tap30    , 'tap30'    );
      AddInput ( i_tap31    , 'tap31'    );
      AddInput ( i_tapmod0  , 'tapmod0'  );
      AddInput ( i_tapmod1  , 'tapmod1'  );
      AddInput ( i_tapmod2  , 'tapmod2'  );
      AddInput ( i_tapmod3  , 'tapmod3'  );
      AddInput ( i_tapmod4  , 'tapmod4'  );
      AddInput ( i_tapmod5  , 'tapmod5'  );
      AddInput ( i_tapmod6  , 'tapmod6'  );
      AddInput ( i_tapmod7  , 'tapmod7'  );
      AddInput ( i_tapmod8  , 'tapmod8'  );
      AddInput ( i_tapmod9  , 'tapmod9'  );
      AddInput ( i_tapmod10 , 'tapmod10' );
      AddInput ( i_tapmod11 , 'tapmod11' );
      AddInput ( i_tapmod12 , 'tapmod12' );
      AddInput ( i_tapmod13 , 'tapmod13' );
      AddInput ( i_tapmod14 , 'tapmod14' );
      AddInput ( i_tapmod15 , 'tapmod15' );
      AddInput ( i_tapmod16 , 'tapmod16' );
      AddInput ( i_tapmod17 , 'tapmod17' );
      AddInput ( i_tapmod18 , 'tapmod18' );
      AddInput ( i_tapmod19 , 'tapmod19' );
      AddInput ( i_tapmod20 , 'tapmod20' );
      AddInput ( i_tapmod21 , 'tapmod21' );
      AddInput ( i_tapmod22 , 'tapmod22' );
      AddInput ( i_tapmod23 , 'tapmod23' );
      AddInput ( i_tapmod24 , 'tapmod24' );
      AddInput ( i_tapmod25 , 'tapmod25' );
      AddInput ( i_tapmod26 , 'tapmod26' );
      AddInput ( i_tapmod27 , 'tapmod27' );
      AddInput ( i_tapmod28 , 'tapmod28' );
      AddInput ( i_tapmod29 , 'tapmod29' );
      AddInput ( i_tapmod30 , 'tapmod30' );
      AddInput ( i_tapmod31 , 'tapmod31' );
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModLFSR.DoSlowTick; // override;
    var
      Trig     : Boolean;
      Reset    : Boolean;
      Clocked  : Boolean;
      Reversed : Boolean;
      GateMode : Boolean;
      i        : Integer;
      M        : Cardinal;
    begin
      Reset    := SignalToLogic( FInputs[ i_reset   ]);
      Trig     := SignalToLogic( FInputs[ i_trig    ]);
      Clocked  := SignalToLogic( FInputs[ i_clocked ]);
      Reversed := SignalToLogic( FInputs[ i_reverse ]);
      GateMode := SignalToLogic( FInputs[ i_gatemode]);

      if   ( Reset and not FOldRes) or ResetFlag
      then begin
        ResetFlag := False;
        FResetReq := True;
      end;

      if   ( not Clocked) or ( Clocked and ( Trig and not FOldTrig))
      then begin
        if FResetReq
        then begin
          FResetReq := False;
          FShifter  := 0;
        end;
      end;

      if Trig and not FOldTrig
      then begin
        M         := 1;
        FFeedback := SignalToLogic( FInputs[ i_evenodd  ]) xor SignalToLogic( FInputs[ i_paritymod]);

        for i := 0 to 31
        do begin
          if SignalToLogic( FInputs[ i_tap0 + i]) xor SignalToLogic( FInputs[ i_tapmod0 + i])
          then FFeedback := FFeedback xor (( FShifter and M) <> 0);

          M := M shl 1;
        end;

        if   Reversed
        then FShifter := ( FShifter shr 1) or ( Cardinal(( SignalToLogic( FInputs[ i_in]) xor FFeedback)) shl 31)
        else FShifter := ( FShifter shl 1) or   Cardinal(( SignalToLogic( FInputs[ i_in]) xor FFeedback));
      end;

      FOutputs[ o_out] := LogicToSignal( FFeedback and ( Trig or GateMode));

      FOldRes  := Reset;
      FOldTrig := Trig;
    end;


    procedure   TModLFSR.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      i : Integer;
      M : Cardinal;
    begin
      M := 1;

      for i := 0 to 31
      do begin
        if ( FShifter and M) = 0
        then FLights[ 1 + i] := ' '
        else FLights[ 1 + i] := 'x';

        M := M shl 1;
      end;

      aCallback( Self, MakeInfo( aPrefix, Name, 'lights', FLights));
    end;


    procedure   TModLFSR.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'  , FInputs [ i_reset  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'   , FInputs [ i_trig   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reverse', FInputs [ i_reverse]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'in'     , FInputs [ i_in     ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'    , FOutputs[ o_out    ]));
    end;


{ ========
  TModStack = class( TMod)
  private
    FLights  : string;
    FStack   : array of Word;
    FSize    : Integer;
    FSP      : Integer;
    FOldPop  : Boolean;
    FOldPush : Boolean;
  public
    property    Size : Integer read FSize write SetSize;
  private
}

    procedure   TModStack.SetSize( aValue: Integer);
    begin
      if aValue <> FSize
      then begin
        Locked := True;

        try
          FSize := aValue;
          FSP   := 0;
          SetLength( FStack, FSize);
        finally
          Locked := ( FSize = 0);
        end;
      end;
    end;


//  public

    constructor TModStack.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      Size := 64;
    end;


    procedure   TModStack.CreateIO; // override;
    begin
      FIsSlow := True;
      SetLength( FLights, 16);
      AddInput ( i_in0  , 'in0'  );
      AddInput ( i_in1  , 'in1'  );
      AddInput ( i_in2  , 'in2'  );
      AddInput ( i_in3  , 'in3'  );
      AddInput ( i_in4  , 'in4'  );
      AddInput ( i_in5  , 'in5'  );
      AddInput ( i_in6  , 'in6'  );
      AddInput ( i_in7  , 'in7'  );
      AddInput ( i_in8  , 'in8'  );
      AddInput ( i_in9  , 'in9'  );
      AddInput ( i_in10 , 'in10' );
      AddInput ( i_in11 , 'in11' );
      AddInput ( i_in12 , 'in12' );
      AddInput ( i_in13 , 'in13' );
      AddInput ( i_in14 , 'in14' );
      AddInput ( i_in15 , 'in15' );
      AddInput ( i_push , 'push' );
      AddInput ( i_pop  , 'pop'  );
      AddOutput( o_out0 , 'out0' );
      AddOutput( o_out1 , 'out1' );
      AddOutput( o_out2 , 'out2' );
      AddOutput( o_out3 , 'out3' );
      AddOutput( o_out4 , 'out4' );
      AddOutput( o_out5 , 'out5' );
      AddOutput( o_out6 , 'out6' );
      AddOutput( o_out7 , 'out7' );
      AddOutput( o_out8 , 'out8' );
      AddOutput( o_out9 , 'out9' );
      AddOutput( o_out10, 'out10');
      AddOutput( o_out11, 'out11');
      AddOutput( o_out12, 'out12');
      AddOutput( o_out13, 'out13');
      AddOutput( o_out14, 'out14');
      AddOutput( o_out15, 'out15');
      AddOutput( o_full , 'full' );
      AddOutput( o_empty, 'empty');
    end;


    destructor  TModStack.Destroy; // override;
    begin
      Size := 0;
      inherited;
    end;


    procedure   TModStack.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'size')
      then Size := Round( aValue)
      else inherited;
    end;


    function    TModStack.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'size')
      then Result := FSize
      else Result := inherited;
    end;


    procedure   TModStack.DoSlowTick; // override;
    var
      Push   : Boolean;
      Pop    : Boolean;
      DoPush : Boolean;
      DoPop  : Boolean;
      i      : Integer;
      M      : Word;
      Value  : Word;
    begin
      Push   := SignalToLogic( FInputs[ i_push]);
      Pop    := SignalToLogic( FInputs[ i_pop ]);
      DoPush := Push and not FOldPush;
      DoPop  := Pop  and not FOldPop;

      if   DoPush xor DoPop
      then begin
        if   DoPush
        then begin
          if FSP < FSize - 1
          then begin
            M     := 1;
            Value := 0;

            for i := 0 to 15
            do begin
              if SignalToLogic( FInputs[ i_in0 + i])
              then Value := Value or M;

              M := M shl 1;
            end;

            Inc( FSP);
            FStack[ FSP] := Value;
          end;
        end
        else begin // DoPop
          if   FSP > 0
          then Dec( FSP);
        end;
      end;

      Value := FStack[ FSP];
      M     := 1;

      for i := 0 to 15
      do begin
        FOutputs[ o_out0 + i] := LogicToSignal( Value and M <> 0);
        M := M shl 1;
      end;

      FOutputs[ o_empty] := LogicToSignal( FSP <=        0);
      FOutputs[ o_full ] := LogicToSignal( FSP >= Size - 1);
      FOldPush := Push;
      FOldPop  := Pop;
    end;


    procedure   TModStack.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      i : Integer;
    begin
      for i := 0 to 15
      do begin
        if SignalToLogic( FOutputs[ o_out0 + i])
        then FLights[ 1 + i] := 'x'
        else FLights[ 1 + i] := ' ';
      end;

      aCallback( Self, MakeInfo( aPrefix, Name, 'lights', FLights));
    end;


    procedure   TModStack.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'push' , FInputs [ i_push ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'pop'  , FInputs [ i_pop  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'empty', FOutputs[ o_empty]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'full' , FOutputs[ o_full ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'depth', FSP + 1           ));
    end;


{ ========
  TModAnalogStack = class( TMod)
  strict private
  const
    i_in    =  0;
    i_push  =  1;
    i_pop   =  2;
  const
    o_out   =  0;
    o_full  =  1;
    o_empty =  2;
  private
    FLights  : string;
    FStack   : array of TSignal;
    FSize    : Integer;
    FSP      : Integer;
    FOldPop  : Boolean;
    FOldPush : Boolean;
  public
    property    Size : Integer read FSize write SetSize;
  private
}

    procedure   TModAnalogStack.SetSize( aValue: Integer);
    begin
      if aValue <> FSize
      then begin
        Locked := True;

        try
          FSize := aValue;
          FSP   := 0;
          SetLength( FStack, FSize);
        finally
          Locked := ( FSize = 0);
        end;
      end;
    end;


//  public

    constructor TModAnalogStack.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      Size := 64;
    end;


    procedure   TModAnalogStack.CreateIO; // override;
    begin
      FIsSlow := True;
      SetLength( FLights, 16);
      AddInput ( i_in   , 'in'   );
      AddInput ( i_push , 'push' );
      AddInput ( i_pop  , 'pop'  );
      AddOutput( o_out  , 'out'  );
      AddOutput( o_full , 'full' );
      AddOutput( o_empty, 'empty');
    end;


    destructor  TModAnalogStack.Destroy; // override;
    begin
      Size := 0;
      inherited;
    end;


    procedure   TModAnalogStack.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'size')
      then Size := Round( aValue)
      else inherited;
    end;


    function    TModAnalogStack.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'size')
      then Result := FSize
      else Result := inherited;
    end;


    procedure   TModAnalogStack.DoSlowTick; // override;
    var
      Push   : Boolean;
      Pop    : Boolean;
      DoPush : Boolean;
      DoPop  : Boolean;
    begin
      Push   := SignalToLogic( FInputs[ i_push]);
      Pop    := SignalToLogic( FInputs[ i_pop ]);
      DoPush := Push and not FOldPush;
      DoPop  := Pop  and not FOldPop;

      if   DoPush xor DoPop
      then begin
        if   DoPush
        then begin
          if FSP < FSize - 1
          then begin
            Inc( FSP);
            FStack[ FSP] := FInputs[ i_in];
          end;
        end
        else begin // DoPop
          if   FSP > 0
          then Dec( FSP);
        end;
      end;


      FOutputs[ o_out  ] := FStack[ FSP];
      FOutputs[ o_empty] := LogicToSignal( FSP <=        0);
      FOutputs[ o_full ] := LogicToSignal( FSP >= Size - 1);
      FOldPush := Push;
      FOldPop  := Pop;
    end;


    procedure   TModAnalogStack.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'push' , FInputs [ i_push ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'pop'  , FInputs [ i_pop  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'empty', FOutputs[ o_empty]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'full' , FOutputs[ o_full ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'depth', FSP + 1          ));
    end;


{ ========
  TModSwanSong = class( TMod)
  strict private
  const
    HUNTER_COUNT = 6;
  private
    FMazeGraph       : TMazeGraph;
    FMazeForth       : TMazeForth;
    FMazeList        : TStringList;
    FMazeType        : Integer;
    FOldHunterClocks : array[ 0 .. HUNTER_COUNT - 1] of Boolean;
    FAngles          : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FStrides         : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FXValues         : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FYValues         : array[ 0 .. HUNTER_COUNT - 1] of TSignal;
    FSwan            : TMazePoint;
    FHunterLocations : TSignalArray;
    FResetReq        : Boolean;
    FMazeTypeChanged : Boolean;
    FMazeExtent      : TMazeRect;
    FSwanOffSet      : TMazePoint;
    FSwanScale       : TMazePoint;
  public
    property    MazeType : Integer read FMazeType write SetMazeType;
  private
}

    procedure   TModSwanSong.SetMazeType( aValue: Integer);
    begin
      if   Assigned( FMazeGraph)
      and  Assigned( FMazeList )
      and  ( FMazeList.Count > 0)
      then begin
        if   ( aValue < 0               )
        or   ( aValue >= FMazeList.Count)
        then aValue := 0;

        if   aValue <> FMazeType
        then begin
          Locked := True;

          try
            FMazeType := aValue;
            FMazeForth.AcceptText( FMazeList[ FMazeType]);
            FMazeExtent   := FMazeGraph.Extent;

            if FMazeExtent.IsNull
            then begin
              FSwanOffSet.X := 0;
              FSwanOffSet.Y := 0;
              FSwanScale .X := 1;
              FSwanScale .Y := 1;
            end
            else begin
              FSwanOffSet.X := FMazeExtent.Left;
              FSwanOffSet.Y := FMazeExtent.Top;
              FSwanScale .X := FMazeExtent.Width;
              FSwanScale .Y := FMazeExtent.Height;
            end;
          finally
            Locked := False;
          end;

          FMazeTypeChanged := True;
        end;
      end;
    end;


    function    TModSwanSong.ScaledSwan: TMazePoint;
    begin
      Result.X := FSwanOffSet.X + FSwan.X * FSwanScale.X;
      Result.Y := FSwanOffSet.Y + FSwan.Y * FSwanScale.Y;
    end;


    procedure   TModSwanSong.DoHunterMoved( const aSender: TObject; anId, aLocation: Integer; aDistance, anAngle, anX, anY: TSignal);
    begin
      if ( anId >= 0) and ( anId < HUNTER_COUNT)
      then begin
        FHunterLocations[ anId] := aLocation;
        FAngles         [ anId] := MathFloatMod( anAngle, TWO_PI) * TWO_PI_REC;
        FStrides        [ anId] := aDistance;

        if   Assigned( FMazeGraph)
        and  not FMazeGraph.Extent.IsNull
        then begin
          FXValues [ anId] := RangeMap( anX, FMazeGraph.Extent.Left, FMazeGraph.Extent.Right , 0.0, 1.0);
          FYValues [ anId] := RangeMap( anY, FMazeGraph.Extent.Top , FMazeGraph.Extent.Bottom, 0.0, 1.0);
        end;
      end;
    end;


//  public

    constructor TModSwanSong.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FMazeGraph := TMazeGraph.Create( HUNTER_COUNT);
      FMazeForth := TMazeForth.Create( FMazeGraph, 1);
      FMazeList  := FMazeForth.CreateExportList;
      FMazeType  := -1;
      SetLength( FHunterLocations, HUNTER_COUNT);
      FMazeGraph.OnHunterMoved := DoHunterMoved;
    end;


    procedure   TModSwanSong.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_swanx        , 'swanx'        );
      AddInput ( i_swany        , 'swany'        );
      AddInput ( i_hunter1clock , 'hunter1clock' );
      AddInput ( i_hunter2clock , 'hunter2clock' );
      AddInput ( i_hunter3clock , 'hunter3clock' );
      AddInput ( i_hunter4clock , 'hunter4clock' );
      AddInput ( i_hunter5clock , 'hunter5clock' );
      AddInput ( i_hunter6clock , 'hunter6clock' );
      AddInput ( i_intype       , 'intype'       );
      AddInput ( i_outtype      , 'outtype'      );
      AddInput ( i_mazetype     , 'mazetype'     );
      AddInput ( i_mazemod      , 'mazemod'      );
      AddInput ( i_hunterrule   , 'hunterrule'   );
      AddInput ( i_scaling      , 'scaling'      );
      AddOutput( o_hunter1angle , 'hunter1angle' );
      AddOutput( o_hunter2angle , 'hunter2angle' );
      AddOutput( o_hunter3angle , 'hunter3angle' );
      AddOutput( o_hunter4angle , 'hunter4angle' );
      AddOutput( o_hunter5angle , 'hunter5angle' );
      AddOutput( o_hunter6angle , 'hunter6angle' );
      AddOutput( o_hunter1stride, 'hunter1stride');
      AddOutput( o_hunter2stride, 'hunter2stride');
      AddOutput( o_hunter3stride, 'hunter3stride');
      AddOutput( o_hunter4stride, 'hunter4stride');
      AddOutput( o_hunter5stride, 'hunter5stride');
      AddOutput( o_hunter6stride, 'hunter6stride');
      AddOutput( o_hunter1x     , 'hunter1x'     );
      AddOutput( o_hunter2x     , 'hunter2x'     );
      AddOutput( o_hunter3x     , 'hunter3x'     );
      AddOutput( o_hunter4x     , 'hunter4x'     );
      AddOutput( o_hunter5x     , 'hunter5x'     );
      AddOutput( o_hunter6x     , 'hunter6x'     );
      AddOutput( o_hunter1y     , 'hunter1y'     );
      AddOutput( o_hunter2y     , 'hunter2y'     );
      AddOutput( o_hunter3y     , 'hunter3y'     );
      AddOutput( o_hunter4y     , 'hunter4y'     );
      AddOutput( o_hunter5y     , 'hunter5y'     );
      AddOutput( o_hunter6y     , 'hunter6y'     );
    end;


    destructor  TModSwanSong.Destroy; // override;
    begin
      FreeAndNil( FMazeList );
      FreeAndNil( FMazeForth);
      FreeAndNil( FMazeGraph);
      inherited;
    end;


    procedure   TModSwanSong.DoSlowTick; // override;
    const
      ScaleFactors : array[ 0 .. 1] of TSignal = (
        1.0,
        12.0 / 128
      );
    var
      HunterClock : Boolean;
      i           : Integer;
      InType      : Integer;
      OutType     : Integer;
      AngleVal    : TSignal;
      Scaling     : Integer;
    begin
      if ResetFlag
      then FResetReq := True;

      if Assigned( FMazeGraph)
      then begin
        for i := 0 to HUNTER_COUNT - 1
        do begin
          HunterClock := SignalToLogic( FInputs[ i_hunter1clock + i]);

          if not FOldHunterClocks[ i] and HunterClock
          then FMazeGraph.MoveHunter( i, True);

          FOldHunterClocks[ i] := HunterClock;
        end;

        InType                  := Round( FInputs[ i_intype]);
        FSwan.X                 := SignalForOutputType( SignalForInputType( FInputs[ i_swanx], InType), outputtype_positive);
        FSwan.Y                 := SignalForOutputType( SignalForInputType( FInputs[ i_swany], InType), outputtype_positive);
        FMazeGraph.SwanPosition := ScaledSwan;
        FMazeGraph.HunterRule   := TmazeHunterRule( Round( FInputs[ i_hunterrule]));
        OutType                 := Round( FInputs[ i_outtype]);
        Scaling                 := Round( FInputs[ i_scaling]);

        for i := 0 to HUNTER_COUNT - 1
        do begin
          if Scaling = 0
          then AngleVal := FAngles [ i]
          else AngleVal := 12.0 * FAngles [ i] * NOTE_SCALING_REC;

          FOutputs[ o_hunter1angle  + i] := SignalForOutputType( SignalForInputType( AngleVal    , inputtype_positive), OutType);
          FOutputs[ o_hunter1stride + i] := SignalForOutputType( SignalForInputType( FStrides[ i], inputtype_positive), OutType);
          FOutputs[ o_hunter1x      + i] := SignalForOutputType( SignalForInputType( FXValues[ i], inputtype_positive), OutType);
          FOutputs[ o_hunter1y      + i] := SignalForOutputType( SignalForInputType( FYValues[ i], inputtype_positive), OutType);
        end;
      end;

      ResetFlag := False;
    end;


    procedure   TModSwanSong.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      if Assigned( FMazeList)
      then MazeType := Round( FInputs[ i_mazetype] + ( FMazeList.Count - 1) * FInputs[ i_mazemod]) mod FMazeList.Count;

      if FResetReq
      then begin
        FResetReq := False;
        FMazeGraph.Reset;
      end;

      for i := 0 to HUNTER_COUNT - 1
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'hunter%d', [ i + 1], AppLocale), LogicToSignal( FOldHunterClocks[ i])));

      if FMazeTypeChanged
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'mazegraph', MazeType));
        FMazeTypeChanged := False;
      end;
    end;


    procedure   TModSwanSong.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'mazegraph', SignalPair( FSwan.X, FSwan.Y)));
    end;


    procedure   TModSwanSong.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'mazegraph', FHunterLocations));
    end;


{ ========
  TModAdc = class( TMod)
  public
}

    procedure   TModAdc.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in   , 'in'   );
      AddOutput( o_out0 , 'out0' );
      AddOutput( o_out1 , 'out1' );
      AddOutput( o_out2 , 'out2' );
      AddOutput( o_out3 , 'out3' );
      AddOutput( o_out4 , 'out4' );
      AddOutput( o_out5 , 'out5' );
      AddOutput( o_out6 , 'out6' );
      AddOutput( o_out7 , 'out7' );
      AddOutput( o_out8 , 'out8' );
      AddOutput( o_out9 , 'out9' );
      AddOutput( o_out10, 'out10');
      AddOutput( o_out11, 'out11');
      AddOutput( o_out12, 'out12');
      AddOutput( o_out13, 'out13');
      AddOutput( o_out14, 'out14');
      AddOutput( o_out15, 'out15');
    end;


    procedure   TModAdc.DoSlowTick; // override;
    var
      aValue : Integer;
      i      : Integer;
    begin
      aValue := SignalToBinary( FInputs[ i_in], o_out15 - o_out0 + 1);

      for i := o_out0 to o_out15
      do begin
        FOutputs[ i] := LogicToSignal(( aValue and 1) = 1);
        aValue := aValue shr 1;
      end;
    end;


    procedure   TModAdc.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := o_out0 to o_out15
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'out%d', [ i - o_out0], AppLocale), FOutputs[ i]));
    end;


{ ========
  TModDac = class( TMod)
  public
}

    procedure   TModDac.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in0 , 'in0' );
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddInput ( i_in5 , 'in5' );
      AddInput ( i_in6 , 'in6' );
      AddInput ( i_in7 , 'in7' );
      AddInput ( i_in8 , 'in8' );
      AddInput ( i_in9 , 'in9' );
      AddInput ( i_in10, 'in10');
      AddInput ( i_in11, 'in11');
      AddInput ( i_in12, 'in12');
      AddInput ( i_in13, 'in13');
      AddInput ( i_in14, 'in14');
      AddInput ( i_in15, 'in15');
      AddOutput( o_out , 'out' );
    end;


    procedure   TModDac.DoSlowTick; // override;
    var
      aValue : Integer;
      i      : Integer;
    begin
      aValue := 0;

      for i := i_in15 downto i_in0
      do begin
        aValue := aValue shl 1;

        if SignalToLogic( FInputs[ i])
        then aValue := aValue or 1;
      end;

      FOutputs[ o_out] := BinaryToSignal( aValue, n_bits);
    end;


    procedure   TModDac.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := i_in0 to i_in15
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'in%d', [ i - i_in0], AppLocale), FInputs[ i]));
    end;


{ ========
  TPDState = 1 .. 12;

  TModPhaseDetect = class( TMod)
  private
    FOldIn1 : Boolean;
    FOldIn2 : Boolean;
    FState  : TPDState;
    FAlpha  : TSignal;
  public
}

    procedure   TModPhaseDetect.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in1        , 'in1'        );
      AddInput ( i_in2        , 'in2'        );
      AddInput ( i_alpha      , 'alpha'      );
      AddInput ( i_alphamod   , 'alphamod'   );
      AddInput ( i_alphamodamt, 'alphamodamt');
      AddInput ( i_type       , 'type'       );
      AddInput ( i_typesel    , 'typesel'    );
      AddOutput( o_out        , 'out'        );
      AddOutput( o_lock       , 'lock'       );
    end;


    procedure   TModPhaseDetect.SetDefaults; // override;
    begin
      FState := 1;
      FInputs[ i_typesel] := -1.0;
      FDezipperMap := [
        i_alpha,
        i_alphamodamt
      ];
    end;


    procedure   TModPhaseDetect.DoSlowTick; // override;
    var
      NewIn1   : Boolean;
      NewIn2   : Boolean;
      In1Up    : Boolean;
      In1Down  : Boolean;
      In2Up    : Boolean;
      In2Down  : Boolean;
      Inp      : TSignal;
      Outp     : TSignal;
      aType    : Integer;
      aTypeSel : Boolean;
    begin
      FAlpha   := Normalize( ExpAttack( FInputs[ i_alpha] + FInputs[ i_alphamod] * FInputs[ i_alphamodamt], 1, 0, 1));
      NewIn1   := SignalToLogic( FInputs[ i_in1    ]);
      NewIn2   := SignalToLogic( FInputs[ i_in2    ]);
      aType    := Round        ( FInputs[ i_type   ]);
      aTypeSel := SignalToLogic( FInputs[ i_typesel]);

      if aTypeSel
      then aType := 2 - aType;

      In1Up   := not FOldIn1 and     NewIn1;
      In1Down :=     FOldIn1 and not NewIn1;
      In2Up   := not FOldIn2 and     NewIn2;
      In2Down :=     FOldIn2 and not NewIn2;

      case FState of
         1 : if in1up   then FState :=  3 else if in2up   then FState :=  2;
         2 : if In1Up   then FState :=  6 else if In2Down then FState :=  4;
         3 : if In1Down then FState :=  5 else if In2Up   then FState :=  6;
         4 : if In1Up   then FState :=  8 else if In2Up   then FState :=  2;
         5 : if In1Up   then FState :=  3 else if In2Up   then FState :=  7;
         6 : if In1Down then FState :=  7 else if in2down then FState :=  8;
         7 : if In1Up   then FState :=  9 else if In2Down then FState :=  1;
         8 : if In1Down then FState :=  1 else if In2Up   then FState := 10;
         9 : if In1Down then FState := 11 else if In2Down then FState :=  3;
        10 : if In1Down then FState :=  2 else if In2Down then FState := 12;
        11 : if In1Up   then FState :=  9 else if In2Down then FState :=  5;
        12 : if In1Down then FState :=  4 else if In2Up   then FState := 10;
      end;

      Outp := FOutputs[ o_out];

      if aType = 0
      then Inp := LogicToSignal( NewIn1 xor NewIn2)
      else if aType = 1
      then begin
        if FState in [ 3, 5, 9, 11]
        then Inp := 1.0                                      // Input frequency is higher
        else if FState in [ 2, 4, 10, 12]
        then Inp := - 1.0                                    // Input frequency is higher
        else Inp := Outp;                                    // Input frequency is the same
      end
      else Inp := FInputs[ i_in1] * FInputs[ i_in2];

      Outp  := Outp + FAlpha * ( Inp - Outp);
      FOutputs[ o_out ] := Normalize( Outp);
      FOutputs[ o_lock] := LogicToSignal( FState in [ 1, 6, 7, 8]);
      FOldIn1 := NewIn1;
      FOldIn2 := NewIn2;
    end;


    procedure   TModPhaseDetect.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'up'  , LogicToSignal( FState in [ 3, 5,  9, 11])));
      aCallback( Self, MakeInfo( aPrefix, Name, 'lock', LogicToSignal( FState in [ 1, 6,  7,  8])));
      aCallback( Self, MakeInfo( aPrefix, Name, 'down', LogicToSignal( FState in [ 2, 4, 10, 12])));
    end;


{ ========
  TModNot = class( TMod)
  public
}

    procedure   TModNot.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
    end;


    procedure   TModNot.SetDefaults; // override;
    var
      i : Integer;
    begin
      for i := i_in1 to i_in4          // Set unused inputs to 1 - just to keep the lights off
      do FInputs[ i] := 1;
    end;


    procedure   TModNot.DoSlowTick; // override;
    var
      i : Integer;
    begin
      for i := i_in1 to i_in4
      do FOutputs[ i] := LogicToSignal( not SignalToLogic( FInputs[ i]));
    end;


    procedure   TModNot.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := o_out1 to o_out4
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'out%d', [ i + 1], AppLocale), FOutputs[ i]));
    end;


{ ========
  TModLogicSelector = class( TMod)
  public
}

    procedure   TModLogicSelector.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_sel , 'sel' );
      AddInput ( i_ina , 'ina' );
      AddInput ( i_inb , 'inb' );
      AddOutput( o_outa, 'outa');
      AddOutput( o_outb, 'outb');
    end;


    procedure   TModLogicSelector.SetDefaults; // override;
    begin
      FInputs[ i_sel] := LogicToSignal( False);
      FInputs[ i_ina] := LogicToSignal( False);
      FInputs[ i_inb] := LogicToSignal( False);
    end;


    procedure   TModLogicSelector.DoSlowTick; // override;
    begin
      if SignalToLogic( FInputs[ i_sel])
      then begin
        FOutputs[ o_outa] := LogicToSignal( SignalToLogic( FInputs[ i_inb]));
        FOutputs[ o_outb] := LogicToSignal( SignalToLogic( FInputs[ i_ina]));
      end
      else begin
        FOutputs[ o_outa] := LogicToSignal( SignalToLogic( FInputs[ i_ina]));
        FOutputs[ o_outb] := LogicToSignal( SignalToLogic( FInputs[ i_inb]));
      end;
    end;


    procedure   TModLogicSelector.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'outa'  , FOutputs[ o_outa]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outb'  , FOutputs[ o_outb]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'select', FInputs [ i_sel ]));
    end;


{ ========
  TModAmplifier = class( TMod)
  public
}

    procedure   TModAmplifier.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_gain, 'gain');
      AddInput ( i_mute, 'mute');
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_in3 , 'in3' );
      AddInput ( i_in4 , 'in4' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
      AddOutput( o_out3, 'out3');
      AddOutput( o_out4, 'out4');
    end;


    procedure   TModAmplifier.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_gain
      ];
    end;


    procedure   TModAmplifier.DoSlowTick; // override;
    var
      i : Integer;
      G : TSignal;
    begin
      G := FInputs[ i_gain] * SignalToMute( FInputs[ i_mute]);

      for i := i_in1 to i_in4
      do FOutputs[ i - i_in1 + o_out1] := FInputs[ i] * G;
    end;


{ ========
  TModScaler = class( TMod)
  public
}

    procedure   TModScaler.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_gain  , 'gain'  );
      AddInput ( i_offset, 'offset');
      AddInput ( i_mode  , 'mode'  );
      AddInput ( i_mute  , 'mute'  );
      AddInput ( i_in1   , 'in1'   );
      AddInput ( i_in2   , 'in2'   );
      AddInput ( i_in3   , 'in3'   );
      AddInput ( i_in4   , 'in4'   );
      AddOutput( o_out1  , 'out1'  );
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_out3  , 'out3'  );
      AddOutput( o_out4  , 'out4'  );
      AddOutput( o_sum   , 'sum'   );   // Sum outpu)t
    end;


    procedure   TModScaler.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_gain,
        i_offset
      ];
    end;


    procedure   TModScaler.DoSlowTick; // override;
    var
      i : Integer;
      G : TSignal;
      O : TSignal;
      S : TSignal;
      T : TSignal;
    begin
      G := FInputs[ i_gain  ] * SignalToMute( FInputs[ i_mute]);
      O := FInputs[ i_offset] * SignalToMute( FInputs[ i_mute]);
      S := 0.0;

      for i := i_in1 to i_in4
      do begin
        T :=  G * FInputs[ i] + O;
        FOutputs[ i - i_in1 + o_out1] := T;
        S := S + T;
      end;

      FOutputs[ o_sum] := S;
    end;


{ ========
  TModPropScaler = class( TMod)
  public
}

    procedure   TModPropScaler.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_p      , 'p'      );
      AddInput ( i_pmod   , 'pmod'   );
      AddInput ( i_pmodamt, 'pmodamt');
      AddInput ( i_in1    , 'in1'    );
      AddInput ( i_in2    , 'in2'    );
      AddInput ( i_in3    , 'in3'    );
      AddInput ( i_in4    , 'in4'    );
      AddOutput( o_out1   , 'out1'   );
      AddOutput( o_out2   , 'out2'   );
      AddOutput( o_out3   , 'out3'   );
      AddOutput( o_out4   , 'out4'   );
    end;


    procedure   TModPropScaler.DoSlowTick; // override;
    var
      p : TSignal;
      i : Integer;
    begin
      p := Clip( FInputs[ i_p] + FInputs[ i_pmod] * FInputs[ i_pmodamt], 0.0, 1.0);

      for i := i_in1 to i_in4
      do FOutputs[ o_out1 + i - i_in1] := Finputs[ i] * ( 1.0 - p) + p;
    end;


{ ========
  TModMixingScaler = class( TMod)
  public
}

    procedure   TModMixingScaler.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_gain    , 'gain'    );
      AddInput ( i_offset  , 'offset'  );
      AddInput ( i_mode    , 'mode'    );
      AddInput ( i_mute    , 'mute'    );
      AddInput ( i_in1     , 'in1'     );
      AddInput ( i_gain2   , 'gain2'   );
      AddInput ( i_offset2 , 'offset2' );
      AddInput ( i_mode2   , 'mode2'   );
      AddInput ( i_mute2   , 'mute2'   );
      AddInput ( i_in2     , 'in2'     );
      AddInput ( i_gain3   , 'gain3'   );
      AddInput ( i_offset3 , 'offset3' );
      AddInput ( i_mode3   , 'mode3'   );
      AddInput ( i_mute3   , 'mute3'   );
      AddInput ( i_in3     , 'in3'     );
      AddInput ( i_gain4   , 'gain4'   );
      AddInput ( i_offset4 , 'offset4' );
      AddInput ( i_mode4   , 'mode4'   );
      AddInput ( i_mute4   , 'mute4'   );
      AddInput ( i_in4     , 'in4'     );
      AddInput ( i_outmute , 'outmute' );
      AddInput ( i_outlevel, 'outlevel');
      AddInput ( i_chain   , 'chain'   );
      AddOutput( o_out1    , 'out1'    );
      AddOutput( o_out2    , 'out2'    );
      AddOutput( o_out3    , 'out3'    );
      AddOutput( o_out4    , 'out4'    );
      AddOutput( o_sum     , 'sum'     );   // Sum outpu)t
    end;


    procedure   TModMixingScaler.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_gain   , i_gain2   , i_gain3   , i_gain4   ,
        i_offset , i_offset2 , i_offset3 , i_offset4 ,
        i_outmute, i_outlevel
      ];
    end;


    procedure   TModMixingScaler.DoSlowTick; // override;
    var
      G : TSignal;
      O : TSignal;
      S : TSignal;
      T : TSignal;
    begin
      S := 0.0;

      G := FInputs[ i_gain  ];
      O := FInputs[ i_offset];
      T := Normalize(( G * FInputs[ i_in1] + O) * SignalToMute( FInputs[ i_mute]));
      S := S + T;
      FOutputs[ o_out1] := T;

      G := FInputs[ i_gain2  ];
      O := FInputs[ i_offset2];
      T := Normalize(( G * FInputs[ i_in2] + O) * SignalToMute( FInputs[ i_mute2]));
      S := S + T;
      FOutputs[ o_out2] := T;

      G := FInputs[ i_gain3  ];
      O := FInputs[ i_offset3];
      T := Normalize(( G * FInputs[ i_in3] + O) * SignalToMute( FInputs[ i_mute3]));
      S := S + T;
      FOutputs[ o_out3] := T;

      G := FInputs[ i_gain4  ];
      O := FInputs[ i_offset4];
      T := Normalize(( G * FInputs[ i_in4] + O) * SignalToMute( FInputs[ i_mute4]));
      S := S + T;
      FOutputs[ o_out4] := T;

      FOutputs[ o_sum] := Normalize( ( S + FInputs[ i_chain]) * FInputs[ i_outmute] * FInputs[ i_outlevel]);
    end;


{ ========
  TModRatio = class( TMod)
  public
}

    procedure   TModRatio.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_multiplier, 'multiplier');
      AddInput ( i_divider   , 'divider'   );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_in1       , 'in1'       );
      AddInput ( i_in2       , 'in2'       );
      AddInput ( i_in3       , 'in3'       );
      AddInput ( i_in4       , 'in4'       );
      AddOutput( o_out1      , 'out1'      );
      AddOutput( o_out2      , 'out2'      );
      AddOutput( o_out3      , 'out3'      );
      AddOutput( o_out4      , 'out4'      );
    end;


    procedure   TModRatio.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_multiplier,
        i_divider
      ];
    end;


    procedure   TModRatio.DoSlowTick; // override;
    var
      i      : Integer;
      MulDiv : TSignal;
      RDiv   : Integer;
    begin
      RDiv := Round( FInputs[ i_divider]);

      if RDiv = 0
      then Muldiv := 1
      else Muldiv := Round( SignalToMute( FInputs[ i_mute]) * FInputs[ i_multiplier]) div RDiv;

      for i := i_in1 to i_in4
      do FOutputs[ i - i_in1 + o_out1] := MulDiv * FInputs[ i];
    end;


{ ========
  TModRange = class( TMod)
  private
    FInLow   : TSignal;
    FInHigh  : TSignal;
    FSigType : Integer;
  public
}

    procedure   TModRangeConverter.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_lowvalue , 'lowvalue' );
      AddInput ( i_highvalue, 'highvalue');
      AddInput ( i_mode     , 'mode'     );
      AddInput ( i_in1      , 'in1'      );
      AddInput ( i_in2      , 'in2'      );
      AddInput ( i_in3      , 'in3'      );
      AddInput ( i_in4      , 'in4'      );
      AddOutput( o_out1     , 'out1'     );
      AddOutput( o_out2     , 'out2'     );
      AddOutput( o_out3     , 'out3'     );
      AddOutput( o_out4     , 'out4'     );
    end;


    procedure   TModRangeConverter.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_lowvalue,
        i_highvalue
      ];
      FMode := s_positive;
    end;


    procedure   TModRangeConverter.DoSlowTick; // override;
    // Called from DoSlowTick when in slow mode, otherwise called directly from the TSynthPatch
    begin
      FInLow  :=        FInputs[ i_lowvalue ];
      FInHigh :=        FInputs[ i_highvalue];
      FMode   := Round( FInputs[ i_mode     ]);

      case FMode of

        s_inverted : begin
          FOutputs[ o_out1] := RangeMap( FInputs[ i_in1],  1, -1, FInLow, FInHigh);
          FOutputs[ o_out2] := RangeMap( FInputs[ i_in2],  1, -1, FInLow, FInHigh);
          FOutputs[ o_out3] := RangeMap( FInputs[ i_in3],  1, -1, FInLow, FInHigh);
          FOutputs[ o_out4] := RangeMap( FInputs[ i_in4],  1, -1, FInLow, FInHigh);
        end;

        s_positive : begin
          FOutputs[ o_out1] := RangeMap( FInputs[ i_in1],  0,  1, FInLow, FInHigh);
          FOutputs[ o_out2] := RangeMap( FInputs[ i_in2],  0,  1, FInLow, FInHigh);
          FOutputs[ o_out3] := RangeMap( FInputs[ i_in3],  0,  1, FInLow, FInHigh);
          FOutputs[ o_out4] := RangeMap( FInputs[ i_in4],  0,  1, FInLow, FInHigh);
        end;

        s_positive_inverted : begin
          FOutputs[ o_out1] := RangeMap( FInputs[ i_in1],  1,  0, FInLow, FInHigh);
          FOutputs[ o_out2] := RangeMap( FInputs[ i_in2],  1,  0, FInLow, FInHigh);
          FOutputs[ o_out3] := RangeMap( FInputs[ i_in3],  1,  0, FInLow, FInHigh);
          FOutputs[ o_out4] := RangeMap( FInputs[ i_in4],  1,  0, FInLow, FInHigh);
        end;

        s_negative : begin
          FOutputs[ o_out1] := RangeMap( FInputs[ i_in1], -1,  0, FInLow, FInHigh);
          FOutputs[ o_out2] := RangeMap( FInputs[ i_in2], -1,  0, FInLow, FInHigh);
          FOutputs[ o_out3] := RangeMap( FInputs[ i_in3], -1,  0, FInLow, FInHigh);
          FOutputs[ o_out4] := RangeMap( FInputs[ i_in4], -1,  0, FInLow, FInHigh);
        end;

        s_negative_inverted : begin
          FOutputs[ o_out1] := RangeMap( FInputs[ i_in1],  0, -1, FInLow, FInHigh);
          FOutputs[ o_out2] := RangeMap( FInputs[ i_in2],  0, -1, FInLow, FInHigh);
          FOutputs[ o_out3] := RangeMap( FInputs[ i_in3],  0, -1, FInLow, FInHigh);
          FOutputs[ o_out4] := RangeMap( FInputs[ i_in4],  0, -1, FInLow, FInHigh);
        end;

        else begin // s_normal
          FOutputs[ o_out1] := RangeMap( FInputs[ i_in1], -1,  1, FInLow, FInHigh);
          FOutputs[ o_out2] := RangeMap( FInputs[ i_in2], -1,  1, FInLow, FInHigh);
          FOutputs[ o_out3] := RangeMap( FInputs[ i_in3], -1,  1, FInLow, FInHigh);
          FOutputs[ o_out4] := RangeMap( FInputs[ i_in4], -1,  1, FInLow, FInHigh);
        end;
      end;
    end;


{ ========
  TModLevelConverter = class( TMod)
  private
    FInMode  : Integer;
    FOutMode : Integer;
  public
}

    procedure   TModLevelConverter.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_inmode , 'inmode' );
      AddInput ( i_outmode, 'outmode');
      AddInput ( i_in1    , 'in1'    );
      AddInput ( i_in2    , 'in2'    );
      AddInput ( i_in3    , 'in3'    );
      AddInput ( i_in4    , 'in4'    );
      AddInput ( i_in5    , 'in5'    );
      AddInput ( i_in6    , 'in6'    );
      AddInput ( i_in7    , 'in7'    );
      AddOutput( o_out1   , 'out1'   );
      AddOutput( o_out2   , 'out2'   );
      AddOutput( o_out3   , 'out3'   );
      AddOutput( o_out4   , 'out4'   );
      AddOutput( o_out5   , 'out5'   );
      AddOutput( o_out6   , 'out6'   );
      AddOutput( o_out7   , 'out7'   );
    end;


    procedure   TModLevelConverter.SetDefaults; // override;
    begin
      FInMode  := s_normal;
      FOutMode := s_positive;
    end;


    procedure   TModLevelConverter.DoSlowTick; // override;
    begin
      FInMode  := Round( FInputs[ i_inmode ]);
      FOutMode := Round( FInputs[ i_outmode]);
      FOutputs[ o_out1] := SignalForOutputType( SignalForInputType( FInputs[ i_in1], FInMode), FOutMode);
      FOutputs[ o_out2] := SignalForOutputType( SignalForInputType( FInputs[ i_in2], FInMode), FOutMode);
      FOutputs[ o_out3] := SignalForOutputType( SignalForInputType( FInputs[ i_in3], FInMode), FOutMode);
      FOutputs[ o_out4] := SignalForOutputType( SignalForInputType( FInputs[ i_in4], FInMode), FOutMode);
      FOutputs[ o_out5] := SignalForOutputType( SignalForInputType( FInputs[ i_in5], FInMode), FOutMode);
      FOutputs[ o_out6] := SignalForOutputType( SignalForInputType( FInputs[ i_in6], FInMode), FOutMode);
      FOutputs[ o_out7] := SignalForOutputType( SignalForInputType( FInputs[ i_in7], FInMode), FOutMode);
    end;


{ ========
  TModTypeFlip = class( TMod)
  private
    FInMode   : Integer;
    FOutModeL : Integer;
    FOutModeR : Integer;
    FSelect   : TSignal;
  public
}

    procedure   TModTypeFlip.CreateIO; // override;
    begin
      FISSlow := True;
      AddInput ( i_in      , 'in'      );
      AddInput ( i_inmode  , 'inmode'  );
      AddInput ( i_outmodel, 'outmodel');
      AddInput ( i_outmoder, 'outmoder');
      AddInput ( i_select  , 'select'  );
      AddInput ( i_slew    , 'slew'    );
      AddOutput( o_outl    , 'outl'    );
      AddOutput( o_outr    , 'outr'    );
    end;


    procedure   TModTypeFlip.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_slew
      ];
    end;


    procedure   TModTypeFlip.DoSlowTick; // override;
    var
      Select  : TSignal;
      Inp     : TSignal;
      OutL    : TSignal;
      OutR    : TSignal;
      Curve   : TSignal;
      Signal1 : TSignal;
      Signal2 : TSignal;
    begin
      Inp       := FInputs[ i_in];
      FInMode   := Round( FInputs[ i_inmode  ]);
      FOutModeL := Round( FInputs[ i_outmodel]);
      FOutModeR := Round( FInputs[ i_outmoder]);
      Curve     := SlewToCurve( FInputs[ i_slew]);
      Select    := SignalForOutputType( LogicToSignal( SignalToLogic( FInputs[ i_select])), s_positive);
      FSelect   := Normalize( Select + Curve * ( FSelect - Select));
      Signal1   := SignalForOutputType( Inp, FOutModeL);
      Signal2   := SignalForOutputType( Inp, FOutModeR);
      OutL      := Signal1 + FSelect * ( Signal2 - Signal1);
      OutR      := Signal2 + FSelect * ( Signal1 - Signal2);
      FOutputs[ o_outl] := OutL;
      FOutputs[ o_outr] := OutR;
    end;


    procedure   TModTypeFlip.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'select', LogicToSignal( FSelect > 0.5)));
    end;


{ ========
  TModFastToSlow = class( TMod)
  private
    FValue : TSignal;
  public
}

    procedure   TModFastToSlow.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in , 'in' );
      AddOutput( o_out, 'out');
    end;


    procedure   TModFastToSlow.DoSlowTick; // override;
    // DoSlowTick executes after DoTick
    begin
      FOutputs[ o_out] := FValue / Control_Decimation;  // Scale accumulated value
      FValue := 0;                                      // and set it back to zero
    end;


    procedure   TModFastToSlow.DoTick; // override;
    // DoTick executes before DoSlowTick
    begin
      FValue := FValue + FInputs[ i_in];           // Accumulate over CONTROL_DEC periods
    end;


{ ========
  TModSlowToFast = class( TMod)
  private
    FOldVal : TSignal;
    FNewVal : TSignal;
    FCount  : Integer;
  public
}

    procedure   TModSlowToFast.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in , 'in' );
      AddOutput( o_out, 'out');
    end;


    procedure   TModSlowToFast.DoSlowTick; // override;
    // DoSlowTick executes after DoTick
    begin
      FoldVal := FNewVal;
      FNewVal := FInputs[ i_in];
      FCount  := 0;
    end;


    procedure   TModSlowToFast.DoTick; // override;
    // DoTick executes before DoSlowTick
    begin
      FOutputs[ o_out] := ( FCount * FNewVal + ( Control_Decimation - FCount) * FOldVal) / Control_Decimation;
      Inc( FCount);
    end;


{ ========
  TModArpeggiator = class( TMod)
  private
    FValues         : TSignalArray;
    FSorted         : TSignalArray;
    FToggled        : Boolean;
    FLength         : Integer;
    FSortOrder      : Integer;
    FMustSort       : Boolean;
    FInFilter       : Integer;
    FInPtr          : Integer;
    FOutPtr         : Integer;
    FOldInTrig      : Boolean;
    FOldOutTrig     : Boolean;
    FOldReset       : Boolean;
    FOldToggled     : Boolean;
    FOldClr         : Boolean;
    FMustClr        : Boolean;
    FInTrigCounter  : Integer;
    FOutTrigCounter : Integer;
    FResetCounter   : Integer;
    FClrCounter     : Integer;
    FPrevInValue    : TSignal;
    FTransparant    : Boolean;
    FPingPongState  : Boolean;
  private
}


    function    TModArpeggiator.CompareLoFirst( anItem1, anItem2: TSignal): Integer;
    begin
      if anItem1 < anItem2
      then Result := -1
      else if anItem1 = anItem2
      then Result :=  0
      else Result :=  1;
    end;


    function    TModArpeggiator.CompareHiFirst( anItem1, anItem2: TSignal): Integer;
    begin
      if anItem1 > anItem2
      then Result := -1
      else if anItem1 = anItem2
      then Result :=  0
      else Result :=  1;
    end;


    function    TModArpeggiator.Comparer( aSortMode: Integer): TKnobsValueCompare;
    begin
      if FToggled
      then begin
        aSortMode := aSortMode + 1;

        if aSortMode > so_random
        then aSortMode := so_unsorted;
      end;

      case aSortMode of
        so_hitolo : Result := CompareHiFirst;
        else        Result := CompareLoFirst;
      end;
    end;


    procedure   TModArpeggiator.SwapItems( anIndex1, anIndex2: Integer);
    var
      aTmpVal : TSignal;
    begin
      aTmpVal            := FSorted[ anIndex1];
      FSorted[ anIndex1] := FSorted[ anIndex2];
      FSorted[ anIndex2] := aTmpVal;
    end;


    procedure   TModArpeggiator.Sort( aLowBound, aHighBound: Integer; aComparer: TKnobsValueCompare);
    var
      i : Integer;
      j : Integer;
      p : Integer;
    begin
      repeat
        i := aLowBound;
        j := aHighBound;
        p := ( aLowBound + aHighBound) shr 1;

        repeat
          while aComparer( FSorted[ i], FSorted[ p]) < 0
          do Inc( i);

          while aComparer( FSorted[ j], FSorted[ p]) > 0
          do Dec( j);

          if i <= j
          then begin
            if i <> j
            then SwapItems( i, j);

            if p = i
            then p := j
            else if p = j
            then p := i;

            Inc( i);
            Dec( j);
          end;
        until i > j;

        if aLowBound < j
        then Sort( aLowBound, j, aComparer);

        aLowBound := i;
      until i >= aHighBound;
    end;


    procedure   TModArpeggiator.Sort; // overload;
    var
      i : Integer;
    begin
      if FSortOrder in [ so_lotohi, so_hitolo, so_pingpong]
      then Sort( 0, FLength - 1, Comparer( FSortOrder))
      else if FSortOrder = so_random
      then begin
        for i := 0 to ( FLength - 1) div 2
        do SwapItems( Random( FLength), Random( FLength));
      end
      else if FSortOrder = so_reversed
      then begin
        for i := 0 to ( FLength - 1) div 2
        do SwapItems( i, FLength - 1 - i);
      end
    end;


    procedure   TModArpeggiator.AppendValue( aValue: TSignal);
    begin
      if FLength > 0
      then begin
        FPrevInValue     := aValue;
        FValues[ FInPtr] := aValue;
        FInPtr := ( FInPtr + 1) mod FLength;
        Move( FValues[ 0], FSorted[ 0], FLength * SizeOf( FValues[ 0]));
      end;
    end;


    procedure   TModArpeggiator.AcceptNewValue( aValue: TSignal);
    var
      i      : Integer;
      Unique : Boolean;
    begin
      if FMustClr
      then begin
        AppendValue( 0);
        FMustClr := False;
      end
      else begin
        case FInFilter of

          if_changed :

            begin
              if aValue <> FValues[ FInPtr]
              then AppendValue( aValue);
            end;

          if_unique :

            begin
              Unique := True;

              for i := 0 to FLength - 1
              do begin
                if FValues[ i] = aValue
                then begin
                  Unique := False;
                  Break;
                end;
              end;

              if Unique
              then AppendValue( aValue);
            end;

          else { if_all } AppendValue( aValue);
        end;
      end;
    end;


//  public

    constructor TModArpeggiator.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      SetLength( FValues, 256);
      SetLength( FSorted, 256);
      FLength := 4;
    end;


    procedure   TModArpeggiator.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in       , 'in'       );
      AddInput ( i_newinput , 'newinput' );
      AddInput ( i_newoutput, 'newoutput');
      AddInput ( i_inmode   , 'inmode'   );
      AddInput ( i_sorting  , 'sorting'  );
      AddInput ( i_length   , 'length'   );
      AddInput ( i_toggle   , 'toggle'   );
      AddInput ( i_reset    , 'reset'    );
      AddInput ( i_clr      , 'clr'      );
      AddInput ( i_reverse  , 'reverse'  );
      AddInput ( i_trans    , 'trans'    );
      AddInput ( i_lengthmod, 'lengthmod');
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModArpeggiator.SetDefaults; // override;
    begin
      FInputs[ i_lengthmod] := 1.0;
    end;


    procedure   TModArpeggiator.DoSlowTick; // override;
    var
      InTrig  : Boolean;
      OutTrig : Boolean;
      Reset   : Boolean;
      Clr     : Boolean;
    begin
      Reset        := SignalToLogic( FInputs[ i_reset    ]);
      InTrig       := SignalToLogic( FInputs[ i_newinput ]);
      OutTrig      := SignalToLogic( FInputs[ i_newoutput]);
      Clr          := SignalToLogic( FInputs[ i_clr      ]);
      FToggled     := SignalToLogic( FInputs[ i_toggle   ]);
      FTransparant := SignalToLogic( FInputs[ i_trans    ]);
      FLength      := Clip( Round( FInputs[ i_length] * FInputs[ i_lengthmod]), 1, 255);
      FSortOrder   := Round( FInputs[ i_sorting]);
      FInFilter    := Round( FInputs[ i_inmode ]);

      if FOldToggled <> FToggled
      then begin
        FMustSort   := True;
        FOldToggled := FToggled;
      end;

      if ResetFlag or ( not FOldReset and Reset)
      then begin
        ResetFlag      := False;
        FOutPtr        := 0;
        FResetCounter  := LedFlashTime( IsSpedUp);
        FPingPongState := False;
      end;

      if not FOldClr and Clr
      then FMustClr := True;

      if FMustClr
      then FClrCounter := LedFlashTime( IsSpedUp);

      if not FOldInTrig and InTrig
      then begin
        AcceptNewValue( FInputs[ i_in]);
        FInTrigCounter := LedFlashTime( IsSpedUp);
        FMustSort      := True;
      end;

      if not FOldOutTrig and OutTrig
      then begin
        if FTransparant
        then FOutputs[ o_out] := FPrevInValue
        else begin
          if FMustSort
          then Sort;

          FOutputs[ o_out] := FSorted[ FOutPtr];

          if FSortOrder = so_pingpong
          then begin
            if SignalToLogic( FInputs[ i_reverse]) xor FPingPongState
            then begin
              FOutPtr := FOutPtr - 1;

              if FOutPtr < 0
              then begin
                FOutPtr        := 0;
                FPingPongState := not FPingPongState;
              end;
            end
            else begin
              FOutPtr := FOutPtr + 1;

              if FOutPtr >= FLength
              then begin
                FOutPtr        := FLength - 1;
                FPingPongState := not FPingPongState;
              end;
            end;
          end
          else begin
            if SignalToLogic( FInputs[ i_reverse])
            then begin
              FOutPtr := FOutPtr - 1;

              if FOutPtr < 0
              then FOutPtr := FLength - 1;
            end
            else begin
              FOutPtr := FOutPtr + 1;

              if FOutPtr >= FLength
              then FOutPtr := 0;
            end;
          end;

          if FSortOrder = so_random
          then FMustSort := True;
        end;

        FOutTrigCounter := LedFlashTime( IsSpedUp);
      end;

      DecToZero( FResetCounter  );
      DecToZero( FInTrigCounter );
      DecToZero( FOutTrigCounter);
      DecToZero( FClrCounter    );
      FOldReset   := Reset;
      FOldOutTrig := OutTrig;
      FOldInTrig  := InTrig;
      FOldClr     := Clr;
    end;


    procedure   TModArpeggiator.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'newinput' , LogicToSignal( FInTrigCounter  > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'newoutput', LogicToSignal( FOutTrigCounter > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'    , LogicToSignal( FResetCounter   > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'toggle'   , LogicToSignal( FToggled           )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clr'      , LogicToSignal( FClrCounter     > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trans'    , LogicToSignal( FTransparant       )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reverse'  , FInputs[ i_reverse                ]));
    end;


{ ========
  TModMarkov = class( TMod)
  private
    FMarkovChain      : TMarkovChain;
    FOrder            : Integer;
    FMaxSize          : Integer;
    FCyclic           : Boolean;
    FOverflow         : Boolean;
    FOldLearn         : Boolean;
    FOldNewOutput     : Boolean;
    FOldClear         : Boolean;
    FOldFull          : Boolean;
    FLearnCounter     : Integer;
    FNewOutputCounter : Integer;
    FClearCounter     : Integer;
    FFullCounter      : Integer;
  private
}

    procedure   TModMarkov.Recreate;
    var
      aNew  : TMarkovChain;
      anOld : TMarkovChain;
    begin
      if ( MaxSize > 0) and ( Order > 0)
      then aNew := TMarkovChain.Create( FCyclic, MaxSize, Order)
      else aNew := nil;

      Locked := True;

      try
        anOld := AtomicExchange( Pointer( FMarkovChain), Pointer( aNew));
      finally
        Locked := False;

        FreeAndNil( anOld);
      end;
    end;


    procedure   TModMarkov.SetMaxSize( aValue: Integer);
    begin
      aValue := Clip( aValue, 0, 256);

      if aValue <> FMaxSize
      then begin
        FMaxSize := aValue;
        Recreate;
      end;
    end;


    procedure   TModMarkov.SetOrder( aValue: Integer);
    begin
      aValue := Clip( aValue, 0, 5);

      if aValue <> FOrder
      then begin
        FOrder := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModMarkov.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      MaxSize := 32;
    end;


    procedure   TModMarkov.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in            , 'in'            );
      AddInput ( i_strength      , 'strength'      );
      AddInput ( i_strengthmod   , 'strengthmod'   );
      AddInput ( i_strengthmodamt, 'strengthmodamt');
      AddInput ( i_mode          , 'mode'          );
      AddInput ( i_learn         , 'learn'         );
      AddInput ( i_newoutput     , 'newoutput'     );
      AddInput ( i_clear         , 'clear'         );
      AddOutput( o_out           , 'out'           );
      AddOutput( o_full          , 'full'          );
    end;


    destructor  TModMarkov.Destroy; // override;
    begin
      MaxSize := 0;
      inherited;
    end;


    procedure   TModMarkov.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'maxsize')
      then MaxSize := Round( aValue)
      else if SameText( aName, 'order')
      then Order := Round( aValue)
      else inherited;
    end;


    function    TModMarkov.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'maxsize')
      then Result := MaxSize
      else if SameText( aName, 'order')
      then Result := Order
      else Result := inherited;
    end;


    procedure   TModMarkov.DoSlowTick; // override;
    var
      Learn     : Boolean;
      NewOutput : Boolean;
      Clear     : Boolean;
    begin
      if Assigned( FMarkovChain)
      then begin
        Clear     := SignalToLogic( FInputs[ i_clear    ]);
        Learn     := SignalToLogic( FInputs[ i_learn    ]);
        NewOutput := SignalToLogic( FInputs[ i_newoutput]);
        FOverflow := FMarkovChain.Overflow;

        if not FOldClear and Clear
        then begin
          FMarkovChain.Clear;
          FClearCounter := LedFlashTime( IsSpedUp);
        end;

        if not FOldLearn and Learn
        then begin
          FMarkovChain.Cyclic := Round( FInputs[ i_mode]) = 1;
          FMarkovChain.Learn( FInputs[ i_in], Clip( FInputs[ i_strength] + FInputs[ i_strengthmod] * FInputs[ i_strengthmodamt], -1.0, + 1.0));
          FLearnCounter := LedFlashTime( IsSpedUp);
        end;

        if not FOldNewOutput and NewOutput
        then begin
          FOutputs[ o_out]  := FMarkovChain.NextValue;
          FNewOutputCounter := LedFlashTime( IsSpedUp);
        end;

        if not FOldFull and FOverflow
        then FFullCounter := LedFlashTime( IsSpedUp);

        FOldClear     := Clear;
        FOldLearn     := Learn;
        FOldNewOutput := NewOutput;
        FOldFull      := FOverflow;
        FOutputs[ o_full] := LogicToSignal( FOverflow);
      end;

      DecToZero( FClearCounter    );
      DecToZero( FLearnCounter    );
      DecToZero( FNewOutputCounter);
      DecToZero( FFullCounter     );
    end;


    procedure   TModMarkov.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'clear'     , LogicToSignal( FClearCounter     > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'learn'     , LogicToSignal( FLearnCounter     > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'newoutput' , LogicToSignal( FNewOutputCounter > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'full'      , LogicToSignal( FFullCounter      > 0)));
    end;


{ ========
  TModAmMod = class( TMod)
    i_mod     = 1;
    i_modamt  = 2;
    i_offset  = 3;
    i_modtype = 4;
  public
}

    procedure   TModAmMod.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in      , 'in'      );
      AddInput ( i_mod     , 'mod'     );
      AddInput ( i_modamt  , 'modamt'  );
      AddInput ( i_offset  , 'offset'  );
      AddInput ( i_modtype , 'modtype' );
      AddInput ( i_amrm    , 'amrm'    );
      AddInput ( i_fxmod   , 'fxmod'   );
      AddInput ( i_fxmodamt, 'fxmodamt');
      AddOutput( o_out     , 'out'     );
    end;


    procedure   TModAmMod.DoSlowTick; // override;
    var
      RM  : TSignal;
      AM  : TSignal;
      FX  : TSignal;
      Mix : TSignal;
    begin
      RM               := SignalForInputType( Normalize( FInputs[ i_mod]), Round( FInputs[ i_modtype]));
      AM               := SignalForOutputType( RM, outputtype_positive);
      FX               := Clip( FInputs[ i_amrm] + Normalize( FInputs[ i_fxmod] * FInputs[ i_fxmodamt]), 0.0, 1.0);
      Mix              := FX * ( RM - AM) + AM;
      FOutputs[ o_out] := Normalize( FInputs[ i_in]) * ( FInputs[ i_offset] + Mix * FInputs[ i_modamt]);
    end;


{ ========
  TModModControl = class( TMod)
  public
}

    procedure   TModModControl.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in     , 'in'     );
      AddInput ( i_mod    , 'mod'    );
      AddInput ( i_intype , 'intype' );
      AddInput ( i_modtype, 'modtype');
      AddInput ( i_outtype, 'outtype');
      AddOutput( o_out    , 'out'    );
    end;


    procedure   TModModControl.DoSlowTick; // override;
    var
      Signal     : TSignal;
      Modulation : TSignal;
      OutValue   : TSignal;
    begin
      Signal           := SignalForOutputType( SignalForInputType( FInputs[ i_in ], Round( FInputs[ i_intype ])), outputtype_positive);
      Modulation       := SignalForOutputType( SignalForInputType( FInputs[ i_mod], Round( FInputs[ i_modtype])), outputtype_positive);
      OutValue         := Modulation * ( Signal - 1) + 1;
      FOutputs[ o_out] := SignalForInputType( SignalForOutputType( OutValue, outputtype_positive), Round( FInputs[ i_outtype]));
    end;


{ ========
  TModShaper = class( TMod)
  public
}

    procedure   TModShaper.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in     , 'in'     );
      AddInput ( i_intype , 'intype' );
      AddInput ( i_shape  , 'shape'  );
      AddInput ( i_outtype, 'outtype');
      AddOutput( o_out    , 'out'    );
    end;


    procedure   TModShaper.DoSlowTick; // override;
    var
      Inp  : TSignal;
      Outp : TSignal;
    begin
      Inp := SignalForOutputType( SignalForInputType( FInputs[ i_in], Round( FInputs[ i_intype])), outputtype_positive);

      case SignalToEnvShape( FInputs[ i_shape]) of
        esLog : Outp := LogAttack( Inp, 1.0, 0.0, 1.0);
        esLin : Outp := LinAttack( Inp, 1.0, 0.0, 1.0);
        esExp : Outp := ExpAttack( Inp, 1.0, 0.0, 1.0);
        else    Outp := SAttack  ( Inp, 1.0, 0.0, 1.0);
      end;

      FOutputs[ o_out] := SignalForOutputType( SignalForInputType( Outp, inputtype_positive), Round( FInputs[ i_outtype]));
    end;


{ ========
  TModAnalogXor = class( TMod)
    i_ina       = 0;
    i_inb       = 1;
    i_intypea   = 2;
    i_intypeb   = 3;
    i_outtype   = 4;
    i_threshold = 5;
    i_tmod      = 6;
    i_tmodamt   = 7;
  public
}

    procedure   TModAnalogXor.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_ina      , 'ina'      );
      AddInput ( i_inb      , 'inb'      );
      AddInput ( i_intypea  , 'intypea'  );
      AddInput ( i_intypeb  , 'intypeb'  );
      AddInput ( i_outtype  , 'outtype'  );
      AddInput ( i_threshold, 'threshold');
      AddInput ( i_tmod     , 'tmod'     );
      AddInput ( i_tmodamt  , 'tmodamt'  );
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModAnalogXor.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_threshold
      ];
    end;


    procedure   TModAnalogXor.DoSlowTick; // override;
    var
      Threshold : TSignal;
      SumValue  : TSignal;
      OutVal    : Boolean;
    begin
      Threshold := FInputs[ i_threshold] + 2 * FInputs[ i_tmod] * FInputs[ i_tmodamt];
      SumValue  :=
        SignalForOutputType( SignalForInputType( FInputs[ i_ina], Round( FInputs[ i_intypea])), outputtype_normal) +
        SignalForOutputType( SignalForInputType( FInputs[ i_inb], Round( FInputs[ i_intypeb])), outputtype_normal);
      OutVal           := ( SumValue >= Threshold) and ( SumValue <= 0.5);
      FOutputs[ o_out] := SignalForOutputType( LogicToSignal( OutVal), Round( FInputs[ i_outtype]));
    end;


{ ========
  TModLogisticMap = class( TMod)
  private
    FOldStart   : Boolean;
    FOldClock   : Boolean;
    FTicks      : Integer;
    FPeriodTime : Integer;
    FOutValue   : TSignal;
    FNextValue  : TSignal;
  public
 }

    procedure   TModLogisticMap.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_clock  , 'clock'  );
      AddInput ( i_start  , 'start'  );
      AddInput ( i_smod   , 'smod'   );
      AddInput ( i_smodamt, 'smodamt');
      AddInput ( i_svalue , 'svalue' );
      AddInput ( i_rmod   , 'rmod'   );
      AddInput ( i_rmodamt, 'rmodamt');
      AddInput ( i_lowr   , 'lowr'   );
      AddInput ( i_highr  , 'highr'  );
      AddInput ( i_xfade  , 'xfade'  );
      AddInput ( i_outtype, 'outtype');
      AddInput ( i_mute   , 'mute'   );
      AddOutput( o_out    , 'out'    );
    end;


    procedure   TModLogisticMap.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_smodamt,
        i_svalue ,
        i_rmodamt,
        i_lowr   ,
        i_highr  ,
        i_xfade
      ];
      FOldStart := False;
      FOldClock := False;
    end;


    procedure   TModLogisticMap.DoSlowTick;  // override;
    var
      Clock  : Boolean;
      Start  : Boolean;
      XFade  : TSignal;
      R      : TSignal;
      OutVal : TSignal;
    begin
      Start := SignalToLogic( FInputs[ i_start]);

      if ( not FOldStart and Start) or ResetFlag
      then begin
        FNextValue  := Clip( FInputs[ i_svalue] + FInputs[ i_smodamt] * FInputs[ i_smod], 0.0, 1.0);
        FPeriodTime := Max( 2, FTicks);
        FTicks      := 0;
      end;

      ResetFlag := False;
      FOldStart := Start;
      Clock     := SignalToLogic( FInputs[ i_clock]);

      if not FOldClock and Clock
      then begin
        FPeriodTime := Max( 2, FTicks);
        FTicks      := 0;
        XFade       := Clip( FInputs[ i_rmod] * FInputs[ i_rmodamt], 0.0, 1.0);
        R           := FInputs[ i_lowr] * ( 1 - XFade) + FInputs[ i_highr] * XFade;
        FOutValue   := FNextValue;
        FNextValue  := Normalize( R * FOutValue * ( 1.0 - FOutValue));
        Nop;
      end
      else if FOldClock and not Clock
      then FPeriodTime := FTicks * 2;

      XFade := Clip((( FPeriodTime - FTicks) / FPeriodTime) * FInputs[ i_xfade], -1.0, 1.0);

      if FOutValue < FNextValue
      then OutVal := 2 * LinDecay ( XFade, 1, FNextValue, FOutValue) - 1  // going up, note that as final and current value are reversed this must call decay ...
      else OutVal := 2 * LinAttack( XFade, 1, FNextValue, FOutValue) - 1; // going down

      if FTicks < MAXINT shr 2    // Cap FTicks value in case the module gets not clocked prolly MAXINT shr 1 would be OK too
      then Inc( FTicks);

      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( OutVal, Round( FInputs[ i_outtype]));
      FOldClock        := Clock;
    end;


{ ========
  TModRotator = class( TMod)
  public
}

    procedure   TModRotator.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_angle , 'angle' );
      AddInput ( i_in1   , 'in1'   );
      AddInput ( i_in2   , 'in2'   );
      AddInput ( i_sctype, 'sctype');
      AddInput ( i_mode  , 'mode'  );
      AddOutput( o_out1  , 'out1'  );
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_sin   , 'sin'   );
      AddOutput( o_cos   , 'cos'   );
    end;


    procedure   TModRotator.SetDefaults; // override;
    begin
      FInputs[ i_angle] := 0;
      FInputs[ i_in1  ] := 0;
      FInputs[ i_in2  ] := 0;
    end;


    procedure   TModRotator.DoSlowTick; // override;
    var
      aCos    : TSignal;
      aSin    : TSignal;
      in1     : TSignal;
      in2     : TSignal;
      aType   : Integer;
      aMode   : Integer;
    begin
      FAngle := FInputs[ i_angle] * TWO_PI;
      SinCos( FAngle, aSin, aCos);
      in1   := FInputs[ i_in1];
      in2   := FInputs[ i_in2];
      aType := Round( FInputs[ i_sctype]);
      aMode := Round( FInputs[ i_mode  ]);

      if aMode = 0
      then begin
        FOutputs[ o_sin ] := SignalForOutputType( aSin, aType);
        FOutputs[ o_cos ] := SignalForOutputType( aCos, aType);
        FOutputs[ o_out1] := aCos * in1 - aSin * in2;
        FOutputs[ o_out2] := aSin * in1 + aCos * in2;
      end
      else begin
        FOutputs[ o_sin ] := SignalForOutputType(   aSin, aType);
        FOutputs[ o_cos ] := SignalForOutputType( - aSin, aType);
        FOutputs[ o_out1] := ( 1 - aSin) * in1;
        FOutputs[ o_out2] := ( 1 + aSin) * in2;
      end;
    end;


{ ========
  TModRotator2 = class( TMod)
  public
}

    procedure   TModRotator2.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_angle, 'angle');
      AddInput ( i_in1  , 'in1'  );
      AddInput ( i_in2  , 'in2'  );
      AddOutput( o_out1 , 'out1' );
      AddOutput( o_out2 , 'out2' );
    end;


    procedure   TModRotator2.SetDefaults; // override;
    begin
      FInputs[ i_angle] := 0;
      FInputs[ i_in1  ] := 0;
      FInputs[ i_in2  ] := 0;
    end;


    procedure   TModRotator2.DoSlowTick; // override;
    var
      anAngle : TSignal;
      aCos    : TSignal;
      aSin    : TSignal;
      in1     : TSignal;
      in2     : TSignal;
    begin
      anAngle := FInputs[ i_angle] * TWO_PI;
      SinCos( anAngle, aSin, aCos);
      in1 := FInputs[ i_in1];
      in2 := FInputs[ i_in2];

      FOutputs[ o_out1] := aCos * in1 - aSin * in2;
      FOutputs[ o_out2] := aSin * in1 + aCos * in2;
    end;


{ ========
  TModMatrixMult = class( TMod)
  strict private
  const
    i_in1    = 0;
    i_in2    = 1;
    i_ina    = 2;
    i_inb    = 3;
    i_inc    = 4;
    i_ind    = 5;
  const
    o_out1   = 0;
    o_out2   = 1;
  public
}

    procedure   TModMatrixMult.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in1 , 'in1' );
      AddInput ( i_in2 , 'in2' );
      AddInput ( i_ina , 'ina' );
      AddInput ( i_inb , 'inb' );
      AddInput ( i_inc , 'inc' );
      AddInput ( i_ind , 'ind' );
      AddOutput( o_out1, 'out1');
      AddOutput( o_out2, 'out2');
    end;


    procedure   TModMatrixMult.SetDefaults; // override;
    begin
      FInputs[ i_ina] := 1.0; // Set identity matrix as the default value
      FInputs[ i_inb] := 0.0;
      FInputs[ i_inc] := 0.0;
      FInputs[ i_ind] := 1.0;
    end;


    procedure   TModMatrixMult.DoSlowTick; // override;
    var
      in1 : TSignal;
      in2 : TSignal;
    begin
      in1               := FInputs[ i_in1];
      in2               := FInputs[ i_in2];
      FOutputs[ o_out1] := FInputs[ i_ina] * in1 + FInputs[ i_inb] * in2;
      FOutputs[ o_out2] := FInputs[ i_inc] * in2 + FInputs[ i_ind] * in2;
    end;


{ ========
  TModRectifier = class( TMod)
  public
}

    procedure   TModRectifier.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_mode  , 'mode'  );
      AddInput ( i_in    , 'in'    );
      AddOutput( o_out   , 'out'   );
      AddOutput( o_outinv, 'outinv');
    end;


    procedure   TModRectifier.SetDefaults; // override;
    begin
      FInputs[ i_in] := 0;
    end;


    procedure   TModRectifier.DoSlowTick; // override;
    var
      Mode : Integer;
      Inp  : TSignal;
    begin
      Mode := Round( FInputs[ i_mode]);
      Inp  :=        FInputs[ i_in  ] ;

      if Mode = 0                                      // Half wave
      then begin
        if SignalToLogic( Inp)
        then FOutputs[ o_out] := Inp
        else FOutputs[ o_out] := 0;
      end
      else if Mode = 1                                 // Full wave
      then FOutputs[ o_out] := Abs( Inp)
      else FOutputs[ o_out] :=      Inp;      // transparent

      FOutputs[ o_outinv] := - FOutputs[ o_out];
    end;


{ ========
  TModMinMax = class( TMod)
  public
}

    procedure   TModMinMax.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in1, 'in1');
      AddInput ( i_in2, 'in2');
      AddOutput( o_min, 'min');
      AddOutput( o_max, 'max');
    end;


    procedure   TModMinMax.SetDefaults; // override;
    begin
      FInputs[ i_in1] := 0;
      FInputs[ i_in2] := 0;
    end;


    procedure   TModMinMax.DoSlowTick; // override;
    var
      in1 : TSignal;
      in2 : TSignal;
    begin
      in1              := FInputs[ i_in1];
      in2              := FInputs[ i_in2];
      FOutputs[ o_min] := Min( in1, in2);
      FOutputs[ o_max] := Max( in1, in2);
    end;


{ ========
  TModWavolver = class( TMod)
  public
}

    procedure   TModWavolver.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in            , 'in'            );
      AddInput ( i_inlevel       , 'inlevel'       );
      AddInput ( i_shape         , 'shape'         );
      AddInput ( i_signal        , 'signal'        );
      AddInput ( i_shapelevel    , 'shapelevel'    );
      AddInput ( i_shapeoffset   , 'shapeoffset'   );
      AddInput ( i_signallevel   , 'signallevel'   );
      AddInput ( i_signaloffset  , 'signaloffset'  );
      AddInput ( i_pulseamp      , 'pulseamp'      );
      AddInput ( i_pulseampmod   , 'pulseampmod'   );
      AddInput ( i_pulseampmodlev, 'pulseampmodlev');
      AddInput ( i_mute          , 'mute'          );
      AddOutput( o_out           , 'out'           );
      AddOutput( o_inactive      , 'inactive'      );
    end;


    procedure   TModWavolver.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_shapelevel    ,
        i_shapeoffset   ,
        i_signallevel   ,
        i_signaloffset  ,
        i_pulseamp      ,
        i_pulseampmodlev,
        i_inlevel
      ];
    end;


    procedure   TModWavolver.DoSlowTick; // override;
    var
      aShape   : TSignal;
      anAbsVal : TSignal;
      aSignal  : TSignal;
      anOutVal : TSignal;
      anAmp    : TSignal;
    begin
      aShape   := FInputs[ i_shape ] * FInputs[ i_shapelevel ] + FInputs[ i_shapeoffset ];
      aSignal  := FInputs[ i_inlevel] * FInputs[ i_in] + FInputs[ i_signal] * FInputs[ i_signallevel] + FInputs[ i_signaloffset];
      anAbsVal := Abs( aSignal);
      anAmp    := FInputs[ i_pulseamp] + FInputs[ i_pulseampmod] * FInputs[ i_pulseampmodlev];

      if anAbsVal > aShape
      then begin
        if aSignal < 0
        then anOutVal := anAmp * aSignal
        else anOutVal :=         aSignal;

        FOutputs[ o_inactive] := LogicToSignal( False);
      end
      else begin
        anOutVal              := 0;
        FOutputs[ o_inactive] := LogicToSignal( True);
      end;

      FOutputs[ o_out] := anOutVal * SignalToMute( FInputs[ i_mute]);
    end;


{ ========
  TModWaveWiper = class( TMod)
  public
}

    procedure   TModWaveWiper.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_break, 'break');
      AddInput ( i_mute , 'mute' );
      AddInput ( i_in1  , 'in1'  );
      AddInput ( i_in2  , 'in2'  );
      AddOutput( o_out1 , 'out1' );
      AddOutput( o_out2 , 'out2' );
    end;


    procedure   TModWaveWiper.SetDefaults; // override;
    begin
      FInputs[ i_in1] := 0;
      FInputs[ i_in2] := 0;
      FDezipperMap := [
      ];
    end;


    procedure   TModWaveWiper.DoSlowTick; // override;
    var
      Insig1   : TSignal;
      Insig2   : TSignal;
      BreakSig : TSignal;
      Mute     : TSignal;
    begin
      InSig1   := FInputs[ i_in1  ];
      InSig2   := FInputs[ i_in2  ];
      BreakSig := FInputs[ i_break];
      Mute     := SignalToMute( FInputs[ i_mute]);
      FOutPuts[ o_out1] := ( Max( Insig1, BreakSig) + Min( InSig2, BreakSig) - BreakSig) * Mute;
      FOutPuts[ o_out2] := ( Max( Insig2, BreakSig) + Min( Insig1, BreakSig) - BreakSig) * Mute;
    end;


{ ========
  TModWaveWrapper = class( TMod)
  private
    FPrevValue  : TSignal;
    FOutValue   : TSignal;
    FStateValue : TSignal;
    FGoingUp    : Boolean;
  public
}

    procedure   TModWaveWrapper.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in       , 'in'       );
      AddInput ( i_mute     , 'mute'     );
      AddInput ( i_lowlevel , 'lowlevel' );
      AddInput ( i_highlevel, 'highlevel');
      AddInput ( i_mod      , 'mod'      );
      AddInput ( i_modamt   , 'modamt'   );
      AddInput ( i_type     , 'type'     );
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModWaveWrapper.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_lowlevel ,
        i_highlevel,
        i_modamt
      ];
    end;


    procedure   TModWaveWrapper.DoSlowTick; // override;
    var
      Modulation  : TSignal;
      LoMirror    : TSignal;
      MirrorDelta : TSignal;
      Amp         : TSignal;
      Mute        : TSignal;
      Signal      : TSignal;
    var
      Difference  : TSignal;
      Invalue     : TSignal;
      LowMirror   : TSignal;
      HighMirror  : TSignal;
    begin
      case Round( FInputs[ i_type]) of

        m_type2 : begin
          // Source :: https://www.muffwiggler.com/forum/viewtopic.php?p=1630760
          InValue      := FInputs[ i_in];
          Modulation   := Normalize( FInputs[ i_mod] * FInputs[ i_modamt]);         // Modulation amount
          LowMirror    := FInputs[ i_lowlevel ] - Modulation;                       // Get low  mirror and subtract modulation
          HighMirror   := FInputs[ i_highlevel] + Modulation;                       // Get high mirror and add      modulation
          Difference   := InValue - FPrevValue;
          FPrevValue   := InValue;

          if FGoingUp
          then FOutValue := FOutValue + Difference
          else FOutValue := FOutValue - Difference;

          if FOutValue > HighMirror
          then begin
            FGoingUp  := not FGoingUp;
            FOutValue := 2.0 * HighMirror - FOutValue;
          end
          else if FOutValue < LowMirror
          then begin
            FGoingUp  := not FGoingUp;
            FOutValue := 2.0 * LowMirror - FOutValue;
          end;
          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * FOutValue;
        end;

        m_type3 : begin
          // Source :: https://www.muffwiggler.com/forum/viewtopic.php?p=1630760 - my take at the smrl variation.
          InValue      := FInputs[ i_in];
          Modulation   := Normalize( FInputs[ i_mod] + FInputs[ i_modamt]);         // Modulation amount
          LowMirror    := Normalize( FInputs[ i_lowlevel ]);                        // Get low  mirror
          HighMirror   := Normalize( FInputs[ i_highlevel]);                        // Get high mirror
          Difference   := Normalize( InValue - FPrevValue);
          FPrevValue   := InValue;

          if FGoingUp
          then FStateValue := Normalize( FStateValue + Difference)
          else FStateValue := Normalize( FStateValue - Difference);

          if FStateValue > HighMirror
          then begin
            FGoingUp    := not FGoingUp;
            FStateValue := 2.0 * HighMirror - FStateValue;
            FOutValue   := FOutValue + Modulation;

            if FOutValue > 1
            then FOutValue := -1;
          end
          else if FStateValue < LowMirror
          then begin
            FGoingUp    := not FGoingUp;
            FStateValue := 2.0 * LowMirror - FStateValue;
            FOutValue   := FOutValue + Modulation;

            if FOutValue > 1
            then FOutValue := -1;
          end;

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * FOutValue;
        end;

        else begin // m_type1
          Modulation  := Normalize( FInputs[ i_mod] * FInputs[ i_modamt]);         // Modulation amount
          LoMirror    := FInputs[ i_lowlevel ] - Modulation;                       // Get low mirror and subtract modulatio
          MirrorDelta := 2.0 * ( FInputs[ i_Highlevel] + Modulation - LoMirror);   // High mirror gets modulation added. calc mirror distance, times 2
          Mute        := SignalToMute( FInputs[ i_mute]);                          // Get Mute multiplier

          if Abs( MirrorDelta) < 1e-6                                              // When mirrors are too close to each other
          then FOutputs[ o_out] := 0                                               // make an exception ...
          else begin
            Amp    := 4.0 / MirrorDelta;                                           // Amplitude reduction factor due to wrapping - times two
            Signal := MathFloatMod( FInputs[ i_in] + LoMirror, MirrorDelta);       // Lift signal by LoMirror, and take signal modulo 2 * Delta

            if Signal > 0.5 * MirrorDelta                                          // Reflect bit above Delta back
            then Signal := MirrorDelta - Signal;

            FOutputs[ o_out] := Normalize(( Amp * Signal - 1.0) * Mute);           // Bring signal into -1, 1 range and apply Mute signal
          end;
        end;
      end;
    end;


{ ========
  TModIntDif = class( TMod)
  private
    FSum        : TSignal;
    FPrevSample : TSignal;
    FMode       : Integer;
  public
}

    procedure   TModIntDif.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in     , 'in'     );
      AddInput ( i_mode   , 'mode'   );
      AddInput ( i_gain   , 'gain'   );
      AddInput ( i_modemod, 'modemod');
      AddOutput( o_int    , 'int'    );
      AddOutput( o_dif    , 'dif'    );
    end;


    procedure   TModIntDif.SetDefaults; // override;
    begin
      FInputs[ i_in] := 0;
    end;


    procedure   TModIntDif.DoSlowTick; // override;
    var
      Inp  : TSignal;
      Dif  : TSignal;
      Gain : TSignal;
    begin
      Inp  := FInputs[ i_in  ];

      if IsFast
      then Gain := FInputs[ i_gain] * System_Rate_Rec
      else Gain := FInputs[ i_gain] * Control_Rate_Rec;

      if ResetFlag
      then begin
        ResetFlag   := False;
        FSum        := Inp;
        FPrevSample := Inp;
      end
      else FSum := Clip( FSum + Gain * Inp, -10, 10);

      if IsFast
      then Gain := FInputs[ i_gain] * System_Rate
      else Gain := FInputs[ i_gain] * Control_Rate;

      Dif   := Clip( Gain * Normalize( Inp - FPrevSample), -10, 10);
      FMode := MathIntMod( Round( FInputs[ i_mode]) + Round( 3 * FInputs[ i_modemod]), 3);

      if FMode = 0                               // Half wave rectification of dif signal
      then begin
        if SignalToLogic( Dif)
        then FOutputs[ o_dif] := Dif
        else FOutputs[ o_dif] := 0;
      end
      else if FMode = 1                           // Full wave rectification of dif signal
      then FOutputs[ o_dif] := Abs( Dif)
      else FOutputs[ o_dif] :=      Dif;         // Transparent

      FOutputs[ o_int] := FSum;
      FPrevSample      := Inp;
    end;


    procedure   TModIntDif.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'mode', Converters.ValueToDisplay( 'Rectifier mode', FMode, 3)));
    end;


{ ========
  TModDif = class( TMod)
  private
    FPrevSample : TSignal;
  public
}

    procedure   TModDif.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in  , 'in'  );
      AddInput ( i_mode, 'mode');
      AddOutput( o_out , 'out' );
    end;


    procedure   TModDif.SetDefaults; // override;
    begin
      FInputs[ i_in] := 0;
    end;


    procedure   TModDif.DoSlowTick; // override;
    var
      Inp  : TSignal;
      Dif  : TSignal;
      Mode : Integer;
    begin
      Inp := FInputs[ i_in];

      if ResetFlag
      then begin
        ResetFlag   := False;
        FPrevSample := Inp;
      end;

      Dif  := Normalize( Inp - FPrevSample);
      Mode := Round( FInputs[ i_mode]);

      if Mode = 0                                // Half wave rectification of dif signal
      then FOutputs[ o_out] := Clip( Dif, 0, Abs( Dif))
      else if Mode = 1                           // Full wave rectification of dif signal
      then FOutputs[ o_out] := Abs( Dif)
      else FOutputs[ o_out] :=      Dif;         // Transparent

      FPrevSample := Inp;
    end;


{ ========
  TModChangeDetector = class( TMod)
  private
    FPrevSample   : TSignal;
    FTrigCount    : Integer;
    FTrigLedCount : Integer;
  public
}

    procedure   TModChangeDetector.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in  , 'in'  );
      AddInput ( i_mode, 'mode');
      AddOutput( o_out , 'out' );
      AddOutput( o_trig, 'trig');
    end;


    procedure   TModChangeDetector.SetDefaults; // override;
    begin
      FInputs[ i_in] := 0;
    end;


    procedure   TModChangeDetector.DoSlowTick; // override;
    var
      Inp  : TSignal;
      Dif  : TSignal;
      Mode : Integer;
    begin
      Inp := FInputs[ i_in];

      if ResetFlag
      then begin
        ResetFlag   := False;
        FPrevSample := Inp;
      end;

      Dif  := Normalize( Inp - FPrevSample);
      Mode := Round( FInputs[ i_mode]);

      if Dif <> 0
      then begin
        case Mode of
           0  : FOutputs[ o_out] :=      Dif;
           else FOutputs[ o_out] := Abs( Dif);
        end;

        FTrigCount    := TrigPulseTime( IsSpedUp);
        FTrigLedCount := LedFlashTime ( IsSpedUp);
      end;

      FOutputs[ o_trig] := LogicToSignal(FTrigCount > 0);

      DecToZero( FTrigCount   );
      DecToZero( FTrigLedCount);
      FPrevSample := Inp;
    end;


    procedure   TModChangeDetector.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig' , LogicToSignal( FTrigLedCount > 0)));
    end;


{ ========
  TModPhasor = class( TMod)
  private
    FPhase      : TSignal;
    FPhaseDelta : TSignal;
    FPosition   : TSignal;
    FOldSync    : Boolean;
    FShape      : Integer;
  public
}

    procedure   TModPhasor.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_frequency, 'frequency');
      AddInput ( i_fm       , 'fm'       );
      AddInput ( i_phase    , 'phase'    );
      AddInput ( i_sync     , 'sync'     );
      AddInput ( i_pmlevel  , 'pmlevel'  );
      AddInput ( i_fmlevel  , 'fmlevel'  );
      AddInput ( i_cents    , 'cents'    );
      AddInput ( i_freq     , 'freq'     );
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModPhasor.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_pmlevel,
        i_fmlevel,
        i_cents
      ];
    end;


    procedure   TModPhasor.DoTick; // override;
    begin
      if ( not FOldSync) and SignalToLogic( FInputs[ i_sync])
      then FPhase  := 0.0;

      FOldSync    := SignalToLogic( FInputs[ i_sync]);
      FPhaseDelta := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents]);
      FPhase      := MathFloatMod( FPhase + FPhaseDelta, 1.0);
      FPosition   := MathFloatMod( FPhase + Clip( FInputs[ i_phase] * FInputs[ i_pmlevel], -1, 1), 1.0);
      FOutputs[ o_out] := FPosition;
    end;


{ ========
  TModBaseOsc = class( TMod)
  private
    FSpeed      : TSignal;
    FProgress   : TSignal;
    FBreakPoint : TSignal;
    FPhase      : TSignal;
    FPhaseDelta : TSignal;
    FPosition   : TSignal;
    FOldSync    : Boolean;
    FShape      : Integer;
    FBandLimit  : TSignal;
  public
}

    procedure   TModBaseOsc.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_phase     , 'phase'     );
      AddInput ( i_sync      , 'sync'      );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_shape     , 'shape'     );
      AddInput ( i_pmlevel   , 'pmlevel'   );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_cents     , 'cents'     );
      AddInput ( i_bandlimit , 'bandlimit' );
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_speed     , 'speed'     );
      AddInput ( i_warp      , 'warp'      );
      AddInput ( i_warpmod   , 'warpmod'   );
      AddInput ( i_warpmodamt, 'warpmodamt');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModBaseOsc.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency ,
        i_pmlevel   ,
        i_fmlevel   ,
        i_cents     ,
        i_warp      ,
        i_warpmodamt
      ];
      FInputs[ i_bandlimit] := 3.0;
      FInputs[ i_speed    ] := 0.5;
      FInputs[ i_warp     ] := 0.5;
      FBreakPoint           := 0.5;
    end;


    procedure   TModBaseOsc.DoTick; // override;
    var
      NewSync  : Boolean;
      EffPhase : TSignal;
    begin
      NewSync := SignalToLogic( FInputs[ i_sync]);

      if ( not FOldSync) and NewSync
      then FPhase  := 0.0;

      FSpeed      := Normalize( FInputs[ i_speed]) * 2;
      FBreakPoint := Clip( FInputs[ i_warp] + FInputs[ i_warpmod] * FInputs[ i_warpmodamt], 0.001, 0.999);
      FPhaseDelta := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents]);
      FPhase      := MathFloatMod( FPhase + FSpeed * FPhaseDelta, 1.0);
      EffPhase    := MathFloatMod( FPhase + FInputs[ i_phase] * FInputs[ i_pmlevel], 1.0);
      FShape      := Round( FInputs[ i_shape    ]);
      FBandLimit  :=        FInputs[ i_bandlimit];
      FOldSync    := NewSync;

      if EffPhase < FBreakPoint
      then FPosition := 0.5 * EffPhase / FBreakPoint
      else FPosition := ( 0.5 * EffPhase + 0.5 - FBreakPoint) / ( 1.0 - FBreakPoint);
    end;


{ ========
  TModModOsc = class( TMod)
  private
    FPhase   : TSignal;
    FOldSync : Boolean;
  public
}

    procedure   TModModOsc.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_sync       , 'sync'       );
      AddInput ( i_ammod      , 'ammod'      );
      AddInput ( i_ammodlev   , 'ammodlev'   );
      AddInput ( i_amplitude  , 'amplitude'  );
      AddInput ( i_centsmod   , 'centsmod'   );
      AddInput ( i_centsmodlev, 'centsmodlev');
      AddInput ( i_cents      , 'cents'      );
      AddInput ( i_pmmod      , 'pmmod'      );
      AddInput ( i_pmmodlev   , 'pmmodlev'   );
      AddInput ( i_phase      , 'phase'      );
      AddInput ( i_freq       , 'freq'       );
      AddInput ( i_fmmod      , 'fmmod'      );
      AddInput ( i_fmmodlev   , 'fmmodlev'   );
      AddInput ( i_frequency  , 'frequency'  );
      AddInput ( i_speed      , 'speed'      );
      AddInput ( i_lfmmod     , 'lfmmod'     );
      AddInput ( i_lfmmodlev  , 'lfmmodlev'  );
      AddInput ( i_lfm        , 'lfm'        );
      AddInput ( i_mute       , 'mute'       );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModModOsc.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_ammodlev    ,
        i_amplitude   ,
        i_centsmodlev ,
        i_cents       ,
        i_pmmodlev    ,
        i_phase       ,
        i_fmmodlev    ,
        i_frequency   ,
        i_lfmmodlev   ,
        i_lfm
      ];
    end;


    procedure   TModModOsc.DoTick; // override;
    const
      OTSF  = 1.0 / 256.0;             // Quarter note positive
    var
      Sync       : Boolean;
      PhaseDelta : TSignal;
      Cents      : TSignal;
      Speed      : TSignal;
      Position   : TSignal;
      Amplitude  : TSignal;
    begin
      Sync := SignalToLogic( FInputs[ i_sync]);

      if ( not FOldSync) and Sync
      then FPhase  := 0.0;

      FOldSync         := Sync;
      Cents            := FInputs[ i_cents] + FInputs[ i_centsmod] * FInputs[ i_centsmodlev] * OTSF;
      Speed            := 2.0 * Normalize( FInputs[ i_speed] + FInputs[ i_lfmmod] * FInputs[ i_lfmmodlev] + FInputs[ i_lfm]);
      PhaseDelta       := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fmmod] * FInputs[ i_fmmodlev] + Cents);
      FPhase           := MathFloatMod( FPhase + Speed * PhaseDelta, 1.0);
      Position         := MathFloatMod( FPhase + FInputs[ i_phase] + FInputs[ i_pmmodlev] * FInputs[ i_pmmod], 1.0);
      Amplitude        := FInputs[ i_amplitude] + FInputs[ i_ammod] * FInputs[ i_ammodlev];
      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * Amplitude * LookupSine( Position);
    end;


{ ========
  TModPdOsc = class( TMod)
  private
    FPhaseMain   : TSignal;
    FPhaseFilter : TSignal;
    FOldSync     : Boolean;
  public
}

    procedure   TModPdOsc.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_mute       , 'mute'       );
      AddInput ( i_sync       , 'sync'       );
      AddInput ( i_cents      , 'cents'      );
      AddInput ( i_centsmod   , 'centsmod'   );
      AddInput ( i_centsmodlev, 'centsmodlev');
      AddInput ( i_freq       , 'freq'       );
      AddInput ( i_fmmod      , 'fmmod'      );
      AddInput ( i_fmmodlev   , 'fmmodlev'   );
      AddInput ( i_frequency  , 'frequency'  );
      AddInput ( i_speed      , 'speed'      );
      AddInput ( i_ratio      , 'ratio'      );
      AddInput ( i_ratiomod   , 'ratiomod'   );
      AddInput ( i_ratiomodlev, 'ratiomodlev');
      AddInput ( i_lfmmod     , 'lfmmod'     );
      AddInput ( i_lfmmodlev  , 'lfmmodlev'  );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModPdOsc.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_cents       ,
        i_centsmodlev ,
        i_centsmodlev ,
        i_fmmodlev    ,
        i_frequency   ,
        i_ratio       ,
        i_ratiomodlev ,
        i_lfmmodlev
      ];
      FInputs[ i_speed] := 0.5;
    end;


    procedure   TModPdOsc.DoTick; // override;
    const
      OTSF  = 1.0 / 256.0;             // Quarter note positive
    var
      Sync       : Boolean;
      PhaseDelta : TSignal;
      Cents      : TSignal;
      Speed      : TSignal;
      Ratio      : TSignal;
    begin
      Sync := SignalToLogic( FInputs[ i_sync]);

      if ( not FOldSync) and Sync
      then FPhaseMain   := 0.0;

      FOldSync     := Sync;
      Cents        := FInputs[ i_cents] + FInputs[ i_centsmod] * FInputs[ i_centsmodlev] * OTSF;
      Speed        := 2.0 * Normalize( FInputs[ i_speed] + FInputs[ i_lfmmod] * FInputs[ i_lfmmodlev]);
      Ratio        := 1.0 + 64.0 * Clip( FInputs[ i_ratio] + FInputs[ i_ratiomod] * FInputs[ i_ratiomodlev], 0.0, 1.0);
      PhaseDelta   := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fmmod] * FInputs[ i_fmmodlev] + Cents);
      FPhaseMain   := MathFloatMod( FPhaseMain + Speed * PhaseDelta, 1.0);
      FPhaseFilter := MathFloatMod( FPhaseMain * Ratio - 0.25, 1.0);
      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - FPhaseMain) * ( 1.0 + LookupSine( FPhaseFilter)) - 1.0);
    end;


{ ========
  TModMasterOsc = class( TMod)
  private
    FOldSync : Boolean;
  public
}

    procedure   TModMasterOsc.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_centsmod   , 'centsmod'   );
      AddInput ( i_centsmodamt, 'centsmodamt');
      AddInput ( i_cents      , 'cents'      );
      AddInput ( i_freq       , 'freq'       );
      AddInput ( i_fmmod      , 'fmmod'      );
      AddInput ( i_fmmodlev   , 'fmmodlev'   );
      AddInput ( i_frequency  , 'frequency'  );
      AddInput ( i_speed      , 'speed'      );
      AddInput ( i_lfmmod     , 'lfmmod'     );
      AddInput ( i_lfmmodlev  , 'lfmmodlev'  );
      AddInput ( i_lfm        , 'lfm'        );
      AddOutput( o_out        , 'out'        );
      AddOutput( o_outexp     , 'outexp'     );
    end;


    procedure   TModMasterOsc.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_centsmodamt ,
        i_cents       ,
        i_fmmodlev    ,
        i_frequency   ,
        i_lfmmodlev   ,
        i_lfm
      ];
      FInputs[ i_speed] := 0.5;
    end;


    procedure   TModMasterOsc.DoSlowTick; // override;
    const
      OTSF  = 1.0 / 256.0;             // Quarter note positive
    var
      Cents  : TSignal;
      Speed  : TSignal;
      LinCon : TSignal;
      Freq   : TSignal;
      ExpCon : TSignal;
    begin
      Speed  := Normalize( 2 * FInputs[ i_speed] + FInputs[ i_lfmmod] * FInputs[ i_lfmmodlev] + FInputs[ i_lfm]);
      Cents  := Normalize( FInputs[ i_cents] + FInputs[ i_centsmod] * FInputs[ i_centsmodamt] * OTSF);
      Freq   := Speed * LookupFrequency( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fmmod] * FInputs[ i_fmmodlev] + Cents);
      LinCon := Freq / ReferenceA;

      if Freq <= 0
      then ExpCon := FrequencyToUnits( 1e-6)      // 1 micro Hz will do for zero Hz I guess (11.5 days or so)
      else ExpCon := FrequencyToUnits( Freq) - NoteNumberToUnits( MiddleNote);

      FOutputs[ o_out   ] := LinCon;
      FOutputs[ o_outexp] := ExpCon;
    end;


{ ========
  TModMultiSine = class( TMod)
  private
    FPhases  : array[ 0 .. 5] of TSignal;
    FOldSync : Boolean;
  public
}

    procedure   TModMultiSine.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_sync      , 'sync'      );
      AddInput ( i_chain     , 'chain'     );
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fmmod     , 'fmmod'     );
      AddInput ( i_fmmodlev  , 'fmmodlev'  );
      AddInput ( i_cents     , 'cents'     );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_fmult1    , 'fmult1'    );
      AddInput ( i_cents1    , 'cents1'    );
      AddInput ( i_lmmod1    , 'lmmod1'    );
      AddInput ( i_fmmod1    , 'fmmod1'    );
      AddInput ( i_ammod1    , 'ammod1'    );
      AddInput ( i_ammodlev1 , 'ammodlev1' );
      AddInput ( i_amplitude1, 'amplitude1');
      AddInput ( i_mute1     , 'mute1'     );
      AddInput ( i_fmult2    , 'fmult2'    );
      AddInput ( i_cents2    , 'cents2'    );
      AddInput ( i_lmmod2    , 'lmmod2'    );
      AddInput ( i_fmmod2    , 'fmmod2'    );
      AddInput ( i_ammod2    , 'ammod2'    );
      AddInput ( i_ammodlev2 , 'ammodlev2' );
      AddInput ( i_amplitude2, 'amplitude2');
      AddInput ( i_mute2     , 'mute2'     );
      AddInput ( i_fmult3    , 'fmult3'    );
      AddInput ( i_cents3    , 'cents3'    );
      AddInput ( i_lmmod3    , 'lmmod3'    );
      AddInput ( i_fmmod3    , 'fmmod3'    );
      AddInput ( i_ammod3    , 'ammod3'    );
      AddInput ( i_ammodlev3 , 'ammodlev3' );
      AddInput ( i_amplitude3, 'amplitude3');
      AddInput ( i_mute3     , 'mute3'     );
      AddInput ( i_fmult4    , 'fmult4'    );
      AddInput ( i_cents4    , 'cents4'    );
      AddInput ( i_lmmod4    , 'lmmod4'    );
      AddInput ( i_fmmod4    , 'fmmod4'    );
      AddInput ( i_ammod4    , 'ammod4'    );
      AddInput ( i_ammodlev4 , 'ammodlev4' );
      AddInput ( i_amplitude4, 'amplitude4');
      AddInput ( i_mute4     , 'mute4'     );
      AddInput ( i_fmult5    , 'fmult5'    );
      AddInput ( i_cents5    , 'cents5'    );
      AddInput ( i_lmmod5    , 'lmmod5'    );
      AddInput ( i_fmmod5    , 'fmmod5'    );
      AddInput ( i_ammod5    , 'ammod5'    );
      AddInput ( i_ammodlev5 , 'ammodlev5' );
      AddInput ( i_amplitude5, 'amplitude5');
      AddInput ( i_mute5     , 'mute5'     );
      AddInput ( i_fmult6    , 'fmult6'    );
      AddInput ( i_cents6    , 'cents6'    );
      AddInput ( i_lmmod6    , 'lmmod6'    );
      AddInput ( i_fmmod6    , 'fmmod6'    );
      AddInput ( i_ammod6    , 'ammod6'    );
      AddInput ( i_ammodlev6 , 'ammodlev6' );
      AddInput ( i_amplitude6, 'amplitude6');
      AddInput ( i_mute6     , 'mute6'     );
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModMultiSine.SetDefaults; // override;
    var
      i : Integer;
      p : Integer;
    begin
      FDezipperMap := [
        i_frequency, i_fmmodlev, i_cents                  ,
        i_fmult1   , i_cents1  , i_ammodlev1, i_amplitude1,
        i_fmult2   , i_cents2  , i_ammodlev2, i_amplitude2,
        i_fmult3   , i_cents3  , i_ammodlev3, i_amplitude3,
        i_fmult4   , i_cents4  , i_ammodlev4, i_amplitude4,
        i_fmult5   , i_cents5  , i_ammodlev5, i_amplitude5,
        i_fmult6   , i_cents6  , i_ammodlev6, i_amplitude6
      ];
      p := 0;

      for i := 1 to 6
      do begin
        FInputs[ i_lmmod1 + p] := 1.0;
        p := p + INSTRIDE;
      end;
    end;


    procedure   TModMultiSine.DoTick; // override;
    var
      i          : Integer;
      p          : Integer;
      Sync       : Boolean;
      PreFreq    : TSignal;
      PhaseDelta : TSignal;
      Tuning     : TSignal;
      Speed      : TSignal;
      Amplitude  : TSignal;
      OutVal     : TSignal;
    begin
      Sync   := SignalToLogic( FInputs[ i_sync]);

      if ( not FOldSync) and Sync
      then begin
        for i := 0 to 5
        do FPhases[ i]  := 0.0;
      end;

      FOldSync := Sync;
      PreFreq  := FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fmmod] * FInputs[ i_fmmodlev] + FInputs[ i_cents];
      OutVal   := FInputs[ i_chain];
      p        := 0;

      for i := 0 to 5
      do begin
        Speed       := LookupFrequency( Finputs[ i_fmult1 + p] * FInputs[ i_lmmod1 + p]) / ReferenceA;
        Tuning      := FInputs[ i_cents1 + p] + FInputs[ i_fmmod1 + p];
        PhaseDelta  := LookupPhaseDelta( PreFreq + Tuning);
        FPhases[ i] := MathFloatMod( FPhases[ i] + Speed * PhaseDelta, 1.0);
        Amplitude   := FInputs[ i_amplitude1 + p] + FInputs[ i_ammod1 + p] * FInputs[ i_ammodlev1 + p];
        OutVal      := OutVal + SignalToMute( FInputs[ i_mute1 + p]) * Amplitude * LookupSine( FPhases[ i]);
        p           := p + INSTRIDE;
      end;

      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * OutVal;
    end;


{ ========
  TModChladnicOsc = class( TMod)
  strict private
    OSC_COUNT     = 11;
  private
    FOldP    : TSignal;
    FPhases  : array[ 0 .. OSC_COUNT - 1] of TSignal;
    FAddends : array[ 0 .. OSC_COUNT - 1] of TSignal;
  public
}

    procedure   TModChladnicOsc.CreateIO; // override;
    begin
      FISFast := True;
      FIsSlow := True;
      AddInput ( i_freq     , 'freq'     );
      AddInput ( i_frequency, 'frequency');
      AddInput ( i_fm       , 'fm'       );
      AddInput ( i_fmlevel  , 'fmlevel'  );
      AddInput ( i_cents    , 'cents'    );
      AddInput ( i_mute     , 'mute'     );
      AddInput ( i_pm       , 'pm'       );
      AddInput ( i_pmlevel  , 'pmlevel'  );
      AddInput ( i_p        , 'p'        );
      AddInput ( i_pmod     , 'pmod'     );
      AddInput ( i_pmodamt  , 'pmodamt'  );
      AddInput ( i_speed    , 'speed'    );
      AddInput ( i_level1   , 'level1'   );
      AddInput ( i_level2   , 'level2'   );
      AddInput ( i_level3   , 'level3'   );
      AddInput ( i_level4   , 'level4'   );
      AddInput ( i_level5   , 'level5'   );
      AddInput ( i_level6   , 'level6'   );
      AddInput ( i_level7   , 'level7'   );
      AddInput ( i_level8   , 'level8'   );
      AddInput ( i_level9   , 'level9'   );
      AddInput ( i_level10  , 'level10'  );
      AddInput ( i_level11  , 'level11'  );
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModChladnicOsc.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_fmlevel  ,
        i_cents    ,
        i_p        ,
        i_pmodamt  ,
        i_level1   ,
        i_level2   ,
        i_level3   ,
        i_level4   ,
        i_level5   ,
        i_level6   ,
        i_level7   ,
        i_level8   ,
        i_level9   ,
        i_level10  ,
        i_level11
      ];

      FInputs[ i_speed] := 0.5;
    end;


    procedure   TModChladnicOsc.DoSlowTick; // override;
    var
      NewP : TSignal;
      i    : Integer;
      m    : TSignal;
    begin
      NewP := Clip( FInputs[ i_pmod] * FInputs[ i_pmodamt] + FInputs[ i_p   ], 1, 8);

      if NewP <> FOldP
      then begin
        m := NewP * NotesPerOctave * NOTE_SCALING_REC;

        for i := 0 to OSC_COUNT - 1
        do FAddends[ i] := m * log2( i + 1);

        FOldP := NewP;
      end;
    end;


//      FSpeed      := Normalize( FInputs[ i_speed]) * 2;
//      FBreakPoint := Clip( FInputs[ i_warp] + FInputs[ i_warpmod] * FInputs[ i_warpmodamt], 0.001, 0.999);
//      FPhaseDelta := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents]);
//      FPhase      := MathFloatMod( FPhase + FSpeed * FPhaseDelta, 1.0);
//      EffPhase    := MathFloatMod( FPhase + FInputs[ i_phase] * FInputs[ i_pmlevel], 1.0);


    procedure   TModChladnicOsc.DoTick; // override;
    var
      i          : Integer;
      PreFreq    : TSignal;
      PhaseDelta : TSignal;
      Speed      : TSignal;
      Phase      : TSignal;
      EffPhase   : TSignal;
      OutVal     : TSignal;
    begin
      OutVal     := 0;
      PreFreq    := FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents];
      Speed      := 2.0 * FInputs[ i_speed];
      Phase      := FInputs[ i_pm] * FInputs[ i_pmlevel];

      for i := 0 to OSC_COUNT - 1
      do begin
        PhaseDelta  := LookupPhaseDelta( PreFreq + FAddends[ i]);
        FPhases[ i] := MathFloatMod( FPhases[ i] + Speed * PhaseDelta, 1.0);
        EffPhase    := MathFloatMod( FPhases[ i] + Phase             , 1.0);
        OutVal      := OutVal + LookupSine( EffPhase);
      end;

      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * OutVal;
    end;


{ ========
  TModBaseTrigOsc = class( TModBaseOsc)
  private
    FState   : TTrigOscState;
    FCounter : TSignal;
  public
}

    procedure   TModbaseTrigOsc.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_trig    , 'trig'    );
      AddInput ( i_count   , 'count'   );
      AddInput ( i_countmod, 'countmod');
    end;


    procedure   TModbaseTrigOsc.SetDefaults; // override;
    begin
      inherited;
      FInputs[ i_countmod] := 1.0;
    end;


    procedure   TModbaseTrigOsc.DoTick; // override;
    var
      FOldCounter : Integer;
    begin
      FSpeed := Normalize( FInputs[ i_speed]) * 2;

      case FState of

        toIdle :
          begin
            FPosition := 0.0;
            FCounter  := 0;

            if SignalToLogic( FInputs[ i_trig])
            then FState := toRunning;
          end;

        toRunning,
        toWaiting :
          begin
            // This is almost the inherited code - except - it ignores phase

            FPhaseDelta := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents]);
            FPosition   := MathFloatMod( FPosition + FSpeed * FPhaseDelta, 1.0);
            FShape      := Round( FInputs[ i_shape]);

            if FState = toRunning
            then begin
              FOldCounter := FCounter;

              if AlmostEqual( FPosition, 0.5, FPhaseDelta)
              then FCounter := FCounter + 1
              else if AlmostEqual( FPosition, 1.0, FPhaseDelta)
              then FCounter := FCounter + 1;

              if FCounter <> FOldCounter
              then begin
                if FCounter >= FInputs[ i_count] * ( Normalize( FInputs[ i_countmod]) + 1.0)
                then FState := toDone
                else FState := toWaiting;
              end;
            end
            else begin
              if
                not AlmostEqual( FPosition, 0.5, FPhaseDelta) and
                not AlmostEqual( FPosition, 1.0, FPhaseDelta)
              then FState := toRunning;
            end;
          end;

        toDone :
          begin
            FPosition := 0.0;

            if FInputs[ i_trig] <= 0.0
            then FState := toIdle;
          end;
      end;
    end;


{ ========
  TModOsc = class( TModBaseOsc)
  public
}

    procedure   TModOSc.DoTick; // override;
    var
      aValue : TSignal;
    begin
      inherited;

      case FShape of
        0  : // sine
          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * LookupSine( FPosition);
        1  : // tri
          begin
            if FPosition < 0.5
            then aValue :=   1
            else aValue := - 1;

            aValue := 4 * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
            FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_out]);
          end;
        2  : // saw
          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 2.0 * FPosition - 1) - PolyBLEP( FPosition, FBandLimit, FPhaseDelta));
        3  : // square
          begin
            if FPosition < 0.5
            then aValue :=   1
            else aValue := - 1;

            FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
          end;
      end;
    end;


{ ========
  TModMultiPhaseOsc = class( TModBaseOsc)
  public
}

    procedure   TModMultiPhaseOsc.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_phaseshift, 'phaseshift');
      AddOutput( o_out2      , 'out2'      );
      AddOutput( o_out3      , 'out3'      );
      AddOutput( o_out4      , 'out4'      );
    end;


    procedure   TModMultiPhaseOsc.DoTick; // override;
    var
      aValue : TSignal;
      aShift : Integer;
      aPos2  : TSignal;
      aPos3  : TSignal;
      aPos4  : TSignal;
    begin
      inherited;
      aShift := Round(FInputs[ i_phaseshift]);

      if aShift = 1                                          // 120 deg, three outputs used
      then begin
        aPos2 := MathFloatMod( FPosition + 0.33333333, 1.0);
        aPos3 := MathFloatMod( FPosition + 0.66666667, 1.0);
        aPos4 := MathFloatMod( FPosition + 0.5       , 1.0); // And add anti-phase as a bonus on the 4th output
      end
      else begin                                             // 90 deg
        aPos2 := MathFloatMod( FPosition + 0.75, 1.0);
        aPos3 := MathFloatMod( FPosition + 0.5 , 1.0);
        aPos4 := MathFloatMod( FPosition + 0.25, 1.0);
      end;

      case FShape of

        0  : // sine

          begin
            FOutputs[ o_out ] := SignalToMute( FInputs[ i_mute]) * LookupSine( FPosition);
            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * LookupSine( aPos2    );
            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * LookupSine( aPos3    );
            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * LookupSine( aPos4    );
          end;

        1  : // tri

          begin
            if FPosition < 0.5 then aValue := 1.0 else aValue := -1.0;

            aValue := 4 * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
            FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_out]);

            if aPos2 < 0.5 then aValue := 1.0 else aValue := -1.0;

            aValue := 4 * ( aValue + PolyBLEP( aPos2, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( aPos2 + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_out2]);

            if aPos3 < 0.5 then aValue := 1.0 else aValue := -1.0;

            aValue := 4 * ( aValue + PolyBLEP( aPos3, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( aPos3 + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_out3]);

            if aPos4 < 0.5 then aValue := 1.0 else aValue := -1.0;

            aValue := 4 * ( aValue + PolyBLEP( aPos4, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( aPos4 + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_out4]);
          end;

        2  : // saw

          begin
            FOutputs[ o_out ] := SignalToMute( FInputs[ i_mute]) * (( 2.0 * FPosition - 1) - PolyBLEP( FPosition, FBandLimit, FPhaseDelta));
            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * (( 2.0 * aPos2     - 1) - PolyBLEP( aPos2    , FBandLimit, FPhaseDelta));
            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * (( 2.0 * aPos3     - 1) - PolyBLEP( aPos3    , FBandLimit, FPhaseDelta));
            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * (( 2.0 * aPos4     - 1) - PolyBLEP( aPos4    , FBandLimit, FPhaseDelta));
          end;

        3  : // square

          begin
            if FPosition < 0.5 then aValue := 1.0 else aValue := -1.0;

            FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));

            if aPos2 < 0.5 then aValue := 1.0 else aValue := -1.0;

            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( aPos2, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( aPos2 + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));

            if aPos3 < 0.5 then aValue := 1.0 else aValue := -1.0;

            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( aPos3, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( aPos3 + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));

            if aPos4 < 0.5 then aValue := 1.0 else aValue := -1.0;

            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( aPos4, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( aPos4 + FBreakPoint, 1.0), FBandLimit, FPhaseDelta));
          end;
      end;
    end;


{ ========
  TModMultiOsc = class( TModBaseOsc)
  public
}

    procedure   TModMultiOsc.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_pwm      , 'pwm'      );
      AddInput ( i_pwmlevel , 'pwmlevel' );
      AddOutput( o_outsine  , 'outsine'  );
      AddOutput( o_outtri   , 'outtri'   );
      AddOutput( o_outsaw   , 'outsaw'   );
      AddOutput( o_outsquare, 'outsquare');
    end;


    procedure   TModMultiOsc.SetDefaults; // override;
    begin
      inherited;
      FDezipperMap := FDezipperMap + [
        i_pwmlevel
      ];
    end;


    procedure   TModMultiOsc.DoTick; // override;
    var
      pwmp   : TSignal;
      aValue : TSignal;
      aMute  : TSignal;
      aBlep  : TSignal;
    begin
      inherited;                                // For phase calculations

      aMute := SignalToMute( FInputs[ i_mute]);
      aBlep := PolyBLEP( FPosition, FBandLimit, FPhaseDelta);
      FOutputs[ o_outsine] := aMute * LookupSine( FPosition);

      if FPosition < 0.5
      then aValue :=   1
      else aValue := - 1;

      aValue := 4 * ( aValue + aBlep - PolyBLEP( FloatMod( FPosition + 0.5, 1.0), FBandLimit, FPhaseDelta));
      FOutputs[ o_outtri] := aMute * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_outtri]);
      FOutputs[ o_outsaw] := aMute * (( 2.0 * FPosition - 1) - aBlep);
      pwmp := Clip( 0.5 * ( 1.0 - FInputs[ i_pwm] * FInputs[ i_pwmlevel]), 0.01, 0.99);

      if FPosition < pwmp
      then aValue :=   1
      else aValue := - 1;

      FOutputs[ o_outsquare] := aMute * ( aValue + aBlep - PolyBLEP( FloatMod( FPosition + ( 1 - pwmp), 1.0), FBandLimit, FPhaseDelta));
    end;


{ ========
  TModTrigOsc = class( TModBaseTrigOsc)
  public
}

    procedure   TModOscTrig.DoTick; // override;
    var
      aValue : TSignal;
    begin
      inherited;

      case FShape of
        0  : // sine

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * LookupSine( FPosition);

        1  : // tri

          begin
            if FPosition < 0.5
            then aValue :=   1
            else aValue := - 1;

            aValue := 4 * aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + 0.5, 1.0), FBandLimit, FPhaseDelta);
            FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( FPhaseDelta * aValue + ( 1 - FPhaseDelta) * FOutputs[ o_out]);
          end;

        2  : // saw

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 2.0 * FPosition - 1) - PolyBLEP( FPosition, FBandLimit, FPhaseDelta));

        3  : // square

          begin
            if FPosition < 0.5
            then aValue :=   1
            else aValue := - 1;

            FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + 0.5, 1.0), FBandLimit, FPhaseDelta));
          end;
      end;
    end;


{ ========
  TModSquare = class( TModBaseOsc)
  private
    FOldPosition  : TSignal;
    FPulseCount   : Integer;
    FPulseCounter : Integer;
  public
}

    procedure   TModSquare.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_pwm     , 'pwm'     );
      AddInput ( i_pwmlevel, 'pwmlevel');
      AddInput ( i_pwmoff  , 'pwmoff'  );
      AddInput ( i_pwmplm  , 'pwmplm'  );
    end;


    procedure   TModSquare.SetDefaults;
    begin
      inherited;
      FDezipperMap := FDezipperMap + [
        i_pwmlevel,
        i_pwmoff
      ];
    end;


    procedure   TModSquare.DoTick; // override;
    var
      pwmp   : TSignal;
      aValue : TSignal;
    begin
      inherited;                                // For phase calculations

      if Round( FInputs[ i_pwmplm]) = 0
      then begin
        pwmp := Clip( 0.5 * ( 1.0 - FInputs[ i_pwm] * FInputs[ i_pwmlevel]) + FInputs[ i_pwmoff], 0.01, 0.99);

        if FPosition < pwmp
        then aValue :=   1
        else aValue := - 1;

        FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + ( 1 - pwmp), 1.0), FBandLimit, FPhaseDelta));
      end
      else begin
        if FPosition < FOldPosition // Wrapped
        then begin
          FPulseCount   := Round( TimeToSampleCount( RangeMap( Clip( FInputs[ i_pwmoff] + FInputs[ i_pwm] * FInputs[ i_pwmlevel], 0.01, 0.99), 0.0, 1.0, 10e-6, 1e-3)));
          FPulseCounter := 0;
          pwmp          := FPosition;
        end
        else begin
          Inc( FPulseCounter);

          if FPulseCounter >= FPulseCount
          then pwmp := 0.0
          else pwmp := FPosition;
        end;

        FOldPosition := FPosition;

        if FPosition < pwmp
        then aValue :=   1
        else aValue := - 1;

        FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + ( 1 - pwmp), 1.0), FBandLimit, FPhaseDelta));
      end;
    end;


{ ========
  TModSquareTrig = class( TModBaseTrigOsc)
  public
}

    procedure   TModSquareTrig.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_pwm     , 'pwm'     );
      AddInput ( i_pwmlevel, 'pwmlevel');
    end;


    procedure   TModSquareTrig.SetDefaults;
    begin
      inherited;
      FDezipperMap := FDezipperMap + [
        i_pwmlevel
      ];
    end;


    procedure   TModSquareTrig.DoTick; // override;
    var
      pwmp   : TSignal;
      aValue : TSignal;
    begin
      inherited;                                // For phase calculations
      pwmp := Clip( 0.5 * ( 1.0 - FInputs[ i_pwm] * FInputs[ i_pwmlevel]), 0.01, 0.99);

      if FPosition < pwmp
      then aValue :=   1
      else aValue := - 1;

      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * ( aValue + PolyBLEP( FPosition, FBandLimit, FPhaseDelta) - PolyBLEP( FloatMod( FPosition + ( 1 - pwmp), 1.0), FBandLimit, FPhaseDelta));
    end;


{ ========
  TModVosim = class( TMod)
  private
    FPhaseDelta : TSignal;
    FPhase      : TSignal;
    FState      : TTrigOScState;
    FCounter    : Integer;
    FLevel      : TSignal;
    FWhisper    : TSignal;
    FDecay      : TSignal;
    FMaxCount   : TSignal;
  public
}

    procedure   TModVosim.CreateIO; // override;
    begin
      FIsFast := True;
      FISSlow := True;
      AddInput ( i_mute     , 'mute'     );
      AddInput ( i_trig     , 'trig'     );
      AddInput ( i_frequency, 'frequency');
      AddInput ( i_freq     , 'freq'     );
      AddInput ( i_fm       , 'fm'       );
      AddInput ( i_fmamt    , 'fmamt'    );
      AddInput ( i_count    , 'count'    );
      AddInput ( i_cm       , 'cm'       );
      AddInput ( i_whisper  , 'whisper'  );
      AddInput ( i_wm       , 'wm'       );
      AddInput ( i_wmamt    , 'wmamt'    );
      AddInput ( i_decay    , 'decay'    );
      AddInput ( i_dm       , 'dm'       );
      AddInput ( i_dmamt    , 'dmamt'    );
      AddInput ( i_amplitude, 'amplitude');
      AddInput ( i_am       , 'am'       );
      AddInput ( i_amamt    , 'amamt'    );
      AddInput ( i_speed    , 'speed'    );
      AddOutput( o_out      , 'out'      );
    end;


    procedure   TModVosim.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_fmamt    ,
        i_whisper  ,
        i_wmamt    ,
        i_decay    ,
        i_dmamt    ,
        i_amplitude,
        i_amamt
      ];
      FInputs[ i_cm   ] := 1.0;
      FInputs[ i_speed] := 0.5;
    end;


    procedure   TModVosim.DoSlowTick; // override;
    begin
      case FState of
        toRunning,
        toWaiting : FPhaseDelta := LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmamt]);
      end;

      FWhisper  := Clip( FInputs[ i_whisper] + FInputs[ i_wm] * FInputs[ i_wmamt], 0.0, 1.0);
      FDecay    := Clip( FInputs[ i_decay  ] + FInputs[ i_dm] * FInputs[ i_dmamt], 0.0, 1.0);
      FMaxCount := Clip( FInputs[ i_count] * FInputs[ i_cm], 1.0, 99.0);
    end;


    procedure   TModVosim.DoTick; // override;
    var
      FOldCounter : Integer;
      Sine        : TSignal;
      Signal      : TSignal;
      AM          : TSignal;
    begin
      case FState of

        toIdle :

          begin
            if SignalToLogic( FInputs[ i_trig])
            then begin
              FPhase   := 0.0;
              FCounter := 0;
              FLevel   := 1.0;
              FState   := toRunning;
            end;
          end;

        toRunning,
        toWaiting :

          begin
            FSpeed  := Normalize( FInputs[ i_speed]) * 2;
            FPhase  := MathFloatMod( FPhase + FSpeed * FPhaseDelta, 1.0);

            if FState = toRunning
            then begin
              FOldCounter := FCounter;

              if AlmostEqual( FPhase , 0.5, FPhaseDelta)
              then FCounter := FCounter + 1
              else if AlmostEqual( FPhase , 1.0, FPhaseDelta)
              then FCounter := FCounter + 1;

              if FCounter <> FOldCounter
              then begin
                if FCounter >= FMaxCount
                then begin
                  FLevel := 0.0;
                  FState := toDone;
                end
                else begin
                  FLevel := FLevel * FDecay;
                  FState := toWaiting;
                end;
              end;
            end
            else begin
              if
                not AlmostEqual( FPhase, 0.5, FPhaseDelta) and
                not AlmostEqual( FPhase, 1.0, FPhaseDelta)
              then FState := toRunning;
            end;

          end;

        toDone :

          begin
            FPhase := 0.0;

            if FInputs[ i_trig] <= 0.0
            then FState := toIdle;
          end;
      end;

      Sine             := LookupSine( FPhase);
      Sine             := Sine * Sine;
      Signal           := FWhisper * ( Random - Sine) + Sine;
      AM               := FInputs[ i_amplitude] + FInputs[ i_am] * FInputs[ i_amamt];
      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * Signal * FLevel * AM;
    end;


{ ========
  TModBaseLfo = class( TMod)
  private
    FSpeed       : TSignal;
    FPhase       : TSignal;
    FPosition    : TSignal;
    FOldRes      : Boolean;
    FShape       : Integer;
    FWrapped     : Boolean;
    FRandomValue : TSignal;
  public
}

    procedure   TModBaseLfo.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_phase     , 'phase'     );
      AddInput ( i_res       , 'res'       );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_pmlevel   , 'pmlevel'   );
      AddInput ( i_shape     , 'shape'     );
      AddInput ( i_range     , 'range'     );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_speed     , 'speed'     );
      AddOutput( o_out       , 'out'       );
      AddOutput( o_sync      , 'sync'      );
    end;


    procedure   TModBaseLfo.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_fmlevel,
        i_pmlevel
      ];
      FInputs[ i_speed] := 0.5;
    end;


    procedure   TModBaseLfo.DoSlowTick; // override;
    var
      Delta : TSignal;
    begin
      if ( not FOldRes) and SignalToLogic( FInputs[ i_res]) or ResetFlag
      then FPhase := 0.0;

      ResetFlag := False;
      FOldRes   := SignalToLogic( FInputs[ i_res  ]);
      FShape    := Round        ( FInputs[ i_shape]);
      FWrapped  := False;

      case Round( FInputs[ i_range]) of
        0  : Delta := LookupLfoFastPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        1  : Delta := LookupLfoMediumPhaseDelta ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        2  : Delta := LookupLfoSlowPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        3  : Delta := LookupLfoBPMPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        4  : Delta := LookupLfoLinPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        5  : Delta := LookupLfoFast2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        6  : Delta := LookupLfoMedium2PhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        else Delta := LookupLfoSlow2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
      end;

      FSpeed      := Normalize( FInputs[ i_speed]) * 2;
      FPhase      := FPhase + FSpeed * Delta;
      FWrapped    := FPhase > 1;
      FPhase      := MathFloatMod( FPhase, 1.0);
      FPosition   := MathFloatMod( FPhase + FInputs[ i_phase] * FInputs[ i_pmlevel], 1.0); // Seems more correct and a tad cheaper

      if FWrapped
      then FRandomValue := 1 - 2 * Random;

      FOutputs[ o_sync] := LogicToSignal( FWrapped);
    end;


    procedure   TModBaseLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'res' , FInputs [ i_res]               ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out' , LogicToSignal( FPosition < 0.5)));
    end;


{ ========
  TModBaseTrigLfo = class( TModBaseLfo)
  private
    FState    : TTrigOScState;
    FCounter  : Integer;
    FDidRun   : Boolean;
    FPrevTrig : Boolean;
  public
}

    procedure   TModBaseTrigLfo.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_count     , 'count'     );
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_pmlevel   , 'pmlevel'   );
      AddInput ( i_shape     , 'shape'     );
      AddInput ( i_range     , 'range'     );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_mode      , 'mode'      );
      AddInput ( i_speed     , 'speed'     );
      AddInput ( i_countmod  , 'countmod'  );
      AddInput ( i_autoready , 'autoready' );
      AddOutput( o_out       , 'out'       );
      AddOutput( o_ready     , 'ready'     );
      AddOutput( o_active    , 'active'    );
    end;


    procedure   TModBaseTrigLfo.SetDefaults; // override;
    begin
      inherited;
      FInputs [ i_countmod] := 1.0;
    end;


    procedure   TModBaseTrigLfo.DoSlowTick; // override;
    var
      FOldCounter : Integer;
      Delta       : TSignal;
      Trig        : Boolean;
    begin
      Trig := SignalToLogic( FInputs[ i_trig]);

      if ( Round( FInputs[ i_mode]) = 1) and Trig and not FPrevTrig
      then ResetFlag := True;

      if ResetFlag
      then begin
        ResetFlag := False;
        FState    := toIdle;
        FDidRun   := False;
      end;

      case FState of

        toIdle :

          begin
            FPosition := 0.0;
            FCounter  := 0;
            if Trig
            then begin
              FState  := toRunning;
              FDidRun := True;
            end;
          end;

        toRunning,
        toWaiting :

          begin
            // This is almost the inherited code - except - it ignores phase and has no freq input

            FShape := Round( FInputs[ i_shape]);

            case Round( FInputs[ i_range]) of
              0  : Delta := LookupLfoFastPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              1  : Delta := LookupLfoMediumPhaseDelta ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              2  : Delta := LookupLfoSlowPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              3  : Delta := LookupLfoBPMPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              4  : Delta := LookupLfoLinPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              5  : Delta := LookupLfoFast2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              6  : Delta := LookupLfoMedium2PhaseDelta( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
              else Delta := LookupLfoSlow2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
            end;

            FSpeed    := Normalize( FInputs[ i_speed]) * 2;
            FPosition := MathFloatMod( FPosition + FSpeed * Delta, 1.0);

            if FState = toRunning
            then begin
              FOldCounter := FCounter;

              if AlmostEqual( FPosition, 0.5, Delta)
              then FCounter := FCounter + 1
              else if AlmostEqual( FPosition, 1.0, Delta)
              then FCounter := FCounter + 1;

              if FCounter <> FOldCounter
              then begin
                if FCounter >= FInputs[ i_count] * Normalize( FInputs[ i_countmod])
                then FState := toDone
                else FState := toWaiting;
              end;
            end
            else begin
              if
                not AlmostEqual( FPosition, 0.5, Delta) and
                not AlmostEqual( FPosition, 1.0, Delta)
              then FState := toRunning;
            end;
          end;

        toDone :

          begin
            FPosition := 0.0;
            if not SignalToLogic( FInputs[ i_trig])
            then FState := toIdle;
          end;
      end;

      FPrevTrig           := Trig;
      FOutputs[ o_ready ] := LogicToSignal(( FState = toIdle) and (FDidRun or ( Round( FInputs[ i_autoready]) = 1)));
      FOutputs[ o_active] := LogicToSignal( FState in [ toRunning, toWaiting]);
    end;


    procedure   TModBaseTrigLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'  , FInputs[ i_trig  ]              ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'ready' , FOutputs[ o_ready]              ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active', FOutputs[ o_active]             ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'   , LogicToSignal( FPosition <= 0.5)));
    end;


{ ========
  TModCounterLfo = class( TMod)
  private
    FPhase : TSignal;
    FSpeed : TSignal;
  public
}

    procedure   TModCounterLfo.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_range     , 'range'     );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_speed     , 'speed'     );
    end;


    procedure   TModCounterLfo.SetDefaults; // override;
    begin
      FInputs[ i_speed] := 0.5;
    end;


    procedure   TModCounterLfo.DoSlowTick; // override;
    var
      Delta   : TSignal;
      DidWrap : Boolean;
    begin
      if ResetFlag
      then begin
        FPhase := Random;
        Wrapped;
      end;

      ResetFlag := False;

      case Round( FInputs[ i_range]) of
        0  : Delta := LookupLfoFastPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        1  : Delta := LookupLfoMediumPhaseDelta ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        2  : Delta := LookupLfoSlowPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        3  : Delta := LookupLfoBPMPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        4  : Delta := LookupLfoLinPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        5  : Delta := LookupLfoFast2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        6  : Delta := LookupLfoMedium2PhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        else Delta := LookupLfoSlow2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
      end;

      FSpeed  := 2.0 * Clip( Finputs[ i_speed], 0.0, 1.0);
      FPhase  := FPhase - FSpeed * Delta;
      DidWrap := False;

      while FPhase < 0
      do begin
        if not DidWrap
        then begin
          Wrapped;
          DidWrap := True;
        end;

        FPhase := FPhase + 1.0;
      end;
    end;


    procedure   TModCounterLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'active', LogicToSignal( FPhase >= 0.5)));
    end;


{ ========
  TModClockGen = class( TMod)
  private
    FPhase16     : TSignal; // Phase acuumulator for * 16 output
    FPhase96     : TSignal; // Phase acuumulator for * 96 output
    FOldRes      : Boolean; // Old reset value
    FOldMute     : Boolean; // Old mute state
    FEven        : Boolean; // Even / Odd cycle for * 16 output
    FState       : Boolean; // * 96 state
    FSwing       : TSignal; // Current Swing value
    FDividerSync : Integer; // Divider for sync output
    FResCount    : Integer; // Reset LED counter
  public
}

    procedure   TModClockGen.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_reset  , 'res'    );
      AddInput ( i_tempo  , 'tempo'  );
      AddInput ( i_tempoex, 'tempoex');
      AddInput ( i_swing  , 'swing'  );
      AddInput ( i_mute   , 'mute'   );
      AddInput ( i_div    , 'div'    );
      AddInput ( i_speed  , 'speed'  );
      AddOutput( o_16     , '16'     );
      AddOutput( o_96     , '96'     );
      AddOutput( o_sync   , 'sync'   );
      AddOutput( o_phase16, 'phase16');
      AddOutput( o_phase96, 'phase96');
    end;


    procedure   TModClockGen.SetDefaults; // override;
    begin
      FInputs[ i_tempoex] := NOT_CONNECTED;
      FInputs[ i_speed  ] := 0.5;
    end;


    procedure   TModClockGen.DoSlowTick; // override;
    var
      Delta96 : TSignal;
      Delta16 : TSignal;
      DidWrap : Boolean;
      Muted   : Boolean;
      Reset   : Boolean;
      SyncDiv : Integer;
      Tempo   : TSignal;
      Speed   : TSignal;
    begin
      Reset   := SignalToLogic( FInputs[ i_reset]);
      Muted   := SignalToMute ( FInputs[ i_mute ]) < 0.5;
      SyncDiv := Round( FInputs[ i_div]);
      Speed   := FInputs[ i_speed] * 2;

      if ResetFlag or ( not FOldRes and Reset) or ( FOldMute and not Muted)
      then begin
        FPhase96     := 0.0;
        FPhase16     := 0.0;
        FEven        := True;
        FState       := True;
        FDividerSync := 0;
        FResCount    := LedFlashTime( IsSpedUp);
        FSwing       := 0.5;
      end;

      if FInputs[ i_tempoex] = NOT_CONNECTED
      then Tempo := FInputs[ i_tempo  ]
      else Tempo := FInputs[ i_tempoex];

      ResetFlag := False;
      FOldRes   := Reset;
      FOldMute  := Muted;
      Delta16   := LookupLfoBPMPhaseDelta( Tempo) * 4 * Speed;
      Delta96   := Delta16 * 6;
      FPhase96  := FPhase96 + Delta96;
      FPhase16  := FPhase16 + Delta16;
      DidWrap   := False;

      while FPhase96 > 1.0
      do begin
        if not DidWrap
        then begin
          DidWrap := True;
          FState  := True;
          Inc( FDividerSync);

          if FDividerSync > 24 * SyncDiv - 1
          then FDividerSync := 0;
        end;

        FPhase96 := FPhase96 - 1.0;
      end;

      if FState
      then begin
        if FPhase96 > 0.5
        then FState := False;
      end;

      DidWrap := False;

      while FPhase16 > 1.0
      do begin
        if not DidWrap
        then begin
          DidWrap := True;
          FEven   := not FEven;

          if FEven
          then FSwing := 0.5
          else FSwing := FInputs[ i_swing];
        end;

        FPhase16 := FPhase16 - 1.0;
      end;

      if FResCount > 0
      then Dec( FResCount);

      if Muted
      then begin
        FOutputs[ o_16     ] := LogicToSignal( False);
        FOutputs[ o_96     ] := LogicToSignal( False);
        FOutputs[ o_sync   ] := LogicToSignal( False);
        FOutputs[ o_phase16] := 0;
        FOutputs[ o_phase96] := 0;
      end
      else begin
        FOutputs[ o_16     ] := LogicToSignal( FPhase16 > FSwing);
        FOutputs[ o_96     ] := LogicToSignal( FState           );
        FOutputs[ o_sync   ] := LogicToSignal( FDividerSync = 0 );
        FOutputs[ o_phase16] := FPhase16;
        FOutputs[ o_phase96] := FPhase96;
      end;
    end;


    procedure   TModClockGen.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'sync', LogicToSignal( FDividerSync = 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res' , LogicToSignal( FResCount    > 0)));
    end;


{ ========
  TModTod = class( TMod)
  public
}

    procedure   TModTod.CreateIO; // override;
    begin
      FIsSlow := True;
      AddOutput( o_out  , 'out' );
      AddOutput( o_outh , 'outh');
      AddOutput( o_outm , 'outm');
      AddOutput( o_outs , 'outs');
    end;


    procedure   TModTod.DoSlowTick; // override;
    const
      H  =     24.0;
      M  = H * 60.0;
      S  = M * 60.0;
    var
      T : TSignal;
    begin
      T                 := Frac( Now);
      FOutputs[ o_out ] := T;                           // Fraction of day
      FOutputs[ o_outh] := MathFloatMod( H  * T, 1.0);  // Fraction of hour
      FOutputs[ o_outm] := MathFloatMod( M  * T, 1.0);  // Fraction of minute
      FOutputs[ o_outs] := MathFloatMod( S  * T, 1.0);  // Fraction of second
    end;


{ ========
  TModLfo = class( TModBaseLfo)
  public
}

    procedure   TModLfo.DoSlowTick; // override;
    begin
      inherited;                                // For phase calculations

      case FShape of

        0  : // sine

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( FPosition), Round( FInputs[ i_outputtype]));

        1  : // tri

          if FPosition <= 0.5
          then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 *         FPosition  - 1, Round( FInputs[ i_outputtype]))
          else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 * ( 1.0 - FPosition) - 1, Round( FInputs[ i_outputtype]));

        2  : // saw

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 2.0 * FPosition - 1, Round( FInputs[ i_outputtype]));

        3  : // square

          if FPosition < 0.5
          then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
          else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

        4  : // random PWM square

          if FRandomValue < 0
          then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
          else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

        5  : // random

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  FRandomValue, Round( FInputs[ i_outputtype]));

      end;
    end;


{ ========
  TModLfoMultiPhase = class( TModBaseLfo)
  public
}

    procedure   TModLfoMultiPhase.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_phaseshift, 'phaseshift');
      AddOutput( o_out2      , 'out2'      );
      AddOutput( o_out3      , 'out3'      );
      AddOutput( o_out4      , 'out4'      );
    end;


    procedure   TModLfoMultiPhase.DoSlowTick; // override;
    var
      aPos2  : TSignal;
      aPos3  : TSignal;
      aPos4  : TSignal;
      aShift : Integer;
    begin
      inherited;                                             // For phase calculations
      aShift := Round(FInputs[ i_phaseshift]);

      if aShift = 1                                          // 120 deg, three outputs used
      then begin
        aPos2 := MathFloatMod( FPosition + 0.33333333, 1.0);
        aPos3 := MathFloatMod( FPosition + 0.66666667, 1.0);
        aPos4 := MathFloatMod( FPosition + 0.5       , 1.0); // And add anti-phase as a bonus on the 4th output
      end
      else begin                                             // 90 deg
        aPos2 := MathFloatMod( FPosition + 0.75, 1.0);
        aPos3 := MathFloatMod( FPosition + 0.5 , 1.0);
        aPos4 := MathFloatMod( FPosition + 0.25, 1.0);
      end;

      case FShape of

        0  : // sine

          begin
            FOutputs[ o_out ] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( FPosition), Round( FInputs[ i_outputtype]));
            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( aPos2    ), Round( FInputs[ i_outputtype]));
            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( aPos3    ), Round( FInputs[ i_outputtype]));
            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( aPos4    ), Round( FInputs[ i_outputtype]));
          end;

        1  : // tri

          begin
            if FPosition <= 0.5
            then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 *         FPosition  - 1, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 * ( 1.0 - FPosition) - 1, Round( FInputs[ i_outputtype]));

            if aPos2 <= 0.5
            then FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 *         aPos2  - 1, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 * ( 1.0 - aPos2) - 1, Round( FInputs[ i_outputtype]));

            if aPos3 <= 0.5
            then FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 *         aPos3  - 1, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 * ( 1.0 - aPos3) - 1, Round( FInputs[ i_outputtype]));

            if aPos4 <= 0.5
            then FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 *         aPos4  - 1, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 * ( 1.0 - aPos4) - 1, Round( FInputs[ i_outputtype]));
          end;

        2  : // saw

          begin
            FOutputs[ o_out ] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 2.0 * FPosition - 1, Round( FInputs[ i_outputtype]));
            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 2.0 * aPos2     - 1, Round( FInputs[ i_outputtype]));
            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 2.0 * aPos3     - 1, Round( FInputs[ i_outputtype]));
            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 2.0 * aPos4     - 1, Round( FInputs[ i_outputtype]));
          end;

        3  : // square

          begin
            if FPosition < 0.5
            then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

            if aPos2 < 0.5
            then FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

            if aPos3 < 0.5
            then FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

            if aPos4 < 0.5
            then FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));
          end;

        4  : // random PWM square

          begin
            if FRandomValue < 0
            then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

            if FRandomValue < 0
            then FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

            if FRandomValue < 0
            then FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

            if FRandomValue < 0
            then FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
            else FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));
          end;

        5  : // random

          begin
            FOutputs[ o_out ] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  FRandomValue, Round( FInputs[ i_outputtype]));
            FOutputs[ o_out2] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  FRandomValue, Round( FInputs[ i_outputtype]));
            FOutputs[ o_out3] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  FRandomValue, Round( FInputs[ i_outputtype]));
            FOutputs[ o_out4] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  FRandomValue, Round( FInputs[ i_outputtype]));
          end;

      end;
    end;


{ ========
  TModLfoTrig = class( TModbaseTrigLfo)
  public
}

    procedure   TModLfoTrig.DoSlowTick; // override;
    begin
      inherited;                                // For phase calculations

      case FShape of

        0  : // sine

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( FPosition), Round( FInputs[ i_outputtype]));

        1  : // tri

          if FPosition <= 0.5
          then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 *         FPosition  - 1, Round( FInputs[ i_outputtype]))
          else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 4.0 * ( 1.0 - FPosition) - 1, Round( FInputs[ i_outputtype]));

        2  : // saw

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( 2.0 * FPosition - 1, Round( FInputs[ i_outputtype]));

        3  : // square

          if FPosition < 0.5
          then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
          else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

        4  : // random PWM square

          if FRandomValue < 0
          then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
          else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

        5  : // random

          FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  FRandomValue, Round( FInputs[ i_outputtype]));

      end;
    end;


{ ========
  TModSquareLfo = class( TModBaseLfo)
  public
}

    procedure   TModSquareLfo.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_pwm     , 'pwm'     );
      AddInput ( i_pwmlevel, 'pwmlevel');
    end;


    procedure   TModSquareLfo.SetDefaults;
    begin
      inherited;
      FDezipperMap := [
        i_frequency,
        i_fmlevel,
        i_pmlevel,
        i_pwmlevel
      ];
    end;

    procedure   TModSquareLfo.DoSlowTick; // override;
    begin
      inherited;                                // For phase calculations

      if FPosition < Clip( 0.5 * ( 1 - FInputs[ i_pwm] * FInputs[ i_pwmlevel]), 0.01, 0.99)
      then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
      else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));
    end;


{ ========
  TModSquareSineLfo = class( TModSquareLfo)
  public
}

    procedure   TModSquareSineLfo.CreateIO; // override;
    begin
      inherited;
      AddOutput( o_sine, 'sine');
    end;


    procedure   TModSquareSineLfo.SetDefaults;
    begin
      inherited;
      FDezipperMap := [
        i_frequency,
        i_fmlevel,
        i_pmlevel,
        i_pwmlevel
      ];
    end;

    procedure   TModSquareSineLfo.DoSlowTick; // override;
    begin
      inherited;                                // For phase calculations

      if FPosition < Clip( 0.5 * ( 1 - FInputs[ i_pwm] * FInputs[ i_pwmlevel]), 0.01, 0.99)
      then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
      else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));

      FOutputs[ o_sine] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( LookupSine( FPosition), Round( FInputs[ i_outputtype]));
    end;


{ ========
  TModTrigSquareLfo = class( TModBaseTrigLfo)
  public
}

    procedure   TModSquareLfoTrig.CreateIO; // override;
    begin
      inherited;
      AddInput( i_pwm     , 'pwm'     );
      AddInput( i_pwmlevel, 'pwmlevel');
    end;


    procedure   TModSquareLfoTrig.SetDefaults;
    begin
      inherited;
      FDezipperMap := [
        i_frequency,
        i_fmlevel,
        i_pmlevel,
        i_pwmlevel
      ];
    end;


    procedure   TModSquareLfoTrig.DoSlowTick; // override;
    begin
      inherited;                                // For phase calculations

      if FPosition < Clip( 0.5 * ( 1.0 - FInputs[ i_pwm] * FInputs[ i_pwmlevel]), 0.01, 0.99)
      then FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(  1.0, Round( FInputs[ i_outputtype]))
      else FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( -1.0, Round( FInputs[ i_outputtype]));
    end;


{ ========
  TModRandSig = class( TModCounterLfo)
  private
    FStart  : TSignal;
    FTarget : TSignal;
  protected
}

    procedure   TModRandSig.Wrapped; // override;
    var
      Dist   : TSignal;
      Lambda : TSignal;
      Mode   : Integer;
    begin
      Dist    := Clip( FInputs[ i_dist] + FInputs[ i_distmod] * FInputs[ i_distmodamt], 0, 1);
      Mode    := Round( FInputs[ i_mode]);
      FStart  := FTarget;

      if Mode = m_exp
      then begin
        Lambda  := Max( FInputs[ i_lambda], 1.0000001);
        FTarget := Clip( Dist * ( RandomExpo( Lambda, False)) + ( 1 - Dist) * FStart, -1, 1);
      end
      else FTarget := Clip( Dist * ( 1 - 2 * Random) + ( 1 - Dist) * FStart, -1, 1);
    end;


//  public

    procedure   TModRandSig.CreateIO; // override;
    begin
      inherited;
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_range     , 'range'     );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_speed     , 'speed'     );
      AddInput ( i_dist      , 'dist'      );
      AddInput ( i_distmod   , 'distmod'   );
      AddInput ( i_distmodamt, 'distmodamt');
      AddInput ( i_lambda    , 'lambda'    );
      AddInput ( i_mode      , 'mode'      );
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModRandSig.SetDefaults; // override;
    begin
      inherited;
      FTarget := 2 * Random - 1;
      Wrapped; // Start with a wrap to make it start moving from the beginning.
      FDezipperMap :=
        FDezipperMap +
        [
          i_dist,
          i_distmodamt
        ];
    end;


    procedure   TModRandSig.DoSlowTick; // override;
    var
      Value : TSignal;
    begin
      inherited;                                          // For phase calculation
      Value := FTarget + FPhase * ( FStart - FTarget);    // Phase goes from 1 -> 0
      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( Value, Round( FInputs[ i_outputtype]));
    end;


{ ========
  TModRandSigs = class( TModCounterLfo)
  private
    FPhases  : array[ 0 .. OUT_COUNT - 1] of TSignal;
    FStarts  : array[ 0 .. OUT_COUNT - 1] of TSignal;
    FTargets : array[ 0 .. OUT_COUNT - 1] of TSignal;
  protected
}

    procedure   TModRandSigs.Wrapped( anIndex: Integer);
    var
      Dist   : TSignal;
      Lambda : TSignal;
      Mode   : Integer;
      FbMod  : TSignal;
    begin
      FbMod             := ( FInputs[ i_distfb] + FInputs[ i_distfbmod] * FInputs[ i_distfbmodamt]) * FOutputs[ MathIntMod( anIndex - 2, OUT_COUNT)];
      Dist              := Clip( FInputs[ i_dist] + FInputs[ i_distmod] * FInputs[ i_distmodamt] * FbMod, 0, 1);
      Mode              := Round( FInputs[ i_mode]);
      FStarts[ anIndex] := FTargets[ anIndex];

      if Mode = m_exp
      then begin
        Lambda := Max( FInputs[ i_lambda], 1.0000001);
        FTargets[ anIndex] := Clip( Dist * ( RandomExpo( Lambda, False)) + ( 1 - Dist) * FStarts[ anIndex], -1, 1);
      end
      else FTargets[ anIndex] := Clip( Dist * ( 1 - 2 * Random) + ( 1 - Dist) * FStarts[ anIndex], -1, 1);
    end;


//  public

    procedure   TModRandSigs.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_frequency   , 'frequency'   );
      AddInput ( i_fm          , 'fm'          );
      AddInput ( i_fmlevel     , 'fmlevel'     );
      AddInput ( i_range       , 'range'       );
      AddInput ( i_mute        , 'mute'        );
      AddInput ( i_outputtype  , 'outputtype'  );
      AddInput ( i_freq        , 'freq'        );
      AddInput ( i_speed       , 'speed'       );
      AddInput ( i_dist        , 'dist'        );
      AddInput ( i_distmod     , 'distmod'     );
      AddInput ( i_distmodamt  , 'distmodamt'  );
      AddInput ( i_lambda      , 'lambda'      );
      AddInput ( i_mode        , 'mode'        );
      AddInput ( i_distfb      , 'distfb'      );
      AddInput ( i_distfbmod   , 'distfbmod'   );
      AddInput ( i_distfbmodamt, 'distfbmodamt');
      AddInput ( i_fmfb        , 'fmfb'        );
      AddInput ( i_fmfbmod     , 'fmfbmod'     );
      AddInput ( i_fmfbmodamt  , 'fmfbmodamt'  );
      AddOutput( o_out1        , 'out1'        );
      AddOutput( o_out2        , 'out2'        );
      AddOutput( o_out3        , 'out3'        );
      AddOutput( o_out4        , 'out4'        );
      AddOutput( o_out5        , 'out5'        );
      AddOutput( o_out6        , 'out6'        );
      AddOutput( o_out7        , 'out7'        );
      AddOutput( o_out8        , 'out8'        );
      AddOutput( o_outinv1     , 'outinv1'     );
      AddOutput( o_outinv2     , 'outinv2'     );
      AddOutput( o_outinv3     , 'outinv3'     );
      AddOutput( o_outinv4     , 'outinv4'     );
      AddOutput( o_outinv5     , 'outinv5'     );
      AddOutput( o_outinv6     , 'outinv6'     );
      AddOutput( o_outinv7     , 'outinv7'     );
      AddOutput( o_outinv8     , 'outinv8'     );
    end;


    procedure   TModRandSigs.SetDefaults; // override;
    var
      i : Integer;
    begin
      FInputs[ i_speed] := 0.5;

      for i := 0 to OUT_COUNT - 1
      do begin
        FTargets[ i] := 2 * Random - 1;
        Wrapped( i);                     // Start with a wrap to make it start moving from the beginning
      end;

      FDezipperMap :=
        FDezipperMap +
        [
          i_frequency   ,
          i_fmlevel     ,
          i_dist        ,
          i_distmodamt  ,
          i_lambda      ,
          i_distfb      ,
          i_distfbmod   ,
          i_distfbmodamt,
          i_fmfb        ,
          i_fmfbmodamt
        ];
    end;


    procedure   TModRandSigs.DoSlowTick; // override;
    var
      Delta    : TSignal;
      DidWrap  : Boolean;
      i        : Integer;
      Value    : TSignal;
      InvValue : TSignal;
      BaseFreq : TSignal;
      FbMod    : TSignal;
      FbFreq   : TSignal;
      Speed    : TSignal;
    begin
      if ResetFlag
      then begin
        for i := 0 to OUT_COUNT - 1
        do begin
          FPhases[ i] := Random;
          Wrapped( i);
        end;
      end;

      ResetFlag := False;
      Speed     := 2.0 * Clip( Finputs[ i_speed], 0.0, 1.0);
      BaseFreq  := FInputs[ i_frequency] + FInputs[ i_freq   ] + FInputs[ i_fm] * FInputs[ i_fmlevel];
      FbMod     := FInputs[ i_fmfb     ] + FInputs[ i_fmfbmod] * FInputs[ i_fmfbmodamt];
      FbFreq    := FOutputs[ o_out8] * FbMod;


      for i := 0 to OUT_COUNT - 1
      do begin
        case Round( FInputs[ i_range]) of
          0  : Delta := LookupLfoFastPhaseDelta   ( BaseFreq + FbFreq);
          1  : Delta := LookupLfoMediumPhaseDelta ( BaseFreq + FbFreq);
          2  : Delta := LookupLfoSlowPhaseDelta   ( BaseFreq + FbFreq);
          3  : Delta := LookupLfoBPMPhaseDelta    ( BaseFreq + FbFreq);
          4  : Delta := LookupLfoLinPhaseDelta    ( BaseFreq + FbFreq);
          5  : Delta := LookupLfoFast2PhaseDelta  ( BaseFreq + FbFreq);
          6  : Delta := LookupLfoMedium2PhaseDelta( BaseFreq + FbFreq);
          else Delta := LookupLfoSlow2PhaseDelta  ( BaseFreq + FbFreq);
        end;

        FbFreq      := FOutputs[ o_out1 + i] * FbMod;
        FPhases[ i] := FPhases[ i] - Speed * Delta;
        DidWrap     := False;

        while FPhases[ i] < 0
        do begin
          if not DidWrap
          then begin
            Wrapped( i);
            DidWrap := True;
          end;

          FPhases[ i] := FPhases[ i] + 1.0;
        end;

        Value    := FTargets[ i] + FPhases[ i] * ( FStarts[ i] - FTargets[ i]);    // Phase goes from 1 -> 0
        InvValue := - Value;
        FOutputs[ o_out1 + i]    := SignalToMute( FInputs[ i_mute]) * SignalForOutputType(    Value, Round( FInputs[ i_outputtype]));
        FOutputs[ o_outinv1 + i] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( InvValue, Round( FInputs[ i_outputtype]));
      end;
    end;


    procedure   TModRandSigs.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := 0 to OUT_COUNT - 1
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'active%d', [ i + 1], AppLocale), LogicToSignal( FPhases[ i] >= 0.5)));
    end;


{ ========
  TModPulses = class( TMod)
  private
    FPhase         : TSignal;
    FState         : TPulsesState;
    FPulseCounter  : Integer;
    FPeriodCounter : Integer;
    FOldFrequency  : TSignal;
    FOldLambda     : TSignal;
    FOldDist       : TSignal;
  public
}

    procedure   TModPulses.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_range     , 'range'     );
      AddInput ( i_dist      , 'dist'      );
      AddInput ( i_distmod   , 'distmod'   );
      AddInput ( i_distmodamt, 'distmodamt');
      AddInput ( i_lambda    , 'lambda'    );
      AddInput ( i_rndmode   , 'rndmode'   );
      AddInput ( i_syncmode  , 'syncmode'  );
      AddInput ( i_outputtype, 'outputtype');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModPulses.SetDefaults; // override;
    begin
      FDezipperMap :=
        [
          i_dist      ,
          i_distmodamt,
          i_lambda
        ];
    end;


    procedure   TModPulses.DoSlowTick; // override;
    var
      Delta      : TSignal;
      Frequency  : TSignal;
      Range      : Integer;
      Dist       : TSignal;
      Lambda     : TSignal;
      RndMode    : Integer;
      SyncMode   : Integer;
      OutputType : Integer;
      Wrap       : Boolean;
    begin
      Frequency  := Max( 1e-6, FInputs[ i_frequency]);
      Range      := Round( FInputs[ i_range]);
      Dist       := FInputs[ i_dist];
      Lambda     := Max( FInputs[ i_lambda], 1.0000001);
      RndMode    := Round( FInputs[ i_rndmode]);
      SyncMode   := Round( FInputs[ i_syncmode]);
      OutputType := Round( FInputs[ i_outputtype]);

      if
        ResetFlag or
        ( FOldFrequency <> Frequency) or
        ( FOldLambda    <> Lambda   ) or
        ( FOldDist      <> Dist     )
      then begin
        ResetFlag := False;
        FPhase    := 0.0;
        FState    := psIdle;
      end;

      FOldFrequency := Frequency;
      FOldLambda    := Lambda;
      FOldDist      := Dist;

      Dist := Clip( Dist + FInputs[ i_distmod] * FInputs[ i_distmodamt], 0, 1);

      case Range of
        0  : Delta := LookupLfoFastPhaseDelta   ( Frequency);
        1  : Delta := LookupLfoMediumPhaseDelta ( Frequency);
        2  : Delta := LookupLfoSlowPhaseDelta   ( Frequency);
        3  : Delta := LookupLfoBPMPhaseDelta    ( Frequency);
        4  : Delta := LookupLfoLinPhaseDelta    ( Frequency);
        5  : Delta := LookupLfoFast2PhaseDelta  ( Frequency);
        6  : Delta := LookupLfoMedium2PhaseDelta( Frequency);
        else Delta := LookupLfoSlow2PhaseDelta  ( Frequency);
      end;

      FPhase := FPhase + Delta;
      Wrap   := False;

      while FPhase > 1.0
      do begin
        FPhase := FPhase - 1.0;
        Wrap   := True;
      end;

      if ( SyncMode = m_sync_async) or Wrap
      then begin
        case FState of

          psIdle : begin
            FPulseCounter := TrigPulseTime( IsSpedUp);
            FLedCounter   := LedFlashTime ( IsSpedUp);

            if RndMode = m_rnd_exp
            then begin
              if SyncMode = m_sync_async
              then FPeriodCounter := Max( 1, Trunc(( Control_Rate / Frequency ) * Dist * RandomExpo( Lambda, True)))
              else FPeriodCounter := Trunc( 25 * Dist * RandomExpo( Lambda, True));
            end
            else begin
              if SyncMode = m_sync_async
              then FPeriodCounter := Max( 1, Trunc(( Control_Rate / Frequency ) * Dist * Random))
              else FPeriodCounter := Trunc( 25 * Dist * Random);
            end;

            FState := psPulsing;
          end;

          else begin
            if FPeriodCounter > 0
            then Dec( FPeriodCounter)
            else FState := psIdle;
          end;
        end;
      end;

      DecToZero( FLedCounter);

      if FState = psPulsing
      then begin
        if FPulseCounter > 0
        then Dec( FPulseCounter)
        else FState := psWaiting;
      end;

      FOutputs[ o_out] := SignalForOutputType( LogicToSignal( FState = psPulsing), OutputType);
    end;


    procedure   TModPulses.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out', LogicToSignal( FLedCounter > 0)));
    end;


{ ========
  TModRandomWalkLfo = class( TModBaseLfo)
  private
    FPosition : TSignal;
    FState    : TRWState;
    FX        : TSignal;
    FY        : TSignal;
    FZ        : TSignal;
    FPrevX    : TSignal;
    FPrevY    : TSignal;
    FPrevZ    : TSignal;
  public
}

    procedure   TModRandomWalkLfo.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_range      , 'range'      );
      AddInput ( i_frequency  , 'frequency'  );
      AddInput ( i_fm         , 'fm'         );
      AddInput ( i_fmlevel    , 'fmlevel'    );
      AddInput ( i_distance   , 'distance'   );
      AddInput ( i_mute       , 'mute'       );
      AddInput ( i_color      , 'color'      );
      AddInput ( i_colormod   , 'colormod'   );
      AddInput ( i_colormodamt, 'colormodamt');
      AddInput ( i_outputtype , 'outputtype' );
      AddInput ( i_lambda     , 'lambda'     );
      AddInput ( i_speed      , 'speed'      );
      AddInput ( i_mode       , 'mode'       );
      AddInput ( i_smode      , 'smode'      );
      AddInput ( i_distancemod, 'distancemod');
      AddOutput( o_outx       , 'outx'       );
      AddOutput( o_outy       , 'outy'       );
      AddOutput( o_outz       , 'outz'       );
      AddOutput( o_sync       , 'sync'       );
    end;


    procedure   TModRandomWalkLfo.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_fmlevel,
        i_distance,
        i_color,
        i_colormodamt
      ];
      FInputs[ i_fm         ] := 0.0;
      FInputs[ i_colormod   ] := 0.0;
      FInputs[ i_distancemod] := 0.0;
      FInputs[ i_speed      ] := 1.0;
    end;


    procedure   TModRandomWalkLfo.DoSlowTick; // override;
    var
      Distance : TSignal;
      Delta    : TSignal;
      Color    : TSignal;
      Mute     : TSignal;
      OutType  : Integer;
      Lambda   : TSignal;
      Mode     : Integer;
      Wrapped  : Boolean;
      OldPos   : TSignal;
      Speed    : TSignal;
    begin
      if ResetFlag
      then begin
        FX := 1.0 - 2.0 * Random;
        FY := 1.0 - 2.0 * Random;
        FZ := 1.0 - 2.0 * Random;
        ResetFlag := False;
      end;

      case Round( FInputs[ i_range]) of
        0  : Delta := LookupLfoFastPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        1  : Delta := LookupLfoMediumPhaseDelta ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        2  : Delta := LookupLfoSlowPhaseDelta   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        3  : Delta := LookupLfoBPMPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        4  : Delta := LookupLfoLinPhaseDelta    ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        5  : Delta := LookupLfoFast2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        6  : Delta := LookupLfoMedium2PhaseDelta( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        else Delta := LookupLfoSlow2PhaseDelta  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
      end;

      OldPos    := FPosition;
      Speed     := Normalize( FInputs[ i_speed]) * 2;
      FPosition := MathFloatMod( FPosition + Speed * Delta, 1.0);
      Wrapped   := FPosition < OldPos;
      Mute      := SignalToMute( FInputs[ i_mute      ]);
      OutType   := Round       ( FInputs[ i_outputtype]);

      case FState of

        trwIdle :

          begin
            if Wrapped
            then begin
              FState   := trwFired;
              Lambda   := Max ( FInputs[ i_lambda  ], 1.0000001);
              Distance := Clip( FInputs[ i_distance] + FInputs[ i_distancemod], -1.0, 1.0);
              Color    := Clip( FInputs[ i_color   ] + FInputs[ i_colormod   ] * FInputs[ i_colormodamt], 0, 1);
              Mode     := Round( FInputs[ i_mode]);
              FPrevX   := FX;
              FPrevY   := FY;
              FPrevZ   := FZ;

              if Mode = m_exp
              then begin
                FX := ( 1 - Color) * FX + Color * ( FX + RandomExpo( Lambda, False) * Distance);
                FY := ( 1 - Color) * FY + Color * ( FY + RandomExpo( Lambda, False) * Distance);
                FZ := ( 1 - Color) * FZ + Color * ( FZ + RandomExpo( Lambda, False) * Distance);
              end
              else begin
                FX := ( 1 - Color) * FX + Color * ( FX + ( 1.0 - 2.0 * Random) * Distance);
                FY := ( 1 - Color) * FY + Color * ( FY + ( 1.0 - 2.0 * Random) * Distance);
                FZ := ( 1 - Color) * FZ + Color * ( FZ + ( 1.0 - 2.0 * Random) * Distance);
              end;

              if FX < -1 then FX := - 2 - FX else if FX > 1 then FX := 2 - FX;
              if FY < -1 then FY := - 2 - FY else if FY > 1 then FY := 2 - FY;
              if FZ < -1 then FZ := - 2 - FZ else if FZ > 1 then FZ := 2 - FZ;

              FOutputs[ o_sync] := LogicToSignal( True);
            end;
          end;

        trwFired :

          begin
            if FPosition > 0.5
            then begin
              FState := trwIdle;
              FOutputs[ o_sync] := LogicToSignal( False);
            end;
          end;

      end;

      if Round( FInputs[ i_smode]) = 1
      then begin
        FOutputs[ o_outx] := Mute * SignalForOutputType( FPosition * FX + ( 1.0 - FPosition) * FPrevX, OutType);
        FOutputs[ o_outy] := Mute * SignalForOutputType( FPosition * FY + ( 1.0 - FPosition) * FPrevY, OutType);
        FOutputs[ o_outz] := Mute * SignalForOutputType( FPosition * FZ + ( 1.0 - FPosition) * FPrevZ, OutType);
      end
      else begin
        FOutputs[ o_outx] := Mute * SignalForOutputType( FX, OutType);
        FOutputs[ o_outy] := Mute * SignalForOutputType( FY, OutType);
        FOutputs[ o_outz] := Mute * SignalForOutputType( FZ, OutType);
      end;
    end;


    procedure   TModRandomWalkLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'outx', FX));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outy', FY));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outz', FZ));
    end;


{ ========
  TModAttractorLfo = class( TModBaseLfo)
  private
    FLorenz  : TLorenzOsc;
    FRossler : TRosslerOsc;
    FX       : TSignal;
    FY       : TSignal;
    FZ       : TSignal;
  public
}

    constructor TModAttractorLfo.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FLorenz  := TLorenzOsc .Create;
      FRossler := TRosslerOsc.Create;
    end;


    procedure   TModAttractorLfo.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_range     , 'range'     );
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_drive     , 'drive'     );
      AddInput ( i_driveamt  , 'driveamt'  );
      AddInput ( i_mode      , 'mode'      );
      AddOutput( o_outx      , 'outx'      );
      AddOutput( o_outy      , 'outy'      );
      AddOutput( o_outz      , 'outz'      );
    end;


    destructor  TModAttractorLfo.Destroy; // override;
    begin
      FLorenz .DisposeOf;
      FRossler.DisposeOf;
      inherited;
    end;


    procedure   TModAttractorLfo.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_fm       ,
        i_fmlevel
      ];
    end;


    procedure   TModAttractorLfo.DoSlowTick; // override;
    var
      Mode      : Integer;
      OutType   : Integer;
      Range     : Integer;
      Mute      : TSignal;
      Frequency : TSignal;
      Drive     : TSignal;
    begin
      if Assigned( FLorenz) and Assigned( FRossler)
      then begin
        Mode    := Round       ( FInputs[ i_mode      ]);
        OutType := Round       ( FInputs[ i_outputtype]);
        Range   := Round       ( FInputs[ i_range     ]);
        Mute    := SignalToMute( FInputs[ i_mute      ]);
        Drive   := FInputs[ i_drive] * FInputs[ i_driveamt];

        case Range of
          0  : Frequency := LookupLfoFastFrequency  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          1  : Frequency := LookupLfoMediumFrequency( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          2  : Frequency := LookupLfoSlowFrequency  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          3  : Frequency := LookupLfoBPMFrequency   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          else Frequency := LookupLfoLinFrequency   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        end;

        case Mode of
          m_lorenz : begin if ResetFlag then FLorenz .Reset; FLorenz .SetFreq( Frequency); FLorenz .Tick( Drive, FX, FY, FZ); end;
          else       begin if ResetFlag then FRossler.Reset; FRossler.SetFreq( Frequency); FRossler.Tick( Drive, FX, FY, FZ); end;
        end;

        FOutputs[ o_outx] := SignalForOutputType( FX * Mute, OutType);
        FOutputs[ o_outy] := SignalForOutputType( FY * Mute, OutType);
        FOutputs[ o_outz] := SignalForOutputType( FZ * Mute, OutType);
      end;

      ResetFlag := False;
    end;


    procedure   TModAttractorLfo.SampleRateChanged; // override;
    begin
      if Assigned( FLorenz ) then FLorenz .SetSampleRate( Control_Rate);
      if Assigned( FRossler) then FRossler.SetSampleRate( Control_Rate);
    end;


    procedure   TModAttractorLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'outx', FX));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outy', FY));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outz', FZ));
    end;


{ ========
  TModVanDerPolLfo = class( TMod)
  private
    FVanDerPol : TVanDerPolOsc;
    FX         : TSignal;
    FY         : TSignal;
  public
}

    constructor TModVanDerPolLfo.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FVanDerPol := TVanDerPolOsc .Create( 32);
    end;


    procedure   TModVanDerPolLfo.CreateIO; // override;
    begin
      FIsSlow := True;

      AddInput ( i_range     , 'range'     );
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_mute      , 'mute'      );
      AddInput ( i_outputtype, 'outputtype');
      AddInput ( i_mu        , 'mu'        );
      AddInput ( i_mumodamt  , 'mumodamt'  );
      AddInput ( i_mumod     , 'mumod'     );
      AddInput ( i_in        , 'in'        );
      AddInput ( i_inamt     , 'inamt'     );

      AddOutput( o_outx      , 'outx'      );
      AddOutput( o_outy      , 'outy'      );

      SampleRateChanged;
    end;


    destructor  TModVanDerPolLfo.Destroy; // override;
    begin
      FVanDerPol.DisposeOf;
      inherited;
    end;


    procedure   TModVanDerPolLfo.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_fm       ,
        i_fmlevel  ,
        i_mu
      ];
    end;


    procedure   TModVanDerPolLfo.DoSlowTick; // override;
    var
      OutType   : Integer;
      Range     : Integer;
      Mute      : TSignal;
      Mu        : TSignal;
      Drive     : TSignal;
      Frequency : TSignal;
    begin
      if Assigned( FVanDerPol)
      then begin
        OutType := Round       ( FInputs[ i_outputtype]);
        Range   := Round       ( FInputs[ i_range     ]);
        Mute    := SignalToMute( FInputs[ i_mute      ]);
        Mu      := FInputs[ i_mu] + FInputs[ i_mumod] * FInputs[ i_mumodamt];
        Drive   := FInputs[ i_in] * FInputs[ i_inamt];

        case Range of
          0  : Frequency := LookupLfoFastFrequency  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          1  : Frequency := LookupLfoMediumFrequency( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          2  : Frequency := LookupLfoSlowFrequency  ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          3  : Frequency := LookupLfoBPMFrequency   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
          else Frequency := LookupLfoLinFrequency   ( FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
        end;

        if ResetFlag
        then FVanDerPol.Reset;

        FVanDerPol.SetFreq( Frequency);
        FVanDerPol.Tick( Drive, Mu, FX, FY);

        FOutputs[ o_outx] := SignalForOutputType( FX * Mute, OutType);
        FOutputs[ o_outy] := SignalForOutputType( FY * Mute, OutType);
      end;

      ResetFlag := False;
    end;


    procedure   TModVanDerPolLfo.SampleRateChanged; // override;
    begin
      if Assigned( FVanDerPol)
      then FVanDerPol.SetSampleRate( Control_Rate);
    end;


    procedure   TModVanDerPolLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'outx', FX));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outy', FY));
    end;


{ ========
  TModNoise = class( TMod)
  public
}

    procedure   TModNoise.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_color      , 'color'      );
      AddInput ( i_colormod   , 'colormod'   );
      AddInput ( i_colormodamt, 'colormodamt');
      AddInput ( i_mute       , 'mute'       );
      AddInput ( i_type       , 'type'       );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModNoise.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_color,
        i_colormodamt
      ];
    end;


    procedure   TModNoise.DoTick; // override;
    var
      Color     : TSignal;
      NoiseType : Integer;
    begin
      Color     := Clip( FInputs[ i_color] + FInputs[ i_colormod] * FInputs[ i_colormodamt], 0, 1);
      NoiseType := Round( FInputs[ i_type]);

      case NoiseType of
        nt_gauss :      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomGaus( 1.0, True)) / Sqrt( 1 + Color));
        nt_exp   :      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomExpo( 3.0, True)) / Sqrt( 1 + Color));
        else { nt_lin } FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * Random                ) / Sqrt( 1 + Color));
      end;

    end;


{ ========
  TModNoiseTrig = class( TMod)
  private
    FPrevInput: TSignal;
  public
}

    procedure   TModNoiseTrig.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_color      , 'color'      );
      AddInput ( i_colormod   , 'colormod'   );
      AddInput ( i_colormodamt, 'colormodamt');
      AddInput ( i_trig       , 'trig'       );
      AddInput ( i_mute       , 'mute'       );
      AddInput ( i_type       , 'type'       );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModNoiseTrig.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_color,
        i_colormodamt
      ];
    end;


    procedure   TModNoiseTrig.DoTick; // override;
    var
      Color     : TSignal;
      Trig      : Boolean;
      NoiseType : Integer;
    begin
      Trig  := SignalToLogic( FInputs[ i_trig]);

      if not FPrevTrig and Trig
      then begin
        Color     := Clip( FInputs[ i_color] + FInputs[ i_colormod] * FInputs[ i_colormodamt], 0, 1);
        NoiseType := Round( FInputs[ i_type]);
        FPrevTrig := Trig;

        case NoiseType of
          nt_gauss :      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomGaus( 1.0, True)) / Sqrt( 1 + Color));
          nt_exp   :      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomExpo( 3.0, True)) / Sqrt( 1 + Color));
          else { nt_lin } FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * Random                ) / Sqrt( 1 + Color));
        end;
      end
      else if FPrevTrig and not Trig
      then FPrevTrig := Trig;
    end;


    procedure   TModNoiseTrig.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig', FInputs[ i_trig]));
    end;


{ ========
  TModNoiseLfo = class( TMod)
  private
    FOutValue : TSignal;
  public
}

    procedure   TModNoiseLfo.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_color      , 'color'      );
      AddInput ( i_colormod   , 'colormod'   );
      AddInput ( i_colormodamt, 'colormodamt');
      AddInput ( i_mute       , 'mute'       );
      AddInput ( i_outputtype , 'outputtype' );
      AddInput ( i_type       , 'type'       );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModNoiseLfo.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_color,
        i_colormodamt
      ];
    end;


    procedure   TModNoiseLfo.DoSlowTick; // override;
    var
      Color     : TSignal;
      NoiseType : Integer;
    begin
      Color     := Clip( FInputs[ i_color] + FInputs[ i_colormod] * FInputs[ i_colormodamt], 0, 1);
      NoiseType := Round( FInputs[ i_type]);

      case NoiseType of
        nt_gauss :      FOutValue := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomGaus( 1.0, True)) / Sqrt( 1 + Color));
        nt_exp   :      FOutValue := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomExpo( 3.0, True)) / Sqrt( 1 + Color));
        else { nt_lin } FOutValue := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * Random                ) / Sqrt( 1 + Color));
      end;

      FOutputs[ o_out] := SignalToMute( FInputs[ i_mute]) * SignalForOutputType( FOutValue, Round( FInputs[ i_outputtype]));
    end;


    procedure   TModNoiseLfo.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'out', FOutValue));
    end;


{ ========
  TModNoiseLfoTrig = class( TMod)
  private
    FPrevTrig : Boolean;
    FOutValue : TSignal;
  public
}

    procedure   TModNoiseLfoTrig.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_color      , 'color'      );
      AddInput ( i_colormod   , 'colormod'   );
      AddInput ( i_colormodamt, 'colormodamt');
      AddInput ( i_trig       , 'trig'       );
      AddInput ( i_mute       , 'mute'       );
      AddInput ( i_outputtype , 'outputtype' );
      AddInput ( i_type       , 'type'       );
      AddOutput( o_out        , 'out'        );
    end;


    procedure   TModNoiseLfoTrig.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_color,
        i_colormodamt
      ];
    end;


    procedure   TModNoiseLfoTrig.DoSlowTick; // override;
    var
      Trig      : Boolean;
      Color     : TSignal;
      NoiseType : Integer;
    begin
      Trig  := SignalToLogic( FInputs[ i_trig]);

      if not FPrevTrig and Trig
      then begin
        Color     := Clip( FInputs[ i_color] + FInputs[ i_colormod] * FInputs[ i_colormodamt], 0, 1);
        NoiseType := Round( FInputs[ i_type]);

        case NoiseType of
          nt_gauss :      FOutValue := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomGaus( 1.0, True)) / Sqrt( 1 + Color));
          nt_exp   :      FOutValue := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * RandomExpo( 3.0, True)) / Sqrt( 1 + Color));
          else { nt_lin } FOutValue := SignalToMute( FInputs[ i_mute]) * (( 1 - Color) * FOutputs[ o_out] + Color * ( 1.0 - 2 * Random                ) / Sqrt( 1 + Color));
        end;

        FOutputs[ o_out] :=  SignalToMute( FInputs[ i_mute]) * SignalForOutputType( FOutValue, Round( FInputs[ i_outputtype]));
        FPrevTrig := Trig;
      end
      else if FPrevTrig and not Trig
      then FPrevTrig := Trig;
    end;


    procedure   TModNoiseLfoTrig.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig', FInputs [ i_trig]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out' , FOutValue        ));
    end;


{ ========
  TModPerlinTone = class( TMod)
  private
    FNoise      : TPerlinNoise;
    FLayerCount : Integer;
    FPhase      : TSignal;
  public
    property    LayerCount : Integer read FLayerCount write SetLayerCount;
  private
}

    procedure   TModPerlinTone.Recreate;
    var
      OldNoise : TPerlinNoise;
      NewNoise : TPerlinNoise;
    begin
      if LayerCount > 0
      then NewNoise := TPerlinNoise.Create( LayerCount, MAX_TONE_LAYER_LEN)
      else NewNoise := nil;

      Locked := True;

      try
        OldNoise := AtomicExchange( Pointer( FNoise), Pointer( NewNoise));
      finally
        Locked := False;
        FreeAndNil( OldNoise);
      end;
    end;


    procedure   TModPerlinTone.SetLayerCount( aValue: Integer);
    begin
      if aValue <> FLayerCount
      then begin
        FLayerCount := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModPerlinTone.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      LayerCount := 8;
    end;


    procedure   TModPerlinTone.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_roughness   , 'roughness'   );
      AddInput ( i_new         , 'new'         );
      AddInput ( i_newinv      , 'newinv'      );
      AddInput ( i_mute        , 'mute'        );
      AddInput ( i_roughmod    , 'roughmod'    );
      AddInput ( i_length      , 'length'      );
      AddInput ( i_lengthmod   , 'lengthmod'   );
      AddInput ( i_lengthmodamt, 'lengthmodamt');
      AddInput ( i_frequency   , 'frequency'   );
      AddInput ( i_freq        , 'freq'        );
      AddInput ( i_fm          , 'fm'          );
      AddInput ( i_fmlevel     , 'fmlevel'     );
      AddInput ( i_cents       , 'cents'       );
      AddOutput( o_out         , 'out'         );
    end;


    destructor  TModPerlinTone.Destroy; // override;
    begin
      FLayerCount := 0;
      Recreate;
      inherited;
    end;


    procedure   TModPerlinTone.SetDefaults; // override;
    begin
      FInputs[ i_roughmod] := 1;
      FInputs[ i_new     ] := 1;
      FDezipperMap := [
        i_roughness
      ];
    end;


    procedure   TModPerlinTone.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'layers')
      then LayerCount := Round( aValue)
      else inherited;
    end;


    function    TModPerlinTone.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'layers')
      then Result := LayerCount
      else Result := inherited;
    end;


    procedure   TModPerlinTone.DoTick; // override;
    var
      MakeNewValue : Boolean;
      Roughness    : TSignal;
      aValue       : TSignal;
      aLength      : TSignal;
    begin
      if Assigned( FNoise) and ( LayerCount > 0)
      then begin
        if ResetFlag
        then begin
          ResetFlag := False;
          FPhase    := 0;
        end
        else begin
          FPhase := FPhase + LookupPhaseDelta( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents]);

          if FPhase > 1
          then FPhase := FPhase - 1;
        end;

        MakeNewValue := SignalToLogic( FInputs[ i_new]) xor SignalToLogic( FInputs[ i_newinv]);
        Roughness    := Normalize( FInputs[ i_roughmod] * FInputs[ i_roughness]);
        aLength      := Clip( Normalize( FInputs[ i_length] + FInputs[ i_lengthmod] * FInputs[ i_lengthmodamt]), 0, 1);
        FNoise.TickLP( aValue, Roughness, aLength, FPhase, MakeNewValue);
        FOutputs[ o_out] := Normalize( aValue * SignalToMute( FInputs[ i_mute]));
      end;
    end;


{ ========
  TModPerlinNoise = class( TMod)
  private
    FNoise       : TPerlinNoise;
    FLayerCount  : Integer;
  public
    property    LayerCount : Integer read FLayerCount write SetLayerCount;
  private
}

    procedure   TModPerlinNoise.Recreate;
    var
      OldNoise : TPerlinNoise;
      NewNoise : TPerlinNoise;
    begin
      if LayerCount > 0
      then NewNoise := TPerlinNoise.Create( LayerCount, MAX_NOISE_LAYER_LEN)
      else NewNoise := nil;

      Locked := True;

      try
        OldNoise := AtomicExchange( Pointer( FNoise), Pointer( NewNoise));
      finally
        Locked := False;
        FreeAndNil( OldNoise);
      end;
    end;


    procedure   TModPerlinNoise.SetLayerCount( aValue: Integer);
    begin
      if aValue <> FLayerCount
      then begin
        FLayerCount := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModPerlinNoise.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      LayerCount := 8;
    end;


    procedure   TModPerlinNoise.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_roughness   , 'roughness'   );
      AddInput ( i_new         , 'new'         );
      AddInput ( i_newinv      , 'newinv'      );
      AddInput ( i_mute        , 'mute'        );
      AddInput ( i_roughmod    , 'roughmod'    );
      AddInput ( i_length      , 'length'      );
      AddInput ( i_lengthmod   , 'lengthmod'   );
      AddInput ( i_lengthmodamt, 'lengthmodamt');
      AddOutput( o_out         , 'out'         );
    end;


    destructor  TModPerlinNoise.Destroy; // override;
    begin
      FLayerCount := 0;
      Recreate;
      inherited;
    end;


    procedure   TModPerlinNoise.SetDefaults; // override;
    begin
      FInputs[ i_roughmod] := 1;
      FInputs[ i_new     ] := 1;
      FDezipperMap := [
        i_roughness
      ];
    end;


    procedure   TModPerlinNoise.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'layers')
      then LayerCount := Round( aValue)
      else inherited;
    end;


    function    TModPerlinNoise.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'layers')
      then Result := LayerCount
      else Result := inherited;
    end;


    procedure   TModPerlinNoise.DoTick; // override;
    var
      MakeNewValue : Boolean;
      Roughness    : TSignal;
      aValue       : TSignal;
      aLength      : TSignal;
    begin
      if Assigned( FNoise) and ( LayerCount > 0)
      then begin
        MakeNewValue := SignalToLogic( FInputs[ i_new]) xor SignalToLogic( FInputs[ i_newinv]);
        Roughness    := Normalize( FInputs[ i_roughmod] * FInputs[ i_roughness]);
        aLength      := Clip( Normalize( FInputs[ i_length] + FInputs[ i_lengthmod] * FInputs[ i_lengthmodamt]), 0, 1);
        FNoise.TickL( aValue, Roughness, aLength, MakeNewValue);
        FOutputs[ o_out] := Normalize( aValue * SignalToMute( FInputs[ i_mute]));
      end;
    end;


{ ========
  TModPerlinNoiseTrig = class( TMod)
  private
    FNoise      : TPerlinNoise;
    FOldTrig    : Boolean;
    FLayerCount : Integer;
  public
    property    LayerCount : Integer read FLayerCount write SetLayerCount;
  private
}

    procedure   TModPerlinNoiseTrig.Recreate;
    var
      OldNoise : TPerlinNoise;
      NewNoise : TPerlinNoise;
    begin
      if LayerCount > 0
      then NewNoise := TPerlinNoise.Create( LayerCount, MAX_NOISE_LAYER_LEN)
      else NewNoise := nil;

      Locked := True;

      try
        OldNoise := AtomicExchange( Pointer( FNoise), Pointer( NewNoise));
      finally
        Locked := False;
        FreeAndNil( OldNoise);
      end;
    end;


    procedure   TModPerlinNoiseTrig.SetLayerCount( aValue: Integer);
    begin
      if aValue <> FLayerCount
      then begin
        FLayerCount := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModPerlinNoiseTrig.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      LayerCount := 8;
    end;


    procedure   TModPerlinNoiseTrig.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_roughness   , 'roughness'   );
      AddInput ( i_new         , 'new'         );
      AddInput ( i_newinv      , 'newinv'      );
      AddInput ( i_mute        , 'mute'        );
      AddInput ( i_roughmod    , 'roughmod')   ;
      AddInput ( i_length      , 'length'      );
      AddInput ( i_lengthmod   , 'lengthmod'   );
      AddInput ( i_lengthmodamt, 'lengthmodamt');
      AddInput ( i_trig        , 'trig'        );
      AddOutput( o_out         , 'out'         );
    end;


    destructor  TModPerlinNoiseTrig.Destroy; // override;
    begin
      FLayerCount := 0;
      Recreate;
      inherited;
    end;


    procedure   TModPerlinNoiseTrig.SetDefaults; // override;
    begin
      FInputs[ i_roughmod] := 1;
      FInputs[ i_new     ] := 1;
      FDezipperMap := [
        i_roughness
      ];
    end;


    procedure   TModPerlinNoiseTrig.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'layers')
      then LayerCount := Round( aValue)
      else inherited;
    end;


    function    TModPerlinNoiseTrig.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'layers')
      then Result := LayerCount
      else Result := inherited;
    end;


    procedure   TModPerlinNoiseTrig.DoTick; // override;
    var
      MakeNewValue : Boolean;
      Roughness    : TSignal;
      aValue       : TSignal;
      Trig         : Boolean;
      aLength      : TSignal;
    begin
      if Assigned( FNoise) and ( LayerCount > 0)
      then begin
        Trig := SignalToLogic( FInputs[ i_trig]);

        if Trig and not FOldTrig
        then begin
          MakeNewValue := SignalToLogic( FInputs[ i_new]) xor SignalToLogic( FInputs[ i_newinv]);
          Roughness    := Normalize( FInputs[ i_roughmod] * FInputs[ i_roughness]);
          aLength      := Clip( Normalize( FInputs[ i_length] + FInputs[ i_lengthmod] * FInputs[ i_lengthmodamt]), 0, 1);
          FNoise.TickL( aValue, Roughness, aLength, MakeNewValue);
          FOutputs[ o_out] := Normalize( aValue * SignalToMute( FInputs[ i_mute]));
        end;

        FOldTrig := Trig;
      end;
    end;


{ ========
  TModAverage = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  private
    FAlpha : TSignal;
  public
}


    procedure   TModAverage.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in         , 'in'         );
      AddInput ( i_alpha      , 'alpha'      );
      AddInput ( i_alphamod   , 'alphamod'   );
      AddInput ( i_alphamodamt, 'alphamodamt');
      AddOutput( o_lp         , 'lp'         );
      AddOutput( o_hp         , 'hp'         );
    end;


    procedure   TModAverage.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_alpha,
        i_alphamodamt
      ];
    end;


    procedure   TModAverage.DoSlowTick; // override;
    var
      Inp   : TSignal;
      Outp  : TSignal;
      Delta : TSignal;
    begin
      FAlpha := Normalize( ExpAttack( FInputs[ i_alpha] + FInputs[ i_alphamod] * FInputs[ i_alphamodamt], 1, 0, 1));

      if ResetFlag
      then begin
        Inp   := 0;
        Outp  := 0;
      end
      else begin
        Inp   := FInputs [ i_in];
        Outp  := FOutputs[ o_lp];
      end;

      Delta           := Normalize( Inp  - Outp);
      Outp            := Normalize( Outp + FAlpha * Delta);
      FOutputs[ o_lp] := Outp;
      FOutputs[ o_hp] := Delta;
      ResetFlag       := False;
    end;


{ ========
  TModUnclick = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha))) - with a fixed alpha
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  public
}

    procedure   TModUnclick.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in , 'in' );
      AddOutput( o_out, 'out');
    end;


    procedure   TModUnclick.DoSlowTick; // override;
    var
      Inp   : TSignal;
      Outp  : TSignal;
      Delta : TSignal;
    begin
      if ResetFlag
      then begin
        Inp   := 0;
        Outp  := 0;
      end
      else begin
        Inp   := FInputs [ i_in ];
        Outp  := FOutputs[ o_out];
      end;

      Delta            := Normalize( Inp  - Outp);
      Outp             := Normalize( Outp + 0.33 * Delta);
      FOutputs[ o_out] := Outp;
      ResetFlag        := False;
    end;


{ ========
  TModMovingAverage = class( TMod)
  private
    FSize    : Integer;
    FPointer : Integer;
    FHistory : array[ 0 .. MAX_SIZE - 1] of TSignal;
  public
}

    procedure   TModMovingAverage.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in  , 'in'  );
      AddInput ( i_size, 'size');
      AddOutput( o_out , 'out' );
    end;


    procedure   TModMovingAverage.SetDefaults; // override;
    var
      i : Integer;
    begin
      FSize := 1;

      for i := Low( FHistory) to High( FHistory)
      do FHistory[ i] := 0.0;

      FPointer := 0;
    end;


    procedure   TModMovingAverage.DoSlowTick; // override;
    var
      i      : Integer;
      OutVal : TSignal;
    begin
      FSize  := Round( FInputs[ i_size]);
      OutVal := 0.0;

      if   ( FSize >= 1       )
      and  ( FSize <= MAX_SIZE)
      then begin
        FHistory[ FPointer] := Normalize( FInputs[ i_in]);

        for i := 0 to FSize - 1
        do OutVal := OutVal + FHistory[ MathIntMod( FPointer - i, FSize)];

        FPointer := MathIntMod( FPointer + 1, MAX_SIZE);
        OutVal   := OutVal / FSize;
      end;

      FOutputs[ o_out] := OutVal;
    end;


{ ========
  TModDCBlock = class( TMod)
  private
    FPrevIn : TSignal;
  public
}

    procedure   TModDCBlock.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in   , 'in'   );
      AddInput ( i_alpha, 'alpha');
      AddOutput( o_out  , 'out'  );
    end;


    procedure   TModDCBlock.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_alpha
      ];
    end;


    procedure   TModDCBlock.DoSlowTick; // override;
    // out <- in - prev-in + lpha * prev-out
    var
      Inp : TSignal;
    begin
      Inp              := FInputs [ i_in ];
      FOutputs[ o_out] := Normalize( Inp - FPrevIn + FInputs[ i_alpha] * FOutputs[ o_out]);
      FPrevIn          := Inp;
    end;


{ ========
  TModFilter6dB = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  private
    FFreq  : TSignal;
    FAlpha : TSignal;
  private
}

    procedure   TModFilter6dB.CalcAlpha;
    begin
      FAlpha := FFreq / ( FFreq + ( System_Rate * TWO_PI_REC));
    end;

//  public

    procedure   TModFilter6dB.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in        , 'in'        );
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_freqmod   , 'freqmod'   );
      AddInput ( i_freqmodamt, 'freqmodamt');
      AddOutput( o_lp        , 'lp'        );
      AddOutput( o_hp        , 'hp'        );
    end;


    procedure   TModFilter6dB.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_freq,
        i_freqmodamt
      ];
    end;


    procedure   TModFilter6dB.DoSlowTick; // override;
    var
      Inp   : TSignal;
      Outp  : TSignal;
      Delta : TSignal;
    begin
      FFreq := LookupFrequency( Normalize( FInputs[ i_freq] + FInputs[ i_freqmod] * FInputs[ i_freqmodamt]));
      CalcAlpha;

      if ResetFlag
      then begin
        Inp   := 0;
        Outp  := 0;
      end
      else begin
        Inp   := FInputs [ i_in];
        Outp  := FOutputs[ o_lp];
      end;

      Delta           := Normalize( Inp  - Outp);
      Outp            := Outp + FAlpha * Delta;
      FOutputs[ o_lp] := Normalize( Outp );
      FOutputs[ o_hp] := Delta;
      ResetFlag       := False;
    end;


    procedure   TModFilter6dB.SampleRateChanged; // override;
    begin
      CalcAlpha;
    end;


{ ========
  TModFilter6dBBP = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  private
    FFreqHP  : TSignal;
    FFreqLP  : TSignal;
    FAlphaHP : TSignal;
    FAlphaLP : TSignal;
    FInp     : TSignal;
    FMid     : TSignal;
  private
}

    procedure   TModFilter6dBBP.CalcAlphaHP;
    begin
      FAlphaHP := FFreqHP / ( FFreqHP + ( System_Rate * TWO_PI_REC));
    end;


    procedure   TModFilter6dBBP.CalcAlphaLP;
    begin
      FAlphaLP := FFreqLP / ( FFreqLP + ( System_Rate * TWO_PI_REC));
    end;


//  public

    procedure   TModFilter6dBBP.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in    , 'in'    );
      AddInput ( i_freqhp, 'freqhp');
      AddInput ( i_freqlp, 'freqlp');
      AddOutput( o_out   , 'out'   );
    end;


    procedure   TModFilter6dBBP.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_freqhp,
        i_freqlp
      ];
    end;


    procedure   TModFilter6dBBP.DoSlowTick; // override;
    var
      NewInp : TSignal;
      Outp   : TSignal;
    begin
      FFreqHP := LookupFrequency( FInputs[ i_freqhp]);
      FFreqLP := LookupFrequency( FInputs[ i_freqlp]);
      CalcAlphaHP;
      CalcAlphaLP;
      NewInp           := FInputs [ i_in ];
      Outp             := FOutputs[ o_out];
      FMid             := ( 1 - FAlphaHP) * Normalize( FMid + NewInp - FInp);
      FInp             := NewInp;
      FOutputs[ o_out] := Normalize( FAlphaLP * ( FMid - Outp) + Outp);
    end;


    procedure   TModFilter6dBBP.SampleRateChanged; // override;
    begin
      CalcAlphaHP;
      CalcAlphaLP;
    end;


{ ========
  TModFilter6dBBPS = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  private
    FFreqHP  : TSignal;
    FFreqLP  : TSignal;
    FAlphaHP : TSignal;
    FAlphaLP : TSignal;
    FInpl    : TSignal;
    FMidl    : TSignal;
    FInpr    : TSignal;
    FMidr    : TSignal;
  private
}

    procedure   TModFilter6dBBPS.CalcAlphaHP;
    begin
      FAlphaHP := FFreqHP / ( FFreqHP + ( System_Rate * TWO_PI_REC));
    end;


    procedure   TModFilter6dBBPS.CalcAlphaLP;
    begin
      FAlphaLP := FFreqLP / ( FFreqLP + ( System_Rate * TWO_PI_REC));
    end;


//  public

    procedure   TModFilter6dBBPS.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_inl   , 'inl'   );
      AddInput ( i_inr   , 'inr'   );
      AddInput ( i_freqhp, 'freqhp');
      AddInput ( i_freqlp, 'freqlp');
      AddOutput( o_outl  , 'outl'  );
      AddOutput( o_outr  , 'outr'  );
    end;


    procedure   TModFilter6dBBPS.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_freqhp,
        i_freqlp
      ];
    end;


    procedure   TModFilter6dBBPS.DoSlowTick; // override;
    var
      NewInpl : TSignal;
      Outpl   : TSignal;
      NewInpr : TSignal;
      Outpr   : TSignal;
    begin
      FFreqHP := LookupFrequency( FInputs[ i_freqhp]);
      FFreqLP := LookupFrequency( FInputs[ i_freqlp]);
      CalcAlphaHP;
      CalcAlphaLP;
      NewInpl           := FInputs [ i_inl ];
      Outpl             := FOutputs[ o_outl];
      FMidl             := Normalize(( 1 - FAlphaHP) * ( FMidl + NewInpl - FInpl));
      FInpl             := NewInpl;
      FOutputs[ o_outl] := Normalize( FAlphaLP * ( FMidl - Outpl) + Outpl);
      NewInpr           := FInputs [ i_inr ];
      Outpr             := FOutputs[ o_outr];
      FMidr             := Normalize( ( 1 - FAlphaHP) * ( FMidr + NewInpr - FInpr));
      FInpr             := NewInpr;
      FOutputs[ o_outr] := Normalize( FAlphaLP * ( FMidr - Outpr) + Outpr);
    end;


    procedure   TModFilter6dBBPS.SampleRateChanged; // override;
    begin
      CalcAlphaHP;
      CalcAlphaLP;
    end;


{ ========
  TModFilter6dBBPSM = class( TMod)
  // Fc    = ( Fs / 2 * PI) * ( alpha / ( 1 - alpha)))
  // alpha = Fc / ( Fc + ( Fx / 2 * PI))
  private
    FFreqHP  : TSignal;
    FFreqLP  : TSignal;
    FAlphaHP : TSignal;
    FAlphaLP : TSignal;
    FInpl    : TSignal;
    FMidl    : TSignal;
    FInpr    : TSignal;
    FMidr    : TSignal;
  private
}

    procedure   TModFilter6dBBPSM.CalcAlphaHP;
    begin
      FAlphaHP := FFreqHP / ( FFreqHP + ( System_Rate * TWO_PI_REC));
    end;


    procedure   TModFilter6dBBPSM.CalcAlphaLP;
    begin
      FAlphaLP := FFreqLP / ( FFreqLP + ( System_Rate * TWO_PI_REC));
    end;


//  public

    procedure   TModFilter6dBBPSM.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_inl       , 'inl'       );
      AddInput ( i_inr       , 'inr'       );
      AddInput ( i_freqhp    , 'freqhp'    );
      AddInput ( i_freqlp    , 'freqlp'    );
      AddInput ( i_freqmod   , 'freqmod'   );
      AddInput ( i_freqmodamt, 'freqmodamt');
      AddOutput( o_outl      , 'outl'      );
      AddOutput( o_outr      , 'outr'      );
    end;


    procedure   TModFilter6dBBPSM.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_freqhp,
        i_freqlp
      ];
    end;


    procedure   TModFilter6dBBPSM.DoSlowTick; // override;
    var
      NewInpl : TSignal;
      Outpl   : TSignal;
      NewInpr : TSignal;
      Outpr   : TSignal;
    begin
      FFreqHP := LookupFrequency( Normalize( FInputs[ i_freqhp] + FInputs[ i_freqmod] * FInputs[ i_freqmodamt]));
      FFreqLP := LookupFrequency( Normalize( FInputs[ i_freqlp] + FInputs[ i_freqmod] * FInputs[ i_freqmodamt]));
      CalcAlphaHP;
      CalcAlphaLP;
      NewInpl           := FInputs [ i_inl ];
      Outpl             := FOutputs[ o_outl];
      FMidl             := Normalize(( 1 - FAlphaHP) * ( FMidl + NewInpl - FInpl));
      FInpl             := NewInpl;
      FOutputs[ o_outl] := Normalize( FAlphaLP * ( FMidl - Outpl) + Outpl);
      NewInpr           := FInputs [ i_inr ];
      Outpr             := FOutputs[ o_outr];
      FMidr             := Normalize( ( 1 - FAlphaHP) * ( FMidr + NewInpr - FInpr));
      FInpr             := NewInpr;
      FOutputs[ o_outr] := Normalize( FAlphaLP * ( FMidr - Outpr) + Outpr);
    end;


    procedure   TModFilter6dBBPSM.SampleRateChanged; // override;
    begin
      CalcAlphaHP;
      CalcAlphaLP;
    end;


{ ========
  TModSVF = class( TMod)
  private
    FFreq  : TSignal;
    FDamp  : TSignal;
    FDrive : TSignal;
  public
}

    (*
      from    : http://www.musicdsp.org/archive.php?classid=3#92
      also see: http://www.music.mcgill.ca/~ich/classes/FiltersChap2.pdf

      input  = input buffer;
      output = output buffer;
      fs     = sampling frequency;
      fc     = cutoff frequency normally something like:
               440.0*pow(2.0, (midi_note - 69.0)/12.0);                 :: We use signals rather, Frequency = Signal * SampleRate / 2
      res    = resonance 0 to 1;
      drive  = internal distortion 0 to 0.1
      freq   = 2.0*sin(PI*MIN(0.25, fc/(fs*2)));  // the fs*2 is because it's double sampled
      damp   = MIN(2.0*(1.0 - pow(res, 0.25)), MIN(2.0, 2.0/freq - freq*0.5));
      notch  = notch output
      low    = low pass output
      high   = high pass output
      band   = band pass output
      peak   = peaking output = low - high
      --
      double sampled svf loop:
      for (i=0; i<numSamples; i++)
      {
        in    = input[i];
        notch = in - damp*band;
        low   = low + freq*band;
        high  = notch - low;
        band  = freq*high + band - drive*band*band*band;
        out   = 0.5*(notch or low or high or band or peak);
        notch = in - damp*band;
        low   = low + freq*band;
        high  = notch - low;
        band  = freq*high + band - drive*band*band*band;
        out  += 0.5*(same out as above);
        output[i] = out;
      }
    *)

    procedure   TModSVF.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in        , 'in'        );
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_q         , 'q'         );
      AddInput ( i_distortion, 'distortion');
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_inlevel   , 'inlevel'   );
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_qm        , 'qm'        );
      AddInput ( i_qmamtp    , 'qmamtp'    );
      AddInput ( i_qmamtn    , 'qmamtn'    );
      AddOutput( o_lp        , 'lp'        );
      AddOutput( o_bp        , 'bp'        );
      AddOutput( o_hp        , 'hp'        );
      AddOutput( o_br        , 'br'        );
    end;


    procedure   TModSVF.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_q,
        i_distortion,
        i_fmlevel,
        i_inlevel
      ];
    end;


    procedure   TModSVF.DoSlowTick; // override;
    begin
      // NOTE: this assumes an SVF sampled at twice the system rate

      FFreq := LookupSvfFrequency( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fm] * FInputs[ i_fmlevel]);
      FDamp := 1 - Clip( FInputs[ i_q] + ( FInputs[ i_qmamtp] - FInputs[ i_qmamtn]) * FInputs[ i_qm] , 0.01, 1);

      // see: http://www.earlevel.com/main/2003/03/02/the-digital-state-variable-filter/
      // instability when FFreq aproaches 1

      if FFreq > 0.99
      then FFreq := 0.99;

      FDrive := Clip( FInputs[ i_distortion] * 0.1, 0.0, 0.1);
    end;


    procedure   TModSVF.DoTick; // override;
    var
      INP : TSignal;
      HP  : TSignal;
      BP  : TSignal;
      LP  : TSignal;
      BR  : TSignal;
    begin
      INP := FInputs [ i_in] * FInputs[ i_inlevel];
      LP  := FOutputs[ o_lp];
      BP  := FOutputs[ o_bp];
      BR := INP - FDamp * BP;
      LP := LP  + FFreq * BP;
      HP := BR  - LP;
      BP := FFreq * HP + BP - FDrive * BP * BP * BP; // with 0 <= Drive <= 0.1
      BR := INP - FDamp * BP;
      LP := LP  + FFreq * BP;
      HP := BR  - LP;
      BP := FFreq * HP + BP - FDrive * BP * BP * BP;
      FOutputs[ o_lp] := Normalize( LP);
      FOutputs[ o_bp] := Normalize( BP);
      FOutputs[ o_hp] := Normalize( HP);
      FOutputs[ o_br] := Normalize( BR);
    end;


{ ========
  TModMoogFilter = class( TMod)
  private
    FFilterClass : TMoogFilterClass;
    FFilter      : TMoogFilter;
  public
    property    FilterClass : TMoogFilterClass read FFilterClass write SetFilterClass;
  private
}

    procedure   TModMoogFilter.Recreate;
    var
      OldFilter : TMoogFilter;
      NewFilter : TMoogFilter;
    begin
      NewFilter := FFilterClass.Create( System_Rate);

      if Assigned( FFilter)
      then begin
        NewFilter.Resonance := FFilter.Resonance;
        NewFilter.CutOff    := FFilter.CutOff;
      end;

      Locked := True;

      try
        OldFilter := AtomicExchange( Pointer( FFilter), Pointer( NewFilter));
        FreeAndnil( OldFilter);
      finally
        Locked := False;
      end;
    end;


    procedure   TModMoogFilter.SetFilterClass( aValue: TMoogFilterClass);
    begin
      if aValue <> FFilterClass
      then begin
        FFilterClass := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModMoogFilter.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FilterClass := THuovilainenMoog; // Which will issue a Recreate
    end;


    procedure   TModMoogFilter.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in        , 'in'       );
      AddInput ( i_frequency , 'frequency');
      AddInput ( i_q         , 'q'        );
      AddInput ( i_fm        , 'fm'       );
      AddInput ( i_fmlevel   , 'fmlevel'  );
      AddInput ( i_inlevel   , 'inlevel'  );
      AddInput ( i_freq      , 'freq'     );
      AddInput ( i_qm        , 'qm'       );
      AddInput ( i_qmamtp    , 'qmamtp'   );
      AddInput ( i_qmamtn    , 'qmamtn'   );
      AddInput ( i_linfm     , 'linfm'    );
      AddInput ( i_linfmamt  , 'linfmamt' );
      AddOutput( o_out       , 'out'      );
    end;


    destructor  TModMoogFilter.Destroy; // override;
    var
      OldFilter : TMoogFilter;
    begin
      Locked := True;

      try
        OldFilter := AtomicExchange( Pointer( FFilter), nil);
        FreeAndnil( OldFilter);
      finally
        Locked := False;
      end;

      inherited;
    end;


    procedure   TModMoogFilter.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_frequency,
        i_q,
        i_fmlevel,
        i_inlevel
      ];
    end;


    procedure   TModMoogFilter.DoSlowTick; // override;
    begin
      if Assigned( FFilter)
      then FFilter.Resonance := Clip( FInputs[ i_q] + ( FInputs[ i_qmamtp] - FInputs[ i_qmamtn]) * FInputs[ i_qm] , 0.01, 1.0);
    end;


    procedure   TModMoogFilter.DoTick; // override;
    const
      FMAX = 12.5e3;
    var
      InVal  : TSignal;
      OutVal : TSignal;
      LFM    : TSignal;
    begin
      if Assigned( FFilter)
      then begin
        LFM            := Clip( FInputs[ i_linfm] * FInputs[ i_linfmamt] * FMAX, 0.0, FMAX);
        FFilter.CutOff := LookupFrequency( FInputs[ i_freq] + FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel]) + LFM;
        InVal          := FInputs [ i_in] * FInputs[ i_inlevel];
        FFilter.Process( InVal, OutVal);
        FOutputs[ o_out] := OutVal;
      end;
    end;


    procedure   TModMoogFilter.SampleRateChanged; // override;
    begin
      Recreate;
    end;


{ ========
  TModPinkFilter = class( TMod)
  private
    FB0 : TSignal;
    FB1 : TSignal;
    FB2 : TSignal;
    FB3 : TSignal;
    FB4 : TSignal;
    FB5 : TSignal;
    FB6 : TSignal;
    FG  : TSignal;
  public
}

    procedure   TModPinkFilter.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_in  , 'in'  );
      AddInput ( i_mode, 'mode');
      AddInput ( i_gain, 'gain');
      AddOutput( o_out , 'out' );
    end;


    procedure   TModPinkFilter.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_gain
      ];
      FInputs[ i_in] := 0;
      FB0            := 0.0;
      FB1            := 0.0;
      FB2            := 0.0;
      FB3            := 0.0;
      FB4            := 0.0;
      FB5            := 0.0;
      FB6            := 0.0;
      SampleRateChanged;
    end;


    procedure   TModPinkFilter.DoTick; // override;
    // http://www.musicdsp.org/files/pink.txt
    var
      White : TSignal;
    begin
      White := Normalize( FInputs[ i_in]);

      case Round( FInputs[ i_mode]) of
        0 : begin
          FB0 := Normalize( 0.99886 * FB0 + White * 0.0555179);
          FB1 := Normalize( 0.99332 * FB1 + White * 0.0750759);
          FB2 := Normalize( 0.96900 * FB2 + White * 0.1538520);
          FB3 := Normalize( 0.86650 * FB3 + White * 0.3104856);
          FB4 := Normalize( 0.55000 * FB4 + White * 0.5329522);
          FB5 := Normalize( -0.7616 * FB5 - White * 0.0168980);
          FOutputs[ o_out] := FInputs[ i_gain] * FG * ( FB0 + FB1 + FB2 + FB3 + FB4 + FB5 + FB6 + White * 0.5362);
          FB6 := White * 0.115926;
        end;
        else begin
          FB0 := Normalize( 0.99765 * FB0 + White * 0.0990460);
          FB1 := Normalize( 0.96300 * FB1 + White * 0.2965164);
          FB2 := Normalize( 0.57000 * FB2 + White * 1.0526913);
          FOutputs[ o_out] := FInputs[ i_gain] * FG * ( FB0 + FB1 + FB2 + White * 0.1848);
        end
      end;
    end;


    procedure   TModPinkFilter.SampleRateChanged; // override;
    begin
      FG  := dBToUnits( -13) * ( 44100 * System_Rate_Rec);
    end;


{ ========
  TModResonator = class( TMod)
  // http://www.katjaas.nl/complexintegrator/complexresonator.html
  private
    FOldRe  : TSignal;
    FOldIm  : TSignal;
    FRadius : TSignal;
    FSin    : TSignal;
    FCos    : TSignal;
  public
}

    procedure   TModResonator.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_freq     , 'freq');
      AddInput ( i_decay    , 'decay');
      AddInput ( i_in       , 'in');
      AddInput ( i_fm       , 'fm');
      AddInput ( i_fmlevel  , 'fmlevel');
      AddInput ( i_frequency, 'frequency');
      AddInput ( i_cents    , 'cents');
      AddInput ( i_pm       , 'pm');
      AddInput ( i_pmlevel  , 'pmlevel');
      AddInput ( i_dm       , 'dm');
      AddInput ( i_dmlevel  , 'dmlevel');
      AddInput ( i_click    , 'click');
      AddInput ( i_inlevel  , 'inlevel');
      AddOutput( o_out      , 'out');
    end;


    procedure   TModResonator.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_decay    ,
        i_fmlevel  ,
        i_frequency,
        i_cents    ,
        i_pmlevel  ,
        i_dmlevel  ,
        i_click
      ];
    end;


    procedure   TModResonator.DoSlowTick; // override;
    const
      PM_ATT = 1.0 / 32.0;
    var
      aFreq     : TSignal;
      anAngle   : TSignal;
      aDecayMod : TSignal;
      aDecay    : TSignal;
    begin
      aFreq     := LookupFrequency( FInputs[ i_freq] + FInputs[ i_frequency] + FInputs[ i_fm] * FInputs[ i_fmlevel] + FInputs[ i_cents]);
      anAngle   := TWO_PI * (( aFreq * System_Rate_Rec) + FInputs[ i_pm] * FInputs[ i_pmlevel] * PM_ATT);
      aDecayMod := FInputs[ i_dm] * FInputs[ i_dmlevel];
      aDecay    := FInputs[ i_decay] * ( 1 + aDecayMod);

      if aDecay <= 10e-6      // decay time is seconds - so that's 10 s minimal time
      then aDecay := 10e-6;

      FRadius := Normalize( exp( - System_Rate_Rec / aDecay));
      SinCos( anAngle, FSin, FCos);
    end;


    procedure   TModResonator.DoTick; // override;
    var
      inRe : TSignal;
      inIm : TSignal;
    begin
      inRe             := FRadius * FOldRe + FInputs[ i_in] * FInputs[ i_inlevel];   // Use Re. for input
      inIm             := FRadius * FOldIm;
      FOldRe           := Normalize( FCos * inRe - FSin * inIm);
      FOldIm           := Normalize( FSin * inRe + FCos * inIm);
      FOutputs[ o_out] := FOldIm + FInputs[ i_click] * ( FOldRe - FOldIm);           // and then Im. for output
    end;


{ ========
  TModFreqShifter = class( TMod)
  // https://github.com/swh/ladspa/blob/master/bode_shifter_1431.xml
  private
    FDelay : array[ 0 .. 255] of TSignal;
    FDPtr  : Integer;
    FPhase : TSignal;
    FDelta : TSignal;
  public
}

    procedure   TModFreqShifter.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in           , 'in'           );
      AddInput ( i_shift        , 'shift'        );
      AddInput ( i_shiftmod     , 'shiftmod'     );
      AddInput ( i_shiftdepth   , 'shiftdepth'   );
      AddInput ( i_shiftdepthamt, 'shiftdepthamt');
      AddOutput( o_up           , 'up'           );
      AddOutput( o_down         , 'down'         );
    end;


    procedure   TModFreqShifter.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_shift        ,
        i_shiftdepth   ,
        i_shiftdepthamt
      ];
    end;


    procedure   TModFreqShifter.DoSlowTick; // override;
    begin
      FDelta :=
        Clip(
          LookupFrequencyCubed( FInputs[ i_shift]) +
          LookupFrequencyLin  ( FInputs[ i_shiftmod] * FInputs[ i_shiftdepthamt] * FInputs[ i_shiftdepth]),
          0.0,
          10000.0
        ) * System_Rate_Rec;
    end;


    procedure   TModFreqShifter.DoTick; // override;
    const
      c_coefficients : array[ 0 .. 99] of TSignal = (
        +0.0008103736, +0.0008457886, +0.0009017196, +0.0009793364,
        +0.0010798341, +0.0012044365, +0.0013544008, +0.0015310235,
        +0.0017356466, +0.0019696659, +0.0022345404, +0.0025318040,
        +0.0028630784, +0.0032300896, +0.0036346867, +0.0040788644,
        +0.0045647903, +0.0050948365, +0.0056716186, +0.0062980419,
        +0.0069773575, +0.0077132300, +0.0085098208, +0.0093718901,
        +0.0103049226, +0.0113152847, +0.0124104218, +0.0135991079,
        +0.0148917649, +0.0163008758, +0.0178415242, +0.0195321089,
        +0.0213953037, +0.0234593652, +0.0257599469, +0.0283426636,
        +0.0312667947, +0.0346107648, +0.0384804823, +0.0430224431,
        +0.0484451086, +0.0550553725, +0.0633242001, +0.0740128560,
        +0.0884368322, +0.1090816773, +0.1412745301, +0.1988673273,
        +0.3326528346, +0.9997730178, -0.9997730178, -0.3326528346,
        -0.1988673273, -0.1412745301, -0.1090816773, -0.0884368322,
        -0.0740128560, -0.0633242001, -0.0550553725, -0.0484451086,
        -0.0430224431, -0.0384804823, -0.0346107648, -0.0312667947,
        -0.0283426636, -0.0257599469, -0.0234593652, -0.0213953037,
        -0.0195321089, -0.0178415242, -0.0163008758, -0.0148917649,
        -0.0135991079, -0.0124104218, -0.0113152847, -0.0103049226,
        -0.0093718901, -0.0085098208, -0.0077132300, -0.0069773575,
        -0.0062980419, -0.0056716186, -0.0050948365, -0.0045647903,
        -0.0040788644, -0.0036346867, -0.0032300896, -0.0028630784,
        -0.0025318040, -0.0022345404, -0.0019696659, -0.0017356466,
        -0.0015310235, -0.0013544008, -0.0012044365, -0.0010798341,
        -0.0009793364, -0.0009017196, -0.0008457886, -0.0008103736
      );
    var
      i       : Integer;
      d       : Word;
      hilb    : TSignal;
      rm1     : TSignal;
      rm2     : TSignal;
    begin
      FDelay[ FDPtr] := Normalize( FInputs[ i_in]);
      hilb := 0;
      d    := FDPtr;

      for i := Low( c_coefficients) to High( c_coefficients)
      do begin
        hilb := hilb + c_coefficients[ i] * FDelay[ d];
        d    := ( d + 254) and 255;
      end;

      FPhase  := MathFloatMod( FPhase + FDelta, 1.0);
      rm1     := hilb * 0.63661978              * LookupSine  ( FPhase);
      rm2     := FDelay[( FDPtr - 99) and 255] * LookupCosine( FPhase);
      FOutputs[ o_down] := ( rm2 - rm1) * 0.5;
      FOutputs[ o_up  ] := ( rm2 + rm1) * 0.5;
      FDPtr := ( FDPtr + 1) and 255;
    end;


{ ========
  TModTiltFilter = class( TMod)
  // source: http://www.musicdsp.org/showone.php?id=267
  strict private
  const
    AMP     = LOG_TWO / 6;
    GFACTOR = 5.0;
  private
    FLGain   : TSignal;
    FHGain   : TSignal;
    FLpOut   : TSignal;
    FA0      : TSignal;
    FB1      : TSignal;
  public
}

    procedure   TModTiltFilter.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in        , 'in'        );
      AddInput ( i_inlevel   , 'inlevel'   );
      AddInput ( i_tilt      , 'tilt'      );
      AddInput ( i_tiltmod   , 'tiltmod'   );
      AddInput ( i_tiltmodamt, 'tiltmodamt');
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_fmmod     , 'fmmod'     );
      AddInput ( i_fmmodamt  , 'fmmodamt'  );
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModTiltFilter.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_inlevel   ,
        i_tilt      ,
        i_tiltmodamt,
        i_freq      ,
        i_fmmodamt
      ];
    end;


    procedure   TModTiltFilter.DoSlowTick; // override;
    var
      g1    : TSignal;
      g2    : TSignal;
      Sr3   : TSignal;
      Gain  : TSignal;
      Omega : TSignal;
      n     : TSignal;
      Freq  : TSignal;
    begin
      Sr3  := System_Rate * 3.0;
      Gain := 6.0 * Clip( Normalize( Finputs[ i_tilt] + FInputs[ i_tiltmod] * FInputs[ i_tiltmodamt]), -1.0, 1.0);

      if Gain > 0
      then begin
        g1 := - GFACTOR * Gain;
        g2 :=             gain;
      end
      else begin
        g1 :=           - gain;
        g2 :=   GFACTOR * Gain;
      end;

      FLGain := Exp( g1 * AMP);
      FHGain := Exp( g2 * AMP);
      Freq   := LookupFrequency( Normalize( FInputs[ i_freq] + FInputs[ i_fmmod] * FInputs[ i_fmmodamt]));
      Omega  := TWO_PI * Freq;
      n      := 1 / ( Sr3 + Omega);
      FA0    := 2 * Omega * n;
      FB1    := ( Sr3 - Omega) * n;
    end;


    procedure   TModTiltFilter.DoTick; // override;
    var
      Inp : TSignal;
    begin
      Inp              := Normalize( FInputs[ i_in] * FInputs[ i_inlevel]);
      FLpOut           := Normalize( FA0 * Inp + FB1 * FLpOut);
      FOutputs[ o_out] := Inp + FLGain * FLpOut + FHGain * Normalize( Inp - FLpOut);
    end;


{ ========
  TModTiltFilterStereo = class( TMod)
  // source: http://www.musicdsp.org/showone.php?id=267
  strict private
  const
    AMP     = LOG_TWO / 6;
    GFACTOR = 5.0;
  private
    FLGain   : TSignal;
    FHGain   : TSignal;
    FLpOut   : TSignal;
    FLpOut2  : TSignal;
    FA0      : TSignal;
    FB1      : TSignal;
  public
}

    procedure   TModTiltFilterStereo.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_in        , 'in'        );
      AddInput ( i_in2       , 'in2'       );
      AddInput ( i_inlevel   , 'inlevel'   );
      AddInput ( i_tilt      , 'tilt'      );
      AddInput ( i_tiltmod   , 'tiltmod'   );
      AddInput ( i_tiltmodamt, 'tiltmodamt');
      AddInput ( i_freq      , 'freq'      );
      AddInput ( i_fmmod     , 'fmmod'     );
      AddInput ( i_fmmodamt  , 'fmmodamt'  );
      AddOutput( o_out       , 'out'       );
      AddOutput( o_out2      , 'out2'      );
    end;


    procedure   TModTiltFilterStereo.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_inlevel   ,
        i_tilt      ,
        i_tiltmodamt,
        i_freq      ,
        i_fmmodamt
      ];
    end;


    procedure   TModTiltFilterStereo.DoSlowTick; // override;
    var
      g1    : TSignal;
      g2    : TSignal;
      Sr3   : TSignal;
      Gain  : TSignal;
      Omega : TSignal;
      n     : TSignal;
      Freq  : TSignal;
    begin
      Sr3  := System_Rate * 3.0;
      Gain := 6.0 * Clip( Normalize( Finputs[ i_tilt] + FInputs[ i_tiltmod] * FInputs[ i_tiltmodamt]), -1.0, 1.0);

      if Gain > 0
      then begin
        g1 := - GFACTOR * Gain;
        g2 :=             gain;
      end
      else begin
        g1 :=           - gain;
        g2 :=   GFACTOR * Gain;
      end;

      FLGain := Exp( g1 * AMP);
      FHGain := Exp( g2 * AMP);
      Freq   := LookupFrequency( Normalize( FInputs[ i_freq] + FInputs[ i_fmmod] * FInputs[ i_fmmodamt]));
      Omega  := TWO_PI * Freq;
      n      := 1 / ( Sr3 + Omega);
      FA0    := 2 * Omega * n;
      FB1    := ( Sr3 - Omega) * n;
    end;


    procedure   TModTiltFilterStereo.DoTick; // override;
    var
      Inp : TSignal;
    begin
      Inp               := Normalize( FInputs[ i_in] * FInputs[ i_inlevel]);
      FLpOut            := Normalize( FA0 * Inp + FB1 * FLpOut);
      FOutputs[ o_out]  := Inp + FLGain * FLpOut + FHGain * Normalize( Inp - FLpOut);

      Inp               := Normalize( FInputs[ i_in2] * FInputs[ i_inlevel]);
      FLpOut2           := Normalize( FA0 * Inp + FB1 * FLpOut2);
      FOutputs[ o_out2] := Inp + FLGain * FLpOut2 + FHGain * Normalize( Inp - FLpOut2);
    end;


{ ========
  TModEnvAR = class( TMod)
  private
    FState        : TARState;
    FValue        : TSignal;
    FCoeffAttack  : TSignal;
    FCoeffRelease : TSignal;
    FEnvValue     : TSignal;
    FTrigCount    : Integer;
  public
}

    procedure   TModEnvAR.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_trig   , 'trig'   );
      AddInput ( i_attack , 'attack' );
      AddInput ( i_release, 'release');
      AddInput ( i_gain   , 'gain'   );
      AddInput ( i_sigin  , 'sigin'  );
      AddInput ( i_mute   , 'mute'   );
      AddInput ( i_hold   , 'hold'   );
      AddInput ( i_amod   , 'amod'   );
      AddInput ( i_rmod   , 'rmod'   );
      AddOutput( o_out    , 'out'    );
      AddOutput( o_inv    , 'inv'    );
      AddOutput( o_sigout , 'sigout' );
    end;


    procedure   TModEnvAR.SetDefaults; // override;
    begin
      FInputs[ i_gain] := 1; // Set default gain to 1, so it'll work unconnected
      FDezipperMap := [
        i_attack,
        i_release
      ];
    end;


    procedure   TModEnvAR.DoTick; // override;
    var
      R     : TSignal;
      Coeff : TSignal;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FState    := arIdle;
        FValue    := 0;
      end;

      case FState of

        arIdle :

          if SignalToLogic( FInputs[ i_trig])
          then begin

            FState     := arA;
            FTrigCount := LedFlashTime( True);
          end;

        arA :

          begin
            R := FValue;

            if R < 0.001
            then R := 0.001;

            Coeff := 1 / Max( 1, TimeToSampleCount( LookupEnvTime( erFast, FInputs[ i_amod] + FInputs[ i_attack]))); // Linear attack
            R     := R + Coeff;

            if R > 0.999
            then begin
              if SignalToLogic( FInputs[ i_trig])
              then begin
                if SignalToLogic( FInputs[ i_hold])
                then FState := arH
                else FState := arR;
              end
              else FState := arR;
            end
            else FValue := R;
          end;

        arH :

          begin
            if not SignalToLogic( FInputs[ i_trig])
            then FState := arR;
          end;

        arR :

          begin
            Coeff := 1.0 + LN0001 / Max( 1, TimeToSampleCount( LookupEnvTime( erFast, FInputs[ i_rmod] + FInputs[ i_release]))); // Exp release
            R     := Normalize( FValue * Coeff);

            if R < 0.001
            then begin
              FValue := 0.0;
              FState := arWait;
            end
            else FValue := R;
          end;

        arWait :

          if not SignalToLogic( FInputs[ i_trig])
          then FState := arIdle;

      end;

      case Round( FInputs[ i_mute]) of
        1  : FEnvValue := Normalize(  EnvelopeSmoothing * FEnvValue                              / ( 1 + EnvelopeSmoothing));
        2  : FEnvValue := Normalize(( EnvelopeSmoothing * FEnvValue + 1)                         / ( 1 + EnvelopeSmoothing));
        else FEnvValue := Normalize(( EnvelopeSmoothing * FEnvValue + FValue * FInputs[ i_gain]) / ( 1 + EnvelopeSmoothing));
      end;

      DecToZero( FTrigCount);
      FOutputs[ o_out   ] :=       FEnvValue;
      FOutputs[ o_inv   ] := 1.0 - FEnvValue;
      FOutputs[ o_sigout] := FEnvValue * FInputs[ i_sigin];
    end;


    procedure   TModEnvAR.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig', LogicToSignal( FTrigCount > 0)));
    end;

{ ========
  TModEnvARRetrig = class( TMod)
  private
    FState     : TARState;
    FValue     : TSignal;
    FArmed     : Boolean;
    FHoldMode  : Boolean;
    FEnvValue  : TSignal;
    FTrigCount : Integer;
  public
}

    procedure   TModEnvARRetrig.CreateIO; // override;
    begin
      FIsFast := True;
      AddInput ( i_trig   , 'trig'   );
      AddInput ( i_attack , 'attack' );
      AddInput ( i_release, 'release');
      AddInput ( i_gain   , 'gain'   );
      AddInput ( i_sigin  , 'sigin'  );
      AddInput ( i_mute   , 'mute'   );
      AddInput ( i_hold   , 'hold'   );
      AddInput ( i_amod   , 'amod'   );
      AddInput ( i_rmod   , 'rmod'   );
      AddOutput( o_out    , 'out'    );
      AddOutput( o_inv    , 'inv'    );
      AddOutput( o_sigout , 'sigout' );
    end;


    procedure   TModEnvARRetrig.SetDefaults; // override;
    begin
      FInputs[ i_gain] := 1; // Set default gain to 1, so it'll work unconnected
      FDezipperMap := [
        i_attack,
        i_release
      ];
    end;


    procedure   TModEnvARRetrig.DoTick; // override;
    var
      R     : TSignal;
      Coeff : TSignal;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FState    := arIdle;
        FValue    := 0;
      end;

      if FArmed and SignalToLogic( FInputs[ i_trig])
      then FState := arIdle;

      FHoldMode := SignalToLogic( FInputs[ i_hold]);

      case FState of

        arIdle :

          if SignalToLogic( FInputs[ i_trig])
          then begin
            FState        := arA;
            FArmed        := False;
            FTrigCount    := LedFlashTime( True);
          end;

        arA :

          begin
            if not FArmed
            then FArmed := not SignalToLogic( FInputs[ i_trig]);

            R := FValue;

            if R < 0.001
            then R := 0.001;

            Coeff := 1 / Max( 1, TimeToSampleCount( LookupEnvTime( erFast, FInputs[ i_amod] + FInputs[ i_attack]))); // Linear attack
            R     := R + Coeff;

            if R > 0.999
            then begin
              if SignalToLogic( FInputs[ i_trig])
              then begin
                if FHoldMode
                then FState := arH
                else FState := arR;
              end
              else FState := arR;
            end
            else FValue := R;
          end;

        arH :

          begin
            if not SignalToLogic( FInputs[ i_trig])
            then FState := arR;
          end;

        arR :

          begin
            if not FArmed
            then FArmed := not SignalToLogic( FInputs[ i_trig]);

            Coeff := 1 + LN0001 / Max( 1, TimeToSampleCount( LookupEnvTime( erFast, FInputs[ i_rmod] + FInputs[ i_release]))); // Exp release
            R     := Normalize( FValue * Coeff);

            if R < 0.001
            then begin
              FValue := 0.0;
              FState := arWait;
            end
            else FValue := R;
          end;

        arWait :

          if not SignalToLogic( FInputs[ i_trig])
          then FState := arIdle;

      end;

      case Round( FInputs[ i_mute]) of
        1  : FEnvValue := Normalize(  EnvelopeSmoothing * FEnvValue                              / ( 1 + EnvelopeSmoothing));
        2  : FEnvValue := Normalize(( EnvelopeSmoothing * FEnvValue + 1)                         / ( 1 + EnvelopeSmoothing));
        else FEnvValue := Normalize(( EnvelopeSmoothing * FEnvValue + FValue * FInputs[ i_gain]) / ( 1 + EnvelopeSmoothing));
      end;

      DecToZero( FTrigCount);
      FOutputs[ o_out   ] :=       FEnvValue;
      FOutputs[ o_inv   ] := 1.0 - FEnvValue;
      FOutputs[ o_sigout] := FEnvValue * FInputs[ i_sigin];
    end;


    procedure   TModEnvARRetrig.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      if FHoldMode
      then aCallback( Self, MakeInfo( aPrefix, Name, 'trig', LogicToSignal( SignalToLogic( FInputs[ i_trig]))))
      else aCallback( Self, MakeInfo( aPrefix, Name, 'trig', LogicToSignal( FTrigCount > 0 )));
    end;


{ ========
  TModEnvAHD = class( TMod)
  private
    FData      : TSignalArray;
    FState     : TAHDState;
    FTime      : Integer;
    FDuration  : Integer;
    FArmed     : Boolean;
    FPrevLevel : TSignal;
    FValue     : TSignal;
    FEnvValue  : TSignal;
    FTrigFlag  : Boolean;
    FTrigCount : Integer;
    FEocCount  : Integer;
    FEoCLed    : Integer;
  public
}

    constructor TModEnvAHD.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      SetLength( FData, 67);       // Should match the viewer width in the designer
    end;


    procedure   TModEnvAHD.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_trig         , 'trig'         );
      AddInput ( i_mode         , 'mode'         );
      AddInput ( i_ashape       , 'ashape'       );
      AddInput ( i_dshape       , 'dshape'       );
      AddInput ( i_attack       , 'attack'       );
      AddInput ( i_hold         , 'hold'         );
      AddInput ( i_decay        , 'decay'        );
      AddInput ( i_gain         , 'gain'         );
      AddInput ( i_sigin        , 'sigin'        );
      AddInput ( i_mute         , 'mute'         );
      AddInput ( i_arange       , 'arange'       );
      AddInput ( i_hrange       , 'hrange'       );
      AddInput ( i_drange       , 'drange'       );
      AddInput ( i_amod         , 'amod'         );
      AddInput ( i_hmod         , 'hmod'         );
      AddInput ( i_dmod         , 'dmod'         );
      AddInput ( i_amodneg      , 'amodneg'      );
      AddInput ( i_hmodneg      , 'hmodneg'      );
      AddInput ( i_dmodneg      , 'dmodneg'      );
      AddInput ( i_amodlevel    , 'amodlevel'    );
      AddInput ( i_hmodlevel    , 'hmodlevel'    );
      AddInput ( i_dmodlevel    , 'dmodlevel'    );
      AddInput ( i_outputtype   , 'outputtype'   );
      AddInput ( i_holdmode     , 'holdmode'     );
      AddInput ( i_swell        , 'swell'        );
      AddInput ( i_swellmod     , 'swellmod'     );
      AddInput ( i_swellmodneg  , 'swellmodneg'  );
      AddInput ( i_swellmodlevel, 'swellmodlevel');
      AddInput ( i_swellshape   , 'swellshape'   );
      AddOutput( o_out          , 'out'          );
      AddOutput( o_sigout       , 'sigout'       );
      AddOutput( o_eoc          , 'eoc'          );
      AddOutput( o_inv          , 'inv'          );
    end;


    procedure   TModEnvAHD.SetDefaults; // override;
    begin
      FInputs[ i_gain] := 1; // Set default gain to 1, so it'll work unconnected
      FDezipperMap := [
        i_attack       ,
        i_hold         ,
        i_decay        ,
        i_amodlevel    ,
        i_hmodlevel    ,
        i_dmodlevel    ,
        i_swellmodlevel
      ];
    end;


    procedure   TModEnvAHD.DoSlowTick; // override;
    var
      Trig     : Boolean;
      AMod     : TSignal;
      HMod     : TSignal;
      DMod     : TSignal;
      HoldMode : Boolean;
      Swell    : TSignal;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FState    := ahdIdle;
        FValue    := 0;
        FEocCount := TrigPulseTime( IsSpedUp);
        FEoCLed   := LedFlashTime ( IsSpedUp);
      end;

      Trig      := FTrigFlag;
      FTrigFlag := False;
      Swell     := Clip( FInputs[ i_swell] + FInputs[ i_swellmodlevel] * ( FInputs[ i_swellmod] - FInputs[ i_swellmodneg]), 0.0, 1.0);

      if FArmed and Trig and ( Round( FInputs[ i_mode]) = 1)
      then begin
        FState    := ahdIdle;
        FEocCount := TrigPulseTime( IsSpedUp);
        FEoCLed   := LedFlashTime ( IsSpedUp);
      end;


      case FState of

        ahdIdle :

          if Trig
          then begin
            FArmed     := False;
            FPrevLevel := FValue;
            FTime      := 0;
            AMod       := ( FInputs[ i_amod] - FInputs[ i_amodneg]) * FInputs[ i_amodlevel];
            FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_arange]), FInputs[ i_attack] + AMod))));
            FState     := ahdA;
            FTrigCount := LedFlashTime( IsSpedUp);
          end;

        ahdA :

          begin
            if not FArmed
            then FArmed := not Trig;

            Inc( FTime);

            if FTime > FDuration
            then begin
              FTime      := 0;
              HMod       := ( FInputs[ i_hmod] - FInputs[ i_hmodneg]) * FInputs[ i_hmodlevel];
              FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_hrange]), FInputs[ i_hold] + HMod))));
              FPrevLevel := FValue;
              FState     := ahdH;
            end
            else begin
              case SignalToEnvShape( FInputs[ i_ashape]) of
                esLog : FValue := LogAttack( FTime, FDuration, FPrevLevel, 1 - Swell);
                esLin : FValue := LinAttack( FTime, FDuration, FPrevLevel, 1 - Swell);
                esExp : FValue := ExpAttack( FTime, FDuration, FPrevLevel, 1 - Swell);
                else    FValue := SAttack  ( FTime, FDuration, FPrevLevel, 1 - Swell);
              end;
            end;
          end;

        ahdH :

          begin
            if not FArmed
            then FArmed := not Trig;

            Inc( FTime);
            HoldMode := SignalToLogic( FInputs[ i_holdmode]);

            case SignalToEnvShape( FInputs[ i_swellshape]) of
              esLog : FValue := LogAttack( FTime, FDuration, FPrevLevel, 1);
              esLin : FValue := LinAttack( FTime, FDuration, FPrevLevel, 1);
              esExp : FValue := ExpAttack( FTime, FDuration, FPrevLevel, 1);
              else    FValue := SAttack  ( FTime, FDuration, FPrevLevel, 1);
            end;

            if ( FTime > FDuration) and not ( HoldMode and not FArmed)
            then begin
              FPrevLevel := FValue;
              FTime      := 0;
              DMod       := ( FInputs[ i_dmod] - FInputs[ i_dmodneg]) * FInputs[ i_dmodlevel];
              FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_drange]), FInputs[ i_decay] + DMod))));
              FState     := ahdD;
            end;
          end;

        ahdD :

          begin
            if not FArmed
            then FArmed := not Trig;

            Inc( FTime);

            if FTime > FDuration
            then FState := ahdWait
            else begin
              case SignalToEnvShape( FInputs[ i_dshape]) of
                esLog : FValue := LogDecay( FTime, FDuration, FPrevLevel, 0);
                esLin : FValue := LinDecay( FTime, FDuration, FPrevLevel, 0);
                esExp : FValue := ExpDecay( FTime, FDuration, FPrevLevel, 0);
                else    FValue := SDecay  ( FTime, FDuration, FPrevLevel, 0);
              end;
            end;
          end;

        ahdWait :

          if not Trig or FArmed
          then begin
            FState    := ahdIdle;
            FEocCount := TrigPulseTime( IsSpedUp);
            FEoCLed   := LedFlashTime ( IsSpedUp);
          end;

      end;

      case Round( FInputs[ i_mute]) of
        1  : FEnvValue := Normalize(  EnvelopeSmoothingSlow * FEnvValue                              / ( 1 + EnvelopeSmoothingSlow));
        2  : FEnvValue := Normalize(( EnvelopeSmoothingSlow * FEnvValue + 1)                         / ( 1 + EnvelopeSmoothingSlow));
        else FEnvValue := Normalize(( EnvelopeSmoothingSlow * FEnvValue + FValue * FInputs[ i_gain]) / ( 1 + EnvelopeSmoothingSlow));
      end;

      FOutputs[ o_out] := SignalForEnvOutputType( FEnvValue, Round( FInputs[ i_outputtype]));
      FOutputs[ o_inv] := 1.0 - FOutputs[ o_out];
      FOutputs[ o_eoc] := LogicToSignal( FEocCount > 0);
      DecToZero( FTrigCount);
      DecToZero( FEocCount );
      DecToZero( FEocLed   );
    end;


    procedure   TModEnvAHD.DoTick; // override;
    begin
      if SignalToLogic( FInputs[ i_trig])
      then FTrigFlag := True;

      FOutputs[ o_sigout] := FInputs[ i_sigin] * FOutputs[ o_out]; // sigout := sigin * out
    end;


    procedure   TModEnvAHD.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'  , LogicToSignal( not ( FState in [ ahdIdle, ahdWait]))));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigin', LogicToSignal(( FTrigCount > 0) or ( SignalToLogic( FInputs[ i_holdmode]) and SignalToLogic( FInputs[ i_trig])))));
      aCallback( Self, MakeInfo( aPrefix, Name, 'eoc'   , LogicToSignal( FEoCLed    > 0)                      ));
    end;


    procedure   TModEnvAHD.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    var
      ATime    : TSignal;
      HTime    : TSignal;
      DTime    : TSignal;
      TTime    : TSignal;
      TTimeRec : TSignal;
      AShape   : TEnvShape;
      DShape   : TEnvShape;
      SShape   : TEnvShape;
      Swell    : TSignal;
      i        : Integer;
      p        : Integer;
      R        : TSignal;
    begin
      ATime  := LookupEnvTime( SignalToEnvRange( FInputs[ i_arange]), FInputs[ i_attack]);
      HTime  := LookupEnvTime( SignalToEnvRange( FInputs[ i_hrange]), FInputs[ i_hold  ]);
      DTime  := LookupEnvTime( SignalToEnvRange( FInputs[ i_drange]), FInputs[ i_decay ]);
      TTime  := ATime + HTime + DTime;

      if TTIme = 0
      then Exit;

      TTimeRec := 67 / TTime;
      ATime    := ATime * TTimeRec;
      HTime    := HTime * TTimeRec;
      DTime    := DTime * TTimeRec;
      AShape   := SignalToEnvShape( FInputs[ i_ashape    ]);
      DShape   := SignalToEnvShape( FInputs[ i_dshape    ]);
      SShape   := SignalToEnvShape( FInputs[ i_swellshape]);
      Swell    := FInputs[ i_swell];

      i := 0;

      while i < ATime
      do begin
        case AShape of
          esLog : R := LogAttack( i, ATime, 0, 1 - Swell);
          esLin : R := LinAttack( i, ATime, 0, 1 - Swell);
          esExp : R := ExpAttack( i, ATime, 0, 1 - Swell);
          else    R := SAttack  ( i, ATime, 0, 1 - Swell);
        end;

        FData[ i] :=  SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
      end;

      p := 0;

      while i < ATime + HTime
      do begin
        case SShape of
          esLog : R := LogAttack( p, HTime, 1 - Swell, 1);
          esLin : R := LinAttack( p, HTime, 1 - Swell, 1);
          esExp : R := ExpAttack( p, HTime, 1 - Swell, 1);
          else    R := SAttack  ( p, HTime, 1 - Swell, 1);
        end;

        FData[ i] := SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
        Inc( p);
      end;

      p := 0;

      while i < 67
      do begin
        case DShape of
          esLog : R := LogDecay( p, DTime, 1, 0);
          esLin : R := LinDecay( p, DTime, 1, 0);
          esExp : R := ExpDecay( p, DTime, 1, 0);
          else    R := SDecay  ( p, DTime, 1, 0);
        end;

        FData[ i] := SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
        Inc( p);
      end;

      aCallback( Self, MakeInfo( aPrefix, Name, 'viewer', FData));
    end;


{ ========
  TModEnvADSR = class( TMod)
  private
    FData      : TSignalArray;
    FState     : TADSRState;
    FTime      : Integer;
    FDuration  : Integer;
    FArmed     : Boolean;
    FPrevLevel : TSignal;
    FValue     : TSignal;
    FEnvValue  : TSignal;
    FTrigFlag  : Boolean;
    FTrigCount : Integer;
    FEocCount  : Integer;
    FEoCLed    : Integer;
  public
}

    constructor TModEnvADSR.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      SetLength( FData, 67);       // Should match the viewer width in the designer
    end;


    procedure   TModEnvADSR.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_trig         , 'trig'         );
      AddInput ( i_mode         , 'mode'         );
      AddInput ( i_ashape       , 'ashape'       );
      AddInput ( i_dshape       , 'dshape'       );
      AddInput ( i_rshape       , 'rshape'       );
      AddInput ( i_attack       , 'attack'       );
      AddInput ( i_decay        , 'decay'        );
      AddInput ( i_sustain      , 'sustain'      );
      AddInput ( i_release      , 'release'      );
      AddInput ( i_gain         , 'gain'         );
      AddInput ( i_sigin        , 'sigin'        );
      AddInput ( i_mute         , 'mute'         );
      AddInput ( i_arange       , 'arange'       );
      AddInput ( i_drange       , 'drange'       );
      AddInput ( i_rrange       , 'rrange'       );
      AddInput ( i_amod         , 'amod'         );
      AddInput ( i_dmod         , 'dmod'         );
      AddInput ( i_smod         , 'smod'         );
      AddInput ( i_rmod         , 'rmod'         );
      AddInput ( i_amodneg      , 'amodneg'      );
      AddInput ( i_dmodneg      , 'dmodneg'      );
      AddInput ( i_smodneg      , 'smodneg'      );
      AddInput ( i_rmodneg      , 'rmodneg'      );
      AddInput ( i_amodlevel    , 'amodlevel'    );
      AddInput ( i_dmodlevel    , 'dmodlevel'    );
      AddInput ( i_smodlevel    , 'smodlevel'    );
      AddInput ( i_rmodlevel    , 'rmodlevel'    );
      AddInput ( i_outputtype   , 'outputtype'   );
      AddInput ( i_swell        , 'swell'        );
      AddInput ( i_swellmod     , 'swellmod'     );
      AddInput ( i_swellmodneg  , 'swellmodneg'  );
      AddInput ( i_swellmodlevel, 'swellmodlevel');
      AddInput ( i_swellshape   , 'swellshape'   );
      AddOutput( o_out          , 'out');
      AddOutput( o_sigout       , 'sigout');
      AddOutput( o_eoc          , 'eoc');
      AddOutput( o_inv          , 'inv');
    end;


    procedure   TModEnvADSR.SetDefaults; // override;
    begin
      FInputs[ i_gain] := 1; // Set default gain to 1, so it'll work unconnected
      FDezipperMap := [
        i_attack,
        i_decay,
        i_sustain,
        i_release,
        i_amodlevel,
        i_dmodlevel,
        i_smodlevel,
        i_rmodlevel
      ];
    end;


    procedure   TModEnvADSR.DoSlowTick; // override;
    var
      Trig   : Boolean;
      AMod   : TSignal;
      DMod   : TSignal;
      SMod   : TSignal;
      RMod   : TSignal;
      SLevel : TSignal;
      Swell  : TSignal;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FState    := adsrIdle;
        FValue    := 0;
        FEocCount := TrigPulseTime( IsSpedUp);
        FEoCLed   := LedFlashTime ( IsSpedUp);
      end;

      Trig      := FTrigFlag;
      FTrigFlag := False;
      AMod      := ( FInputs[ i_amod] - FInputs[ i_amodneg]) * FInputs[ i_amodlevel];
      DMod      := ( FInputs[ i_dmod] - FInputs[ i_dmodneg]) * FInputs[ i_dmodlevel];
      SMod      := ( FInputs[ i_smod] - FInputs[ i_smodneg]) * FInputs[ i_smodlevel];
      RMod      := ( FInputs[ i_rmod] - FInputs[ i_rmodneg]) * FInputs[ i_rmodlevel];
      Swell     := Clip( FInputs[ i_swell] + FInputs[ i_swellmodlevel] * ( FInputs[ i_swellmod] - FInputs[ i_swellmodneg]), 0.0, 1.0);

      if FArmed and Trig and ( Round( FInputs[ i_mode]) = 1)
      then begin
        FState    := adsrIdle;
        FEocCount := TrigPulseTime( IsSpedUp);
        FEoCLed   := LedFlashTime ( IsSpedUp);
      end;

      case FState of

        adsrIdle :

          if Trig
          then begin
            FArmed     := False;
            FPrevLevel := FValue;
            FTime      := 0;
            FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_arange]), FInputs[ i_attack] + AMod))));
            FState     := adsrA;
            FTrigCount := LedFlashTime( IsSpedUp);
          end;

        adsrA :

          begin
            if not FArmed
            then FArmed := not Trig;

            Inc( FTime);

            if FTime > FDuration
            then begin
              FPrevLevel := FValue;
              FTime      := 0;
              FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_drange]), FInputs[ i_decay] + DMod))));
              FState     := adsrD;
            end
            else begin
              case SignalToEnvShape( FInputs[ i_ashape ]) of
                esLog : FValue := LogAttack( FTime, FDuration, FPrevLevel, 1.0 - Swell / 3.0);
                esLin : FValue := LinAttack( FTime, FDuration, FPrevLevel, 1.0 - Swell / 3.0);
                esExp : FValue := ExpAttack( FTime, FDuration, FPrevLevel, 1.0 - Swell / 3.0);
                else    FValue := SAttack  ( FTime, FDuration, FPrevLevel, 1.0 - Swell / 3.0);
              end;
            end;
          end;

        adsrD :

          begin
            if not FArmed
            then FArmed := not Trig;

            Inc( FTime);

            if FTime > FDuration
            then begin
              FPrevLevel := FValue;
//            FDuration  := Max( ShortestPossibleEnvelopeTime, 3 * Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_drange]), FInputs[ i_decay] + DMod))));
              FTime      := 0;
              FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_rrange]), FInputs[ i_release] + RMod))));
              FState     := adsrS;
            end
            else begin
              SLevel := FInputs[ i_sustain] + SMod;

              case SignalToEnvShape( FInputs[ i_dshape ]) of
                esLog : FValue := LogDecay( FTime, FDuration, FPrevLevel, SLevel);
                esLin : FValue := LinDecay( FTime, FDuration, FPrevLevel, SLevel);
                esExp : FValue := ExpDecay( FTime, FDuration, FPrevLevel, SLevel);
                else    FValue := SDecay  ( FTime, FDuration, FPrevLevel, SLevel);
              end;
            end;
          end;

        adsrS :

          begin
            if not FArmed
            then FArmed := not Trig;

            if FArmed
            then begin
              FPrevLevel := FValue;
              FTime      := 0;
              FDuration  := Max( ShortestPossibleEnvelopeTime, Round( SlowTimeToSampleCount( LookupEnvTime( SignalToEnvRange( FInputs[ i_rrange]), FInputs[ i_release] + RMod))));
              FState     := adsrR;
            end
            else begin
              Inc( FTime);
              SLevel := FInputs[ i_sustain] + SMod;

              case SignalToEnvShape( FInputs[ i_swellshape ]) of
                esLog : FValue := LogAttack( FTime, FDuration, FPrevLevel, SLevel + Swell * ( 1 - SLevel));
                esLin : FValue := LinAttack( FTime, FDuration, FPrevLevel, SLevel + Swell * ( 1 - SLevel));
                esExp : FValue := ExpAttack( FTime, FDuration, FPrevLevel, SLevel + Swell * ( 1 - SLevel));
                else    FValue := SAttack  ( FTime, FDuration, FPrevLevel, SLevel + Swell * ( 1 - SLevel));
              end;
            end;
          end;

        adsrR :

          begin
            if not FArmed
            then FArmed := not Trig;

            Inc( FTime);

            if FTime > FDuration
            then FState := adsrWait
            else begin
              case SignalToEnvShape( FInputs[ i_rshape ]) of
                esLog : FValue := LogDecay( FTime, FDuration, FPrevLevel, 0);
                esLin : FValue := LinDecay( FTime, FDuration, FPrevLevel, 0);
                esExp : FValue := ExpDecay( FTime, FDuration, FPrevLevel, 0);
                else    FValue := SDecay  ( FTime, FDuration, FPrevLevel, 0);
              end;
            end;
          end;

        adsrWait :

          if not Trig or FArmed
          then begin
            FState    := adsrIdle;
            FEocCount := TrigPulseTime( IsSpedUp);
            FEoCLed   := LedFlashTime ( IsSpedUp);
          end;

      end;

      case Round( FInputs[ i_mute]) of
        1  : FEnvValue := Normalize(  EnvelopeSmoothingSlow * FEnvValue                              / ( 1 + EnvelopeSmoothingSlow));
        2  : FEnvValue := Normalize(( EnvelopeSmoothingSlow * FEnvValue + 1 * 1)                     / ( 1 + EnvelopeSmoothingSlow));
        else FEnvValue := Normalize(( EnvelopeSmoothingSlow * FEnvValue + FValue * FInputs[ i_gain]) / ( 1 + EnvelopeSmoothingSlow));
      end;

      FOutputs[ o_out] := SignalForEnvOutputType( FEnvValue, Round( FInputs[ i_outputtype]));
      FOutputs[ o_inv] := 1.0 - FOutputs[ o_out];
      FOutputs[ o_eoc] := LogicToSignal( FEocCount > 0);
      DecToZero( FTrigCount);
      DecToZero( FEocCount );
      DecToZero( FEoCLed   );
    end;


    procedure   TModEnvADSR.DoTick; // override;
    begin
      if SignalToLogic( FInputs[ i_trig])
      then FTrigFlag := True;
      FOutputs[ o_sigout] := FInputs[ i_sigin] * FOutputs[ o_out]; // sigout := sigin * out
    end;


    procedure   TModEnvADSR.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'  , LogicToSignal( not ( FState in [ adsrIdle, adsrWait]))));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigin', LogicToSignal( FTrigCount > 0)                        ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'eoc'   , LogicToSignal( FEoCLed    > 0)                        ));
    end;


    procedure   TModEnvADSR.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    var
      ATime      : TSignal;
      DTime      : TSignal;
      STime      : TSignal;
      RTime      : TSignal;
      TTime      : TSignal;
      TTimeRec   : TSignal;
      AShape     : TEnvShape;
      DShape     : TEnvShape;
      SShape     : TEnvShape;
      RShape     : TEnvShape;
      Swell      : TSignal;
      SLevel     : TSignal;
      i          : Integer;
      p          : Integer;
      R          : TSignal;
      StartLevel : TSignal;
      EndLevel   : TSignal;
    begin
      ATime  := LookupEnvTime( SignalToEnvRange( FInputs[ i_arange]), FInputs[ i_attack ]);
      DTime  := LookupEnvTime( SignalToEnvRange( FInputs[ i_drange]), FInputs[ i_decay  ]);
      RTime  := LookupEnvTime( SignalToEnvRange( FInputs[ i_rrange]), FInputs[ i_release]);
      STime  := ( ATime + DTime + RTime) * 0.333333;
      TTime  := ATime + DTime + STime + RTime;

      if TTIme = 0
      then Exit;

      TTimeRec := 67 / TTime;
      ATime    := ATime * TTimeRec;
      DTime    := DTime * TTimeRec;
      RTime    := RTime * TTimeRec;
      STime    := STime * TTimeRec;
      AShape   := SignalToEnvShape( FInputs[ i_ashape    ]);
      DShape   := SignalToEnvShape( FInputs[ i_dshape    ]);
      SShape   := SignalToEnvShape( FInputs[ i_swellshape]);
      RShape   := SignalToEnvShape( FInputs[ i_rshape    ]);
      SLevel   := FInputs[ i_sustain];
      Swell    := FInputs[ i_swell  ];

      StartLevel := 0;
      EndLevel   := 1 - Swell / 3;
      i          := 0;

      while i < ATime
      do begin
        case AShape of
          esLog : R := LogAttack( i, ATime, StartLevel, EndLevel);
          esLin : R := LinAttack( i, ATime, StartLevel, EndLevel);
          esExp : R := ExpAttack( i, ATime, StartLevel, EndLevel);
          else    R := SAttack  ( i, ATime, StartLevel, EndLevel);
        end;

        FData[ i] := SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
      end;

      StartLevel := 1 - Swell / 3;
      EndLevel   := SLevel;
      p          := 0;

      while i < ATime + DTime
      do begin
        case DShape of
          esLog : R := LogDecay( p, DTime, StartLevel, EndLevel);
          esLin : R := LinDecay( p, DTime, StartLevel, EndLevel);
          esExp : R := ExpDecay( p, DTime, StartLevel, EndLevel);
          else    R := SDecay  ( p, DTime, StartLevel, EndLevel);
        end;

        FData[ i] := SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
        Inc( p);
      end;

      StartLevel := SLevel;
      EndLevel   := SLevel + Swell * ( 1 - SLevel);
      p          := 0;

      while i < ATime + DTime + STime
      do begin
        case SShape of
          esLog : R := LogAttack( p, STime, StartLevel, EndLevel);
          esLin : R := LinAttack( p, STime, StartLevel, EndLevel);
          esExp : R := ExpAttack( p, STime, StartLevel, EndLevel);
          else    R := SAttack  ( p, STime, StartLevel, EndLevel);
        end;

        FData[ i] := SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
        Inc( p);
      end;

      StartLevel := SLevel + Swell * ( 1 - SLevel);
      EndLevel   := 0;
      p          := 0;

      while i < 67
      do begin
        case RShape of
          esLog : R := LogDecay( p, RTime, StartLevel, EndLevel);
          esLin : R := LinDecay( p, RTime, StartLevel, EndLevel);
          esExp : R := ExpDecay( p, RTime, StartLevel, EndLevel);
          else    R := SDecay  ( p, RTime, StartLevel, EndLevel);
        end;

        FData[ i] := SignalForEnvOutputType( R, Round( FInputs[ i_outputtype]));
        Inc( i);
        Inc( p);
      end;

      aCallback( Self, MakeInfo( aPrefix, Name, 'viewer', FData));
    end;


{ ========
  TModEnvControl = class( TMod)
  private
    FLevel        : TSignal;
    FTrig         : Boolean;
    FState        : TARState;
    FValue        : TSignal;
    FEnvValue     : TSignal;
    FTrigCount    : Integer;
    FArmed        : Boolean;
    FCoeffAttack  : TSignal;
    FCoeffRelease : TSignal;
    FEocCount     : Integer;
    FEoCLed       : Integer;
  public
}

    procedure   TModEnvControl.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_trig   , 'trig'   );
      AddInput ( i_attack , 'attack' );
      AddInput ( i_release, 'release');
      AddInput ( i_level  , 'level'  );
      AddInput ( i_mute   , 'mute'   );
      AddInput ( i_hold   , 'hold'   );
      AddOutput( o_out    , 'out'    );
      AddOutput( o_eoc    , 'eoc'    );
    end;


    procedure   TModEnvControl.DoTick; // override;
    begin
      FTrig := SignalToLogic( FInputs[ i_trig]);
    end;


    procedure   TModEnvControl.DoSlowTick; // override;
    var
      R : TSignal;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FState    := arIdle;
        FValue    := 0;
        FEocCount := TrigPulseTime( IsSpedUp);
        FEoCLed   := LedFlashTime ( IsSpedUp);
      end;

      FOutputs[ o_eoc] := LogicToSignal( False);
      FLevel := Clip( FInputs[ i_level], 0.0, 1.0);

      if FArmed and FTrig
      then FState := arIdle;

      case FState of

        arIdle :

          if FTrig
          then begin
            FCoeffAttack  :=         FLevel  / Max( 1, TimeToSampleCount( LookupEnvTime( erFast, FInputs[ i_attack ]))); // Linear attack
            FCoeffRelease := ( 1.0 - FLevel) / Max( 1, TimeToSampleCount( LookupEnvTime( erFast, FInputs[ i_release]))); // Linear release
            FState        := arA;
            FArmed        := False;
            FValue        := 0;
            FEnvValue     := 0.0;
            FTrigCount    := LedFlashTime( IsSpedUp);
          end;

        arA :

          begin
            if not FArmed
            then FArmed := not FTrig;

            R := FValue;

            if R < 0.001
            then R := 0.001;

            R := R + FCoeffAttack;

            if R > FLevel
            then begin
              if SignalToLogic( FInputs[ i_trig])
              then begin
                if SignalToLogic( FInputs[ i_hold])
                then FState := arH
                else FState := arR;
              end;
            end
            else FValue := R;
          end;

        arH :

          begin
            if not SignalToLogic( FInputs[ i_trig])
            then FState := arR;
          end;

        arR :

          begin
            if not FArmed
            then FArmed := not FTrig;

            R := FValue;
            R := R + FCoeffRelease;

            if R > 1.0
            then begin
              FValue    := 1.0;
              FState    := arWait;
              FEocCount := TrigPulseTime( IsSpedUp);
              FEoCLed   := LedFlashTime ( IsSpedUp);
            end
            else FValue := R;
          end;

        arWait :

          begin
            if not FTrig or FArmed
            then FState := arIdle;
          end;

      end;

      case Round( FInputs[ i_mute]) of
        1  : FEnvValue := Normalize(  EnvelopeSmoothingSlow * FEnvValue           / ( 1 + EnvelopeSmoothingSlow));
        else FEnvValue := Normalize(( EnvelopeSmoothingSlow * FEnvValue + FValue) / ( 1 + EnvelopeSmoothingSlow));
      end;

      FTrig := False;
      DecToZero( FEocCount );
      DecToZero( FEocLed   );
      DecToZero( FTrigCount);
      FOutputs[ o_out] := FEnvValue;
      FOutputs[ o_eoc] := LogicToSignal( FEocCount > 0);
    end;


    procedure   TModEnvControl.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig', LogicToSignal( FTrigCount > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'eoc' , LogicToSignal( FEoCLed    > 0)));
    end;


{ ========
  TModSequencer = class( TMod)
  private
    FOutValue : TSignal;
    FSteps    : Integer;
    FCounter  : Integer;
    FInState  : TSeqInState;
    FOldRes   : Boolean;
  public
}

    procedure   TModSequencer.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_trig  , 'trig'  );
      AddInput ( i_reset , 'reset' );
      AddInput ( i_dir   , 'dir'   );
      AddInput ( i_chain , 'chain' );
      AddInput ( i_rnd   , 'rnd'   );
      AddInput ( i_active, 'active');
      AddInput ( i_step1 , 'step1' );
      AddInput ( i_step2 , 'step2' );
      AddInput ( i_step3 , 'step3' );
      AddInput ( i_step4 , 'step4' );
      AddInput ( i_step5 , 'step5' );
      AddInput ( i_step6 , 'step6' );
      AddInput ( i_step7 , 'step7' );
      AddInput ( i_step8 , 'step8' );
      AddInput ( i_step9 , 'step9' );
      AddInput ( i_step10, 'step10');
      AddInput ( i_step11, 'step11');
      AddInput ( i_inv   , 'inv'   );
      AddOutput( o_out   , 'out'   );
      AddOutput( o_trig1 , 'trig1' );
      AddOutput( o_trig2 , 'trig2' );
      AddOutput( o_trig3 , 'trig3' );
      AddOutput( o_trig4 , 'trig4' );
      AddOutput( o_trig5 , 'trig5' );
      AddOutput( o_trig6 , 'trig6' );
      AddOutput( o_trig7 , 'trig7' );
      AddOutput( o_trig8 , 'trig8' );
      AddOutput( o_trig9 , 'trig9' );
      AddOutput( o_trig10, 'trig10');
      AddOutput( o_trig11, 'trig11');
    end;


    procedure   TModSequencer.SetDefaults; // override;
    begin
      FOutValue          := 0;    // Neutral output
      FInputs[ i_dir   ] := 1;    // Make seq run forwards by default.
      FInputs[ i_chain ] := 0;    // Chain input to zero
      FInputs[ i_active] := 1;    // Sequencer active when active input not connected
      FCounter           := 0;
    end;


    procedure   TModSequencer.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'steps')
      then FSteps := Clip( Ceil( aValue), 1, 11)
      else inherited;
    end;


    function    TModSequencer.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'steps')
      then Result := FSteps
      else Result := inherited;
    end;


    procedure   TModSequencer.DoTick; // override;
    begin
      if not FOldRes and SignalToLogic( FInputs[ i_reset])
      then begin
        FInState  := seqClkHigh;
        FCounter  := 0;
        FOutValue := 0;
      end;

      FOldRes := SignalToLogic( FInputs[ i_reset]);
    end;


    procedure   TModSequencer.DoSlowTick; // override;
    var
      i : Integer;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FInState  := seqClkHigh;
        FCounter  := 0;
        FOutValue := 0;
      end;

      case FInState of

        seqClkLow :

          begin
            if SignalToLogic( FInputs[ i_trig])
            then begin
              if FSteps > 0
              then begin
                FInState := seqClkHigh;

                if SignalToLogic( FInputs[ i_dir])
                then FCounter := ( FCounter + 1) mod FSteps
                else begin
                  if FCounter <= 0
                  then FCounter := FSteps - 1
                  else Dec( FCounter);
                end;
              end;
            end;
          end;

        seqClkHigh :

          begin
            if not SignalToLogic( FInputs[ i_trig])
            then FInState := seqClkLow;
          end;

      end; // case

      if SignalToLogic( FInputs[ i_active])
      then begin
        if Round( FInputs[ i_inv]) = 0
        then FOutValue := FInputs[ FCounter + i_step1] - MiddleNote * NOTE_SCALING_REC
        else FOutValue := MiddleNote * NOTE_SCALING_REC - FInputs[ FCounter + i_step1];
      end
      else FOutValue := 0;

      FOutputs[ o_out] := FOutValue + FInputs[ i_chain];

      for i := o_trig1 to o_trig11
      do begin
        if ( i = FCounter + o_trig1) and SignalToLogic( FInputs[ i_active])
        then FOutputs[ i] := LogicToSignal( True )
        else FOutputs[ i] := LogicToSignal( False);
      end;
    end;


    procedure   TModSequencer.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'  , FInputs [ i_trig  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset' , FInputs [ i_reset ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active', FInputs [ i_active]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'   , FInputs [ i_dir   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'inv'   , FInputs [ i_inv   ]));

      for i := o_trig1 to o_trig11
      do begin
        if ( i = FCounter + o_trig1) and SignalToLogic( FInputs[ i_active])
        then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i - o_trig1 + 1], AppLocale), 1.0))
        else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i - o_trig1 + 1], AppLocale), 0.0));
      end;
    end;


{ ========
  TModSeqSeq = class( TMod)
  private
    FState           : Word;
    FSteps           : Integer;
    FSequenceCounter : Integer;
    FPeriodCounter   : Integer;
    FOldRes          : Boolean;
    FOldTrig         : Boolean;
    FClkCnt          : Integer;
  public
}

    procedure   TModSeqSeq.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_res    , 'res'    );
      AddInput ( i_trig   , 'trig'   );
      AddInput ( i_dir    , 'dir'    );
      AddInput ( i_resmod , 'resmod' );
      AddOutput( o_outres , 'outres' );
      AddOutput( o_outtrig, 'outtrig');
      AddOutput( o_outdir , 'outdir' );
      AddOutput( o_out1   , 'out1'   );
      AddOutput( o_out2   , 'out2'   );
      AddOutput( o_out3   , 'out3'   );
      AddOutput( o_out4   , 'out4'   );
      AddOutput( o_out5   , 'out5'   );
      AddOutput( o_out6   , 'out6'   );
      AddOutput( o_out7   , 'out7'   );
      AddOutput( o_out8   , 'out8'   );
      AddOutput( o_out9   , 'out9'   );
      AddOutput( o_out10  , 'out10'  );
      AddOutput( o_out11  , 'out11'  );
      AddOutput( o_out12  , 'out12'  );
      AddOutput( o_out13  , 'out13'  );
      AddOutput( o_out14  , 'out14'  );
      AddOutput( o_out15  , 'out15'  );
      AddOutput( o_out16  , 'out16'  );
    end;


    procedure   TModSeqSeq.SetDefaults; // override;
    begin
      FInputs[ i_dir]  :=  1;        // Go forwards with unconnected dir input
      FInputs[ i_res]  := -1;        // No reset on unconnected input
      FSequenceCounter := 16;
      FPeriodCounter   := 24;
    end;


    procedure   TModSeqSeq.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'steps')
      then FSteps := Clip( Ceil( aValue), 1, 256)
      else inherited;
    end;


    function    TModSeqSeq.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'steps')
      then Result := FSteps
      else Result := inherited;
    end;


    procedure   TModSeqSeq.DoSlowTick; // override;
    const
      MaxCount = 16;
    var
      Direction : Boolean;
      Trig      : Boolean;
      Res       : Boolean;
      i         : Integer;
      DoChange  : Boolean;
      EndOfSeq  : Boolean;
      MustReset : Boolean;
      ResMod    : Integer;
    begin
      DoChange  := False;
      MustReset := False;

      ResMod    := Round        ( FInputs[ i_resmod]);
      Direction := SignalToLogic( FInputs[ i_dir   ]);
      Trig      := SignalToLogic( FInputs[ i_trig  ]);
      Res       := SignalToLogic( FInputs[ i_res   ]);
      EndOfSeq  := FPeriodCounter <= 0;

      if (( not FOldRes) and Res) or ResetFlag or EndOfSeq
      then begin
        if Direction
        then begin                                             // Forwards , reset to first state
          FState           :=  0;
          FSequenceCounter := 15;
        end
        else begin                                             // Backwards , reset to Prev state
          FState           :=   FSteps shr 4;
          FSequenceCounter := ( FSteps and 15) - 1;
        end;

        FClkCnt        := c_clkdelay;
        FPeriodCounter := FSteps;
        DoChange       := True;
        MustReset      := MustReset or ( ResMod in [ m_res1, m_res3]);   // Propagate reset on received reset or end of sequence in mode 1 and 3
        ResetFlag      := False;
        FOldTrig       := True;
      end
      else begin
        if Trig and not FOldTrig
        then begin
          if not EndOfSeq
          then Dec( FPeriodCounter);

          if FSequenceCounter > 0
          then Dec( FSequenceCounter)
          else begin
            FSequenceCounter := 15;

            if Direction
            then begin                                         // Forwards, go to nexxt state
              FState := FState + 1;

              if FState > FSteps shr 4
              then FState := 0;
            end
            else begin                                         // Backwards, go to previous state
              if FState > 0
              then Dec( FState)
              else FState := Fsteps shr 4;
            end;

            MustReset := MustReset or ( ResMod in [ m_res2, m_res3]);    // Issue reset after state change in mode 2 and 3
            DoChange  := True;
          end;

          FClkCnt := c_clkdelay;
        end
        else if FOldTrig and not Trig
        then begin
          DoChange := True;
          FClkCnt  := c_clkdelay;
        end;

        FOldTrig := Trig;
      end;

      FOldRes := Res;

      if FClkCnt > 0
      then begin
        Dec( FClkCnt);
        if FClkCnt = 0
        then FOutputs[ o_outtrig] := LogicToSignal( Trig);
      end;

      if DoChange
      then begin
        for i := 0 to MaxCount - 1
        do begin
          if i = FState
          then FOutputs[ o_out1 + i] := LogicToSignal( True )
          else FOutputs[ o_out1 + i] := LogicToSignal( False);
        end;

        FOutputs[ o_outres] := FInputs[ i_res];
        FOutputs[ o_outdir] := FInputs[ i_dir];
      end;

      if MustReset
      then begin
        FOutputs[ o_outres ] := LogicToSignal( True);
        FOutputs[ o_outtrig] := LogicToSignal( True);
      end;
    end;


    procedure   TModSeqSeq.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := o_out1 to o_out16
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ 1 + i - o_out1], AppLocale), FOutputs[ i]));

      aCallback( Self, MakeInfo( aPrefix, Name, 'trig', FInputs[ i_trig]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res' , FInputs[ i_res ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir' , FInputs[ i_dir ]));
    end;


{ ========
  TModSeq16 = class( TMod)
  private
    FOutValue   : TSignal;
    FOutValue2  : TSignal;
    FOutValue3  : TSignal;
    FOutValue4  : TSignal;
    FReset      : Boolean;
    FNextValue  : TSignal;
    FNextValue2 : TSignal;
    FNextValue3 : TSignal;
    FNextValue4 : TSignal;
    FForwards   : Boolean;
    FActive     : Boolean;
    FChainValue : TSignal;
    FSteps      : Integer;
    FCounter    : Integer;
    FInState    : TSeqInState;
    FOldRes     : Boolean;
    FTicks      : Integer;
    FPeriodTime : Integer;
    FInvert     : Boolean;
    FRandom     : Boolean;
    FOldRandom  : Boolean;
    FRndReq     : Boolean;
    FFreeze     : Boolean;
    FOldTrig    : Boolean;
    FGateMode   : Boolean;
    FSkips      : set of 0 .. 15;
    FTimeWarp   : TSignal;
  private
}

    procedure   TModSeq16.SampleInputs;
    begin
      FSteps      := FPreSteps;
      FInState    := seqClkHigh;
      FChainValue := FInputs[ i_chain];
      FActive     := SignalToLogic( FInputs[ i_active  ]) and ( FSteps > 0);
      FForwards   := SignalToLogic( FInputs[ i_dir     ]);
      FReset      := SignalToLogic( FInputs[ i_res     ]);
      FFreeze     := SignalToLogic( FInputs[ i_hold    ]);
      FGateMode   := SignalToLogic( FInputs[ i_gatemode]);
    end;


    procedure   TModSeq16.MakeNextValue;
    var
      FMode : Integer;
    begin
      FMode      := Round( FInputs[ i_mode]);
      FOutValue  := FNextValue;
      FOutValue2 := FNextValue2;
      FOutValue3 := FNextValue3;
      FOutValue4 := FNextValue4;

      if FActive
      then begin
        if FMode = m_note
        then begin
          if FInvert
          then begin
            FNextValue  := NoteNumberToUnits( MiddleNote) - FInputs[  FCounter               + i_step1];
            FNextValue2 := NoteNumberToUnits( MiddleNote) - FInputs[( FCounter +  4 ) and 15 + i_step1];
            FNextValue3 := NoteNumberToUnits( MiddleNote) - FInputs[( FCounter +  8 ) and 15 + i_step1];
            FNextValue4 := NoteNumberToUnits( MiddleNote) - FInputs[( FCounter + 12 ) and 15 + i_step1];
          end
          else begin
            FNextValue  := FInputs[  FCounter              + i_step1] - NoteNumberToUnits( MiddleNote);
            FNextValue2 := FInputs[( FCounter +  4) and 15 + i_step1] - NoteNumberToUnits( MiddleNote);
            FNextValue3 := FInputs[( FCounter +  8) and 15 + i_step1] - NoteNumberToUnits( MiddleNote);
            FNextValue4 := FInputs[( FCounter + 12) and 15 + i_step1] - NoteNumberToUnits( MiddleNote);
          end;
        end
        else if FMode in [ m_lin11, m_lin01, m_db0]
        then begin
          if FInvert
          then begin
            FNextValue  := 1.0 - FInputs[  FCounter              + i_step1];
            FNextValue2 := 1.0 - FInputs[( FCounter +  4) and 15 + i_step1];
            FNextValue3 := 1.0 - FInputs[( FCounter +  8) and 15 + i_step1];
            FNextValue3 := 1.0 - FInputs[( FCounter + 12) and 15 + i_step1];
          end
          else begin
            FNextValue  := FInputs[  FCounter              + i_step1];
            FNextValue2 := FInputs[( FCounter +  4) and 15 + i_step1];
            FNextValue3 := FInputs[( FCounter +  8) and 15 + i_step1];
            FNextValue4 := FInputs[( FCounter + 12) and 15 + i_step1];
          end;
        end
        else if FMode in [ m_lin44, m_lin04, m_db4]
        then begin
          if FInvert
          then begin
            FNextValue  := 4.0 - FInputs[  FCounter              + i_step1];
            FNextValue2 := 4.0 - FInputs[( FCounter +  4) and 15 + i_step1];
            FNextValue3 := 4.0 - FInputs[( FCounter +  8) and 15 + i_step1];
            FNextValue4 := 4.0 - FInputs[( FCounter + 12) and 15 + i_step1];
          end
          else begin
            FNextValue  := FInputs[  FCounter              + i_step1];
            FNextValue2 := FInputs[( FCounter +  4) and 15 + i_step1];
            FNextValue3 := FInputs[( FCounter +  8) and 15 + i_step1];
            FNextValue4 := FInputs[( FCounter + 12) and 15 + i_step1];
          end;
        end
        else begin
          if FInvert
          then begin
            FNextValue  := 1.0 - FInputs[  FCounter              + i_step1];
            FNextValue2 := 1.0 - FInputs[( FCounter +  4) and 15 + i_step1];
            FNextValue3 := 1.0 - FInputs[( FCounter +  8) and 15 + i_step1];
            FNextValue4 := 1.0 - FInputs[( FCounter + 12) and 15 + i_step1];
          end
          else begin
            FNextValue  := FInputs[  FCounter              + i_step1];
            FNextValue2 := FInputs[( FCounter +  4) and 15 + i_step1];
            FNextValue3 := FInputs[( FCounter +  8) and 15 + i_step1];
            FNextValue4 := FInputs[( FCounter + 12) and 15 + i_step1];
          end;
        end;
      end
      else begin
        FNextValue  := FChainValue;
        FNextValue2 := FChainValue;
        FNextValue3 := FChainValue;
        FNextValue4 := FChainValue;
      end;
    end;


    procedure   TModSeq16.FixSkips;
    var
      i : Integer;
    begin
      FSkips     := [];
      FSkipCount := 0;

      for i := 0 to 15
      do begin
        if SignalToLogic( FInputs[ i_skip1 + i])
        then begin
          FSkips := FSkips + [ i];

          if i < FSteps
          then Inc( FSkipCount);
        end;
      end;
    end;


//  public

    procedure   TModSeq16.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig     , 'trig'     );
      AddInput ( i_res      , 'res'      );
      AddInput ( i_dir      , 'dir'      );
      AddInput ( i_chain    , 'chain'    );
      AddInput ( i_rnd      , 'rnd'      );
      AddInput ( i_active   , 'active'   );
      AddInput ( i_mode     , 'mode'     );
      AddInput ( i_xfade    , 'xfade'    );
      AddInput ( i_curveup  , 'curveup'  );
      AddInput ( i_curvedown, 'curvedown');
      AddInput ( i_step1    , 'step1'    );
      AddInput ( i_step2    , 'step2'    );
      AddInput ( i_step3    , 'step3'    );
      AddInput ( i_step4    , 'step4'    );
      AddInput ( i_step5    , 'step5'    );
      AddInput ( i_step6    , 'step6'    );
      AddInput ( i_step7    , 'step7'    );
      AddInput ( i_step8    , 'step8'    );
      AddInput ( i_step9    , 'step9'    );
      AddInput ( i_step10   , 'step10'   );
      AddInput ( i_step11   , 'step11'   );
      AddInput ( i_step12   , 'step12'   );
      AddInput ( i_step13   , 'step13'   );
      AddInput ( i_step14   , 'step14'   );
      AddInput ( i_step15   , 'step15'   );
      AddInput ( i_step16   , 'step16'   );
      AddInput ( i_xfademod , 'xfademod' );
      AddInput ( i_inv      , 'inv'      );
      AddInput ( i_rnd2     , 'rnd2'     );
      AddInput ( i_hold     , 'hold'     );
      AddInput ( i_skip1    , 'skip1'    );
      AddInput ( i_skip2    , 'skip2'    );
      AddInput ( i_skip3    , 'skip3'    );
      AddInput ( i_skip4    , 'skip4'    );
      AddInput ( i_skip5    , 'skip5'    );
      AddInput ( i_skip6    , 'skip6'    );
      AddInput ( i_skip7    , 'skip7'    );
      AddInput ( i_skip8    , 'skip8'    );
      AddInput ( i_skip9    , 'skip9'    );
      AddInput ( i_skip10   , 'skip10'   );
      AddInput ( i_skip11   , 'skip11'   );
      AddInput ( i_skip12   , 'skip12'   );
      AddInput ( i_skip13   , 'skip13'   );
      AddInput ( i_skip14   , 'skip14'   );
      AddInput ( i_skip15   , 'skip15'   );
      AddInput ( i_skip16   , 'skip16'   );
      AddInput ( i_clockskip, 'clockskip');
      AddInput ( i_gatemode , 'gatemode' );
      AddInput ( i_rnd3     , 'rnd3'     );
      AddOutput( o_out      , 'out'      );
      AddOutput( o_outres   , 'outres'   );
      AddOutput( o_outtrig  , 'outtrig'  );
      AddOutput( o_outdir   , 'outdir'   );
      AddOutput( o_out2     , 'out2'     );
      AddOutput( o_out3     , 'out3'     );
      AddOutput( o_out4     , 'out4'     );
      AddOutput( o_stepout1 , 'stepout1' );
      AddOutput( o_stepout2 , 'stepout2' );
      AddOutput( o_stepout3 , 'stepout3' );
      AddOutput( o_stepout4 , 'stepout4' );
      AddOutput( o_stepout5 , 'stepout5' );
      AddOutput( o_stepout6 , 'stepout6' );
      AddOutput( o_stepout7 , 'stepout7' );
      AddOutput( o_stepout8 , 'stepout8' );
      AddOutput( o_stepout9 , 'stepout9' );
      AddOutput( o_stepout10, 'stepout10');
      AddOutput( o_stepout11, 'stepout11');
      AddOutput( o_stepout12, 'stepout12');
      AddOutput( o_stepout13, 'stepout13');
      AddOutput( o_stepout14, 'stepout14');
      AddOutput( o_stepout15, 'stepout15');
      AddOutput( o_stepout16, 'stepout16');
      AddOutput( o_timewarp , 'timewarp' );
      AddOutput( o_outraw   , 'outraw'   );
      AddOutput( o_outraw2  , 'outraw2'  );
      AddOutput( o_outraw3  , 'outraw3'  );
      AddOutput( o_outraw4  , 'outraw4'  );
    end;


    procedure   TModSeq16.SetDefaults; // override;
    var
      i : Integer;
    begin
      inherited;
      FInputs[ i_dir     ] := 1.0;         // Make seq run forwards by default.
      FInputs[ i_chain   ] := 0.0;         // Chain input to zero wwhen not connected
      FInputs[ i_active  ] := 1.0;         // Sequencer active when active input not connected
      FInputs[ i_xfademod] := 1.0;         // Full xfade range
      FInputs[ i_mode    ] := m_note;      // Default startup mode is notes
      FNextValue         := 0;             // Neutral output
      FPeriodTime        := 2;
      FTicks             := 0;
      FTimeWarp          := 1.0;
      FSkipCount         := 0;

      for i := 0 to 15
      do FInputs[ i_skip1 + i] := 0.0;

      SampleInputs;

      if FForwards
      then FCounter := 0
      else FCounter := FSteps - 1;
    end;


    procedure   TModSeq16.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'steps')
      then FPreSteps := Clip( Ceil( aValue), 1, 16)
      else inherited;
    end;


    function    TModSeq16.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'steps')
      then Result := FPreSteps
      else Result := inherited;
    end;


    procedure   TModSeq16.DoSlowTick; // override;
    var
      XFade   : TSignal;
      OutVal  : TSignal;
      OutVal2 : TSignal;
      OutVal3 : TSignal;
      OutVal4 : TSignal;
      Count   : Integer;
      NewTrig : Boolean;
      ClkSkip : Boolean;
      DoTrig  : Boolean;
      i       : Integer;
    begin
      if ResetFlag
      then begin
        SampleInputs;
        FPeriodTime := Max( 2, FTicks);
        FTicks      := 0;
        FNextValue  := 0;

        if FForwards
        then FCounter := 0
        else FCounter := FSteps - 1;

        FixSkips;
        MakeNextValue;
      end
      else begin
        NewTrig := SignalToLogic( FInputs[ i_trig     ]);
        ClkSkip := SignalToLogic( FInputs[ i_clockskip]);

        case FInState of

          seqClkLow :

            begin
              if NewTrig and not FOldTrig and not ClkSkip
              then begin
                SampleInputs;

                if ResetFlag or ( not FOldRes and FReset)
                then begin
                  FPeriodTime := Max( 2, FTicks);
                  FTicks      := 0;

                  if FForwards
                  then FCounter := 0
                  else FCounter := FSteps - 1;
                end
                else begin
                  if FSteps > 0
                  then begin
                 // SampleInputs;
                    FPeriodTime := Max( 2, FTicks);
                    FTicks      := 0;
                    FixSkips;

                    if FSkips <> [ 0 .. 15]
                    then begin
                      Count := 15;

                      repeat
                        Dec( Count);

                        if FForwards
                        then FCounter := ( FCounter + 1) mod FSteps
                        else begin
                          if FCounter <= 0
                          then FCounter := FSteps - 1
                          else Dec( FCounter);
                        end;
                      until not ( FCounter in FSkips) or ( Count < 0);
                    end;
                  end;
                end;

                MakeNextValue;
              end;
            end;

          seqClkHigh :

            begin
              if not NewTrig and FOldTrig
              then begin
                FInState    := seqClkLow;
                FPeriodTime := FTicks * 2;
              end;
            end;
        end; // case

        FOldTrig := NewTrig;
      end;

      XFade   := Clip((( FPeriodTime - FTicks) / FPeriodTime) * FInputs[ i_xfade] * FInputs[ i_xfademod], -1, 1);
      FInvert := SignalToLogic( FInputs[ i_inv]);

      if FOutValue < FNextValue      // going up, note that as final and current value are reversed this must call decay ...
      then begin
        case Round( FInputs[ i_curveup]) of
          0  : OutVal := LogDecay( XFade, 1, FNextValue, FOutValue);
          1  : OutVal := LinDecay( XFade, 1, FNextValue, FOutValue);
          2  : OutVal := ExpDecay( XFade, 1, FNextValue, FOutValue);
          else OutVal := SDecay  ( XFade, 1, FNextValue, FOutValue);
        end;
      end
      else begin                     // going down
        case Round( FInputs[ i_curvedown]) of
          0  : OutVal := LogAttack( XFade, 1, FNextValue, FOutValue);
          1  : OutVal := LinAttack( XFade, 1, FNextValue, FOutValue);
          2  : OutVal := ExpAttack( XFade, 1, FNextValue, FOutValue);
          else OutVal := SAttack  ( XFade, 1, FNextValue, FOutValue);
        end;
      end;

      if FOutValue2 < FNextValue2
      then begin
        case Round( FInputs[ i_curveup]) of
          0  : OutVal2 := LogDecay( XFade, 1, FNextValue2, FOutValue2);
          1  : OutVal2 := LinDecay( XFade, 1, FNextValue2, FOutValue2);
          2  : OutVal2 := ExpDecay( XFade, 1, FNextValue2, FOutValue2);
          else OutVal2 := SDecay  ( XFade, 1, FNextValue2, FOutValue2);
        end;
      end
      else begin                     // going down
        case Round( FInputs[ i_curvedown]) of
          0  : OutVal2 := LogAttack( XFade, 1, FNextValue2, FOutValue2);
          1  : OutVal2 := LinAttack( XFade, 1, FNextValue2, FOutValue2);
          2  : OutVal2 := ExpAttack( XFade, 1, FNextValue2, FOutValue2);
          else OutVal2 := SAttack  ( XFade, 1, FNextValue2, FOutValue2);
        end;
      end;

      if FOutValue3 < FNextValue3
      then begin
        case Round( FInputs[ i_curveup]) of
          0  : OutVal3 := LogDecay( XFade, 1, FNextValue3, FOutValue3);
          1  : OutVal3 := LinDecay( XFade, 1, FNextValue3, FOutValue3);
          2  : OutVal3 := ExpDecay( XFade, 1, FNextValue3, FOutValue3);
          else OutVal3 := SDecay  ( XFade, 1, FNextValue3, FOutValue3);
        end;
      end
      else begin                     // going down
        case Round( FInputs[ i_curvedown]) of
          0  : OutVal3 := LogAttack( XFade, 1, FNextValue3, FOutValue3);
          1  : OutVal3 := LinAttack( XFade, 1, FNextValue3, FOutValue3);
          2  : OutVal3 := ExpAttack( XFade, 1, FNextValue3, FOutValue3);
          else OutVal3 := SAttack  ( XFade, 1, FNextValue3, FOutValue3);
        end;
      end;

      if FOutValue4 < FNextValue4
      then begin
        case Round( FInputs[ i_curveup]) of
          0  : OutVal4 := LogDecay( XFade, 1, FNextValue4, FOutValue4);
          1  : OutVal4 := LinDecay( XFade, 1, FNextValue4, FOutValue4);
          2  : OutVal4 := ExpDecay( XFade, 1, FNextValue4, FOutValue4);
          else OutVal4 := SDecay  ( XFade, 1, FNextValue4, FOutValue4);
        end;
      end
      else begin                     // going down
        case Round( FInputs[ i_curvedown]) of
          0  : OutVal4 := LogAttack( XFade, 1, FNextValue4, FOutValue4);
          1  : OutVal4 := LinAttack( XFade, 1, FNextValue4, FOutValue4);
          2  : OutVal4 := ExpAttack( XFade, 1, FNextValue4, FOutValue4);
          else OutVal4 := SAttack  ( XFade, 1, FNextValue4, FOutValue4);
        end;
      end;

      FRandom  := SignalToLogic( FInputs[ i_rnd2]);
      FRandom3 := SignalToLogic( FInputs[ i_rnd3]);

      if not FOldRandom and FRandom
      then FRndReq := True;

      if not FOldRandom3 and FRandom3
      then FRndReq3 := True;

      if not FFreeze
      then begin
        FOutputs[ o_out    ] := OutVal;
        FOutputs[ o_out2   ] := OutVal2;
        FOutputs[ o_out3   ] := OutVal3;
        FOutputs[ o_out4   ] := OutVal4;
        FOutputs[ o_outraw ] := FNextValue;
        FOutputs[ o_outraw2] := FNextValue2;
        FOutputs[ o_outraw3] := FNextValue3;
        FOutputs[ o_outraw4] := FNextValue4;
        FOutputs[ o_outres ] := FInputs[ i_res ];
        FOutputs[ o_outtrig] := FInputs[ i_trig];
        FOutputs[ o_outdir ] := FInputs[ i_dir ];
      end;

      DoTrig := FActive and ( SignalToLogic( FInputs[ i_trig]) or FGateMode);

      for i := 0 to 15
      do FOutputs[ o_stepout1 + i] := LogicToSignal(( FCounter = i) and DoTrig);

      if FActive and ( FSteps > 0)
      then FTimeWarp := ( FSteps - FSkipCount) / FSteps
      else FTimeWarp := 1.0;

      FOutputs[ o_timewarp] := FTimeWarp;

      if FTicks < MAXINT shr 2    // Cap FTicks value in case the module gets not clocked prolly MAXINT shr 1 would be OK too
      then Inc( FTicks);

      FOldRes     := FReset;
      ResetFlag   := False;
      FOldRandom  := FRandom;
      FOldRandom3 := FRandom3;
    end;


    procedure   TModSeq16.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'     , FInputs [ i_trig      ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'    , FInputs [ i_res       ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active'   , FInputs [ i_active    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'      , FInputs [ i_dir       ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'inv'      , FInputs [ i_inv       ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clockskip', FInputs [ i_clockskip ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'hold'     , LogicToSignal( FFreeze)));

      for i := i_step1 to i_step16
      do begin
        if ( i = FCounter + i_step1) and SignalToLogic( FInputs[ i_active])
        then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i - i_step1 + 1], AppLocale), 1.0))
        else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i - i_step1 + 1], AppLocale), 0.0));
      end;
    end;


    procedure   TModSeq16.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // override;
    begin
      if FRndReq
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'rnd', 1));
        FRndReq := False;
      end;

      if FRndReq3
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'rndone', 16));
        FRndReq3 := False;
      end;
    end;


{ ========
  TModVcps = class( TMod)
  // Voltage controlled pattern sequence, after : http://electro-music.com/forum/topic-37829.html
  private
    FOutValue    : TSignal;
    FPattern     : TSignal;
    FLength      : Integer;
    FStart       : Integer;
    FReset       : Boolean;
    FForwards    : Boolean;
    FSteps       : Integer;
    FCounter     : Integer;
    FInState     : TSeqInState;
    FOldRes      : Boolean;
    FInvert      : Boolean;
    FClear       : Boolean;
    FOldClear    : Boolean;
    FClrReq      : Boolean;
    FRandom      : Boolean;
    FOldRandom   : Boolean;
    FRndReq      : Boolean;
    FRandom3     : Boolean;
    FOldRandom3  : Boolean;
    FRndReq3     : Boolean;
    FOldTrig     : Boolean;
    FActiveSteps : Word;
  private
}

    procedure   TModVcps.SampleAdc;
    const
      Mask = 1 shl 15;
    var
      i       : Integer;
      AdcVals : Word;
    begin
      FActiveSteps := 0;
      AdcVals      := GrayCode( Round( FStart + FLength * Clip( FPattern, 0.0, 1.0)));

      for i := i_stepmode1 to i_stepmode16
      do begin
        FActiveSteps := FActiveSteps shr 1;

        case Round( FInputs[ i]) of
          0 : FActiveSteps := FActiveSteps or ( AdcVals and Mask);  //  O   adc    => active( adc)
          1 : ;                                                     //  -   Off    => always inactive
          2 : FActiveSteps := FActiveSteps or Mask;                 //  |   Active => always active
        end;

        AdcVals := AdcVals shl 1;
      end;
    end;


    procedure   TModVcps.SampleInputs;
    var
      aStart  : Integer;
      aLength : Integer;
      anEnd   : Integer;
    begin
      FSteps       := Clip( Round( FInputs[ i_steps] * FInputs[ i_stepsmod]), 1, 16);
      FForwards    := SignalToLogic( FInputs[ i_dir]);
      FReset       := SignalToLogic( FInputs[ i_res]);
      FPattern     := FInputs[ i_pattern];
      aStart       := Round( FInputs[ i_start ] + 0.45);
      aLength      := Round( FInputs[ i_length] + 0.45);
      anEnd        := aStart + aLength;
      FStart       := 1 shl aStart;
      FLength      := ( 1 shl anEnd) - ( 1 shl aStart);
    end;


    procedure   TModVcps.MakeNextValue;
    var
      FMode : Integer;
    begin
      FMode := Round( FInputs[ i_mode]);

      if FMode = m_note
      then begin
        if FInvert
        then FOutValue  := NoteNumberToUnits( MiddleNote) - FInputs[  FCounter + i_step1]
        else FOutValue  := FInputs[  FCounter + i_step1] - NoteNumberToUnits( MiddleNote);
      end
      else if FMode in [ m_lin11, m_lin01, m_db0]
      then begin
        if FInvert
        then FOutValue  := 1.0 - FInputs[  FCounter + i_step1]
        else FOutValue  := FInputs[  FCounter + i_step1];
      end
      else if FMode in [ m_lin44, m_lin04, m_db4]
      then begin
        if FInvert
        then FOutValue  := 4.0 - FInputs[  FCounter + i_step1]
        else FOutValue  := FInputs[  FCounter + i_step1];
      end
      else begin
        if FInvert
        then FOutValue  := 1.0 - FInputs[  FCounter + i_step1]
        else FOutValue  := FInputs[  FCounter + i_step1];
      end;
    end;


//  public

    procedure   TModVcps.CreateIO;
    begin
      FIsSlow := True;
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_res       , 'res'       );
      AddInput ( i_dir       , 'dir'       );
      AddInput ( i_inv       , 'inv'       );
      AddInput ( i_pattern   , 'pattern'   );
      AddInput ( i_steps     , 'steps'     );
      AddInput ( i_stepsmod  , 'stepsmod'  );
      AddInput ( i_mode      , 'mode'      );
      AddInput ( i_start     , 'start'     );
      AddInput ( i_length    , 'length'    );
      AddInput ( i_rnd2      , 'rnd2'      );
      AddInput ( i_rnd3      , 'rnd3'      );
      AddInput ( i_clr2      , 'clr2'      );
      AddInput ( i_step1     , 'step1'     );
      AddInput ( i_step2     , 'step2'     );
      AddInput ( i_step3     , 'step3'     );
      AddInput ( i_step4     , 'step4'     );
      AddInput ( i_step5     , 'step5'     );
      AddInput ( i_step6     , 'step6'     );
      AddInput ( i_step7     , 'step7'     );
      AddInput ( i_step8     , 'step8'     );
      AddInput ( i_step9     , 'step9'     );
      AddInput ( i_step10    , 'step10'    );
      AddInput ( i_step11    , 'step11'    );
      AddInput ( i_step12    , 'step12'    );
      AddInput ( i_step13    , 'step13'    );
      AddInput ( i_step14    , 'step14'    );
      AddInput ( i_step15    , 'step15'    );
      AddInput ( i_step16    , 'step16'    );
      AddInput ( i_stepmode1 , 'stepmode1' );
      AddInput ( i_stepmode2 , 'stepmode2' );
      AddInput ( i_stepmode3 , 'stepmode3' );
      AddInput ( i_stepmode4 , 'stepmode4' );
      AddInput ( i_stepmode5 , 'stepmode5' );
      AddInput ( i_stepmode6 , 'stepmode6' );
      AddInput ( i_stepmode7 , 'stepmode7' );
      AddInput ( i_stepmode8 , 'stepmode8' );
      AddInput ( i_stepmode9 , 'stepmode9' );
      AddInput ( i_stepmode10, 'stepmode10');
      AddInput ( i_stepmode11, 'stepmode11');
      AddInput ( i_stepmode12, 'stepmode12');
      AddInput ( i_stepmode13, 'stepmode13');
      AddInput ( i_stepmode14, 'stepmode14');
      AddInput ( i_stepmode15, 'stepmode15');
      AddInput ( i_stepmode16, 'stepmode16');
      AddOutput( o_out       , 'out'       );
      AddOutput( o_outgate   , 'outgate'   );
      AddOutput( o_outtrig   , 'outtrig'   );
    end;


    procedure   TModVcps.SetDefaults;
    begin
      inherited;
      FInputs[ i_dir ]     := 1.0;         // Make seq run forwards by default.
      FInputs[ i_stepsmod] := 1.0;         // Use all steps when no modulation is applied
      FInputs[ i_mode]     := m_lin01;     // Default startup mode is linear[0,1]
      SampleInputs;
      SampleAdc;

      if FForwards
      then FCounter := 0
      else FCounter := FSteps - 1;
    end;


    procedure   TModVcps.DoSlowTick;
    var
      OutVal  : TSignal;
      NewTrig : Boolean;
      DoTrig  : Boolean;
      Mask    : Word;
      Active  : Boolean;
    begin
      SampleInputs;

      if ResetFlag or ( not FOldRes and FReset)
      then begin
        if FForwards
        then FCounter := 0
        else FCounter := FSteps - 1;

        SampleAdc;
        MakeNextValue;
      end
      else begin
        NewTrig := SignalToLogic( FInputs[ i_trig]);

        if NewTrig <> FOldTrig
        then begin
          case FInState of

            seqClkLow :

              begin
                if NewTrig and not FOldTrig
                then begin
                  SampleAdc;

                  if FForwards
                  then FCounter := ( FCounter + 1) mod FSteps
                  else begin
                    if FCounter <= 0
                    then FCounter := FSteps - 1
                    else Dec( FCounter);
                  end;

                  MakeNextValue;
                end;
              end;

            seqClkHigh :

              begin
                if not NewTrig and FOldTrig
                then FInState := seqClkLow;
              end;
          end; // case

          FOldTrig := NewTrig;
        end;
      end;

      FInvert  := SignalToLogic( FInputs[ i_inv]);
      OutVal   := FOutValue;
      FClear   := SignalToLogic( FInputs[ i_clr2]);
      FRandom  := SignalToLogic( FInputs[ i_rnd2]);
      FRandom3 := SignalToLogic( FInputs[ i_rnd3]);

      if not FOldClear and FClear
      then FClrReq := True;

      if not FOldRandom and FRandom
      then FRndReq := True;

      if not FOldRandom3 and FRandom3
      then FRndReq3 := True;

      Mask   := 1 shl FCounter;
      Active := ( FActiveSteps and Mask) <> 0;
      DoTrig := SignalToLogic( FInputs[ i_trig]);
      FOutputs[ o_outgate   ] := LogicToSignal( Active);
      FOutputs[ o_outtrig   ] := LogicToSignal( Active and DoTrig);

      if Active
      then FOutputs[ o_out] := OutVal;

      FOldRes     := FReset;
      ResetFlag   := False;
      FOldClear   := FClear;
      FOldRandom  := FRandom;
      FOldRandom3 := FRandom3;
    end;


    procedure   TModVcps.GatherLights ( const aPrefix: string; const aCallback: TLightsHandler);
    var
      i : Integer;
      M : Word;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'   , FInputs [ i_trig   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'  , FInputs [ i_res    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'    , FInputs [ i_dir    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'inv'    , FInputs [ i_inv    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigout', FOutputs[ o_outtrig]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gateout', FOutputs[ o_outgate]));

      M := FActiveSteps;

      for i := 0 to i_step16 - i_step1 - 1
      do begin
        if ( i = FCounter) and (( M and 1) <> 0)
        then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i + 1], AppLocale), 1.0))
        else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i + 1], AppLocale), 0.0));

        M := M shr 1;
      end;
    end;


    procedure   TModVcps.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler);
    begin
      if FClrReq
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'clr', 16));
        FClrReq := False;
      end;

      if FRndReq
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'rnd', 1));
        FRndReq := False;
      end;

      if FRndReq3
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'rndone', 16));
        FRndReq3 := False;
      end;
    end;


{ ========
  TModSeq32 = class( TMod)
  strict private
  const
    o_out1        =   0;
    o_out2        =   1;
    o_out3        =   2;
    o_trigout1    =   3;
    o_trigout2    =   4;
    o_trigout3    =   5;
    o_div161      =   6;
    o_div162      =   7;
    o_div163      =   8;
  const
    SIZE      = 32;
    CHANNELS  =  3;
    FULL_SIZE = CHANNELS * SIZE;
    MAX_COUNT = 16;
  type
    TS32PlayMode = ( s32_3x32, s32_1x96);
    TS32LinkMode = ( s32_lnk , s32_ulnk);
  private
    FNotes        : array[ 0 .. FULL_SIZE - 1] of TSignal;
    FDurations    : array[ 0 .. FULL_SIZE - 1] of Integer;
    FSkips        : array[ 0 .. FULL_SIZE - 1] of Boolean;
    FStops        : array[ 0 .. FULL_SIZE - 1] of Boolean;
    FOldClock1    : Boolean;
    FOldClock2    : Boolean;
    FOldClock3    : Boolean;
    FOldReset     : Boolean;
    FCounter1     : Integer;
    FCounter2     : Integer;
    FCounter3     : Integer;
    FCounterDiv1  : Integer;
    FCounterDiv2  : Integer;
    FCounterDiv3  : Integer;
    FStep1        : Integer;
    FStep2        : Integer;
    FStep3        : Integer;
    FDisplayStep1 : Integer;
    FDisplayStep2 : Integer;
    FDisplayStep3 : Integer;
    FPlayMode     : TS32PlayMode;
    FLinkMode     : TS32LinkMode;
    FClkCount1    : Integer;
    FClkCount2    : Integer;
    FClkCount3    : Integer;
    FResCount     : Integer;
    FReverse      : Boolean;
    FInverse      : Boolean;
  private
    property    PlayMode : TS32PlayMode read FPlayMode write SetPlayMode;
    property    LinkMode : TS32LinkMode read FLinkMode write SetLinkMode;
  private
}

    procedure   TModSeq32.SetPlayMode( aValue: TS32PlayMode);
    begin
      if aValue <> FPlayMode
      then FPlayMode := aValue;
    end;


    procedure   TModSeq32.SetLinkMode( aValue: TS32LinkMode);
    begin
      if aValue <> FLinkMode
      then FLinkMode := aValue;
    end;


    procedure   TModSeq32.NextStep( Which: Integer);
    var
      StepDir : Integer;
    const
      Lookup : array[ 0 .. FULL_SIZE - 1] of Integer =
      (
        // Maps step index to display position index
        // also see GatherStringData

         0,  3,  6,  9, 12, 15, 18, 21,
        24, 27, 30, 33, 36, 39, 42, 45,
        48, 51, 54, 57, 60, 63, 66, 69,
        72, 75, 78, 81, 84, 87, 90, 93,
         1,  4,  7, 10, 13, 16, 19, 22,
        25, 28, 31, 34, 37, 40, 43, 46,
        49, 52, 55, 58, 61, 64, 67, 70,
        73, 76, 79, 82, 85, 88, 91, 94,
         2,  5,  8, 11, 14, 17, 20, 23,
        26, 29, 32, 35, 38, 41, 44, 47,
        50, 53, 56, 59, 62, 65, 68, 71,
        74, 77, 80, 83, 86, 89, 92, 95
      );
    begin
      if FReverse
      then StepDir := -1
      else StepDir :=  1;

      case Which of

        1 :
          begin
            case PlayMode of
              s32_3x32 : FStep1 := MathIntMod( FStep1 + StepDir,      SIZE);
              else       FStep1 := MathIntMod( FStep1 + StepDir, FULL_SIZE);
            end;

            FDisplayStep1 := Lookup[( FStep1 + 0 * SIZE) mod FULL_SIZE];
            FCounter1     := 0;
            FDurations[( FStep1 + 0 * SIZE) mod FULL_SIZE] := Round( FInputs[ i_duration11 + ( 0 * SIZE + FStep1) mod FULL_SIZE]);

            if FInverse
            then FNotes[( FStep1 + 0 * SIZE) mod FULL_SIZE] := NoteNumberToUnits( 12 * ( Finputs[ i_octave11 + ( 0 * SIZE + FStep1) mod FULL_SIZE] - 2) - FInputs[ i_note11 + ( 0 * SIZE + FStep1) mod FULL_SIZE])
            else FNotes[( FStep1 + 0 * SIZE) mod FULL_SIZE] := NoteNumberToUnits( 12 * ( Finputs[ i_octave11 + ( 0 * SIZE + FStep1) mod FULL_SIZE] - 2) + FInputs[ i_note11 + ( 0 * SIZE + FStep1) mod FULL_SIZE]);
          end;

        2 :
          begin
            case PlayMode of
              s32_3x32 : FStep2 := MathIntMod( FStep2 + StepDir,      SIZE);
              else       FStep2 := MathIntMod( FStep2 + StepDir, FULL_SIZE);
            end;

            FDisplayStep2 := Lookup[( FStep2 + 1 * SIZE) mod FULL_SIZE];
            FCounter2     := 0;
            FDurations[( FStep2 + 1 * SIZE) mod FULL_SIZE] := Round( FInputs[ i_duration11 + ( 1 * SIZE + FStep2) mod FULL_SIZE]);

            if FInverse
            then FNotes[( FStep2 + 1 * SIZE) mod FULL_SIZE] := NoteNumberToUnits( 12 * ( Finputs[ i_octave11 + ( 1 * SIZE + FStep2) mod FULL_SIZE] - 2) - FInputs[ i_note11 + ( 1 * SIZE + FStep2) mod FULL_SIZE])
            else FNotes[( FStep2 + 1 * SIZE) mod FULL_SIZE] := NoteNumberToUnits( 12 * ( Finputs[ i_octave11 + ( 1 * SIZE + FStep2) mod FULL_SIZE] - 2) + FInputs[ i_note11 + ( 1 * SIZE + FStep2) mod FULL_SIZE]);
          end;

        3 :
          begin
            case PlayMode of
              s32_3x32 : FStep3 := MathIntMod( FStep3 + StepDir,      SIZE);
              else       FStep3 := MathIntMod( FStep3 + StepDir, FULL_SIZE);
            end;

            FDisplayStep3 := Lookup[( FStep3 + 2 * SIZE) mod FULL_SIZE];
            FCounter3     := 0;
            FDurations[( FStep3 + 2 * SIZE) mod FULL_SIZE] := Round( FInputs[ i_duration11 + ( 2 * SIZE + FStep3) mod FULL_SIZE]);

            if FInverse
            then FNotes[( FStep3 + 2 * SIZE) mod FULL_SIZE] := NoteNumberToUnits( 12 * ( Finputs[ i_octave11 + ( 2 * SIZE + FStep3) mod FULL_SIZE] - 2) - FInputs[ i_note11 + ( 2 * SIZE + FStep3) mod FULL_SIZE])
            else FNotes[( FStep3 + 2 * SIZE) mod FULL_SIZE] := NoteNumberToUnits( 12 * ( Finputs[ i_octave11 + ( 2 * SIZE + FStep3) mod FULL_SIZE] - 2) + FInputs[ i_note11 + ( 2 * SIZE + FStep3) mod FULL_SIZE]);
          end;

      end;
    end;


    procedure   TModSeq32.NextCount( Which: Integer);
    begin
      case Which of

        1 :
          begin
            FCounter1    := ( FCounter1    + 1) mod MAX_COUNT;
            FCounterDiv1 := ( FCounterDiv1 + 1) mod MAX_COUNT;

            if   ( FCounter1 = 0)
            or   (( LinkMode = s32_free) and ( FCounter1 > FDurations[( FStep1 + 0 * SIZE) mod FULL_SIZE]))
            then NextStep( Which);
          end;

        2 :
          begin
            FCounter2    := ( FCounter2    + 1) mod MAX_COUNT;
            FCounterDiv2 := ( FCounterDiv2 + 1) mod MAX_COUNT;

            if   ( FCounter2 = 0)
            or   (( LinkMode = s32_free) and ( FCounter2 > FDurations[( FStep2 + 1 * SIZE) mod FULL_SIZE]))
            then NextStep( Which);
          end;

        3 :
          begin
            FCounter3    := ( FCounter3    + 1) mod MAX_COUNT;
            FCounterDiv3 := ( FCounterDiv3 + 1) mod MAX_COUNT;

            if   ( FCounter3 = 0)
            or   (( LinkMode = s32_free) and ( FCounter3 > FDurations[( FStep3 + 2 * SIZE) mod FULL_SIZE]))
            then NextStep( Which);
          end;

      end;
    end;


    procedure   TModSeq32.ResetState;
    begin
      FStep1       := -1;
      FCounter1    := -1;
      FCounterDiv1 := -1;
      NextCount( 1);
      FStep2       := -1;
      FCounter2    := -1;
      FCounterDiv2 := -1;
      NextCount( 2);
      FStep3       := -1;
      FCounter3    := -1;
      FCounterDiv3 := -1;
      NextCount( 3);
    end;


//  public

    procedure   TModSeq32.CreateIO; // override;
    var
      i : Integer;
      j : Integer;
      p : Integer;
    begin
      FIsSlow := True;
      p := i_note11;

      for i := 1 to CHANNELS
      do begin
        for j := 1 to SIZE
        do begin
          AddInput( p, Format( 'note%d%d', [ i, j], AppLocale));
          Inc( p);
        end;
      end;

      for i := 1 to CHANNELS
      do begin
        for j := 1 to SIZE
        do begin
          AddInput( p, Format( 'octave%d%d', [ i, j], AppLocale));
          Inc( p);
        end;
      end;

      for i := 1 to CHANNELS
      do begin
        for j := 1 to SIZE
        do begin
          AddInput( p, Format( 'duration%d%d', [ i, j], AppLocale));
          Inc( p);
        end;
      end;

      AddInput( i_clk1        , 'clk1'       );
      AddInput( i_clk2        , 'clk2'       );
      AddInput( i_clk3        , 'clk3'       );
      AddInput( i_res         , 'res'        );
      AddInput( i_active      , 'active'     );
      AddInput( i_playmode    , 'playmode'   );
      AddInput ( i_linkmode   , 'linkmode'   );
      AddInput ( i_gatemode   , 'gatemode'   );
      AddInput ( i_playmodemod, 'playmodemod');
      AddInput ( i_linkmodemod, 'linkmodemod');
      AddInput ( i_gatemodemod, 'gatemodemod');
      AddInput ( i_reverse    , 'reverse'    );
      AddInput ( i_inverse    , 'inverse'    );

      AddOutput( o_out1       , 'out1'       );
      AddOutput( o_out2       , 'out2'       );
      AddOutput( o_out3       , 'out3'       );
      AddOutput( o_trigout1   , 'trigout1'   );
      AddOutput( o_trigout2   , 'trigout2'   );
      AddOutput( o_trigout3   , 'trigout3'   );
      AddOutput( o_div161     , 'div161'     );
      AddOutput( o_div162     , 'div162'     );
      AddOutput( o_div163     , 'div163'     );
    end;


    procedure   TModSeq32.SetDefaults; // override;
    begin
      inherited;
      FInputs[ i_active] := 1.0;         // Sequencer active when active input not connected
    end;


    procedure   TModSeq32.DoSlowTick; // override;
    var
      NewReset    : Boolean;
      NewClock1   : Boolean;
      NewClock2   : Boolean;
      NewClock3   : Boolean;
      MustOutput1 : Boolean;
      MustOutput2 : Boolean;
      MustOutput3 : Boolean;
      GateMode    : Integer; // [ T, G, R]
    begin
      MustOutput1 := False;
      MustOutput2 := False;
      MustOutput3 := False;
      PlayMode    := TS32PlayMode(( Round( FInputs[ i_playmode]) + Integer( SignalToLogic( FInputs[ i_playmodemod])) and 1));
      LinkMode    := TS32LinkMode(( Round( FInputs[ i_linkmode]) + Integer( SignalToLogic( FInputs[ i_linkmodemod])) and 1));
      GateMode    := MathIntMod( Round( FInputs[ i_gatemode]) + Round( 3 * FInputs[ i_gatemodemod]), 3);
      FReverse    := SignalToLogic( FInputs[ i_reverse ]);
      FInverse    := SignalToLogic( FInputs[ i_inverse ]);
      NewClock1   := SignalToLogic( FInputs[ i_clk1    ]);
      NewClock2   := SignalToLogic( FInputs[ i_clk2    ]);
      NewClock3   := SignalToLogic( FInputs[ i_clk3    ]);

      if   SignalToLogic( FInputs[ i_active])
      or   ResetFlag
      then begin
        NewReset := SignalToLogic( FInputs[ i_res]);

        if   ( NewReset and not FOldReset)
        or   ResetFlag
        then begin
          ResetFlag   := False;
          MustOutput1 := True;
          MustOutput2 := True;
          MustOutput3 := True;
          FResCount   := LedFlashTime( IsSpedUp);
          ResetState;
        end
        else begin
          if NewClock1 and not FOldClock1
          then begin
            MustOutput1 := True;
            FClkCount1  := LedFlashTime( IsSpedUp);
            NextCount( 1);
          end;

          if NewClock2 and not FOldClock2
          then begin
            MustOutput2 := True;
            FClkCount2  := LedFlashTime( IsSpedUp);
            NextCount( 2);
          end;

          if NewClock3 and not FOldClock3
          then begin
            MustOutput3 := True;
            FClkCount3  := LedFlashTime( IsSpedUp);
            NextCount( 3);
          end;
        end;

        if MustOutput1
        then begin
          FOutputs[ o_div161  ] := LogicToSignal( FCounterDiv1 < 8);
          FOutputs[ o_out1    ] := FNotes[( FStep1 + 0 * SIZE) mod FULL_SIZE];
        end;

        if MustOutput2
        then begin
          FOutputs[ o_div162  ] := LogicToSignal( FCounterDiv2 < 8);
          FOutputs[ o_out2    ] := FNotes[( FStep2 + 1 * SIZE) mod FULL_SIZE];
        end;

        if MustOutput3
        then begin
          FOutputs[ o_div163  ] := LogicToSignal( FCounterDiv3 < 8);
          FOutputs[ o_out3    ] := FNotes[( FStep3 + 2 * SIZE) mod FULL_SIZE];
        end;

        case GateMode of

          0 : // T - one clock out for the first count only
            begin
              FOutputs[ o_trigout1] := LogicToSignal(( FCounter1 = 0) and NewClock1);
              FOutputs[ o_trigout2] := LogicToSignal(( FCounter2 = 0) and NewClock2);
              FOutputs[ o_trigout3] := LogicToSignal(( FCounter3 = 0) and NewClock3);
            end;

          1 : // G - one clock out for the full time the step is active
            begin
              FOutputs[ o_trigout1] := LogicToSignal( FCounter1 < FDurations[( FStep1 + 0 * SIZE) mod FULL_SIZE]);
              FOutputs[ o_trigout2] := LogicToSignal( FCounter2 < FDurations[( FStep2 + 1 * SIZE) mod FULL_SIZE]);
              FOutputs[ o_trigout3] := LogicToSignal( FCounter3 < FDurations[( FStep3 + 2 * SIZE) mod FULL_SIZE]);
            end;

          2 : // R - repeated clock outs for the time the step is active
            begin
              FOutputs[ o_trigout1] := LogicToSignal(( FCounter1 < FDurations[( FStep1 + 0 * SIZE) mod FULL_SIZE]) and NewClock1);
              FOutputs[ o_trigout2] := LogicToSignal(( FCounter2 < FDurations[( FStep2 + 1 * SIZE) mod FULL_SIZE]) and NewClock2);
              FOutputs[ o_trigout3] := LogicToSignal(( FCounter3 < FDurations[( FStep3 + 2 * SIZE) mod FULL_SIZE]) and NewClock3);
            end;

        end;

        FOldReset := NewReset;
      end;

      FOldClock1 := NewClock1;
      FOldClock2 := NewClock2;
      FOldClock3 := NewClock3;
      DecToZero( FClkCount1);
      DecToZero( FClkCount2);
      DecToZero( FClkCount3);
      DecToZero( FResCount);
    end;


    procedure   TModSeq32.GatherLights ( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'clk1'   , LogicToSignal( FClkCount1 > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clk2'   , LogicToSignal( FClkCount2 > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clk3'   , LogicToSignal( FClkCount3 > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'    , LogicToSignal( FResCount  > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reverse', LogicToSignal( FReverse      )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'inverse', LogicToSignal( FInverse      )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'act'    , FInputs [ i_active  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'div161' , FOutputs[ o_div161  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'div162' , FOutputs[ o_div162  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'div163' , FOutputs[ o_div163  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig1'  , FOutputs[ o_trigout1]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig2'  , FOutputs[ o_trigout2]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig3'  , FOutputs[ o_trigout3]));
    end;


    procedure   TModSeq32.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      i  : Integer;
      p  : Integer;
      m  : Integer;
      L1 : string;
      L2 : string;
      L3 : string;
      L4 : string;
    begin
      L1 := '';
      L2 := '';
      L3 := '';
      L4 := '';

      for i := 0 to FULL_SIZE - 1
      do begin
        p := i div 24;
        m := p mod  4;

        case m of

          0 :
            begin
              if i in [ FDisplayStep1, FDisplayStep2, FDisplayStep3]
              then L1 := L1 + 'x'
              else L1 := L1 + ' ';

              if Length( L1) in [ 3, 7, 11, 15, 19, 23, 27]
              then L1 := L1 + '|';
            end;

          1 :
            begin
              if i in [ FDisplayStep1, FDisplayStep2, FDisplayStep3]
              then L2 := L2 + 'x'
              else L2 := L2 + ' ';

              if Length( L2) in [ 3, 7, 11, 15, 19, 23, 27]
              then L2 := L2 + '|';
            end;

          2 :
            begin
              if i in [ FDisplayStep1, FDisplayStep2, FDisplayStep3]
              then L3 := L3 + 'x'
              else L3 := L3 + ' ';

              if Length( L3) in [ 3, 7, 11, 15, 19, 23, 27]
              then L3 := L3 + '|';
            end;

          3 :
            begin
              if i in [ FDisplayStep1, FDisplayStep2, FDisplayStep3]
              then L4 := L4 + 'x'
              else L4 := L4 + ' ';

              if Length( L4) in [ 3, 7, 11, 15, 19, 23, 27]
              then L4 := L4 + '|';
            end;

        end;

        aCallback( Self, MakeInfo( aPrefix, Name, 'lights1', L1));
        aCallback( Self, MakeInfo( aPrefix, Name, 'lights2', L2));
        aCallback( Self, MakeInfo( aPrefix, Name, 'lights3', L3));
        aCallback( Self, MakeInfo( aPrefix, Name, 'lights4', L4));
      end;

    end;


{ ========
  TModTextSequencer = class( TMod)
  private
    FOutValue   : TSignal;
    FStepCount  : Integer;
    FStepValues : TSignalArray;
    FCounter    : Integer;
    FInState    : TSeqInState;
    FOldRes     : Boolean;
  public
}

    procedure   TModTextSequencer.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig   , 'trig'   );
      AddInput ( i_reset  , 'reset'  );
      AddInput ( i_dir    , 'dir'    );
      AddInput ( i_mode   , 'mode'   );
      AddInput ( i_chain  , 'chain'  );
      AddOutput( o_out    , 'out'    );
      AddOutput( o_gateout, 'gateout');
    end;


    procedure   TModTextSequencer.SetDefaults; // override;
    begin
      FOutValue            := 0;
      FInputs [ i_dir    ] := 1; // Make seq run forward by default.
      FInputs [ i_chain  ] := 0; // Clear chain input
      FOutputs[ o_gateout] := LogicToSignal( False);
    end;


    procedure   TModTextSequencer.DoSlowTick; // override;
    var
      CurrentValue : TSIgnal;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FInState  := seqClkLow;
        FCounter  := 0;
        FOutValue := 0;
      end;

      if not FOldRes and SignalToLogic( FInputs[ i_reset])
      then begin
        FInState := seqClkLow;
        FCounter := 0;
      end;

      case FInState of

        seqClkLow :

          begin
            if SignalToLogic( FInputs[ i_trig])
            then begin
              FInState := seqClkHigh;

              if FStepCount > 0
              then begin
                if FInputs[ i_mode] = 0
                then begin
                  if SignalToLogic( FInputs[ i_dir])
                  then FCounter := ( FCounter + 1) mod FStepCount
                  else begin
                    if FCounter <= 0
                    then FCounter := FStepCount - 1
                    else Dec( FCounter);
                  end;
                end
                else FCounter := Random( FStepCount);
              end;

              CurrentValue := FStepValues[ FCounter];

              if not AlmostEqual( CurrentValue, NO_NOTE, DELTA)
              then begin
                FOutValue := CurrentValue - MiddleNote * NOTE_SCALING_REC;
                FOutputs[ o_gateout] := LogicToSignal( True);
              end;
            end;
          end;

        seqClkHigh :

          begin
            if FInputs[ i_trig] <= 0
            then begin
              FInState := seqClkLow;
              FOutputs[ o_gateout] := LogicToSignal( False);
            end;
          end;

      end;

      FOutputs[ o_out] := FOutValue + FInputs[ i_chain];
      FOldRes := SignalToLogic( FInputs[ i_reset]);
    end;


    procedure   TModTextSequencer.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig' , FInputs[ i_trig    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset', FInputs[ i_reset   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'  , FInputs[ i_dir     ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step' , FCounter + 1        ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gate' , FOutputs[ o_gateout]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'steps', FStepCount          ));
    end;


    procedure   TModTextSequencer.SetStepValues( const aValues: TSignalArray);
    var
      i       : Integer;
      aLength : Integer;
    begin
      FStepCount := 0;
      aLength := Length( aValues);
      SetLength( FStepValues, aLength);

      for i := 0 to aLength - 1
      do FStepValues[ i] := aValues[ i];

      FStepCount := aLength;
    end;


{ ========
  TModSeqStep = class( TMod)
  private
    FTime    : TSignal;
    FState   : TSeqStepState;
    FOldRes  : Boolean;
  public
}

    procedure   TModSeqStep.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_triginfwd , 'triginfwd' );
      AddInput ( i_triginbwd , 'triginbwd' );
      AddInput ( i_chain     , 'chain'     );
      AddInput ( i_reset     , 'reset'     );
      AddInput ( i_dir       , 'dir'       );
      AddInput ( i_note      , 'note'      );
      AddInput ( i_time      , 'time'      );
      AddOutput( o_trigoutfwd, 'trigoutfwd');
      AddOutput( o_trigoutbwd, 'trigoutbwd');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModSeqStep.SetDefaults; // override;
    begin
      FInputs[ i_dir] := 1; // Make seq run forward by default.
      FDezipperMap := [
        i_time
      ];
    end;


    procedure   TModSeqStep.DoSlowTick; // override;
    begin
      if ResetFlag
      then begin
        ResetFlag := False;
        FState := sssIdle;
        FOutputs[ o_trigoutfwd] := LogicToSignal( False);                        // Do not trigger next in forward  direction
        FOutputs[ o_trigoutbwd] := LogicToSignal( False);                        // Do not trigger next in backward direction
      end;

      if not FOldRes and SignalToLogic( FInputs[ i_reset])
      then begin                                                                 // Reset from inactive to active
        FState := sssIdle;
        FOutputs[ o_trigoutfwd] := LogicToSignal( False);                        // Do not trigger next in forward  direction
        FOutputs[ o_trigoutbwd] := LogicToSignal( False);                        // Do not trigger next in backward direction
      end;

      case FState of

        sssIdle :

          begin
            if SignalToLogic( FInputs[ i_dir])
            then begin                                                           // running Forwards
              if SignalToLogic( FInputs[ i_triginfwd])                           // Forward trig in
              then begin
                FState := sssActive;
                FTime  := SlowTimeToSampleCount( FInputs[ i_time]);              // Duration
                FOutputs[ o_trigoutfwd] := LogicToSignal( False);                // Do not trigger next in forward  direction
                FOutputs[ o_trigoutbwd] := LogicToSignal( False);                // Do not trigger next in backward direction
              end;
            end
            else begin                                                           // running Backwards
              if SignalToLogic( FInputs[ i_triginbwd])                           // Backward trig in
              then begin
                FState := sssActive;
                FTime  := SlowTimeToSampleCount( FInputs[ i_time]);              // Duration
                FOutputs[ o_trigoutfwd] := LogicToSignal( False);                // Do not trigger next in forward  direction
                FOutputs[ o_trigoutbwd] := LogicToSignal( False);                // Do not trigger next in backward direction
              end;
            end;
          end;

        sssActive :

          begin
            FTime := FTime - 1;

            if FTime <= 0
            then begin
              FState := sssWaiting;
              if SignalToLogic( FInputs[ i_dir])                                 // running Forwards
              then FOutputs[ o_trigoutfwd] := LogicToSignal( True)               // Trigger next in forward  direction
              else FOutputs[ o_trigoutbwd] := LogicToSignal( True);              // Trigger next in backward direction
            end;
          end;

        sssWaiting :

          begin
            if SignalToLogic( FInputs[ i_dir])
            then begin                                                           // running Forwards
              if not SignalToLogic( FInputs[ i_triginfwd])                       // Forward trig in fell off
              then begin
                FState := sssIdle;
              end;
            end
            else begin                                                           // running Backwards
              if not SignalToLogic( FInputs[ i_triginbwd])                       // Backward trig in fell off
              then begin
                FState := sssIdle;
              end;
            end;
          end;

      end;

      if FState = sssActive
      then FOutputs[ o_out] := FInputs[ i_note] - MiddleNote * NOTE_SCALING_REC  // Own value to output
      else FOutputs[ o_out] := FInputs[ i_chain];                                // Chained value to output (was compensated there already)

      FOldRes := SignalToLogic( FInputs[ i_reset]);
    end;


    procedure   TModSeqStep.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigfwd', FInputs[ i_triginfwd]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigbwd', FInputs[ i_triginbwd]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'  , FInputs[ i_reset    ]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'    , FInputs[ i_dir      ]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active' , IfThen( FState = sssActive, 1.0, 0.0)));
    end;


{ ========
  TModSeqClockStep = class( TMod)
  private
    FCount      : Integer;
    FState      : TSeqStepState;
    FOldRes     : Boolean;
    FOldClock   : Boolean;
    FTrigInFwd  : Boolean;
    FTrigInBwd  : Boolean;
    FDir        : Boolean;
    FTrigOutFwd : Boolean;
    FTrigOutBwd : Boolean;
  public
}

    procedure   TModSeqClockStep.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_triginfwd , 'triginfwd' );
      AddInput ( i_triginbwd , 'triginbwd' );
      AddInput ( i_chain     , 'chain'     );
      AddInput ( i_reset     , 'reset'     );
      AddInput ( i_dir       , 'dir'       );
      AddInput ( i_clock     , 'clock'     );
      AddInput ( i_note      , 'note'      );
      AddInput ( i_counts    , 'counts'    );
      AddOutput( o_trigoutfwd, 'trigoutfwd');
      AddOutput( o_trigoutbwd, 'trigoutbwd');
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModSeqClockStep.SetDefaults; // override;
    begin
      FInputs[ i_dir] := 1; // Make seq run forward by default.
    end;


    procedure   TModSeqClockStep.DoSlowTick; // override;
    begin
      if ResetFlag
      then begin
        ResetFlag   := False;
        FState      := sssIdle;
        FTrigOutFwd := False;
        FTrigOutBwd := False;
        FOutputs[ o_trigoutfwd] := LogicToSignal( False);                       // Do not trigger next in forward  direction
        FOutputs[ o_trigoutbwd] := LogicToSignal( False);                       // Do not trigger next in backward direction
        FOutputs[ o_out] := FInputs[ i_chain];                                  // Chained value to output
      end;

      if not FOldRes and SignalToLogic( FInputs[ i_reset])
      then begin                                                                // Reset from inactive to active - reset is asynchroneous
        FState      := sssIdle;
        FTrigOutFwd := False;
        FTrigOutBwd := False;
        FOutputs[ o_trigoutfwd] := LogicToSignal( False);                       // Do not trigger next in forward  direction
        FOutputs[ o_trigoutbwd] := LogicToSignal( False);                       // Do not trigger next in backward direction
        FOutputs[ o_out] := FInputs[ i_chain];                                  // Chained value to output
      end;

      FTrigInFwd := SignalToLogic( FInputs[ i_triginfwd]);
      FTrigInBwd := SignalToLogic( FInputs[ i_triginbwd]);
      FDir       := SignalToLogic( FInputs[ i_dir]);

      case FState of

        sssIdle :

          begin
            if ( FDir and FTrigInFwd) or (( not FDir) and FTrigInBwd)
            then begin                                                          // Trigger seen
              FState      := sssActive;
              FCount      := Ceil( FInputs[ i_counts]);                         // Duration
              FTrigOutFwd := False;                                             // Do not trigger next in forward  direction
              FTrigOutBwd := False;                                             // Do not trigger next in backward direction
            end;
          end;

        sssWaiting :

          begin
            if ( FDir and ( not FTrigInFwd)) or (( not FDir) and ( not FTrigInBwd))
            then begin
              FState      := sssIdle;                                           // Trigger fell off - go idle
              FTrigOutFwd := False;                                             // Do not trigger next in forward  direction
              FTrigOutBwd := False;                                             // Do not trigger next in backward direction
            end;
          end;
      end;

      if not FOldClock and SignalToLogic( FInputs[ i_clock])
      then begin                                                                // Clock from inactive to active - state changes should only occur here
        case FState of
          sssActive :
            begin
              FCount := FCount - 1;

              if FCount <= 1
              then begin
                FCount := 0;
                FState := sssWaiting;

                if FDir                                                         // running Forwards
                then begin
                  FTrigOutFwd := True;                                          // Trigger next in forward  direction
                  FTrigOutBwd := False;                                         // No Trigger next in backward  direction
                end
                else begin
                  FTrigOutFwd := False;                                         // No Trigger next in forward direction
                  FTrigOutBwd := True;                                          // Trigger next in backward direction
                end;
              end;
            end;
        end;
      end;

      FOutputs[ o_trigoutfwd] := LogicToSignal( FTrigOutFwd);                   // Clock out internal state to ouptputs
      FOutputs[ o_trigoutbwd] := LogicToSignal( FTrigOutBwd);

      if FState = sssActive
      then FOutputs[ o_out] := FInputs[ i_note] - MiddleNote * NOTE_SCALING_REC // Own     value to output
      else FOutputs[ o_out] := FInputs[ i_chain];                               // Chained value to output (was compensated there already)

      FOldRes   := SignalToLogic( FInputs[ i_triginbwd]);
      FOldClock := SignalToLogic( FInputs[ i_clock    ]);
    end;


    procedure   TModSeqClockStep.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigfwd', FInputs[ i_triginfwd]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trigbwd', FInputs[ i_triginbwd]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'  , FInputs[ i_reset    ]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'    , FInputs[ i_dir      ]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clock'  , FInputs[ i_clock    ]                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'count'  , FCount                               ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active' , IfThen( FState = sssActive, 1.0, 0.0)));
    end;


{ ========
  TModPattern = class( TMod)
  private
    FPosition : Integer;
    FCounter  : Integer;
    FOldTrig  : Boolean;
    FOldRes   : Boolean;
    FOutVal   : Boolean;
  public
}

    procedure   TModPattern.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_res   , 'res'   );
      AddInput ( i_trig  , 'trig'  );
      AddInput ( i_step1 , 'step1' );
      AddInput ( i_step2 , 'step2' );
      AddInput ( i_step3 , 'step3' );
      AddInput ( i_step4 , 'step4' );
      AddInput ( i_step5 , 'step5' );
      AddInput ( i_step6 , 'step6' );
      AddInput ( i_step7 , 'step7' );
      AddInput ( i_step8 , 'step8' );
      AddInput ( i_mode1 , 'mode1' );
      AddInput ( i_mode2 , 'mode2' );
      AddInput ( i_mode3 , 'mode3' );
      AddInput ( i_mode4 , 'mode4' );
      AddInput ( i_mode5 , 'mode5' );
      AddInput ( i_mode6 , 'mode6' );
      AddInput ( i_mode7 , 'mode7' );
      AddInput ( i_mode8 , 'mode8' );
      AddInput ( i_active, 'active');
      AddOutput( o_out   , 'out'   );
      AddOutput( o_out1  , 'out1'  );
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_out3  , 'out3'  );
      AddOutput( o_out4  , 'out4'  );
      AddOutput( o_out5  , 'out5'  );
      AddOutput( o_out6  , 'out6'  );
      AddOutput( o_out7  , 'out7'  );
      AddOutput( o_out8  , 'out8'  );
    end;


    procedure   TModPattern.SetDefaults; // override;
    begin
      FInputs[ i_active] := 1.0;
    end;


    procedure   TModPattern.DoSlowTick; // override;
    var
      Trig   : Boolean;
      Res    : Boolean;
      Steps  : Integer;
      Mode   : Integer;
      Active : Boolean;
      i      : Integer;
      p      : Integer;
    begin
      Res    := SignalToLogic( FInputs[ i_res   ]);
      Trig   := SignalToLogic( FInputs[ i_trig  ]);
      Active := SignalToLogic( FInputs[ i_active]);
      Steps  := Round( FInputs[ i_step1 + FPosition]);

      if ( not FOldRes and Res) or ResetFlag
      then FPosition := 0;

      if not FOldTrig and Trig and Active
      then begin
        if FCounter > 0
        then Dec( FCounter);

        FOutVal := ( FCounter = 0) and ( Steps > 0);

        if FCounter <= 0
        then begin
          p := FPosition;

          for i := i_step1 to i_last
          do begin
            p    := ( p + 1) mod ( i_last - i_step1 + 1);
            Mode := Round( FInputs[ i_mode1 + p]);

            case Mode of
              sm_rnd1          : Steps := Random   ( Round( FInputs[ i_step1 + p]) + 1);
              sm_rnd2          : Steps := RandomDiv( Round( FInputs[ i_step1 + p]) , 2);
              sm_rnd3          : Steps := RandomDiv( Round( FInputs[ i_step1 + p]) , 3);
              sm_skip          : Steps := 0;
              else { sm_normal } Steps := Round( FInputs[ i_step1 + p]);
            end;

            if Steps > 0
            then begin
              FPosition := p;
              Break;
            end;
          end;

          FCounter := Steps;
        end;
      end;

      FOutputs[ o_out] := LogicToSignal( FOutVal and Trig and Active);

      for i := 1 to 8
      do begin
        if i - 1 = FPosition
        then FOutputs[ o_out + i] := LogicToSignal( FOutVal and Trig and Active)
        else FOutputs[ o_out + i] := LogicToSignal( False);
      end;

      ResetFlag := False;
      FOldRes   := Res;
      FOldTrig  := Trig;
    end;


    procedure   TModPattern.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'  , FInputs [ i_trig  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'   , FInputs [ i_res   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'   , FOutputs[ o_out   ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active', FInputs [ i_active]));

      for i := 1 to 8
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i], AppLocale), LogicToSignal( SignalToLogic( FOutputs[ o_out]) and ( FPosition + 1 = i))));
    end;


{ ========
  TModSeqChord = class( TMod)
  private
}

    function    TModSeqChord.CalcChord( aDegree, aType, anInversion, aMode: Integer): TTetrad;
    const
      Shifts : array[ 0 .. 6] of Integer = (
        0, 2, 4, 5, 7, 9, 11
      );
    var
      i : Integer;
    begin
      // Determine base chord and inversion
      // Result := ScalesAndInversions[ aMode, aType, anInversion];
      // Transpose to the requested degree and return that.

      for i := Low( Result) to High( Result)
      do begin
        Result[ i] := ScalesAndInversions[ aMode, aType, anInversion, i];
        Result[ i] := Result[ i] + Shifts[ aDegree mod 7];
      end;
    end;


//  public

    procedure   TModSeqChord.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_active0  , 'active0'  ); 
      AddInput ( i_active1  , 'active1'  ); 
      AddInput ( i_active2  , 'active2'  ); 
      AddInput ( i_active3  , 'active3'  ); 
      AddInput ( i_gatein0  , 'gatein0'  ); 
      AddInput ( i_gatein1  , 'gatein1'  ); 
      AddInput ( i_gatein2  , 'gatein2'  ); 
      AddInput ( i_gatein3  , 'gatein3'  ); 
      AddInput ( i_notein0  , 'notein0'  ); 
      AddInput ( i_notein1  , 'notein1'  ); 
      AddInput ( i_notein2  , 'notein2'  ); 
      AddInput ( i_notein3  , 'notein3'  ); 
      AddInput ( i_degree0  , 'degree0'  ); 
      AddInput ( i_degree1  , 'degree1'  ); 
      AddInput ( i_degree2  , 'degree2'  ); 
      AddInput ( i_degree3  , 'degree3'  ); 
      AddInput ( i_type0    , 'type0'    ); 
      AddInput ( i_type1    , 'type1'    ); 
      AddInput ( i_type2    , 'type2'    ); 
      AddInput ( i_type3    , 'type3'    ); 
      AddInput ( i_inver0   , 'inver0'   ); 
      AddInput ( i_inver1   , 'inver1'   ); 
      AddInput ( i_inver2   , 'inver2'   ); 
      AddInput ( i_inver3   , 'inver3'   ); 
      AddInput ( i_octave0  , 'octave0'  ); 
      AddInput ( i_octave1  , 'octave1'  ); 
      AddInput ( i_octave2  , 'octave2'  ); 
      AddInput ( i_octave3  , 'octave3'  ); 
      AddInput ( i_transpose, 'transpose');
      AddInput ( i_gate     , 'gate'     );
      AddInput ( i_slew     , 'slew'     );
      AddInput ( i_reduce   , 'reduce'   );
      AddOutput( o_gateout0 , 'gateout0' );
      AddOutput( o_gateout1 , 'gateout1' );
      AddOutput( o_gateout2 , 'gateout2' );
      AddOutput( o_gateout3 , 'gateout3' );
      AddOutput( o_noteout0 , 'noteout0' );
      AddOutput( o_noteout1 , 'noteout1' );
      AddOutput( o_noteout2 , 'noteout2' );
      AddOutput( o_noteout3 , 'noteout3' );
    end;


    procedure   TModSeqChord.DoSlowTick; // override;
    var
      Slew       : TSignal;
      Out0       : TSignal;
      Out1       : TSignal;
      Out2       : TSignal;
      Out3       : TSignal;
      Gate0      : Boolean;
      Gate1      : Boolean;
      Gate2      : Boolean;
      Gate3      : Boolean;
      Degree     : Integer;
      ScaleType  : Integer;
      Inversion  : Integer;
      Octave     : Integer;
      Transpose  : Integer;
      Notes      : TTetrad;
      Gate       : Boolean;
      Active     : Boolean;
      Mode       : Integer;
    begin
      Slew      := SlewToCurve( FInputs[ i_slew]);
      Transpose := Round( UnitsToNoteNumber( FInputs[ i_transpose]));
      Octave    := 0;
      Out0      := FInputs[ i_notein0];  // When we do nothing here we pass the inputs
      Out1      := FInputs[ i_notein1];
      Out2      := FInputs[ i_notein2];
      Out3      := FInputs[ i_notein3];
      Gate0     := False;
      Gate1     := False;
      Gate2     := False;
      Gate3     := False;
      Active    := False;
      Mode      := Round( FInputs[ i_reduce]);

      if( SignalToLogic( FInputs[ i_active0]))
      then begin
        Active    := True;
        Degree    := Round( FInputs[ i_degree0 ]);
        ScaleType := Round( FInputs[ i_type0   ]);
        Inversion := Round( FInputs[ i_inver0  ]);
        Octave    := Round( FInputs[ i_octave0 ]) - 3;
        Notes     := CalcChord( Degree, ScaleType, Inversion, Mode);
      end
      else if( SignalToLogic( FInputs[ i_active1]))
      then begin
        Active    := True;
        Degree    := Round( FInputs[ i_degree1 ]);
        ScaleType := Round( FInputs[ i_type1   ]);
        Inversion := Round( FInputs[ i_inver1  ]);
        Octave    := Round( FInputs[ i_octave1 ]) - 3;
        Notes     := CalcChord( Degree, ScaleType, Inversion, Mode);
      end
      else if( SignalToLogic( FInputs[ i_active2]))
      then begin
        Active    := True;
        Degree    := Round( FInputs[ i_degree2 ]);
        ScaleType := Round( FInputs[ i_type2   ]);
        Inversion := Round( FInputs[ i_inver2  ]);
        Octave    := Round( FInputs[ i_octave2 ]) - 3;
        Notes     := CalcChord( Degree, ScaleType, Inversion, Mode);
      end
      else if( SignalToLogic( FInputs[ i_active3]))
      then begin
        Active    := True;
        Degree    := Round( FInputs[ i_degree3 ]);
        ScaleType := Round( FInputs[ i_type3   ]);
        Inversion := Round( FInputs[ i_inver3  ]);
        Octave    := Round( FInputs[ i_octave3 ]) - 3;
        Notes     := CalcChord( Degree, ScaleType, Inversion, Mode);
      end;

      if Active
      then begin
        Gate := SignalToLogic( FInputs[ i_gate]);
        Out0 := NoteNumberToUnits( Notes[ 0] + Transpose + 12 * Octave - MiddleNote); Gate0 := Gate;
        Out1 := NoteNumberToUnits( Notes[ 1] + Transpose + 12 * Octave - MiddleNote); Gate1 := Gate;
        Out2 := NoteNumberToUnits( Notes[ 2] + Transpose + 12 * Octave - MiddleNote); Gate2 := Gate;
        Out3 := NoteNumberToUnits( Notes[ 3] + Transpose + 12 * Octave - MiddleNote); Gate3 := Gate;
      end;

      FOutputs[ o_noteout0] := Normalize( Out0 + Slew * ( FOutputs[ o_noteout0] - Out0));
      FOutputs[ o_noteout1] := Normalize( Out1 + Slew * ( FOutputs[ o_noteout1] - Out1));
      FOutputs[ o_noteout2] := Normalize( Out2 + Slew * ( FOutputs[ o_noteout2] - Out2));
      FOutputs[ o_noteout3] := Normalize( Out3 + Slew * ( FOutputs[ o_noteout3] - Out3));
      FOutputs[ o_gateout0] := LogicToSignal( Gate0 or SignalToLogic( FInputs[ i_gatein0]));
      FOutputs[ o_gateout1] := LogicToSignal( Gate1 or SignalToLogic( FInputs[ i_gatein1]));
      FOutputs[ o_gateout2] := LogicToSignal( Gate2 or SignalToLogic( FInputs[ i_gatein2]));
      FOutputs[ o_gateout3] := LogicToSignal( Gate3 or SignalToLogic( FInputs[ i_gatein3]));
    end;


    procedure   TModSeqChord.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'gate'       , FInputs [ i_gate    ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active0'    , FInputs [ i_active0 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active1'    , FInputs [ i_active1 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active2'    , FInputs [ i_active2 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active3'    , FInputs [ i_active3 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gatein0'    , FInputs [ i_gatein0 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gatein0'    , FInputs [ i_gatein0 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gatein1'    , FInputs [ i_gatein1 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gatein2'    , FInputs [ i_gatein2 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gatein3'    , FInputs [ i_gatein3 ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gateout0'   , FOutputs[ o_gateout0]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gateout1'   , FOutputs[ o_gateout1]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gateout2'   , FOutputs[ o_gateout2]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gateout3'   , FOutputs[ o_gateout3]));
    end;


{ ========
  TModProbSequencer = class( TMod)
  private
    FCurrentNote  : TSignal;
    FCurrNoteNr   : Integer;
    FOldTrig      : Boolean;
    FScale        : Integer;
    FScaleChanged : Boolean;
    FNotes        : TScaleNoteSet;
    FPrevProb     : TSIgnal;
    FProb         : array[ -1 .. 127] of TSignal;
  private
}

    function    TModProbSequencer.FixProb( aProb: TSignal): TSignal;
    begin
      if ( aProb < 0.0) or ( aProb > 0.99)
      then Result := 0.0
      else Result := aProb;
    end;


    function    TModProbSequencer.Search( aValue: TSignal): Integer;
    var
      P : Integer;
      L : Integer;
      H : Integer;
    begin
      L      := 0;
      H      := High( FProb) - 1;
      Result := 0;

      while L <= H
      do begin
        P := ( L + H) div 2;

        if aValue < FProb[ P]
        then H := P - 1
        else if ( P = H) or ( aValue < FProb[ P + 1])
        then begin
          Result := P + 1;
          Break;
        end
        else L := P + 1;
      end;
    end;


//  public

    procedure   TModProbSequencer.CreateIO; // override;
    begin
      FISSlow := True;
      AddInput ( i_note0       , 'note0'       );
      AddInput ( i_note1       , 'note1'       );
      AddInput ( i_note2       , 'note2'       );
      AddInput ( i_note3       , 'note3'       );
      AddInput ( i_note4       , 'note4'       );
      AddInput ( i_note5       , 'note5'       );
      AddInput ( i_note6       , 'note6'       );
      AddInput ( i_note7       , 'note7'       );
      AddInput ( i_note8       , 'note8'       );
      AddInput ( i_note9       , 'note9'       );
      AddInput ( i_note10      , 'note10'      );
      AddInput ( i_note11      , 'note11'      );
      AddInput ( i_prob0       , 'prob0'       );
      AddInput ( i_prob1       , 'prob1'       );
      AddInput ( i_prob2       , 'prob2'       );
      AddInput ( i_prob3       , 'prob3'       );
      AddInput ( i_prob4       , 'prob4'       );
      AddInput ( i_prob5       , 'prob5'       );
      AddInput ( i_prob6       , 'prob6'       );
      AddInput ( i_prob7       , 'prob7'       );
      AddInput ( i_prob8       , 'prob8'       );
      AddInput ( i_prob9       , 'prob9'       );
      AddInput ( i_prob10      , 'prob10'      );
      AddInput ( i_prob11      , 'prob11'      );
      AddInput ( i_ctrl0       , 'ctrl0'       );
      AddInput ( i_ctrl1       , 'ctrl1'       );
      AddInput ( i_ctrl2       , 'ctrl2'       );
      AddInput ( i_ctrl3       , 'ctrl3'       );
      AddInput ( i_ctrl4       , 'ctrl4'       );
      AddInput ( i_ctrl5       , 'ctrl5'       );
      AddInput ( i_ctrl6       , 'ctrl6'       );
      AddInput ( i_ctrl7       , 'ctrl7'       );
      AddInput ( i_ctrl8       , 'ctrl8'       );
      AddInput ( i_ctrl9       , 'ctrl9'       );
      AddInput ( i_ctrl10      , 'ctrl10'      );
      AddInput ( i_ctrl11      , 'ctrl11'      );
      AddInput ( i_chain       , 'chain'       );
      AddInput ( i_trig        , 'trig'        );
      AddInput ( i_rnd         , 'rnd'         );
      AddInput ( i_clr         , 'clr'         );
      AddInput ( i_tilt        , 'tilt'        );
      AddInput ( i_tiltmod     , 'tiltmod'     );
      AddInput ( i_tiltmodamt  , 'tiltmodamt'  );
      AddInput ( i_basenote    , 'basenote'    );
      AddInput ( i_lownote     , 'lownote'     );
      AddInput ( i_highnote    , 'highnote'    );
      AddInput ( i_scale       , 'scale'       );
      AddInput ( i_smooth      , 'smooth'      );
      AddInput ( i_smoothmod   , 'smoothmod'   );
      AddInput ( i_smoothmodamt, 'smoothmodamt');
      AddInput ( i_active      , 'active'      );
      AddInput ( i_scalemod    , 'scalemod'    );
      AddOutput( o_out         , 'out'         );
    end;


    procedure   TModProbSequencer.SetDefaults; // override;
    var
      i : Integer;
    begin
      for i := 0 to 11
      do FInputs[ i + i_ctrl0] := 1;

      FInputs[ i_active] := 1.0;
      FScaleChanged      := True;
    end;


    procedure   TModProbSequencer.DoSlowTick; // override;
    var
      i          : Integer;
      imod       : Integer;
      Trig       : Boolean;
      Tilt       : TSignal;
      Scale      : Integer;
      Pivot      : TSignal;
      LowNote    : Integer;
      HighNote   : Integer;
      Prev       : Integer;
      Prob       : TSignal;
      TopProb    : TSignal;
      Smooth     : TSignal;
      Active     : Boolean;
      Modulation : Integer;
    begin
      Modulation := Round(( FInputs[ i_scalemod] - 0.5 * SystemScales.CountRec) * ( SystemScales.Count - 1));
      Scale      := ( Round( FInputs[ i_scale]) + Modulation ) mod SystemScales.Count;
      Active     := SignalToLogic( FInputs[ i_active]);

      if ( Scale <> FScale) or ResetFlag
      then begin
        FScale := Scale;
        FNotes := SystemScales.ScaleNotes[ FScale];

        for i := 0 to 11                                                         // Set button values from scale
        do begin
          if i in FNotes
          then FInputs[ i + i_note0 ] := 1
          else FInputs[ i + i_note0 ] := 0;
        end;

        FScaleChanged := True;                                                   // User feedback flag
      end;

      ResetFlag := False;
      Trig      := SignalToLogic( FInputs[ i_trig]);
      Prev      := High( FProb);
      TopProb   := FProb[ Prev];

      if not Trig and FOldTrig and Active                                        // Downgoing edge, calculate next note
      then begin
        LowNote  := Round( UnitsToNoteNumber( FInputs[ i_lownote ]));
        HighNote := Round( UnitsToNoteNumber( FInputs[ i_highnote]));
        Pivot    := LowNote + ( HighNote - LowNote) div 2;
        Tilt     := Clip( FInputs[ i_tilt] + FInputs[ i_tiltmod] * FInputs[ i_tiltmodamt], -1.0, 1.0);

        for i := 0 to Prev
        do begin
          if ( i >= LowNote) and ( i <= HighNote)
          then begin
            imod := i mod 12;
            Prob := ( 0.5 + ( i - Pivot) * Tilt) * FInputs[ imod + i_ctrl0];
            FProb[ i] :=
              FProb[ i - 1] +                                                    // NB: FProb[ -1] = 0, has to be set in Create();
              FixProb(
                Clip( FInputs[ imod + i_note0], 0, 1) *
                FInputs[ imod + i_prob0] *
                Prob
              );
          end
          else FProb[ i] := FProb[ i - 1];
        end;

      end
      else if Trig and not FOldTrig and ( TopProb > 0) and Active                // Upgoing edge, generate output
      then begin
        Smooth       := Clip( Finputs[ i_smooth] + FInputs[ i_smoothmod] * FInputs[ i_smoothmodamt], 0, 1);
        Prob         := Random * TopProb;
        FPrevProb    := Clip( Prob + Smooth * ( FPrevProb - Prob), 0, 0.9999 * TopProb);
        FCurrNoteNr  := Search( FPrevProb);                                      // FCurrNoteNr used for user feedback
        FCurrentNote := NoteNumberToUnits( FCurrNoteNr - Round( MiddleNote) + 12 * FInputs[ i_basenote]);
      end;

      if Active
      then FOutputs[ o_out] := FCurrentNote + FInputs[ i_chain]
      else FOutputs[ o_out] :=                FInputs[ i_chain];

      FOldTrig := Trig;
    end;


    procedure   TModProbSequencer.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      N : Integer;
      i : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'  , FInputs [ i_trig  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active', FInputs [ i_active]));
      N := FCurrNoteNr mod 12;

      for i := 0 to 11
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'curr%d', [ i], AppLocale), LogicToSignal( N = i)));

      if FScaleChanged
      then begin
        for i := 0 to 11                                                         // Set button looks from scale
        do begin
          if i in FNotes
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'note%d', [ i]), 1))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'note%d', [ i]), 0));
        end;

        FScaleChanged := False;
      end;
    end;


    procedure   TModProbSequencer.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'scale', SystemScales.ScaleName[ FScale]));
    end;


{ ========
  TModSeqPattern = class( TMod)
  const
    BIT_COUNT     = 24;
    STATE_COUNT   = 8 * 8 * 8;
    TOT_BIT_COUNT = STATE_COUNT * BIT_COUNT;
  type
    TMode = ( mRun, mEdit);
  private
    FBits1       : TBitArray;           // Bit inactive (0) or active (1)
    FBits2       : TBitArray;           // Bit is short (0) or long   (1)
    FSkips       : TBitArray;
    FResets      : TBitArray;
    FMode        : TMode;
    FCyclic      : Boolean;
    FPart        : Integer;
    FSection     : Integer;
    FElement     : Integer;
    FEditPart    : Integer;
    FEditSection : Integer;
    FEditElement : Integer;
    FOldClk      : Boolean;
    FClk         : Boolean;
    FNotClk      : Boolean;
    FOldRes      : Boolean;
    FRes         : Boolean;
    FESP         : array[ 0 .. 7, 0 .. 7, 0 .. 7 ] of Integer; // order: E, S, P
    FBitsChanged : Integer;
  private
    property    BitsChanged : Boolean read GetBitsChanged write SetBitsChanged;
  private
}

    function    TModSeqPattern.GetBitsChanged : Boolean;
    begin
      Result := FBitsChanged > 0;
    end;


    procedure   TModSeqPattern.SetBitsChanged( aValue: Boolean);
    begin
      if aValue
      then Inc( FBitsChanged)
      else if FBitsChanged > 0
      then Dec( FBitsChanged);
    end;


    procedure   TModSeqPattern.CreateBitArray( var anArray: TBitArray; aSize: Integer);
    begin
      anArray := TBitArray.Create;
      anArray.BitCount := aSize;
    end;


    procedure   TModSeqPattern.CopyBits( aSrcIndex, aDstIndex, aCount: Integer);
    var
      i : Integer;
    begin
      for i := 0 to aCount - 1
      do begin
        FBits1[ aDstIndex + i] := FBits1[ aSrcIndex + i];
        FBits2[ aDstIndex + i] := FBits2[ aSrcIndex + i];
      end;

      BitsChanged := True;
    end;


    procedure   TModSeqPattern.ClearBits( aDstIndex, aCount: Integer);
    var
      i : Integer;
    begin
      for i := 0 to aCount - 1
      do begin
        FBits1[ aDstIndex + i] := 0;
        FBits2[ aDstIndex + i] := 0;
      end;

      BitsChanged := True;
    end;


    function    TModSeqPattern.ResetSequence: Boolean;
    begin
      Result := False;

      if FCyclic
      then begin
        FPart    := 0;
        FSection := 0;
        FElement := 0;
        Result   := True;

        BitsChanged := True;
      end
      else begin
        FPart    := 8;
        FSection := 8;
        FElement := 8;

        BitsChanged := True;
      end;
    end;


    procedure   TModSeqPattern.IncStep;
    begin
      repeat
        Inc( FElement);
      until ( not StepIsSkip) or ( FElement >= 8) or StepIsReset;

      if FElement >= 8
      then begin
        FElement := 0;

        repeat
          Inc( FSection);
        until ( not StepIsSkip) or ( FSection >= 8) or StepIsReset;

        if FSection >= 8
        then begin
          FSection := 0;

          repeat
            Inc( FPart);
          until ( not StepIsSkip) or ( FPart >= 8) or StepIsReset;
        end;
      end;
    end;


    function    TModSeqPattern.ESPIndex( E, S, P: Integer): Integer; // overload;
    begin
      Result := E + 8 * ( S + 8 * P);
    end;


    function    TModSeqPattern.ESPIndex: Integer; // overload;
    begin
      Result := ESPIndex( FElement, FSection, FPart);
    end;


    function    TModSeqPattern.BitStartIndex: Integer;
    begin
      Result := BIT_COUNT * ESPIndex;
    end;


    function    TModSeqPattern.ESPEditIndex: Integer;
    begin
      Result := ESPIndex( FEditElement, FEditSection, FEditPart);
    end;


    function    TModSeqPattern.BitStartEditIndex: Integer;
    begin
      Result := BIT_COUNT * ESPEditIndex;
    end;


    function    TModSeqPattern.StepIsSkip: Boolean;
    begin
      Result := False;

      if FPart < 8
      then Result := FSkips[ ESPIndex] = 1;
    end;


    function    TModSeqPattern.StepIsReset: Boolean;
    begin
      Result := False;

      if FPart < 8
      then Result := FResets[ ESPIndex] = 1;
    end;


    procedure   TModSeqPattern.SetOutputs;
    var
      i : Integer;
      S : Integer;
      V : Boolean;
    begin
      S := BitStartIndex;

      for i := 0 to BIT_COUNT - 1
      do begin
        V := FBits1[ i + S] = 1;

        if V
        then begin
          if FBits2[ i + S] = 1
          then FOutputs[ i + o_out1] := LogicToSignal( True)
          else FOutputs[ i + o_out1] := LogicToSignal( SignalToLogic( FInputs[ i_clk]));
        end
        else FOutputs[ i + o_out1] := LogicToSignal( False);
      end;
    end;


    procedure   TModSeqPattern.ClearOutputs;
    var
      i : Integer;
    begin
      for i := 0 to BIT_COUNT - 1
      do FOutputs[ i + o_out1] := LogicToSignal( False);
    end;


    procedure   TModSeqPattern.NextStep;
    begin
      if FPart < 8
      then IncStep;

      if ( FPart >= 8) or StepIsReset
      then begin
        if ResetSequence
        then IncStep;
      end;

      if ( FPart < 8) and not ( StepIsSkip or StepIsReset)
      then SetOutputs
      else begin
        FPart    := 8;
        FElement := 8;
        FSection := 8;
        ClearOutputs;
      end;

      BitsChanged := FMode <> mspEdit;
    end;


    function    TModSeqPattern.BitDataValue: TSignalArray;
    var
      i : Integer;
    begin
      SetLength( Result, TOT_BIT_COUNT);

      for i := 0 to TOT_BIT_COUNT - 1
      do Result[ i] := BitsToValue( i);
    end;


    function    TModSeqPattern.EspDataValue: TSignalArray;
    var
      e : Integer;
      s : Integer;
      p : Integer;
      i : Integer;
    begin
      SetLength( Result, 8 * 8 * 8);

      for p := 0 to 7
      do begin
        for s := 0 to 7
        do begin
          for e := 0 to 7
          do begin
            i := e + 8 * ( s + 8 * p);
            Result[ i] := FESP[ e, s, p];
          end;
        end;
      end;
    end;


    procedure   TModSeqPattern.FixSkipsAndResets( anIndex, aNewValue: Integer); // overload;
    begin
      if ( anIndex >= 0) and ( anIndex < STATE_COUNT)
      then begin
        case aNewValue of
          0, 1 : begin FSkips [ anIndex] := 0; FResets[ anIndex] := 0; end;
          2    : begin FSkips [ anIndex] := 1; FResets[ anIndex] := 0; end;
          3    : begin FSkips [ anIndex] := 0; FResets[ anIndex] := 1; end;
        end;
      end;
    end;


    procedure   TModSeqPattern.FixSkipsAndResets( anE, anS, anP, aNewValue: Integer); // overload;
    begin
      FixSkipsAndResets( anE + 8 * ( anS + 8 * anP), aNewValue);
    end;


    procedure   TModSeqPattern.FixSkipsAndResets( aNewValue: Integer); // overload;
    begin
      FixSkipsAndResets( FEditElement, FEditSection, FEditPart, aNewValue);
    end;


    procedure   TModSeqPattern.IncBitValue( anIndex: Integer);
    begin
      ValueToBits( anIndex, ( BitsToValue( anIndex) + 1) mod 3);
      BitsChanged := True;
    end;


    function    TModSeqPattern.BitsToValue( anIndex: Integer): Integer;
    begin
      Result := 0;

      if ( anIndex >= 0) and ( anIndex < TOT_BIT_COUNT)
      then begin
        if FBits1[ anIndex] = 1
        then begin
          if FBits2[ anIndex] = 1
          then Result := 2
          else Result := 1;
        end;
      end;
    end;


    procedure   TModSeqPattern.ValueToBits( anIndex, aValue: Integer);
    begin
      if ( anIndex >= 0) and ( anIndex < TOT_BIT_COUNT)
      then begin
        case aValue of
          0 : begin FBits1[ anIndex] := 0; FBits2[ anIndex] := 0; end;
          1 : begin FBits1[ anIndex] := 1; FBits2[ anIndex] := 0; end;
          2 : begin FBits1[ anIndex] := 1; FBits2[ anIndex] := 1; end;
        end;
      end;
    end;


//  public

    constructor TModSeqPattern.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      CreateBitArray( FBits1 , TOT_BIT_COUNT);
      CreateBitArray( FBits2 , TOT_BIT_COUNT);
      CreateBitArray( FSkips , STATE_COUNT  );
      CreateBitArray( FResets, STATE_COUNT  );
      FPart        := 8;
      FSection     := 8;
      FElement     := 8;
      FEditPart    := 0;
      FEditSection := 0;
      FEditElement := 0;
      BitsChanged  := True;
    end;


    procedure   TModSeqPattern.CreateIO; // override;
    begin
      FIsSlow := True;
      FIsFast := True;
      AddInput ( i_bits1 , 'bits1' );
      AddInput ( i_bits2 , 'bits2' );
      AddInput ( i_bits3 , 'bits3' );
      AddInput ( i_bits4 , 'bits4' );
      AddInput ( i_bits5 , 'bits5' );
      AddInput ( i_bits6 , 'bits6' );
      AddInput ( i_bits7 , 'bits7' );
      AddInput ( i_bits8 , 'bits8' );
      AddInput ( i_bits9 , 'bits9' );
      AddInput ( i_bits10, 'bits10');
      AddInput ( i_bits11, 'bits11');
      AddInput ( i_bits12, 'bits12');
      AddInput ( i_bits13, 'bits13');
      AddInput ( i_bits14, 'bits14');
      AddInput ( i_bits15, 'bits15');
      AddInput ( i_bits16, 'bits16');
      AddInput ( i_bits17, 'bits17');
      AddInput ( i_bits18, 'bits18');
      AddInput ( i_bits19, 'bits19');
      AddInput ( i_bits20, 'bits20');
      AddInput ( i_bits21, 'bits21');
      AddInput ( i_bits22, 'bits22');
      AddInput ( i_bits23, 'bits23');
      AddInput ( i_bits24, 'bits24');
      AddInput ( i_loop  , 'loop'  );
      AddInput ( i_mode  , 'mode'  );
      AddInput ( i_clk   , 'clk'   );
      AddInput ( i_act   , 'act'   );
      AddInput ( i_res   , 'res'   );
      AddInput ( i_p1    , 'p1'    );
      AddInput ( i_p2    , 'p2'    );
      AddInput ( i_p3    , 'p3'    );
      AddInput ( i_p4    , 'p4'    );
      AddInput ( i_p5    , 'p5'    );
      AddInput ( i_p6    , 'p6'    );
      AddInput ( i_p7    , 'p7'    );
      AddInput ( i_p8    , 'p8'    );
      AddInput ( i_s1    , 's1'    );
      AddInput ( i_s2    , 's2'    );
      AddInput ( i_s3    , 's3'    );
      AddInput ( i_s4    , 's4'    );
      AddInput ( i_s5    , 's5'    );
      AddInput ( i_s6    , 's6'    );
      AddInput ( i_s7    , 's7'    );
      AddInput ( i_s8    , 's8'    );
      AddInput ( i_e1    , 'e1'    );
      AddInput ( i_e2    , 'e2'    );
      AddInput ( i_e3    , 'e3'    );
      AddInput ( i_e4    , 'e4'    );
      AddInput ( i_e5    , 'e5'    );
      AddInput ( i_e6    , 'e6'    );
      AddInput ( i_e7    , 'e7'    );
      AddInput ( i_e8    , 'e8'    );
      AddInput ( i_copye , 'copye' );
      AddInput ( i_copys , 'copys' );
      AddInput ( i_copyp , 'copyp' );
      AddInput ( i_clre  , 'clre'  );
      AddInput ( i_clrs  , 'clrs'  );
      AddInput ( i_clrp  , 'clrp'  );
      AddInput ( i_reset , 'reset' );
      AddOutput( o_out1  , 'out1'  );
      AddOutput( o_out2  , 'out2'  );
      AddOutput( o_out3  , 'out3'  );
      AddOutput( o_out4  , 'out4'  );
      AddOutput( o_out5  , 'out5'  );
      AddOutput( o_out6  , 'out6'  );
      AddOutput( o_out7  , 'out7'  );
      AddOutput( o_out8  , 'out8'  );
      AddOutput( o_out9  , 'out9'  );
      AddOutput( o_out10 , 'out10' );
      AddOutput( o_out11 , 'out11' );
      AddOutput( o_out12 , 'out12' );
      AddOutput( o_out13 , 'out13' );
      AddOutput( o_out14 , 'out14' );
      AddOutput( o_out15 , 'out15' );
      AddOutput( o_out16 , 'out16' );
      AddOutput( o_out17 , 'out17' );
      AddOutput( o_out18 , 'out18' );
      AddOutput( o_out19 , 'out19' );
      AddOutput( o_out20 , 'out20' );
      AddOutput( o_out21 , 'out21' );
      AddOutput( o_out22 , 'out22' );
      AddOutput( o_out23 , 'out23' );
      AddOutput( o_out24 , 'out24' );
      AddOutput( o_done  , 'done'  );
    end;


    destructor  TModSeqPattern.Destroy; // override;
    begin
      Locked       := True;
      FBitsChanged := 0;
      FResets.DisposeOf;
      FSkips .DisposeOf;
      FBits2 .DisposeOf;
      FBits1 .DisposeOf;
      inherited;
    end;


    procedure   TModSeqPattern.Pulse( anInput: Integer); // override;
    var
      V            : Integer;
      i            : Integer;
      j            : Integer;
      IndexChanged : Boolean;
    begin
      if anInput = i_reset
      then ResetFlag := True;

      if FMode = mspEdit
      then begin
        case anInput of

          i_bits1 .. i_bits24 :

            IncBitValue( BitStartEditIndex + anInput - i_bits1);

          i_e1 .. i_e8 : // Change one element

            begin
              V            := anInput - i_e1;
              IndexChanged := ESPIndex( V, FEditSection, FEditPart) <> ESPEditIndex;
              FEditElement := V;

              if IndexChanged
              then V :=   FESP[ FEditElement, FEditSection, FEditPart]
              else V := ( FESP[ FEditElement, FEditSection, FEditPart] + 1) mod 4;

              FESP[ FEditElement, FEditSection, FEditPart] := V;
              FixSkipsAndResets( V);

              BitsChanged := True;
            end;

          i_s1 .. i_s8 : // Change eight elements

            begin
              V            := anInput - i_s1;
              IndexChanged := ESPIndex( FEditElement, V, FEditPart) <> ESPEditIndex;
              FEditSection := V;

              if IndexChanged
              then begin
                V := FESP[ FEditElement, FEditSection, FEditPart];
                FixSkipsAndResets( V);
              end
              else begin
                V := ( FESP[ FEditElement, FEditSection, FEditPart] + 1) mod 4;

                for i := 0 to 7
                do begin
                  FESP[ i, FEditSection, FEditPart] := V;
                  FixSkipsAndResets( i, FEditSection, FEditPart, V);
                end;
              end;

              BitsChanged := True;
            end;

          i_p1 .. i_p8 : // Change 64 elelements

            begin
              V            := anInput - i_p1;
              IndexChanged := ESPIndex( FEditElement, FEditSection, V) <> ESPEditIndex;
              FEditPart    := V;

              if IndexChanged
              then begin
                V := FESP[ FEditElement, FEditSection, FEditPart];
                FixSkipsAndResets( V);
              end
              else begin
                V := ( FESP[ FEditElement, FEditSection, FEditPart] + 1) mod 4;

                for i := 0 to 7
                do begin
                  for j := 0 to 7
                  do begin
                    FESP[ j, i, FEditPart] := V;
                    FixSkipsAndResets( j, i, FEditPart, V);
                  end;
                end;
              end;

              BitsChanged := True;
            end;

          i_copye :

            begin
              if ( FEditElement > 0) and ( FEditElement < 8)
              then CopyBits( BIT_COUNT * ( FEditElement - 1), BIT_COUNT * ( FEditElement), BIT_COUNT);
            end;

          i_copys :

            begin
              if ( FEditSection > 0) and ( FEditSection < 8)
              then CopyBits( 8 * BIT_COUNT * ( FEditSection - 1), 8 * BIT_COUNT * ( FEditSection), 8 * BIT_COUNT);
            end;

          i_copyp :

            begin
              if ( FEditPart > 0) and ( FEditPart < 8)
              then CopyBits( 8 * 8 * BIT_COUNT * ( FEditPart - 1), 8 * 8 * BIT_COUNT * ( FEditPart), 8 * 8 * BIT_COUNT);
            end;

          i_clre  :

            begin
              if ( FEditElement >= 0 ) and ( FEditElement < 8)
              then ClearBits( BIT_COUNT * FEditElement, BIT_COUNT);
            end;

          i_clrs  :

            begin
              if ( FEditSection >= 0) and ( FEditSection < 8)
              then ClearBits( 8 * BIT_COUNT * FEditSection, 8 * BIT_COUNT);
            end;

          i_clrp  :

            begin
              if ( FEditPart >= 0) and ( FEditPart < 8)
              then ClearBits( 8 * 8 * BIT_COUNT * FEditPart, 8 * 8 * BIT_COUNT);
            end;

        end;
      end
      else begin // Run mode
        case anInput of

          i_reset :

            ResetFlag := True;

          i_e1 .. i_e8 :

            begin
              FElement    := anInput - i_e1;
              BitsChanged := True;
            end;

          i_s1 .. i_s8 :

            begin
              FSection    := anInput - i_s1;
              BitsChanged := True;
            end;

          i_p1 .. i_p8 :
            begin
              FPart       := anInput - i_p1;
              BitsChanged := True;
            end;

        end;
      end;
    end;


    procedure   TModSeqPattern.SetStringValue( const aName, aValue: string); // override;
    var
      i       : Integer;
      V       : Integer;
      BitData : TSignalArray;
      E       : Integer;
      S       : Integer;
      P       : Integer;
      L       : Integer;
    begin
      if SameText( aName, 'bitdata')
      then begin
        BitData := StrToSignalArray( aValue);

        for i := 0 to Length( BitData) - 1
        do ValueToBits( i, Round( BitData[ i]));

        BitsChanged := True;
      end
      else if SameText( aName, 'espdata')
      then begin
        BitData := StrToSignalArray( aValue);
        i       := 0;
        L       := Length( BitData);

        for P := 0 to 7
        do begin
          for S := 0 to 7
          do begin
            for E := 0 to 7
            do begin
              V := Round( BitData[ i]);

              if V in [ 2, 3]
              then Nop;

              FESP[ E, S, P] := V;
              FixSkipsAndResets( E, S, P, V);
              Inc( i);

              if i >= L
              then break;
            end;

            if i >= L
            then break;
          end;

          if i >= L
          then break;
        end;
      end
      else inherited;
    end;


    function    TModSeqPattern.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'bitdata')
      then Result := SignalArrayToStr( BitDataValue)
      else if SameText( aName, 'espdata')
      then Result := SignalArrayToStr( EspDataValue)
      else Result := inherited;
    end;


    procedure   TModSeqPattern.SetDefaults; // override;
    begin
      FInputs[ i_act] := 1.0; // Make it work with unconnected act input
    end;


    procedure   TModSeqPattern.DoTick; // override;
    // audio rate processing for clk and res - to not miss any short spikes
    var
      Clock : Boolean;
      Reset : Boolean;
    begin

      Clock := SignalToLogic( FInputs[ i_clk]);
      Reset := SignalToLogic( FInputs[ i_res]);

      if Clock and not FOldClk
      then FClk := True
      else if not Clock and FOldClk
      then FNotClk := True;

      if Reset and not FOldRes
      then FRes := True;

      FOldClk := Clock;
      FOldRes := Reset;
    end;


    procedure   TModSeqPattern.DoSlowTick; // override;
    var
      Active : Boolean;
    begin
      Active  := SignalToLogic( FInputs[ i_act]);
      FMode   := TMode( Round( FInputs[ i_mode]));
      FCyclic := Round( FInputs[ i_loop]) = 1;

      if FRes or ResetFlag
      then begin
        FPart    := 0;
        FSection := 0;
        FElement := 0;
      end;

      if Active
      then begin
        if FClk
        then NextStep
        else if FNotClk
        then SetOutputs;
      end
      else begin
        FPart    := 8;
        FSection := 8;
        FElement := 8;
      end;

      FOutputs[ o_done] := LogicToSignal( FPart >= 8);
      FRes      := False;
      FClk      := False;
      FNotClk   := False;
      ResetFlag := False;
    end;


    procedure   TModSeqPattern.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
      V : Integer;
      S : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'done', FOutputs[ o_done]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'act' , FInputs [ i_act ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clk' , FInputs [ i_clk ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res' , FInputs [ i_res ]));

      for i := 0 to BIT_COUNT - 1
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'bit%d', [ i + 1], AppLocale), FOutputs[ i + o_out1]));

      if FMode = mspEdit
      then begin
        if BitsChanged
        then begin
          BitsChanged := False;
          S           := BitStartEditIndex;

          for i := 0 to BIT_COUNT - 1
          do begin
            V := BitsToValue( i + S);
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'bits%d', [ i + 1], AppLocale), V));
          end;
        end;

        for i := 0 to 7
        do begin
          // i : Element
          if ( i = FEditElement) and ( FESP[ i, FEditSection, FEditPart] in [ 0, 1])
          then begin
            FESP[ i, FEditSection, FEditPart] := 1;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 1));
          end
          else if FESP[ i, FEditSection, FEditPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), FESP[ i, FEditSection, FEditPart]))
          else begin
            FESP[ i, FEditSection, FEditPart] := 0;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 0));
          end;

          // i : Section
          if ( i = FEditSection) and ( FESP[ FEditElement, i, FEditPart] in [ 0, 1])
          then begin
            FESP[ FEditElement, i, FEditPart] := 1;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 1));
          end
          else if FESP[ FEditElement, i, FEditPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), FESP[ FEditElement, i, FEditPart]))
          else begin
            FESP[ FEditElement, i, FEditPart] := 0;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 0));
          end;

          // i : Part
          if ( i = FEditPart) and ( FESP[ FEditElement, FEditSection, i] in [ 0, 1])
          then begin
            FESP[ FEditElement, FEditSection, i] := 1;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 1));
          end
          else if FESP[ FEditElement, FEditSection, i] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), FESP[ FEditElement, FEditSection, i]))
          else begin
            FESP[ FEditElement, FEditSection, i] := 0;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 0));
          end;
        end;
      end
      else begin // Run mode otherwise
        if BitsChanged
        then begin
          BitsChanged := False;
          S           := BitStartIndex;

          for i := 0 to BIT_COUNT - 1
          do begin
            V := BitsToValue( i + S);
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'bits%d', [ i + 1], AppLocale), V));
          end;
        end;

        for i := 0 to 7
        do begin
          // i : Element
          if ( i = FElement) and ( FESP[ i, FSection, FPart] in [ 0, 1])
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 1))
          else if FESP[ i, FSection, FPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), FESP[ i, FSection, FPart]))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 0));

          // i : Section
          if ( i = FSection) and ( FESP[ FElement, i, FPart] in [ 0, 1])
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 1))
          else if FESP[ FElement, i, FPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), FESP[ FElement, i, FPart]))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 0));

          // i : Part
          if ( i = FPart) and ( FESP[ FElement, FSection, i] in [ 0, 1])
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 1))
          else if FESP[ FElement, FSection, i] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), FESP[ FElement, FSection, i]))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 0));
        end;
      end;
    end;


    procedure   TModSeqPattern.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    // Must collect the programmed pattern data so it can be stored with the patch
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'bitdata', BitDataValue));
      aCallback( Self, MakeInfo( aPrefix, Name, 'espdata', EspDataValue));
    end;


{ ========
  TModSeqValues = class( TMod)
  strict private
  const
    VALUE_COUNT     = 8;
    STATE_COUNT     = 8 * 8 * 8;
    TOT_VALUE_COUNT = STATE_COUNT * VALUE_COUNT;
  type
    TMode = ( msvRun, msvEdit);
  private
    FValues        : TSignalArray;
    FSkips         : TBitArray;
    FResets        : TBitArray;
    FMode          : TMode;
    FCyclic        : Boolean;
    FPart          : Integer;
    FSection       : Integer;
    FElement       : Integer;
    FEditPart      : Integer;
    FEditSection   : Integer;
    FEditElement   : Integer;
    FOldClk        : Boolean;
    FClk           : Boolean;
    FNotClk        : Boolean;
    FOldRes        : Boolean;
    FRes           : Boolean;
    FESP           : array[ 0 .. 7, 0 .. 7, 0 .. 7 ] of Integer; // order: E, S, P
    FValuesChanged : Integer;
  private
    property    ValuesChanged: Boolean read GetValuesChanged write SetValuesChanged;
  private
}

    procedure   TModSeqValues.CreateBitArray( var anArray: TBitArray; aSize: Integer);
    begin
      anArray := TBitArray.Create;
      anArray.BitCount := aSize;
    end;


    procedure   TModSeqValues.CopyValues( aSrcIndex, aDstIndex, aCount: Integer);
    var
      i : Integer;
    begin
      for i := 0 to aCount - 1
      do FValues[ aDstIndex + i] := FValues[ aSrcIndex + i];

      ValuesChanged := True;
    end;


    procedure   TModSeqValues.ClearValues( aDstIndex, aCount: Integer);
    var
      i : Integer;
    begin
      for i := 0 to aCount - 1
      do FValues[ aDstIndex + i] := 0;

      ValuesChanged := True;
    end;


    function    TModSeqValues.ResetSequence: Boolean;
    begin
      Result := False;

      if FCyclic
      then begin
        FPart    := 0;
        FSection := 0;
        FElement := 0;
        Result   := True;
      end
      else begin
        FPart    := 8;
        FSection := 8;
        FElement := 8;
      end;

      ValuesChanged := True;
    end;


    procedure   TModSeqValues.IncStep;
    begin
      repeat
        Inc( FElement);
      until ( not StepIsSkip) or ( FElement >= 8) or StepIsReset;

      if FElement >= 8
      then begin
        FElement := 0;

        repeat
          Inc( FSection);
        until ( not StepIsSkip) or ( FSection >= 8) or StepIsReset;

        if FSection >= 8
        then begin
          FSection := 0;

          repeat
            Inc( FPart);
          until ( not StepIsSkip) or ( FPart >= 8) or StepIsReset;
        end;
      end;
    end;


    function    TModSeqValues.ESPIndex( E, S, P: Integer): Integer; // overload;
    begin
      Result := E + 8 * ( S + 8 * P);
    end;


    function    TModSeqValues.ESPIndex: Integer; // overload;
    begin
      Result := ESPIndex( FElement, FSection, FPart);
    end;


    function    TModSeqValues.ValueStartIndex: Integer;
    begin
      Result := VALUE_COUNT * ESPIndex;
    end;


    function    TModSeqValues.ESPEditIndex: Integer;
    begin
      Result := ESPIndex( FEditElement, FEditSection, FEditPart);
    end;


    function    TModSeqValues.ValueStartEditIndex: Integer;
    begin
      Result := VALUE_COUNT * ESPEditIndex;
    end;


    function    TModSeqValues.StepIsSkip: Boolean;
    begin
      Result := False;

      if FPart < 8
      then Result := FSkips[ ESPIndex] = 1;
    end;


    function    TModSeqValues.StepIsReset: Boolean;
    begin
      Result := False;

      if FPart < 8
      then Result := FResets[ ESPIndex] = 1;
    end;


    procedure   TModSeqValues.SetOutputs;
    var
      i : Integer;
    begin
      // todo : look at FValMode ... internal range always is 0 - 127
      for i := o_out1 to o_out8
      do FOutputs[ i] := FValues[ ValueStartIndex + i - o_out1];
    end;


    procedure   TModSeqValues.ClearOutputs;
    var
      i : Integer;
    begin
      // todo : look at FValMode ... internal range always is 0 - 127
      for i := o_out1 to o_out8
      do FOutputs[ i] := 0;
    end;


    procedure   TModSeqValues.NextStep;
    begin
      if FPart < 8
      then IncStep;

      if ( FPart >= 8) or StepIsReset
      then begin
        if ResetSequence
        then IncStep;
      end;

      if ( FPart < 8) and not ( StepIsSkip or StepIsReset)
      then SetOutputs
      else begin
        FPart    := 8;
        FElement := 8;
        FSection := 8;
        ClearOutputs;
      end;

      ValuesChanged := FMode <> msvEdit;
    end;


    function    TModSeqValues.EspDataValue: TSignalArray;
    var
      e : Integer;
      s : Integer;
      p : Integer;
      i : Integer;
    begin
      SetLength( Result, 8 * 8 * 8);

      for p := 0 to 7
      do begin
        for s := 0 to 7
        do begin
          for e := 0 to 7
          do begin
            i := e + 8 * ( s + 8 * p);
            Result[ i] := FESP[ e, s, p];
          end;
        end;
      end;
    end;


    procedure   TModSeqValues.FixSkipsAndResets( anIndex, aNewValue: Integer); // overload;
    begin
      if ( anIndex >= 0) and ( anIndex < STATE_COUNT)
      then begin
        case aNewValue of
          0, 1 : begin FSkips [ anIndex] := 0; FResets[ anIndex] := 0; end;
          2    : begin FSkips [ anIndex] := 1; FResets[ anIndex] := 0; end;
          3    : begin FSkips [ anIndex] := 0; FResets[ anIndex] := 1; end;
        end;
      end;
    end;


    procedure   TModSeqValues.FixSkipsAndResets( anE, anS, anP, aNewValue: Integer); // overload;
    begin
      FixSkipsAndResets( anE + 8 * ( anS + 8 * anP), aNewValue);
    end;


    procedure   TModSeqValues.FixSkipsAndResets( aNewValue: Integer); // overload;
    begin
      FixSkipsAndResets( FEditElement, FEditSection, FEditPart, aNewValue);
    end;


    function    TModSeqValues.GetValuesChanged: Boolean;
    begin
      Result := FValuesChanged > 0;
    end;


    procedure   TModSeqValues.SetValuesChanged( aValue: Boolean);
    begin
      if aValue
      then Inc( FValuesChanged)
      else if FValuesChanged > 0
      then Dec( FValuesChanged);
    end;


//  public

    constructor TModSeqValues.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      CreateBitArray( FSkips ,     STATE_COUNT);
      CreateBitArray( FResets,     STATE_COUNT);
      SetLength     ( FValues, TOT_VALUE_COUNT);
      FPart         := 8;
      FSection      := 8;
      FElement      := 8;
      FEditPart     := 0;
      FEditSection  := 0;
      FEditElement  := 0;
      ValuesChanged := True;
    end;


    procedure   TModSeqValues.CreateIO; // override;
    begin
      FIsSlow := True;
      FIsFast := True;
      AddInput ( i_value1 , 'value1' );
      AddInput ( i_value2 , 'value2' );
      AddInput ( i_value3 , 'value3' );
      AddInput ( i_value4 , 'value4' );
      AddInput ( i_value5 , 'value5' );
      AddInput ( i_value6 , 'value6' );
      AddInput ( i_value7 , 'value7' );
      AddInput ( i_value8 , 'value8' );
      AddInput ( i_loop   , 'loop'   );
      AddInput ( i_mode   , 'mode'   );
      AddInput ( i_clk    , 'clk'    );
      AddInput ( i_act    , 'act'    );
      AddInput ( i_res    , 'res'    );
      AddInput ( i_p1     , 'p1'     );
      AddInput ( i_p2     , 'p2'     );
      AddInput ( i_p3     , 'p3'     );
      AddInput ( i_p4     , 'p4'     );
      AddInput ( i_p5     , 'p5'     );
      AddInput ( i_p6     , 'p6'     );
      AddInput ( i_p7     , 'p7'     );
      AddInput ( i_p8     , 'p8'     );
      AddInput ( i_s1     , 's1'     );
      AddInput ( i_s2     , 's2'     );
      AddInput ( i_s3     , 's3'     );
      AddInput ( i_s4     , 's4'     );
      AddInput ( i_s5     , 's5'     );
      AddInput ( i_s6     , 's6'     );
      AddInput ( i_s7     , 's7'     );
      AddInput ( i_s8     , 's8'     );
      AddInput ( i_e1     , 'e1'     );
      AddInput ( i_e2     , 'e2'     );
      AddInput ( i_e3     , 'e3'     );
      AddInput ( i_e4     , 'e4'     );
      AddInput ( i_e5     , 'e5'     );
      AddInput ( i_e6     , 'e6'     );
      AddInput ( i_e7     , 'e7'     );
      AddInput ( i_e8     , 'e8'     );
      AddInput ( i_copye  , 'copye'  );
      AddInput ( i_copys  , 'copys'  );
      AddInput ( i_copyp  , 'copyp'  );
      AddInput ( i_clre   , 'clre'   );
      AddInput ( i_clrs   , 'clrs'   );
      AddInput ( i_clrp   , 'clrp'   );
      AddInput ( i_reset  , 'reset'  );
      AddInput ( i_valmode, 'valmode');
      AddOutput( o_out1   , 'out1'   );
      AddOutput( o_out2   , 'out2'   );
      AddOutput( o_out3   , 'out3'   );
      AddOutput( o_out4   , 'out4'   );
      AddOutput( o_out5   , 'out5'   );
      AddOutput( o_out6   , 'out6'   );
      AddOutput( o_out7   , 'out7'   );
      AddOutput( o_out8   , 'out8'   );
      AddOutput( o_done   , 'done'   );
    end;


    destructor  TModSeqValues.Destroy; // override;
    begin
      Locked         := True;
      FValuesChanged := 0;
      FResets.DisposeOf;
      FSkips .DisposeOf;
      inherited;
    end;


    procedure   TModSeqValues.Pulse( anInput: Integer); // override;
    var
      V            : Integer;
      i            : Integer;
      j            : Integer;
      IndexChanged : Boolean;
    begin
      if anInput = i_reset
      then ResetFlag := True;

      if FMode = msvEdit
      then begin
        case anInput of

          i_e1 .. i_e8 : // Change one element

            begin
              V            := anInput - i_e1;
              IndexChanged := ESPIndex( V, FEditSection, FEditPart) <> ESPEditIndex;
              FEditElement := V;

              if IndexChanged
              then V :=   FESP[ FEditElement, FEditSection, FEditPart]
              else V := ( FESP[ FEditElement, FEditSection, FEditPart] + 1) mod 4;

              FESP[ FEditElement, FEditSection, FEditPart] := V;
              FixSkipsAndResets( V);
              ValuesChanged := True;
            end;

          i_s1 .. i_s8 : // Change eight elements

            begin
              V            := anInput - i_s1;
              IndexChanged := ESPIndex( FEditElement, V, FEditPart) <> ESPEditIndex;
              FEditSection := V;

              if IndexChanged
              then begin
                V := FESP[ FEditElement, FEditSection, FEditPart];
                FixSkipsAndResets( V);
              end
              else begin
                V := ( FESP[ FEditElement, FEditSection, FEditPart] + 1) mod 4;

                for i := 0 to 7
                do begin
                  FESP[ i, FEditSection, FEditPart] := V;
                  FixSkipsAndResets( i, FEditSection, FEditPart, V);
                end;
              end;

              ValuesChanged := True;
            end;

          i_p1 .. i_p8 : // Change 64 elelements

            begin
              V            := anInput - i_p1;
              IndexChanged := ESPIndex( FEditElement, FEditSection, V) <> ESPEditIndex;
              FEditPart    := V;

              if IndexChanged
              then begin
                V := FESP[ FEditElement, FEditSection, FEditPart];
                FixSkipsAndResets( V);
              end
              else begin
                V := ( FESP[ FEditElement, FEditSection, FEditPart] + 1) mod 4;

                for i := 0 to 7
                do begin
                  for j := 0 to 7
                  do begin
                    FESP[ j, i, FEditPart] := V;
                    FixSkipsAndResets( j, i, FEditPart, V);
                  end;
                end;
              end;

              ValuesChanged := True;
            end;

          i_copye :

            begin
              if ( FEditElement > 0) and ( FEditElement < 8)
              then CopyValues( VALUE_COUNT * ( FEditElement - 1), VALUE_COUNT * ( FEditElement), VALUE_COUNT);
            end;

          i_copys :

            begin
              if ( FEditSection > 0) and ( FEditSection < 8)
              then CopyValues( 8 * VALUE_COUNT * ( FEditSection - 1), 8 * VALUE_COUNT * ( FEditSection), 8 * VALUE_COUNT);
            end;

          i_copyp :

            begin
              if ( FEditPart > 0) and ( FEditPart < 8)
              then CopyValues( 8 * 8 * VALUE_COUNT * ( FEditPart - 1), 8 * 8 * VALUE_COUNT * ( FEditPart), 8 * 8 * VALUE_COUNT);
            end;

          i_clre  :

            begin
              if ( FEditElement >= 0 ) and ( FEditElement < 8)
              then ClearValues( VALUE_COUNT * FEditElement, VALUE_COUNT);
            end;

          i_clrs  :

            begin
              if ( FEditSection >= 0) and ( FEditSection < 8)
              then ClearValues( 8 * VALUE_COUNT * FEditSection, 8 * VALUE_COUNT);
            end;

          i_clrp  :

            begin
              if ( FEditPart >= 0) and ( FEditPart < 8)
              then ClearValues( 8 * 8 * VALUE_COUNT * FEditPart, 8 * 8 * VALUE_COUNT);
            end;

        end;
      end
      else begin // Run mode
        case anInput of
          i_reset      : ResetFlag := True;
          i_e1 .. i_e8 : begin FElement  := anInput - i_e1; ValuesChanged := True; end;
          i_s1 .. i_s8 : begin FSection  := anInput - i_s1; ValuesChanged := True; end;
          i_p1 .. i_p8 : begin FPart     := anInput - i_p1; ValuesChanged := True; end;
        end;
      end;
    end;


    procedure   TModSeqValues.SetStringValue( const aName, aValue: string); // override;
    var
      i       : Integer;
      V       : Integer;
      EspData : TSignalArray;
      E       : Integer;
      S       : Integer;
      P       : Integer;
      L       : Integer;
    begin
      if SameText( aName, 'valuedata')
      then begin
        if aValue <> ''
        then FValues := StrToSignalArray( aValue);

        ValuesChanged := True;
      end
      else if SameText( aName, 'espdata')
      then begin
        EspData := StrToSignalArray( aValue);
        i       := 0;
        L       := Length( EspData);

        for P := 0 to 7
        do begin
          for S := 0 to 7
          do begin
            for E := 0 to 7
            do begin
              V := Round( EspData[ i]);

              if V in [ 2, 3]
              then Nop;

              FESP[ E, S, P] := V;
              FixSkipsAndResets( E, S, P, V);
              Inc( i);

              if i >= L
              then break;
            end;

            if i >= L
            then break;
          end;

          if i >= L
          then break;
        end;
      end
      else inherited;
    end;


    function    TModSeqValues.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'valuedata')
      then Result := SignalArrayToStr( FValues)
      else if SameText( aName, 'espdata')
      then Result := SignalArrayToStr( EspDataValue)
      else Result := inherited;
    end;


    procedure   TModSeqValues.SetDefaults; // override;
    begin
      FInputs[ i_act] := 1.0; // Make it work with unconnected act input
    end;


    procedure   TModSeqValues.DoTick; // override;
    // audio rate processing for clk and res - to not miss any short spikes
    var
      Clock : Boolean;
      Reset : Boolean;
    begin
      Clock := SignalToLogic( FInputs[ i_clk]);
      Reset := SignalToLogic( FInputs[ i_res]);

      if Clock and not FOldClk
      then FClk := True
      else if not Clock and FOldClk
      then FNotClk := True;

      if Reset and not FOldRes
      then FRes := True;

      FOldClk := Clock;
      FOldRes := Reset;
    end;


    procedure   TModSeqValues.DoSlowTick; // override;
    var
      Active : Boolean;
      i      : Integer;
    begin
      Active   := SignalToLogic( FInputs[ i_act]);
      FMode    := TMode( Round( FInputs[ i_mode]));
      FCyclic  := Round( FInputs[ i_loop]) = 1;
      FValMode := Round( FInputs[ i_valmode]);

      if FMode = msvEdit
      then begin
        for i := i_value1 to i_value8
        do FValues[ ValueStartEditIndex + i - i_value1] := FInputs[ i];
      end;

      if FRes or ResetFlag
      then begin
        FPart    := 0;
        FSection := 0;
        FElement := 0;
      end;

      if Active
      then begin
        if FClk
        then NextStep
        else if FNotClk
        then SetOutputs;
      end
      else begin
        FPart    := 8;
        FSection := 8;
        FElement := 8;
      end;

      FOutputs[ o_done] := LogicToSignal( FPart >= 8);
      FRes      := False;
      FClk      := False;
      FNotClk   := False;
      ResetFlag := False;
    end;


    procedure   TModSeqValues.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'done', FOutputs[ o_done]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'act' , FInputs [ i_act ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'clk' , FInputs [ i_clk ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res' , FInputs [ i_res ]));

      if FMode = msvEdit
      then begin
        for i := 0 to 7
        do begin
          // i : Element
          if ( i = FEditElement) and ( FESP[ i, FEditSection, FEditPart] in [ 0, 1])
          then begin
            FESP[ i, FEditSection, FEditPart] := 1;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 1));
          end
          else if FESP[ i, FEditSection, FEditPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), FESP[ i, FEditSection, FEditPart]))
          else begin
            FESP[ i, FEditSection, FEditPart] := 0;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 0));
          end;

          // i : Section
          if ( i = FEditSection) and ( FESP[ FEditElement, i, FEditPart] in [ 0, 1])
          then begin
            FESP[ FEditElement, i, FEditPart] := 1;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 1));
          end
          else if FESP[ FEditElement, i, FEditPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), FESP[ FEditElement, i, FEditPart]))
          else begin
            FESP[ FEditElement, i, FEditPart] := 0;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 0));
          end;

          // i : Part
          if ( i = FEditPart) and ( FESP[ FEditElement, FEditSection, i] in [ 0, 1])
          then begin
            FESP[ FEditElement, FEditSection, i] := 1;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 1));
          end
          else if FESP[ FEditElement, FEditSection, i] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), FESP[ FEditElement, FEditSection, i]))
          else begin
            FESP[ FEditElement, FEditSection, i] := 0;
            aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 0));
          end;
        end;
      end
      else begin // Run mode otherwise
        for i := 0 to 7
        do begin
          // i : Element
          if ( i = FElement) and ( FESP[ i, FSection, FPart] in [ 0, 1])
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 1))
          else if FESP[ i, FSection, FPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), FESP[ i, FSection, FPart]))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'e%d', [ i + 1], AppLocale), 0));

          // i : Section
          if ( i = FSection) and ( FESP[ FElement, i, FPart] in [ 0, 1])
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 1))
          else if FESP[ FElement, i, FPart] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), FESP[ FElement, i, FPart]))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 's%d', [ i + 1], AppLocale), 0));

          // i : Part
          if ( i = FPart) and ( FESP[ FElement, FSection, i] in [ 0, 1])
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 1))
          else if FESP[ FElement, FSection, i] in [ 2, 3]
          then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), FESP[ FElement, FSection, i]))
          else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'p%d', [ i + 1], AppLocale), 0));
        end;
      end;
    end;


    procedure   TModSeqValues.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // override;
    var
      i : Integer;
      S : Integer;
    begin
      if ValuesChanged
      then begin
        ValuesChanged := False;
        S             := ValueStartEditIndex;

        for i := 0 to VALUE_COUNT - 1
        do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'value%d', [ i + 1], AppLocale), FValues[ i + S]));
      end;
    end;


    procedure   TModSeqValues.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'valuedata', FValues     ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'espdata'  , EspDataValue));
    end;


{ ========
  TModSlew = class( TMod)
  public
}

    procedure   TModSlew.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in  , 'in'  );
      AddInput ( i_slew, 'slew'); 
      AddOutput( o_out , 'out' );
    end;


    procedure   TModSlew.DoSlowTick; // override;
    var
      Inp   : TSignal;
      Curve : TSignal;
    begin
      Inp   := Normalize( FInputs[ i_in]);
      Curve := SlewToCurve( FInputs[ i_slew]);
      FOutputs[ o_out] := Normalize( Inp + Curve * ( FOutputs[ o_out] - Inp));
    end;


{ ========
  TModRMS = class( TMod)
  private
    FRMSCalculator : TRMSCalculator;
    FPrevTime      : TSignal;
    FWasSpedUp     : Boolean;
  public
}

    constructor TModRMS.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FPrevTime      := 101;                          // ms
      FWasSpedUp     := IsSpedUp;
      FRMSCalculator := TRMSCalculator.Create;
      FRMSCalculator.AvgTime := FPrevTime;
      FRMSCalculator.IsFast  := IsSpedUp;
      SampleRateChanged;
    end;


    procedure   TModRMS.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in  , 'in'  ); 
      AddInput ( i_time, 'time');
      AddOutput( o_out , 'out' );
    end;


    destructor  TModRMS.Destroy; // override;
    begin
      FreeAndNil( FRMSCalculator);
      inherited;
    end;


    procedure   TModRMS.DoSlowTick; // override;
    var
      OutVal : TSignal;
      Time   : TSignal;
      SpedUp : Boolean;
    begin
      Time := LookupEnvTime( erMedium, FInputs[ i_time]);

      if Time <> FPrevTime
      then begin
        FRMSCalculator.AvgTime := Time;
        FPrevTime              := Time;
      end;

      SpedUp := IsSpedUp;

      if SpedUp <> FWasSpedUp
      then begin
        SampleRateChanged;
        FRMSCalculator.IsFast := SpedUp;
        FWasSpedUp            := SpedUp;
      end;

      FRMSCalculator.Tick( Normalize( FInputs[ i_in]), OutVal);
      FOutputs[ o_out] := Normalize( OutVal);
    end;


    procedure   TModRMS.SampleRateChanged; // override;
    begin
      FRMSCalculator.SampleRateChanged;
    end;


{ ========
  TModCsvData = class( TMod)
  private
    FMemory   : Array[ 0 .. 15] of TSignalArray;
    FComments : TStringList;
    FFileName : string;
    FChannels : Integer;
    FCounter  : Integer;
    FSteps    : Integer;
    FInState  : TSeqInState;
    FOldRes   : Boolean;
    FMode     : Integer;
    FStopped  : Boolean;
  public
    property    FileName: string read FFileName write SetFileName;
  private
 }

    procedure   TModCsvData.LoadFile;

      procedure AddSignal( aMemoryIndex: Integer; aValue: TSignal);
      var
        aLength : Integer;
      begin
        aLength := Length( FMemory[ aMemoryIndex]);
        SetLength( FMemory[ aMemoryIndex], aLength + 1);
        FMemory[ aMemoryIndex][ aLength] := aValue;
      end;

    var
      aData  : TStringList;
      aparts : TStringList;
      aLine  : string;
      i      : Integer;
      j      : Integer;
      p      : Integer;
      S      : string;
      T      : string;
      Cmt    : string;
      First  : Boolean;
    begin
      aData := TStringList.Create;

      try
        if FileExists( FFileName)
        then begin
          aData.LoadFromFile( FFileName);
          First := True;

          for i := Low( FMemory) to High( FMemory)
          do begin
            for j := Low( FMemory[ i]) to High( FMemory[ i])
            do FMemory[ i][ j] := 0.0;
          end;

          FComments.Clear;

          for aLine in aData
          do begin
            // Strip comment, but keep end of line comment for comment label in module
            p := Pos( '#', aLine);

            if p > 0
            then begin
              Cmt := Trim( Copy( aLine, p + 1, Length( aLine)));
              S   := Trim( Copy( aLine, 1, p - 1))
            end
            else begin
              Cmt := '';
              S   := Trim( aLine);
            end;

            // if line not empty after that handle it
            if Length( S) > 0
            then begin
              aParts := Explode( S, ',');

              try
                if First                           // First line determines column count
                then begin
                  FChannels := aParts.Count;
                  First     := False;
                end;

                for i := 0 to FChannels - 1
                do begin
                  if i > aParts.Count - 1
                  then AddSignal( i, 0)            // When not enough data on this line set remaining columns to zero
                  else begin
                    T := Trim( aParts[ i]);
                    AddSignal( i, StrToFloatDef( T, 0));
                  end;
                end;
              finally
                aParts.DisposeOf;
              end;

              FComments.Add( Cmt);
            end;
          end;

          FSteps   := Length( FMemory[ 0]);
          FCounter := Min( FCounter, FSteps - 1);  // Do not address outside data area
        end;
      finally
        aData.DisposeOf;
      end;
    end;


    procedure   TModCsvData.SetFileName( const aValue: string);
    begin
      if  aValue <> FFileName
      then begin
        FFileName := aValue;
        LoadFile;
      end;
    end;


//  public

    constructor TModCsvData.Create( aParent: TSynthPatch; const aName: string); // override;
    var
      i : Integer;
    begin
      inherited;

      for i := Low( FMemory) to High( FMemory)
      do FMemory[ i] := nil;

      FComments := TStringList.Create;
    end;


    procedure   TModCsvData.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := True;
      AddInput ( i_reset, 'reset');
      AddInput ( i_trig , 'trig' );
      AddInput ( i_dir  , 'dir'  );
      AddInput ( i_mode , 'mode' );
      AddOutput( o_out1 , 'out1' );
      AddOutput( o_out2 , 'out2' );
      AddOutput( o_out3 , 'out3' );
      AddOutput( o_out4 , 'out4' );
      AddOutput( o_out5 , 'out5' );
      AddOutput( o_out6 , 'out6' );
      AddOutput( o_out7 , 'out7' );
      AddOutput( o_out8 , 'out8' );
      AddOutput( o_out9 , 'out9' );
      AddOutput( o_out10, 'out10');
      AddOutput( o_out11, 'out11');
      AddOutput( o_out12, 'out12');
      AddOutput( o_out13, 'out13');
      AddOutput( o_out14, 'out14');
      AddOutput( o_out15, 'out15');
      AddOutput( o_out16, 'out16');
    end;


    destructor  TModCsvData.Destroy; // override;
    var
      i : Integer;
    begin
      FreeAndNil( FComments);

      for i := Low( FMemory) to High( FMemory)
      do FreeAndNil( FMemory[ i]);

      inherited;
    end;


    procedure   TModCsvData.SetDefaults; // override;
    begin
      FInputs[ i_dir] := 1; // Make seq run forward by default.
    end;


    procedure   TModCsvData.SetStringValue( const aName, aValue: string); // override;
    begin
      if SameText( aName, 'filename')
      then FileName := aValue
      else inherited;
    end;


    function    TModCsvData.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'filename')
      then Result := FFileName
      else Result := inherited;
    end;


    procedure   TModCsvData.DoTick; // override;
    // Follow inputs at audio rate
    var
      OldCounter : Integer;
      Forwards   : Boolean;
    begin
      Forwards := SignalToLogic( FInputs[ i_dir]);

      if ResetFlag or ( not FOldRes and SignalToLogic( FInputs[ i_reset]))
      then begin
        ResetFlag := False;
        FInState  := seqClkLow;
        if Forwards
        then FCounter := 0
        else FCounter := FSteps - 1;
        FStopped   := False;
      end;

      FOldRes := SignalToLogic( FInputs[ i_reset]);

      if ( FSteps > 0) and not FStopped
      then begin
        FMode      := Round( FInputs[ i_mode]);
        OldCounter := FCounter;

        case FInState of

          seqClkLow :

            begin
              if SignalToLogic( FInputs[ i_trig])
              then begin
                FInState := seqClkHigh;

                if Forwards
                then begin
                  FCounter := ( FCounter + 1) mod FSteps;

                  if ( FCounter < OldCounter) and ( FMode = 1)
                  then FStopped := True;
                end
                else begin
                  if FCounter <= 0
                  then begin
                    FCounter := FSteps - 1;

                    if FMode = 1
                    then FStopped := True;
                  end
                  else Dec( FCounter);
                end;
              end;
            end;

          seqClkHigh :

            begin
              if FInputs[ i_trig] <= 0
              then FInState := seqClkLow;
            end;

        end;
      end;
    end;


    procedure   TModCsvData.DoSlowTick; // override;
    // generate output at control rate
    var
      i : Integer;
    begin
      if ( FSteps > 0) and ( FChannels > 0)
      then begin
        for i := o_out1 to Min( FChannels - 1, o_out16)
        do FOutputs[ i] := FMemory[ i - o_out1][ FCounter];
      end;
    end;


    procedure   TModCsvData.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'    , FInputs[ i_trig ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset'   , FInputs[ i_reset]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'     , FInputs[ i_dir  ]));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step'    , FCounter + 1     ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'steps'   , FSteps           ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'channels', FChannels        ));
    end;


    procedure   TModCsvData.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      Cmt : string;
    begin
      if FCounter < FComments.Count
      then Cmt := FComments[ FCounter]
      else Cmt := '';

      aCallback( Self, MakeInfo( aPrefix, Name, 'comment', Cmt));
    end;


{ ========
  TModKleeBit = class( TMod)
  private
    FPattern    : Byte;
    FOldStep    : Boolean;
    FOldLoad    : Boolean;
    FBus1       : Boolean;
    FBus2       : Boolean;
    FBus3       : Boolean;
    FAnalog     : TSignal;
    FLoopMode   : Boolean;
    FInvertMode : Boolean;
  private
}

    procedure   TModKleeBit.LoadPreset;
    var
      i : Integer;
    begin
      FPattern := 0;

      for i := i_pre1 to i_pre8
      do begin
        FPattern := FPattern shr 1;

        if SignalToLogic( FInputs[ i])
        then FPattern := FPattern or 128;
      end;
    end;


    procedure   TModKleeBit.Step;
    var
      InBit   : Boolean;
      Carry   : Boolean;
      i       : Integer;
      Bit     : Boolean;
      MainLev : TSignal;
    begin
      FInvertMode := SignalToLogic( FInputs[ i_invertmode]) xor SignalToLogic( FInputs[ i_invertinvert]);
      FLoopMode   := SignalToLogic( FInputs[ i_loopmode  ]) xor SignalToLogic( FInputs[ i_loopinvert  ]);
      Carry       := ( FPattern and 128 <> 0) xor FInvertMode;

      if FLoopMode
      then InBit := Carry
      else InBit := SignalToLogic( FInputs[ i_inbit]);

      FPattern := FPattern shl 1;

      if Inbit
      then FPattern := FPattern or 1;

      FBus1   := SignalToLogic( FInputs[ i_inbus1   ]);
      FBus2   := SignalToLogic( FInputs[ i_inbus2   ]);
      FBus3   := SignalToLogic( FInputs[ i_inbus3   ]);
      FAnalog :=                FInputs[ i_inanalog ];
      MainLev :=                FInputs[ i_mainlevel] * FInputs[ i_mainlevelmod];

      for i := 0 to 7
      do begin
        Bit := FPattern and ( 1 shl i) <> 0;

        if Bit
        then FAnalog := FAnalog + FInputs[ i_level1 + i] * MainLev;

        case Round( FInputs[ i_bussel1 + i]) of
          0 : FBus1 := FBus1 or Bit;
          1 : FBus2 := FBus2 or Bit;
          2 : FBus3 := FBus3 or Bit;
        end;
      end;

      FOutputs[ o_outbit   ] := LogicToSignal( Carry);
      FOutputs[ o_outanalog] := FAnalog;
      FOutputs[ o_outbus1  ] := LogicToSignal( FBus1      );
      FOutputs[ o_outbus2  ] := LogicToSignal( FBus2      );
      FOutputs[ o_outbus3  ] := LogicToSignal( FBus3      );
      FOutputs[ o_outloop  ] := LogicToSignal( FLoopMode  );
      FOutputs[ o_outinvert] := LogicToSignal( FInvertMode);
    end;


//  public

    procedure   TModKleeBit.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_instep      , 'instep'      );
      AddInput ( i_inload      , 'inload'      );
      AddInput ( i_loopmode    , 'loopmode'    );
      AddInput ( i_inanalog    , 'inanalog'    );
      AddInput ( i_inbus1      , 'inbus1'      );
      AddInput ( i_inbus2      , 'inbus2'      );
      AddInput ( i_inbus3      , 'inbus3'      );
      AddInput ( i_pre1        , 'pre1'        );
      AddInput ( i_pre2        , 'pre2'        );
      AddInput ( i_pre3        , 'pre3'        );
      AddInput ( i_pre4        , 'pre4'        );
      AddInput ( i_pre5        , 'pre5'        );
      AddInput ( i_pre6        , 'pre6'        );
      AddInput ( i_pre7        , 'pre7'        );
      AddInput ( i_pre8        , 'pre8'        );
      AddInput ( i_level1      , 'level1'      );
      AddInput ( i_level2      , 'level2'      );
      AddInput ( i_level3      , 'level3'      );
      AddInput ( i_level4      , 'level4'      );
      AddInput ( i_level5      , 'level5'      );
      AddInput ( i_level6      , 'level6'      );
      AddInput ( i_level7      , 'level7'      );
      AddInput ( i_level8      , 'level8'      );
      AddInput ( i_bussel1     , 'bussel1'     );
      AddInput ( i_bussel2     , 'bussel2'     );
      AddInput ( i_bussel3     , 'bussel3'     );
      AddInput ( i_bussel4     , 'bussel4'     );
      AddInput ( i_bussel5     , 'bussel5'     );
      AddInput ( i_bussel6     , 'bussel6'     );
      AddInput ( i_bussel7     , 'bussel7'     );
      AddInput ( i_bussel8     , 'bussel8'     );
      AddInput ( i_inbit       , 'inbit'       );
      AddInput ( i_invertmode  , 'invertmode'  );
      AddInput ( i_mainlevel   , 'mainlevel'   );
      AddInput ( i_loopinvert  , 'loopinvert'  );
      AddInput ( i_invertinvert, 'invertinvert');
      AddInput ( i_mainlevelmod, 'mainlevelmod');
      AddOutput( o_outanalog   , 'outanalog'   );
      AddOutput( o_outbus1     , 'outbus1'     );
      AddOutput( o_outbus2     , 'outbus2'     );
      AddOutput( o_outbus3     , 'outbus3'     );
      AddOutput( o_outbit      , 'outbit'      );
      AddOutput( o_outloop     , 'outloop'     );
      AddOutput( o_outinvert   , 'outinvert'   );
    end;


    procedure   TModKleeBit.SetDefaults; // override;
    begin
      FInputs[ i_inanalog    ] := 0.0;
      FInputs[ i_inbus1      ] := 0.0;
      FInputs[ i_inbus2      ] := 0.0;
      FInputs[ i_inbus3      ] := 0.0;
      FInputs[ i_instep      ] := 0.0;
      FInputs[ i_inload      ] := 0.0;
      FInputs[ i_loopinvert  ] := 0.0;
      FInputs[ i_invertinvert] := 0.0;
      FInputs[ i_mainlevelmod] := 1.0;
      LoadPreset;
    end;


    procedure   TModKleeBit.DoSlowTick; // override;
    var
      aValue : Boolean;
    begin
      if ResetFlag
      then SetDefaults;

      ResetFlag := False;

      aValue := SignalToLogic( FInputs[ i_inload]);
      if not FOldLoad and aValue
      then LoadPreset;

      FOldLoad := aValue;

      aValue := SignalToLogic( FInputs[ i_instep]);

      if not FOldStep and aValue
      then Step;

      FOldStep := aValue;
    end;


    procedure   TModKleeBit.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step1' , LogicToSignal( FPattern and   1 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step2' , LogicToSignal( FPattern and   2 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step3' , LogicToSignal( FPattern and   4 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step4' , LogicToSignal( FPattern and   8 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step5' , LogicToSignal( FPattern and  16 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step6' , LogicToSignal( FPattern and  32 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step7' , LogicToSignal( FPattern and  64 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step8' , LogicToSignal( FPattern and 128 <> 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'loop'  , LogicToSignal( FLoopMode            )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'invert', LogicToSignal( FInvertMode          )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'bus1'  , FOutputs[ o_outbus1]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'bus2'  , FOutputs[ o_outbus2]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'bus3'  , FOutputs[ o_outbus3]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'outbit', FOutputs[ o_outbit ]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step'  , FInputs [ i_instep ]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'load'  , FInputs [ i_inload ]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'inbit' , FInputs [ i_inbit  ]                 ));
    end;


{ ========
  TModSeqRandom = class( TMod)
  private
    FValues         : array[ Byte] of TScaleNote;
    FPattern        : array[ Byte] of TScaleNote;
    FCount          : Integer;
    FScale          : Integer;
    FBaseNote       : TScaleNote;
    FLowNote        : TScaleNote;
    FHighNote       : TScaleNote;
    FNoteCount      : Integer;
    FScaleNotes     : array[ TScaleNote] of TScaleNote;
    FOldTrig        : Boolean;
    FOldOne         : Boolean;
    FOldAll         : Boolean;
    FPosition       : Byte;
    FDistribution   : TSignal;
    FTrigCount      : Integer;
    FChangeOneCount : Integer;
    FChangeAllCount : Integer;
  private
}

    procedure   TModSeqRandom.MakeScale;
    var
      aScale : TScaleNoteSet;
      i      : TScaleNote;
    begin
      aScale     := TransPoseAndCompleteScale( SystemScales.ScaleNotes[ FScale], FLowNote, FHighNote, FBaseNote, 12);
      FNoteCount := 0;

      for i := Low( i) to High( i)
      do begin
        if i in aScale
        then begin
          FScaleNotes[ FNoteCount] := i;
          Inc( FNoteCount);
        end;
      end;
    end;


    function    TModSeqRandom.RandomNote( aPosition: Byte; aDistribution: TSignal): TScaleNote;
    var
      aRandomValue : TScaleNote;
      anOldValue   : TScaleNote;
      aNewValue    : TScaleNote;
    begin
      aRandomValue := Random( FNoteCount);
      anOldValue   := FValues[ aPosition];
      aNewValue    := Round( aDistribution * aRandomValue + ( 1 - aDistribution) * anOldValue);
      Result       := FScaleNotes[ aNewValue];
      FValues[ aPosition] := aNewValue;
    end;


    procedure   TModSeqRandom.ChangeScale( aScale: Integer);
    begin
      FScale := aScale;
      MakeScale;
      ChangeAllNotes;
    end;


    procedure   TModSeqRandom.ChangeOneNote( aPosition: Byte);
    begin
      FPattern[ FPosition] := RandomNote( aPosition, FDistribution); // correlated random selection
    end;


    procedure   TModSeqRandom.ChangeAllNotes;
    var
      i : Integer;
    begin
      for i := Low( FPattern) to High( FPattern)
      do FPattern[ i] := RandomNote( i, 1.0);       // totally random selection
    end;


    procedure   TModSeqRandom.Step;
    begin
      if FPosition < FCount - 1
      then Inc( FPosition)
      else FPosition := Low( FPosition);
    end;


//  public

    procedure   TModSeqRandom.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_changeone , 'changeone' );
      AddInput ( i_scale     , 'scale'     );
      AddInput ( i_basenote  , 'basenote'  );
      AddInput ( i_lownote   , 'lownote'   );
      AddInput ( i_highnote  , 'highnote'  );
      AddInput ( i_distmod   , 'distmod'   );
      AddInput ( i_distmodamt, 'distmodamt');
      AddInput ( i_dist      , 'dist'      );
      AddInput ( i_chain     , 'chain'     );
      AddInput ( i_active    , 'active'    );
      AddInput ( i_scalemod  , 'scalemod'  );
      AddOutput( o_out       , 'out'       );
      AddInput ( i_changeall , 'changeall' );
    end;


    procedure   TModSeqRandom.SetDefaults; // override;
    begin
      ChangeScale( 1);
      ChangeScale( 0);
      FInputs[ i_active] := 1.0;
    end;


    procedure   TModSeqRandom.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'count')
      then FCount := Clip( Ceil( aValue), 1, 256)
      else inherited;
    end;


    function    TModSeqRandom.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'count')
      then Result := FCount
      else Result := inherited;
    end;


    procedure   TModSeqRandom.DoSlowTick; // override;
    var
      aBool       : Boolean;
      anInt       : Integer;
      aChangeOne  : Boolean;
      aChangeAll  : Boolean;
      aScale      : Integer;
      anActive    : Boolean;
      aModulation : Integer;
    begin
      if ResetFlag
      then begin
        FPosition := 0;
        ChangeAllNotes;
      end;


      ResetFlag     := False;
      FDistribution := Clip( FInputs[ i_dist] + FInputs[ i_distmod] * FInputs[ i_distmodamt], 0.0, 1.0);
      anActive      := SignalToLogic( FInputs[ i_active]);
      aBool         := False;
      anInt         := Round( 12 * FInputs[ i_basenote]);

      if anInt <> FBaseNote
      then begin
        aBool     := True;
        FBaseNote := anInt;
      end;

      anInt := Round( UnitsToNoteNumber( FInputs[ i_lownote]));

      if anInt <> FLowNote
      then begin
        aBool    := True;
        FLowNote := anInt;
      end;

      anInt := Round( UnitsToNoteNumber( FInputs[ i_highnote]));

      if anInt <> FHighNote
      then begin
        aBool     := True;
        FHighNote := anInt;
      end;

      aModulation := Round(( FInputs[ i_scalemod] - 0.5 * SystemScales.CountRec) * ( SystemScales.Count - 1));
      aScale      := ( Round( FInputs[ i_scale]) + aModulation ) mod SystemScales.Count;
   // aScale := ( Round( FInputs[ i_scale]) + Round(( FInputs[ i_scalemod] - SystemScales.CountRec) * ( SystemScales.Count - 1))) mod SystemScales.Count;

      if aBool or ( aScale <> FScale)
      then ChangeScale( aSCale);

      aBool := SignalToLogic( FInputs[ i_trig]) and anActive;

      if not FOldTrig and aBool
      then begin
        FTrigCount := LedFlashTime( IsSpedUp);
        aChangeOne := SignalToLogic( FInputs[ i_changeone]);
        aChangeAll := SignalToLogic( FInputs[ i_changeall]);

        if not FOldOne and aChangeOne
        then begin
          FChangeOneCount := LedFlashTime( IsSpedUp);
          ChangeOneNote( FPosition);
        end;

        if not FOldAll and aChangeAll
        then begin
          FChangeAllCount := LedFlashTime( IsSpedUp);
          ChangeAllNotes;
        end;

        FOldOne := aChangeOne;
        FOldAll := aChangeAll;
        Step;
      end;

      DecToZero( FTrigCount);
      DecToZero( FChangeOneCount);
      DecToZero( FChangeAllCount);
      FOldTrig := aBool;

      if anActive
      then FOutputs[ o_out] := NoteNumberToUnits( FPattern[ FPosition] - MiddleNote) + FInputs[ i_chain]
      else FOutputs[ o_out] := FInputs[ i_chain];
    end;


    procedure   TModSeqRandom.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'     , LogicToSignal( FTrigCount      > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'changeone', LogicToSignal( FChangeOneCount > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'changeall', LogicToSignal( FChangeAllCount > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active'   , FInputs[ i_active]                 ));
    end;


    procedure   TModSeqRandom.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'scale', SystemScales.ScaleName[ FScale]));
    end;


{ ========
  TModSeqRandVal = class( TMod)
  private
    FValues       : array[ Byte] of TSignal;
    FCount        : Integer;
    FLowValue     : TSignal;
    FHighValue    : TSignal;
    FOldTrig      : Boolean;
    FOldOne       : Boolean;
    FPosition     : Byte;
    FDistribution : TSignal;
    FOldValue     : TSignal;
    FTrigCount    : Integer;
    FChangeCount  : Integer;
  private
}

    function    TModSeqRandVal.RandomValue( aPosition: Byte; aDistribution: TSignal): TSignal;
    var
      aRandomValue : TSignal;
      anOldValue   : TSignal;
    begin
      aRandomValue := Random;
      anOldValue   := FValues[ aPosition];
      Result       := aDistribution * aRandomValue + ( 1 - aDistribution) * anOldValue;
    end;


    function    TModSeqRandVal.ScaledValue( aPosition: Byte): TSignal;
    begin
      Result := FValues[ aPosition] * ( FHighValue - FLowValue) + FLowValue;
    end;


    procedure   TModSeqRandVal.ChangeOneValue( aPosition: Byte);
    begin
      FValues[ FPosition] := RandomValue( aPosition, FDistribution); // correlated random selection
    end;


    procedure   TModSeqRandVal.Step;
    begin
      if FPosition < FCount - 1
      then Inc( FPosition)
      else FPosition := Low( FPosition);
    end;


//  public

    procedure   TModSeqRandVal.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_trig      , 'trig'      );
      AddInput ( i_changeone , 'changeone' );
      AddInput ( i_basevalue , 'basevalue' );
      AddInput ( i_lowvalue  , 'lowvalue'  );
      AddInput ( i_highvalue , 'highvalue' );
      AddInput ( i_distmod   , 'distmod'   );
      AddInput ( i_distmodamt, 'distmodamt');
      AddInput ( i_dist      , 'dist'      );
      AddInput ( i_chain     , 'chain'     );
      AddInput ( i_quantize  , 'quantize'  );
      AddInput ( i_qsteps    , 'qsteps'    );
      AddInput ( i_slew      , 'slew'      );
      AddInput ( i_slewmode  , 'slewmode'  );
      AddInput ( i_active    , 'active'    );
      AddOutput( o_out       , 'out'       );
    end;


    procedure   TModSeqRandVal.SetDefaults; // override;
    var
      i : Integer;
    begin
      FDezipperMap := [
        i_basevalue ,
        i_lowvalue  ,
        i_highvalue ,
        i_distmodamt,
        i_dist      ,
        i_slew
      ];

      for i := Low( FValues) to High( FValues)
      do FValues[ i] := Random;

      FInputs[ i_active] := 1.0;
    end;


    procedure   TModSeqRandVal.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if SameText( aName, 'count')
      then FCount := Clip( Ceil( aValue), 1, 256)
      else inherited;
    end;


    function    TModSeqRandVal.GetInternal( const aName: string): TSignal; // override;
    begin
      if SameText( aName, 'count')
      then Result := FCount
      else Result := inherited;
    end;


    procedure   TModSeqRandVal.DoSlowTick; // override;
    var
      aBool    : Boolean;
      aChange  : Boolean;
      Steps    : Integer;
      Slew     : TSignal;
      SlewMod  : Integer;
      NewValue : TSignal;
      Active   : Boolean;
    begin
      if ResetFlag
      then FPosition := 0;

      ResetFlag     := False;
      FDistribution := Clip( FInputs[ i_dist] + FInputs[ i_distmod] * FInputs[ i_distmodamt], 0.0, 1.0);
      FLowValue     := FInputs[ i_lowvalue ];
      FHighValue    := FInputs[ i_highvalue];
      Active        := SignalToLogic( FInputs[ i_active]);
      aBool         := SignalToLogic( FInputs[ i_trig]);

      if not FOldTrig and aBool and Active
      then begin
        FTrigCount := LedFlashTime( IsSpedUp);
        aChange    := SignalToLogic( FInputs[ i_changeone]);

        if not FOldOne and aChange
        then begin
          FChangeCount := LedFlashTime( IsSpedUp);
          ChangeOneValue( FPosition);
        end;

        FOldOne := aChange;
        Step;
      end;

      FOldTrig := aBool;
      Slew     := SlewToCurve( FInputs[ i_slew]);
      SlewMod  := Round( FInputs[ i_slewmode]);
      NewValue := ScaledValue( FPosition) + FInputs[ i_basevalue];

      // FOutputs[ o_noteout0] := Normalize( Out0 + Slew * ( FOutputs[ o_noteout0] - Out0));

      if SlewMod = 1                                                      // Filter is pre-quantization
      then begin
        NewValue  := Normalize( NewValue + Slew * ( FOldValue - NewValue));
        FOldValue := NewValue;
      end;

      if Round( FInputs[ i_quantize]) = 1                                 // Quantization is on
      then begin
        Steps    := Max( 1, Round( FInputs[ i_qsteps]));
        NewValue := Round( Steps * NewValue ) / Steps;
      end;

      if SlewMod = 0                                                      // Filter is post-quantization
      then begin
        NewValue  := Normalize( NewValue + Slew * ( FOldValue - NewValue));
        FOldValue := NewValue;
      end;

      DecToZero( FTrigCount);
      DecToZero( FChangeCount);

      if Active
      then FOutputs[ o_out] := NewValue + Normalize( FInputs[ i_chain])
      else FOutputs[ o_out] :=            Normalize( FInputs[ i_chain]);
    end;


    procedure   TModSeqRandVal.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'     , LogicToSignal( FTrigCount   > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'changeone', LogicToSignal( FChangeCount > 0)));
      aCallback( Self, MakeInfo( aPrefix, Name, 'active'   , FInputs[ i_active]              ));
    end;


{ ========
  TModRewriter = class( TMod)
  private
    FSteps     : Integer;
    FRuleA     : string;
    FRuleB     : string;
    FRuleC     : string;
    FRuleD     : string;
    FRuleE     : string;
    FPatterns  : array[ 0 .. 8] of string;
    FStep      : Integer;
    FSymbol    : Char;
    FOldStep   : Boolean;
    FTrigCount : Integer;
  public
    property    Steps: Integer read FSteps write SetSteps;
    property    RuleA: string  read FRuleA write SetRuleA;
    property    RuleB: string  read FRuleB write SetRuleB;
    property    RuleC: string  read FRuleC write SetRuleC;
    property    RuleD: string  read FRuleD write SetRuleD;
    property    RuleE: string  read FRuleE write SetRuleE;
  private
}

    procedure   TModRewriter.MakePatterns;
    var
      i : Integer;
      j : Integer;
    begin
      FPatterns[ Low( FPatterns)] := 'A';

      for i := Low( FPatterns) + 1 to High( FPatterns)
      do begin
        FPatterns[ i] := '';

        for j := Low( FPatterns[ i - 1]) to High( FPatterns[ i - 1])
        do begin
          case Fpatterns[ i - 1, j] of
            'A' : FPatterns[ i] := FPatterns[ i] + RuleA;
            'B' : FPatterns[ i] := FPatterns[ i] + RuleB;
            'C' : FPatterns[ i] := FPatterns[ i] + RuleC;
            'D' : FPatterns[ i] := FPatterns[ i] + RuleD;
            'E' : FPatterns[ i] := FPatterns[ i] + RuleE;
          end;
        end;
      end;
    end;


    function    TModRewriter.StripString( const aValue: string): string;
    var
      C : Char;
    begin
      Result := '';

      for C in aValue
      do begin
        if CharInSet( C, ['a' .. 'e', 'A' .. 'E'])
        then Result := Result + UpCase( C);
      end;
    end;


//  private

    procedure   TModRewriter.SetSteps( aValue: Integer);
    begin
      if aValue <> FSteps
      then begin
        FSteps := aValue;
        MakePatterns;
      end;
    end;


    procedure   TModRewriter.SetRuleA( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleA
      then begin
        FRuleA := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriter.SetRuleB( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleB
      then begin
        FRuleB := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriter.SetRuleC( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleC
      then begin
        FRuleC := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriter.SetRuleD( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleD
      then begin
        FRuleD := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriter.SetRuleE( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleE
      then begin
        FRuleE := S;
        MakePatterns;
      end;
    end;


//  public

    procedure   TModRewriter.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_step    , 'step'    );
      AddInput ( i_steps   , 'steps'   );
      AddInput ( i_stepsmod, 'stepsmod');
      AddInput ( i_mode    , 'mode'    );
      AddOutput( o_a       , 'a'       );
      AddOutput( o_b       , 'b'       );
      AddOutput( o_c       , 'c'       );
      AddOutput( o_d       , 'd'       );
      AddOutput( o_e       , 'e'       );
    end;


    procedure   TModRewriter.SetDefaults; // override;
    begin
      FInputs[ i_stepsmod] := 1;
    end;


    procedure   TModRewriter.SetStringValue( const aName, aValue: string); // override;
    begin
      if      SameText( aName, 'a')
      then RuleA := aValue
      else if SameText( aName, 'b')
      then RuleB := aValue
      else if SameText( aName, 'c')
      then RuleC := aValue
      else if SameText( aName, 'd')
      then RuleD := aValue
      else if SameText( aName, 'e')
      then RuleE := aValue
      else inherited;
    end;


    function    TModRewriter.GetStringValue( const aName: string): string; // override;
    begin
      if      SameText( aName, 'a')
      then Result := RuleA
      else if SameText( aName, 'b')
      then Result := RuleB
      else if SameText( aName, 'c')
      then Result := RuleC
      else if SameText( aName, 'd')
      then Result := RuleD
      else if SameText( aName, 'e')
      then Result := RuleE
      else Result := inherited;
    end;


    procedure   TModRewriter.DoSlowTick; // override;
    var
      Step  : Boolean;
      InVal : TSignal;
      Pat   : Integer;
    begin
      Pat := Clip( Round( FInputs[ i_steps] * FInputs[ i_stepsmod]), 0, 8);

      if ResetFlag
      then FStep := 0;

      if Length( FPatterns[ Pat]) > 0
      then begin
        InVal := FInputs[ i_step];
        Step  := SignalToLogic( InVal);

        case Round( FInputs[ i_mode ]) of
          1  : Inval := LogicToSignal( Step);                 // pulse  mode
          2  : begin                                          // signal mode
              if not Step
              then InVal := LogicToSignal( False);
            end;
          else InVal := LogicToSignal( True);                 // square mode
        end;

        if Step and not FOldStep
        then begin
          if ( FStep > High( FPatterns[ Pat])) or ( FStep < Low( FPatterns[ Pat]))
          then FStep := Low( FPatterns[ Pat]);

          FTrigCount := LedFlashTime( IsSpedUp);
          FSymbol    := FPatterns[ Pat, FStep];
          Inc( FStep);
        end;

        InVal          := Clip( InVal, 0, 1);
        FOutputs[ o_a] := Inval * LogicToSignal( FSymbol = 'A');
        FOutputs[ o_b] := Inval * LogicToSignal( FSymbol = 'B');
        FOutputs[ o_c] := Inval * LogicToSignal( FSymbol = 'C');
        FOutputs[ o_d] := Inval * LogicToSignal( FSymbol = 'D');
        FOutputs[ o_e] := Inval * LogicToSignal( FSymbol = 'E');
        FOldStep       := Step;
      end;

      ResetFlag := False;
      DecToZero( FTrigCount);
    end;


    procedure   TModRewriter.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step', LogicToSignal( FTrigCount >  0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'a'   , LogicToSignal( FSymbol    = 'A')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'b'   , LogicToSignal( FSymbol    = 'B')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'c'   , LogicToSignal( FSymbol    = 'C')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'd'   , LogicToSignal( FSymbol    = 'D')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'e'   , LogicToSignal( FSymbol    = 'E')));
    end;


{ ========
  TModRewriterNote = class( TMod)
  private
    FSteps     : Integer;
    FRuleA     : string;
    FRuleB     : string;
    FRuleC     : string;
    FRuleD     : string;
    FRuleE     : string;
    FRuleF     : string;
    FRuleG     : string;
    FRuleH     : string;
    FPatterns  : array[ 0 .. 12] of string;
    FStep      : Integer;
    FSymbol    : Char;
    FOldStep   : Boolean;
    FTrigCount : Integer;
  public
    property    Steps: Integer read FSteps write SetSteps;
    property    RuleA: string  read FRuleA write SetRuleA;
    property    RuleB: string  read FRuleB write SetRuleB;
    property    RuleC: string  read FRuleC write SetRuleC;
    property    RuleD: string  read FRuleD write SetRuleD;
    property    RuleE: string  read FRuleE write SetRuleE;
    property    RuleF: string  read FRuleF write SetRuleF;
    property    RuleG: string  read FRuleG write SetRuleG;
    property    RuleH: string  read FRuleH write SetRuleH;
  private
}

    procedure   TModRewriterNote.MakePatterns;
    var
      i : Integer;
      j : Integer;
    begin
      FPatterns[ Low( FPatterns)] := 'A';

      for i := Low( FPatterns) + 1 to High( FPatterns)
      do begin
        FPatterns[ i] := '';

        for j := Low( FPatterns[ i - 1]) to High( FPatterns[ i - 1])
        do begin
          case Fpatterns[ i - 1, j] of
            'A' : FPatterns[ i] := FPatterns[ i] + RuleA;
            'B' : FPatterns[ i] := FPatterns[ i] + RuleB;
            'C' : FPatterns[ i] := FPatterns[ i] + RuleC;
            'D' : FPatterns[ i] := FPatterns[ i] + RuleD;
            'E' : FPatterns[ i] := FPatterns[ i] + RuleE;
            'F' : FPatterns[ i] := FPatterns[ i] + RuleF;
            'G' : FPatterns[ i] := FPatterns[ i] + RuleG;
            'H' : FPatterns[ i] := FPatterns[ i] + RuleH;
          end;
        end;
      end;
    end;


    function    TModRewriterNote.StripString( const aValue: string): string;
    var
      C : Char;
    begin
      Result := '';

      for C in aValue
      do begin
        if CharInSet( C, ['a' .. 'h', 'A' .. 'H'])
        then Result := Result + UpCase( C);
      end;
    end;


//  private

    procedure   TModRewriterNote.SetSteps( aValue: Integer);
    begin
      if aValue <> FSteps
      then begin
        FSteps := aValue;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleA( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleA
      then begin
        FRuleA := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleB( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleB
      then begin
        FRuleB := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleC( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleC
      then begin
        FRuleC := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleD( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleD
      then begin
        FRuleD := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleE( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleE
      then begin
        FRuleE := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleF( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleF
      then begin
        FRuleF := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleG( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleG
      then begin
        FRuleG := S;
        MakePatterns;
      end;
    end;


    procedure   TModRewriterNote.SetRuleH( const aValue: string);
    var
      S : string;
    begin
      S := StripString( aValue);

      if S <> FRuleH
      then begin
        FRuleH := S;
        MakePatterns;
      end;
    end;


//   public

    procedure   TModRewriterNote.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_step    , 'step'    );
      AddInput ( i_steps   , 'steps'   );
      AddInput ( i_stepsmod, 'stepsmod');
      AddInput ( i_notea   , 'notea'   );
      AddInput ( i_noteb   , 'noteb'   );
      AddInput ( i_notec   , 'notec'   );
      AddInput ( i_noted   , 'noted'   );
      AddInput ( i_notee   , 'notee'   );
      AddInput ( i_notef   , 'notef'   );
      AddInput ( i_noteg   , 'noteg'   );
      AddInput ( i_noteh   , 'noteh'   );
      AddInput ( i_chain   , 'chain'   );
      AddInput ( i_mode    , 'mode'    );
      AddInput ( i_chaina  , 'chaina'  );
      AddInput ( i_chainb  , 'chainb'  );
      AddInput ( i_chainc  , 'chainc'  );
      AddInput ( i_chaind  , 'chaind'  );
      AddInput ( i_chaine  , 'chaine'  );
      AddInput ( i_chainf  , 'chainf'  );
      AddInput ( i_chaing  , 'chaing'  );
      AddInput ( i_chainh  , 'chainh'  );
      AddOutput( o_out     , 'out'     );
    end;


    procedure   TModRewriterNote.SetDefaults; // override;
    begin
      FInputs[ i_stepsmod] := 1;
    end;


    procedure   TModRewriterNote.SetStringValue( const aName, aValue: string); // override;
    begin
      if      SameText( aName, 'a')
      then RuleA := aValue
      else if SameText( aName, 'b')
      then RuleB := aValue
      else if SameText( aName, 'c')
      then RuleC := aValue
      else if SameText( aName, 'd')
      then RuleD := aValue
      else if SameText( aName, 'e')
      then RuleE := aValue
      else if SameText( aName, 'f')
      then RuleF := aValue
      else if SameText( aName, 'g')
      then RuleG := aValue
      else if SameText( aName, 'h')
      then RuleH := aValue
      else inherited;
    end;


    function    TModRewriterNote.GetStringValue( const aName: string): string; // override;
    begin
      if      SameText( aName, 'a')
      then Result := RuleA
      else if SameText( aName, 'b')
      then Result := RuleB
      else if SameText( aName, 'c')
      then Result := RuleC
      else if SameText( aName, 'd')
      then Result := RuleD
      else if SameText( aName, 'e')
      then Result := RuleE
      else if SameText( aName, 'f')
      then Result := RuleF
      else if SameText( aName, 'g')
      then Result := RuleG
      else if SameText( aName, 'h')
      then Result := RuleH
      else Result := inherited;
    end;


    procedure   TModRewriterNote.DoSlowTick; // override;
    var
      Step  : Boolean;
      Pat   : Integer;
    begin
      Pat := Clip( Round( FInputs[ i_steps] * FInputs[ i_stepsmod]), 0, 8);

      if ResetFlag
      then FStep := 0;

      if Length( FPatterns[ Pat]) > 0
      then begin
        Step  := SignalToLogic( FInputs[ i_step]);

        if Step and not FOldStep
        then begin
          if ( FStep > High( FPatterns[ Pat])) or ( FStep < Low( FPatterns[ Pat]))
          then FStep := Low( FPatterns[ Pat]);

          FTrigCount := LedFlashTime( IsSpedUp);
          FSymbol    := FPatterns[ Pat, FStep];

          if Round( FInputs[ i_mode]) = m_note
          then begin
            case FSymbol of
              'A' : FOutputs[ o_out] := FInputs[ i_notea] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chaina];
              'B' : FOutputs[ o_out] := FInputs[ i_noteb] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chainb];
              'C' : FOutputs[ o_out] := FInputs[ i_notec] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chainc];
              'D' : FOutputs[ o_out] := FInputs[ i_noted] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chaind];
              'E' : FOutputs[ o_out] := FInputs[ i_notee] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chaine];
              'F' : FOutputs[ o_out] := FInputs[ i_notef] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chainf];
              'G' : FOutputs[ o_out] := FInputs[ i_noteg] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chaing];
              'H' : FOutputs[ o_out] := FInputs[ i_noteh] - NoteNumberToUnits( MiddleNote) + FInputs[ i_chain] + FInputs[ i_chainh];
            end;
          end
          else begin
            case FSymbol of
              'A' : FOutputs[ o_out] := FInputs[ i_notea] + FInputs[ i_chain] + FInputs[ i_chaina];
              'B' : FOutputs[ o_out] := FInputs[ i_noteb] + FInputs[ i_chain] + FInputs[ i_chainb];
              'C' : FOutputs[ o_out] := FInputs[ i_notec] + FInputs[ i_chain] + FInputs[ i_chainc];
              'D' : FOutputs[ o_out] := FInputs[ i_noted] + FInputs[ i_chain] + FInputs[ i_chaind];
              'E' : FOutputs[ o_out] := FInputs[ i_notee] + FInputs[ i_chain] + FInputs[ i_chaine];
              'F' : FOutputs[ o_out] := FInputs[ i_notef] + FInputs[ i_chain] + FInputs[ i_chainf];
              'G' : FOutputs[ o_out] := FInputs[ i_noteg] + FInputs[ i_chain] + FInputs[ i_chaing];
              'H' : FOutputs[ o_out] := FInputs[ i_noteh] + FInputs[ i_chain] + FInputs[ i_chainh];
            end;
          end;

          Inc( FStep);
        end;

        FOldStep  := Step;
      end;

      ResetFlag := False;
      DecToZero( FTrigCount);
    end;


    procedure   TModRewriterNote.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step', LogicToSignal( FTrigCount >  0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'a'   , LogicToSignal( FSymbol    = 'A')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'b'   , LogicToSignal( FSymbol    = 'B')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'c'   , LogicToSignal( FSymbol    = 'C')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'd'   , LogicToSignal( FSymbol    = 'D')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'e'   , LogicToSignal( FSymbol    = 'E')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'f'   , LogicToSignal( FSymbol    = 'F')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'g'   , LogicToSignal( FSymbol    = 'G')));
      aCallback( Self, MakeInfo( aPrefix, Name, 'h'   , LogicToSignal( FSymbol    = 'H')));
    end;


{ ========
  TModTuneSmithy = class( TMod)
  private
    FSeed       : string;
    FTuneSmithy : TTuneSmithy;
    FStep       : Integer;
    FOldStep    : Boolean;
    FTrigCount  : Integer;
  public
    property    Seed: string read FSeed write SetSeed;
  private
}

    procedure   TModTuneSmithy.Recreate;
    var
      aNew  : TTuneSmithy;
      anOld : TTuneSmithy;
    begin
      if Length( Seed) > 0
      then aNew := TTuneSmithy.Create( Seed)
      else aNew := nil;

      Locked := True;

      try
        anOld := AtomicExchange( Pointer( FTuneSmithy), Pointer( aNew));
      finally
        Locked := False;
        FreeAndNil( anOld);
      end;
    end;


    procedure   TModTuneSmithy.SetSeed( const aValue: string);
    begin
      if aValue <> FSeed
      then begin
        FSeed := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModTuneSmithy.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FSeed := '';
    end;


    procedure   TModTuneSmithy.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_step , 'step' );
      AddInput ( i_steps, 'steps');
      AddInput ( i_chain, 'chain');
      AddOutput( o_out  , 'out'  );
    end;


    destructor  TModTuneSmithy.Destroy; // override;
    begin
      Seed := '';
      Inherited;
    end;


    procedure   TModTuneSmithy.SetStringValue( const aName, aValue: string); // override;
    begin
      if SameText( aName, 'seed')
      then Seed := aValue
      else inherited;
    end;


    function    TModTuneSmithy.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'seed')
      then Result := Seed
      else Result := inherited;
    end;


    procedure   TModTuneSmithy.DoSlowTick; // override;
    var
      Step  : Boolean;
      Steps : Integer;
    begin
      if ResetFlag
      then FStep := 0;

      Steps := Round( FInputs[ i_steps]);

      if Assigned( FTuneSmithy) and ( Length( FSeed) > 0)
      then begin
        Step := SignalToLogic( FInputs[ i_step]);

        if Step and not FOldStep
        then begin
          FTrigCount := LedFlashTime( IsSpedUp);
          Inc( FStep);
        end;

        if FStep >= Steps
        then FStep := 0;

        FOldStep := Step;
        FOutputs[ o_out] := NoteNumberToUnits( FTuneSmithy.Element( FStep)) + FInputs[ i_chain];
      end;


      ResetFlag := False;
      DecToZero( FTrigCount);
    end;


    procedure   TModTuneSmithy.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step', LogicToSignal( FTrigCount >  0 )));
    end;


{ ========
  TModNeuralPulser = class( TMod)
  private
    FOutValues : array[ 0 .. 7] of Boolean;
    FCounters  : array[ 0 .. 7] of Integer;
    FOldClock  : Boolean;
    FOldReset  : Boolean;
  public
}

    procedure   TModNeuralPulser.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_clock  , 'clock'  );
      AddInput ( i_reset  , 'reset'  );
      AddInput ( i_outmode, 'outmode');
      AddInput ( i_exc1a  , 'exc1a'  );
      AddInput ( i_exc1b  , 'exc1b'  );
      AddInput ( i_inh1   , 'inh1'   );
      AddInput ( i_mode1  , 'mode1'  );
      AddInput ( i_exc2a  , 'exc2a'  );
      AddInput ( i_exc2b  , 'exc2b'  );
      AddInput ( i_inh2   , 'inh2'   );
      AddInput ( i_mode2  , 'mode2'  );
      AddInput ( i_exc3a  , 'exc3a'  );
      AddInput ( i_exc3b  , 'exc3b'  );
      AddInput ( i_inh3   , 'inh3'   );
      AddInput ( i_mode3  , 'mode3'  );
      AddInput ( i_exc4a  , 'exc4a'  );
      AddInput ( i_exc4b  , 'exc4b'  );
      AddInput ( i_inh4   , 'inh4'   );
      AddInput ( i_mode4  , 'mode4'  );
      AddInput ( i_exc5a  , 'exc5a'  );
      AddInput ( i_exc5b  , 'exc5b'  );
      AddInput ( i_inh5   , 'inh5'   );
      AddInput ( i_mode5  , 'mode5'  );
      AddInput ( i_exc6a  , 'exc6a'  );
      AddInput ( i_exc6b  , 'exc6b'  );
      AddInput ( i_inh6   , 'inh6'   );
      AddInput ( i_mode6  , 'mode6'  );
      AddInput ( i_exc7a  , 'exc7a'  );
      AddInput ( i_exc7b  , 'exc7b'  );
      AddInput ( i_inh7   , 'inh7')  ;
      AddInput ( i_mode7  , 'mode7'  );
      AddInput ( i_exc8a  , 'exc8a'  );
      AddInput ( i_exc8b  , 'exc8b'  );
      AddInput ( i_inh8   , 'inh8'   );
      AddInput ( i_mode8  , 'mode8'  );
      AddOutput( o_out1   , 'out1'   );
      AddOutput( o_out2   , 'out2'   );
      AddOutput( o_out3   , 'out3'   );
      AddOutput( o_out4   , 'out4'   );
      AddOutput( o_out5   , 'out5'   );
      AddOutput( o_out6   , 'out6'   );
      AddOutput( o_out7   , 'out7'   );
      AddOutput( o_out8   , 'out8'   );
    end;


    procedure   TModNeuralPulser.DoSlowTick; // override;

      procedure ClockNeuron( inh, exca, excb, mode, counter: Integer);
      var
        aCount : Integer;
      begin
        aCount := 0;

        if( FInputs[ inh] >= 0.5)
        then begin
          Dec( aCount);
          FCounters[ counter] := 0;
        end
        else begin
          if FInputs[ exca] >= 0.5
          then Inc( aCount);

          if FInputs[ excb] >= 0.5
          then Inc( aCount);
        end;

        if aCount >= FInputs[ mode]
        then FCounters[ counter] := NeuronTime( IsSpedUp);
      end;

    var
      NewReset : Boolean;
      NewClock : Boolean;
      i        : Integer;
    begin
      NewReset := SignalToLogic( FInputs[ i_reset]);
      NewClock := SignalToLogic( FInputs[ i_clock]);

      if ResetFlag or ( NewReset and not FOldReset)
      then begin
        for i := 0 to 7 do
        begin
          FOutValues[ i] := False;
          FCounters [ i] := 0;
        end;
      end;

      if not FOldClock and NewClock
      then begin
        ClockNeuron( i_inh1, i_exc1a, i_exc1b, i_mode1, 0);
        ClockNeuron( i_inh2, i_exc2a, i_exc2b, i_mode2, 1);
        ClockNeuron( i_inh3, i_exc3a, i_exc3b, i_mode3, 2);
        ClockNeuron( i_inh4, i_exc4a, i_exc4b, i_mode4, 3);
        ClockNeuron( i_inh5, i_exc5a, i_exc5b, i_mode5, 4);
        ClockNeuron( i_inh6, i_exc6a, i_exc6b, i_mode6, 5);
        ClockNeuron( i_inh7, i_exc7a, i_exc7b, i_mode7, 6);
        ClockNeuron( i_inh8, i_exc8a, i_exc8b, i_mode8, 7);
      end;

      for i := 0 to 7
      do begin
        DecToZero( FCounters[ i]);
        FOutValues[ i] := FCounters[ i] > 0;
        FOutputs[ o_out1 + i] := SignalForOutputType( LogicToSignal( FOutValues[ i]), Round( FInputs[ i_outmode]));
      end;

      FOldReset := NewReset;
      FOldClock := NewClock;
      ResetFlag := False;
    end;


    procedure   TModNeuralPulser.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := Low( FOutValues) to High( FOutValues)
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'out%d', [ i + 1], AppLocale) , LogicToSignal( FOutValues[ i])));
    end;


{ ========
  TModCellAut = class( TMod)
  private
    FEngine     : TElementaryCA;
    FRule       : Integer;
    FSteps      : Integer;
    FFaults     : Integer;
    FSumCount   : Integer;
    FSumMult    : TSignal;
    FStep       : Integer;
    FStepCount  : Integer;
    FOldStep    : Boolean;
    FOldReset   : Boolean;
    FTrigCount  : Integer;
    FResetCount : Integer;
  public
    property    Rule     : Integer read FRule     write SetRule;
    property    Steps    : Integer read FSteps    write SetSteps;
    property    Faults   : Integer read FFaults   write setFaults;
    property    SumCount : Integer read FSumCount write SetSumCount;
  private
}

    procedure   TModCellAut.Recreate;
    var
      FOldEngine : TElementaryCA;
      FNewEngine : TElementaryCA;
      i          : Integer;
    begin
      FNewEngine := TElementaryCA.Create;
      FNewEngine.Rule      := FRule;
      FNewEngine.FaultRate := FFaults / 10000;
      FNewEngine.SumCount  := FSumCount;

      for i := 0 to FSteps - 1
      do FNewEngine.Execute;

      Locked := True;

      try
        FOldEngine := AtomicExchange( Pointer( FEngine), Pointer( FNewEngine));
        FStepCount := 1 + 2 * FSteps;
        FreeAndNil( FOldEngine);
      finally
        Locked := False;
      end;
    end;


    procedure   TModCellAut.SetSteps( aValue: Integer);
    begin
      if aValue <> FSteps
      then begin
        FSteps := aValue;
        Recreate;
      end;
    end;


    procedure   TModCellAut.SetRule( aValue: Integer);
    begin
      if aValue <> FRule
      then begin
        FRule := aValue;
        Recreate;
      end;
    end;


    procedure   TModCellAut.SetFaults( aValue: Integer);
    begin
      if aValue <> FFaults
      then begin
        FFaults := aValue;
        Recreate;
      end;
    end;


    procedure   TModCellAut.SetSumCount( aValue: Integer);
    begin
      if aValue <> FSumCount
      then begin
        FSumCount := aValue;
        FSumMult  := 1.0 / ( 1 shl aValue);
        Recreate;
      end;
    end;


//  public

    constructor TModCellAut.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FEngine  := TElementaryCA.Create;
      SumCount := 3;
    end;


    procedure   TModCellAut.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_step  , 'step'  );
      AddInput ( i_reset , 'reset' );
      AddInput ( i_dir   , 'dir'   );
      AddOutput( o_out   , 'out'   );
      AddOutput( o_outsum, 'outsum');
    end;


    destructor  TModCellAut.Destroy; // override;
    begin
      Locked := True;
      FreeAndNil( FEngine);
      inherited;
    end;


    procedure   TModCellAut.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if      SameText( aName, 'steps' ) then Steps    := Round( aValue)
      else if SameText( aName, 'rule'  ) then Rule     := Round( aValue)
      else if SameText( aName, 'faults') then Faults   := Round( aValue)
      else if SameText( aName, 'sumcnt') then SumCount := Round( aValue)
      else inherited;
    end;


    function    TModCellAut.GetInternal( const aName: string): TSignal; // override;
    begin
      if      SameText( aName, 'steps' ) then Result := Steps
      else if SameText( aName, 'rule'  ) then Result := Rule
      else if SameText( aName, 'faults') then Result := Faults
      else if SameText( aName, 'sumcnt') then Result := SumCount
      else Result := inherited;
    end;


    procedure   TModCellAut.DoSlowTick; // override;
    var
      Reset : Boolean;
      Step  : Boolean;
      Dir   : Boolean;
    begin
      if Assigned( FEngine)
      then begin
        Reset := SignalToLogic( FInputs[ i_reset]);
        Step  := SignalToLogic( FInputs[ i_step ]);

        if ResetFlag or ( not FOldReset and Reset)
        then begin
          FStep := 0;
          FResetCount := LedFlashTime( IsSpedUp);
        end;

        if not FOldStep and Step
        then begin
          Dir := SignalToLogic( FInputs[ i_dir   ]);

          if Dir
          then begin
            if ( FStep < 0) or ( FStep >= FStepCount)
            then FStep := FStepCount - 1;
          end
          else begin
            if ( FStep < 0) or ( FStep >= FStepCount)
            then FStep := 0;
          end;

          FTrigCount := LedFlashTime( IsSpedUp);
          FOutputs[ o_out]    := LogicToSignal( FEngine.Bit[ FStep] <> 0);
          FOutputs[ o_outsum] := FEngine.Sum[ FStep] * FSumMult;

          if Dir
          then begin
            if FStep > 0
            then Dec( FStep)
            else FStep := FStepCount - 1
          end
          else begin
            if FStep < FStepCount - 1
            then Inc( FStep)
            else FStep := 0;
          end
        end;

        DecToZero( FResetCount);
        DecToZero( FTrigCount );
        ResetFlag := False;
        FOldReset := Reset;
        FOldStep  := Step;;
      end;
    end;


    procedure   TModCellAut.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step' , LogicToSignal( FTrigCount  >  0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset', LogicToSignal( FResetCount >  0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'out'  , FOutputs[ o_out]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'dir'  , FInputs [ i_dir]                 ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'cstep', FStep + 1                        ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'steps', FStepCount                       ));
    end;


{ ========
  TModCellAut2 = class( TMod)
  private
    FEngine     : TElementaryCA;
    FRule       : Integer;
    FSteps      : Integer;
    FFaults     : Integer;
    FSumCount   : Integer;
    FSumMult    : TSignal;
    FStep       : Integer;
    FStepCount  : Integer;
    FOldStep    : Boolean;
    FOldReset   : Boolean;
    FTrigCount  : Integer;
    FResetCount : Integer;
  public
    property    Rule     : Integer read FRule     write SetRule;
    property    Steps    : Integer read FSteps    write SetSteps;
    property    Faults   : Integer read FFaults   write setFaults;
    property    SumCount : Integer read FSumCount write SetSumCount;
  private
}

    procedure   TModCellAut2.Recreate;
    var
      FOldEngine : TLimitedCA;
      FNewEngine : TLimitedCA;
    begin
      FNewEngine := TLimitedCA.Create( WIDTH, FSteps);
      FNewEngine.Rule := FRule;
      Locked := True;

      try
        FOldEngine := AtomicExchange( Pointer( FEngine), Pointer( FNewEngine));
        FreeAndNil( FOldEngine);
      finally
        Locked := False;
      end;
    end;


    procedure   TModCellAut2.SetRule( aValue: Integer);
    begin
      if aValue <> FRule
      then begin
        FRule        := aValue;
        FEngine.Rule := aValue;
      end;
    end;


    procedure   TModCellAut2.SetFaults( aValue: Integer);
    begin
      if aValue <> FFaults
      then begin
        FFaults           := aValue;
        FEngine.FaultRate := aValue / 10000;
      end;
    end;


    procedure   TModCellAut2.SetSteps( aValue: Integer);
    begin
      if aValue <> FSteps
      then begin
        FSteps := aValue;
        Recreate;
      end;
    end;


//  public

    constructor TModCellAut2.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FEngine  := TLimitedCA.Create( WIDTH, 16);
    end;


    procedure   TModCellAut2.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_step     , 'step'     );
      AddInput ( i_reset    , 'reset'    );
      AddInput ( i_rule     , 'rule'     );
      AddInput ( i_rulemod  , 'rulemod'  );
      AddInput ( i_faults   , 'faults'   );
      AddInput ( i_faultsmod, 'faultsmod');
      AddInput ( i_mode     , 'mode'     );
      AddOutput( o_out1     , 'out1'     );
      AddOutput( o_out2     , 'out2'     );
      AddOutput( o_out3     , 'out3'     );
      AddOutput( o_out4     , 'out4'     );
      AddOutput( o_out5     , 'out5'     );
      AddOutput( o_out6     , 'out6'     );
      AddOutput( o_out7     , 'out7'     );
      AddOutput( o_out8     , 'out8'     );
      AddOutput( o_out9     , 'out9'     );
      AddOutput( o_out10    , 'out10'    );
      AddOutput( o_out11    , 'out11'    );
      AddOutput( o_out12    , 'out12'    );
      AddOutput( o_out13    , 'out13'    );
      AddOutput( o_out14    , 'out14'    );
      AddOutput( o_out15    , 'out15'    );
    end;


    procedure   TModCellAut2.SetDefaults; // override;
    begin
      FInputs[ i_rulemod  ] := 1.0;
      FInputs[ i_faultsmod] := 1.0;
    end;


    destructor  TModCellAut2.Destroy; // override;
    begin
      Locked := True;
      FreeAndNil( FEngine);
      inherited;
    end;


    procedure   TModCellAut2.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if      SameText( aName, 'rule'  ) then Rule   := Round( aValue)
      else if SameText( aName, 'faults') then Faults := Round( aValue)
      else if SameText( aName, 'steps' ) then Steps  := Round( aValue)
      else inherited;
    end;


    function    TModCellAut2.GetInternal( const aName: string): TSignal; // override;
    begin
      if      SameText( aName, 'rule'  ) then Result := Rule
      else if SameText( aName, 'faults') then Result := Faults
      else if SameText( aName, 'steps' ) then Result := Steps
      else Result := inherited;
    end;


    procedure   TModCellAut2.DoSlowTick; // override;
    var
      Reset : Boolean;
      Step  : Boolean;
      Mode  : Integer;
      i     : Integer;
    begin
      if Assigned( FEngine)
      then begin
        Reset  := SignalToLogic( FInputs[ i_reset]);
        Step   := SignalToLogic( FInputs[ i_step ]);
        Mode   := Round( FInputs[ i_mode]);
        Rule   := Round(( FInputs[ i_rule  ] + 0.5) * FInputs[ i_rulemod  ]);
        Faults := Round(( FInputs[ i_faults] + 0.5) * FInputs[ i_faultsmod]);


        if ResetFlag or ( not FOldReset and Reset)
        then begin
          FResetCount := LedFlashTime( IsSpedUp);
          FEngine.Clear;
        end;


        case Mode of
          1 :                                                 // Pulse mode
            for i := 0 to WIDTH - 1
            do FOutputs[ o_out1 + i] := LogicToSignal(( FEngine.Bit[ i] <> 0) and Step);

          2 :                                                 // Signal mode
            for i := 0 to WIDTH - 1
            do begin
              if FEngine.Bit[ i] <> 0
              then FOutputs[ o_out1 + i] := FInputs[ i_step]
              else FOutputs[ o_out1 + i] := 0;
            end;

          else                                                // Square mode
            for i := 0 to WIDTH - 1
            do FOutputs[ o_out1 + i] := LogicToSignal( FEngine.Bit[ i] <> 0);
        end;

        if not FOldStep and Step
        then begin
          FEngine.Execute;
          FTrigCount := LedFlashTime( IsSpedUp);
        end;

        DecToZero( FResetCount);
        DecToZero( FTrigCount );
        ResetFlag := False;
        FOldReset := Reset;
        FOldStep  := Step;
      end;
    end;


    procedure   TModCellAut2.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step' , LogicToSignal( FTrigCount  >  0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'reset', LogicToSignal( FResetCount >  0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'cstep', FEngine.Count + 1                ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'steps', Steps                            ));

      for i := 0 to WIDTH - 1
      do aCallback( Self, MakeInfo( aPrefix, Name, Format( 'out%d', [ i + 1], AppLocale), FEngine.Bit[ i]));
    end;


{ ========
  TModLifeSeq = class( TMod)
  private
    FEngine      : TGameOfLife;
    FStops       : TGameOfLife;
    FExecCounter : Integer;
    FOldStep     : Boolean;
    FOldChange   : Boolean;
    FOldRes      : Boolean;
    FTrigCount   : Integer;
    FChangeCount : Integer;
    FResCount    : Integer;
    FPosition    : Integer;
    FRandomFlag  : Boolean;
    FClearFlag   : Boolean;
    FReversed    : Boolean;
    FLeftStop    : Integer;
    FRightStop   : Integer;
  public
    property    GridValue  : string read GetGridValue  write SetGridValue;
    property    StopsValue : string read GetStopsValue write SetStopsValue;
  private
}

    function    TModLifeSeq.GetGridValue: string;
    begin
      if Assigned( FEngine)
      then Result := FEngine.AsString;
    end;


    procedure   TModLifeSeq.SetGridValue( const aValue: string);
    begin
      if Assigned( FEngine)
      then FEngine.AsString := aValue;
    end;


    function    TModLifeSeq.GetStopsValue: string;
    begin
      if Assigned( FStops)
      then Result := FStops.AsString;
    end;


    procedure   TModLifeSeq.SetStopsValue( const aValue: string);
    var
      i : Integer;
    begin
      if Assigned( FStops)
      then begin
        FStops.AsString := aValue;

        FLeftStop  := -1;
        FRightStop := m_size;

        for i := 0 to m_size - 1
        do begin
          if FStops.Pixel[ i, 0]
          then begin
            FLeftStop := i;
            Break;
          end;
        end;

        for i := m_size - 1 downto 0
        do begin
          if FStops.Pixel[ i, 0]
          then begin
            FRightStop := i;
            Break;
          end;
        end;

        if FRightStop = FLeftStop
        then FLeftStop := -1;
      end;
    end;


    procedure   TModLifeSeq.CellChanged( aSender: TObject; anX, anY: Integer; IsAlive: Boolean);
    // Note: called from audio thread
    // done: Implement me ... maybe
    begin
    end;


//  public

    constructor TModLifeSeq.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      Locked             := True;
      FEngine            := TGameOfLife.Create( m_size, m_size, CellChanged);
      FEngine.FaultRate  := 0;
      FEngine.FillChance := 0.1;
      FEngine.Mode       := lmLife;
      FEngine.Wrapped    := lwXY;
      FEngine.Column     := 0;
      FEngine.SumMode    := smBin;
      FStops             := TGameOfLife.Create( m_size, 1, nil);
      FEngine.FaultRate  := 0;
      FEngine.FillChance := 0;
      FEngine.Mode       := lmFreeze;
      FEngine.Wrapped    := lwOff;
      FEngine.Column     := 0;
      FEngine.SumMode    := smBin;
      FReversed          := False;
      Locked             := False;
    end;


    procedure   TModLifeSeq.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_step        , 'step'        );
      AddInput ( i_change      , 'change'      );
      AddInput ( i_changemute  , 'changemute'  );
      AddInput ( i_res         , 'res'         );
      AddInput ( i_mode        , 'mode'        );
      AddInput ( i_wrapmode    , 'wrapmode'    );
      AddInput ( i_summode     , 'summode'     );
      AddInput ( i_faultrate   , 'faultrate'   );
      AddInput ( i_lowcount    , 'lowcount'    );
      AddInput ( i_randomamount, 'randomamount');
      AddInput ( i_rnd         , 'rnd'         );
      AddInput ( i_clr         , 'clr'         );
      AddInput ( i_chain       , 'chain'       );
      AddInput ( i_rev         , 'rev'         );
      AddOutput( o_sum         , 'sum'         );
      AddOutput( o_out1        , 'out1'        );
      AddOutput( o_out2        , 'out2'        );
      AddOutput( o_out3        , 'out3'        );
      AddOutput( o_out4        , 'out4'        );
      AddOutput( o_out5        , 'out5'        );
      AddOutput( o_out6        , 'out6'        );
      AddOutput( o_out7        , 'out7'        );
      AddOutput( o_out8        , 'out8'        );
      AddOutput( o_out9        , 'out9'        );
      AddOutput( o_out10       , 'out10'       );
      AddOutput( o_out11       , 'out11'       );
      AddOutput( o_out12       , 'out12'       );
      AddOutput( o_out13       , 'out13'       );
      AddOutput( o_out14       , 'out14'       );
      AddOutput( o_out15       , 'out15'       );
      AddOutput( o_out16       , 'out16'       );
      AddOutput( o_out17       , 'out17'       );
      AddOutput( o_out18       , 'out18'       );
      AddOutput( o_out19       , 'out19'       );
      AddOutput( o_out20       , 'out20'       );
      AddOutput( o_out21       , 'out21'       );
      AddOutput( o_out22       , 'out22'       );
      AddOutput( o_out23       , 'out23'       );
      AddOutput( o_out24       , 'out24'       );
      AddOutput( o_out25       , 'out25'       );
      AddOutput( o_out26       , 'out26'       );
      AddOutput( o_out27       , 'out27'       );
      AddOutput( o_out28       , 'out28'       );
      AddOutput( o_out29       , 'out29'       );
      AddOutput( o_out30       , 'out30'       );
      AddOutput( o_out31       , 'out31'       );
      AddOutput( o_out32       , 'out32'       );
      AddOutput( o_sum1        , 'sum1'        );
      AddOutput( o_sum2        , 'sum2'        );
      AddOutput( o_sum3        , 'sum3'        );
      AddOutput( o_sum4        , 'sum4'        );
      AddOutput( o_sum5        , 'sum5'        );
      AddOutput( o_sum6        , 'sum6'        );
      AddOutput( o_sum7        , 'sum7'        );
      AddOutput( o_sum8        , 'sum8'        );
      AddOutput( o_sum9        , 'sum9'        );
      AddOutput( o_sum10       , 'sum10'       );
      AddOutput( o_sum11       , 'sum11'       );
      AddOutput( o_sum12       , 'sum12'       );
      AddOutput( o_sum13       , 'sum13'       );
      AddOutput( o_sum14       , 'sum14'       );
      AddOutput( o_sum15       , 'sum15'       );
      AddOutput( o_sum16       , 'sum16'       );
      AddOutput( o_sum17       , 'sum17'       );
      AddOutput( o_sum18       , 'sum18'       );
      AddOutput( o_sum19       , 'sum19'       );
      AddOutput( o_sum20       , 'sum20'       );
      AddOutput( o_sum21       , 'sum21'       );
      AddOutput( o_sum22       , 'sum22'       );
      AddOutput( o_sum23       , 'sum23'       );
      AddOutput( o_sum24       , 'sum24'       );
      AddOutput( o_sum25       , 'sum25'       );
      AddOutput( o_sum26       , 'sum26'       );
      AddOutput( o_sum27       , 'sum27'       );
      AddOutput( o_sum28       , 'sum28'       );
      AddOutput( o_sum29       , 'sum29'       );
      AddOutput( o_sum30       , 'sum30'       );
      AddOutput( o_sum31       , 'sum31'       );
      AddOutput( o_sum32       , 'sum32'       );
    end;


    destructor  TModLifeSeq.Destroy; // override;
    begin
      Locked := True;
      FreeAndNil( FEngine);
      FreeAndNil( FStops );
      inherited;
    end;


    procedure   TModLifeSeq.SetStringValue( const aName, aValue: string); // override;
    begin
      if SameText( aName, 'grid')
      then GridValue := aValue
      else if SameText( aName, 'stops')
      then StopsValue := aValue
      else inherited;
    end;


    function    TModLifeSeq.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'grid')
      then Result := GridValue
      else if SameText( aName, 'stops')
      then Result := StopsValue
      else Result := inherited;
    end;


    procedure   TModLifeSeq.Pulse( anInput: Integer); // override;
    begin
      case anInput of
        i_rnd : FRandomFlag := True;
        i_clr : FClearFlag  := True;
      end;
    end;


    procedure   TModLifeSeq.DoSlowTick; // override;
    var
      i          : Integer;
      j          : Integer;
      Step       : Boolean;
      Change     : Boolean;
      ChangeMute : Boolean;
      Res        : Boolean;
      Count      : Integer;
      MinCount   : Integer;
    begin
      if Assigned( FEngine)
      then begin
        FEngine.FaultRate  :=                       FInputs[ i_faultrate   ] / 1000;
        FEngine.FillChance :=                       FInputs[ i_randomamount] /  100;
        FEngine.Mode       := TLifeMode    ( Round( FInputs[ i_mode        ]));
        FEngine.Wrapped    := TLifeWrapMode( Round( FInputs[ i_wrapmode    ]));
        FEngine.SumMode    := TLifeSumMode ( Round( FInputs[ i_summode     ]));
        MinCount           :=                Round( Finputs[ i_lowcount    ] );
        Count              := FEngine.FillCount;
        Res                := SignalToLogic       ( FInputs[ i_res         ] );
        Step               := SignalToLogic       ( FInputs[ i_step        ] );
        ChangeMute         := Round               ( FInputs[ i_changemute  ] ) <> 0;
        Change             := SignalToLogic       ( FInputs[ i_change      ] ) and not ChangeMute;

        if FClearFlag
        then begin
          FEngine.Clear;
          FChanged := True;
        end;

        if  ( FEngine.Mode <> lmFreeze) and (( Count < MinCount) and ( FExecCounter = 0)) or FRandomFlag
        then begin
          FEngine.FillRandom;
          FChanged := True;
        end;

        if ResetFlag or (( not FOldRes) and Res)
        then begin
          if FReversed
          then FPosition := FRightStop - 1
          else FPosition := FLeftStop  + 1;

          FResCount := LedFlashTime( IsSpedUp);
        end;

        if ( FEngine.Mode <> lmFreeze) and (( not FOldChange) and Change) and ( FExecCounter = 0)
        then begin
          FExecCounter := FEngine.XSize;
          FChangeCount := LedFlashTime( IsSpedUp);
        end;

        if FExecCounter > 0
        then FChanged := FEngine.ExecuteCol( FEngine.XSize - FExecCounter);

        if ( not FOldStep) and Step
        then begin
          FReversed := SignalToLogic( FInputs[ i_rev]);

          if FReversed
          then begin
            FPosition := FPosition - 1;

            if FPosition <= FLeftStop
            then FPosition := FRightStop - 1;
          end
          else begin
            FPosition := FPosition + 1;

            if FPosition >= FRightStop
            then FPosition := FLeftStop + 1;
          end;

          FTrigCount := LedFlashTime( IsSpedUp);
        end;

        FEngine.Column := FPosition;

        for i := 0 to m_size - 1
        do begin
          FOutputs[ o_out1 + i] := LogicToSignal( FEngine.Pixel[ FPosition, i] and Step);
          FOutputs[ o_sum1 + i] := 0;

          for j := 0 to m_size - 1
          do FOutputs[ o_sum1 + i] := FOutputs[ o_sum1 + i] + ( Ord( FEngine.Pixel[ j, i]) / 32.0);
        end;

        FOutputs[ o_sum] := FEngine.ColFunction + FInputs[ i_chain];
        FOldStep         := Step;
        FOldChange       := Change;
        FOldRes          := Res;
        ResetFlag        := False;
        FClearFlag       := False;
        FRandomFlag      := False;
        DecToZero( FResCount   );
        DecToZero( FChangeCount);
        DecToZero( FTrigCount  );
        DecToZero( FExecCounter);
      end;
    end;


    procedure   TModLifeSeq.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'step'  , LogicToSignal( FTrigCount   > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'change', LogicToSignal( FChangeCount > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'   , LogicToSignal( FResCount    > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'rev'   , LogicToSignal( FReversed        )));
    end;


    procedure   TModLifeSeq.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // override;
    begin
      if IsSpedUp
      then aCallback( Self, MakeInfo( aPrefix, Name, 'grid', SignalPair( -1                , 0.0)))
      else aCallback( Self, MakeInfo( aPrefix, Name, 'grid', SignalPair( FPosition / m_size, 0.0)));
    end;


    procedure   TModLifeSeq.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      if FChanged
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'grid', FEngine.AsString));
        FChanged := False;
      end;
    end;


{ ========
  TModSmallLifeSeq = class( TMod)
  private
    FEngine      : TGameOfLife;
    FExecCounter : Integer;
    FOldXStep    : Boolean;
    FOldYStep    : Boolean;
    FOldChange   : Boolean;
    FOldRes      : Boolean;
    FTrigXCount  : Integer;
    FTrigYCount  : Integer;
    FChangeCount : Integer;
    FResCount    : Integer;
    FXPosition   : Integer;
    FYPosition   : Integer;
    FRandomFlag  : Boolean;
    FClearFlag   : Boolean;
    FReversed    : Boolean;
    FChanged     : Boolean;
  public
    property    GridValue  : string read GetGridValue  write SetGridValue;
  private
}

    function    TModSmallLifeSeq.GetGridValue: string;
    begin
      if Assigned( FEngine)
      then Result := FEngine.AsString;
    end;


    procedure   TModSmallLifeSeq.SetGridValue( const aValue: string);
    begin
      if Assigned( FEngine)
      then FEngine.AsString := aValue;
    end;


    procedure   TModSmallLifeSeq.CellChanged( aSender: TObject; anX, anY: Integer; IsAlive: Boolean);
    // Note: called from audio thread
    // done: Implement me ... maybe
    begin
    end;


//  public

    constructor TModSmallLifeSeq.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      Locked             := True;
      FEngine            := TGameOfLife.Create( m_size, m_size, CellChanged);
      FEngine.FaultRate  := 0;
      FEngine.FillChance := 0.1;
      FEngine.Mode       := lmLife;
      FEngine.Wrapped    := lwXY;
      FEngine.Column     := 0;
      FEngine.SumMode    := smBin;
      FEngine.FaultRate  := 0;
      FEngine.FillChance := 0;
      FEngine.Mode       := lmFreeze;
      FEngine.Wrapped    := lwOff;
      FEngine.Column     := 0;
      FEngine.SumMode    := smBin;
      FReversed          := False;
      Locked             := False;
    end;


    procedure   TModSmallLifeSeq.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_xstep       , 'xstep'       );
      AddInput ( i_ystep       , 'ystep'       );
      AddInput ( i_change      , 'change'      );
      AddInput ( i_res         , 'res'         );
      AddInput ( i_mode        , 'mode'        );
      AddInput ( i_wrapmode    , 'wrapmode'    );
      AddInput ( i_faultrate   , 'faultrate'   );
      AddInput ( i_lowcount    , 'lowcount'    );
      AddInput ( i_randomamount, 'randomamount');
      AddInput ( i_rnd         , 'rnd'         );
      AddInput ( i_clr         , 'clr'         );
      AddInput ( i_rev         , 'rev'         );
      AddOutput( o_out         , 'out'         );
      AddOutput( o_sum         , 'sum'         );
      AddOutput( o_rowout1     , 'rowout1'     );
      AddOutput( o_rowout2     , 'rowout2'     );
      AddOutput( o_rowout3     , 'rowout3'     );
      AddOutput( o_rowout4     , 'rowout4'     );
      AddOutput( o_rowout5     , 'rowout5'     );
      AddOutput( o_rowout6     , 'rowout6'     );
      AddOutput( o_rowout7     , 'rowout7'     );
      AddOutput( o_rowout8     , 'rowout8'     );
      AddOutput( o_rowout9     , 'rowout9'     );
      AddOutput( o_rowout10    , 'rowout10'    );
      AddOutput( o_rowout11    , 'rowout11'    );
      AddOutput( o_rowout12    , 'rowout12'    );
      AddOutput( o_rowsum1     , 'rowsum1'     );
      AddOutput( o_rowsum2     , 'rowsum2'     );
      AddOutput( o_rowsum3     , 'rowsum3'     );
      AddOutput( o_rowsum4     , 'rowsum4'     );
      AddOutput( o_rowsum5     , 'rowsum5'     );
      AddOutput( o_rowsum6     , 'rowsum6'     );
      AddOutput( o_rowsum7     , 'rowsum7'     );
      AddOutput( o_rowsum8     , 'rowsum8'     );
      AddOutput( o_rowsum9     , 'rowsum9'     );
      AddOutput( o_rowsum10    , 'rowsum10'    );
      AddOutput( o_rowsum11    , 'rowsum11'    );
      AddOutput( o_rowsum12    , 'rowsum12'    );
      AddOutput( o_colout1     , 'colout1'     );
      AddOutput( o_colout2     , 'colout2'     );
      AddOutput( o_colout3     , 'colout3'     );
      AddOutput( o_colout4     , 'colout4'     );
      AddOutput( o_colout5     , 'colout5'     );
      AddOutput( o_colout6     , 'colout6'     );
      AddOutput( o_colout7     , 'colout7'     );
      AddOutput( o_colout8     , 'colout8'     );
      AddOutput( o_colout9     , 'colout9'     );
      AddOutput( o_colout10    , 'colout10'    );
      AddOutput( o_colout11    , 'colout11'    );
      AddOutput( o_colout12    , 'colout12'    );
      AddOutput( o_colsum1     , 'colsum1'     );
      AddOutput( o_colsum2     , 'colsum2'     );
      AddOutput( o_colsum3     , 'colsum3'     );
      AddOutput( o_colsum4     , 'colsum4'     );
      AddOutput( o_colsum5     , 'colsum5'     );
      AddOutput( o_colsum6     , 'colsum6'     );
      AddOutput( o_colsum7     , 'colsum7'     );
      AddOutput( o_colsum8     , 'colsum8'     );
      AddOutput( o_colsum9     , 'colsum9'     );
      AddOutput( o_colsum10    , 'colsum10'    );
      AddOutput( o_colsum11    , 'colsum11'    );
      AddOutput( o_colsum12    , 'colsum12'    );
    end;


    destructor  TModSmallLifeSeq.Destroy; // override;
    begin
      Locked := True;
      FreeAndNil( FEngine);
      inherited;
    end;


    procedure   TModSmallLifeSeq.SetStringValue( const aName, aValue: string); // override;
    begin
      if SameText( aName, 'grid')
      then GridValue := aValue
      else inherited;
    end;


    function    TModSmallLifeSeq.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'grid')
      then Result := GridValue
      else Result := inherited;
    end;


    procedure   TModSmallLifeSeq.Pulse( anInput: Integer); // override;
    begin
      case anInput of
        i_rnd : FRandomFlag := True;
        i_clr : FClearFlag  := True;
      end;
    end;


    procedure   TModSmallLifeSeq.DoSlowTick; // override;
    var
      row        : Integer;
      col        : Integer;
      XStep      : Boolean;
      YStep      : Boolean;
      Change     : Boolean;
      Res        : Boolean;
      Count      : Integer;
      MinCount   : Integer;
      OutVal     : Boolean;
      OutSum     : TSignal;
      LocOutVal  : Boolean;
      LocOutSum  : TSignal;
    begin
      if Assigned( FEngine)
      then begin
        FEngine.FaultRate  :=                       FInputs[ i_faultrate   ] / 1000;
        FEngine.FillChance :=                       FInputs[ i_randomamount] /  100;
        FEngine.Mode       := TLifeMode    ( Round( FInputs[ i_mode        ]));
        FEngine.Wrapped    := TLifeWrapMode( Round( FInputs[ i_wrapmode    ]));
        FEngine.SumMode    := smLin;
        MinCount           :=                Round( Finputs[ i_lowcount    ] );
        Count              := FEngine.FillCount;
        Res                := SignalToLogic       ( FInputs[ i_res         ] );
        XStep              := SignalToLogic       ( FInputs[ i_xstep       ] );
        YStep              := SignalToLogic       ( FInputs[ i_ystep       ] );
        Change             := SignalToLogic       ( FInputs[ i_change      ] );
        FReversed          := SignalToLogic       ( FInputs[ i_rev         ] );

        if FClearFlag
        then begin
          FEngine.Clear;
          FChanged := True;
        end;

        if  ( FEngine.Mode <> lmFreeze) and (( Count < MinCount) and ( FExecCounter = 0)) or FRandomFlag
        then begin
          FEngine.FillRandom;
          FChanged := True;
        end;

        if ResetFlag or (( not FOldRes) and Res)
        then begin
          if FReversed
          then begin
            FXPosition := 0;
            FYPosition := 0;
          end
          else begin
            FXPosition := m_size - 1;
            FYPosition := m_size - 1;
          end;

          FResCount := LedFlashTime( IsSpedUp);
        end;

        if ( FEngine.Mode <> lmFreeze) and (( not FOldChange) and Change) and ( FExecCounter = 0)
        then begin
          FExecCounter := FEngine.XSize;
          FChangeCount := LedFlashTime( IsSpedUp);
        end;

        if FExecCounter > 0
        then FChanged := FEngine.ExecuteCol( FEngine.XSize - FExecCounter);

        if ( not FOldXStep) and XStep
        then begin

          if FReversed
          then begin
            FXPosition := FXPosition - 1;

            if FXPosition <= 0
            then FXPosition := m_size - 1;
          end
          else begin
            FXPosition := FXPosition + 1;

            if FXPosition >= m_size
            then FXPosition := 0;
          end;

          FTrigXCount := LedFlashTime( IsSpedUp);
        end;

        if ( not FOldYStep) and YStep
        then begin

          if FReversed
          then begin
            FYPosition := FYPosition - 1;

            if FYPosition <= 0
            then FYPosition := m_size - 1;
          end
          else begin
            FYPosition := FYPosition + 1;

            if FYPosition >= m_size
            then FYPosition := 0;
          end;

          FTrigYCount := LedFlashTime( IsSpedUp);
        end;

        OutVal := False;
        OutSum := 0.0;

        for col := 0 to m_size - 1
        do begin
          LocOutSum := 0.0;

          for row := 0 to m_size - 1
          do begin
            LocOutSum := LocOutSum + Ord( FEngine.Pixel[ col, row]);

            if col = FXPosition
            then begin
              LocOutVal                  := FEngine.Pixel[ FXPosition, row] and XStep;
              FOutputs[ o_rowout1 + row] := LogicToSignal( LocOutVal);
              OutVal                     := OutVal or LocOutVal;
            end;

            LocOutSum                  := LocOutSum / 12.0;
            FOutputs[ o_colsum1 + col] := LocOutSum;
          end;

          OutSum := OutSum + LocOutSum / 2.0;
        end;

        for row := 0 to m_size - 1
        do begin
          LocOutSum := 0.0;

          for col := 0 to m_size - 1
          do begin
            LocOutSum := LocOutSum + Ord( FEngine.Pixel[ col, row]);

            if row = FYPosition
            then begin
              LocOutVal                  := FEngine.Pixel[ col, FYPosition] and YStep;
              FOutputs[ o_colout1 + col] := LogicToSignal( LocOutVal);
              OutVal                     := OutVal or LocOutVal;
            end;

            LocOutSum                  := LocOutSum / 12.0;
            FOutputs[ o_rowsum1 + row] := LocOutSum;
          end;

          OutSum := OutSum + LocOutSum / 2.0;
        end;

        FOutputs[ o_out] := LogicToSignal( OutVal);
        FOutputs[ o_sum] := OutSum;
        FOldXStep        := XStep;
        FOldYStep        := YStep;
        FOldChange       := Change;
        FOldRes          := Res;
        ResetFlag        := False;
        FClearFlag       := False;
        FRandomFlag      := False;
        DecToZero( FResCount   );
        DecToZero( FChangeCount);
        DecToZero( FTrigXCount );
        DecToZero( FTrigYCount );
        DecToZero( FExecCounter);
      end;
    end;


    procedure   TModSmallLifeSeq.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'xstep' , LogicToSignal( FTrigXCount  > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'ystep' , LogicToSignal( FTrigYCount  > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'change', LogicToSignal( FChangeCount > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'   , LogicToSignal( FResCount    > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'rev'   , LogicToSignal( FReversed        )));
    end;


    procedure   TModSmallLifeSeq.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // override;
    begin
      if IsSpedUp
      then aCallback( Self, MakeInfo( aPrefix, Name, 'grid', SignalPair( -1                 , -1                 )))
      else aCallback( Self, MakeInfo( aPrefix, Name, 'grid', SignalPair( FXPosition / m_size, FYPosition / m_size)));
    end;


    procedure   TModSmallLifeSeq.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      if FChanged
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'grid', FEngine.AsString));
        FChanged := False;
      end;
    end;


{ ========
  TModVitaPHoBium = class( TMod)
  private
    FEngine       : TGameOfLife;
    FFwdIndex     : array of Integer;
    FRevIndex     : array of Integer;
    FLiveCount    : Integer;
    FExecCounter  : Integer;
    FOldClock     : Boolean;
    FOldStep      : Boolean;
    FOldChange    : Boolean;
    FOldRes       : Boolean;
    FResCount     : Integer;
    FStepCount    : Integer;
    FChangeCount  : Integer;
    FClockCount   : Integer;
    FTrigCount    : Integer;
    FTrigLedCount : Integer;
    FXPosition    : Integer;
    FYPosition    : Integer;
    FRandomFlag   : Boolean;
    FClearFlag    : Boolean;
    FReversed     : Boolean;
    FChanged      : Boolean;
    FOldX         : Integer;
    FOldY         : Integer;
    FSkipDead     : Boolean;
  public
    property    GridValue  : string read GetGridValue  write SetGridValue;
  private
}

    function    TModVitaPHoBium.GetGridValue: string;
    begin
      if Assigned( FEngine)
      then Result := FEngine.AsString;
    end;


    procedure   TModVitaPHoBium.SetGridValue( const aValue: string);
    begin
      if Assigned( FEngine)
      then FEngine.AsString := aValue;
    end;


    procedure   TModVitaPHoBium.CellChanged( aSender: TObject; anX, anY: Integer; IsAlive: Boolean);
    // Note: called from audio thread
    //
    // Example .. for a 3 x 3 grid ... with cells 1, 4 and 8 being alive (marked by a *
    // where the cell number (or Pos later on) = X + 3 * Y. Also FLiveCount = 3;
    //
    //                             +-----+-----+-----+
    //                             | 0   | 1 * | 2   |
    //                             +-----+-----+-----+
    //                             | 3   | 4 * | 5   |
    //                             +-----+-----+-----+
    //                             | 6   | 7   | 8 * |
    //                             +-----+-----+-----+
    //
    // Then the index arrays FFwdIndex and FRevIndex should be:
    //
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    //                      index  |  0  |  1  |  2  |  3  |  4  |  5  |  6  |  7  |  8  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    // Then in FFwdIndex we values | -1, |  4  | -1  | -1  |  8  | -1  | -1  | -1  |  1  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    // and in  FRevIndex we have   | -1  |  8  | -1  | -1  |  1  | -1  | -1  | -1  |  4  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    //
    // That is -1 for the dead cells and for the live cells the index of the next live one (or the previous
    // live one in the reverse index)
    //
    // Now say ... item 4 (X=1, Y=1, so Pos = 4) dies .. so in the Fwd index now item 1 needs to point to where 4
    // pointed, so to 8 we can find the fwd cell that needs to be changed (1) in the Rev index at position 4 and
    // the value it has to be changed to (8).
    //
    // So we find:
    //
    //   NewFwdPos = FRevIndex[ Pos] = FRevIndex[ 4] = 1  - the cell that needs to be changed in FFwdIndex
    //   NewFwdVal = FFwdIndex[ Pos] = FFwdIndex[ 4] = 8  - the value that cell needs to be changed to
    //
    // Likewise for the reverse index we find:
    //
    //   NewRevPos = FFwdIndex[ Pos] = FFwdIndex[ 4] = 8  - the cell that needs to be changed in FFwdIndex
    //   NewRevVal = FRevIndex[ Pos] = FRevIndex[ 4] = 1  - the value that cell needs to be changed to
    //
    // Then finally the value at index Pos (the died cell) needs to be set to -1 in both FFwdIndex and FRevIndex.
    // and also we decremtnt FLiveCount by so it's 2 now.
    //
    // So now things look like:
    //
    //                             +-----+-----+-----+
    //                             | 0   | 1 * | 2   |
    //                             +-----+-----+-----+
    //                             | 3   | 4   | 5   |
    //                             +-----+-----+-----+
    //                             | 6   | 7   | 8 * |
    //                             +-----+-----+-----+
    //
    // Then the index arrays FFwdIndex and FRevIndex should be ( with FLiveCount being 2)
    //
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    //                      index  |  0  |  1  |  2  |  3  |  4  |  5  |  6  |  7  |  8  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    // Then in FFwdIndex we values | -1, |  8  | -1  | -1  | -1  | -1  | -1  | -1  |  1  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    // and in  FRevIndex we have   | -1  |  8  | -1  | -1  | -1  | -1  | -1  | -1  |  1  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    //
    // and when cell 1 dies too :

    //                             +-----+-----+-----+
    //                             | 0   | 1   | 2   |
    //                             +-----+-----+-----+
    //                             | 3   | 4   | 5   |
    //                             +-----+-----+-----+
    //                             | 6   | 7   | 8 * |
    //                             +-----+-----+-----+
    //
    // Then the index arrays FFwdIndex and FRevIndex should be ( with FLiveCount being 1):
    //
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    //                      index  |  0  |  1  |  2  |  3  |  4  |  5  |  6  |  7  |  8  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    // Then in FFwdIndex we values | -1, | -1  | -1  | -1  | -1  | -1  | -1  | -1  |  8  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    // and in  FRevIndex we have   | -1  | -1  | -1  | -1  | -1  | -1  | -1  | -1  |  8  |
    //                             +-----+-----+-----+-----+-----+-----+-----+-----+-----+
    //
    //
    // Insertion of a live cell is a bit more work though, and it nees some searching, but at least it only nees to be
    // done once instead of on each iteration.
    //
    // Ok, let's make cell 2 alive now, 2 will have to point at 8 then and 8 must point at 2.
    //
    // The only thing we then know is that Pos = 2 and from that we'll have to do some searching ...
    // just see the algorithm below for the IsLive == True case
    //
    var
      Pos       : Integer;
      NewFwdVal : Integer;
      NewRevVal : Integer;
      NewFwdPos : Integer;
      NewRevPos : Integer;
      i         : Integer;
      p         : Integer;
    begin
      Pos := anX + m_xsize * anY;                  // That is the position for the cell which died or came alive

      if   ( Pos >= 0     )                        // Make sure to have a valid Pos so we wont addres outside
      and  ( Pos <  m_size)                        // the index arrays later on
      then begin
        if IsAlive                                 // Is the changed cell alive now?
        then begin
          if FLiveCount <= 0                        // No live nodes currently ... just insert the new one pointing to itself
          then begin
            FFwdIndex[ Pos] := Pos;
            FRevIndex[ Pos] := Pos;
            FLiveCount      := 1;                   // and there is one live node now
          end
          else begin
            if   ( FFwdIndex[ Pos] < 0)            // Only turn on dead cells
            and  ( FRevIndex[ Pos] < 0)
            then begin
              p := Pos;

              for i := 0 to m_size - 1             // at most m_size - 1 lookups, the msize - 1 case would be the new one .. but then count would have been zero
              do begin
                p := MathIntMod( p + 1, m_size);   // C: p = ( p + 1) % m_size

                if FFwdIndex[ p] >= 0
                then Break;
              end;

              FFwdIndex[ FRevIndex[ p]] := Pos;
              FFwdIndex[ Pos          ] := p;
              FRevIndex[ Pos          ] := FRevIndex[ p];
              FRevIndex[ p            ] := Pos;
              Inc( FLiveCount);
            end;
          end;
        end
        else begin                                 // A cell died ... remove it from Fwd and Rev index chains
          if  ( FFwdIndex[ Pos] >= 0)              // First check if it was actually alive, don't kill the dead
          and ( FRevIndex[ Pos] >= 0)
          and ( FLiveCount      > 0 )
          then begin
            NewFwdPos := FRevIndex[ Pos];          // Find the fwd position to be changed
            NewFwdVal := FFwdIndex[ Pos];          // Find the new value that has to be at the fwd position
            NewRevPos := NewFwdVal;
            NewRevVal := NewFwdPos;

            if  ( NewFwdVal >= 0)                  // Make sure to have valid indices and link values
            and ( NewFwdPos >= 0)
            then begin
              FFwdIndex[ NewFwdPos] := NewFwdVal;  // Then set the new stuff
              FRevIndex[ NewRevPos] := NewRevVal;
              Dec( FLiveCount);
            end;
          end;

          FFwdIndex[ Pos] := -1;                   // And finally, and always. clear the links for the removed item
          FRevIndex[ Pos] := -1;
        end;
      end;
    end;


    procedure   TModVitaPHoBium.FindNextAlive( var XPos, YPos: Integer);
    var
      p : Integer;
    begin
      p := XPos + m_xsize * YPos;

      if FReversed
      then begin
        if FRevIndex[ p] >= 0
        then p := FRevIndex[ p]
        else p := MathIntMod( p - 1, m_size);
      end
      else begin
        if FFwdIndex[ p] >= 0
        then p := FFwdIndex[ p]
        else p := MathIntMod( p + 1, m_size);
      end;

      XPos := p mod m_xsize;
      YPos := p div m_xsize;
    end;


//  public

    constructor TModVitaPHoBium.Create( aParent: TSynthPatch; const aName: string); // override;
    var
      i : Integer;
    begin
      inherited;
      Locked     := True;
      FLiveCount := 0;                    // No living cells yet
      SetLength( FFwdIndex, m_size);      // Setup index arrays
      SetLength( FRevIndex, m_size);

      for i := 0 to m_size - 1            // Mark all cels as dead in the indices
      do begin
        FFwdIndex[ i] := -1;
        FRevIndex[ i] := -1;
      end;

      FEngine            := TGameOfLife.Create( m_xsize, m_ysize, CellChanged);
      FEngine.FaultRate  := 0;
      FEngine.FillChance := 0.1;
      FEngine.Mode       := lmLife;
      FEngine.Wrapped    := lwXY;
      FEngine.Column     := 0;
      FEngine.SumMode    := smBin;
      FEngine.FaultRate  := 0;
      FEngine.FillChance := 0;
      FEngine.Mode       := lmFreeze;
      FEngine.Wrapped    := lwOff;
      FEngine.Column     := 0;
      FEngine.SumMode    := smLin;
      FReversed          := False;
      Locked             := False;
    end;


    destructor  TModVitaPHoBium.Destroy; // override;
    begin
      Locked := True;
      FreeAndNil( FEngine);
      inherited;
    end;


    procedure   TModVitaPHoBium.SetStringValue( const aName, aValue: string); // override;
    begin
      if SameText( aName, 'grid')
      then GridValue := aValue
      else inherited;
    end;


    function    TModVitaPHoBium.GetStringValue( const aName: string): string; // override;
    begin
      if SameText( aName, 'grid')
      then Result := GridValue
      else Result := inherited;
    end;


    procedure   TModVitaPHoBium.Pulse( anInput: Integer); // override;
    begin
      case anInput of
        i_rnd : FRandomFlag := True;
        i_clr : FClearFlag  := True;
      end;
    end;


    procedure   TModVitaPHoBium.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_clock       , 'clock'       );
      AddInput ( i_in          , 'in'          );
      AddInput ( i_res         , 'res'         );
      AddInput ( i_rev         , 'rev'         );
      AddInput ( i_change      , 'change'      );
      AddInput ( i_step        , 'step'        );
      AddInput ( i_x           , 'x'           );
      AddInput ( i_y           , 'y'           );
      AddInput ( i_mode        , 'mode'        );
      AddInput ( i_wrapmode    , 'wrapmode'    );
      AddInput ( i_faultrate   , 'faultrate'   );
      AddInput ( i_randomamount, 'randomamount');
      AddInput ( i_rnd         , 'rnd'         );
      AddInput ( i_skipmode    , 'skipmode'    );
      AddInput ( i_skipctrl    , 'skipctrl'    );
      AddInput ( i_clr         , 'clr'         );
      AddOutput( o_out         , 'out'         );
      AddOutput( o_gate        , 'gate'        );
      AddOutput( o_trig        , 'trig'        );
      AddOutput( o_xout        , 'xout'        );
      AddOutput( o_yout        , 'Yout'        );
    end;


    procedure   TModVitaPHoBium.SetDefaults; // override;
    begin
      FInputs[ i_x   ] := NOT_CONNECTED;
      FInputs[ i_y   ] := NOT_CONNECTED;
      FInputs[ i_step] := NOT_CONNECTED;
      FInputs[ i_rev ] := LogicToSignal( false);
    end;


    procedure   TModVitaPHoBium.DoSlowTick; // override;
    var
      Clock   : Boolean;
      Step    : Boolean;
      Change  : Boolean;
      Res     : Boolean;
      Reset   : Boolean;
      NoteNr  : Integer;
      X       : TSignal;
      Y       : TSignal;
    begin
      if Assigned( FEngine)
      then begin
        FEngine.FaultRate  :=                       FInputs[ i_faultrate   ] / 1000;
        FEngine.FillChance :=                       FInputs[ i_randomamount] /  100;
        FEngine.Mode       := TLifeMode    ( Round( FInputs[ i_mode        ]));
        FEngine.Wrapped    := TLifeWrapMode( Round( FInputs[ i_wrapmode    ]));
        Res                := SignalToLogic       ( FInputs[ i_res         ]);
        Step               := SignalToLogic       ( FInputs[ i_step        ]);
        Change             := SignalToLogic       ( FInputs[ i_change      ]);
        Clock              := SignalToLogic       ( FInputs[ i_clock       ]);
        X                  :=                       FInputs[ i_x           ];
        Y                  :=                       FInputs[ i_y           ];
        FReversed          := SignalToLogic       ( FInputs[ i_rev         ]);
        FSkipDead          := SignalToLogic       ( FInputs[ i_skipmode    ]) xor ( SignalToLogic( FInputs[ i_skipctrl]));

        if FClearFlag
        then begin
          FEngine.Clear;
          FChanged := True;
        end;

        if  FRandomFlag
        then begin
          FEngine.FillRandom;
          FChanged := True;
        end;

        Reset := ( not FOldRes) and Res;

        if ResetFlag or Reset
        then begin
          if Reset
          then begin
            FEngine.Clear;
            FChanged := True;
          end;

          if FReversed
          then begin
            FXPosition := 0;
            FYPosition := 0;
          end
          else begin
            FXPosition := m_xsize - 1;
            FYPosition := m_ysize - 1;
          end;

          FResCount := LedFlashTime( IsSpedUp);
        end;

        if (( not FOldChange) and Change) and ( FExecCounter = 0)
        then begin
          FExecCounter := FEngine.XSize;
          FChangeCount := LedFlashTime( IsSpedUp);
        end;

        if Clock and not FOldClock
        then begin
          NoteNr := MathIntMod( Round( UnitsToNoteNumber( FInputs[ i_in])), m_xsize * m_ysize);
          FEngine.Pixel[ NoteNr mod m_xsize, NoteNr div m_xsize] := True;
          FClockCount := LedFlashTime( IsSpedUp);
          FChanged    := True;
        end;

        if FExecCounter > 0
        then FChanged := FEngine.ExecuteCol( FEngine.XSize - FExecCounter);

        if FInputs[ i_step] = NOT_CONNECTED                                 // use X and Y inputs asynchroneously
        then begin
          FXPosition := Clip( Round( X * m_xsize + 0.5), 0, m_xsize - 1);
          FYPosition := Clip( Round( Y * m_ysize + 0.5), 0, m_ysize - 1);

          if FReversed
          then begin
            FXPosition := m_xsize - 1 - FXPosition;
            FYPosition := m_ysize - 1 - FYPosition;
          end;

          if ( FOldX <> FXPosition)                                         // no need to do anything when no change
          or ( FOldY <> FYPosition)
          then begin
            if FEngine.Pixel[ FXPosition, FYPosition]
            then begin
              NoteNr            := m_xsize * FYPosition + FXPosition;
              FOutputs[ o_out ] := NoteNumberToUnits( NoteNr);
              FOutputs[ o_gate] := LogicToSignal( True);
              FTrigLedCount     := LedFlashTime ( IsSpedUp);
              FTrigCount        := TrigPulseTime( IsSpedUp);
            end
            else FOutputs[ o_gate] := LogicToSignal( False);
          end;
        end
        else begin
          if ( not FOldStep) and Step                                       // if X or Y connected clock in X and Y
          then begin
            if ( X <> NOT_CONNECTED)
            or ( Y <> NOT_CONNECTED)
            then begin
              FXPosition := Clip( Round( X * m_xsize + 0.5), 0, m_xsize - 1);
              FYPosition := Clip( Round( Y * m_ysize + 0.5), 0, m_ysize - 1);

              if FReversed
              then begin
                FXPosition := m_xsize - 1 - FXPosition;
                FYPosition := m_ysize - 1 - FYPosition;
              end;
            end
            else begin                                                      // and otherwise step on each upgoing step edge
              if FSkipDead
              then FindNextAlive( FXPosition, FYPosition)
              else begin
                if FReversed
                then begin
                  FXPosition := FXPosition - 1;

                  if FXPosition <= 0
                  then begin
                    FXPosition := m_xsize - 1;
                    FYPosition := FYPosition - 1;

                    if FYPosition <= 0
                    then FYPosition := m_ysize - 1;
                  end;
                end
                else begin
                  FXPosition := FXPosition + 1;

                  if FXPosition >= m_xsize
                  then begin
                    FXPosition := 0;
                    FYPosition := FYPosition + 1;

                    if FYPosition >= m_ysize
                    then FYPosition := 0;
                  end;
                end;
              end;
            end;

            if FEngine.Pixel[ FXPosition, FYPosition]
            then begin
              NoteNr            := m_xsize * FYPosition + FXPosition;
              FOutputs[ o_out ] := NoteNumberToUnits( NoteNr);
              FOutputs[ o_gate] := LogicToSignal( True);
              FTrigLedCount     := LedFlashTime ( IsSpedUp);
              FTrigCount        := TrigPulseTime( IsSpedUp);
            end
            else FOutputs[ o_gate] := LogicToSignal( False);

            FStepCount := LedFlashTime( IsSpedUp);
          end;
        end;

        FOutputs[ o_trig] := LogicToSignal( FTrigCount > 0);
        FOutputs[ o_xout] := FXPosition / ( m_xsize - 1);
        FOutputs[ o_yout] := FYPosition / ( m_ysize - 1);
        FOldStep          := Step;
        FOldChange        := Change;
        FOldRes           := Res;
        FOldClock         := Clock;
        ResetFlag         := False;
        FClearFlag        := False;
        FRandomFlag       := False;
        FOldX             := FXPosition;
        FOldY             := FYPosition;
        DecToZero( FResCount    );
        DecToZero( FChangeCount );
        DecToZero( FStepCount   );
        DecToZero( FExecCounter );
        DecToZero( FClockCount  );
        DecToZero( FTrigCount   );
        DecToZero( FTrigLedCount);
      end;
    end;


    procedure   TModVitaPHoBium.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'clock'   , LogicToSignal( FClockCount   > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'step'    , LogicToSignal( FStepCount    > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'change'  , LogicToSignal( FChangeCount  > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'res'     , LogicToSignal( FResCount     > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'rev'     , LogicToSignal( FReversed         )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'trig'    , LogicToSignal( FTrigLedCount > 0 )));
      aCallback( Self, MakeInfo( aPrefix, Name, 'gate'    , FOutputs[ o_gate    ]             ));
      aCallback( Self, MakeInfo( aPrefix, Name, 'skipctrl', LogicToSignal( FSkipDead)         ));
    end;


    procedure   TModVitaPHoBium.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // override;
    begin
      if IsSpedUp
      then aCallback( Self, MakeInfo( aPrefix, Name, 'grid', SignalPair( -1                 , -1                   )))
      else aCallback( Self, MakeInfo( aPrefix, Name, 'grid', SignalPair( FXPosition / m_xsize, FYPosition / m_ysize)));
    end;


    procedure   TModVitaPHoBium.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    begin
      if FChanged
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'grid', FEngine.AsString));
        FChanged := False;
      end;
    end;


{ ========
  TModLut = class( TMod)
  private
    FStepCount  : Integer;
    FPosition   : TSignal;
    FPosition2  : TSignal;
    FPosition3  : TSignal;
    FPosition4  : TSignal;
    FRandom     : Boolean;
    FOldRandom  : Boolean;
    FRndReq     : Boolean;
    FRandom3    : Boolean;
    FOldRandom3 : Boolean;
    FRndReq3    : Boolean;
  public
}

    procedure   TModLut.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_select   , 'select'   );
      AddInput ( i_mode     , 'mode'     );
      AddInput ( i_xfade    , 'xfade'    );
      AddInput ( i_chain    , 'chain'    );
      AddInput ( i_step1    , 'step1'    );
      AddInput ( i_step2    , 'step2'    );
      AddInput ( i_step3    , 'step3'    );
      AddInput ( i_step4    , 'step4'    );
      AddInput ( i_step5    , 'step5'    );
      AddInput ( i_step6    , 'step6'    );
      AddInput ( i_step7    , 'step7'    );
      AddInput ( i_step8    , 'step8'    );
      AddInput ( i_step9    , 'step9'    );
      AddInput ( i_step10   , 'step10'   );
      AddInput ( i_step11   , 'step11'   );
      AddInput ( i_step12   , 'step12'   );
      AddInput ( i_step13   , 'step13'   );
      AddInput ( i_step14   , 'step14'   );
      AddInput ( i_step15   , 'step15'   );
      AddInput ( i_step16   , 'step16'   );
      AddInput ( i_steps    , 'steps'    );
      AddInput ( i_stepsmod , 'stepsmod' );
      AddInput ( i_rnd2     , 'rnd2'     );
      AddInput ( i_xfademod , 'xfademod' );
      AddInput ( i_rnd3     , 'rnd3'     );
      AddOutput( o_out      , 'out'      );
      AddOutput( o_out2     , 'out2'     );
      AddOutput( o_out3     , 'out3'     );
      AddOutput( o_out4     , 'out4'     );
      AddOutput( o_stepout1 , 'stepout1' );
      AddOutput( o_stepout2 , 'stepout2' );
      AddOutput( o_stepout3 , 'stepout3' );
      AddOutput( o_stepout4 , 'stepout4' );
      AddOutput( o_stepout5 , 'stepout5' );
      AddOutput( o_stepout6 , 'stepout6' );
      AddOutput( o_stepout7 , 'stepout7' );
      AddOutput( o_stepout8 , 'stepout8' );
      AddOutput( o_stepout9 , 'stepout9' );
      AddOutput( o_stepout10, 'stepout10');
      AddOutput( o_stepout11, 'stepout11');
      AddOutput( o_stepout12, 'stepout12');
      AddOutput( o_stepout13, 'stepout13');
      AddOutput( o_stepout14, 'stepout14');
      AddOutput( o_stepout15, 'stepout15');
      AddOutput( o_stepout16, 'stepout16');
      AddOutput( o_outraw   , 'outraw'   );
      AddOutput( o_outraw2  , 'outraw2'  );
      AddOutput( o_outraw3  , 'outraw3'  );
      AddOutput( o_outraw4  , 'outraw4'  );
    end;


    procedure   TModLut.SetDefaults; // override;
    begin
      inherited;
      FInputs[ i_chain   ] := 0.0;         // Chain input to zero wwhen not connected
      FInputs[ i_mode    ] := m_note;      // Default startup mode is notes
      FInputs[ i_stepsmod] := 1.0;         // Set full steps range
      FInputs[ i_xfademod] := 1.0;         // Set full xfade range
      FPosition            := 0;           // Neutral output
      FDezipperMap         := [ i_xfade];
    end;


    procedure   TModLut.DoSlowTick; // override;
    var
      XFade     : TSignal;
      Step      : TSignal;
      Position  : Integer;
      Position2 : Integer;
      Position3 : Integer;
      Position4 : Integer;
      Delta     : TSignal;
      Delta2    : TSignal;
      Delta3    : TSignal;
      Delta4    : TSignal;
      Next      : Integer;
      Next2     : Integer;
      Next3     : Integer;
      Next4     : Integer;
      Curr      : TSignal;
      Curr2     : TSignal;
      Curr3     : TSignal;
      Curr4     : TSignal;
      i         : Integer;
    begin
      XFade      := FInputs[ i_xfade] * FInputs[ i_xfademod];
      FStepCount := Round( FInputs[ i_steps] * FInputs[ i_stepsmod]);
      Step       := FStepCount * Clip( FInputs[ i_select], 0, 1);
      FPosition  := MathFloatMod( Step       , 16.0);
      FPosition2 := MathFloatMod( Step +  4.0, 16.0);
      FPosition3 := MathFloatMod( Step +  8.0, 16.0);
      FPosition4 := MathFloatMod( Step + 12.0, 16.0);
      Position   := Trunc( FPosition );
      Position2  := Trunc( FPosition2);
      Position3  := Trunc( FPosition3);
      Position4  := Trunc( FPosition4);
      Next       := ( Position  + 1) and 15;
      Next2      := ( Position2 + 1) and 15;
      Next3      := ( Position3 + 1) and 15;
      Next4      := ( Position4 + 1) and 15;
      Delta      := XFade * ( FPosition  - Position );
      Delta2     := XFade * ( FPosition2 - Position2);
      Delta3     := XFade * ( FPosition3 - Position3);
      Delta4     := XFade * ( FPosition4 - Position4);
      Curr       := FInputs[ i_step1 + Position ];
      Curr2      := FInputs[ i_step1 + Position2];
      Curr3      := FInputs[ i_step1 + Position3];
      Curr4      := FInputs[ i_step1 + Position4];

      if Round( FInputs[ i_mode]) = m_note
      then begin
        FOutputs[ o_out    ] := FInputs[ i_chain] + Curr  + Delta  * ( FInputs[ i_step1 + Next ] - Curr ) - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_out2   ] := FInputs[ i_chain] + Curr2 + Delta2 * ( FInputs[ i_step1 + Next2] - Curr2) - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_out3   ] := FInputs[ i_chain] + Curr3 + Delta3 * ( FInputs[ i_step1 + Next3] - Curr3) - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_out4   ] := FInputs[ i_chain] + Curr4 + Delta4 * ( FInputs[ i_step1 + Next4] - Curr4) - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_outraw ] := FInputs[ i_chain] + Curr  - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_outraw2] := FInputs[ i_chain] + Curr2 - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_outraw3] := FInputs[ i_chain] + Curr3 - NoteNumberToUnits( MiddleNote);
        FOutputs[ o_outraw4] := FInputs[ i_chain] + Curr4 - NoteNumberToUnits( MiddleNote);
      end
      else begin
        FOutputs[ o_out    ] := FInputs[ i_chain] + Curr  + Delta  * ( FInputs[ i_step1 + Next ] - Curr );
        FOutputs[ o_out2   ] := FInputs[ i_chain] + Curr2 + Delta2 * ( FInputs[ i_step1 + Next2] - Curr2);
        FOutputs[ o_out3   ] := FInputs[ i_chain] + Curr3 + Delta3 * ( FInputs[ i_step1 + Next3] - Curr3);
        FOutputs[ o_out4   ] := FInputs[ i_chain] + Curr4 + Delta4 * ( FInputs[ i_step1 + Next4] - Curr4);
        FOutputs[ o_outraw ] := FInputs[ i_chain] + Curr ;
        FOutputs[ o_outraw2] := FInputs[ i_chain] + Curr2;
        FOutputs[ o_outraw3] := FInputs[ i_chain] + Curr3;
        FOutputs[ o_outraw4] := FInputs[ i_chain] + Curr4;
      end;

      for i := 0 to 15
      do FOutputs[ o_stepout1 + i] := LogicToSignal(( Position = i));

      FRandom  := SignalToLogic( FInputs[ i_rnd2]);
      FRandom3 := SignalToLogic( FInputs[ i_rnd3]);

      if not FOldRandom and FRandom
      then FRndReq := True;

      if not FOldRandom3 and FRandom3
      then FRndReq3 := True;

      FOldRandom  := FRandom;
      FOldRandom3 := FRandom3;
    end;


    procedure   TModLut.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i : Integer;
    begin
      for i := i_step1 to i_step16
      do begin
        if i = Round( FPosition) + i_step1
        then aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i - i_step1 + 1], AppLocale), 1.0))
        else aCallback( Self, MakeInfo( aPrefix, Name, Format( 'step%d', [ i - i_step1 + 1], AppLocale), 0.0));
      end;
    end;


    procedure   TModLut.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // override;
    begin
      if FRndReq
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'rnd', 1));
        FRndReq := False;
      end;

      if FRndReq3
      then begin
        aCallback( Self, MakeInfo( aPrefix, Name, 'rndone', 16));
        FRndReq3 := False;
      end;
    end;


{ ========
  TModMultiCompare = class( TMod)
  private
    FPosition : Integer;
  public
}

    procedure   TModMultiCompare.CreateIO; // override;
    begin
      FIsSlow := True;
      AddInput ( i_in      , 'in'      );
      AddInput ( i_inamt   , 'inamt'   );
      AddInput ( i_inamtmod, 'inamtmod');
      AddInput ( i_level1  , 'level1'  );
      AddInput ( i_level2  , 'level2'  );
      AddInput ( i_level3  , 'level3'  );
      AddInput ( i_level4  , 'level4'  );
      AddInput ( i_level5  , 'level5'  );
      AddInput ( i_level6  , 'level6'  );
      AddInput ( i_level7  , 'level7'  );
      AddInput ( i_level8  , 'level8'  );
      AddInput ( i_bussel1 , 'bussel1' );
      AddInput ( i_bussel2 , 'bussel2' );
      AddInput ( i_bussel3 , 'bussel3' );
      AddInput ( i_bussel4 , 'bussel4' );
      AddInput ( i_bussel5 , 'bussel5' );
      AddInput ( i_bussel6 , 'bussel6' );
      AddInput ( i_bussel7 , 'bussel7' );
      AddInput ( i_bussel8 , 'bussel8' );
      AddOutput( o_out1    , 'out1'    );
      AddOutput( o_out2    , 'out2'    );
      AddOutput( o_out3    , 'out3'    );
      AddOutput( o_pos1    , 'pos1'    );
      AddOutput( o_pos2    , 'pos2'    );
      AddOutput( o_pos3    , 'pos3'    );
      AddOutput( o_pos4    , 'pos4'    );
      AddOutput( o_pos5    , 'pos5'    );
      AddOutput( o_pos6    , 'pos6'    );
      AddOutput( o_pos7    , 'pos7'    );
      AddOutput( o_pos8    , 'pos8'    );
    end;


    procedure   TModMultiCompare.SetDefaults; // override;
    begin
      FDezipperMap := [
        i_inamt,
        i_level1,
        i_level2,
        i_level3,
        i_level4,
        i_level5,
        i_level6,
        i_level7,
        i_level8
      ];
      FInputs[ i_inamtmod] := 1.0;
    end;


    procedure   TModMultiCompare.DoSlowTick; // override;
    var
      i       : Integer;
      aLevel  : TSignal;
      aLevels : array[ 0 .. 8] of TSignal;
      anOut1  : TSignal;
      anOut2  : TSignal;
      anOut3  : TSignal;
    begin
      aLevel      := FInputs[ i_in] * FInputs[ i_inamt] * FInputs[ i_inamtmod] * 4.0;
      aLevels[ 0] :=               FInputs[ i_level1];
      aLevels[ 1] := aLevels[ 0] + FInputs[ i_level2];
      aLevels[ 2] := aLevels[ 1] + FInputs[ i_level3];
      aLevels[ 3] := aLevels[ 2] + FInputs[ i_level4];
      aLevels[ 4] := aLevels[ 3] + FInputs[ i_level5];
      aLevels[ 5] := aLevels[ 4] + FInputs[ i_level6];
      aLevels[ 6] := aLevels[ 5] + FInputs[ i_level7];
      aLevels[ 7] := aLevels[ 6] + FInputs[ i_level8];
      aLevels[ 8] := aLevels[ 7] + 1e1