unit poetry_generator;

interface

uses

  KnobsUtils;

type

  TPoetryGenerator = class
  //
  // Source  : http://www.jonathanmackenzie.net/portfolio/PHD.pdf
  // Details : USING STRANGE ATTRACTORS. TO MODEL SOUND. Submitted to. The University of
  //           London for the Degree of Doctor of Philosophy. Jonathan Mackenzie.
  // Also    : http://www.jonathanmackenzie.net/portfolio/project_phd.html
  // Uses    : The Poetry Generation Algorithm from Chapter 8 of the Thesis.
  //
  private
    FHistory     : TSignalArray;
    FSeed        : TSignalArray;
    FInPointer   : Integer;
    FSeedPointer : Integer;
    FEpsilon     : TSignal;
  private
    function    GetHistoryCount : Integer;
    function    GetSeedCount    : Integer;
  private
    function    Equal( A, B: TSignal): Boolean;
    function    RandomFindSeed: Integer;
    procedure   ShiftSeed( aValue: TSignal);
  private
    property    HistoryCount : Integer read GetHistoryCount;
    property    SeedCount    : Integer read GetSeedCount;
  public
    constructor Create( aSeedLength, aHistorySize: Word; anEpsilon: TSignal);
    procedure   AddHistory( const aValue: TSignal);
    procedure   MakeRandomSeed;
    function    Tick: TSignal;
  end;



implementation


uses

 Math;


{ ========
  TPoetryGenerator = class
  private
    FHistory     : TSignalArray;
    FSeed        : TSignalArray;
    FInPointer   : Integer;
    FSeedPointer : Integer;
    FEpsilon     : TSignal;
  private
    property    HistoryCount : Integer read GetHistoryCount;
    property    SeedCount    : Integer read GetSeedCount;
  private
}

    function    TPoetryGenerator.GetHistoryCount: Integer;
    begin
      Result := Length( FHistory);
    end;


    function    TPoetryGenerator.GetSeedCount: Integer;
    begin
      Result := Length( FSeed);
    end;


//  private

    function    TPoetryGenerator.Equal( A, B: TSignal): Boolean;
    begin
      Result := Abs( A - B) < FEpsilon;
    end;


    function    TPoetryGenerator.RandomFindSeed: Integer;
    var
      p     : Integer;
      i     : Integer;
      j     : Integer;
      Found : Boolean;
    begin
      Result := -1;
      p      := Random( HistoryCount);                  // Start Seed search at a random position
      i      := 0;

      while i < HistoryCount
      do begin
        Found := True;
        j     := 0;

        while j < SeedCount
        do begin
          if Equal(
            FHistory[ ( p + i + j) mod HistoryCount],
            FSeed[ ( FSeedPointer + j) mod SeedCount]
          )
          then Inc( j)
          else begin
            Found := False;
            Inc( i);
            Break;
          end
        end;

        if Found
        then begin
          Result := ( i + p) mod HistoryCount;
          Break;
        end;

      end;
    end;


    procedure   TPoetryGenerator.ShiftSeed( aValue: TSignal);
    begin
      FSeedPointer := ( FSeedPointer + 1) mod SeedCount;
      FSeed[ ( FSeedPointer + SeedCount - 1) mod SeedCount] := aValue;
    end;

//  public


    constructor TPoetryGenerator.Create( aSeedLength, aHistorySize: Word; anEpsilon: TSignal);
    var
      i : Integer;
    begin
      inherited Create;
      SetLength( FHistory, aHistorySize);
      SetLength( FSeed   , aSeedLength );
      FEpsilon := anEpsilon;

      for i := 0 to SeedCount - 1
      do FSeed[ i] := 0;

      for i := 0 to HistoryCount - 1
      do FHistory[ i] := 0;

      FInPointer   := 0;
      FSeedPointer := 0;
    end;


    procedure   TPoetryGenerator.AddHistory( const aValue: TSignal);
    begin
      FHistory[ FInPointer] := aValue;
      FInPointer := ( FInPointer + 1) mod HistoryCount;
      MakeRandomSeed;
    end;


    procedure   TPoetryGenerator.MakeRandomSeed;
    var
      p : Integer;
      i : Integer;
    begin
      p := Random( HistoryCount);

      for i := 0 to SeedCount - 1
      do begin
        FSeed[ ( FSeedPointer + i) mod SeedCount] := FHistory[ p];
        p := ( p + 1) mod HistoryCount;
      end;
    end;


    function    TPoetryGenerator.Tick: TSignal;
    var
      i : Integer;
    begin
      i := RandomFindSeed;

      if i >= 0
      then ShiftSeed( FHistory[ ( i + SeedCount) mod HistoryCount])
      else MakeRandomSeed;

      Result := FSeed[ FSeedPointer];
    end;


end.

