unit scala;

interface

uses

  System.Classes, System.SysUtils, System.Math,

  Rationals, KnobsUtils, KnobsConversions;


type

  EScala           = Exception;
  EScalaReject     = class( EScala);

  TScalaNoteType   = ( stRational, stCents);

  TScalaNoteValue = record
    NoteType : TScalaNoteType;
    case TScalaNoteType of
      stRational    : ( Rational: TRational);
      stCents       : ( Cents   : TSignal  );
   {end}
  end;

  TScalaScaleNoteMap = record
    Scale     : Integer;
    Units     : TSignal;
    Frequency : TSignal;
  end;

  TScalaTuningTable = array[ TMidiSeptet] of TScalaScaleNoteMap;

  TScalaMapping = class( TStringList)
  // This maps key numbers to scale numbers.
  strict private
    type TParseState = (
      msInitial,
      msHasMapSize,
      msHasFirstNote,
      msHasLastNote,
      msHasMiddleNote,
      msHasReferenceNote,
      msHasReferenceFrequency,
      msHasScaleDegree
    );
  private
    FFileName           : string;           // File name for file  that the mapping was read from
    FParseState         : TParseState;      // The current file parse state
    FMapLimit           : Integer;          // Size of map. The pattern repeats every so many keys:
    FFirstNote          : Integer;          // First MIDI note number to retune
    FLastNote           : Integer;          // Last  MIDI note number to retune
    FMiddleNote         : Integer;          // Middle note where the first entry of the mapping is mapped to
    FReferenceNote      : Integer;          // Reference note for which frequency is given
    FReferenceFrequency : TSignal;          // Frequency to tune the above note to (Hz)
    FScaleDegree        : Integer;          // Scale degree to consider as formal octave (determines difference in pitch
                                            // between adjacent mapping patterns)
    FMapping            : array of Integer; // The numbers represent scale degrees mapped to keys. The first entry is
                                            // for the given middle note, the next for subsequent higher keys.
                                            // For an unmapped key, put in an "x". At the end, unmapped keys may be left
                                            // out.
    FlastMapped         : Integer;
    FLookupTable        : array[ TMidiSeptet] of Integer; // The resulting mapping from key number to scale number
  private
    procedure   SetFileName( const aValue: string);
    function    GetMappingCount: Integer;
    function    GetMapping( anIndex: Integer): Integer;
  private
    procedure   Reject( const aMsg: string);
    procedure   RejectFmt( const aFmt: string; anArgs: array of const);
    procedure   AddMapping( aLineNr: Integer; aValue: Integer);
    procedure   ParseLine( aLineNr: Integer; const aLine: string);
    procedure   Parse;
    procedure   FillTable;
    procedure   SetDefaultMapping;
  public
    constructor Create;
    function    Lookup( aKeyNr: TMidiSeptet): Integer;
  public
    property    FileName                    : string  read FFileName           write SetFileName;
    property    MappingCount                : Integer read GetMappingCount;
    property    FirstNote                   : Integer read FFirstNote;
    property    LastNote                    : Integer read FLastNote;
    property    MiddleNote                  : Integer read FMiddleNote;
    property    ReferenceNote               : Integer read FReferenceNote;
    property    ReferenceFrequency          : TSignal read FReferenceFrequency;
    property    ScaleDegree                 : Integer read FScaleDegree;
    property    Mapping[ anIndex: Integer]  : Integer read GetMapping;
  end;


  TScalaScale = class( TStringList)
  // This maps scale numbers to note numbers, or indirectly key numbers to note numbers
  strict private
    type TParseState = (
      spInitial,
      spHasDescription,
      spHasNoteCount
    );
  private
    FFileName          : string;
    FparseState        : TParseState;
    FDescription       : string;
    FPromisedNoteCount : Integer;
    FKeyMapping        : TScalaMapping;
    FNotes             : array of TScalaNoteValue;
    FIntervals         : array of TSignal;
    FLookupTable       : TScalaTuningTable;
  private
    function    GetFirstNote         : Integer;
    function    GetLastNote          : Integer;
    function    GetMiddleNote        : Integer;
    function    GetReferenceNote     : Integer;
    function    GetReferenceFrequency: TSignal;
    function    GetScaleDegree       : Integer;
    procedure   SetFileName          ( const aValue: string);
    function    GetKeyMapFileName    : string;
    procedure   SetKeyMapFileName    ( const aValue: string);
    function    GetNoteCount         : Integer;
    function    GetNote( anIndex: Integer): TScalaNoteValue;
    function    GetNoteType( anIndex: Integer): TScalaNoteType;
    function    GetRational( anIndex: Integer): Trational;
    function    GetCents   ( anIndex: Integer): TSignal;
  private
    procedure   Reject( const aMsg: string);
    procedure   RejectFmt( const aFmt: string; anArgs: array of const);
    procedure   AddRational( aNumerator: Int64; aDenominator: Int64 = 1);
    procedure   AddCents( aValue: TSignal);
    procedure   ParseCents   ( aLineNr: Integer; const aLine: string);
    procedure   ParseRational( aLineNr: Integer; const aLine: string);
    procedure   ParseLine    ( aLineNr: Integer; const aLine: string);
    procedure   MakeCents;
    procedure   BuildNotes;
    procedure   BuildIntervals( aTranspose: Integer);
    procedure   Transpose     ( aTranspose: Integer);
    procedure   FillTable;
    procedure   Parse;
    procedure   SetDefaults;
    function    NoteToMultiplier( aNote: Integer): TSignal;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Recalculate;
    function    NoteNumberToUnits      ( aNoteNr      : TMidiSeptet): TSignal;  // Key note to Wren internal units
    function    NoteNumberToScaleDegree( aNoteNr      : TMidiSeptet): Integer;  // Key note to scale note
    function    ScaleDegreeToUnits     ( aScaleDegree : Integer    ): TSignal;  // Scale note to Wren internal units
  public
    property    FirstNote                   : Integer         read GetFirstNote;
    property    LastNote                    : Integer         read GetLastNote;
    property    MiddleNote                  : Integer         read GetMiddleNote;
    property    ReferenceNote               : Integer         read GetReferenceNote;
    property    ReferenceFrequency          : TSignal         read GetReferenceFrequency;
    property    ScaleDegree                 : Integer         read GetScaleDegree;
    property    FileName                    : string          read FFileName             write SetFileName;
    property    KeyMapFileName              : string          read GetKeyMapFileName     write SetKeyMapFileName;
    property    Description                 : string          read FDescription;
    property    NoteCount                   : Integer         read GetNoteCount;
    property    Note    [ anIndex: Integer] : TScalaNoteValue read GetNote;
    property    NoteType[ anIndex: Integer] : TScalaNoteType  read GetNotetype;
    property    Rational[ anIndex: Integer] : TRational       read GetRational;
    property    Cents   [ anIndex: Integer] : TSignal         read GetCents;
  end;



implementation


  function  ScanNumber( const aLine: string): string;
  const
    Allowed = [ '0' .. '9', '.'];
  var
    i : Integer;
    S : string;
  begin
    S      := Trim( aLine);
    Result := '';

    for i := 1 to Length( S)
    do begin
      if CharInSet( S[ i], Allowed)
      then Result := Result + S[ i]
      else Break;
    end;
  end;


{ ========
  TScalaMappingFile = class( TStringList)
  strict private
    type TParseState = (
      msInitial,
      msHasMapSize,
      msHasFirstNote,
      msHasLastNote,
      msHasMiddleNote,
      msHasReferenceNote,
      msHasReferenceFrequency,
      msHasScaleDegree
    );
  private
    FFileName           : string;           // File name for file  that the mapping was read from
    FParseState         : TParseState;      // The current file parse state
    FMapLimit           : Integer;          // Size of map. The pattern repeats every so many keys:
    FFirstNote          : Integer;          // First MIDI note number to retune
    FLastNote           : Integer;          // Last  MIDI note number to retune
    FMiddleNote         : Integer;          // Middle note where the first entry of the mapping is mapped to
    FReferenceNote      : Integer;          // Reference note for which frequency is given
    FReferenceFrequency : TSignal;          // Frequency to tune the above note to (Hz)
    FScaleDegree        : Integer;          // Scale degree to consider as formal octave (determines difference in pitch
                                            // between adjacent mapping patterns)
    FMapping            : array of Integer; // The numbers represent scale degrees mapped to keys. The first entry is
                                            // for the given middle note, the next for subsequent higher keys.
                                            // For an unmapped key, put in an "x". At the end, unmapped keys may be left
                                            // out.
    FlastMapped         : Integer;
    FLookupTable        : array[ TMidiSeptet] of Integer; // The resulting mapping from key number to scale number
  public
    property    FileName                    : string  read FFileName           write SetFileName;
    property    MappingCount                : Integer read GetMappingCount;
    property    FirstNote                   : Integer read FFirstNote;
    property    LastNote                    : Integer read FLastNote;
    property    MiddleNote                  : Integer read FMiddleNote;
    property    ReferenceNote               : Integer read FReferenceNote;
    property    ReferenceFrequency          : TSignal read FReferenceFrequency;
    property    ScaleDegree                 : Integer read FScaleDegree;
    property    Mapping[ anIndex: Integer]  : Integer read GetMapping;
  private
}

    procedure   TScalaMapping.SetFileName( const aValue: string);
    begin
      FFileName   := aValue;
      FlastMapped := 0;

      if FileExists( FFileName)
      then begin
        LoadFromFile( FFileName);
        Parse;
      end
      else SetDefaultMapping;
    end;


    function    TScalaMapping.GetMappingCount: Integer;
    begin
      Result := Length( FMapping);
    end;


    function    TScalaMapping.GetMapping( anIndex: Integer): Integer;
    begin
      if ( anIndex >= 0) and ( anIndex < MappingCount)
      then Result := FMapping[ anIndex]
      else Result := -1;
    end;


//  private

    procedure   TScalaMapping.Reject( const aMsg: string);
    begin
      raise EScalaReject.Create( aMsg);
    end;


    procedure   TScalaMapping.RejectFmt( const aFmt: string; anArgs: array of const);
    begin
      Reject( Format( aFmt, anArgs));
    end;


    procedure   TScalaMapping.AddMapping( aLineNr: Integer; aValue: Integer);
    begin
      if  MappingCount < FMapLimit
      then begin
        SetLength( FMapping, MappingCount + 1);
        FMapping[ MappingCount - 1] := aValue;
      end
    end;


    procedure   TScalaMapping.ParseLine( aLineNr: Integer; const aLine: string);
    var
      N : Integer;
      F : TSignal;
      p : Integer;
      S : string;
    begin
      if ( Pos( '!', aLine) <> 1) and ( aLine <> '')
      then begin
        case FparseState of

          msInitial :
            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FMapLimit   := N;
                FparseState := msHasMapSize;
              end
              else RejectFmt( 'Invalid map size in line %d', [ aLineNr]);
            end;

          msHasMapSize :
            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FFirstNote  := N;
                FparseState := msHasFirstNote;
              end
              else RejectFmt( 'Invalid first note in line %d', [ aLineNr]);
            end;

          msHasFirstNote :
            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FLastNote   := N;
                FparseState := msHasLastNote;
              end
              else RejectFmt( 'Invalid last note in line %d', [ aLineNr]);
            end;

          msHasLastNote :
            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FMiddleNote := N;
                FparseState := msHasMiddleNote;
              end
              else RejectFmt( 'Invalid middle note in line %d', [ aLineNr]);
            end;

          msHasMiddleNote :
            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FReferenceNote := N;
                FparseState    := msHasReferenceNote;
              end
              else RejectFmt( 'Invalid reference note in line %d', [ aLineNr]);
            end;

          msHasReferenceNote :
            begin
              F := StrToFloatDef( ScanNumber( aLine), NaN);

              if not IsNan( F)
              then begin
                FReferenceFrequency := F;
                FparseState         := msHasReferenceFrequency;
              end
              else RejectFmt( 'Invalid reference frequency in line %d', [ aLineNr]);
            end;

          msHasReferenceFrequency :
            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FScaleDegree := N;
                FparseState  := msHasScaleDegree;
              end
              else RejectFmt( 'Invalid scale degree in line %d', [ aLineNr]);
            end;

          msHasScaleDegree :
            begin
              p := Pos( '!', aLine);

              if p > 0
              then S := Trim( Copy( aLine, 1, p - 1))
              else S := Trim( aLine);

              if Length( S) > 0
              then begin
                if CharInSet( S[ Length( S)], [ 'x', 'X'])
                then AddMapping( aLineNr, FLastMapped)
                else begin
                  N := StrToIntDef( ScanNumber( S), -1);

                  if N >= 0
                  then begin
                    AddMapping( aLineNr, N);
                    FLastMapped := N;
                  end
                  else AddMapping( aLineNr, N)
                end;
              end
            end;

        end; // case
      end;   // if
    end;


    procedure   TScalaMapping.Parse;
    var
      i : Integer;
    begin
      SetLength( FMapping, 0);
      FparseState := msInitial;

      for i := 0 to Count - 1
      do ParseLine( i + 1, Trim( Strings[ i]));

      while MappingCount < FMapLimit
      do AddMapping( 0, FLastNote);

      FillTable;
    end;


    procedure   TScalaMapping.FillTable;
    var
      i : Integer;
      m : Integer;
      d : Integer;
    begin
      if FMapLimit <= 0 // Use default 1:1 mapping
      then begin
        if ScaleDegree = 0
        then FScaleDegree := 12;

        FMapLimit := ScaleDegree;
        SetLength( FMapping, ScaleDegree);

        for i := 0 to ScaleDegree - 1
        do FMapping[ i] := i;
      end;

      for i := Low( FLookupTable) to High( FLookupTable)
      do begin
        m := MathIntMod(( i - MiddleNote), MappingCount);
        d := MathIntDiv(( i - MiddleNote), MappingCount);
        FLookupTable[ i] := FMapping[ m] + ScaleDegree * d;
      end;
    end;


    procedure   TScalaMapping.SetDefaultMapping;
    begin
      FMapLimit           :=   0; // Uses default mapping
      FFirstNote          :=   0;
      FLastNote           := 127;
      FMiddleNote         :=   0;
      FReferenceNote      :=  69;
      FReferenceFrequency := 440;
      FScaleDegree        :=   0;
      FillTable;
    end;


//  public

    constructor TScalaMapping.Create;
    begin
      inherited Create;
      SetDefaultMapping;
    end;


    function    TScalaMapping.Lookup( aKeyNr: TMidiSeptet): Integer;
    begin
      if aKeyNr in [ FirstNote .. LastNote]
      then Result := FLookupTable[ aKeyNr]
      else Result := 0;
    end;



{ ========
  TScalaScaleFile = class( TStringList)
  strict private
    type TParseState = (
      spInitial,
      spHasDescription,
      spHasNoteCount
    );
  private
    FFileName          : string;
    FparseState        : TParseState;
    FDescription       : string;
    FPromisedNoteCount : Integer;
    FKeyMapping        : TScalaMapping;
    FNotes             : array of TScalaNoteValue;
    FIntervals         : array of TSignal;
    FLookupTable       : TScalaTuningTable;
  public
    property    FirstNote                   : Integer         read GetFirstNote;
    property    LastNote                    : Integer         read GetLastNote;
    property    MiddleNote                  : Integer         read GetMiddleNote;
    property    ReferenceNote               : Integer         read GetReferenceNote;
    property    ReferenceFrequency          : TSignal         read GetReferenceFrequency;
    property    ScaleDegree                 : Integer         read GetScaleDegree;
    property    FileName                    : string          read FFileName             write SetFileName;
    property    KeyMapFileName              : string          read GetKeyMapFileName     write SetKeyMapFileName;
    property    Description                 : string          read FDescription;
    property    NoteCount                   : Integer         read GetNoteCount;
    property    Note    [ anIndex: Integer] : TScalaNoteValue read GetNote;
    property    NoteType[ anIndex: Integer] : TScalaNoteType  read GetNotetype;
    property    Rational[ anIndex: Integer] : TRational       read GetRational;
    property    Cents   [ anIndex: Integer] : TSignal         read GetCents;
  private
}

    function    TScalaScale.GetFirstNote: Integer;
    begin
      Result := FKeyMapping.FirstNote;
    end;


    function    TScalaScale.GetLastNote: Integer;
    begin
      Result := FKeyMapping.LastNote;
    end;


    function    TScalaScale.GetMiddleNote: Integer;
    begin
      Result := FKeyMapping.MiddleNote;
    end;


    function    TScalaScale.GetReferenceNote: Integer;
    begin
      Result := FKeyMapping.ReferenceNote;
    end;


    function    TScalaScale.GetReferenceFrequency: TSignal;
    begin
      Result := FKeyMapping.ReferenceFrequency;
    end;


    function    TScalaScale.GetScaleDegree: Integer;
    begin
      Result := FKeyMapping.ScaleDegree;
    end;


    procedure   TScalaScale.SetFileName( const aValue: string);
    begin
      FFileName := aValue;

      if FileExists( FFileName)
      then begin
        LoadFromFile( FFileName);
        Parse;
      end
      else SetDefaults;
    end;


    function    TScalaScale.GetKeyMapFileName: string;
    begin
      Result := FKeyMapping.FileName;
    end;


    procedure   TScalaScale.SetKeyMapFileName( const aValue: string);
    begin
      if aValue <> FKeyMapping.FileName
      then begin
        FKeyMapping.FileName := aValue;
        Parse;                              // todo: Should reparse, or else multiple transpositions take place?
     // FillTable;                          // Parse already does FillTable, and FillTable does the trasposition
      end;
    end;


    function    TScalaScale.GetNoteCount: Integer;
    begin
      Result := Length( FNotes);
    end;


    function    TScalaScale.GetNote( anIndex: Integer): TScalaNoteValue;
    begin
      Result := FNotes[ anIndex];
    end;


    function    TScalaScale.GetNoteType( anIndex: Integer): TScalaNoteType;
    begin
      Result := FNotes[ anIndex].NoteType;
    end;


    function    TScalaScale.GetRational( anIndex: Integer): Trational;
    begin
      Result := 0;

      if NoteType[ anIndex] = stRational
      then Result := FNotes[ anIndex].Rational
      else Reject( 'note type is not rational')
    end;


    function    TScalaScale.GetCents( anIndex: Integer): TSignal;
    begin
      Result := 0;

      if NoteType[ anIndex] = stCents
      then Result := FNotes[ anIndex].Cents
      else Reject( 'Note type is not cents');
    end;


//  private

    procedure   TScalaScale.Reject( const aMsg: string);
    begin
      raise EScalaReject.Create( aMsg);
    end;


    procedure   TScalaScale.RejectFmt( const aFmt: string; anArgs: array of const);
    begin
      Reject( Format( aFmt, anArgs));
    end;


    procedure   TScalaScale.AddRational( aNumerator: Int64; aDenominator: Int64 = 1);
    begin
      SetLength( FNotes, Length( FNotes) + 1);
      FNotes[ NoteCount - 1].NoteType := stRational;
      FNotes[ NoteCount - 1].Rational := TRational.Create( aNumerator, aDenominator);
    end;


    procedure   TScalaScale.AddCents( aValue: TSignal);
    begin
      SetLength( FNotes, Length( FNotes) + 1);
      FNotes[ NoteCount - 1].NoteType := stCents;
      FNotes[ NoteCount - 1].Cents    := aValue;
    end;


    procedure   TScalaScale.ParseCents( aLineNr: Integer; const aLine: string);
    var
      N : TSignal;
    begin
      N := StrToFloatDef( ScanNumber( aLine), NaN);

      if IsNan( N)
      then RejectFmt( 'Invalid cents spec in line %d', [ aLineNr])
      else AddCents( N);
    end;


    procedure   TScalaScale.ParseRational( aLineNr: Integer; const aLine: string);
    var
      S      : string;
      p      : Integer;
      aParts : TStringList;
      N1     : Integer;
      N2     : Integer;
      aCode  : Integer;
    begin
      p := Pos( ' ', aLine);

      if p <= 0
      then p := Length( aLine);

      S := Copy( aLine, 1, p);
      aParts := Explode( S, '/');

      try
        if aParts.Count = 1
        then begin
          Val( ScanNumber( aParts[ 0]), N1, aCode);

          if aCode = 0
          then AddRational( N1)
          else RejectFmt( 'invalid note in line %d', [ aLineNr]);
        end
        else if aParts.Count = 2
        then begin
          Val( ScanNumber( aParts[ 0]), N1, aCode);
          if aCode = 0

          then begin
            Val( ScanNumber( aParts[ 1]), N2, aCode);
            if aCode = 0
            then AddRational( N1, N2)
            else RejectFmt( 'invalid note in line %d', [ aLineNr]);
          end
          else RejectFmt( 'invalid note in line %d', [ aLineNr]);
        end
        else RejectFmt( 'invalid note in line %d', [ aLineNr]);
      finally
        aParts.DisposeOf;
      end;
    end;


    procedure   TScalaScale.ParseLine( aLineNr: Integer; const aLine: string);
    var
      N : Integer;
    begin
      if ( Pos( '!', aLine) <> 1) and ( aLine <> '')
      then begin
        case FparseState of

          spInitial :

            begin
              FDescription := aLine;
              FparseState  := spHasDescription;
            end;

          spHasDescription :

            begin
              N := StrToIntDef( ScanNumber( aLine), -1);

              if N >= 0
              then begin
                FPromisedNoteCount := N;
                AddRational( 1);                       // Add the implied 1/1 at the beginning
                FparseState := spHasNoteCount;
              end
              else RejectFmt( 'Invalid note count in line %d', [ aLineNr]);
            end;

          spHasNoteCount :

            begin
              if Pos( '.', aLine) > 0
              then ParseCents   ( aLineNr, aLine)
              else ParseRational( aLineNr, aLine);
            end;

        end;
      end;
    end;


    procedure   TScalaScale.MakeCents;
    var
      i      : Integer;
      aValue : TSignal;
      aCents : TSignal;
    begin
      for i := 0 to NoteCount - 1
      do begin
        if Note[ i].NoteType = stRational
        then begin
          aValue := Note[ i].Rational.AsDouble;
          aCents := 1200 * Log2( aValue);
          FNotes[ i].NoteType := stCents;
          FNotes[ i].Cents    := aCents;
        end;
      end;
    end;


    procedure   TScalaScale.BuildNotes;
    var
      i : Integer;
    begin
      // Note : FNotes[ 0] always is the base note .. the 1/1 implicit thing, so start at offset 1 here

      FNotes[ 1].NoteType := stCents;
      FNotes[ 1].Cents    := 0;

      for i := 1 to NoteCount - 1
      do begin
        FNotes[ i].NoteType := stCents;
        FNotes[ i].Cents := FNotes[ i - 1].Cents + FIntervals[ i - 1];
      end;
    end;


    procedure   TScalaScale.BuildIntervals( aTranspose: Integer);
    var
      i : Integer;
    begin
      MakeCents;

      SetLength( FIntervals, NoteCount - 1);

      for i := 0 to Length( FIntervals) - 1
      do FIntervals[ MathIntMod( i - aTranspose, Length( FIntervals))] := Note[ i + 1].Cents - Note[ i].Cents;
    end;


    procedure   TScalaScale.Transpose( aTranspose: Integer);
    begin
      BuildIntervals( aTranspose);
      BuildNotes;
    end;


    procedure   TScalaScale.FillTable;
    var
      aFrequency    : TSignal;
      anOctave      : Integer;
      aNote         : Integer;
      anOctaveMult  : TSignal;
      i             : Integer;
      aTranspose    : Integer;
      aScaleLength  : Integer;
    begin
      // Last item in FNotes (index NoteCount - 1) holds the 'octave' multiplier
      aScaleLength := NoteCount - 1;
      anOctaveMult := NoteToMultiplier( aScaleLength);
      aTranspose   := MathIntMod( ReferenceNote, aScaleLength);

      Transpose( aTranspose);

      for i := Low( FLookupTable) to High( FLookupTable)
      do begin
        anOctave    := MathIntDiv( i - ReferenceNote, aScaleLength);
        aNote       := MathIntMod( i - ReferenceNote, aScaleLength);
        aFrequency  := ReferenceFrequency * Power( anOctaveMult, anOctave) * NoteToMultiplier( aNote);
        FLookupTable[ i].Scale     := i - MiddleNote;
        FLookupTable[ i].Units     := FrequencyToUnits( aFrequency);
        FLookupTable[ i].Frequency := aFrequency;
      end;
    end;


    procedure   TScalaScale.Parse;
    var
      i : Integer;
    begin
      SetLength( FNotes, 0);
      FparseState := spInitial;

      for i := 0 to Count - 1                    // We are a stringlist .. so we have Count and Strings
      do ParseLine( i + 1, Trim( Strings[ i]));

      if FPromisedNoteCount <> NoteCount - 1
      then Reject( 'Given note count does not match the actual number of note specifications found');

      FillTable;
    end;


    procedure   TScalaScale.SetDefaults;
    var
      i : Integer;
      v : TSignal;
    begin
      AddRational( 1, 1);
      v := 100.0;

      for i := 1 to 11 do
      begin
        AddCents( v);
        v := v + 100.0;
      end;

      AddRational( 2, 1);
      FillTable;
    end;


    function    TScalaScale.NoteToMultiplier( aNote: Integer): TSignal;
    begin
      if FNotes[ aNote].NoteType = stRational
      then Result := FNotes[ aNote].Rational.AsDouble
      else Result := Power( 2, FNotes[ aNote].Cents / 1200.0);
    end;


//  public

    constructor TScalaScale.Create;
    begin
      inherited Create;
      FKeyMapping := TScalaMapping.Create;
      SetDefaults;
    end;


    destructor  TScalaScale.Destroy; // override;
    begin
      FreeAndNil( FKeyMapping);
      inherited;
    end;


    procedure   TScalaScale.Recalculate;
    begin
      Parse;
    end;


    function    TScalaScale.NoteNumberToUnits( aNoteNr: TMidiSeptet): TSignal;
    // Key note to Wren internal units
    var
      aScaleDegree : Integer;
    begin
      aScaleDegree := NoteNumberToScaleDegree( aNoteNr     );
      Result       := ScaleDegreeToUnits     ( aScaleDegree);
    end;


    function    TScalaScale.NoteNumberToScaleDegree( aNoteNr: TMidiSeptet): Integer;
    // Key note to scale note
    begin
      Result := FKeyMapping.Lookup( aNoteNr + MiddleNote) + ReferenceNote;
    end;


    function    TScalaScale.ScaleDegreeToUnits( aScaleDegree : Integer): TSignal;
    // Scale note to Wren internal units
    begin
      Result := FLookupTable[ Clip( aScaleDegree, Low( FLookupTable), High( FLookupTable))].Units - ( KnobsConversions.MiddleNote / 128);
    end;


end.

