unit synth_info;

{

   COPYRIGHT 2015 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.Classes, System.SysUtils, System.TypInfo,

  KnobsUtils, knobs2013, KnobsConversions, module_defs;

type

  ESynthInfo      = class( Exception);
  EControlMapping = class( ESynthInfo);


  TOnLookupModuleClass = function ( const aModuleType: Integer): TClass;

  TKnobsControlKind = (
    ckInput,
    ckOutput,
    ckInternal,
    ckString,
    ckInternalString
  );

  TKnobsSignalSpec = record
  public
    Index                  : Byte;
    ClassType              : TClass;
    Name                   : string;
    ControlType            : string;
    ControlKind            : TKnobsControlKind;
    SignalType             : TSignalType;
    StepCount              : Integer;
    Position               : Integer;
    DefaultPosition        : Integer;
    Value                  : string;
    PairedWith             : string;
    Locked                 : Boolean;
    Dezippered             : Boolean;
    DynamicStepCount       : Boolean;
    AllowsAutomation       : Boolean;
    MidiCC                 : Byte;
    AllowRandomization     : Boolean;
    ShowAllowRandomization : Boolean;
    AllowVariations        : Boolean;
    VariationCount         : Integer;
  end;

  TKnobsSignalSpecs = array of TKnobsSignalSpec;


  TKnobsModuleSignalSpec = class
  private
    FModuleName             : string;
    FModuleType             : TKnobsModuleType;
    FModuleClass            : TClass;
    FTitle                  : string;
    FComment                : string;
    FSignalSpecs            : TKnobsSignalSpecs;
    FInputCount             : Integer;
    FOutputCount            : Integer;
    FInternalCount          : Integer;
    FStringCount            : Integer;
    FInternalStringCount    : Integer;
    FDezipperMap            : TByteset;
    FAllowSignalConversion  : Boolean;
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FHasSideEffects         : Boolean;
    FTemplateName           : string;
    FActiveVariation        : Integer;
  private
    function    GetSignalCount: Integer;
    function    GetSignalSpec( anIndex: Integer): TKnobsSignalSpec;
    procedure   AddSignal(
      const aName             : string;
      aClassType              : TClass;
      const aControlType      : string;
      aControlKind            : TKnobsControlKind;
      aSignalType             : TSignalType;
      aStepCount              : Integer;
      aPosition               : Integer;
      aDefaultPosition        : Integer;
      const aValue            : string;
      const aPaired           : string;
      aLocked                 : Boolean;
      aDezip                  : Boolean;
      aDynamic                : Boolean;
      anAllowsAutomation      : Boolean;
      aMidiCC                 : Byte;
      anAllowRandomization    : Boolean;
      aShowAllowRandomization : Boolean;
      anAllowVariations       : Boolean;
      aVariationCount         : Integer
    );
    procedure   AddValuedControl( const aValue: TKnobsValuedControl);
    procedure   AddConnector( const aValue: TKnobsConnector);
    procedure   AddFileSelector( const aValue: TKnobsFileSelector);
    procedure   AddPad( const aValue: TKnobsPad);
    procedure   AddDisplay( const aValue: TKnobsDisplay);
    procedure   MakeDezipperMap;
    procedure   DumpItem( anInd, anIndex: Integer; const aStringList: TStringList);
  public
    constructor Create( const aModule: TKnobsCustomModule);
    function    CountControlKind( aControlKind: TKnobsControlKind): Integer;
    function    FindSignalOfKindByName( const aName: string; aControlKind: TKnobsControlKind): Integer;
    procedure   ApplyControlMapping       ( const aVars: array of Byte; const aNames: array of string; aControlKind: TKnobsControlKind);
    procedure   ApplyInputMapping         ( const aVars: array of Byte; const aNames: array of string);
    procedure   ApplyOutputMapping        ( const aVars: array of Byte; const aNames: array of string);
    procedure   ApplyInternalMapping      ( const aVars: array of Byte; const aNames: array of string);
    procedure   ApplyStringMapping        ( const aVars: array of Byte; const aNames: array of string);
    procedure   ApplyInternalStringMapping( const aVars: array of Byte; const aNames: array of string);
    procedure   TestApply;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
  public
    property    ModuleName                    : string           read FModuleName;
    property    ModuleType                    : TKnobsModuleType read FModuleType;
    property    ModuleClass                   : TClass           read FModuleClass;
    property    Title                         : string           read FTitle;
    property    Comment                       : string           read FComment;
    property    SignalCount                   : Integer          read GetSignalCount;
    property    SignalSpec[ anIndex: Integer] : TKnobsSignalSpec read GetSignalSpec;
    property    DezipperMap                   : TByteSet         read FDezipperMap;
    property    AllowSignalConversion         : Boolean          read FAllowSignalConversion;
    property    AllowRandomization            : Boolean          read FAllowRandomization;
    property    ShowAllowRandomization        : Boolean          read FShowAllowRandomization;
    property    HasSideEffects                : Boolean          read FHasSideEffects;
    property    TemplateName                  : string           read FTemplateName;
    property    ActiveVariation               : Integer          read FActiveVariation;
  end;


  TKnobsModuleSignalSpecs = class
  private
    FSpecs : array of TKnobsModuleSignalSpec;
  private
    function    GetModuleCount: Integer;
    function    GetModuleName ( aModuleIndex: Integer): string;
    function    GetModuleType ( aModuleIndex: Integer): TKnobsModuleType;
    function    GetSignalCount( aModuleIndex: Integer): Integer;
    function    GetSignalSpec ( aModuleIndex, aSignalIndex: Integer): TKnobsSignalSpec;
    procedure   DumpItem( anInd, anIndex: Integer; const aStringList: TStringList);
  public
    constructor Create( const anEditorPatch: TKnobsWirePanel);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Load( const anEditorPatch: TKnobsWirePanel);
    procedure   AddModule( const aModule: TKnobsCustomModule);
    procedure   Dump( const aStringList: TStringList);                                                         overload;
    procedure   Dump( const aFileName: string);                                                                overload;
  public
    property    ModuleCount                                       : Integer          read GetModuleCount;
    property    ModuleName [ aModuleIndex: Integer]               : string           read GetModuleName;
    property    ModuleType [ aModuleIndex: Integer]               : TKnobsModuleType read GetModuleType;
    property    SignalCount[ aModuleIndex: Integer]               : Integer          read GetSignalCount;
    property    SignalSpec [ aModuleIndex, aSignalIndex: Integer] : TKnobsSignalSpec read GetSignalSpec;
  end;


  function  ControlKindToStr( const aValue: TKnobsControlKind): string;

var

  GOnLookupModuleClass : TOnLookupModuleClass;



implementation



  function  ControlKindToStr( const aValue: TKnobsControlKind): string;
  begin
    Result := GetEnumName( TypeInfo( TKnobsControlKind), Ord( aValue));
  end;


  function  LookupModuleClass( aModuleType: TKnobsModuleType): TClass;
  begin
    if Assigned( GOnLookupModuleClass)
    then Result := GOnLookupModuleClass( aModuleType)
    else Result := nil;
  end;


{ ========
  TKnobsControlKind = (
    ckInput,
    ckOutput,
    ckInternal,
    ckString,
    ckInternalString
  );

  TKnobsSignalSpec = record
  public
    Index            : Byte;
    ClassType        : TClass;
    Name             : string;
    ControlType      : string;
    ControlKind      : TKnobsControlKind;
    SignalType       : TSignalType;
    StepCount        : Integer;
    Position         : Integer;
    DefaultPosition  : Integer;
    Value            : string;
    PairedWith       : string;
    Locked           : Boolean;
    Dezippered       : Boolean;
    DynamicStepCount : Boolean;
  end;

  TKnobsSignalSpecs = array of TKnobsSignalSpec;

  TKnobsModuleSignalSpec = class
  private
    FModuleName             : string;
    FModuleType             : TKnobsModuleType;
    FModuleClass            : TClass;
    FTitle                  : string;
    FComment                : string;
    FSignalSpecs            : TKnobsSignalSpecs;
    FInputCount             : Integer;
    FOutputCount            : Integer;
    FInternalCount          : Integer;
    FStringCount            : Integer;
    FInternalStringCount    : Integer;
    FDezipperMap            : TByteset;
    FAllowSignalConversion  : Boolean;
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FHasSideEffects         : Boolean;
    FTemplateName           : string;
    FActiveVariation        : Integer;
  public
    property    ModuleName                    : string           read FModuleName;
    property    ModuleType                    : TKnobsModuleType read FModuleType;
    property    ModuleClass                   : TClass           read FModuleClass;
    property    Title                         : string           read FTitle;
    property    Comment                       : string           read FComment;
    property    SignalCount                   : Integer          read GetSignalCount;
    property    SignalSpec[ anIndex: Integer] : TKnobsSignalSpec read GetSignalSpec;
    property    DezipperMap                   : TByteSet         read FDezipperMap;
    property    AllowSignalConversion         : Boolean          read FAllowSignalConversion;
    property    AllowRandomization            : Boolean          read FAllowRandomization;
    property    ShowAllowRandomization        : Boolean          read FShowAllowRandomization;
    property    HasSideEffects                : Boolean          read FHasSideEffects;
    property    TemplateName                  : string           read FTemplateName;
    property    ActiveVariation               : Integer          read FActiveVariation;
  private
}


    function    TKnobsModuleSignalSpec.GetSignalCount: Integer;
    begin
      Result := Length( FSignalSpecs);
    end;


    function    TKnobsModuleSignalSpec.GetSignalSpec( anIndex: Integer): TKnobsSignalSpec;
    begin
      Result := FSignalSpecs[ anIndex];
    end;


    procedure   TKnobsModuleSignalSpec.AddSignal(
      const aName             : string;
      aClassType              : TClass;
      const aControlType      : string;
      aControlKind            : TKnobsControlKind;
      aSignalType             : TSignalType;
      aStepCount              : Integer;
      aPosition               : Integer;
      aDefaultPosition        : Integer;
      const aValue            : string;
      const aPaired           : string;
      aLocked                 : Boolean;
      aDezip                  : Boolean;
      aDynamic                : Boolean;
      anAllowsAutomation      : Boolean;
      aMidiCC                 : Byte;
      anAllowRandomization    : Boolean;
      aShowAllowRandomization : Boolean;
      anAllowVariations       : Boolean;
      aVariationCount         : Integer
    );
    var
      anIndex    : Integer;
      aShortName : string;
      aParts     : TStringList;
    begin
      SetLength( FSignalSpecs, SignalCount + 1);

      case aControlKind of
        ckInput          : begin anIndex := FInputCount         ; Inc( FInputCount         ); end;
        ckOutput         : begin anIndex := FOutputCount        ; Inc( FOutputCount        ); end;
        ckInternal       : begin anIndex := FInternalCount      ; Inc( FInternalCount      ); end;
        ckString         : begin anIndex := FStringCount        ; Inc( FStringCount        ); end;
        ckInternalString : begin anIndex := FInternalStringCount; Inc( FInternalStringCount); end;
        else                     anIndex := -1;
      end;

      aParts := Explode( aName, '_');
      try
        if aparts.Count = 2
        then aShortName := aParts[ 1]
        else aShortName := 'invalid name: ''' + aName + '''';
      finally
        aParts.DisposeOf;
      end;

      FSignalSpecs[ SignalCount - 1].Index                  := anIndex;
      FSignalSpecs[ SignalCount - 1].ClassType              := aClassType;
      FSignalSpecs[ SignalCount - 1].Name                   := aShortName;
      FSignalSpecs[ SignalCount - 1].ControlType            := aControlType;
      FSignalSpecs[ SignalCount - 1].ControlKind            := aControlKind;
      FSignalSpecs[ SignalCount - 1].SignalType             := aSignalType;
      FSignalSpecs[ SignalCount - 1].StepCount              := aStepCount;
      FSignalSpecs[ SignalCount - 1].Position               := aPosition;
      FSignalSpecs[ SignalCount - 1].DefaultPosition        := aDefaultPosition;
      FSignalSpecs[ SignalCount - 1].Value                  := aValue;
      FSignalSpecs[ SignalCount - 1].PairedWith             := aPaired;
      FSignalSpecs[ SignalCount - 1].Locked                 := aLocked;
      FSignalSpecs[ SignalCount - 1].Dezippered             := aDezip;
      FSignalSpecs[ SignalCount - 1].DynamicStepCount       := aDynamic;
      FSignalSpecs[ SignalCount - 1].AllowsAutomation       := anAllowsAutomation;
      FSignalSpecs[ SignalCount - 1].MidiCC                 := aMidiCC;
      FSignalSpecs[ SignalCount - 1].AllowRandomization     := anAllowRandomization;
      FSignalSpecs[ SignalCount - 1].ShowAllowRandomization := aShowAllowRandomization;
      FSignalSpecs[ SignalCount - 1].AllowVariations        := anAllowVariations;
      FSignalSpecs[ SignalCount - 1].VariationCount         := aVariationCount;
    end;


    procedure   TKnobsModuleSignalSpec.AddValuedControl( const aValue: TKnobsValuedControl);
    var
      aControlKind : TKnobsControlKind;
      aPairing     : string;
      aKnob        : TKnobsKnob;
      aParts       : TStringList;
    begin
      if aValue.Tag < 0
      then aControlKind := ckInternal
      else aControlKind := ckInput;

      aPairing := '';

      if aValue is TKnobsKnob
      then begin
        aKnob := TKnobsKnob( aValue).PairedWith;
        if Assigned( aKnob)
        then begin
          aParts := Explode( aKnob.Name, '_');

          try
            if aParts.Count = 2
            then aPairing := aParts[ 1];
          finally
            aParts.DisposeOf;
          end;
        end;
      end;

      AddSignal(
        aValue.Name,
        aValue.ClassType,
        aValue.ControlType,
        aControlKind,
        stControl,
        aValue.StepCount,
        aValue.KnobPosition,
        aValue.DefaultPosition,
        FloatToStr( aValue.AsValue),
        aPairing,
        aValue.Locked,
        aValue.Dezippered,
        aValue.DynamicStepCount,
        aValue.AllowAutomation,
        aValue.AssignedMIDICC,
        aValue.AllowRandomization,
        aValue.ShowAllowRandomization,
        aValue.AllowVariations,
        aValue.VariationCount
      );
    end;


    procedure   TKnobsModuleSignalSpec.AddConnector( const aValue: TKnobsConnector);
    var
      aControlKind : TKnobsControlKind;
    begin
      if aValue is TKnobsInput
      then aControlKind := ckInput
      else aControlKind := ckOutput;

      AddSignal( aValue.Name, aValue.ClassType, '', aControlKind, aValue.Signaltype, -1, -1, -1, '0', '', False, False, False, False, 0, False, False, False, 0);
    end;


    procedure   TKnobsModuleSignalSpec.AddFileSelector( const aValue: TKnobsFileSelector);
    begin
      AddSignal( aValue.Name, aValue.ClassType, '', ckInternalString, stControl, -1, -1, -1, aValue.DataFileName, '', False, False, False, False, 0, False, False, False, 0);
    end;


    procedure   TKnobsModuleSignalSpec.AddPad( const aValue: TKnobsPad);
    begin
      AddSignal( aValue.Name + 'x', aValue.ClassType, aValue.ControlType, ckInput, stControl, aValue.StepCount, aValue.KnobPositionX, aValue.DefaultPositionX, FloatToStr( aValue.AsValueX), '', False, True, False, False, 0, aValue.AllowRandomization, aValue.ShowAllowRandomization, aValue.AllowVariations, aValue.VariationCount);
      AddSignal( aValue.Name + 'y', aValue.ClassType, aValue.ControlType, ckInput, stControl, aValue.StepCount, aValue.KnobPositionY, aValue.DefaultPositionY, FloatToStr( aValue.AsValueY), '', False, True, False, False, 0, aValue.AllowRandomization, aValue.ShowAllowRandomization, aValue.AllowVariations, aValue.VariationCount);
    end;


    procedure   TKnobsModuleSignalSpec.AddDisplay( const aValue: TKnobsDisplay);
    begin
      if aValue.MustNotify
      then AddSignal( aValue.Name, aValue.ClassType, '', ckString, stControl, -1, -1, -1, aValue.TextValue, '', False, False, False, False, 0, False, False, False, 0);
    end;


    procedure   TKnobsModuleSignalSpec.MakeDezipperMap;
    const
      MethodName = 'MakeDezipperMap';
    var
      i : Integer;
      q : Byte;
    begin
      FDezipperMap := [];

      for i := 0 to SignalCount - 1
      do begin
        if SignalSpec[ i].Dezippered
        then begin
          if SignalSpec[ i].ControlKind = ckInput
          then begin
            q := SignalSpec[ i].Index;

            if q in FDezipperMap
            then raise EControlMapping.CreateFmt( '%s.%s: Duplicate index %d in dezipper map', [ UnitName, MethodName, q])
            else FDezipperMap := FDezipperMap + [ q];
          end
          else raise EControlMapping.CreateFmt( '%s.%s: ControlKind for Dezipperd signal %d named %s is %s, only ckInput signals should be dezippered.', [ UnitName, MethodName, i, SignalSpec[ i].Name, ControlKindToStr( SignalSpec[ i].ControlKind)])
        end;
      end;
    end;


    procedure   TKnobsModuleSignalSpec.DumpItem( anInd, anIndex: Integer; const aStringList: TStringList);
    begin
      if Assigned( aStringList) and ( anIndex >= 0) and ( anIndex < SignalCount)
      then begin
        aStringList.Add( Format( '%s('                                , [ Indent( anInd)                                                                            ], AppLocale));
        aStringList.Add( Format(   '%sIndex                  = %d'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].Index                        ], AppLocale));
        aStringList.Add( Format(   '%sClassType              = %s'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].ClassType.ClassName          ], AppLocale));
        aStringList.Add( Format(   '%sName                   = ''%s''', [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].Name                         ], AppLocale));
        aStringList.Add( Format(   '%sControlType            = ''%s''', [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].ControlType                  ], AppLocale));
        aStringList.Add( Format(   '%sControlKind            = %s'    , [ Indent( anInd + 1), ControlKindToStr( FSignalSpecs[ anIndex].ControlKind                 )], AppLocale));
        aStringList.Add( Format(   '%sSignalType             = %s'    , [ Indent( anInd + 1), SignalTypeToStr ( FSignalSpecs[ anIndex].SignalType                  )], AppLocale));
        aStringList.Add( Format(   '%sStepCount              = %d'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].StepCount                    ], AppLocale));
        aStringList.Add( Format(   '%sPosition               = %d'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].Position                     ], AppLocale));
        aStringList.Add( Format(   '%sDefaultPosition        = %d'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].DefaultPosition              ], AppLocale));
        aStringList.Add( Format(   '%sValue                  = ''%s''', [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].Value                        ], AppLocale));
        aStringList.Add( Format(   '%sPairedWith             = ''%s''', [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].PairedWith                   ], AppLocale));
        aStringList.Add( Format(   '%sLocked                 = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].Locked                , True)], AppLocale));
        aStringList.Add( Format(   '%sDezippered             = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].Dezippered            , True)], AppLocale));
        aStringList.Add( Format(   '%sDynamicStepCount       = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].DynamicStepCount      , True)], AppLocale));
        aStringList.Add( Format(   '%sAllowsAutomation       = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].AllowsAutomation      , True)], AppLocale));
        aStringList.Add( Format(   '%sMidiCC                 = %d'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].MidiCC                       ], AppLocale));
        aStringList.Add( Format(   '%sAllowRandomization     = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].AllowRandomization    , True)], AppLocale));
        aStringList.Add( Format(   '%sShowAllowRandomization = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].ShowAllowRandomization, True)], AppLocale));
        aStringList.Add( Format(   '%sAllowVariations        = %s'    , [ Indent( anInd + 1), BoolToStr       ( FSignalSpecs[ anIndex].AllowVariations       , True)], AppLocale));
        aStringList.Add( Format(   '%sVariationCount         = %d'    , [ Indent( anInd + 1),                   FSignalSpecs[ anIndex].VariationCount               ], AppLocale));
        aStringList.Add( Format( '%s)'                                , [ Indent( anInd)                                                                            ], AppLocale));
      end;
    end;

//  public


    constructor TKnobsModuleSignalSpec.Create( const aModule: TKnobsCustomModule);
    var
      i : Integer;
    begin
      inherited Create;

      FInputCount          := 0;
      FOutputCount         := 0;
      FInternalCount       := 0;
      FStringCount         := 0;
      FInternalStringCount := 0;

      if Assigned( aModule)
      then begin
        FModuleName             := aModule.Name;
        FModuleType             := aModule.ModuleType;
        FModuleClass            := LookupModuleClass( FModuleType);
        FTitle                  := aModule.Title;
        FComment                := aModule.Comment;
        FAllowSignalConversion  := aModule.AllowSignalConversion;
        FAllowRandomization     := aModule.AllowRandomization;
        FShowAllowRandomization := aModule.ShowAllowRandomization;
        FHasSideEffects         := aModule.HasSideEffects;
        FTemplateName           := aModule.TemplateName;
        FActiveVariation        := aModule.ActiveVariation;

        with aModule
        do begin
          for i := 0 to ControlCount - 1
          do begin
            if      Controls[ i] is TKnobsValuedControl then AddValuedControl( TKnobsValuedControl( Controls[ i]))
            else if Controls[ i] is TKnobsConnector     then AddConnector    ( TKnobsConnector    ( Controls[ i]))
            else if Controls[ i] is TKnobsFileSelector  then AddFileSelector ( TKnobsFileSelector ( Controls[ i]))
            else if Controls[ i] is TKnobsPad           then AddPad          ( TKnobsPad          ( Controls[ i]))
            else if Controls[ i] is TKnobsDisplay       then AddDisplay      ( TKnobsDisplay      ( Controls[ i]));
          end;
        end;
      end;

      MakeDezipperMap;
    end;


    function    TKnobsModuleSignalSpec.CountControlKind( aControlKind: TKnobsControlKind): Integer;
    begin
      Result := -1;
      case aControlKind of
        ckInput          : Result := FInputCount         ;
        ckOutput         : Result := FOutputCount        ;
        ckInternal       : Result := FInternalCount      ;
        ckString         : Result := FStringCount        ;
        ckInternalString : Result := FInternalStringCount;
      end;
    end;


    function    TKnobsModuleSignalSpec.FindSignalOfKindByName( const aName: string; aControlKind: TKnobsControlKind): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to SignalCount - 1
      do begin
        if SameText( aName, SignalSpec[ i].Name) and ( aControlKind = SignalSpec[ i].ControlKind)
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TKnobsModuleSignalSpec.ApplyControlMapping( const aVars: array of Byte; const aNames: array of string; aControlKind: TKnobsControlKind);
    const
      MethodName = 'ApplyControlMapping';
    var
      i     : Integer;
      p     : Integer;
      q     : Integer;
      Check : TByteSet;
    begin
      if Length( aVars) = CountControlKind( aControlKind)
      then begin
        if Length( aVars) = Length( aNames)
        then begin
          Check := [];

          for i := Low( aVars) to High( aVars)
          do begin
            p := FindSignalOfKindByName( aNames[ i], aControlKind);

            if p >= 0
            then begin
              q := aVars[ i];

              if q in Check
              then raise EControlMapping.CreateFmt( '%s.%s: Duplicate index %d in control map', [ UnitName, MethodName, q])
              else begin
                FSignalSpecs[ p].Index := q;
                Check := Check + [ q];
              end;
            end
            else raise EControlMapping.CreateFmt( '%s.%s: could not find a signal named %s of kind %s in the signals inferred from the designer', [ UnitName, MethodName, aNames[ i], ControlKindToStr( aControlKind)]);
          end;
          if aControlKind = ckInput
          then MakeDezipperMap;
        end
        else raise EControlMapping.CreateFmt( '%s.%s: mismatch in sizes of aVars (%d) and aNames (%d), the size given for aVars matches the size inferred from the designer', [ UnitName, MethodName, Length( aVars), Length( aNames)]);
      end
      else
        raise
          EControlMapping.
            CreateFmt(
              '%s.%s: mismatch in size of passed aVars (%d) and actual signals of kind %s present (%d) for Module type %s as inferred from designer',
              [
                UnitName,
                MethodName,
                Length( aVars),
                ControlKindToStr( aControlKind),
                CountControlKind( aControlKind),
                ModuleClass
              ]
            );
    end;


    procedure   TKnobsModuleSignalSpec.ApplyInputMapping( const aVars: array of Byte; const aNames: array of string);
    begin
      ApplyControlMapping( aVars, aNames, ckInput);
    end;


    procedure   TKnobsModuleSignalSpec.ApplyOutputMapping( const aVars: array of Byte; const aNames: array of string);
    begin
      ApplyControlMapping( aVars, aNames, ckOutput);
    end;


    procedure   TKnobsModuleSignalSpec.ApplyInternalMapping( const aVars: array of Byte; const aNames: array of string);
    begin
      ApplyControlMapping( aVars, aNames, ckInternal);
    end;


    procedure   TKnobsModuleSignalSpec.ApplyStringMapping( const aVars: array of Byte; const aNames: array of string);
    begin
      ApplyControlMapping( aVars, aNames, ckString);
    end;


    procedure   TKnobsModuleSignalSpec.ApplyInternalStringMapping( const aVars: array of Byte; const aNames: array of string);
    begin
      ApplyControlMapping( aVars, aNames, ckInternalString);
    end;


    procedure   TKnobsModuleSignalSpec.TestApply;
    begin
      ApplyInputMapping( [ 1, 2], [ 'aName1', 'aName2']);
    end;


    procedure   TKnobsModuleSignalSpec.Dump( anInd: Integer; const aStringList: TStringList);
    var
      i          : Integer;
      aClassName : string;
    begin
      if Assigned( aStringList)
      then begin
        if Assigned( ModuleClass)
        then aClassName := ModuleClass.ClassName
        else aClassName := 'nil';

        aStringList.Add( Format( '%sTKnobsModuleSignalSpec('          , [ Indent( anInd    )                                             ], AppLocale));
        aStringList.Add( Format  ( '%sModuleName             = ''%s''', [ Indent( anInd + 1),               ModuleName                   ], AppLocale));
        aStringList.Add( Format  ( '%sModuleType             = %d'    , [ Indent( anInd + 1),               ModuleType                   ], AppLocale));
        aStringList.Add( Format  ( '%sModuleClass            = %s'    , [ Indent( anInd + 1),               aClassname                   ], AppLocale));
        aStringList.Add( Format  ( '%sTitle                  = ''%s''', [ Indent( anInd + 1),               Title                        ], AppLocale));
        aStringList.Add( Format  ( '%sComment                = ''%s''', [ Indent( anInd + 1),               Comment                      ], AppLocale));
        aStringList.Add( Format  ( '%sInputCount             = %d'    , [ Indent( anInd + 1),               FInputCount                  ], AppLocale));
        aStringList.Add( Format  ( '%sOutputCount            = %d'    , [ Indent( anInd + 1),               FOutputCount                 ], AppLocale));
        aStringList.Add( Format  ( '%sInternalCount          = %d'    , [ Indent( anInd + 1),               FInternalCount               ], AppLocale));
        aStringList.Add( Format  ( '%sStringCount            = %d'    , [ Indent( anInd + 1),               FStringCount                 ], AppLocale));
        aStringList.Add( Format  ( '%sInternalStringCount    = %d'    , [ Indent( anInd + 1),               FInternalStringCount         ], AppLocale));
        aStringList.Add( Format  ( '%sDezipperMap            = %s'    , [ Indent( anInd + 1), ByteSetToStr( DezipperMap                 )], AppLocale));
        aStringList.Add( Format  ( '%sAllowSignalConversion  = %s'    , [ Indent( anInd + 1), BoolToStr   ( AllowSignalConversion , True)], AppLocale));
        aStringList.Add( Format  ( '%sAllowRandomization     = %s'    , [ Indent( anInd + 1), BoolToStr   ( AllowRandomization    , True)], AppLocale));
        aStringList.Add( Format  ( '%sShowAllowRandomization = %s'    , [ Indent( anInd + 1), BoolToStr   ( ShowAllowRandomization, True)], AppLocale));
        aStringList.Add( Format  ( '%sHasSideEffects         = %s'    , [ Indent( anInd + 1), BoolToStr   ( AllowSignalConversion , True)], AppLocale));
        aStringList.Add( Format  ( '%sTemplateName           = %s'    , [ Indent( anInd + 1), BoolToStr   ( AllowSignalConversion , True)], AppLocale));
        aStringList.Add( Format  ( '%sActiveVariation        = %d'    , [ Indent( anInd + 1),               ActiveVariation              ], AppLocale));
        aStringList.Add( Format  ( '%sSpecs('                         , [ Indent( anInd + 1)                                             ], AppLocale));

        for i := 0 to SignalCount - 1
        do DumpItem( anInd + 2, i, aStringList);

        aStringList.Add( Format  ( '%s)'                              , [ Indent( anInd + 1)                                             ], AppLocale));
        aStringList.Add( Format( '%s)'                                , [ Indent( anInd    )                                             ], AppLocale));
      end;
    end;


{ ========
  TKnobsModuleSignalSpecs = class
  private
    FSpecs : array of TKnobsModuleSignalSpec;
  public
    property    ModuleCount                                       : Integer          read GetModuleCount;
    property    ModuleName [ aModuleIndex: Integer]               : string           read GetModuleName;
    property    ModuleType [ aModuleIndex: Integer]               : TKnobsModuleType read GetModuleType;
    property    SignalCount[ aModuleIndex: Integer]               : Integer          read GetSignalCount;
    property    SignalSpec [ aModuleIndex, aSignalIndex: Integer] : TKnobsSignalSpec read GetSignalSpec;
  private
}

    function    TKnobsModuleSignalSpecs.GetModuleCount: Integer;
    begin
      Result := Length( FSpecs);
    end;


    function    TKnobsModuleSignalSpecs.GetModuleName( aModuleIndex: Integer): string;
    begin
      Result := FSpecs[ aModuleIndex].ModuleName;
    end;


    function    TKnobsModuleSignalSpecs.GetModuleType( aModuleIndex: Integer): TKnobsModuleType;
    begin
      Result := FSpecs[ aModuleIndex].ModuleType;
    end;


    function    TKnobsModuleSignalSpecs.GetSignalCount( aModuleIndex: Integer): Integer;
    begin
      Result := FSpecs[ aModuleIndex].SignalCount;
    end;


    function    TKnobsModuleSignalSpecs.GetSignalSpec( aModuleIndex, aSignalIndex: Integer): TKnobsSignalSpec;
    begin
      Result := FSpecs[ aModuleIndex].SignalSpec[ aSignalIndex];
    end;


    procedure   TKnobsModuleSignalSpecs.DumpItem( anInd, anIndex: Integer; const aStringList: TStringList);
    begin
      if Assigned( aStringList) and ( anIndex >= 0) and ( anIndex < ModuleCount)
      then FSpecs[ anIndex].Dump( anInd, aStringList);
    end;

//  public


    constructor TKnobsModuleSignalSpecs.Create( const anEditorPatch: TKnobsWirePanel);
    begin
      inherited Create;
      Load( anEditorPatch);
    end;


    destructor  TKnobsModuleSignalSpecs.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TKnobsModuleSignalSpecs.Clear;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do FreeAndNil( FSpecs[ i]);

      SetLength( FSpecs, 0);
    end;


    procedure   TKnobsModuleSignalSpecs.Load( const anEditorPatch: TKnobsWirePanel);
    var
      i : Integer;
    begin
      Clear;

      if Assigned( anEditorPatch)
      then begin
        for i := 0 to anEditorPatch.ModuleCount - 1
        do AddModule( anEditorPatch.Module[ i]);
      end;
    end;


    procedure   TKnobsModuleSignalSpecs.AddModule( const aModule: TKnobsCustomModule);
    begin
      SetLength( FSpecs, ModuleCount + 1);
      FSpecs[ ModuleCount - 1] := TKnobsModuleSignalSpec.Create( aModule);
    end;


    procedure   TKnobsModuleSignalSpecs.Dump( const aStringList: TStringList); // overload;
    var
      i : Integer;
    begin
      if Assigned( aStringList)
      then begin
        aStringList.Add( 'TKnobsSignalSpecs(');

        for i:= 0 to ModuleCount - 1
        do DumpItem( 1, i, aStringList);

        aStringList.Add( ')');
      end;
    end;


    procedure   TKnobsModuleSignalSpecs.Dump( const aFileName: string); // overload;
    var
      aStringList: TStringList;
    begin
      aStringList := TStringList.Create;
      try
        Dump( aStringList);
        aStringList.SaveToFile( aFileName);
      finally
        aStringList.DisposeOf;
      end;
    end;



end.

