unit wave_dlg;

{

   COPYRIGHT 2013 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  WinApi.Windows, WinApi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics, Vcl.Controls,
  Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.Buttons, Vcl.ComCtrls, Vcl.ExtCtrls, System.Math,

  Globals, KnobsUtils, KnobsConversions, portaudio, knobs2013;

type

  TOnShowAsioPanel = procedure( aSender: TObject; aDeviceId: TPaDeviceIndex; aHandle: HWND) of object;
  TOnRestartAudio  = procedure( aSender: TObject; aDebug: Boolean)                          of object;
  TOnLog           = procedure( aSender: TObject; const aMsg: string)                       of object;


  TFormWaveDeviceSelect = class(TForm)
    PanelBottom: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    PanelMain: TPanel;
    Label29: TLabel;
    EditBufferSize: TEdit;
    Label6: TLabel;
    BitBtnTestAudio: TBitBtn;
    CheckBoxDebug: TCheckBox;
    PanelPortAudio: TPanel;
    LabelPAVersion: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    LabelInChannels: TLabel;
    LabelOutChannels: TLabel;
    LabelAsioIn: TLabel;
    LabelAsioOut: TLabel;
    ComboBoxAPI: TComboBox;
    ComboBoxWaveInPA: TComboBox;
    ComboBoxWaveOutPA: TComboBox;
    BitBtnShowAsioPanel: TBitBtn;
    ComboBoxIn1: TComboBox;
    ComboBoxIn2: TComboBox;
    ComboBoxIn3: TComboBox;
    ComboBoxIn4: TComboBox;
    ComboBoxIn5: TComboBox;
    ComboBoxIn6: TComboBox;
    ComboBoxIn7: TComboBox;
    ComboBoxIn8: TComboBox;
    ComboBoxOut1: TComboBox;
    ComboBoxOut2: TComboBox;
    ComboBoxOut3: TComboBox;
    ComboBoxOut4: TComboBox;
    ComboBoxOut5: TComboBox;
    ComboBoxOut6: TComboBox;
    ComboBoxOut7: TComboBox;
    ComboBoxOut8: TComboBox;
    procedure ComboBoxAPIChange(Sender: TObject);
    procedure ComboBoxWaveInPAChange(Sender: TObject);
    procedure ComboBoxWaveOutPAChange(Sender: TObject);
    procedure EditBufferSizeChange(Sender: TObject);
    procedure BitBtnShowAsioPanelClick(Sender: TObject);
    procedure BitBtnTestAudioClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ComboBoxInChannelChange(Sender: TObject);
    procedure ComboBoxOutChannelChange(Sender: TObject);
  private
    FSelectedInputPA     : string;
    FSelectedOutputPA    : string;
    FSelectedInputIdPA   : DWORD;
    FSelectedOutputIdPA  : DWORD;
    FMaxInChannels       : Integer;
    FMaxOutChannels      : Integer;
    FIsAsioIn            : Boolean;
    FIsAsioOut           : Boolean;
  private
    FInitialized         : Boolean;
  private
    FPADetected          : Boolean;
    FPAInitialized       : Boolean;
    FPAVersion           : string;
    FPaApiInfo           : array of TPaHostApiInfo;
    FPaInputDevices      : array of TPaDeviceInfo;
    FPaInputDeviceIDs    : array of TPaDeviceIndex;
    FPaOutputDevices     : array of TPaDeviceInfo;
    FPaOutputDeviceIDs   : array of TPaDeviceIndex;
    FSelectedAPI         : string;
    FSelectedAPIId       : Integer;
    FBufferSize          : Integer;
    FOnShowAsioPanel     : TOnShowAsioPanel;
    FOnRestartAudio      : TOnRestartAudio;
    FOnLog               : TOnLog;
    FDebug               : Boolean;
  private
    FInChannels          : TStringList;
    FOutChannels         : TStringList;
    FInMasks             : array[ 0 .. 7] of Integer;
    FOutMasks            : array[ 0 .. 7] of Integer;
  private
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    procedure   SetDebug( aValue: Boolean);
  private
    function    ProbePortAudio: Boolean;
    procedure   SetPAVersion( const aValue: string);
    procedure   InitializePortAudio;
    procedure   AddApiInfo( const anApiInfo: PPaHostApiInfo);
    procedure   ClearPAInputDevices;
    procedure   ClearPAOutputDevices;
    procedure   AddPAInputDevice ( anIndex:TPaDeviceIndex; aDeviceInfo: PPaDeviceInfo);
    procedure   AddPAOutputDevice( anIndex:TPaDeviceIndex; aDeviceInfo: PPaDeviceInfo);
    procedure   FillPADevices( const aCaller: string);
    procedure   ShowAsioPanel;
    procedure   RestartAudio;
  private
    procedure   FixComboBox( const aBox: TComboBox; const anItems: TStringList; aChannel: Integer);
    procedure   FixInputStuff;
    procedure   FixOutputStuff;
    procedure   SetSelectedInputPA   ( const aValue: string);
    procedure   SetSelectedOutputPA  ( const aValue: string);
    procedure   SetSelectedInputIdPA ( aValue: DWORD);
    procedure   SetSelectedOutputIdPA( aValue: DWORD);
  private
    procedure   SetSelectedAPI       ( const aValue: string);
    procedure   SetSelectedAPIID     ( aValue: Integer);
  private
    function    GetPaApiCount: Integer;
    function    GetPaHostApi( anIndex: Integer): TPaHostApiInfo;
    function    GetPaInputDevicecount : Integer;
    function    GetPaOutputDevicecount: Integer;
    function    GetPaInDeviceId       : TPaDeviceIndex;
    function    GetPaInDevice         : TPaDeviceInfo ;
    function    GetPaOutDeviceId      : TPaDeviceIndex;
    function    GetPaOutDevice        : TPaDeviceInfo ;
  private
    procedure   SetBufferSize( aValue: Integer);
  private
    procedure   SelectInputChannel( const aSender: TComboBox);
    procedure   SelectOutputChannel( const aSender: TComboBox);
    function    GetInMaskCount: Integer;
    function    GetInMask( anIndex: Integer): Integer;
    procedure   SetInMask( anIndex, aTag: Integer);
    procedure   ClearInChannels;
    function    GetOutMaskCount: Integer;
    function    GetOutMask( anIndex: Integer): Integer;
    procedure   SetOutMask( anIndex, aTag: Integer);
    procedure   ClearOutChannels;
  public
    procedure   Initialize;
    function    Execute: Boolean;
    procedure   ClearInMasks;
    procedure   ClearOutMasks;
    function    CreateAsioInStreamInfo : TPaAsioStreamInfo;
    function    CreateAsioOutStreamInfo: TPaAsioStreamInfo;
    procedure   FreeAsioStreamInfo ( var aValue: TPaAsioStreamInfo);
  public
    property    SelectedInputPA    : string  read FSelectedInputPA    write SetSelectedInputPA   ;
    property    SelectedOutputPA   : string  read FSelectedOutputPA   write SetSelectedOutputPA  ;
    property    SelectedInputIdPA  : DWORD   read FSelectedInputIdPA  write SetSelectedInputIdPA ;
    property    SelectedOutputIdPA : DWORD   read FSelectedOutputIdPA write SetSelectedOutputIdPA;
  public
    property    PADetected         : Boolean read FPADetected;
    property    PAInitialized      : Boolean read FPAInitialized      write FPAInitialized;
    property    PAVersion          : string  read FPAVersion;
    property    SelectedAPI        : string  read FSelectedAPI        write SetSelectedAPI;
    property    SelectedAPIID      : Integer read FSelectedAPIID      write SetSelectedAPIID;
    property    IsAsioIn           : Boolean read FIsAsioIn;
    property    IsAsioOut          : Boolean read FIsAsioOut;
    property    MaxInChannels      : Integer read FMaxInChannels;
    property    MaxOutChannels     : Integer read FMaxOutChannels;
  public
    property    PaApiCount                          : Integer        read GetPaApiCount;
    property    PaHostApis       [ anIndex: Integer]: TPaHostApiInfo read GetPaHostApi;
    property    PaInputdeviceCount                  : Integer        read GetPaInputDevicecount;
    property    PaOutputdeviceCount                 : Integer        read GetPaOutputDevicecount;
    property    PaInDeviceId                        : TPaDeviceIndex read GetPaInDeviceId;
    property    PaInDevice                          : TPaDeviceInfo  read GetPaInDevice;
    property    PaOutDeviceId                       : TPaDeviceIndex read GetPaOutDeviceId;
    property    PaOutDevice                         : TPaDeviceInfo  read GetPaOutDevice;
    property    InMaskCount                         : Integer        read GetInMaskCount;
    property    OutMaskCount                        : Integer        read GetOutMaskCount;
    property    InMasks [ anIndex: Integer]         : Integer        read GetInMask              write SetInMask;
    property    OutMasks[ anIndex: Integer]         : Integer        read GetOutMask             write SetOutMask;
  public
    property    Debug            : Boolean          read FDebug           write SetDebug;
    property    BufferSize       : Integer          read FBufferSize      write SetBufferSize;
    property    OnShowAsioPanel  : TOnShowAsioPanel read FOnShowAsioPanel write FOnShowAsioPanel;
    property    OnRestartAudio   : TOnRestartAudio  read FOnRestartAudio  write FOnRestartAudio;
    property    OnLog            : TOnLog           read FOnLog           write FOnLog;
  end;

var

  FormWaveDeviceSelect: TFormWaveDeviceSelect;



implementation



{$R *.dfm}


// User area

const

  SApiToUse = '<Select API to use>';


//  private

    procedure   TFormWaveDeviceSelect.Log( const aMsg: string);
    begin
      if Assigned( FOnLog)
      then FOnLog( Self, aMsg);
    end;


    procedure   TFormWaveDeviceSelect.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


    procedure   TFormWaveDeviceSelect.SetDebug( aValue: Boolean);
    begin
      if aValue <> FDebug
      then begin
        FDebug                := aValue;
        CheckBoxDebug.Checked := aValue;
      end;
    end;


//  private

    function    TFormWaveDeviceSelect.ProbePortAudio: Boolean;
    begin
      Result := False;
      SetPAVersion( 'not detected or not properly initialized');

      if PAInitialized
      then begin
        try
          SetPAVersion( string( Pa_GetVersionText));
          Result := True;
        except
          on E: Exception
          do KilledException( E);
        end;
      end;

      FPADetected := Result;
    end;


    procedure   TFormWaveDeviceSelect.SetPAVersion( const aValue: string);
    begin
      FPAVersion             := aValue;
      LabelPAVersion.Caption := Format( 'PortAudio version: %s', [ aValue]);
    end;


    procedure   TFormWaveDeviceSelect.InitializePortAudio;
    var
      i         : Integer;
      aCount    : Integer;
      anApiInfo : PPaHostApiInfo;
    begin
      try
        Log( 'initializing PortAudio');
        ClearPAInputDevices;
        ClearPAOutputDevices;

        FSelectedAPIId      := -1;
        FSelectedAPI        := '';
        FSelectedInputIdPA  := DWORD( -1);
        FSelectedOutputIdPA := DWORD( -1);

        aCount := Pa_GetHostApiCount;
        ComboBoxAPI.Clear;
        ComboBoxAPI.Text := SApiToUse;

        for i := 0 to aCount - 1
        do begin
          anApiInfo := Pa_GetHostApiInfo( i);
          if Assigned( anApiInfo)
          then begin
            AddApiInfo( anApiInfo);
            ComboBoxAPI.Items.Add( string( anApiInfo^.name));
          end;
        end;

        LogFmt( 'initialized PortAudio %d Host APIs found', [ aCount]);
      except
        on E: Exception
        do begin
          SetPAVersion( 'setup error - could not query API interfaces');
          KilledException( E);
          LogFmt( 'PortAudio initialization failed: %s', [ E.Message]);
        end;
      end;
    end;


    procedure   TFormWaveDeviceSelect.AddApiInfo( const anApiInfo: PPaHostApiInfo);
    begin
      if Assigned( anApiInfo)
      then begin
        SetLength( FPaApiInfo, Length( FPaApiInfo) + 1);
        FPaApiInfo[ Length( FPaApiInfo) - 1] := anApiInfo^;
      end;
    end;


    procedure   TFormWaveDeviceSelect.ClearPAInputDevices;
    begin
      ComboBoxWaveInPA.Clear;
      ComboBoxWaveInPA.Text := '';
      SetLength( FPaInputDevices, 0);
    end;


    procedure   TFormWaveDeviceSelect.ClearPAOutputDevices;
    begin
      ComboBoxWaveOutPA.Clear;
      ComboBoxWaveOutPA.Text := '';
      SetLength( FPaOutputDevices, 0);
    end;


    procedure   TFormWaveDeviceSelect.AddPAInputDevice( anIndex:TPaDeviceIndex; aDeviceInfo: PPaDeviceInfo);
    var
      aName  : string;
      anIns  : Integer;
      anOuts : Integer;
    begin
      aName  := string( aDeviceInfo.name);
      anIns  := aDeviceInfo.maxInputChannels;
      anOuts := aDeviceInfo.maxOutputChannels;
      LogFmt( 'AddPAInputDevice( %d, ''%s'', %d ins, %d outs)', [ anIndex, aName, anIns, anOuts]);
      SetLength( FPaInputDevices  , PaInputdeviceCount + 1);
      SetLength( FPaInputDeviceIDs, PaInputdeviceCount + 1);
      FPaInputDevices  [ PaInputdeviceCount - 1] := aDeviceInfo^;
      FPaInputDeviceIDs[ PaInputdeviceCount - 1] := anIndex;
      ComboBoxWaveInPA.Items.Add( aName);
    end;


    procedure   TFormWaveDeviceSelect.AddPAOutputDevice( anIndex:TPaDeviceIndex; aDeviceInfo: PPaDeviceInfo);
    var
      aName  : string;
      anIns  : Integer;
      anOuts : Integer;
    begin
      aName  := string( aDeviceInfo.name);
      anIns  := aDeviceInfo.maxInputChannels;
      anOuts := aDeviceInfo.maxOutputChannels;
      LogFmt( 'AddPAOutputDevice( %d, ''%s'', %d ins, %d outs)', [ anIndex, aName, anIns, anOuts]);
      SetLength( FPaOutputDevices  , PaOutputdeviceCount + 1);
      SetLength( FPaOutputDeviceIDs, PaOutputdeviceCount + 1);
      FPaOutputDevices  [ PaOutputdeviceCount - 1] := aDeviceInfo^;
      FPaOutputDeviceIDs[ PaOutputdeviceCount - 1] := anIndex;
      ComboBoxWaveOutPA.Items.Add( aName);
    end;


    procedure   TFormWaveDeviceSelect.FillPADevices( const aCaller: string);
    var
      aCount      : TPaDeviceIndex;
      i           : TPaDeviceIndex;
      aDeviceInfo : PPaDeviceInfo;
      aName       : string;
      anIncount   : Integer;
      anOutCount  : Integer;
    begin
      ClearPAInputDevices;
      ClearPAOutputDevices;
      FixInputStuff;
      FixOutputStuff;
      aCount := Pa_GetDeviceCount;
      LogFmt( 'FillPADevices: count = %d, called from ''%s''', [ aCount, aCaller]);

      for i := 0 to aCount - 1
      do begin
        aDeviceInfo := Pa_GetDeviceInfo( i);

        if   Assigned( aDeviceInfo)
        then begin
          if   aDeviceInfo.hostApi = SelectedAPIID
          then begin
            if aDeviceInfo.maxInputChannels >= 2
            then AddPAInputDevice( i, aDeviceInfo);

            if aDeviceInfo.maxOutputChannels >= 2
            then AddPAOutputDevice( i, aDeviceInfo);
          end
          else begin
            if Debug
              then begin
              aName      := string( aDeviceInfo.name);
              anIncount  := aDeviceInfo.maxInputChannels;
              anOutcount := aDeviceInfo.maxOutputChannels;
              LogFmt( 'FillPADevices: NOT added: %d, ''%s'' %d ins, %d outs)', [ i, aName, anInCount, anOutCount]);
            end;
          end;
        end
        else LogFmt( 'FillPADevices: NIL device %d, called from ''%s''', [ i, aCaller]);
      end;
    end;


    procedure   TFormWaveDeviceSelect.ShowAsioPanel;
    begin
      if Assigned( FOnShowAsioPanel)
      then begin
        if IsAsioOut
        then FOnShowAsioPanel( Self, FPaOutputDeviceIDs[ SelectedOutputIdPA], Handle)
        else if IsAsioIn
        then FOnShowAsioPanel( Self, FPaInputDeviceIDs [ SelectedInputIdPA ], Handle);
      end;
    end;


    procedure   TFormWaveDeviceSelect.RestartAudio;
    begin
      if Assigned( FOnRestartAudio)
      then FOnRestartAudio( Self, Debug);
    end;


//  private

    procedure   TFormWaveDeviceSelect.FixComboBox( const aBox: TComboBox; const anItems: TStringList; aChannel: Integer);
    begin
      with aBox
      do begin
        Items.Assign( anItems);

        if   ( Items.Count > 0)
        and  ( aChannel    > 0)
        then begin
          ItemIndex := aChannel;
          Text      := Items[ ItemIndex];
        end
        else begin
          ItemIndex := -1;
          Text      := '';
        end;
      end;
    end;


    procedure   TFormWaveDeviceSelect.FixInputStuff;
    var
      i     : Integer;
      aName : PAnsiChar;
    begin
      FmaxInChannels          := PaInDevice.maxInputChannels;
      LabelInChannels.Caption := Format( '(%d)', [ MaxInChannels]);
      FIsAsioIn               := PaHostApis[ PaInDevice.hostApi]._type = paASIO;
      LabelAsioIn.Visible     := IsAsioIn;

      FInChannels.Add( '-');

      if IsAsioIn
      then begin
        for i := 0 to MaxInChannels - 1
        do begin
          PaAsio_GetInputChannelName( PaInDeviceId, i, @ aName);
          FInChannels.Add( Format( '%d: %s', [ i + 1, aName], AppLocale));
        end;
      end
      else ClearInChannels;

      FixComboBox( ComboBoxIn1, FInChannels, InMasks[ 0]);
      FixComboBox( ComboBoxIn2, FInChannels, InMasks[ 1]);
      FixComboBox( ComboBoxIn3, FInChannels, InMasks[ 2]);
      FixComboBox( ComboBoxIn4, FInChannels, InMasks[ 3]);
      FixComboBox( ComboBoxIn5, FInChannels, InMasks[ 4]);
      FixComboBox( ComboBoxIn6, FInChannels, InMasks[ 5]);
      FixComboBox( ComboBoxIn7, FInChannels, InMasks[ 6]);
      FixComboBox( ComboBoxIn8, FInChannels, InMasks[ 7]);
    end;


    procedure   TFormWaveDeviceSelect.FixOutputStuff;
    var
      i     : Integer;
      aName : PAnsiChar;
    begin
      FMaxOutChannels          := PaOutDevice.maxOutputChannels;
      LabelOutChannels.Caption := Format( '(%d)', [ MaxOutChannels]);
      FIsAsioOut               := PaHostApis[ PaOutDevice.hostApi]._type = paASIO;
      LabelAsioOut.Visible     := IsAsioOut;

      FOutChannels.Add( '-');

      if IsAsioOut
      then begin
        for i := 0 to MaxOutChannels - 1
        do begin
          PaAsio_GetOutputChannelName( PaOutDeviceId, i, @ aName);
          FOutChannels.Add( Format( '%d: %s', [ i + 1, aName], AppLocale));
        end;
      end
      else ClearOutChannels;

      FixComboBox( ComboBoxOut1, FOutChannels, Outmasks[ 0]);
      FixComboBox( ComboBoxOut2, FOutChannels, Outmasks[ 1]);
      FixComboBox( ComboBoxOut3, FOutChannels, Outmasks[ 2]);
      FixComboBox( ComboBoxOut4, FOutChannels, Outmasks[ 3]);
      FixComboBox( ComboBoxOut5, FOutChannels, Outmasks[ 4]);
      FixComboBox( ComboBoxOut6, FOutChannels, Outmasks[ 5]);
      FixComboBox( ComboBoxOut7, FOutChannels, Outmasks[ 6]);
      FixComboBox( ComboBoxOut8, FOutChannels, Outmasks[ 7]);
    end;


    procedure   TFormWaveDeviceSelect.SetSelectedInputPA( const aValue: string);
    var
      i     : Integer;
      Found : Integer;
    begin
      if   not FInitialized
      or   not PADetected
      then Exit;

      if aValue <> FSelectedInputPA
      then begin
        with ComboBoxWaveInPA
        do begin
          Found := -1;

          for i := 0 to Items.Count - 1
          do begin
            if SameText( aValue, Items[ i])
            then begin
              Found := i;
              Break;
            end;
          end;

          if Found >= 0
          then begin
            ItemIndex          := Found;
            Text               := Items[ Found];
            FSelectedInputPA   := Text;
            FSelectedInputIdPA := Found;
          end;
        end;
      end;
    end;


    procedure   TFormWaveDeviceSelect.SetSelectedOutputPA( const aValue: string);
    var
      i     : Integer;
      Found : Integer;
    begin
      if   not FInitialized
      or   not PADetected
      then Exit;

      if aValue <> FSelectedOutputPA
      then begin
        with ComboBoxWaveOutPA
        do begin
          Found := -1;

          for i := 0 to Items.Count - 1
          do begin
            if SameText( aValue, Items[ i])
            then begin
              Found := i;
              Break;
            end;
          end;

          if Found >= 0
          then begin
            ItemIndex           := Found;
            Text                := Items[ Found];
            FSelectedOutputPA   := Text;
            FSelectedOutputIdPA := Found;
          end;
        end;
      end;
    end;

    procedure   TFormWaveDeviceSelect.SetSelectedInputIdPA( aValue: DWORD);
    begin
      LogFmt( 'SetSelectedInputIdPA( %d)', [ aValue]);

      if   not FInitialized
      or   not PADetected
      then Exit;

   // if aValue <> FSelectedInputIdPA
   // then begin
        if ( aValue <> DWORD( -1)) and ( Integer( aValue) < ComboBoxWaveInPA.Items.Count)
        then begin
          FSelectedInputIdPA := aValue;
          SelectedInputPA    := ComboBoxWaveInPA.Items[ aValue];
          FixInputStuff;
          BitBtnShowAsioPanel.Enabled := IsAsioIn or IsAsioOut;
        end
        else begin
          SelectedInputPA    := '';
          FSelectedInputIdPA := DWORD( -1);
        end;
   // end;
    end;


    procedure   TFormWaveDeviceSelect.SetSelectedOutputIdPA( aValue: DWORD);
    begin
      LogFmt( 'SetSelectedOutputIdPA( %d)', [ aValue]);

      if   not FInitialized
      or   not PADetected
      then Exit;

   // if aValue <> FSelectedOutputIdPA
   // then begin
        if ( aValue <> DWORD( -1)) and ( Integer( aValue) < ComboBoxWaveOutPA.Items.Count)
        then begin
          FSelectedOutputIdPA := aValue;
          SelectedOutputPA    := ComboBoxWaveOutPA.Items[ aValue];
          FixOutputStuff;
          BitBtnShowAsioPanel.Enabled := IsAsioIn or IsAsioOut;
        end
        else begin
          SelectedOutputPA    := '';
          FSelectedOutputIdPA := DWORD( -1);
        end;
   // end;
    end;


//  private

    procedure   TFormWaveDeviceSelect.SetSelectedAPI( const aValue: string);
    var
      i     : Integer;
      Found : Integer;
    begin
      LabelInChannels .Caption := '(0)';
      LabelOutChannels.Caption := '(0)';

      if   not FInitialized
      or   not PADetected
      then Exit;

   // if aValue <> FSelectedAPI
   // then begin
        with ComboBoxAPI
        do begin
          Found := -1;

          for i := 0 to Items.Count - 1
          do begin
            if SameText( aValue, Items[ i])
            then begin
              Found := i;
              Break;
            end;
          end;

          if Found >= 0
          then begin
            ItemIndex     := Found;
            Text          := Items[ Found];
            FSelectedAPI  := Text;
            SelectedAPIId := Found;
            FillPADevices( Format( 'SetSelectedAPI( const aValue: ''%s'')', [ aValue], AppLocale));
          end
          else ComboBoxAPI.Text := SApiToUse;
        end;
   // end;
    end;


    procedure   TFormWaveDeviceSelect.SetSelectedAPIID( aValue: Integer);
    begin
      if   not FInitialized
      or   not PADetected
      then Exit;

      if aValue <> FSelectedAPIID
      then begin
        if ( aValue >= 0) and ( aValue < ComboBoxAPI.Items.Count)
        then begin
          FSelectedAPIID := aValue;
          SelectedAPI    := ComboBoxAPI.Items[ aValue];
          FillPADevices( Format( 'SetSelectedAPIID( aValue: %d)', [ aValue], AppLocale));
        end
        else begin
          FSelectedAPIId := -1;
          SelectedAPI    := '';
        end;
      end;
    end;


//  private

    function    TFormWaveDeviceSelect.GetPaApiCount: Integer;
    begin
      Result := Length( FPaApiInfo);
    end;


    function    TFormWaveDeviceSelect.GetPaHostApi( anIndex: Integer): TPaHostApiInfo;
    begin
      if ( anIndex >= 0) and ( anIndex < PaApiCount)
      then Result := FPaApiInfo[ anIndex]
      else begin
        Result._type               := -1;
        Result.name                := 'invalid host API';
        Result.deviceCount         := 0;
        Result.defaultInputDevice  := -1;
        Result.defaultOutputDevice := -1;
      end;
    end;


    function    TFormWaveDeviceSelect.GetPaInputDevicecount: Integer;
    begin
      Result := Length( FPaInputDevices);
    end;


    function    TFormWaveDeviceSelect.GetPaOutputDevicecount: Integer;
    begin
      Result := Length( FPaOutputDevices);
    end;


    function    TFormWaveDeviceSelect.GetPaInDeviceId: TPaDeviceIndex;
    begin
      if FSelectedInputIdPA < Cardinal( PaInputDeviceCount)
      then Result := FPaInputDeviceIDs[ FSelectedInputIdPA]
      else Result := -1;
    end;


    function    TFormWaveDeviceSelect.GetPaInDevice: TPaDeviceInfo;
    begin
      if FSelectedInputIdPA < Cardinal( PaInputDeviceCount)
      then Result := FPaInputDevices[ FSelectedInputIdPA]
      else begin
        with Result
        do begin
          structVersion           := 0;             {* this is struct version 2 *}
          hostApi                 := -1;            {* note this is a host API index, not a type id*}
          maxInputChannels        := 0;
          maxOutputChannels       := 0;
        end;
      end;
    end;


    function    TFormWaveDeviceSelect.GetPaOutDeviceId: TPaDeviceIndex;
    begin
      if FSelectedOutputIdPA < Cardinal( PaOutputDeviceCount)
      then Result := FPaOutputDeviceIDs[ FSelectedOutputIdPA]
      else Result := -1;
    end;


    function    TFormWaveDeviceSelect.GetPaOutDevice: TPaDeviceInfo;
    begin
      if FSelectedOutputIdPA < Cardinal( PaOutputDeviceCount)
      then Result := FPaOutputDevices[ FSelectedOutputIdPA]
      else begin
        with Result
        do begin
          structVersion           := 0;             {* this is struct version 2 *}
          hostApi                 := -1;            {* note this is a host API index, not a type id*}
          maxInputChannels        := 0;
          maxOutputChannels       := 0;
        end;
      end;
    end;


//  private

    procedure   TFormWaveDeviceSelect.SetBufferSize( aValue: Integer);
    begin
      if aValue <> FBufferSize
      then begin
        FBufferSize         := aValue;
        EditBufferSize.Text := IntToStr( aValue);
      end;
    end;


//  private

    procedure   TFormWaveDeviceSelect.SelectInputChannel( const aSender: TComboBox);
    begin
      InMasks[ aSender.Tag] := aSender.ItemIndex;
    end;


    procedure   TFormWaveDeviceSelect.SelectOutputChannel( const aSender: TComboBox);
    begin
      OutMasks[ aSender.Tag] := aSender.ItemIndex;
    end;


    function    TFormWaveDeviceSelect.GetInMaskCount: Integer;
    var
      i : Integer;
    begin
      Result := 0;

      for i := Low( FInMasks) to High( FInMasks)
      do begin
        if InMasks[ i] <= 0
        then Break
        else Inc( Result);
      end;
    end;


    function    TFormWaveDeviceSelect.GetInMask( anIndex: Integer): Integer;
    begin
      Result := FInMasks[ anIndex];
    end;


    procedure   TFormWaveDeviceSelect.SetInMask( anIndex, aTag: Integer);
    begin
      if aTag <> FInMasks[ anIndex]
      then begin
        FInMasks[ anIndex] := aTag;

        case anIndex of
          0 : ComboBoxIn1.ItemIndex := aTag;
          1 : ComboBoxIn2.ItemIndex := aTag;
          2 : ComboBoxIn3.ItemIndex := aTag;
          3 : ComboBoxIn4.ItemIndex := aTag;
          4 : ComboBoxIn5.ItemIndex := aTag;
          5 : ComboBoxIn6.ItemIndex := aTag;
          6 : ComboBoxIn7.ItemIndex := aTag;
          7 : ComboBoxIn8.ItemIndex := aTag;
        end;
      end;
    end;


    procedure   TFormWaveDeviceSelect.ClearInChannels;
    begin
      ClearInMasks;
      FInChannels.Clear;
    end;


    function    TFormWaveDeviceSelect.GetOutMaskCount: Integer;
    var
      i : Integer;
    begin
      Result := 0;

      for i := Low( FOutMasks) to High( FOutMasks)
      do begin
        if OutMasks[ i] <= 0
        then break
        else Inc( Result);
      end;
    end;


    function    TFormWaveDeviceSelect.GetOutMask( anIndex: Integer): Integer;
    begin
      Result := FOutMasks[ anIndex];
    end;


    procedure   TFormWaveDeviceSelect.SetOutMask( anIndex, aTag: Integer);
    begin
      if aTag <> FOutMasks[ anIndex]
      then begin
        FOutMasks[ anIndex] := aTag;

        case anIndex of
          0 : ComboBoxOut1.ItemIndex := aTag;
          1 : ComboBoxOut2.ItemIndex := aTag;
          2 : ComboBoxOut3.ItemIndex := aTag;
          3 : ComboBoxOut4.ItemIndex := aTag;
          4 : ComboBoxOut5.ItemIndex := aTag;
          5 : ComboBoxOut6.ItemIndex := aTag;
          6 : ComboBoxOut7.ItemIndex := aTag;
          7 : ComboBoxOut8.ItemIndex := aTag;
        end;
      end;
    end;


    procedure   TFormWaveDeviceSelect.ClearOutChannels;
    begin
      ClearOutMasks;
      FOutChannels.Clear;
    end;


//  public

    procedure   TFormWaveDeviceSelect.Initialize;
    begin
      if FInitialized
      then Exit;

      Log( 'initializing');

      if ProbePortAudio
      then begin
        Log( '  PortAudio probed');
        InitializePortAudio;
      end
      else begin
        Log( '  PortAudio not probed');
        ComboBoxAPI.Text := '';
        ComboBoxWaveInPA .Clear; ComboBoxWaveInPA .Text := '';
        ComboBoxWaveOutPA.Clear; ComboBoxWaveOutPA.Text := '';
      end;

      FInitialized := True;
      Log( 'initialized');
    end;


    function    TFormWaveDeviceSelect.Execute: Boolean;
    begin
      if not FInitialized
      then Result := False
      else Result := ShowModal = mrOk;
    end;


    procedure   TFormWaveDeviceSelect.ClearInMasks;
    var
      i : Integer;
    begin
      for i := Low( FInMasks) to High( FInMasks)
      do InMasks[ i] := 0;
    end;


    procedure   TFormWaveDeviceSelect.ClearOutMasks;
    var
      i : Integer;
    begin
      for i := Low( FOutMasks) to High( FOutMasks)
      do OutMasks[ i] := 0;
    end;


    function    TFormWaveDeviceSelect.CreateAsioInStreamInfo: TPaAsioStreamInfo;
    // AsioStreamInfo can be freed after OpenStream returned : see FreeAsioStremInfo
    var
      aMaskCount : Integer;
      aMasks     : PPaChannelSelectors;
      i          : Integer;
    begin
      aMaskCount := Min( InMaskCount, MaxInChannels);
      aMasks     := nil;

      if aMaskCount > 0
      then begin
        GetMem( aMasks, aMaskCount * SizeOf( TPaChannelSelector));

        for i := 0 to aMaskCount - 1
        do aMasks^[ i] := InMasks[ i] - 1;
      end;

      with Result
      do begin
        size             := SizeOf( Result);
        hostApiType      := paASIO;
        version          := 1;
        channelselectors := aMasks;

        if Assigned( aMasks)
        then flags := paAsioUseChannelSelectors
        else flags := 0;
      end;
    end;


    function    TFormWaveDeviceSelect.CreateAsioOutStreamInfo: TPaAsioStreamInfo;
    // AsioStreamInfo can be freed after OpenStream returned : see FreeAsioStremInfo
    var
      aMaskCount : Integer;
      aMasks     : PPaChannelSelectors;
      i          : Integer;
    begin
      aMaskCount := Min( OutMaskCount, MaxOutChannels);
      aMasks     := nil;

      if aMaskCount > 0
      then begin
        GetMem( aMasks, aMaskCount * SizeOf( TPaChannelSelector));

        for i := 0 to aMaskCount - 1
        do aMasks^[ i] := OutMasks[ i] - 1;
      end;

      with Result
      do begin
        size             := SizeOf( Result);
        hostApiType      := paASIO;
        version          := 1;
        channelselectors := aMasks;

        if Assigned( amasks)
        then flags := paAsioUseChannelSelectors
        else flags := 0;
      end;
    end;


    procedure   TFormWaveDeviceSelect.FreeAsioStreamInfo( var aValue: TPaAsioStreamInfo);
    // This can be called after OpenStream returned, it does not free the record as
    // Delphi will take care of that once it goes out of scope, but the contained
    // channleselectors are not automatically freed.
    begin
      if Assigned( aValue.channelselectors)
      then begin
        FreeMem( aValue.channelselectors);
        aValue.channelselectors := nil;
      end;
    end;


// Delphi area

procedure TFormWaveDeviceSelect.FormCreate(Sender: TObject);
begin
  FInChannels  := TStringList.Create;
  FOutChannels := TStringList.Create;
  BufferSize   := 4096;
  BufferSize   :=    0;
end;


procedure TFormWaveDeviceSelect.FormDestroy(Sender: TObject);
begin
  FreeAndNil( FInChannels );
  FreeAndNil( FOutChannels);
end;


procedure TFormWaveDeviceSelect.EditBufferSizeChange(Sender: TObject);
var
  aValue : Integer;
begin
  aValue := StrToIntDef( EditBufferSize.Text, -1);

  if aValue <> -1
  then BufferSize := aValue;
end;


procedure TFormWaveDeviceSelect.BitBtnTestAudioClick(Sender: TObject);
begin
  RestartAudio;
end;


procedure TFormWaveDeviceSelect.BitBtnShowAsioPanelClick(Sender: TObject);
begin
  ShowAsioPanel;
end;


procedure TFormWaveDeviceSelect.ComboBoxAPIChange(Sender: TObject);
begin
  SelectedAPIID := ComboBoxAPI.ItemIndex;
end;


procedure TFormWaveDeviceSelect.ComboBoxInChannelChange(Sender: TObject);
begin
  SelectInputChannel( Sender as TComboBox);
end;


procedure TFormWaveDeviceSelect.ComboBoxOutChannelChange(Sender: TObject);
begin
  SelectOutputChannel( Sender as TComboBox)
end;


procedure TFormWaveDeviceSelect.ComboBoxWaveInPAChange(Sender: TObject);
begin
  SelectedInputIdPA := ComboBoxWaveInPA.ItemIndex;
end;


procedure TFormWaveDeviceSelect.ComboBoxWaveOutPAChange(Sender: TObject);
begin
  SelectedOutputIdPA := ComboBoxWaveOutPA.ItemIndex;
end;

end.

