unit wave_tools;

{

   COPYRIGHT 2013 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.Classes, System.SysUtils;

type

  TStereoSamples = array[ 0 .. MaxInt div SizeOf( Cardinal) - 1] of Cardinal;
  PStereoSamples = ^TStereoSamples;

  TStereoWaveWriter = class( TFileStream)
  private
    FSampleRate  : Cardinal;
    FSampleCount : Cardinal;
  private
    procedure   WriteHeader(aSampleRate : Cardinal);
    procedure   FixHeader;
  public
    constructor Create( const aFileName: string; aSampleRate: Cardinal);
    destructor  Destroy;                                         override;
    procedure   WriteSamples( aSamples: PStereoSamples; aCount: Cardinal);
    procedure   WriteSample( aSample: Cardinal);
    function    Duration: Cardinal; // in ms
  end;


implementation


type

  TFourCC = packed array[ 0 .. 3] of AnsiChar;

  TWaveStreamHeader = packed record
    RiffId       : TFourCC;
    RiffLength   : Cardinal;
    WaveId       : TFourCC;
    FmtId        : TFourCC;
    FmtSize      : Cardinal;
    WaveFormat   : Word;
    ChannelCount : Word;
    SampleRate   : Cardinal;
    StreamSpeed  : Cardinal;
    BlockAlign   : Word;
    BitCount     : Word;
    DataId       : TFourCC;
    DataSize     : Cardinal;
  end;


{ ========
  TStereoWaveWriter = class( TFileStream)
  private
    FSampleRate  : Cardinal;
    FSampleCount : Cardinal;
  private
}

    procedure   TStereoWaveWriter.WriteHeader( aSampleRate: Cardinal);
    var
      aHeader : TWaveStreamHeader;
    begin
      with aHeader
      do begin
        RiffId       := 'RIFF';                      //  0:4 - RIFF.
        RiffLength   :=  0;                          //  4:4 - Length : file size - 8 (not known yet)
        WaveId       := 'WAVE';                      //  8:4 - WAVE.
        FmtId        := 'fmt ';                      // 12:4 - fmt .
        FmtSize      := 16;                          // 16:4 - size of fmt chunk
        WaveFormat   := 1;                           // 20:2 - PCM format (1)
        ChannelCount := 2;                           // 22:2 - 2 channels
        SampleRate   := aSampleRate;                 // 24:4 - Sample rate
        StreamSpeed  := aSampleRate * 2 * 16 div 8;  // 28:4 - stream speed     - Samples_per_second*Number_of_channels*Bits_per_Sample/8
        BlockAlign   := 2 * 16 div 8;                // 32:2 - Block alignment  - Number_of_channels*Bits_per_Sample/8
        BitCount     := 16;                          // 34:2 - 16 bits / sample
        DataId       := 'data';                      // 36:4 - data.
        DataSize     := 0;                           // 40:4 - Number of bytes in data section (not known yet, filesize - 44)
      end;
      Write( aHeader, SizeOf( aHeader));
    end;

    procedure   TStereoWaveWriter.FixHeader;
    begin
      Seek( 4, soFromBeginning);                     // size of RIFF chunk
      WriteData( Cardinal( Size -  8));
      Seek( 40, soFromBeginning);                    // size of data chunk
      WriteData( Cardinal( Size - 44));
    end;

//  public

    constructor TStereoWaveWriter.Create( const aFileName: string; aSampleRate: Cardinal);
    begin
      inherited Create( aFileName, fmCreate or fmOpenReadWrite); // This will open the underlying file
      FSampleCount := 0;
      FSampleRate  := aSampleRate;
      Seek( 0, soFromBeginning);
      WriteHeader( aSampleRate);
    end;

    destructor  TStereoWaveWriter.Destroy; // override;
    begin
      FixHeader;
      Seek( 0, soFromEnd);                           // Set file pointer to end of file
      inherited;                                     // This will close the underlying file
    end;

    procedure   TStereoWaveWriter.WriteSamples( aSamples: PStereoSamples; aCount: Cardinal);
    begin
      WriteBuffer( aSamples, aCount * SizeOf( Cardinal));
      Inc( FSampleCount, aCount);
    end;

    procedure   TStereoWaveWriter.WriteSample( aSample: Cardinal);
    begin
      WriteData( aSample);
      Inc( FSampleCount);
    end;

    function    TStereoWaveWriter.Duration: Cardinal;
    begin
      Result := ( Int64( 1000) * Int64( FSampleCount)) div Int64( FSampleRate);
    end;



end.
