unit windowing;

{

  (C) COPYRIGHT 2014 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

{
  Windowing functions to be used with FFT conversions
}

interface

uses

  System.Math, System.SysUtils,

  KnobsUtils;

type

  TFFTWindow      = class;
  TFFTWindowClass = class of TFFTWindow;


  TFFTWindow = class
  private
    FLocks : Integer;
    FSize  : Integer;
    FTable : TSignalArray;
  private
    procedure   Lock;
    procedure   UnLock;
    function    GetLocked: Boolean;
    procedure   SetSize( aValue: Integer);
    function    GetTable( anIndex: Integer): TSignal;
  protected
    procedure   Calculate;                                           virtual; abstract;
  public
    constructor Create( aSize: Integer);                             virtual;
    procedure   ApplyTo( var aData: TSignalArray);                   virtual;
    function    SameParams( aWindow: TFFTWindow): Boolean;           virtual;
    function    SameAs    ( aWindow: TFFTWindow): Boolean;
  public
    property    Size                    : Integer read FSize      write SetSize;
    property    Locked                  : Boolean read GetLocked;
    property    Table[ anIndex: Integer]: TSignal read GetTable; default;
  end;

  TFFTRectWindow = class( TFFTWindow)
  protected
    procedure   Calculate;                                          override;
  end;

  TFFTTriangularWindow = class( TFFTWindow)
  private
    FL : Integer;
  protected
    procedure   Calculate;                                          override;
    procedure   SetParams( L: Integer);
  public
    constructor Create( aSize: Integer);                            override;
    function    SameParams( aWindow: TFFTWindow): Boolean;          override;
  end;

  TFFTWelchWindow = class( TFFTWindow)
  protected
    procedure   Calculate;                                          override;
  end;

  TFFTHammingWindow = class( TFFTWindow)
  private
    FAlpha : TSignal;
    FBeta  : TSignal;
  protected
    procedure   Calculate;                                          override;
  public
    constructor Create( aSize: Integer);                            override; // Defaults to a Hann window
    function    SameParams( aWindow: TFFTWindow): Boolean;          override;
    procedure   SetParams( Alpha, Beta: TSignal);
  end;

  TFFTBlackmanWindow = class( TFFTWindow)
  private
    FAlpha : TSignal;
  protected
    procedure   Calculate;                                          override;
  public
    constructor Create( aSize: Integer);                            override;
    function    SameParams( aWindow: TFFTWindow): Boolean;          override;
    procedure   SetParams( Alpha: TSignal);
  end;

  TFFTNuttallWindow = class( TFFTWindow)
  private
    Fa0 : TSignal;
    Fa1 : TSignal;
    Fa2 : TSignal;
    Fa3 : TSignal;
  protected
    procedure   Calculate;                                          override;
  public
    constructor Create( aSize: Integer);                            override;
    function    SameParams( aWindow: TFFTWindow): Boolean;          override;
    procedure   SetParams( a0, a1, a2, a3: TSignal);
  end;

  TFFTBlackmanNutallWindow = class( TFFTNuttallWindow)
  public
    constructor Create( aSize: Integer);                            override;
  end;

  TFFTBlackmanHarrisWindow = class( TFFTNuttallWindow)
  public
    constructor Create( aSize: Integer);                            override;
  end;



implementation


{ ========
  TFFTWindow = class
  private
    FLocks : Integer;
    FSize  : Integer;
    FTable : TSignalArray;
  public
    property    Size                    : Integer read FSize      write SetSize;
    property    Locked                  : Boolean read GetLocked;
    property    Table[ anIndex: Integer]: TSignal read GetTable; default;
  private
}

    procedure   TFFTWindow.Lock;
    begin
      Inc( FLocks);
    end;

    procedure   TFFTWindow.UnLock;
    begin
      if FLocks > 0
      then Dec( FLocks);
    end;

    function    TFFTWindow.GetLocked: Boolean;
    begin
      Result := FLocks <> 0;
    end;

    procedure   TFFTWindow.SetSize( aValue: Integer);
    begin
      if  aValue <> FSize
      then begin
        Lock;
        try
          FSize := aValue;
          SetLength( FTable, aValue);
          Calculate;
        finally
          Unlock;
        end;
      end;
    end;

    function    TFFTWindow.GetTable( anIndex: Integer): TSignal;
    begin
      if ( anIndex< 0) or ( anIndex >= Size)
      then Result := 0
      else Result := FTable[ anIndex];
    end;

//  public

    constructor TFFTWindow.Create( aSize: Integer); // virtual;
    begin
      inherited Create;
      Size := aSize;
    end;

    procedure   TFFTWindow.ApplyTo( var aData: TSignalArray);
    var
      n : Integer;
    begin
      if ( FSize = Length( aData)) and not Locked
      then begin
        for n := 0 to FSize - 1
        do aData[ n] := aData[ n] * FTable[ n];
      end;
    end;

    function    TFFTWindow.SameParams( aWindow: TFFTWindow): Boolean; // virtual;
    begin
      Result := True;
    end;

    function    TFFTWindow.SameAs( aWindow: TFFTWindow): Boolean;
    begin
      Result := False;
      if Assigned( aWindow)
      then Result := ( aWindow.ClassType = ClassType) and ( aWindow.Size = Size) and SameParams( aWindow);
    end;


{ ========
  TFFTRectWindow = class( TFFTWindow)
  protected
}

    procedure   TFFTRectWindow.Calculate; // override;
    var
      n : Integer;
    begin
      for n := 0 to Size - 1
      do FTable[ n] := 1;
    end;


{ ========
  TFFTTriangularWindow = class( TFFTWindow)
  private
    FL : Integer;
  protected
}

    procedure   TFFTTriangularWindow.Calculate; // override;
    var
      n  : Integer;
      h1 : TSignal;
      h2 : TSignal;
    begin
      if FL > 0
      then begin
        h1 := Size / 2;
        h2 := FL   / 2;
        for n := 0 to Size - 1
        do FTable[ n] := 1 - Abs(( n - h1) / h2);
      end;
    end;

    procedure   TFFTTriangularWindow.SetParams( L: Integer);
    begin
      if L <> FL
      then begin
        Lock;
        try
          FL := L;
          Calculate;
        finally
          UnLock;
        end;
      end;
    end;

//  public

    constructor TFFTTriangularWindow.Create( aSize: Integer); // override;
    begin
      SetParams( aSize);
      inherited;
    end;

    function    TFFTTriangularWindow.SameParams( aWindow: TFFTWindow): Boolean; // override;
    begin
      Result :=
        ( aWindow is TFFTTriangularWindow)        and
        ( TFFTTriangularWindow( aWindow).FL = FL);
    end;


{ ========
  TFFTWelchWindow = class( TFFTWindow)
  protected
}

    procedure   TFFTWelchWindow.Calculate; // override;
    var
      n : Integer;
      h : TSignal;
    begin
      h := Size/ 2;
      for n := 0 to Size - 1
      do FTable[ n] := 1 - Sqr(( n - h) / h)
    end;


{ ========
  TFFTHammingWindow = class( TFFTWindow)
  private
    FAlpha : TSignal;
    FBeta  : TSignal;
  protected
}

    procedure   TFFTHammingWindow.Calculate; // override;
    var
      n : Integer;
    begin
      for n := 0 to Size - 1
      do
        FTable[ n] :=
          FAlpha -
          FBeta * Cos( 2 * PI * n / Size);
    end;

//  public

    constructor TFFTHammingWindow.Create( aSize: Integer); // override;
    begin
      inherited;
      SetParams( 0.5, 0.5);
    end;

    function    TFFTHammingWindow.SameParams( aWindow: TFFTWindow): Boolean; // override;
    begin
      Result :=
        ( aWindow is TFFTHammingWindow)                and
        ( TFFTHammingWindow( aWindow).FAlpha = FAlpha) and
        ( TFFTHammingWindow( aWindow).FBeta  = FBeta )
    end;

    procedure   TFFTHammingWindow.SetParams( Alpha, Beta: TSignal);
    begin
      if ( Alpha <> FAlpha) or ( Beta <> FBeta)
      then begin
        Lock;
        try
          FAlpha := Alpha;
          FBeta  := Beta;
          Calculate;
        finally
          UnLock;
        end;
      end;
    end;


{ ========
  TFFTBlackmanWindow = class( TFFTWindow)
  private
    FAlpha : TSignal;
  protected
}

    procedure   TFFTBlackmanWindow.Calculate; // override;
    var
      n  : Integer;
      a0 : TSignal;
      a1 : TSignal;
      a2 : TSignal;
    begin
      a0 := ( 1 - FAlpha) / 2;
      a1 := 0.5;
      a2 := FAlpha / 2;
      for n := 0 to Size - 1
      do
        FTable[ n] :=
          a0                           -
          a1 * Cos( 2 * PI * n / Size) +
          a2 * Cos( 4 * PI * n / Size);
    end;

//  public

    constructor TFFTBlackmanWindow.Create( aSize: Integer); // override;
    begin
      inherited;
      SetParams( 0.16);
    end;

    function    TFFTBlackmanWindow.SameParams( aWindow: TFFTWindow): Boolean; // override;
    begin
      Result :=
        ( aWindow is TFFTBlackmanWindow)                and
        ( TFFTBlackmanWindow( aWindow).FAlpha = FAlpha);
    end;

    procedure   TFFTBlackmanWindow.SetParams( Alpha: TSignal);
    begin
      if Alpha <> FAlpha
      then begin
        Lock;
        try
          FAlpha := Alpha;
          Calculate;
        finally
          UnLock;
        end;
      end;
    end;


{ ========
  TFFTNuttallWindow = class( TFFTWindow)
  private
    Fa0 : TSignal;
    Fa1 : TSignal;
    Fa2 : TSignal;
    Fa3 : TSignal;
  protected
}

    procedure   TFFTNuttallWindow.Calculate; // override;
    var
      n : Integer;
    begin
      for n := 0 to Size - 1
      do
        FTable[ n] :=
          Fa0                           -
          Fa1 * Cos( 2 * PI * n / Size) +
          Fa2 * Cos( 4 * PI * n / Size) -
          Fa3 * Cos( 6 * PI * n / Size);
    end;

//  public

    constructor TFFTNuttallWindow.Create( aSize: Integer); // override;
    begin
      inherited;
      SetParams( 0.355769, 0.487396, 0.144232, 0.012604);
    end;

    function    TFFTNuttallWindow.SameParams( aWindow: TFFTWindow): Boolean; // override;
    begin
      Result :=
        ( aWindow is TFFTNuttallWindow)          and
        ( TFFTNuttallWindow( aWindow).Fa0 = Fa0) and
        ( TFFTNuttallWindow( aWindow).Fa1 = Fa1) and
        ( TFFTNuttallWindow( aWindow).Fa2 = Fa2) and
        ( TFFTNuttallWindow( aWindow).Fa3 = Fa3);
    end;

    procedure   TFFTNuttallWindow.SetParams( a0, a1, a2, a3: TSignal);
    begin
      if ( a0 <> Fa0) or ( a1 <> Fa1) or ( a2 <> Fa2) or ( a3 <> Fa3)
      then begin
        Lock;
        try
          Fa0 := a0;
          Fa1 := a1;
          Fa2 := a2;
          Fa3 := a3;
          Calculate;
        finally
          UnLock;
        end;
      end;
    end;


{ ========
  TFFTBlackmanNutallWindow = class( TFFTNuttallWindow)
  public
}

    constructor TFFTBlackmanNutallWindow.Create( aSize: Integer); // override;
    begin
      inherited;
      SetParams( 0.3635819, 0.4891775, 0.1365995, 0.0106411);
    end;


{ ========
  TFFTBlackmanHarrisWindow = class( TFFTNuttallWindow)
  public
}

    constructor TFFTBlackmanHarrisWindow.Create( aSize: Integer); // override;
    begin
      inherited;
      SetParams( 0.35875, 0.48829, 0.14128, 0.01168);
    end;



end.

