unit BSplines;

{
  Original code comments are:

  ------------------------------------------------------------------------------

  This code was written by : M. v. Engeland

  This code is copyright 2000 by
  M. v. Engeland

  This is the followup of the B-splines component posted in 1997.
  Some bugs were fixed, and some changes were made. The splines now use
  dynamic allocation of memory when needed when adding points to the spline.
  If the number of points to be added is known in advance, the the precise
  amount of memory can be allocated setting the capacity property.

  The conditions for using this component however are not changed:
  You may use this component in any way you want to, whether it is for
  commercial purposes or not, as long as you don't hold me responsible for
  any disasters it may cause. But I would appreciate it if you would send me
  an e-mail (martijn@dutw38.wbmt.tudelft.nl) to tell me what you use it for
  because over the last three years I've learned that it has been used for
  a wide variaty of interesting applications which it was initially never
  intended for. Besides, it's nice to know how your offspring's doing.
  Also all comments and/or remarks are welcome.

  See the demo program for how to use the component.
  Special attention is payed on the possibility of interpolating the
  vertices. As you may or may not know, B-Splines normally do not
  interpolate the used controlpoints. Thanks to simple matrix calculation
  however it is possible to interpolate the controlpoints by calculating
  new vertices in such a way that the spline interpolates the original ones.

  ------------------------------------------------------------------------------
  ------------------------------------------------------------------------------

  2016-05-21: Changed to XE5 code for use in Wren by BlueHell

  - code layout changed to be better readable to me
  - renamed used units to XE5 conventions, removed unused units.
  - changed extended data type to Double
  - changed single   data type to Double
  - changed build to built where it was past tense
  - removed the registration code for the component. I did not need it, and it
    should be in a design time package anyway when needed.

  ------------------------------------------------------------------------------

  Modifications are (C) COPYRIGHT 2016 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl

}


interface

uses

  WinApi.Windows, System.SysUtils, System.Classes, Vcl.Graphics, System.Math;


const

  MaxFragments            = 600;               // The maximum of straight line segments allowed for drawing the spline
  MaxResults              = MaxFragments + 10; // Max. number of calculated intersections
  MaxInterpolatedVertices = 250;               // The maximum number of vertices that can be interpolated, up to 16000 allowed
  MaxCalcSteps            = 150;               // Number of steps for numerical intersection calculating
  MaxError                = 1e-5;              // Max error for intersection calculating
  MaxIterations           = 80;
  VerticesIncrement       = 25;                // Number of vertices to allocate memory for when the count property exceeds the current capacity


type

  TDataType = Double;

  TVertex = record
    X : TDataType;
    Y : TDataType;
  end;

  ESplines = class( Exception);


  // The following dynamic array is used to store the desired user-specified controlpoints
  T2DPointList = array[ 1 .. 1] of TVertex;
  P2DPointList = ^T2DPointList;


  // The vertexlist is used internally to make the spline interpolate the controlpoints
  TVertexList   = array[ 0 .. 0] of TVertex;
  P2DVertexList = ^TVertexList;


  // The knuckle list stores a flag to see whether a point is a knuckle or not
  TKnuckleList = array[ 1 .. 1] of Boolean;
  PKnuckleList = ^TKnuckleList;


  // T2DResults is a record with calculatedvalues at a specific point when for ex. the x-value is known
  T2DResults = record
    NumberOfHits : Integer;
    Parameter    : array[ 1 .. MaxResults] of TDataType;
    Point        : array[ 1 .. MaxResults] of TVertex;
  end;


  // 2D B-spline class:
  TBSpline = class // 2D B-Spline object
  private
    FPointCount   : Integer;
    FCapacity     : Integer;
    FPointList    : P2DPointList;
    FVertexList   : P2DVertexList;
    FKnuckleList  : PKnuckleList;
    FBuilt        : Boolean;
    FVertexCount  : Integer;
    FInterpolated : Boolean;
    FMin          : TVertex;
    FMax          : TVertex;
    FFragments    : Integer;
  private
    procedure   SetBuilt( aValue: Boolean);
    procedure   SetCapacity( aValue: Integer);                                                                  virtual;
    procedure   SetInterpolated( const aValue: Boolean);
    procedure   SetFragments( const aValue: Integer);
    function    GetPoint  ( anIndex: Integer): TVertex;                                                         virtual;
    procedure   SetPoint  ( anIndex: Integer; aValue : TVertex);                                                virtual;
    function    GetVertex ( anIndex: Integer): TVertex;                                                         virtual;
    procedure   SetVertex ( anIndex: Integer; aValue : TVertex);                                                virtual;
    function    GetKnuckle( anIndex: Integer): Boolean;                                                         virtual;
    procedure   SetKnuckle( anIndex: Integer; aValue : Boolean);                                                virtual;
    function    GetVertexCount: Integer;
    procedure   Interpolate;                                                                                    virtual;
    procedure   FillMatrix;                                                                                     virtual;
    procedure   PhantomPoints;
  public
    procedure   AddPoint( aVertex: TVertex);                                                                    virtual;
    procedure   Extents( var aMin, aMax: TVertex);                                                              virtual;
    procedure   Clear;                                                                                          virtual;
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Draw( aCanvas: TCanvas; aColor, aDotColor: TColor; aDotSize: Integer);
    procedure   DeletePoint( anIndex: Integer);
    procedure   InsertPoint( anIndex: Integer; aVertex: TVertex);
    procedure   Invert;                            // Inverse the controlpoints, eg the last point first and vice versa
    function    KnownX( anX: TDataType; var Results: T2DResults): Boolean;                                      virtual;
    function    KnownY( anY: TDataType; var Results: T2DResults): Boolean;                                      virtual;
    function    Value           ( aParameter: Double): TVertex;                                                 virtual;
    function    FirstDerivative ( aParameter: Double): TVertex;                                                 virtual;
    function    SecondDerivative( aParameter: Double): TVertex;    { second derivative in a point }             virtual;
    function    Rebuild: Boolean;                                                                               virtual;
  public
    property    Built                       : Boolean read FBuilt         write SetBuilt;
    property    PointCount                  : Integer read FPointCount;
    property    Capacity                    : Integer read FCapacity      write SetCapacity;
    property    Fragments                   : Integer read FFragments     write SetFragments;
    property    Interpolated                : Boolean read FInterpolated  write SetInterpolated;
    property    VertexCount                 : Integer read GetVertexCount;
    property    Point  [ anIndex : Integer] : TVertex read GetPoint       write SetPoint;
    property    Vertex [ anIndex : Integer] : TVertex read GetVertex      write SetVertex;
    property    Knuckle[ anIndex : Integer] : Boolean read GetKnuckle     write SetKnuckle;
    property    Min                         : TVertex read FMin;
    property    Max                         : TVertex read FMax;
  end;


  TSplines = class( TComponent) // Component to store all B-Splines in
  private
    FSplines : TList;
  private
    function    GetCount: Integer;
    function    GetSpline( anIndex : Integer): TBSpline;
    procedure   SetSpline( anIndex : Integer; const aValue: TBSpline);
  public
    constructor Create( anOwner : TComponent);                                                                 override;
    destructor  Destroy;                                                                                       override;
    procedure   AddSpline( const aBSpline: TBSpline);
    procedure   Clear;
    procedure   InsertSpline( aPos : Integer; const aBSpline: TBSpline);
    procedure   DeleteSpline( const aBSpline: TBSpline);
  public
    property    Items[ anIndex : Integer] : TBSpline read GetSpline write SetSpline;
    property    Count                     : Integer  read GetCount;
  end;


  function Vertex( const anX, anY: TDataType): TVertex;


implementation



// The following tpes are used for the interpolation routines

type

  TMatrixRow = array[ 1 .. 1] of TDataType;
  PMatrixRow = ^TMatrixRow;
  TMatrix    = array[ 1 .. MaxInterpolatedVertices] of PMatrixRow;

var

  Matrix : TMatrix;


  function Vertex( const anX, anY: TDataType): TVertex;
  begin
    Result.X := anX;
    Result.Y := anY;
  end;


  function DistPP2D( P1, P2 : TVertex): TDataType;
  begin
    Result := Sqrt( Sqr( P2.X - P1.X) + Sqr( P2.Y - P1.Y));
  end;


{ ========
  TBSpline = class // 2D B-Spline object
  private
    FPointCount   : Integer;
    FCapacity     : Integer;
    FPointList    : P2DPointList;
    FVertexList   : P2DVertexList;
    FKnuckleList  : PKnuckleList;
    FBuilt        : Boolean;
    FVertexCount  : Integer;
    FInterpolated : Boolean;
    FMin          : TVertex;
    FMax          : TVertex;
    FFragments    : Integer;
  public
    property    Built                       : Boolean read FBuilt         write SetBuilt;
    property    PointCount                  : Integer read FPointCount;
    property    Capacity                    : Integer read FCapacity      write SetCapacity;
    property    Fragments                   : Integer read FFragments     write SetFragments;
    property    Interpolated                : Boolean read FInterpolated  write SetInterpolated;
    property    VertexCount                 : Integer read GetVertexCount;
    property    Point  [ anIndex : Integer] : TVertex read GetPoint       write SetPoint;
    property    Vertex [ anIndex : Integer] : TVertex read GetVertex      write SetVertex;
    property    Knuckle[ anIndex : Integer] : Boolean read GetKnuckle     write SetKnuckle;
    property    Min                         : TVertex read FMin;
    property    Max                         : TVertex read FMax;
  private
}

    procedure   TBSpline.SetBuilt( aValue: Boolean);
    begin
      if not aValue
      then begin
        // Release allocated memory for vertices
        if Assigned( FVertexList) and Built
        then Freemem( FVertexList, ( FVertexCount + 2) * SizeOf( TVertex));

        FVertexCount := 0;
        FVertexList  := nil;

        // Clear extents
        FMin.X := 0;
        FMin.Y := 0;
        FMax.X := 1;
        FMax.Y := 1;
      end;

      FBuilt := aValue;
    end;


    procedure   TBSpline.SetCapacity( aValue: Integer);
    var
      CurrentSize : Word;
      NewSize     : Word;
      OldPoints   : P2DPointList;
      OldKnuckle  : PKnuckleList;
    begin
      if aValue <> FCapacity
      then begin
        CurrentSize  := Capacity * SizeOf( TVertex);
        NewSize      := aValue * SizeOf( TVertex);
        OldPoints    := FPointList;
        FPointList   := nil;
        OldKnuckle   := FKnuckleList;
        FKnuckleList := nil;

        if aValue > 0
        then begin
          GetMem( FPointList, NewSize);
          GetMem( FKnuckleList, aValue);
          FillChar( FKnuckleList^, aValue, 0);

          if Capacity <> 0
          then begin
            Move( OldKnuckle^, FKnuckleList^, Capacity   );
            Move( OldPoints^ , FPointList^  , CurrentSize);
          end;
        end;

        if CurrentSize <> 0
        then begin
          Freemem( OldPoints , CurrentSize);
          Freemem( OldKnuckle, Capacity   );
        end;

        FCapacity := aValue;
      end;
    end;


    procedure   TBSpline.SetFragments( const aValue: Integer);
    begin
      if FFragments <> aValue
      then begin
        FFragments := aValue;

        if FFragments > MaxFragments
        then FFragments := MaxFragments;
      end;
    end;


    procedure   TBSpline.SetInterpolated( const aValue: Boolean);
    begin
      if aValue <> FInterpolated
      then begin
        FInterpolated := aValue;
        Built         := False;
      end;
    end;


    function   TBSpline.GetPoint( anIndex: Integer) : TVertex;
    begin
      if ( anIndex >= 1) and ( anIndex <= FPointCount)
      then Result := FPointList^[ anIndex]

      else begin
        Result.X := 0;
        Result.Y := 0;
        Result.Y := 0;

        raise ESplines.Create( 'List index out of bounds in ' + ClassName + '.FGetPoint. (' + IntToStr( anIndex) + ').');
      end;
    end;


    procedure   TBSpline.SetPoint( anIndex: Integer; aValue: TVertex);
    begin
      if (anIndex >= 1) and (anIndex <= FPointCount)
      then begin
        FPointList^[anIndex] := aValue;
        Built                := False;
      end
      else raise ESplines.Create( 'List index out of bounds in ' + ClassName + '.FSetPoint. (' + IntToStr( anIndex) + ').');
    end;


    function   TBSpline.GetVertex( anIndex: Integer): TVertex;
    begin
      Result.X := 0;
      Result.Y := 0;

      if not Built
      then begin
        if FPointCount > 1
        then Rebuild
        else Exit;
      end;

      if (anIndex >= 0) and (anIndex <= VertexCount + 1)
      then Result := FVertexList^[anIndex]
      else raise ESplines.Create( 'List index out of bounds in ' + ClassName + '.FGetVertex. (' + IntToStr( anIndex) + ').');
    end;


    procedure   TBSpline.SetVertex( anIndex: Integer; aValue: TVertex);
    begin
      if ( anIndex >= 0) and ( anIndex <= VertexCount + 1)
      then FVertexList^[anIndex] := aValue
      else raise ESplines.Create( 'List index out of bounds in ' + ClassName + '.FSetVertex. (' + IntToStr( anIndex) + ').');
    end;


    function    TBSpline.GetKnuckle( anIndex: Integer): Boolean;
    begin
      if ( anIndex = 1) or ( anIndex = FPointCount)
      then Result := False
      else if ( anIndex > 0) and ( anIndex <= FPointCount)
      then Result := FKnuckleList^[anIndex]
      else raise ESplines.Create( 'List index out of bounds in ' + ClassName + 'FGetKnuckle. (' + IntToStr(anIndex) + ').');
    end;


    procedure   TBSpline.SetKnuckle( anIndex: Integer; aValue: Boolean);
    begin
      if ( anIndex > 0) and ( anIndex <= FPointCount)
      then begin
        FKnuckleList^[ anIndex] := aValue;
        Built                   := False;
      end
      else raise ESplines.Create( 'List index out of bounds in ' + ClassName + '.FSetKnuckle. (' + IntToStr( anIndex) + ').');
    end;


    function TBSpline.GetVertexCount: Integer;
    begin
      if not Built
      then Rebuild;

      Result := FVertexCount;
    end;


    function    TBSpline.Rebuild: Boolean;
    var
      i        : Integer;
      j        : Integer;
      Vertex2D : TVertex;
    begin
      if FPointCount > 1
      then begin
        if Assigned( FVertexList)
        then begin
          Freemem( FVertexList, ( FVertexCount + 2) * SizeOf( TVertex));
          FVertexList := nil;
        end;

        FVertexCount := 0;

        for i := 1 to FPointCount
        do begin
          if Knuckle[ i]
          then Inc( FVertexCount, 3)
          else Inc( FVertexCount, 1);
        end;

        GetMem( FVertexList, ( FVertexCount + 2) * SizeOf( TVertex));

        j := 0;

        for i := 1 to FPointCount
        do begin
          Vertex2D := Point[ i];

          if Knuckle[ i]
          then begin
            FVertexList^[ j + 1] := Vertex2D;
            FVertexList^[ j + 2] := Vertex2D;
            Inc( j, 2);
          end;

          FVertexList^[ j + 1] := FPointList^[ i];

          if i = 1
          then begin
            FMin := Vertex2D;
            FMax := FMin;
          end
          else begin
            if Vertex2D.X < FMin.X
            then FMin.X := Vertex2D.X;

            if Vertex2D.Y < FMin.Y
            then FMin.Y := Vertex2D.Y;

            if Vertex2D.X > FMax.X
            then FMax.X := Vertex2D.X;

            if Vertex2D.Y > FMax.Y
            then FMax.Y := Vertex2D.Y;
          end;

          Inc(j);
        end;

        if Interpolated
        then begin
          for i := 1 to FVertexCount
          do begin
            GetMem  ( Matrix[ i] , FVertexCount * SizeOf( TDataType));
            FillChar( Matrix[ i]^, FVertexCount * SizeOf( TDataType), 0);
          end;

          FillMatrix;
          Interpolate;

          for i := 1 to FVertexCount
          do begin
            Freemem( Matrix[ i], FVertexCount * SizeOf( TDataType));
            Matrix[ i] := nil;
          end;
        end;
      end;

      Built  := True;
      Result := Built;
      PhantomPoints;
    end;


    procedure   TBSpline.Interpolate;
    var
      i      : Integer;
      j      : Integer;
      k      : Integer;
      Factor : Double;
      Tmp    : P2DVertexList;
    begin
      if ( FVertexCount < MaxInterpolatedVertices) and ( FVertexCount > 2)
      then begin
        GetMem( Tmp, ( FVertexCount + 2) * SizeOf( TVertex));

        for i := 1 to FVertexCount
        do begin
          for j := i + 1 to FVertexCount
          do begin
            Factor := Matrix[ j]^[ i] / Matrix[ i]^[ i];

            for k := 1 to FVertexCount
            do Matrix[ j]^[ k] := Matrix[ j]^[ k] - Factor * Matrix[ i]^[ k];

            FVertexList^[ j].X := FVertexList^[ j].X - Factor * FVertexList^[ j - 1].X;
            FVertexList^[ j].Y := FVertexList^[ j].Y - Factor * FVertexList^[ j - 1].Y;
          end;
        end;

        Tmp^[ FVertexCount].X := FVertexList^[ FVertexCount].X / Matrix[ FVertexCount]^[ FVertexCount];
        Tmp^[ FVertexCount].Y := FVertexList^[ FVertexCount].Y / Matrix[ FVertexCount]^[ FVertexCount];

        for i := FVertexCount - 1 downto 1
        do begin
          Tmp^[ i].X := ( 1 / Matrix[ i]^[ i]) * ( FVertexList^[ i].X - Matrix[ i]^[ i + 1] * Tmp^[ i + 1].X);
          Tmp^[ i].Y := ( 1 / Matrix[ i]^[ i]) * ( FVertexList^[ i].Y - Matrix[ i]^[ i + 1] * Tmp^[ i + 1].Y);
        end;

        if Assigned( FVertexList)
        then
        begin
          Freemem( FVertexList, ( FVertexCount + 2) * SizeOf( TVertex));
          FVertexList := nil;
        end;
        FVertexList := Tmp;
      end;
    end;


    function    TBSpline.KnownX( anX: TDataType; var Results: T2DResults): Boolean;
    var
      UpperLimit : Integer;
      Counter    : Integer;
      Min        : Double;
      Max        : Double;
      Parameter  : Double;
      Error      : Double;
      Finished   : Boolean;
      P1         : TVertex;
      P2         : TVertex;
      Output     : TVertex;
    begin
      Result := False;

      if not Built and not Rebuild
      then Exit;

      Results.NumberOfHits := 0;

      if PointCount = 0
      then Exit;

      if PointCount = 2
      then begin
        P1 := Point[1];
        P2 := Point[2];

        if P1.X > P2.X
        then begin
          Output := P1;
          P1     := P2;
          P2     := Output;
        end;

        if ( P1.X <= anX) and ( P2.X >= anX)
        then begin
          if Abs( P1.X - P2.X) < 1e6
          then Parameter := 0.5
          else Parameter := ( anX - P1.X) / ( P2.X - P1.X);

          Results.NumberOfHits                       := Results.NumberOfHits + 1;
          Results.Parameter[ Results.NumberOfHits]   := Parameter;
          Results.Point    [ Results.NumberOfHits]   := Value( Parameter);
          Results.Point    [ Results.NumberOfHits].X := anX;
        end;
      end
      else begin
        UpperLimit := 1;
        P1         := Point[ 1];

        repeat
          Finished := False;

          repeat
            P2 := Value( UpperLimit / MaxCalcSteps);

            if (( P1.X <= anX) and ( P2.X >= anX)) or (( P1.X >= anX) and ( P2.X <= anX))
            then Finished := True;

            if not Finished
            then begin
              if UpperLimit = MaxCalcSteps
              then begin
                Result := Results.NumberOfHits > 0;
                Exit;
              end
              else begin
                Inc( UpperLimit);
                P1 := P2;
              end;
            end;
          until Finished;

          Max     := UpperLimit        / MaxCalcSteps;
          Min     := ( UpperLimit - 1) / MaxCalcSteps;
          Counter := 0;

          repeat
            if Abs( P1.X - P2.X) < 1e-6
            then Parameter := 0.5 * ( Min + Max)
            else if ( P1.X <= anX) and ( P2.X >= anX)
            then Parameter := Min + ( Max - Min) * (( anX - P1.X) / ( P2.X - P1.X))
            else Parameter := Min + ((P1.X - anX) * (Max - Min) / (P1.X - P2.X));

            if Parameter < Min
            then Parameter := Min;

            if Parameter > Max
            then Parameter := Max;

            Output := Value(Parameter);

            if anX = 0
            then Error := Abs( Output.X - anX)
            else Error := Abs(( anX - Output.X) / anX);

            if Output.X > anX
            then begin
              if ( P1.X <= anX) and ( P2.X >= anX)
              then begin
                Max := Parameter;
                P2  := Output;
              end
              else begin
                Min := Parameter;
                P1  := Output;
              end;
            end
            else begin
              if ( P1.X <= anX) and ( P2.X >= anX)
              then begin
                Min := Parameter;
                P1  := Output;
              end
              else begin
                Max := Parameter;
                P2  := Output;
              end;
            end;

            Inc( Counter);
          until ( Error < MaxError) or ( Counter > MaxIterations);

          if Results.NumberOfHits >= MaxResults
          then raise ESplines.Create( 'Max. number of results exceeded in TBSpline.KnownX');

          Results.NumberOfHits                       := Results.NumberOfHits + 1;
          Results.Parameter[ Results.NumberOfHits]   := Parameter;
          Results.Point    [ Results.NumberOfHits]   := Value(Parameter);
          Results.Point    [ Results.NumberOfHits].X := anX;
          Max                                        := UpperLimit / MaxCalcSteps;
          P1                                         := Value( Max);
          Inc( UpperLimit);
        until UpperLimit > MaxCalcSteps;
      end;

      Result := Results.NumberOfHits > 0;
    end;


    function   TBSpline.KnownY( anY: TDataType; var Results: T2DResults): Boolean;
    var
      UpperLimit : Integer;
      Counter    : Integer;
      Min        : Double;
      Max        : Double;
      Parameter  : Double;
      Error      : Double;
      Finished   : Boolean;
      P1         : TVertex;
      P2         : TVertex;
      Output     : TVertex;
    begin
      Result := False;

      if not Built and not Rebuild
      then Exit;

      Results.NumberOfHits := 0;

      if PointCount = 0
      then Exit;

      if PointCount = 2
      then begin
        P1 := Point[ 1];
        P2 := Point[ 2];

        if P1.Y > P2.Y
        then begin
          Output := P1;
          P1     := P2;
          P2     := Output;
        end;

        if ( P1.Y <= anY) and ( P2.Y >= anY)
        then begin
          if Abs( P1.Y - P2.Y) < 1e6
          then Parameter := 0.5
          else Parameter := ( anY - P1.Y) / ( P2.Y - P1.Y);

          Results.NumberOfHits                       := Results.NumberOfHits + 1;
          Results.Parameter[ Results.NumberOfHits]   := Parameter;
          Results.Point    [ Results.NumberOfHits]   := Value( Parameter);
          Results.Point    [ Results.NumberOfHits].Y := anY;
        end;
      end
      else begin
        UpperLimit := 1;
        P1         := Point[1];

        repeat
          Finished := False;

          repeat
            P2 := Value( UpperLimit / MaxCalcSteps);

            if (( P1.Y <= anY) and ( P2.Y >= anY)) or (( P1.Y >= anY) and ( P2.Y <= anY))
            then Finished := True;

            if not Finished
            then begin
              if UpperLimit = MaxCalcSteps
              then begin
                Result := Results.NumberOfHits > 0;
                Exit;
              end
              else begin
                Inc( UpperLimit);
                P1 := P2;
              end;
            end;
          until Finished;

          Max     := UpperLimit        / MaxCalcSteps;
          Min     := ( UpperLimit - 1) / MaxCalcSteps;
          Counter := 0;

          repeat
            if Abs( P1.Y - P2.Y) < 1e-6
            then Parameter := 0.5 * ( Min + Max)
            else if ( P1.Y <= anY) and ( P2.Y >= anY)
            then Parameter := Min + ( Max - Min) * (( anY - P1.Y) / ( P2.Y - P1.Y))
            else Parameter := Min + (( P1.Y - anY) * ( Max - Min) / ( P1.Y - P2.Y));

            if Parameter < Min
            then Parameter := Min;

            if Parameter > Max
            then Parameter := Max;

            Output := Value( Parameter);

            if anY = 0
            then Error := Abs( Output.Y - anY)
            else Error := Abs(( anY - Output.Y) / anY);

            if Output.Y > anY
            then begin
              if ( P1.Y <= anY) and ( P2.Y >= anY)
              then begin
                Max := Parameter;
                P2  := Output;
              end
              else begin
                Min := Parameter;
                P1  := Output;
              end;
            end
            else begin
              if ( P1.Y <= anY) and ( P2.Y >= anY)
              then begin
                Min := Parameter;
                P1  := Output;
              end
              else begin
                Max := Parameter;
                P2  := Output;
              end;
            end;

            Inc( Counter);
          until ( Error < MaxError) or ( Counter > MaxIterations);

          if Results.NumberOfHits >= MaxResults
          then raise ESplines.Create( 'Max. number of results exceeded in TBSpline.KnownY');

          Results.NumberOfHits                       := Results.NumberOfHits + 1;
          Results.Parameter[ Results.NumberOfHits]   := Parameter;
          Results.Point    [ Results.NumberOfHits]   := Value(Parameter);
          Results.Point    [ Results.NumberOfHits].Y := anY;
          Max                                        := UpperLimit / MaxCalcSteps;
          P1                                         := Value(Max);
          Inc( UpperLimit);
        until UpperLimit > MaxCalcSteps;
      end;
      Result := Results.NumberOfHits > 0;
    end;


    procedure   TBSpline.AddPoint( aVertex: TVertex);
    begin
      if PointCount = Capacity
      then Capacity := Capacity + VerticesIncrement;

      Inc( FPointCount);
      Point[ PointCount] := aVertex;
      Built              := False;
    end;


    constructor TBSpline.Create;
    begin
      inherited;
      FPointList   := nil;
      FVertexList  := nil;
      FKnuckleList := nil;
      FCapacity    := 0;
      Clear;
    end;


    procedure TBSpline.DeletePoint( anIndex: Integer);
    var
      i : Integer;
    begin
      if PointCount > 0
      then
      begin
        Dec( FPointCount);

        for i := anIndex to PointCount
        do begin
          FPointList^  [ i] := FPointList^  [ i + 1];
          FKnuckleList^[ i] := FKnuckleList^[ i + 1];
        end;

        Built := False;
      end;

      if PointCount = 0
      then Clear;
    end;


    destructor TBSpline.Destroy;
    begin
      Clear;
      inherited;
    end;


    procedure TBSpline.Draw( aCanvas: TCanvas; aColor, aDotColor: TColor; aDotSize: Integer);
    var
      i          : Integer;
      ParamValue : TDataType;
      V          : TVertex;
    begin
      aCanvas.Pen.Color := aColor;

      for i := 0 to Fragments                   // Draw the spline in 'Fragments' steps
      do begin
        ParamValue := i / Fragments;            // parameter value must be in the range 0.0-1.0
        V          := Value( ParamValue);

        // todo: BH speed up drawing
        // Used moveto / lineto method for demo-drawing, but using
        // the Canvas.polyline method is SIGNIFICANTLY FASTER!!

        if i = 0
        then aCanvas.MoveTo( Round( V.X), Round( V.Y))
        else aCanvas.LineTo( Round( V.X), Round( V.Y));
      end;

      if aDotSize > 0
      then begin
        for i := 1 to PointCount
        do begin
          V := Point[ i];
          aCanvas.Pen  .Color := aDotColor;
          aCanvas.Brush.Color := aDotColor;
          aCanvas.Ellipse( Round( V.X) - aDotSize, Round( V.Y) - aDotSize, Round( V.X) + aDotSize + 1, Round( V.Y) + aDotSize + 1);
        end;
      end;
    end;


    procedure TBSpline.Extents( var aMin, aMax: TVertex);
    var
      i : Integer;
      P : TVertex;
    begin
      for i := 1 to PointCount
      do begin
        P := Point[ i];

        if P.X < aMin.X
        then aMin.X := P.X;

        if P.X > aMax.X
        then aMax.X := P.X;

        if P.Y < aMin.Y
        then aMin.Y := P.Y;

        if P.Y > aMax.Y
        then aMax.Y := P.Y;
      end;
    end;


    procedure TBSpline.InsertPoint( anIndex: Integer; aVertex: TVertex);
    var
      i : Integer;
    begin
      if ( anIndex >= 0) and ( anIndex <= PointCount)
      then begin
        if PointCount = Capacity
        then Capacity := Capacity + VerticesIncrement;

        Inc( FPointCount);

        for i := PointCount downto anIndex + 1
        do begin
          FPointList^  [ i] := FPointList^  [ i - 1];
          FKnuckleList^[ i] := FKnuckleList^[ i - 1];
        end;

        FPointList^  [ anIndex] := aVertex;
        FKnuckleList^[ anIndex] := False;
        Built                   := False;
      end
      else raise ESplines.Create( 'Index out of range');
    end;


    procedure TBSpline.Invert;
    var
      OldPoints   : P2DPointList;
      OldVertices : P2DVertexList;
      OldKnuckle  : PKnuckleList;
      i           : Integer;
    begin
      Exit; // todo: BH :: Exit ??? WTF ??? Backup current data

      OldPoints    := FPointList;
      FPointList   := nil;
      OldKnuckle   := FKnuckleList;
      FKnuckleList := nil;
      OldVertices  := FVertexList;
      FVertexList  := nil;

      // Prepare new arrays
      GetMem( FPointList  , Capacity * SizeOf( TVertex));
      GetMem( FKnuckleList, Capacity);
      GetMem( FVertexList , ( Capacity + 2) * SizeOf( TVertex));

      // Initialize knuckle list
      FillChar( FKnuckleList^, Capacity, 0);

      // Copy controlpoints
      for i := 1 to PointCount
      do begin
        FPointList^  [ i] := OldPoints^ [ PointCount - i + 1];
        FKnuckleList^[ i] := OldKnuckle^[ PointCount - i + 1];
      end;

      // Copy vertices
      if Assigned( OldVertices) and ( FVertexCount <> 0) and Built
      then begin
        for i := 0 to VertexCount + 1
        do FVertexList^[ i] := OldVertices^[ FVertexCount - i + 1];
      end;

      // Destroy old arrays
      Freemem( OldPoints, Capacity * SizeOf( TVertex));

      if Assigned( OldVertices) and ( FVertexCount <> 0) and Built
      then Freemem( OldVertices, ( Capacity + 2) * SizeOf( TVertex));

      Freemem( OldKnuckle, Capacity);
    end;


    procedure TBSpline.Clear;
    begin
      if Assigned( FVertexList) and ( VertexCount > 0)
      then begin
        Freemem( FVertexList, ( FVertexCount + 2) * SizeOf( TVertex));
        FVertexList := nil;
      end;

      FPointCount   := 0;
      FVertexCount  := 0;
      Built         := False;
      Capacity      := 0;
      FInterpolated := False;
      FFragments    := 100;
    end;


    procedure TBSpline.PhantomPoints;
    var
      i : Integer;
    begin
      if VertexCount > 1
      then begin
        i := 0;
        FVertexList^[ i              ].X := 2 * FVertexList^[ i + 1      ].X - FVertexList^[ i + 2          ].X;
        FVertexList^[ i              ].Y := 2 * FVertexList^[ i + 1      ].Y - FVertexList^[ i + 2          ].Y;
        FVertexList^[ VertexCount + 1].X := 2 * FVertexList^[ VertexCount].X - FVertexList^[ VertexCount - 1].X;
        FVertexList^[ VertexCount + 1].Y := 2 * FVertexList^[ VertexCount].Y - FVertexList^[ VertexCount - 1].Y;
      end;
    end;


    procedure TBSpline.FillMatrix;
    var
      i : Integer;
      j : Integer;
    begin
      if ( FVertexCount > 2) and ( FVertexCount <= MaxInterpolatedVertices)
      then begin
        for i := 2 to FVertexCount - 1
        do begin
          Matrix[ i]^[ i - 1] := 1 / 6;
          Matrix[ i]^[ i    ] := 2 / 3;
          Matrix[ i]^[ i + 1] := 1 / 6;
        end;

        Matrix[ 1]^           [ 1           ] := 1;
        Matrix[ FVertexCount]^[ FVertexCount] := 1;
        i := 3;

        while i < FVertexCount - 1
        do begin
          if
            ( Abs( FVertexList^[ i].X - FVertexList^[ i - 1].X) < 1e-5) and ( Abs( FVertexList^[ i + 1].X - FVertexList^[ i].X) < 1e-5) and
            ( Abs( FVertexList^[ i].Y - FVertexList^[ i - 1].Y) < 1e-5) and ( Abs( FVertexList^[ i + 1].Y - FVertexList^[ i].Y) < 1e-5)
          then begin
            for j := i - 1 to i + 1
            do begin
              Matrix[ j]^[ j - 1] := 0;
              Matrix[ j]^[ j    ] := 1;
              Matrix[ j]^[ j + 1] := 0;
            end;

            Inc( i, 2);
          end
          else Inc( i);
        end;
      end;
    end;


    function TBSpline.Value( aParameter: Double): TVertex;
    var
      c    : Integer;
      S    : Integer;
      E    : Integer;
      Dist : Double;
      Mix  : Double;
      Mid  : TDataType;
    begin
      Result.X := 0;
      Result.Y := 0;

      if FPointCount < 2
      then Exit;

      if not Built and not Rebuild or not Assigned( FVertexList)
      then Exit;

      Mid := ( VertexCount - 1) * aParameter + 1;
      S   := Trunc( Mid - 1);

      if S < 0
      then S := 0;

      E := S + 3;

      if S > FVertexCount + 1
      then S := FVertexCount + 1;

      for c := S to E
      do begin
        Dist := Abs( c - Mid);

        if Dist < 2
        then begin
          if Dist < 1
          then Mix := 4 / 6 - Dist * Dist + 0.5 * Dist * Dist * Dist
          else Mix := (2 - Dist) * ( 2 - Dist) * ( 2 - Dist) / 6;

          Result.X := Result.X + FVertexList^[ c].X * Mix;
          Result.Y := Result.Y + FVertexList^[ c].Y * Mix;
        end;
      end;
    end;


    function TBSpline.FirstDerivative( aParameter: Double): TVertex;
    var
      c    : Integer;
      S    : Integer;
      E    : Integer;
      Dist : Double;
      Mix  : Double;
      Mid  : Double;
    begin
      Result.X := 0;
      Result.Y := 0;

      if FPointCount < 2
      then Exit;

      if not Built and not Rebuild
      then Exit;

      Mid := ( VertexCount - 1) * aParameter + 1;
      S   := Trunc( Mid - 1);

      if S < 0
      then S := 0;

      E := S + 3;

      if S > FVertexCount + 1
      then S   := FVertexCount + 1;

      for c := S to E
      do begin
        Dist := c - Mid;

        if ( Dist > - 2) and ( Dist <= - 1)
        then Mix := ( 2 + Dist) * ( 2 + Dist) * 0.5
        else if ( Dist >= - 1) and ( Dist <= 0)
        then Mix := (- 2 * Dist - 1.5 * Dist * Dist)
        else if ( Dist >= 0) and ( Dist <= 1)
        then Mix := ( - 2 * Dist + 1.5 * Dist * Dist)
        else if ( Dist >= 1) and ( Dist < 2)
        then Mix := - ( 2 - Dist) * ( 2 - Dist) * 0.5
        else Mix := 0;

        Result.X := Result.X + FVertexList^[ c].X * Mix;
        Result.Y := Result.Y + FVertexList^[ c].Y * Mix;
      end;
    end;


    function TBSpline.SecondDerivative( aParameter: Double): TVertex;
    var
      c    : Integer;
      S    : Integer;
      E    : Integer;
      Dist : Double;
      Mix  : Double;
      Mid  : Double;
    begin
      Result.X := 0;
      Result.Y := 0;

      if FPointCount < 2
      then Exit;

      if not Built and not Rebuild
      then Exit;

      Mid := ( VertexCount - 1) * aParameter + 1;
      S   := Trunc( Mid - 1);

      if S < 0
      then S := 0;

      E := S + 3;

      if S > FVertexCount + 1
      then S := FVertexCount + 1;

      for c := S to E
      do begin
        Dist := c - Mid;

        if ( Dist >= - 2) and ( Dist <= - 1)
        then Mix := 2 + Dist
        else if ( Dist >= - 1) and ( Dist <= 0)
        then Mix := - 2 - 3 * Dist
        else if (Dist >= 0) and (Dist <= 1)
        then Mix := - 2 + 3 * Dist
        else if ( Dist >= 1) and ( Dist <= 2)
        then Mix := 2 - Dist
        else Mix := 0;

        Result.X := Result.X + FVertexList^[ c].X * Mix;
        Result.Y := Result.Y + FVertexList^[ c].Y * Mix;
      end;
    end;



{ ========
  TSplines = class( TComponent) // Component to store all B-Splines in
  private
    FSplines : TList;
  private
}

    function TSplines.GetCount: Integer;
    begin
      Result := FSplines.Count;
    end;


    function TSplines.GetSpline( anIndex: Integer): TBSpline;
    begin
      Result := FSplines[ anIndex];
    end;


    procedure TSplines.SetSpline( anIndex: Integer; const aValue: TBSpline);
    begin
      FSplines[ anIndex] := aValue;
    end;


    constructor TSplines.Create( anOwner: TComponent);
    begin
      inherited;
      FSplines := TList.Create;
    end;


    destructor TSplines.Destroy;
    begin
      FreeAndNil( FSplines);
      inherited;
    end;


    procedure TSplines.AddSpline( const aBSpline: TBSpline);
    begin
      FSplines.Add( aBSpline);
    end;


    procedure TSplines.Clear;
    var
      i : Integer;
    begin
      // First destroy the splines
      for i := 1 to Count
      do Items[i - 1].Destroy;

      // then their reference in the list
      FSplines.Clear;
    end;


    procedure TSplines.InsertSpline( aPos : Integer; const aBSpline: TBSpline);
    begin
      FSplines.Insert( aPos, aBSpline);
    end;


    procedure TSplines.DeleteSpline( const aBSpline: TBSpline);
    var
      i : Integer;
    begin
      i := FSplines.IndexOf( aBSpline);

      if i >= 0
      then begin
        FSplines.Delete( i);
        FSplines.Pack;
      end;
    end;


end.
