unit knobs2013;

{

  (C) COPYRIGHT 1999 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

{$ifdef DEBUG}
//  {$define DEBUG_GENE}
{$endif}

// {$DEFINE USE_HASH}

interface

uses

  WinApi.Windows, WinApi.Messages, WinApi.MMSystem,

  System.SysUtils, System.Types, System.StrUtils, System.Math, System.Classes, System.Contnrs, System.UITypes,

  Vcl.Graphics, Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.ExtCtrls, Vcl.Buttons, Vcl.StdCtrls, Vcl.ComCtrls, Vcl.Themes,

  System.TypInfo, System.Generics.Collections, System.Rtti,

  Vcl.Clipbrd,

  KnobsUtils, ComplexMath, KnobsConversions, KnobsWorms, FrmScala, BSplines, {KnobsDesigner,} KnobsMaze, KnobsLightning;

const

  KNOBS_UM_CONNECTIONADD    = WM_USER +   0;  // Async messaging identifiers
  KNOBS_UM_CONNECTIONREMOVE = WM_USER +   1;
  KNOBS_UM_COLORCHANGE      = WM_USER +   2;
  KNOBS_UM_HIGHLIGHT        = WM_USER +   3;
  KNOBS_UM_USER             = WM_USER + 256;

  DO_RESTACK      = True; NO_RESTACK      = not DO_RESTACK;
  DO_DRAG         = True; NO_DRAG         = not DO_DRAG;
  DO_STRICT       = True; NO_STRICT       = not DO_STRICT;
  DO_COPY_MIDI    = True; NO_COPY_MIDI    = not DO_COPY_MIDI;
  DO_REDRAW_WIRES = True; NO_REDRAW_WIRES = not DO_REDRAW_WIRES;
  DO_NOTIFY       = True; NO_NOTIFY       = not DO_NOTIFY;
  DO_COPY_TUNING  = True; NO_COPY_TUNING  = not DO_COPY_TUNING;

  CL_LOCKED       = clRed;                  // Color for lock indication
  CL_FOCUS        = clRed;
  CL_RANDOMIZABLE = clAqua;
  CL_MIDI         = clBlue;
  CL_MIDI_FOCUS   = clFuchsia;
  CL_CURSOR       = clBlue;
  CL_DOT          = clBlue;
  CL_ACTIVEDOT    = clFuchsia;
  CL_GRIDDOT      = $009D6871;
  MAX_VARIATIONS  = 8;                      // Number of variations to be used for a control / module / patch

  MOD_X_UNIT      = 250;
  MOD_Y_UNIT      =  15;

  HISTORY_STR_LEN         =  100;           // Maximum string length for string based history items
  CONTROL_HISTORY_COUNT   =   10;           // Maximum number of history items in the control context menu
  MODULE_HISTORY_COUNT    =   15;           // Maximum number of hisstory items in the module  context menu
  EDITOR_HISTORY_COUNT    =   36;           // Maximum number of history items in the editor  context menu
  VIEWER_HISTORY_COUNT    = 1024;           // Maximum number of history items in the history viewer screen, !! <= DEFAULT_AUTOMTION_SIZE !!
  DEFAULT_AUTOMATION_SIZE = 8192;           // Default size of the automation history store

  RANGE_COUNT             = 1;              // Number of ranges present in the system
  RANGE_MARKER            = 0;              // Index into Ranges for the 'Marker' range
  NAME_MARKER             = 'Marker';       // Name for the 'Marker' range


var

  GLooksPath       : string  = '';          // Global path for the looks data


type

  TKnobsTimeBtnOptions = set of (
    tbFocusRect,
    tbAllowTimer
  );


  TKnobsVariationSet = set of 0 .. MAX_VARIATIONS - 1;


  TKnobsLedShape = (
    lsRound,
    lsSquare
  );


  TKnobsModuleFlag = (
    mfSelected ,
    mfCapturing
  );
  TKnobsModuleFlags = set of TKnobsModuleFlag;


  TKnobsConnectOp = (
    coConnect,
    coDisconnect
  );


  TKnobsReadMode = (
    rmAppend,                    // Adds the read mudules to an editor patch                             - leaves GUID as is
    rmReplace,                   // Removes the existing modules and replaces them with the read modules - changes GUID to the one read in from the patch
    rmParams                     // Pastes parameter from one patch to another one                       = leaves GUID as is
  );


  TKnobsWriteMode = (
    wmAll,
    wmSelected,
    wmParams
  );


  TKnobsDisplayType = (
    dtSingleFromLow,
    dtSingleFromHigh,
    dtFilledFromLow,
    dtFilledFromHigh
  );


  TKnobsDisplayOrientation = (
    doHorizontal,
    doVertical
  );


  TKnobsButtonsDisplayMode = (
    bdmHorizontal,
    bdmVertical
  );


  TKnobsDVStyle = (        // DataViewer design time styles
    dvsBlank,              // Paint nothing
    dvsSine,               // Paint a sine
    dvsTri,                // triangle
    dvsSaw,                // saw
    dvsSquare,             // pulse / square
    dvsNoise,              // noise / random
    dvsEnvAr,              // AR   type envelope
    dvsEnvAhd,             // AHD  type envelope
    dvsEnvAdsr,            // ADSR type envelope
    dvsXYScope             // XY scope display type
  );

  TKnobsDMShape = (        // DataMaker predefined wave shapes
    dmsClear,              // Clear to initial state
    dmsSine,               // Sine
    dmsCosine,             // Cosine
    dmsArcTan,             // Arc tangent
    dmsBell,               // Bell shaped curve
    dmsTri,                // Triangle wave
    dmsSaw,                // Sawtooth wave
    dmsSquare3,            // square  3% DC
    dmsSquare10,           // square 10% DC
    dmsSquare33,           // square 33% DC
    dmsSquare50,           // square 50% DC
    dmsSquare67,           // square 67% DC
    dmsSquare90,           // square 90% DC
    dmsSquare97,           // square 97% DC
    dmsNoise,              // Random, each time different
    dmsSwapLeftRight,      // Horizontal flip
    dmsSwapTopBottom,      // Vertical   flip
    dmsRotate180,          // Rotate 180 degrees
    dmsFixLoop,            // Make the first point match the last one - for now a first order approach (i.e. derivative may have a jump).
    dmsFixScale,           // Make the graph fill the entire Y range
    dmsSaveToFile,         // Save the graph to a file
    dmsLoadFromFile        // Load the graph from a file
  );


  TKnobsDataMakerType = (  // DtaMker polarity type
    dmtBipolar ,
    dmtUniPolar
  );


  TKnobsDMGraphMode = (    // DataMaker graph mode
    dmgLinear  ,
    dmgSplines ,
    dmgSteps
  );


  TKnobsPairingMode = (    // Pairing mode for paired knobs
    pmOff,                 // Pairing is off
    pmNormal,              // Pairing is normal, both knobs have the same position
    pmMirrored             // Pairing is anti normal, knobs have mirrored positions
  );


  TKnobsControlSrc = (     // Control source for a knob change
    kcsUser,               // by user
    kcsAutomation          // by automation
  );


  EKnobs                 = class( Exception);
  EKnobsWarning          = class( EKnobs);
  EKnobsCorruptPatch     = class( EKnobs);
  EKnobsNoDuplicates     = class( EKnobs);
  EKnobsShortCircuit     = class( EKnobs);
  EKnobsDuplicateItem    = class( EKnobs);
  EKnobsHistoryTypeError = class( EKnobs);


  TKnobsBezierPoints = array[ 0 .. 3] of TPoint;
  TKnobsBezierParams = record
    Angle1    : Integer;
    Angle2    : Integer;
    Strength1 : Integer;
    Strength2 : Integer;
  end;


  TKnobsVariationValues      =          array[ 0 .. MAX_VARIATIONS - 1] of TSignal;
  TKnobsMultiVariationValues = array of array[ 0 .. MAX_VARIATIONS - 1] of TSignal;
  PKnobsEditData             = ^TKnobsEditData;
  TKnobsEditHistory          = class;
  TKnobsValuedControl        = class;
  TKnobsSelector             = class;
  TKnobsConnector            = class;
  TKnobsDisplay              = class;
  TKnobsEditLabel            = class;
  TKnobsCustomModule         = class;
  TKnobsModule               = class;
  TKnobsWirePanel            = class;
  TKnobsModuleSelector       = class;
  TKnobsModuleButton         = class;
  TKnobsModuleClass          = class of TKnobsCustomModule;
  TKnobsValuedControlClass   = class of TKnobsValuedControl;
  TStringArray               = TArray<string>;


  TKnobsOnValueChange        = procedure( const aSender: TObject; const aPath, aControlType: string; aValue: TSignal; IsFinal, IsAutomation: Boolean) of object;
  TKnobsKnobPositionChanged  = procedure( const aSender: TKnobsValuedControl; const aPath, aControlType: string; aPosition: Integer)                  of object;
  TKnobsOnTextChanged        = procedure( const aSender: TObject; const aPath, aValue: string)                                                        of object;
  TKnobsOnPointChanged       = procedure( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: TSignal)                           of object;
  TKnobsOnValuedCtrlRemoved  = procedure( const aSender: TKnobsValuedControl)                                                                         of object;
  TKnobsOnModuleButtonClick  = procedure( const aSender: TObject; aModuleclass: TKnobsModuleType)                                                     of object;
  TKnobsOnGetGlyph           = function ( const aSender: TObject; const aBitmapsFolder, aName: string): TBitmap                                       of object;
  TKnobsOnHistoryChange      = procedure( const aSender: TObject; anUndoCount, aRedoCount, aSavedMarker: Integer)                                     of object;
  TKnobsOnRecompile          = procedure( const aSender: TKnobsWirePanel; ModulesChanged: Boolean)                                                    of object;
  TKnobsOnWireAdded          = procedure( const aSender: TObject; const aSrcModule, aSrcConnector, aDstModule, aDstConnector: string)                 of object;
  TKnobsOnWireRemoved        = procedure( const aSender: TObject; const aSrcModule, aSrcConnector, aDstModule, aDstConnector: string)                 of object;
  TKnobsOnEditLabelChanged   = procedure( const aSender: TObject; const aValue: string)                                                               of object;
  TOnCreateModuleBitmap      = function ( const aModuleType: TKnobsModuleType; UseCache: Boolean): TBitmap                                            of object;
  TOnReadModuleComment       = function ( const aModuleType: TKnobsModuleType): string                                                                of object;
  TOnShowPopup               = procedure( const aSender: TKnobsWirePanel; const aControl: TControl)                                                   of object;
  TOnUnfocus                 = procedure( const aSender: TObject)                                                                                     of object;
  TKnobsOnVisitControl       = function ( const aControl: TControl; aControlClass: TControlClass; const aUserData: TObject): Boolean                  of object;
  TOnGetTabColor             = function ( const aSender: TObject; const aPageName: string; aMixColor: TColor; aMix: Byte): TColor                     of object;
  TOnGetModuleColor          = function ( const aSender: TObject; const aModuleType: TKnobsModuleType): TColor                                        of object;
  TOnPopupEditorShow         = procedure( const aSender: TObject)                                                                                     of object;
  TOnPopupEditorHide         = procedure( const aSender: TObject)                                                                                     of object;
  TOnLoadDataGraph           = function ( const aSender: TObject): string                                                                             of object;
  TOnSaveDataGraph           = procedure( const aSender: TObject; const aValue: string)                                                               of object;
  TOnLoadGridControl         = function ( const aSender: TObject): string                                                                             of object;
  TOnSaveGridControl         = procedure( const aSender: TObject; const aValue: string)                                                               of object;
  TOnActiveVariationChanged  = procedure( const aSender: TObject; aVariation: Integer)                                                                of object;
  TOnModulesRenamed          = procedure( const aSender: TObject; const anOldNames, aNewNames: TStringArray)                                          of object;
  TOnAddEditData             = procedure( const aSender: TKnobsValuedControl; const aData: PKnobsEditData)                                            of object;
  TOnKnobsGridClick          = procedure( const aSender: TObject; aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer)                      of object;
  TOnKnobsGridMouseMove      = procedure( const aSender: TObject; aShift: TShiftState; anX, anY: Integer)                                             of object;
  TOnKnobsRightClick         = procedure( const aSender: TObject)                                                                                     of object;
  TOnLoadCaptions            = procedure
                               (
                                 const aSender     : TObject;
                                 const aModule     : TKnobsCustomModule;
                                 const aControl    : TKnobsSelector;
                                 const aDependency : TKnobsValuedControl
                               ) of object;


  TKnobsUMConnectionChange = record
    Msg         : Cardinal;
    Source      : TKnobsConnector;
    Destination : TKnobsConnector;
    Result      : LRESULT;
  end;
  TKnobsUMConnectionAdd   = TKnobsUMConnectionChange;
  TKnobUMConnectionRemove = TKnobsUMConnectionChange;


  TKnobsUMColorChange = record
    Msg    : Cardinal;
    Source : TKnobsConnector;
    Color  : TColor;
    Result : LRESULT;
  end;


  TKnobsUMHighlight = record
    Msg         : Cardinal;
    Source      : TKnobsConnector;
    NotUsed2    : LPARAM;
    Result      : LRESULT;
  end;


  TKnobsUMButtonUp = record
    Msg         : Cardinal;
    Source      : TKnobsModuleButton;
    NotUsed     : LPARAM;
    Result      : LRESULT;
  end;


  IKnobsPatchReader = interface ['{2D211C63-38F4-4779-9556-188B3A08DA8F}']
  // Prototype patch reader

    function    GetWarnOnStructMismatch: Boolean;
    procedure   SetWarnOnStructMismatch( aValue: Boolean);
    function    GetTuningChanged: Boolean;

    procedure   ReadParams(
      const aData      : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    );

    procedure   ReadString(
      const aData      : string;
      const aWirePanel : TKnobsWirePanel;
      aDistance        : TPoint;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode;
      const aFileName  : string
    );

    procedure   ReadFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    );

    procedure   ImportFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    );

    procedure   ReadFromClipboard(
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    );

    procedure   ParamsFromFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    );

    procedure   ParamsFromClipboard(
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    );

    property WarnOnStructMismatch : Boolean read GetWarnOnStructMismatch write SetWarnOnStructMismatch;
    property TuningChanged        : Boolean read GetTuningChanged;
  end;


  IKnobsPatchWriter = interface ['{8BC9E523-A979-480B-B123-38894DB7B424}']
  // Prototype patch writer

    function    WriteString(
      const aWirePanel : TKnobsWirePanel;
      const aFileName  : string = '';
      aWriteMode       : TKnobsWriteMode = wmAll
    ): string;

    procedure   WriteFile(
      const aWirePanel : TKnobsWirePanel;
      const aFileName  : string;
      aWriteMode       : TKnobsWriteMode = wmAll
    );

    procedure   WriteToClipboard(
      const aWirePanel : TKnobsWirePanel;
      aWriteMode       : TKnobsWriteMode = wmAll
    );
  end;


  IKnobsAutomatable = interface ['{4E93C29C-B93C-435C-A440-0F8ED1F3E33B}']
    function    GetAutomationName: string;
    function    GetAllowAutomation: Boolean;
    procedure   SetAllowAutomation( aValue: Boolean);
    function    GetAllowRecording: Boolean;
    procedure   SetAllowRecording( aValue: Boolean);
    function    GetAssignedMIDICC: Byte;
    procedure   SetAssignedMIDICC( aValue: Byte);
    function    GetStepCount : Integer;
    procedure   SetAutomationPosition( aValue: Integer);
    procedure   SetPlayPosition      ( aValue: Integer);
    procedure   SetScaledPosition    ( anIndex: Integer; aValue: TSignal);

    property    AutomationName                    : string  read GetAutomationName;
    property    AllowAutomation                   : Boolean read GetAllowAutomation write SetAllowAutomation;
    property    AllowRecording                    : Boolean read GetAllowRecording  write SetAllowRecording;
    property    AssignedMIDICC                    : Byte    read GetAssignedMIDICC  write SetAssignedMIDICC;
    property    StepCount                         : Integer read GetStepCount;
    property    AutomationPosition                : Integer                         write SetAutomationPosition;
    property    PlayPosition                      : Integer                         write SetPlayPosition;
    property    ScaledPosition[ anIndex: Integer] : TSignal                         write SetScaledPosition;
  end;


  IKnobsVariation = interface ['{30856A2C-9160-49E4-A8F5-0F636AD6794B}']
    function    GetVariationCount: Integer;
    function    GetAllowVariations: Boolean;
    function    GetActiveVariation: Integer;
    procedure   SetActiveVariation( aValue: Integer);
    function    GetVariationsAsString: string;
    procedure   SetVariationsAsString( const aValue: string);
    function    GetVariationValue( anIndex: Integer): TSignal;
    procedure   SetVariationValue( anIndex: Integer; aValue: TSignal);
    function    KnobValueToVariationValue( aValue: TSignal): TSignal;
    function    VariationValueToKnobValue( aValue: TSignal): TSignal;
    procedure   SetDefaultVariations;
    procedure   CountGene( var aCount: Integer);
    procedure   FillGene  ( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    procedure   AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
    procedure   SetRandomValue( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   SetRandomValue( anAmount: TSignal);                                                            overload;
    procedure   Randomize     ( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   Randomize     ( anAmount: TSignal);                                                            overload;
    procedure   Mutate        ( aProb, aRange: TSignal; aVariation: Integer);                                  overload;
    procedure   Mutate        ( aProb, aRange: TSignal);                                                       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer);       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);                   overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                           overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom: Integer);                                       overload;
    procedure   LiveMorph     ( anAmount: TSignal);
    function    CanAllowRandomization: Boolean;
    function    CanRandomize: Boolean;
    function    GetAllowRandomization: Boolean;
    procedure   SetAllowRandomization( aValue: Boolean);
    function    GetShowAllowRandomization: Boolean;
    procedure   SetShowAllowRandomization( aValue: Boolean);
    function    GetControlType: string;
    procedure   SetControlType( const aValue: string);

    property    VariationCount                    : Integer read GetVariationCount;
    property    AllowVariations                   : Boolean read GetAllowVariations;
    property    ActiveVariation                   : Integer read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string  read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal read GetVariationValue         write SetVariationValue;
    property    AllowRandomization                : Boolean read GetAllowRandomization     write SetAllowRandomization;
    property    ShowAllowRandomization            : Boolean read GetShowAllowRandomization write SetShowAllowRandomization;
    property    ControlType                       : string  read GetControlType            write SetControlType;
  end;


  TKnobsEditKind = (
    wekNothing ,
    wekKnob    ,
    wekSelector
  );


  TKnobsEditData = record
  private
    FAssociate : TKnobsValuedControl;
    FSelected  : Boolean;
    FVariation : Integer;
    FTimeStamp : TKnobsTimeStamp;
    FKind      : TKnobsEditKind;
    FPath      : string;
    FCType     : string;
    FPrevValue : TValue;
    FValue     : TValue;
    FPrevPos   : Integer;
    FPosition  : Integer;
    FSteps     : Integer;
  public
    procedure   Create(
      const anAssociate : TKnobsValuedControl;
      const aTimeStamp  : TKnobsTimeStamp;
      const aKind       : TKnobsEditKind;
      const aPath       : string
    );
    function    CreateFromString( const anEditor: TKnobsWirePanel; const aStringValue: string): Boolean;
    function    AsString: string;
    function    AsFileString: string;
    procedure   Select( DoSelect: Boolean);
    procedure   UpdateNames( const anOldFriendlyNames, aNewFriendlyNames: TStringArray);
  public
    function    GetAssociate    : TKnobsValuedControl;
  end;


  IKnobsUndoable = interface ['{5BE404E5-BA04-4A18-970A-53EB131F3F61}']
    procedure   Clear;
    procedure   AddEditFor( const aTimeStamp: TKnobsTimeStamp; const aPrefix: string; const anAutomationData: TKnobsEditHistory);
    procedure   AddEdit( const aTimeStamp: TKnobsTimeStamp);
    procedure   FillStrings( const aList: TStrings; FileFill: Boolean; MaxItems: Integer);
    function    CreateStringList( aMaxItems: Integer): TStringList;
    procedure   UndoItem( anIndex: Integer);
    procedure   PlayItem( anIndex: Integer);
    function    Last: TKnobsEditData;
    function    FreeCount: Integer;
    function    FillCount: Integer;
    procedure   SetSize( aValue: Integer);
    function    GetSize  : Integer;
    procedure   SetAllowUnchanged( aValue: Boolean);
    function    GetAllowUnchanged: Boolean;
    procedure   SaveToStringList  ( const aStringList: TStringList; const aGuid: string);
    procedure   LoadFromStringList( const anEditor: TKnobsWirePanel; const aStringList: TStringList; const aGuid: string);
    procedure   CopyFrom( const aHistory: TKnobsEditHistory);
    procedure   DeselectAll;
    procedure   SelectItem    ( anIndex: Integer);
    function    IsItemSelected( anIndex: Integer): Boolean;
    procedure   UndoSelected;

    property    Size           : Integer read GetSize           write SetSize;
    property    AllowUnchanged : Boolean read GetAllowUnchanged write SetAllowUnchanged; // When set, allows for 'unchanged value events' to be recorded
  end;


  TKnobsComponent = class
  private
    FComponent      : TComponent;
    FComponentClass : TComponentClass;
  public
    constructor Create( const aComponent: TComponent);
  end;


  TKnobsComponentData = class( TDictionary< string, TKnobsComponent>)
  protected
    procedure   ValueNotify( const aValue: TKnobsComponent; anAction: TCollectionNotification);                override;
  public
    procedure   InsertComponent( const aComponent: TComponent);
    procedure   RemoveComponent( const aComponent: TComponent);
    function    Lookup( const aKey: string; const aClass: TComponentClass): TKnobsComponent;
  end;


  TKnobsGraphicControl = class( TGraphicControl)
  private
    function    GetModule: TKnobsCustomModule;
    function    GetModuleName   : string;
  public
    property    Module        : TKnobsCustomModule  read GetModule;
    property    ModuleName    : string              read GetModuleName;
  end;


  TKnobsData = class( TGraphicControl)
  private
    FShortName : string;
    FSize      : Integer;
    FData      : TSignalArray;
    FComment   : string;
    FStrVal    : string;
    FOnChanged : TKnobsOnTextChanged;
  private
    function    CheckChanged( const aNewValue: TSignalArray): Boolean;
    procedure   Changed;
    procedure   SetSize( aValue: Integer);
    function    GetData: TSignalArray;
    procedure   SetData( const aValue: TSignalArray);
    function    GetAsString: string;
    procedure   SetAsString( const aValue: string);
    function    GetModule: TKnobsCustomModule;
  protected
    procedure   TextChanged( const aNewValue: string; DoNotify: Boolean);                                       virtual;
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   WMNCHitTest( var aMessage: TWMNCHitTest);                                          message WM_NCHITTEST;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( aOwner: TComponent);                                                                   override;
    destructor  Destroy;                                                                                       override;
  public
    property    ShortName : string              read FShortName;
    property    Data      : TSignalArray        read GetData     write SetData;
    property    AsString  : string              read GetAsString write SetAsString;
    property    Module    : TKnobsCustomModule  read GetModule;
  published
    property    OnChanged : TKnobsOnTextChanged read FOnChanged  write FOnChanged;
    property    Size      : Integer             read FSize       write SetSize;
    property    Comment   : string              read FComment    write FComment;
    property    Width                                                                                        default 12;
    property    Height                                                                                       default 12;
  end;


  TKnobsLed = class( TKnobsGraphicControl)
  // An LED with one shot mode
  private
    FTime        : Cardinal;
    FTimer       : TTimer;
    FColorBorder : TColor;
    FColorOff    : TColor;
    FColorOn     : TColor;
    FHighLight   : Boolean;
    FActive      : Boolean;
    FShape       : TKnobsLedShape;
  private
    procedure   TimerFired( aSender: TObject);
    procedure   StartTimer( aValue: Cardinal);
    procedure   StopTimer;
    procedure   SetTime       ( aValue: Cardinal      );
    procedure   SetActive     ( aValue: Boolean       );
    procedure   SetColorBorder( aValue: TColor        );
    procedure   SetColorOn    ( aValue: TColor        );
    procedure   SetColorOff   ( aValue: TColor        );
    procedure   SetHighLight  ( aValue: Boolean       );
    procedure   SetShape      ( aValue: TKnobsLedShape);
  protected
    procedure   Paint;                                                                                         override;
  public
    constructor Create( aOwner: TComponent);                                                                   override;
    destructor  Destroy;                                                                                       override;
    procedure   Toggle;
  published
    property    Time        : Cardinal       read FTime        write SetTime        default 0;
    property    Active      : Boolean        read FActive      write SetActive      default True;
    property    ColorBorder : TColor         read FColorBorder write SetColorBorder default clGray;
    property    ColorOn     : TColor         read FColorOn     write SetColorOn     default clLime;
    property    ColorOff    : TColor         read FColorOff    write SetColorOff    default clGreen;
    property    HighLight   : Boolean        read FHighLight   write SetHighLight   default True;
    property    Shape       : TKnobsLedShape read FShape       write SetShape       default lsRound;
    property    Height                                                              default 15;
    property    Width                                                               default 15;
    property    Hint;
    property    ShowHint                                                            default False;
    property    ParentShowHint                                                      default False;
    property    Dragcursor;
    property    Dragmode;
    property    Visible;
    property    OnDragDrop;
    property    OnDragOver;
    property    OnEndDrag;
    property    OnMouseDown;
    property    OnMouseMove;
    property    OnMouseUp;
    property    ParentColor                                                         default False;
    property    OnClick;
  end;


  TKnobsClicker = class( TSpeedButton)
  // Repeating speed button, semi flat ... used as a sub element in knobs and sliders
  private
    FRepeatTimer : TTimer;
    FRepeatCount : Integer;
    FCanRepeat   : Boolean;
    FMouseDown   : Boolean;
    FShowFocus   : Boolean;
    FBitmap      : TBitmap;
  private
    procedure   TimerExpired( aSender: TObject);
    procedure   SetShowFocus( aValue: Boolean);
  private
    function    FindWirePanel: TKnobsWirePanel;
  protected
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   AssignBitmap( aBitmap: TBitmap);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   Click;                                                                                         override;
  private
    property    ShowFocus: Boolean read FShowFocus write SetShowFocus;
  public
    property    CanRepeat: Boolean read FCanRepeat write FCanRepeat default True;
  published
    property    StyleElements                                       default [];
  end;


  TKnobsEditHistory = class( TInterfacedObject, IKnobsUndoable)
  private
    FAssociate      : TKnobsValuedControl;
    FData           : TArray<TKnobsEditData>;
    FSize           : Integer;
    FAllowUnchanged : Boolean;
    FmaxStrLen      : Integer;
    FFreeCount      : Integer;
    FFillCount      : Integer;
    FInp            : Integer;
    FOutp           : Integer;
    FAllowNonAuto   : Boolean;
    FRecordAll      : Boolean;
  private
    function    GetSize       : Integer;
    procedure   SetSize       ( aValue: Integer);
    procedure   SetAllowUnchanged( aValue: Boolean);
    function    GetAllowUnchanged: Boolean;
    function    GetItem( anIndex: Integer): TKnobsEditData;
    procedure   SetItem( anIndex: Integer; const aValue: TKnobsEditData);
    function    GetData( anIndex: Integer): PKnobsEditData;
  public
    constructor Create(
      const anAssociate : TKnobsValuedControl;
      aHistorySize      : Integer;
      aMaxStringLength  : Integer;
      DoAllowUnchanged  : Boolean
    );                                                                                                         overload;
    constructor Create( const aSource: TKnobsEditHistory);                                                     overload;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   AddData( const aData: TKnobsEditData);
    procedure   AddEditFor( const aTimeStamp: TKnobsTimeStamp; const aPrefix: string; const anAutomationData: TKnobsEditHistory);
    procedure   AddEdit( const aTimeStamp: TKnobsTimeStamp);
    procedure   FillStrings( const aList: TStrings; FileFill: Boolean; aMaxItems: Integer);
    function    CreateStringList( aMaxItems: Integer): TStringList;
    procedure   UndoItem( anIndex: Integer);
    procedure   PlayItem( anIndex: Integer);
    function    Last: TKnobsEditData;
    function    FreeCount: Integer;
    function    FillCount: Integer;
    procedure   SaveToStringList  ( const aStringList: TStringList; const aGuid: string);
    procedure   LoadFromStringList( const anEditor: TKnobsWirePanel; const aStringList: TStringList; const aGuid: string);
    procedure   CopyFrom( const aHistory: TKnobsEditHistory);
    procedure   DeselectAll;
    procedure   SelectItem    ( anIndex: Integer);
    function    IsItemSelected( anIndex: Integer): Boolean;
    procedure   UndoSelected;
    procedure   UpdateNames( const anOldFriendlyNames, aNewFriendlyNames: TStringArray);
    function    CanRecord( aControl: TKnobsValuedControl): Boolean;
    procedure   RemoveControl( const aControl: TKnobsValuedControl);
  public
    property    AllowNonAuto            : Boolean        read FAllowNonAuto     write FAllowNonAuto;
    property    RecordAll               : Boolean        read FRecordAll        write FRecordAll;
    property    Size                    : Integer        read GetSize           write SetSize;
    property    Item[ anIndex: Integer] : TKnobsEditData read GetItem           write SetItem;                  default;
    property    Data[ anIndex: Integer] : PKnobsEditData read GetData;
    property    AllowUnchanged          : Boolean        read GetAllowUnchanged write SetAllowUnchanged;
  end;


  TKnobsRangeSpec = record
    Name      : string;
    IsMorph   : Boolean;
    Value     : TSignal;
  end;

  TKnobsRangeSpecs = TArray<TKnobsRangeSpec>;
  PKnobsRangeSpecs = ^ TKnobsRangeSpecs;


  TKnobsRange = record
    LowValue  : TSignal;
    HighValue : TSignal;
  end;


  TKnobsRangeData = TArray<TKnobsRange>;


  TKnobsRanges = class
  private
    FData        : TKnobsRangeData;
    FSpecs       : PKnobsRangeSpecs;
    FActiveRange : Integer;
  private
    function    GetHasActiveRange : Boolean;
    function    GetCount          : Integer;
    function    GetLowRangeAsStr  : string;
    procedure   SetLowRangeAsStr  ( const aValue: string);
    function    GetHighRangeAsStr : string;
    procedure   SetHighRangeAsStr ( const aValue: string);
    function    GetRange      ( anIndex: Integer): TKnobsRange;
    procedure   SetRange      ( anIndex: Integer; const aValue: TKnobsRange);
    function    GetName       ( anIndex: Integer): string;
    function    GetIsMorph    ( anIndex: Integer): Boolean;
    function    GetValue      ( anIndex: Integer): TSignal;
    function    GetLowValue   ( anIndex: Integer): TSignal;
    procedure   SetLowValue   ( anIndex: Integer; aValue: TSignal);
    function    GetHighValue  ( anIndex: Integer): TSignal;
    procedure   SetHighValue  ( anIndex: Integer; aValue: TSignal);
    function    GetHasRangeFor( anIndex: Integer): Boolean;
  public
    constructor Create( const aSpecs: PKnobsRangeSpecs);
    procedure   Clear;
    procedure   CopyFrom( const aValue: TKnobsRanges);
  public
    property    HasActiveRange                 : Boolean     read GetHasActiveRange;
    property    ActiveRange                    : Integer     read FActiveRange      write FActiveRange;
    property    Count                          : Integer     read GetCount;
    property    LowRangeAsStr                  : string      read GetLowRangeAsStr  write SetLowRangeAsStr;
    property    HighRangeAsStr                 : string      read GetHighRangeAsStr write SetHighRangeAsStr;
    property    Range      [ anIndex: Integer] : TKnobsRange read GetRange          write SetRange;
    property    Name       [ anIndex: Integer] : string      read GetName;
    property    IsMorph    [ anIndex: Integer] : Boolean     read GetIsMorph;
    property    Value      [ anIndex: Integer] : TSignal     read GetValue;
    property    LowValue   [ anIndex: Integer] : TSignal     read GetLowValue       write SetLowValue;
    property    HighValue  [ anIndex: Integer] : TSignal     read GetHighValue      write SetHighValue;
    property    HasRangeFor[ anIndex: Integer] : Boolean     read GetHasRangeFor;
  end;


  TKnobsValuedControl = class( TCustomControl, IKnobsAutomatable, IKnobsVariation, IKnobsUndoable)
  // Base type for valued controls, ones that can have an effect on an external model
  private
    FEditHistory            : TKnobsEditHistory;          // A small local undo history
    FDisplay                : TKnobsDisplay;
    FShortName              : string;
    FControlSrc             : TKnobsControlSrc;           // User or automation
    FMIDIColor              : TColor;                     // Accent color when under MIDI control
    FFocusColor             : TColor;                     // Accent color when the control has keyboard and mouse(wheel) focus
    FMIDIFocusColor         : TColor;                     // Accent color when under MIDI control and having focus
    FFriendlyName           : string;
    F_DesignerRemarks       : string;                     // Remarks to be used at design time - informative only
    FDynamicStepCount       : Boolean;                    // True when StepCount is not set at design time but at run time (by some magic)
    FStepCount              : Integer;                    // Number of steps in the total knob travel
    FLocked                 : Boolean;
    FPrevPosition           : Integer;                    // Previous knob position
    FKnobPosition           : Integer;                    // Current  knob position
    FDefaultPosition        : Integer;                    // default  knob position;
    FControlType            : string;                     // The control's behaviour
    FDezippered             : Boolean;                    // Whether dezippering should be used
    FAllowAutomation        : Boolean;
    FIsIndicator            : Boolean;                    // When true the value is set from an external source
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FAllowRecording         : Boolean;
    FShowAllowRecording     : Boolean;
    FRanges                 : TKnobsRanges;
    FAssignedMIDICC         : Byte;
    FActiveVariation        : Integer;
    FVariationValues        : TKnobsVariationValues;      // Values for variations
    FCurrentValue           : TSignal;                    // Current variation value, after variation selection,
    FLiveMorph              : TSignal;                    // Current LiveMorph position
    FShowSimpleHint         : Boolean;
    FOnValueChanged         : TKnobsOnValueChange;        // Called with the new value
    FOnKnobPositionChanged  : TKnobsKnobPositionChanged;  // Called with the new position
    FPopupCount             : Integer;
    FPopupWindow            : THintWindow;
    FWheelSupport           : Boolean;
    FWheelSensitivity       : Integer;
    FWheelCounter           : Integer;
    FOnUnFocus              : TOnUnFocus;
  private
    function    GetModule: TKnobsCustomModule;
  private
    function    GetModuleName   : string;
    function    GetModuleTitle  : string;
    function    GetLowRangeAsStr  : string;
    procedure   SetLowRangeAsStr  ( const aValue: string);
    function    GetHighRangeAsStr : string;
    procedure   SetHighRangeAsStr ( const aValue: string);
  private
    procedure   FinalizeKnobPosition;
  private
    function    GetAutomationName        : string;
    procedure   SetMIDIColor             ( aValue: TColor);
    procedure   SetFocusColor            ( aValue: TColor);
    procedure   SetMIDIFocusColor        ( aValue: TColor);
    procedure   SetKnobPosition          ( aValue: Integer);                                                    virtual;
    procedure   SetAutomationPosition    ( aValue: Integer);
    procedure   SetPlayPosition          ( aValue: Integer);
    procedure   SetScaledPosition        ( anIndex: Integer; aValue: TSignal);
    procedure   SetDefaultPosition       ( aValue: Integer);                                                    virtual;
    function    GetStepCount             : Integer;
    procedure   SetStepCount             ( aValue: Integer);                                                    virtual;
    function    GetControlType           : string;
    procedure   SetFriendlyName          ( const aValue: string);
    procedure   SetControlType           ( const aValue: string);                                               virtual;
    function    GetDynamicHint           : string;                                                              virtual;
    function    GetAsHint                : string;
    function    GetAsPrevDisplay         : string;
    function    GetAsPrevValue           : TSignal;
    function    GetAsDisplay             : string;
    function    GetAsValue               : TSignal;
    function    GetControlName           : string;
    procedure   SetOnValueChanged        ( aValue: TKnobsOnValueChange);
    function    GetAllowAutomation       : Boolean;
    procedure   SetAllowAutomation       ( aValue: Boolean);
    function    GetAllowRandomization    : Boolean;
    procedure   SetAllowRandomization    ( aValue: Boolean);
    function    GetShowAllowRandomization: Boolean;
    procedure   SetShowAllowRandomization( aValue: Boolean);
    procedure   SetIsIndicator           ( aValue: Boolean);
    function    GetAllowRecording        : Boolean;
    procedure   SetAllowRecording        ( aValue: Boolean);
    function    GetShowAllowRecording    : Boolean;
    procedure   SetShowAllowRecording    ( aValue: Boolean);
    function    GetAssignedMIDICC        : Byte;
    procedure   SetAssignedMIDICC        ( aValue: Byte);
    procedure   SetDynamicStepCount      ( aValue: Boolean);
    procedure   SetDisplay               ( aValue: TKnobsDisplay);
    procedure   SetShowSimpleHint        ( aValue: Boolean);
    procedure   NotifyPeers;                                                                                    virtual;
    procedure   DoWheel( aSender: TObject; aShift: TShiftState; aWheelDelta: Integer; aMousePos: TPoint; var Handled: Boolean);
    function    GetRangeCount: Integer;
    function    GetActiveRange: Integer;
    procedure   SetActiveRange( aValue: Integer);
    procedure   SetRangeValue( aValue: TSignal);
    function    GetVariationCount: Integer;
    function    GetAllowVariations: Boolean;
    function    GetActiveVariation: Integer;
    procedure   SetActiveVariation( aValue: Integer);
    function    GetVariationsAsString: string;
    procedure   SetVariationsAsString( const aValue: string);
    function    GetVariationValue( anIndex: Integer): TSignal;
    procedure   SetVariationValue( anIndex: Integer; aValue: TSignal);
    procedure   SetCurrentValue( aValue: TSignal);
    function    KnobValueToVariationValue( aValue: TSignal): TSignal;
    function    VariationValueToKnobValue( aValue: TSignal): TSignal;
    function    VariationValueToStr( aValue: TSignal): string;
    procedure   SetDefaultVariations;
    procedure   CountGene( var aCount: Integer);
    procedure   FillGene  ( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    procedure   AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
    function    ScaleToRange  ( aValue: TSignal; anIndex: Integer): TSignal;
    function    ScaleToMarkers( aValue: TSignal): TSignal;
    function    ClipToMarkers ( aValue: TSignal): TSignal;
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   SetLocked( aValue: Boolean);                                                                    virtual;
    procedure   FixParam;
    procedure   ValueChanged( IsFinal: Boolean);                                                                virtual;
    function    FindModule   : TKnobsCustomModule;
    function    FindWirePanel: TKnobsWirePanel;
    procedure   ActivatePopup;
    procedure   ShowPopup;
    procedure   HidePopup;
    procedure   ForceHidePopup;
    procedure   HandleWheelUp  ( anAmount: Integer; aShift: TShiftState);                                       virtual;
    procedure   HandleWheelDown( anAmount: Integer; aShift: TShiftState);                                       virtual;
    procedure   UnFocus;                                                                                        virtual;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   FixBitmaps;                                                                                     virtual;
    procedure   SetDefault;
    procedure   FixDisplay;
    procedure   Dump( var aFile: TextFile; anInd: Integer);
    procedure   BeginStateChange;
    procedure   EndStateChange;
    procedure   AddToHistory;
    procedure   CopyRangesFrom( const aControl: TKnobsValuedControl);
  public
    procedure   FixLiveMorph;                                                                                      virtual;
    procedure   SetRandomValue( anAmount: TSignal; aVariation: Integer);                                 overload; virtual;
    procedure   SetRandomValue( anAmount: TSignal);                                                      overload; virtual;
    procedure   Randomize     ( anAmount: TSignal; aVariation: Integer);                                 overload; virtual;
    procedure   Randomize     ( anAmount: TSignal);                                                      overload; virtual;
    procedure   Mutate        ( aProb, aRange: TSignal; aVariation: Integer);                            overload; virtual;
    procedure   Mutate        ( aProb, aRange: TSignal);                                                 overload; virtual;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); overload; virtual;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);             overload; virtual;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                     overload; virtual;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom: Integer);                                 overload; virtual;
    procedure   LiveMorph     ( anAmount: TSignal);                                                                virtual;
    function    CanAllowRandomization: Boolean;
    function    CanRandomize: Boolean;
    procedure   FixActiveRange  ( aValue: Integer);
    procedure   SetValueForRange( aRange: Integer; aValue: TSignal);
  public
    property    ModuleName                        : string            read GetModuleName;
    property    ModuleTitle                       : string            read GetModuleTitle;
    property    LowRangeAsStr                     : string            read GetLowRangeAsStr          write SetLowRangeAsStr;
    property    HighRangeAsStr                    : string            read GetHighRangeAsStr         write SetHighRangeAsStr;
  public
    property    EditHistory                       : TKnobsEditHistory read FEditHistory implements IKnobsUndoable;
  public
    property    ShortName                         : string            read FShortName;
    property    AsPrevDisplay                     : string            read GetAsPrevDisplay;
    property    AsDisplay                         : string            read GetAsDisplay;
    property    AsHint                            : string            read GetAsHint;
    property    AsPrevValue                       : TSignal           read GetAsPrevValue;
    property    AsValue                           : TSignal           read GetAsValue;
    property    PrevPosition                      : Integer           read FPrevPosition;
    property    ControlName                       : string            read GetControlName;
    property    WheelSupport                      : Boolean           read FWheelSupport             write FWheelSupport;
    property    WheelSensitivity                  : Integer           read FWheelSensitivity         write FWheelSensitivity;
    property    WheelCounter                      : Integer           read FWheelCounter             write FWheelCounter;
    property    AutomationPosition                : Integer                                          write SetAutomationPosition;
    property    PlayPosition                      : Integer                                          write SetPlayPosition;
    property    ScaledPosition[ anIndex: Integer] : TSignal                                          write SetScaledPosition;
    property    RangeCount                        : Integer           read GetRangeCount;
    property    ActiveRange                       : Integer           read GetActiveRange            write SetActiveRange;
    property    RangeValue                        : TSignal                                          write SetRangeValue;
    property    VariationCount                    : Integer           read GetVariationCount;
    property    AllowVariations                   : Boolean           read GetAllowVariations;
    property    ActiveVariation                   : Integer           read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string            read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal           read GetVariationValue         write SetVariationValue;
    property    CurrentValue                      : TSignal           read FCurrentValue             write SetCurrentValue;
  published
    property    MIDIColor                         : TColor            read FMIDIColor                write SetMIDIColor              default CL_MIDI;
    property    FocusColor                        : TColor            read FFocusColor               write SetFocusColor             default CL_FOCUS;
    property    MIDIFocusColor                    : TColor            read FMIDIFocusColor           write SetMIDIFocusColor         default CL_MIDI_FOCUS;
    property    Action;
    property    FriendlyName                      : string            read FFriendlyName             write SetFriendlyName;
    property    _DesignerRemarks                  : string            read F_DesignerRemarks         write F_DesignerRemarks;
    property    Dezippered                        : Boolean           read FDezippered               write FDezippered               default False;
    property    AllowAutomation                   : Boolean           read GetAllowAutomation        write SetAllowAutomation        default True;
    property    AllowRandomization                : Boolean           read FAllowRandomization       write SetAllowRandomization     default False;
    property    ShowAllowRandomization            : Boolean           read GetShowAllowRandomization write SetShowAllowRandomization default False;
    property    AllowRecording                    : Boolean           read GetAllowRecording         write SetAllowRecording         default False;
    property    ShowAllowRecording                : Boolean           read GetShowAllowRecording     write SetShowAllowRecording     default False;
    property    IsIndicator                       : Boolean           read FIsIndicator              write SetIsIndicator            default False;
    property    AssignedMIDICC                    : Byte              read GetAssignedMIDICC         write SetAssignedMIDICC         default 0;
    property    DynamicStepCount                  : Boolean           read FDynamicStepCount         write SetDynamicStepCount       default False;
    // NOTE: declaration order is important here to avoid loss of knob resolution on a load action ...
    property    Display                           : TKnobsDisplay     read FDisplay                  write SetDisplay;
    property    StepCount                         : Integer           read GetStepCount              write SetStepCount              default 1;
    property    KnobPosition                      : Integer           read FKnobPosition             write SetKnobPosition           default 0;
    property    DefaultPosition                   : Integer           read FDefaultPosition          write SetDefaultPosition        default 0;
    property    ControlType                       : string            read GetControlType            write SetControlType;
    property    Locked                            : Boolean           read FLocked                   write SetLocked;
    // END NOTE.
    property    StyleElements                                                                                                        default [];
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    OnValueChanged         : TKnobsOnValueChange       read FOnValueChanged        write SetOnValueChanged;
    property    OnKnobPositionChanged  : TKnobsKnobPositionChanged read FOnKnobPositionChanged write FOnKnobPositionChanged;
    property    OnUnFocus              : TOnUnFocus                read FOnUnFocus             write FOnUnFocus;
    property    OnClick;
    property    ShowHint                                                                                                             default False;
    property    ShowSimpleHint         : Boolean              read FShowSimpleHint             write SetShowSimpleHint               default False;
    property    ParentShowHint                                                                                                       default False;
  end;


  TKnobsXYControl = class( TCustomControl, IKnobsVariation)
  // Base type for two valued controls, ones that can have an effect on an external model
  private
    FFriendlyName           : string;
    F_DesignerRemarks       : string;
    FDynamicStepCount       : Boolean;
    FStepCount              : Integer;             // Number of steps in the total knob travel, same for X and Y directions
    FPrevPositionX          : Integer;             // Previous knob X position
    FPrevPositionY          : Integer;             // Previous knob X position;
    FKnobPositionX          : Integer;             // Current  knob X position
    FKnobPositionY          : Integer;             // Current  knob Y position
    FDefaultPositionX       : Integer;             // default  knob X position;
    FDefaultPositionY       : Integer;             // default  knob Y position;
    FControlType            : string;              // The control's behaviour
    FOnValueChanged         : TKnobsOnValueChange; // Called with the new value- twice, once for X, once for Y
    FPopupCount             : Integer;
    FPopupWindow            : THintWindow;
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FActiveVariation        : Integer;
    FVariationValuesX       : TKnobsVariationValues;
    FVariationValuesY       : TKnobsVariationValues;
    FLiveMorph              : TSignal;
    FCurrentValueX          : TSignal;
    FCurrentValueY          : TSignal;
    FControlSrc             : TKnobsControlSrc;
    FOnUnFocus              : TOnUnFocus;
  private
    function    GetModule: TKnobsCustomModule;
  private
    function    GetModuleName   : string;
  private
    procedure   FinalizeKnobPositionX;
    procedure   FinalizeKnobPositionY;
    function    GetControlType           : string;
    procedure   SetControlType           ( const aValue: string);
    function    GetAllowRandomization    : Boolean;
    procedure   SetAllowRandomization    ( aValue: Boolean);
    function    GetShowAllowRandomization: Boolean;
    procedure   SetShowAllowRandomization( aValue: Boolean);
    function    GetVariationCount: Integer;
    function    GetAllowVariations: Boolean;
    function    GetActiveVariation: Integer;
    procedure   SetActiveVariation( aValue: Integer);
    function    GetVariationsAsString: string;
    procedure   SetVariationsAsString( const aValue: string);
    function    GetVariationValue( anIndex: Integer): TSignal;
    procedure   SetVariationValue( anIndex: Integer; aValue: TSignal);
    procedure   SetCurrentValueX( aValue: TSignal);
    procedure   SetCurrentValueY( aValue: TSignal);
    function    KnobValueToVariationValue( aValue: TSignal): TSignal;
    function    VariationValueToKnobValue( aValue: TSignal): TSignal;
    procedure   SetDefaultVariations;
    procedure   CountGene( var aCount: Integer);
    procedure   FillGene  ( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    procedure   AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
  private
    procedure   SetKnobPositionX   ( aValue: Integer);                                                          virtual;
    procedure   SetKnobPositionY   ( aValue: Integer);                                                          virtual;
    procedure   SetDefaultPositionX( aValue: Integer);                                                          virtual;
    procedure   SetDefaultPositionY( aValue: Integer);                                                          virtual;
    procedure   SetStepCount       ( aValue: Integer);                                                          virtual;
    function    GetAsHint          : string;
    function    GetAsValueX        : TSignal;
    function    GetAsValueY        : TSignal;
    procedure   SetOnValueChanged  ( aValue: TKnobsOnValueChange);
    procedure   SetDynamicStepCount( aValue: Boolean);
    procedure   NotifyPeers;                                                                                    virtual;
  protected
    procedure   FixParam;
    procedure   ValueChanged( IsFinal: Boolean);                                                                virtual;
    function    FindModule   : TKnobsCustomModule;
    function    FindWirePanel: TKnobsWirePanel;
    procedure   ActivatePopup;
    procedure   ShowPopup;
    procedure   HidePopup;
    procedure   ForceHidePopup;
    procedure   UnFocus;                                                                                        virtual;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    procedure   SetDefault;
    procedure   Dump( var aFile: TextFile; anInd: Integer);
    procedure   BeginStateChange;
    procedure   EndStateChange;
  public
    procedure   FixLiveMorphX;
    procedure   FixLiveMorphY;
    procedure   SetRandomValue( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   SetRandomValue( anAmount: TSignal);                                                            overload;
    procedure   Randomize     ( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   Randomize     ( anAmount: TSignal);                                                            overload;
    procedure   Mutate        ( aProb, aRange: TSignal; aVariation: Integer);                                  overload;
    procedure   Mutate        ( aProb, aRange: TSignal);                                                       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer);       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);                   overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                           overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom: Integer);                                       overload;
    procedure   LiveMorph     ( anAmount: TSignal);
    function    CanAllowRandomization: Boolean;
    function    CanRandomize: Boolean;
  public
    property    ModuleName                        : string              read GetModuleName;
  public
    property    AsHint                            : string              read GetAsHint;
    property    AsValueX                          : TSignal             read GetAsValueX;
    property    AsValueY                          : TSignal             read GetAsValueY;
    property    PrevPositionX                     : Integer             read FKnobPositionX;
    property    PrevPositionY                     : Integer             read FKnobPositionY;
    property    VariationCount                    : Integer             read GetVariationCount;
    property    AllowVariations                   : Boolean             read GetAllowVariations;
    property    ActiveVariation                   : Integer             read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string              read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal             read GetVariationValue         write SetVariationValue;
    property    CurrentValueX                     : TSignal             read FCurrentValueX            write SetCurrentValueX;
    property    CurrentValueY                     : TSignal             read FCurrentValueY            write SetCurrentValueY;
  published
    property    Action;
    property    AllowRandomization                : Boolean             read GetAllowRandomization     write SetAllowRandomization     default False;
    property    ShowAllowRandomization            : Boolean             read GetShowAllowRandomization write SetShowAllowRandomization default False;
    property    FriendlyName                      : string              read FFriendlyName             write FFriendlyName;
    property    _DesignerRemarks                  : string              read F_DesignerRemarks         write F_DesignerRemarks;
    property    DynamicStepCount                  : Boolean             read FDynamicStepCount         write SetDynamicStepCount       default False;
    // NOTE: declaration order is important here to avoid loss of resolution on a load action ...
    property    StepCount                         : Integer             read FStepCount                write SetStepCount              default 1;
    property    KnobPositionX                     : Integer             read FKnobPositionX            write SetKnobPositionX          default 0;
    property    KnobPositionY                     : Integer             read FKnobPositionY            write SetKnobPositionY          default 0;
    property    DefaultPositionX                  : Integer             read FDefaultPositionX         write SetDefaultPositionX       default 0;
    property    DefaultPositionY                  : Integer             read FDefaultPositionY         write SetDefaultPositionY       default 0;
    property    ControlType                       : string              read FControlType              write SetControlType;
    // END NOTE.
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    OnValueChanged                    : TKnobsOnValueChange read FOnValueChanged           write SetOnValueChanged;
    property    OnUnFocus                         : TOnUnFocus          read FOnUnFocus                write FOnUnFocus;
    property    ShowHint                                                                                                               default False;
    property    ParentShowHint                                                                                                         default False;
    property    OnClick;
  end;


  TKnobsSelectorChoice = class;

  TKnobsSelector = class( TKnobsValuedControl)
  // A selector input type, lets user select from a list of items
  private
    FCaptions          : TStringList;
    FGlyphs            : TImageList;
    FDependency        : TKnobsValuedControl;
    FBorderColor       : TColor;
    FBorderColorSingle : TColor;
    FChoice            : TKnobsSelectorChoice;
    FMouseInControl    : Boolean;
    FOnRightClick      : TOnKnobsRightClick;
  private
    procedure   SetControlType           ( const aValue: string);                                              override;
  private
    procedure   DoCaptionsChanged( aSender: TObject);
    procedure   FixStepCount;
    procedure   SetCaptions         ( const aValue: TStringList);
    procedure   SetGlyphs           ( const aValue: TImageList );
    procedure   SetBorderColor      ( aValue: TColor);
    procedure   SetBorderColorSingle( aValue: TColor);
    procedure   SetChoice( const aValue: TKnobsSelectorChoice);
    function    AllDashes: Boolean;
  private
    procedure   WMSetFocus  ( var aMessage: TWMSetFocus  );                                       message   WM_SETFOCUS;
    procedure   WMKillFocus ( var aMessage: TWMKillFocus );                                       message  WM_KILLFOCUS;
    procedure   WMGetDlgCode( var aMessage: TWMGetDlgCode);                                       message WM_GETDLGCODE;
    procedure   CMMouseEnter( var aMessage: TMessage);                                            message CM_MOUSEENTER;
    procedure   CMMouseLeave( var aMessage: TMessage);                                            message CM_MOUSELEAVE;
  protected
    procedure   DoClick( aSender: TObject);
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   KeyDown( var Key: Word; Shift: TShiftState);                                                   override;
    procedure   Paint;                                                                                         override;
    procedure   LoadCaptions;
    function    GetDynamicHint: string;                                                                        override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   FixPeers;
    procedure   SetKnobPosition( aValue  : Integer);                                                                override;
    procedure   FixLiveMorph;                                                                                       override;
    procedure   SetRandomValue ( anAmount: TSignal; aVariation: Integer);                                 overload; override;
    procedure   SetRandomValue ( anAmount: TSignal);                                                      overload; override;
    procedure   Randomize      ( anAmount: TSignal; aVariation: Integer);                                 overload; override;
    procedure   Randomize      ( anAmount: TSignal);                                                      overload; override;
    procedure   Mutate         ( aProb, aRange: TSignal; aVariation: Integer);                            overload; override;
    procedure   Mutate         ( aProb, aRange: TSignal);                                                 overload; override;
    procedure   MateWith       ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); overload; override;
    procedure   MateWith       ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);             overload; override;
    procedure   Morph          ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                     overload; override;
    procedure   Morph          ( anAmount: TSignal; aDad, aMom: Integer);                                 overload; override;
    procedure   LiveMorph      ( anAmount: TSignal);                                                                override;
  published
    property    Captions          : TStringList          read FCaptions          write SetCaptions;
    property    Glyphs            : TImageList           read FGlyphs            write SetGlyphs;
    property    Color                                                                                       default clGray;
    property    BorderColor       : TColor               read FBorderColor       write SetBorderColor       default clYellow;
    property    BorderColorSingle : TColor               read FBorderColorSingle write SetBorderColorSingle default clWhite;
    property    Choice            : TKnobsSelectorChoice read FChoice            write SetChoice;
    property    Font;
    property    Dependency        : TKnobsValuedControl  read FDependency        write FDependency  ;
    property    OnRightClick      : TOnKnobsRightClick   read FOnRightClick      write FOnRightClick;
  end;


  TKnobsKnob = class( TKnobsValuedControl)
  // A rotary knob, or linear horizontal or vertical control, mouse wheel works too.
  private
    FDownClicker       : TKnobsClicker;
    FUpClicker         : TKnobsClicker;
    FShift             : TShiftState;
    FMouseDown         : Boolean;
    FGraphic           : TBitmap;
    FMouseInControl    : Boolean;
    FTransparent       : Boolean;
    FPairedWith        : TKnobsKnob;
    FPairingMode       : TKnobsPairingMode;
    FPairingLock       : Integer;
    FControlMode       : TDistanceMode;
    FFinalizationTimer : TTimer;
    FTresholdPast      : Boolean;
    FFinalizations     : Integer;
  private
    procedure   InitializeFinalizationTimer;
    procedure   FinalizeFinalizationTimer;
    procedure   StartFinalizationTimer;
    procedure   StopFinalizationTimer;
    procedure   FinalizationTimerFired( aSender: TObject);
  private
    procedure   WMSetFocus  ( var aMessage: TWMSetFocus  );                                       message   WM_SETFOCUS;
    procedure   WMKillFocus ( var aMessage: TWMKillFocus );                                       message  WM_KILLFOCUS;
    procedure   WMGetDlgCode( var aMessage: TWMGetDlgCode);                                       message WM_GETDLGCODE;
    procedure   CMMouseEnter( var aMessage: TMessage);                                            message CM_MOUSEENTER;
    procedure   CMMouseLeave( var aMessage: TMessage);                                            message CM_MOUSELEAVE;
    procedure   OnDownClick( aSender: TObject);
    procedure   OnUpClick  ( aSender: TObject);
    procedure   NotifyPeers;                                                                                   override;
    procedure   LockPairing;
    procedure   UnlockPairing;
    function    PairingLocked: Boolean;
    procedure   FixPairing;
  private
    procedure   SetTransparent    ( aValue: Boolean          );
    procedure   SetKnobPosition   ( aValue: Integer          );                                                override;
    procedure   SetPairedWith     ( const aValue: TKnobsKnob );
    procedure   SetPairingMode    ( aValue: TKnobsPairingMode);
    procedure   SetControlMode    ( aValue: TDistanceMode    );                                                 virtual;
  private
    function    CenterPoint: TPoint;
    procedure   BeginMove ( aPoint: TPoint; aShift: TShiftState);                                               virtual;
    procedure   EndMove   ( aPoint: TPoint; aShift: TShiftState);                                               virtual;
    procedure   UpdateMove( aPoint: TPoint; aShift: TShiftState);                                               virtual;
  protected
    procedure   SetLocked( aValue: Boolean);                                                                   override;
    function    CreateClicker( aLeft, aTop: Integer; aNotifyEvent: TNotifyEvent; aBitmap: TBitmap): TKnobsClicker; virtual;
    procedure   Paint;                                                                                         override;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer);                                                 override;
    procedure   HandleWheelUp  ( anAmount: Integer; aShift: TShiftState);                                      override;
    procedure   HandleWheelDown( anAmount: Integer; aShift: TShiftState);                                      override;
    procedure   KeyDown( var Key: Word; Shift: TShiftState);                                                   override;
    procedure   AssignGraphic;                                                                                  virtual;
    procedure   Loaded;                                                                                        override;
    function    WheelAmount: Integer;                                                                           virtual;
    procedure   SetSize;                                                                                        virtual;
    procedure   CreateClickers;                                                                                 virtual;
    procedure   InvalidateDisplay;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   FixBitmaps;                                                                                    override;
  published
    property    Action;
    property    Transparent  : Boolean           read FTransparent   write SetTransparent default False;
    property    Color;
    property    TabOrder;
    property    TabStop;
    property    ParentColor                                                                               default True;
    property    PairedWith   : TKnobsKnob        read FPairedWith    write SetPairedWith;
    property    PairingMode  : TKnobsPairingMode read FPairingMode   write SetPairingMode;
    property    ControlMode  : TDistanceMode     read FControlMode   write SetControlMode;
    property    ShowHint                                                                                  default False;
    property    ParentShowHint                                                                            default False;
  end;


  TKnobsSmallKnob = class( TKnobsKnob)
  // A bit smaller than a TKnobsKnob
  protected
    procedure   AssignGraphic;                                                                                 override;
  end;


  TKnobsNoKnob = class( TKnobsKnob)
  // No knob at all, just the clickers
  protected
    procedure   AssignGraphic;                                                                                 override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  end;


  TKnobsSlider = class( TKnobsKnob)
  // A vertical slider type of control
  private
    procedure   SetControlMode( aValue: TDistanceMode);                                                        override;
    procedure   DoResize( aSender: TObject);
  protected
    procedure   AssignGraphic;                                                                                 override;
    function    CreateClicker( aLeft, aTop: Integer; aNotifyEvent: TNotifyEvent; aBitmap: TBitmap): TKnobsClicker; override;
    procedure   Paint;                                                                                         override;
    procedure   SetSize;                                                                                       override;
    procedure   CreateClickers;                                                                                override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  end;


  TKnobsHSlider = class( TKnobsKnob)
  // A horizontal slider type of control
  private
    procedure   SetControlMode( aValue: TDistanceMode);                                                        override;
    procedure   DoResize( aSender: TObject);
  protected
    procedure   AssignGraphic;                                                                                 override;
    function    CreateClicker( aLeft, aTop: Integer; aNotifyEvent: TNotifyEvent; aBitmap: TBitmap): TKnobsClicker; override;
    procedure   Paint;                                                                                         override;
    procedure   SetSize;                                                                                       override;
    procedure   CreateClickers;                                                                                override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  end;


  TKnobsPad = class( TKnobsXYControl)
  // A pad control horizontal and vertical control, no mouse wheel
  private
    FBackGround        : TBitmap;
    FOrigBackGround    : TBitmap;
    FBorderColor       : TColor;
    FFocusColor        : TColor;
    FDotColor          : TColor;
    FDotSize           : Integer;
    FMouseDown         : Boolean;
    FMouseInControl    : Boolean;
    FFinalizationTimer : TTimer;
    FTresholdPast      : Boolean;
    FFinalizations     : Integer;
    FCursorColor       : TColor;
    FSendColorValues   : Boolean;
  private
    procedure   SetBackGround ( aValue: TBitmap);
    procedure   SetBorderColor( aValue: TColor );
    procedure   SetFocusColor ( aValue: TColor );
    procedure   SetDotColor   ( aValue: TColor );
    procedure   SetDotSize    ( aValue: Integer);
    procedure   SetCursorColor;
    function    GetRedValue   : TSignal;
    function    GetGreenValue : TSignal;
    function    GetBlueValue  : TSignal;
  private
    procedure   InitializeFinalizationTimer;
    procedure   FinalizeFinalizationTimer;
    procedure   StartFinalizationTimer;
    procedure   StopFinalizationTimer;
    procedure   FinalizationTimerFired( aSender: TObject);
  private
    procedure   WMSetFocus  ( var aMessage: TWMSetFocus  );                                       message   WM_SETFOCUS;
    procedure   WMKillFocus ( var aMessage: TWMKillFocus );                                       message  WM_KILLFOCUS;
    procedure   WMGetDlgCode( var aMessage: TWMGetDlgCode);                                       message WM_GETDLGCODE;
    procedure   CMMouseEnter( var aMessage: TMessage);                                            message CM_MOUSEENTER;
    procedure   CMMouseLeave( var aMessage: TMessage);                                            message CM_MOUSELEAVE;
  private
    procedure   SetKnobPositionX( aValue: Integer);                                                            override;
    procedure   SetKnobPositionY( aValue: Integer);                                                            override;
    function    CenterPoint: TPoint;
    function    ScalePoint( const aValue: TPoint): TPoint;
    procedure   BeginMove ( aPoint: TPoint; aShift: TShiftState);                                               virtual;
    procedure   EndMove   ( aPoint: TPoint; aShift: TShiftState);                                               virtual;
    procedure   UpdateMove( aPoint: TPoint; aShift: TShiftState);                                               virtual;
  protected
    procedure   Paint;                                                                                         override;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer);                                                 override;
    procedure   KeyDown( var Key: Word; Shift: TShiftState);                                                   override;
    procedure   Loaded;                                                                                        override;
    procedure   ValueChanged( IsFinal: Boolean);                                                               override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   LoadBackGroundFile( const aFileName: string);
    procedure   ClearBackGroundFile;
  public
    property    RedValue        : TSignal read GetRedValue;
    property    GreenValue      : TSignal read GetGreenValue;
    property    BlueValue       : TSignal read GetBlueValue;
  published
    property    Background      : TBitmap read FBackGround      write SetBackGround;
    property    BorderColor     : TColor  read FBorderColor     write SetBorderColor         default clSilver;
    property    FocusColor      : TColor  read FFocusColor      write SetFocusColor          default CL_FOCUS;
    property    DotColor        : TColor  read FDotColor        write SetDotColor            default CL_DOT;
    property    DotSize         : Integer read FDotSize         write SetDotSize             default 3;
    property    SendColorValues : Boolean read FSendColorValues write FSendColorValues       default false;
    property    Action;
    property    Color;
    property    TabOrder;
    property    TabStop;
    property    ParentColor                                                                  default True;
    property    ParentShowHint                                                               default False;
    property    ShowHint                                                                     default False;
  end;


  TKnobsValuedButton = class( TKnobsValuedControl)
  // Single button
  private
    FButton     : TSpeedButton;
    FColorFocus : TColor;
    FUserColor  : TColor;
    FUserFlat   : Boolean;
  private
    procedure   SetKnobPosition    ( aValue: Integer);                                                         override;
    function    GetCaption         : string;
    procedure   SetCaption         ( const aValue: string);
    function    GetGlyph           : TBitmap;
    procedure   SetGlyph           ( const aValue: TBitmap);
    function    GetHint            : string;
    procedure   SetHint            ( const aValue: string);
    function    GetGroupIndex      : Integer;
    procedure   SetGroupIndex      ( aValue: Integer);
    function    GetAllowAllUp      : Boolean;
    procedure   SetAllowAllUp      ( aValue: Boolean);
  private
    function    GetAlignWithMargins: Boolean;
    procedure   SetAlignWithMargins( aValue: Boolean);
    function    GetFlat            : Boolean;
    procedure   SetFlat            ( aValue: Boolean);
    function    GetFont            : TFont;
    procedure   SetFont            ( const aValue: TFont);
    function    GetLayout          : TButtonLayout;
    procedure   SetLayout          ( aValue: TButtonLayout);
    function    GetMargin          : Integer;
    procedure   SetMargin          ( aValue: Integer);
    function    GetMargins         : TMargins;
    procedure   SetMargins         ( aValue: TMargins);
    function    GetNumGlyphs       : TNumGlyphs;
    procedure   SetNumGlyphs       ( aValue: TNumGlyphs);
    function    GetParentFont      : Boolean;
    procedure   SetParentFont      ( aValue: Boolean);
    function    GetParentShowHint  : Boolean;
    procedure   SetParentShowHint  ( aValue: Boolean);
    function    GetShowHint        : Boolean;
    procedure   SetShowHint        ( aValue: Boolean);
    function    Gettransparent     : Boolean;
    procedure   SetTransparent     ( aValue: Boolean);
    function    GetVisible         : Boolean;
    procedure   SetVisible         ( aValue: Boolean);
  private
    procedure   OnButtonClick( aSender: TObject);
  protected
    procedure   WMSetFocus  ( var aMessage: TWMSetFocus  );                                       message   WM_SETFOCUS;
    procedure   WMKillFocus ( var aMessage: TWMKillFocus );                                       message  WM_KILLFOCUS;
    procedure   WMGetDlgCode( var aMessage: TWMGetDlgCode);                                       message WM_GETDLGCODE;
    procedure   KeyDown     ( var Key: Word; Shift: TShiftState);                                              override;
    procedure   SetBiDiMode        ( aValue: TBiDiMode);                                                       override;
    procedure   SetParentBiDiMode  ( aValue: Boolean);                                                         override;
    function    GetEnabled         : Boolean;                                                                  override;
    procedure   SetEnabled         ( aValue: Boolean);                                                         override;
  protected
    procedure   Resize;                                                                                        override;
    procedure   Loaded;                                                                                        override;
    procedure   SetDefaults;                                                                                    virtual;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  published
    property    Caption          : string        read GetCaption          write SetCaption                               ;
    property    Glyph            : TBitmap       read GetGlyph            write SetGlyph                                 ;
    property    Hint             : string        read GetHint             write SetHint                                  ;
    property    GroupIndex       : Integer       read GetGroupIndex       write SetGroupIndex       default             1;
    property    AllowAllUp       : Boolean       read GetAllowAllUp       write SetAllowAllUp       default         false;
    property    Height                                                                              default            18;
    property    Width                                                                               default            18;
    property    AlignWithMargins : Boolean       read GetAlignWithMargins write SetAlignWithMargins default         False;
    property    Color                                                                               default      clSilver;
    property    ColorFocus       : TColor        read FColorFocus         write FColorFocus         default     clSkyBlue;
    property    Flat             : Boolean       read GetFlat             write SetFlat             default         False;
    property    Font             : TFont         read GetFont             write SetFont                                  ;
    property    Layout           : TButtonLayout read GetLayout           write SetLayout           default   blGlyphLeft;
    property    Margin           : Integer       read GetMargin           write SetMargin           default            -1;
    property    Margins          : TMargins      read GetMargins          write SetMargins                               ;
    property    NumGlyphs        : TNumGlyphs    read GetNumGlyphs        write SetNumGlyphs        default             1;
    property    ParentFont       : Boolean       read GetParentFont       write SetParentFont       default          True;
    property    ParentShowHint   : Boolean       read GetParentShowHint   write SetParentShowHint   default         False;
    property    ShowHint         : Boolean       read GetShowHint         write SetShowHint         default         False;
    property    Transparent      : Boolean       read GetTransparent      write SetTransparent      default          True;
    property    Visible          : Boolean       read GetVisible          write SetVisible          default          True;
  end;


  TKnobsValuedButtons = class( TKnobsValuedControl)
  // Radio buttons
  private
    FButtons          : array of TSpeedButton;
    FCaptions         : TStrings;
    FImageIndices     : TStrings;
    FGlyphs           : TImageList;
    FHints            : TStrings;
    FGroupIndex       : Integer;
    FAlignWithMargins : Boolean;
    FAllowAllUp       : Boolean;
    FFlat             : Boolean;
    FFont             : TFont;
    FLayout           : TButtonLayout;
    FDisplayMode      : TKnobsButtonsDisplayMode;
    FMargin           : Integer;
    FMargins          : TMargins;
    FNumGlyphs        : TNumGlyphs;
    FParentFont       : Boolean;
    FParentShowHint   : Boolean;
    FShowHint         : Boolean;
    FTransparent      : Boolean;
    FVisible          : Boolean;
    FColorFocus       : TColor;
    FUserColor        : TColor;
    FUserFlat         : Boolean;
  private
    procedure   FixDimensions;
    procedure   FixGlyph( anIndex: Integer; aBitmap: TBitmap);
    function    FindGlyphIndex( anIndex: Integer): Integer;
  private
    procedure   FixAlignWithMargins;
    procedure   FixBiDiMode;
    procedure   FixCaptions;
    procedure   FixEnabled;
    procedure   FixFlat;
    procedure   FixFont;
    procedure   FixGlyphs;
    procedure   FixHints;
    procedure   FixLayout;
    procedure   FixMargin;
    procedure   FixMargins;
    procedure   FixNumGlyphs;
    procedure   FixParentFont;
    procedure   FixParentShowHint;
    procedure   FixParentBiDiMode;
    procedure   FixShowHint;
    procedure   FixTransparent;
    procedure   FixVisible;
    procedure   FixMode( aNewStepCount: Integer);
  private
    procedure   SetKnobPosition    ( aValue: Integer);                                                         override;
    procedure   SetStepCount       ( aValue: Integer);                                                         override;
    function    GetCaptions        : TStrings;
    procedure   SetCaptions        ( const aValue: TStrings);
    function    GetImageIndices    : TStrings;
    procedure   SetImageIndices    ( const aValue: TStrings);
    function    GetGlyphs          : TImageList;
    procedure   SetGlyphs          ( const aValue: TImageList);
    function    GetHints           : TStrings;
    procedure   SetHints           ( const aValue: TStrings);
    procedure   SetGroupIndex      ( aValue: Integer);
    procedure   SetAllowAllUp      ( aValue: Boolean);
  private
    procedure   SetAlignWithMargins( aValue: Boolean);
    procedure   SetFlat            ( aValue: Boolean);
    procedure   SetFont            ( const aValue: TFont);
    procedure   SetLayout          ( aValue: TButtonLayout);
    procedure   SetDisplayMode     ( aValue: TKnobsButtonsDisplayMode);
    procedure   SetMargin          ( aValue: Integer);
    procedure   SetMargins         ( aValue: TMargins);
    procedure   SetNumGlyphs       ( aValue: TNumGlyphs);
    procedure   SetParentFont      ( aValue: Boolean);
    procedure   SetParentShowHint  ( aValue: Boolean);
    procedure   SetShowHint        ( aValue: Boolean);
    procedure   SetTransparent     ( aValue: Boolean);
    procedure   SetVisible         ( aValue: Boolean);
  private
    procedure   OnButtonClick( aSender: TObject);
    procedure   CreateButton;
  protected
    procedure   WMSetFocus  ( var aMessage: TWMSetFocus  );                                       message   WM_SETFOCUS;
    procedure   WMKillFocus ( var aMessage: TWMKillFocus );                                       message  WM_KILLFOCUS;
    procedure   WMGetDlgCode( var aMessage: TWMGetDlgCode);                                       message WM_GETDLGCODE;
    procedure   KeyDown     ( var Key: Word; Shift: TShiftState);                                              override;
    procedure   SetBiDiMode        ( aValue: TBiDiMode);                                                       override;
    procedure   SetParentBiDiMode  ( aValue: Boolean);                                                         override;
    procedure   SetEnabled         ( aValue: Boolean);                                                         override;
  protected
    procedure   Resize;                                                                                        override;
    procedure   Loaded;                                                                                        override;
    procedure   FixButtons;                                                                                     virtual;
    procedure   SetDefaults;                                                                                    virtual;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  published
    property    StepCount                                                                                    default             2;
    property    Captions         : TStrings                 read GetCaptions       write SetCaptions                              ;
    property    ImageIndices     : TStrings                 read GetImageIndices   write SetImageIndices                          ;
    property    Glyphs           : TImageList               read GetGlyphs         write SetGlyphs                                ;
    property    Hints            : TStrings                 read GetHints          write SetHints                                 ;
    property    GroupIndex       : Integer                  read FGroupIndex       write SetGroupIndex       default             1;
    property    AllowAllUp       : Boolean                  read FAllowAllUp       write SetAllowAllUp       default         false;
    property    Height                                                                                       default            25;
    property    Width                                                                                        default            51;
    property    AlignWithMargins : Boolean                  read FAlignWithMargins write SetAlignWithMargins default         False;
    property    Color                                                                                        default      clSilver;
    property    ColorFocus       : TColor                   read FColorFocus       write FColorFocus         default     clSkyBlue;
    property    Flat             : Boolean                  read FFlat             write SetFlat             default         False;
    property    Font             : TFont                    read FFont             write SetFont                                  ;
    property    Layout           : TButtonLayout            read FLayout           write SetLayout           default   blGlyphLeft;
    property    DisplayMode      : TKnobsButtonsDisplayMode read FDisplayMode      write SetDisplayMode      default bdmHorizontal;
    property    Margin           : Integer                  read FMargin           write SetMargin           default            -1;
    property    Margins          : TMargins                 read FMargins          write SetMargins                               ;
    property    NumGlyphs        : TNumGlyphs               read FNumGlyphs        write SetNumGlyphs        default             1;
    property    ParentFont       : Boolean                  read FParentFont       write SetParentFont       default          True;
    property    ParentShowHint   : Boolean                  read FParentShowHint   write SetParentShowHint   default         False;
    property    ShowHint         : Boolean                  read FShowHint         write SetShowHint         default         False;
    property    Transparent      : Boolean                  read FTransparent      write SetTransparent      default          True;
    property    Visible          : Boolean                  read FVisible          write SetVisible          default          True;
  end;


  TKnobsDisplayEditor = class( TMemo)
  // An multi line editor for TDisplay
  private
    FDisplay : TKnobsDisplay;
  private
    procedure   EndEdit;
    procedure   WMKillFocus( var aMessage: TWMKillFocus );                                         message WM_KILLFOCUS;
    procedure   CNKeyDown(var Message: TWMKeyDown);                                                message   CN_KEYDOWN;
    procedure   Do_Exit   ( aSender: TObject);
    procedure   Do_KeyDown( aSender: TObject; var aKey: Word; aShift: TShiftState);
  protected
    procedure   SetDisplay( aDisplay: TKnobsDisplay);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  end;


  TKnobsTextControl = class( TStaticText)
  // Used for text input, the base type for a couple of text input controls
  private
    FBorderColor      : TColor;
    FMustSave         : Boolean;
    FMustNotify       : Boolean;
    FSavedColor       : TColor;
    FPrevValue        : string;
    FOnChanged        : TKnobsOnTextChanged;
  strict private
    class constructor Create;
    class destructor  Destroy;
  private
    function    GetModuleName   : string;
  private
    procedure   SetBorderColor( aColor: TColor);
    function    FindWirePanel: TKnobsWirePanel;
    function    GetModule: TKnobsCustomModule;
    procedure   Do_KeyDown( aSender: TObject; var aKey: Word; aShift: TShiftState);
  protected
    procedure   HandleReturnKey;                                                                      virtual; abstract;
    procedure   HandleDoubleClick;                                                                    virtual; abstract;
    function    GetTextValue: string;                                                                 virtual; abstract;
    procedure   SetTextValue( const aValue: string);                                                  virtual; abstract;
    procedure   CreateParams( var aParams: TCreateParams);                                                     override;
    procedure   SetText( const aValue: string);
    procedure   TextChanged( const aNewValue: string; DoNotify: Boolean);                                       virtual;
    procedure   HandleTextChange( const aValue: string);                                                        virtual;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   WMSetFocus ( var aMessage: TWMSetFocus );                                          message  WM_SETFOCUS;
    procedure   WMKillFocus( var aMessage: TWMKillFocus);                                          message WM_KILLFOCUS;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    procedure   Edit;                                                                                           virtual;
    procedure   BeginStateChange;
    procedure   EndStateChange;
  public
    property    ModuleName    : string              read GetModuleName;
  public
    property    Module        : TKnobsCustomModule  read GetModule;
  published
    property    TextValue     : string              read GetTextValue     write SetTextValue;
    property    Ctl3D                                                                          default False;
    property    ParentColor                                                                    default False;
    property    Color                                                                          default clGray;
    property    Width                                                                          default 33;
    property    Height                                                                         default 15;
    property    AutoSize                                                                       default False;
    property    BorderStyle                                                                    default sbsSingle;
    property    TabStop                                                                        default True;
    property    StyleElements                                                                  default [ seBorder];
    property    BorderColor   : TColor              read FBorderColor     write SetBorderColor default clGray;
    property    MustSave      : Boolean             read FMustSave        write FMustSave      default False;
    property    MustNotify    : Boolean             read FMustNotify      write FMustNotify    default False;
    property    OnChanged     : TKnobsOnTextChanged read FOnChanged       write FOnChanged;
    property    OnClick;
  end;


  TKnobsDisplay = class( TKnobsTextControl)
  // Can be linked to aTKnob for auto update
  // Can have an editor to manually edit the value
  private
    FShortName       : string;
    FEditor          : TKnobsDisplayEditor;
    FController      : TKnobsValuedControl;
    FHasEditor       : Boolean;
    FHasParentEditor : Boolean;
    FHasScrollbar    : Boolean;
    FMultiLine       : Boolean;
  strict private
    class constructor Create;
    class destructor  Destroy;
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   HandleReturnKey;                                                                               override;
    procedure   HandleDoubleClick;                                                                             override;
    function    GetTextValue: string;                                                                          override;
    procedure   SetTextValue( const aValue: string);                                                           override;
    procedure   TextChanged( const aNewValue: string; DoNotify: Boolean);                                      override;
    procedure   WMNCHitTest( var aMessage: TWMNCHitTest);                                          message WM_NCHITTEST;
    procedure   LockKnob;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  public
    property    ShortName       : string              read FShortName;
    property    Editor          : TKnobsDisplayEditor read FEditor          write FEditor;
  published
    property    HasEditor       : Boolean             read FHasEditor       write FHasEditor       default False;
    property    HasParentEditor : Boolean             read FHasParentEditor write FHasParentEditor default False;
    property    HasScrollbar    : Boolean             read FHasScrollbar    write FHasScrollbar    default False;
    property    MultiLine       : Boolean             read FMultiLine       write FMultiLine       default False;
    property    BorderColor                                                                        default clSilver;
  end;


  TKnobsSelectorChoice = class( TKnobsDisplay)
  // Allows for making a selection from items in a selector
  // which then can be used to allow for selector modulatio
  private
    FSelector  : TKnobsSelector;
    FSelection : TByteSet;
    FNames     : TStringList;
    FChoice    : Byte;
  private
    procedure   SetSelector( const aValue: TKnobsSelector);
    procedure   SetNames( const aValue: TStringList);
    procedure   SetChoice( aValue: Byte);
    function    GetSelected( anIndex: Byte): Boolean;
    procedure   SetSelected( anIndex: Byte; aValue: Boolean);
  protected
    procedure   HandleReturnKey;                                                                               override;
    procedure   HandleDoubleClick;                                                                             override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   SelectItem  ( anItem: Byte);
    procedure   UnselectItem( anItem: Byte);
  public
    property    Names                    : TStringList    read FNames      write SetNames;
    property    Choice                   : Byte           read FChoice     write SetChoice;
    property    Selected[ anIndex: Byte] : Boolean        read GetSelected write SetSelected;
  published
    property    Selector                 : TKnobsSelector read FSelector   write SetSelector;
  end;


  TKnobsFileSelector = class( TKnobsTextControl)
  // A file browser thingie
  private
    FFileSelector : TOpenDialog;
    FDataFileName : string;
    FDataFileExt  : string;
    FDataFileMsg  : string;
  strict private
    class var FScalaCounter : Integer;
  strict private
    class constructor Create;
    class destructor  Destroy;
  private
    procedure   SetDataFileName( const aValue: string);
    procedure   FixCaption;
    procedure   SelectFile;
  protected
    procedure   HandleReturnKey;                                                                               override;
    procedure   HandleDoubleClick;                                                                             override;
    function    GetTextValue: string;                                                                          override;
    procedure   SetTextValue( const aValue: string);                                                           override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  published
    property    DataFileName : string read FDataFileName write SetDataFileName;
    property    DataFileExt  : string read FDataFileExt  write FDataFileExt;
    property    DataFileMsg  : string read FDataFileMsg  write FDataFileMsg;
  end;


  TKnobsIndicator = class( TKnobsGraphicControl)
  // an LED
  private
    FShortName  : string;
    FActive     : Boolean;
    FActiveBm   : TBitmap;
    FInActiveBm : TBitmap;
  private
    procedure   SetActive        ( aValue: Boolean);
    procedure   SetBitmapActive  ( aValue: TBitmap);
    procedure   SetBitmapInactive( aValue: TBitmap);
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   AssignBitmaps;                                                                                  virtual;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    procedure   FixBitmaps;                                                                                     virtual;
  public
    property    BitmapActive   : TBitmap read FActiveBm   write SetBitmapActive;
    property    BitmapInactive : TBitmap read FInactiveBm write SetBitmapInactive;
    property    ShortName      : string  read FShortName;
  published
    property    Active         : Boolean read FActive     write SetActive default False;
    property    Visible;
    property    ShowHint                                                  default False;
    property    ParentShowHint                                            default False;
    property    OnClick;
  end;


  TKnobsIndicatorBar = class( TKnobsGraphicControl)
  // A bar of indicator LEDs, horizontal or vertical
  private
    FShortName      : string;
    FIndicatorCount : Integer;
    FValue          : TSignal;
    FPeakValue      : TSignal;
    FValeyValue     : TSignal;
    FShowPeakValue  : Boolean;
    FShowValeyValue : Boolean;
    FValueLowMid    : TSignal;
    FValueMidHigh   : TSignal;
    FDisplayType    : TKnobsDisplayType;
    FOrientation    : TKnobsDisplayOrientation;
    FSpacing        : Integer;
    FBitmap         : TBitmap;
    FActiveLowBm    : TBitmap;
    FInActiveLowBm  : TBitmap;
    FActiveMidBm    : TBitmap;
    FInActiveMidBm  : TBitmap;
    FActiveHighBm   : TBitmap;
    FInActiveHighBm : TBitmap;
    FPeakColor      : TColor;
    FValeyColor     : TColor;
  private
    function    GetMaxBmWidth : Integer;
    function    GetMaxBmHeight: Integer;
    procedure   FixDimensions;
  private
    procedure   SetIndicatorCount( aValue: Integer                 );
    procedure   SetValue         ( aValue: TSignal                 );
    procedure   SetPeakValue     ( aValue: TSignal                 );
    procedure   SetValeyValue    ( aValue: TSignal                 );
    procedure   SetShowPeakValue ( aValue: Boolean                 );
    procedure   SetShowValeyValue( aValue: Boolean                 );
    procedure   SetValueLowMid   ( aValue: TSignal                 );
    procedure   SetValueMidHigh  ( aValue: TSignal                 );
    procedure   SetDisplayType   ( aValue: TKnobsDisplayType       );
    procedure   SetOrientation   ( aValue: TKnobsDisplayOrientation);
    procedure   SetSpacing       ( aValue: Integer                 );
    procedure   SetActiveLowBm   ( aValue: TBitmap                 );
    procedure   SetInActiveLowBm ( aValue: TBitmap                 );
    procedure   SetActiveMidBm   ( aValue: TBitmap                 );
    procedure   SetInActiveMidBm ( aValue: TBitmap                 );
    procedure   SetActiveHighBm  ( aValue: TBitmap                 );
    procedure   SetInActiveHighBm( aValue: TBitmap                 );
    procedure   SetPeakColor     ( aValue: TColor                  );
    procedure   SetValeyColor    ( aValue: TColor                  );
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   AssignBitmaps;                                                                                  virtual;
    function    PaintActive( anIndex: Integer): Boolean;                                                        virtual;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   FixBitmaps;                                                                                     virtual;
  public
    property    ShortName      : string                   read FShortName;
    property    ActiveLowBm    : TBitmap                  read FActiveLowBm    write SetActiveLowBm;
    property    InActiveLowBm  : TBitmap                  read FInActiveLowBm  write SetInActiveLowBm;
    property    ActiveMidBm    : TBitmap                  read FActiveMidBm    write SetActiveMidBm;
    property    InActiveMidBm  : TBitmap                  read FInActiveMidBm  write SetInActiveMidBm;
    property    ActiveHighBm   : TBitmap                  read FActiveHighBm   write SetActiveHighBm;
    property    InActiveHighBm : TBitmap                  read FInActiveHighBm write SetInActiveHighBm;
  published
    property    IndicatorCount : Integer                  read FIndicatorCount write SetIndicatorCount default 10;
    property    Value          : TSignal                  read FValue          write SetValue          nodefault;
    property    PeakValue      : TSignal                  read FPeakValue      write SetPeakValue      nodefault;
    property    ValeyValue     : TSignal                  read FValeyValue     write SetValeyValue     nodefault;
    property    ShowPeakValue  : Boolean                  read FShowPeakValue  write SetShowPeakValue  default False;
    property    ShowValeyValue : Boolean                  read FShowValeyValue write SetShowValeyValue default False;
    property    ValueLowMid    : TSignal                  read FValueLowMid    write SetValueLowMid    nodefault;
    property    ValueMidHigh   : TSignal                  read FValueMidHigh   write SetValueMidHigh   nodefault;
    property    DisplayType    : TKnobsDisplayType        read FDisplayType    write SetDisplayType    default dtFilledFromLow;
    property    Orientation    : TKnobsDisplayOrientation read FOrientation    write SetOrientation    default doHorizontal;
    property    Spacing        : Integer                  read FSpacing        write SetSpacing        default 2;
    property    PeakColor      : TColor                   read FPeakColor      write SetPeakColor      default clWhite;
    property    ValeyColor     : TColor                   read FValeyColor     write SetValeyColor     default clBlack;
    property    Visible;
    property    Enabled;
  published
    property    OnClick;
    property    OnDblClick;
  end;


  TKnobsIndicatorText = class( TCustomLabel)
  // A text element that can be used for feedback from synth engine to synth editor
  private
    FShortName     : string;
    FValue         : TSignal;
    FDisplayFormat : string;
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   SetValue( aValue: TSignal);
    procedure   SetDisplayFormat( const aValue: string);
    procedure   DoDrawText( var aRect: TRect; aFlags: Longint);                                                override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  public
    property    ShortName     : string  read FShortName;
  published
    property    DisplayFormat : string  read FDisplayFormat write SetDisplayFormat;
    property    Value         : TSignal read FValue         write SetValue          nodefault;
    property    Width                                                               default 10;
    property    Height                                                              default 10;
    property    AutoSize                                                            default False;
    property    Align                                                               default alNone;
    property    Alignment                                                           default taCenter;
    property    Color                                                               default clGray;
    property    Transparent                                                         default True;
    property    Visible                                                             default True;
    property    WordWrap                                                            default False;
    property    ParentColor                                                         default False;
    property    ParentFont                                                          default False;
    property    ShowHint                                                            default False;
    property    ParentShowHint                                                      default False;
    property    ShowAccelChar                                                       default False;
    property    Layout;
    property    Anchors;
    property    BiDiMode;
    property    Caption;
    property    Constraints;
    property    Enabled;
    property    FocusControl;
    property    Font;
    property    ParentBiDiMode;
    property    PopupMenu;
    property    OnClick;
  end;


  TKnobsMazeGraphViewer = class( TKnobsGraphicControl)
  private
    FMazeGraph     : TMazeGraph;
    FMazeForth     : TMazeForth;
    FShortName     : string;
    FSwanX         : TSignal;
    FSwanY         : TSignal;
    FColor         : TColor;
    FBorderColor   : TColor;
    FSwanColor     : TColor;
    FHunterColor   : TColor;
    FSwanSize      : Integer;
    FHunterSize    : Integer;
    FForthFileName : string;
    FImportList    : TStringList;
    FCurrentImport : Integer;
    FShowingError  : Boolean;
    FOnError       : TmazeOnError;
  private
    function    GetHunterCount   : Integer;
    procedure   SetHunterCount   ( aValue: Integer);
    function    GetHunterPosition( anIndex: Integer): Integer;
    procedure   SetHunterPosition( anIndex: Integer; aValue: Integer);
    procedure   SetSwanX         ( aValue: TSignal);
    procedure   SetSwanY         ( aValue: TSignal);
    procedure   SetColor         ( aValue: TColor);
    procedure   SetBorderColor   ( aValue: TColor);
    procedure   SetSwanColor     ( aValue: TColor);
    procedure   SetHunterColor   ( aValue: TColor);
    procedure   SetSwanSize      ( aValue: Integer);
    procedure   SetHunterSize    ( aValue: Integer);
    procedure   SetForthFileName ( const aValue: string);
  private
    procedure   InitMazeStuff;
    procedure   DoneMazeStuff;
    procedure   ImportForthExports;
    procedure   PerformForthImport;
  protected
    procedure   DoMazeError( const aSender: TObject; aType: TMazeErrorType; const aMsg: string);
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    function    FindModule: TKnobsCustomModule;
    procedure   WMEraseBackground( var aMsg: TWMEraseBkgnd);                                      message WM_ERASEBKGND;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   MoveHunter ( anIndex: Integer; atRandom: Boolean);
    procedure   MoveHunters( atRandom: Boolean);
    procedure   SelectMazeType( anIndex: Integer);
    function    ImportList: TStringList;
    procedure   ExecuteForthText( const S: string);
    procedure   ReloadMazeForth;
  public
    property    ShortName                         : string              read FShortName;
    property    ForthFileName                     : string              read FForthFileName  write SetForthFileName;
  public
    property    HunterPosition[ anIndex: Integer] : Integer             read GetHunterPosition write SetHunterPosition;
  published
    property    SwanX                             : TSignal             read FSwanX          write SetSwanX;            // default 0.5;
    property    SwanY                             : TSignal             read FSwanY          write SetSwanY;            // default 0.5;
    property    Color                             : TColor              read FColor          write SetColor                default $009E9E9E;
    property    BorderColor                       : TColor              read FBorderColor    write SetBorderColor          default clGray;
    property    SwanColor                         : TColor              read FSwanColor      write SetSwanColor            default clPurple;
    property    HunterColor                       : TColor              read FHunterColor    write SetHunterColor          default clBlue;
    property    SwanSize                          : Integer             read FSwanSize       write SetSwanSize             default 4;
    property    HunterSize                        : Integer             read FHunterSize     write SetHunterSize           default 3;
    property    HunterCount                       : Integer             read GetHunterCount  write SetHunterCount          default 4;
    property    Height                                                                                                    default 242;
    property    Width                                                                                                     default 242;
    property    OnError                           : TMazeOnError        read FOnError        write FOnError;
  end;


  TKnobsLightningViewer = class( TKnobsGraphicControl)
  // A viewer for TLightningEngine
  private
    FEngine         : TLightningEngine;
    FShortName      : string;
    FColor          : TColor;
    FBorderColor    : TColor;
    FLightningColor : TColor;
    FEta            : TSignal;
    FShowField      : Boolean;
    FStartRandom    : Boolean;
    FRandomNew      : TSignal;
    FStepCount      : Integer;
    FSteps          : Integer;
  private
    function    GetAsString: string;
    procedure   SetAsString( const aValue: string);
    function    GetRawPotential: TRawPotential;
    procedure   SetRawPotential( const aValue: TRawPotential);
    function    GetRawIsShape: TRawIsShape;
    procedure   SetRawIsShape( const aValue: TRawIsShape);
    procedure   SetColor         ( aValue: TColor);
    procedure   SetBorderColor   ( aValue: TColor);
    procedure   SetLightningColor( aValue: TColor);
    procedure   SetEta( aValue: TSignal);
    procedure   SetShowField( aValue: Boolean);
    procedure   SetStartRandom( aValue: Boolean);
    procedure   SetRandomNew( aValue: TSignal);
    procedure   SetStepCount( aValue: Integer);
    procedure   DoResize( aSender: TObject);
  protected
    procedure   Loaded;                                                                                        override;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   Step;
  public
    property    ShortName      : string        read FShortName;
    property    AsString       : string        read GetAsString     write SetAsString;
    property    RawPotential   : TRawPotential read GetRawPotential write SetRawPotential;
    property    RawIsShape     : TRawIsShape   read GetRawIsShape   write SetRawIsShape;
  published
    property    Color          : TColor  read FColor          write SetColor          default clBlack;
    property    BorderColor    : TColor  read FBorderColor    write SetBorderColor    default clGray;
    property    LightningColor : TColor  read FLightningColor write SetLightningColor default clYellow;
    property    Eta            : TSignal read FEta            write SetEta;
    property    ShowField      : Boolean read FShowField      write SetShowField;
    property    StartRandom    : Boolean read FStartRandom    write SetStartRandom;
    property    RandomNew      : TSignal read FRandomNew      write SetRandomNew;
    property    StepCount      : Integer read FStepCount      write SetStepCount;
  end;


  TKnobsDataViewer = class( TKnobsGraphicControl)
  // a viewer
  private
    FShortName   : string;
    FBackGround  : TBitmap;
    FYData       : TSignalArray;
    FXYData      : TSignalPairFifo;
    FXYPoints    : Integer;
    FXOffset     : TSignal;
    FYOffset     : TSignal;
    FMaxValue    : TSignal;
    FMinValue    : TSignal;
    FXZoom       : TSignal;
    FYZoom       : TSignal;
    FBorderColor : TColor;
    FLineColor   : TColor;
    FFillColor   : TColor;
    FStyle       : TKnobsDVStyle;
  private
    procedure   SetBackGround ( aValue: TBitmap);
    procedure   SetYData      ( const aValue: TSignalArray);
    procedure   SetXYData     ( const aValue: TSignalPairFifo);
    procedure   SetXOffset    ( aValue: TSignal);
    procedure   SetYOffset    ( aValue: TSignal);
    procedure   SetMaxValue   ( aValue: TSignal);
    procedure   SetMinValue   ( aValue: TSignal);
    procedure   SetXZoom      ( aValue: TSignal);
    procedure   SetYZoom      ( aValue: TSignal);
    procedure   SetLineColor  ( aValue: TColor );
    procedure   SetFillColor  ( aValue: TColor );
    procedure   SetBorderColor( aValue: TColor );
    procedure   SetStyle      ( aValue: TKnobsDVStyle);
    procedure   SetXYPoints   ( aValue: Integer);
  private
    procedure   FillBlank;
    procedure   FillSine;
    procedure   FillTri;
    procedure   FillSaw;
    procedure   FillSquare;
    procedure   FillNoise;
    procedure   FillEnvAr;
    procedure   FillEnvAhd;
    procedure   FillEnvAdsr;
    procedure   FillXYScope;
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   WMNCHitTest( var aMessage: TWMNCHitTest);                                         message  WM_NCHITTEST;
    procedure   WMEraseBackground( var aMsg: TWMEraseBkgnd);                                      message WM_ERASEBKGND;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    function    HasNegativeData: Boolean;
    function    ScaleData        ( anIndex: Integer): Integer;
    function    ScaleDataNegative( anIndex: Integer): Integer;
    function    ScaleX           ( anIndex: Integer): Integer;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  public
    property    ShortName   : string             read FShortName;
    property    YData       : TSignalArray       read FYData       write SetYData;
    property    XYData      : TSignalPairFifo    read FXYData      write SetXYData;
  published
    property    Background  : TBitmap            read FBackGround  write SetBackGround;
    property    XOffset     : TSignal            read FXOffset     write SetXOffset;    // default  0;
    property    YOffset     : TSignal            read FYOffset     write SetYOffset;    // default 75;
    property    MaxValue    : TSignal            read FMaxValue    write SetMaxValue;   // default High( SmallInt);
    property    MinValue    : TSignal            read FMinValue    write SetMinValue;   // default Low ( SmallInt);
    property    XZoom       : TSignal            read FXZoom       write SetXZoom;      // default 1;
    property    YZoom       : TSignal            read FYZoom       write SetYZoom;      // default 1;
    property    Color                                                                   default clGray;
    property    LineColor   : TColor             read FLineColor   write SetLineColor   default clWhite;
    property    FillColor   : TColor             read FFillColor   write SetFillColor   default $009E9E9E;
    property    BorderColor : TColor             read FBorderColor write SetBorderColor default clGray;
    property    Style       : TKnobsDVStyle      read FStyle       write SetStyle       default dvsNoise;
    property    XYPoints    : Integer            read FXYPoints    write SetXYPoints    default 255;
    property    Width                                                                   default 150;
    property    Height                                                                  default 60;
    property    OnClick;
  end;


  TKnobsDataMaker = class( TKnobsGraphicControl, IKnobsVariation)
  // a graphical data generator - has a minimum of two data points, beyond that user can add points with a double click
  // or control click into an empty area and can move points by double clicking on a point, removal is by moving it
  // outside the control, except the leftmost and rightmost points can not be removed.
  private
    FShortName              : string;
    FPrevData               : TSignalPairarray;
    FData                   : TSignalPairArray;
    FDataType               : TKnobsDataMakerType;
    FXZoom                  : TSignal;
    FYZoom                  : TSignal;
    FXPan                   : TSignal;
    FYPan                   : TSignal;
    FAutoScale              : Boolean;
    FAutoLoop               : Boolean;
    FGluedLeft              : Boolean;
    FGluedRight             : Boolean;
    FAutoHGrid              : Boolean;
    FAutoVGrid              : Boolean;
    FBorderColor            : TColor;
    FLineColor              : TColor;
    FGridColor              : TColor;
    FCursorColor            : TColor;
    FDotColor               : TColor;
    FActiveDotColor         : TColor;
    FTransparent            : Boolean;
    FShowGrid               : Boolean;
    FShowXCursor            : Boolean;
    FShowYCursor            : Boolean;
    FGraphMode              : TKnobsDMGraphMode;
    FDotSize                : Integer;
    FCursorX                : TSignal;
    FCursorY                : TSignal;
    FLastX                  : TSignal;
    FLastY                  : TSignal;
    FAllowXMoves            : Boolean;
    FAllowYMoves            : Boolean;
    FAllowPointDeletion     : Boolean;
    FAllowPointInsertion    : Boolean;
    FMouseDown              : Boolean;
    FCursorXWasShown        : Boolean;
    FCursorYWasShown        : Boolean;
    FFollowCursor           : Boolean;
    FSelectedPoint          : Integer;
    FControlType            : string;
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FMultiVariationValues   : TKnobsMultiVariationValues;
    FCurrentValues          : TSignalArray;
    FLiveMorph              : TSignal;
    FActiveVariation        : Integer;
    FOnChanged              : TKnobsOnTextChanged;
    FOnPointChanged         : TKnobsOnPointChanged;
    FOnPointAdded           : TKnobsOnPointChanged;
    FOnPointRemoved         : TKnobsOnPointChanged;
    FOnLoadDataGraph        : TOnLoadDataGraph;
    FOnSaveDataGraph        : TOnSaveDataGraph;
  private
    procedure   SetDataType   ( aValue: TKnobsDataMakerType);
    procedure   SetAutoScale  ( aValue: Boolean);
    procedure   SetAutoLoop   ( aValue: Boolean);
    procedure   SetGluedLeft  ( aValue: Boolean);
    procedure   SetGluedRight ( aValue: Boolean);
    procedure   SetAutoHGrid  ( aValue: Boolean);
    procedure   SetAutoVGrid  ( aValue: Boolean);
    function    GetControlType: string;
    procedure   SetControlType( const aValue: string);
    function    GetAllowRandomization: Boolean;
    procedure   SetAllowRandomization( aValue: Boolean);
    function    GetShowAllowRandomization: Boolean;
    procedure   SetShowAllowRandomization( aValue: Boolean);
    procedure   FixLiveMorph;
    procedure   SetRandomValue( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   SetRandomValue( anAmount: TSignal);                                                            overload;
    procedure   Randomize     ( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   Randomize     ( anAmount: TSignal);                                                            overload;
    procedure   Mutate        ( aProb, aRange: TSignal; aVariation: Integer);                                  overload;
    procedure   Mutate        ( aProb, aRange: TSignal);                                                       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer);       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);                   overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                           overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom: Integer);                                       overload;
    procedure   LiveMorph     ( anAmount: TSignal);
    function    CanAllowRandomization: Boolean;
    function    CanRandomize: Boolean;
    function    GetVariationCount: Integer;
    function    GetAllowVariations: Boolean;
    function    GetActiveVariation: Integer;
    procedure   SetActiveVariation( aValue: Integer);
    function    GetVariationsAsString: string;
    procedure   SetVariationsAsString( const aValue: string);
    function    GetVariationValue( anIndex: Integer): TSignal;
    procedure   SetVariationValue( anIndex: Integer; aValue: TSignal);
    function    GetCurrentValue( anIndex: Integer): TSignal;
    procedure   SetCurrentValue( anIndex: Integer; aValue: TSignal);
    function    KnobValueToVariationValue( aValue: TSignal): TSignal;
    function    VariationValueToKnobValue( aValue: TSignal): TSignal;
    procedure   SetDefaultVariations;
    procedure   CountGene( var aCount: Integer);
    procedure   FillGene  ( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    procedure   AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
  private
    function    GetDataCount  : Integer;
    procedure   ClearData;
    procedure   ClearAllData;
    procedure   AddData( const aData: TSignalPair);                                                            overload;
    procedure   AddData( anX, anY: TSignal);                                                                   overload;
    procedure   DeleteData( anIndex: Integer; FlagInvalidation: Boolean);
    procedure   PointChanged( anIndex: Integer; anX, anY: TSignal);
    procedure   PointAdded  ( anIndex: Integer; anX, anY: TSignal);
    procedure   PointRemoved( anIndex: Integer; anX, anY: TSignal);
    procedure   ChangeData( anIndex: Integer; anX, anY: TSignal);
    procedure   SetPrevData;
    procedure   SetData          ( const aValue: TSignalPairArray);
    procedure   SetXZoom         ( aValue: TSignal);
    procedure   SetYZoom         ( aValue: TSignal);
    procedure   SetXPan          ( aValue: TSignal);
    procedure   SetYPan          ( aValue: TSignal);
    procedure   SetLineColor     ( aValue: TColor );
    procedure   SetBorderColor   ( aValue: TColor );
    procedure   SetGridColor     ( aValue: TColor );
    procedure   SetCursorColor   ( aValue: TColor );
    procedure   SetDotColor      ( aValue: TColor );
    procedure   SetActiveDotColor( aValue: TColor);
    procedure   SetTransparent   ( aValue: Boolean);
    procedure   SetShowGrid      ( aValue: Boolean);
    procedure   SetShowXCursor   ( aValue: Boolean);
    procedure   SetShowYCursor   ( aValue: Boolean);
    procedure   SetGraphMode     ( aValue: TKnobsDMGraphMode);
    procedure   SetDotSize       ( aValue: Integer);
    procedure   SetCursorX       ( aValue: TSignal);
    procedure   SetCursorY       ( aValue: TSignal);
    function    GetCursorXY      : TSignalPair;
    procedure   SetCursorXY      ( const aValue: TSignalPair);
    function    GetAsString      : string;
    procedure   SetAsString      ( const aValue: string);
  private
    function    ScaleY( anIndex: Integer): Integer;                                                            overload;
    function    ScaleX( anIndex: Integer): Integer;                                                            overload;
    function    ScaleY( aValue : TSignal): Integer;                                                            overload;
    function    ScaleX( aValue : TSignal): Integer;                                                            overload;
    function    UnscaleX( aValue: Integer): TSignal;
    function    UnscaleY( aValue: Integer): TSignal;
    function    IsNearPoint( anX, anY: TSignal; aData: TSignalPair): Boolean;
    function    FindDataPoint( anX, anY: TSignal): Integer;                                                    overload;
    function    FindDataPoint( anX, anY: Integer): Integer;                                                    overload;
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   WMEraseBackground( var aMsg: TWMEraseBkgnd);                                      message WM_ERASEBKGND;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer);                                                 override;
    procedure   Paint;                                                                                         override;
    procedure   TextChanged( const aNewValue: string; DoNotify: Boolean);                                       virtual;
    procedure   HandleTextChange( const aValue: string);                                                        virtual;
    function    FindWirePanel: TKnobsWirePanel;
    procedure   SyncVariations;
    procedure   ValueChanged;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   MakeWave( aShape: TKnobsDMShape);
    procedure   Scale;
    procedure   MakeLoopable;
    procedure   SaveToFile;
    procedure   LoadFromFile;
    procedure   BeginStateChange;
    procedure   EndStateChange;
  public
    property    ShortName                         : string               read FShortName;
    property    Data                              : TSignalPairArray     read FData                     write SetData;
    property    DataCount                         : Integer              read GetDataCount;
    property    CursorX                           : TSignal              read FCursorX                  write SetCursorX;
    property    CursorY                           : TSignal              read FCursorY                  write SetCursorY;
    property    CursorXY                          : TSignalPair          read GetCursorXY               write SetCursorXY;
    property    AsString                          : string               read GetAsString               write SetAsString;
    property    FollowCursor                      : Boolean              read FFollowCursor;
    property    VariationCount                    : Integer              read GetVariationCount;
    property    AllowVariations                   : Boolean              read GetAllowVariations;
    property    ActiveVariation                   : Integer              read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string               read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal              read GetVariationValue         write SetVariationValue;
    property    CurrentValue  [ anIndex: Integer] : TSignal              read GetCurrentValue           write SetCurrentValue;
  published
    property    DataType                          : TKnobsDataMakerType  read FDataType                 write SetDataType               default dmtBipolar;
    property    AllowXMoves                       : Boolean              read FAllowXMoves              write FAllowXMoves              default True;
    property    AllowYMoves                       : Boolean              read FAllowYMoves              write FAllowYMoves              default True;
    property    AllowPointDeletion                : Boolean              read FAllowPointDeletion       write FAllowPointDeletion       default True;
    property    AllowPointInsertion               : Boolean              read FAllowPointInsertion      write FAllowPointInsertion      default True;
    property    AutoScale                         : Boolean              read FAutoScale                write SetAutoScale              default False;
    property    AutoLoop                          : Boolean              read FAutoLoop                 write SetAutoLoop               default False;
    property    GluedLeft                         : Boolean              read FGluedLeft                write SetGluedLeft              default True;
    property    GluedRight                        : Boolean              read FGluedRight               write SetGluedRight             default True;
    property    AutoHGrid                         : Boolean              read FAutoHGrid                write SetAutoHGrid              default False;
    property    AutoVGrid                         : Boolean              read FAutoVGrid                write SetAutoVGrid              default False;
    property    XZoom                             : TSignal              read FXZoom                    write SetXZoom;              // default 1;
    property    YZoom                             : TSignal              read FYZoom                    write SetYZoom;              // default 1;
    property    XPan                              : TSignal              read FXPan                     write SetXPan;               // default 0;
    property    YPan                              : TSignal              read FYPan                     write SetYPan;               // default 0;
    property    Color                                                                                                                   default clGray;
    property    LineColor                         : TColor               read FLineColor                write SetLineColor              default clWhite;
    property    BorderColor                       : TColor               read FBorderColor              write SetBorderColor            default clGray;
    property    GridColor                         : TColor               read FGridColor                write SetGridColor              default clSilver;
    property    CursorColor                       : TColor               read FCursorColor              write SetCursorColor            default CL_CURSOR;
    property    DotColor                          : TColor               read FDotColor                 write SetDotColor               default CL_DOT;
    property    ActiveDotColor                    : TColor               read FActiveDotColor           write SetActiveDotColor         default CL_ACTIVEDOT;
    property    Transparent                       : Boolean              read FTransparent              write SetTransparent            default True;
    property    ShowGrid                          : Boolean              read FShowGrid                 write SetShowGrid               default True;
    property    ShowXCursor                       : Boolean              read FShowXCursor              write SetShowXCursor            default True;
    property    ShowYCursor                       : Boolean              read FShowYCursor              write SetShowYCursor            default True;
    property    GraphMode                         : TKnobsDMGraphMode    read FGraphMode                write SetGraphMode              default dmgLinear;
    property    DotSize                           : Integer              read FDotSize                  write SetDotSize                default   3;
    property    Width                                                                                                                   default 150;
    property    Height                                                                                                                  default  60;
    property    Align;
    property    ControlType                       : string               read GetControlType            write SetControlType;
    property    AllowRandomization                : Boolean              read GetAllowRandomization     write SetAllowRandomization     default False;
    property    ShowAllowRandomization            : Boolean              read GetShowAllowRandomization write SetShowAllowRandomization default False;
    property    OnChanged                         : TKnobsOnTextChanged  read FOnChanged                write FOnChanged;
    property    OnPointChanged                    : TKnobsOnPointChanged read FOnPointChanged           write FOnPointChanged;
    property    OnPointAdded                      : TKnobsOnPointChanged read FOnPointAdded             write FOnPointAdded;
    property    OnPointRemoved                    : TKnobsOnPointChanged read FOnPointRemoved           write FOnPointRemoved;
  end;


  TKnobsGridProto = class( TKnobsGraphicControl)
  private
    FShortName         : string;
    FPixelXSize        : Integer;
    FPixelYSize        : Integer;
    FDataWidth         : Integer;
    FDataHeight        : Integer;
    FData              : TKnobsDataPlane;
    FBorderColor       : TColor;
    FLineColor         : TColor;
    FGridColor         : TColor;
    FCursorColor       : TColor;
    FDotColor          : TColor;
    FActiveDotColor    : TColor;
    FTransparent       : Boolean;
    FShowGrid          : Boolean;
    FCursorX           : Integer;
    FCursorY           : Integer;
    FShowCursorX       : Boolean;
    FShowCursorY       : Boolean;
    FMouseDown         : Boolean;
    FBlockUpdates      : Boolean;
    FOnChanged         : TKnobsOnTextChanged;
    FOnLoadGridControl : TOnLoadGridControl;
    FOnSaveGridControl : TOnSaveGridControl;
  private
    procedure   SetLineColor     ( aValue: TColor );
    procedure   SetBorderColor   ( aValue: TColor );
    procedure   SetGridColor     ( aValue: TColor );
    procedure   SetCursorColor   ( aValue: TColor );
    procedure   SetDotColor      ( aValue: TColor );
    procedure   SetActiveDotColor( aValue : TColor);
    procedure   SetTransparent   ( aValue: Boolean);
    procedure   SetShowGrid      ( aValue: Boolean);
    function    GetCursorX       : TSignal;
    procedure   SetCursorX       ( aValue: TSignal);
    function    GetCursorY       : TSignal;
    procedure   SetCursorY       ( aValue: TSignal);
    function    GetCursorXY      : TSignalPair;
    procedure   SetCursorXY      ( aValue: TSignalPair);
    procedure   SetShowCursorX   ( aValue: Boolean);
    procedure   SetShowCursorY   ( aValue: Boolean);
    procedure   SetDataWidth     ( aValue: Integer);
    procedure   SetDataHeight    ( aValue: Integer);
    function    GetPixel         ( X, Y: Integer): Boolean;
    procedure   SetPixel         ( X, Y: Integer; aValue: Boolean);
    function    GetAsString      : string;
    procedure   SetAsString      ( const aValue: string);
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   WMEraseBackground( var aMsg: TWMEraseBkgnd);                                      message WM_ERASEBKGND;
    procedure   Clicked      ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);        virtual; abstract;
    procedure   RightClicked ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);        virtual; abstract;
    procedure   ClickEnded   ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);        virtual; abstract;
    procedure   DoubleClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);        virtual; abstract;
    procedure   MouseMoved   ( aShift: TShiftState; anX, anY: Integer);                               virtual; abstract;
    procedure   MouseDown    ( aButton: TMouseButton; aShift: TShiftState; X, Y: Integer);                     override;
    procedure   MouseUp      ( aButton: TMouseButton; aShift: TShiftState; X, Y: Integer);                     override;
    procedure   MouseMove    ( aShift: TShiftState; X, Y: Integer);                                            override;
    procedure   Paint;                                                                                         override;
    procedure   TextChanged( const aNewValue: string; DoNotify: Boolean);                                       virtual;
    procedure   HandleTextChange( const aValue: string);                                                        virtual;
    function    FindWirePanel: TKnobsWirePanel;
    procedure   ValueChanged;
    procedure   CreateData;
    procedure   MouseToGrid( aMouseX, aMouseY: Integer; var aGridX, aGridY: Integer);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   FillRandom( anOnChance: TSignal);
    procedure   SaveToFile;
    procedure   LoadFromFile;
    procedure   ClearData;
    procedure   SetData( const aValue: TKnobsDataPlane);
    function    GetData: TKnobsDataPlane;
    procedure   BeginStateChange;
    procedure   EndStateChange;
  public
    property    ShortName             : string             read FShortName;
    property    CursorX               : TSignal            read GetCursorX         write SetCursorX;
    property    CursorY               : TSignal            read GetCursorY         write SetCursorY;
    property    CursorXY              : TSignalPair        read GetCursorXY        Write SetCursorXY;
    property    Pixel[ X, Y: Integer] : Boolean            read GetPixel           write SetPixel;
    property    AsString              : string             read GetAsString        write SetAsString;
  published
    property    Color                                                                                       default clGray;
    property    Width                                                                                       default 241;
    property    Height                                                                                      default 241;
    property    LineColor         : TColor                 read FLineColor         write SetLineColor       default clWhite;
    property    BorderColor       : TColor                 read FBorderColor       write SetBorderColor     default clGray;
    property    GridColor         : TColor                 read FGridColor         write SetGridColor       default clGray;
    property    CursorColor       : TColor                 read FCursorColor       write SetCursorColor     default CL_CURSOR;
    property    DotColor          : TColor                 read FDotColor          write SetDotColor        default CL_GRIDDOT;
    property    ActiveDotColor    : TColor                 read FActiveDotColor    write SetActiveDotColor  default CL_ACTIVEDOT;
    property    Transparent       : Boolean                read FTransparent       write SetTransparent     default True;
    property    ShowGrid          : Boolean                read FShowGrid          write SetShowGrid        default True;
    property    ShowCursorX       : Boolean                read FShowCursorX       write SetShowCursorX     default True;
    property    ShowCursorY       : Boolean                read FShowCursorY       write SetShowCursorY     default True;
    property    DataWidth         : Integer                read FDataWidth         write SetDataWidth       default 60;
    property    DataHeight        : Integer                read FDataHeight        write SetDataHeight      default 60;
    property    OnChanged         : TKnobsOnTextChanged    read FOnChanged         write FOnChanged;
    property    OnLoadGridControl : TOnLoadGridControl     read FOnLoadGridControl write FOnLoadGridControl;
    property    OnSaveGridControl : TOnSaveGridControl     read FOnSaveGridControl write FOnSaveGridControl;
  end;


  TKnobsGridControl = class( TKnobsGridProto)
  protected
    procedure   Clicked      ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   RightClicked ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   ClickEnded   ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   DoubleClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   MouseMoved   ( aShift: TShiftState; anX, anY: Integer);                                        override;
  end;


  TKnobsAutomationGrid = class( TKnobsGridProto)
  private
    FOnClick       : TOnKnobsGridClick;
    FOnDoubleClick : TOnKnobsGridClick;
    FOnRightClick  : TOnKnobsGridClick;
    FOnClickEnd    : TOnKnobsGridClick;
    FOnMouseMove   : TOnKnobsGridMouseMove;
  protected
    procedure   Clicked      ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   RightClicked ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   ClickEnded   ( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   DoubleClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer);                 override;
    procedure   MouseMoved   ( aShift: TShiftState; anX, anY: Integer);                                        override;
  published
    property    Align                                                                                    default alNone;
    property    OnClick       : TOnKnobsGridClick     read FOnClick       write FOnClick      ;
    property    OnDoubleClick : TOnKnobsGridClick     read FOnDoubleClick write FOnDoubleClick;
    property    OnRightClick  : TOnKnobsGridClick     read FOnRightClick  write FOnRightClick ;
    property    OnClickEnd    : TOnKnobsGridClick     read FOnClickEnd    write FOnClickEnd   ;
    property    OnMouseMove   : TOnKnobsGridMouseMove read FOnMouseMove   write FOnMouseMove  ;
  end;


  TKnobsConnector = class( TKnobsGraphicControl)
  // to connect wires to
  private
    FBitmapAudio           : TBitmap;
    FBitmapControl         : TBitmap;
    FBitmapLogic           : TBitmap;
    FBitmapControlLogic    : TBitmap;
    FSignalType            : TSignalType;
    FWireColor             : TColor;
    FBorderColor           : TColor;
    FUseBorders            : Boolean;
    FUseTypedBorders       : Boolean;
    FLinks                 : TList;   // Connections, not wires
    FMouseDown             : Boolean;
    FConnected             : Boolean; // True when at least one connection exists
    FHighLight             : Boolean; // Highlighted connections
    FCustomWireColor       : Boolean; // True when a custom wire color was set
    FAllowSignalConversion : Boolean; // True when this connector may be sped up by connecting a fast signal to it
    FIsSpedUp              : Boolean; // True when this connector was actually sped up
    FVisited               : Boolean;
    FIsMove                : Boolean; // True when the mouse action indicates a connection move (or delete) (instead of a make)
  private
    procedure   FixSpeedUp;
    procedure   SetConnected( aValue: Boolean);
    procedure   SetIsSpedUp ( aValue: Boolean);
    function    GetModuleName: string;
    function    GetDefaultWireColor: TColor;
    function    GetEffectiveColor: TColor;
    function    GetEffectiveSignalType: TSignalType;
    procedure   SetHighLight      ( aValue: Boolean);
    procedure   SetWireColor      ( aValue: TColor );
    procedure   SetBorderColor    ( aValue: TColor );
    procedure   SetUseBorders     ( aValue: Boolean);
    procedure   SetUseTypedBorders( aValue: Boolean);
    procedure   SetSignalType( aValue: TSignalType);
    procedure   SetAllowSignalConversion( aValue: Boolean);
  private
    function    FindWirePanel: TKnobsWirePanel;
    procedure   AddLink   ( aConnector: TKnobsConnector);
    procedure   DeleteLink( aConnector: TKnobsConnector);
    function    HasDirectlinkTo( aConnector: TKnobsConnector): Boolean;
    function    FindConnector( aScreenPoint: TPoint): TKnobsConnector;
    function    CenterPoint: TPoint;
    procedure   DrawLine( X, Y: Integer);
    procedure   SendWireNotification( Msg: Cardinal; aConnector: TKnobsConnector);
    procedure   SendColorNotification( aColor: TColor);
    procedure   SendHighlightNotification;
    procedure   ScrollMouseInView( anX, anY: LongInt);
  protected
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer);                                                 override;
    function    FindOutput: TKnobsConnector;
    function    IsConnectedToOutput: Boolean;
    procedure   FindDistanceToOutput( var aDistance: Integer);
    function    IsConnectedTo( aConnector: TKnobsConnector): Boolean;
    function    CanAccept( aSource: TKnobsConnector; isMove: Boolean): Boolean;
    procedure   AssignBitmap;                                                                         virtual; Abstract;
    procedure   MoveLinksTo   ( aConnector: TKnobsConnector);
    procedure   ConnectTo     ( aConnector: TKnobsConnector);
    procedure   DisconnectFrom( aConnector: TKnobsConnector);
    function    DisConnectAll: Boolean;
    procedure   DisConnectDownStream;
    procedure   DisConnectDownStreamRecursive;
    procedure   ChangeWireColor( aColor: TColor);
    function    SelectBitmap: TBitmap;
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   FixBitmaps;                                                                                     virtual;
    procedure   Dump     ( var aFile: TextFile; anInd: Integer);
    procedure   DumpFlat ( var aFile: TextFile; anInd: Integer);
    procedure   DumpLinks( var aFile: TextFile; anInd: Integer);
    procedure   Log( aLogClass: TLogClass; const aMsg: string);
    function    DistanceToOutput: Integer;
  protected
    property    HighLight             : Boolean            read FHighLight             write SetHighLight;
  public
    property    Connected             : Boolean            read FConnected             write SetConnected;
    property    ModuleName            : string             read GetModuleName;
    property    DefaultWireColor      : TColor             read GetDefaultWireColor;
    property    EffectiveColor        : TColor             read GetEffectiveColor;
    property    CustomWireColor       : Boolean            read FCustomWireColor;
    property    EffectiveSignalType   : TSignalType        read GetEffectiveSignalType;
    property    IsSpedUp              : Boolean            read FIsSpedUp              write SetIsSpedUp;
  published
    property    Signaltype            : TSignalType        read FSignalType            write SetSignalType            default stControl;
    property    WireColor             : TColor             read FWireColor             write SetWireColor             default clBlue;
    property    BorderColor           : TColor             read FBorderColor           write SetBorderColor           default clWhite;
    property    UseBorders            : Boolean            read FUseBorders            write SetUseBorders            default True;
    property    UseTypedBorders       : Boolean            read FUseTypedBorders       write SetUseTypedBorders       default False;
    property    AllowSignalConversion : Boolean            read FAllowSignalConversion write SetAllowSignalConversion default False;
    property    Visible;
    property    OnClick;
  end;

  TKnobsConnectors = array of TKnobsConnector;


  TKnobsInput = class( TKnobsConnector)
  // Specialized connector with it's own bitmap
  protected
    procedure   AssignBitmap;                                                                                  override;
  end;


  TKnobsOutput = class( TKnobsConnector)
  // Specialized connector with it's own bitmap
  protected
    procedure   AssignBitmap;                                                                                  override;
  end;


  TKnobsTextLabel = class( TCustomLabel)
  // Just a small print TLabel
  private
    FIsDisplay    : Boolean;
    FDisplayColor : TColor;
    FShortName    : string;
  private
    function    GetModule       : TKnobsCustomModule;
    function    GetModuleName   : string;
    procedure   SetIsDisplay    ( aValue: Boolean);
    procedure   SetDisplayColor ( aValue: TColor);
  protected
    procedure   SetName( const aNewName: TComponentName);                                                      override;
    procedure   DoDrawText( var aRect: TRect; aFlags: Longint);                                                override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  public
    property    ShortName     : string         read FShortName;
    property    ModuleName    : string         read GetModuleName;
  published
    property    IsDisplay     : Boolean        read FIsDisplay       write SetIsDisplay                   default false;
    property    DisplayColor  : TColor         read FDisplayColor    write SetDisplayColor                default clWhite;
    property    Align;
    property    Alignment;
    property    Anchors;
    property    AutoSize;
    property    BiDiMode;
    property    Caption;
    property    Color;
    property    Constraints;
    property    DragCursor;
    property    DragKind;
    property    DragMode;
    property    Enabled                                                                                   default False;
    property    FocusControl;
    property    Font;
    property    ParentBiDiMode;
    property    ParentColor;
    property    ParentFont;
    property    ParentShowHint                                                                            default False;
    property    PopupMenu;
    property    ShowAccelChar;
    property    ShowHint                                                                                  default False;
    property    Transparent;
    property    Layout;
    property    Visible;
    property    WordWrap;
    property    OnClick;
  end;


  TKnobsLabelEditor = class( TEdit)
  // An editor for TEditLabel
  private
    FLabel : TKnobsEditLabel;
  private
    procedure   WMKillFocus( var aMessage: TWMKillFocus );                                         message WM_KILLFOCUS;
    procedure   CNKeyDown(var Message: TWMKeyDown);                                                message   CN_KEYDOWN;
    procedure   Do_Exit   ( aSender: TObject);
    procedure   Do_KeyDown( aSender: TObject; var aKey: Word; aShift: TShiftState);
  protected
    procedure   SetLabel( aLabel: TKnobsEditLabel);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  end;


  TKnobsChoiceSelector = class( TComboBox)
  private
    FChoice : TKnobsSelectorChoice;
  private
    procedure   WMKillFocus( var aMessage: TWMKillFocus );                                         message WM_KILLFOCUS;
    procedure   Do_Exit( aSender: TObject);
  protected
    procedure   SetSelectorChoise( aChoice: TKnobsSelectorChoice);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  end;


  TKnobsEditLabel = class( TKnobsTextLabel)
  // a label with a popup editor so it's caption can be edited.
  private
    FOnChanged : TKnobsOnEditLabelChanged;
  private
    function    GetModule: TKnobsCustomModule;
  protected
    procedure   CMHintShow(var Message: TCMHintShow);                                               message CM_HINTSHOW;
    procedure   SetText( const aValue: string);
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  public
    property    Module: TKnobsCustomModule read GetModule;
  published
    property    Enabled                                                                                    default True;
    property    OnChanged: TKnobsOnEditLabelChanged read FOnChanged write FOnChanged;
  end;


  TKnobsBox = class( TBevel)
  // A bevel that can be disabled
  private
    function    GetModule: TKnobsCustomModule;
  private
    function    GetModuleName   : string;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  public
    property    ModuleName    : string         read GetModuleName;
  published
    property    Enabled                                                                                   default False;
    property    OnClick;
  end;


  TKnobsWire = class
  // Wire, visual aspect of a connection
  private
    FColor        : TColor;
    FSource       : TKnobsConnector;
    FDestination  : TKnobsConnector;
    FHighlight    : Boolean;
    FBezierParams : TKnobsBezierParams;
  private
    function    GetSourceName        : string;
    function    GetDestinationName   : string;
    function    GetSourceModule      : TKnobsCustomModule;
    function    GetDestinationModule : TKnobsCustomModule;
    function    GetFullName          : string;
  public
    constructor Create( aSource, aDestination: TKnobsConnector);
    procedure   Wiggle;
    procedure   Dump( var aFile: TextFile; anInd: Integer);
    procedure   FixWire;
    procedure   SwapConnectors;
    function    Involves( aModule: TKnobsCustomModule): Boolean;
    function    AsString: string;
    function    AsCompactString: string;
  public
    property    Source            : TKnobsConnector    read FSource;
    property    Destination       : TKnobsConnector    read FDestination;
    property    SourceName        : string             read GetSourceName;
    property    DestinationName   : string             read GetDestinationName;
    property    SourceModule      : TKnobsCustomModule read GetSourceModule;
    property    DestinationModule : TKnobsCustomModule read GetDestinationModule;
    property    BezierParams      : TKnobsBezierParams read FBezierParams;
    property    FullName          : string             read GetFullName;
  end;


  TKnobsWireOverlay = class( TCustomControl)
  //
  // Responsibilty to maintain connections is assumed by the TConnectors
  // themselves. Wires are maintained here, but it's a cache really to ease
  // wire painting. The wires are maintained by the connectors theselves that
  // send their info to us through the WirePanel (by means of UM_CONNEXTIONxx
  // and UM_HIGHLIGHT messages). The WirePanel imforms us by calling the
  // AddWire, DelWire and Highlight methodes.
  //
  // All Wires are owned by FWires, so our destructor will eventually destroy
  // all dangling wires. Nevertheless, TConnector still must, on destruction,
  // clean up the wires it is responsible for .. it wouldn't look good other
  // wise in the user interface.
  //
  private
    FWires         : TObjectList;
    FCurvedLines   : Boolean;
    FWireThickness : Integer;
    FWiresVisible  : Boolean;
    FBlockCount    : Integer;
    FPaintCount    : Integer;
    FPaintClocks   : Int64;
  private
    procedure   WMNCHitTest( var aMessage: TWMNCHitTest);                                          message WM_NCHITTEST;
  private
    function    GetWire         ( anIndex: Integer): TKnobsWire;
    function    GetSource       ( anIndex: Integer): TKnobsConnector;
    function    GetDestination  ( anIndex: Integer): TKnobsConnector;
    procedure   SetCurvedLines  ( aValue: Boolean );
    procedure   SetWireThickNess( aValue: Integer );
    procedure   SetWiresVisible ( aValue: Boolean );
  protected
    procedure   Paint;                                                                                         override;
    procedure   PaintWire( aDC: HDC; aWire: TKnobsWire);
    function    FindWire ( aSource, aDestination: TKnobsConnector): Integer;
    procedure   AddWire  ( aSource, aDestination: TKnobsConnector);
    procedure   DelWire  ( aSource, aDestination: TKnobsConnector);
    procedure   Connect  ( aSource, aDestination: TKnobsConnector);
    procedure   HandleHighlight;
    procedure   HandleColorChange( aSource: TKnobsConnector; aColor: TColor);
    function    WireCount: Integer;
    procedure   WiggleWires;
    procedure   ToggleWires;
    function    WiresOff: Boolean;
    procedure   WiresOn( TurnOn: Boolean);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   InvalidateWires;
    function    Disconnect( aConnector: TKnobsConnector): Boolean;
    procedure   Dump( var aFile: TextFile; anInd: Integer);
    procedure   Log( aLogClass: TLogClass; const aMsg: string);
    procedure   BlockUpdates;
    procedure   UnBlockUpdates;
    procedure   FixAllWires;
  public
    property    Wire       [ anIndex: Integer]: TKnobsWire      read GetWire;
    property    Source     [ anIndex: Integer]: TKnobsConnector read GetSource;
    property    Destination[ anIndex: Integer]: TKnobsConnector read GetDestination;
  public
    property    CurvedLines      : Boolean read FCurvedLines   write SetCurvedLines                        default True;
    property    WireThickness    : Integer read FWireThickness write SetWireThickness                         nodefault;
    property    WiresVisible     : Boolean read FWiresVisible  write SetWiresVisible                       default True;
    property    BlockCount       : Integer read FBlockCount;
    property    PaintCount       : Integer read FPaintCount;
    property    PaintClocks      : Int64   read FPaintClocks;
  published
    property    Left                                                                                       stored False;
    property    Top                                                                                        stored False;
    property    Width                                                                                      stored False;
    property    Height                                                                                     stored False;
  end;


  TKnobsModuleList = class( TList)
  // List of TCustomModule - some comon ops defined here
  private
    FonLog     : TKnobsOnLog;
    FUseIndex  : Boolean;
    FIndex     : TStringList;
    FClassList : TClassList;
  private
    function    GetModule      (       anIndex: Integer): TKnobsCustomModule;
    function    GetModuleByName( const anIndex: string ): TKnobsCustomModule;
    function    GetSelected( anIndex: Integer): Boolean;
    procedure   SetSelected( anIndex: Integer; aValue: Boolean);
    function    GetAnySelected: Boolean;
    function    GetNoneSelected: Boolean;
  public
    constructor Create( aUseIndex: Boolean);
    destructor  Destroy;                                                                                       override;
    procedure   BuildIndex;
    function    FindModule( const aModule: TKnobsCustomModule): Integer;                                       overload;
    function    FindModule( const aName : string): TKnobsCustomModule;                                         overload;
    procedure   FreeModule( anIndex: Integer; const aCallback: TKnobsOnValuedCtrlRemoved);                     overload;
    procedure   FreeModule( var aModule: TKnobsCustomModule; const aCallback: TKnobsOnValuedCtrlRemoved);      overload;
    function    FreeModules( const aCallback: TKnobsOnValuedCtrlRemoved): Boolean;
    function    FreeSelectedModules( const aCallback: TKnobsOnValuedCtrlRemoved): Boolean;
    procedure   SelectAll;
    procedure   UnselectAll;
    procedure   SelectUnique( aModule: TKnobsCustomModule);
    procedure   FixSelection( aModule: TKnobsCustomModule);
    procedure   SetSelectedModuleColor( aValue: TColor);
    procedure   SetSelectedModuleColorDefault( const aColorFunc: TOnGetModuleColor);
    procedure   InvertSelection;
    procedure   InvalidateDisplays;
  public
    //* Some helpers for selector population and component registration
    function    FindType( aModuleType: TKnobsModuleType): TKnobsCustomModule;
    procedure   PopulateSelector( aModuleSelector: TKnobsModuleSelector; const aBitmapsFolder: string);
    procedure   UnPopulateSelector( aModuleSelector: TKnobsModuleSelector);
    function    CreateModule( aWirePanel: TKnobsWirePanel; aModuleType: TKnobsModuleType; MustDrag, StandardColor: Boolean): TKnobsCustomModule;
    function    ChangeModule( aWirePanel: TKnobsWirePanel; var aTarget: TKnobsCustomModule; aWantedType: TKnobsModuleType; aCallBack: TKnobsOnValuedCtrlRemoved): TKnobsCustomModule;
    procedure   CollectModuleComponents( const aModule: TKnobsCustomModule);
    procedure   RegisterClassList( doRegister: Boolean);
    procedure   RegisterComponents( doRegister: Boolean);
    procedure   Dump( var aFile: TextFile; anInd: Integer);
    procedure   Log( aLogClass: TLogClass; const aMsg: string);
  public
    property    UseIndex                         : Boolean            read FUseIndex;
    property    Module  [ anIndex: Integer]      : TKnobsCustomModule read GetModule;                           default;
    property    Selected[ anIndex: Integer]      : Boolean            read GetSelected     write SetSelected;
    property    ByName  [ const anIndex: string] : TKnobsCustomModule read GetModuleByName;
    property    AnySelected                      : Boolean            read GetAnySelected;
    property    NoneSelected                     : Boolean            read GetNoneSelected;
    property    OnLog                            : TKnobsOnLog        read FOnLog          write FOnLog;
  end;


  TKnobsHistoryItemKind = (
    hikPatch       ,
    hikSignal      ,
    hikStringValue
  );

  TKnobsHistoryItem = record
  private
    FKind  : TKnobsHistoryItemKind;
    FValue : TValue;
  public
    procedure   RetrievalError( WantedType: TKnobsHistoryItemKind);
    procedure   InitAsPatch      ( const aValue : TKnobsWirePanel);
    procedure   InitAsSignal     ( const aValue : TSignal        );
    procedure   InitAsStringValue( const aValue : string         );
    function    Kind: TKnobsHistoryItemKind;
    procedure   GetAsPatch( const aValue: TKnobsWirePanel);
    function    GetAsSignal: TSignal;
    function    GetAsStringValue: string;
    function    IsKind( aValue: TKnobsHistoryItemKind): Boolean;
    function    IsSameAs( const aHistoryItem: TKnobsHistoryItem): Boolean;
    function    Description: string;
  end;


  TKnobsHistoryStack = class( TList<TKnobsHistoryItem>)
  // A stack of wirepanels for undo / redo
  private
    FMaxHistory : Integer;
  private
    procedure   SetMaxHistory( aValue: Integer);
  public
    procedure   PushPatch ( const aWirePanel: TKnobsWirePanel);
    procedure   PopPatch  ( const aWirePanel: TKnobsWirePanel);
    procedure   PushString( const aValue: string);
    function    PopString: string;
    procedure   PushSignal( aSignal: TSignal);
    function    PopSignal : TSignal;
    function    CreateDescription( aMaxCount: Integer): TStringList;
  public
    property    MaxHistory : Integer read FMaxHistory write SetMaxHistory;
  end;


  TKnobsUndoRedoHistory = class( TObject)
  // A history of wirepanels for undo / redo
  private
    FOnHistoryChange : TKnobsOnHistoryChange;
    FMaxHistory      : Integer;
    FLockCount       : Integer;
    FUndos           : TKnobsHistoryStack;
    FRedos           : TKnobsHistoryStack;
    FSavedMarker     : Integer;
  private
    function    GetLocked: Boolean;
    function    GetUndoCount: Integer;
    function    GetRedoCount: Integer;
    procedure   SetmaxHistory( aValue: Integer);
  private
    procedure   Lock;
    function    UnLock: Boolean;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   HistoryChanged;
    procedure   SaveState( const aWirePanel: TKnobsWirePanel);
    procedure   Undo     ( const aWirePanel: TKnobsWirePanel);
    procedure   Redo     ( const aWirePanel: TKnobsWirePanel);
    procedure   Clear;
    procedure   SetSavedMarker;
  public
    property    Locked          : Boolean               read GetLocked;
    property    UndoCount       : Integer               read GetUndoCount;
    property    RedoCount       : Integer               read GetRedoCount;
    property    MaxHistory      : Integer               read FMaxHistory      write SetmaxHistory;
  public
    property    OnHistoryChange : TKnobsOnHistoryChange read FOnHistoryChange write FOnHistoryChange;
  end;


  TKnobsWirePanel = class( TScrollingWinControl, IKnobsUndoable)
  // Contains the modules and the wire overlay - scrollable.
  private
    FEditHistory               : TKnobsEditHistory;       // A small local undo history
    FGuid                      : TGUID;
    FOffsetLeft                : Integer;
    FOffsetTop                 : Integer;
    FTitle                     : string;
    FVersion                   : Integer;
    FMouseDown                 : Boolean;
    FWireOverlay               : TKnobsWireOverlay;
    FModules                   : TKnobsModuleList;
    FFilename                  : string;
    FSnapX                     : Integer;
    FSnapY                     : Integer;
    FControlMode               : TDistanceMode;
    FWheelSupportOnKnobs       : Boolean;
    FWheelSensitivity          : Integer;
    FModuleOpacity             : Byte;
    FModuleFlatness            : Boolean;
    FModuleTexture             : Boolean;
    FUseTitleHints             : Boolean;
    FUseConnectorBorders       : Boolean;
    FUseTypedConnectorBorders  : Boolean;
    FModuleColor               : TColor;
    FConnectorBorderColor      : TColor;
    FSelectorColor             : TColor;
    FSelectorBorderColor       : TColor;
    FSelectorBorderColorSingle : TColor;
    FDisplayColor              : TColor;
    FDisplayBorderColor        : TColor;
    FKnobMIDIColor             : TColor;
    FKnobFocusColor            : TColor;
    FKnobMIDIFocusColor        : TColor;
    FViewerColor               : TColor;
    FViewerBorderColor         : TColor;
    FViewerLineColor           : TColor;
    FViewerFillColor           : TColor;
    FIndBarPeakColor           : TColor;
    FIndBarValeyColor          : TColor;
    FSnapActive                : Boolean;
    FAllowRandomization        : Boolean;
    FShowAllowRandomization    : Boolean;
    FWormUpdatesBlocked        : Integer;
    FEditLockCounter           : Integer;
    FPatchReader               : IKnobsPatchReader;
    FPatchWriter               : IKnobsPatchWriter;
    FOnLog                     : TKnobsOnLog;
    FOnHistoryChange           : TKnobsOnHistoryChange;
    FHistory                   : TKnobsUndoRedoHistory;
    FActiveVariation           : Integer;
    FActiveRange               : Integer;
    FOnValueChanged            : TKnobsOnValueChange;
    FOnTextChanged             : TKnobsOnTextChanged;
    FOnRecompile               : TKnobsOnRecompile;
    FOnWireAdded               : TKnobsOnWireAdded;
    FOnWireRemoved             : TKnobsOnWireRemoved;
    FOnShowPopup               : TOnShowPopup;
    FOnUnFocus                 : TOnUnFocus;
    FOnLoadDataGraph           : TOnLoadDataGraph;
    FOnSaveDataGraph           : TOnSaveDataGraph;
    FOnLoadGridControl         : TOnLoadGridControl;
    FOnSaveGridControl         : TOnSaveGridControl;
    FOnActiveVariationChanged  : TOnActiveVariationChanged;
    FOnLoadCaptions            : TOnLoadCaptions;
    FOnEditHistoryChanged      : TNotifyEvent;
    FOnModulesRenamed          : TOnModulesRenamed;
    FOnModulesRenamedFriendly  : TOnModulesRenamed;
    FOnAddEditData             : TOnAddEditData;
  private
    procedure   UMConnectionAdd   ( var aMessage: TKnobsUMConnectionAdd  );           message    KNOBS_UM_CONNECTIONADD;
    procedure   UMConnectionRemove( var aMessage: TKnobUMConnectionRemove);           message KNOBS_UM_CONNECTIONREMOVE;
    procedure   UMColorChange     ( var aMessage: TKnobsUMColorChange    );           message      KNOBS_UM_COLORCHANGE;
    procedure   UMHighlight       ( var aMessage: TKnobsUMHighlight      );           message        KNOBS_UM_HIGHLIGHT;
  private
    function    GetModule          ( anIndex: Integer): TKnobsCustomModule;
    function    GetWire            ( anIndex: Integer): TKnobsWire;
    function    GetSource          ( anIndex: Integer): TKnobsConnector;
    function    GetDestination     ( anIndex: Integer): TKnobsConnector;
  private
    function    GetGuid                     : string;
    procedure   SetGuid                     ( const aValue: string);
    procedure   SetOffsetLeft               ( aValue: Integer);
    procedure   SetOffsetTop                ( aValue: Integer);
    function    GetCurvedLines              : Boolean;
    procedure   SetCurvedLines              ( aValue: Boolean);
    function    GetWireThickness            : Integer;
    procedure   SetWireThickness            ( aValue: Integer);
    function    GetModuleCount              : Integer;
    function    GetWireCount                : Integer;
    function    GetMaxHistory               : Integer;
    procedure   SetMaxHistory               ( aValue: Integer);
    function    GetBlockCount               : Integer;
    function    GetPaintCount               : Integer;
    function    GetPaintClocks              : Int64;
    function    GetScrollOffset             : TPoint;
    function    GetVariationCount           : Integer;
    procedure   SetControlMode              ( aValue: TDistanceMode);
    procedure   SetModuleOpacity            ( aValue: Byte);
    procedure   SetModuleFlatness           ( aValue: Boolean);
    procedure   SetModuleTexture            ( aValue: Boolean);
    procedure   SetUseTitleHints            ( aValue: Boolean);
    procedure   SetUseConnectorBorders      ( aValue: Boolean);
    procedure   SetUseTypedConnectorBorders ( aValue: Boolean);
    procedure   SetModuleColor              ( aValue: TColor);
    procedure   SetConnectorBorderColor     ( aValue: TColor);
    procedure   SetSelectorColor            ( aValue: TColor);
    procedure   SetSelectorBorderColor      ( aValue: TColor);
    procedure   SetSelectorBorderColorSingle( aValue: TColor);
    procedure   SetDisplayColor             ( aValue: TColor);
    procedure   SetDisplayBorderColor       ( aValue: TColor);
    procedure   SetKnobMIDIColor            ( aValue: TColor);
    procedure   SetKnobFocusColor           ( aValue: TColor);
    procedure   SetKnobMIDIFocusColor       ( aValue: TColor);
    procedure   SetViewerColor              ( aValue: TColor);
    procedure   SetViewerBorderColor        ( aValue: TColor);
    procedure   SetViewerLineColor          ( aValue: TColor);
    procedure   SetViewerFillColor          ( aValue: TColor);
    procedure   SetIndBarPeakColor          ( aValue: TColor);
    procedure   SetIndBarValeyColor         ( aValue: TColor);
    procedure   SetWheelSupportOnKnobs      ( aValue: Boolean);
    procedure   SetWheelSensitivity         ( aValue: Integer);
    procedure   SetAllowRandomization       ( aValue: Boolean);
    procedure   SetShowAllowRandomization   ( aValue: Boolean);
    procedure   SetActiveVariation          ( aValue: Integer);
    procedure   SetActiveRange              ( aValue: Integer);
    procedure   SetRangeValue               ( aValue: TSignal);
  private
    procedure   StrToConnection( const S: string);
  protected
    procedure   HandleRightClick( const aSender: TControl);
    procedure   CreateParams(var Params: TCreateParams);                                                       override;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer);                                                 override;
    procedure   DragStart( aPoint: TPoint);                                                                     virtual;
    procedure   DragEnd  ( aPoint: TPoint);                                                                     virtual;
    procedure   DragMove ( aPoint: TPoint);                                                                     virtual;
    procedure   SelectModulesInRect( aRect: TRect);
    function    ModuleIndex( aModule: TKnobsCustomModule): Integer;
    procedure   Notification( aComponent: TComponent; anOperation: TOperation);                                override;
    procedure   Connect( aSource, aDestination: TKnobsConnector);
    procedure   Restack( DoRestack, RedrawWires: Boolean);
    procedure   Resnap;
    procedure   FixNames( aStart: Integer);
    procedure   SolveNamingConflictsWith( aDstWirePanel: TKnobsWirePanel);
    function    Snap( const aPoint: TPoint): TPoint;
    procedure   DoHistoryChanged( const aSender: TObject; anUndoCount, aRedoCount, aSavedMarker: Integer);
    procedure   ValueChanged( aSender: TObject; const aPath, aControlType: string; aValue: TSignal; IsFinal, IsAutomation: Boolean); virtual;
    procedure   TextChanged ( aSender: TObject; const aPath, aValue: string);
    procedure   Recompile( ModulesChanged: Boolean);
    procedure   ScrollPointInView( aPoint: TPoint);                                                            overload;
    procedure   ScrollPointInView( anX, anY: LongInt);                                                         overload;
    function    LoadDataGraph: string;
    procedure   SaveDataGraph( const aValue: string);
    function    LoadGridControl: string;
    procedure   SaveGridControl( const aValue: string);
    procedure   LoadCaptionsFor( const aModule: TKnobsCustomModule; const aControl: TKnobsSelector; const aDependency: TKnobsValuedControl);
    procedure   AcceptAutomationData( const aSender: TKnobsValuedControl; const aData: PKnobsEditData);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   BuildModuleIndex;
    procedure   PrepareGraph;
    procedure   SortGraph;
    procedure   DumpGraph( const aStrings: TStringList);                                                       overload;
    procedure   DumpGraph( const aFileName: string);                                                           overload;
    procedure   FixAllSynthParams( IsAutomation: Boolean);
    procedure   BeginStateChange( IsStructural: Boolean);
    procedure   EndStateChange( MustRecompile, ModulesChanged: Boolean);
    procedure   Undo;
    procedure   Redo;
    procedure   ClearUndoRedo;
    procedure   SetSavedMarker;
    procedure   Dump( const aFileName: string; DoAppend: Boolean);                                             overload;
    procedure   Dump( var aFile: TextFile);                                                                    overload;
    procedure   DumpModules( var aFile: TextFile; anInd: Integer);
    procedure   DumpWires( var aFile: TextFile; anInd: Integer);
    procedure   Log( aLogClass: TLogClass; const aMsg: string);
    procedure   LogFmt( aLogClass: TLogClass; const aFmt: string; const anArgs: array of const);
    function    FindModule( const aName: string): TKnobsCustomModule;
    function    FindFirstSelectedModule: TKnobsCustomModule;
    function    GetModuleAlternateTypes( aModule: TKnobsCustomModule): string;
  public
    procedure   SetAllowAllRandomization( aValue: Boolean);
  public
    procedure   InvalidateWires;
    procedure   InvalidateDisplays;
    procedure   Disconnect( aConnector: TKnobsConnector);
    procedure   FreeModule( aModule: TKnobsCustomModule; const aCallback: TKnobsOnValuedCtrlRemoved);
    procedure   FreeModules( const aCallback: TKnobsOnValuedCtrlRemoved);
    procedure   FreeSelectedModules( const aCallback: TKnobsOnValuedCtrlRemoved);
    function    HasEditPopup: Boolean;
    procedure   SetSelectedModuleColor( aValue: TColor);
    procedure   SetSelectedModuleColorDefault( const aColorFunc: TOnGetModuleColor);
    procedure   SelectAllModules;
    procedure   UnSelectAllModules;
    procedure   SelectUnique( aModule: TKnobsCustomModule);
    procedure   FixSelection( aModule: TKnobsCustomModule);
    procedure   AddToSelection( const aModuleName: string);
    procedure   InvertModuleSelection;
    function    FindTitle( const aModuleName: string): string;
    function    CopyToString( aWriteMode: TKnobsWriteMode): string;
    procedure   CopyFromString( const S: string; CopyMidiCC: Boolean; aReadMode: TKnobsReadMode);
    procedure   CopyTo( aPanel: TKnobsWirePanel; CopyMidiCC: Boolean; aWriteMode: TKnobsWriteMode; aReadMode: TKnobsReadMode);
    function    SameStructure( aPanel: TKnobsWirePanel): Boolean;
    function    CopyParamsFrom( aSrc: TKnobsWirePanel; CopyStrict: Boolean): Boolean;
    procedure   AddModules( aSrc: TKnobsWirePanel; aDistance: TPoint; DoRestack, RedrawWires, MustDrag, CopyMidiCC: Boolean); virtual;
    procedure   AddWire( const aSrc, aDst: string);                                                            overload;
    procedure   AddWire( const aSrcDst: string);                                                               overload;
    procedure   WiggleWires;
    procedure   ToggleWires;
    function    WiresOff: Boolean;
    procedure   WiresOn( TurnOn: Boolean);
    procedure   FixAllWires;
    procedure   FixUserSettings;
    procedure   EditHistoryChanged;
    procedure   ClearHistory;
    procedure   FixBitmaps;
    procedure   BlockWireUpdates;
    procedure   UnblockWireUpdates;
    procedure   Search( const aValue: string);
    procedure   FindUnconnectedModules;
    procedure   UnFocus( const aSender: TObject);
    procedure   FindGhosts( const aGhosts: TStringList);
    function    CountGhosts: Integer;
    procedure   SetOnClickFor( aComponentName: string; aUserId: Integer; aHandler: TNotifyEvent);
  public
    procedure   ConnectorDelete    ( const aConnector: TKnobsConnector);
    procedure   ConnectorBreak     ( const aConnector: TKnobsConnector);
    procedure   ConnectorDisconnect( const aConnector: TKnobsConnector);
    procedure   SetWireColor( const aConnector: TKnobsConnector; const aColor: TColor);
    procedure   SetWireColorDefault( const aConnector: TKnobsConnector);
   function     VisitControls( aModuleType: TKnobsModuleType; aControlType: TControlClass; const aName: string; const aHandler: TKnobsOnVisitControl; const aUserData: TObject): Boolean;
    procedure   SetTitleFont ( const aValue: TFont);
    procedure   SetModuleFont( const aValue: TFont);
    procedure   CollectCCControls( const aList: TStringList);
    function    CreateMidiCCList: TStringList;
    procedure   UnassignMidiCC( aMidiCC: Byte);
    procedure   SetRandomValue( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   SetRandomValue( anAmount: TSignal);                                                            overload;
    procedure   Randomize     ( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   Randomize     ( anAmount: TSignal);                                                            overload;
    procedure   Mutate        ( aProb, aRange: TSignal; aVariation: Integer);                                  overload;
    procedure   Mutate        ( aProb, aRange: TSignal);                                                       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer);       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);                   overload;
    procedure   Mate          ( anXProb, aMutProb, aMutRange: TSignal);
    procedure   Morph         ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                           overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom: Integer);                                       overload;
    procedure   Morph;                                                                                         overload;
    procedure   LiveMorph     ( anAmount: TSignal);
    procedure   SyncControlRandomization( const aControlTypeName: string; aValue: Boolean);
    procedure   SyncModuleRandomization ( aModuleType: TKnobsModuleType; aValue: Boolean);
    procedure   CountGene( var aCount: Integer);
    procedure   FillGene       ( const aGene : TKnobsGene; aVariation: Integer);
    procedure   FillWorm       ( const aWorm : TKnobsWorm; aVariation: Integer);
    procedure   FillWormPanel  ( const aPanel: TKnobsWormPanel);                                               overload;
    procedure   FillWormPanel  ( const aPanel: TKnobsWormPanel; aVariations: TKnobsVariationSet);              overload;
    procedure   AcceptGene     ( const aGene : TKnobsGene; aVariation: Integer);
    function    EditChangesLocked: Boolean;
    procedure   LockEditChanges;
    procedure   UnlockEditChanges;
    procedure   CreateValueSnapShot( const aHistory: TKnobsEditHistory);
    procedure   FixActiveRange  ( aValue: Integer);
    procedure   SetValueForRange( aRange: Integer; aValue: TSignal);
  public
    property    EditHistory      : TKnobsEditHistory                  read FEditHistory       implements IKnobsUndoable;
    property    PatchReader      : IKnobsPatchReader                  read FPatchReader               write FPatchReader;
    property    PatchWriter      : IKnobsPatchWriter                  read FPatchWriter               write FPatchWriter;
  public
    property    Module     [ anIndex: Integer]: TKnobsCustomModule    read GetModule;
    property    Wire       [ anIndex: Integer]: TKnobsWire            read GetWire;
    property    Source     [ anIndex: Integer]: TKnobsConnector       read GetSource;
    property    Destination[ anIndex: Integer]: TKnobsConnector       read GetDestination;
  public
    property    OnHistoryChange           : TKnobsOnHistoryChange     read FOnHistoryChange           write FOnHistoryChange;
    property    OnLog                     : TKnobsOnLog               read FOnLog                     write FonLog;
    property    Filename                  : string                    read FFilename                  write FFilename;
    property    ModuleCount               : Integer                   read GetModuleCount;
    property    WireCount                 : Integer                   read GetWireCount;
    property    BlockCount                : Integer                   read GetBlockCount;
    property    PaintCount                : Integer                   read GetPaintCount;
    property    PaintClocks               : Int64                     read GetPaintClocks;
    property    ScrollOffset              : TPoint                    read GetScrollOffset;
    property    ActiveVariation           : Integer                   read FActiveVariation           write SetActiveVariation;
    property    VariationCount            : Integer                   read GetVariationCount;
    property    ActiveRange               : Integer                   read FActiveRange               write SetActiveRange;
    property    RangeValue                : TSignal                                                   write SetRangeValue;
    property    Guid                      : string                    read GetGuid                    write SetGuid;
  published
    property    MaxHistory                : Integer                   read GetMaxHistory              write SetMaxHistory                default 100;
    property    OnValueChanged            : TKnobsOnValueChange       read FOnValueChanged            write FOnValueChanged;
    property    OnTextChanged             : TKnobsOnTextChanged       read FOnTextChanged             write FOnTextChanged;
    property    OnRecompile               : TKnobsOnRecompile         read FOnRecompile               write FOnRecompile;
    property    OnWireAdded               : TKnobsOnWireAdded         read FOnWireAdded               write FOnWireAdded;
    property    OnWireRemoved             : TKnobsOnWireRemoved       read FOnWireRemoved             write FOnWireRemoved;
    property    OnShowPoup                : TOnShowPopup              read FOnShowPopup               write FOnShowPopup;
    property    OnUnFocus                 : TOnUnFocus                read FOnUnFocus                 write FOnUnFocus;
    property    OnLoadDataGraph           : TOnLoadDataGraph          read FOnLoadDataGraph           write FOnLoadDataGraph;
    property    OnSaveDataGraph           : TOnSaveDataGraph          read FOnSaveDataGraph           write FOnSaveDataGraph;
    property    OnLoadGridControl         : TOnLoadGridControl        read FOnLoadGridControl         write FOnLoadGridControl;
    property    OnSaveGridControl         : TOnSaveGridControl        read FOnSaveGridControl         write FOnSaveGridControl;
    property    OnActiveVariationChanged  : TOnActiveVariationChanged read FOnActiveVariationChanged  write FOnActiveVariationChanged;
    property    OnLoadCaptions            : TOnLoadCaptions           read FOnLoadCaptions            write FOnLoadCaptions;
    property    OnEditHistoryChanged      : TNotifyEvent              read FOnEditHistoryChanged      write FOnEditHistoryChanged;
    property    OnModulesRenamed          : TOnModulesRenamed         read FOnModulesRenamed          write FOnModulesRenamed;
    property    OnModulesRenamedFriendly  : TOnModulesRenamed         read FOnModulesRenamedFriendly  write FOnModulesRenamedFriendly;
    property    OnAddEditData             : TOnAddEditData            read FOnAddEditData             write FOnAddEditData;
    property    OffsetLeft                : Integer                   read FOffsetLeft                write SetOffsetLeft                default 4;
    property    OffsetTop                 : Integer                   read FOffsetTop                 write SetOffsetTop                 default 4;
    property    CurvedLines               : Boolean                   read GetCurvedLines             write SetCurvedLines               default True;
    property    WireThickness             : Integer                   read GetWireThickness           write SetWireThickness             default 1;
    property    Title                     : string                    read FTitle                     write FTitle;
    property    Version                   : Integer                   read FVersion                   write FVersion;
    property    SnapX                     : Integer                   read FSnapX                     write FSnapX                       default MOD_Y_UNIT;
    property    SnapY                     : Integer                   read FSnapY                     write FSnapY                       default MOD_X_UNIT;
    property    SnapActive                : Boolean                   read FSnapActive                write FSnapActive                  default True;
    property    AllowRandomization        : Boolean                   read FAllowRandomization        write SetAllowRandomization        default False;
    property    ShowAllowRandomization    : Boolean                   read FShowAllowRandomization    write SetShowAllowRandomization    default False;
    property    ControlMode               : TDistanceMode             read FControlMode               write SetControlMode               default dmCircular;
    property    WheelSupportOnKnobs       : Boolean                   read FWheelSupportOnKnobs       write SetWheelSupportOnKnobs;
    property    WheelSensitivity          : Integer                   read FWheelSensitivity          write SetWheelSensitivity          default 50;
    property    ModuleOpacity             : Byte                      read FModuleOpacity             write SetModuleOpacity             default 223;
    property    ModuleFlatness            : Boolean                   read FModuleFlatness            write SetModuleFlatness            default False;
    property    ModuleTexture             : Boolean                   read FModuleTexture             write SetModuleTexture             default True;
    property    UseTitleHints             : Boolean                   read FUseTitleHints             write SetUseTitleHints             default True;
    property    UseConnectorBorders       : Boolean                   read FUseConnectorBorders       write SetUseConnectorBorders       default True;
    property    UseTypedConnectorBorders  : Boolean                   read FUseTypedConnectorBorders  write SetUseTypedConnectorBorders  default False;
    property    ModuleColor               : TColor                    read FModuleColor               write SetModuleColor               default clWhite;
    property    ConnectorBorderColor      : TColor                    read FConnectorBorderColor      write SetConnectorBorderColor      default clWhite;
    property    SelectorColor             : TColor                    read FSelectorColor             write SetSelectorColor             default clGray;
    property    SelectorBorderColor       : TColor                    read FSelectorBorderColor       write SetSelectorBorderColor       default clYellow;
    property    SelectorBorderColorSingle : TColor                    read FSelectorBorderColorSingle write SetSelectorBorderColorSingle default clWhite;
    property    DisplayColor              : TColor                    read FDisplayColor              write SetDisplayColor              default clGray;
    property    DisplayBorderColor        : TColor                    read FDisplayBorderColor        write SetDisplayBorderColor        default clSilver;
    property    KnobMIDIColor             : TColor                    read FKnobMIDIColor             write SetKnobMIDIColor             default CL_MIDI;
    property    KnobFocusColor            : TColor                    read FKnobFocusColor            write SetKnobFocusColor            default CL_FOCUS;
    property    KnobMIDIFocusColor        : TColor                    read FKnobMIDIFocusColor        write SetKnobMIDIFocusColor        default CL_MIDI_FOCUS;
    property    ViewerColor               : TColor                    read FViewerColor               write SetViewerColor               default clGray;
    property    ViewerBorderColor         : TColor                    read FViewerBorderColor         write SetViewerBorderColor         default clGray;
    property    ViewerLineColor           : TColor                    read FViewerLineColor           write SetViewerLineColor           default clWhite;
    property    ViewerFillColor           : TColor                    read FViewerFillColor           write SetViewerFillColor           default $009E9E9E;
    property    IndBarPeakColor           : TColor                    read FIndBarPeakColor           write SetIndBarPeakColor           default clWhite;
    property    IndBarValeyColor          : TColor                    read FIndBarValeyColor          write SetIndBarValeyColor          default clBlack;
    property    Color                                                                                                                    default $00313131;
    property    Align;
    property    Anchors;
    property    Constraints;
    property    ParentColor;
    property    ParentShowHint                                                                                                           default False;
    property    ParentBackground;
    property    PopupMenu;
    property    ShowHint                                                                                                                 default False;
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    Top;
    property    Left;
    property    Width;
    property    Height;
    property    HorzScrollBar;
    property    VertScrollBar;
    property    StyleElements default [];
    property    OnClick;
  end;


  TKnobsCustomModule = class( TCustomPanel, IKnobsUndoable)
  // Base type for TModule, it has a bitmap and a title mainly
  private
    FEditHistory               : TKnobsEditHistory;      // A small local undo history
    FAnchor                    : TPoint;                 // Drag origin
    FDistance                  : TPoint;                 // Drag distance
    FTresholdPast              : Boolean;                // Set when dragged over a minimum distance
    FFlags                     : TKnobsModuleFlags;      // Selected / Capturing
    FTitleLabel                : TKnobsEditLabel;        // All editor modules can have a title label, it may be nil
    FTitle                     : string;                 // User editible module title
    FSavedTitle                : string;                 // A copy of the regular FTitle
    FUseAlternateTitle         : Boolean;                // When active shows module name (in short form) instead of the title
    FComment                   : string;                 // Comment as displayed on mouseover in the module selector
    FPicture                   : TPicture;               // Image used for the module selector
    FDocs                      : TStrings;               // Help text for generated context sensitive help
    FModuleType                : TKnobsModuleType;       // The type, used to couple editor modules to synth modules
    FComponentData             : TKnobsComponentData;    // A hashed list of inserted components - on the long name
    FExternalName              : string;                 // For TModExternal - the filename for the dynamically loaded code
    FModuleAlternateTypes      : string;                 // a space separated list of laternate module IDs
    FPageName                  : string;                 // The page in the module selector this module will be on,
                                                         // when the parent is a TTabSheet and that TTabSheet has a non empty
                                                         // Caption that value will be used instead.
    FBitmap                    : TBitmap;                // Overlay bitmap for module
    FUseTitleHints             : Boolean;
    FUseConnectorBorders       : Boolean;
    FUseTypedConnectorBorders  : Boolean;
    FConnectorBorderColor      : TColor;
    FSelectorColor             : TColor;
    FSelectorBorderColor       : TColor;
    FSelectorBorderColorSingle : TColor;
    FDisplayColor              : TColor;
    FDisplayBorderColor        : TColor;
    FKnobMIDIColor             : TColor;
    FKnobFocusColor            : TColor;
    FKnobMIDIFocusColor        : TColor;
    FViewerColor               : TColor;
    FViewerBorderColor         : TColor;
    FViewerLineColor           : TColor;
    FViewerFillColor           : TColor;
    FIndBarPeakColor           : TColor;
    FIndBarValeyColor          : TColor;
    FOpacity                   : Byte;                   // Opacity for background bitmap
    FFlat                      : Boolean;                // When true draw a flat module
    FTexture                   : Boolean;                // When true draw a texture bitmap for the module, otherwise use .. hmm .. some solid color
    FControlMode               : TDistanceMode;          // Knobs control mode, circular, horizontal or vertical
    FWheelSupportOnKnobs       : Boolean;                // True when MouseWheel ccan control knobs
    FWheelSensitivity          : Integer;                // MouseWheel sensitivity
    FAllowSignalConversion     : Boolean;                // True when dynamic signal conversions are supported by this module
    FAllowRandomization        : Boolean;                // True when patch randomization allowed on this module
    FShowAllowRandomization    : Boolean;                // When true and randomization is not allowed show a special border
    FHasSideEffects            : Boolean;                // True when this module is connected even when inputs only are connected
    FTemplateName              : string;                 // Name of template used to load values from or to save values to, if any.
    FActiveVariation           : Integer;
    FActiveRange               : Integer;
    FOldSpedUp                 : Boolean;
    FVisited                   : Boolean;                // Used in Graph searches
    FVisitOrder                : Integer;                // Used in Graph searches
    FPreCount                  : Integer;                // Used in Graph searches
    FPostCount                 : Integer;                // Used in Graph searches
    FOnValueChanged            : TKnobsOnValueChange;    // Called with the new value
    FOnTextChanged             : TKnobsOnTextChanged;    // Called with the new text
    FOnLog                     : TKnobsOnLog;            // To atach an external logger
    FOnUnFocus                 : TOnUnFocus;             // Called when a control could be unfocussed
  private
    procedure   IncludeFlags( aFlags: TKnobsModuleFlags);
    procedure   ExcludeFlags( aFlags: TKnobsModuleFlags);
    function    HasFlags( aFlags: TKnobsModuleFlags): Boolean;
    procedure   FixSpedUpControls( MustSpeedUp: Boolean);
    procedure   ComponentInserted( const aComponent: TComponent);
    procedure   ComponentRemoved ( const aComponent: TComponent);
  private
    function    GetSelected : Boolean;
    procedure   SetSelected ( aValue: Boolean);
    function    GetCapturing: Boolean;
    procedure   SetCapturing( aValue: Boolean);
  private
    procedure   SetPicture                  ( const aValue: TPicture);
    procedure   SetDocs                     ( const aValue: TStrings);
    function    GetPageName                 : string;
    procedure   SetPageName                 ( const aValue: string);
    procedure   SetModuleType               ( aValue : TKnobsModuleType);
    function    GetIsSpedUp                 : Boolean;
    procedure   SetActiveVariation          ( aValue: Integer);
    procedure   SetActiveRange              ( aValue: Integer);
    procedure   SetRangeValue               ( aValue: TSignal);
    procedure   SetTitle                    ( const aValue: string);
    procedure   SetUseAlternateTitle        ( aValue: Boolean);
    function    GetFriendlyName             : string;
    procedure   SetTitleLabel               ( const aValue: TKnobsEditLabel);
    procedure   SetUseTitleHints            ( aValue : Boolean);
    procedure   SetUseConnectorBorders      ( aValue : Boolean);
    procedure   SetUseTypedConnectorBorders ( aValue : Boolean);
    procedure   SetConnectorBorderColor     ( aValue : TColor);
    procedure   SetSelectorColor            ( aValue : TColor);
    procedure   SetSelectorBorderColor      ( aValue : TColor);
    procedure   SetSelectorBorderColorSingle( aValue : TColor);
    procedure   SetDisplayColor             ( aValue : TColor);
    procedure   SetDisplayBorderColor       ( aValue : TColor);
    procedure   SetKnobMIDIColor            ( aValue : TColor);
    procedure   SetKnobFocusColor           ( aValue : TColor);
    procedure   SetKnobMIDIFocusColor       ( aValue : TColor);
    procedure   SetViewerColor              ( aValue : TColor);
    procedure   SetViewerBorderColor        ( aValue : TColor);
    procedure   SetViewerLineColor          ( aValue : TColor);
    procedure   SetViewerFillColor          ( aValue : TColor);
    procedure   SetIndBarPeakColor          ( aValue : TColor);
    procedure   SetIndBarValeyColor         ( aValue : TColor);
    procedure   SetOpacity                  ( aValue : Byte);
    procedure   SetFlat                     ( aValue : Boolean);
    procedure   SetTexture                  ( aValue : Boolean);
    procedure   SetControlMode              ( aValue : TDistanceMode);
    procedure   SetWheelSupportOnKnobs      ( aValue : Boolean);
    procedure   SetWheelSensitivity         ( aValue : Integer);
    procedure   SetAllowSignalConversion    ( aValue : Boolean);
    procedure   SetAllowRandomization       ( aValue : Boolean);
    procedure   SetShowAllowRandomization   ( aValue : Boolean);
  protected
    procedure   WMEraseBackground( var aMsg: TWMEraseBkgnd);                                      message WM_ERASEBKGND;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer);                                                 override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
  protected
    procedure   InvalidateKnobs;
    procedure   InvalidateDisplays;
    procedure   ValueChanged( aSender: TObject; const aPath, aControlType: string; aValue: TSignal; IsFinal, IsAutomation: Boolean); virtual;
    procedure   TextChanged ( aSender: TObject; const aPath, aValue: string);
    function    FindWirePanel: TKnobsWirePanel;
    function    Snap( const aPoint: TPoint): TPoint;
    procedure   Paint;                                                                                         override;
    procedure   Loaded;                                                                                        override;
    procedure   AssignBitmap;                                                                                   virtual;
    procedure   ScrollSelfInView;
    procedure   HandleSelection( CtrlKeyActive: Boolean);
    procedure   BeginMove        ( anAnchor : TPoint; aShift: TShiftState);
    procedure   Move             ( aDistance: TPoint; aShift: TShiftState);
    procedure   MoveSelected     ( aDistance: TPoint; aShift: TShiftState);
    procedure   FinalMoveSelected( aDistance: TPoint; aShift: TShiftState);
    procedure   ReleasePeerCapture;
    procedure   Notification( aComponent: TComponent; anOperation: TOperation);                                override;
    procedure   FixNames;
    function    FindComponentData ( const aName: string; const aClass: TComponentClass): TKnobsComponent;
    function    FindControl       ( const aName: string; const aClass: TComponentClass): TComponent;
    procedure   FixComponentData;
    procedure   LoadCaptionsFor( const aControl:  TKnobsSelector; const aDependency: TKnobsValuedControl);
  protected
    procedure   PreVisit ( var aParam: Integer);
    procedure   PostVisit( var aParam: Integer);
    procedure   Visit    ( var aParam1, aParam2: Integer);
    procedure   DumpGraph( const aStrings: TStringList);
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   ClearHistory;
    procedure   FixControlInsertions;
    procedure   FixBitmaps;
    procedure   FixAllSynthParams( IsAutomation: Boolean);
    procedure   FixOwnership( aRoot, aControl: TWinControl);
    function    Clone( anOwner: TComponent; aParent: TWinControl; anOffset: TPoint; MustDrag, CopyMidiCC: Boolean): TKnobsCustomModule;
    procedure   SetKnobValue     ( const aType, aName: string; aValue: Integer);
    procedure   SetDetailValue   ( const aType, aName: string; const aValue: string);
    procedure   SetLockOn        ( const aType, aName: string);
    procedure   SetCCOn          ( const aType, aName: string; aValue: Byte);
    procedure   SetRndOn         ( const aType, aName: string; aValue: Byte; WasRead: Boolean);
    procedure   SetVarOn         ( const aType, aName, aValue: string; WasRead: Boolean);
    procedure   SetLowHighMarksOn( const aType, aName: string; const aLow, aHigh: string);
    procedure   SetSignal        ( const aName: string; const aValue: TSignal        );
    procedure   SetLight         ( const aName: string; const aValue: TSignal        );
    procedure   SetData          ( const aName: string; const aValue: TSignalArray   );
    procedure   SetXYData        ( const aName: string; const aValue: TSignalPairFifo);
    procedure   SetInfo          ( const aName: string; const aValue: string         );
    procedure   SetCursorData    ( const aName: string; const aValue: TSignalPair    );
    procedure   SetGridData      ( const aName: string; const aValue: string         );
    procedure   Dump( var aFile: TextFile; anInd: Integer);
    procedure   Log( aLogClass: TLogClass; const aMsg: string);
    procedure   LogFmt( aLogClass: TLogClass; const aFmt: string; const anArgs: array of const);
    function    AsBitmap: TBitmap;
    procedure   SetTitleFont ( const aValue: TFont);
    procedure   SetModuleFont( const aValue: TFont);
  public
    function    FindValuedControl( const aName: string): TKnobsValuedControl;
    function    FindTextControl  ( const aName: string): TKnobsTextControl;
    function    FindXYControl    ( const aName: string): TKnobsXYControl;
    function    FindDisplay      ( const aName: string): TKnobsDisplay;
    function    FindFileSelector ( const aName: string): TKnobsFileSelector;
    function    FindConnector    ( const aName: string): TKnobsConnector;
    function    FindDataMaker    ( const aName: string): TKnobsDataMaker;
    function    FindGridControl  ( const aName: string): TKnobsGridControl;
    function    FindData         ( const aName: string): TKnobsData;
    function    FindAutomatable  ( const aName: string): IKnobsAutomatable;
    function    FindVariation    ( const aName: string): IKnobsVariation;
  public
    procedure   Restack( DoRestack, RedrawWires: Boolean);
    procedure   FixupInsertion;
    procedure   FixDisplays;
    procedure   Disconnect;
    procedure   SelectUnique;
    procedure   FixSelection;
    procedure   CollectInputsWithBaseName ( var anInputs : TKnobsConnectors; const aBaseName: string);
    procedure   CollectOutputsWithBaseName( var anOutputs: TKnobsConnectors; const aBaseName: string);
    procedure   UnFocus( const aSender: TObject);                                                               virtual;
    function    IsConnected: Boolean;
    procedure   CopyParamsFrom( aModule: TKnobsCustomModule; CopyStrict: Boolean);
    function    VisitControls( aControlType: TControlClass; const aName: string; const aHandler: TKnobsOnVisitControl; const aUserData: TObject): Boolean;
    procedure   CollectCCControls( const aList: TStringList);
    procedure   UnassignMidiCC( aMidiCC: Byte);
    procedure   SetRandomValue( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   SetRandomValue( anAmount: TSignal);                                                            overload;
    procedure   Randomize     ( anAmount: TSignal; aVariation: Integer);                                       overload;
    procedure   Randomize     ( anAmount: TSignal);                                                            overload;
    procedure   Mutate        ( aProb, aRange: TSignal; aVariation: Integer);                                  overload;
    procedure   Mutate        ( aProb, aRange: TSignal);                                                       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer);       overload;
    procedure   MateWith      ( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer);                   overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom, aVariation: Integer);                           overload;
    procedure   Morph         ( anAmount: TSignal; aDad, aMom: Integer);                                       overload;
    procedure   LiveMorph     ( anAmount: TSignal);
    procedure   AllowAllRandomization( aValue: Boolean);
    procedure   SyncControlRandomization( const aControlTypeName: string; aValue: Boolean);
    function    HasRandomizableControls: Boolean;
    procedure   CountGene( var aCount: Integer);
    procedure   FillGene  ( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    procedure   AcceptGene( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    procedure   CreateValueSnapShot( aTimeStamp: TKnobsTimeStamp; const aHistory: TKnobsEditHistory);
    procedure   FixActiveRange  ( aValue: Integer);
    procedure   SetValueForRange( aRange: Integer; aValue: TSignal);
  public
    property    EditHistory               : TKnobsEditHistory      read FEditHistory          implements IKnobsUndoable;
  public
    property    Selected                  : Boolean                read GetSelected                write SetSelected;
    property    Capturing                 : Boolean                read GetCapturing               write SetCapturing;
  public
    property    UseAlternateTitle         : Boolean                read FUseAlternateTitle         write SetUseAlternateTitle;
    property    Title                     : string                 read FTitle                     write SetTitle;
    property    FriendlyName              : string                 read GetFriendlyName;
    property    Comment                   : string                 read FComment                   write FComment;
    property    Picture                   : TPicture               read FPicture                   write SetPicture;
    property    Docs                      : TStrings               read FDocs                      write SetDocs;
    property    PageName                  : string                 read GetPageName                write SetPageName;
    property    ModuleType                : TKnobsModuleType       read FModuleType                write SetModuleType;
    property    ExternalName              : string                 read FExternalName              write FExternalName;
    property    ModuleAlternateTypes      : string                 read FModuleAlternateTypes      write FModuleAlternateTypes;
    property    TemplateName              : string                 read FTemplateName              write FTemplateName;
    property    IsSpedUp                  : Boolean                read GetIsSpedUp;
    property    ActiveVariation           : Integer                read FActiveVariation           write SetActiveVariation;
    property    ActiveRange               : Integer                read FActiveRange               write SetActiveRange;
    property    RangeValue                : TSignal                                                write SetRangeValue;
    property    OnLog                     : TKnobsOnLog            read FOnLog                     write FOnLog;
  published
    property    StyleElements                                                                                                         default [];
    property    Color                                                                                                                 default clWhite;
    property    UseTitleHints             : Boolean                read FUseTitleHints             write SetUseTitleHints             default True;
    property    UseConnectorBorders       : Boolean                read FUseConnectorBorders       write SetUseConnectorBorders       default True;
    property    UseTypedConnectorBorders  : Boolean                read FUseTypedConnectorBorders  write SetUseTypedConnectorBorders  default False;
    property    ConnectorBorderColor      : TColor                 read FConnectorBorderColor      write SetConnectorBorderColor      default clWhite;
    property    SelectorColor             : TColor                 read FSelectorColor             write SetSelectorColor             default clGray;
    property    SelectorBorderColor       : TColor                 read FSelectorBorderColor       write SetSelectorBorderColor       default clYellow;
    property    SelectorBorderColorSingle : TColor                 read FSelectorBorderColorSingle write SetSelectorBorderColorSingle default clWhite;
    property    DisplayColor              : TColor                 read FDisplayColor              write SetDisplayColor              default clGray;
    property    DisplayBorderColor        : TColor                 read FDisplayBorderColor        write SetDisplayBorderColor        default clSilver;
    property    KnobMIDIColor             : TColor                 read FKnobMIDIColor             write SetKnobMIDIColor             default CL_MIDI;
    property    KnobFocusColor            : TColor                 read FKnobFocusColor            write SetKnobFocusColor            default CL_FOCUS;
    property    KnobMIDIFocusColor        : TColor                 read FKnobMIDIFocusColor        write SetKnobMIDIFocusColor        default CL_MIDI_FOCUS;
    property    ViewerColor               : TColor                 read FViewerColor               write SetViewerColor               default clGray;
    property    ViewerBorderColor         : TColor                 read FViewerBorderColor         write SetViewerBorderColor         default clGray;
    property    ViewerLineColor           : TColor                 read FViewerLineColor           write SetViewerLineColor           default clWhite;
    property    ViewerFillColor           : TColor                 read FViewerFillColor           write SetViewerFillColor           default $009E9E9E;
    property    IndBarPeakColor           : TColor                 read FIndBarPeakColor           write SetIndBarPeakColor           default clWhite;
    property    IndBarValeyColor          : TColor                 read FIndBarValeyColor          write SetIndBarValeyColor          default clBlack;
    property    TitleLabel                : TKnobsEditLabel        read FTitleLabel                write SetTitleLabel;
    property    Opacity                   : Byte                   read FOpacity                   write SetOpacity                   default 223;
    property    Flat                      : Boolean                read FFlat                      write SetFlat                      default False;
    property    Texture                   : Boolean                read FTexture                   write SetTexture                   default True;
    property    ControlMode               : TDistanceMode          read FControlMode               write SetControlMode;
    property    WheelSupportOnKnobs       : Boolean                read FWheelSupportOnKnobs       write SetWheelSupportOnKnobs;
    property    WheelSensitivity          : Integer                read FWheelSensitivity          write SetWheelSensitivity;
    property    AllowSignalConversion     : Boolean                read FAllowSignalConversion     write SetAllowSignalConversion     default False;
    property    AllowRandomization        : Boolean                read FAllowRandomization        write SetAllowRandomization        default False;
    property    ShowAllowRandomization    : Boolean                read FShowAllowRandomization    write SetShowAllowRandomization    default False;
    property    HasSideEffects            : Boolean                read FHasSideEffects            write FHasSideEffects              default False;
    property    OnClick;
  published
    property    OnValueChanged            : TKnobsOnValueChange    read FOnValueChanged            write FOnValueChanged;
    property    OnTextChanged             : TKnobsOnTextChanged    read FOnTextChanged             write FOnTextChanged;
    property    OnUnFocus                 : TOnUnFocus             read FOnUnFocus                 write FOnUnFocus;
  end;


  TKnobsModule = class( TKnobsCustomModule)
  // exports stuff of TCustomModule
  published
    property    BiDiMode;
    property    Constraints;
    property    Enabled;
    property    Color;
    property    Ctl3D;
    property    Font;
    property    ParentBiDiMode;
    property    ParentColor;
    property    ParentCtl3D;
    property    ParentFont;
    property    ParentShowHint     default False;
    property    PopupMenu;
    property    ShowHint           default False;
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    OnContextPopup;
    property    OnEnter;
    property    OnExit;
    property    OnMouseDown;
    property    OnMouseMove;
    property    OnMouseUp;
    property    OnMouseWheel;
    property    OnMouseWheelDown;
    property    OnMouseWheelUp;
    property    OnResize;
    property    Width;
    property    Height;
    property    Tag;
    property    Title;
    property    Comment;
    property    Picture;
    property    Docs;
    property    PageName;
    property    ModuleType;
    property    ModuleAlternateTypes;
    property    BevelInner;
    property    BevelOuter;
    property    BevelWidth;
  end;


  TKnobsModuleButton = class( TSpeedButton)
  // Button as used in module selector, on click it creates a module
  // of the type that got registered with it.
  private
    FPageName   : string;
    FModuleType : TKnobsModuleType;
    FModuleName : string;
    FShowing    : Boolean;
  private
    procedure   SetShowing( aValue: Boolean);
  protected
    procedure   Paint;                                                                                         override;
  public
    procedure   FreeGlyph;
    procedure   LoadGlyph( const aGlyph: TBitmap);
    procedure   Initialize( const aPageName: string; aModuleType: TKnobsModuleType; const aModuleName: string; const aGlyph: TBitmap);
  public
    property    PageName   : string           read FPageName;
    property    ModuleType : TKnobsModuleType read FModuleType;
    property    ModuleName : string           read FModuleName;
    property    Showing    : Boolean          read FShowing write SetShowing;
  end;


  TKnobsModuleSeparator = class( TKnobsModuleButton)
  // Separator space in module selector
  end;


  TKnobsShifterBtn = class( TBitBtn)
  // A bitBtn with repeat, used for the shifter buttons in TKnobsModuleSelector - or general purpose
  private
    FRepeatTimer : TTimer;
    FRepeatCount : Integer;
    FMouseDown   : Boolean;
  private
    procedure   StartTimer( ams: Cardinal);
    procedure   StopTimer;
    procedure   TimerFired( aSender: TObject);
  protected
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp  ( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   SetEnabled( aValue: Boolean);                                                                  override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  end;


  TKnobsModuleSelector = class( TPageControl)
  // A tabbed thingy having TModuleButtons on it's pages.
  strict private
  const
    ButtonWidth     = 23;
    ButtonHeight    = 23;
    SeparatorWidth  =  7;
    SeparatorHeight = 23;
    ShifterWidth    = ( 3 * ButtonWidth) div 4;
  private
    FRegisteredModules   : TKnobsModuleList;
    FOnModuleButtonClick : TKnobsOnModuleButtonClick;
    FOnGetGlyph          : TKnobsOnGetGlyph;
    FButtonsEnabled      : Boolean;
    FAllowDuplicates     : Boolean;
    FTimer               : TTimer;
    FTabColorsHi         : array of TColor;
    FTabColorsLo         : array of TColor;
    FSearchName          : string;
    FLastPage            : TTabSheet;
  strict private
    class constructor Create;
    class destructor  Destroy;
  private
    procedure   DoResize    ( aSender: TObject);
    procedure   DoTabChanged( aSender: TObject);
    procedure   DoLeftClick ( aSender: TObject);
    procedure   DoRightClick( aSender: TObject);
    procedure   StartTimeout;
    procedure   StopTimeout;
    procedure   TimeOutFired( aSender: TObject);
    procedure   AllButtonsUp;
  private
    procedure   SetButtonsEnabled( aValue: Boolean);
    function    AddPage          ( const aName: string): TTabSheet;
    function    PageButtonCount  ( const aName: string): Integer;
    function    PageExists       ( const aName: string): Boolean;
    function    PageByName       ( const aName: string): TTabSheet;
    function    FindModulePage   ( aModuleType: TKnobsModuleType): TTabSheet;
    function    FindModuleButton ( aModuleType: TKnobsModuleType): TKnobsModuleButton;
    procedure   DoModuleMouseDown( aSender: TObject; aButton: TMouseButton; aShift: TShiftState; X, Y: Integer);
  private
    procedure   ShowShifters      ( aPage: TTabsheet; DoShow: Boolean);
    procedure   EnableLeftShifter ( aPage: TTabsheet; DoMakeVisible: Boolean);
    procedure   EnableRightShifter( aPage: TTabsheet; DoMakeVisible: Boolean);
    function    GetShiftersVisible( aPage: TTabsheet): Boolean;
    procedure   SetShiftersVisible( aPage: TTabSheet; aValue: Boolean);
    function    GetLeftmostButton ( aPage: TTabsheet): Integer;
    procedure   SetLeftmostButton ( aPage: TTabsheet; aValue: Integer);
    procedure   ReOrderButtons    ( aPage: TTabSheet);
    function    CalcFitting       ( aPage: TTabSheet): Boolean;
    function    CalcButtonsWidth  ( aPage: TTabSheet): Integer;
    procedure   FixPageLayout     ( aPage: TTabSheet);
    procedure   FormatPage        ( aPage: TTabSheet);
    procedure   ShiftButtonsLeft  ( aPage: TTabSheet);
    procedure   ShiftButtonsRight ( aPage: TTabSheet);
  private
    property    ShiftersVisible[ anIndex: TTabSheet]: Boolean read GetShiftersVisible write SetShiftersVisible;
    property    LeftmostButton [ anIndex: TTabSheet]: Integer read GetLeftmostButton  write SetLeftmostButton;
  protected
    procedure   Notification( aComponent: TComponent; anOperation: TOperation);                                override;
    procedure   DrawTab( aTabIndex: Integer; const aRect: TRect; anActive: Boolean);                           override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   RegisterModuleType  ( const aPageName: string; aModuleType: TKnobsModuleType; const aBitmapsFolder, aModuleName: string);
    procedure   UnregisterModuleType( aModuleType: TKnobsModuleType);
    procedure   UnregisterAllTypes;
    procedure   GetTabColors( const aGetter: TOnGetTabColor; aMixColor: TColor; aMixHi, aMixLo: Byte);
    function    GetTabColorHi( anIndex: Integer): TColor;
    function    GetTabColorLo( anIndex: Integer): TColor;
    procedure   Search( const aName: string);
    procedure   ReloadGlyphs( const aBitmapsFolder: string);
  published
    property    OnModuleButtonClick : TKnobsOnModuleButtonClick read FonModuleButtonClick write FOnModuleButtonClick;
    property    OnGetGlyph          : TKnobsOnGetGlyph          read FOnGetGlyph          write FonGetGlyph;
    property    ButtonsEnabled      : Boolean                   read FButtonsEnabled      write SetButtonsEnabled;
    property    AllowDuplicates     : Boolean                   read FAllowDuplicates     write FAllowDuplicates     default True;
    property    ParentColor;
  end;


  TKnobsHintWindow = class( THintWindow)
  // Hint popup window for module selector
  private
    FModuleType : string;
    FBitmap     : TBitmap;
    FComment    : string;
  protected
    function    MakeBitmap: TBitmap;
    procedure   ParseHint( var aHintStr: string);
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   ActivateHint( aRect: TRect; const aHint: string);                                              override;
  published
    property    Caption;
  end;


  TKnobsSimpleHintWindow = class( THintWindow)
  // Hint popup window for controls
  protected
    procedure   Paint;                                                                                         override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
  end;


  TKnobsTimerSpeedButton = class( TSpeedButton)
  private
    FRepeatTimer    : TTimer;
    FRepeatCount    : Integer;
    FTimeBtnOptions : TKnobsTimeBtnOptions;
  private
    procedure   TimerExpired( Sender: TObject);                                                                 virtual;
    procedure   WMSetFocus ( var aMessage: TWMSetFocus);                                            message WM_SETFOCUS;
    procedure   WMKillFocus( var aMessage: TWMKillFocus);                                          message WM_KILLFOCUS;
  protected
    procedure   Paint;                                                                                         override;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                           override;
    procedure   MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer);                             override;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
  published
    property    TimeBtnOptions: TKnobsTimeBtnOptions read FTimeBtnOptions write FTimeBtnOptions              default [];
    property    StyleElements                                                                                default [];
  end;


  TKnobsDelayedSpeedButton = Class( TKnobsTimerSpeedButton)
  private
    FOnDelayedClick : TNotifyEvent;
  private
    procedure   TimerExpired( Sender: TObject);                                                                override;
    procedure   DelayedClick;
  published
    property    OnDelayedClick: TNotifyEvent read FOnDelayedClick write FOnDelayedClick;
  end;


var

  GOnLoadCaptions         : TOnLoadCaptions         = nil;   // Global caption loader for dynamic count selectors
  GOnCreateModuleBitmap   : TOnCreateModuleBitmap   = nil;   // To query FrmStore for a module's looks
  GOnReadModuleComment    : TOnReadModuleComment    = nil;   // To query FrmStore for a module's help text
  GOnPopupEditorShow      : TOnPopupEditorShow      = nil;   // To inform a main program about a popup editor being shown
  GOnPopupEditorHide      : TOnPopupEditorHide      = nil;   // To inform a main program about a popup editor being hidden

  function  RangeCount   : Integer;
  function  RangeName    ( anIndex: Integer): string;
  function  RangeIsMorph ( anIndex: Integer): Boolean;
  procedure SetRangeValue( anIndex: Integer; aValue: TSignal);
  function  RangeValue   ( anIndex: Integer): TSignal;
  function  CreateRangeNames: TStringList;

  procedure InvalidateEditors;                             // Needed after a theme change.
  procedure CreateBitmaps( const aFolderName: string);     // Load module symbols from a disk folder or from a resource file
  function  IsAutomatable( const anObject: TObject): Boolean;
  function  IsVariation  ( const anObject: TObject): Boolean;
  function  IsUndoable   ( const anObject: TObject): Boolean;
  procedure ResetTimeStamp;
  function  GetTimeStamp: TKnobsTimeStamp;
  function  ModuleFriendlyToLongName( const aValue: string): string;
  function  ModuleLongToFriendlyName( const aValue: string): string;


var

  GUseNoteNames : Boolean = False;    // True when note names should be used in frequency displays


implementation



{$R KnobsBitmaps.res}


type

  TknobsTextControlStyleHook = class( TStaticTextStyleHook)
  strict protected
    procedure   PaintNC( aCanvas: TCanvas);                                                                    override;
  end;


  TKnobsModuleSelectorStyleHook = class( TTabControlStyleHook)
  protected
    procedure   DrawTab( aCanvas: TCanvas; anIndex: Integer);                                                  override;
  end;


  TWinControlClass       = class( TWinControl      );
  TCustomTabControlClass = class( TCustomTabControl);


const

  InitRepeatPause     = 330;          // Initial pause before slow repeat kicks in (200 caused accidental initial double clicks)
  SlowRepeatPause     = 100;          // RepeatPause slow repeat
  FastRepeatPause     =  30;          // RepeatPause fast repeat;
  SlowRepeats         =  15;          // After SlowRepeats repeats we change from slow to fast repeat
  ClickerSize         =  10;          // Height for clickers


  // Bitmap Id's as used in the compiled in resource 'Bitmaps.res'

  idbmDn              =   1;          // Down pointing triangle
  idbmUp              =   2;          // Up pointing triangle
  idbmConIn           =   3;          // Input connector - round
  idbmConOut          =   4;          // Output connector - square
  idbmIndActive       =   5;          // Active LED
  idbmIndInactive     =   6;          // Inactivee LED
  idbmKnob            =   7;          // Large knob
  idbmSmallKnob       =   8;          // Small knob
  idbmSkin            =   9;          // Texture overlay
  idbmIndLowActive    =  10;          // Active LED for low values (green)
  idbmIndLowInactive  =  11;          // Inactive LED for low values (green)
  idbmIndMidActive    =  12;          // Active LED for mid values (yellow)
  idbmIndMidInactive  =  13;          // Inactive LED for mid values (yellow)
  idbmIndHighActive   =  14;          // Active LED for high values (red)
  idbmIndHighInactive =  15;          // Inactive LED for high values (red)
  idbmNoKnob          =  16;          // Bitmap for invisible knobs - clickers only
  idbmSkin2           =  17;          // Alternate texture overlay
  idbmKnobNew         =  18;          // Alternate large knob
  idbmSmallKnobNew    =  19;          // Alternate small knob
  idbmConInTrans      =  20;          // Alternate input connector (round)
  idbmConOutTrans     =  21;          // Alternate output connector (square)


var

  GAnchor             : TPoint               = ( x: 0; y: 0);   // Ussed for mouse drag operaations
  GStartPoint         : TPoint               = ( x: 0; y: 0);   // Ussed for mouse drag operaations
  GDistance           : TPoint               = ( x: 0; y: 0);   // Ussed for mouse drag operaations
  GLabelEditor        : TKnobsLabelEditor    = nil;             // The label editor popup
  GDisplayEditor      : TKnobsDisplayEditor  = nil;             // The display (multiline) editor popup
  GChoiceSelector     : TKnobsChoiceSelector = nil;             // The choice selector popup
  GScalaSelector      : TFormScala           = nil;             // The scala file load form
  GTimeStamp          : TKnobsTimeStamp      = 0;               // Relative time since last ResetTimeStamp.


   // Shared bitmaps, loaded from 'Bitmaps.res' at startup

  bmDn                 : TBitmap = nil;  // Down pointing triangle
  bmUp                 : TBitmap = nil;  // Up pointing triangle
  bmConAudioIn         : TBitmap = nil;  // Input connector  - round  - audio rate
  bmConAudioOut        : TBitmap = nil;  // Output connector - square - audio rate
  bmConControlIn       : TBitmap = nil;  // Input connector  - round  - control rate
  bmConControlOut      : TBitmap = nil;  // Output connector - square - control rate
  bmConControlLogicIn  : TBitmap = nil;  // Input connector  - round  - logic control rate
  bmConControlLogicOut : TBitmap = nil;  // Output connector - square - logic control rate
  bmConLogicIn         : TBitmap = nil;  // Input connector  - round  - logic rate
  bmConLogicOut        : TBitmap = nil;  // Output connector - square - logic rate
  bmIndActive          : TBitmap = nil;  // Active LED
  bmIndInactive        : TBitmap = nil;  // Inactive LED
  bmIndLowActive       : TBitmap = nil;  // Active LED for low values (green)
  bmIndLowInactive     : TBitmap = nil;  // Inactive LED for low values (green)
  bmIndMidActive       : TBitmap = nil;  // Active LED for mid values (yellow)
  bmIndMidInactive     : TBitmap = nil;  // Inactive LED for mid values (yellow)
  bmIndHighActive      : TBitmap = nil;  // Active LED for high values (red)
  bmIndHighInactive    : TBitmap = nil;  // Inactive LED for high values (red)
  bmKnob               : TBitmap = nil;  // Large knob
  bmSmallKnob          : TBitmap = nil;  // Small knob
  bmNoKnob             : TBitmap = nil;  // Bitmap for invisible knobs - clickers only
  bmSkin               : TBitmap = nil;  // Texture overlays


  GRangeSpecs : TKnobsRangeSpecs = nil;


  function  RangeCount: Integer;
  begin
    Result := Length( GRangeSpecs);
  end;


  function  RangeName( anIndex: Integer): string;
  begin
    Result := GRangeSpecs[ anIndex].Name;
  end;


  function  RangeIsMorph( anIndex: Integer): Boolean;
  begin
    Result := GRangeSpecs[ anIndex].IsMorph;
  end;


  procedure SetRangeValue( anIndex: Integer; aValue: TSignal);
  begin
    if   ( anIndex >= 0        )
    and  ( anIndex < RangeCount)
    and  Assigned( GRangeSpecs )
    then GRangeSpecs[ anIndex].Value := aValue;
  end;


  function  RangeValue( anIndex: Integer): TSignal;
  begin
    if   ( anIndex >= 0        )
    and  ( anIndex < RangeCount)
    and  Assigned( GRangeSpecs )
    then Result := GRangeSpecs[ anIndex].Value
    else Result := 0.0;
  end;


  procedure FillRangeNameList( const aStrings: TStrings);
  var
    i : Integer;
  begin
    if   Assigned( aStrings)
    then begin
      aStrings.Clear;

      for i := 0 to RangeCount - 1
      do aStrings.Add( RangeName( i));
    end;
  end;


  function  CreateRangeNames: TStringList;
  begin
    Result := TStringList.Create;
    FillRangeNameList( Result);
  end;


  function  IsAutomatable( const anObject: TObject): Boolean;
  begin
    Result := Supports( anObject, IKnobsAutomatable);
  end;


  function  IsVariation( const anObject: TObject): Boolean;
  begin
    Result := Supports( anObject, IKnobsVariation);
  end;


  function  IsUndoable( const anObject: TObject): Boolean;
  begin
    Result := Supports( anObject, IKnobsUndoable);
  end;


  procedure ResetTimeStamp;
  begin
    GTimeStamp := Now64;
  end;


  function  GetTimeStamp: TKnobsTimeStamp;
  begin
    Result := Now64 - GTimeStamp;
  end;


  procedure PopupEditorShow( const aSender: TObject);
  begin
    if   Assigned( GOnPopupEditorShow)
    then GOnPopupEditorShow( aSender);
  end;


  procedure PopupEditorHide( const aSender: Tobject);
  begin
    if   Assigned( GOnPopupEditorHide)
    then GOnPopupEditorHide( aSender);
  end;


  procedure InvalidateEditors; // Needed after a theme change.
  begin
    FreeAndNil( GLabelEditor   );
    FreeAndNil( GDisplayEditor );
    FreeAndNil( GChoiceSelector);
  end;


  procedure NeedsLabelEditor;
  begin
    if   GLabelEditor = nil
    then TKnobsLabelEditor.Create( Application);
  end;


  procedure NeedsDisplayEditor;
  begin
    if   GDisplayEditor = nil
    then TKnobsDisplayEditor.Create( Application);
  end;


  procedure NeedsChoiceEditor;
  begin
    if   GChoiceSelector = nil
    then TKnobsChoiceSelector.Create( Application);
  end;


  function  CreateModuleBitmap( aModuleType: TKnobsModuleType; UseCache: Boolean): TBitmap;
  begin
    if   Assigned( GOnCreateModuleBitmap)
    then Result := GOnCreateModuleBitmap( aModuleType, UseCache)
    else Result := nil;
  end;


  function  ReadModuleComment( aModuleType: TKnobsModuleType): string;
  begin
    if   Assigned( GOnReadModuleComment)
    then Result := GOnReadModuleComment( aModuleType)
    else Result := '';
  end;


  procedure LoadBitmap( var aBitmap: TBitmap; anIndex: Integer; const aFolderName, aFileName: string);
  // Loads one bitmap from 'Bitmaps.res' or from a file in the application;s bitmaps folder
  var
    fname : string;
  begin
    if   not Assigned( aBitmap)
    then aBitmap := TBitmap.Create;

    fname := Format( '%s\%s.bmp', [ aFolderName, aFileName]);

    if   FileExists( fname)
    then begin
      try
        aBitmap.LoadFromFile( fname);
      except
        aBitmap.LoadFromResourceID( HInstance, anIndex);
      end;
    end
    else aBitmap.LoadFromResourceID( HInstance, anIndex);

    aBitmap.TransparentMode := tmAuto;
    aBitmap.Transparent     := True;
  end;


  procedure CreateBitmaps( const aFolderName: string);
  // Loads all bitmaps from 'Bitmaps.res' or from a user supplied folder
  begin
    LoadBitmap( bmDn                , idbmDn             , aFolderName, 'skin_down'                       );   // Down arrow on buttons
    LoadBitmap( bmUp                , idbmUp             , aFolderName, 'skin_up'                         );   // up arrow   on buttons
    LoadBitmap( bmConAudioIn        , idbmConIn          , aFolderName, 'skin_connector_audio_in'         );   // input connector
    LoadBitmap( bmConAudioOut       , idbmConOut         , aFolderName, 'skin_connector_audio_out'        );   // output connector
    LoadBitmap( bmConControlIn      , idbmConIn          , aFolderName, 'skin_connector_control_in'       );   // input connector
    LoadBitmap( bmConControlOut     , idbmConOut         , aFolderName, 'skin_connector_control_out'      );   // output connector
    LoadBitmap( bmConLogicIn        , idbmConIn          , aFolderName, 'skin_connector_logic_in'         );   // input connector
    LoadBitmap( bmConLogicOut       , idbmConOut         , aFolderName, 'skin_connector_logic_out'        );   // output connector
    LoadBitmap( bmConControlLogicIn , idbmConIn          , aFolderName, 'skin_connector_control_logic_in' );   // input connector
    LoadBitmap( bmConControlLogicOut, idbmConOut         , aFolderName, 'skin_connector_control_logic_out');   // output connector
    LoadBitmap( bmIndActive         , idbmIndActive      , aFolderName, 'skin_indicator_active'           );   // active moduel LED
    LoadBitmap( bmIndInactive       , idbmIndInactive    , aFolderName, 'skin_indicator_inactive'         );   // Inactive module LED
    LoadBitmap( bmIndLowActive      , idbmIndLowActive   , aFolderName, 'skin_indicator_low_active'       );   // Active low value VU led (green)
    LoadBitmap( bmIndLowInactive    , idbmIndLowInactive , aFolderName, 'skin_indicator_low_inactive'     );   // Inactive low value VU led (green)
    LoadBitmap( bmIndMidActive      , idbmIndMidActive   , aFolderName, 'skin_indicator_mid_active'       );   // Active mid value VU led (yellow)
    LoadBitmap( bmIndMidInactive    , idbmIndMidInactive , aFolderName, 'skin_indicator_midinactive'      );   // Inactive mid value VU led (yellow)
    LoadBitmap( bmIndHighActive     , idbmIndHighActive  , aFolderName, 'skin_indicator_high_active'      );   // Active high value VU led (red)
    LoadBitmap( bmIndHighInactive   , idbmIndHighInactive, aFolderName, 'skin_indicator_high_inactive'    );   // Inactive high value VU led (red)
    LoadBitmap( bmKnob              , idbmKnobNew        , aFolderName, 'skin_knob'                       );   // Large knob
    LoadBitmap( bmSmallKnob         , idbmSmallKnobNew   , aFolderName, 'skin_small_knob'                 );   // Small knob
    LoadBitmap( bmNoKnob            , idbmNoKnob         , aFolderName, 'skin_no_knob'                    );   // No knob - does probably nothing
    LoadBitmap( bmSkin              , idbmSkin2          , aFolderName, 'skin_module'                     );   // Module overlay
  end;


  procedure FreeBitmaps;
  // Free the bitmaps loaded at program startup
  begin
    FreeAndnil( bmKnob              );
    FreeAndnil( bmSmallKnob         );
    FreeAndnil( bmNoKnob            );
    FreeAndnil( bmIndActive         );
    FreeAndnil( bmIndInactive       );
    FreeAndnil( bmIndLowActive      );
    FreeAndnil( bmIndLowInactive    );
    FreeAndnil( bmIndMidActive      );
    FreeAndnil( bmIndMidInactive    );
    FreeAndnil( bmIndHighActive     );
    FreeAndnil( bmIndHighInactive   );
    FreeAndnil( bmConAudioIn        );
    FreeAndnil( bmConAudioOut       );
    FreeAndnil( bmConControlIn      );
    FreeAndnil( bmConControlOut     );
    FreeAndnil( bmConLogicIn        );
    FreeAndnil( bmConLogicOut       );
    FreeAndnil( bmConControlLogicIn );
    FreeAndnil( bmConControlLogicOut);
    FreeAndnil( bmDn                );
    FreeAndnil( bmUp                );
    FreeAndnil( bmSkin              );
  end;


  procedure   CalcBezierCurve( var Points: TKnobsBezierPoints; const Params: TKnobsBezierParams);
  // Used for calculating wire bending - for rotate etc. see 'knobutils.pas'.
  var
    Dir1 : TPoint;
    Dir2 : TPoint;
  begin
    Dir1       := Rotate( Points[ 0] - Points[ 3], Params.Angle1);
    Dir2       := Rotate( Points[ 0] - Points[ 3], Params.Angle2);
    Points[ 1] := Points[ 0] - DivPoint( MultPoint( Dir1, Params.Strength1), 100);
    Points[ 2] := Points[ 3] + DivPoint( MultPoint( Dir2, Params.Strength2), 100);
  end;


  function  ConnectorName( aConnector: TKnobsConnector): string;
  begin
    if   Assigned( aConnector)
    then begin
      if   aConnector.HasParent
      then Result := Format( '%s.%s', [ aConnector.Parent.Name, aConnector.Name], AppLocale)
      else Result := Format( '<nil>.%s', [ aConnector.Name], AppLocale);
    end
    else Result := '<nil>.<nil>';
  end;


  function  WireName( aSource, aDestination: TKnobsConnector): string;
  begin
    Result := Format( '%s -> %s', [ ConnectorName( aSource), ConnectorName( aDestination)], AppLocale);
  end;


  function  ModuleFriendlyToLongName( const aValue: string): string;
  begin
    Result := StringReplace( aValue, 'mod ', 'TKnobsModule', [ rfReplaceAll, rfIgnoreCase]);
  end;


  function  ModuleLongToFriendlyName( const aValue: string): string;
  begin
    Result := StringReplace( aValue, 'TKnobsModule', 'mod ', [ rfReplaceAll, rfIgnoreCase]);
  end;


{ ========
  TknobsTextControlStyleHook = class( TStaticTextStyleHook)
  strict protected
}

    procedure   TknobsTextControlStyleHook.PaintNC( aCanvas: TCanvas); // override;
    var
      aBuffer : TBitmap;
      R       : Trect;
      aColor  : TColor;
    begin
      if   not StyleServices.Available
      then Exit;

      R := Rect( 0, 0, Control.Width, Control.Height);

      if   ( R.Width = 0) or ( R.Height = 0)
      then Exit;

      if   TKnobsTextControl( Control).BorderStyle = sbsNone
      then Exit;

      aBuffer := TBitMap.Create;

      try
        aBuffer.Width  := R.Width;
        aBuffer.Height := R.Height;
        aColor         := TKnobsTextControl( Control).BorderColor;
        Frame3D( aBuffer.Canvas, R, aColor, aColor, 1);
        ExcludeClipRect( aCanvas.Handle, 1, 1, Control.Width - 1, Control.Height - 1);
        aCanvas.Draw( 0, 0, aBuffer);
      finally
        aBuffer.Free;
      end;
    end;


{ ========
  TKnobsModuleSelectorStyleHook = class( TTabControlStyleHook)
  protected
}

    procedure   TKnobsModuleSelectorStyleHook.DrawTab( aCanvas: TCanvas; anIndex: Integer);
    var
      LDetails    : TThemedElementDetails;
      LImageIndex : Integer;
      LThemedTab  : TThemedTab;
      LIconRect   : TRect;
      R           : TRect;
      LayoutR     : TRect;
      LImageW     : Integer;
      LImageH     : Integer;
      DxImage     : Integer;
      LTextX      : Integer;
      LTextY      : Integer;
      LTextColor  : TColor;


        procedure DrawControlText( const S: string; var R: TRect; Flags: Cardinal);
        var
          TextFormat: TTextFormatFlags;
        begin
          aCanvas.Font       := TWinControlClass( Control).Font;
          TextFormat         := TTextFormatFlags( Flags);
          aCanvas.Font.Color := LTextColor;
          StyleServices.DrawText( aCanvas.Handle, LDetails, S, R, TextFormat, aCanvas.Font.Color);
        end;


        procedure AngleTextOut2( aCanvas: TCanvas; Angle, X, Y: Integer; const Text: string);
        var
          LSavedDC: Integer;
        begin
          LSavedDC := SaveDC( aCanvas.Handle);

          try
            SetBkMode( aCanvas.Handle, TRANSPARENT);
            aCanvas.Font.Orientation := Angle;
            aCanvas.TextOut( X, Y, Text);
          finally
            RestoreDC( aCanvas.Handle, LSavedDC);
          end;
        end;


    begin
      if   ( Images <> nil) and ( anIndex < Images.Count)
      then begin
        LImageW := Images.Width;
        LImageH := Images.Height;
        DxImage := 3;
      end
      else begin
        LImageW := 0;
        LImageH := 0;
        DxImage := 0;
      end;

      R := TabRect[ anIndex];

      if   R.Left < 0
      then Exit;

      if   TabPosition in [ tpTop, tpBottom]
      then begin
        if   anIndex <> TabIndex
        then begin
          InflateRect( R, 0, -1);
          OffsetRect ( R, 0,  1);
        end;
      end
      else if anIndex <> TabIndex
      then Dec( R.Left , 1)
      else Dec( R.Right, 1);

      aCanvas.Font.Assign( TPageControl( Control).Font);
      LayoutR    := R;
      LThemedTab := ttTabDontCare;

      // Get the type of the active tab

      case TabPosition of

        tpTop:

          begin
            if   anIndex = TabIndex
            then LThemedTab := ttTabItemSelected
            else if ( anIndex = HotTabIndex) and MouseInControl
            then LThemedTab := ttTabItemHot
            else LThemedTab := ttTabItemNormal;
          end;

        tpLeft:

          begin
            if   anIndex = TabIndex
            then LThemedTab := ttTabItemLeftEdgeSelected
            else if ( anIndex = HotTabIndex) and MouseInControl
            then LThemedTab := ttTabItemLeftEdgeHot
            else LThemedTab := ttTabItemLeftEdgeNormal;
          end;

        tpBottom:

          begin
            if   anIndex = TabIndex
            then LThemedTab := ttTabItemBothEdgeSelected
            else if ( anIndex = HotTabIndex) and MouseInControl
            then LThemedTab := ttTabItemBothEdgeHot
            else LThemedTab := ttTabItemBothEdgeNormal;
          end;

        tpRight:

          begin
            if   anIndex = TabIndex
            then LThemedTab := ttTabItemRightEdgeSelected
            else if (anIndex = HotTabIndex) and MouseInControl
            then LThemedTab := ttTabItemRightEdgeHot
            else LThemedTab := ttTabItemRightEdgeNormal;
          end;
      end;

      // draw the tab

      if   StyleServices.Available
      then begin
        if   Control is TKnobsModuleSelector
        then begin
          LDetails := StyleServices.GetElementDetails( LThemedTab); // necesary for DrawControlText and draw the icon
          InflateRect( R, -1, 0);                                   // adjust the size of the tab creating a blank space between the tabs

          if   LThemedTab in [ ttTabItemHot, ttTabItemLeftEdgeHot, ttTabItemRightEdgeHot, ttTabItemBothEdgeHot]
          then aCanvas.Brush.Color := TKnobsModuleSelector( Control).GetTabColorHi( anIndex)
          else aCanvas.Brush.Color := TKnobsModuleSelector( Control).GetTabColorLo( anIndex); // get the color

          aCanvas.Pen.Width := 1;

          if   anIndex = TabIndex
          then aCanvas.Pen.Color := clWhite
          else aCanvas.Pen.Color := aCanvas.Brush.Color;

          aCanvas.Rectangle( R);
        end
        else begin
          LDetails := StyleServices.GetElementDetails( LThemedTab);                    // necesary for  DrawControlText
          StyleServices.DrawElement( aCanvas.Handle, LDetails, R);
        end;
      end;

      // get the anIndex of the image (icon)

      if   Control is TCustomTabControl
      then LImageIndex := TCustomTabControlClass( Control).GetImageIndex( anIndex)
      else LImageIndex := anIndex;

      // draw the image

      if   ( Images <> nil) and ( LImageIndex >= 0) and ( LImageIndex < Images.Count)
      then begin
        LIconRect := LayoutR;

        case TabPosition of
          tpTop    ,
          tpBottom :

            begin
              LIconRect.Left  := LIconRect.Left + DxImage;
              LIconRect.Right := LIconRect.Left + LImageW;
              LayoutR.Left    := LIconRect.Right;
              LIconRect.Top   := LIconRect.Top + ( LIconRect.Bottom - LIconRect.Top) div 2 - LImageH div 2;

              if   ( TabPosition = tpTop) and ( anIndex = TabIndex)
              then OffsetRect( LIconRect, 0, -1)
              else if ( TabPosition = tpBottom) and ( anIndex = TabIndex)
              then OffsetRect( LIconRect, 0, 1);
            end;

          tpLeft :

            begin
              LIconRect.Bottom := LIconRect.Bottom - DxImage;
              LIconRect.Top    := LIconRect.Bottom - LImageH;
              LayoutR.Bottom   := LIconRect.Top;
              LIconRect.Left   := LIconRect.Left + ( LIconRect.Right - LIconRect.Left) div 2 - LImageW div 2;
            end;

          tpRight :

            begin
              LIconRect.Top    := LIconRect.Top + DxImage;
              LIconRect.Bottom := LIconRect.Top + LImageH;
              LayoutR.Top      := LIconRect.Bottom;
              LIconRect.Left   := LIconRect.Left + ( LIconRect.Right - LIconRect.Left) div 2 - LImageW div 2;
            end;
        end;

        if   StyleServices.Available
        then StyleServices.DrawIcon( aCanvas.Handle, LDetails, LIconRect, Images.Handle, LImageIndex);
      end;

      // draw the text of the tab

      if   StyleServices.Available
      then begin

        // here you set the color of the font

        if   LThemedTab in [ ttTabItemHot, ttTabItemLeftEdgeHot, ttTabItemRightEdgeHot, ttTabItemBothEdgeHot]
        then LTextColor := clWhite
        else LTextColor := clBlack;

        if   ( TabPosition = tpTop) and ( anIndex = TabIndex)
        then OffsetRect( LayoutR, 0, -1)
        else if (TabPosition = tpBottom) and ( anIndex = TabIndex)
        then OffsetRect( LayoutR, 0, 1);

        if   TabPosition = tpLeft
        then begin
          LTextX := LayoutR.Left + ( LayoutR.Right  - LayoutR.Left) div 2 - aCanvas.TextHeight( Tabs[ anIndex]) div 2;
          LTextY := LayoutR.Top  + ( LayoutR.Bottom - LayoutR.Top ) div 2 + aCanvas.TextWidth ( Tabs[ anIndex]) div 2;
          aCanvas.Font.Color := LTextColor;
          AngleTextOut2( aCanvas, 900, LTextX, LTextY, Tabs[ anIndex]);
        end
        else if TabPosition = tpRight
        then begin
          LTextX := LayoutR.Left + ( LayoutR.Right  - LayoutR.Left) div 2 + aCanvas.TextHeight( Tabs[ anIndex]) div 2;
          LTextY := LayoutR.Top  + ( LayoutR.Bottom - LayoutR.Top ) div 2 - aCanvas.TextWidth ( Tabs[ anIndex]) div 2;
          aCanvas.Font.Color := LTextColor;
          AngleTextOut2( aCanvas, - 900, LTextX, LTextY, Tabs[anIndex]);
        end
        else DrawControlText( Tabs[ anIndex], LayoutR, DT_VCENTER or DT_CENTER or DT_SINGLELINE  or DT_NOCLIP);
      end;
    end;


{ ========
  TKnobsComponent = class
  private
    FComponent      : TComponent;
    FComponentClass : TComponentClass;
  public
}

    constructor TKnobsComponent.Create( const aComponent: TComponent);
    begin
      FComponent      := aComponent;
      FComponentClass := TComponentClass( aComponent.ClassType);
    end;


{ ========
  TKnobsComponentData = class( TDictionary<string, TKnobsComponent>)
  protected
}

    procedure   TKnobsComponentData.ValueNotify( const aValue: TKnobsComponent; anAction: TCollectionNotification); // override;
    begin
      inherited;

      if   anAction = cnRemoved
      then aValue.DisposeOf;
    end;


//  public

    procedure   TKnobsComponentData.InsertComponent( const aComponent: TComponent);
    var
      C : TKnobsComponent;
    begin
      if   Assigned( aComponent) and ( aComponent.Name <> '')
      then begin
        C := TKnobsComponent.Create( aComponent);
        Add( aComponent.Name , C)
      end;
    end;


    procedure   TKnobsComponentData.RemoveComponent( const aComponent: TComponent);
    begin
      if   Assigned( aComponent) and ( aComponent.Name <> '')
      then Remove( aComponent.Name);
    end;


    function    TKnobsComponentData.Lookup( const aKey: string; const aClass: TComponentClass): TKnobsComponent;
    begin
      if   not TryGetValue( aKey, Result)
      then Result := nil
      else begin
        if   ( Assigned( aClass) and ( not Result.FComponentClass.InheritsFrom( aClass)))
        then Result := nil;
      end;
    end;


{ ========
  TKnobsGraphicControl = class( TGraphicControl)
  public
    property    Module        : TKnobsCustomModule  read GetModule;
    property    ModuleName    : string              read GetModuleName;
  private
}

    function    TKnobsGraphicControl.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then result := TKnobsCustomModule( Owner)
      else result := nil;
    end;


    function    TKnobsGraphicControl.GetModuleName: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


{ ========
  TKnobsData = class( TGraphicControl)
  private
    FShortName : string;
    FSize      : Integer;
    FData      : TSignalArray;
    FComment   : string;
    FOnChanged : TKnobsOnTextChanged;
  public
    property    Data     : TSignalArray           read GetData     write SetData;
    property    AsString : string                 read GetAsString write SetAsString;
    property    Module   : TKnobsCustomModule     read GetModule;
  published
    property    TextChanged : TKnobsOnTextChanged read FOnChanged  write FOnChanged;
    property    Size     : Integer                read FSize       write SetSize;
    property    Comment  : string                 read FComment    write FComment;
    property    Width                                                                                        default 12;
    property    Height                                                                                       default 12;
  private
}

    function    TKnobsData.CheckChanged( const aNewValue: TSignalArray): Boolean;
    var
      i : Integer;
    begin
      Result := Length( aNewValue) <> Length( FData);

      if   not Result
      then begin
        for i := 0 to Length( FData) - 1
        do begin
          if   FData[ i] <> aNewValue[ i]
          then begin
            Result := True;
            Break;
          end;
        end;
      end;
    end;


    procedure   TKnobsData.Changed;
    var
      aModule : TKnobsCustomModule;
    begin
      FStrVal := AsString;

      if   Assigned( FOnChanged)
      then FOnChanged( Self, Name, FStrVal)
      else begin
        aModule := Module;

        if   Assigned( aModule)
        then Module.TextChanged( Self, MakePath( [ aModule.Name, Name]), FStrVal);
      end;
    end;


    procedure   TKnobsData.SetSize( aValue: Integer);
    var
      NewData : TSignalArray;
      OldData : TSignalArray;
    begin
      if   aValue <> FSize
      then begin
        SetLength( NewData, aValue);
        OldData := FData;
        Move( FData[ 0], NewData[ 0], Min( aValue, FSize) * SizeOf( TSignal));

        if   aValue > FSize
        then begin
          FData := NewData;
          FSize := aValue;
        end
        else begin
          FSize := aValue;
          FData := NewData;
        end;

        SetLength( OldData, 0);
        Changed;
      end;
    end;


    function    TKnobsData.GetData: TSignalArray;
    begin
      SetLength( Result, FSize);
      Move( FData[ 0], Result[ 0], FSize * SizeOf( TSignal));
    end;


    procedure   TKnobsData.SetData( const aValue: TSignalArray);
    var
      DoChanged : Boolean;
    begin
      DoChanged := CheckChanged( aValue);

      if   DoChanged
      then begin
        if   Length( aValue) > Length( FData)
        then begin
          SetLength( FData, Length( aValue));
          FSize := Length( aValue);
          Move( aValue[ 0], FData[ 0], FSize * SizeOf( TSignal));
        end
        else begin
          FSize := Length( aValue);
          SetLength( FData, Length( aValue));
          Move( aValue[ 0], FData[ 0], FSize * SizeOf( TSignal));
        end;

        FStrVal := SignalArrayToStr( FData);
        Changed;
      end;
    end;


//  protected

    procedure   TKnobsData.TextChanged( const aNewValue: string; DoNotify: Boolean); // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnChanged)
      then FOnChanged( Self, Name, aNewValue)
      else begin
        aModule := Module;

        if   Assigned( aModule) and DoNotify
        then Module.TextChanged( Self, MakePath( [ aModule.Name, Name]), aNewValue);
      end;
    end;


    procedure   TKnobsData.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsData.WMNCHitTest( var aMessage: TWMNCHitTest); // message WM_NCHITTEST;
    begin
      if   csDesigning in ComponentState
      then inherited
      else aMessage.Result := HTTRANSPARENT;
    end;


    procedure   TKnobsData.Paint; // override;
    begin
      if   csDesigning in ComponentState
      then begin
        Canvas.Pen.Color := clBlack;
        Canvas.Rectangle( 0, 0, Width, Height);
        Canvas.Font.Height := 8;
        Canvas.TextOut( 4, 2, 'D');
      end;
    end;


//  public

    function    TKnobsData.GetAsString: string;
    begin
      Result  := FStrVal;
    end;


    procedure   TKnobsData.SetAsString( const aValue: string);
    var
      Data : TSignalArray;
    begin
      if   aValue <> FStrVal
      then begin
        FStrVal := aValue;
        Data    := StrToSignalArray( FStrVal);

        if   CheckChanged( Data)
        then begin
          FData := Data;
          Changed;
        end;
      end;
    end;


    function    TKnobsData.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then result := TKnobsCustomModule( Owner)
      else result := nil;
    end;


//  public

    constructor TKnobsData.Create( aOwner: TComponent); // override;
    begin
      inherited;
      Size    := 0;
      Width   := 12;
      Height  := 12;
      FStrVal := '';
    end;


    destructor  TKnobsData.Destroy; // override;
    begin
      Size := 0;
      inherited;
    end;


{ ========
  TKnobsLed = class( TKnobsGraphicControl)
  private
    FTime       : Cardinal;
    FTimer      : TTimer;
    FColorOff   : TColor;
    FColorOn    : TColor;
    FHighLight  : Boolean;
    FActive     : Boolean;
    FShape      : TKnobsLedShape;
  published
    property    Time        : Cardinal       read FTime        write SetTime        default 0;
    property    Active      : Boolean        read FActive      write SetActive      default True;
    property    ColorBorder : TColor         read FColorBorder write SetColorBorder default clGray;
    property    ColorOn     : TColor         read FColorOn     write SetColorOn     default clLime;
    property    ColorOff    : TColor         read FColorOff    write SetColorOff    default clGreen;
    property    HighLight   : Boolean        read FHighLight   write SetHighLight   default True;
    property    Shape       : TKnobsLedShape read FShape       write SetShape       default lsRound;
    property    Height                                                              default 15;
    property    Width                                                               default 15;
    property    Hint;
    property    ShowHint                                                            default False;
    property    ParentShowHint                                                      default False;
    property    Dragcursor;
    property    Dragmode;
    property    OnDragDrop;
    property    OnDragOver;
    property    OnEndDrag;
    property    OnMouseDown;
    property    OnMouseMove;
    property    OnMouseUp;
    property    ParentColor                                                         default False;
    property    OnClick;
  private
}

    procedure   TKnobsLed.TimerFired( aSender: TObject);
    begin
      if   FTime <> 0
      then begin
        Active := False;
        Stoptimer;
      end;
    end;


    procedure   TKnobsLed.StartTimer( aValue: Cardinal);
    begin
      if   not ( csDesigning in ComponentState)
      then begin
        if   not Assigned( FTimer)
        then FTimer := TTimer.Create( nil);

        if   Assigned( FTimer)
        then begin
          FTimer.OnTimer  := nil;
          FTimer.Enabled  := False;
          FTimer.Interval := FTime;
          FTimer.Enabled  := True;
          FTimer.OnTimer  := TimerFired;
        end;
      end;
    end;


    procedure   TKnobsLed.StopTimer;
    begin
      if   Assigned( FTimer)
      then begin
        FTimer.OnTimer := nil;
        FTimer.Enabled := False;
        FreeAndNil( FTimer);
      end;
    end;


    procedure   TKnobsLed.SetTime( aValue: Cardinal);
    begin
      if   aValue <> FTime
      then begin
        FTime := aValue;

        if   ( FTime > 0) and Active
        then Starttimer( FTime)
      end;
    end;


    procedure   TKnobsLed.SetActive( aValue: Boolean);
    begin
      if   csDestroying in ComponentState
      then Exit;

      if   aValue <> FActive
      then begin
        FActive := aValue;
        Invalidate;
      end;

      if   FActive and ( FTime > 0)
      then StartTimer( FTime)
      else StopTimer;
    end;


    procedure   TKnobsLed.SetColorBorder( aValue: TColor);
    begin
      if   csDestroying in ComponentState
      then Exit;

      if   aValue <> FColorBorder
      then begin
        FColorBorder := aValue;
        Invalidate;
      end;
    end;

    procedure   TKnobsLed.SetColorOn( aValue: TColor);
    begin
      if   csDestroying in ComponentState
      then Exit;

      if   aValue <> FColorOn
      then begin
        FColorOn := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsLed.SetColorOff( aValue: TColor);
    begin
      if   csDestroying in ComponentState
      then Exit;

      if   aValue <> FColorOff
      then begin
        FColorOff := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsLed.SetHighLight( aValue: Boolean);
    begin
      if   csDestroying in ComponentState
      then Exit;

      if   aValue <> FHighLight
      then begin
        FHighLight := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsLed.SetShape( aValue: TKnobsLedShape);
    begin
      if   csDestroying in ComponentState
      then Exit;

      if   aValue <> FShape
      then begin
        FShape := aValue;
        Invalidate;
      end;
    end;


//  protected

    procedure   TKnobsLed.Paint; // override;
    begin
      Canvas.Pen.Width := 1;
      Canvas.Pen.Color := ColorBorder;

      if   Active
      then Canvas.Brush.Color := ColorOn
      else begin
        Canvas.Brush.Color := ColorOff;

        if   ParentColor
        then Canvas.Brush.Style := bsClear;
      end;

      case Shape of

        lsRound :

          begin
            Canvas.Ellipse( 0, 0, Width, Height);

            if   FHighLight
            then begin
              Canvas.Pen.Color := clWhite;
              Canvas.Pen.Width := 2;
              Canvas.Arc(
                Width  * 3 div 4         ,
                Height * 3 div 4         ,
                Width  - Width  * 3 div 4,
                Height - Height * 3 div 4,
                Width  div 2             ,
                0                        ,
                0                        ,
                Height div 2
              );
            end;
          end;

        lsSquare :

          begin
            Canvas.Rectangle( 0, 0, Width, Height);
          end;

      end;
    end;


//  public

    constructor TKnobsLed.Create( aOwner: TComponent); // override;
    begin
      inherited;
      Height         := 15;
      Width          := 15;
      Shape          := lsRound;
      ColorBorder    := clGray;
      ColorOn        := clLime;
      ColorOff       := clGreen;
      HighLight      := True;
      Active         := True;
      ParentColor    := False;
      ShowHint       := False;
      ParentShowHint := False;
    end;


    destructor  TKnobsLed.Destroy; // override;
    begin
      StopTimer;
      inherited;
    end;


    procedure   TKnobsLed.Toggle;
    begin
      if   csDestroying in ComponentState
      then Exit;

      Active := not Active;
    end;


{ ========
  TKnobsClicker = class( TSpeedButton)
  // Repeating speed button, semi flat ...
  private
    FRepeatTimer : TTimer;
    FRepeatCount : Integer;
    FCanRepeat   : Boolean;
    FMouseDown   : Boolean;
    FShowFocus   : Boolean;
    FBitmap      : TBitmap;
  private
    property    ShowFocus: Boolean read FShowFocus write SetShowFocus;
  public
    property    CanRepeat: Boolean read FCanRepeat write FCanRepeat default True;
  private
}

    procedure   TKnobsClicker.TimerExpired( aSender: TObject);
    begin
      with FRepeatTimer
      do begin
        if   FRepeatCount > SlowRepeats
        then Interval := FastRepeatPause
        else Interval := SlowRepeatPause;
      end;

      if   FMouseDown and MouseCapture
      then begin
        try
          Click;
        except
          FRepeatTimer.Enabled := False;
          raise;
        end;
      end;

      Inc( FRepeatCount);
    end;


    procedure   TKnobsClicker.SetShowFocus( aValue: Boolean);
    begin
      if
        ( Parent is TKnobsNoKnob ) or
        ( Parent is TKnobsSlider ) or
        ( Parent is TKnobsHSlider)
      then begin
        Flat   := False;
        Glyph  := FBitmap;
        Invalidate;
      end
      else begin
        if   aValue <> FShowFocus
        then begin
          FShowFocus  := aValue;
          Flat        := not FShowFocus;

          if   ShowFocus
          then Glyph := FBitmap
          else Glyph := nil;

          Invalidate;
        end;
      end;
    end;


//  private

    function    TKnobsClicker.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


//  protected

    procedure   TKnobsClicker.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP : TKnobsWirePanel;
    begin
      inherited;

      if   FMouseDown
      then Exit;

      if   ( Button = mbLeft) and CanRepeat
      then begin
        FMouseDown   := True;
        FRepeatCount := 0;

        if   not Assigned( FRepeatTimer)
        then FRepeatTimer := TTimer.Create( Self);

        with FRepeatTimer
        do begin
          OnTimer  := TimerExpired;
          Interval := InitRepeatPause;
          Enabled  := True;
        end;
      end
      else if   ( Button = mbRight) and Assigned( Parent) and ( Parent is TKnobsKnob)
      then begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.HandleRightClick( Parent);
      end;
    end;


    procedure   TKnobsClicker.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   Assigned( FRepeatTimer)
      then FRepeatTimer.Enabled  := False;

      FMouseDown := False;
    end;


    procedure   TKnobsClicker.AssignBitmap( aBitmap: TBitmap);
    begin
      FBitmap := aBitmap;
      Invalidate;
    end;


//  public

    constructor TKnobsClicker.Create( anOwner: TComponent); // override;
    begin
      inherited;

      CanRepeat := True;

      if
        ( anOwner is TKnobsNoKnob ) or
        ( anOwner is TKnobsSlider ) or
        ( anOwner is TKnobsHSlider)
      then begin
        Transparent := False;
        Flat        := False;
      end
      else begin
        Transparent := True;
        Flat        := True;
      end;

      StyleElements := [];
    end;


    destructor  TKnobsClicker.Destroy; // override;
    begin
      if   Assigned( FRepeatTimer)
      then FRepeatTimer.DisposeOf;

      inherited;
    end;


    procedure   TKnobsClicker.Click; // override;
    begin
      inherited;

      if   HasParent
      then Parent.SetFocus;
    end;


{ ========
  TKnobsEditData = record
  private
    FAssociate : TKnobsValuedControl;
    FSelected  : Boolean;
    FTimeStamp : TKnobsTimeStamp;
    FKind      : TKnobsEditKind;
    FPath      : string;
    FCType     : string;
    FPrevValue : TValue;
    FValue     : TValue;
    FPrevPos   : Integer;
    FPosition  : Integer;
    FSteps     : Integer;
  public
}

    procedure   TKnobsEditData.Create(
      const anAssociate : TKnobsValuedControl;
      const aTimeStamp  : TKnobsTimeStamp;
      const aKind       : TKnobsEditKind;
      const aPath       : string
    );
    begin
      FAssociate := anAssociate ;
      FTimeStamp := aTimeStamp  ;
      FKind      := aKind       ;
      FPath      := aPath       ;

      if   Assigned( anAssociate)
      then begin
        FVariation := anAssociate.ActiveVariation;
        FCType     := anAssociate.ControlType;
        FPrevPos   := anAssociate.PrevPosition;
        FPosition  := anAssociate.KnobPosition;
        FSteps     := anAssociate.StepCount;
      end
      else FKind := wekNothing;

      case FKind of

        wekNothing  :

          begin
            FPrevValue := 0;
            FValue     := 0;
          end;

        wekKnob     :

          begin
            FPrevValue := anAssociate.AsPrevValue;
            FValue     := anAssociate.AsValue;
          end;

        wekSelector :

          begin
            FPrevValue := anAssociate.AsPrevDisplay;
            FValue     := anAssociate.AsDisplay;
          end;
      end;

    end;


    function    TKnobsEditData.CreateFromString( const anEditor: TKnobsWirePanel; const aStringValue: string): Boolean;
    var
      aItems       : TStringList;
      bItems       : TStringList;
      aVariation   : Integer;
      aTimeStamp   : TKnobsTimeStamp;
      aPath        : string;
      anFPath      : string;
      aPrevPos     : Integer;
      aPos         : Integer;
      aSteps       : Integer;
      aModuleName  : string;
      aControlName : string;
      aModule      : TKnobsCustomModule;
      anAssociate  : TKnobsValuedControl;
      aControlType : string;
      aKind        : TKnobsEditKind;
      aPrevValue   : TValue;
      aValue       : TValue;
    begin
      Result := False;

      if   Assigned( anEditor) and not aStringValue.IsEmpty
      then begin
        aItems := Explode( aStringValue, ',');

        try
          if   aItems.Count = 6
          then begin
            TrimStrings( aItems);
            aTimeStamp   := StrToIntDef( aItems[ 0], -1);
            aVariation   := StrToIntDef( aItems[ 1],  0);
            anFPath      := StripQuotes( aItems[ 2], '''', '''');
            aPath        := ModuleFriendlyToLongName( anFPath);
            aControlName := StripQuotes( aItems[ 3], '''', '''');
            aPrevPos     := StrToIntDef( aItems[ 4], -1);
            aPos         := StrToIntDef( aItems[ 5], -1);

            if  ( aTimeStamp >= 0         )
            and ( aPos       >= 0         )
            and ( aPrevPos   >= 0         )
            and ( not aControlName.IsEmpty)
            then begin
              bItems := Explode( aPath, ' ');

              try
                if   bItems.Count = 3
                then begin
                  TrimStrings( bItems);
                  aModuleName  := bItems[ 0];
                  aModule      := anEditor.FindModule( aModuleName);

                  if   Assigned( aModule)
                  then begin
                    aModule.FixControlInsertions;
                    anAssociate := aModule.FindValuedControl( aControlName);

                    if   Assigned( anAssociate)
                    then begin
                      aSteps       := anAssociate.StepCount;
                      aControlType := anAssociate.ControlType;

                      if   anAssociate is TKnobsSelector
                      then begin
                        aKind      := wekSelector;
                        aValue     := Converters.PosToValue( aControlType, aPos    , aSteps);
                        aPrevValue := Converters.PosToValue( aControlType, aPrevPos, aSteps);
                      end
                      else begin
                        aKind      := wekKnob;
                        aValue     := Converters.PosToValue( aControlType, aPos    , aSteps);
                        aPrevValue := Converters.PosToValue( aControlType, aPrevPos, aSteps);
                      end;

                      FAssociate := anAssociate;
                      FSelected  := False;
                      FVariation := aVariation;
                      FTimeStamp := aTimeStamp;
                      FKind      := aKind;
                      FPath      := anFPath;
                      FCType     := aControlType;
                      FPrevValue := aPrevValue;
                      FValue     := aValue;
                      FPrevPos   := aPrevPos;
                      FPosition  := aPos;
                      FSteps     := aSteps;
                      Result     := True;
                    end;
                  end;
                end;
              finally
                bItems.DisposeOf;
              end;
            end;
          end;
        finally
          aItems.DisposeOf;
        end;
      end
    end;


    function    TKnobsEditData.AsString: string;
    begin
      case FKind of
        wekKnob :
          begin
            if   ( FSteps < 0) or ( FPosition < 0) or ( FPrevPos < 0)
            then Result := Format( '%s | %d | %g | %g', [ FPath, FVariation, FPrevValue.AsExtended, FValue.AsExtended], AppLocale)
            else Result := Format( '%s | %d | %s | %s', [ FPath, FVariation, Converters.PosToDisplay( FCType, FPrevPos, FSteps), Converters.PosToDisplay( FCType, FPosition, FSteps)], AppLocale);
          end;
        wekSelector :
                 Result := Format( '%s | %d | %s | %s', [ FPath, FVariation, FPrevValue.AsString, FValue.AsString], AppLocale);
        else     Result := Format( '%s | ? | ? | ?'   , [ FPath                                                  ], AppLocale);
      end;
    end;


    function    TKnobsEditData.AsFileString: string;
    begin
      case FKind of

        wekKnob, wekSelector :

          if   Assigned( FAssociate)
          then
            Result :=
              Format(
                '%d, %d, ''%s'', ''%s'', %d, %d',
                [
                  FTimeStamp     ,
                  FVariation     ,
                  FPath          ,
                  FAssociate.Name,
                  FPrevPos       ,
                  FPosition
                ],
                AppLocale
              )
            else Result := '';

        else Result := '';
      end;
    end;


    procedure   TKnobsEditData.Select( DoSelect: Boolean);
    begin
      FSelected := DoSelect;
    end;


    procedure   TKnobsEditData.UpdateNames( const anOldFriendlyNames, aNewFriendlyNames: TStringArray);
    var
      i : Integer;
      O : string;
      N : string;
    begin
        for    i := Min( Length( anOldFriendlyNames), Length( aNewFriendlyNames)) - 1
        downto 0
        do begin
          O := anOldFriendlyNames[ i];

          if   FPath.Contains( O)
          then begin
            N     := aNewFriendlyNames[ i];
            FPath := StringReplace( FPath, O, N, [ rfReplaceAll, rfIgnoreCase]);
            Break;
          end;
        end;
    end;


//  public

    function    TKnobsEditData.GetAssociate: TKnobsValuedControl;
    begin
      Result := FAssociate;
    end;


{ ========
  TKnobsEditHistory = class( TInterfacedObject, IKnobsUndoable)
  private
    FAssociate      : TKnobsValuedControl;
    FData           : TArray<TKnobsEditData>;
    FSize           : Integer;
    FAllowUnchanged : Boolean;
    FmaxStrLen      : Integer;
    FFreeCount      : Integer;
    FFillCount      : Integer;
    FInp            : Integer;
    FOutp           : Integer;
    FAllowNonAuto   : Boolean;
    FRecordAll      : Boolean;
  public
    property    AllowNonAuto            : Boolean        read FAllowNonAuto     write FAllowNonAuto;
    property    FRecordAll              : Boolean        read FRecordAll        write FRecordAll;
    property    Size                    : Integer        read GetSize           write SetSize;
    property    Item[ anIndex: Integer] : TKnobsEditData read GetItem           write SetItem;                  default;
    property    Data[ anIndex: Integer] : PKnobsEditData read GetData;
    property    AllowUnchanged          : Boolean        read GetAllowUnchanged write SetAllowUnchanged;
  private
}

    function    TKnobsEditHistory.GetSize: Integer;
    begin
      Result := FSize;
    end;


    procedure   TKnobsEditHistory.SetSize( aValue: Integer);
    begin
      if   aValue <> FSize
      then begin
        FFreeCount := 0;
        FFillCount := 0;
        FSize      := aValue;
        SetLength( FData, FSize);
        Clear;
      end;
    end;


    procedure   TKnobsEditHistory.SetAllowUnchanged( aValue: Boolean);
    begin
      FAllowUnchanged := aValue;
    end;


    function    TKnobsEditHistory.GetAllowUnchanged: Boolean;
    begin
      Result := FAllowUnchanged;
    end;


    function    TKnobsEditHistory.GetItem( anIndex: Integer): TKnobsEditData;
    begin
      if  ( anIndex >= 0       )
      and ( anIndex < FillCount)
      then Result := FData[ MathIntMod( FOutp + anIndex, FSize)]
      else begin
        Result.FAssociate := nil;
        Result.FSelected  := False;
        Result.FTimeStamp := -1;
        Result.FKind      := wekNothing;
        Result.FPath      := '';
        Result.FCType     := '';
        Result.FPrevValue := 0;
        Result.FValue     := 0;
        Result.FPrevPos   := -1;
        Result.FPosition  := -1;
        Result.FSteps     := 0;
      end;
    end;


    procedure   TKnobsEditHistory.SetItem( anIndex: Integer; const aValue: TKnobsEditData);
    begin
      if  ( anIndex >= 0       )
      and ( anIndex < FillCount)
      then FData[ MathIntMod( FOutp + anIndex, FSize)] := aValue;
    end;


    function    TKnobsEditHistory.GetData( anIndex: Integer): PKnobsEditData;
    begin
      if  ( anIndex >= 0       )
      and ( anIndex < FillCount)
      then Result := @ FData[ MathIntMod( FOutp + anIndex, FSize)]
      else Result := nil;
    end;


//  public

    constructor TKnobsEditHistory.Create(
      const anAssociate : TKnobsValuedControl;
      aHistorySize      : Integer;
      aMaxStringLength  : Integer;
      DoAllowUnchanged  : Boolean
    ); // overload;
    begin
      inherited Create;
      FAllowNonAuto   := True;
      FRecordAll      := True;
      FAssociate      := anAssociate;
      FSize           := aHistorySize;
      FmaxStrLen      := aMaxStringLength;
      FAllowUnchanged := DoAllowUnchanged;
      SetLength( FData, FSize);
      Clear;
    end;


    constructor TKnobsEditHistory.Create( const aSource: TKnobsEditHistory); // overload;
    begin
      inherited Create;
      CopyFrom( aSource);
    end;


    destructor  TKnobsEditHistory.Destroy; // override;
    begin
      FFreeCount := 0;
      FFillCount := 0;
      SetLength( FData, 0);
      inherited;
    end;


    procedure   TKnobsEditHistory.Clear;
    begin
      FInp       := 0;
      FOutp      := 0;
      FFillCount := 0;
      FFreeCount := FSize;
    end;


    procedure   TKnobsEditHistory.AddData( const aData: TKnobsEditData);
    var
      anAssociate : TKnobsValuedControl;
    begin
      AnAssociate := aData.GetAssociate;

      if   Assigned( anAssociate)
      then begin
        if   CanRecord( anAssociate)
        then begin
          if   FFreeCount = 0
          then begin
            FOutp := ( FOutp + 1) mod FSize;
            Inc( FFreeCount);
            Dec( FFillCount);
          end;

          FData[ FInp] := aData;
          FInp         := ( FInp + 1) mod FSize;
          Dec( FFreeCount);
          Inc( FFillCount);
        end;
      end;
    end;


    procedure   TKnobsEditHistory.AddEditFor( const aTimeStamp: TKnobsTimeStamp; const aPrefix: string; const anAutomationData: TKnobsEditHistory);
    var
      aData : TKnobsEditData;
      aPath : string;
    begin
      if   Assigned( anAutomationData)
      then begin
        if   anAutomationData.CanRecord( FAssociate)
        then begin
          if   aPrefix = ''
          then aPath := FAssociate.ShortName
          else aPath := Format( '%s   %s', [ aPrefix, FAssociate.ShortName], AppLocale);

          if   FAssociate is TKnobsSelector
          then begin
            aData.
              Create(
                FAssociate ,
                aTimeStamp ,
                wekSelector,
                aPath
              );
          end
          else begin
            aData.
              Create(
                FAssociate,
                aTimeStamp,
                wekKnob   ,
                aPath
              );
          end;

          anAutomationData.AddData( aData);
        end;
      end;
    end;


    procedure   TKnobsEditHistory.AddEdit( const aTimeStamp: TKnobsTimeStamp); // overload;
    begin
      AddEditFor( aTimeStamp, '', Self);
    end;


    procedure   TKnobsEditHistory.FillStrings( const aList: TStrings; FileFill: Boolean; aMaxItems: Integer);
    var
      i : Integer;
      p : Integer;
      S : string;
    begin
      if   Assigned( aList)
      then begin
        aList.Clear;

        if   FileFill
        then begin
          p := FOutp;
          aList.BeginUpdate;

          try
            for i := 0 to Min( FFillCount, aMaxItems) - 1
            do begin
              S := FData[ p].AsFileString;
              p := MathIntMod( p + 1, FSize);

              if   S <> ''
              then aList.Add( S);
            end;
          finally
            aList.EndUpdate;
          end;
        end
        else begin
          p := MathIntMod( FInp - 1, FSize);
          aList.BeginUpdate;

          try
            for i := 0 to Min( FFillCount, aMaxItems) - 1
            do begin
              S := FData[ p].AsString;
              p := MathIntMod( p - 1, FSize);

              if   S <> ''
              then aList.Add( S);
            end;
          finally
            aList.EndUpdate;
          end;
        end;
      end;
    end;


    function    TKnobsEditHistory.CreateStringList( aMaxItems: Integer): TStringList;
    begin
      Result := TStringList.Create;
      FillStrings( Result, False, aMaxItems);
    end;


    procedure   TKnobsEditHistory.UndoItem( anIndex: Integer);
    var
      p : Integer;
    begin
      if   ( anIndex >= 0) and ( anIndex < FFillCount)
      then begin
        p := MathIntMod( FInp - anIndex - 1, FSize);
        FData[ p].Select( False);

        if   Assigned( FAssociate)
        then begin
          FAssociate.FPrevPosition := FAssociate.KnobPosition;
          FAssociate.KnobPosition  := FData[ p].FPrevPos;
          FAssociate.AddToHistory;                                  // reinsert the undone one as last for a redo
        end
        else if Assigned( FData[ p].FAssociate)
        then begin
          FData[ p].FAssociate.FPrevPosition := FData[ p].FAssociate.KnobPosition;
          FData[ p].FAssociate.KnobPosition  := FData[ p].FPrevPos;
          FData[ p].FAssociate.AddToHistory;                        // reinsert the undone one as last for a redo
        end;
      end;
    end;


    procedure   TKnobsEditHistory.PlayItem( anIndex: Integer);
    var
      p : Integer;
    begin
      if   ( anIndex >= 0) and ( anIndex < FFillCount)
      then begin
        p := MathIntMod( FOutp + anIndex, FSize);

        if   Assigned( FAssociate)
        then FAssociate.PlayPosition := FData[ p].FPosition
        else if Assigned( FData[ p].FAssociate)
        then FData[ p].FAssociate.PlayPosition := FData[ p].FPosition;
      end;
    end;


    function    TKnobsEditHistory.Last: TKnobsEditData;
    begin
      if   FFillCount > 0
      then Result := FData[ MathIntMod( FInp - 1, FSize)]
      else begin
        Result.FKind      := wekNothing;
        Result.FPath      := '';
        Result.FCType     := '';
        Result.FPrevValue :=  0;
        Result.FValue     :=  0;
        Result.FPrevPos   := -1;
        Result.FPosition  := -1;
        Result.FSteps     := -1;
      end;
    end;


    function    TKnobsEditHistory.FreeCount: Integer;
    begin
      Result := FFreeCount;
    end;


    function    TKnobsEditHistory.FillCount: Integer;
    begin
      Result := FFillCount;
    end;


    procedure   TKnobsEditHistory.SaveToStringList( const aStringList: TStringList; const aGuid: string);
    begin
      if   Assigned( aStringList)
      then begin
        FillStrings( aStringList, True, Size);
        aStringList.Insert( 0, aGuid);
      end;
    end;


    procedure   TKnobsEditHistory.LoadFromStringList( const anEditor: TKnobsWirePanel; const aStringList: TStringList; const aGuid: string);
    var
      i     : Integer;
      aData : TKnobsEditData;
    begin
      if  Assigned( aStringList)
      and ( aStringList.Count > 0)
      then begin
        if   aGuid = aStringList[ 0]
        then begin
          Clear;

          for i := 1 to aStringList.Count - 1
          do begin
            if   aData.CreateFromString( anEditor, aStringList[ i])
            then AddData( aData);
          end;
        end;
      end;
    end;


    procedure   TKnobsEditHistory.CopyFrom( const aHistory: TKnobsEditHistory);
    var
      i : Integer;
      p : Integer;
    begin
      Clear;

      if   Assigned( aHistory)
      then begin
        AllowNonAuto := aHistory.AllowNonAuto;
        RecordAll    := aHistory.RecordAll;

        if   Size < aHistory.Size
        then Size := aHistory.Size;

        p := aHistory.FOutp;

        for i := 0 to aHistory.FFillCount - 1
        do begin
          AddData( aHistory.FData[ p]);
          p := MathIntMod( p + 1, aHistory.FSize);
        end;
      end;
    end;


    procedure   TKnobsEditHistory.DeselectAll;
    var
      i : Integer;
      p : Integer;
    begin
      p := FOutp;

      for i := 0 to FFillCount - 1
      do begin
        FData[ p].Select( False);
        p := MathIntMod( p + 1, FSize);
      end;
    end;

    procedure   TKnobsEditHistory.SelectItem( anIndex: Integer);
    begin
      if   ( anIndex >= 0) and ( anIndex < FFillCount)
      then FData[ MathIntMod( FInp - anIndex - 1, FSize)].Select( True);
    end;


    function    TKnobsEditHistory.IsItemSelected( anIndex: Integer): Boolean;
    begin
      if   ( anIndex >= 0) and ( anIndex < FFillCount)
      then Result := FData[ MathIntMod( FInp - anIndex - 1, FSize)].FSelected
      else Result := False;
    end;


    procedure   TKnobsEditHistory.UndoSelected;
    // This undoes selected items, reversed, newest one is undone first
    var
      i : Integer;
    begin
      i := 0;

      while i < FFillCount
      do begin
        if   IsItemSelected( i)
        then UndoItem( i);

        Inc( i);
      end;
    end;


    procedure   TKnobsEditHistory.UpdateNames( const anOldFriendlyNames, aNewFriendlyNames: TStringArray);
    var
      i : Integer;
      p : Integer;
    begin
      p := FOutp;

      for i := 0 to FFillCount - 1
      do begin
        FData[ p].UpdateNames( anOldFriendlyNames, aNewFriendlyNames);
        p := MathIntMod( p + 1, FSize);
      end;
    end;


    function    TKnobsEditHistory.CanRecord( aControl: TKnobsValuedControl): Boolean;
    begin
      Result :=
        Assigned( aControl)                                     // When there is no associate there is no data
        and (
              AllowUnchanged                                    // Except when pulses are allowed ...
          or ( aControl.PrevPosition <> aControl.KnobPosition)  // ... unchanged positions need not be saved.
        )
        and ( aControl.AllowAutomation or AllowNonAuto)
        and ( aControl.AllowRecording  or RecordAll   );
    end;


    procedure   TKnobsEditHistory.RemoveControl( const aControl: TKnobsValuedControl);
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to FillCount - 1
      do begin
        if Item[ i].GetAssociate = aControl
        then begin
          Nop;

          for j := i to FillCount - 2
          do Item[ j] := Item[ j + 1];

          if FOutp < FSize - 1
          then Inc( FOutp)
          else FOutp := 0;

          Dec( FFillCount);
          Inc( FFreeCount);
        end;
      end;
    end;


{ ========
  TKnobsRange = record
    Name      : string;
    IsMorph   : Boolean;
    LowValue  : TSignal;
    HighValue : TSignal;
  end;


  TKnobsRangeData = TArray<TKnobsRange>;


  TKnobsRanges = class
  private
    FData        : TKnobsRangeData;
    FSpecs       : PKnobsRangeSpecs;
    FActiveRange : Integer;
  public
    property    HasActiveRange                 : Boolean     read GetHasActiveRange;
    property    ActiveRange                    : Integer     read FActiveRange      write FActiveRange;
    property    Count                          : Integer     read GetCount;
    property    Range      [ anIndex: Integer] : TKnobsRange read GetRange          write SetRange;
    property    Name       [ anIndex: Integer] : string      read GetName;
    property    IsMorph    [ anIndex: Integer] : Boolean     read GetIsMorph;
    property    LowValue   [ anIndex: Integer] : TSignal     read GetLowValue       write SetLowValue;
    property    HighValue  [ anIndex: Integer] : TSignal     read GetHighValue      write SetHighValue;
    property    HasRangeFor[ anIndex: Integer] : Boolean     read GetHasRangeFor;
  private
}

    function    TKnobsRanges.GetHasActiveRange : Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to Count - 1
      do begin
        if   HasRangeFor[ i]
        then begin
          Result := True;
          Break;
        end;
      end;
    end;


    function    TKnobsRanges.GetCount: Integer;
    begin
      Result := Length( FData);
    end;


    function    TKnobsRanges.GetLowRangeAsStr: string;
    var
      i : Integer;
    begin
      Result := '';

      if   HasActiveRange
      then begin
        for i := 0 to RangeCount - 1
        do begin
          if   Result = ''
          then Result := Format( '%g'    , [         LowValue[ i]], AppLocale)
          else Result := Format( '%s, %g', [ Result, LowValue[ i]], AppLocale);
        end;
      end;
    end;


    procedure   TKnobsRanges.SetLowRangeAsStr( const aValue: string);
    var
      aParts : TStringList;
      i      : Integer;
      aVal   : TSignal;
    begin
      if   aValue <> ''
      then begin
        aParts := Explode( aValue, ',');

        try
          TrimStrings( aParts);

          for i := 0 to aParts.Count - 1
          do begin
            if   i >= RangeCount
            then Break
            else begin
              aVal := StrToFloatDef( aParts[ i], Nan);

              if   IsNan( aVal)
              then Break
              else LowValue[ i] := Clip( aVal, 0.0, 1.0);
            end;
          end;
        finally
          aParts.DisposeOf;
        end;
      end;
    end;


    function    TKnobsRanges.GetHighRangeAsStr: string;
    var
      i : Integer;
    begin
      Result := '';

      if   HasActiveRange
      then begin
        for i := 0 to RangeCount - 1
        do begin
          if   Result = ''
          then Result := Format( '%g'    , [         HighValue[ i]], AppLocale)
          else Result := Format( '%s, %g', [ Result, HighValue[ i]], AppLocale);
        end;
      end;
    end;


    procedure   TKnobsRanges.SetHighRangeAsStr( const aValue: string);
    var
      aParts : TStringList;
      i      : Integer;
      aVal   : TSignal;
    begin
      if   aValue <> ''
      then begin
        aParts := Explode( aValue, ',');

        try
          TrimStrings( aParts);

          for i := 0 to aParts.Count - 1
          do begin
            if   i >= RangeCount
            then Break
            else begin
              aVal := StrToFloatDef( aParts[ i], Nan);

              if   IsNan( aVal)
              then Break
              else HighValue[ i] := Clip( aVal, 0.0, 1.0);
            end;
          end;
        finally
          aParts.DisposeOf;
        end;
      end;
    end;


    function    TKnobsRanges.GetRange( anIndex: Integer): TKnobsRange;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < RangeCount)
      then Result := FData[ anIndex]
      else begin
        Result.LowValue  := 0.0;
        Result.HighValue := 1.0;
      end;
    end;


    procedure   TKnobsRanges.SetRange( anIndex: Integer; const aValue: TKnobsRange);
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < RangeCount)
      then FData[ anIndex] := aValue;
    end;


    function    TKnobsRanges.GetName( anIndex: Integer): string;
    begin
      if   Assigned( FSpecs)
      then Result := FSpecs^[ anIndex].Name
      else Result := 'undefined';
    end;


    function    TKnobsRanges.GetIsMorph( anIndex: Integer): Boolean;
    begin
      Result := Assigned( FSpecs) and FSpecs^[ anIndex].IsMorph;
    end;


    function    TKnobsRanges.GetValue( anIndex: Integer): TSignal;
    begin
      if   Assigned( FSpecs)
      then Result := FSpecs^[ anIndex].Value
      else Result := 0.0;
    end;


    function    TKnobsRanges.GetLowValue( anIndex: Integer): TSignal;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < RangeCount)
      then Result := FData[ anIndex].LowValue
      else Result := 0.0;
    end;


    procedure   TKnobsRanges.SetLowValue( anIndex: Integer; aValue: TSignal);
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < RangeCount)
      then begin
        aValue := Clip( aValue, 0.0, 1.0);
        FData[ anIndex].LowValue := aValue;
      end;
    end;


    function    TKnobsRanges.GetHighValue( anIndex: Integer): TSignal;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < RangeCount)
      then Result := FData[ anIndex].HighValue
      else Result := 1.0;
    end;


    procedure   TKnobsRanges.SetHighValue( anIndex: Integer; aValue: TSignal);
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < RangeCount)
      then begin
        aValue := Clip( aValue, 0.0, 1.0);
        FData[ anIndex].HighValue := aValue;
      end;
    end;


    function    TKnobsRanges.GetHasRangeFor( anIndex: Integer): Boolean;
    begin
      Result :=
          ( anIndex >= 0        )
      and ( anIndex < RangeCount)
      and (( FData[ anIndex].LowValue  > 0.0) or ( FData[ anIndex].HighValue < 1.0));
    end;


//  public

    constructor TKnobsRanges.Create( const aSpecs: PKnobsRangeSpecs);
    begin
      inherited Create;
      FSpecs := aSpecs;

      if   Assigned( aSpecs)
      then begin
        SetLength( FData, Length( aSpecs^));
        Clear;
      end;
    end;


    procedure   TKnobsRanges.Clear;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do begin
        LowValue [ i] := 0.0;
        HighValue[ i] := 1.0;
      end;
    end;


    procedure   TKnobsRanges.CopyFrom( const aValue: TKnobsRanges);
    var
      i : Integer;
    begin
      Clear;

      if   Assigned( aValue)
      then begin
        FSpecs := aValue.FSpecs;

        if   Assigned( FSpecs)
        then SetLength( FData, Length( FSpecs^))
        else SetLength( FData, 0);

        Clear;

        for i := 0 to Count - 1
        do FData[ i] := aValue.FData[ i];

        FActiveRange := aValue.FActiveRange;
      end;
    end;



{ ========
  TKnobsValuedControl = class( TCustomControl, IKnobsAutomatable, IKnobsVariation)
  // Base type for valued controls, ones that can have an effect on an external model
  private
    FEditHistory            : TKnobsEditHistory;          // A small local undo history
    FDisplay                : TKnobsDisplay;
    FShortName              : string;
    FControlSrc             : TKnobsControlSrc;           // User or automation
    FMIDIColor              : TColor;                     // Accent color when under MIDI control
    FFocusColor             : TColor;                     // Accent color when the control has keyboard and mouse(wheel) focus
    FMIDIFocusColor         : TColor;                     // Accent color when under MIDI control and having focus
    FFriendlyName           : string;
    F_DesignerRemarks       : string;                     // Remarks to be used at design time - informative only
    FDynamicStepCount       : Boolean;                    // True when StepCount is not set at design time but at run time (by some magic)
    FStepCount              : Integer;                    // Number of steps in the total knob travel
    FLocked                 : Boolean;
    FPrevPosition           : Integer;                    // Previous knob position
    FKnobPosition           : Integer;                    // Current  knob position
    FDefaultPosition        : Integer;                    // default  knob position;
    FControlType            : string;                     // The control's behaviour
    FDezippered             : Boolean;                    // Whether dezippering should be used
    FAllowAutomation        : Boolean;
    FIsIndicator            : Boolean;                    // When true the value is set from an external source
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FAllowRecording         : Boolean;
    FShowAllowRecording     : Boolean;
    FRanges                 : TKnobsRanges;
    FAssignedMIDICC         : Byte;
    FActiveVariation        : Integer;
    FVariationValues        : TKnobsVariationValues;      // Values for variations
    FCurrentValue           : TSignal;                    // Current variation value, after variation selection,
    FLiveMorph              : TSignal;                    // Current LiveMorph position
    FShowSimpleHint         : Boolean;
    FOnValueChanged         : TKnobsOnValueChange;        // Called with the new value
    FOnKnobPositionChanged  : TKnobsKnobPositionChanged;  // Called with the new position
    FPopupCount             : Integer;
    FPopupWindow            : THintWindow;
    FWheelSupport           : Boolean;
    FWheelSensitivity       : Integer;
    FWheelCounter           : Integer;
    FOnUnFocus              : TOnUnFocus;
  public
    property    ModuleName                        : string            read GetModuleName;
    property    ModuleTitle                       : string            read GetModuleTitle;
    property    LowRangeAsStr                     : string            read GetLowRangeAsStr          write SetLowRangeAsStr;
    property    HighRangeAsStr                    : string            read GetHighRangeAsStr         write SetHighRangeAsStr;
  public
    property    EditHistory                       : TKnobsEditHistory read FEditHistory implements IKnobsUndoable;
  public
    property    ShortName                         : string            read FShortName;
    property    AsPrevDisplay                     : string            read GetAsPrevDisplay;
    property    AsDisplay                         : string            read GetAsDisplay;
    property    AsHint                            : string            read GetAsHint;
    property    AsPrevValue                       : TSignal           read GetAsPrevValue;
    property    AsValue                           : TSignal           read GetAsValue;
    property    PrevPosition                      : Integer           read FPrevPosition;
    property    ControlName                       : string            read GetControlName;
    property    WheelSupport                      : Boolean           read FWheelSupport             write FWheelSupport;
    property    WheelSensitivity                  : Integer           read FWheelSensitivity         write FWheelSensitivity;
    property    WheelCounter                      : Integer           read FWheelCounter             write FWheelCounter;
    property    AutomationPosition                : Integer                                          write SetAutomationPosition;
    property    PlayPosition                      : Integer                                          write SetPlayPosition;
    property    ScaledPosition[ anIndex: Integer] : TSignal                                          write SetScaledPosition;
    property    RangeCount                        : Integer           read GetRangeCount;
    property    ActiveRange                       : Integer           read GetActiveRange            write SetActiveRange;
    property    RangeValue                        : TSignal                                          write SetRangeValue;
    property    VariationCount                    : Integer           read GetVariationCount;
    property    AllowVariations                   : Boolean           read GetAllowVariations;
    property    ActiveVariation                   : Integer           read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string            read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal           read GetVariationValue         write SetVariationValue;
    property    CurrentValue                      : TSignal           read FCurrentValue             write SetCurrentValue;
  published
    property    MIDIColor                         : TColor            read FMIDIColor                write SetMIDIColor              default CL_MIDI;
    property    FocusColor                        : TColor            read FFocusColor               write SetFocusColor             default CL_FOCUS;
    property    MIDIFocusColor                    : TColor            read FMIDIFocusColor           write SetMIDIFocusColor         default CL_MIDI_FOCUS;
    property    Action;
    property    FriendlyName                      : string            read FFriendlyName             write SetFriendlyName;
    property    _DesignerRemarks                  : string            read F_DesignerRemarks         write F_DesignerRemarks;
    property    Dezippered                        : Boolean           read FDezippered               write FDezippered               default False;
    property    AllowAutomation                   : Boolean           read GetAllowAutomation        write SetAllowAutomation        default True;
    property    AllowRandomization                : Boolean           read FAllowRandomization       write SetAllowRandomization     default False;
    property    ShowAllowRandomization            : Boolean           read GetShowAllowRandomization write SetShowAllowRandomization default False;
    property    AllowRecording                    : Boolean           read GetAllowRecording         write SetAllowRecording         default True;
    property    ShowAllowRecording                : Boolean           read GetShowAllowRecording     write SetShowAllowRecording     default False;
    property    IsIndicator                       : Boolean           read FIsIndicator              write SetIsIndicator            default False;
    property    AssignedMIDICC                    : Byte              read GetAssignedMIDICC         write SetAssignedMIDICC         default 0;
    property    DynamicStepCount                  : Boolean           read FDynamicStepCount         write SetDynamicStepCount       default False;
    // NOTE: declaration order is important here to avoid loss of knob resolution on a load action ...
    property    Display                           : TKnobsDisplay     read FDisplay                  write SetDisplay;
    property    StepCount                         : Integer           read GetStepCount              write SetStepCount              default 1;
    property    KnobPosition                      : Integer           read FKnobPosition             write SetKnobPosition           default 0;
    property    DefaultPosition                   : Integer           read FDefaultPosition          write SetDefaultPosition        default 0;
    property    ControlType                       : string            read GetControlType            write SetControlType;
    property    Locked                            : Boolean           read FLocked                   write SetLocked;
    // END NOTE.
    property    StyleElements                                                                                                        default [];
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    OnValueChanged         : TKnobsOnValueChange       read FOnValueChanged        write SetOnValueChanged;
    property    OnKnobPositionChanged  : TKnobsKnobPositionChanged read FOnKnobPositionChanged write FOnKnobPositionChanged;
    property    OnUnFocus              : TOnUnFocus                read FOnUnFocus             write FOnUnFocus;
    property    OnClick;
    property    ShowHint                                                                                                             default False;
    property    ShowSimpleHint         : Boolean              read FShowSimpleHint             write SetShowSimpleHint               default False;
    property    ParentShowHint                                                                                                       default False;
  private
}

    function    TKnobsValuedControl.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then result := TKnobsCustomModule( Owner)
      else result := nil;
    end;


//  private

    function    TKnobsValuedControl.GetModuleName: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


    function    TKnobsValuedControl.GetModuleTitle: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Title
      else Result := '';
    end;


    function    TKnobsValuedControl.GetLowRangeAsStr: string;
    begin
      if   Assigned( FRanges)
      then Result := FRanges.LowRangeAsStr
      else Result := ''
    end;


    procedure   TKnobsValuedControl.SetLowRangeAsStr( const aValue: string);
    begin
      if   Assigned( FRanges)
      then FRanges.LowRangeAsStr := aValue;
    end;


    function    TKnobsValuedControl.GetHighRangeAsStr: string;
    begin
      if   Assigned( FRanges)
      then Result := FRanges.HighRangeAsStr
      else Result := ''
    end;


    procedure   TKnobsValuedControl.SetHighRangeAsStr( const aValue: string);
    begin
      if   Assigned( FRanges)
      then FRanges.HighRangeAsStr := aValue;
    end;


    procedure   TKnobsValuedControl.FinalizeKnobPosition;
    begin
      KnobPosition    := KnobPosition;      // why ???? - is this for 'one step' controls or for selectors?
      DefaultPosition := DefaultPosition;   // why ???? - it will issue ValueChanged .. but thats don here anyway

      if   FControlSrc = kcsAutomation
      then CurrentValue                     := KnobValueToVariationValue( KnobPosition)
      else VariationValue[ ActiveVariation] := KnobValueToVariationValue( KnobPosition);

      Invalidate;
      Hint := AsHint;
      ValueChanged( True);
    end;


//  private

    procedure   TKnobsValuedControl.SetDefaultPosition( aValue: Integer); // virtual;
    var
      i : Integer;
    begin
      aValue := Clip( aValue, 0, StepCount - 1);

      if   aValue <> FDefaultPosition
      then begin
        FDefaultPosition := aValue;

        if   csDesigning in ComponentState
        then begin
          for i := 0 to Length( FVariationValues) - 1
          do FVariationValues[ i] := KnobValueToVariationValue( DefaultPosition);
        end;
      end;
    end;


    function    TKnobsValuedControl.GetStepCount: Integer;
    begin
      Result := FStepCount;
    end;


    procedure   TKnobsValuedControl.SetStepCount( aValue: Integer); // virtual;
    begin
      if   aValue <= 0
      then aValue := 1;

      if   aValue <> FStepCount
      then begin
        FStepCount := aValue;
        FinalizeKnobPosition;
      end;
    end;


    function    TKnobsValuedControl.GetAutomationName: string;
    begin
      Result := Name;
    end;


    procedure   TKnobsValuedControl.SetMIDIColor( aValue: TColor);
    begin
      if   aValue <> FMIDIColor
      then begin
        FMIDIColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetFocusColor( aValue: TColor);
    begin
      if   aValue <> FFocusColor
      then begin
        FFocusColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetMIDIFocusColor( aValue: TColor);
    begin
      if   aValue <> FMIDIFocusColor
      then begin
        FMIDIFocusColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetKnobPosition( aValue: Integer); // virtual;
    begin
      if   DynamicStepCount and ( aValue > StepCount - 1)
      then StepCount := aValue + 1;

      aValue := Clip( aValue, 0, StepCount - 1);

      if   aValue <> FKnobPosition
      then begin
        FKnobPosition := aValue;

        if   csDesigning in ComponentState
        then DefaultPosition := aValue;

     // KnobPosition    := KnobPosition;      // Copy of CommonKnobPosition                          // recursive, and why ???
     // DefaultPosition := DefaultPosition;                                                                            why ???

        if   FControlSrc = kcsAutomation
        then CurrentValue                     := KnobValueToVariationValue( KnobPosition)            // recursive, but okay, as in must be done
        else VariationValue[ ActiveVariation] := KnobValueToVariationValue( KnobPosition);           // recursive, but okay, as in must be done

        if   Assigned( FOnKnobPositionChanged)
        then FOnKnobPositionChanged( self, Name, ControlType, KnobPosition);

        ValueChanged( False);
        Hint := AsHint;
        ActivatePopup;
        Invalidate;
      end
      else if StepCount = 1 // Change to same value .. pulse .. for single valued controls only
      then begin
        ValueChanged( True);

        if   Assigned( FOnKnobPositionChanged)
        then FOnKnobPositionChanged( self, Name, ControlType, KnobPosition);
      end;
    end;


    procedure   TKnobsValuedControl.SetAutomationPosition( aValue: Integer); // virtual;
    var
      OldCtrlSrc : TKnobsControlSrc;
    begin
      OldCtrlSrc  := FControlSrc;
      FControlSrc := kcsAutomation;

      try
        SetKnobPosition( aValue);
      finally
        FControlSrc := OldCtrlSrc;
      end;
    end;


    procedure   TKnobsValuedControl.SetPlayPosition( aValue: Integer);
    var
      OldCtrlSrc : TKnobsControlSrc;
    begin
      OldCtrlSrc  := FControlSrc;
      FControlSrc := kcsAutomation;

      try
        SetKnobPosition( aValue);
      finally
        FControlSrc := OldCtrlSrc;
      end;
    end;


    procedure   TKnobsValuedControl.SetScaledPosition( anIndex: Integer; aValue: TSignal);
    var
      OldCtrlSrc : TKnobsControlSrc;
    begin
      OldCtrlSrc  := FControlSrc;
      FControlSrc := kcsAutomation;

      try
        if   FRanges.HasRangeFor[ anIndex]
        then begin
          SetKnobPosition(
            Round(
              VariationValueToKnobValue(
                ScaleToRange(
                  aValue ,
                  anIndex
                )
              )
            )
          );
        end;
      finally
        FControlSrc := OldCtrlSrc;
      end;
    end;


    function    TKnobsValuedControl.GetControlType: string;
    begin
      Result := FControlType;
    end;


    procedure   TKnobsValuedControl.SetFriendlyName( const aValue: string);
    begin
      if   aValue <> FriendlyName
      then begin
        FFriendlyName := aValue;
        Hint          := AsHint;
      end;
    end;


    procedure   TKnobsValuedControl.SetControlType( const aValue: string); // virtual;
    var
      OldCtrlSrc : TKnobsControlSrc;
    begin
      if   aValue <> FControlType
      then begin
        OldCtrlSrc  := FControlSrc;
        FControlSrc := kcsAutomation;

        try
          FControlType := aValue;
          FinalizeKnobPosition;
        finally
          FControlSrc := OldCtrlSrc;
        end;
      end;
    end;


    function    TKnobsValuedControl.GetDynamicHint: string; // virtual;
    begin
      Result := Converters.PosToHint( ControlType, KnobPosition, StepCount);
    end;


    function    TKnobsValuedControl.GetAsHint: string;
    var
      aParts : TStringList;
      S      : string;
    begin
      if   ShowSimpleHint
      then Result := FriendlyName
      else begin
        if   csDesigning in ComponentState
        then Result := Converters.PosToDisplay( ControlType, KnobPosition, StepCount)
        else begin
          aParts := Explode( Name, '_');
          try
            if   FriendlyName = ''
            then S := aParts[ aParts.Count - 1]
            else if AllowAutomation
            then S := Format( '%s'^M'%s', [ aParts[ aParts.Count - 1], FriendlyName], AppLocale)
            else S := FriendlyName;

            if   AllowAutomation and ( AssignedMIDICC <> 0)
            then S := Format( '%s'^M'CC:%d', [ S, AssignedMIDICC], AppLocale)
            else if AllowAutomation
            then S := Format( 'A %s', [ S], AppLocale);

            if   CanAllowRandomization and CanRandomize
            then S := Format( 'R %s', [ S], AppLocale);

            Result := Format( '%s'^M'%s', [ S, GetDynamicHint], AppLocale);

            if   FRanges.HasRangeFor[ ActiveRange]
            then begin
              S := Format( '%s  ..  %s', [ VariationValueToStr( FRanges.LowValue[ ActiveRange]), VariationValueToStr( FRanges.HighValue[ ActiveRange])], AppLocale);
              Result := Format( '%s'^M'%s', [ Result, S], AppLocale);
            end;

          finally
            aParts.DisposeOf;
          end;
        end;
      end;
    end;


    function    TKnobsValuedControl.GetAsPrevDisplay: string;
    begin
      Result := Converters.PosToDisplay( ControlType, PrevPosition, StepCount);
    end;


    function    TKnobsValuedControl.GetAsPrevValue: TSignal;
    begin
      Result := Converters.PosToValue( ControlType, PrevPosition, StepCount);
    end;


    function    TKnobsValuedControl.GetAsDisplay: string;
    var
      p : Integer;
    begin
      Result := Converters.PosToDisplay( ControlType, KnobPosition, StepCount);
      p      := Pos( '~~', Result);

      if p > 0
      then begin
        if GUseNoteNames
        then Result := Trim( Copy( Result, p + 2, Length( Result)))
        else Result := Trim( Copy( Result,     1, p - 1          ));
      end;
    end;


    function    TKnobsValuedControl.GetAsValue: TSignal;
    begin
      // todo : do not take the value from te display - not until PosToValue is fixed
      // for now just use the knob value @@123

      { if Locked and Assigned( FDisplay)
      then begin
        Result := 0;
        ParseTextValue( ControlType, FDisplay.Text, Result)
      end
      else } Result := Converters.PosToValue( ControlType, KnobPosition, StepCount);
    end;


    function    TKnobsValuedControl.GetControlName: string;
    var
      aParts : TStringList;
    begin
      if   FriendlyName <> ''
      then begin
        Result := FriendlyName;

        if   Assigned( Parent) and ( Parent is TKnobsModule)
        then Result := TKnobsModule( Parent).Name + ': ' + Result;
      end
      else begin
        Result := Name;
        aParts := Explode( Result, '_');

        try
          if   ( aParts.Count = 2) and Assigned( Parent) and ( Parent is TKnobsModule)
          then Result := TKnobsModule( Parent).Name + ': ' + aParts[ 1];
        finally
          aParts.DisposeOf;
        end;
      end;
    end;


    procedure   TKnobsValuedControl.SetOnValueChanged( aValue: TKnobsOnValueChange); // virtual;
    begin
      FOnValueChanged := aValue;
      ValueChanged( True);
    end;


    function    TKnobsValuedControl.GetAllowAutomation: Boolean;
    begin
      Result := FAllowAutomation;
    end;


    procedure   TKnobsValuedControl.SetAllowAutomation( aValue: Boolean);
    begin
      if   aValue <> FAllowAutomation
      then begin
        FAllowAutomation := aValue;
        Hint             := AsHint;
        Invalidate;
      end;
    end;


    function    TKnobsValuedControl.GetAllowRandomization: Boolean;
    begin
      Result := FAllowRandomization;
    end;


    procedure   TKnobsValuedControl.SetAllowRandomization( aValue: Boolean);
    begin
      if   aValue <> FAllowRandomization
      then begin
        FAllowRandomization := aValue;
        Hint                := AsHint;
        Invalidate;
      end;
    end;


    function    TKnobsValuedControl.GetShowAllowRandomization: Boolean;
    begin
      Result := FShowAllowRandomization;
    end;


    procedure   TKnobsValuedControl.SetShowAllowRandomization( aValue: Boolean);
    begin
      if   aValue <> FShowAllowRandomization
      then begin
        FShowAllowRandomization := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetIsIndicator( aValue: Boolean); // virtual;
    begin
      if   aValue <> FIsIndicator
      then begin
        if   FIsIndicator
        then begin
          AllowAutomation    := False;
          AllowRandomization := False;
        end;

        FIsIndicator := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsValuedControl.GetAllowRecording: Boolean;
    begin
      Result := FAllowRecording;
    end;


    procedure   TKnobsValuedControl.SetAllowRecording( aValue: Boolean);
    begin
      if   aValue <> FAllowRecording
      then begin
        FAllowRecording := aValue;
        Hint            := AsHint;
        Invalidate;
      end;
    end;


    function    TKnobsValuedControl.GetShowAllowRecording: Boolean;
    begin
      Result := FShowAllowRecording;
    end;


    procedure   TKnobsValuedControl.SetShowAllowRecording( aValue: Boolean);
    begin
      if   aValue <> FShowAllowRecording
      then begin
        FShowAllowRecording := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsValuedControl.GetAssignedMIDICC: Byte;
    begin
      Result := FAssignedMIDICC;
    end;


    procedure   TKnobsValuedControl.SetAssignedMIDICC( aValue: Byte);
    begin
      if   aValue <> FAssignedMIDICC
      then begin
        FAssignedMIDICC := aValue;
        Hint := AsHint;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetDynamicStepCount( aValue: Boolean);
    begin
      if   aValue <> FDynamicStepCount
      then begin
        FDynamicStepCount := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetDisplay( aValue: TKnobsDisplay);
    begin
      FDisplay := aValue;

      if   Assigned( FDisplay)
      then begin
        if   not Locked
        then FDisplay.SetText( AsDisplay);

        FDisplay.FController := Self;
      end;
    end;


    procedure   TKnobsValuedControl.SetShowSimpleHint( aValue: Boolean);
    begin
      if   aValue <> FShowSimpleHint
      then begin
        FShowSimpleHint := aValue;
        Hint            := AsHint;
      end;
    end;


    procedure   TKnobsValuedControl.NotifyPeers; // virtual;
    begin
      ActivatePopup;
    end;


    procedure   TKnobsValuedControl.DoWheel( aSender: TObject; aShift: TShiftState; aWheelDelta: Integer; aMousePos: TPoint; var Handled: Boolean);
    begin
      if   WheelSupport
      and  not ( csDesigning in ComponentState)
      and  not Locked
      then begin
        {$Q-R-}
        FWheelCounter := FWheelCounter + aWheelDelta;
        {$Q+R+}

        if   aWheelDelta > 0
        then HandleWheelUp  (   aWheelDelta, aShift)
        else if aWheelDelta < 0
        then HandleWheelDown( - aWheelDelta, aShift)
        else FWheelCounter := 0;

        Handled := aWheelDelta <> 0;
      end
      else Handled := False;
    end;


    function    TKnobsValuedControl.GetRangeCount: Integer;
    begin
      Result := FRanges.Count;
    end;


    function    TKnobsValuedControl.GetActiveRange: Integer;
    begin
      Result := FRanges.ActiveRange;
    end;


    procedure   TKnobsValuedControl.SetActiveRange( aValue: Integer);
    begin
      if   aValue <> ActiveRange
      then begin
        FRanges.ActiveRange := aValue;
        Hint := AsHint;
        Invalidate;
      end;
    end;


    procedure   TKnobsValuedControl.SetRangeValue( aValue: TSignal);
    begin
      if   FRanges.IsMorph    [ ActiveRange]
      and  FRanges.HasRangeFor[ ActiveRange]
      then begin
        ScaledPosition[ ActiveRange] := aValue;
        Hint := AsHint;
      end;
    end;


    function    TKnobsValuedControl.GetVariationCount: Integer;
    begin
      Result := Length( FVariationValues);
    end;


    function    TKnobsValuedControl.GetAllowVariations: Boolean;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := FindModule;
      Result  := AllowAutomation and AllowRandomization and ( not Assigned( aModule) or aModule.AllowRandomization);
    end;


    function    TKnobsValuedControl.GetActiveVariation: Integer;
    begin
      Result := FActiveVariation;
    end;


    procedure   TKnobsValuedControl.SetActiveVariation( aValue: Integer);
    begin
      aValue := Clip( aValue, 0, VariationCount - 1);

      if   aValue <> FActiveVariation
      then begin
        FActiveVariation := aValue;

        if   VariationCount > 0
        then begin
          if   AllowVariations
          then CurrentValue := VariationValue[ FActiveVariation]
          else CurrentValue := VariationValue[ 0               ];
        end;
      end;
    end;


    function    TKnobsValuedControl.GetVariationsAsString: string;
    var
      i : Integer;
    begin
      Result := Format( '%d', [ VariationCount], AppLocale);

      for i := 0 to VariationCount - 1
      do Result := Result + Format( ',%g', [ VariationValue[ i]], AppLocale);
    end;


    procedure   TKnobsValuedControl.SetVariationsAsString( const aValue: string);
    var
      aParts : TStringList;
      aCount : Integer;
      i      : Integer;
    begin
      aParts := Explode( aValue, ',');

      try
        if   aParts.Count > 0
        then begin
          aCount := StrToIntDef( aParts[ 0], -1);

          if   aCount = VariationCount
          then begin
            for i := 0 to VariationCount - 1
            do begin
              if   AllowVariations
              then begin
                FControlSrc := kcsAutomation;
                VariationValue[ i] := StrToFloatDef( aParts[ i + 1], 0);
              end
              else begin
                FControlSrc := kcsAutomation;
                VariationValue[ i] := StrToFloatDef( aParts[     1], 0);
              end;
            end;
          end;
        end;
      finally
        FControlSrc := kcsUser;
        aParts.DisposeOf;
      end;
    end;


    function    TKnobsValuedControl.GetVariationValue( anIndex: Integer): TSignal;
    begin
      anIndex := Clip( anIndex, 0, VariationCount - 1);

      if   VariationCount > 0
      then begin
        if   AllowVariations
        then Result := FVariationValues[ anIndex]
        else Result := FVariationValues[ 0];
      end
      else Result := 0;

      Result := KnobValueToVariationValue( VariationValueToKnobValue( Result)); // round to knob values.
    end;


    procedure   TKnobsValuedControl.SetVariationValue( anIndex: Integer; aValue: TSignal);
    var
      i             : Integer;
      WasAutomation : Boolean;
    begin
      anIndex := Clip( anIndex, 0, VariationCount - 1);
      aValue  := KnobValueToVariationValue( VariationValueToKnobValue( aValue)); // round to knob values.

      if   VariationCount > 0
      then begin
        if   AllowVariations
        then FVariationValues[ anIndex] := aValue
        else begin
          for i := 0 to VariationCount - 1
          do FVariationValues[ i] := aValue;
        end;
      end;

      if   ( anIndex = ActiveVariation) or not AllowVariations or ( VariationCount = 0)
      then begin
        WasAutomation := FControlSrc = kcsAutomation;
        CurrentValue  := aValue;

        if   WasAutomation
        then FControlSrc := kcsAutomation;
      end;
    end;


    procedure   TKnobsValuedControl.SetCurrentValue( aValue: TSignal);
    begin
      if   aValue <> FCurrentValue
      then begin
        FCurrentValue      := aValue;
        AutomationPosition := Round( VariationValueToKnobValue( aValue));
      end;
    end;


    function    TKnobsValuedControl.KnobValueToVariationValue( aValue: TSignal): TSignal;
    begin
      if   StepCount > 1
      then Result := aValue / ( StepCount - 1)
      else Result := 0;
    end;


    function    TKnobsValuedControl.VariationValueToKnobValue( aValue: TSignal): TSignal;
    begin
      if   StepCount > 0
      then Result := Round( aValue * ( StepCount - 1))
      else Result := 0;
    end;


    function    TKnobsValuedControl.VariationValueToStr( aValue: TSignal): string;
    begin
      Result := Converters.PosToDisplay( ControlType, Round( VariationValueToKnobValue( aValue)), StepCount)
    end;


    procedure   TKnobsValuedControl.SetDefaultVariations;
    var
      i : Integer;
    begin
      for i := 0 to VariationCount - 1
      do VariationValue[ i] := KnobValueToVariationValue( KnobPosition);
    end;


    procedure   TKnobsValuedControl.CountGene( var aCount: Integer);
    begin
      aCount := aCount + 1;
    end;


    procedure   TKnobsValuedControl.FillGene( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    begin
      if   Assigned( aGene)
      then begin
        if   anIndex < aGene.Size
        then begin
          if   ( aVariation >= 0) and ( aVariation < MAX_VARIATIONS)
          then aGene[ anIndex] := FVariationValues[ aVariation]
          else aGene[ anIndex] := CurrentValue;

          anIndex := anIndex + 1;
        end;
      end;
    end;


    procedure   TKnobsValuedControl.AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
    begin
      if   Assigned( aGene)
      then begin
        if   anIndex < aGene.Size
        then begin
          if   not Locked
          and  AllowAutomation
          and  AllowRandomization
          and  DoAllowRandomization
          then begin
            if   ( aVariation >= 0) and ( aVariation < MAX_VARIATIONS)
            then VariationValue[ aVariation] := aGene[ anIndex]
            else CurrentValue := aGene[ anIndex];
          end;

          anIndex := anIndex + 1;
        end;
      end;
    end;


    function    TKnobsValuedControl.ScaleToRange( aValue: TSignal; anIndex: Integer): TSignal;
    begin
      if   RangeIsMorph( anIndex)
      then Result := RangeMap( aValue, 0.0, 1.0, FRanges.LowValue[ anIndex], FRanges.HighValue[ anIndex])
      else Result := aValue;
    end;


    function    TKnobsValuedControl.ScaleToMarkers( aValue: TSignal): TSignal;
    var
      HalfWay : TSignal;
    begin
      if   FRanges.LowValue[ 0] <= FRanges.HighValue[ 0]
      then Result := RangeMap( aValue, 0.0, 1.0, FRanges.LowValue[ 0], FRanges.HighValue[ 0])
      else begin
        HalfWay := ( FRanges.LowValue[ 0] + FRanges.HighValue[ 0]) / 2.0;

        if   aValue > HalfWay
        then Result := RangeMap( aValue, HalfWay, 1.0    , FRanges.LowValue[ 0], 1.0                  )
        else Result := RangeMap( aValue, 0.0    , HalfWay, 0.0                 , FRanges.HighValue[ 0]);
      end;
    end;


    function    TKnobsValuedControl.ClipToMarkers( aValue: TSignal): TSignal;
    begin
      if   FRanges.LowValue[ 0] <= FRanges.HighValue[ 0]
      then Result := Clip  ( aValue, FRanges.LowValue [ 0], FRanges.HighValue[ 0])
      else Result := UnClip( aValue, FRanges.HighValue[ 0], FRanges.LowValue [ 0]);
    end;


//  protected

    procedure   TKnobsValuedControl.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsValuedControl.SetLocked( aValue: Boolean); // virtual;
    begin
      if   aValue <> FLocked
      then begin
        FLocked := aValue;
        Invalidate;
        ValueChanged( True);
      end;
    end;


    procedure   TKnobsValuedControl.FixParam;
    var
      OldCtrlSrc : TKnobsControlSrc;
    begin
      OldCtrlSrc  := FControlSrc;
      FControlSrc := kcsAutomation;

      try
        ValueChanged( True);
      finally
        FControlSrc := OldCtrlSrc;
      end;
    end;


    procedure   TKnobsValuedControl.ValueChanged( IsFinal: Boolean);
    var
      aModule : TKnobsCustomModule;
    begin
      NotifyPeers;

      if   IsFinal and ( FControlSrc = kcsUser)
      then AddToHistory;

      if   Assigned( FOnValueChanged)
      then FOnValueChanged( Self, Name, ControlType, AsValue, IsFinal, FControlSrc = kcsAutomation)
      else begin
        aModule := FindModule;

        if   Assigned( aModule)
        then aModule.ValueChanged( Self, MakePath( [ aModule.Name, Name]), ControlType, AsValue, IsFinal, FControlSrc = kcsAutomation);
      end;
    end;


    function    TKnobsValuedControl.FindModule: TKnobsCustomModule;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsCustomModule)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsCustomModule
      then Result := TKnobsCustomModule( aParent);
    end;


    function    TKnobsValuedControl.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    procedure   TKnobsValuedControl.ActivatePopup;
    var
      P : TPoint;
      R : TRect;
    begin
      if   Assigned( FPopupWindow) // and ( ControlType <> '')
      then begin
        R := FPopupWindow.CalcHintRect( 200, Hint, nil);
        P := ClientToScreen( Point( Width, Height));
        R.Offset( P.X, P.Y);
        FPopupWindow.ActivateHint( R, Hint);
      end;
    end;


    procedure   TKnobsValuedControl.ShowPopup;
    begin
      if   ( FPopupCount = 0) // and ( ControlType <> '')
      then begin
        FPopupWindow := TKnobsSimpleHintWindow.Create( self);
        FPopupWindow.DoubleBuffered := True;
        ActivatePopup;
      end;

      Inc( FPopupCount);
    end;


    procedure   TKnobsValuedControl.HidePopup;
    begin
      if   FPopupCount > 0
      then Dec( FPopupCount);

      if   FPopupCount = 0
      then FreeAndNil( FPopupWindow);
    end;


    procedure   TKnobsValuedControl.ForceHidePopup;
    begin
      FPopupCount := 0;
      FreeAndNil( FPopupWindow);
    end;


    procedure   TKnobsValuedControl.HandleWheelUp( anAmount: Integer; aShift: TShiftState); // virtual;
    begin
      // Nothing here - abstract ...
    end;


    procedure   TKnobsValuedControl.HandleWheelDown( anAmount: Integer; aShift: TShiftState); // virtual;
    begin
      // Nothing here - abstract ...
    end;


    procedure   TKnobsValuedControl.UnFocus; // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnUnFocus)
      then FOnUnFocus( Self)
      else begin
        aModule := FindModule;

        if   Assigned( aModule)
        then aModule.UnFocus( Self);
      end;
    end;


//  public

    constructor TKnobsValuedControl.Create( anOwner: TComponent); // override;
    var
      i : Integer;
    begin
      inherited;
      FEditHistory           := TKnobsEditHistory.Create( Self, CONTROL_HISTORY_COUNT, HISTORY_STR_LEN, False);
      FRanges                := TKnobsRanges.Create( @ GRangeSpecs);
      ActiveRange            := -1;
      FWheelSupport          := False;
      FWheelSensitivity      := 50;
      StepCount              := 1;
      KnobPosition           := 0;
      FPrevPosition          := KnobPosition;
      DefaultPosition        := KnobPosition;
      OnMouseWheel           := DoWheel;
      ShowHint               := False;
      ParentShowHint         := False;
      AllowAutomation        := True;
      AllowRandomization     := False;
      ShowAllowRandomization := False;
      AllowRecording         := False;
      ShowAllowRecording     := False;
      AssignedMIDICC         := 0;
      StyleElements          := [];
      MIDIColor              := CL_MIDI;
      FocusColor             := CL_FOCUS;
      MIDIFocusColor         := CL_MIDI_FOCUS;
      CurrentValue           := KnobValueToVariationValue( KnobPosition);

      for i := 0 to Length( FVariationValues) - 1
      do FVariationValues[ i] := KnobValueToVariationValue( DefaultPosition);
    end;


    destructor  TKnobsValuedControl.Destroy; // override;
    begin
      if   Assigned( FEditHistory)
      then FEditHistory.DisposeOf;

      FEditHistory := nil;
      FreeAndNil( FRanges);
      inherited;
    end;


    procedure   TKnobsValuedControl.FixBitmaps; // virtual;
    begin
      // Nothing here currently
    end;


    procedure   TKnobsValuedControl.SetDefault;
    begin
      Locked        := False;
      FPrevPosition := KnobPosition;
      KnobPosition  := DefaultPosition;
      AddToHistory;
    end;


    procedure   TKnobsValuedControl.FixDisplay;
    var
      OldPrev : Integer;
      Old     : Integer;
    begin
      OldPrev := PrevPosition;
      Old     := KnobPosition;

      if   KnobPosition > 0
      then KnobPosition := KnobPosition - 1
      else KnobPosition := KnobPosition + 1;

      FPrevPosition := OldPrev;
      KnobPosition  := Old;
    end;


    procedure   TKnobsValuedControl.Dump( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, Format( 'ValuedControl "%s" = %f', [ Name, AsValue], AppLocale));
    end;


    procedure   TKnobsValuedControl.BeginStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      FPrevPosition := KnobPosition;
      aWp           := FindWirePanel;

      if   Assigned( aWp)
      then aWp.BeginStateChange( False);
    end;


    procedure   TKnobsValuedControl.EndStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.EndStateChange( False, False);
    end;


    procedure   TKnobsValuedControl.AddToHistory;
    var
      M : TKnobsCustomModule;
      W : TKnobsWirePanel;
      L : TKnobsEditData;
    begin
      if   Assigned( EditHistory)
      then begin
        EditHistory.AddEdit( GetTimeStamp);
        L := EditHistory.Last;

        if   L.FKind <> wekNothing
        then begin
          M := FindModule;

          if   Assigned( M)
          then begin
            M.EditHistory.AddData( L);
            W := FindWirePanel;

            if   Assigned( W)
            then begin
              L.Select( False);
              L.FPath := Format( '%s    %s   %s', [ M.FriendlyName, M.Title, L.FPath], AppLocale);
              W.EditHistory.AddData( L);
              W.EditHistoryChanged;
              W.AcceptAutomationData( Self, @ L);
            end;
          end;
        end;
      end;
    end;


    procedure   TKnobsValuedControl.CopyRangesFrom( const aControl: TKnobsValuedControl);
    begin
      if   Assigned( aControl)
      and  Assigned( aControl.FRanges)
      and  Assigned( FRanges)
      then FRanges.CopyFrom( aControl.FRanges);
    end;


//  public

    procedure   TKnobsValuedControl.FixLiveMorph; // virtual;
    var
      anIndex   : Integer;
      aFraction : TSignal;
    begin
      anIndex      := Trunc( FLiveMorph);
      aFraction    := FLiveMorph - anIndex;
      CurrentValue := VariationValue[ anIndex] + aFraction * ( VariationValue[ anIndex + 1] - VariationValue[ anIndex]);
    end;


    procedure   TKnobsValuedControl.SetRandomValue( anAmount: TSignal; aVariation: Integer); // overload; virtual;
    var
      OldControlSrc : TKnobsControlSrc;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        OldControlSrc := FControlSrc;
        FControlSrc   := kcsAutomation;

        try
       // VariationValue[ aVariation] := VariationValue[ aVariation] + anAmount * ( Random - VariationValue[ aVariation]);
          VariationValue[ aVariation] := ScaleToMarkers( Random);
        finally
          FControlSrc := OldControlSrc;
        end;
      end;
    end;


    procedure   TKnobsValuedControl.SetRandomValue( anAmount: TSignal); // overload; virtual;
    begin
      SetRandomValue( anAmount, ActiveVariation);
    end;


    procedure   TKnobsValuedControl.Randomize( anAmount: TSignal; aVariation: Integer); // overload; virtual;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        VariationValue[ aVariation] := ScaleToMarkers( Random);
        FixLiveMorph;
      end;
    end;


    procedure   TKnobsValuedControl.Randomize( anAmount: TSignal); // overload; virtual;
    begin
      Randomize( anAmount, ActiveVariation);
    end;


    procedure   TKnobsValuedControl.Mutate( aProb, aRange: TSignal; aVariation: Integer); // overload; virtual;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        if   Random < aProb
        then begin
          VariationValue[ aVariation] := ClipToMarkers( VariationValue[ aVariation] + aRange * ( Random - VariationValue[ aVariation]));
          FixLiveMorph;
        end;
      end;
    end;


    procedure   TKnobsValuedControl.Mutate( aProb, aRange: TSignal); // overload; virtual;
    begin
      Mutate( aProb, aRange, ActiveVariation);
    end;


    procedure   TKnobsValuedControl.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); // overload; virtual;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        if Random < anXProb
        then VariationValue[ aVariation] := ClipToMarkers( VariationValue[ aMom])
        else VariationValue[ aVariation] := ClipToMarkers( VariationValue[ aDad]);

        Mutate( aMutProb, aMutRange, aVariation);
        FixLiveMorph;
      end;
    end;


    procedure   TKnobsValuedControl.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer); // overload; virtual;
    begin
      MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsValuedControl.Morph( anAmount: TSignal; aDad, aMom, aVariation: Integer); // overload; virtual;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        VariationValue[ aVariation] := ClipToMarkers( VariationValue[ aDad] + anAmount * ( VariationValue[ aMom] - VariationValue[ aDad]));
        FixLiveMorph;
      end;
    end;


    procedure   TKnobsValuedControl.Morph( anAmount: TSignal; aDad, aMom: Integer); // overload; virtual;
    begin
      Morph( anAmount, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsValuedControl.LiveMorph( anAmount: TSignal); // virtual;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anAmount  := Clip( anAmount, 0.0, 0.9999) * ( MAX_VARIATIONS - 1);

        if   anAmount <> FLiveMorph
        then begin
          FLiveMorph := anAmount;
          FixLiveMorph;
        end;
      end;
    end;


    function    TKnobsValuedControl.CanAllowRandomization: Boolean;
    begin
      Result := AllowAutomation and ( StepCount > 1);
    end;


    function    TKnobsValuedControl.CanRandomize: Boolean;
    begin
      Result := AllowRandomization
        and not Locked
        and Converters.IsRandomizable( ControlType)
        and not IsIndicator;
    end;


    procedure   TKnobsValuedControl.FixActiveRange( aValue: Integer);
    begin
      FRanges.ActiveRange := aValue;
      Hint := AsHint;
      Invalidate;
    end;


    procedure   TKnobsValuedControl.SetValueForRange( aRange: Integer; aValue: TSignal);
    begin
      if   FRanges.IsMorph    [ aRange]
      and  FRanges.HasRangeFor[ aRange]
      then ScaledPosition[ aRange] := aValue;
    end;


{ ========
  TKnobsXYControl = class( TCustomControl)
  // Base type for two valued controls, ones that can have an effect on an external model
  private
    FFriendlyName           : string;
    F_DesignerRemarks       : string;
    FDynamicStepCount       : Boolean;
    FStepCount              : Integer;             // Number of steps in the total knob travel, same for X and Y directions
    FPrevPositionX          : Integer;             // Previous knob X position
    FPrevPositionY          : Integer;             // Previous knob X position;
    FKnobPositionX          : Integer;             // Current  knob X position
    FKnobPositionY          : Integer;             // Current  knob Y position
    FDefaultPositionX       : Integer;             // default  knob X position;
    FDefaultPositionY       : Integer;             // default  knob Y position;
    FControlType            : string;              // The control's behaviour
    FOnValueChanged         : TKnobsOnValueChange; // Called with the new value- twice, once for X, once for Y
    FPopupCount             : Integer;
    FPopupWindow            : THintWindow;
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FActiveVariation        : Integer;
    FVariationValuesX       : TKnobsVariationValues;
    FVariationValuesY       : TKnobsVariationValues;
    FLiveMorph              : TSignal;
    FCurrentValueX          : TSignal;
    FCurrentValueY          : TSignal;
    FControlSrc             : TKnobsControlSrc;
    FOnUnFocus              : TOnUnFocus;
  public
    property    ModuleName                        : string              read GetModuleName;
  public
    property    AsHint                            : string              read GetAsHint;
    property    AsValueX                          : TSignal             read GetAsValueX;
    property    AsValueY                          : TSignal             read GetAsValueY;
    property    PrevPositionX                     : Integer             read FKnobPositionX;
    property    PrevPositionY                     : Integer             read FKnobPositionY;
    property    VariationCount                    : Integer             read GetVariationCount;
    property    AllowVariations                   : Boolean             read GetAllowVariations;
    property    ActiveVariation                   : Integer             read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string              read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal             read GetVariationValue         write SetVariationValue;
    property    CurrentValueX                     : TSignal             read FCurrentValueX            write SetCurrentValueX;
    property    CurrentValueY                     : TSignal             read FCurrentValueY            write SetCurrentValueY;
  published
    property    Action;
    property    AllowRandomization                : Boolean             read GetAllowRandomization     write SetAllowRandomization     default False;
    property    ShowAllowRandomization            : Boolean             read GetShowAllowRandomization write SetShowAllowRandomization default False;
    property    FriendlyName                      : string              read FFriendlyName             write FFriendlyName;
    property    _DesignerRemarks                  : string              read F_DesignerRemarks         write F_DesignerRemarks;
    property    DynamicStepCount                  : Boolean             read FDynamicStepCount         write SetDynamicStepCount       default False;
    // NOTE: declaration order is important here to avoid loss of resolution on a load action ...
    property    StepCount                         : Integer             read FStepCount                write SetStepCount              default 1;
    property    KnobPositionX                     : Integer             read FKnobPositionX            write SetKnobPositionX          default 0;
    property    KnobPositionY                     : Integer             read FKnobPositionY            write SetKnobPositionY          default 0;
    property    DefaultPositionX                  : Integer             read FDefaultPositionX         write SetDefaultPositionX       default 0;
    property    DefaultPositionY                  : Integer             read FDefaultPositionY         write SetDefaultPositionY       default 0;
    property    ControlType                       : string              read FControlType              write SetControlType;
    // END NOTE.
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    OnValueChanged                    : TKnobsOnValueChange read FOnValueChanged           write SetOnValueChanged;
    property    OnUnFocus                         : TOnUnFocus          read FOnUnFocus                write FOnUnFocus;
    property    ShowHint                                                                                                               default False;
    property    ParentShowHint                                                                                                         default False;
    property    OnClick;
  private
}

    function    TKnobsXYControl.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then result := TKnobsCustomModule( Owner)
      else result := nil;
    end;


//  private

    function    TKnobsXYControl.GetModuleName: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


    procedure   TKnobsXYControl.FinalizeKnobPositionX;
    begin
      KnobPositionX    := KnobPositionX;
      DefaultPositionX := DefaultPositionX;

      if   FControlSrc = kcsAutomation
      then CurrentValueX                            := KnobValueToVariationValue( KnobPositionX)
      else VariationValue[ 2 * ActiveVariation + 0] := KnobValueToVariationValue( KnobPositionX);

      Invalidate;
      Hint := AsHint;
      ValueChanged( True);
    end;


    procedure   TKnobsXYControl.FinalizeKnobPositionY;
    begin
      KnobPositionY    := KnobPositionY;
      DefaultPositionY := DefaultPositionY;

      if   FControlSrc = kcsAutomation
      then CurrentValueY                            := KnobValueToVariationValue( KnobPositionY)
      else VariationValue[ 2 * ActiveVariation + 1] := KnobValueToVariationValue( KnobPositionY);

      Invalidate;
      Hint := AsHint;
      ValueChanged( True);
    end;


    function    TKnobsXYControl.GetControlType: string;
    begin
      Result := ControlType;
    end;


    procedure   TKnobsXYControl.SetControlType( const aValue: string);
    begin
      if   aValue <> FControlType
      then begin
        FControlType := aValue;
        FinalizeKnobPositionX;
        FinalizeKnobPositionY;
      end;
    end;


    function    TKnobsXYControl.GetAllowRandomization: Boolean;
    begin
      Result := FAllowRandomization;
    end;


    procedure   TKnobsXYControl.SetAllowRandomization( aValue: Boolean);
    begin
      if   aValue <> FAllowRandomization
      then begin
        FAllowRandomization := aValue;
        Hint := AsHint;
        Invalidate;
      end;
    end;


    function    TKnobsXYControl.GetShowAllowRandomization: Boolean;
    begin
      Result := FShowAllowRandomization;
    end;


    procedure   TKnobsXYControl.SetShowAllowRandomization( aValue: Boolean);
    begin
      if   aValue <> FShowAllowRandomization
      then begin
        FShowAllowRandomization := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsXYControl.GetVariationCount: Integer;
    begin
      Result := Length( FVariationValuesX) + Length( FVariationValuesY);
    end;


    function    TKnobsXYControl.GetAllowVariations: Boolean;
    begin
      Result := AllowRandomization;
    end;


    function    TKnobsXYControl.GetActiveVariation: Integer;
    begin
      Result := FActiveVariation;
    end;


    procedure   TKnobsXYControl.SetActiveVariation( aValue: Integer);
    begin
      aValue := Clip( aValue, 0, VariationCount - 1);

      if   aValue <> FActiveVariation
      then begin
        FActiveVariation := aValue;

        if   VariationCount > 0
        then begin
          if   AllowVariations
          then begin
            CurrentValueX := FVariationValuesX[ FActiveVariation];
            CurrentValueY := FVariationValuesY[ FActiveVariation];
          end
          else begin
            CurrentValueX := FVariationValuesX[ 0];
            CurrentValueY := FVariationValuesY[ 0];
          end;
        end;
      end;
    end;


    function    TKnobsXYControl.GetVariationsAsString: string;
    var
      i : Integer;
    begin
      Result := Format( '%d', [ VariationCount], AppLocale);

      for i := 0 to VariationCount - 1
      do Result := Result + Format( ',%g', [ VariationValue[ i]], AppLocale);
    end;


    procedure   TKnobsXYControl.SetVariationsAsString( const aValue: string);
    var
      aParts : TStringList;
      aCount : Integer;
      i      : Integer;
    begin
      aParts := Explode( aValue, ',');

      try
        if   aParts.Count > 0
        then begin
          aCount := StrToIntDef( aParts[ 0], -1);

          if   ( aCount = VariationCount) and ( aCount = aParts.Count - 1)
          then begin
            for i := 0 to VariationCount - 1
            do begin
              if   AllowVariations
              then begin
                FControlSrc := kcsAutomation;
                VariationValue[ i] := StrToFloatDef( aParts[ i + 1], 0)
              end
              else begin
                FControlSrc := kcsAutomation;

                if   Odd( i)
                then VariationValue[ i] := StrToFloatDef( aParts[ 2], 0)
                else VariationValue[ i] := StrToFloatDef( aParts[ 1], 0);
              end;
            end;
          end;
        end;
      finally
        FControlSrc := kcsUser;
        aParts.DisposeOf;
      end;
    end;


    function    TKnobsXYControl.GetVariationValue( anIndex: Integer): TSignal;
    begin
      anIndex := Clip( anIndex, 0, VariationCount - 1);

      if   VariationCount > 0
      then begin
        if   AllowVariations
        then begin
          if   Odd( anIndex)
          then Result := FVariationValuesY[ anIndex div 2]
          else Result := FVariationValuesX[ anIndex div 2];
        end
        else begin
          if   Odd( anIndex)
          then Result := FVariationValuesY[ 0]
          else Result := FVariationValuesX[ 0];
        end;
      end
      else Result := 0;

      Result := KnobValueToVariationValue( VariationValueToKnobValue( Result)); // round to knob values.
    end;


    procedure   TKnobsXYControl.SetVariationValue( anIndex: Integer; aValue: TSignal);
    var
      i             : Integer;
      WasAutomation : Boolean;
    begin
      anIndex := Clip( anIndex, 0, VariationCount - 1);
      aValue  := KnobValueToVariationValue( VariationValueToKnobValue( aValue)); // round to knob values.

      if   VariationCount > 0
      then begin
        if   AllowVariations
        then begin
          if   Odd( anIndex)
          then FVariationValuesY[ anIndex div 2] := aValue
          else FVariationValuesX[ anIndex div 2] := aValue;
        end
        else begin
          if   Odd( anIndex)
          then begin
            for i := 0 to VariationCount div 2 - 1
            do FVariationValuesY[ i] := aValue;
          end
          else begin
            for i := 0 to VariationCount div 2 - 1
            do FVariationValuesX[ i] := aValue;
          end;
        end;
      end;

      if   (( anIndex div 2) = ActiveVariation) or not AllowVariations or ( VariationCount = 0)
      then begin
        WasAutomation := FControlSrc = kcsAutomation;

        if   Odd( anIndex)
        then CurrentValueY := aValue
        else CurrentValueX := aValue;

        if   WasAutomation
        then FControlSrc := kcsAutomation;
      end;
    end;


    procedure   TKnobsXYControl.SetCurrentValueX( aValue: TSignal);
    begin
      if   aValue <> FCurrentValueX
      then begin
        FCurrentValueX := aValue;
        FControlSrc    := kcsAutomation;

        try
          KnobPositionX := Round( VariationValueToKnobValue( aValue));
        finally
          FControlSrc := kcsUser;
        end;
      end;
    end;


    procedure   TKnobsXYControl.SetCurrentValueY( aValue: TSignal);
    begin
      if   aValue <> FCurrentValueY
      then begin
        FCurrentValueY := aValue;
        FControlSrc    := kcsAutomation;

        try
          KnobPositionY := Round( VariationValueToKnobValue( aValue));
        finally
          FControlSrc := kcsUser;
        end;
      end;
    end;


    function    TKnobsXYControl.KnobValueToVariationValue( aValue: TSignal): TSignal;
    begin
      if   StepCount > 1
      then Result := aValue / ( StepCount - 1)
      else Result := 0;
    end;


    function    TKnobsXYControl.VariationValueToKnobValue( aValue: TSignal): TSignal;
    begin
      if   StepCount > 0
      then Result := Round( aValue * ( StepCount - 1))
      else Result := 0;
    end;


    procedure   TKnobsXYControl.SetDefaultVariations;
    var
      i : Integer;
    begin
      for i := 0 to VariationCount - 1
      do begin
        if   Odd( i)
        then VariationValue[ i] := KnobValueToVariationValue( KnobPositionY)
        else VariationValue[ i] := KnobValueToVariationValue( KnobPositionX);
      end;
    end;


    procedure   TKnobsXYControl.CountGene( var aCount: Integer);
    begin
      aCount := aCount + 2;
    end;


    procedure   TKnobsXYControl.FillGene( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    begin
      if   Assigned( aGene)
      then begin
        if   anIndex < aGene.Size - 1
        then begin
          if   ( aVariation >= 0) and ( aVariation < MAX_VARIATIONS)
          then begin
            aGene[ anIndex + 0] := FVariationValuesX[ aVariation];
            aGene[ anIndex + 1] := FVariationValuesY[ aVariation];
          end
          else begin
            aGene[ anIndex + 0] := CurrentValueX;
            aGene[ anIndex + 1] := CurrentValueY;
          end;

          anIndex := anIndex + 2;
        end;
      end;
    end;


    procedure   TKnobsXYControl.AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
    begin
      if   Assigned( aGene)
      then begin
        if   anIndex < aGene.Size - 1
        then begin
          if   AllowRandomization
          and  DoAllowRandomization
          then begin
            if   ( aVariation >= 0            )
            and  ( aVariation < MAX_VARIATIONS)
            then begin
              VariationValue[ 2 * aVariation + 0] := aGene[ anIndex + 0];
              VariationValue[ 2 * aVariation + 1] := aGene[ anIndex + 1];
            end
            else begin
              CurrentValueX := aGene[ anIndex + 0];
              CurrentValueY := aGene[ anIndex + 1];
            end;
          end;

          anIndex := anIndex + 2;
        end;
      end;
    end;


//  private

    procedure   TKnobsXYControl.SetDefaultPositionX( aValue: Integer); // virtual;
    var
      i : Integer;
    begin
      aValue := Clip( aValue, 0, StepCount - 1);

      if   aValue <> FDefaultPositionX
      then begin
        FDefaultPositionX := aValue;

        if   csDesigning in ComponentState
        then begin
          for i := 0 to Length( FVariationValuesX) - 1
          do FVariationValuesX[ i] := KnobValueToVariationValue( DefaultPositionX);
        end;
      end;
    end;


    procedure   TKnobsXYControl.SetDefaultPositionY( aValue: Integer); // virtual;
    var
      i : Integer;
    begin
      aValue := Clip( aValue, 0, StepCount - 1);

      if   aValue <> FDefaultPositionY
      then begin
        FDefaultPositionY := aValue;

        if   csDesigning in ComponentState
        then begin
          for i := 0 to Length( FVariationValuesY) - 1
          do FVariationValuesY[ i] := KnobValueToVariationValue( DefaultPositionY);
        end;
      end;
    end;


    procedure   TKnobsXYControl.SetKnobPositionX( aValue: Integer); // virtual;
    begin
      if   DynamicStepCount and ( aValue > StepCount - 1)
      then StepCount := aValue + 1;

      aValue := Clip( aValue, 0, StepCount - 1);

      if   aValue <> FKnobPositionX
      then begin
        FKnobPositionX := aValue;

        if   csDesigning in ComponentState
        then DefaultPositionX := FKnobPositionX;

     // KnobPositionX    := KnobPositionX;
     // DefaultPositionX := DefaultPositionX;

        if   FControlSrc = kcsAutomation
        then CurrentValueX                            := KnobValueToVariationValue( KnobPositionX)
        else VariationValue[ 2 * ActiveVariation + 0] := KnobValueToVariationValue( KnobPositionX);

        Invalidate;
        Hint := AsHint;
        ValueChanged( False);
        ActivatePopup;
      end;
    end;


    procedure   TKnobsXYControl.SetKnobPositionY( aValue: Integer); // virtual;
    begin
      if   DynamicStepCount and ( aValue > StepCount - 1)
      then StepCount := aValue + 1;

      aValue := Clip( aValue, 0, StepCount - 1);

      if   aValue <> FKnobPositionY
      then begin
        FKnobPositionY := aValue;

        if   csDesigning in ComponentState
        then DefaultPositionY := FKnobPositionY;

     // KnobPositionY    := KnobPositionY;
     // DefaultPositionY := DefaultPositionY;

        if   FControlSrc = kcsAutomation
        then CurrentValueY                            := KnobValueToVariationValue( KnobPositionY)
        else VariationValue[ 2 * ActiveVariation + 1] := KnobValueToVariationValue( KnobPositionY);

        Invalidate;
        Hint := AsHint;
        ValueChanged( False);
        ActivatePopup;
      end;
    end;


    procedure   TKnobsXYControl.SetStepCount( aValue: Integer); // virtual;
    begin
      if   aValue <= 0
      then aValue := 1;

      if   aValue <> FStepCount
      then begin
        FStepCount   := aValue;
        FinalizeKnobPositionX;
        FinalizeKnobPositionY;
      end;
    end;


    function    TKnobsXYControl.GetAsHint: string;
    var
      aParts : TStringList;
      S      : string;
    begin
      if   csDesigning in ComponentState
      then Result := Format( '%s, %s', [ Converters.PosToDisplay( ControlType, KnobPositionX, StepCount), Converters.PosToDisplay( ControlType, KnobPositionY, StepCount)], AppLocale)
      else begin
        aParts := Explode( Name, '_');

        try
          if   FriendlyName = ''
          then S := aParts[ aParts.Count - 1]
          else S := Format( '%s'^M'%s', [ aParts[ aParts.Count - 1], FriendlyName], AppLocale);

          Result := Format( '%s'^M'%s, %s', [ S, Converters.PosToHint( ControlType, KnobPositionX, StepCount), Converters.PosToHint( ControlType, KnobPositionY, StepCount)], AppLocale);
        finally
          aParts.DisposeOf;
        end;
      end;
    end;


    function    TKnobsXYControl.GetAsValueX: TSignal;
    begin
      Result := Converters.PosToValue( ControlType, KnobPositionX, StepCount);
    end;


    function    TKnobsXYControl.GetAsValueY: TSignal;
    begin
      Result := Converters.PosToValue( ControlType, KnobPositionY, StepCount);
    end;


    procedure   TKnobsXYControl.SetOnValueChanged( aValue: TKnobsOnValueChange);
    begin
      FOnValueChanged := aValue;
      ValueChanged( True);
    end;


    procedure   TKnobsXYControl.SetDynamicStepCount( aValue: Boolean);
    begin
      if   aValue <> FDynamicStepCount
      then begin
        FDynamicStepCount := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsXYControl.NotifyPeers; // virtual;
    begin
      ActivatePopup;
    end;


//  protected

    procedure   TKnobsXYControl.FixParam;
    var
      OldCtrlSrc : TKnobsControlSrc;
    begin
      OldCtrlSrc  := FControlSrc;
      FControlSrc := kcsAutomation;

      try
        ValueChanged( True);
      finally
        FControlSrc := OldCtrlSrc;
      end;
    end;


    procedure   TKnobsXYControl.ValueChanged( IsFinal: Boolean); // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      NotifyPeers;

      if   Assigned( FOnValueChanged)
      then begin
        FOnValueChanged( Self, Name + 'x', ControlType, AsValueX, IsFinal, FControlSrc = kcsAutomation);
        FOnValueChanged( Self, Name + 'y', ControlType, AsValueY, IsFinal, FControlSrc = kcsAutomation);
      end
      else begin
        aModule := FindModule;

        if   Assigned( aModule)
        then begin
          aModule.ValueChanged( Self, MakePath( [ aModule.Name, Name + 'x']), ControlType, AsValueX, IsFinal, FControlSrc = kcsAutomation);
          aModule.ValueChanged( Self, MakePath( [ aModule.Name, Name + 'y']), ControlType, AsValueY, IsFinal, FControlSrc = kcsAutomation);
        end;
      end;
    end;


    function    TKnobsXYControl.FindModule: TKnobsCustomModule;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsCustomModule)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsCustomModule
      then Result := TKnobsCustomModule( aParent);
    end;


    function    TKnobsXYControl.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    procedure   TKnobsXYControl.ActivatePopup;
    var
      P : TPoint;
      R : TRect;
    begin
      if   Assigned( FPopupWindow)
      and  ( ControlType <> '')
      then begin
        R := FPopupWindow.CalcHintRect( 200, Hint, nil);
        P := ClientToScreen( Point( Width, Height));
        R.Offset( P.X, P.Y);
        FPopupWindow.ActivateHint( R, Hint);
      end;
    end;


    procedure   TKnobsXYControl.ShowPopup;
    begin
      if   ( FPopupCount = 0)
      and  ( ControlType <> '')
      then begin
        FPopupWindow := TKnobsSimpleHintWindow.Create( self);
        FPopupWindow.DoubleBuffered := True;
        ActivatePopup;
      end;

      Inc( FPopupCount);
    end;


    procedure   TKnobsXYControl.HidePopup;
    begin
      if   FPopupCount > 0
      then Dec( FPopupCount);

      if   FPopupCount = 0
      then FreeAndNil( FPopupWindow);
    end;


    procedure   TKnobsXYControl.ForceHidePopup;
    begin
      FPopupCount := 0;
      FreeAndNil( FPopupWindow);
    end;


    procedure   TKnobsXYControl.UnFocus; // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnUnFocus)
      then FOnUnFocus( Self)
      else begin
        aModule := FindModule;

        if   Assigned( aModule)
        then aModule.UnFocus( Self);
      end;
    end;


//  public

    constructor TKnobsXYControl.Create( anOwner: TComponent); // override;
    var
      i : Integer;
    begin
      inherited;
      StepCount          := 1;
      KnobPositionX      := 0;
      KnobPositionY      := 0;
      FPrevPositionX     := KnobPositionX;
      FPrevPositionY     := KnobPositionY;
      DefaultPositionX   := KnobPositionX;
      DefaultPositionY   := KnobPositionY;
      AllowRandomization := False;
      ShowHint           := False;
      ParentShowHint     := False;

      for i := 0 to VariationCount div 2 - 1
      do begin
        FVariationValuesX[ i] := KnobValueToVariationValue( DefaultPositionX);
        FVariationValuesY[ i] := KnobValueToVariationValue( DefaultPositionY);
      end;
    end;


    procedure   TKnobsXYControl.SetDefault;
    begin
      FPrevPositionX := KnobPositionX;
      FPrevPositionY := KnobPositionY;
      KnobPositionX  := DefaultPositionX;
      KnobPositionY  := DefaultPositionY;
    end;

    procedure   TKnobsXYControl.Dump( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, Format( 'XYControl "%s" = %f, %f', [ Name, AsValueX, AsValueY], AppLocale));
    end;


    procedure   TKnobsXYControl.BeginStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      FPrevPositionX := KnobPositionX;
      FPrevPositionY := KnobPositionY;
      aWp            := FindWirePanel;

      if   Assigned( aWp)
      then aWp.BeginStateChange( False);
    end;


    procedure   TKnobsXYControl.EndStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.EndStateChange( False, False);
    end;


//  public

    procedure   TKnobsXYControl.FixLiveMorphX;
    var
      anIndex   : Integer;
      aFraction : TSignal;
    begin
      anIndex       := Trunc( FLiveMorph);
      aFraction     := FLiveMorph - anIndex;
      CurrentValueX := VariationValue[ 2 * anIndex + 0] + aFraction * ( VariationValue[ 2 * ( anIndex + 1) + 0] - VariationValue[ 2 * anIndex + 0]);
    end;


    procedure   TKnobsXYControl.FixLiveMorphY;
    var
      anIndex   : Integer;
      aFraction : TSignal;
    begin
      anIndex       := Trunc( FLiveMorph);
      aFraction     := FLiveMorph - anIndex;
      CurrentValueY := VariationValue[ 2 * anIndex + 1] + aFraction * ( VariationValue[ 2 * ( anIndex + 1) + 1] - VariationValue[ 2 * anIndex + 1]);
    end;


    procedure   TKnobsXYControl.SetRandomValue( anAmount: TSignal; aVariation: Integer); // overload;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        VariationValue[ 2 * aVariation + 0] := VariationValue[ 2 * aVariation + 0] + anAmount * ( Random - VariationValue[ 2 * aVariation + 0]);
        VariationValue[ 2 * aVariation + 1] := VariationValue[ 2 * aVariation + 1] + anAmount * ( Random - VariationValue[ 2 * aVariation + 1]);
      end;
    end;


    procedure   TKnobsXYControl.SetRandomValue( anAmount: TSignal); // overload;
    begin
      SetRandomValue( anAmount, ActiveVariation);
    end;


    procedure   TKnobsXYControl.Randomize( anAmount: TSignal; aVariation: Integer); // overload;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        VariationValue[ 2 * aVariation + 0] := VariationValue[ 2 * aVariation + 0] + anAmount * ( Random - VariationValue[ 2 * aVariation + 0]);
        FixLiveMorphX;
        VariationValue[ 2 * aVariation + 1] := VariationValue[ 2 * aVariation + 1] + anAmount * ( Random - VariationValue[ 2 * aVariation + 1]);
        FixLiveMorphY;
      end;
    end;


    procedure   TKnobsXYControl.Randomize( anAmount: TSignal); // overload;
    begin
      Randomize( anAmount, ActiveVariation);
    end;


    procedure   TKnobsXYControl.Mutate( aProb, aRange: TSignal; aVariation: Integer); // overload;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        if   Random < aProb
        then begin
          VariationValue[ 2 * aVariation + 0] := VariationValue[ 2 * aVariation + 0] + aRange * ( Random - VariationValue[ 2 * aVariation + 0]);
          FixLiveMorphX;
        end;

        if   Random < aProb
        then begin
          VariationValue[ 2 * aVariation + 1] :=  VariationValue[ 2 * aVariation + 1] + aRange * ( Random - VariationValue[ 2 * aVariation + 1]);
          FixLiveMorphY;
        end;
      end;
    end;


    procedure   TKnobsXYControl.Mutate( aProb, aRange: TSignal); // overload;
    begin
      Mutate( aProb, aRange, ActiveVariation);
    end;


    procedure   TKnobsXYControl.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); // overload;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        if   Random < anXProb
        then VariationValue[ 2 * aVariation + 0] := VariationValue[ 2 * aMom + 0]
        else VariationValue[ 2 * aVariation + 0] := VariationValue[ 2 * aDad + 0];
        FixLiveMorphX;

        if   Random < anXProb
        then VariationValue[ 2 * aVariation + 1] := VariationValue[ 2 * aMom + 1]
        else VariationValue[ 2 * aVariation + 1] := VariationValue[ 2 * aDad + 1];
        FixLiveMorphY;

        Mutate( aMutProb, aMutRange, aVariation);
      end;
    end;


    procedure   TKnobsXYControl.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer); // overload;
    begin
      MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsXYControl.Morph( anAmount: TSignal; aDad, aMom, aVariation: Integer); // overload;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        VariationValue[ 2 * aVariation + 0] := VariationValue[ 2 * aDad + 0] + anAmount * ( VariationValue[ 2 * aMom + 0] - VariationValue[ 2 * aDad + 0]);
        FixLiveMorphX;
        VariationValue[ 2 * aVariation + 1] := VariationValue[ 2 * aDad + 1] + anAmount * ( VariationValue[ 2 * aMom + 1] - VariationValue[ 2 * aDad + 1]);
        FixLiveMorphY;
      end;
    end;


    procedure   TKnobsXYControl.Morph( anAmount: TSignal; aDad, aMom: Integer); // overload;
    begin
      Morph( anAmount, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsXYControl.LiveMorph( anAmount: TSignal);
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anAmount  := Clip( anAmount, 0.0, 0.9999) * ( MAX_VARIATIONS - 1);

        if   anAmount <> FLiveMorph
        then begin
          FLiveMorph := anAmount;
          FixLiveMorphX;
          FixLiveMorphY;
        end;
      end;
    end;


    function    TKnobsXYControl.CanAllowRandomization: Boolean;
    begin
      Result := StepCount > 1;
    end;


    function    TKnobsXYControl.CanRandomize: Boolean;
    begin
      Result := AllowRandomization and Converters.IsRandomizable( ControlType);
    end;



{ ========
  TKnobsSelector = class( TKnobsValuedControl)
  private
    FCaptions          : TStringList;
    FGlyphs            : TImageList;
    FBorderColor       : TColor;
    FBorderColorSingle : TColor;
    FMouseInControl    : Boolean;
    FOnRightClick      : TOnKnobsRightClick;
  published
    property    Captions          : TStringList        read FCaptions          write SetCaptions;
    property    Glyphs            : TImageList         read FGlyphs            write SetGlyphs;
    property    Color                                                                                     default clGray;
    property    BorderColor       : TColor             read FBorderColor       write SetBorderColor       default clYellow;
    property    BorderColorSingle : TColor             read FBorderColorSingle write SetBorderColorSingle default clWhite;
    property    Font;
    property    OnRightClick      : TOnKnobsRightClick read FOnRightClick      write FOnRightClick;
  private
}

    procedure   TKnobsSelector.SetControlType( const aValue: string); // override;
   // var
   //   aCaptions : TStringList;
    begin
      inherited;

      // todo: the below is not working ... something changes the captions after ... @@captions@@

    {
      if Converters.IsEnumerable( aValue)
      then begin
        aCaptions := Converters.CreateEnumeration( aValue);

        if   Assigned( aCaptions)
        then begin
          try
            Captions := aCaptions;
          finally
            aCaptions.DisposeOf;
          end;
        end;
      end;
    }

    end;


//  private

    procedure   TKnobsSelector.DoCaptionsChanged( aSender: TObject);
    begin
      invalidate;
    end;


    procedure   TKnobsSelector.FixStepCount;
    begin
      if   Assigned( FGlyphs) and Assigned( FCaptions)
      then StepCount := Min( FGlyphs.Count, FCaptions.Count)   // Allows for some glyphs to be left out at the end
      else if Assigned( FGlyphs)
      then StepCount := FGlyphs.Count                          // Otherwise, usually, the glyph count determines the step count
      else if Assigned( FCaptions)
      then StepCount := FCaptions.Count                        // Or alternatively the captions
      else StepCount := 0;                                     // Nothing assigned, no selection possible
    end;


    procedure   TKnobsSelector.SetCaptions( const aValue: TStringList);
    begin
      FCaptions.Assign( aValue);
      FixStepCount;
      Invalidate;
    end;


    procedure   TKnobsSelector.SetGlyphs( const aValue: TImageList);
    begin
      FGlyphs := aValue;
      FixStepCount;
    end;


    procedure   TKnobsSelector.SetBorderColor( aValue: TColor);
    begin
      if   aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsSelector.SetBorderColorSingle( aValue: TColor);
    begin
      if   aValue <> FBorderColorSingle
      then begin
        FBorderColorSingle := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsSelector.SetChoice( const aValue: TKnobsSelectorChoice);
    var
      FOldChoice : TKnobsSelectorChoice;
    begin
      if aValue <> FChoice
      then begin
        FOldChoice := FChoice;
        FChoice    := aValue;

        if Assigned( FOldChoice)
        then begin
          FOldChoice.Clear;
          FOldChoice.Selector := nil;
        end;

        if Assigned( FChoice)
        then begin
          FChoice.Selector := self;
          FChoice.Names := Captions;
        end;
      end;
    end;


    function    TKnobsSelector.AllDashes: Boolean;
    // Must at times check for a situation where all captions are '-' as for dynamic selectors
    // this could be caused by errors in user supplied files. In such a case moving to a different
    // selector item could result in the program hanging in an endless loop.
    var
      i : Integer;
    begin
      Result := Assigned( Captions) and ( Captions.Count > 0);       // An empty list does not have any '-' items.

      if   Result
      then begin
        for i := 0 to Captions.Count - 1
        do begin
          if   Captions[ i] <> '-'
          then begin
            Result := False;
            Break;
          end;
        end;
      end;
    end;


//  private

    procedure   TKnobsSelector.WMSetFocus( var aMessage: TWMSetFocus  ); // message WM_SETFOCUS;
    begin
      Invalidate;
      inherited;
    end;


    procedure   TKnobsSelector.WMKillFocus( var aMessage: TWMKillFocus ); // message WM_KILLFOCUS;
    begin
      Invalidate;
      inherited;
    end;


    procedure   TKnobsSelector.WMGetDlgCode( var aMessage: TWMGetDlgCode); // message WM_GETDLGCODE;
    begin
      aMessage.Result := DLGC_WANTARROWS;
    end;


    procedure   TKnobsSelector.CMMouseEnter( var aMessage: TMessage); // message CM_MOUSEENTER;
    begin
      if   not FMouseInControl
      then begin
        if   TabStop
        and  ( GetFocus <> Handle)
        and  CanFocus
        then SetFocus;

        ShowPopup;
        FMouseInControl := True;
      end;
      inherited;
    end;


    procedure   TKnobsSelector.CMMouseLeave( var aMessage: TMessage); // message CM_MOUSELEAVE;
    begin
      if   FMouseInControl
      then begin
        FMouseInControl := False;
        HidePopup;
        UnFocus;
      end;

      inherited;
    end;


//  protected

    procedure   TKnobsSelector.DoClick( aSender: TObject);
    begin
      BeginStateChange;

      try
        if   not AllDashes
        and  ( Captions.Count > 0)
        and  not IsIndicator
        and  ( StepCount > 1)           // mod operator has a bug .. anything mod 1 should be zero, 1 mod 1 returns 1 tho
        then begin                      // but can't find a a next value anyway when stepcount <= 1, so just exclude it
          repeat
            KnobPosition := ( KnobPosition + 1) mod StepCount
          until Captions[ KnobPosition] <> '-';
        end;

        FinalizeKnobPosition;           // does Invalidate
      finally
        EndStateChange;
      end;
    end;


    procedure   TKnobsSelector.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP: TKnobsWirePanel;
    begin
      if   Button = mbRight
      then begin
        if   Assigned( FOnRightClick)
        then FOnRightClick( Self);

        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.HandleRightClick( Self);
      end
      else inherited;
    end;


    procedure   TKnobsSelector.KeyDown( var Key: Word; Shift: TShiftState); // override;
    begin
      if   not Locked
      then begin
        case Key of

          VK_UP :

            begin
              if   not AllDashes
              then begin
                if   IsIndicator or ( StepCount = 1)
                then FinalizeKnobPosition
                else begin
                  BeginStateChange;

                  try
                    repeat
                      KnobPosition := KnobPosition - 1
                    until ( KnobPosition = 0) or ( Captions[ KnobPosition] <> '-');

                    FinalizeKnobPosition;
                  finally
                    EndStateChange;
                  end;
                end;
              end;

              Key := 0;
            end;

          VK_DOWN :

            begin
              if   not AllDashes
              then begin
                if   IsIndicator or ( StepCount = 1)
                then FinalizeKnobPosition
                else begin
                  BeginStateChange;

                  try
                    if   KnobPosition < StepCount - 1
                    then begin
                      KnobPosition := KnobPosition + 1;

                      if   ( Captions[ KnobPosition] = '-')
                      and  ( KnobPosition < StepCount - 1)
                      then KnobPosition := KnobPosition + 1;

                      FinalizeKnobPosition;
                    end;
                  finally
                    EndStateChange;
                  end;
                end;
              end;

              Key := 0;
            end;

          VK_SPACE :

            begin
              if   not AllDashes
              then begin
                if   IsIndicator or ( StepCount = 1)
                then FinalizeKnobPosition
                else begin
                  BeginStateChange;

                  try
                    repeat
                      if   KnobPosition >= StepCount - 1
                      then KnobPosition := 0
                      else KnobPosition := KnobPosition + 1;
                    until Captions[ KnobPosition] <> '-';

                    FinalizeKnobPosition;
                  finally
                    EndStateChange;
                  end;
                end;
              end;

              Key := 0;
            end;

          else inherited;
        end;
      end;
    end;


    procedure   TKnobsSelector.Paint; // override;
    var
      aTextSize : TSize;
      aLeft     : Integer;
      aTop      : Integer;
    begin
      with Canvas
      do begin
        if   AllowAutomation and ( AssignedMIDICC <> 0) and Focused
        then Pen.Color := MIDIFocusColor
        else if AllowAutomation and ( AssignedMIDICC <> 0)
        then Pen.Color := MIDIColor
        else if StepCount = 1
        then Pen.Color := BorderColorSingle
        else Pen.Color := BorderColor;

        if   ShowAllowRandomization and AllowRandomization
        then Pen.Color := CL_RANDOMIZABLE;

        if   Focused
        then Pen.Color := FocusColor;

        Pen.Width   := 1;
        Brush.Style := bsSolid;
        Brush.Color := Color;
        Font        := Self.Font;
        Rectangle( 0, 0, Width, Height);

        if   Assigned( FGlyphs)
        then FGlyphs.Draw( Canvas, ( Width - FGlyphs.Width) div 2, ( Height - FGlyphs.Height) div 2, KnobPosition, True)
        else begin
          if   Assigned( Captions) and ( KnobPosition >= 0)
          and  ( KnobPosition < Captions.Count)
          then Caption := TabsToSpaces( Captions[ KnobPosition])
          else begin
            if   DynamicStepCount
            then Caption := 'dynamically loaded'
            else Caption := 'invalid';
          end;

          aTextSize := TextExtent( Caption);
          aLeft     := ( Self.Width  - aTextSize.cx) div 2;
          aTop      := ( Self.Height - aTextSize.cy) div 2;
          TextOut( aLeft, aTop, Caption);
        end;

        Brush.Style := bsClear;
        Rectangle( 0, 0, Width, Height);
      end;
    end;


    procedure   TKnobsSelector.LoadCaptions;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := FindModule;

      if   Assigned( aModule)
      then aModule.LoadCaptionsFor( Self, FDependency)
      else if Assigned( GOnLoadCaptions)
      then GOnLoadCaptions( Self, nil, Self, FDependency);

      FixStepCount;
    end;


    function    TKnobsSelector.GetDynamicHint: string; // override;
    begin
      if   DynamicStepCount
      and  Assigned( FCaptions)
      and  ( KnobPosition >= 0)
      and  ( KnobPosition < FCaptions.Count)
      then Result := FCaptions[ KnobPosition]
      else Result := inherited;
    end;


//  public

    constructor TKnobsSelector.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle       := ControlStyle - [ csOpaque] + [ csSetCaption];
      FCaptions          := TStringList.Create;
      FCaptions.OnChange := DoCaptionsChanged;
      KnobPosition       :=  0;
      Height             := 16;
      Width              := 35;
      Color              := clGray;
      Font.Color         := clWhite;
      BorderColor        := clYellow;
      BorderColorSingle  := clWhite;
      TabStop            := True;
      AllowAutomation    := True;
      AllowRandomization := False;
      OnClick            := DoClick;
    end;


    destructor  TKnobsSelector.Destroy; // override;
    begin
      FreeAndNil( FCaptions);   // Free captions only, glyphs are not owned but a reference to a global image list.
      Choice := nil;
      inherited;
    end;


    procedure   TKnobsSelector.FixPeers;
    begin
      FinalizeKnobPosition;
    end;


    procedure   TKnobsSelector.SetKnobPosition( aValue: Integer); // override;
    var
      GoDown : Boolean;
    begin
      if   Assigned( Captions)
      and  (
        ( Captions.Count = 0) or
        Assigned( FDependency)
      )
      and  DynamicStepCount
      then begin
        LoadCaptions;

        if   aValue >= StepCount
        then aValue := 0;
      end;

      if   StepCount > 1
      then inherited;

      if   DynamicStepCount
      and  Assigned( FDependency)
      and  ( StepCount = 1)
      then inherited
      else begin
        if   not AllDashes
        and  Assigned( Captions)
        and  ( Captions.Count > KnobPosition)
        and  ( StepCount > 1)
        then begin
          GoDown := aValue < KnobPosition;

          while Captions[ KnobPosition] = '-'
          do begin
            if   GoDown
            then begin
              if   aValue = 0
              then aValue := Captions.Count - 1
              else Dec( aValue);
            end
            else aValue := ( aValue + 1) mod Captions.Count;

            inherited;
          end;
        end;
      end;
    end;


    procedure   TKnobsSelector.FixLiveMorph; // override;
    var
      anIndex   : Integer;
      aFraction : TSignal;
    begin
      anIndex      := Trunc( FLiveMorph);
      aFraction    := FLiveMorph - anIndex;
      CurrentValue := VariationValue[ anIndex] + aFraction * ( VariationValue[ anIndex + 1] - VariationValue[ anIndex]);
    end;


    procedure   TKnobsSelector.SetRandomValue( anAmount: TSignal; aVariation: Integer); // overload; override;
    var
      aKnobPosition : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      and  Assigned( Captions)
      and  not AllDashes
      then begin
        aKnobPosition := Random( Captions.Count);

        repeat
          if   Captions[ aKnobPosition] = '-'
          then aKnobPosition := ( aKnobPosition + 1) mod Captions.Count;
        until Captions[ aKnobPosition] <> '-';

        SetVariationValue( aVariation, KnobValueToVariationValue( aKnobPosition));
      end;
    end;


    procedure   TKnobsSelector.SetRandomValue( anAmount: TSignal); //  overload; override;
    begin
      SetRandomValue( anAmount, ActiveVariation);
    end;


    procedure   TKnobsSelector.Randomize( anAmount: TSignal; aVariation: Integer); // overload; override;
    var
      aKnobPosition : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      and  Assigned( Captions)
      and  not AllDashes
      then begin
        aKnobPosition := Random( Captions.Count);

        repeat
          if   Captions[ aKnobPosition] = '-'
          then aKnobPosition := ( aKnobPosition + 1) mod Captions.Count;
        until Captions[ aKnobPosition] <> '-';

        SetVariationValue( aVariation, KnobValueToVariationValue( aKnobPosition));
        FixLiveMorph;
      end;
    end;


    procedure   TKnobsSelector.Randomize( anAmount: TSignal); //  overload; override;
    begin
      Randomize( anAmount, ActiveVariation);
    end;


    procedure   TKnobsSelector.Mutate( aProb, aRange: TSignal; aVariation: Integer); // overload; override;
    var
      aKnobPosition : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      and  Assigned( Captions)
      and  not AllDashes
      then begin
        if   Random < aProb
        then begin
          aKnobPosition := Random( Captions.Count);

          repeat
            if   Captions[ aKnobPosition] = '-'
            then aKnobPosition := ( aKnobPosition + 1) mod Captions.Count;
          until Captions[ aKnobPosition] <> '-';

          SetVariationValue( aVariation, KnobValueToVariationValue( aKnobPosition));
          FixLiveMorph;
        end;
      end;
    end;


    procedure   TKnobsSelector.Mutate( aProb, aRange: TSignal); // overload; override;
    begin
      Mutate( aProb, aRange, ActiveVariation);
    end;


    procedure   TKnobsSelector.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); // overload; override;
    begin
      if  CanAllowRandomization
      and  CanRandomize
      and  not AllDashes
      then begin
        if   Random < anXProb
        then VariationValue[ aVariation] := VariationValue[ aMom]
        else VariationValue[ aVariation] := VariationValue[ aDad];

        FixLiveMorph;
        Mutate( aMutProb, aMutRange, aVariation);
      end;
    end;


    procedure   TKnobsSelector.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer); // overload; override;
    begin
      MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsSelector.Morph( anAmount: TSignal; aDad, aMom, aVariation: Integer); // overload; override;
    var
      aKnobPosition : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      and  not AllDashes
      then begin
        aKnobPosition := Round( VariationValueToKnobValue( VariationValue[ aDad] + anAmount * ( VariationValue[ aMom] - VariationValue[ aDad])));

        repeat
          if   Captions[ aKnobPosition] = '-'
          then aKnobPosition := ( aKnobPosition + 1) mod Captions.Count;
        until Captions[ aKnobPosition] <> '-';

        SetVariationValue( aVariation, KnobValueToVariationValue( aKnobPosition));
        FixLiveMorph;
      end;
    end;


    procedure   TKnobsSelector.Morph( anAmount: TSignal; aDad, aMom: Integer); // overload; override;
    begin
      Morph( anAmount, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsSelector.LiveMorph( anAmount: TSignal); // override;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      and  not AllDashes
      then begin
        anAmount  := Clip( anAmount, 0.0, 0.9999) * ( MAX_VARIATIONS - 1);

        if   anAmount <> FLiveMorph
        then begin
          FLiveMorph := anAmount;
          FixLiveMorph;
        end;
      end;
    end;


{ ========
  TKnobsKnob = class( TKnobsValuedControl)
  // A rotary knob, or linear horizontal or vertical control, mouse wheel works too.
  private
    FDownClicker       : TKnobsClicker;
    FUpClicker         : TKnobsClicker;
    FShift             : TShiftState;
    FMouseDown         : Boolean;
    FGraphic           : TBitmap;
    FMouseInControl    : Boolean;
    FTransparent       : Boolean;
    FPairedWith        : TKnobsKnob;
    FPairingMode       : TKnobsPairingMode;
    FPairingLock       : Integer;
    FControlMode       : TDistanceMode;
    FFinalizationTimer : TTimer;
    FTresholdPast      : Boolean;
    FFinalizations     : Integer;
  published
    property    Action;
    property    Transparent  : Boolean           read FTransparent   write SetTransparent default False;
    property    Color;
    property    TabOrder;
    property    TabStop;
    property    ParentColor                                                                               default True;
    property    PairedWith   : TKnobsKnob        read FPairedWith    write SetPairedWith;
    property    PairingMode  : TKnobsPairingMode read FPairingMode   write SetPairingMode;
    property    ControlMode  : TDistanceMode     read FControlMode   write SetControlMode;
    property    ShowHint                                                                                  default False;
    property    ParentShowHint                                                                            default False;
  private
}

    procedure   TKnobsKnob.InitializeFinalizationTimer;
    begin
      FFinalizations     := 0;
      FFinalizationTimer := TTimer.Create( nil);
      StopFinalizationTimer;
      FFinalizationTimer.OnTimer := FinalizationTimerFired;
    end;


    procedure   TKnobsKnob.FinalizeFinalizationTimer;
    begin
      if   Assigned( FFinalizationTimer)
      then begin
        FFinalizationTimer.OnTimer := nil;
        StopFinalizationTimer;
        FreeAndNil( FFinalizationTimer);
      end;
    end;


    procedure   TKnobsKnob.StartFinalizationTimer;
    begin
      if   FFinalizations = 0
      then begin
        BeginStateChange;
        Inc( FFinalizations);
      end;

      StopFinalizationTimer;
      FFinalizationTimer.Interval := 739;
      FFinalizationTimer.Enabled  := True;
    end;


    procedure   TKnobsKnob.StopFinalizationTimer;
    begin
      FFinalizationTimer.Enabled := False;
    end;


    procedure   TKnobsKnob.FinalizationTimerFired( aSender: TObject);
    begin
      StopFinalizationTimer;
      FinalizeKnobPosition;
      FFinalizations := 0;
      EndStateChange;
    end;


//  private

    procedure   TKnobsKnob.WMSetFocus( var aMessage: TWMSetFocus ); // message WM_SETFOCUS;
    begin
      FUpClicker  .ShowFocus := True;
      FDownClicker.ShowFocus := True;
      Invalidate;

      inherited;
    end;


    procedure   TKnobsKnob.WMKillFocus( var aMessage: TWMKillFocus); // message WM_KILLFOCUS;
    begin
      FUpClicker  .ShowFocus := False;
      FDownClicker.ShowFocus := False;
      Invalidate;

      inherited;
    end;


    procedure   TKnobsKnob.WMGetDlgCode( var aMessage: TWMGetDlgCode); // message WM_GETDLGCODE;
    begin
      aMessage.Result := DLGC_WANTARROWS;
    end;


    procedure TKnobsKnob.CMMouseEnter( var aMessage: TMessage);
    begin
      inherited;
      ShowPopup;

      if   not FMouseInControl
      and  Enabled
      and  ( GetCapture = 0)
      then begin
        SetFocus;
        FMouseInControl := True;
        Invalidate;
      end;
    end;


    procedure TKnobsKnob.CMMouseLeave( var aMessage: TMessage);
    begin
      inherited;
      HidePopup;

      if   FMouseInControl
      and  Enabled
      and  not Dragging
      then begin
        FMouseInControl := False;
        UnFocus;
        Invalidate;
      end;
    end;


    procedure   TKnobsKnob.OnDownClick( aSender: TObject);
    begin
      if   not Locked
      then begin
        try
          if   ssCtrl in FShift
          then begin
            FRanges.LowValue[ ActiveRange]  := FRanges.LowValue[ ActiveRange]  - ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else if ssShift in FShift
          then begin
            FRanges.HighValue[ ActiveRange] := FRanges.HighValue[ ActiveRange] - ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else begin
            StartFinalizationTimer;
            KnobPosition := KnobPosition - 1;
          end;
        except
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TKnobsKnob.OnUpClick( aSender: TObject);
    begin
      if   not Locked
      then begin
        try
          if   ssCtrl in FShift
          then begin
            FRanges.LowValue [ ActiveRange] := FRanges.LowValue[ ActiveRange]  + ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else if ssShift in FShift
          then begin
            FRanges.HighValue[ ActiveRange] := FRanges.HighValue[ ActiveRange] + ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else begin
            StartFinalizationTimer;
            KnobPosition := KnobPosition + 1;
          end;
        except
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TKnobsKnob.NotifyPeers; // Override;
    begin
      inherited;

      // todo : once the display edit mechanism got fixed ... uncomment somme stuff below @@123

      if   Assigned( FDisplay) { and not Locked }
      then FDisplay.SetText( AsDisplay);
    end;


    procedure   TKnobsKnob.LockPairing;
    begin
      Inc( FPairingLock);
    end;


    procedure   TKnobsKnob.UnlockPairing;
    begin
      if   PairingLocked
      then Dec( FPairingLock);
    end;


    function    TKnobsKnob.PairingLocked: Boolean;
    begin
      Result := FPairingLock > 0;
    end;


    procedure   TKnobsKnob.FixPairing;
    begin
      if   ( PairingMode <> pmOff)
      and  Assigned( PairedWith)
      and  ( StepCount = PairedWith.StepCount)
      and  not PairingLocked
      then begin
        LockPairing;

        try
          case PairingMode of
            pmNormal   : PairedWith.KnobPosition :=                        KnobPosition    ;
            pmMirrored : PairedWith.KnobPosition := PairedWith.StepCount - KnobPosition - 1;
          end;
        finally
          UnlockPairing;
        end;
      end;
    end;


    procedure   TKnobsKnob.SetTransparent( aValue: Boolean);
    begin
      if   aValue <> FTransparent
      then begin
        FTransparent := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsKnob.SetKnobPosition( aValue: Integer); // override;
    var
      OldVal : Integer;
    begin
      OldVal := KnobPosition;
      inherited;                            // todo: Hmm .. this is recursive (trough FixKnobPosition and some more stuff)

      if   OldVal <> KnobPosition
      then FixPairing;
    end;


    procedure   TKnobsKnob.SetPairedWith( const aValue: TKnobsKnob);
    begin
      if   aValue <> FPairedWith
      then begin
        if   not PairingLocked
        then begin
          LockPairing;

          try
            if   Assigned( aValue)
            then aValue.PairedWith := Self
            else begin
              if   Assigned( FPairedWith)
              then begin
                FPairedWith.PairingMode := pmOff;
                FPairedWith.PairedWith  := nil;
              end;

              PairingMode := pmOff;
            end;

            FPairedWith := aValue;
          finally
            UnlockPairing;
          end;

          FixPairing;
        end;
      end;
    end;


    procedure   TKnobsKnob.SetPairingMode( aValue: TKnobsPairingMode);
    begin
      if   aValue <> FPairingMode
      then begin
        if   not PairingLocked
        then begin
          LockPairing;

          try
            FPairingMode := aValue;

            if   Assigned( FPairedWith)
            then FPairedWith.PairingMode := aValue;
          finally
            UnlockPairing;
          end;

          FixPairing;
        end;
      end;
    end;


    procedure   TKnobsKnob.SetControlMode( aValue: TDistanceMode); // virtual;
    begin
      FControlMode := aValue;
    end;


    function    TKnobsKnob.CenterPoint: TPoint;
    begin
      Result.X := Width         div 2;
      Result.Y := ( Height - 1) div 2;
    end;


    procedure   TKnobsKnob.BeginMove( aPoint: TPoint; aShift: TShiftState); // override;
    var
      P : TPoint;
      A : Extended;
    begin
      FShift := aShift;
      BeginStateChange;               // This will lock UndoRedo as well

      case ControlMode of

        dmCircular :

          begin
            P := PolarToPoint( CenterPoint, ValuesToAngle( 0, KnobPosition, StepCount - 1), Width / 2);

            with ClientToScreen( P)
            do SetCursorPos( X, Y);
          end;

        dmHorizontal :

          begin
            if   StepCount > 0
            then begin
              A := Clip( KnobPosition / StepCount, 0, 1);
              P := CenterPoint + Point( Round( -100 + A * 200), GAnchor.Y);
            end
            else P := CenterPoint;

            with ClientToScreen( P)
            do SetCursorPos( X, Y);
          end;

        dmVertical :

          begin
            if   StepCount > 0
            then begin
              A := Clip( KnobPosition / StepCount, 0, 1);
              P := CenterPoint + Point( GAnchor.X, Round( 100 -  A * 200));
            end
            else P := CenterPoint;

            with ClientToScreen( P)
            do SetCursorPos( X, Y);
          end;

      end;

      GDistance   := CenterPoint - P;
      GStartPoint := GAnchor;
      GAnchor     := CenterPoint;
      Application.ActivateHint( ClientToScreen( CenterPoint));
    end;


    procedure   TKnobsKnob.EndMove( aPoint: TPoint; aShift: TShiftState); // override;
    var
      aVal : TSignal;
    begin
      with ClientToScreen( GStartPoint)
      do SetCursorPos( X, Y);

      Application.HideHint;
      aVal := DistanceToValue( 0, StepCount - 1, aPoint, ControlMode);

      if   ssCtrl in FShift
      then begin
        FRanges.LowValue[ ActiveRange ]  := aVal / ( StepCount - 1);
        Hint := AsHint;
        Invalidate;
      end
      else if ssShift in FShift
      then begin
        FRanges.HighValue[ ActiveRange] := aVal / ( StepCount - 1);
        Hint := AsHint;
        Invalidate;
      end
      else begin
        KnobPosition := Round( aVal);
        FinalizeKnobPosition;
      end;

      EndStateChange;                 // This will unlock UndoRedo as well
    end;


    procedure   TKnobsKnob.UpdateMove( aPoint: TPoint; aShift: TShiftState); // override;
    var
      aVal : TSignal;
    begin
      aVal := DistanceToValue( 0, StepCount - 1, aPoint, ControlMode);

      if   ssCtrl in FShift
      then begin
        FRanges.LowValue [ ActiveRange]  := aVal / ( StepCount - 1);
        Hint := AsHint;
        Invalidate;
      end
      else if ssShift in FShift
      then begin
        FRanges.HighValue[ ActiveRange] := aVal / ( StepCount - 1);
        Hint := AsHint;
        Invalidate;
      end
      else KnobPosition := Round( aVal)
    end;


//  protected

    procedure   TKnobsKnob.SetLocked( aValue: Boolean); // override;
    begin
      inherited;

      if   Assigned( FUpClicker)
      then FUpClicker.Enabled := not aValue;

      if   Assigned( FDownClicker)
      then FDownClicker.Enabled := not aValue;
    end;


    function    TKnobsKnob.CreateClicker( aLeft, aTop: Integer; aNotifyEvent: TNotifyEvent; aBitmap: TBitmap): TKnobsClicker; // virtual
    begin
      Result := TKnobsClicker.Create( Self);

      with Result
      do begin
        Left           := aLeft;
        Top            := aTop;
        Width          := Self.Width div 2 + 1;
        Height         := ClickerSize;
        OnClick        := aNotifyEvent;
        Parent         := Self;
        ParentShowHint := False;
        ShowHint       := False;
        AssignBitmap( aBitmap);
        SetShowFocus( ClassType = TKnobsNoKnob);
      end;
    end;


    procedure   TKnobsKnob.Paint; // override;

      function GetParentColor : TColor;
      var
        LDetails : TThemedElementDetails;
        aColor   : TColor;
      begin
        Result := clBtnFace;

        if   Assigned( Parent)
        then begin
          if   Parent is TKnobsCustomModule
          then Result := TKnobsCustomModule( Parent).Color
          else begin
            if   ( seClient in Parent.StyleElements) and StyleServices.Available
            then begin
              // Set design time color

              if Parent is TPanel
              then Result := TPanel( Parent).Color
              else if Parent is TGroupBox
              then Result := TGroupBox( Parent).Color;

              // Unless theming changed it

              LDetails := StyleServices.GetElementDetails( tpPanelBackground);

              if   StyleServices.GetElementColor( LDetails, ecFillColor, aColor)
              and  ( aColor <> clNone)
              then Result := aColor;
            end
            else begin
              if   Parent is TPanel
              then Result := TPanel( Parent).Color
              else if Parent is TGroupBox
              then Result := TGroupBox( Parent).Color
            end;
          end;
        end;
      end;

      function GetParentOpacity : Byte;
      begin
        Result := 255;

        if   Assigned( Parent)
        and  ( Parent is TKnobsCustomModule)
        then Result := TKnobsCustomModule( Parent).Opacity;
      end;

      function GetParentBitmap : TBitmap;
      begin
        Result := nil;

        if   Assigned( Parent)
        and  ( Parent is TKnobsCustomModule)
        then Result := TKnobsCustomModule( Parent).FBitmap;
      end;

      function GetParentWidth : Integer;
      begin
        Result := Self.Width;

        if   Assigned( Parent)
        and  ( Parent is TControl)
        then Result := TControl( Parent).Width;
      end;

      function GetParentHeight : Integer;
      begin
        Result := Self.Height;

        if   Assigned( Parent)
        and  ( Parent is TControl)
        then Result := TControl( Parent).Height;
      end;

    const
      LockSpotSize = 3;
    var
      aReplica       : TBitmap;
      aParentBmp     : TBitmap;
      aLineStart     : TPoint;
      aLineEnd1      : TPoint;
      aLineEnd2      : TPoint;
      anAngle1       : TSignal;
      anAngle2       : TSignal;
      aParentColor   : TColor;
      aParentOpacity : Byte;
      aParentWidth   : Integer;
      aParentHeight  : Integer;
    begin
      aParentColor   := GetParentColor;
      aParentOpacity := GetParentOpacity;
      aParentBmp     := GetParentBitmap;
      aParentWidth   := GetParentWidth;
      aParentHeight  := GetParentHeight;
      aReplica       := nil;

      try
        // When the Parent has an overlay make a local replica.
        if   Assigned( aParentBmp)
        then begin
          aReplica := TBitmap.Create;
          aReplica.Width  := aParentWidth;
          aReplica.Height := aParentHeight;
          aReplica.Canvas.StretchDraw( Rect( 0, 0, aReplica.Width, aReplica.Height), aParentBmp);
        end;

        with Canvas
        do begin
          // Paint background in Parent's color

          Pen.Color   := aParentColor;
          Pen.Width   := 1;
          Brush.Style := bsSolid;
          Brush.Color := aParentColor;
          Rectangle( 0, 0, width, height);

          // When the Parent has an overlay, paint it

          if   Assigned( aReplica)
          then Draw( -Left, -Top, aReplica, aParentOpacity);

          // When the knob has a graphic for itself, paint it

          if   Assigned( FGraphic)
          then begin
            FGraphic.AlphaFormat := afDefined;
            FGraphic.Transparent := False;
            Draw( 1, 1, FGraphic);
          end;

          // Paint automation limits where needed

          if   AllowAutomation
          and  FRanges.HasActiveRange
          then begin
            if   FRanges.HasRangeFor[ ActiveRange]
            then begin

              // Paint an arc indicating range limits

              anAngle1   := ValuesToAngle( 0, FRanges.LowValue[ ActiveRange], 1);
              aLineEnd1  := PolarToPoint( CenterPoint, anAngle1, Width div 2 - 3);
              anAngle2   := ValuesToAngle( 0, FRanges.HighValue[ ActiveRange], 1);
              aLineEnd2  := PolarToPoint( CenterPoint, anAngle2, Width div 2 - 3);
              Pen.Width  := 3;

              if   anAngle1 < anAngle2
              then begin
                Pen.Color := clRed;
                Arc( 3, 3, Width - 3, Height - 3, aLineEnd1.X, aLineEnd1.Y, aLineEnd2.X, aLineEnd2.Y);
              end
              else begin
                Pen.Color := clBlue;
                Arc( 3, 3, Width - 3, Height - 3, aLineEnd2.X, aLineEnd2.Y, aLineEnd1.X, aLineEnd1.Y);
              end;
            end
            else begin

              // Paint an arc indicating range limits on some other range

              anAngle1   := ValuesToAngle( 0.0, 1.0, 1.0);
              aLineEnd1  := PolarToPoint( CenterPoint, anAngle1, Width div 2 - 3);
              anAngle2   := ValuesToAngle( 0.0, 0.0, 1.0);
              aLineEnd2  := PolarToPoint( CenterPoint, anAngle2, Width div 2 - 3);
              Pen.Width  := 3;

              Pen.Color := clGray;
              Arc( 3, 3, Width - 3, Height - 3, aLineEnd1.X, aLineEnd1.Y, aLineEnd2.X, aLineEnd2.Y);
            end;
          end;

          // Paint the position indicator

          if   Focused
          and  not Locked
          then begin
            Pen.Color := FocusColor;
            Pen.Width := 3;
          end
          else begin
            if   AllowAutomation
            and  ( AssignedMIDICC <> 0)
            then begin
              Pen.Color := MIDIColor;
              Pen.Width := 2;
            end
            else begin
              Pen.Color := clBlack;
              Pen.Width := 1;
            end;
          end;

          anAngle1   := ValuesToAngle( 0, KnobPosition, StepCount - 1);
          aLineStart := PolarToPoint( CenterPoint, anAngle1, 1);
          aLineEnd1  := PolarToPoint( CenterPoint, anAngle1, Width div 2 - 3);
          MoveTo( aLineStart.x, aLineStart.y);
          LineTo( aLineEnd1.x , aLineEnd1.y );

          // Paint he locked indicator or the selected and/or the MIDI indicator

          if   Locked
          then begin
            Pen.Color   := CL_LOCKED;
            Pen.Width   := 1;
            Brush.Color := CL_LOCKED;
            Ellipse(
              CenterPoint.X - LockSpotSize,
              CenterPoint.Y - LockSpotSize,
              CenterPoint.X + LockSpotSize + 1,
              CenterPoint.Y + LockSpotSize + 1
            );
          end;

          if AllowAutomation and ( AssignedMIDICC <> 0) and Focused
          then begin
            Pen.Color   := MIDIFocusColor;
            Pen.Width   := 2;
            Brush.Style := bsClear;
            Rectangle( 1, 1, width, height);
          end
          else if Focused
          then begin
            Pen.Color   := FocusColor;
            Pen.Width   := 2;
            Brush.Style := bsClear;
            Rectangle( 1, 1, width, height);
          end
          else if AllowRandomization and ShowAllowRandomization
          then begin
            Pen.Color   := CL_RANDOMIZABLE;
            Pen.Width   := 2;
            Brush.Style := bsClear;
            Rectangle( 1, 1, width, height);
          end
          else if AllowAutomation and ( AssignedMIDICC <> 0)
          then begin
            Pen.Color   := MIDIColor;
            Pen.Width   := 2;
            Brush.Style := bsClear;
            Rectangle( 1, 1, width, height);
          end;
        end;
      finally
        FreeAndNil( aReplica);
      end;
    end;


    procedure   TKnobsKnob.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP: TKnobsWirePanel;
    begin
      inherited;

      if   FMouseDown
      then Exit;

      if   ( Button = mbLeft) and not Locked
      then begin
        if   TabStop
        and  ( GetFocus <> Handle)
        and  CanFocus
        then SetFocus;

        GAnchor       := Point( X, Y);
        FTresholdPast := False;
        BeginMove( GAnchor, Shift);
        FMouseDown := True;
      end
      else if Button = mbRight
      then begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.HandleRightClick( Self);
      end;
    end;


    procedure   TKnobsKnob.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   FMouseDown
      then begin
        FMouseDown  := False;

        if   FTresholdPast
        then EndMove( Point( X - GAnchor.x, Y - GAnchor.y), Shift)
        else EndMove( GAnchor, Shift);
      end;
    end;


    procedure   TKnobsKnob.MouseMove( Shift: TShiftState; X, Y: Integer); // override;
    var
      aDistance : TPoint;
    begin
      inherited;

      if   FMouseDown
      then begin
        aDistance := Point( X, Y) - GAnchor;

        if   not FTresholdPast
        then FTresholdPast := Distance( aDistance) > 10;

        if   FTresholdPast
        then UpdateMove( Point( X - GAnchor.x, Y - GAnchor.y), Shift);
      end;
    end;


    procedure   TKnobsKnob.HandleWheelUp( anAmount: Integer; aShift: TShiftState); // override;
    var
      WA : Integer;
    begin
      try
        WA := WheelAmount;

        while WheelCounter > WA
        do begin
          if   ssCtrl in aShift
          then begin
            FRanges.LowValue[ ActiveRange]  := FRanges.LowValue[ ActiveRange]  + ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else if ssShift in aShift
          then begin
            FRanges.HighValue[ ActiveRange] := FRanges.HighValue[ ActiveRange] + ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else begin
            StartFinalizationTimer;
            KnobPosition := KnobPosition + 1;
          end;

          WheelCounter := WheelCounter - WA;
        end;
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TKnobsKnob.HandleWheelDown( anAmount: Integer; aShift: TShiftState); // override;
    var
      WA : Integer;
    begin
      try
        WA := WheelAmount;

        while WheelCounter < -WA
        do begin
          if   ssCtrl in aShift
          then begin
            FRanges.LowValue[ ActiveRange]  := FRanges.LowValue[ ActiveRange]  - ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else if ssShift in aShift
          then begin
            FRanges.HighValue[ ActiveRange] := FRanges.HighValue[ ActiveRange] - ( 1 / ( StepCount - 1));
            Hint := AsHint;
          end
          else begin
            StartFinalizationTimer;
            KnobPosition := KnobPosition - 1;
          end;

          WheelCounter := WheelCounter + WA;
        end;
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TKnobsKnob.KeyDown( var Key: Word; Shift: TShiftState); // override;
    begin
      if   not Locked
      then begin
        case Key of
          VK_UP   : begin FShift := Shift; FUpClicker  .Click; Key := 0; end;
          VK_DOWN : begin FShift := Shift; FDownClicker.Click; Key := 0; end;
          else      inherited;
        end;
      end;
    end;


    procedure   TKnobsKnob.AssignGraphic; // virtual;
    begin
      FGraphic := bmKnob;
    end;


    procedure   TKnobsKnob.Loaded; // override;
    begin
      inherited;

      NotifyPeers;
      Hint := AsHint;
    end;


    function    TKnobsKnob.WheelAmount: Integer; // virtual;
    begin
      Result := Max( Trunc((( 100 - Clip( WheelSensitivity, 0, 99)) * 256) / Max( StepCount, 1)), 1);
    end;


    procedure   TKnobsKnob.SetSize; // virtual;
    begin
      Width  := FGraphic.Width  + 2;
      Height := FGraphic.Height + 3;
    end;


    procedure   TKnobsKnob.CreateClickers; // virtual;
    begin
      FDownClicker := CreateClicker( 0          , Height - ClickerSize, OnDownClick, bmDn);
      FUpClicker   := CreateClicker( Width div 2, Height - ClickerSize, OnUpClick  , bmUp);
    end;


    procedure   TKnobsKnob.InvalidateDisplay;
    begin
      NotifyPeers;
      Hint := AsHint;
    end;


//  public

    constructor TKnobsKnob.Create( anOwner: TComponent); // override;
    begin
      inherited;
      InitializeFinalizationTimer;
      ControlStyle := ControlStyle - [ csSetCaption, csOpaque];
      AssignGraphic;
      SetSize;
      CreateClickers;
      DoubleBuffered     := True;
      TabStop            := True;
      Color              := clSilver;
      Transparent        := False;
      ParentColor        := True;
      ShowHint           := False;
      ParentShowHint     := False;
      AllowAutomation    := True;
      AllowRandomization := False;
    end;


    destructor  TKnobsKnob.Destroy; // override;
    begin
      FinalizeFinalizationTimer;

      if   Assigned( FDisplay)
      then FDisplay.FController := nil;

      inherited;
    end;


    procedure   TKnobsKnob.FixBitmaps; // override;
    begin
      AssignGraphic;
      FUpClicker  .AssignBitmap( bmUp);
      FDownClicker.AssignBitmap( bmDn);
    end;


{ ========
  TKnobsSmallKnob = class( TKnobsKnob)
  // just a bit smaller than TKnob
  protected
}

    procedure   TKnobsSmallKnob.AssignGraphic; // Overide;
    begin
      FGraphic := bmSmallKnob;
    end;


{ ========
  TKnobsNoKnob = class( TKnobsKnob)
  // No knob at all, just the clickers
  protected
}

    procedure   TKnobsNoKnob.AssignGraphic; // Overide;
    begin
      FGraphic := bmNoKnob;
    end;


//  public

    constructor TKnobsNoKnob.Create( anOwner: TComponent); // override;
    begin
      inherited;

      FDownClicker.Top  := -1;        // this sucks ...
      FUpClicker  .Top  := -1;
      FUpClicker  .Left := FUpClicker.Left - 1;
    end;


{ ========
  TKnobsSlider = class( TKnobsKnob)
  // A vertical slider type of control
  private
}

    procedure   TKnobsSlider.SetControlMode( aValue: TDistanceMode); // override;
    begin
      FControlMode := dmVertical;
    end;


    procedure   TKnobsSlider.DoResize( aSender: TObject);
    begin
      FDownClicker.Top := Height - ClickerSize;
    end;


//  protected

    procedure   TKnobsSlider.AssignGraphic; // override;
    begin
      FGraphic := nil;
    end;


    function    TKnobsSlider.CreateClicker( aLeft, aTop: Integer; aNotifyEvent: TNotifyEvent; aBitmap: TBitmap): TKnobsClicker; // override;
    begin
      Result := TKnobsClicker.Create( Self);

      with Result
      do begin
        Left        := aLeft;
        Top         := aTop;
        Width       := ClickerSize;
        Height      := ClickerSize;
        OnClick     := aNotifyEvent;
        Parent      := Self;
        Transparent := True;
        AssignBitmap( aBitmap);
        SetShowFocus( True);
      end;
    end;


    procedure   TKnobsSlider.Paint; // override;
    var
      HPos : Integer;
    begin
      with Canvas
      do begin
        if   StepCount > 1
        then begin
          try
            HPos := Round( RangeMap( KnobPosition, 0, StepCount - 1, Height - ClickerSize, ClickerSize))
          except
            HPos := 15;
          end;
        end
        else HPos := 15;

        if AllowAutomation and ( AssignedMIDICC <> 0) and Focused
        then begin
          Pen.Color := MIDIFocusColor;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          MoveTo( 1        , HPos);
          LineTo( width - 1, HPos);
        end
        else if Focused
        then begin
          Pen.Color := FocusColor;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          MoveTo( 1        , HPos);
          LineTo( width - 1, HPos);
        end
        else if AllowAutomation and ( AssignedMIDICC <> 0)
        then begin
          Pen.Color := MIDIColor;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);

          if Locked
          then begin
            Pen.Color := CL_LOCKED;
            Pen.Width := 12
          end
          else if AllowRandomization and ShowAllowRandomization
          then begin
            Pen.Color := CL_RANDOMIZABLE;
            Pen.Width := 12
          end;

          MoveTo( 1        , HPos);
          LineTo( width - 1, HPos);
        end
        else if locked
        then begin
          Pen.Color := CL_LOCKED;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          Pen.Width := 12;
          MoveTo( 1        , HPos);
          LineTo( width - 1, HPos);
        end
        else if AllowRandomization and ShowAllowRandomization
        then begin
          Pen.Color := CL_RANDOMIZABLE;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          Pen.Width := 12;
          MoveTo( 1        , HPos);
          LineTo( width - 1, HPos);
        end
        else begin
          Pen.Color := clBlack;
          Pen.Width := 1;
          Brush.Style := bsClear;
          Rectangle( 0, 0, width, height);
          Pen.Width := 2;
          MoveTo( 0        , HPos);
          LineTo( width - 1, HPos);
        end;
      end;
    end;


    procedure   TKnobsSlider.SetSize; // override;
    begin
      Width  :=     ClickerSize;
      Height := 2 * ClickerSize + 30;
    end;


    procedure   TKnobsSlider.CreateClickers; // override;
    begin
      FDownClicker := CreateClicker( 0, Height - ClickerSize, OnDownClick, bmDn);
      FUpClicker   := CreateClicker( 0, 0                   , OnUpClick  , bmUp);
      FDownClicker.Width := FDownClicker.Width + 2;
      FUpClicker  .Width := FUpClicker  .Width + 2;
    end;


//  public

    constructor TKnobsSlider.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FControlMode := dmVertical;
      OnResize     := DoResize;
    end;


{ ========
  TKnobsHSlider = class( TKnobsKnob)
  // A vertical slider type of control
  private
}

    procedure   TKnobsHSlider.SetControlMode( aValue: TDistanceMode); // override;
    begin
      FControlMode := dmHorizontal;
    end;


    procedure   TKnobsHSlider.DoResize( aSender: TObject);
    begin
      FUpClicker.Left := Width - ClickerSize;
    end;


//  protected

    procedure   TKnobsHSlider.AssignGraphic; // override;
    begin
      FGraphic := nil;
    end;


    function    TKnobsHSlider.CreateClicker( aLeft, aTop: Integer; aNotifyEvent: TNotifyEvent; aBitmap: TBitmap): TKnobsClicker; // override;
    begin
      Result := TKnobsClicker.Create( Self);

      with Result
      do begin
        Left    := aLeft;
        Top     := aTop;
        Width   := ClickerSize;
        Height  := ClickerSize;
        OnClick := aNotifyEvent;
        Parent  := Self;
        Transparent := True;
        AssignBitmap( aBitmap);
        SetShowFocus( True);
      end;
    end;


    procedure   TKnobsHSlider.Paint; // override;
    var
      VPos : Integer;
    begin
      with Canvas
      do begin
        if   StepCount > 1
        then begin
          try
            VPos := Round( RangeMap( KnobPosition, 0, StepCount - 1, Width - ClickerSize, ClickerSize))
          except
            VPos := 15;
          end;
        end
        else VPos := 15;

        VPos := Width - VPos;

        if AllowAutomation and ( AssignedMIDICC <> 0) and Focused
        then begin
          Pen.Color := MIDIFocusColor;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          MoveTo( VPos, 1         );
          LineTo( VPos, height - 1);
        end
        else if Focused
        then begin
          Pen.Color := FocusColor;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          MoveTo( VPos, 1         );
          LineTo( VPos, height - 1);
        end
        else if AllowAutomation and ( AssignedMIDICC <> 0)
        then begin
          Pen.Color := MIDIColor;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);

          if   Locked
          then begin
            Pen.Color := CL_LOCKED;
            Pen.Width := 12
          end
          else if AllowRandomization and ShowAllowRandomization
          then begin
            Pen.Color := CL_RANDOMIZABLE;
            Pen.Width := 12
          end;

          MoveTo( VPos, 1         );
          LineTo( VPos, height - 1);
        end
        else if locked
        then begin
          Pen.Color := CL_LOCKED;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          Pen.Width := 12;
          MoveTo( VPos, 1         );
          LineTo( VPos, height - 1);
        end
        else if AllowRandomization and ShowAllowRandomization
        then begin
          Pen.Color := CL_RANDOMIZABLE;
          Pen.Width := 2;
          Brush.Style := bsClear;
          Rectangle( 1, 1, width, height);
          Pen.Width := 12;
          MoveTo( VPos, 1         );
          LineTo( VPos, height - 1);
        end
        else begin
          Pen.Color := clBlack;
          Pen.Width := 1;
          Brush.Style := bsClear;
          Rectangle( 0, 0, width, height);
          Pen.Width := 2;
          MoveTo( VPos, 1         );
          LineTo( VPos, height - 1);
        end;
      end;
    end;


    procedure   TKnobsHSlider.SetSize; // override;
    begin
      Width  := 2 * ClickerSize + 30;
      Height :=     ClickerSize;
    end;


    procedure   TKnobsHSlider.CreateClickers; // override;
    begin
      FDownClicker := CreateClicker( 0, 0                  , OnDownClick, bmDn);
      FUpClicker   := CreateClicker( Width - ClickerSize, 0, OnUpClick  , bmUp);
      FDownClicker.Height := FDownClicker.Height + 2;
      FUpClicker  .Height := FUpClicker  .Height + 2;
    end;


//  public

    constructor TKnobsHSlider.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FControlMode := dmHorizontal;
      OnResize     := DoResize;
    end;


{ ========
  TKnobsPad = class( TKnobsXYControl)
  // A pad control horizontal and vertical control, no mouse wheel
  private
    FBackGround        : TBitmap;
    FOrigBackGround    : TBitmap;
    FBorderColor       : TColor;
    FFocusColor        : TColor;
    FDotColor          : TColor;
    FDotSize           : Integer;
    FMouseDown         : Boolean;
    FMouseInControl    : Boolean;
    FFinalizationTimer : TTimer;
    FTresholdPast      : Boolean;
    FFinalizations     : Integer;
    FCursorColor       : TColor;
    FSendColorValues   : Boolean;
  public
    property    RedValue        : TSignal read GetRedValue;
    property    GreenValue      : TSignal read GetGreenValue;
    property    BlueValue       : TSignal read GetBlueValue;
  published
    property    Background      : TBitmap read FBackGround      write SetBackGround;
    property    BorderColor     : TColor  read FBorderColor     write SetBorderColor         default clSilver;
    property    FocusColor      : TColor  read FFocusColor      write SetFocusColor          default CL_FOCUS;
    property    DotColor        : TColor  read FDotColor        write SetDotColor            default CL_DOT;
    property    DotSize         : Integer read FDotSize         write SetDotSize             default 3;
    property    SendColorValues : Boolean read FSendColorValues write FSendColorValues       default false;
    property    Action;
    property    Color;
    property    TabOrder;
    property    TabStop;
    property    ParentColor                                                                  default True;
    property    ParentShowHint                                                               default False;
    property    ShowHint                                                                     default False;
  private
}

    procedure   TKnobsPad.SetBackGround( aValue: TBitmap);
    begin
      if Assigned( FBackGround)
      then begin
        FBackGround.Assign( aValue);
        Invalidate;
      end;
    end;


    procedure   TKnobsPad.SetBorderColor( aValue: TColor);
    begin
      if   aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        invalidate;
      end;
    end;


    procedure   TKnobsPad.SetFocusColor ( aValue: TColor );
    begin
      if   aValue <> FFocusColor
      then begin
        FFocusColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsPad.SetDotColor ( aValue: TColor );
    begin
      if   aValue <> FDotColor
      then begin
        FDotColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsPad.SetDotSize( aValue: Integer);
    begin
      if   aValue <> FDotSize
      then begin
        FDotSize := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsPad.SetCursorColor;
    var
      XPos : Integer;
      YPos : Integer;
    begin
      if Assigned( FBackGround)
      then begin
        XPos := Round( RangeMap( KnobPositionX, 0, StepCount - 1, 0         , Width - 1));
        YPos := Round( RangeMap( KnobPositionY, 0, StepCount - 1, Height - 1, 0        ));
        FCursorColor := FBackGround.Canvas.Pixels[ XPos, YPos];
      end;
    end;


    function    TKnobsPad.GetRedValue: TSignal;
    begin
      Result := RangeMap( GetRValue( FCursorColor), 0.0, 255.0, -1.0, 1.0);
    end;


    function    TKnobsPad.GetGreenValue: TSignal;
    begin
      Result := RangeMap( GetGValue( FCursorColor), 0.0, 255.0, -1.0, 1.0);
    end;


    function    TKnobsPad.GetBlueValue: TSignal;
    begin
      Result := RangeMap( GetBValue( FCursorColor), 0.0, 255.0, -1.0, 1.0);
    end;


//  private

    procedure   TKnobsPad.InitializeFinalizationTimer;
    begin
      FFinalizations     := 0;
      FFinalizationTimer := TTimer.Create( nil);
      StopFinalizationTimer;
      FFinalizationTimer.OnTimer := FinalizationTimerFired;
    end;


    procedure   TKnobsPad.FinalizeFinalizationTimer;
    begin
      FFinalizationTimer.OnTimer := nil;
      StopFinalizationTimer;
      FreeAndNil( FFinalizationTimer);
    end;


    procedure   TKnobsPad.StartFinalizationTimer;
    begin
      if   FFinalizations = 0
      then begin
        BeginStateChange;
        Inc( FFinalizations);
      end;

      StopFinalizationTimer;
      FFinalizationTimer.Interval := 739;
      FFinalizationTimer.Enabled  := True;
    end;


    procedure   TKnobsPad.StopFinalizationTimer;
    begin
      FFinalizationTimer.Enabled := False;
    end;


    procedure   TKnobsPad.FinalizationTimerFired( aSender: TObject);
    begin
      StopFinalizationTimer;
      FinalizeKnobPositionX;
      FinalizeKnobPositionY;
      FFinalizations := 0;
      EndStateChange;
    end;


//  private

    procedure   TKnobsPad.WMSetFocus( var aMessage: TWMSetFocus); // message WM_SETFOCUS;
    begin
      Invalidate;
      inherited;
    end;


    procedure   TKnobsPad.WMKillFocus( var aMessage: TWMKillFocus); // message WM_KILLFOCUS;
    begin
      Invalidate;
      inherited;
    end;


    procedure   TKnobsPad.WMGetDlgCode( var aMessage: TWMGetDlgCode); // message WM_GETDLGCODE;
    begin
      aMessage.Result := DLGC_WANTARROWS;
    end;


    procedure   TKnobsPad.CMMouseEnter( var aMessage: TMessage); // message CM_MOUSEENTER;
    begin
      inherited;
      ShowPopup;

      if   not FMouseInControl
      and  Enabled
      and  ( GetCapture = 0)
      then begin
        FMouseInControl := True;
        Invalidate;
      end;
    end;


    procedure   TKnobsPad.CMMouseLeave( var aMessage: TMessage); // message CM_MOUSELEAVE;
    begin
      inherited;
      HidePopup;

      if   FMouseInControl
      and  Enabled
      and  not Dragging
      then begin
        FMouseInControl := False;
        Invalidate;
      end;
    end;


//  private

    procedure   TKnobsPad.SetKnobPositionX( aValue: Integer); // override;
    begin
      inherited;
      SetCursorColor;
    end;


    procedure   TKnobsPad.SetKnobPositionY( aValue: Integer); // override;
    begin
      inherited;
      SetCursorColor;
    end;


    function    TKnobsPad.CenterPoint: TPoint;
    begin
      Result.X := Width  div 2;
      Result.Y := Height div 2;
    end;


    function    TKnobsPad.ScalePoint( const aValue: TPoint): TPoint;
    begin
      Result.X := ( aValue.X * 2) div 3;
      Result.Y := ( aValue.Y * 2) div 3;
    end;


    procedure   TKnobsPad.BeginMove( aPoint: TPoint; aShift: TShiftState); // virtual;
    var
      P : TPoint;
      X : Extended;
      Y : Extended;
    begin
      BeginStateChange;               // This will lock UndoRedo as well

      if   StepCount > 0
      then begin
        X := Clip( KnobPositionX / StepCount, 0, 1);
        Y := Clip( KnobPositionY / StepCount, 0, 1);
        P := CenterPoint + Point( Round( - 150 + X * 300), Round( 150 - Y * 300));
      end
      else P := CenterPoint;

      with ClientToScreen( P)
      do SetCursorPos( X, Y);

      SetCursorColor;
      GDistance   := CenterPoint - P;
      GStartPoint := GAnchor;
      GAnchor     := CenterPoint;
      Application.ActivateHint( ClientToScreen( CenterPoint));
    end;


    procedure   TKnobsPad.EndMove( aPoint: TPoint; aShift: TShiftState); // virtual;
    var
      aVal : TSignal;
    begin
      with ClientToScreen( GStartPoint)
      do SetCursorPos( X, Y);

      Application.HideHint;
      aVal          := DistanceToValue( 0, StepCount - 1, ScalePoint( aPoint), dmHorizontal);
      KnobPositionX := Round( aVal);
      FinalizeKnobPositionX;
      aVal          := DistanceToValue( 0, StepCount - 1, ScalePoint( aPoint), dmVertical);
      KnobPositionY := Round( aVal);
      FinalizeKnobPositionY;
      SetCursorColor;
      EndStateChange;                 // This will unlock UndoRedo as well
    end;


    procedure   TKnobsPad.UpdateMove( aPoint: TPoint; aShift: TShiftState); // virtual;
    var
      aVal : TSignal;
    begin
      aVal          := DistanceToValue( 0, StepCount - 1, ScalePoint( aPoint), dmHorizontal);
      KnobPositionX := Round( aVal);
      aVal          := DistanceToValue( 0, StepCount - 1, ScalePoint( aPoint), dmVertical);
      KnobPositionY := Round( aVal);
      SetCursorColor;
    end;


//  protected

    procedure   TKnobsPad.Paint; // override;
    var
      HPos   : Integer;
      VPos   : Integer;
      DLight : TColor;
      DDark  : TColor;
    begin
      with Canvas
      do begin
        if   Assigned( FBackGround)
        and  not FBackGround.Empty
        then StretchDraw( Rect( 0, 0, Height, Width), FBackGround)
        else begin
          Pen.Width   := 1;
          Brush.Color := Color;
          Brush.Style := bsClear;
          Pen.Color   := BorderColor;
          Rectangle( 0, 0, Width, Height);
        end;

        HPos := CenterPoint.X;
        VPos := CenterPoint.Y;

        if   StepCount > 1
        then begin
          try
            HPos := Round( RangeMap( KnobPositionX, 0, StepCount - 1, 2        , Height - 4));
            VPos := Round( RangeMap( KnobPositionY, 0, StepCount - 1, Width - 4, 2         ));
          except
            on E: Exception
            do KilledException( E);
          end;
        end;

        Pen  .Width := 1;
        Pen  .Style := psSolid;
        Brush.Style := bsSolid;
        DLight      := InterpolateColors( DotColor, clWhite, 128);
        DDark       := InterpolateColors( DotColor, clBlack,  64);

        if   Focused
        then Pen.Color := FocusColor
        else Pen.Color := DotColor;

        Brush.Color := DotColor;
        Ellipse( Rect( HPos - DotSize, VPos - DotSize, HPos + DotSize + 1, VPos + DotSize + 1));
        Dec( HPos, DotSize div 2);
        Dec( VPos, DotSize div 2);
        Pen  .Color := DLight;
        Brush.Color := DLight;
        Ellipse( Rect( HPos - DotSize div 2, VPos - DotSize div 2, HPos + DotSize div 2 + 1, VPos + DotSize div 2 + 1));
        Inc( HPos, DotSize);
        Inc( VPos, DotSize);
        Pen  .Color := DDark;
        Brush.Color := DDark;
        Ellipse( Rect( HPos - DotSize div 2 - 1, VPos - DotSize div 2 - 1, HPos + DotSize div 2, VPos + DotSize div 2));

        if CanRandomize and ShowAllowRandomization
        then begin
          Pen.Color   := CL_RANDOMIZABLE;
          Pen.Width   := 1;
          Brush.Style := bsClear;
          Rectangle( 0, 0, Width, Height);
        end;
      end;
    end;


    procedure   TKnobsPad.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP: TKnobsWirePanel;
    begin
      inherited;

      if   FMouseDown
      then Exit;

      if   ( Button = mbLeft)
      then begin
        if   TabStop
        and  ( GetFocus <> Handle)
        and  CanFocus
        then SetFocus;

        GAnchor       := Point( X, Y);
        FTresholdPast := False;
        BeginMove( GAnchor, Shift);
        FMouseDown := True;
      end
      else if Button = mbRight
      then begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.HandleRightClick( Self);
      end;
    end;


    procedure   TKnobsPad.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   FMouseDown
      then begin
        FMouseDown  := False;

        if   FTresholdPast
        then EndMove( Point( X - GAnchor.x, Y - GAnchor.y), Shift)
        else EndMove( GAnchor, Shift);
      end;
    end;


    procedure   TKnobsPad.MouseMove( Shift: TShiftState; X, Y: Integer); // override;
    var
      aDistance : TPoint;
    begin
      inherited;

      if   FMouseDown
      then begin
        aDistance := Point( X, Y) - GAnchor;

        if   not FTresholdPast
        then FTresholdPast := Distance( aDistance) > 10;

        if   FTresholdPast
        then UpdateMove( Point( X - GAnchor.x, Y - GAnchor.y), Shift);
      end;

    end;

    procedure   TKnobsPad.KeyDown( var Key: Word; Shift: TShiftState); // override;
    begin
      case Key of
        VK_RIGHT : begin StartFinalizationTimer; KnobPositionX := KnobPositionX + 1; Key := 0; end;
        VK_LEFT  : begin StartFinalizationTimer; KnobPositionX := KnobPositionX - 1; Key := 0; end;
        VK_UP    : begin StartFinalizationTimer; KnobPositionY := KnobPositionY + 1; Key := 0; end;
        VK_DOWN  : begin StartFinalizationTimer; KnobPositionY := KnobPositionY - 1; Key := 0; end;
        else      inherited;
      end;
    end;


    procedure   TKnobsPad.Loaded; // override;
    begin
      inherited;

      NotifyPeers;
      Hint := AsHint;
    end;


    procedure   TKnobsPad.ValueChanged( IsFinal: Boolean); // override;
    var
      aModule : TKnobsCustomModule;
    begin
      inherited;

      if   SendColorValues
      and Assigned( FBackGround)
      then begin
        if   Assigned( FOnValueChanged)
        then begin
          FOnValueChanged( Self, Name + 'r', ControlType, RedValue  , IsFinal, FControlSrc = kcsAutomation);
          FOnValueChanged( Self, Name + 'g', ControlType, GreenValue, IsFinal, FControlSrc = kcsAutomation);
          FOnValueChanged( Self, Name + 'b', ControlType, BlueValue , IsFinal, FControlSrc = kcsAutomation);
        end
        else begin
          aModule := FindModule;

          if   Assigned( aModule)
          then begin
            aModule.ValueChanged( Self, MakePath( [ aModule.Name, Name + 'r']), ControlType, RedValue  , IsFinal, FControlSrc = kcsAutomation);
            aModule.ValueChanged( Self, MakePath( [ aModule.Name, Name + 'g']), ControlType, GreenValue, IsFinal, FControlSrc = kcsAutomation);
            aModule.ValueChanged( Self, MakePath( [ aModule.Name, Name + 'b']), ControlType, BlueValue , IsFinal, FControlSrc = kcsAutomation);
          end;
        end;
      end;
    end;


//  public

    constructor TKnobsPad.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle := ControlStyle + [ csOpaque] - [ csSetCaption];
      FBackGround  := TBitmap.Create;
      InitializeFinalizationTimer;
      DoubleBuffered := True;
      TabStop        := True;
      ShowHint       := False;
      Color          := clSilver;
      BorderColor    := clSilver;
      FocusColor     := CL_FOCUS;
      DotColor       := CL_DOT;
      DotSize        := 3;
      ParentColor    := True;
      Width          := 155;
      Height         := 155;
      ShowHint       := False;
      ParentShowHint := False;
    end;


    destructor  TKnobsPad.Destroy; // override;
    begin
      FinalizeFinalizationTimer;
      FreeAndNil( FBackGround);

      if Assigned( FOrigBackGround)
      then FreeAndNil( FOrigBackGround);

      inherited;
    end;


    procedure   TKnobsPad.LoadBackGroundFile( const aFileName: string);
    begin
      if Assigned( FBackGround)
      then begin
        if not Assigned( FOrigBackGround)
        then begin
          FOrigBackGround := TBitmap.Create;
          FOrigBackGround.Assign( FBackGround);
        end;

        try
          FBackGround.LoadFromFile( aFileName);
          Invalidate;
        except
          ClearBackGroundFile;
        end;
      end;
    end;


    procedure   TKnobsPad.ClearBackGroundFile;
    begin
      if Assigned( FOrigBackGround)
      then begin
        FBackGround.Assign( FOrigBackGround);
        Invalidate;
      end;
    end;


{ ========
  TKnobsValuedButton = class( TKnobsValuedControl)
  private
    FButton     : TSpeedButton;
    FColorFocus : TColor;
    FUserColor  : TColor;
    FUserFlat   : Boolean;
  published
    property    Caption          : string        read GetCaption          write SetCaption                               ;
    property    Glyph            : TBitmap       read GetGlyph            write SetGlyph                                 ;
    property    Hint             : string        read GetHint             write SetHint                                  ;
    property    GroupIndex       : Integer       read GetGroupIndex       write SetGroupIndex       default             1;
    property    AllowAllUp       : Boolean       read GetAllowAllUp       write SetAllowAllUp       default         false;
    property    Height                                                                              default            18;
    property    Width                                                                               default            18;
    property    AlignWithMargins : Boolean       read GetAlignWithMargins write SetAlignWithMargins default         False;
    property    Color                                                                               default      clSilver;
    property    ColorFocus       : TColor        read FColorFocus         write FColorFocus         default     clSkyBlue;
    property    Flat             : Boolean       read GetFlat             write SetFlat             default         False;
    property    Font             : TFont         read GetFont             write SetFont                                  ;
    property    Layout           : TButtonLayout read GetLayout           write SetLayout           default   blGlyphLeft;
    property    Margin           : Integer       read GetMargin           write SetMargin           default            -1;
    property    Margins          : TMargins      read GetMargins          write SetMargins                               ;
    property    NumGlyphs        : TNumGlyphs    read GetNumGlyphs        write SetNumGlyphs        default             1;
    property    ParentFont       : Boolean       read GetParentFont       write SetParentFont       default          True;
    property    ParentShowHint   : Boolean       read GetParentShowHint   write SetParentShowHint   default         False;
    property    ShowHint         : Boolean       read GetShowHint         write SetShowHint         default         False;
    property    Transparent      : Boolean       read GetTransparent      write SetTransparent      default          True;
    property    Visible          : Boolean       read GetVisible          write SetVisible          default          True;
  private
}

    procedure   TKnobsValuedButton.SetKnobPosition( aValue: Integer); // override;
    begin
      if   Assigned( FButton)
      then FButton.Down := SignalToLogic( aValue);

      inherited;
    end;


    function    TKnobsValuedButton.GetCaption: string;
    begin
      if   Assigned( FButton)
      then Result := FButton.Caption
      else Result := '';
    end;


    procedure   TKnobsValuedButton.SetCaption( const aValue: string);
    begin
      if   Assigned( FButton)
      then FButton.Caption := aValue;
    end;


    function    TKnobsValuedButton.GetGlyph: TBitmap;
    begin
      if   Assigned( FButton)
      then Result := FButton.Glyph
      else Result := nil;
    end;


    procedure   TKnobsValuedButton.SetGlyph( const aValue: TBitmap);
    begin
      if   Assigned( FButton)
      then FButton.Glyph.Assign( aValue);
    end;


    function    TKnobsValuedButton.GetHint: string;
    begin
      if   Assigned( FButton)
      then Result := FButton.Hint
      else Result := '';
    end;


    procedure   TKnobsValuedButton.SetHint( const aValue: string);
    begin
      if   Assigned( FButton)
      then FButton.Hint := aValue;
    end;


    function    TKnobsValuedButton.GetGroupIndex: Integer;
    begin
      if   Assigned( FButton)
      then Result := FButton.GroupIndex
      else Result := 0;
    end;


    procedure   TKnobsValuedButton.SetGroupIndex( aValue: Integer);
    begin
      if   Assigned( FButton)
      then FButton.GroupIndex := aValue;
    end;


    function    TKnobsValuedButton.GetAllowAllUp: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.AllowAllUp
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetAllowAllUp( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.AllowAllUp := aValue;
    end;


//  private

    function    TKnobsValuedButton.GetAlignWithMargins: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.AlignWithMargins
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetAlignWithMargins( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.AlignWithMargins := aValue;
    end;


    function    TKnobsValuedButton.GetFlat: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.Flat
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetFlat( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.Flat := aValue;
    end;


    function    TKnobsValuedButton.GetFont: TFont;
    begin
      if   Assigned( FButton)
      then Result := FButton.Font
      else Result := nil;
    end;


    procedure   TKnobsValuedButton.SetFont( const aValue: TFont);
    begin
      if   Assigned( FButton)
      then FButton.Font.Assign( aValue);
    end;


    function    TKnobsValuedButton.GetLayout: TButtonLayout;
    begin
      if   Assigned( FButton)
      then Result := FButton.Layout
      else Result := blGlyphLeft;
    end;


    procedure   TKnobsValuedButton.SetLayout( aValue: TButtonLayout);
    begin
      if   Assigned( FButton)
      then FButton.Layout := aValue;
    end;


    function    TKnobsValuedButton.GetMargin: Integer;
    begin
      if   Assigned( FButton)
      then Result := FButton.Margin
      else Result := -1;
    end;


    procedure   TKnobsValuedButton.SetMargin( aValue: Integer);
    begin
      if   Assigned( FButton)
      then FButton.Margin := aValue;
    end;


    function    TKnobsValuedButton.GetMargins: TMargins;
    begin
      if   Assigned( FButton)
      then Result := FButton.Margins
      else Result := nil;
    end;


    procedure   TKnobsValuedButton.SetMargins( aValue: TMargins);
    begin
      if   Assigned( FButton)
      then FButton.Margins := aValue;
    end;


    function    TKnobsValuedButton.GetNumGlyphs: TNumGlyphs;
    begin
      if   Assigned( FButton)
      then Result := FButton.NumGlyphs
      else Result := 1;
    end;


    procedure   TKnobsValuedButton.SetNumGlyphs( aValue: TNumGlyphs);
    begin
      if   Assigned( FButton)
      then FButton.NumGlyphs := aValue;
    end;


    function    TKnobsValuedButton.GetParentFont: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.ParentFont
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetParentFont( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.ParentFont := aValue;
    end;


    function    TKnobsValuedButton.GetParentShowHint: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.ParentShowHint
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetParentShowHint( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.ParentShowHint := aValue;
    end;


    function    TKnobsValuedButton.GetShowHint: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.ShowHint
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetShowHint( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.ShowHint := aValue;
    end;


    function    TKnobsValuedButton.Gettransparent: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.Transparent
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetTransparent( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.Transparent := aValue;
    end;


    function    TKnobsValuedButton.GetVisible: Boolean;
    begin
      if   Assigned( FButton)
      then Result := FButton.Visible
      else Result := True;
    end;


    procedure   TKnobsValuedButton.SetVisible( aValue: Boolean);
    begin
      if   Assigned( FButton)
      then FButton.Visible := aValue;
    end;


//  private

    procedure   TKnobsValuedButton.OnButtonClick( aSender: TObject);
    begin
      if   Assigned( FButton)
      then begin
        if   StepCount = 1
        then FButton.Down := False
        else KnobPosition := Round( LogicToSignal( FButton.Down));
      end;

      FinalizeKnobPosition;
    end;


//  protected

    procedure   TKnobsValuedButton.WMSetFocus( var aMessage: TWMSetFocus); // message WM_SETFOCUS;
    begin
      inherited;
      FUserColor := Color;
      FUserFlat  := Flat;
      Color      := ColorFocus;
      Flat       := not Flat;
    end;


    procedure   TKnobsValuedButton.WMKillFocus( var aMessage: TWMKillFocus); // message WM_KILLFOCUS;
    begin
      inherited;
      Color := FUserColor;
      Flat  := FUserFlat;
    end;


    procedure   TKnobsValuedButton.WMGetDlgCode( var aMessage: TWMGetDlgCode); // message WM_GETDLGCODE;
    begin
      aMessage.Result := DLGC_WANTARROWS;
    end;


    procedure   TKnobsValuedButton.KeyDown( var Key: Word; Shift: TShiftState); // override;
    begin
      if   not Assigned( FButton)
      then Exit;

      case Key of
        VK_UP  , VK_RIGHT : begin FButton.Down := False; Key := 0; end;
        VK_DOWN, VK_LEFT  : begin FButton.Down := True ; Key := 0; end;
        else                inherited;
      end;
    end;


    procedure   TKnobsValuedButton.SetBiDiMode( aValue: TBiDiMode); // override;
    begin
      if   Assigned( FButton)
      then FButton.BiDiMode := aValue;
    end;


    procedure   TKnobsValuedButton.SetParentBiDiMode( aValue: Boolean); // override;
    begin
      if   Assigned( FButton)
      then FButton.ParentBiDiMode := aValue;
    end;


    function    TKnobsValuedButton.GetEnabled: Boolean; // override;
    begin
      if   Assigned( FButton)
      then Result := FButton.Enabled
      else Result := False;
    end;


    procedure   TKnobsValuedButton.SetEnabled( aValue: Boolean); // override;
    begin
      if   Assigned( FButton)
      then FButton.Enabled := aValue;
    end;


//  protected

    procedure   TKnobsValuedButton.Resize; // override;
    begin
      inherited;

      if   Assigned( FButton)
      then FButton.SetBounds( 0, 0, Width, Height);
    end;


    procedure   TKnobsValuedButton.Loaded; // override;
    begin
      inherited;
      Width := Width - 1; // A resize is needed to make the button visible ... weird
      Width := Width + 1;
    end;


    procedure   TKnobsValuedButton.SetDefaults; // virtual;
    begin
      StepCount       := 1;
      KnobPosition    := 0;
      GroupIndex      := 1;
      AllowAllUp      := false;
      Height          := 18;
      Width           := 18;
      Color           := clSilver;
      Flat            := False;
      Layout          := blGlyphLeft;
      Margin          := -1;
      NumGlyphs       := 1;
      ParentFont      := True;
      ParentShowHint  := False;
      ShowHint        := False;
      Transparent     := True;
      Visible         := True;
    end;


//  public

    constructor TKnobsValuedButton.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle   := ControlStyle - [ csSetCaption] + [ csOpaque];
      ShowHint       := False;
      ParentShowHint := False;
      FButton        := TSpeedButton.Create( Self);
      FButton.Parent := Self;
      FButton.SetBounds( 0, 0, Width, Height);
      FButton.OnClick := OnButtonClick;
      SetDefaults;
      DoubleBuffered := True;
    end;


    destructor  TKnobsValuedButton.Destroy; // override;
    begin
      FButton.OnClick := nil;
      FreeAndNil( FButton);
      inherited;
    end;


{ ========
  TKnobsValuedButtons = class( TKnobsValuedControl)
  private
    FButtons          : array of TSpeedButton;
    FCaptions         : TStrings;
    FImageIndices     : TStrings;
    FGlyphs           : TImageList;
    FHints            : TStrings;
    FGroupIndex       : Integer;
    FAlignWithMargins : Boolean;
    FAllowAllUp       : Boolean;
    FFlat             : Boolean;
    FFont             : TFont;
    FLayout           : TButtonLayout;
    FDisplayMode      : TKnobsButtonsDisplayMode;
    FMargin           : Integer;
    FMargins          : TMargins;
    FNumGlyphs        : TNumGlyphs;
    FParentFont       : Boolean;
    FParentShowHint   : Boolean;
    FShowHint         : Boolean;
    FTransparent      : Boolean;
    FVisible          : Boolean;
    FColorFocus       : TColor;
    FUserColor        : TColor;
    FUserFlat         : Boolean;
  published
    property    StepCount                                                                                    default             2;
    property    Captions         : TStrings                 read GetCaptions       write SetCaptions                              ;
    property    ImageIndices     : TStrings                 read GetImageIndices   write SetImageIndices                          ;
    property    Glyphs           : TImageList               read GetGlyphs         write SetGlyphs                                ;
    property    Hints            : TStrings                 read GetHints          write SetHints                                 ;
    property    GroupIndex       : Integer                  read FGroupIndex       write SetGroupIndex       default             1;
    property    AllowAllUp       : Boolean                  read FAllowAllUp       write SetAllowAllUp       default         false;
    property    Height                                                                                       default            25;
    property    Width                                                                                        default            51;
    property    AlignWithMargins : Boolean                  read FAlignWithMargins write SetAlignWithMargins default         False;
    property    Color                                                                                        default      clSilver;
    property    ColorFocus       : TColor                   read FColorFocus       write FColorFocus         default     clSkyBlue;
    property    Flat             : Boolean                  read FFlat             write SetFlat             default         False;
    property    Font             : TFont                    read FFont             write SetFont                                  ;
    property    Layout           : TButtonLayout            read FLayout           write SetLayout           default   blGlyphLeft;
    property    DisplayMode      : TKnobsButtonsDisplayMode read FDisplayMode      write SetDisplayMode      default bdmHorizontal;
    property    Margin           : Integer                  read FMargin           write SetMargin           default            -1;
    property    Margins          : TMargins                 read FMargins          write SetMargins                               ;
    property    NumGlyphs        : TNumGlyphs               read FNumGlyphs        write SetNumGlyphs        default             1;
    property    ParentFont       : Boolean                  read FParentFont       write SetParentFont       default          True;
    property    ParentShowHint   : Boolean                  read FParentShowHint   write SetParentShowHint   default         False;
    property    ShowHint         : Boolean                  read FShowHint         write SetShowHint         default         False;
    property    Transparent      : Boolean                  read FTransparent      write SetTransparent      default          True;
    property    Visible          : Boolean                  read FVisible          write SetVisible          default          True;
  private
}

    procedure   TKnobsValuedButtons.FixDimensions;
    var
      i : Integer;
      w : Integer;
      h : Integer;
    begin
      case DisplayMode of

        bdmHorizontal :

          if   StepCount > 0
          then begin
            w := Width div StepCount;
            h := Height;

            if   AlignWithMargins
            then begin
              w := w - Margins.Left - Margins.Right;
              h := h - Margins.Top  - Margins.Bottom;
            end;

            for i := 0 to StepCount - 1
            do begin
              FButtons[ i].Width  := w;
              FButtons[ i].Height := h;
            end;
          end;

        bdmVertical :

          if   StepCount > 0
          then begin
            w := Width;
            h := Height div StepCount;

            if   AlignWithMargins
            then begin
              w := w - Margins.Left - Margins.Right;
              h := h - Margins.Top  - Margins.Bottom;
            end;

            for i := 0 to StepCount - 1
            do begin
              FButtons[ i].Width  := w;
              FButtons[ i].Height := h;
            end;
          end;
      end;
    end;


    procedure   TKnobsValuedButtons.FixGlyph( anIndex: Integer; aBitmap: TBitmap);
    begin
      FButtons[ anIndex].Glyph.Assign( aBitmap);
    end;


    function    TKnobsValuedButtons.FindGlyphIndex( anIndex: Integer): Integer;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < FImageIndices.Count)
      then Result := StrToIntDef( FImageIndices[ anIndex], -1)
      else Result := -1;
    end;


//  private

    procedure   TKnobsValuedButtons.FixAlignWithMargins;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].AlignWithMargins := AlignWithMargins;

      FixDimensions;
    end;


    procedure   TKnobsValuedButtons.FixBiDiMode;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].BiDiMode := BiDiMode;
    end;


    procedure   TKnobsValuedButtons.FixCaptions;
    var
      i : Integer;
    begin
      if   Assigned( FCaptions)
      then begin
        for i := 0 to StepCount - 1
        do begin
          if   i < FCaptions.Count
          then FButtons[ i].Caption := FCaptions[ i]
          else FButtons[ i].Caption := '';
        end;
      end;
    end;


    procedure   TKnobsValuedButtons.FixEnabled;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Enabled := Enabled;
    end;


    procedure   TKnobsValuedButtons.FixFlat;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Flat := Flat;
    end;


    procedure   TKnobsValuedButtons.FixFont;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Font.Assign( Font);
    end;


    procedure   TKnobsValuedButtons.FixGlyphs;
    var
      i   : Integer;
      aBm : TBitmap;
      idx : Integer;
    begin
      if   Assigned( FGlyphs)
      then begin
        aBm := TBitmap.Create;

        try
          for i := 0 to StepCount - 1
          do begin
            idx := FindGlyphIndex( i);

            if   ( idx >= 0) and ( idx < FGlyphs.Count)
            then begin
              FGlyphs.GetBitmap( idx, aBm);
              FixGlyph( i, aBm);
            end
            else FixGlyph( i, nil);
          end;
        finally
          aBm.DisposeOf;
        end;
      end;
    end;


    procedure   TKnobsValuedButtons.FixHints;
    var
      i : Integer;
    begin
      if   Assigned( FHints)
      then begin
        for i := 0 to StepCount - 1
        do begin
          if   i < FHints.Count
          then FButtons[ i].Hint := FHints[ i]
          else FButtons[ i].Hint := '';
        end;
      end;
    end;


    procedure   TKnobsValuedButtons.FixLayout;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Layout := Layout;
    end;


    procedure   TKnobsValuedButtons.FixMargin;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Margin := Margin;
    end;


    procedure   TKnobsValuedButtons.FixMargins;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Margins.Assign( FMargins);

      FixDimensions;
    end;


    procedure   TKnobsValuedButtons.FixNumGlyphs;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].NumGlyphs := NumGlyphs;
    end;


    procedure   TKnobsValuedButtons.FixParentFont;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].ParentFont := ParentFont;
    end;


    procedure   TKnobsValuedButtons.FixParentShowHint;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].ParentShowHint := ParentShowHint;
    end;


    procedure   TKnobsValuedButtons.FixParentBiDiMode;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].ParentBiDiMode := ParentBiDiMode;
    end;


    procedure   TKnobsValuedButtons.FixShowHint;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].ShowHint := ShowHint;
    end;


    procedure   TKnobsValuedButtons.FixTransparent;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Transparent := Transparent;
    end;


    procedure   TKnobsValuedButtons.FixVisible;
    var
      i : Integer;
    begin
      for i := 0 to StepCount - 1
      do FButtons[ i].Visible := Visible;
    end;


//  private

    procedure   TKnobsValuedButtons.FixMode( aNewStepCount: Integer);
    var
      aComponent: TComponent;
    begin
      while aNewStepCount < Length( FButtons)
      do begin
        aComponent := FButtons[ Length( FButtons) - 1];
        SetLength( FButtons, Length( FButtons) - 1);
        Owner.RemoveComponent( aComponent);
        aComponent.DisposeOf;
      end;

      FStepCount := aNewStepCount;

      while aNewStepCount > Length( FButtons)
      do CreateButton;

      FixDimensions;
    end;


    procedure   TKnobsValuedButtons.SetKnobPosition( aValue: Integer); // override;
    begin
      if   aValue <> FKnobPosition
      then begin
        inherited;

        if   ( aValue >= 0) and ( aValue < StepCount)
        then FButtons[ aValue].Down := True;

        FinalizeKnobPosition;
      end;
    end;


    procedure   TKnobsValuedButtons.SetStepCount( aValue: Integer);
    begin
      if   aValue <= 0
      then aValue := 1;

      if   aValue <> StepCount
      then begin
        FixMode( aValue);
        FixGlyphs;
        FixCaptions;
        FixHints;
        FixDimensions;
        FinalizeKnobPosition;
      end;
    end;


    function    TKnobsValuedButtons.GetCaptions: TStrings;
    begin
      Result := FCaptions;
    end;


    procedure   TKnobsValuedButtons.SetCaptions( const aValue: TStrings);
    begin
      FCaptions.Assign( aValue);
      FixCaptions;
    end;


    function    TKnobsValuedButtons.GetImageIndices: TStrings;
    begin
      Result := FImageIndices;
    end;


    procedure   TKnobsValuedButtons.SetImageIndices( const aValue: TStrings);
    begin
      FImageIndices.Assign( aValue);
      FixGlyphs;
    end;


    function    TKnobsValuedButtons.GetGlyphs: TImageList;
    begin
      Result := FGlyphs;
    end;


    procedure   TKnobsValuedButtons.SetGlyphs( const aValue: TImageList);
    begin
      FGlyphs := aValue;
      FixGlyphs;
    end;


    function    TKnobsValuedButtons.GetHints: TStrings;
    begin
      Result := FHints;
    end;


    procedure   TKnobsValuedButtons.SetHints( const aValue: TStrings);
    begin
      FHints.Assign( aValue);
      FixHints;
    end;


    procedure   TKnobsValuedButtons.SetGroupIndex( aValue: Integer);
    var
      i : Integer;
    begin
      if   aValue <> FGroupIndex
      then begin
        FGroupIndex := aValue;

        for i := 0 to StepCount - 1
        do FButtons[ i].GroupIndex := FGroupIndex;
      end;
    end;


    procedure   TKnobsValuedButtons.SetAllowAllUp( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FAllowAllUp
      then begin
        FAllowAllUp := aValue;

        for i := 0 to StepCount - 1
        do FButtons[ i].AllowAllUp := FAllowAllUp;
      end;
    end;


    procedure   TKnobsValuedButtons.SetAlignWithMargins( aValue: Boolean);
    begin
      if   aValue <> FAlignWithMargins
      then begin
        FAlignWithMargins := aValue;
        FixAlignWithMargins;
      end;
    end;


    procedure   TKnobsValuedButtons.SetFlat( aValue: Boolean);
    begin
      if   avalue <> FFlat
      then begin
        FFlat := aValue;
        FixFlat;
      end;
    end;


    procedure   TKnobsValuedButtons.SetFont( const aValue: TFont);
    begin
      FFont.Assign( aValue);
      FixFont;
    end;


    procedure   TKnobsValuedButtons.SetLayout( aValue: TButtonLayout);
    begin
      if   aValue <> FLayout
      then begin
        FLayout := aValue;
        FixLayout;
      end;
    end;


    procedure   TKnobsValuedButtons.SetDisplayMode( aValue: TKnobsButtonsDisplayMode);
    begin
      if   aValue <> FDisplayMode
      then begin
        FDisplayMode := aValue;
        FixMode( StepCount);
      end;
    end;


    procedure   TKnobsValuedButtons.SetMargin( aValue: Integer);
    begin
      if   avalue <> FMargin
      then begin
        FMargin := aValue;
        FixMargin;
      end;
    end;


    procedure   TKnobsValuedButtons.SetMargins( aValue: TMargins);
    begin
      FMargins.Assign( aValue);
      FixMargins;
    end;


    procedure   TKnobsValuedButtons.SetNumGlyphs( aValue: TNumGlyphs);
    begin
      if   avalue <> FNumGlyphs
      then begin
        FNumGlyphs := aValue;
        FixNumGlyphs;
      end;
    end;


    procedure   TKnobsValuedButtons.SetParentFont( aValue: Boolean);
    begin
      if   aValue <> FParentFont
      then begin
        FParentFont := aValue;
        FixParentFont;
      end;
    end;


    procedure   TKnobsValuedButtons.SetParentShowHint( aValue: Boolean);
    begin
      if   aValue <> FParentShowHint
      then begin
        FParentShowHint := aValue;
        FixParentShowHint;

        if   ParentShowHint
        then ShowHint := False;
      end;
    end;


    procedure   TKnobsValuedButtons.setShowHint( aValue: Boolean);
    begin
      if   aValue <> FShowHint
      then begin
        FShowHint := aValue;
        FixShowHint;

        if   ShowHint
        then ParentShowHint := False;
      end;
    end;


    procedure   TKnobsValuedButtons.SetTransparent( aValue: Boolean);
    begin
      if   aValue <> FTransparent
      then begin
        FTransparent := aValue;
        FixTransparent;
      end;
    end;


    procedure   TKnobsValuedButtons.SetVisible( aValue: Boolean);
    begin
      if   aValue <> FVisible
      then begin
        FVisible := aValue;
        FixVisible;
      end;
    end;


//  private

    procedure   TKnobsValuedButtons.OnButtonClick( aSender: TObject);
    var
      aTag: Integer;
    begin
      aTag := TSpeedButton( aSender).Tag;
      KnobPosition := aTag;
      FinalizeKnobPosition;
    end;


    procedure   TKnobsValuedButtons.CreateButton;
    var
      aButton : TSpeedButton;
    begin
      aButton := TSpeedButton.Create( Self);

      with aButton do
      begin
        Parent     := Self;
        AllowAllUp := Self.AllowAllUp;
        GroupIndex := Self.GroupIndex;
        Name       := '';
        Tag        := Length( FButtons);
        OnClick    := OnButtonClick;
        Align      := alLeft;
        SetLength( FButtons, Length( FButtons) + 1);
        FButtons[ Length( FButtons) - 1] := aButton;
      end;

      StepCount := Length( FButtons);
    end;


//  protected

    procedure   TKnobsValuedButtons.WMSetFocus( var aMessage: TWMSetFocus); // message WM_SETFOCUS;
    begin
      inherited;
      FUserColor := Color;
      FUserFlat  := Flat;
      Color      := ColorFocus;
      Flat       := not Flat;
    end;


    procedure   TKnobsValuedButtons.WMKillFocus( var aMessage: TWMKillFocus); // message WM_KILLFOCUS;
    begin
      inherited;
      Color := FUserColor;
      Flat  := FUserFlat;
    end;


    procedure   TKnobsValuedButtons.WMGetDlgCode( var aMessage: TWMGetDlgCode); // message WM_GETDLGCODE;
    begin
      aMessage.Result := DLGC_WANTARROWS;
    end;


    procedure   TKnobsValuedButtons.KeyDown( var Key: Word; Shift: TShiftState); // override;
    begin
      case Key of
        VK_UP  , VK_RIGHT : begin if KnobPosition < StepCount - 1 then KnobPosition := KnobPosition + 1; Key := 0; end;
        VK_DOWN, VK_LEFT  : begin if KnobPosition > 0             then KnobPosition := KnobPosition - 1; Key := 0; end;
        else                inherited;
      end;
    end;


    procedure   TKnobsValuedButtons.SetBiDiMode( aValue: TBiDiMode); // Override;
    begin
      if   aValue <> BiDiMode
      then begin
        inherited;
        FixBiDiMode;
      end;
    end;


    procedure   TKnobsValuedButtons.SetEnabled( aValue: Boolean); // Override;
    begin
      if   aValue <> Enabled
      then begin
        inherited;
        FixEnabled;
      end;
    end;


    procedure   TKnobsValuedButtons.SetParentBiDiMode( aValue: Boolean); // Override;
    begin
      if   aValue <> ParentBiDiMode
      then begin
        inherited;
        FixParentBiDiMode;
      end;
    end;


//  protected

    procedure   TKnobsValuedButtons.Resize; // override;
    begin
      inherited;
      FixDimensions;
    end;


    procedure   TKnobsValuedButtons.Loaded; // override;
    begin
      // FixCaptions;
      FixButtons;
    end;


    procedure   TKnobsValuedButtons.FixButtons; // virtual;
    begin
      FixAlignWithMargins;
      FixBiDiMode;
      FixCaptions;
      FixEnabled;
      FixFlat;
      FixFont;
      FixGlyphs;
      FixHints;
      FixLayout;
      FixMargin;
      FixMargins;
      FixNumGlyphs;
      FixParentFont;
      FixParentShowHint;
      FixParentBiDiMode;
      FixShowHint;
      FixTransparent;
      FixVisible;
    end;


    procedure   TKnobsValuedButtons.SetDefaults; // virtual;
    begin
      StepCount       := 2;
      GroupIndex      := 1;
      AllowAllUp      := False;
      Height          := 25;
      Width           := 51;
      Color           := clSilver;
      ColorFocus      := clSkyBlue;
      Flat            := False;
      Layout          := blGlyphLeft;
      Margin          := -1;
      NumGlyphs       := 1;
      ParentFont      := True;
      ParentShowHint  := False;
      ShowHint        := False;
      Transparent     := True;
      Visible         := True;
    end;


//  public

    constructor TKnobsValuedButtons.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle  := ControlStyle + [ csOpaque];
      FCaptions     := TStringList.Create;
      FHints        := TStringList.Create;
      FImageIndices := TStringList.Create;
      FFont         := TFont      .Create;
      FMargins      := TMargins   .Create( Self);
      SetDefaults;
    end;


    destructor  TKnobsValuedButtons.Destroy; // override;
    begin
      FFont        .DisposeOf;
      FImageIndices.DisposeOf;
      FHints       .DisposeOf;
      FCaptions    .DisposeOf;
      inherited;
    end;


{ ========
  TKnobsDisplayEditor = class( TMemo)
  // An multi line editor for TDisplay
  private
    FDisplay : TKnobsDisplay;
  private
}

    procedure   TKnobsDisplayEditor.EndEdit;
    begin
      Visible      := False;
      PopupEditorHide( Self);

      if   Assigned( FDisplay)
      then FDisplay.EndStateChange;
    end;


    procedure   TKnobsDisplayEditor.WMKillFocus( var aMessage: TWMKillFocus ); // message WM_KILLFOCUS;
    begin
      inherited;
      EndEdit;
    end;


    procedure   TKnobsDisplayEditor.Do_Exit( aSender: TObject);
    begin
      EndEdit;

      if   Assigned( FDisplay)
      then FDisplay.SetFocus;
    end;


    procedure   TKnobsDisplayEditor.CNKeyDown( var Message: TWMKeyDown); // message CN_KEYDOWN;
    // Copied from TWinControl.CNKeyDown - but left out the check for IsShortcut
    // as that one was 'stealing away' the menu shortcut keys resulting in the DEL
    // key (VK_DELETE) not working for editing the editor's text. Now this control
    // behaves like there are no menu shortcuts at all, and that is what is wanted
    // here.
    var
      Mask: Integer;
    begin
      with Message
      do begin
        Result := 1;
        UpdateUIState( Message.CharCode);

        if   not ( csDesigning in ComponentState)
        then begin
          if   Perform( CM_CHILDKEY, CharCode, Winapi.Windows.LPARAM( Self)) <> 0
          then Exit;

          Mask := 0;

          case CharCode of
            VK_TAB                                      : Mask := DLGC_WANTTAB;
            VK_LEFT, VK_RIGHT, VK_UP, VK_DOWN           : Mask := DLGC_WANTARROWS;
            VK_RETURN, VK_EXECUTE, VK_ESCAPE, VK_CANCEL : Mask := DLGC_WANTALLKEYS;
          end;

          if  ( Mask <> 0)
          and ( Perform( CM_WANTSPECIALKEY, CharCode, 0) = 0)
          and ( Perform( WM_GETDLGCODE, 0, 0) and Mask = 0)
          and ( GetParentForm( Self).Perform( CM_DIALOGKEY, CharCode, KeyData) <> 0)
          then Exit;
        end;

        Result := 0;
      end;
    end;


    procedure   TKnobsDisplayEditor.Do_KeyDown( aSender: TObject; var aKey: Word; aShift: TShiftState);

      procedure HandleReturn;
      begin
        EndEdit;
        aKey := 0;

        if   Assigned( FDisplay)
        then begin
          FDisplay.SetText( Text);
          FDisplay.LockKnob;
        end;
      end;

    begin
      if   aShift = []
      then begin
        case aKey Of

          VK_ESCAPE :

            begin
              EndEdit;
              aKey := 0;
            end;

          VK_RETURN :

            if   not FDisplay.MultiLine
            then HandleReturn;

        end;
      end
      else if aShift = [ ssCtrl]
      then begin
        case aKey Of

          VK_RETURN : HandleReturn;

        end;
      end;
    end;


//  protected

    procedure   TKnobsDisplayEditor.SetDisplay( aDisplay: TKnobsDisplay);
    var
      aLeft   : Integer;
      aTop    : Integer;
      aWidth  : Integer;
      aHeight : Integer;
    begin
      FDisplay := aDisplay;
      Parent   := aDisplay.Parent;
      Text     := aDisplay.Caption;

      if   aDisplay.HasParentEditor
      then SetBounds( 0, 0, Parent.Width, Parent.Height)
      else begin
        aLeft   := 0;
        aTop    := aDisplay.Top -  2;
        aWidth  := Parent.Width;
        aHeight := aDisplay.Height +  4;

        if   aTop < 0
        then aTop := 0;

        if   aTop + aHeight > Parent.Height
        then aTop := Parent.Height - aHeight;

        SetBounds( aLeft, aTop, aWidth, aHeight);
      end;

      if   aDisplay.HasScrollbar
      then ScrollBars := ssVertical
      else ScrollBars := ssNone;

      Font         := aDisplay.Font;
      Color        := clWhite;
      Font.Color   := clBlack;
      Visible      := True;
      WantReturns  := aDisplay.MultiLine;
      WantTabs     := False;
      SetFocus;
      PopupEditorShow( Self);
    end;


//  public

    constructor TKnobsDisplayEditor.Create( anOwner: TComponent); // override;
    begin
      if   Assigned( GDisplayEditor)
      then Exit;

      inherited;
      ControlStyle   := ControlStyle + [ csOpaque];
      Visible        := False;
      Left           := 1;
      Top            := 1;
      OnExit         := Do_Exit;
      OnKeyDown      := Do_KeyDown;
      GDisplayEditor := Self;
    end;


    destructor  TKnobsDisplayEditor.Destroy; // override;
    begin
      if   Assigned( Owner)
      then Owner.RemoveComponent( Self);

      GDisplayEditor := nil;
      PopupEditorHide( Self);
      inherited;
    end;


{ ========
  TKnobsTextControl = class( TStaticText)
  private
    FBorderColor      : TColor;
    FMustSave         : Boolean;
    FMustNotify       : Boolean;
    FSavedColor       : TColor;
    FPrevValue        : string;
    FOnChanged        : TKnobsOnTextChanged;
  public
    property    ModuleName    : string              read GetModuleName;
  public
    property    Module        : TKnobsCustomModule  read GetModule;
  published
    property    TextValue     : string              read GetTextValue     write SetTextValue;
    property    Ctl3D                                                                          default False;
    property    ParentColor                                                                    default False;
    property    Color                                                                          default clGray;
    property    Width                                                                          default 33;
    property    Height                                                                         default 15;
    property    AutoSize                                                                       default False;
    property    BorderStyle                                                                    default sbsSingle;
    property    TabStop                                                                        default True;
    property    StyleElements                                                                  default [ seBorder];
    property    BorderColor   : TColor              read FBorderColor     write SetBorderColor default clGray;
    property    MustSave      : Boolean             read FMustSave        write FMustSave      default False;
    property    MustNotify    : Boolean             read FMustNotify      write FMustNotify    default False;
    property    OnChanged     : TKnobsOnTextChanged read FOnChanged       write FOnChanged;
    property    OnClick;
  strict private
 }

    class constructor TKnobsTextControl.Create;
    begin
      TStyleManager.Engine.RegisterStyleHook( TKnobsTextControl, TknobsTextControlStyleHook);
    end;


    class destructor  TKnobsTextControl.Destroy;
    begin
      TStyleManager.Engine.UnregisterStyleHook( TKnobsTextControl, TknobsTextControlStyleHook);
    end;

//  private

    function    TKnobsTextControl.GetModuleName: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


    procedure   TKnobsTextControl.SetBorderColor( aColor: TColor);
    begin
      if   aColor <> FBorderColor
      then begin
        FBorderColor := aColor;
        RecreateWnd;
      end;
    end;


    function    TKnobsTextControl.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    function    TKnobsTextControl.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then Result := TKnobsCustomModule( Owner)
      else Result := nil;
    end;


    procedure   TKnobsTextControl.Do_KeyDown( aSender: TObject; var aKey: Word; aShift: TShiftState);
    begin
      if   aShift = []
      then begin
        case aKey of

          VK_RETURN:

            begin
              HandleReturnKey;
              aKey := 0;
            end;

        end;
      end;
    end;


 // protected

    procedure   TKnobsTextControl.CreateParams( var aParams: TCreateParams); // override;
    begin
      inherited;
   // aParams.Style := aParams.Style and not SS_NOTIFY;
    end;


    procedure   TKnobsTextControl.SetText( const aValue: string);
    begin
      if   TextValue <> aValue
      then begin
        try
          HandleTextChange( aValue);
          TextValue := aValue;
        except
          on E: EAbort
          do KilledException( E)  // Absorb aborts
          else raise;             // Re-raise errors
        end;
      end;
    end;


    procedure   TKnobsTextControl.TextChanged( const aNewValue: string; DoNotify: Boolean); // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnChanged)
      then FOnChanged( Self, Name, aNewValue)
      else begin
        aModule := Module;

        if   Assigned( aModule) and ( DoNotify or MustNotify)
        then Module.TextChanged( Self, MakePath( [ aModule.Name, Name]), aNewValue);
      end;
    end;


    procedure   TKnobsTextControl.HandleTextChange( const aValue: string); // virtual;
    begin
      TextChanged( aValue, DO_NOTIFY);
    end;


    procedure   TKnobsTextControl.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    // Ctrl + click should fire up some text editor
    // Right clicks ripple up to the wire panel for right mouse menu support
    // other events should be handled by the containing module.
    var
      aModule : TKnobsCustomModule;
      aWP     : TKnobsWirePanel;
    begin
      SetFocus;

      if   csDesigning in ComponentState
      then inherited
      else begin
        if   ( ssCtrl in Shift) and ( Button = mbLeft)
        then HandleDoubleClick
        else if Button = mbRight
        then begin
          aWP := FindWirePanel;

          if   Assigned( aWP)
          then aWP.HandleRightClick( Self);
        end
        else begin
          // Pass it on to the containing module
          aModule := Module;

          if   Assigned( aModule)
          then begin
            aModule.MouseCapture := True;
            aModule.MouseDown( Button, Shift, X + Left, Y + Top);
          end;
        end;
      end;
    end;


    procedure   TKnobsTextControl.WMSetFocus( var aMessage: TWMSetFocus); // message WM_SETFOCUS;
    begin
      inherited;
      FSavedColor := Color;
      Color       := clBlack;
    end;


    procedure   TKnobsTextControl.WMKillFocus( var aMessage: TWMKillFocus); // message WM_KILLFOCUS;
    begin
      inherited;
      Color := FSavedColor;
    end;


//  public

    constructor TKnobsTextControl.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle      := ControlStyle - [ csSetCaption] + [ csOpaque];
      StyleElements     := [ seBorder];
      DoubleBuffered    := True;
      Ctl3D             := False;
      AutoSize          := False;
      BorderStyle       := sbsSingle;
      Width             := 33;
      Height            := 15;
      Font.Color        := clWhite;
      ParentColor       := False;
      Color             := clGray;
      BorderColor       := clGray;
      FSavedColor       := Color;
      TabStop           := True;
      MustSave          := False;
      MustNotify        := False;
      OnKeyDown         := Do_KeyDown;
    end;


    procedure   TKnobsTextControl.Edit; // virtual;
    begin
      HandleDoubleClick;
    end;


    procedure   TKnobsTextControl.BeginStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      FPrevValue := TextValue;
      aWp        := FindWirePanel;

      if   Assigned( aWp)
      then aWp.BeginStateChange( False);
    end;


    procedure   TKnobsTextControl.EndStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.EndStateChange( False, False);
    end;


{ ========
  TKnobsDisplay = class( TKnobsTextControl)
  // Can be linked to aTKnob for auto update
  // Can have an editor to manually edit the value
  private
    FShortName       : string;
    FEditor          : TKnobsDisplayEditor;
    FController      : TKnobsValuedControl;
    FHasEditor       : Boolean;
    FHasParentEditor : Boolean;
    FHasScrollbar    : Boolean;
    FMultiLine       : Boolean;
  public
    property    ShortName       : string              read FShortName;
    property    Editor          : TKnobsDisplayEditor read FEditor          write FEditor;
  published
    property    HasEditor       : Boolean             read FHasEditor       write FHasEditor       default False;
    property    HasParentEditor : Boolean             read FHasParentEditor write FHasParentEditor default False;
    property    HasScrollbar    : Boolean             read FHasScrollbar    write FHasScrollbar    default False;
    property    MultiLine       : Boolean             read FMultiLine       write FMultiLine       default False;
    property    BorderColor                                                                        default clSilver;
  strict private
}

    class constructor TKnobsDisplay.Create;
    begin
      TStyleManager.Engine.RegisterStyleHook( TKnobsDisplay, TknobsTextControlStyleHook);
    end;


    class destructor  TKnobsDisplay.Destroy;
    begin
      TStyleManager.Engine.UnregisterStyleHook( TKnobsDisplay, TknobsTextControlStyleHook);
    end;


//  protected

    procedure   TKnobsDisplay.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsDisplay.HandleReturnKey; // override;
    begin
      BeginStateChange;
      NeedsDisplayEditor;
      GDisplayEditor.SetDisplay( Self);
    end;


    procedure   TKnobsDisplay.HandleDoubleClick; // override;
    begin
      BeginStateChange;
      NeedsDisplayEditor;
      GDisplayEditor.SetDisplay( Self);
    end;


    function    TKnobsDisplay.GetTextValue: string; // override;
    begin
      Result := Caption;
    end;


    procedure   TKnobsDisplay.SetTextValue( const aValue: string); // override;
    begin
      Caption := aValue;
    end;


    procedure   TKnobsDisplay.TextChanged( const aNewValue: string; DoNotify: Boolean); // override;
    begin
      if   not Assigned( FController)
      or   (( Assigned( FController) and FController.Locked))
      then inherited;
    end;


    procedure   TKnobsDisplay.WMNCHitTest( var aMessage: TWMNCHitTest); // message WM_NCHITTEST;
    begin
      if   ( csDesigning in ComponentState) or HasEditor
      then inherited
      else aMessage.Result := HTTRANSPARENT;
    end;


    procedure   TKnobsDisplay.LockKnob;
    begin
      if   Assigned( FController)
      then FController.Locked := True;
    end;


//  public

    constructor TKnobsDisplay.Create( anOwner: TComponent); // override;
    begin
      inherited;
      TextValue       := '';
      HasEditor       := False;
      HasParentEditor := False;
      HasScrollbar    := False;
      MultiLine       := False;
      BorderColor     := clSilver;
    end;


    destructor  TKnobsDisplay.Destroy; // override;
    begin
      if   FController is TKnobsKnob
      then TKnobsKnob( FController).Display := nil;

      inherited;
    end;


{ ========
  TKnobsSelectorChoice = class( TKnobsDisplay)
  // Allows for making a selection from items in a selector
  // which then can be used to allow for selector modulatio
  private
    FSelector  : TKnobsSelector;
    FSelection : TByteSet;
    FNames     : TStringList;
    FChoice    : Byte;
  public
    property    Names                    : TStringList    read FNames      write SetNames;
    property    Choice                   : Byte           read FChoice     write SetChoice;
    property    Selected[ anIndex: Byte] : Boolean        read GetSelected write SetSelected;
  published
    property    Selector                 : TKnobsSelector read FSelector   write SetSelector;
  private
}

    procedure   TKnobsSelectorChoice.SetSelector( const aValue: TKnobsSelector);
    var
      FOldSelector: TKnobsSelector;
    begin
      if FSelector <> aValue
      then begin
        FOldSelector := FSelector;
        FSelector    := aValue;

        if Assigned( FOldSelector)
        then FOldSelector.Choice := nil;

        if Assigned( FSelector)
        then FSelector.Choice := self;
      end;
    end;


    procedure   TKnobsSelectorChoice.SetNames( const aValue: TStringList);
    var
      i : Integer;
    begin
      FNames.Clear;
      FSelection := [];

      if   Assigned( aValue)
      then begin
        for i:= 0 to aValue.Count - 1
        do begin
          if aValue[ i] <> '-'
          then FNames.Add( aValue[ i]);
        end;
      end;
    end;


    procedure   TKnobsSelectorChoice.SetChoice( aValue: Byte);
    begin
      if   Enabled
      and  ( aValue < FNames.Count)
      and  ( aValue <> FChoice   )
      then begin
        FChoice   := aValue;
        TextValue := FNames[ aValue];

        if Assigned( FSelector)
        then FSelector.KnobPosition := aValue;
      end;
    end;


    function    TKnobsSelectorChoice.GetSelected( anIndex: Byte): Boolean;
    begin
      Result := anIndex in FSelection;
    end;


    procedure   TKnobsSelectorChoice.SetSelected( anIndex: Byte; aValue: Boolean);
    begin
      if   aValue
      then SelectItem  ( anIndex)
      else UnselectItem( anIndex);
    end;


//  protected

    procedure   TKnobsSelectorChoice.HandleReturnKey; // override;
    begin
      BeginStateChange;
      NeedsChoiceEditor;
      GChoiceSelector.SetSelectorChoise( Self);
    end;


    procedure   TKnobsSelectorChoice.HandleDoubleClick; // override;
    begin
      BeginStateChange;
      NeedsChoiceEditor;
      GChoiceSelector.SetSelectorChoise( Self);
    end;


//  public

    constructor TKnobsSelectorChoice.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FNames := TStringList.Create;
    end;


    destructor  TKnobsSelectorChoice.Destroy; // override;
    begin
      Selector := nil;
      Clear;
      FNames.DisposeOf;
      inherited;
    end;


    procedure   TKnobsSelectorChoice.Clear;
    begin
      FSelection := [];
      FNames.Clear;
    end;


    procedure   TKnobsSelectorChoice.SelectItem( anItem: Byte);
    begin
      if   anItem < FNames.Count
      then FSelection := FSelection + [ anItem];
    end;


    procedure   TKnobsSelectorChoice.UnselectItem( anItem: Byte);
    begin
      FSelection := FSelection - [ anItem];
    end;


{ ========
  TKnobsFileSelector = class( TKnobsTextControl)
  private
    FFileSelector : TOpenDialog;
    FDataFileName : string;
    FDataFileExt  : string;
    FDataFileMsg  : string;
  strict private
    class var FScalaCounter : Integer;
  published
    property    DataFileName : string read FDataFileName write SetDataFileName;
    property    DataFileExt  : string read FDataFileExt  write FDataFileExt;
    property    DataFileMsg  : string read FDataFileMsg  write FDataFileMsg;
  strict private
}

    class constructor TKnobsFileSelector.Create;
    begin
      TStyleManager.Engine.RegisterStyleHook( TKnobsFileSelector, TknobsTextControlStyleHook);

      if   not Assigned( GScalaSelector)
      then begin
        GScalaSelector := TFormScala.Create( nil);
        FScalaCounter  := 0;
      end
      else Inc( FScalaCounter);
    end;


    class destructor  TKnobsFileSelector.Destroy;
    begin
      if   FScalaCounter > 0
      then Dec( FScalaCounter);

      if   FScalaCounter <= 0
      then FreeAndNil( GScalaSelector);

      TStyleManager.Engine.UnregisterStyleHook( TKnobsFileSelector, TknobsTextControlStyleHook);
    end;


//  private

    procedure   TKnobsFileSelector.SetDataFileName( const aValue: string);
    begin
      FDataFileName := aValue;
      FixCaption;
    end;


    procedure   TKnobsFileSelector.FixCaption;
    begin
      if   FDataFileName = ''
      then Caption := ' Ctrl click to select a file'
      else Caption := ' ' + ExtractFileName( FDataFileName);
    end;


    procedure   TKnobsFileSelector.SelectFile;
    var
      aFileType : TScalaFileType;
    begin
      if
        SameText( DataFileExt, '.scl') or
        SameText( DataFileExt, '.kbm')
      then begin
        with GScalaSelector
        do begin
          if   FileExists( DataFileName)
          then begin
            FileName   := DataFileName;
            InitialDir := ExtractFilePath( DataFileName);
          end
          else InitialDir := ApplicationPath;

          if   SameText( DataFileExt, '.scl')
          then aFileType := ftSCL
          else aFileType := ftKBM;

          if   Execute( aFileType)
          then SetText( FileName);
        end;
      end
      else begin
        with FFileSelector
        do begin
          DefaultExt := DataFileExt;
          Title      := DataFileMsg;

          // todo : make a mapping from file extension to description and use that to register new fle types
          //        in, then reduce the following code to something like :
          //
          //          Ext    := Copy( DataFileExt, 2, Length( DataFileExt))
          //          Filter := Format( '%0:s (*.%1:s)|*.%1:s|All files (*.*)|*.*', [Map[ Ext], Ext], AppLocale)
          //
          //        Registration would then be something like :
          //
          //         RegisterExtension( const anExtension, aDescription: string);

          if   SameText( DataFileExt, '.wav')
          then Filter := 'wave files (*.wav)|*.wav|All files (*.*)|*.*'
          else if SameText( DataFileExt, '.mid')
          then Filter := 'midi files (*.mid)|*.mid|All files (*.*)|*.*';

          if   FileExists( DataFileName)
          then begin
            FileName   := DataFileName;
            InitialDir := ExtractFilePath( DataFileName);
          end
          else InitialDir := ApplicationPath;

          if   Execute
          then SetText( FileName);
        end;
      end;
    end;


//  protected

    procedure   TKnobsFileSelector.HandleReturnKey; // override;
    begin
      SelectFile;
    end;


    procedure   TKnobsFileSelector.HandleDoubleClick; // override;
    begin
      SelectFile;
    end;


    function    TKnobsFileSelector.GetTextValue: string; // override;
    begin
      Result := DataFileName;
    end;


    procedure   TKnobsFileSelector.SetTextValue( const aValue: string); // override;
    begin
      DataFileName := aValue;
    end;


//  public

    constructor TKnobsFileSelector.Create( anOwner: TComponent); // override;
    begin
      inherited;

      FFileSelector := TOpenDialog.Create( nil);
      Caption       := 'Ctrl click to select a file';
      TextValue     := '';

      with FFileSelector
      do begin
        DefaultExt := '';
        Title      := 'select a file';
        Filter     := '';
      end;
    end;


    destructor  TKnobsFileSelector.Destroy; // override;
    begin
      FreeAndNil( FFileSelector );
      inherited;
    end;


{ ========
  TKnobsIndicator = class( TKnobsGraphicControl)
  // an LED
  private
    FShortName  : string;
    FActive     : Boolean;
    FActiveBm   : TBitmap;
    FInActiveBm : TBitmap;
  public
    property    BitmapActive   : TBitmap read FActiveBm   write SetBitmapActive;
    property    BitmapInactive : TBitmap read FInactiveBm write SetBitmapInactive;
    property    ShortName      : string  read FShortName;
  published
    property    Active         : Boolean read FActive     write SetActive default False;
    property    Visible;
    property    ShowHint                                                  default False;
    property    ParentShowHint                                            default False;
    property    OnClick;
  private
}

    procedure   TKnobsIndicator.SetActive( aValue: Boolean);
    begin
      if   aValue <> FActive
      then begin
        FActive := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicator.SetBitmapActive( aValue: TBitmap);
    begin
      FActiveBm.Assign( aValue);

      if   FActive
      then Invalidate;
    end;


    procedure   TKnobsIndicator.SetBitmapInactive( aValue: TBitmap);
    begin
      FInactiveBm.Assign( aValue);

      if   not FActive
      then Invalidate;
    end;


//  protected

    procedure   TKnobsIndicator.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsIndicator.AssignBitmaps; // virtual;
    begin
      FInactiveBm := bmIndInactive;
      FActiveBm   := bmIndActive  ;
      FInActiveBm.Transparent := False;
      FActiveBm  .Transparent := False;
    end;


    procedure   TKnobsIndicator.Paint; // override;
    begin
      inherited;

      with Canvas
      do begin
        if   Active
        then Draw( 0, 0, FActiveBm  )
        else Draw( 0, 0, FInactiveBm);
      end;
    end;


//  public

    constructor TKnobsIndicator.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle   := ControlStyle + [ csOpaque];
      AssignBitmaps;
      Width          := Max( FActiveBm.Width , FInactiveBm.Width );
      Height         := Max( FActiveBm.Height, FInactiveBm.Height);
      FActive        := False;
      Enabled        := False;
      ShowHint       := False;
      ParentShowHint := False;
    end;


    procedure   TKnobsIndicator.FixBitmaps; // virtual;
    begin
      AssignBitmaps;
    end;


{ ========
  TKnobsIndicatorBar = class( TKnobsGraphicControl)
  // A bar of indicators, horizontal or vertical
  private
    FIndicatorCount : Integer;
    FValue          : TSignal;
    FPeakValue      : TSignal;
    FValeyValue     : TSignal;
    FShowPeakValue  : Boolean;
    FShowValeyValue : Boolean;
    FValueLowMid    : TSignal;
    FValueMidHigh   : TSignal;
    FDisplayType    : TKnobsDisplayType;
    FOrientation    : TKnobsDisplayOrientation;
    FSpacing        : Integer;
    FBitmap         : TBitmap;
    FActiveLowBm    : TBitmap;
    FInActiveLowBm  : TBitmap;
    FActiveMidBm    : TBitmap;
    FInActiveMidBm  : TBitmap;
    FActiveHighBm   : TBitmap;
    FInActiveHighBm : TBitmap;
    FPeakColor      : TColor;
    FValeyColor     : TColor;
  public
    property    ActiveLowBm    : TBitmap                  read FActiveLowBm    write SetActiveLowBm;
    property    InActiveLowBm  : TBitmap                  read FInActiveLowBm  write SetInActiveLowBm;
    property    ActiveMidBm    : TBitmap                  read FActiveMidBm    write SetActiveMidBm;
    property    InActiveMidBm  : TBitmap                  read FInActiveMidBm  write SetInActiveMidBm;
    property    ActiveHighBm   : TBitmap                  read FActiveHighBm   write SetActiveHighBm;
    property    InActiveHighBm : TBitmap                  read FInActiveHighBm write SetInActiveHighBm;
  published
    property    IndicatorCount : Integer                  read FIndicatorCount write SetIndicatorCount default 10;
    property    Value          : TSignal                  read FValue          write SetValue          nodefault;
    property    PeakValue      : TSignal                  read FPeakValue      write SetPeakValue      nodefault;
    property    ValeyValue     : TSignal                  read FValeyValue     write SetValeyValue     nodefault;
    property    ShowPeakValue  : Boolean                  read FShowPeakValue  write SetShowPeakValue  default False;
    property    ShowValeyValue : Boolean                  read FShowValeyValue write SetShowValeyValue default False;
    property    ValueLowMid    : TSignal                  read FValueLowMid    write SetValueLowMid    nodefault;
    property    ValueMidHigh   : TSignal                  read FValueMidHigh   write SetValueMidHigh   nodefault;
    property    DisplayType    : TKnobsDisplayType        read FDisplayType    write SetDisplayType    default dtFilledFromLow;
    property    Orientation    : TKnobsDisplayOrientation read FOrientation    write SetOrientation    default doHorizontal;
    property    Spacing        : Integer                  read FSpacing        write SetSpacing        default 2;
    property    PeakColor      : TColor                   read FPeakColor      write SetPeakColor      default clWhite;
    property    ValeyColor     : TColor                   read FValeyColor     write SetValeyColor     default clBlack;
    property    Visible;
    property    Enabled;
  published
    property    OnClick;
    property    OnDblClick;
  private
}

    function    TKnobsIndicatorBar.GetMaxBmWidth: Integer;
    begin
      Result := 0;
      Result := Max( Result,   ActiveLowBm .Width);
      Result := Max( Result, InActiveLowBm .Width);
      Result := Max( Result,   ActiveMidBm .Width);
      Result := Max( Result, InActiveMidBm .Width);
      Result := Max( Result,   ActiveHighBm.Width);
      Result := Max( Result, InActiveHighBm.Width);
    end;


    function    TKnobsIndicatorBar.GetMaxBmHeight: Integer;
    begin
      Result := 0;
      Result := Max( Result,   ActiveLowBm .Height);
      Result := Max( Result, InActiveLowBm .Height);
      Result := Max( Result,   ActiveMidBm .Height);
      Result := Max( Result, InActiveMidBm .Height);
      Result := Max( Result,   ActiveHighBm.Height);
      Result := Max( Result, InActiveHighBm.Height);
    end;


    procedure   TKnobsIndicatorBar.FixDimensions;
    begin
      if   Orientation = doHorizontal
      then begin
        Width  := IndicatorCount * ( GetMaxBmWidth + Spacing);
        Height :=                    GetMaxBmHeight;
      end
      else begin
        Width  :=                    GetMaxBmWidth;
        Height := IndicatorCount * ( GetMaxBmHeight + Spacing);
      end;

      FBitmap.Width  := Self.Width;
      FBitmap.Height := Self.Height;
    end;


//  private

    procedure   TKnobsIndicatorBar.SetIndicatorCount( aValue: Integer);
    begin
      if   aValue <> FIndicatorCount
      then begin
        FIndicatorCount := aValue;

        if   ( Value < 0) or ( Value > aValue)
        then Value := 3 * aValue div 2;

        FixDimensions;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetValue( aValue: TSignal);
    begin
      if   aValue <> FValue
      then begin
        FValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetPeakValue( aValue: TSignal);
    begin
      if   aValue <> FPeakValue
      then begin
        FPeakValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetValeyValue( aValue: TSignal);
    begin
      if   aValue <> FValeyValue
      then begin
        FValeyValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetShowPeakValue( aValue: Boolean);
    begin
      if   aValue <> FShowPeakValue
      then begin
        FShowPeakValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetShowValeyValue( aValue: Boolean);
    begin
      if   aValue <> FShowValeyValue
      then begin
        FShowValeyValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetValueLowMid( aValue: TSignal);
    begin
      if   aValue <> FValueLowMid
      then begin
        FValueLowMid := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetValueMidHigh( aValue: TSignal);
    begin
      if   aValue <> FValueMidHigh
      then begin
        FValueMidHigh := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetDisplayType( aValue: TKnobsDisplayType);
    begin
      if   aValue <> FDisplayType
      then begin
        FDisplayType := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetOrientation( aValue: TKnobsDisplayOrientation);
    begin
      if   aValue <> FOrientation
      then begin
        FOrientation := aValue;
        FixDimensions;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetSpacing( aValue: Integer);
    begin
      if   aValue <> FSpacing
      then begin
        FSpacing := aValue;
        FixDimensions;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetActiveLowBm( aValue: TBitmap);
    begin
      FActiveLowBm.Assign( aValue);
      FixDimensions;
      Invalidate;
    end;


    procedure   TKnobsIndicatorBar.SetInActiveLowBm( aValue: TBitmap);
    begin
      FInActiveLowBm.Assign( aValue);
      FixDimensions;
      Invalidate;
    end;


    procedure   TKnobsIndicatorBar.SetActiveMidBm( aValue: TBitmap);
    begin
      FActiveMidBm.Assign( aValue);
      FixDimensions;
      Invalidate;
    end;


    procedure   TKnobsIndicatorBar.SetInActiveMidBm( aValue: TBitmap);
    begin
      FInActiveMidBm.Assign( aValue);
      FixDimensions;
      Invalidate;
    end;


    procedure   TKnobsIndicatorBar.SetActiveHighBm( aValue: TBitmap);
    begin
      FActiveHighBm.Assign( aValue);
      FixDimensions;
      Invalidate;
    end;


    procedure   TKnobsIndicatorBar.SetInActiveHighBm( aValue: TBitmap);
    begin
      FInActiveHighBm.Assign( aValue);
      FixDimensions;
      Invalidate;
    end;


    procedure   TKnobsIndicatorBar.SetPeakColor( aValue: TColor);
    begin
      if   aValue <> FPeakColor
      then begin
        FPeakColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorBar.SetValeyColor( aValue: TColor);
    begin
      if   aValue <> FValeyColor
      then begin
        FValeyColor := aValue;
        Invalidate;
      end;
    end;


//  protected

    procedure   TKnobsIndicatorBar.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsIndicatorBar.AssignBitmaps; // virtual;
    begin
      FInactiveLowBm  := bmIndLowInactive;
      FActiveLowBm    := bmIndLowActive;
      FInactiveMidBm  := bmIndMidInactive;
      FActiveMidBm    := bmIndMidActive;
      FInactiveHighBm := bmIndHighInactive;
      FActiveHighBm   := bmIndHighActive;
      FInactiveLowBm .Transparent := False;
      FActiveLowBm   .Transparent := False;
      FInactiveMidBm .Transparent := False;
      FActiveMidBm   .Transparent := False;
      FInactiveHighBm.Transparent := False;
      FActiveHighBm  .Transparent := False;
      FixDimensions;
    end;


    function    TKnobsIndicatorBar.PaintActive( anIndex: Integer): Boolean;
    begin
      Result := False;

      case DisplayType of
        dtSingleFromLow, dtSingleFromHigh : Result := anIndex = Value - 1;
        dtFilledFromLow, dtFilledFromHigh : Result := anIndex < Value;
      end;
    end;


    procedure   TKnobsIndicatorBar.Paint; // override;
    var
      i  : Integer;
      L  : Integer;
      T  : Integer;
      W  : Integer;
      H  : Integer;
      LA : Integer;
      TA : Integer;
    begin
      inherited;
      W := GetMaxBmWidth  + Spacing;
      H := GetMaxBmHeight + Spacing;

      if   Orientation = doHorizontal
      then begin
        TA := 0;

        if   DisplayType in [ dtSingleFromHigh, dtFilledFromHigh]
        then begin
          W  := - W;
          LA := Width + W;
        end
        else LA := 0;
      end
      else begin
        LA := 0;

        if   DisplayType in [ dtSingleFromLow, dtFilledFromLow]
        then begin
          H  := - H;
          TA := Height + H;
        end
        else TA := 0;
      end;

      L := LA;
      T := TA;

      with FBitmap.Canvas
      do begin
        Brush.Color := clBlack;
        Rectangle( 0, 0, Width, Height);

        for i := 0 to IndicatorCount - 1
        do begin
          if   PaintActive( i)
          then begin
            if   i < ValueLowMid
            then Draw( L, T, FActiveLowBm  )
            else if i >= ValueMidHigh
            then Draw( L, T, FActiveHighBm)
            else Draw( L, T, FActiveMidBm);
          end
          else begin
            if   i < ValueLowMid
            then Draw( L, T, FInActiveLowBm  )
            else if i >= ValueMidHigh
            then Draw( L, T, FInActiveHighBm)
            else Draw( L, T, FInActiveMidBm);
          end;

          if   Orientation = doHorizontal
          then L := L + W
          else T := T + H;
        end;

        if   ShowPeakValue
        then begin
          Pen.Color := PeakColor;
          Pen.Width := 3;

          if   Orientation = doHorizontal
          then begin
            L := LA + ( Ceil( PeakValue) - 1) * W + 1;
            MoveTo( L, Height div 2);
            LineTo( L, Height      );
          end
          else begin
            T := TA + ( Ceil( PeakValue) - 1) * H - 1;
            MoveTo( Width div 2, T);
            LineTo( Width      , T);
          end;
        end;

        if   ShowValeyValue
        then begin
          Pen.Color := ValeyColor;
          Pen.Width := 3;

          if   Orientation = doHorizontal
          then begin
            L := LA + ( Ceil( ValeyValue) - 1) * W + 1;
            MoveTo( L, 0           );
            LineTo( L, Height div 2);
          end
          else begin
            T := TA + ( Ceil( ValeyValue) - 1) * H - 1;
            MoveTo( 0           , T);
            LineTo( Width div 2 , T);
          end;
        end;
      end;

      Canvas.Brush.Color := clWhite;
      Canvas.Draw( 0, 0, FBitmap);
    end;


//  public

    constructor TKnobsIndicatorBar.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FBitmap := TBitmap.Create;
      AssignBitmaps;
      ControlStyle   := ControlStyle + [ csOpaque];
      ShowPeakValue  := False;
      ShowValeyValue := False;
      IndicatorCount := 10;
      Value          :=  6;
      PeakValue      :=  0;
      ValueLowMid    :=  5;
      ValueMidHigh   :=  8;
      DisplayType    := dtFilledFromLow;
      Orientation    := doHorizontal;
      Spacing        := 2;
      FPeakColor     := clWhite;
      FValeyColor    := clBlack;

      FixDimensions;
    end;


    destructor  TKnobsIndicatorBar.Destroy; // override;
    begin
      FBitmap.DisposeOf;
      inherited;
    end;


    procedure   TKnobsIndicatorBar.FixBitmaps; // virtual;
    begin
      AssignBitmaps;
    end;


{ ========
  TKnobsIndicatorText = class( TCustomLabel)
  private
    FShortName     : string;
    FValue         : TSignal;
    FDisplayFormat : string;
  public
    property    ShortName     : string  read FShortName;
  published
    property    DisplayFormat : string  read FDisplayFormat write SetDisplayFormat;
    property    Value         : TSignal read FValue         write SetValue          nodefault;
    property    Width                                                               default 10;
    property    Height                                                              default 10;
    property    AutoSize                                                            default False;
    property    Align                                                               default alNone;
    property    Alignment                                                           default taCenter;
    property    Color                                                               default clGray;
    property    Transparent                                                         default True;
    property    Visible                                                             default True;
    property    WordWrap                                                            default False;
    property    ParentColor                                                         default False;
    property    ParentFont                                                          default False;
    property    ShowHint                                                            default False;
    property    ParentShowHint                                                      default False;
    property    ShowAccelChar                                                       default False;
    property    Layout;
    property    Anchors;
    property    BiDiMode;
    property    Caption;
    property    Constraints;
    property    Enabled;
    property    FocusControl;
    property    Font;
    property    ParentBiDiMode;
    property    PopupMenu;
    property    OnClick;
  protected
}

    procedure   TKnobsIndicatorText.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsIndicatorText.SetValue( aValue: TSignal);
    begin
      if   aValue <> FValue
      then begin
        FValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorText.SetDisplayFormat( const aValue: string);
    begin
      if   aValue <> FDisplayFormat
      then begin
        FDisplayFormat := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsIndicatorText.DoDrawText( var aRect: TRect; aFlags: Longint); // override;
    var
      Text : string;
    begin
      Text   := Format( DisplayFormat, [ FValue]);
      aFlags := aFlags or DT_NOPREFIX;
      aFlags := DrawTextBiDiModeFlags( aFlags);
      Canvas.Font := Font;
      DrawText( Canvas.Handle, PChar( Text), Length( Text), aRect, aFlags);
    end;


//  public

    constructor TKnobsIndicatorText.Create( anOwner: TComponent); // override;
    begin
      inherited;
      Width          := 10;
      Height         := 10;
      AutoSize       := false;
      Align          := alNone;
      Alignment      := taCenter;
      Color          := clGray;
      Transparent    := True;
      Visible        := True;
      WordWrap       := False;
      ParentColor    := False;
      ParentFont     := False;
      ShowHint       := False;
      ParentShowHint := False;
      ShowAccelChar  := False;
      DisplayFormat  := '%.0f';

      with Font
      do begin
        Name  := 'Small Fonts';
        Size  := 7;
        Color := clWhite;
      end;
    end;


{ ========
  TKnobsMazeGraphViewer = class( TKnobsGraphicControl)
  private
    FMazeGraph     : TMazeGraph;
    FMazeForth     : TMazeForth;
    FShortName     : string;
    FSwanX         : TSignal;
    FSwanY         : TSignal;
    FColor         : TColor;
    FBorderColor   : TColor;
    FSwanColor     : TColor;
    FHunterColor   : TColor;
    FSwanSize      : Integer;
    FHunterSize    : Integer;
    FForthFileName : string;
    FImportList    : TStringList;
    FCurrentImport : Integer;
    FShowingError  : Boolean;
    FOnError       : TmazeOnError;
  public
    property    ShortName                         : string              read FShortName;
    property    ForthFileName                     : string              read FForthFileName  write SetForthFileName;
  public
    property    HunterPosition[ anIndex: Integer] : Integer             read GetHunterPosition write SetHunterPosition;
  published
    property    SwanX                             : TSignal             read FSwanX          write SetSwanX;            // default 0.5;
    property    SwanY                             : TSignal             read FSwanY          write SetSwanY;            // default 0.5;
    property    Color                             : TColor              read FColor          write SetColor                default $009E9E9E;
    property    BorderColor                       : TColor              read FBorderColor    write SetBorderColor          default clGray;
    property    SwanColor                         : TColor              read FSwanColor      write SetSwanColor            default clPurple;
    property    HunterColor                       : TColor              read FHunterColor    write SetHunterColor          default clBlue;
    property    SwanSize                          : Integer             read FSwanSize       write SetSwanSize             default 4;
    property    HunterSize                        : Integer             read FHunterSize     write SetHunterSize           default 3;
    property    HunterCount                       : Integer             read GetHunterCount  write SetHunterCount          default 4;
    property    Height                                                                                                    default 242;
    property    Width                                                                                                     default 242;
    property    OnError                           : TMazeOnError        read FOnError        write FOnError;
  private
}

    function    TKnobsMazeGraphViewer.GetHunterCount: Integer;
    begin
      if   Assigned( FMazeGraph)
      then Result := FMazeGraph.HunterCount
      else Result := 0;
    end;


    procedure   TKnobsMazeGraphViewer.SetHunterCount( aValue: Integer);
    begin
      if   Assigned( FMazeGraph)
      and  ( aValue <> HunterCount)
      then FMazeGraph.HunterCount := aValue;
    end;


    function    TKnobsMazeGraphViewer.GetHunterPosition( anIndex: Integer): Integer;
    begin
      if   Assigned( FMazeGraph)
      then Result := FMazeGraph.Hunter[ anIndex].Location
      else Result := -1;
    end;


    procedure   TKnobsMazeGraphViewer.SetHunterPosition( anIndex: Integer; aValue: Integer);
    begin
      if   Assigned( FMazeGraph)
      then begin
        FMazeGraph.Hunter[ anIndex].Location := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetSwanX( aValue: TSignal);
    begin
      if aValue <> FSwanX
      then begin
        FSwanX := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetSwanY( aValue: TSignal);
    begin
      if aValue <> FSwanY
      then begin
        FSwanY := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetColor( aValue: TColor);
    begin
      if aValue <> FColor
      then begin
        FColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetBorderColor( aValue: TColor);
    begin
      if aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetSwanColor( aValue: TColor);
    begin
      if aValue <> FSwanColor
      then begin
        FSwanColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetHunterColor( aValue: TColor);
    begin
      if aValue <> FHunterColor
      then begin
        FHunterColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetSwanSize( aValue: Integer);
    begin
      if aValue <> FSwanSize
      then begin
        FSwanSize := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetHunterSize( aValue: Integer);
    begin
      if aValue <> FHunterSize
      then begin
        FHunterSize := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetForthFileName( const aValue: string);
    begin
      if aValue <> FForthFileName
      then begin
        FForthFileName := aValue;
        ReloadMazeForth;
      end;
    end;


//  private

    procedure   TKnobsMazeGraphViewer.InitMazeStuff;
    begin
      FMazeGraph             := TMazeGraph.Create( 4);
      FMazeForth             := TMazeForth.Create( FMazeGraph, 0);
      FMazeGraph.SwanColor   := SwanColor;
      FMazeGraph.SwanSize    := SwanSize;
      FMazeGraph.HunterColor := HunterColor;
      FMazeGraph.HunterSize  := HunterSize;
      FMazeGraph.OnError     := DoMazeError;
    end;


    procedure   TKnobsMazeGraphViewer.DoneMazeStuff;
    begin
      FreeAndNil( FMazeForth );
      FreeAndNil( FMazeGraph );
      FreeAndNil( FImportList);
    end;


    procedure   TKnobsMazeGraphViewer.ImportForthExports;
    begin
      if Assigned( FImportList)
      then FreeAndNil( FImportList);

      FCurrentImport := -1;
      FImportList    := FMazeForth.CreateExportList;

      if   Assigned( FImportList)
      and  ( FImportList.Count > 0)
      then begin
        FCurrentImport := 0;
        PerformForthImport;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.PerformForthImport;
    begin
      if   Assigned( FImportList)
      and  ( FCurrentImport >= 0)
      then ExecuteForthText( FImportList[ FCurrentImport]);
    end;


//  protected

    procedure   TKnobsMazeGraphViewer.DoMazeError( const aSender: TObject; aType: TMazeErrorType; const aMsg: string);
    var
      aDialogType : TMsgDlgType;
    begin
      if Assigned( FOnError)
      then FOnError( aSender, aType, aMsg)
      else begin
        if   not FShowingError
        and  not ( csDesigning in ComponentState)
        and  ( aType <> mzInfo)
        then begin
          FShowingError := True;

          try
            case aType of
              mzError       : aDialogType := mtError;
              mzWarning     : aDialogType := mtWarning;
              mzInfo        : aDialogType := mtInformation;
              mzInterpreter : aDialogType := mtError;
              else            aDialogType := mtWarning;
            end;

            MessageDlg(
              Format( 'Maze error (type %s):'^M'%s', [ MazeErrorTypeToStr( aType), aMsg], AppLocale),
              aDialogType,
              [ mbOk],
              0
            );
          finally
            FShowingError := False;
          end;
        end;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    function    TKnobsMazeGraphViewer.FindModule: TKnobsCustomModule;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsCustomModule)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsCustomModule
      then Result := TKnobsCustomModule( aParent);
    end;


    procedure   TKnobsMazeGraphViewer.WMEraseBackground( var aMsg: TWMEraseBkgnd); // message WM_ERASEBKGND;
    begin
      aMsg.Result := -1;
    end;


    procedure   TKnobsMazeGraphViewer.Paint; // override;
    var
      anOffset   : TMazePoint;
      aScale     : TSignal;
      ScaledSwan : TMazePoint;
    begin
      if Assigned( FMazeGraph)
      then begin
        if FMazeGraph.Extent.IsNull
        then begin
          aScale       := Width;
          anOffset     := MazePoint( Width / 2.0, Height / 2.0);
          ScaledSwan.X := Width  / 2.0;
          ScaledSwan.Y := Height / 2.0;
        end
        else begin
          aScale       := Min( ( Width - 6) / FMazeGraph.Extent.Width, ( Height - 6) / FMazeGraph.Extent.Height);
          anOffset     := MazePoint( - FMazeGraph.Extent.Left * aScale + 3, - FMazeGraph.Extent.Top  * aScale + 3);
          ScaledSwan.X := RangeMap( SwanX, 0, 1, FMazeGraph.Extent.Left, FMazeGraph.Extent.Right );
          ScaledSwan.Y := RangeMap( SwanY, 0, 1, FMazeGraph.Extent.Top , FMazeGraph.Extent.Bottom);

          if ScaledSwan.X < FMazeGraph.Extent.Left
          then ScaledSwan.X := 2.0 * FMazeGraph.Extent.Left - ScaledSwan.X
          else if ScaledSwan.X > FMazeGraph.Extent.Right
          then ScaledSwan.X := 2.0 * FMazeGraph.Extent.Right - ScaledSwan.X;

          if ScaledSwan.Y < FMazeGraph.Extent.Top
          then ScaledSwan.Y := 2.0 * FMazeGraph.Extent.Top - ScaledSwan.Y
          else if ScaledSwan.Y > FMazeGraph.Extent.Bottom
          then ScaledSwan.Y := 2.0 * FMazeGraph.Extent.Bottom - ScaledSwan.Y;
        end;

        FMazeGraph.SwanPosition := ScaledSwan;
        FMazeGraph.SwanColor    := SwanColor;
        FMazeGraph.HunterColor  := HunterColor;
        FMazeGraph.SwanSize     := SwanSize;
        FMazeGraph.HunterSize   := HunterSize;
        Canvas.Pen.Width        := 1;
        Canvas.Pen.Color        := BorderColor;
        Canvas.Brush.Color      := Color;
        Canvas.Rectangle( 0, 0, Width, Height);
        FMazeGraph.Paint( Canvas, aScale, anOffset);
      end;
    end;


//  public

    constructor TKnobsMazeGraphViewer.Create( anOwner: TComponent); // override;
    begin
      inherited;
      ControlStyle := ControlStyle + [ csOpaque];
      Height       := 242;
      Width        := 242;
      Color        := $009E9E9E;
      BorderColor  := clGray;
      SwanColor    := clPurple;
      HunterColor  := clBlue;
      SwanSize     := 4;
      HunterSize   := 3;
      InitMazeStuff;
    end;


    destructor  TKnobsMazeGraphViewer.Destroy; // override;
    begin
      DoneMazeStuff;
      inherited;
    end;


    procedure   TKnobsMazeGraphViewer.MoveHunter( anIndex: Integer; atRandom: Boolean);
    begin
      if   Assigned( FMazeForth)
      then begin
        FMazeGraph.MoveHunter( anIndex, atRandom);
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.MoveHunters( atRandom: Boolean);
    begin
      if   Assigned( FMazeForth)
      then begin
        FMazeGraph.MoveHunters( atRandom);
        Invalidate;
      end;
    end;


    procedure   TKnobsMazeGraphViewer.SelectMazeType( anIndex: Integer);
    begin
      if   Assigned( FImportList)
      and  ( anIndex >= 0)
      and  ( anIndex < FImportList.Count)
      then ExecuteForthText( FImportList[ anIndex]);
    end;


    function    TKnobsMazeGraphViewer.ImportList: TStringList;
    begin
      Result := FImportList;
    end;


    procedure   TKnobsMazeGraphViewer.ExecuteForthText( const S: string);
    begin
      FMazeForth.AcceptText( S);
      Invalidate;
    end;


    procedure   TKnobsMazeGraphViewer.ReloadMazeForth;
    begin
      FMazeForth.Initialize;
      FMazeForth.LoadFile( FForthFileName);
      ImportForthExports;
    end;


{ ========
  TKnobsLightningViewer = class( TKnobsGraphicControl)
  // A viewer for TLightningEngine
  private
    FEngine         : TLightningEngine;
    FShortName      : string;
    FColor          : TColor;
    FBorderColor    : TColor;
    FLightningColor : TColor;
    FEta            : TSignal;
    FShowField      : Boolean;
    FStartRandom    : Boolean;
    FRandomNew      : TSignal;
  public
    property    ShortName      : string read FShortName;
    property    AsString       : string read GetAsString     write SetAsString;
  published
    property    Color          : TColor  read FColor          write SetColor          default clBlack;
    property    BorderColor    : TColor  read FBorderColor    write SetBorderColor    default clGray;
    property    LightningColor : TColor  read FLightningColor write SetLightningColor default clYellow;
    property    Eta            : TSignal read FEta            write SetEta;
    property    ShowField      : Boolean read FShowField      write SetShowField;
    property    StartRandom    : Boolean read FStartRandom    write SetStartRandom;
    property    RandomNew      : TSignal read FRandomNew      write SetRandomNew;
  private
}

    function    TKnobsLightningViewer.GetAsString: string;
    begin
      if Assigned( FEngine)
      then Result := FEngine.AsString
      else Result := '';
    end;


    procedure   TKnobsLightningViewer.SetAsString( const aValue: string);
    begin
      if Assigned( FEngine)
      then begin
        FEngine.AsString := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsLightningViewer.GetRawPotential: TRawPotential;
    begin
      if Assigned( FEngine)
      then Result := FEngine.RawPotential
      else Result := nil;
    end;


    procedure   TKnobsLightningViewer.SetRawPotential( const aValue: TRawPotential);
    begin
      if Assigned( FEngine)
      then Fengine.RawPotential := aValue;
    end;


    function    TKnobsLightningViewer.GetRawIsShape: TRawIsShape;
    begin
      if Assigned( FEngine)
      then Result := FEngine.RawIsShape
      else Result := nil;
    end;


    procedure   TKnobsLightningViewer.SetRawIsShape( const aValue: TRawIsShape);
    begin
      if Assigned( FEngine)
      then Fengine.RawIsShape := aValue;
    end;


    procedure   TKnobsLightningViewer.SetColor( aValue: TColor);
    begin
      if aValue <> FColor
      then begin
        FColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsLightningViewer.SetBorderColor( aValue: TColor);
    begin
      if aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsLightningViewer.SetLightningColor( aValue: TColor);
    begin
      if aValue <> FLightningColor
      then begin
        FLightningColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsLightningViewer.SetEta( aValue: TSignal);
    begin
      if aValue <> FEta
      then begin
        FEta := aValue;

        if Assigned( FEngine)
        then FEngine.Eta := FEta;
      end;
    end;


    procedure   TKnobsLightningViewer.SetShowField( aValue: Boolean);
    begin
      if aValue <> FShowField
      then begin
        FShowField := aValue;

        if Assigned( FEngine)
        then FEngine.ShowField := FShowField;

        Invalidate;
      end;
    end;


    procedure   TKnobsLightningViewer.SetStartRandom( aValue: Boolean);
    begin
      if aValue <> FStartRandom
      then begin
        FStartRandom := aValue;

        if Assigned( FEngine)
        then FEngine.StartRandom := FStartRandom;
      end;
    end;


    procedure   TKnobsLightningViewer.SetRandomNew( aValue: TSignal);
    begin
      if aValue <> FRandomNew
      then begin
        FRandomNew := aValue;

        if Assigned( FEngine)
        then FEngine.RandomNew := FRandomNew;
      end;
    end;


    procedure   TKnobsLightningViewer.SetStepCount( aValue: Integer);
    begin
      if aValue < 1
      then aValue := 1;

      if aValue <> FStepCount
      then FStepCount := aValue;
    end;


    procedure   TKnobsLightningViewer.DoResize( aSender: TObject);
    begin
      if Assigned( FEngine)
      then FEngine.SetSize( Width - 2, Height - 2);
    end;


//  protected

    procedure   TKnobsLightningViewer.Loaded; // override;
    begin
      inherited;

      DoResize( Self);
    end;


    procedure   TKnobsLightningViewer.Paint; // override;
    type
      TLine = array[ 0 .. MaxInt div SizeOf( TColor) - 1] of TColor;
      PLine = ^TLine;
    var
      aBmp    : TBitmap;
      aLine   : PLine;
      Scaling : TSignal;
      i       : Integer;
      j       : Integer;
      t       : Integer;
    begin
      if Assigned( FEngine)
      then begin
        Canvas.Pen.Color := BorderColor;
        Canvas.Pen.Width := 1;
        Canvas.Brush.Style := bsClear;
        Canvas.Rectangle( 0, 0, Width, Height);

        aBmp := TBitmap.Create;

        try
          if FEngine.MaxValue = FEngine.MinValue
          then Scaling := 1
          else Scaling := 100 / ( FEngine.MaxValue - FEngine.MinValue);

          aBmp.PixelFormat := pf32bit;
          aBmp.Width       := FEngine.Width;
          aBmp.Height      := FEngine.Height;

          for j := 0 to FEngine.Height - 1
          do begin
            aLine := aBmp.ScanLine[ j];

            for i := 0 to FEngine.Width - 1
            do begin
              if FEngine.IsShape[ i, j]
              then aLine^[ i] := LightningColor
              else if FEngine.ShowField
              then begin
                t := Round( 64 * ( 1 - Cos( Scaling * FEngine.MaxValue - FEngine.Potential[ i, j]))) and $ff;
                aLine^[ i] := RGB( t, t, t) or Cardinal( Color);
              end
              else aLine^[ i] := Color;
            end;
          end;

          Canvas.Draw( 1, 1, aBmp);
        finally
          aBmp.DisposeOf;
        end;
      end;
    end;


//    procedure   TKnobsLightningViewer.Paint; // override;
//    var
//      Scaling : TSignal;
//      i       : Integer;
//      j       : Integer;
//      t       : Integer;
//      C       : TColor;
//    begin
//      if Assigned( FEngine)
//      then begin
//        Canvas.Pen.Color := BorderColor;
//        Canvas.Pen.Width := 1;
//        Canvas.Brush.Style := bsClear;
//        Canvas.Rectangle( 0, 0, Width, Height);
//
//        if FEngine.MaxValue = FEngine.MinValue
//        then Scaling := 1
//        else Scaling := 100 / ( FEngine.MaxValue - FEngine.MinValue);
//
//        for i := 0 to FEngine.Width - 1
//        do begin
//          for j := 0 to FEngine.Height - 1
//          do begin
//            if FEngine.IsShape[ i, j]
//            then C := LightningColor
//            else if FEngine.ShowField
//            then begin
//              t := Round( 64 * ( 1 - Cos( Scaling * FEngine.MaxValue - FEngine.Potential[ i, j]))) and $ff;
//              C := RGB( t, t, t) or Color;
//           // C := InterpolateColors( RGB( t, t, t), Color, 128);
//           // t := 64 * ( 1 - Cos( Scaling * FLightningEngine.MaxValue - FLightningEngine.Potential[ i, j]));
//           // C := RGB( Round( t) and $ff, Round( t) and $ff, Round( t) and $ff);
//            end
//            else C := Color;
//
//            Canvas.Pixels[ i + 1, j + 1] := C;
//          end;
//        end;
//      end;
//    end;


//  public

    constructor TKnobsLightningViewer.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FEngine        := TLightningEngine.Create( Width - 2, Height - 2, 1, 0, False); // Create with dummy values .. to be fixed later ...
      Color          := clBlack;
      BorderColor    := clGray;
      LightningColor := clYellow;
      Eta            := 9.0;
      ShowField      := True;
      StartRandom    := False;
      RandomNew      := 0.0;
      OnResize       := DoResize;
      FSteps         := 0;
      FStepCount     := 0;
    end;


    destructor  TKnobsLightningViewer.Destroy; // override;
    begin
      OnResize := nil;
      FreeAndNil( FEngine);
      inherited;
    end;


    procedure   TKnobsLightningViewer.Step;
    begin
      if Assigned( FEngine)
      then begin
        FEngine.Step;
        Inc( FSteps);

        if FSteps > StepCount
        then begin
          Invalidate;
          FSteps := 0;
        end;
      end;
    end;


{ ========
  TKnobsDataViewer = class( TKnobsGraphicControl)
  // a viewer
  private
    FShortName   : string;
    FBackGround  : TBitmap;
    FYData       : TSignalArray;
    FXYData      : TSignalPairFifo;
    FXYPoints    : Integer;
    FXOffset     : TSignal;
    FYOffset     : TSignal;
    FMaxValue    : TSignal;
    FMinValue    : TSignal;
    FXZoom       : TSignal;
    FYZoom       : TSignal;
    FBorderColor : TColor;
    FLineColor   : TColor;
    FFillColor   : TColor;
    FStyle       : TKnobsDVStyle;
  public
    property    ShortName   : string             read FShortName;
    property    YData       : TSignalArray       read FYData       write SetYData;
    property    XYData      : TSignalPairFifo    read FXYData      write SetXYData;
  published
    property    Background  : TBitmap            read FBackGround  write SetBackGround;
    property    XOffset     : TSignal            read FXOffset     write SetXOffset;    // default  0;
    property    YOffset     : TSignal            read FYOffset     write SetYOffset;    // default 75;
    property    MaxValue    : TSignal            read FMaxValue    write SetMaxValue;   // default High( SmallInt);
    property    MinValue    : TSignal            read FMinValue    write SetMinValue;   // default Low ( SmallInt);
    property    XZoom       : TSignal            read FXZoom       write SetXZoom;      // default 1;
    property    YZoom       : TSignal            read FYZoom       write SetYZoom;      // default 1;
    property    LineColor   : TColor             read FLineColor   write SetLineColor   default clWhite;
    property    FillColor   : TColor             read FFillColor   write SetFillColor   default clSilver;
    property    BorderColor : TColor             read FBorderColor write SetBorderColor default clGray;
    property    Style       : TKnobsDVStyle      read FStyle       write SetStyle       default dvsNoise;
    property    XYPoints    : Integer            read FXYPoints    write SetXYPoints    default 255;
    property    Width                                                                   default 150;
    property    Height                                                                  default 60;
    property    Color                                                                   default clGray;
  private
}

    procedure   TKnobsDataViewer.SetBackGround( aValue: TBitmap);
    begin
      FBackGround.Assign( aValue);
      Invalidate;
    end;


    procedure   TKnobsDataViewer.SetYData( const aValue: TSignalArray);
    begin
      FYData := aValue;
      Invalidate;
    end;


    procedure   TKnobsDataViewer.SetXYData( const aValue: TSignalPairFifo);
    begin
      if   Assigned( aValue)
      and  ( aValue.FillCount > 0)
      then begin
        while aValue.FillCount > 0
        do FXYData.Append( aValue.GetData);

        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetXOffset( aValue: TSignal);
    begin
      if   aValue <> FXOffset
      then begin
        FXOffset := aValue;
        Invalidate;
      end;
    end;

    procedure   TKnobsDataViewer.SetYOffset( aValue: TSignal);
    begin
      if   aValue <> FYOffset
      then begin
        FYOffset := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetMaxValue( aValue: TSignal);
    begin
      if   aValue <> FMaxValue
      then begin
        FMaxValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetMinValue( aValue: TSignal);
    begin
      if   aValue <> FMinValue
      then begin
        FMinValue := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetXZoom( aValue: TSignal);
    begin
      if   aValue <> FXZoom
      then begin
        FXZoom := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetYZoom( aValue: TSignal);
    begin
      if   aValue <> FYZoom
      then begin
        FYZoom := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetLineColor( aValue: TColor );
    begin
      if   aValue <> FLineColor
      then begin
        FLineColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetFillColor( aValue: TColor);
    begin
      if   aValue <> FFillColor
      then begin
        FFillColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetBorderColor( aValue: TColor );
    begin
      if   aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetStyle( aValue: TKnobsDVStyle);
    begin
      if   aValue <> FStyle
      then begin
        FStyle := aValue;

        case aValue of
          dvsBlank   : FillBlank;
          dvsSine    : FillSine;
          dvsTri     : FillTri;
          dvsSaw     : FillSaw;
          dvsSquare  : FillSquare;
          dvsNoise   : FillNoise;
          dvsEnvAr   : FillEnvAr;
          dvsEnvAhd  : FillEnvAhd;
          dvsEnvAdsr : FillEnvAdsr;
          dvsXYScope : FillXYScope;
          else         FillBlank;
        end;

        Invalidate;
      end;
    end;


    procedure   TKnobsDataViewer.SetXYPoints( aValue: Integer);
    begin
      if   aValue <> FXYPoints
      then begin
        FXYPoints := aValue;
      end;
    end;


//  private

    procedure   TKnobsDataViewer.FillBlank;
    begin
      SetLength( FYData, 0);
    end;


    procedure   TKnobsDataViewer.FillSine;
    var
      i : Integer;
    begin
      SetLength( FYData, Width);

      for i := 0 to Width
      do FYData[ i] := ( MaxValue - MinValue) * ( Sin( 2 * pi * i / width) + 1) / 2;
    end;


    procedure   TKnobsDataViewer.FillTri;
    var
      i         : Integer;
      PhaseAccu : TSignal;
      HalfValue : TSignal;
    begin
      SetLength( FYData, Width);
      HalfValue := ( MaxValue - MinValue) / 2;

      for i := 0 to Width
      do begin
        PhaseAccu := i / Width;

        if   PhaseAccu <= 0.5
        then FYData[ i] := HalfValue * 4 * PhaseAccu
        else FYData[ i] := HalfValue * 4 * ( 1 - PhaseAccu);
      end;
    end;


    procedure   TKnobsDataViewer.FillSaw;
    var
      i         : Integer;
      PhaseAccu : TSignal;
      HalfValue : TSignal;
    begin
      SetLength( FYData, Width);
      HalfValue := ( MaxValue - MinValue) / 2;

      for i := 0 to Width
      do begin
        PhaseAccu := i / Width;
        FYData[ i] := HalfValue * 2 * PhaseAccu;
      end;
    end;


    procedure   TKnobsDataViewer.FillSquare;
    var
      i         : Integer;
      PhaseAccu : TSignal;
    begin
      SetLength( FYData, Width);

      for i := 0 to Width
      do begin
        PhaseAccu := i / Width;

        if   PhaseAccu < 0.5
        then FYData[ i] := MinValue
        else FYData[ i] := MaxValue;
      end;
    end;


    procedure   TKnobsDataViewer.FillNoise;
    var
      i : Integer;
    begin
      SetLength( FYData, Width div 5);

      for i := 0 to Length( FYData) - 1
      do FYData[ i] := MinValue + ( MaxValue - MinValue) * Random;
    end;


    procedure   TKnobsDataViewer.FillEnvAr;
    var
      i    : Integer;
      Knee : Integer;
    begin
      SetLength( FYData, Width);
      Knee := Width div 5;

      for i := 0 to Length( FYData) - 1
      do begin
        if   i < Knee
        then FYData[ i] := LinAttack( i, Knee, MinValue, MaxValue)
        else FYData[ i] := ExpDecay( i - Knee, Width - Knee, MaxValue, MinValue);
      end;
    end;


    procedure   TKnobsDataViewer.FillEnvAhd;
    var
      i     : Integer;
      Knee1 : Integer;
      Knee2 : Integer;
    begin
      SetLength( FYData, Width);
      Knee1 := Width div 6;
      Knee2 := Width div 3;

      for i := 0 to Length( FYData) - 1
      do begin
        if   i < Knee1
        then FYData[ i] := LinAttack( i, Knee1, MinValue, MaxValue)
        else if i < Knee2
        then FYData[ i] := MaxValue
        else FYData[ i] := ExpDecay( i - Knee2, Width - Knee2, MaxValue, MinValue);
      end;
    end;


    procedure   TKnobsDataViewer.FillEnvAdsr;
    var
      i     : Integer;
      Knee1 : Integer;
      Knee2 : Integer;
      Knee3 : Integer;
      Sust  : TSignal;
    begin
      SetLength( FYData, Width);
      Knee1 := Width div 8;
      Knee2 := Width div 2;
      Knee3 := 2 * Width div 3;
      Sust  := MinValue + ( MaxValue - MinValue) / 3;

      for i := 0 to Length( FYData) - 1
      do begin
        if   i < Knee1
        then FYData[ i] := LinAttack( i, Knee1, MinValue, MaxValue)
        else if i < Knee2
        then FYData[ i] := ExpDecay( i - Knee1, Knee3 - Knee2, MaxValue, Sust)
        else if i < Knee3
        then FYData[ i] := Sust
        else FYData[ i] := ExpDecay( i - Knee3, Width - Knee3, Sust, MinValue);
      end;
    end;


    procedure   TKnobsDataViewer.FillXYScope;
    const
      AngleCount = 5;
      Angles : array[ 0 .. AngleCount - 1] of TSignal = (
        1.5, 2.5, 3.5, 4.5, 5.5
      );
    var
      Angle : Integer;
      Rad   : TSignal;
      X     : TSignal;
      Y     : TSignal;
      Typ   : Integer;
    begin
      FXYData.Clear;
      Typ := Random( AngleCount);

      for Angle := 0 to 5 * 360
      do begin
        Rad := DegToRad( Angle);
        X   := Sin( Angles[ Typ] * Rad);
        Y   := Cos( 3.01         * Rad);
        FXYData.Append( 0.95 * X, 0.95 * Y);
      end;
    end;


//  protected

    procedure   TKnobsDataViewer.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsDataViewer.WMNCHitTest( var aMessage: TWMNCHitTest);  // message WM_NCHITTEST;
    begin
      if   csDesigning in ComponentState
      then inherited
      else aMessage.Result := HTTRANSPARENT;
    end;


    procedure   TKnobsDataViewer.WMEraseBackground( var aMsg: TWMEraseBkgnd); // message WM_ERASEBKGND;
    begin
      aMsg.Result := -1;
    end;


    procedure   TKnobsDataViewer.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aModule: TKnobsCustomModule;
    begin
      if   csDesigning in ComponentState
      then inherited
      else begin
        // Pass it on
        aModule := Module;

        if   Assigned( aModule)
        then begin
          aModule.MouseCapture := True;
          aModule.MouseDown( Button, Shift, X + Left, Y + Top);
        end;
      end;
    end;


    function    TKnobsDataViewer.HasNegativeData: Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to Length( FYData) - 1
      do begin
        if   FYData[ i] < 0
        then begin
          Result := True;
          Break;
        end;
      end;
    end;


    function    TKnobsDataViewer.ScaleData( anIndex: Integer): Integer;
    var
      aVal: TSignal;
    begin
      aVal   := Clip( FYData[ anIndex] , MinValue, MaxValue);
      Result := Height - Round( Clip( aVal * YZoom * ( Height / ( MaxValue - MinValue)) + YOffset, 0, Height));
    end;


    function    TKnobsDataViewer.ScaleDataNegative( anIndex: Integer): Integer;
    var
      aVal: TSignal;
    begin
      aVal   := Clip( 1 - Abs( FYData[ anIndex]), MinValue, MaxValue);
      Result := Height - Round( Clip( aVal * YZoom * ( Height / ( MaxValue - MinValue)) + YOffset, 0, Height));
    end;


    function    TKnobsDataViewer.ScaleX( anIndex: Integer): Integer;
    begin
      Result := Round( Clip( anIndex * XZoom * ( Width / Length( FYData)) - XOffset, 0, Width));
    end;


    procedure   TKnobsDataViewer.Paint; // override;
    var
      i     : Integer;
      X     : Integer;
      Y     : Integer;
      LastX : Integer;
      LastY : Integer;
      aData : TSignalPair;
    begin
      inherited;

      with Canvas
      do begin
        if   Assigned( FBackGround)
        and  not FBackGround.Empty
        then StretchDraw( Rect( 0, 0, Height, Width), FBackGround)
        else begin
          Pen.Width   := 1;
          Pen.Color   := BorderColor;
          Brush.Color := Color;
          Rectangle( 0, 0, Width, Height);
        end;

        if   ( Length( FYData) > 0)
        or  ( FXYData.Size > 0)
        then begin
          if   ( Style in [ dvsEnvAr, dvsEnvAhd, dvsEnvAdsr])
          and  HasNegativeData
          then begin
            Pen.Width := 1;
            Pen.Color := FillColor;

            for i := 1 to Length( FYData) - 1
            do begin
              MoveTo( ScaleX( i), 0);
              LineTo( ScaleX( i), ScaleDataNegative( i));
            end;

            Pen.Color := LineColor;
            Pen.Width := 1;
            MoveTo( ScaleX( 0), ScaleDataNegative( 0));

            for i := 1 to Length( FYData) - 1
            do LineTo( ScaleX( i), ScaleDataNegative( i));
          end
          else if ( Style = dvsXYScope)
          then begin
            ClipRect.Inflate( -2, -2);
            Pen.Color := LineColor;
            Pen.Width := 1;
            LastX := 0;
            LastY := 0;

            for i := 0 to FXYData.Size - 1
            do begin
              aData := FXYData.GetData;
              X := Round( ( 0.5 * ( Width  - 5) * ( 1 + aData.Left ))) + 2;
              Y := Round( ( 0.5 * ( Height - 5) * ( 1 - aData.Right))) + 2;

              if   i = 0
              then MoveTo( X, Y)
              else begin
                if   ( X <> LastX)
                or   ( Y <> LastY)
                then LineTo( X, Y);
              end;

              LastX := X;
              LastY := Y;
            end;

            if   csDesigning in ComponentState
            then FillXYScope;
          end
          else begin
            Pen.Width := 1;
            Pen.Color := FillColor;

            for i := 1 to Length( FYData) - 1
            do begin
              MoveTo( ScaleX( i), Height - 1);
              LineTo( ScaleX( i), ScaleData( i));
            end;

            Pen.Color := LineColor;
            Pen.Width := 1;
            MoveTo( ScaleX( 0), ScaleData( 0));

            for i := 1 to Length( FYData) - 1
            do LineTo( ScaleX( i), ScaleData( i));
          end;
        end;
      end;
    end;


//  public

    constructor TKnobsDataViewer.Create( anOwner: TComponent); // override;
    begin
      inherited;

      FXYData := TSignalPairFifo.Create( System_Rate);
      FXYData.CountOverflows := False;
      FXYData.AllowOverflows := True;

      ControlStyle := ControlStyle + [ csOpaque];
      FBackGround  := TBitmap.Create;
      XOffset      := 0;
      YOffset      := 30;
      MaxValue     := High( SmallInt);
      MinValue     := Low ( SmallInt);
      XZoom        := 1;
      YZoom        := 1;
      Width        := 150;
      Height       := 60;
      LineColor    := clWhite;
      FillColor    := $009E9E9E;
      BorderColor  := clGray;
      Color        := clGray;
      Style        := dvsNoise;
    end;


    destructor  TKnobsDataViewer.Destroy; // override;
    begin
      FreeAndNil( FXYData);
      FreeAndNil( FBackGround);
      inherited;
    end;


{ ========
  TKnobsDataMaker = class( TKnobsGraphicControl)
  // a graphical data generator
  private
    FShortName              : string;
    FPrevData               : TSignalPairarray;
    FData                   : TSignalPairArray;
    FDataType               : TKnobsDataMakerType;
    FXZoom                  : TSignal;
    FYZoom                  : TSignal;
    FXPan                   : TSignal;
    FYPan                   : TSignal;
    FAutoScale              : Boolean;
    FAutoLoop               : Boolean;
    FGluedLeft              : Boolean;
    FGluedRight             : Boolean;
    FAutoHGrid              : Boolean;
    FAutoVGrid              : Boolean;
    FBorderColor            : TColor;
    FLineColor              : TColor;
    FGridColor              : TColor;
    FCursorColor            : TColor;
    FDotColor               : TColor;
    FActiveDotColor         : TColor;
    FTransparent            : Boolean;
    FShowGrid               : Boolean;
    FShowXCursor            : Boolean;
    FShowYCursor            : Boolean;
    FGraphMode              : TKnobsDMGraphMode;
    FDotSize                : Integer;
    FCursorX                : TSignal;
    FCursorY                : TSignal;
    FLastX                  : TSignal;
    FLastY                  : TSignal;
    FAllowXMoves            : Boolean;
    FAllowYMoves            : Boolean;
    FAllowPointDeletion     : Boolean;
    FAllowPointInsertion    : Boolean;
    FMouseDown              : Boolean;
    FCursorXWasShown        : Boolean;
    FCursorYWasShown        : Boolean;
    FFollowCursor           : Boolean;
    FSelectedPoint          : Integer;
    FControlType            : string;
    FAllowRandomization     : Boolean;
    FShowAllowRandomization : Boolean;
    FMultiVariationValues   : TKnobsMultiVariationValues;
    FCurrentValues          : TSignalArray;
    FLiveMorph              : TSignal;
    FActiveVariation        : Integer;
    FOnChanged              : TKnobsOnTextChanged;
    FOnPointChanged         : TKnobsOnPointChanged;
    FOnPointAdded           : TKnobsOnPointChanged;
    FOnPointRemoved         : TKnobsOnPointChanged;
    FOnLoadDataGraph        : TOnLoadDataGraph;
    FOnSaveDataGraph        : TOnSaveDataGraph;
  public
    property    ShortName                         : string               read FShortName;
    property    Data                              : TSignalPairArray     read FData                     write SetData;
    property    DataCount                         : Integer              read GetDataCount;
    property    Module                            : TKnobsCustomModule   read GetModule;
    property    CursorX                           : TSignal              read FCursorX                  write SetCursorX;
    property    CursorY                           : TSignal              read FCursorY                  write SetCursorY;
    property    CursorXY                          : TSignalPair          read GetCursorXY               write SetCursorXY;
    property    AsString                          : string               read GetAsString               write SetAsString;
    property    FollowCursor                      : Boolean              read FFollowCursor;
    property    VariationCount                    : Integer              read GetVariationCount;
    property    AllowVariations                   : Boolean              read GetAllowVariations;
    property    ActiveVariation                   : Integer              read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string               read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal              read GetVariationValue         write SetVariationValue;
    property    CurrentValue  [ anIndex: Integer] : TSignal              read GetCurrentValue           write SetCurrentValue;
  published
    property    ShortName                         : string               read FShortName;
    property    Data                              : TSignalPairArray     read FData                     write SetData;
    property    DataCount                         : Integer              read GetDataCount;
    property    CursorX                           : TSignal              read FCursorX                  write SetCursorX;
    property    CursorY                           : TSignal              read FCursorY                  write SetCursorY;
    property    CursorXY                          : TSignalPair          read GetCursorXY               write SetCursorXY;
    property    AsString                          : string               read GetAsString               write SetAsString;
    property    FollowCursor                      : Boolean              read FFollowCursor;
    property    VariationCount                    : Integer              read GetVariationCount;
    property    AllowVariations                   : Boolean              read GetAllowVariations;
    property    ActiveVariation                   : Integer              read GetActiveVariation        write SetActiveVariation;
    property    VariationsAsString                : string               read GetVariationsAsString     write SetVariationsAsString;
    property    VariationValue[ anIndex: Integer] : TSignal              read GetVariationValue         write SetVariationValue;
    property    CurrentValue  [ anIndex: Integer] : TSignal              read GetCurrentValue           write SetCurrentValue;
  published
    property    DataType                          : TKnobsDataMakerType  read FDataType                 write SetDataType               default dmtBipolar;
    property    AllowXMoves                       : Boolean              read FAllowXMoves              write FAllowXMoves              default True;
    property    AllowYMoves                       : Boolean              read FAllowYMoves              write FAllowYMoves              default True;
    property    AllowPointDeletion                : Boolean              read FAllowPointDeletion       write FAllowPointDeletion       default True;
    property    AllowPointInsertion               : Boolean              read FAllowPointInsertion      write FAllowPointInsertion      default True;
    property    AutoScale                         : Boolean              read FAutoScale                write SetAutoScale              default False;
    property    AutoLoop                          : Boolean              read FAutoLoop                 write SetAutoLoop               default False;
    property    GluedLeft                         : Boolean              read FGluedLeft                write SetGluedLeft              default True;
    property    GluedRight                        : Boolean              read FGluedRight               write SetGluedRight             default True;
    property    AutoHGrid                         : Boolean              read FAutoHGrid                write SetAutoHGrid              default False;
    property    AutoVGrid                         : Boolean              read FAutoVGrid                write SetAutoVGrid              default False;
    property    XZoom                             : TSignal              read FXZoom                    write SetXZoom;              // default 1;
    property    YZoom                             : TSignal              read FYZoom                    write SetYZoom;              // default 1;
    property    XPan                              : TSignal              read FXPan                     write SetXPan;               // default 0;
    property    YPan                              : TSignal              read FYPan                     write SetYPan;               // default 0;
    property    Color                                                                                                                   default clGray;
    property    LineColor                         : TColor               read FLineColor                write SetLineColor              default clWhite;
    property    BorderColor                       : TColor               read FBorderColor              write SetBorderColor            default clGray;
    property    GridColor                         : TColor               read FGridColor                write SetGridColor              default clSilver;
    property    CursorColor                       : TColor               read FCursorColor              write SetCursorColor            default CL_CURSOR;
    property    DotColor                          : TColor               read FDotColor                 write SetDotColor               default CL_DOT;
    property    ActiveDotColor                    : TColor               read FActiveDotColor           write SetActiveDotColor         default CL_ACTIVEDOT;
    property    Transparent                       : Boolean              read FTransparent              write SetTransparent            default True;
    property    ShowGrid                          : Boolean              read FShowGrid                 write SetShowGrid               default True;
    property    ShowXCursor                       : Boolean              read FShowXCursor              write SetShowXCursor            default True;
    property    ShowYCursor                       : Boolean              read FShowYCursor              write SetShowYCursor            default True;
    property    GraphMode                         : TKnobsDMGraphMode    read FGraphMode                write SetGraphMode              default dmgLinear;
    property    DotSize                           : Integer              read FDotSize                  write SetDotSize                default   3;
    property    Width                                                                                                                   default 150;
    property    Height                                                                                                                  default  60;
    property    Align;
    property    ControlType                       : string               read GetControlType            write SetControlType;
    property    AllowRandomization                : Boolean              read GetAllowRandomization     write SetAllowRandomization     default False;
    property    ShowAllowRandomization            : Boolean              read GetShowAllowRandomization write SetShowAllowRandomization default False;
    property    OnChanged                         : TKnobsOnTextChanged  read FOnChanged                write FOnChanged;
    property    OnPointChanged                    : TKnobsOnPointChanged read FOnPointChanged           write FOnPointChanged;
    property    OnPointAdded                      : TKnobsOnPointChanged read FOnPointAdded             write FOnPointAdded;
    property    OnPointRemoved                    : TKnobsOnPointChanged read FOnPointRemoved           write FOnPointRemoved;
  private
}

    procedure   TKnobsDataMaker.SetDataType( aValue: TKnobsDataMakerType);
    begin
      if   FDataType <> aValue
      then begin
        FDataType := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetAutoScale( aValue: Boolean);
    begin
      if   aValue <> FAutoScale
      then begin
        FAutoScale := aValue;

        if   AutoScale
        then ValueChanged;
      end;
    end;


    procedure   TKnobsDataMaker.SetAutoLoop( aValue: Boolean);
    begin
      if   aValue <> FAutoLoop
      then begin
        FAutoLoop := aValue;

        if   AutoLoop
        then ValueChanged;
      end;
    end;


    procedure   TKnobsDataMaker.SetGluedLeft( aValue: Boolean);
    begin
      if   aValue <> FGluedLeft
      then begin
        FGluedLeft := aValue;
      end;
    end;


    procedure   TKnobsDataMaker.SetGluedRight( aValue: Boolean);
    begin
      if   aValue <> FGluedRight
      then begin
        FGluedRight := aValue;
      end;
    end;


    procedure   TKnobsDataMaker.SetAutoHGrid( aValue: Boolean);
    begin
      if   aValue <> FAutoHGrid
      then begin
        FAutoHGrid := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetAutoVGrid( aValue: Boolean);
    begin
      if   aValue <> FAutoVGrid
      then begin
        FAutoVGrid := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsDataMaker.GetControlType: string;
    begin
      Result := FControlType;
    end;


    procedure   TKnobsDataMaker.SetControlType( const aValue: string);
    begin
      if   aValue <> FControlType
      then begin
        FControlType := aValue;

        Invalidate;
      end;
    end;


    function    TKnobsDataMaker.GetAllowRandomization: Boolean;
    begin
      Result := FAllowRandomization;
    end;


    procedure   TKnobsDataMaker.SetAllowRandomization( aValue: Boolean);
    begin
      if   aValue <> FAllowRandomization
      then begin
        FAllowRandomization := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsDataMaker.GetShowAllowRandomization: Boolean;
    begin
      Result := FShowAllowRandomization;
    end;


    procedure   TKnobsDataMaker.SetShowAllowRandomization( aValue: Boolean);
    begin
      if   aValue <> FShowAllowRandomization
      then begin
        FShowAllowRandomization := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.FixLiveMorph;
    var
      anIndex   : Integer;
      aFraction : TSignal;
      i         : Integer;
      anOffset0 : Integer;
      anOffset1 : Integer;
      anIndex0  : Integer;
      anIndex1  : Integer;
    begin
      anIndex   := Trunc( FLiveMorph);
      aFraction := FLiveMorph - anIndex;
      anOffset0 := DataCount * anIndex;
      anOffset1 := anOffset0 + DataCount;

      for i := 0 to DataCount - 1
      do begin
        anIndex0 := anOffset0 + i;
        anIndex1 := anOffset1 + i;
        CurrentValue[ i] := VariationValue[ anIndex0] + aFraction * ( VariationValue[ anIndex1] - VariationValue[ anIndex0]);
      end;
    end;


    procedure   TKnobsDataMaker.SetRandomValue( anAmount: TSignal; aVariation: Integer); // overload;
    // Set random values for one variation
    var
      i        : Integer;
      anOffset : Integer;
      anIndex  : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anOffset := DataCount * aVariation;

        for i := 0 to DataCount - 1
        do begin
          anIndex := anOffset + i;
          VariationValue[ anIndex] := VariationValue[ anIndex] + anAmount * ( Random - VariationValue[ anIndex]);
        end;
      end;
    end;


    procedure   TKnobsDataMaker.SetRandomValue( anAmount: TSignal); // overload;
    begin
      SetRandomValue( anAmount, ActiveVariation);
    end;


    procedure   TKnobsDataMaker.Randomize( anAmount: TSignal; aVariation: Integer); // overload;
    // Set random values for one variation
    var
      i        : Integer;
      anOffset : Integer;
      anIndex  : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anOffset := DataCount * aVariation;

        for i := 0 to DataCount - 1
        do begin
          anIndex := anOffset + i;
          VariationValue[ anIndex] := VariationValue[ anIndex] + anAmount * ( Random - VariationValue[ anIndex]);
        end;

        FixLiveMorph;
      end;
    end;


    procedure   TKnobsDataMaker.Randomize( anAmount: TSignal); // overload;
    begin
      Randomize( anAmount, ActiveVariation);
    end;


    procedure   TKnobsDataMaker.Mutate( aProb, aRange: TSignal; aVariation: Integer); // overload;
    // Mutate one variation, a slightly different form of randomization
    var
      i        : Integer;
      anOffset : Integer;
      anIndex  : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anOffset := DataCount * aVariation;

        for i := 0 to DataCount - 1
        do begin
          if   Random < aProb
          then begin
            anIndex := anOffset + i;
            VariationValue[ anIndex] := VariationValue[ anIndex] + aRange * ( Random - VariationValue[ anIndex]);
          end;
        end;

        FixLiveMorph;
      end;
    end;


    procedure   TKnobsDataMaker.Mutate( aProb, aRange: TSignal); // overload;
    begin
      Mutate( aProb, aRange, ActiveVariation);
    end;


    procedure   TKnobsDataMaker.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); // overload;
    // Mate mom and a dad into one variation, mom and dad are variations as well, and it is possible
    // to mate the result into a mom or dad variation, or to mate a variation with itself, or otherwise ...
    var
      i           : Integer;
      anOffset    : Integer;
      anOffsetMom : Integer;
      anOffsetDad : Integer;
      anIndex     : Integer;
      anIndexMom  : Integer;
      anIndexDad  : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anOffset    := DataCount * aVariation;
        anOffsetMom := DataCount * aMom;
        anOffsetDad := DataCount * aDad;

        for i := 0 to DataCount - 1
        do begin
          anIndex    := anOffset + i;
          anIndexMom := anOffsetMom + i;
          anIndexDad := anOffsetDad + i;

          if   Random < anXProb
          then VariationValue[ anIndex] := VariationValue[ anIndexMom]
          else VariationValue[ anIndex] := VariationValue[ anIndexDad];
        end;

        FixLiveMorph;
        Mutate( aMutProb, aMutRange, aVariation);
      end;
    end;


    procedure   TKnobsDataMaker.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer); // overload;
    begin
      MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsDataMaker.Morph( anAmount: TSignal; aDad, aMom, aVariation: Integer); // overload;
    // Morph between a mom and a dad into a certain variation, again all combinations are possible
    // just as with the mate procedure.
    var
      i           : Integer;
      anOffset    : Integer;
      anOffsetMom : Integer;
      anOffsetDad : Integer;
      anIndex     : Integer;
      anIndexMom  : Integer;
      anIndexDad  : Integer;
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anOffset    := DataCount * aVariation;
        anOffsetMom := DataCount * aMom;
        anOffsetDad := DataCount * aDad;

        for i := 0 to DataCount - 1
        do begin
          anIndex    := anOffset    + i;
          anIndexMom := anOffsetMom + i;
          anIndexDad := anOffsetDad + i;
          VariationValue[ anIndex] := VariationValue[ anIndexDad] + anAmount * ( VariationValue[ anIndexMom] - VariationValue[ anIndexDad]);
        end;

        FixLiveMorph;
      end;
    end;


    procedure   TKnobsDataMaker.Morph( anAmount: TSignal; aDad, aMom: Integer); // overload;
    begin
      Morph( anAmount, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsDataMaker.LiveMorph( anAmount: TSignal);
    // Make a morph between two adjacent variations, e.g. v and v + 1, say
    begin
      if   CanAllowRandomization
      and  CanRandomize
      then begin
        anAmount  := Clip( anAmount, 0.0, 0.999999) * ( MAX_VARIATIONS - 1);

        if   anAmount <> FLiveMorph
        then begin
          FLiveMorph := anAmount;
          FixLiveMorph;
        end;
      end;
    end;


    function    TKnobsDataMaker.CanAllowRandomization: Boolean;
    begin
      Result := True;
    end;


    function    TKnobsDataMaker.CanRandomize: Boolean;
    begin
      Result := AllowRandomization;
    end;


    function    TKnobsDataMaker.GetVariationCount: Integer;
    begin
      Result := DataCount * MAX_VARIATIONS;
    end;


    function    TKnobsDataMaker.GetAllowVariations: Boolean;
    begin
      Result := AllowRandomization;
    end;


    function    TKnobsDataMaker.GetActiveVariation: Integer;
    begin
      Result := FActiveVariation;
    end;


    procedure   TKnobsDataMaker.SetActiveVariation( aValue: Integer);
    var
      i : Integer;
    begin
      aValue := Clip( aValue, 0, VariationCount - 1);

      if   aValue <> FActiveVariation
      then begin
        FActiveVariation := aValue;

        if   VariationCount > 0
        then begin
          if   AllowVariations
          then begin
            for i := 0 to DataCount - 1
            do CurrentValue[ i] := FMultiVariationValues[ i, FActiveVariation];
          end
          else begin
            for i := 0 to DataCount - 1
            do CurrentValue[ i] := FMultiVariationValues[ i, 0];
          end;
        end;
      end;
    end;


    function    TKnobsDataMaker.GetVariationsAsString: string;
    var
      i : Integer;
    begin
      Result := Format( '%d', [ VariationCount], AppLocale);

      for i := 0 to VariationCount - 1
      do Result := Result + Format( ',%g', [ VariationValue[ i]], AppLocale);
    end;


    procedure   TKnobsDataMaker.SetVariationsAsString( const aValue: string);
    var
      aParts : TStringList;
      aCount : Integer;
      i      : Integer;
    begin
      aParts := Explode( aValue, ',');

      try
        if   aParts.Count > 0
        then begin
          aCount := StrToIntDef( aParts[ 0], -1);

          if   aCount = VariationCount
          then begin
            for i := 0 to VariationCount - 1
            do begin
              if   AllowVariations
              then VariationValue[ i] := StrToFloatDef( aParts[ i               + 1], 0)
              else VariationValue[ i] := StrToFloatDef( aParts[ i mod DataCount + 1], 0);
            end;
          end
          else SyncVariations;
        end;
      finally
        aParts.DisposeOf;
      end;
    end;


    function    TKnobsDataMaker.GetVariationValue( anIndex: Integer): TSignal;
    var
      aVarIndex   : Integer;
      aParamIndex : Integer;
    begin
      anIndex := Clip( anIndex, 0, VariationCount - 1);

      if   VariationCount > 0
      then begin
        aVarIndex   := anIndex div DataCount;
        aParamIndex := anIndex mod DataCount;

        if   AllowVariations
        then Result := FMultiVariationValues[ aParamIndex, aVarIndex]
        else Result := FMultiVariationValues[ aParamIndex, 0        ];
      end
      else Result := 0;
    end;


    procedure   TKnobsDataMaker.SetVariationValue( anIndex: Integer; aValue: TSignal);
    var
      aVarIndex   : Integer;
      aParamIndex : Integer;
      i           : Integer;
    begin
      anIndex     := Clip( anIndex, 0, VariationCount - 1);
      aVarIndex   := anIndex div DataCount;
      aParamIndex := anIndex mod DataCount;

      if   VariationCount > 0
      then begin
        if   AllowVariations
        then FMultiVariationValues[ aParamIndex, aVarIndex] := aValue
        else begin
          for i := 0 to MAX_VARIATIONS - 1
          do FMultiVariationValues[ aParamIndex, i] := aValue;
        end;
      end;

      if   ( aVarIndex = ActiveVariation) or not AllowVariations or ( VariationCount = 0)
      then CurrentValue[ aParamIndex] := aValue;
    end;


    function    TKnobsDataMaker.GetCurrentValue( anIndex: Integer): TSignal;
    begin
      Result := FCurrentValues[ anIndex];
    end;


    procedure   TKnobsDataMaker.SetCurrentValue( anIndex: Integer; aValue: TSignal);
    begin
      if   aValue <> FCurrentValues[ anIndex]
      then begin
        FCurrentValues[ anIndex] := aValue;
        Data[ anIndex].Y := VariationValueToKnobValue( aValue);
        ValueChanged;
      end;
    end;


    function    TKnobsDataMaker.KnobValueToVariationValue( aValue: TSignal): TSignal;
    begin
      Result := RangeMap( aValue, -1.0, 1.0, 0.0, 1.0);
    end;


    function    TKnobsDataMaker.VariationValueToKnobValue( aValue: TSignal): TSignal;
    begin
      Result := RangeMap( aValue, 0.0, 1.0, -1.0, 1.0);
    end;


    procedure   TKnobsDataMaker.SetDefaultVariations;
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to DataCount - 1
      do begin
        FCurrentValues[ i] := KnobValueToVariationValue( FData[ i].Y);

        for j := 0 to MAX_VARIATIONS - 1
        do FMultiVariationValues[ i, j] := FCurrentValues[ i];
      end;
    end;


    procedure   TKnobsDataMaker.CountGene( var aCount: Integer);
    begin
      aCount := aCount + DataCount;
    end;


    procedure   TKnobsDataMaker.FillGene( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    var
      i : Integer;
    begin
      if   Assigned( aGene)
      then begin
        if   anIndex < aGene.Size - DataCount
        then begin
          if   ( aVariation >= 0) and ( aVariation < MAX_VARIATIONS)
          then begin
            for i := 0 to DataCount - 1
            do begin
              aGene[ anIndex] := FMultiVariationValues[ i, aVariation];
              anIndex := anIndex + 1;
            end;
          end
          else begin
            for i := 0 to DataCount - 1
            do begin
              aGene[ anIndex] := FCurrentValues[ i];
              anIndex := anIndex + 1;
            end;
          end;
        end;
      end;
    end;


    procedure   TKnobsDataMaker.AcceptGene( const aGene: TKnobsGene; aVariation: Integer; DoAllowRandomization: Boolean; var anIndex: Integer);
    var
      i          : Integer;
      WasChanged : Boolean;
      OldData    : TSignal;
    begin
      if   Assigned( aGene)
      then begin
        WasChanged := False;

        if   anIndex < aGene.Size - DataCount
        then begin
          for i := 0 to DataCount - 1
          do begin
            if   AllowRandomization
            and  DoAllowRandomization
            then begin
              if   ( aVariation >= 0) and ( aVariation < MAX_VARIATIONS)
              then begin
                OldData                               := FData[ i].Y;
                FMultiVariationValues[ i, aVariation] := aGene[ anIndex];
                FData[ i].Y                           := VariationValueToKnobValue( aGene[ anIndex]);

                if   not WasChanged
                and  ( FData[ i].Y <> OldData)
                then WasChanged := True;
              end
              else begin
                OldData            := FData[ i].Y;
                FCurrentValues[ i] := aGene[ anIndex];
                FData[ i].Y        := VariationValueToKnobValue( FCurrentValues[ i]);

                if   not WasChanged
                and  ( FData[ i].Y <> OldData)
                then WasChanged := True;
              end;
            end;

            anIndex := anIndex + 1;
          end;

          if   WasChanged
          then ValueChanged;
        end;
      end;
    end;


//  private

    function    TKnobsDataMaker.GetDataCount: Integer;
    begin
      Result := Length( FData);
    end;


    procedure   TKnobsDataMaker.ClearData;
    var
      Changed : Boolean;
      X       : TSignal;
      Y       : TSignal;
    begin
      Changed := False;

      if   DataCount = 0
      then begin
        AddData( 0.0, 0.0);
        Changed := True;
      end;

      if   DataCount = 1
      then begin
        FData[ 0].X := 0.0;
        AddData( 1.0, 0.0);
        Changed := True;
      end;

      if   DataCount = 2
      then begin
        FData[ 1].X := 1.0;
        Changed := True;
      end;

      while DataCount > 2
      do begin
        X := FData[ DataCount - 2].X;
        Y := FData[ DataCount - 2].Y;
        DeleteData( DataCount - 2, False);
        PointRemoved( DataCount - 2, X, Y);
        Changed := True;
      end;

      if   Changed
      then begin
        ValueChanged;
        SyncVariations;
      end;
    end;


    procedure   TKnobsDataMaker.ClearAllData;
    // All data being cleared should not be painted ...
    begin
      SetLength( FData                , 0);
      SetLength( FMultiVariationValues, 0);
      SetLength( FCurrentValues       , 0);
    end;


    procedure   TKnobsDataMaker.AddData( const aData: TSignalPair); // overload;
    var
      i : Integer;
      j : Integer;
      p : Integer;
    begin
      p := DataCount;

      for i := 0 to DataCount - 2
      do begin
        if   ( aData.X > FData[ i    ].X)
        and  ( aData.X < FData[ i + 1].X)
        then begin
          p := i + 1;
          Break;
        end;
      end;

      SetLength( FData                , DataCount + 1);
      SetLength( FMultiVariationValues, DataCount    );
      SetLength( FCurrentValues       , DataCount    );

      for i := DataCount - 2 downto p
      do begin
        FData         [ i + 1] := FData         [ i];
        FCurrentValues[ i + 1] := FCurrentValues[ i];

        for j := 0 to MAX_VARIATIONS - 1
        do FMultiVariationValues[ i + 1, j] := FMultiVariationValues[ i, j];
      end;

      FData         [ p].X := aData.X;
      FData         [ p].Y := aData.Y;
      FCurrentValues[ p]   := KnobValueToVariationValue( aData.Y);

      for j := 0 to MAX_VARIATIONS - 1
      do FMultiVariationValues[ p, j] := KnobValueToVariationValue( FData[ p].Y);

      PointAdded( p, aData.X, aData.Y);
      ValueChanged;
    end;


    procedure   TKnobsDataMaker.AddData( anX, anY: TSignal); // overload;
    var
      aPair : TSignalPair;
    begin
      aPair.X := anX;
      aPair.Y := anY;
      AddData( aPair);
    end;


    procedure   TKnobsDataMaker.DeleteData( anIndex: Integer; FlagInvalidation: Boolean);
    var
      i : Integer;
      j : Integer;
      X : TSignal;
      Y : TSignal;
    begin
      if   ( anIndex > 0)
      and  ( anIndex < DataCount - 1)
      then begin
        X := FData[ anIndex].X;
        Y := FData[ anIndex].Y;

        for i := anIndex to DataCount - 2
        do begin
          FData[ i] := FData[ i + 1];

          for j := 0 to MAX_VARIATIONS - 1
          do FMultiVariationValues[ i, j] := FMultiVariationValues[ i + 1, j];
        end;

        SetLength( FData                , DataCount - 1);
        SetLength( FMultiVariationValues, DataCount    );
        SetLength( FCurrentValues       , DataCount    );
        PointRemoved( anIndex, X, Y);;

        if   FlagInvalidation
        then ValueChanged;
      end;
    end;


    procedure   TKnobsDataMaker.PointChanged( anIndex: Integer; anX, anY: TSignal);
//    var
//      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnPointChanged)
      then FOnPointChanged( Self, Name, anIndex, anX, anY)
//      else begin
//        aModule := Module;
//
//        if   Assigned( aModule) and DoNotify
//        then Module.PointChanged( Self, MakePath( [ aModule.Name, Name]), anX, anY);
//      end;
    end;


    procedure   TKnobsDataMaker.PointAdded( anIndex: Integer; anX, anY: TSignal);
    begin
      if   Assigned( FOnPointAdded)
      then FOnPointAdded( Self, Name, anIndex, anX, anY)
    end;


    procedure   TKnobsDataMaker.PointRemoved( anIndex: Integer; anX, anY: TSignal);
    begin
      if   Assigned( FOnPointRemoved)
      then FOnPointRemoved( Self, Name, anIndex, anX, anY)
    end;


    procedure   TKnobsDataMaker.ChangeData( anIndex: Integer; anX, anY: TSignal);
    var
      anOldX : TSignal;
      anOldY : TSignal;
    begin
      if   (( anIndex > 0            ) or not GluedLeft )
      and  (( anIndex < DataCount - 1) or not GluedRight)
      then begin
        anOldX := FData[ anIndex].X;
        anOldY := FData[ anIndex].Y;

        if   AllowXMoves
        then FData[ anIndex].X  := Clip( anX,  FData[ anIndex - 1].X, FData[ anIndex + 1].X);

        if   AllowYMoves
        then FData[ anIndex].Y := Clip( anY, -1.0, 1.0);

        FMultiVariationValues[ anIndex, FActiveVariation] := KnobValueToVariationValue( FData[ anIndex].Y);

        if  ( anOldX <> FData[ anIndex].X)
        or  ( anOldY <> FData[ anIndex].Y)
        then begin
          PointChanged( anIndex, anX, anY);
          ValueChanged;
        end;
      end
      else if anIndex = 0
      then begin
        anOldX := FData[ anIndex].X;
        anOldY := FData[ anIndex].Y;

        FData[ anIndex].X  := 0.0;
        FData[ anIndex].Y := Clip( anY, -1.0, 1.0);
        FMultiVariationValues[ anIndex, FActiveVariation] := KnobValueToVariationValue( FData[ anIndex].Y);

        if   ( anOldX <> FData[ anIndex].X)
        or   ( anOldY <> FData[ anIndex].Y)
        then begin
          PointChanged( anIndex, anX, anY);
          ValueChanged;
        end;
      end
      else if anIndex = DataCount - 1
      then begin
        anOldX := FData[ anIndex].X;
        anOldY := FData[ anIndex].Y;

        FData[ anIndex].X  := 1.0;
        FData[ anIndex].Y := Clip( anY, -1.0, 1.0);
        FMultiVariationValues[ anIndex, FActiveVariation] := KnobValueToVariationValue( FData[ anIndex].Y);

        if   ( anOldX <> FData[ anIndex].X)
        or   ( anOldY <> FData[ anIndex].Y)
        then begin
          PointChanged( anIndex, anX, anY);
          ValueChanged;
        end;
      end;
    end;


    procedure   TKnobsDataMaker.SetPrevData;
    var
      i : Integer;
    begin
      SetLength( FPrevData, Length( FData));

      for i := 0 to Length( FData) - 1
      do FPrevData[ i] := FData[ i];
    end;


    procedure   TKnobsDataMaker.SetData( const aValue: TSignalPairArray);
    var
      i : Integer;
    begin
      ClearData;

      if   Length( aValue) > 1
      then begin
        ClearAllData;

        for i := 0 to Length( aValue) - 1
        do AddData( aValue[ i]);
      end;

      ValueChanged;
      SyncVariations;
    end;


    procedure   TKnobsDataMaker.SetXZoom( aValue: TSignal);
    begin
      if   aValue <> FXZoom
      then begin
        FXZoom := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetYZoom( aValue: TSignal);
    begin
      if   aValue <> FYZoom
      then begin
        FYZoom := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetXPan( aValue: TSignal);
    begin
      if   aValue <> FXPan
      then begin
        FXPan := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetYPan( aValue: TSignal);
    begin
      if   aValue <> FYPan
      then begin
        FYPan := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetLineColor( aValue: TColor);
    begin
      if   aValue <> FLineColor
      then begin
        FLineColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetBorderColor( aValue: TColor);
    begin
      if   aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetGridColor( aValue: TColor);
    begin
      if   aValue <> FGridColor
      then begin
        FGridColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetCursorColor( aValue: TColor);
    begin
      if   aValue <> FCursorColor
      then begin
        FCursorColor := aValue;
        Invalidate;
      end;
    end;

    procedure   TKnobsDataMaker.SetDotColor( aValue: TColor);
    begin
      if   aValue <> FDotColor
      then begin
        FDotColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetActiveDotColor( aValue: TColor);
    begin
      if   aValue <> FActiveDotColor
      then begin
        FActiveDotColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetTransparent( aValue: Boolean);
    begin
      if   aValue <> FTransparent
      then begin
        FTransparent := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetShowGrid( aValue: Boolean);
    begin
      if   aValue <> FShowGrid
      then begin
        FShowGrid := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetShowXCursor( aValue: Boolean);
    begin
      if   aValue <> FShowXCursor
      then begin
        FShowXCursor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetShowYCursor( aValue: Boolean);
    begin
      if   aValue <> FShowYCursor
      then begin
        FShowYCursor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetGraphMode( aValue: TKnobsDMGraphMode);
    begin
      if   aValue <> FGraphMode
      then begin
        FGraphMode := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetDotSize( aValue: Integer);
    begin
      if   aValue <> FDotSize
      then begin
        FDotSize := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetCursorX( aValue: TSignal);
    begin
      if   aValue <> FCursorX
      then begin
        FCursorX := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SetCursorY( aValue: TSignal);
    begin
      if   aValue <> FCursorY
      then begin
        FCursorY := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsDataMaker.GetCursorXY: TSignalPair;
    begin
      Result.X := CursorX;
      Result.Y := CursorY;
    end;


    procedure   TKnobsDataMaker.SetCursorXY( const aValue: TSignalPair);
    begin
      if   ( CursorX <> aValue.X)
      or   ( CursorY <> aValue.Y)
      then begin
        FCursorX := aValue.X;
        FCursorY := aValue.Y;
        Invalidate;
      end;
    end;


    function    TKnobsDataMaker.GetAsString: string;
    begin
      Result := SignalPairArrayToStr( FData);
    end;


    procedure   TKnobsDataMaker.SetAsString( const aValue: string);
    begin
      FData := StrToSignalPairArray( aValue, True, 0.0, 1.0);
      ValueChanged;
      SyncVariations;
    end;


//  private

    function    TKnobsDataMaker.ScaleY( anIndex: Integer): Integer; // overload;
    begin
      Result := ScaleY( FData[ anIndex].Y);
    end;


    function    TKnobsDataMaker.ScaleX( anIndex: Integer): Integer; // overload;
    begin
      Result := ScaleX( FData[ anIndex].X)
    end;


    function    TKnobsDataMaker.ScaleY( aValue: TSignal): Integer; // overload;
    var
      H : Integer;
    begin
      if   DataType = dmtBipolar
      then begin
        H      := Height div 2 - 1;
        Result := H - Round( Clip( aValue * YZoom * H + YPan, - H, H)) + 1;
      end
      else begin
        H      := Height - 1;
        Result := H - Round( Clip( aValue * YZoom * H + YPan, - H, H));
      end;
    end;


    function    TKnobsDataMaker.ScaleX( aValue: TSignal): Integer; // overload;
    var
      W : Integer;
    begin
      W      := Width - 3;
      Result := 1 + Round( Clip( aValue * XZoom * W + XPan, 0, W));
    end;


    function    TKnobsDataMaker.UnscaleX( aValue: Integer): TSignal;
    begin
      Clip( aValue, 1, Width - 2);
      Result := RangeMap( aValue, 1, Width - 2, 0.0, 1.0);
    end;


    function    TKnobsDataMaker.UnscaleY( aValue: Integer): TSignal;
    begin
      Clip( aValue, 1, Height - 2);

      if   DataType = dmtBipolar
      then Result := RangeMap( aValue, 1, Height - 2, 1.0, -1.0)
      else Result := RangeMap( aValue, 1, Height - 2, 1.0,  0.0);
    end;


    function    TKnobsDataMaker.IsNearPoint( anX, anY: TSignal; aData: TSignalPair): Boolean;
    const
      DeltaX = 0.025;
      DeltaY = 2 * DeltaX;
    begin
      Result :=
        ( aData.X >= anX - DeltaX) and
        ( aData.X <= anX + DeltaX) and
        ( aData.Y >= anY - DeltaY) and
        ( aData.Y <= anY + DeltaY);
    end;


    function    TKnobsDataMaker.FindDataPoint( anX, anY: TSignal): Integer; // overload;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to DataCount - 1
      do begin
        if   IsNearPoint( anX, anY, FData[ i])
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TKnobsDataMaker.FindDataPoint( anX, anY: Integer): Integer; // overload;
    var
      ScaledX : TSignal;
      ScaledY : TSignal;
    begin
      ScaledX := UnscaleX( anX);
      ScaledY := UnscaleY( anY);
      Result  := FindDataPoint( ScaledX, ScaledY);
    end;


//  protected

    procedure   TKnobsDataMaker.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsDataMaker.WMEraseBackground( var aMsg: TWMEraseBkgnd); // message WM_ERASEBKGND;
    begin
      aMsg.Result := -1;
    end;


    procedure   TKnobsDataMaker.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP : TKnobsWirePanel;
    begin
      inherited;

      if   not ( csDesigning in ComponentState)
      then begin
        if   FMouseDown
        then Exit;

        if   Button = mbLeft
        then begin
          if
              (
                   (( ssDouble in Shift) and not ( ssCtrl   in Shift))
                or (( ssCtrl   in Shift) and not ( ssDouble in Shift))
              )
          and AllowPointInsertion
          then AddData( UnscaleX( X), UnscaleY( Y));

          FSelectedPoint := FindDataPoint( X, Y);

          if   FSelectedPoint >= 0
          then begin
            FMouseDown       := True;
            FCursorXWasShown := ShowXCursor;
            FCursorYWasShown := ShowYCursor;
            FFollowCursor    := False;
            ShowXCursor      := True;
            ShowYCursor      := True;
            FLastX           := CursorX;
            FLastY           := CursorY;
            CursorX          := UnscaleX( X);
            CursorY          := UnscaleY( Y);
            BeginStateChange;
          end;
        end
        else if Button = mbRight
        then begin
          aWp := FindWirePanel;

          if   Assigned( aWP)
          then aWP.HandleRightClick( Self);
        end;
      end;
    end;


    procedure   TKnobsDataMaker.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    const
      SafetyPixels = 25;
    begin
      inherited;

      if   not ( csDesigning in ComponentState)
      then begin
        if   FMouseDown
        then begin
          FMouseDown    := False;
          FFollowCursor := True;
          ShowXCursor   := FCursorXWasShown;
          ShowYCursor   := FCursorYWasShown;
          CursorX       := FLastX;
          CursorY       := FLastY;

          if
            (
              ( X <        - SafetyPixels) or                                          // if dragged outside view
              ( X > Width  + SafetyPixels) or
              ( Y <        - SafetyPixels) or
              ( Y > Height + SafetyPixels)
            )                                                                          // but not first or last point ... unless not glued
          and  (( FSelectedPoint > 0            ) or not GluedLeft )
          and  (( FSelectedPoint < DataCount - 1) or not GluedRight)
          and  AllowPointDeletion                                                      // and also not when forbidden
          then DeleteData( FSelectedPoint, True)                                       // then Remove it
          else begin
            if   FSelectedPoint >= 0                                                   // otherwise move it (which looks at GluedXXX again)
            then ChangeData( FSelectedPoint, UnscaleX( X), UnscaleY( Y))
          end;

          EndStateChange;
        end;
      end;
    end;


    procedure   TKnobsDataMaker.MouseMove( Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   not ( csDesigning in ComponentState)
      then begin
        if   FMouseDown
        then begin
          if   AllowXMoves
          then CursorX := UnscaleX( X);

          if   AllowYMoves
          then CursorY := UnscaleY( Y);
        end
        else begin
          if   FindDataPoint( X, Y) >= 0
          then Cursor := crHandPoint
          else Cursor := crDefault;
        end;
      end;
    end;


    procedure   TKnobsDataMaker.Paint; // override;
    var
      i       : Integer;
      X       : Integer;
      Y       : Integer;
      DLight  : TColor;
      DDark   : TColor;
      aSpline : TBSpline;
    begin
      inherited;

      with Canvas
      do begin
        // Draw border and background

        Pen.Mode    := pmCopy;
        Pen.Style   := psSolid;
        Brush.Color := Color;

        if CanRandomize and ShowAllowRandomization
        then begin
          Pen.Color := CL_RANDOMIZABLE;
          Pen.Width := 1;
        end
        else begin
          Pen.Color := BorderColor;
          Pen.Width := 1;
        end;

        if   Transparent
        then Brush.Style := bsClear
        else Brush.Style := bsSolid;

        Rectangle( 0, 0, Width, Height);

        if   ShowGrid
        then begin
          // Draw horizontal grid
          Pen.Color := GridColor;
          Pen.Width := 1;
          Pen.Style := psSolid;

          if   DataType = dmtBipolar
          then begin
            for i := -5 to 5
            do begin
              if   i = 0
              then Pen.Width := 2
              else Pen.Width := 1;

              MoveTo( ScaleX( 0.0), ScaleY( i / 5.0));
              LineTo( ScaleX( 1.0), ScaleY( i / 5.0));
            end;
          end
          else begin
            for i := 0 to 10
            do begin
              if   i = 5
              then Pen.Width := 2
              else Pen.Width := 1;

              MoveTo( ScaleX( 0.0), ScaleY( i / 10.0));
              LineTo( ScaleX( 1.0), ScaleY( i / 10.0));
            end;
          end;

          // Draw vertical grid

          for i := 0 to 10
          do begin
            if   i = 5
            then Pen.Width := 2
            else Pen.Width := 1;

            MoveTo( ScaleX( i / 10.0), ScaleY( - 1.0));
            LineTo( ScaleX( i / 10.0), ScaleY(   1.0));
          end;
        end;

        // Draw data

        if   GraphMode = dmgSplines
        then begin
          aSpline := TBSpline.Create;

          try
            aSpline.Fragments := Width;

            for i := 0 to DataCount - 1
            do aSpline.AddPoint( Vertex( ScaleX( FData[ i].X), ScaleY( FData[ i].Y)));

            // Draw the spline without it's internal dot drawing by using DotSize = 0
            aSpline.Draw( Canvas, LineColor, DotColor, 0);
          finally
            aSpline.DisposeOf;
          end;
        end
        else if GraphMode = dmgLinear
        then begin
          if   DataCount > 0
          then begin
            Pen.Color   := LineColor;
            Pen.Width   := 1;
            Pen.Style   := psSolid;
            X           := ScaleX( 0);
            Y           := ScaleY( 0);
            MoveTo( X, Y);

            for i := 1 to DataCount - 1
            do begin
              X := ScaleX( i);
              Y := ScaleY( i);
              LineTo( X, Y);
            end;

          end;
        end
        else begin
          if   DataCount > 0
          then begin
            Pen.Color   := LineColor;
            Pen.Width   := 1;
            Pen.Style   := psSolid;
            X           := ScaleX( 0);
            Y           := ScaleY( 0);
            MoveTo( X, Y);

            for i := 1 to DataCount - 1
            do begin
              X := ScaleX( i);
              LineTo( X, Y);
              Y := ScaleY( i);
              LineTo( X, Y);
            end;

          end;
        end;

        // Draw dots if needed

        if   DotSize > 0
        then begin
          Pen  .Style := psSolid;
          Brush.Style := bsSolid;
          DLight      := InterpolateColors( DotColor, clWhite, 128);
          DDark       := InterpolateColors( DotColor, clBlack,  64);

          for i := 0 to DataCount - 1
          do begin
            X := ScaleX( i);
            Y := ScaleY( i);
            Pen  .Color := DotColor;
            Brush.Color := DotColor;
            Ellipse( Rect( X - DotSize, Y - DotSize, X + DotSize + 1, Y + DotSize + 1));

            Dec( X, DotSize div 2);
            Dec( Y, DotSize div 2);
            Pen  .Color := DLight;
            Brush.Color := DLight;
            Ellipse( Rect( X - DotSize div 2, Y - DotSize div 2, X + DotSize div 2 + 1, Y + DotSize div 2 + 1));

            Inc( X, DotSize);
            Inc( Y, DotSize);
            Pen  .Color := DDark;
            Brush.Color := DDark;
            Ellipse( Rect( X - DotSize div 2 - 1, Y - DotSize div 2 - 1, X + DotSize div 2, Y + DotSize div 2));
          end;
        end;

        // Draw cursors where needed

        if   ShowXCursor
        then begin
          Pen.Color := CursorColor;
          Pen.Width := 1;
          Pen.Style := psSolid;
          Pen.Mode  := pmCopy;
          MoveTo( ScaleX( CursorX), ScaleY( - 1.0));
          LineTo( ScaleX( CursorX), ScaleY(   1.0));
        end;

        if   ShowYCursor
        then begin
          Pen.Color := CursorColor;
          Pen.Width := 1;
          Pen.Style := psSolid;
          Pen.Mode  := pmCopy;
          MoveTo( ScaleX( 0.0), ScaleY( CursorY));
          LineTo( ScaleX( 1.0), ScaleY( CursorY));
        end;
      end;
    end;


    procedure   TKnobsDataMaker.TextChanged( const aNewValue: string; DoNotify: Boolean); // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnChanged)
      then FOnChanged( Self, Name, aNewValue)
      else begin
        aModule := Module;

        if   Assigned( aModule) and DoNotify
        then Module.TextChanged( Self, MakePath( [ aModule.Name, Name]), aNewValue);
      end;
    end;


    procedure   TKnobsDataMaker.HandleTextChange( const aValue: string); // virtual;
    begin
      TextChanged( aValue, DO_NOTIFY);
    end;


    function    TKnobsDataMaker.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    procedure   TKnobsDataMaker.SyncVariations;
    begin
      if   Length( FMultiVariationValues) <> Length( FData)
      then begin
        SetLength( FMultiVariationValues, Length( FData));
        SetLength( FCurrentValues       , Length( FData));
        SetDefaultVariations;
      end;
    end;


    procedure   TKnobsDataMaker.ValueChanged;
    begin
      if   AutoLoop
      then MakeLoopable;

      if   AutoScale
      then Scale;

      HandleTextChange( AsString);
      Invalidate;
    end;


//  public

    constructor TKnobsDataMaker.Create( anOwner: TComponent); // override;
    begin
      inherited;
      SetLength( FData, 2);
      FDataType              := dmtBipolar;
      FData[ 0].X            := 0.0;
      FData[ 0].Y            := 0.0;
      FData[ 1].X            := 1.0;
      FData[ 1].Y            := 0.0;
      SyncVariations;
      XZoom                  := 1.0;
      YZoom                  := 1.0;
      XPan                   := 0.0;
      YPan                   := 0.0;
      Width                  := 150;
      Height                 := 60;
      AutoLoop               := false;
      AutoScale              := False;
      GluedLeft              := True;
      GluedRight             := True;
      AutoHGrid              := False;
      AutoVGrid              := False;
      LineColor              := clWhite;
      BorderColor            := clYellow;
      Color                  := clGray;
      GridColor              := clSilver;
      CursorColor            := CL_CURSOR;
      DotColor               := CL_DOT;
      ActiveDotColor         := CL_ACTIVEDOT;
      ShowGrid               := True;
      ShowXCursor            := True;
      ShowYCursor            := True;
      DotSize                := 3;
      Transparent            := True;
      CursorX                := 0.0;
      CursorY                := 0.0;
      FFollowCursor          := True;
      AllowXMoves            := True;
      AllowYMoves            := True;
      AllowPointDeletion     := True;
      AllowPointInsertion    := True;
      AllowRandomization     := False;
      ShowAllowRandomization := False;
    end;


    destructor  TKnobsDataMaker.Destroy; // override;
    begin
      inherited;
    end;


    procedure   TKnobsDataMaker.MakeWave( aShape: TKnobsDMShape);

      procedure MakeClear;
      begin
        SetLength( FData, 2);
        FData[ 0].X := 0.0;
        FData[ 0].Y := 0.0;
        FData[ 1].X := 1.0;
        FData[ 1].Y := 0.0;
      end;

      procedure MakeSine;
      var
        i : Integer;
      begin
        SetLength( FData, 15);

        for i := 0 to 14
        do begin
          FData[ i].X := i / 14;
          FData[ i].Y := Sin( TWO_PI * i / 14);
        end;
      end;

      procedure MakeCosine;
      var
        i : Integer;
      begin
        SetLength( FData, 15);

        for i := 0 to 14
        do begin
          FData[ i].X := i / 14;
          FData[ i].Y := Cos( TWO_PI * i / 14);
        end;
      end;

      procedure MakeArcTan;
      var
        i : Integer;
      begin
        SetLength( FData, 15);

        for i := 0 to 14
        do begin
          FData[ i].X := i / 14;
          FData[ i].Y := ArcTan( ( i - 7)) / HALF_PI;
        end;
      end;

      procedure MakeBell;
      var
        i : Integer;
      begin
        SetLength( FData, 15);

        for i := 0 to 14
        do begin
          FData[ i].X := i / 14;
          FData[ i].Y := 2 * Sin( PI * i / 14) * Sin( PI * i / 14) - 1;
        end;
      end;

      procedure MakeTri;
      begin
        SetLength( FData, 3);
        FData[ 0].X :=  0.0;
        FData[ 0].Y := -1.0;
        FData[ 1].X :=  0.5;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  1.0;
        FData[ 2].Y := -1.0;
      end;

      procedure MakeSaw;
      begin
        SetLength( FData, 2);
        FData[ 0].X :=  0.0;
        FData[ 0].Y := -1.0;
        FData[ 1].X :=  1.0;
        FData[ 1].Y :=  1.0;
      end;

      procedure MakeSquare3;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.029999999;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.030000000;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeSquare10;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.099999999;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.100000000;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeSquare33;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.333333332;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.333333333;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeSquare50;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.499999999;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.500000000;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeSquare67;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.666666666;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.666666667;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeSquare90;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.899999999;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.900000000;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeSquare97;
      begin
        SetLength( FData, 4);
        FData[ 0].X :=  0.0;
        FData[ 0].Y :=  1.0;
        FData[ 1].X :=  0.969999999;
        FData[ 1].Y :=  1.0;
        FData[ 2].X :=  0.970000000;
        FData[ 2].Y := -1.0;
        FData[ 3].X :=  1.0;
        FData[ 3].Y := -1.0;
      end;

      procedure MakeNoise;
      var
        i : Integer;
      begin
        SetLength( FData, 15);

        for i := 0 to 14
        do begin
          FData[ i].X:= i / 14;
          FData[ i].Y:= 2 * Random - 1;
        end;
      end;

      procedure MakeHFlip;
      var
        i : Integer;
      begin
        for i := 0 to DataCount - 1
        do FData[ i].X:= 1.0 - FData[ i].X;
      end;

      procedure MakeVFlip;
      var
        i : Integer;
      begin
        for i := 0 to DataCount - 1
        do FData[ i].Y := - FData[ i].Y;
      end;

      procedure MakeRotate;
      var
        i : Integer;
      begin
        for i := 0 to DataCount - 1
        do begin
          FData[ i].X := 1.0 - FData[ i].X;
          FData[ i].Y := -     FData[ i].Y;
        end;
      end;

    begin
      case aShape of
        dmsClear         : MakeClear   ;          // Clear to initial state
        dmsSine          : MakeSine    ;          // Sine
        dmsCosine        : MakeCosine  ;          // Cosine
        dmsArcTan        : MakeArcTan  ;          // Arc tangent
        dmsBell          : MakeBell    ;          // Bell shaped curve
        dmsTri           : MakeTri     ;          // Triangle wave
        dmsSaw           : MakeSaw     ;          // Sawtooth wave
        dmsSquare3       : MakeSquare3 ;          // square  3% DC
        dmsSquare10      : MakeSquare10;          // square 10% DC
        dmsSquare33      : MakeSquare33;          // square 33% DC
        dmsSquare50      : MakeSquare50;          // square 50% DC
        dmsSquare67      : MakeSquare67;          // square 67% DC
        dmsSquare90      : MakeSquare90;          // square 90% DC
        dmsSquare97      : MakeSquare97;          // square 97% DC
        dmsNoise         : MakeNoise   ;          // Random, each time different
        dmsSwapLeftRight : MakeHFlip   ;          // Horizontal flip
        dmsSwapTopBottom : MakeVFlip   ;          // Vertical   flip
        dmsRotate180     : MakeRotate  ;          // Rotate 180 degrees
        dmsFixLoop       : MakeLoopable;          // Make the first point match the last one - for now a first order approach
        dmsFixScale      : Scale;                 // Make the graph fill the entire Y range
        dmsSaveToFile    : SaveToFile  ;          // Save the graph to a file
        dmsLoadFromFile  : LoadFromFile;          // Load the graph from a file
      end;

      ValueChanged;
      SyncVariations;
    end;


    procedure   TKnobsDataMaker.Scale;
    var
      MinVal : TSignal;
      MaxVal : TSignal;
      i      : Integer;
    begin
      MinVal :=  1000;
      MaxVal := -1000;

      for i := 0 to Length( FData) - 1
      do begin
        if   FData[ i].Y < MinVal
        then MinVal := FData[ i].Y;

        if   FData[ i].Y > MaxVal
        then MaxVal := FData[ i].Y;
      end;

      if   MinVal <> MaxVal
      then begin
        for i := 0 to Length( FData) - 1
        do FData[ i].Y := RangeMap( FData[ i].Y, MinVal, MaxVal, -1, 1);
      end;

      Invalidate;
    end;


    procedure   TKnobsDataMaker.MakeLoopable;
    var
      MidPoint  : TSignal;
      LastPoint : Integer;
    begin
      if   DataCount >= 2
      then begin
        LastPoint           := DataCount - 1;
        MidPoint            := ( FData[ 0].Y + FData[ LastPoint].Y) * 0.5;
        FData[ 0        ].Y := MidPoint;
        FData[ LastPoint].Y := MidPoint;
        Invalidate;
      end;
    end;


    procedure   TKnobsDataMaker.SaveToFile;
    var
      aWP : TKnobsWirePanel;
    begin
      if   Assigned( FOnSaveDataGraph)
      then FOnSaveDataGraph( Self, AsString)
      else begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.SaveDataGraph( AsString);
      end;
    end;


    procedure   TKnobsDataMaker.LoadFromFile;
    var
      aWP : TKnobsWirePanel;
      S   : string;
    begin
      S := '';

      if   Assigned( FOnLoadDataGraph)
      then S := FOnLoadDataGraph( Self)
      else begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then S := aWP.LoadDataGraph;
      end;

      if   S <> ''
      then AsString := S;
    end;


    procedure   TKnobsDataMaker.BeginStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      SetPrevData;
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.BeginStateChange( False);
    end;


    procedure   TKnobsDataMaker.EndStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.EndStateChange( False, False);
    end;


{ ========
  TKnobsGridProto = class( TKnobsGraphicControl)
  private
    FShortName         : string;
    FPixelXSize        : Integer;
    FPixelYSize        : Integer;
    FDataWidth         : Integer;
    FDataHeight        : Integer;
    FData              : TKnobsDataPlane;
    FBorderColor       : TColor;
    FLineColor         : TColor;
    FGridColor         : TColor;
    FCursorColor       : TColor;
    FDotColor          : TColor;
    FActiveDotColor    : TColor;
    FTransparent       : Boolean;
    FShowGrid          : Boolean;
    FCursorX           : Integer;
    FCursorY           : Integer;
    FShowCursorX       : Boolean;
    FShowCursorY       : Boolean;
    FMouseDown         : Boolean;
    FBlockUpdates      : Boolean;
    FOnChanged         : TKnobsOnTextChanged;
    FOnLoadGridControl : TOnLoadGridControl;
    FOnSaveGridControl : TOnSaveGridControl;
  public
    property    ShortName             : string             read FShortName;
    property    CursorX               : TSignal            read GetCursorX         write SetCursorX;
    property    CursorY               : TSignal            read GetCursorY         write SetCursorY;
    property    CursorXY              : TSignalPair        read GetCursorXY        Write SetCursorXY;
    property    Pixel[ X, Y: Integer] : Boolean            read GetPixel           write SetPixel;
    property    AsString              : string             read GetAsString        write SetAsString;
  published
    property    Color                                                                                       default clGray;
    property    Width                                                                                       default 241;
    property    Height                                                                                      default 241;
    property    LineColor         : TColor                 read FLineColor         write SetLineColor       default clWhite;
    property    BorderColor       : TColor                 read FBorderColor       write SetBorderColor     default clGray;
    property    GridColor         : TColor                 read FGridColor         write SetGridColor       default clGray;
    property    CursorColor       : TColor                 read FCursorColor       write SetCursorColor     default CL_CURSOR;
    property    DotColor          : TColor                 read FDotColor          write SetDotColor        default CL_GRIDDOT;
    property    ActiveDotColor    : TColor                 read FActiveDotColor    write SetActiveDotColor  default CL_ACTIVEDOT;
    property    Transparent       : Boolean                read FTransparent       write SetTransparent     default True;
    property    ShowGrid          : Boolean                read FShowGrid          write SetShowGrid        default True;
    property    ShowCursorX       : Boolean                read FShowCursorX       write SetShowCursorX     default True;
    property    ShowCursorY       : Boolean                read FShowCursorY       write SetShowCursorY     default True;
    property    DataWidth         : Integer                read FDataWidth         write SetDataWidth       default 60;
    property    DataHeight        : Integer                read FDataHeight        write SetDataHeight      default 60;
    property    OnChanged         : TKnobsOnTextChanged    read FOnChanged         write FOnChanged;
    property    OnLoadGridControl : TOnLoadGridControl     read FOnLoadGridControl write FOnLoadGridControl;
    property    OnSaveGridControl : TOnSaveGridControl     read FOnSaveGridControl write FOnSaveGridControl;
  private
}

    procedure   TKnobsGridProto.SetLineColor( aValue: TColor);
    begin
      if   aValue <> FLineColor
      then begin
        FLineColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetBorderColor( aValue: TColor);
    begin
      if   aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetGridColor( aValue: TColor);
    begin
      if   aValue <> FGridColor
      then begin
        FGridColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetCursorColor( aValue: TColor);
    begin
      if   aValue <> FCursorColor
      then begin
        FCursorColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetDotColor( aValue: TColor);
    begin
      if   aValue <> FDotColor
      then begin
        FDotColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetActiveDotColor( aValue: TColor);
    begin
      if   aValue <> FActiveDotColor
      then begin
        FActiveDotColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetTransparent( aValue: Boolean);
    begin
      if   aValue <> FTransparent
      then begin
        FTransparent := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetShowGrid( aValue: Boolean);
    begin
      if   aValue <> FShowGrid
      then begin
        FShowGrid := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsGridProto.GetCursorX : TSignal;
    begin
      if   FDataWidth > 0
      then Result := FCursorX / FDataWidth
      else Result := 1;
    end;


    procedure   TKnobsGridProto.SetCursorX( aValue: TSignal);
    begin
      if   aValue <> CursorX
      then begin
        FCursorX := Round( aValue * FDataWidth);
        Invalidate;
      end;
    end;


    function    TKnobsGridProto.GetCursorY: TSignal;
    begin
      if   FDataHeight > 0
      then Result := FCursorY / FDataHeight
      else Result := 1;
    end;


    procedure   TKnobsGridProto.SetCursorY( aValue: TSignal);
    begin
      if   aValue <> CursorY
      then begin
        FCursorY := Round( aValue * FDataHeight);
        Invalidate;
      end;
    end;


    function    TKnobsGridProto.GetCursorXY: TSignalPair;
    begin
      Result := SignalPair( CursorX, CursorY);
    end;


    procedure   TKnobsGridProto.SetCursorXY( aValue: TSignalPair);
    begin
      if   ( CursorX <> aValue.X) or ( CursorY <> aValue.Y)
      then begin
        FCursorX := Round( aValue.X * FDataWidth );
        FCursorY := Round( aValue.Y * FDataHeight);
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetShowCursorX( aValue: Boolean);
    begin
      if   aValue <> FShowCursorX
      then begin
        FShowCursorX := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetShowCursorY( aValue: Boolean);
    begin
      if   aValue <> FShowCursorY
      then begin
        FShowCursorY := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsGridProto.SetDataWidth( aValue: Integer);
    begin
      if   aValue <> FDataWidth
      then begin
        FDataWidth := aValue;
        CreateData;

        if   csDesigning in ComponentState
        then FillRandom( 0.333);
      end;
    end;


    procedure   TKnobsGridProto.SetDataHeight( aValue: Integer);
    begin
      if   aValue <> FDataHeight
      then begin
        FDataHeight := aValue;
        CreateData;

        if   csDesigning in ComponentState
        then FillRandom( 0.333);
      end;
    end;


    function    TKnobsGridProto.GetPixel ( X, Y: Integer): Boolean;
    begin
      if   ( X >= 0)
      and  ( X < DataWidth)
      and  ( Y >= 0)
      and  ( Y < DataHeight)
      then Result := FData[ Y, X]
      else Result := False;
    end;


    procedure   TKnobsGridProto.SetPixel( X, Y: Integer; aValue: Boolean);
    begin
      if   ( X >= 0)
      and  ( X < DataWidth)
      and  ( Y >= 0)
      and  ( Y < DataHeight)
      and  ( aValue <> FData[ Y, X])
      then begin
        FData[ Y, X] := aValue;
        Invalidate;
        ValueChanged;
      end;
    end;


    function    TKnobsGridProto.GetAsString: string;
    begin
      Result := DataPlaneToStr( FData);
    end;


    procedure   TKnobsGridProto.SetAsString( const aValue: string);
    begin
      if   not FBlockUpdates
      then begin
        try
          FBlockUpdates := True;
          StrToDataPlane( aValue, FData);
          Invalidate;
          ValueChanged;
        finally
          FBlockUpdates := False;
        end;
      end;
    end;


//  protected

    procedure   TKnobsGridProto.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsGridProto.WMEraseBackground( var aMsg: TWMEraseBkgnd); // message WM_ERASEBKGND;
    begin
      aMsg.Result := -1;
    end;


    procedure   TKnobsGridProto.MouseDown( aButton: TMouseButton; aShift: TShiftState; X, Y: Integer); // override;
    var
      aGridX : Integer;
      aGridY : Integer;
    begin
      MouseToGrid( X, Y, aGridX, aGridY);

      if   ssDouble in aShift
      then DoubleClicked( aButton, aShift, aGridX, aGridY)
      else if [ ssLeft] = aShift
      then Clicked( aButton, aShift, aGridX, aGridY)
      else if ( ssLeft in aShift) and ( aShift * [ ssShift, ssCtrl] <> [])
      then FMouseDown := True
      else if aButton = mbRight
      then RightClicked( aButton, aShift, aGridX, aGridY)
      else inherited;
    end;


    procedure   TKnobsGridProto.MouseUp( aButton: TMouseButton; aShift: TShiftState; X, Y: Integer); // override;
    var
      aGridX : Integer;
      aGridY : Integer;
    begin
      if   FMouseDown
      then begin
        MouseToGrid( X, Y, aGridX, aGridY);

        FMouseDown := False;
        ClickEnded( aButton, aShift, aGridX, aGridY);
      end;
    end;


    procedure   TKnobsGridProto.MouseMove( aShift: TShiftState; X, Y: Integer); // override;
    var
      aGridX : Integer;
      aGridY : Integer;
    begin
      if   FMouseDown
      and  ( aShift * [ ssShift, ssCtrl] <> [])
      then begin
        MouseToGrid( X, Y, aGridX, aGridY);
        MouseMoved( aShift, aGridX, aGridY);
      end;
    end;


    procedure   TKnobsGridProto.Paint; // override;
    var
      i : Integer;
      j : Integer;
      R : TRect;
    begin
      inherited;

      FPixelXSize := Width  div DataWidth;
      FPixelYSize := Height div DataHeight;

      with Canvas
      do begin
        // Draw border and background

        Pen.Color   := BorderColor;
        Pen.Width   := 1;
        Pen.Style   := psSolid;
        Pen.Mode    := pmCopy;
        Brush.Color := Color;

        if   Transparent
        then Brush.Style := bsClear
        else Brush.Style := bsSolid;

        Rectangle( 0, 0, Width, Height);

        // Draw data cells

        Pen.Color   := FDotColor;
        Brush.Color := FDotColor;
        Brush.Style := bsSolid;

        for i := 0 to Length( FData) - 1
        do begin
          for j := 0 to Length( FData[ i]) - 1
          do begin
            if   FData[ i, j]
            then begin
              R := Rect( j * FPixelXSize + 1, i * FPixelYSize + 1, ( j + 1) * FPixelXSize, ( i + 1) * FPixelYSize);

              if   ClipRect.IntersectsWith( R)
              then Rectangle( R);
            end;
          end;
        end;

        if   ShowGrid
        then begin
          // Draw grid

          Pen.Color := GridColor;
          Pen.Width := 1;
          Pen.Style := psSolid;

          for i := 0 to DataHeight
          do begin
            MoveTo( 0    , i * FPixelYSize);
            LineTo( Width, i * FPixelYSize);
          end;

          for i := 0 to DataWidth
          do begin
            MoveTo( i * FPixelXSize, 0     );
            LineTo( i * FPixelXSize, Height);
          end;
        end;

        // Draw cursors where needed

        if   ShowCursorX
        and  ( FCursorX >= 0)
        then begin
          Pen.Color := CursorColor;
          Pen.Width := 1;
          Pen.Style := psSolid;
          Pen.Mode  := pmCopy;
          MoveTo( FPixelXSize div 2 + FPixelXSize * FCursorX, 0     );
          LineTo( FPixelXSize div 2 + FPixelXSize * FCursorX, Height);
        end;

        if   ShowCursorY
        and  ( FCursorY >= 0)
        then begin
          Pen.Color := CursorColor;
          Pen.Width := 1;
          Pen.Style := psSolid;
          Pen.Mode  := pmCopy;
          MoveTo( 0    , FPixelYSize div 2 + FPixelYSize * FCursorY);
          LineTo( Width, FPixelYSize div 2 + FPixelYSize * FCursorY);
        end;
      end;
    end;


    procedure   TKnobsGridProto.TextChanged( const aNewValue: string; DoNotify: Boolean); // virtual;
    var
      aModule : TKnobsCustomModule;
    begin
      if   Assigned( FOnChanged)
      then FOnChanged( Self, Name, aNewValue)
      else begin
        aModule := Module;

        if   Assigned( aModule) and DoNotify
        then Module.TextChanged( Self, MakePath( [ aModule.Name, Name]), aNewValue);
      end;
    end;


    procedure   TKnobsGridProto.HandleTextChange( const aValue: string); // virtual;
    begin
      TextChanged( aValue, DO_NOTIFY);
    end;


    function    TKnobsGridProto.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    procedure   TKnobsGridProto.ValueChanged;
    begin
      HandleTextChange( AsString);
    end;


    procedure   TKnobsGridProto.CreateData;
    var
      i : Integer;
    begin
      SetLength( FData, DataHeight);

      for i := 0 to DataHeight - 1
      do SetLength( FData[ i], DataWidth);
    end;


    procedure   TKnobsGridProto.MouseToGrid( aMouseX, aMouseY: Integer; var aGridX, aGridY: Integer);
    begin
      FPixelXSize := Width   div DataWidth;
      FPixelYSize := Height  div DataHeight;
      aGridX      := aMouseX div FPixelXSize;
      aGridY      := aMouseY div FPixelYSize;
    end;


//  public

    constructor TKnobsGridProto.Create( anOwner: TComponent); // override;
    begin
      inherited;
      CursorX        := -1;
      CursorY        := -1;
      Color          := clGray;
      LineColor      := clWhite;
      BorderColor    := clGray;
      GridColor      := clSilver;
      CursorColor    := CL_CURSOR;
      DotColor       := CL_GRIDDOT;
      ActiveDotColor := CL_ACTIVEDOT;
      Transparent    := True;
      ShowGrid       := True;
      ShowCursorX    := True;
      ShowCursorY    := True;
      Width          := 241;
      Height         := 241;
      DataWidth      :=  60;
      DataHeight     :=  60;
    end;


    destructor  TKnobsGridProto.Destroy; // override;
    begin
      ClearData;
      inherited;
    end;


    procedure   TKnobsGridProto.FillRandom( anOnChance: TSignal);
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to Length( FData) - 1
      do begin
        for j := 0 to Length( FData[ i]) - 1
        do FData[ i, j] := Random < anOnChance;
      end;

      Invalidate;
      ValueChanged;
    end;


    procedure   TKnobsGridProto.SaveToFile;
    var
      aWP : TKnobsWirePanel;
    begin
      if   Assigned( FOnSaveGridControl)
      then FOnSaveGridControl( Self, AsString)
      else begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.SaveGridControl( AsString);
      end;
    end;


    procedure   TKnobsGridProto.LoadFromFile;
    var
      aWP : TKnobsWirePanel;
      S   : string;
    begin
      S := '';

      if   Assigned( FOnLoadGridControl)
      then S := FOnLoadGridControl( Self)
      else begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then S := aWP.LoadGridControl;
      end;

      if   S <> ''
      then AsString := S;
    end;


    procedure   TKnobsGridProto.ClearData;
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to Length( FData) - 1
      do begin
        for j := 0 to Length( FData[ i]) - 1
        do Pixel[ j, i] := False;
      end;
    end;


    procedure   TKnobsGridProto.SetData( const aValue: TKnobsDataPlane);
    var
      i : Integer;
      j : Integer;
    begin
      SetLength( FData, Length( aValue));

      for i := 0 to Length( aValue) - 1
      do begin
        SetLength( FData[ i], Length( aValue[ i]));

        for j:= 0 to Length( aValue[ i]) - 1
        do FData[ i, j] := aValue[ i, j];
      end;

      Invalidate;
      ValueChanged;
    end;


    function    TKnobsGridProto.GetData: TKnobsDataPlane;
    begin
      Result := FData;
    end;


    procedure   TKnobsGridProto.BeginStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.BeginStateChange( False);
    end;


    procedure   TKnobsGridProto.EndStateChange;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.EndStateChange( False, False);
    end;


{ ========
  TKnobsGridControl = class( TKnobsGridProto)
  protected
}

    procedure   TKnobsGridControl.Clicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
      Pixel[ anX, anY] := not Pixel[ anX, anY];
    end;


    procedure   TKnobsGridControl.RightClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWP)
      then aWP.HandleRightClick( Self);
    end;


    procedure   TKnobsGridControl.ClickEnded( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
      ValueChanged;
    end;


    procedure   TKnobsGridControl.DoubleClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
    end;


    procedure   TKnobsGridControl.MouseMoved( aShift: TShiftState; anX, anY: Integer); // override;
    begin
      if   aShift * [ ssLeft, ssShift, ssCtrl] = [ ssLeft, ssCtrl]
      then Pixel[ anX, anY] := False
      else if aShift * [ ssLeft, ssShift, ssCtrl] = [ ssLeft, ssShift]
      then Pixel[ anX, anY] := True;
    end;


{ ========
  TKnobsAutomationGrid = class( TKnobsGridControl)
  private
    FOnClick       : TOnKnobsGridClick;
    FOnDoubleClick : TOnKnobsGridClick;
    FOnRightClick  : TOnKnobsGridClick;
    FOnClickEnd    : TOnKnobsGridClick;
    FOnMouseMove   : TOnKnobsGridMouseMove;
  published
    property    OnClick       : TOnKnobsGridClick     read FOnClick       write FOnClick      ;
    property    OnDoubleClick : TOnKnobsGridClick     read FOnDoubleClick write FOnDoubleClick;
    property    OnRightClick  : TOnKnobsGridClick     read FOnRightClick  write FOnRightClick ;
    property    OnClickEnd    : TOnKnobsGridClick     read FOnClickEnd    write FOnClickEnd   ;
    property    OnMouseMove   : TOnKnobsGridMouseMove read FOnMouseMove   write FOnMouseMove  ;
  protected
}

    procedure   TKnobsAutomationGrid.Clicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
      if   Assigned( FOnClick)
      then FOnClick( Self, aButton, aShift, anX, anY);
    end;


    procedure   TKnobsAutomationGrid.RightClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
      if   Assigned( FOnRightClick)
      then FOnRightClick( Self, aButton, aShift, anX, anY);
    end;


    procedure   TKnobsAutomationGrid.ClickEnded( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
      if   Assigned( FOnClickEnd)
      then FOnClickEnd( Self, aButton, aShift, anX, anY);
    end;


    procedure   TKnobsAutomationGrid.DoubleClicked( aButton: TMouseButton; aShift: TShiftState; anX, anY: Integer); // override;
    begin
      if   Assigned( FOnDoubleClick)
      then FOnDoubleClick( Self, aButton, aShift, anX, anY);
    end;


    procedure   TKnobsAutomationGrid.MouseMoved( aShift: TShiftState; anX, anY: Integer); // override;
    begin
      if   Assigned( FOnMouseMove)
      then FOnMouseMove( Self, aShift, anX, anY);
    end;


{ ========
  TKnobsConnector = class( TKnobsGraphicControl)
  // to connect wires to
  private
    FBitmapAudio           : TBitmap;
    FBitmapControl         : TBitmap;
    FBitmapLogic           : TBitmap;
    FBitmapControlLogic    : TBitmap;
    FSignalType            : TSignalType;
    FWireColor             : TColor;
    FBorderColor           : TColor;
    FUseBorders            : Boolean;
    FUseTypedBorders       : Boolean;
    FLinks                 : TList;   // Connections, not wires
    FMouseDown             : Boolean;
    FConnected             : Boolean; // True when at least one connection exists
    FHighLight             : Boolean; // Highlighted connections
    FCustomWireColor       : Boolean; // True when a custom wire color was set
    FAllowSignalConversion : Boolean; // True when this connector may be sped up by connecting a fast signal to it
    FIsSpedUp              : Boolean; // True when this coonector was actually sped up
    FVisited               : Boolean;
    FIsMove                : Boolean; // True when the mouse action indicates a connection move (or delete) (instead of a make)
  protected
    property    HighLight             : Boolean            read FHighLight             write SetHighLight;
  public
    property    Connected             : Boolean            read FConnected             write SetConnected;
    property    ModuleName            : string             read GetModuleName;
    property    DefaultWireColor      : TColor             read GetDefaultWireColor;
    property    EffectiveColor        : TColor             read GetEffectiveColor;
    property    CustomWireColor       : Boolean            read FCustomWireColor;
    property    EffectiveSignalType   : TSignalType        read GetEffectiveSignalType;
    property    IsSpedUp              : Boolean            read FIsSpedUp              write SetIsSpedUp;
  public
    property    Connected             : Boolean            read FConnected             write SetConnected;
    property    Module                : TKnobsCustomModule read GetModule;
    property    ModuleName            : string             read GetModuleName;
    property    DefaultWireColor      : TColor             read GetDefaultWireColor;
    property    EffectiveColor        : TColor             read GetEffectiveColor;
    property    CustomWireColor       : Boolean            read FCustomWireColor;
    property    EffectiveSignalType   : TSignalType        read GetEffectiveSignalType;
    property    IsSpedUp              : Boolean            read FIsSpedUp              write SetIsSpedUp;
  published
    property    Signaltype            : TSignalType        read FSignalType            write SetSignalType            default stControl;
    property    WireColor             : TColor             read FWireColor             write SetWireColor             default clBlue;
    property    BorderColor           : TColor             read FBorderColor           write SetBorderColor           default clWhite;
    property    UseBorders            : Boolean            read FUseBorders            write SetUseBorders            default True;
    property    UseTypedBorders       : Boolean            read FUseTypedBorders       write SetUseTypedBorders       default False;
    property    AllowSignalConversion : Boolean            read FAllowSignalConversion write SetAllowSignalConversion default False;
    property    Visible;
  private
}

    procedure   TKnobsConnector.FixSpeedUp;
    var
      anOutput  : TKnobsConnector;
      aNewValue : Boolean;
    begin
      aNewValue := False;

      if   AllowSignalConversion
      then begin
        anOutput := FindOutput;

        if   Connected and Assigned( anOutput) and SignalIsFast( anOutput.EffectiveSignaltype)
        then aNewValue := not SignalIsFast( SignalType);
      end;

      IsSpedUp := aNewValue;
    end;


    procedure   TKnobsConnector.SetConnected( aValue: Boolean);
    begin
      if   aValue <> FConnected
      then begin
        FConnected := aValue;
        FixSpeedUp;
        Invalidate;
      end;
    end;


    procedure   TKnobsConnector.SetIsSpedUp( aValue: Boolean);
    begin
      if   aValue <> FIsSpedUp
      then begin
        FIsSpedUp := aValue;

        if   Assigned( Module)
        then Module.Invalidate
        else Invalidate;
      end;
    end;


    function    TKnobsConnector.GetModuleName: string;
    var
      aModule: TKnobsCustomModule;
    begin
      aModule := Module;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


    function    TKnobsConnector.GetDefaultWireColor: TColor;
    begin
      Result := SignalTypeToColor( Signaltype);
    end;


    function    TKnobsConnector.GetEffectiveColor: TColor;
    begin
      Result := DefaultWireColor;

      if   ( not SignalIsFast( SignalType))
      and  ( IsSpedUp or ( Assigned( Module) and Module.IsSpedUp))
      then begin
        case SignalType of
          stControl    : Result := SignalTypeToColor( stAudio);
          stContrLogic : Result := SignalTypeToColor( stLogic);
        end;
      end;
    end;


    function    TKnobsConnector.GetEffectiveSignalType: TSignalType;
    begin
      Result := SignalType;

      if   not SignalIsFast( SignalType)
      and  ( IsSpedUp or ( Assigned( Module) and Module.IsSpedUp))
      then begin
        case SignalType of
          stControl    : Result := stAudio;
          stContrLogic : Result := stLogic;
        end;
      end;
    end;


    procedure   TKnobsConnector.SetHighLight( aValue: Boolean);
    var
      i : Integer;
    const
      Depth : Integer = 0;
    begin
      if   aValue <> FHighLight
      then begin
        if   not FVisited
        then begin
          try
            FVisited   := True;
            FHighLight := aValue;
            Invalidate;
            Inc( Depth);

            try
              with FLinks
              do
                for i := 0 to Count - 1
                do TKnobsConnector( Items[ i]).HighLight := aValue;
            finally
              Dec( Depth);
            end;

          finally
            FVisited := False;
          end;
        end;
      end;

      if   Depth = 0
      then SendHighlightNotification;
    end;


    procedure   TKnobsConnector.SetWireColor( aValue: TColor);
    begin
      if   aValue <> FWireColor
      then begin
        FWireColor       := aValue;
        FCustomWireColor := WireColor <> SignalTypeToColor( SignalType);
        SendColorNotification( WireColor);
        Invalidate;
      end;
    end;


    procedure   TKnobsConnector.SetBorderColor( aValue: TColor);
    begin
      if   aValue <> FBorderColor
      then begin
        FBorderColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsConnector.SetUseBorders ( aValue: Boolean);
    begin
      if   aValue <> FUseBorders
      then begin
        FUseBorders := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsConnector.SetUseTypedBorders( aValue: Boolean);
    begin
      if   aValue <> FUseTypedBorders
      then begin
        FUseTypedBorders := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsConnector.SetSignalType( aValue: TSignalType);
    begin
      if   aValue <> FSignalType
      then begin
        FSignalType := aValue;
        WireColor   := DefaultWireColor;
      end;
    end;


    procedure   TKnobsConnector.SetAllowSignalConversion( aValue: Boolean);
    begin
      if   Assigned( Module)
      then aValue := aValue and Module.AllowSignalConversion;

      if   aValue <> FAllowSignalConversion
      then begin
        FAllowSignalConversion := aValue;
        Invalidate;
      end;
    end;


    function    TKnobsConnector.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent :=  aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    procedure   TKnobsConnector.AddLink( aConnector: TKnobsConnector);
    begin
      if   not HasDirectlinkTo( aConnector)
      then begin
        FLinks.Add( aConnector);
        Connected := True;
      end;
    end;


    procedure   TKnobsConnector.DeleteLink( aConnector: TKnobsConnector);
    begin
      Connected := False;

      with FLinks
      do begin
        Remove( aConnector);
        Connected := Count > 0;
      end;
    end;


    function    TKnobsConnector.HasDirectlinkTo( aConnector: TKnobsConnector): Boolean;
    begin
      Result := FLinks.IndexOf( aConnector) >= 0;
    end;


    function    TKnobsConnector.FindConnector( aScreenPoint: TPoint): TKnobsConnector;
    var
      aControl : TControl;
    begin
      Result := nil;
      aControl := FindDragTarget( aScreenPoint, True);

      if   Assigned( aControl)
      and  ( aControl is TKnobsConnector)
      then Result := TKnobsConnector( aControl);
    end;


    procedure   TKnobsConnector.DrawLine( X, Y: Integer);
    var
      aDC    : HDC;
      aPen   : HPen;
      OldObj : HGDIOBJ;
      aColor : TColor;
      aWp    : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then begin

        ScrollMouseInView( X, Y);
        aColor := clWhite;
        aDC    := GetDCEx( aWp.Handle, 0, DCX_PARENTCLIP);
        IntersectClipRect( aDC, 0, 0, aWp.ClientWidth, aWp.ClientHeight);

        try
          SetRop2( aDC, R2_XORPEN);
          aPen   := CreatePen( PS_SOLID, 2, ColorToRGB( aColor));
          OldObj := SelectObject( aDC, aPen);

          try
            with aWp.ScreenToClient( ClientToScreen( CenterPoint))
            do MoveToEx( aDC, X, Y, nil);

            with aWp.ScreenToClient( ClientToScreen( Point( X, Y)))
            do LineTo( aDC, X, Y);
          finally
            SelectObject( aDC, OldObj);
            DeleteObject( aPen);
          end;

        finally
          ReleaseDC( aWp.Handle, aDC);
        end;
      end;
    end;


    procedure   TKnobsConnector.SendWireNotification( Msg: Cardinal; aConnector: TKnobsConnector);
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.Perform( Msg, WPARAM( Self), LPARAM( aConnector));
    end;


    procedure   TKnobsConnector.SendColorNotification( aColor: TColor);
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.Perform( KNOBS_UM_COLORCHANGE, WPARAM( Self), LPARAM( aColor));
    end;


    procedure   TKnobsConnector.SendHighlightNotification;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.Perform( KNOBS_UM_HIGHLIGHT, WPARAM( Self), 0);
    end;


    procedure   TKnobsConnector.ScrollMouseInView( anX, anY: LongInt);
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.ScrollPointInView( aWp.ScreenToClient( ClientToScreen( Point( anX, anY))));
    end;


//  protected

    function    TKnobsConnector.CenterPoint: TPoint;
    begin
      with Result
      do begin
        x := Width  div 2;
        y := Height div 2;
      end;
    end;


    procedure   TKnobsConnector.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP : TKnobsWirePanel;
    begin
      inherited;

      if   FMouseDown
      then Exit;

      if   Button = mbLeft
      then begin
        FMouseDown := True;
        FIsMove    := ssDouble in Shift;
        GDistance  := Point( X, Y);
        DrawLine( X, Y);
        HighLight := True;
      end
      else if Button = mbRight
      then begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.HandleRightClick( Self);
      end;
    end;


    procedure   TKnobsConnector.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aConnector : TKnobsConnector;
    begin
      inherited;

      if   FMouseDown
      then begin
        with GDistance
        do DrawLine( X, Y);

        FMouseDown    := False;
        FIsMove       := FIsMove or ( ssCtrl in Shift);
        HighLight     := False;
        Screen.Cursor := crdefault;
        aConnector    := FindConnector( ClientToScreen( Point( X, Y)));

        if   Assigned( aConnector)
        and  aConnector.CanAccept( Self, FIsMove)
        then begin
          if   FIsMove
          then MoveLinksTo( aConnector)
          else ConnectTo  ( aConnector);
        end
        else begin
          if   not Assigned( aConnector)
          and  FIsMove
          then DisconnectAll;
        end;
      end;
    end;


    procedure   TKnobsConnector.MouseMove( Shift: TShiftState; X, Y: Integer); // override;
    var
      aConnector : TKnobsConnector;
      isMove     : Boolean;
    begin
      inherited;

      if   FMouseDown
      then begin
        with GDistance
        do DrawLine( X, Y);

        GDistance := Point( X, Y);

        with GDistance
        do DrawLine( X, Y);

        isMove     := FIsMove or ( ssCtrl in Shift);
        aConnector := FindConnector( ClientToScreen( Point( X, Y)));

        if   Assigned( aConnector)
        then begin
          if   aConnector.CanAccept( Self, isMove)
          then begin
            if   isMove
            then Screen.Cursor := crMultiDrag
            else Screen.Cursor := crHandPoint;
          end
          else Screen.Cursor := crDefault;
        end
        else Screen.Cursor := crdefault;
      end;
    end;


    function    TKnobsConnector.FindOutput: TKnobsConnector;
    var
      i : Integer;
    begin
      if   Self is TKnobsOutput
      then Result := Self
      else begin
        Result := nil;

        if   not FVisited
        then begin
          FVisited := True;

          try
            with FLinks
            do
              for i := 0 to Count - 1
              do begin
                Result := TKnobsConnector( Items[ i]).FindOutput;

                if   Assigned( Result)
                then Break;
              end;
          finally
            FVisited := False;
          end;
        end;
      end;
    end;


    function    TKnobsConnector.IsConnectedToOutput: Boolean;
    begin
      Result := FindOutput <> nil;
    end;


    procedure   TKnobsConnector.FindDistanceToOutput( var aDistance: Integer);

    // Should at the top level be called with aDistance set to -1
    // so that when no originating output can be found the final
    // value of aDistance will still be -1.
    // When this connector is an output, just set aDistance to 0 and return,
    // otherwise iterate over all links to find one that ends on an output.
    // If so, that recursive call will return 0, which then will be
    // inceremented before this function returns .. i.e it is on further away
    // for every return executed.

    var
      i : Integer;
    begin
      if   Self is TKnobsOutput
      then aDistance := 0
      else begin
        if   not FVisited
        then begin
          FVisited := True;

          try
            with FLinks
            do
              for i := 0 to Count - 1
              do begin
                TKnobsConnector( Items[ i]).FindDistanceToOutput( aDistance);

                if   aDistance >= 0
                then begin
                  Inc( aDistance);
                  Break;
                end;
              end;
          finally
            FVisited := False;
          end;
        end;
      end;
    end;


    function    TKnobsConnector.IsConnectedTo( aConnector: TKnobsConnector): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      if   not FVisited
      then begin
        FVisited := True;

        try
          with FLinks
          do
            for i := 0 to Count - 1
            do begin
              Result :=
                HasDirectlinkTo( aConnector) or
                TKnobsConnector( Items[ i]).IsConnectedTo( aConnector);

              if   Result
              then Break;
            end;
        finally
          FVisited := False;
        end;
      end;
    end;


    function    TKnobsConnector.CanAccept( aSource: TKnobsConnector; isMove: Boolean): Boolean;
    begin
      Result := False;

      if   Assigned( aSource) and ( aSource <> Self)
      then begin
        if   isMove
        then begin
          Result :=
            ( aSource is TKnobsOutput) or
            not ( IsConnectedToOutput and aSource.IsConnectedToOutput);
        end
        else begin
          if   IsConnectedToOutput
          then Result := not aSource.IsConnectedToOutput
          else Result := not IsConnectedTo( aSource);
        end;
      end;
    end;


    procedure   TKnobsConnector.MoveLinksTo( aConnector: TKnobsConnector);
    var
      aMoved  : TKnobsConnector;
      aWp     : TKnobsWirePanel;
      Changed : Boolean;
    begin
      if   Assigned( aConnector)
      and  ( aConnector <> Self)
      then begin
        Changed := False;
        aWp     := FindWirePanel;

        if   Assigned( aWp)
        then begin
          with FLinks
          do begin
            if   Count > 0
            then begin
              aWp.BeginStateChange( True);

              try
                while Count > 0
                do begin
                  aMoved := TKnobsConnector( Items[ Count - 1]);
                  DisConnectFrom( aMoved);
                  aConnector.ConnectTo( aMoved);
                  Changed := True;
                end;
              finally
                aWp.EndStateChange( Changed, False);
              end;
            end;
          end;
        end;
      end;
    end;


    procedure   TKnobsConnector.ConnectTo( aConnector: TKnobsConnector);
    var
      aWp     : TKnobsWirePanel;
      Changed : Boolean;
    begin
      if   Assigned( aConnector)
      and  aConnector.CanAccept( Self, False)
      then begin
        Changed := False;
        aWp     := FindWirePanel;

        if   Assigned( aWp)
        then begin
          aWp.BeginStateChange( True);

          try
            AddLink( aConnector);
            aConnector.AddLink( Self);
            SendWireNotification( KNOBS_UM_CONNECTIONADD, aConnector);
            Changed := True;
          finally
            aWp.EndStateChange( Changed, False);
          end;
        end;
      end;
    end;


    procedure   TKnobsConnector.DisconnectFrom( aConnector: TKnobsConnector);
    var
      aWp     : TKnobsWirePanel;
      Changed : Boolean;
    begin
      if   HasDirectlinkTo( aConnector)
      then begin
        Changed := False;
        aWp     := FindWirePanel;

        if   Assigned( aWp)
        then begin
          aWp.BeginStateChange( True);
          try
            DeleteLink( aConnector);
            aConnector.DeleteLink( Self);
            SendWireNotification( KNOBS_UM_CONNECTIONREMOVE, aConnector);
            Changed := True;
          finally
            aWp.EndStateChange( Changed, False);
          end;
        end;
      end;
    end;


    function   TKnobsConnector.DisConnectAll: Boolean;
    var
      i        : Integer;
      OldLinks : TList;
      aWp      : TKnobsWirePanel;
      Changed  : Boolean;
    begin
      Changed := False;

      if   FLinks.Count > 0
      then begin
        Changed := False;
        aWp     := FindWirePanel;

        if   Assigned( aWp)
        then begin
          aWp.BeginStateChange( True);

          try
            OldLinks := TList.Create;

            try
              with FLinks do
              begin
                for i := 0 to Count - 1
                do OldLinks.Add( Items[ i]);

                while Count > 0
                do begin
                  DisconnectFrom( FLinks[ Count - 1]);
                  Changed := True;
                end;
              end;

              with OldLinks
              do begin
                if   Count > 1
                then begin
                  for i := 1 to Count - 1
                  do begin
                    if   TKnobsConnector( Items[ i - 1]).HasParent
                    and  TKnobsConnector( Items[ i    ]).HasParent
                    then begin
                      TKnobsConnector( Items[ i - 1]).ConnectTo( TKnobsConnector( Items[ i]));
                      Changed := True;
                    end;
                  end;
                end;
              end;
            finally
              OldLinks.DisposeOf;
            end;
          finally
            aWp.EndStateChange( Changed, False);
          end;
        end;
      end;

      Result := Changed;
    end;


    procedure   TKnobsConnector.DisConnectDownStream;
    // todo : Should disconnect the downstream links, currently breaks one upstream link too
    var
      aWp     : TKnobsWirePanel;
      Changed : Boolean;
    begin
      if   FLinks.Count > 0
      then begin
        Changed := False;
        aWp     := FindWirePanel;

        if   Assigned( aWp)
        then begin
          aWp.BeginStateChange( True);

          try
            with FLinks do
            begin
              while Count > 0
              do begin
                DisconnectFrom( FLinks[ Count - 1]);
                Changed := True;
              end;
            end;
          finally
            aWp.EndStateChange( Changed, False);
          end;
        end;
      end;
    end;


    procedure   TKnobsConnector.DisConnectDownStreamRecursive;
    var
      aWp     : TKnobsWirePanel;
      Changed : Boolean;
    begin
      if   ( FLinks.Count > 0)
      and  not FVisited
      then begin
        Changed := False;
        aWp     := FindWirePanel;

        if   Assigned( aWp)
        then begin
          aWp.BeginStateChange( True);
          FVisited := True;

          try
            with FLinks do
            begin
              while Count > 0
              do begin
                if   TKnobsConnector( FLinks[ Count - 1]).DisConnectAll
                then Changed := True;
              end;
            end;
          finally
            FVisited := False;
            aWp.EndStateChange( Changed, False);
          end;
        end;
      end;
    end;


    procedure   TKnobsConnector.ChangeWireColor( aColor: TColor);
    var
      anOutput: TKnobsConnector;
    begin
      anOutput := FindOutput;

      if   Assigned( anOutput)
      then anOutput.WireColor := aColor;
    end;


    function    TKnobsConnector.SelectBitmap: TBitmap;
    begin
      case SignalType of
        stControl    : Result := FBitmapControl;
        stLogic      : Result := FBitmapLogic;
        stContrLogic : Result := FBitmapControlLogic;
        else           Result := FBitmapAudio;
      end;
    end;


    procedure   TKnobsConnector.Paint; // override;
    begin
      with Canvas
      do begin
        Draw( 0, 0, SelectBitmap);

        if   FHighLight or FConnected
        then begin
          Pen.Width   := 1;
          Brush.Style := bsSolid;

          if   FHighLight
          then begin
            Brush.Color := ColorHighlight;
            Pen  .Color := ColorHighlight;
          end
          else if FConnected
          then begin
            Brush.Color := EffectiveColor;
            Pen  .Color := EffectiveColor;
          end;

          if   UseBorders
          then begin
            if   Self is TKnobsOutput
            then Rectangle( 0, 0, Width, Height);

            Ellipse( 0, 0, Width, Height);
          end
          else Ellipse( 1, 1, Width - 1, Height - 1);

          Pen.Color := clWhite;
          Arc(
            1         ,
            1         ,
            Width - 1,
            Height - 1,
            Width div 2              ,
            0                        ,
            0                        ,
            Height
          );
          Pen.Color := clGray;
          Arc(
            1         ,
            1         ,
            Width - 1,
            Height - 1,
            0                        ,
            Width div 2              ,
            Height                   ,
            0
          );
        end;

        if UseBorders
        then begin
          if   UseTypedBorders
          then Pen.Color := EffectiveColor
          else Pen.Color := BorderColor;

          Brush.Style := bsClear;

          if   Self is TKnobsOutput
          then Rectangle( 0, 0, Width, Height)
          else Ellipse( 0, 0, Width, Height)
        end;
      end;
    end;


//  public

    constructor TKnobsConnector.Create( anOwner: TComponent); // override;
    begin
      inherited;
      AssignBitmap;
      Width            := FBitmapAudio.Width;
      Height           := FBitmapAudio.Height;
      FLinks           := TList.Create;
      FSignalType      := stControl;
      FWireColor       := DefaultWireColor;
      FUseBorders      := True;
      FUseTypedBorders := False;
      FBorderColor     := clWhite;
    end;


    destructor  TKnobsConnector.Destroy; // override;
    begin
      DisConnectAll;
      FLinks.DisposeOf;
      inherited;
    end;


    procedure   TKnobsConnector.FixBitmaps; // virtual;
    begin
      AssignBitmap;
    end;


    procedure   TKnobsConnector.Dump( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, Format( 'connector     : %s', [ ConnectorName( self)], AppLocale));
      DumpLinks( aFile, anInd + 1);
      WriteLnInd( aFile, anInd,         'end connector');
    end;


    procedure   TKnobsConnector.DumpFlat( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, Format( 'links to : %s', [ ConnectorName( Self)], AppLocale));
    end;


    procedure   TKnobsConnector.DumpLinks( var aFile: TextFile; anInd: Integer);
    var
      i : Integer;
    begin
      for i := 0 to FLinks.Count - 1
      do TKnobsConnector( FLinks[ i]).DumpFlat( aFile, anInd);
    end;


    procedure   TKnobsConnector.Log( aLogClass: TLogClass; const aMsg: string);
    begin
      if   Parent is TKnobsCustomModule
      then TKnobsCustomModule( Parent).Log( aLogClass, Format( 'connector %s( %s)', [ ConnectorName( Self), aMsg], AppLocale));
    end;


    function    TKnobsConnector.DistanceToOutput: Integer;
    begin

      // Return the distance to the output, there should be one at most
      // (as code elswhere should prohibit output to output connections,
      // raising an EShortCircuit on that).
      // When an output can not be reached return -1
      // When this connector is an output return 0

      Result := -1;                     // Assume no connection to an output ...
      FindDistanceToOutput( Result);
    end;


{ ========
  TKnobsInput = class( TKnobsConnector)
  // Specialized connector with it's own bitmap
  protected
}

    procedure   TKnobsInput.AssignBitmap; // override;
    begin
      FBitmapAudio        := bmConAudioIn       ; FBitmapAudio       .Transparent := True;
      FBitmapControl      := bmConControlIn     ; FBitmapControl     .Transparent := True;
      FBitmapLogic        := bmConLogicIn       ; FBitmapLogic       .Transparent := True;
      FBitmapControlLogic := bmConControlLogicIn; FBitmapControlLogic.Transparent := True;
    end;


{ ========
  TKnobsOutput = class( TKnobsConnector)
  // Specialized connector with it's own bitmap
}

    procedure   TKnobsOutput.AssignBitmap; // override;
    begin
      FBitmapAudio        := bmConAudioOut       ; FBitmapAudio       .Transparent := True;
      FBitmapControl      := bmConControlOut     ; FBitmapControl     .Transparent := True;
      FBitmapLogic        := bmConLogicOut       ; FBitmapLogic       .Transparent := True;
      FBitmapControlLogic := bmConControlLogicOut; FBitmapControlLogic.Transparent := True;
    end;


{ ========
  TKnobsTextLabel = class( TCustomLabel)
  // Just a small print TLabel
  private
    FIsDisplay    : Boolean;
    FDisplayColor : TColor;
    FShortName    : string;
  public
    property    ShortName     : string         read FShortName;
    property    ModuleName    : string         read GetModuleName;
  published
    property    IsDisplay     : Boolean        read FIsDisplay       write SetIsDisplay                   default false;
    property    DisplayColor  : TColor         read FDisplayColor    write SetDisplayColor                default clWhite;
    property    Align;
    property    Alignment;
    property    Anchors;
    property    AutoSize;
    property    BiDiMode;
    property    Caption;
    property    Color;
    property    Constraints;
    property    DragCursor;
    property    DragKind;
    property    DragMode;
    property    Enabled                                                                                   default False;
    property    FocusControl;
    property    Font;
    property    ParentBiDiMode;
    property    ParentColor;
    property    ParentFont;
    property    ParentShowHint                                                                            default False;
    property    PopupMenu;
    property    ShowAccelChar;
    property    ShowHint                                                                                  default False;
    property    Transparent;
    property    Layout;
    property    Visible;
    property    WordWrap;
    property    OnClick;
  protected
}

    function    TKnobsTextLabel.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then result := TKnobsCustomModule( Owner)
      else result := nil;
    end;


    function    TKnobsTextLabel.GetModuleName: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


    procedure   TKnobsTextLabel.SetIsDisplay( aValue: Boolean);
    begin
      if aValue <> FIsDisplay
      then begin
        FIsDisplay := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsTextLabel.SetDisplayColor ( aValue: TColor);
    begin
      if aValue <> FDisplayColor
      then begin
        FDisplayColor := aValue;
        Invalidate;
      end;
    end;


//  protected

    procedure   TKnobsTextLabel.SetName( const aNewName: TComponentName); // override;
    begin
      inherited;
      FShortName := ParsePostfix( Name);
    end;


    procedure   TKnobsTextLabel.DoDrawText( var aRect: TRect; aFlags: Longint); // override;
    var
      Text: string;
    begin
      Text   := GetLabelText;
      aFlags := aFlags or DT_NOPREFIX;
      aFlags := DrawTextBiDiModeFlags( aFlags);
      Canvas.Font := Font;

      if IsDisplay
      then begin
        Canvas.Font.Color := DisplayColor;
        Canvas.Font.Style := [ fsBold];
      end;

      DrawText( Canvas.Handle, PChar( Text), Length( Text), aRect, aFlags);
    end;


//  public

    constructor TKnobsTextLabel.Create( anOwner: TComponent); // override;
    begin
      inherited;

      with Font
      do begin
        Name := 'Small Fonts';
        Size := 7;
      end;

      ShowHint       := False;
      ParentShowHint := False;
      Enabled        := False;
      Transparent    := True;
      IsDisplay      := False;
      DisplayColor   := clWhite;
    end;


{ ========
  TKnobsLabelEditor = class( TEdit)
  // An editor for TEditLabel
  private
    FLabel : TKnobsEditLabel;
  private
}

    procedure   TKnobsLabelEditor.WMKillFocus( var aMessage: TWMKillFocus); //  message WM_KILLFOCUS;
    begin
      inherited;
      Visible := False;
      PopupEditorHide( Self);
    end;


    procedure   TKnobsLabelEditor.CNKeyDown( var Message: TWMKeyDown); // message CN_KEYDOWN;
    // Copied from TWinControl.CNKeyDown - but left out the check for IsShortcut
    // as that one was 'stealing away' the menu shortcut keys resulting in the DEL
    // key (VK_DELETE) not working for editing the editor's text. Now this control
    // behaves like there are no menu shortcuts at all, and that is what is wanted
    // here.
    var
      Mask: Integer;
    begin
      with Message
      do begin
        Result := 1;
        UpdateUIState( Message.CharCode);

        if   not ( csDesigning in ComponentState)
        then begin
          if   Perform( CM_CHILDKEY, CharCode, Winapi.Windows.LPARAM( Self)) <> 0
          then Exit;

          Mask := 0;

          case CharCode of
            VK_TAB                                      : Mask := DLGC_WANTTAB;
            VK_LEFT, VK_RIGHT, VK_UP, VK_DOWN           : Mask := DLGC_WANTARROWS;
            VK_RETURN, VK_EXECUTE, VK_ESCAPE, VK_CANCEL : Mask := DLGC_WANTALLKEYS;
          end;
          if
            ( Mask <> 0)                                    and
            ( Perform( CM_WANTSPECIALKEY, CharCode, 0) = 0) and
            ( Perform( WM_GETDLGCODE, 0, 0) and Mask = 0)   and
            ( GetParentForm( Self).Perform( CM_DIALOGKEY, CharCode, KeyData) <> 0)
          then Exit;
        end;

        Result := 0;
      end;
    end;


    procedure   TKnobsLabelEditor.Do_Exit( aSender: TObject);
    begin
      Visible := False;
      PopupEditorHide( Self);
    end;


    procedure   TKnobsLabelEditor.Do_KeyDown( aSender: TObject; var aKey: Word; aShift: TShiftState);
    begin
      if   aShift = []
      then begin
        case aKey Of

          VK_RETURN :

            begin
              FLabel.SetText( Text);
              Visible := False;
              PopupEditorHide( Self);
              aKey    := 0;
            end;

          VK_ESCAPE :

            begin
              Visible := False;
              PopupEditorHide( Self);
              aKey    := 0;
            end;

        end;
      end;
    end;


//  protected

    procedure   TKnobsLabelEditor.SetLabel( aLabel: TKnobsEditLabel);
    begin
      FLabel  := aLabel;
      Parent  := aLabel.Parent;
      Text    := aLabel.Caption;
      Visible := True;
      PopupEditorShow( Self);
      SetFocus;
    end;


//  public

    constructor TKnobsLabelEditor.Create( anOwner: TComponent); // override;
    begin
      if   Assigned( GLabelEditor)
      then Exit;

      inherited;
      ControlStyle := ControlStyle + [ csOpaque];
      Visible      := False;
      Left         := 1;
      Top          := 1;
      OnExit       := Do_Exit;
      OnKeyDown    := Do_KeyDown;
      GLabelEditor := Self;
    end;


    destructor  TKnobsLabelEditor.Destroy; // override;
    begin
      if   Assigned( Owner)
      then Owner.RemoveComponent( Self);

      PopupEditorHide( Self);
      GLabelEditor := nil;
      inherited;
    end;

{ ========
  TKnobsChoiceSelector = class( TComboBox)
  private
    FChoice : TKnobsSelectorChoice;
  private
}

    procedure   TKnobsChoiceSelector.WMKillFocus( var aMessage: TWMKillFocus ); // message WM_KILLFOCUS;
    begin
      inherited;
      Visible := False;
      PopupEditorHide( Self);
    end;


    procedure   TKnobsChoiceSelector.Do_Exit( aSender: TObject);
    begin
      Visible := False;
      PopupEditorHide( Self);
    end;


//  protected

    procedure   TKnobsChoiceSelector.SetSelectorChoise( aChoice: TKnobsSelectorChoice);
    begin
      FChoice := aChoice;
      Parent  := aChoice.Parent;
      Items.Assign( aChoice.Names);

      // todo : fix the selection

      Visible := True;
      PopupEditorShow( Self);
      SetFocus;
    end;


//  public

    constructor TKnobsChoiceSelector.Create( anOwner: TComponent); // override;
    begin
      if   Assigned( GChoiceSelector)
      then Exit;

      inherited;
      ControlStyle    := ControlStyle + [ csOpaque];
      Visible         := False;
      Left            := 1;
      Top             := 1;
      OnExit          := Do_Exit;
      GChoiceSelector := Self;
    end;


    destructor  TKnobsChoiceSelector.Destroy; // override;
    begin
      if   Assigned( Owner)
      then Owner.RemoveComponent( Self);

      GChoiceSelector := nil;
      PopupEditorHide( Self);
      inherited;
    end;


{ ========
  TKnobsEditLabel = class( TKnobsTextLabel)
  // a label with a popup editor so it's caption can be edited.
  private
    FOnChanged : TKnobsOnEditLabelChanged;
  public
    property    Module: TKnobsCustomModule read GetModule;
  published
    property    OnChanged: TKnobsOnEditLabelChanged read FOnChanged write FOnChanged;
  private
}

    function    TKnobsEditLabel.GetModule: TKnobsCustomModule;
    begin
      if   Parent is TKnobsCustomModule
      then Result := TKnobsCustomModule( Parent)
      else Result := nil;
    end;


//  protected

    procedure   TKnobsEditLabel.CMHintShow(var Message: TCMHintShow); // message CM_HINTSHOW;
    var
      M : TKnobsCustomModule;
    begin
      if   not ( csDesigning in ComponentState)
      then begin
        M := Module;

        if   Assigned( M)
        then Hint := Format( '%s [%s]', [ M.Title, M.FriendlyName], AppLocale);
      end;

      inherited;
    end;


    procedure   TKnobsEditLabel.SetText( const aValue: string);
    begin
      Caption := aValue;

      if   Assigned( Module)
      then Module.Title := aValue;

      if   Assigned( FOnChanged)
      then FOnChanged( Self, aValue);
    end;


    procedure   TKnobsEditLabel.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;

    // We only need the double click here to start a label editor, all other
    // events should be handled by the containing module.

    var
      aModule : TKnobsCustomModule;
    begin
      if   not ( csDesigning in ComponentState)
      then begin
        if   ( ssDouble in Shift)
        and  ( Button = mbLeft)
        then begin

          // Handle the double click

          NeedsLabelEditor;
          GLabelEditor.SetLabel( Self);
        end
        else begin

          // Pass it on

          aModule := Module;

          if   Assigned( aModule)
          then begin
            aModule.MouseCapture := True;
            aModule.MouseDown( Button, Shift, X + Left, Y + Top);
          end;
        end;
      end;
    end;


//  public

    constructor TKnobsEditLabel.Create( anOwner: TComponent); // override;
    begin
      inherited;
      AutoSize       := False;
      Width          := 75;
      Enabled        := True;
      ShowHint       := True;
      ParentShowHint := True;
    end;


{ ========
  TKnobsBox = class( TBevel)
  // A disablable bevel
  private
    property    ModuleName    : string         read GetModuleName;
  published
    property    Enabled                                                                                   default False;
    property    OnClick;
  private
}

    function    TKnobsBox.GetModule: TKnobsCustomModule;
    begin
      if   Owner is TKnobsCustomModule
      then result := TKnobsCustomModule( Owner)
      else result := nil;
    end;


//  private

    function    TKnobsBox.GetModuleName: string;
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := GetModule;

      if   Assigned( aModule)
      then Result := aModule.Name
      else Result := '';
    end;


//  public

    constructor TKnobsBox.Create( anOwner: TComponent); // override;
    begin
      inherited;
      Enabled := False;
    end;


{
  TKnobsWire = class
  // Wire, visual aspect of a connection
  private
    FColor        : TColor;
    FSource       : TKnobsConnector;
    FDestination  : TKnobsConnector;
    FHighlight    : Boolean;
    FBezierParams : TKnobsBezierParams;
  public
    property    Source          : TKnobsConnector    read FSource;
    property    Destination     : TKnobsConnector    read FDestination;
    property    SourceName      : string        read GetSourceName;
    property    DestinationName : string        read GetDestinationName;
    property    BezierParams    : TKnobsBezierParams read FBezierParams;
    property    FullName        : string        read GetFullName;
  private
}

    function    TKnobsWire.GetSourceName: string;
    begin
      Result := ConnectorName( Source.FindOutput); // Trace to ultimate source (an output)
    end;


    function    TKnobsWire.GetDestinationName: string;
    begin
      Result := ConnectorName( Destination); // Use factual destination
    end;


    function    TKnobsWire.GetSourceModule: TKnobsCustomModule;
    var
      aFinal : TKnobsConnector;
    begin
      Result := nil;

      if   Assigned( Source)
      then begin
        aFinal := Source.FindOutput; // Trace to ultimate source (an output)

        if   Assigned( aFinal)
        then Result := aFinal.Module;
      end;
    end;


    function    TKnobsWire.GetDestinationModule: TKnobsCustomModule;
    begin
      Result := nil;

      if   Assigned( Destination)
      then Result := Destination.Module; // Use factual destination
    end;


    function    TKnobsWire.GetFullName: string;
    begin
      Result := Format( '%s -> %s', [ ConnectorName( Source), ConnectorName( Destination)], AppLocale);
    end;


//  public

    constructor TKnobsWire.Create( aSource, aDestination: TKnobsConnector);
    var
      anOutput: TKnobsConnector;
    begin
      Assert( Assigned( aSource) and Assigned( aDestination));
      inherited Create;
      FSource      := aSource;
      FDestination := aDestination;
      anOutput     := aSource.FindOutput;

      if   Assigned( anOutput)
      then FColor := anOutput.WireColor
      else FColor := ColorUnknown;

      Wiggle;
    end;


    procedure   TKnobsWire.Wiggle;
    begin
      with FBezierParams
      do begin
        Angle1    := 15 + Random( 21);
        Strength1 := 20 + Random( 21);
        Angle2    := 20 + Random( 21);
        Strength2 := 20 + Random( 21);
      end;
    end;


    procedure   TKnobsWire.Dump( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, Format( 'wire : %s', [ WireName( Source, Destination)], AppLocale));
    end;


    procedure   TKnobsWire.FixWire;
    // When the wire is input to input fix it such such that the Source connector is closest
    // to the originating output connector. When there is no output in the chain  just
    // do something arbitrary - which must then be fixed later once an output gets to
    // be connected. Also fix rate smartness here.
    begin
      Source     .FixSpeedUp;
      Destination.FixSpeedUp;

      if   ( Source      is TKnobsInput)
      and  ( Destination is TKnobsInput)
      then begin
        if   Source.DistanceToOutput > Destination.DistanceToOutput
        then SwapConnectors;
      end;
    end;


    procedure   TKnobsWire.SwapConnectors;
    var
      aTmp: TKnobsConnector;
    begin
      aTmp         := FSource;
      FSource      := FDestination;
      FDestination := aTmp;
    end;


    function    TKnobsWire.Involves( aModule: TKnobsCustomModule): Boolean;
    begin
      Result := ( SourceModule = aModule) or ( DestinationModule = aModule);
    end;


    function    TKnobsWire.AsString: string;
    begin
      Result :=
        Format(
          'wire( class = ''%s'' src = ''%s.%s'' dst = ''%s.%s'')',
          [
            ClassName,
            Source     .ModuleName,
            Source     .Name,
            Destination.ModuleName,
            Destination.Name
          ]
        )
    end;


    function    TKnobsWire.AsCompactString: string;
    begin
      Result :=
        Format(
          '%s.%s %s.%s',
          [
            Source     .ModuleName,
            Source     .Name,
            Destination.ModuleName,
            Destination.Name
          ]
        )
    end;


{ ========
  TKnobsWireOverlay = class( TCustomControl)
  //
  // Responsibilty to maintain connections is assumed by the TConnectors
  // themselves. Wires are maintained here, but it's a cache really to ease
  // wire painting. The wires are maintained by the connectors theselves that
  // send their info to us through the WirePanel (by means of UM_CONNEXTIONxx
  // and UM_HIGHLIGHT messages). The WirePanel imforms us by calling the
  // AddWire, DelWire and Highlight methodes.
  //
  // All Wires are owned by FWires, so our destructor will eventually destroy
  // all dangling wires. Nevertheless, TConnector still must, on destruction,
  // clean up the wires it is responsible for .. it wouldn't look good other
  // wise.
  //
  private
    FWires         : TObjectList;
    FCurvedLines   : Boolean;
    FWireThickness : Integer;
    FWiresVisible  : Boolean;
    FBlockCount    : Integer;
    FPaintCount    : Integer;
    FPaintClocks   : Int64;
  public
    property    Wire       [ anIndex: Integer]: TKnobsWire      read GetWire;
    property    Source     [ anIndex: Integer]: TKnobsConnector read GetSource;
    property    Destination[ anIndex: Integer]: TKnobsConnector read GetDestination;
  public
    property    CurvedLines      : Boolean read FCurvedLines      write SetCurvedLines    default True;
    property    WireThickness    : Integer read FWireThickness    write SetWireThickness  nodefault;
    property    BlockCount       : Integer read FBlockCount;
    property    PaintCount       : Integer read FPaintCount;
    property    PaintClocks      : Int64   read FPaintClocks;
  published
    property    Left   stored False;
    property    Top    stored False;
    property    Width  stored False;
    property    Height stored False;
  private
}

    procedure   TKnobsWireOverlay.WMNCHitTest( var aMessage: TWMNCHitTest); // message WM_NCHITTESTq;
    begin
      if   csDesigning in ComponentState
      then inherited
      else aMessage.Result := HTTRANSPARENT;
    end;


    function    TKnobsWireOverlay.GetWire( anIndex: Integer): TKnobsWire;
    begin
      Result := TKnobsWire( FWires[ anIndex]);
    end;


    function    TKnobsWireOverlay.GetSource( anIndex: Integer): TKnobsConnector;
    begin
      Result := Wire[ anIndex].FSource;
    end;


    function    TKnobsWireOverlay.GetDestination( anIndex: Integer): TKnobsConnector;
    begin
      Result := Wire[ anIndex].FDestination;
    end;


    procedure   TKnobsWireOverlay.SetCurvedLines( aValue: Boolean);
    begin
      if   aValue <> FCurvedLines
      then begin
        FCurvedLines := aValue;
        InvalidateWires;
      end;
    end;


    procedure   TKnobsWireOverlay.SetWireThickNess( aValue: Integer);
    begin
      if   aValue <> FWireThickness
      then begin
        FWireThickness := aValue;
        InvalidateWires;
      end;
    end;


    procedure   TKnobsWireOverlay.SetWiresVisible( aValue: Boolean);
    begin
      if   aValue <> FWiresVisible
      then begin
        FWiresVisible := aValue;
        InvalidateWires;
      end;
    end;


//   protected

    procedure   TKnobsWireOverlay.Paint; // override;
    var
      i : Integer;
      O : TKnobsConnector;
    begin
      if   FBlockCount = 0
      then begin
        with Canvas
        do begin
          if   WiresVisible
          then Pen.Width := WireThickness + 1
          else Pen.Width := 1;

          for i := 0 to WireCount - 1
          do begin
            with Wire[ i]
            do begin
              O := FSource.FindOutput;

              if   not Assigned( O)
              then Pen.Color := ColorUnknown
              else if FHighlight
              then Pen.Color := ColorHighlight
              else begin
                if   O.CustomWireColor
                then Pen.Color := O.WireColor
                else Pen.Color := O.EffectiveColor;
              end
            end;

            PaintWire( Handle, Wire[ i]);
          end;
        end;
      end;
    end;


    procedure   TKnobsWireOverlay.PaintWire( aDC: HDC; aWire: TKnobsWire);
    var
      Points : TKnobsBezierPoints;
    begin
      with aWire
      do begin
        if   FSource.HasParent
        and  FDestination.HasParent
        and  ( WireThickness > 0)
        and  WiresVisible
        then begin
          with FSource
          do Points[ 0] := ClientToScreen( CenterPoint);

          Points[ 0] := ScreenToClient( Points[ 0]);

          with FDestination
          do Points[ 3] := ClientToScreen( CenterPoint);

          Points[ 3] := ScreenToClient( Points[ 3]);

          if   CurvedLines
          then begin
            CalcBezierCurve( Points, aWire.BezierParams);
            PolyBezier( aDC, Points, 4);
          end
          else begin
            with Points[ 0]
            do MoveToEx( aDC, X, Y, nil);

            with Points[ 3]
            do LineTo  ( aDC, X, Y     );
          end;
        end;
      end;
    end;


    function    TKnobsWireOverlay.FindWire( aSource, aDestination: TKnobsConnector): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to WireCount - 1
      do begin
        if
          (( Wire[ i].FSource = aSource     ) and ( Wire[ i].FDestination = aDestination)) or
          (( Wire[ i].FSource = aDestination) and ( Wire[ i].FDestination = aSource     ))
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TKnobsWireOverlay.AddWire( aSource, aDestination: TKnobsConnector);
    var
      anIndex      : Integer;
      aWire        : TKnobsWire;
      aDistanceSrc : Integer;
      aDistanceDst : Integer;
    begin
      anIndex := FindWire( aSource, aDestination);

      if   anIndex < 0
      then begin
        if   ( aSource      is TKnobsOutput)
        and  ( aDestination is TKnobsOutput)
        then raise EKnobsShortCircuit.Create( 'output to output connection detected ... should not be there')
        else if ( aSource is TKnobsInput) and ( aDestination is TKnobsOutput)
        then aWire := TKnobsWire.Create( aDestination, aSource)
        else if ( aSource is TKnobsOutput) and ( aDestination is TKnobsInput)
        then aWire := TKnobsWire.Create( aSource, aDestination)
        else begin

          // For input to input connections the source should be closest to originating output.
          // When there is no output the wire might still be added with the connectors in the
          // wrong order, this must be fixed before patch compilation with a call to
          // FixAllWires - we will not do that here as the order is irrelevant for the editor
          // itself. So just add the wire here in an arbitrary way. And also, there may not be
          // an output in the chain.

          aDistanceSrc := aSource     .DistanceToOutput;
          aDistanceDst := aDestination.DistanceToOutput;

          if   aDistanceSrc > aDistanceDst
          then aWire := TKnobsWire.Create( aDestination, aSource     )
          else aWire := TKnobsWire.Create( aSource     , aDestination);
        end;

        FWires.Add( aWire);
        InvalidateWires;
      end;
    end;


    procedure   TKnobsWireOverlay.DelWire( aSource, aDestination: TKnobsConnector);
    var
      anIndex : Integer;
    begin
      anIndex := FindWire( aSource, aDestination);

      if   anIndex >= 0
      then begin
        FWires.Delete( anIndex);
        InvalidateWires;
      end;
    end;


    procedure   TKnobsWireOverlay.Connect( aSource, aDestination: TKnobsConnector);
    begin
      if   Assigned( aSource)
      then aSource.ConnectTo( aDestination);
    end;


    procedure   TKnobsWireOverlay.HandleHighlight;
    var
      i : Integer;
    begin
      for i := 0 to WireCount - 1
      do begin
        with Wire[ i]
        do FHighlight := FSource.HighLight or FDestination.HighLight;
      end;

      Invalidate;
    end;


    procedure   TKnobsWireOverlay.HandleColorChange( aSource: TKnobsConnector; aColor: TColor);
    begin
      InvalidateWires;
    end;


    function    TKnobsWireOverlay.WireCount: Integer;
    begin
      if   Assigned( FWires)
      then Result := FWires.Count
      else Result := 0;
    end;


    procedure   TKnobsWireOverlay.WiggleWires;
    var
      i : Integer;
    begin
      if   CurvedLines
      and  Assigned( FWires)
      then begin
        for i := 0 to WireCount - 1
        do Wire[ i].Wiggle;

        InvalidateWires;
      end;
    end;


    procedure   TKnobsWireOverlay.ToggleWires;
    begin
      WiresVisible := not WiresVisible;
    end;


    function    TKnobsWireOverlay.WiresOff: Boolean;
    begin
      Result       := WiresVisible;
      WiresVisible := False;
    end;


    procedure   TKnobsWireOverlay.WiresOn( TurnOn: Boolean);
    begin
      WiresVisible := TurnOn;
    end;


//   public

    constructor TKnobsWireOverlay.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FBlockCount        := 0;
      FWires             := TObjectList.Create;
      Align              := alClient;
      FCurvedLines       := True;
      FWireThickness     := 1;
      FWiresVisible      := True;
      FWires.OwnsObjects := True;
    end;


    destructor  TKnobsWireOverlay.Destroy; // override;
    begin
      FreeAndnil( FWires);
      inherited;
    end;


    procedure   TKnobsWireOverlay.InvalidateWires;

      function CreateRgn( aDC: HDC; aWire: TKnobsWire): HRGN;
      begin
        BeginPath ( aDC);
        PaintWire ( aDC, aWire);
        EndPath   ( aDC);
        WidenPath ( aDC);
        Result := PathToRegion( aDC);
      end;

    var
      DC       : HDC;
      i        : Integer;
      HRegion1 : HRGN;
      HRegion2 : HRGN;
      SavedPen : HPEN;
      Clocks   : Int64;
    begin
      if   HasParent
      and  Parent.HasParent
      and  ( FBlockCount = 0)
      then begin
        Inc( FPaintCount);
        Clocks := GetCpuClockCycleCount;

        with Canvas
        do begin
          Pen.Width := FWireThickness;
          DC        := CreateCompatibleDC( Handle);
          SavedPen  := SelectObject( DC, Pen.Handle);
        end;

        try
          if   ( WireCount > 0)
          and  ( WireThickness > 0)
          and  WiresVisible
          then begin
            HRegion1 := CreateRgn( DC, Wire[ 0]);

            for i := 1 to WireCount - 1
            do begin
              HRegion2 := CreateRgn( DC, Wire[ i]);
              CombineRgn( HRegion2, HRegion2, HRegion1, RGN_OR);
              DeleteObject( HRegion1);
              HRegion1 := HRegion2;
            end;
          end
          else HRegion1 := CreateRectRgn( 0, 0, 1, 1);

          SetWindowRgn( Handle, HRegion1, True);
          BringToFront; // see other __1 remarks - it will paint behind .. otherwise
        finally
          SelectObject( DC, SavedPen);
          DeleteDC( DC);
        end;

        Invalidate; // And tell windows we want a repaint.
        FPaintClocks := GetCpuClockCycleCount - Clocks;
      end;
    end;


    function    TKnobsWireOverlay.Disconnect( aConnector: TKnobsConnector): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := WireCount - 1 downto 0
      do begin
        with Wire[ i]
        do begin
          if   ( FSource      = aConnector)
          or   ( FDestination = aConnector)
          then begin
            FWires.Delete( i);
            Result := True;
          end;
        end;
      end;
    end;


    procedure   TKnobsWireOverlay.Dump( var aFile: TextFile; anInd: Integer);
    var
      i : Integer;
    begin
      for i := 0 to FWires.Count - 1
      do TKnobsWire( FWires[ i]).Dump( aFile, anInd);
    end;


    procedure   TKnobsWireOverlay.Log( aLogClass: TLogClass; const aMsg: string);
    begin
      if   Parent is TKnobsWirePanel
      then TKnobsWirePanel( Parent).Log( aLogClass, Format( 'wire overlay %s( %s)', [ Name, aMsg], AppLocale));
    end;


    procedure   TKnobsWireOverlay.BlockUpdates;
    begin
      Inc( FBlockCount);
    end;


    procedure   TKnobsWireOverlay.UnBlockUpdates;
    begin
      if   FBlockCount > 0
      then begin
        Dec( FBlockCount);

        if   FBlockCount = 0
        then InvalidateWires;
      end;
    end;


    procedure   TKnobsWireOverlay.FixAllWires;
    var
      i : Integer;
    begin

      // Fix all wires that are input to input such that the Source connector is closest
      // to the originating output connector. When there is no output in the chain  just
      // do something arbitrary - which must then be fixed later once an output gets to
      // be connected. Must be called before a patch is translated into a synth patch.

      for i := 0 to WireCount - 1
      do Wire[ i].FixWire;
    end;



{ =======
  TModuleList = class( TList)
  private
    FonLog     : TKnobsOnLog;
    FUseIndex  : Boolean;
    FIndex     : TStringList;
    FClassList : TClassList;
  public
    property    UseIndex                         : Boolean            read FUseIndex;
    property    Module  [ anIndex: Integer]      : TKnobsCustomModule read GetModule;                           default;
    property    Selected[ anIndex: Integer]      : Boolean            read GetSelected     write SetSelected;
    property    ByName  [ const anIndex: string] : TKnobsCustomModule read GetModuleByName;
    property    AnySelected                      : Boolean            read GetAnySelected;
    property    NoneSelected                     : Boolean            read GetNoneSelected;
    property    OnLog                            : TKnobsOnLog        read FOnLog          write FOnLog;
  private
}

    function    TKnobsModuleList.GetModule( anIndex: Integer): TKnobsCustomModule;
    begin
      Result := TKnobsCustomModule( Items[ anIndex]);
    end;


    function    TKnobsModuleList.GetModuleByName( const anIndex: string ): TKnobsCustomModule;
    var
      i : Integer;
    begin
      Result := nil;

      for i := 0 to Count - 1
      do begin
        if   SameText( anIndex, Module[ i].Name)
        then begin
          Result := Module[ i];
          Break;
        end;
      end;
    end;


    function    TKnobsModuleList.GetSelected( anIndex: Integer): Boolean;
    begin
      Result := Module[ anIndex].Selected;
    end;


    procedure   TKnobsModuleList.SetSelected( anIndex: Integer; aValue: Boolean);
    begin
      Module[ anIndex].Selected := aValue;
    end;


    function    TKnobsModuleList.GetAnySelected: Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to Count - 1
      do begin
        if   Selected[ i]
        then begin
          Result := True;
          Break;
        end;
      end;
    end;


    function    TKnobsModuleList.GetNoneSelected: Boolean;
    var
      i : Integer;
    begin
      Result := True;

      for i := 0 to Count - 1
      do begin
        if   Selected[ i]
        then begin
          Result := False;
          Break;
        end;
      end;
    end;


//  public

    constructor TKnobsModuleList.Create( aUseIndex: Boolean);
    begin
      inherited Create;
      FClassList := TClassList.Create;
      FUSeIndex  := aUseIndex;

      if   FUseIndex
      then begin
        FIndex               := TStringList.Create;
        FIndex.OwnsObjects   := False;
        FIndex.Sorted        := True;
        FIndex.Duplicates    := dupError;
        FIndex.CaseSensitive := False;
      end;
    end;


    destructor  TKnobsModuleList.Destroy; // override;
    begin
      inherited;

      if   UseIndex
      then FreeAndNil( FIndex);

      FreeAndNil( FClassList);
    end;


    procedure   TKnobsModuleList.BuildIndex;
    var
      i : Integer;
      M : TKnobsCustomModule;
    begin
      if   UseIndex
      then begin
        FIndex.Clear;

        for i := 0 to Count - 1
        do begin
          M := Module[ i];
          FIndex.AddObject( M.Name, M);
        end;
      end;
    end;


    function    TKnobsModuleList.FindModule( const aModule: TKnobsCustomModule): Integer; // overload;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to Count - 1
      do begin
        if   Module[ i] = aModule
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TKnobsModuleList.FindModule( const aName: string): TKnobsCustomModule; // overload;
    var
      anIndex : Integer;
      i       : Integer;
    begin
      Result  := nil;
      anIndex := -1;

      if   UseIndex
      then anIndex := FIndex.IndexOf( aName)
      else begin
        for i := 0 to Count - 1
        do begin
          if   SameText( Module[ i].Name, aName)
          then begin
            anIndex := i;
            Break;
          end;
        end;
      end;

      if   anIndex >= 0
      then Result  := TKnobsCustomModule( FIndex.Objects[ anIndex]);
    end;


    procedure   TKnobsModuleList.FreeModule( anIndex: Integer; const aCallback: TKnobsOnValuedCtrlRemoved); // overload;
    var
      aModule : TKnobsCustomModule;
      i       : Integer;
    begin
      aModule := Module[ anIndex];

      if   Assigned( aModule)
      then begin
        for i := 0 to aModule.ControlCount - 1
        do begin
          if   ( aModule.Controls[ i] is TKnobsValuedControl)
          and  Assigned( aCallback)
          then aCallback( TKnobsValuedControl( aModule.Controls[ i]));
        end;
      end;

      Delete( anIndex);
      aModule.DisposeOf;
    end;


    procedure   TKnobsModuleList.FreeModule( var aModule: TKnobsCustomModule; const aCallback: TKnobsOnValuedCtrlRemoved); // overload;
    var
      anIndex : Integer;
    begin
      anIndex := FindModule( aModule);

      if   anIndex >= 0
      then FreeModule( anIndex, aCallback);

      aModule := nil;
    end;


    function    TKnobsModuleList.FreeModules( const aCallback: TKnobsOnValuedCtrlRemoved): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := Count - 1 downto 0
      do begin
        FreeModule( i, aCallback);
        Result := True;
      end;
    end;


    function    TKnobsModuleList.FreeSelectedModules( const aCallback: TKnobsOnValuedCtrlRemoved): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := Count - 1 downto 0
      do begin
        if   Selected[ i]
        then begin
          FreeModule( i, aCallback);
          Result := True;
        end;
      end;
    end;


    procedure   TKnobsModuleList.SelectAll;
    var
      i : Integer;
    begin
      for i := Count - 1 downto 0
      do Selected[ i] := True;
    end;


    procedure   TKnobsModuleList.UnselectAll;
    var
      i : Integer;
    begin
      for i := Count - 1 downto 0
      do Selected[ i] := False;
    end;


    procedure   TKnobsModuleList.SelectUnique( aModule: TKnobsCustomModule);
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do Module[ i].Selected := Module[ i] = aModule;
    end;


    procedure   TKnobsModuleList.FixSelection( aModule: TKnobsCustomModule);
    begin

      // When no modules are selected select aModule uniquely

      if   Assigned( aModule) and NoneSelected
      then SelectUnique( aModule);
    end;


    procedure   TKnobsModuleList.SetSelectedModuleColor( aValue: TColor);
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do begin
        if   Selected[ i]
        then Module[ i].Color := aValue;
      end;
    end;


    procedure   TKnobsModuleList.SetSelectedModuleColorDefault( const aColorFunc: TOnGetModuleColor);
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do begin
        if   Selected[ i]
        then Module[ i].Color := aColorFunc( Self, Module[ i].ModuleType);
      end;
    end;


    procedure   TKnobsModuleList.InvertSelection;
    var
      i : Integer;
    begin
      for i := Count - 1 downto 0
      do Selected[ i] := not Selected[ i];
    end;


    procedure   TKnobsModuleList.InvalidateDisplays;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do Module[ i].InvalidateDisplays;
    end;


//  public

    // Some helpers for selector population and component registration

    function    TKnobsModuleList.FindType( aModuleType: TKnobsModuleType): TKnobsCustomModule;
    var
      i : Integer;
    begin

      // Finds the first enabled module of the ModuleType specified

      Result := nil;

      for i := 0 to Count - 1
      do begin
        if   ( Module[ i].ModuleType = aModuleType)
        and  ( Module[ i]).Enabled
        then begin
          Result := Module[ i];
          Break;
        end;
      end;
    end;


    function TabIndexSorter( Item1, Item2: Pointer): Integer;

      function CompareModules( m1, m2: TKnobsCustomModule): Integer;
      begin
        if   m1.TabOrder > m2.TabOrder
        then Result := 1
        else if m1.TabOrder < m2.TabOrder
        then Result := -1
        else Result := 0;
      end;

    var
      m1 : TKnobsCustomModule;
      m2 : TKnobsCustomModule;
      t1 : TTabSheet;
      t2 : TTabSheet;
    begin
      m1 := TKnobsCustomModule( Item1);
      m2 := TKnobsCustomModule( Item2);
      t1 := nil;

      if   Assigned( m1.Parent)
      and  ( m1.Parent is TTabSheet)
      then t1 := TTabSheet( m1.Parent);

      t2 := nil;

      if   Assigned( m2.Parent)
      and  ( m2.Parent is TTabSheet)
      then t2 := TTabSheet( m2.Parent);

      if   not Assigned( t1)
      or   not Assigned( t2)
      then Result := CompareModules( m1, m2)
      else begin
        if   t1.TabIndex > t2.TabIndex
        then Result := 1
        else if t1.TabIndex < t2.TabIndex
        then Result := -1
        else Result := CompareModules( m1, m2);
      end;
    end;


    procedure   TKnobsModuleList.PopulateSelector( aModuleSelector: TKnobsModuleSelector; const aBitmapsFolder: string);
    var
      i : Integer;
    begin

      // Populate a module selector using all the modules contained in the list

      UnPopulateSelector( aModuleSelector);
      Sort( TabIndexSorter);

      if   Assigned( aModuleSelector)
      then begin
        for i := 0 to Count - 1
        do begin
          if   Module[ i].Enabled
          then begin
            with Module[ i]
            do aModuleSelector.RegisterModuleType( PageName, ModuleType, aBitmapsFolder, Title);
          end;
        end;
      end;
    end;


    procedure   TKnobsModuleList.UnPopulateSelector( aModuleSelector: TKnobsModuleSelector);
    begin
      if   Assigned( aModuleSelector)
      then aModuleSelector.UnregisterAllTypes;
    end;


    function    TKnobsModuleList.CreateModule( aWirePanel: TKnobsWirePanel; aModuleType: TKnobsModuleType; MustDrag, StandardColor: Boolean): TKnobsCustomModule;
    var
      aModule : TKnobsCustomModule;
    begin

      // Factory method, create a clone from the specified module type when an
      // enabled module of that type can be found in the list.
      // This one is called from the module selector only, it should set
      // the one and only module into drag mode. See __2 for other cases
      // (also in KnobParser)
      // See Clone method .. that one fixes it

      Result  := nil;
      aModule := FindType( aModuleType);

      if   Assigned( aModule)
      then begin
        Result := aModule.Clone( aWirepanel, aWirePanel, Point( 0, 0), MustDrag, False);

        if   Assigned( Result)
        then begin
          Result.Opacity                := aWirePanel.ModuleOpacity;
          Result.Flat                   := aWirePanel.ModuleFlatness;
          Result.Texture                := aWirePanel.ModuleTexture;
          Result.WheelSupportOnKnobs    := aWirePanel.WheelSupportOnKnobs;
          Result.WheelSensitivity       := aWirePanel.WheelSensitivity;
          Result.ControlMode            := aWirePanel.ControlMode;
          Result.ShowAllowRandomization := aWirePanel.ShowAllowRandomization;
          Result.AllowRandomization     := False;

          if   not StandardColor
          then Result.Color := aWirePanel.ModuleColor;
        end;
      end;
    end;


    function    TKnobsModuleList.ChangeModule( aWirePanel: TKnobsWirePanel; var aTarget: TKnobsCustomModule; aWantedType: TKnobsModuleType; aCallBack: TKnobsOnValuedCtrlRemoved): TKnobsCustomModule;
    var
      aModule : TKnobsCustomModule;
      aTop    : Integer;
      aLeft   : Integer;
    begin
      Result  := nil;

      if   Assigned( aTarget)
      then begin
        aModule := FindType( aWantedType);
        aLeft   := aTarget.Left;
        aTop    := aTarget.Top;

        if   Assigned( aModule)
        then begin
          Result := aModule.Clone( aWirepanel, aWirePanel, Point( aLeft, aTop), NO_DRAG, False);

          if   Assigned( Result)
          then begin
            Result.Opacity                := aTarget.Opacity;
            Result.WheelSupportOnKnobs    := aTarget.WheelSupportOnKnobs;
            Result.WheelSensitivity       := aTarget.WheelSensitivity;
            Result.ControlMode            := aTarget.ControlMode;
            Result.AllowRandomization     := aTarget.AllowRandomization;
            Result.ShowAllowRandomization := aTarget.ShowAllowRandomization;
          end;
        end;

        aWirePanel.FreeModule( aTarget, aCallBack);
      end;
    end;


    procedure   TKnobsModuleList.CollectModuleComponents( const aModule: TKnobsCustomModule);
    var
      p : Integer;
      i : Integer;
    begin
      if   Assigned( aModule)
      then begin
        p := FClassList.IndexOf( aModule.ClassType);

        if   p < 0
        then FClassList.Add( aModule.ClassType);

        // Register component classes for all components direcly parented by
        // the module.

        for i := 0 to aModule.ControlCount - 1
        do begin
          if   aModule.Controls[ i] is TPersistent
          then begin
            p := FClassList.IndexOf( aModule.Controls[ i].ClassType);

            if   p < 0
            then FClassList.Add( aModule.Controls[ i].ClassType);
          end;
        end;
      end;
    end;


    procedure   TKnobsModuleList.RegisterClassList( doRegister: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to FClassList.Count - 1
      do begin
        if   doRegister
        then Registerclass  ( TPersistentClass( FClassList[ i]))
        else UnRegisterClass( TPersistentClass( FClassList[ i]));
      end;
    end;


    procedure   TKnobsModuleList.RegisterComponents( doRegister: Boolean);
    var
      i : Integer;
    begin

      // Register or unregister all module classes and all component's classes
      // used on all modules with Delphi so that they can be used with the
      // component streaming mechanism.

      for i := 0 to Count - 1
      do CollectModuleComponents( Module[ i]);

      RegisterClassList( doRegister);
    end;


    procedure   TKnobsModuleList.Dump( var aFile: TextFile; anInd: Integer);
    var
      i : Integer;
    begin
      WriteLnInd( afile, anInd, '.');

      for i := 0 to Count - 1
      do Module[ i].Dump( aFile, anInd);

      WriteLnInd( afile, anInd, '.');
    end;


    procedure   TKnobsModuleList.Log( aLogClass: TLogClass; const aMsg: string);
    begin
      if   Assigned( FOnLog)
      then FOnLog( Self, aLogClass, Format( 'TKnobsModuleList( %s)', [ aMsg], AppLocale));
    end;


{ ========
  TKnobsHistoryItemKind = (
    hikPatch      ,
    hikSignal     ,
    hikStringValue
  );

  TKnobsHistoryItem = record
  private
    FKind  : TKnobsHistoryItemKind;
    FValue : TValue;
  public
}

    procedure   TKnobsHistoryItem.RetrievalError( WantedType: TKnobsHistoryItemKind);
    begin
      raise EKnobsHistoryTypeError.Create( 'Type mismatch');
    end;


    procedure   TKnobsHistoryItem.InitAsPatch( const aValue : TKnobsWirePanel);
    begin
      if   Assigned( aValue)
      then begin
        FKind  := hikPatch;
        FValue := aValue.PatchWriter.WriteString( aValue, '', wmAll)
      end;
    end;


    procedure   TKnobsHistoryItem.InitAsSignal( const aValue : TSignal);
    begin
      FKind  := hikSignal;
      FValue := aValue;
    end;


    procedure   TKnobsHistoryItem.InitAsStringValue( const aValue : string);
    begin
      FKind  := hikStringValue;
      FValue := aValue;
    end;


    function    TKnobsHistoryItem.Kind: TKnobsHistoryItemKind;
    begin
      Result := FKind;
    end;


    procedure   TKnobsHistoryItem.GetAsPatch( const aValue: TKnobsWirePanel);
    begin
      if   Assigned( aValue)
      then begin
        if   IsKind( hikPatch)
        then aValue.PatchReader.ReadString( FValue.AsString, aValue, Point( 0, 0), True, False, True, True, rmReplace, '')
        else RetrievalError( hikPatch);
      end;
    end;


    function    TKnobsHistoryItem.GetAsSignal: TSignal;
    begin
      Result := UNDEFINED;

      if   IsKind( hikSignal)
      then Result := FValue.AsExtended
      else RetrievalError( hikSignal);
    end;


    function    TKnobsHistoryItem.GetAsStringValue: string;
    begin
      Result := '';

      if   IsKind( hikStringValue)
      then Result := FValue.AsString
      else RetrievalError( hikStringValue);
    end;


    function    TKnobsHistoryItem.IsKind( aValue: TKnobsHistoryItemKind): Boolean;
    begin
      Result := aValue = Kind;
    end;


    function    TKnobsHistoryItem.IsSameAs( const aHistoryItem: TKnobsHistoryItem): Boolean;
    begin
      Result := aHistoryItem.IsKind( Kind);

      if   Result
      then begin
        if   IsKind( hikSignal)
        then Result := FValue.AsExtended = aHistoryItem.FValue.AsExtended
        else Result := FValue.AsString   = aHistoryItem.FValue.AsString;
      end;
    end;


    function    TKnobsHistoryItem.Description: string;
    begin
      case Kind of
        hikPatch       : Result := 'patch change';
        hikSignal      : Result := FValue.AsString;
        hikStringValue : Result := FValue.AsString;
      end;
    end;


{ ========
  TKnobsHistoryStack = class( TList<TKnobsHistoryItem>)
  private
    FMaxHistory : Integer;
  public
    property    MaxHistory : Integer read FMaxHistory write SetMaxHistory;
  private
}

    procedure   TKnobsHistoryStack.SetMaxHistory( aValue: Integer);
    begin
      if   aValue <> FMaxHistory
      then begin
        FMaxHistory := aValue;

        while Count > MaxHistory
        do Delete( 0);
      end;
    end;


//  public

    procedure   TKnobsHistoryStack.PushPatch( const aWirePanel: TKnobsWirePanel);
    var
      anItem: TKnobsHistoryItem;
    begin
      if   Assigned( aWirePanel)
      then begin
        anItem.InitAsPatch( aWirePanel);

        if   ( Count = 0)
        or   not anItem.IsSameAs( Last)
        then begin
          Add( anItem);

          while Count > MaxHistory
          do Delete( 0);
        end;
      end;
    end;


    procedure   TKnobsHistoryStack.PopPatch( const aWirePanel: TKnobsWirePanel);
    begin
      if   Count > 0
      then begin
        Last.GetAsPatch( aWirePanel);
        Delete( Count - 1);
      end;
    end;


    procedure   TKnobsHistoryStack.PushString( const aValue: string);
    var
      anItem: TKnobsHistoryItem;
    begin
      anItem.InitAsStringValue( aValue);

      if   ( Count = 0)
      or  not anItem.IsSameAs( Last)
      then begin
        Add( anItem);

        while Count > MaxHistory
        do Delete( 0);
      end;
    end;


    function    TKnobsHistoryStack.PopString: string;
    begin
      Result := '';

      if   Count > 0
      then begin
        Result := Last.GetAsStringValue;
        Delete( Count - 1);
      end;
    end;


    procedure   TKnobsHistoryStack.PushSignal( aSignal: TSignal);
    var
      anItem: TKnobsHistoryItem;
    begin
      anItem.InitAsSignal( aSignal);

      if   ( Count = 0)
      or   not anItem.IsSameAs( Last)
      then begin
        Add( anItem);

        while Count > MaxHistory
        do Delete( 0);
      end;
    end;


    function    TKnobsHistoryStack.PopSignal : TSignal;
    begin
      Result := UNDEFINED;

      if   Count > 0
      then begin
        Result := Last.GetAsSignal;
        Delete( Count - 1);
      end;
    end;


    function    TKnobsHistoryStack.CreateDescription( aMaxCount: Integer): TStringList;
    var
      i : Integer;
    begin
      Result := TStringList.Create;

      if   aMaxCount > 0
      then begin
        for i := Count - 1 downto 0
        do begin
          Result.Add( List[ i].Description);
          Dec( aMaxCount);

          if   aMaxCount <= 0
          then Break;
        end;
      end;
    end;


{ ========
  TKnobsUndoRedoHistory = class( TObject)
  private
    FOnHistoryChange : TKnobsOnHistoryChange;
    FMaxHistory      : Integer;
    FLockCount       : Integer;
    FUndos           : TKnobsHistoryStack;
    FRedos           : TKnobsHistoryStack;
  public
    property    Locked          : Boolean               read GetLocked;
    property    UndoCount       : Integer               read GetUndoCount;
    property    RedoCount       : Integer               read GetRedoCount;
    property    MaxHistory      : Integer               read FMaxHistory      write SetmaxHistory;
  public
    property    OnHistoryChange : TKnobsOnHistoryChange read FOnHistoryChange write FOnHistoryChange;
  private
}

    function    TKnobsUndoRedoHistory.GetLocked: Boolean;
    begin
      Result := FLockCount > 0;
    end;


    function    TKnobsUndoRedoHistory.GetUndoCount: Integer;
    begin
      Result := FUndos.Count;
    end;


    function    TKnobsUndoRedoHistory.GetRedoCount: Integer;
    begin
      Result := FRedos.Count;
    end;


    procedure   TKnobsUndoRedoHistory.SetmaxHistory( aValue: Integer);
    begin
      if   aValue <> FMaxHistory
      then begin
        FMaxHistory := aValue;
        FUndos.MaxHistory := aValue;
        FRedos.MaxHistory := aValue;
      end;
    end;


//  private

    procedure   TKnobsUndoRedoHistory.Lock;
    begin
      Inc( FLockCount);
    end;


    function    TKnobsUndoRedoHistory.UnLock: Boolean;
    begin
      if   FLockCount > 0
      then Dec( FLockCount);

      Result := FLockCount = 0;
    end;


//  public

    constructor TKnobsUndoRedoHistory.Create;
    begin
      inherited Create;
      FMaxHistory       := 1;
      FUndos            := TKnobsHistoryStack.Create;
      FRedos            := TKnobsHistoryStack.Create;
      FUndos.MaxHistory := 1;
      FRedos.MaxHistory := 1;
    end;


    destructor  TKnobsUndoRedoHistory.Destroy; // override;
    begin
      FreeAndNil( FUndos);
      FreeAndNil( FRedos);
      inherited;
    end;


    procedure   TKnobsUndoRedoHistory.HistoryChanged;
    begin
      if   Assigned( FOnHistoryChange)
      then FOnHistoryChange( Self, UndoCount, RedoCount, FSavedMarker);
    end;


    procedure   TKnobsUndoRedoHistory.SaveState( const aWirePanel: TKnobsWirePanel);
    begin
      if   not Locked
      then begin
        FRedos.Clear;
        FUndos.PushPatch( aWirePanel);
        HistoryChanged;
      end;
    end;


    procedure   TKnobsUndoRedoHistory.Undo( const aWirePanel: TKnobsWirePanel);
    begin
      if   UndoCount > 0
      then begin
        FRedos.PushPatch( aWirePanel);

        try
          Lock;                    // Prevent undo from changing history
          FUndos.PopPatch( aWirePanel);
        finally
          UnLock;
        end;

        HistoryChanged;
      end;
    end;


    procedure   TKnobsUndoRedoHistory.Redo( const aWirePanel: TKnobsWirePanel);
    begin
      if   RedoCount > 0
      then begin
        FUndos.PushPatch( aWirePanel);

        try
          Lock;                    // Prevent redo from changing history
          FRedos.PopPatch( aWirePanel);
        finally
          UnLock;
        end;

        HistoryChanged;
      end;
    end;


    procedure   TKnobsUndoRedoHistory.Clear;
    begin
      FUndos.Clear;
      FRedos.Clear;
      FSavedMarker := 0;
      HistoryChanged;
    end;


    procedure   TKnobsUndoRedoHistory.SetSavedMarker;
    begin
      if   UndoCount <> FSavedMarker
      then begin
        FSavedMarker := UndoCount;
        HistoryChanged;
      end;
    end;


{ ========
  TKnobsWirePanel = class( TScrollingWinControl)
  // Contains the modules and the wire overlay - scrollable.
  private
    FEditHistory               : TKnobsEditHistory;       // A small local undo history
    FGuid                      : TGUID;
    FOffsetLeft                : Integer;
    FOffsetTop                 : Integer;
    FTitle                     : string;
    FVersion                   : Integer;
    FMouseDown                 : Boolean;
    FWireOverlay               : TKnobsWireOverlay;
    FModules                   : TKnobsModuleList;
    FFilename                  : string;
    FSnapX                     : Integer;
    FSnapY                     : Integer;
    FControlMode               : TDistanceMode;
    FWheelSupportOnKnobs       : Boolean;
    FWheelSensitivity          : Integer;
    FModuleOpacity             : Byte;
    FModuleFlatness            : Boolean;
    FModuleTexture             : Boolean;
    FUseTitleHints             : Boolean;
    FUseConnectorBorders       : Boolean;
    FUseTypedConnectorBorders  : Boolean;
    FModuleColor               : TColor;
    FConnectorBorderColor      : TColor;
    FSelectorColor             : TColor;
    FSelectorBorderColor       : TColor;
    FSelectorBorderColorSingle : TColor;
    FDisplayColor              : TColor;
    FDisplayBorderColor        : TColor;
    FKnobMIDIColor             : TColor;
    FKnobFocusColor            : TColor;
    FKnobMIDIFocusColor        : TColor;
    FViewerColor               : TColor;
    FViewerBorderColor         : TColor;
    FViewerLineColor           : TColor;
    FViewerFillColor           : TColor;
    FIndBarPeakColor           : TColor;
    FIndBarValeyColor          : TColor;
    FSnapActive                : Boolean;
    FAllowRandomization        : Boolean;
    FShowAllowRandomization    : Boolean;
    FWormUpdatesBlocked        : Integer;
    FEditLockCounter           : Integer;
    FPatchReader               : IKnobsPatchReader;
    FPatchWriter               : IKnobsPatchWriter;
    FOnLog                     : TKnobsOnLog;
    FOnHistoryChange           : TKnobsOnHistoryChange;
    FHistory                   : TKnobsUndoRedoHistory;
    FActiveVariation           : Integer;
    FActiveRange               : Integer;
    FOnValueChanged            : TKnobsOnValueChange;
    FOnTextChanged             : TKnobsOnTextChanged;
    FOnRecompile               : TKnobsOnRecompile;
    FOnWireAdded               : TKnobsOnWireAdded;
    FOnWireRemoved             : TKnobsOnWireRemoved;
    FOnShowPopup               : TOnShowPopup;
    FOnUnFocus                 : TOnUnFocus;
    FOnLoadDataGraph           : TOnLoadDataGraph;
    FOnSaveDataGraph           : TOnSaveDataGraph;
    FOnLoadGridControl         : TOnLoadGridControl;
    FOnSaveGridControl         : TOnSaveGridControl;
    FOnActiveVariationChanged  : TOnActiveVariationChanged;
    FOnLoadCaptions            : TOnLoadCaptions;
    FOnEditHistoryChanged      : TNotifyEvent;
    FOnModulesRenamed          : TOnModulesRenamed;
    FOnModulesRenamedFriendly  : TOnModulesRenamed;
    FOnAddEditData             : TOnAddEditData;
  public
    property    EditHistory      : TKnobsEditHistory                  read FEditHistory       implements IKnobsUndoable;
    property    PatchReader      : IKnobsPatchReader                  read FPatchReader               write FPatchReader;
    property    PatchWriter      : IKnobsPatchWriter                  read FPatchWriter               write FPatchWriter;
  public
    property    Module     [ anIndex: Integer]: TKnobsCustomModule    read GetModule;
    property    Wire       [ anIndex: Integer]: TKnobsWire            read GetWire;
    property    Source     [ anIndex: Integer]: TKnobsConnector       read GetSource;
    property    Destination[ anIndex: Integer]: TKnobsConnector       read GetDestination;
  public
    property    OnHistoryChange           : TKnobsOnHistoryChange     read FOnHistoryChange           write FOnHistoryChange;
    property    OnLog                     : TKnobsOnLog               read FOnLog                     write FonLog;
    property    Filename                  : string                    read FFilename                  write FFilename;
    property    ModuleCount               : Integer                   read GetModuleCount;
    property    WireCount                 : Integer                   read GetWireCount;
    property    BlockCount                : Integer                   read GetBlockCount;
    property    PaintCount                : Integer                   read GetPaintCount;
    property    PaintClocks               : Int64                     read GetPaintClocks;
    property    ScrollOffset              : TPoint                    read GetScrollOffset;
    property    ActiveVariation           : Integer                   read FActiveVariation           write SetActiveVariation;
    property    VariationCount            : Integer                   read GetVariationCount;
    property    ActiveRange               : Integer                   read FActiveRange               write SetActiveRange;
    property    RangeValue                : TSignal                                                   write SetRangeValue;
    property    Guid                      : string                    read GetGuid                    write SetGuid;
  published
    property    MaxHistory                : Integer                   read GetMaxHistory              write SetMaxHistory                default 100;
    property    OnValueChanged            : TKnobsOnValueChange       read FOnValueChanged            write FOnValueChanged;
    property    OnTextChanged             : TKnobsOnTextChanged       read FOnTextChanged             write FOnTextChanged;
    property    OnRecompile               : TKnobsOnRecompile         read FOnRecompile               write FOnRecompile;
    property    OnWireAdded               : TKnobsOnWireAdded         read FOnWireAdded               write FOnWireAdded;
    property    OnWireRemoved             : TKnobsOnWireRemoved       read FOnWireRemoved             write FOnWireRemoved;
    property    OnShowPoup                : TOnShowPopup              read FOnShowPopup               write FOnShowPopup;
    property    OnUnFocus                 : TOnUnFocus                read FOnUnFocus                 write FOnUnFocus;
    property    OnLoadDataGraph           : TOnLoadDataGraph          read FOnLoadDataGraph           write FOnLoadDataGraph;
    property    OnSaveDataGraph           : TOnSaveDataGraph          read FOnSaveDataGraph           write FOnSaveDataGraph;
    property    OnLoadGridControl         : TOnLoadGridControl        read FOnLoadGridControl         write FOnLoadGridControl;
    property    OnSaveGridControl         : TOnSaveGridControl        read FOnSaveGridControl         write FOnSaveGridControl;
    property    OnActiveVariationChanged  : TOnActiveVariationChanged read FOnActiveVariationChanged  write FOnActiveVariationChanged;
    property    OnLoadCaptions            : TOnLoadCaptions           read FOnLoadCaptions            write FOnLoadCaptions;
    property    OnEditHistoryChanged      : TNotifyEvent              read FOnEditHistoryChanged      write FOnEditHistoryChanged;
    property    OnModulesRenamed          : TOnModulesRenamed         read FOnModulesRenamed          write FOnModulesRenamed;
    property    OnModulesRenamedFriendly  : TOnModulesRenamed         read FOnModulesRenamedFriendly  write FOnModulesRenamedFriendly;
    property    OnAddEditData             : TOnAddEditData            read FOnAddEditData             write FOnAddEditData;
    property    OffsetLeft                : Integer                   read FOffsetLeft                write SetOffsetLeft                default 4;
    property    OffsetTop                 : Integer                   read FOffsetTop                 write SetOffsetTop                 default 4;
    property    CurvedLines               : Boolean                   read GetCurvedLines             write SetCurvedLines               default True;
    property    WireThickness             : Integer                   read GetWireThickness           write SetWireThickness             default 1;
    property    Title                     : string                    read FTitle                     write FTitle;
    property    Version                   : Integer                   read FVersion                   write FVersion;
    property    SnapX                     : Integer                   read FSnapX                     write FSnapX                       default MOD_Y_UNIT;
    property    SnapY                     : Integer                   read FSnapY                     write FSnapY                       default MOD_X_UNIT;
    property    SnapActive                : Boolean                   read FSnapActive                write FSnapActive                  default True;
    property    AllowRandomization        : Boolean                   read FAllowRandomization        write SetAllowRandomization        default False;
    property    ShowAllowRandomization    : Boolean                   read FShowAllowRandomization    write SetShowAllowRandomization    default False;
    property    ControlMode               : TDistanceMode             read FControlMode               write SetControlMode               default dmCircular;
    property    WheelSupportOnKnobs       : Boolean                   read FWheelSupportOnKnobs       write SetWheelSupportOnKnobs;
    property    WheelSensitivity          : Integer                   read FWheelSensitivity          write SetWheelSensitivity          default 50;
    property    ModuleOpacity             : Byte                      read FModuleOpacity             write SetModuleOpacity             default 223;
    property    ModuleFlatness            : Boolean                   read FModuleFlatness            write SetModuleFlatness            default False;
    property    ModuleTexture             : Boolean                   read FModuleTexture             write SetModuleTexture             default True;
    property    UseTitleHints             : Boolean                   read FUseTitleHints             write SetUseTitleHints             default True;
    property    UseConnectorBorders       : Boolean                   read FUseConnectorBorders       write SetUseConnectorBorders       default True;
    property    UseTypedConnectorBorders  : Boolean                   read FUseTypedConnectorBorders  write SetUseTypedConnectorBorders  default False;
    property    ModuleColor               : TColor                    read FModuleColor               write SetModuleColor               default clWhite;
    property    ConnectorBorderColor      : TColor                    read FConnectorBorderColor      write SetConnectorBorderColor      default clWhite;
    property    SelectorColor             : TColor                    read FSelectorColor             write SetSelectorColor             default clGray;
    property    SelectorBorderColor       : TColor                    read FSelectorBorderColor       write SetSelectorBorderColor       default clYellow;
    property    SelectorBorderColorSingle : TColor                    read FSelectorBorderColorSingle write SetSelectorBorderColorSingle default clWhite;
    property    DisplayColor              : TColor                    read FDisplayColor              write SetDisplayColor              default clGray;
    property    DisplayBorderColor        : TColor                    read FDisplayBorderColor        write SetDisplayBorderColor        default clSilver;
    property    KnobMIDIColor             : TColor                    read FKnobMIDIColor             write SetKnobMIDIColor             default CL_MIDI;
    property    KnobFocusColor            : TColor                    read FKnobFocusColor            write SetKnobFocusColor            default CL_FOCUS;
    property    KnobMIDIFocusColor        : TColor                    read FKnobMIDIFocusColor        write SetKnobMIDIFocusColor        default CL_MIDI_FOCUS;
    property    ViewerColor               : TColor                    read FViewerColor               write SetViewerColor               default clGray;
    property    ViewerBorderColor         : TColor                    read FViewerBorderColor         write SetViewerBorderColor         default clGray;
    property    ViewerLineColor           : TColor                    read FViewerLineColor           write SetViewerLineColor           default clWhite;
    property    ViewerFillColor           : TColor                    read FViewerFillColor           write SetViewerFillColor           default $009E9E9E;
    property    IndBarPeakColor           : TColor                    read FIndBarPeakColor           write SetIndBarPeakColor           default clWhite;
    property    IndBarValeyColor          : TColor                    read FIndBarValeyColor          write SetIndBarValeyColor          default clBlack;
    property    Color                                                                                                                    default $00313131;
    property    Align;
    property    Anchors;
    property    Constraints;
    property    ParentColor;
    property    ParentShowHint                                                                                                           default False;
    property    ParentBackground;
    property    PopupMenu;
    property    ShowHint                                                                                                                 default False;
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    Top;
    property    Left;
    property    Width;
    property    Height;
    property    HorzScrollBar;
    property    VertScrollBar;
    property    StyleElements default [];
    property    OnClick;
  protected
}

    procedure   TKnobsWirePanel.UMConnectionAdd( var aMessage: TKnobsUMConnectionAdd); // message UM_CONNECTIONADD;
    var
      aSrcModule    : string;
      aSrcConnector : string;
      aDstModule    : string;
      aDstConnector : string;
    begin
      FWireOverlay.AddWire( aMessage.Source, aMessage.Destination);

      if   Assigned( FOnWireAdded)
      then begin
        aSrcModule    := aMessage.Source     .ModuleName;
        aSrcConnector := aMessage.Source     .Name;
        aDstModule    := aMessage.Destination.ModuleName;
        aDstConnector := aMessage.Destination.Name;
        FOnWireAdded( Self, aSrcModule, aSrcConnector, aDstModule, aDstConnector);
      end;
    end;


    procedure   TKnobsWirePanel.UMConnectionRemove( var aMessage: TKnobUMConnectionRemove); // message UM_CONNECTIONREMOVE;
    var
      aSrcModule    : string;
      aSrcConnector : string;
      aDstModule    : string;
      aDstConnector : string;
    begin
      FWireOverlay.DelWire( aMessage.Source, aMessage.Destination);

      if   Assigned( FOnWireRemoved)
      then begin
        aSrcModule    := aMessage.Source     .ModuleName;
        aSrcConnector := aMessage.Source     .Name;
        aDstModule    := aMessage.Destination.ModuleName;
        aDstConnector := aMessage.Destination.Name;
        FOnWireRemoved( Self, aSrcModule, aSrcConnector, aDstModule, aDstConnector);
      end;
    end;


    procedure   TKnobsWirePanel.UMColorChange( var aMessage: TKnobsUMColorChange); // message UM_COLORCHANGE;
    begin
      FWireOverlay.HandleColorChange( aMessage.Source, aMessage.Color);
    end;


    procedure   TKnobsWirePanel.UMHighlight( var aMessage: TKnobsUMHighlight); // message UM_HIGHLIGHT;
    begin
      FWireOverlay.HandleHighlight;
    end;


//  private

    function    TKnobsWirePanel.GetModule( anIndex: Integer): TKnobsCustomModule;
    begin
      Result := FModules[ anIndex];
    end;


    function    TKnobsWirePanel.GetWire( anIndex: Integer): TKnobsWire;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.Wire[ anIndex]
      else Result := nil;
    end;


    function    TKnobsWirePanel.GetSource( anIndex: Integer): TKnobsConnector;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.Source[ anIndex]
      else Result := nil;
    end;


    function    TKnobsWirePanel.GetDestination( anIndex: Integer): TKnobsConnector;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.Destination[ anIndex]
      else Result := nil;
    end;


    function    TKnobsWirePanel.GetGuid: string;
    begin
      try
        if   IsEqualGUID( FGUID, TGUID.Empty)
        then CreateGUID( FGUID);

        Result := GUIDToString( FGUID);
      except
        Result := GUIDToString( FGUID);
      end;
    end;


    procedure   TKnobsWirePanel.SetGuid( const aValue: string);
    begin
      // Setting an invalid GUID will result in creation of a fresh one
      if   not SameText( aValue, GUID)
      then begin
        try
          FGuid := StringToGUID( aValue);
        except
          CreateGUID( FGUID);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.SetOffsetLeft( aValue: Integer);
    begin
      if   aValue <> FOffsetLeft
      then begin
        FOffsetLeft := aValue;
        Resnap;
        Restack( DO_RESTACK, DO_REDRAW_WIRES);
      end;
    end;


    procedure   TKnobsWirePanel.SetOffsetTop( aValue: Integer);
    begin
      if   aValue <> FOffsetTop
      then begin
        FOffsetTop := aValue;
        Resnap;
        Restack( DO_RESTACK, DO_REDRAW_WIRES);
      end;
    end;


    function    TKnobsWirePanel.GetCurvedLines: Boolean;
    begin
      Result := FWireOverlay.CurvedLines;
    end;


    procedure   TKnobsWirePanel.SetCurvedLines( aValue: Boolean);
    begin
      FWireOverlay.CurvedLines := aValue;
    end;


    function    TKnobsWirePanel.GetWireThickness: Integer;
    begin
      Result := FWireOverlay.WireThickness;
    end;


    procedure   TKnobsWirePanel.SetWireThickness( aValue: Integer);
    begin
      FWireOverlay.WireThickness := aValue;
    end;


    function    TKnobsWirePanel.GetModuleCount: Integer;
    begin
      if   Assigned( FModules)
      then Result := FModules.Count
      else Result := 0;
    end;


    function    TKnobsWirePanel.GetWireCount: Integer;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.WireCount
      else Result := 0;
    end;


    function    TKnobsWirePanel.GetMaxHistory: Integer;
    begin
      Result := FHistory.MaxHistory;
    end;


    procedure   TKnobsWirePanel.SetMaxHistory( aValue: Integer);
    begin
      FHistory.MaxHistory := aValue;
    end;


    function    TKnobsWirePanel.GetBlockCount: Integer;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.BlockCount
      else Result := -1;
    end;


    function    TKnobsWirePanel.GetPaintCount: Integer;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.PaintCount
      else Result := -1;
    end;


    function    TKnobsWirePanel.GetPaintClocks: Int64;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.PaintClocks
      else Result := -1;
    end;


    function    TKnobsWirePanel.GetScrollOffset: TPoint;
    begin
      Result := Point( HorzScrollBar.Position, VertScrollBar.Position);
    end;


    function    TKnobsWirePanel.GetVariationCount: Integer;
    begin
      Result := MAX_VARIATIONS;
    end;


    procedure   TKnobsWirePanel.SetControlMode( aValue: TDistanceMode);
    begin
      if   aValue <> FControlMode
      then begin
        FControlMode := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetModuleOpacity( aValue: Byte);
    begin
      if   aValue <> FModuleOpacity
      then begin
        FModuleOpacity := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetModuleFlatness( aValue: Boolean);
    begin
      if aValue <> FModuleFlatness
      then begin
        FModuleFlatness := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetModuleTexture( aValue: Boolean);
    begin
      if aValue <> FModuleTexture
      then begin
        FModuleTexture := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetUseTitleHints( aValue: Boolean);
    begin
      if   aValue <> FUseTitleHints
      then begin
        FUseTitleHints := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetUseConnectorBorders( aValue: Boolean);
    begin
      if   aValue <> FUseConnectorBorders
      then begin
        FUseConnectorBorders := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetUseTypedConnectorBorders( aValue: Boolean);
    begin
      if   aValue <> FUseTypedConnectorBorders
      then begin
        FUseTypedConnectorBorders := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetModuleColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FModuleColor
      then begin
        FModuleColor := aValue;

        for i := 0 to ModuleCount - 1
        do Module[ i].Color := ModuleColor;
      end;
    end;


    procedure   TKnobsWirePanel.SetConnectorBorderColor( aValue: TColor);
    begin
      if   aValue <> FConnectorBorderColor
      then begin
        FConnectorBorderColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetSelectorColor( aValue: TColor);
    begin
      if   aValue <> FSelectorColor
      then begin
        FSelectorColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetSelectorBorderColor( aValue: TColor);
    begin
      if   aValue <> FSelectorBorderColor
      then begin
        FSelectorBorderColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetSelectorBorderColorSingle( aValue: TColor);
    begin
      if   aValue <> FSelectorBorderColorSingle
      then begin
        FSelectorBorderColorSingle := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetDisplayColor( aValue: TColor);
    begin
      if   aValue <> FDisplayColor
      then begin
        FDisplayColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetDisplayBorderColor( aValue: TColor);
    begin
      if   aValue <> FDisplayBorderColor
      then begin
        FDisplayBorderColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetKnobMIDIColor( aValue: TColor);
    begin
      if   aValue <> FKnobMIDIColor
      then begin
        FKnobMIDIColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetKnobFocusColor( aValue: TColor);
    begin
      if   aValue <> FKnobFocusColor
      then begin
        FKnobFocusColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetKnobMIDIFocusColor( aValue: TColor);
    begin
      if   aValue <> FKnobMIDIFocusColor
      then begin
        FKnobMIDIFocusColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetViewerColor( aValue: TColor);
    begin
      if   aValue <> FViewerColor
      then begin
        FViewerColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetViewerBorderColor( aValue: TColor);
    begin
      if   aValue <> FViewerBorderColor
      then begin
        FViewerBorderColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetViewerLineColor( aValue: TColor);
    begin
      if   aValue <> FViewerLineColor
      then begin
        FViewerLineColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetViewerFillColor( aValue: TColor);
    begin
      if   aValue <> FViewerFillColor
      then begin
        FViewerFillColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetIndBarPeakColor( aValue: TColor);
    begin
      if   aValue <> FIndBarPeakColor
      then begin
        FIndBarPeakColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetIndBarValeyColor( aValue: TColor);
    begin
      if   aValue <> FIndBarValeyColor
      then begin
        FIndBarValeyColor := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetWheelSupportOnKnobs( aValue: Boolean);
    begin
      if   aValue <> FWheelSupportOnKnobs
      then begin
        FWheelSupportOnKnobs := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetWheelSensitivity( aValue: Integer);
    begin
      if   aValue <> FWheelSensitivity
      then begin
        FWheelSensitivity := aValue;
        FixUserSettings;
      end;
    end;


    procedure   TKnobsWirePanel.SetAllowRandomization( aValue: Boolean);
    var
      i : Integer;
    begin
      FAllowRandomization := aValue;

      for i := 0 to ControlCount - 1
      do begin
        if   ( Controls[ i] is TKnobsCustomModule)
        and  TKnobsCustomModule( Controls[ i]).Selected
        then TKnobsCustomModule( Controls[ i]).AllowRandomization := aValue;
      end;
    end;


    procedure   TKnobsWirePanel.SetShowAllowRandomization( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FShowAllowRandomization
      then begin
        FShowAllowRandomization := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsCustomModule
          then TKnobsCustomModule( Controls[ i]).ShowAllowRandomization := aValue;
        end;
      end;
    end;


    procedure   TKnobsWirePanel.SetActiveVariation( aValue: Integer);
    var
      i : Integer;
    begin
      aValue := Clip( aValue, 0, MAX_VARIATIONS);

      if   aValue <> FActiveVariation
      then begin
        FActiveVariation := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   ( Controls[ i] is TKnobsCustomModule)
          then TKnobsCustomModule( Controls[ i]).ActiveVariation := FActiveVariation;
        end;

        if   Assigned( FOnActiveVariationChanged)
        then FOnActiveVariationChanged( Self, FActiveVariation);
      end;
    end;


    procedure   TKnobsWirePanel.SetActiveRange( aValue: Integer);
    var
      i : Integer;
    begin
      if   aValue <> FActiveRange
      then begin
        FActiveRange := aValue;

        for i := 0 to ModuleCount - 1
        do Module[ i].ActiveRange := FActiveRange;
      end;
    end;


    procedure   TKnobsWirePanel.SetRangeValue( aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].RangeValue := aValue;
    end;


    procedure   TKnobsWirePanel.StrToConnection( const S: string);
    const
      Nums : set of AnsiChar = [ '0' .. '9'];
      Ascs : set of AnsiChar = [ '0' .. '9', 'A' .. 'Z', 'a' .. 'z', '_'];
    var
      p        : Integer;
      i        : Integer;
      SrcModNm : string;
      SrcConNm : string;
      DstModNm : string;
      DstConNm : string;
      SrcMod   : TKnobsCustomModule;
      SrcCon   : TKnobsConnector;
      DstMod   : TKnobsCustomModule;
      DstCon   : TKnobsConnector;
    begin

      p := 1;
      i := 1;

      if   Length( S) > 0
      then begin

        while ( p <= Length( S)) and CharInSet( S[ p], Ascs)
        do Inc( p);

        SrcModNm := Copy( S, i, p - i);

        if   ( p > Length( S)) or ( S[ p] <> '.')
        then Exit
        else Inc( p);

        i := p;

        while ( p <= Length( S)) and CharInSet( S[ p], Ascs)
        do Inc( p);

        SrcConNm := Copy( S, i, p - i);

        if   ( p > Length( S))
        or   ( S[ p] <> ' ')
        then Exit
        else Inc( p);

        i := p;

        while ( p <= Length( S)) and CharInSet( S[ p], Ascs)
        do Inc( p);

        DstModNm := Copy( S, i, p - i);

        if   ( p > Length( S))
        or   ( S[ p] <> '.')
        then Exit
        else Inc( p);

        i := p;

        while ( p <= Length( S)) and CharInSet( S[ p], Ascs)
        do Inc( p);

        DstConNm := Copy( S, i, p - i);

        try
          SrcMod :=        FindComponent( SrcModNm) as TKnobsCustomModule;
          SrcCon := SrcMod.FindComponent( SrcConNm) as TKnobsConnector;
          DstMod :=        FindComponent( DstModNm) as TKnobsCustomModule;
          DstCon := DstMod.FindComponent( DstConNm) as TKnobsConnector;
          Connect( SrcCon, DstCon);
        except
          on Exception
          do
            raise
              EKnobsCorruptPatch.
                Create(
                  'The patch is corrupted or incompatible:'^M +
                  'can''t make all the connections.'
                );
        end;

      end
      else
        raise
          EKnobsCorruptPatch.
            Create(
              'The patch is corrupted, incompatible or not a patch.'^M
            );
    end;


//  protected

    procedure   TKnobsWirePanel.HandleRightClick( const aSender: TControl);
    begin
      if   Assigned( FOnShowPopup)
      then FOnShowPopup( Self, aSender);
    end;


    procedure   TKnobsWirePanel.CreateParams(var Params: TCreateParams); // override
    begin
      inherited CreateParams( Params);

      with Params.WindowClass
      do Style := Style and not ( CS_HREDRAW or CS_VREDRAW);
    end;


    procedure   TKnobsWirePanel.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      if   FMouseDown
      then Exit;

      if   Button = mbRight
      then HandleRightClick( Self)
      else if not FMouseDown and ( Button = mbLeft)
      then begin
        FMouseDown := True;
        GAnchor    := Point( X, Y);
        GDistance  := Point( 0, 0);
        DragStart( GAnchor  );
        DragMove ( GDistance);
      end;
    end;


    procedure   TKnobsWirePanel.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      if   FMouseDown
      then begin
        DragMove( GDistance);
        GDistance.x := X - GAnchor.x;
        GDistance.y := Y - GAnchor.y;
        FMouseDown := False;
        DragEnd( GDistance);
      end;
    end;


    procedure   TKnobsWirePanel.MouseMove( Shift: TShiftState; X, Y: Integer); // override;
    begin
      if   FMouseDown
      then begin
        DragMove( GDistance);
        GDistance.x := X - GAnchor.x;
        GDistance.y := Y - GAnchor.y;
        DragMove( GDistance);
      end
    end;


    procedure   TKnobsWirePanel.DragStart( aPoint: TPoint); // virtual;
    begin
      // Nothing
    end;


    procedure   TKnobsWirePanel.DragEnd( aPoint: TPoint); // virtual;
    begin
      SelectModulesInRect(
        Rect(
          GAnchor.x,
          GAnchor.y,
          GAnchor.x + aPoint.x,
          GAnchor.y + aPoint.y
        )
      );
    end;


    procedure   TKnobsWirePanel.DragMove( aPoint: TPoint); // virtual;
    var
      aDC      : HDC;
      aRect    : TRect;
      aPen     : HPen;
      aBrush   : HBrush;
      aColor   : TColor;
      OldPen   : HGDIOBJ;
      OldBrush : HGDIOBJ;
    begin
      aDC := GetDCEx( Handle, 0, DCX_PARENTCLIP);
      IntersectClipRect( aDC, 0, 0, ClientWidth, ClientHeight);

      try
        with aRect
        do begin
          Left   := GAnchor.x;
          Top    := GAnchor.y;
          Right  := Left + aPoint.x;
          Bottom := Top  + aPoint.y;

          if   Left > Right
          then Exchange( Left, Right);

          if   Top > Bottom
          then Exchange( Bottom, Top);
        end;

        SetRop2( aDC, R2_XORPEN);
        aColor   := clWhite;
        aPen     := CreatePen( PS_SOLID, 2, ColorToRGB( aColor));
        OldPen   := SelectObject( aDC, aPen);
        aBrush   := GetStockObject( NULL_BRUSH);
        OldBrush := SelectObject( aDC, aBrush);

        try
          Rectangle( aDC, aRect.Left, aRect.Top, aRect.Left + aRect.Width, aRect.Top + aRect.Height);
        finally
          SelectObject( aDC, OldBrush);
          SelectObject( aDC, OldPen  );
          DeleteObject( aPen);
        end;
      finally
        ReleaseDC( Handle, aDC);
      end;
    end;


    procedure   TKnobsWirePanel.SelectModulesInRect( aRect: TRect);
    var
      i : Integer;
    begin
      with aRect
      do begin
        if   Left > Right
        then Exchange( Left, Right);

        if   Top > Bottom
        then Exchange( Bottom, Top);
      end;

      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsCustomModule
        then begin
          with TKnobsCustomModule( Controls[ i])
          do Selected := RectanglesOverlap( aRect, Rect( Left, Top, Left + Width, Top + Height));
        end;
      end;
    end;


    function    TKnobsWirePanel.ModuleIndex( aModule: TKnobsCustomModule): Integer;
    begin
      Result := FModules.IndexOf( aModule);
    end;


    procedure   TKnobsWirePanel.Notification( aComponent: TComponent; anOperation: TOperation); // override;
    var
      i : Integer;
    begin
      inherited;

      if   aComponent is TKnobsCustomModule
      then begin
        if   anOperation = opInsert
        then begin
          i := ModuleIndex( TKnobsCustomModule( aComponent));

          if   i < 0
          then FModules.Add( aComponent);
        end
        else if anOperation = opRemove
        then begin
          i := ModuleIndex( TKnobsCustomModule( aComponent));

          if   i >= 0
          then begin
            with FModules
            do begin
              Delete( i);
              Pack;
            end;
          end;
        end;
      End
    end;


    procedure   TKnobsWirePanel.Connect( aSource, aDestination: TKnobsConnector);
    begin
      if   Assigned( FWireOverlay)
      then FWireOverlay.Connect( aSource, aDestination);
    end;


    procedure   TKnobsWirePanel.Restack( DoRestack, RedrawWires: Boolean);
    begin
      if   ModuleCount > 0
      then Module[ 0].Restack( DoRestack, RedrawWires); // Which will restack all modules
    end;


    procedure   TKnobsWirePanel.Resnap;
    var
      i : Integer;
      p : TPoint;
    begin
      for i := 0 to ModuleCount  - 1
      do begin
        with Module[ i]
        do begin
          p    := Snap( Point( Left, Top));
          Left := P.X;
          Top  := P.Y;
        end;
      end;
    end;


    procedure   TKnobsWirePanel.FixNames( aStart: Integer);
    var
      i                  : Integer;
      aModule            : TKnobsCustomModule;
      anOldNames         : TStringArray;
      aNewNames          : TStringArray;
      anOldFriendlyNames : TStringArray;
      aNewFriendlyNames  : TStringArray;
    begin
      SetLength( anOldNames        , ModuleCount);
      SetLength( aNewNames         , ModuleCount);
      SetLength( anOldFriendlyNames, ModuleCount);
      SetLength( aNewFriendlyNames , ModuleCount);

      for i := 0 to ModuleCount - 1                             // First rename is to avoid name clashes in 2nd rename
      do begin
        anOldNames[ i]         := Module[ i].Name;
        anOldFriendlyNames[ i] := Module[ i].FriendlyName + ' ';
        Module[ i].Name        := Format( 'a%d', [ i], AppLocale);
      end;

      for i := 0 to ModuleCount - 1                             // No modules have a name of the form <clssname><n> yet
      do begin
        aModule := Module[ i];
        aModule.Name          := Format( 'TKnobsModule%d', [ aStart + i], AppLocale);
        aNewNames[ i]         := Module[ i].Name;
        aNewFriendlyNames[ i] := Module[ i].FriendlyName + ' ';

        if   Assigned( aModule.FTitleLabel)
        then begin
          if   aModule.FUseAlternateTitle
          then aModule.FTitleLabel.Text := aModule.FriendlyName
          else aModule.FTitleLabel.Text := aModule.Title;
        end;
      end;

      if   Assigned( EditHistory)
      then begin
        EditHistory.UpdateNames( anOldFriendlyNames, aNewFriendlyNames);
        EditHistoryChanged;
      end;

      if   Assigned( FOnModulesRenamed)
      then FOnModulesRenamed( Self, anOldNames, aNewNames);

      if   Assigned( FOnModulesRenamedFriendly)
      then FOnModulesRenamedFriendly( Self, anOldFriendlyNames, aNewFriendlyNames);
    end;


    procedure   TKnobsWirePanel.SolveNamingConflictsWith( aDstWirePanel: TKnobsWirePanel);
    begin

      // Make own module names unique in aDstWirePanel

      if   Assigned( aDstWirePanel)
      then begin
        aDstWirePanel.FixNames( 0);            // Dst numbered 0 .. n-1
        FixNames( aDstWirePanel.ModuleCount);  // and self numbered n .. m-1
      end;
    end;


    function    TKnobsWirePanel.Snap( const aPoint: TPoint): TPoint;
    begin
      if   SnapActive
      and  ( SnapX <> 0)
      and  ( SnapY <> 0)
      then begin
        Result.x := OffsetLeft + (( aPoint.x - OffsetLeft + SnapX div 2) div SnapX) * SnapX;
        Result.y := OffsetTop  + (( aPoint.y - OffsetTop  + SnapY div 2) div SnapY) * SnapY;
      end
      else Result := aPoint;
    end;


    procedure   TKnobsWirePanel.DoHistoryChanged( const aSender: TObject; anUndoCount, aRedoCount, aSavedMarker: Integer);
    begin
      if   Assigned( FOnHistoryChange)
      then begin
        try
          FOnHistoryChange( aSender, anUndoCount, aRedoCount, aSavedMarker);
        except
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.ValueChanged( aSender: TObject; const aPath, aControlType: string; aValue: TSignal; IsFinal, IsAutomation: Boolean); // virtual;
    begin
      if   Assigned( FOnValueChanged)
      then begin
        try
          if   ( not IsAutomation)
          or   ( IsAutomation and Lock( FWormUpdatesBlocked))
          then begin
            try
              FOnValueChanged( aSender, aPath, aControlType, aValue, IsFinal, IsAutomation);
            finally
              if   IsAutomation
              then Unlock( FWormUpdatesBlocked);
            end;
          end;
        except
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.TextChanged( aSender: TObject; const aPath, aValue: string);
    begin
      if   Assigned( FOnTextChanged)
      then begin
        try
          FOnTextChanged( aSender, aPath, aValue);
        except
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.Recompile( ModulesChanged: Boolean);
    begin
      if   Assigned( FOnRecompile)
      then begin
        try
          FOnRecompile( Self, ModulesChanged);
        except
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.ScrollPointInView( aPoint: TPoint); // overload;
    begin
      ScrollPointInView( aPoint.X, aPoint.Y);
    end;


    procedure   TKnobsWirePanel.ScrollPointInView( anX, anY: LongInt); // overload;
    const
      Clearance = 33;
    var
      aRect: TRect;
    begin
      aRect.TopLeft     := Point( anX - Clearance, anY - Clearance);
      aRect.BottomRight := Point( anX + Clearance, anY + Clearance);
      Dec( aRect.Left  , HorzScrollBar.Margin);
      Inc( aRect.Right , HorzScrollBar.Margin);
      Dec( aRect.Top   , VertScrollBar.Margin);
      Inc( aRect.Bottom, VertScrollBar.Margin);

      if   aRect.Left < 0
      then begin
        with HorzScrollBar
        do Position := Position + aRect.Left
      end
      else if aRect.Right > ClientWidth
      then begin
        if   aRect.Right - aRect.Left > ClientWidth
        then aRect.Right := aRect.Left + ClientWidth;

        with HorzScrollBar
        do Position := Position + aRect.Right - ClientWidth;
      end;

      if   aRect.Top < 0
      then begin
        with VertScrollBar
        do Position := Position + aRect.Top;
      end
      else if aRect.Bottom > ClientHeight
      then begin
        if   aRect.Bottom - aRect.Top > ClientHeight
        then aRect.Bottom := aRect.Top + ClientHeight;

        with VertScrollBar
        do Position := Position + aRect.Bottom - ClientHeight;
      end;
    end;


    function    TKnobsWirePanel.LoadDataGraph: string;
    begin
      if   Assigned( FOnLoadDataGraph)
      then Result := FOnLoadDataGraph( Self)
      else Result := '';
    end;

    procedure   TKnobsWirePanel.SaveDataGraph( const aValue: string);
    begin
      if   Assigned( FOnSaveDataGraph)
      then FOnSaveDataGraph( Self, aValue);
    end;


    function    TKnobsWirePanel.LoadGridControl: string;
    begin
      if   Assigned( FOnLoadGridControl)
      then Result := FOnLoadGridControl( Self)
      else Result := '';
    end;

    procedure   TKnobsWirePanel.SaveGridControl( const aValue: string);
    begin
      if   Assigned( FOnSaveGridControl)
      then FOnSaveGridControl( Self, aValue);
    end;


    procedure   TKnobsWirePanel.LoadCaptionsFor( const aModule: TKnobsCustomModule; const aControl: TKnobsSelector; const aDependency: TKnobsValuedControl);
    begin
      if   Assigned( FOnLoadCaptions)
      then FOnLoadCaptions( Self, aModule, aControl, aDependency)
      else if Assigned( GOnLoadCaptions)
      then GOnLoadCaptions( Self, aModule, aControl, aDependency);
    end;


    procedure   TKnobsWirePanel.AcceptAutomationData( const aSender: TKnobsValuedControl; const aData: PKnobsEditData);
    begin
      if   Assigned( FOnAddEditData)
      then FOnAddEditData( aSender, aData);
    end;


//  public

    constructor TKnobsWirePanel.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FEditHistory    := TKnobsEditHistory.Create( nil, DEFAULT_AUTOMATION_SIZE, HISTORY_STR_LEN, False);
      ControlStyle    :=
        [
          csAcceptsControls,
          csCaptureMouse,
          csClickEvents,
          csSetCaption,
          csDoubleClicks,
          csPannable
        ];
      OffsetLeft                   :=  4;
      OffsetTop                    :=  4;
      AutoScroll                   := True;
      Width                        := 185;
      Height                       :=  41;
      SnapX                        := MOD_X_UNIT;
      SnapY                        := MOD_Y_UNIT;
      SnapActive                   := True;
      FModules                     := TKnobsModuleList     .Create( True);
      FWireOverlay                 := TKnobsWireOverlay    .Create( Self);
      FHistory                     := TKnobsUndoRedoHistory.Create;
      MaxHistory                   := 100;
      FControlMode                 := dmHorizontal;
      ControlMode                  := dmCircular;
      UseTitleHints                := True;
      UseConnectorBorders          := True;
      UseTypedConnectorBorders     := False;
      ModuleColor                  := clWhite;
      ConnectorBorderColor         := clWhite;
      SelectorColor                := clGray;
      SelectorBorderColor          := clYellow;
      SelectorBorderColorSingle    := clWhite;
      DisplayColor                 := clGray;
      DisplayBorderColor           := clSilver;
      KnobMIDIColor                := CL_MIDI;
      KnobFocusColor               := CL_FOCUS;
      KnobMIDIFocusColor           := CL_MIDI_FOCUS;
      ViewerColor                  := clGray;
      ViewerBorderColor            := clGray;
      ViewerLineColor              := clWhite;
      ViewerFillColor              := $009E9E9E;
      IndBarPeakColor              := clWhite;
      IndBarValeyColor             := clBlack;
      Color                        := $00313131;
      ModuleOpacity                := 223;
      ModuleFlatness               := False;
      ModuleTexture                := True;
      WheelSensitivity             := 50;
      AllowRandomization           := False;
      ShowAllowRandomization       := False;
      StyleElements                := [];
      ShowHint                     := False;
      ParentShowHint               := False;
      FWireOverlay.Parent          := Self;
      ActiveRange                  := -1;
      FHistory    .OnHistoryChange := DoHistoryChanged;
    end;


    destructor  TKnobsWirePanel.Destroy; // override;
    begin
      inherited;                      // This will call into "Notification", which
      FreeAndNil( FModules       );   // accesses FModules, so free FModules last.
      FreeAndNil( FHistory       );   // and do this later even ...
      FreeAndNil( FEditHistory   );
    end;


    procedure   TKnobsWirePanel.BuildModuleIndex;
    begin
      FModules.BuildIndex;
    end;


    procedure   TKnobsWirePanel.PrepareGraph;
    var
      i         : Integer;
      aCounter1 : Integer;
      aCounter2 : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].FVisited := False;

      aCounter1 := 1;
      aCounter2 := 0;

      for i := 0 to ModuleCount - 1
      do Module[ i].Visit( aCounter1, aCounter2);
    end;



    function CycleSort( Item1, Item2: Pointer): Integer;
    var
      M1 : TKnobsCustomModule;
      M2 : TKnobsCustomModule;
    begin
      if   Assigned( Item1)
      and  Assigned( Item2)
      then begin
        M1 := TKnobsCustomModule( Item1);
        M2 := TKnobsCustomModule( Item2);

        if   M1.FPostCount < M2.FPostCount
        then Result := 1
        else if M1.FPostCount > M2.FPostCount
        then Result := -1
        else begin
          if   M1.Left < M2.Left
          then Result := 1
          else if M1.Left > M2.Left
          then Result := -1
          else begin
            if   M1.Top < M2.Top
            then Result := 1
            else if M1.Top > M2.Top
            then Result := -1
            else Result := 0;
          end;
        end;
      end
      else Result := 0;
    end;


    procedure   TKnobsWirePanel.SortGraph;
    begin
      Restack( DO_RESTACK, NO_REDRAW_WIRES);
      PrepareGraph;

      if   Assigned( FModules)
      then FModules.Sort( CycleSort);
    end;


    procedure   TKnobsWirePanel.DumpGraph( const aStrings: TStringList); // overload;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].DumpGraph( aStrings);
    end;


    procedure   TKnobsWirePanel.DumpGraph( const aFileName: string); // overload;
    var
      aStrings: TStringList;
    begin
      aStrings := TStringList.Create;

      try
        DumpGraph( aStrings);
        aStrings.SaveToFile( aFileName);
      finally
        aStrings.DisposeOf;
      end;
    end;


    procedure   TKnobsWirePanel.FixAllSynthParams( IsAutomation: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].FixAllSynthParams( IsAutomation);
    end;


    procedure   TKnobsWirePanel.BeginStateChange( IsStructural: Boolean);
    begin
      if
           IsStructural
      and  Assigned( FPatchReader)
      and  Assigned( FPatchWriter)
      then FHistory.SaveState( Self);

      FHistory.Lock;
    end;


    procedure   TKnobsWirePanel.EndStateChange( MustRecompile, ModulesChanged: Boolean);
    begin
      if   FHistory.UnLock
      and  MustRecompile
      then Recompile( ModulesChanged);
    end;


    procedure   TKnobsWirePanel.Undo;
    begin
      if   Assigned( FPatchReader)
      and  Assigned( FPatchWriter)
      then begin
        FHistory.Undo( Self);
        Recompile( True);
        UnSelectAllModules;
        FixAllSynthParams( False);  // todo: raaaaar .. this is needed ... non comprendo .. anyway .. otherwise stuff doesnt properly start up on the initial patch ...
      end
    end;


    procedure   TKnobsWirePanel.Redo;
    begin
      if   Assigned( FPatchReader)
      and  Assigned( FPatchWriter)
      then begin
        FHistory.Redo( Self);
        Recompile( True);
        UnSelectAllModules;
        FixAllSynthParams( False);  // todo: raaaaar .. this is needed ... non comprendo .. anyway .. otherwise stuff doesnt properly start up on the initial patch ...
      end;
    end;


    procedure   TKnobsWirePanel.ClearUndoRedo;
    begin
      FHistory.Clear;
    end;


    procedure   TKnobsWirePanel.SetSavedMarker;
    begin
      FHistory.SetSavedMarker;
    end;


    procedure   TKnobsWirePanel.Dump( const aFileName: string; DoAppend: Boolean); // overload;
    var
      aFile: TextFile;
    begin
      AssignFile( aFile, aFileName);

      if   FileExists( aFileName)
      then begin
        if   DoAppend
        then Append ( aFile)
        else Rewrite( aFile);
      end
      else Rewrite( aFile);

      try
        Dump( aFile);
      finally
        CloseFile( aFile);
      end;
    end;


    procedure   TKnobsWirePanel.Dump( var aFile: TextFile); // overload;
    begin
      WriteLn     ( aFile,         '---- wirepanel.dump -----------------------------------');
      WriteLn     ( aFile, Format( '---- %s', [ FormatDateTime( 'yyyy-mm-dd hh:nn:ss', Now)], AppLocale));
      DumpModules ( aFile, 1);
      WriteLn     ( aFile);
      DumpWires   ( aFile, 1);;
      WriteLn     ( aFile,         '---- end wirepanel.dump -------------------------------');
      WriteLn     ( aFile);
    end;


    procedure   TKnobsWirePanel.DumpModules( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, 'modules');

      if   Assigned( FModules)
      then FModules.Dump( aFile, anInd + 1);

      WriteLnInd( aFile, anInd, 'end modules');
    end;


    procedure   TKnobsWirePanel.DumpWires( var aFile: TextFile; anInd: Integer);
    begin
      WriteLnInd( aFile, anInd, 'wires');

      if   Assigned( FWireOverlay)
      then FWireOverlay.Dump( aFile, anInd + 1);

      WriteLnInd( aFile, anInd, 'end wires');
    end;


    procedure   TKnobsWirePanel.Log( aLogClass: TLogClass; const aMsg: string);
    begin
      if   Assigned( FOnLog)
      then FOnLog( Self, aLogClass, aMsg);
    end;


    procedure   TKnobsWirePanel.LogFmt( aLogClass: TLogClass; const aFmt: string; const anArgs: array of const);
    begin
      Log( aLogClass, Format( aFmt, anArgs, AppLocale));
    end;

{$ifdef USE_HASH}

    function    TKnobsWirePanel.FindModule( const aName: string): TKnobsCustomModule;
    begin
      Result := FModules.FindModule( aName);
    end;

{$else}

    function    TKnobsWirePanel.FindModule( const aName: string): TKnobsCustomModule;
    var
      i : Integer;
    begin
      Result := nil;

      for i := 0 to ModuleCount - 1
      do begin
        if   SameText( Module[ i].Name, aName)
        then begin
          Result := Module[ i];
          Break;
        end;
      end;
    end;

{$endif}

    function    TKnobsWirePanel.FindFirstSelectedModule: TKnobsCustomModule;
    var
      i : Integer;
    begin
      Result := nil;

      for i := 0 to ModuleCount - 1
      do begin
        if   Module[ i].Selected
        then begin
          Result := Module[ i];
          Break;
        end;
      end;
    end;


    function    TKnobsWirePanel.GetModuleAlternateTypes( aModule: TKnobsCustomModule): string;
    begin
      Result := '';

      if   Assigned( aModule)
      then Result := aModule.ModuleAlternateTypes;
    end;


//  public

    procedure   TKnobsWirePanel.SetAllowAllRandomization( aValue: Boolean);
    var
      i : Integer;
    begin
      FAllowRandomization := aValue;

      for i := 0 to ControlCount - 1
      do begin
        if   ( Controls[ i] is TKnobsCustomModule)
        then TKnobsCustomModule( Controls[ i]).AllowAllRandomization( aValue)
      end;
    end;


//  public

    procedure   TKnobsWirePanel.InvalidateWires;
    begin
      FWireOverlay.InvalidateWires;
    end;


    procedure   TKnobsWirePanel.InvalidateDisplays;
    begin
      FModules.InvalidateDisplays;
    end;


    procedure   TKnobsWirePanel.Disconnect( aConnector: TKnobsConnector);
    var
      Changed : Boolean;
    begin
      if   Assigned( FWireOverlay)
      then begin
        Changed := False;
        BeginStateChange( True);

        try
          Changed := FWireOverlay.Disconnect( aConnector);
        finally
          EndStateChange( Changed, False)
        end;
      end;
    end;


    procedure   TKnobsWirePanel.FreeModule( aModule: TKnobsCustomModule; const aCallback: TKnobsOnValuedCtrlRemoved);
    begin
      BeginStateChange( True);
      BlockWireUpdates;

      try
        FModules.FreeModule( aModule, aCallback);
      finally
        UnBlockWireUpdates;
        EndStateChange( True, True);
      end;
    end;


    procedure   TKnobsWirePanel.FreeModules( const aCallback: TKnobsOnValuedCtrlRemoved);
    var
      Changed : Boolean;
    begin
      BeginStateChange( True);
      BlockWireUpdates;
      Changed := False;

      try
        Changed := FModules.FreeModules( aCallback);
      finally
        UnBlockWireUpdates;
        EndStateChange( Changed, Changed);
      end;
    end;


    procedure   TKnobsWirePanel.FreeSelectedModules( const aCallback: TKnobsOnValuedCtrlRemoved);
    var
      Changed : Boolean;
    begin
      if   not HasEditPopup
      then begin
        BeginStateChange( True);
        BlockWireUpdates;
        Changed := False;

        try
          Changed := FModules.FreeSelectedModules( aCallback);
        finally
          UnBlockWireUpdates;
          EndStateChange( Changed, Changed);
        end;
      end;
    end;


    function    TKnobsWirePanel.HasEditPopup: Boolean;
    begin
      Result :=
        ( Assigned( GDisplayEditor) and GDisplayEditor.Visible) or
        ( Assigned( GLabelEditor  ) and GLabelEditor  .Visible)
    end;


    procedure   TKnobsWirePanel.SetSelectedModuleColor( aValue: TColor);
    begin
      FModules.SetSelectedModuleColor( aValue);
    end;


    procedure   TKnobsWirePanel.SetSelectedModuleColorDefault( const aColorFunc: TOnGetModuleColor);
    begin
      FModules.SetSelectedModuleColorDefault( aColorFunc);
    end;


    procedure   TKnobsWirePanel.SelectAllModules;
    begin
      FModules.SelectAll;
    end;


    procedure   TKnobsWirePanel.UnSelectAllModules;
    begin
      FModules.UnSelectAll;
    end;


    procedure   TKnobsWirePanel.SelectUnique( aModule: TKnobsCustomModule);
    begin
      FModules.SelectUnique( aModule);
    end;


    procedure   TKnobsWirePanel.FixSelection( aModule: TKnobsCustomModule);
    begin

      // When no modules are selected select aModule uniquely

      FModules.FixSelection( aModule);
    end;


    procedure   TKnobsWirePanel.AddToSelection( const aModuleName: string);
    var
      aModule : TKnobsCustomModule;
    begin
      aModule := FindModule( aModuleName);

      if   Assigned( aModule)
      then aModule.Selected := True;
    end;


    procedure   TKnobsWirePanel.InvertModuleSelection;
    begin
      FModules.InvertSelection;
    end;


    function    TKnobsWirePanel.FindTitle( const aModuleName: string): string;
    var
      aModule : TKnobsCustomModule;
    begin
      Result := '';
      aModule := FindModule( aModuleName);

      if   Assigned( aModule)
      then Result := aModule.Title;
    end;


    function    TKnobsWirePanel.CopyToString( aWriteMode: TKnobsWriteMode): string;
    begin
      Result := '';

      if   Assigned( FPatchWriter)
      then Result := FPatchWriter.WriteString( Self, '', aWriteMode);
    end;


    procedure   TKnobsWirePanel.CopyFromString( const S: string; CopyMidiCC: Boolean; aReadMode: TKnobsReadMode);
    begin
      if   Assigned( FPatchReader)
      then FPatchReader.ReadString( S, Self, Point( 0, 0), True, False, CopyMidiCC, False, rmReplace, '');
    end;


    procedure   TKnobsWirePanel.CopyTo( aPanel: TKnobsWirePanel; CopyMidiCC: Boolean; aWriteMode: TKnobsWriteMode; aReadMode: TKnobsReadMode);
    begin
      if   Assigned( aPanel)
      then aPanel.CopyFromString( CopyToString( aWriteMode), CopyMidiCC, aReadMode);
    end;


    function    TKnobsWirePanel.SameStructure( aPanel: TKnobsWirePanel): Boolean;
    var
      i : Integer;
      p : Integer;
    begin

      // For a structural match to be present there needs be a one to one module type mapping from one panel to the other.
      // Module order is important here (imagine the case where two identically typed modules are involved).
      // We only look at the selected modules in self, but look at all the modules in aPanel, regardless of selection.
      // The easy way would be to make a copy of the selected modules of Self and compare that to aPanel. The wires are
      // irrelevant for a structural match.

      Result := False;

      if   Assigned( aPanel)
      then begin
        Result := True;
        p      := 0;

        for i := 0 to ModuleCount - 1
        do begin
          if   Module[ i].Selected
          then begin
            if
              ( p >= aPanel.ModuleCount) or
              ( aPanel.Module[ p].ModuleType <> Module[ i].ModuleType)
            then begin
              Result := False;
              Break;
            end;
            Inc( p);
          end;
        end;
      end;
    end;


    function   TKnobsWirePanel.CopyParamsFrom( aSrc: TKnobsWirePanel; CopyStrict: Boolean): Boolean;
    var
      i : Integer;
      p : Integer;
    begin

      // When the structure of aSrc matches the structure of the selection in Self
      // then copy the parameter values from aSrc over to the selection of Self
      //
      // First check structures, the structure of the selection in Self needs to match the structure of aSrc (regardless
      // of selection there). If so copy over all parameter values from aSrc to the selection in Self. For a structural
      // match to be present there needs be a one to one module type mapping from one panel to the other. Only when that
      // helds a parameter copy can be performed. Module order is important here (imagine the case where two identically
      // typed modules are involved), so a sort should be performed here, likely, on the screen positons ... hmm. The
      // origin of aSrc is assumed to be a copy (action) of selected modules from a patch .. this may not be a current
      // invariant .. need to check that ... ok, modules are written in module order always when using a PatchWriter.
      // The easy way would be to make a copy of the selected modules of Self and copy that to aPanel. The wires are
      // irrelevamt for a parameter copy.

      Result := False;  // Signal failure

      if   Assigned( aSrc)
      then begin
        if   SameStructure( aSrc)
        or   (( aSrc.ModuleCount = 1) and ( ModuleCount = 1) and ( not CopyStrict))
        then begin
          p := 0;

          for i := 0 to ModuleCount - 1
          do begin
            if   Module[ i].Selected
            and  ( p < aSrc.ModuleCount)
            then begin

              // The SameStructure call checked Module[ i] and aSrc.Module[ p] to be of the same type so we must not
              // check that here in order to not let p and i not get out of sync. The CopyParamsFrom method for a
              // module however should still perform a type check, and as such it may not actually perform a param
              // copy.

              Module[ i].CopyParamsFrom( aSrc.Module[ p], CopyStrict);
              Inc( p);
            end;
          end;

          Result := True; // Signal success
        end;
      end;
    end;


    procedure   TKnobsWirePanel.AddModules( aSrc: TKnobsWirePanel; aDistance: TPoint; DoRestack, RedrawWires, MustDrag, CopyMidiCC: Boolean); // override;
    var
      i             : Integer;
      M             : TKnobsCustomModule;
      ConnectErrors : Integer;
      ModuleErrors  : Integer;
    begin

      // This one is called only from the KnobParser either when a patch file
      // (or a set of modules) is loaded or when a wirepanel was pasted from the
      // clipboard; when pasted from the clipboard, or when it is a set of modules
      // to be added in, the newly pasted modules should be set into drag mode,
      // for other cases see __2 (also in KnobParser although that one will end
      // up here and nothing special is done there except for setting the
      // Dragging parameter to true when the Addition was the result of a paste
      // from the clip board, or when a set of modules was loaded from a file).
      // Correction : also after a Ctrl+Drag (copy) operation we end up here ...
      // in that case dragging will be false, which is the correct way to handle
      // this ... luck ... :shock:
      // See Clone method - that one fixes it for single module insertions.

      if   not HasEditPopup
      then begin
        BeginStateChange( True);   // Blocks undo/redo updates
        BlockWireUpdates;          // Blocks screen paints
        DisableAlign;

        try
          UnSelectAllModules;
          aSrc.SolveNamingConflictsWith( Self);  // Make Src names unique in Self

          // When dragging the following code should not be performed as this
          // represents a final insertion into the wirepanel which will move
          // any existing modules around ... this will be tricky .... __2
          // we could maybe treat this by inserting the modules a a negative
          // offset (which is not normally possible) and setting an approriate
          // distance instead of the one passed as an argument to us.

          ModuleErrors := 0;

          for i := 0 to aSrc.ModuleCount - 1
          do begin
            try
              M := aSrc.Module[ i];

              if   M.Color = TColor( $ffffffff)
              then M.Color := Self.ModuleColor;

              M.Opacity := ModuleOpacity;
              M.Flat    := ModuleFlatness;
              M.Texture := ModuleTexture;
              M.Clone(
                Self,
                Self,
                Snap( Point( M.Left, M.Top ) + aDistance),
                MustDrag,
                CopyMidiCC
              // ).FixupInsertion; // .Restack( DoRestack, RedrawWires);
              ).Restack( DoRestack, RedrawWires);
              // todo : **1 :: try to get rid of offset - 1000, see other **1 remarks too - ommitting the .restack causes
              //               a GPF later on after a copy / paste - but not restacking otherwise does the right thing,
              //               fixupInsertions avoids a crash it seems (like Restack does .. erm fixup calls restack ...).
            except
              Inc( ModuleErrors);
            end;
          end;

          if   ModuleErrors > 0
          then LogFmt( LC_COMPILER, 'Could not make all modules (%d was/were lost)', [ ModuleErrors]);

          ConnectErrors := 0;

          for i := 0 to aSrc.WireCount - 1
          do begin
            try
              StrToConnection(
                Format
                (
                  '%s %s',
                  [
                    ConnectorName( aSrc.Source     [ i]),
                    ConnectorName( aSrc.Destination[ i])
                  ],
                  AppLocale
                )
              );
            except
              Inc( ConnectErrors);
            end;
          end;

          if   ConnectErrors > 0
          then LogFmt( LC_COMPILER, 'Could not make all connections (%d was/were lost)', [ ConnectErrors]);
        finally
          FixUserSettings;
          UnBlockWireUpdates;
          EndStateChange( True, True);
          EnableAlign;
        end;
      end;
    end;


    procedure   TKnobsWirePanel.AddWire( const aSrc, aDst: string); // overload;
    begin
      BeginStateChange( True);

      try
        StrToConnection( Format( '%s %s', [ aSrc, aDst], AppLocale));
      finally
        EndStateChange( True, False);
      end;
    end;


    procedure   TKnobsWirePanel.AddWire( const aSrcDst: string); // overload;
    begin
      BeginStateChange( True);

      try
        StrToConnection( aSrcDst);
      finally
        EndStateChange( True, False);
      end;
    end;


    procedure   TKnobsWirePanel.WiggleWires;
    begin
      FWireOverlay.WiggleWires;
    end;


    procedure   TKnobsWirePanel.ToggleWires;
    begin
      FWireOverlay.ToggleWires;
    end;


    function    TKnobsWirePanel.WiresOff: Boolean;
    begin
      if   Assigned( FWireOverlay)
      then Result := FWireOverlay.WiresOff
      else Result := True;
    end;


    procedure   TKnobsWirePanel.WiresOn( TurnOn: Boolean);
    begin
      if   Assigned( FWireOverlay)
      then FWireOverlay.WiresOn( TurnOn);
    end;


    procedure   TKnobsWirePanel.FixAllWires;
    begin
      if   Assigned( FWireOverlay)
      then FWireOverlay.FixAllWires;
    end;


    procedure   TKnobsWirePanel.FixUserSettings;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do begin
        Module[ i].ControlMode               := ControlMode              ;
        Module[ i].WheelSupportOnKnobs       := WheelSupportOnKnobs      ;
        Module[ i].WheelSensitivity          := WheelSensitivity         ;
        Module[ i].Opacity                   := ModuleOpacity            ;
        Module[ i].Flat                      := ModuleFlatness           ;
        Module[ i].Texture                   := ModuleTexture            ;
        Module[ i].UseTitleHints             := UseTitleHints            ;
        Module[ i].UseConnectorBorders       := UseConnectorBorders      ;
        Module[ i].UseTypedConnectorBorders  := UseTypedConnectorBorders ;
        Module[ i].ConnectorBorderColor      := ConnectorBorderColor     ;
        Module[ i].SelectorColor             := SelectorColor            ;
        Module[ i].SelectorBorderColor       := SelectorBorderColor      ;
        Module[ i].SelectorBorderColorSingle := SelectorBorderColorSingle;
        Module[ i].DisplayColor              := DisplayColor             ;
        Module[ i].DisplayBorderColor        := DisplayBorderColor       ;
        Module[ i].KnobMIDIColor             := KnobMIDIColor            ;
        Module[ i].KnobFocusColor            := KnobFocusColor           ;
        Module[ i].KnobMIDIFocusColor        := KnobMIDIFocusColor       ;
        Module[ i].ViewerColor               := ViewerColor              ;
        Module[ i].ViewerBorderColor         := ViewerBorderColor        ;
        Module[ i].ViewerLineColor           := ViewerLineColor          ;
        Module[ i].ViewerFillColor           := ViewerFillColor          ;
        Module[ i].IndBarPeakColor           := IndBarPeakColor          ;
        Module[ i].IndBarValeyColor          := IndBarValeyColor         ;
        Module[ i].ShowAllowRandomization    := ShowAllowRandomization   ;
      end;
    end;


    procedure   TKnobsWirePanel.EditHistoryChanged;
    begin
      if   Assigned( FOnEditHistoryChanged)
      and  not EditChangesLocked
      then FOnEditHistoryChanged( Self);
    end;


    procedure   TKnobsWirePanel.ClearHistory;
    var
      i : Integer;
    begin
      EditHistory.Clear;
      LockEditChanges;

      try
        for i := 0 to ModuleCount - 1
        do Module[ i].ClearHistory;
      finally
        UnlockEditChanges;
      end;
    end;


    procedure   TKnobsWirePanel.FixBitmaps;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].FixBitmaps;
    end;


    procedure   TKnobsWirePanel.BlockWireUpdates;
    begin
      FWireOverlay.BlockUpdates;
    end;


    procedure   TKnobsWirePanel.UnblockWireUpdates;
    begin
      FWireOverlay.UnBlockUpdates;
    end;


    procedure   TKnobsWirePanel.Search( const aValue: string);
    var
      i     : Integer;
      M     : TKnobsCustomModule;
      U     : string;
      N     : string;
      O     : string;
      IsNum : Boolean;
      aVal  : Integer;
    begin
      U     := UpperCase( aValue);
      aVal  := StrToIntDef( U, -1);
      IsNum := aVal >= 0;

      for i := 0 to ModuleCount - 1
      do begin
        M := Module[ i];
        N := UpperCase( M.Title);
        O := UpperCase( M.Name);
        M.Selected := ( Length( U) > 0) and (( Pos( U, N) > 0) or ( IsNum and ( Pos( U, O) > 0)));
      end;
    end;


    procedure   TKnobsWirePanel.FindUnconnectedModules;
    var
      i       : Integer;
      aModule : TKnobsCustomModule;
    begin
      for i := 0 to ModuleCount - 1
      do begin
        aModule          := Module[ i];
        aModule.Selected := not aModule.IsConnected;
      end;
    end;


    procedure   TKnobsWirePanel.UnFocus( const aSender: TObject);
    begin
      if   Assigned( FOnUnFocus)
      then FOnUnFocus( aSender);
    end;


    procedure   TKnobsWirePanel.FindGhosts( const aGhosts: TStringList);
    var
      i : Integer;
    begin
      if   Assigned( aGhosts)
      then begin
        for i := 0 to ModuleCount - 1
        do begin
          if   ( Module[ i].Top  < 0)
          or   ( Module[ i].Left < 0)
          then aGhosts.Add( Module[ i].Name);
        end;
      end;
    end;


    function    TKnobsWirePanel.CountGhosts: Integer;
    var
      i : Integer;
    begin
      Result := 0;

      for i := 0 to ModuleCount - 1
      do begin
        if   ( Module[ i].Top  < 0)
        or   ( Module[ i].Left < 0)
        then Inc( Result);
      end;
    end;


    procedure   TKnobsWirePanel.SetOnClickFor( aComponentName: string; aUserId: Integer; aHandler: TNotifyEvent);
    var
      aComponent : TComponent;
      i          : Integer;
    begin
      aComponent := FindComponent( aComponentName);

      if   not Assigned( aComponent)
      then begin
        for i := 0 to ModuleCount - 1
        do begin
          aComponent := Module[ i].FindComponent( aComponentName);

          if   Assigned( aComponent)
          then Break;
        end;
      end;

      if   Assigned( aComponent)
      then begin
        aComponent.Tag := aUserId;

        if      aComponent is TKnobsLed           then TKnobsLed          ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsValuedControl then TKnobsValuedControl( aComponent).OnClick := aHandler
        else if aComponent is TKnobsXYControl     then TKnobsXYControl    ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsTextControl   then TKnobsTextControl  ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsIndicator     then TKnobsIndicator    ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsIndicatorBar  then TKnobsIndicatorBar ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsIndicatorText then TKnobsIndicatorText( aComponent).OnClick := aHandler
        else if aComponent is TKnobsDataViewer    then TKnobsDataViewer   ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsConnector     then TKnobsConnector    ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsTextLabel     then TKnobsTextLabel    ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsBox           then TKnobsBox          ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsWirePanel     then TKnobsWirePanel    ( aComponent).OnClick := aHandler
        else if aComponent is TKnobsCustomModule  then TKnobsCustomModule ( aComponent).OnClick := aHandler;
      end;
    end;


//  public

    procedure   TKnobsWirePanel.ConnectorDelete( const aConnector: TKnobsConnector);
    var
      anOutput: TKnobsConnector;
    begin

      // Disconnects every connector on this tree

      if   Assigned( aConnector)
      then begin
        anOutput := aConnector.FindOutput;

        if   Assigned( anOutput)
        then anOutput  .DisConnectDownStreamRecursive
        else aConnector.DisConnectDownStreamRecursive;
      end;
    end;


    procedure   TKnobsWirePanel.ConnectorBreak( const aConnector: TKnobsConnector);
    begin

      // Should disconnect the downstream links.
      // todo: currently breaks the upstream one too

      if   Assigned( aConnector)
      then aConnector.DisConnectDownStream;
    end;


    procedure   TKnobsWirePanel.ConnectorDisconnect( const aConnector: TKnobsConnector);
    begin

      // Takes this connector out of the connection tree

      if   Assigned( aConnector)
      then aConnector.DisConnectAll;
    end;


    procedure   TKnobsWirePanel.SetWireColor( const aConnector: TKnobsConnector; const aColor: TColor);
    begin
      if   Assigned( aConnector)
      then aConnector.ChangeWireColor( aColor);
    end;


    procedure   TKnobsWirePanel.SetWireColorDefault( const aConnector: TKnobsConnector);
    begin
      if   Assigned( aConnector)
      then aConnector.ChangeWireColor( aConnector.DefaultWireColor);
    end;


    function    TKnobsWirePanel.VisitControls( aModuleType: TKnobsModuleType; aControlType: TControlClass; const aName: string; const aHandler: TKnobsOnVisitControl; const aUserData: TObject): Boolean;
    var
      i : Integer;
    begin
      Result := True;

      for i := 0 to ModuleCount - 1
      do begin
        if   not ( aModuleType = 0)
        or   ( Module[ i].ModuleType = aModuleType)
        then begin
          Result := Module[ i].VisitControls( aControlType, aName, aHandler, aUserData);

          if   not Result
          then Break;
        end;
      end;
    end;


    procedure   TKnobsWirePanel.SetTitleFont( const aValue: TFont);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].SetTitleFont( aValue);
    end;


    procedure   TKnobsWirePanel.SetModuleFont( const aValue: TFont);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].SetModuleFont( aValue);
    end;


    procedure   TKnobsWirePanel.CollectCCControls( const aList: TStringList);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].CollectCCControls( aList);
    end;


    function    TKnobsWirePanel.CreateMidiCCList: TStringList;
    begin
      Result := TStringList.Create;
      CollectCCControls( Result);
    end;


    procedure   TKnobsWirePanel.UnassignMidiCC( aMidiCC: Byte);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].UnassignMidiCC( aMidiCC)
    end;


    procedure   TKnobsWirePanel.SetRandomValue( anAmount: TSignal; aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      if   FindFirstSelectedModule = nil                        // When no modules are selected randomize all modules
      then begin
        for i := 0 to ModuleCount - 1
        do Module[ i].SetRandomValue( anAmount, aVariation);
      end
      else begin                                                // otherwise the seleced ones only
        for i := 0 to ModuleCount - 1
        do begin
          if   Module[ i].Selected
          then Module[ i].SetRandomValue( anAmount, aVariation);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.SetRandomValue( anAmount: TSignal); // overload;
    begin
      SetRandomValue( ActiveVariation);
    end;


    procedure   TKnobsWirePanel.Randomize( anAmount: TSignal; aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].Randomize( anAmount, aVariation);
    end;


    procedure   TKnobsWirePanel.Randomize( anAmount: TSignal); // overload;
    var
      i : Integer;
    begin
      for i := 1 to MAX_VARIATIONS - 2
      do Randomize( anAmount, i);
    end;


    procedure   TKnobsWirePanel.Mutate( aProb, aRange: TSignal; aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].Mutate( aProb, aRange, aVariation);
    end;


    procedure   TKnobsWirePanel.Mutate( aProb, aRange: TSignal); // overload;
    var
      i : Integer;
    begin
      for i := 1 to MAX_VARIATIONS - 2
      do Mutate( aProb, aRange, i);
    end;


    procedure   TKnobsWirePanel.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, aVariation);
    end;


    procedure   TKnobsWirePanel.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer); // overload;
    begin
      MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsWirePanel.Mate( anXProb, aMutProb, aMutRange: TSignal);
    var
      i : Integer;
    begin
      for i := 1 to MAX_VARIATIONS - 2
      do MateWith( anXProb, aMutProb, aMutRange, 0, MAX_VARIATIONS - 1, i);
    end;


    procedure   TKnobsWirePanel.Morph( anAmount: TSignal; aDad, aMom, aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].Morph( anAmount, aDad, aMom, aVariation);
    end;


    procedure   TKnobsWirePanel.Morph( anAmount: TSignal; aDad, aMom: Integer); // overload;
    begin
      Morph( anAmount, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsWirePanel.Morph; // Overload;
    var
      i : Integer;
    begin
      for i := 1 to MAX_VARIATIONS - 2
      do Morph(( i - 1) / ( MAX_VARIATIONS - 2), 0, MAX_VARIATIONS - 1, i);
    end;


    procedure   TKnobsWirePanel.LiveMorph( anAmount: TSignal);
    var
      i : Integer;
    begin
      if   Lock( FWormUpdatesBlocked)
      then begin
        BeginStateChange( False);

        try
          for i := 0 to ModuleCount - 1
          do Module[ i].LiveMorph( anAmount);
        finally
          EndStateChange( False, False);
          Unlock( FWormUpdatesBlocked);
        end;
      end;
    end;


    procedure   TKnobsWirePanel.SyncControlRandomization( const aControlTypeName: string; aValue: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].SyncControlRandomization( aControlTypeName, aValue);
    end;


    procedure   TKnobsWirePanel.SyncModuleRandomization( aModuleType: TKnobsModuleType; aValue: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do begin
        if   Module[ i].ModuleType = aModuleType
        then Module[ i].AllowRandomization := aValue;
      end;
    end;


    procedure   TKnobsWirePanel.CountGene( var aCount: Integer);
    var
      i : Integer;
    begin
      aCount := 0;

      for i := 0 to ModuleCount - 1
      do Module[ i].CountGene( aCount);
    end;


    procedure   TKnobsWirePanel.FillGene( const aGene: TKnobsGene; aVariation: Integer);
    var
      i       : Integer;
      aCount  : Integer;
      anIndex : Integer;
    begin
      if   Assigned( aGene)
      then begin
        {$ifdef DEBUG_GENE} LogFmt( LC_RANDOMIZER, 'TKnobsWirePanel.FillGene.Enter for variation %d', [ aVariation]); {$endif}

        aCount     := 0;
        CountGene( aCount);
        aGene.Size := aCount;
        anIndex    := 0;

        for i := 0 to ModuleCount - 1
        do Module[ i].FillGene( aGene, aVariation, anIndex);

        {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.FillGene.Leave'); {$endif}
      end;
    end;


    procedure   TKnobsWirePanel.FillWorm( const aWorm: TKnobsWorm; aVariation: Integer);
    begin
      if   Assigned( aWorm)
      then begin
        {$ifdef DEBUG_GENE} LogFmt( LC_RANDOMIZER, 'TKnobsWirePanel.FillWorm.Enter for variation %d', [ aVariation]); {$endif}

        FillGene( aWorm.Gene, aVariation);
        aWorm.TriggerPaint;

        {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.FillWorm.Leave'); {$endif}
      end;
    end;


    procedure   TKnobsWirePanel.FillWormPanel( const aPanel: TKnobsWormPanel);
    var
      i      : Integer;
      aCount : Integer;
    begin
      if   Assigned( aPanel)
      and  ( aPanel.WormCount = VariationCount)
      then begin
        {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.FillWormPanel.Enter'); {$endif}

        aCount := 0;
        CountGene( aCount);
        aPanel.WormSize := aCount;

        for i := 0 to VariationCount - 1
        do FillWorm( aPanel.Worm[ i], i);

        {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.FillWormPanel.Leave'); {$endif}
      end;
    end;


    procedure   TKnobsWirePanel.FillWormPanel( const aPanel: TKnobsWormPanel; aVariations: TKnobsVariationSet); // overload;
    var
      i      : Integer;
      aCount : Integer;
    begin
      if   Assigned( aPanel)
      and ( aPanel.WormCount = VariationCount)
      then begin
        {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.FillWormPanel.Enter'); {$endif}

        aCount := 0;
        CountGene( aCount);
        aPanel.WormSize := aCount;

        if   aVariations = []
        then FillWorm( aPanel.Worm[ ActiveVariation], ActiveVariation)
        else begin
          for i := 0 to VariationCount - 1
          do begin
            if   i in aVariations
            then FillWorm( aPanel.Worm[ i], i);
          end;
        end;

        {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.FillWormPanel.Leave'); {$endif}
      end;
    end;


    procedure   TKnobsWirePanel.AcceptGene( const aGene : TKnobsGene; aVariation: Integer);
    var
      i       : Integer;
      anIndex : Integer;
    begin
      if   Assigned( aGene)
      then begin
        if   Lock( FWormUpdatesBlocked)
        then begin
          try
            {$ifdef DEBUG_GENE} LogFmt( LC_RANDOMIZER, 'TKnobsWirePanel.AcceptGene.Enter, active variation = %d', [ aVariation]); {$endif}

            BeginStateChange( False);
            anIndex := 0;

            try
              for i := 0 to ModuleCount - 1
              do Module[ i].AcceptGene( aGene, aVariation, anIndex);
            finally
              EndStateChange( False, False);
            end;

            {$ifdef DEBUG_GENE} Log( LC_RANDOMIZER, 'TKnobsWirePanel.AcceptGene.Leave'); {$endif}
          finally
            Unlock( FWormUpdatesBlocked);
          end;
        end;
      end;
    end;


    function    TKnobsWirePanel.EditChangesLocked: Boolean;
    begin
      Result := FEditLockCounter > 0;
    end;


    procedure   TKnobsWirePanel.LockEditChanges;
    begin
      Inc( FEditLockCounter);
    end;


    procedure   TKnobsWirePanel.UnlockEditChanges;
    begin
      if   EditChangesLocked
      then Dec( FEditLockCounter);

      if   not EditChangesLocked
      then EditHistoryChanged;
    end;


    procedure   TKnobsWirePanel.CreateValueSnapShot( const aHistory: TKnobsEditHistory);
    var
      i         : Integer;
      aTimeStamp: TKnobsTimeStamp;
    begin
      // Will record only the changed values, which initially is anything having a non-default value

      if   Assigned( aHistory)
      then begin
        aTimeStamp := GetTimeStamp;      // For a snapshot make all changes happen at the same time

        for i := 0 to ModuleCount - 1
        do Module[ i].CreateValueSnapshot( aTimeStamp, aHistory);
      end;
    end;


    procedure   TKnobsWirePanel.FixActiveRange( aValue: Integer);
    var
      i : Integer;
    begin
      FActiveRange := aValue;

      for i := 0 to ModuleCount - 1
      do Module[ i].FixActiveRange( FActiveRange);
    end;


    procedure   TKnobsWirePanel.SetValueForRange( aRange: Integer; aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to ModuleCount - 1
      do Module[ i].SetValueForRange( aRange, aValue);
    end;


{ ========
  TKnobsCustomModule = class( TCustomPanel)
  // Base type for TModule, it has a bitmap and a title mainly
  private
    FEditHistory               : TKnobsEditHistory;      // A small local undo history
    FAnchor                    : TPoint;                 // Drag origin
    FDistance                  : TPoint;                 // Drag distance
    FTresholdPast              : Boolean;                // Set when dragged over a minimum distance
    FFlags                     : TKnobsModuleFlags;      // Selected / Capturing
    FTitleLabel                : TKnobsEditLabel;        // All editor modules can have a title label, it may be nil
    FTitle                     : string;                 // User editible module title
    FSavedTitle                : string;                 // A copy of the regular FTitle
    FUseAlternateTitle         : Boolean;                // When active shows module name (in short form) instead of the title
    FComment                   : string;                 // Comment as displayed on mouseover in the module selector
    FPicture                   : TPicture;               // Image used for the module selector
    FDocs                      : TStrings;               // Help text for generated context sensitive help
    FModuleType                : TKnobsModuleType;       // The type, used to couple editor modules to synth modules
    FComponentData             : TKnobsComponentData;    // A hashed list of inserted components - on the long name
    FExternalName              : string;                 // For TModExternal - the filename for the dynamically loaded code
    FModuleAlternateTypes      : string;                 // a space separated list of laternate module IDs
    FPageName                  : string;                 // The page in the module selector this module will be on,
                                                         // when the parent is a TTabSheet and that TTabSheet has a non empty
                                                         // Caption that value will be used instead.
    FBitmap                    : TBitmap;                // Overlay bitmap for module
    FUseTitleHints             : Boolean;
    FUseConnectorBorders       : Boolean;
    FUseTypedConnectorBorders  : Boolean;
    FConnectorBorderColor      : TColor;
    FSelectorColor             : TColor;
    FSelectorBorderColor       : TColor;
    FSelectorBorderColorSingle : TColor;
    FDisplayColor              : TColor;
    FDisplayBorderColor        : TColor;
    FKnobMIDIColor             : TColor;
    FKnobFocusColor            : TColor;
    FKnobMIDIFocusColor        : TColor;
    FViewerColor               : TColor;
    FViewerBorderColor         : TColor;
    FViewerLineColor           : TColor;
    FViewerFillColor           : TColor;
    FIndBarPeakColor           : TColor;
    FIndBarValeyColor          : TColor;
    FOpacity                   : Byte;                   // Opacity for background bitmap
    FOpacity                   : Byte;                   // Opacity for background bitmap
    FControlMode               : TDistanceMode;          // Knobs control mode, circular, horizontal or vertical
    FWheelSupportOnKnobs       : Boolean;                // True when MouseWheel ccan control knobs
    FWheelSensitivity          : Integer;                // MouseWheel sensitivity
    FAllowSignalConversion     : Boolean;                // True when dynamic signal conversions are supported by this module
    FAllowRandomization        : Boolean;                // True when patch randomization allowed on this module
    FShowAllowRandomization    : Boolean;                // When true and randomization is not allowed show a special border
    FHasSideEffects            : Boolean;                // True when this module is connected even when inputs only are connected
    FTemplateName              : string;                 // Name of template used to load values from or to save values to, if any.
    FActiveVariation           : Integer;
    FActiveRange               : Integer;
    FOldSpedUp                 : Boolean;
    FVisited                   : Boolean;                // Used in Graph searches
    FVisitOrder                : Integer;                // Used in Graph searches
    FPreCount                  : Integer;                // Used in Graph searches
    FPostCount                 : Integer;                // Used in Graph searches
    FOnValueChanged            : TKnobsOnValueChange;    // Called with the new value
    FOnTextChanged             : TKnobsOnTextChanged;    // Called with the new text
    FOnLog                     : TKnobsOnLog;            // To atach an external logger
    FOnUnFocus                 : TOnUnFocus;             // Called when a control could be unfocussed
  public
    property    EditHistory               : TKnobsEditHistory      read FEditHistory          implements IKnobsUndoable;
  public
    property    Selected                  : Boolean                read GetSelected                write SetSelected;
    property    Capturing                 : Boolean                read GetCapturing               write SetCapturing;
  public
    property    UseAlternateTitle         : Boolean                read FUseAlternateTitle         write SetUseAlternateTitle;
    property    Title                     : string                 read FTitle                     write SetTitle;
    property    FriendlyName              : string                 read GetFriendlyName;
    property    Comment                   : string                 read FComment                   write FComment;
    property    Picture                   : TPicture               read FPicture                   write SetPicture;
    property    Docs                      : TStrings               read FDocs                      write SetDocs;
    property    PageName                  : string                 read GetPageName                write SetPageName;
    property    ModuleType                : TKnobsModuleType       read FModuleType                write SetModuleType;
    property    ExternalName              : string                 read FExternalName              write FExternalName;
    property    ModuleAlternateTypes      : string                 read FModuleAlternateTypes      write FModuleAlternateTypes;
    property    TemplateName              : string                 read FTemplateName              write FTemplateName;
    property    IsSpedUp                  : Boolean                read GetIsSpedUp;
    property    ActiveVariation           : Integer                read FActiveVariation           write SetActiveVariation;
    property    ActiveRange               : Integer                read FActiveRange               write SetActiveRange;
    property    RangeValue                : TSignal                                                write SetRangeValue;
    property    OnLog                     : TKnobsOnLog            read FOnLog                     write FOnLog;
  published
    property    StyleElements                                                                                                         default [];
    property    Color                                                                                                                 default clWhite;
    property    UseTitleHints             : Boolean                read FUseTitleHints             write SetUseTitleHints             default True;
    property    UseConnectorBorders       : Boolean                read FUseConnectorBorders       write SetUseConnectorBorders       default True;
    property    UseTypedConnectorBorders  : Boolean                read FUseTypedConnectorBorders  write SetUseTypedConnectorBorders  default False;
    property    ConnectorBorderColor      : TColor                 read FConnectorBorderColor      write SetConnectorBorderColor      default clWhite;
    property    SelectorColor             : TColor                 read FSelectorColor             write SetSelectorColor             default clGray;
    property    SelectorBorderColor       : TColor                 read FSelectorBorderColor       write SetSelectorBorderColor       default clYellow;
    property    SelectorBorderColorSingle : TColor                 read FSelectorBorderColorSingle write SetSelectorBorderColorSingle default clWhite;
    property    DisplayColor              : TColor                 read FDisplayColor              write SetDisplayColor              default clGray;
    property    DisplayBorderColor        : TColor                 read FDisplayBorderColor        write SetDisplayBorderColor        default clSilver;
    property    KnobMIDIColor             : TColor                 read FKnobMIDIColor             write SetKnobMIDIColor             default CL_MIDI;
    property    KnobFocusColor            : TColor                 read FKnobFocusColor            write SetKnobFocusColor            default CL_FOCUS;
    property    KnobMIDIFocusColor        : TColor                 read FKnobMIDIFocusColor        write SetKnobMIDIFocusColor        default CL_MIDI_FOCUS;
    property    ViewerColor               : TColor                 read FViewerColor               write SetViewerColor               default clGray;
    property    ViewerBorderColor         : TColor                 read FViewerBorderColor         write SetViewerBorderColor         default clGray;
    property    ViewerLineColor           : TColor                 read FViewerLineColor           write SetViewerLineColor           default clWhite;
    property    ViewerFillColor           : TColor                 read FViewerFillColor           write SetViewerFillColor           default $009E9E9E;
    property    IndBarPeakColor           : TColor                 read FIndBarPeakColor           write SetIndBarPeakColor           default clWhite;
    property    IndBarValeyColor          : TColor                 read FIndBarValeyColor          write SetIndBarValeyColor          default clBlack;
    property    TitleLabel                : TKnobsEditLabel        read FTitleLabel                write SetTitleLabel;
    property    Opacity                   : Byte                   read FOpacity                   write SetOpacity                   default 223;
    property    Flat                      : Boolean                read FFlat                      write SetFlat                      default False;
    property    Texture                   : Boolean                read FTexture                   write SetTexture                   default True;
    property    ControlMode               : TDistanceMode          read FControlMode               write SetControlMode;
    property    WheelSupportOnKnobs       : Boolean                read FWheelSupportOnKnobs       write SetWheelSupportOnKnobs;
    property    WheelSensitivity          : Integer                read FWheelSensitivity          write SetWheelSensitivity;
    property    AllowSignalConversion     : Boolean                read FAllowSignalConversion     write SetAllowSignalConversion     default False;
    property    AllowRandomization        : Boolean                read FAllowRandomization        write SetAllowRandomization        default False;
    property    ShowAllowRandomization    : Boolean                read FShowAllowRandomization    write SetShowAllowRandomization    default False;
    property    HasSideEffects            : Boolean                read FHasSideEffects            write FHasSideEffects              default False;
    property    OnClick;
  published
    property    OnValueChanged            : TKnobsOnValueChange    read FOnValueChanged            write FOnValueChanged;
    property    OnTextChanged             : TKnobsOnTextChanged    read FOnTextChanged             write FOnTextChanged;
    property    OnUnFocus                 : TOnUnFocus             read FOnUnFocus                 write FOnUnFocus;
  private
}

    procedure   TKnobsCustomModule.IncludeFlags( aFlags: TKnobsModuleFlags);
    begin
      FFlags := FFlags + aFlags;
    end;


    procedure   TKnobsCustomModule.ExcludeFlags( aFlags: TKnobsModuleFlags);
    begin
      FFlags := FFlags - aFlags;
    end;


    function    TKnobsCustomModule.HasFlags( aFlags: TKnobsModuleFlags): Boolean;
    begin
      Result := aFlags <= FFlags;
    end;


    procedure   TKnobsCustomModule.FixSpedUpControls( MustSpeedUp: Boolean);
    var
      i : Integer;
      V : TKnobsValuedControl;
    begin
      if   MustSpeedUp
      then begin                                                     // Speed up
        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then begin
            V := TKnobsValuedControl( Controls[ i]);

            if      SameText( V.ControlType, 'EnvTimeFast'  ) then V.ControlType := 'EnvTimeFastAr'
            else if SameText( V.ControlType, 'EnvTimeMedium') then V.ControlType := 'EnvTimeMediumAr'
            else if SameText( V.ControlType, 'EnvTimeSlow'  ) then V.ControlType := 'EnvTimeSlowAr';
          end;
        end;
      end
      else begin                                                     // Speed Down
        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then begin
            V := TKnobsValuedControl( Controls[ i]);

            if      SameText( V.ControlType, 'EnvTimeFastAr'  ) then V.ControlType := 'EnvTimeFast'
            else if SameText( V.ControlType, 'EnvTimeMediumAr') then V.ControlType := 'EnvTimeMedium'
            else if SameText( V.ControlType, 'EnvTimeSlowAr'  ) then V.ControlType := 'EnvTimeSlow';
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.ComponentInserted( const aComponent: TComponent);
    begin
      if   Assigned( aComponent)
      and  not ( aComponent is TKnobsCustomModule)
      and  ( aComponent.Name <> '')
      then begin
        if   not Assigned( FComponentData)
        then FComponentData := TKnobsComponentData.Create;

        FComponentData.InsertComponent( aComponent);
      end;
    end;


    procedure   TKnobsCustomModule.ComponentRemoved( const aComponent: TComponent);
    var
      aWp : TKnobsWirePanel;
    begin
      if   Assigned( aComponent)
      then begin
        if   aComponent is TKnobsConnector
        then begin
          aWp := FindWirePanel;

          if   Assigned( aWp)
          then aWp.Disconnect( TKnobsConnector( aComponent));
        end;

        if   Assigned( FComponentData)
        and  not ( aComponent is TKnobsCustomModule)
        then FComponentData.RemoveComponent( aComponent);
      end;
    end;


//  private

    function    TKnobsCustomModule.GetSelected: Boolean;
    begin
      Result := HasFlags( [ mfSelected]);
    end;


    procedure   TKnobsCustomModule.SetSelected( aValue: Boolean);
    const
      Colors : array[ Boolean] of TColor = ( clSilver, clWhite);
    begin
      if   aValue <> Selected
      then begin
        if   aValue
        then IncludeFlags( [ mfSelected])
        else ExcludeFlags( [ mfSelected]);

        if   Assigned( FTitleLabel)
        then begin
          with FTitleLabel
          do begin
            Color       := Colors[ Selected];
            Transparent := not Selected;
          end;
        end;
      end;
    end;


    function    TKnobsCustomModule.GetCapturing: Boolean;
    begin
      Result := HasFlags( [ mfCapturing]);
    end;


    procedure   TKnobsCustomModule.SetCapturing( aValue: Boolean);
    begin
      if   aValue <> Capturing
      then begin
        if   aValue
        then IncludeFlags( [ mfCapturing])
        else ExcludeFlags( [ mfCapturing]);
      end;
    end;


//  private

    procedure   TKnobsCustomModule.SetPicture( const aValue: TPicture);
    begin
      FPicture.Assign( aValue);
    end;


    procedure   TKnobsCustomModule.SetDocs( const aValue: TStrings);
    begin
      FDocs.Assign( aValue);
    end;


    function    TKnobsCustomModule.GetPageName: string;
    begin
      if   HasParent
      and  ( Parent is TTabSheet)
      and  ( TTabSheet( Parent).Caption <> '')
      then Result := TTabSheet( Parent).Caption
      else Result := FPageName;
    end;


    procedure   TKnobsCustomModule.SetPageName( const aValue: string);
    begin
      if   HasParent
      and  ( Parent is TTabSheet)
      and  ( TTabSheet( Parent).Caption <> '')
      then FPageName := TTabSheet( Parent).Caption
      else FPageName := aValue;
    end;


    procedure   TKnobsCustomModule.SetModuleType( aValue : TKnobsModuleType);
    begin
      if   aValue <> FModuleType
      then FModuleType := aValue;
    end;


    function    TKnobsCustomModule.GetIsSpedUp: Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsConnector
        then begin
          if   TKnobsConnector( Controls[ i]).IsSpedUp
          then begin
            Result := True;
            Break;
          end;
        end;
      end;

      if   Result <> FOldSpedUp
      then begin
        FixSpedUpControls( Result);
        FOldSpedUp := Result;
      end;
    end;


    procedure   TKnobsCustomModule.SetActiveVariation( aValue: Integer);
    var
      i : Integer;
    begin
      aValue := Clip( aValue, 0, MAX_VARIATIONS - 1);

      if   aValue <> FActiveVariation
      then begin
        FActiveVariation := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).ActiveVariation := FActiveVariation;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetActiveRange( aValue: Integer);
    var
      i : Integer;
    begin
      if   aValue <> FActiveRange
      then begin
        FActiveRange := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then TKnobsValuedControl( Controls[ i]).ActiveRange := FActiveRange;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetRangeValue( aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).RangeValue := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SetTitle( const aValue: string);
    begin
      if   aValue <> FTitle
      then begin
        if   not UseAlternateTitle
        then begin
          FTitle      := aValue;
          FSavedTitle := aValue;
        end;

        if   Assigned( FTitleLabel)
        then begin
          if   FUseAlternateTitle
          then FTitleLabel.Text := FriendlyName
          else FTitleLabel.Text := FTitle;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetUseAlternateTitle( aValue: Boolean);
    begin
      if   aValue <> FUseAlternateTitle
      then begin
        FUseAlternateTitle := aValue;

        if   Assigned( FTitleLabel)
        then begin
          if   FUseAlternateTitle
          then FTitleLabel.Text := FriendlyName
          else FTitleLabel.Text := Title;
        end;
      end;
    end;


    function    TKnobsCustomModule.GetFriendlyName: string;
    begin
      Result := ModuleLongToFriendlyName( Name);
    end;


    procedure   TKnobsCustomModule.SetTitleLabel( const aValue: TKnobsEditLabel);
    begin
      FTitleLabel := aValue;

      if   Assigned( FTitleLabel)
      and  not ( ComponentState * [ csDestroying, csLoading] = [])
      then FTitleLabel.Caption := Title;
    end;


    procedure   TKnobsCustomModule.SetSelectorColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FSelectorColor
      then begin
        FSelectorColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsSelector
          then TKnobsSelector( Controls[ i]).Color := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetUseTitleHints( aValue : Boolean);
    begin
      if   aValue <> FUseTitleHints
      then begin
        FUseTitleHints := aValue;

        if   Assigned( TitleLabel)
        then TitleLabel.ShowHint := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SetUseConnectorBorders( aValue : Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FUseConnectorBorders
      then begin
        FUseConnectorBorders := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsConnector
          then TKnobsConnector( Controls[ i]).UseBorders := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetUseTypedConnectorBorders( aValue : Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FUseTypedConnectorBorders
      then begin
        FUseTypedConnectorBorders := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsConnector
          then TKnobsConnector( Controls[ i]).UseTypedBorders := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetConnectorBorderColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FConnectorBorderColor
      then begin
        FConnectorBorderColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsConnector
          then TKnobsConnector( Controls[ i]).BorderColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetSelectorBorderColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FSelectorBorderColor
      then begin
        FSelectorBorderColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsSelector
          then TKnobsSelector( Controls[ i]).BorderColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetSelectorBorderColorSingle( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FSelectorBorderColorSingle
      then begin
        FSelectorBorderColorSingle := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsSelector
          then TKnobsSelector( Controls[ i]).BorderColorSingle := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetDisplayColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FDisplayColor
      then begin
        FDisplayColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsDisplay
          then TKnobsDisplay( Controls[ i]).Color := aValue
          else if Controls[ i] is TKnobsGridControl
          then TKnobsGridControl( Controls[ i]).Color := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetDisplayBorderColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FDisplayBorderColor
      then begin
        FDisplayBorderColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsDisplay
          then TKnobsDisplay( Controls[ i]).BorderColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetKnobMIDIColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FKnobMIDIColor
      then begin
        FKnobMIDIColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then TKnobsValuedControl( Controls[ i]).MIDIColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetKnobFocusColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FKnobFocusColor
      then begin
        FKnobFocusColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then TKnobsValuedControl( Controls[ i]).FocusColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetKnobMIDIFocusColor( aValue : TColor);
    var
      i : Integer;
    begin
      if   aValue <> FKnobMIDIFocusColor
      then begin
        FKnobMIDIFocusColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then TKnobsValuedControl( Controls[ i]).MIDIFocusColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetViewerColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FViewerColor
      then begin
        FViewerColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsDataViewer
          then TKnobsDataViewer( Controls[ i]).Color := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetViewerBorderColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FViewerBorderColor
      then begin
        FViewerBorderColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsDataViewer
          then TKnobsDataViewer( Controls[ i]).BorderColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetViewerLineColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FViewerLineColor
      then begin
        FViewerLineColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsDataViewer
          then TKnobsDataViewer( Controls[ i]).LineColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetViewerFillColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FViewerFillColor
      then begin
        FViewerFillColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsDataViewer
          then TKnobsDataViewer( Controls[ i]).FillColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetIndBarPeakColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FIndBarPeakColor
      then begin
        FIndBarPeakColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsIndicatorBar
          then TKnobsIndicatorBar( Controls[ i]).PeakColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetIndBarValeyColor( aValue: TColor);
    var
      i : Integer;
    begin
      if   aValue <> FIndBarValeyColor
      then begin
        FIndBarValeyColor := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsIndicatorBar
          then TKnobsIndicatorBar( Controls[ i]).ValeyColor := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetOpacity( aValue: Byte);
    begin
      if   FOpacity <> aValue
      then begin
        FOpacity := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsCustomModule.SetFlat( aValue: Boolean);
    begin
      if   aValue <> FFlat
      then begin
        FFlat := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsCustomModule.SetTexture( aValue: Boolean);
    begin
      if   aValue <> FTexture
      then begin
        FTexture := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsCustomModule.SetControlMode( aValue: TDistanceMode);
    var
      i : Integer;
    begin
      if   aValue <> FControlMode
      then begin
        FControlMode := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsKnob
          then TKnobsKnob( Controls[ i]).ControlMode := FControlMode;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetWheelSupportOnKnobs( aValue: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).WheelSupport := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SetWheelSensitivity( aValue: Integer);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).WheelSensitivity := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SetAllowSignalConversion( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FAllowSignalConversion
      then begin
        FAllowSignalConversion := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsConnector
          then TKnobsConnector( Controls[ i]).AllowSignalConversion := AllowSignalConversion;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetAllowRandomization( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FAllowRandomization
      then begin
        FAllowRandomization := aValue;
        Invalidate;

        if   ShowAllowRandomization
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   IsVariation( Controls[ i])
            then ( Controls[ i] as IKnobsVariation).ShowAllowRandomization := AllowRandomization;
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetShowAllowRandomization( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FShowAllowRandomization
      then begin
        FShowAllowRandomization := aValue;

        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).ShowAllowRandomization := aValue and AllowRandomization;
        end;

        Invalidate;
      end;
    end;


//  protected

    procedure   TKnobsCustomModule.WMEraseBackground( var aMsg: TWMEraseBkgnd); // message WM_ERASEBKGND;
    begin
      InvalidateKnobs;
      aMsg.Result := -1;
    end;


    procedure   TKnobsCustomModule.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    var
      aWP: TKnobsWirePanel;
    begin
      inherited;

      if   Capturing
      then Exit;

      if   Button = mbLeft
      then begin
        HandleSelection( ssCtrl in Shift);
        SetCapture( Handle);
        Capturing     := True;
        FTresholdPast := False;
        BeginMove( Point( X, Y), Shift);
      end
      else if Button = mbRight
      then begin
        aWP := FindWirePanel;

        if   Assigned( aWP)
        then aWP.HandleRightClick( Self);
      end;

      SendToBack; // See __1 remarks  - needed to keep wires on top of modules
    end;


    procedure   TKnobsCustomModule.MouseMove( Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   Capturing
      then begin
        if   FTresholdPast
        then MoveSelected( FDistance, Shift);

        FDistance := Point( X, Y) - FAnchor;

        if   not FTresholdPast
        then FTresholdPast := Distance( FDistance) > 8;

        if   FTresholdPast
        then begin
          if   ssCtrl in Shift
          then Selected := True;

          MoveSelected( FDistance, Shift);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   Capturing
      then begin
        ReleaseCapture;
        ReleasePeerCapture;

        if   FTresholdPast
        then begin
          if   HasParent
          then Parent.DisableAlign;

          try
            FinalMoveSelected( FDistance, Shift)
          finally
            if   HasParent
            then Parent.EnableAlign;
          end;
        end;

        SendToBack;  // See __1 remarks  - needed to keep wires on top of modules
      end;
    end;


    //  protected

    procedure   TKnobsCustomModule.InvalidateKnobs;
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsKnob
        then TKnobsKnob( Controls[ i]).Invalidate;
      end;
    end;


    procedure   TKnobsCustomModule.InvalidateDisplays;
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsKnob
        then TKnobsKnob( Controls[ i]).InvalidateDisplay;
      end;
    end;


    procedure   TKnobsCustomModule.ValueChanged( aSender: TObject; const aPath, aControlType: string; aValue: TSignal; IsFinal, IsAutomation: Boolean); // virtual;
    var
      aWp: TKnobsWirePanel;
    begin
      if   Assigned( FOnValueChanged)
      then FOnValueChanged( aSender, aPath, aControlType, aValue, IsFinal, IsAutomation)
      else begin
        aWp := FindWirePanel;

        if   Assigned( aWp)
        then aWp.ValueChanged( aSender, MakePath( [ aWp.Name, aPath]), aControlType, aValue, IsFinal, IsAutomation);
      end;
    end;


    procedure   TKnobsCustomModule.TextChanged( aSender: TObject; const aPath, aValue: string);
    var
      aWp: TKnobsWirePanel;
    begin
      if   Assigned( FOnTextChanged)
      then FOnTextChanged( aSender, aPath, aValue)
      else begin
        aWp := FindWirePanel;

        if   Assigned( aWp)
        then aWp.TextChanged( aSender, MakePath( [ aWp.Name, aPath]), aValue);
      end;
    end;


    function    TKnobsCustomModule.FindWirePanel: TKnobsWirePanel;
    var
      aParent : TWinControl;
    begin
      Result  := nil;
      aParent := Parent;

      while Assigned( aParent) and not ( aParent is TKnobsWirePanel)
      do aParent := aParent.Parent;

      if   aParent is TKnobsWirePanel
      then Result := TKnobsWirePanel( aParent);
    end;


    function    TKnobsCustomModule.Snap( const aPoint: TPoint): TPoint;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then Result := aWp.Snap( aPoint)
      else Result := aPoint;
    end;


    procedure   TKnobsCustomModule.Paint; // override;
    const
      UTC = clGray;
    var
      aBmp: TBitmap;
    begin
      aBmp := TBitmap.Create;

      try
        if Flat
        then begin
          aBmp.Width  := Self.Width;
          aBmp.Height := Self.Height;
        end
        else begin
          aBmp.Width  := Self.Width  - 2;
          aBmp.Height := Self.Height - 2;
        end;

        if Texture
        then aBmp.Canvas.StretchDraw( Rect( 0, 0, aBmp.Width, aBmp.Height), FBitmap)
        else begin
          aBmp.Canvas.Pen  .Color := UTC;
          aBmp.Canvas.Pen  .Style := psSolid;
          aBmp.Canvas.Brush.Color := UTC;
          aBmp.Canvas.Brush.Style := bsSolid;
          aBmp.Canvas.Rectangle( 0, 0, aBmp.Width, aBmp.Height);
        end;

        with Canvas
        do begin
          Pen  .Color := Color;
          Pen  .Width := 1;
          Pen  .Style := psSolid;
          Brush.Color := Color;
          Brush.Style := bsSolid;

          if Flat
          then begin
            Rectangle( 0, 0, Width, Height);
            Draw( 0, 0, aBmp, Opacity);
          end
          else begin
            Rectangle( 1, 1, Width - 1, Height - 1);
            Draw( 1, 1, aBmp, Opacity);
            Pen.Color := InterpolateColors( clWhite, Color, 50);
            MoveTo( 0        , Height - 1);
            LineTo( 0        , 0         );
            LineTo( Width    , 0         );
            Pen.Color := InterpolateColors( clBlack, Color, 50);
            MoveTo( 0        , Height - 1);
            LineTo( Width - 1, Height - 1);
            LineTo( Width - 1, 0         );
          end;

          // If enabled show a red inner border for modules that allow for randomization

          if   AllowRandomization
          and  ShowAllowRandomization
          then begin
            Pen.Color := CL_RANDOMIZABLE;
            MoveTo( 1        , Height - 2);
            LineTo( 1        , 1         );
            LineTo( Width - 1, 1         );
            MoveTo( 1        , Height - 2);
            LineTo( Width - 2, Height - 2);
            LineTo( Width - 2, 1         );
          end
        end;
      finally
        aBmp.DisposeOf;
      end;
    end;


    procedure   TKnobsCustomModule.Loaded; // override;
    var
      p : TPoint;
    begin
      inherited;
      p    := Snap( Point( Left, Top));
      Left := P.X;
      Top  := P.Y;
    end;


    procedure   TKnobsCustomModule.AssignBitmap; // virtual;
    begin
      FBitmap := bmSkin;
      FBitmap.Transparent := False;
    end;


    procedure   TKnobsCustomModule.ScrollSelfInView;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      and  ( aWp.BlockCount = 0)
      then aWp.ScrollInView( Self);
    end;


    procedure   TKnobsCustomModule.HandleSelection( CtrlKeyActive: Boolean);
    begin
      if   CtrlKeyActive
      then Selected := not Selected
      else begin
        if   not Selected
        then SelectUnique;
      end;

      BringToFront; // See __1 remarks around
    end;


    procedure   TKnobsCustomModule.BeginMove( anAnchor: TPoint; aShift: TShiftState);
    begin
      FAnchor   := anAnchor;
      FDistance := Point( 0, 0);
    end;


    procedure   TKnobsCustomModule.Move( aDistance: TPoint; aShift: TShiftState);
    var
      aWp      : TKnobsWirePanel;
      aDC      : HDC;
      aRect    : TRect;
      aRgn     : HRGN;
      aPen     : HPen;
      aBrush   : HBrush;
      aColor   : TColor;
      OldPen   : HGDIOBJ;
      OldBrush : HGDIOBJ;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then begin
        aRect.Left   := Left + aDistance.X + 1;
        aRect.Top    := Top  + aDistance.Y + 1;
        aRect.Right  := aRect.Left + Width  - 2;
        aRect.Bottom := aRect.Top  + Height - 2;
        aDC := GetDCEx( aWp.Handle, 0, DCX_PARENTCLIP);

        try
          aRgn := CreateRectRgn( aWp.ClientRect.Left, aWp.ClientRect.Top, aWp.ClientRect.Right, aWp.ClientRect.Bottom);
          SelectClipRgn( aDC, aRgn);
          DeleteObject( aRgn);
          SetRop2( aDC, R2_XORPEN);
          aColor   := clWhite;
          aPen     := CreatePen( PS_SOLID, 2, ColorToRGB( aColor));
          OldPen   := SelectObject( aDC, aPen);
          aBrush   := GetStockObject( NULL_BRUSH);
          OldBrush := SelectObject( aDC, aBrush);

          try
            Rectangle( aDC, aRect.Left, aRect.Top, aRect.Left + aRect.Width, aRect.Top + aRect.Height);
          finally
            SelectObject( aDC, OldBrush);
            SelectObject( aDC, OldPen  );
            DeleteObject( aPen);
          end;
        finally
          ReleaseDC( aWp.Handle, aDC);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.MoveSelected( aDistance: TPoint; aShift: TShiftState);
    var
      i     : Integer;
      aWp   : TKnobsWirePanel;
      aPt   : TPoint;
      aRect : TRect;
    begin
      aWp   := FindWirePanel;
      aPt   := aDistance;
      aRect := Rect( Left, Top, Left + Width, Top + Height);
      aPt.Offset( aRect.CenterPoint);

      if   Assigned( aWp)
      then begin
        with aWp
        do begin
          aWp.ScrollPointInView( aWp.ScreenToClient( ClientToScreen( aPt)));

          for i := 0 to ModuleCount - 1
          do begin
            with Module[ i]
            do begin
              if   Selected
              then Move( aDistance, aShift);
            end;
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.FinalMoveSelected( aDistance: TPoint; aShift: TShiftState);
    var
      P   : TPoint;
      i   : Integer;
      aWp : TKnobsWirePanel;
    begin

      // When CtrlKeyActive is true this is a clone operation and the set of
      // newly created modules should be set into drag mode. See __2 for other
      // cases (also in KnobParser). This will end up in TKnobsWirePanel.AddModules
      // when CtrlKeyActive is true, but as we set the filename to '' and not
      // to '<clip>' it will not go into drag mode (which was just finished
      // for the ctrl+drag mode ...).

      aWp := FindWirePanel;

      if   Assigned( aWp)
      then begin
        MoveSelected( FDistance, aShift);

        with aWp
        do begin
          if   ssCtrl in aShift    // When CtrlKeyActive this was a /copy/ action, a /move/ otherwise
          then PatchReader.ReadString( PatchWriter.WriteString( aWp, '', wmSelected), aWp, aDistance, True, False, False, True, rmAppend, '')
          else begin
            aWp.BeginStateChange( False);
            aWp.BlockWireUpdates;
            DisableAlign;

            try
              for i := 0 to ModuleCount - 1
              do begin
                with Module[ i]
                do begin
                  if   Selected
                  then begin
                    P := Snap( Point( Left, Top) + ScrollOffset + aDistance);
                    if P.X < OffsetLeft then P.X := OffsetLeft;
                    if P.Y < OffsetTop  then P.Y := OffsetTop;
                    P := P - ScrollOffset;
                    SetBounds( P.X, P.Y, Width, Height);
                  end;
                end;
              end;

              // ReStack( DO_RESTACK, DO_REDRAW_WIRES);
              ReStack( DO_RESTACK, NO_REDRAW_WIRES);
              // todo :: **1 :: the - 1000 offset stuff
            finally
              aWp.UnBlockWireUpdates;   // This will invalidate wires, when the lock count gets zero
              aWp.EndStateChange( False, False);
              EnableAlign;
            end;
          end;
        end;

        ScrollSelfInView;
      end;
    end;


    procedure   TKnobsCustomModule.ReleasePeerCapture;
    var
      i   : Integer;
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then begin
        with aWp
        do begin
          for i := 0 to ModuleCount - 1
          do Module[ i].Capturing := False;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.Notification( aComponent: TComponent; anOperation: TOperation); // override;
    begin
      inherited;

      case anOperation of
        opInsert : ComponentInserted( aComponent);
        opRemove : ComponentRemoved ( aComponent);
      end;
    end;


    procedure   TKnobsCustomModule.FixNames;
    var
      aWp : TKnobsWirePanel;
    begin
      aWp := FindWirePanel;

      if   Assigned( aWp)
      then aWp.FixNames( 0);
    end;


    function    TKnobsCustomModule.FindComponentData( const aName: string; const aClass: TComponentClass): TKnobsComponent;
    begin
      if   Assigned( FComponentData)
      then Result := FComponentData.Lookup( aName, aClass)
      else Result := nil;
    end;


    function    TKnobsCustomModule.FindControl( const aName: string; const aClass: TComponentClass): TComponent;
    var
      C : TKnobsComponent;
    begin
      C := FindComponentData( aName, aClass);

      if   Assigned( C)
      then Result := C.FComponent
      else Result := nil;
    end;


    procedure   TKnobsCustomModule.FixComponentData;
    var
      i : Integer;
    begin
      if   Assigned( FComponentData)
      then FComponentData.Clear;

      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TComponent
        then ComponentInserted( TComponent( Controls[ i]));
      end;
    end;


    procedure   TKnobsCustomModule.SetTitleFont( const aValue: TFont);
    var
      i : Integer;
    begin
      if   Assigned( aValue)
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsEditLabel
          then TKnobsEditLabel( Controls[ i]).Font := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetModuleFont( const aValue: TFont);
    var
      i : Integer;
    begin
      if   Assigned( aValue)
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   ( Controls[ i] is TKnobsTextLabel)
          and  not ( Controls[ i] is TKnobsEditLabel)
          then TKnobsTextLabel( Controls[ i]).Font := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.LoadCaptionsFor( const aControl:  TKnobsSelector; const aDependency: TKnobsValuedControl);
    var
      aWirePanel : TKnobsWirePanel;
    begin
      aWirePanel := FindWirePanel;

      if   Assigned( aWirePanel)
      then aWirePanel.LoadCaptionsFor( Self, aControl, aDependency)
      else if Assigned( GOnLoadCaptions)
      then GOnLoadCaptions( Self, Self, aControl, aDependency);
    end;


//  protected

    procedure   TKnobsCustomModule.PreVisit( var aParam: Integer);
    begin
      FPreCount := aParam;
      aParam    := aParam + 1;
    end;


    procedure   TKnobsCustomModule.PostVisit( var aParam: Integer);
    begin
      FPostCount := aParam;
      aParam     := aParam + 1;
    end;


    procedure   TKnobsCustomModule.Visit( var aParam1, aParam2: Integer);
    var
      i : Integer;
      j : Integer;
      C : TKnobsConnector;
      T : TKnobsConnector;
      M : TKnobsCustomModule;
    begin
      if   not FVisited
      then begin
        FVisited    := True;
        FVisitOrder := aParam2;
        aParam2     := aParam2 + 1;
        PreVisit( aParam1);

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsConnector
          then begin
            C := TKnobsConnector( Controls[ i]);

            if   C is TKnobsOutput
            then begin
              for j := 0 to C.FLinks.Count - 1
              do begin
                T := C.FLinks[ j];

                if   Assigned( T)
                then begin
                  M := T.Module;

                  if   Assigned( M) { and not M.FVisited }
                  then M.Visit( aParam1, aParam2);
                end;
              end;
            end;
          end;
        end;
      end;

      PostVisit( aParam1);
    end;


    procedure   TKnobsCustomModule.DumpGraph( const aStrings: TStringList);
    begin
      aStrings.Add( Format( 'name : %14s title : %20s, visit order : %4d, pre : %4d, post: %4d, x : %4d, y : %4d, visited : %s', [ Name, Title, FVisitOrder, FPreCount, FPostCount, Left, Top, BoolToStr( FVisited, True)], AppLocale));
    end;


//  public

    constructor TKnobsCustomModule.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FEditHistory              := TKnobsEditHistory.Create( nil, MODULE_HISTORY_COUNT, HISTORY_STR_LEN, False);
      FComponentData            := TKnobsComponentData.Create;
      Color                     := clWhite;
      UseTitleHints             := True;
      UseConnectorBorders       := True;
      UseTypedConnectorBorders  := False;
      ConnectorBorderColor      := clWhite;
      SelectorColor             := clGray;
      SelectorBorderColor       := clYellow;
      SelectorBorderColorSingle := clWhite;
      DisplayColor              := clGray;
      DisplayBorderColor        := clSilver;
      KnobMIDIColor             := CL_MIDI;
      KnobFocusColor            := CL_FOCUS;
      KnobMIDIFocusColor        := CL_MIDI_FOCUS;
      ViewerColor               := clGray;
      ViewerBorderColor         := clGray;
      ViewerLineColor           := clWhite;
      ViewerFillColor           := $009E9E9E;
      IndBarPeakColor           := clWhite;
      IndBarValeyColor          := clBlack;
      AllowRandomization        := False;
      ShowAllowRandomization    := False;
      FPicture                  := TPicture.Create;
      FDocs                     := TStringList.Create;
      ControlStyle              :=
        [
          csAcceptsControls,
          csCaptureMouse,
          csClickEvents,
          csDoubleClicks,
          csOpaque
        ];
      StyleElements             := [];
      AssignBitmap;
      Width                     :=     MOD_X_UNIT;
      Height                    := 4 * MOD_Y_UNIT;
      Name                      := '';
      FOpacity                  := 223;
      FFlat                     := False;
      FTexture                  := True;
      FTitle                    := '';
      DoubleBuffered            := True;
      ActiveRange               := -1;
    end;


    destructor  TKnobsCustomModule.Destroy; // override;
    begin
      Disconnect;
      FDocs   .DisposeOf;
      FPicture.DisposeOf;
      inherited;
      FComponentData.DisposeOf;
      FEditHistory.DisposeOf;
    end;


    procedure   TKnobsCustomModule.ClearHistory;
    var
      i : Integer;
    begin
      EditHistory.Clear;

      for i := 0 to ControlCount - 1
      do begin
        if   IsUndoable( Controls[ i])
        then ( Controls[ i] as IKnobsUndoable).Clear;
      end;
    end;


    procedure   TKnobsCustomModule.FixControlInsertions;
    var
      i : Integer;
      C : TControl;
    begin
      for i := 0 to ControlCount - 1
      do begin
        C := Controls[ i];

        if   not Assigned( FindComponentData( C.Name, TControl))
        then ComponentInserted( C);
      end;
    end;


    procedure   TKnobsCustomModule.FixBitmaps;
    var
      i : Integer;
    begin
      AssignBitmap;

      for i := 0 to ControlCount - 1
      do begin
        if      Controls[ i] is TKnobsValuedControl then TKnobsValuedControl( Controls[ i]).FixBitmaps
        else if Controls[ i] is TKnobsIndicator     then TKnobsIndicator    ( Controls[ i]).FixBitmaps
        else if Controls[ i] is TKnobsIndicatorBar  then TKnobsIndicatorBar ( Controls[ i]).FixBitmaps
        else if Controls[ i] is TKnobsConnector     then TKnobsConnector    ( Controls[ i]).FixBitmaps
      end;
    end;


    procedure   TKnobsCustomModule.FixAllSynthParams( IsAutomation: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if
          ( not IsAutomation)               or
          ( not IsVariation( Controls[ i])) or
          (
            AllowRandomization                                       and
            IsVariation( Controls[ i])                               and
            ( Controls[ i] as IKnobsVariation).CanAllowRandomization and
            ( Controls[ i] as IKnobsVariation).CanRandomize
          )
        then begin
          if   Controls[ i] is TKnobsValuedControl
          then begin
            if   TKnobsValuedControl( Controls[ i]).StepCount > 1
            then TKnobsValuedControl( Controls[ i]).FixParam;
          end
          else if Controls[ i] is TKnobsXYControl
          then TKnobsXYControl( Controls[ i]).FixParam
          else if
            (( Controls[ i] is TKnobsDisplay     ) and TKnobsDisplay( Controls[ i]).HasEditor) or
             ( Controls[ i] is TKnobsFileSelector)
          then begin
            with TKnobsTextControl( Controls[ i])
            do TextChanged( TextValue, NO_NOTIFY);
          end
          else if Controls[ i] is TKnobsDataMaker
          then begin
            with TKnobsDataMaker( Controls[ i])
            do TextChanged( AsString, DO_NOTIFY);
          end
          else if Controls[ i] is TKnobsGridControl
          then begin
            with TKnobsGridControl( Controls[ i])
            do TextChanged( AsString, DO_NOTIFY);
          end
          else if Controls[ i] is TKnobsData
          then begin
            with TKnobsData( Controls[ i])
            do TextChanged( AsString, DO_NOTIFY);
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.FixOwnership( aRoot, aControl: TWinControl);
    var
      i : Integer;
    begin

      // Recursively make all controls parented by 'aControl' to be owned by
      // 'aRoot', this in order to allow 'aRoot' to write such controls on
      // a stream easily.

      if   Assigned( aControl)
      and  Assigned( aRoot   )
      then begin
        with aControl
        do begin
          for i := 0 to ControlCount - 1
          do begin
            with Controls[ i]
            do begin
              if   ( Owner <> aRoot)
              and  ( Parent = aRoot)
              then begin
                Owner.RemoveComponent( Controls[ i]);
                aRoot.InsertComponent( Controls[ i]);

                if   Controls[ i] is TWinControl
                then FixOwnerShip( aRoot, TWinControl( Controls[ i]));
              end;
            end;
          end;
        end;
      end;
    end;


    function    TKnobsCustomModule.Clone( anOwner: TComponent; aParent: TWinControl; anOffset: TPoint; MustDrag, CopyMidiCC: Boolean): TKnobsCustomModule;
    var
      S : TStream;
      i : Integer;
      j : Integer;
   // V : TKnobsSelector;
   // C : TStringList;
    begin

      // Make all children parented by self to be owned by self, so they will be
      // streamed out. Then write self on a stream and read it back into a new
      // TCustomModule. Finally make the clone owned and parented as indicated in
      // the parameters passed, then position it relative to it's parent's (top, left).

      FixOwnerShip( Self, Self);
      S := TMemoryStream.Create;

      try
        S.WriteComponent( Self);
        S.Seek( 0, 0);
        Result := TKnobsCustomModule( S.ReadComponent( nil));

        if   Assigned( Result)
        then begin

          // ReadComponent fux up zero values in ValuedControls, fix 'm manually then.
          // Same for empty strings in TextControls.
          // Same for empty style elements in controls.
          // And TKnobsDataMaker.AsString is not a published property, so do that manually as well.
          // MidiCC needs manual work to clear it on the clone.
          // Variations is not published, so do that manually as well.
          // ActiveVariation is not published.
          // Range stuff is not published

          for i := 0 to ControlCount - 1
          do begin
            Result.Controls[ i].StyleElements := Controls[ i].StyleElements;

            if   Controls[ i] is TKnobsValuedControl
            then begin
              TKnobsValuedControl( Result.Controls[ i]).KnobPosition     := TKnobsValuedControl( Controls[ i]).KnobPosition;
              TKnobsValuedControl( Result.Controls[ i]).DefaultPosition  := TKnobsValuedControl( Controls[ i]).DefaultPosition;
              TKnobsValuedControl( Result.Controls[ i]).FPrevPosition    := TKnobsValuedControl( Controls[ i]).DefaultPosition;
              TKnobsValuedControl( Result.Controls[ i]).CopyRangesFrom(     TKnobsValuedControl( Controls[ i]));
              TKnobsValuedControl( Result.Controls[ i]).ActiveRange      := TKnobsValuedControl( Controls[ i]).ActiveRange;
            end
            else if Controls[ i] is TKnobsXYControl
            then begin
              TKnobsXYControl( Result.Controls[ i]).KnobPositionX     := TKnobsXYControl( Controls[ i]).KnobPositionX;
              TKnobsXYControl( Result.Controls[ i]).DefaultPositionX  := TKnobsXYControl( Controls[ i]).DefaultPositionX;
              TKnobsXYControl( Result.Controls[ i]).FPrevPositionX    := TKnobsXYControl( Controls[ i]).DefaultPositionX;
              TKnobsXYControl( Result.Controls[ i]).KnobPositionY     := TKnobsXYControl( Controls[ i]).KnobPositionY;
              TKnobsXYControl( Result.Controls[ i]).DefaultPositionY  := TKnobsXYControl( Controls[ i]).DefaultPositionY;
              TKnobsXYControl( Result.Controls[ i]).FPrevPositionY    := TKnobsXYControl( Controls[ i]).DefaultPositionY;
            end
            else if Controls[ i] is TKnobsTextControl
            then ( Result.Controls[ i] as TKnobsTextControl).TextValue := TKnobsTextControl( Controls[ i]).TextValue
            else if Controls[ i] is TKnobsDataMaker
            then ( Result.Controls[ i] as TKnobsDataMaker).AsString := TKnobsDataMaker( Controls[ i]).AsString
            else if Controls[ i] is TKnobsGridControl
            then ( Result.Controls[ i] as TKnobsGridControl).AsString := TKnobsGridControl( Controls[ i]).AsString
            else if Controls[ i] is TKnobsData
            then ( Result.Controls[ i] as TKnobsData).AsString := TKnobsData( Controls[ i]).AsString
            else if Controls[ i] is TKnobsMazeGraphViewer
            then ( Result.Controls[ i] as TKnobsMazeGraphViewer).ImportForthExports;

            if   IsAutomatable( Controls[ i])
            then begin
              if   not CopyMidiCC
              then ( Result.Controls[ i] as IKnobsAutomatable).AssignedMIDICC := 0; // Do not allow a MIDI CC to be set in a clone
            end;

            if   IsVariation( Controls[ i])
            then begin
              for j := 0 to ( Result.Controls[ i] as IKnobsVariation).VariationCount - 1
              do ( Result.Controls[ i] as IKnobsVariation).VariationValue[ j] := ( Controls[ i] as IKnobsVariation).VariationValue[ j];
            end;

            Result.ActiveVariation := ActiveVariation;

            // todo: the below is not working ... something changes the captions after ... @@captions@@

          {

            if Controls[ i] is TKnobsSelector
            then begin
              V := TKnobsSelector( Controls[ i]);

              if V.ControlType <> ''
              then begin
                if Converters.IsEnumerable( V.ControlType)
                then begin
                  C := Converters.CreateEnumeration( V.ControlType);

                  if Assigned( C)
                  then begin
                    try
                      V.Captions := C;
                    finally
                      C.DisposeOf;
                    end;
                  end;
                end;
              end;
            end;
          }

          end;

          if   Assigned( anOwner)
          then anOwner.InsertComponent( Result);

          Result.ActiveRange := ActiveRange;

          with Result
          do begin
            if   MustDrag
            then begin
              Left := anOffset.X;
              Top  := anOffset.Y - 1000;
              // Top  := anOffset.Y;
              // todo : **1 :: try to get rid of offset - 1000, see other **1 remarks too
            end
            else begin
              Left := anOffset.X;
              Top  := anOffset.Y;

              if   Top < 0
              then Top := 0;
            end;

            Parent   := aParent;
            Selected := True;
          end;

          with Result
          do begin
            if   MustDrag
            and  HasParent
            and  Parent.HasParent
            then begin
              SetCapture( Handle);
              Capturing := True;
            end;

            if   not MustDrag
            then SelectUnique;
          end;
        end;
      finally
        S.DisposeOf;
      end;
    end;


    procedure   TKnobsCustomModule.SetKnobValue( const aType, aName: string; aValue: Integer);
    var
      aValuedControl : TKnobsValuedControl;
      anXYControl    : TKnobsXYControl;
      aSuffix        : string;
    begin
      if
        SameText( aType, 'TKnobsSelector'     ) or
        SameText( aType, 'TKnobsKnob'         ) or
        SameText( aType, 'TKnobsSmallKnob'    ) or
        SameText( aType, 'TKnobsNoKnob'       ) or
        SameText( aType, 'TKnobsSlider'       ) or
        SameText( aType, 'TKnobsHSlider'      ) or
        SameText( aType, 'TKnobsValuedButton' ) or
        SameText( aType, 'TKnobsValuedButtons')
      then begin
        aValuedControl := FindValuedControl( aName);

        if   Assigned( aValuedControl)
        then aValuedControl.KnobPosition := aValue;
      end
      else if
        SameText( aType, 'TKnobsPad')
      then begin
        aSuffix     := Copy( aName, Length( aName), 1);
        anXYControl := FindXYControl( Copy( aName, 1, Length( aName) - 1));

        if   Assigned( anXYControl)
        then begin
          if   SameText( aSuffix, 'x')
          then anXYControl.KnobPositionX := aValue
          else if SameText( aSuffix, 'y')
          then anXYControl.KnobPositionY := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetDetailValue( const aType, aName: string; const aValue: string);
    var
      aDisplay      : TKnobsDisplay;
      aFileSelector : TKnobsFileSelector;
      aConnector    : TKnobsConnector;
      aDataMaker    : TKnobsDataMaker;
      aGridControl  : TKnobsGridControl;
      aData         : TKnobsData;
    begin
      if   SameText( aType, 'TKnobsDisplay')
      then begin
        if   SameText( aName, 'waveplayer_display_filename')   // Some trickery to be able to read old patches
        then begin                                             // which did not have the file selector component
          aFileSelector := FindFileSelector( 'waveplayer_fileselector_filename');

          if   Assigned( aFileSelector)
          then aFileSelector.SetText( aValue);
        end
        else begin
          aDisplay := FindDisplay( aName);

          if   Assigned( aDisplay)
          then aDisplay.SetText( aValue);
        end;
      end
      else if SameText( aType, 'TKnobsFileSelector')
      then begin
        aFileSelector := FindFileSelector( aName);

        if   Assigned( aFileSelector)
        then aFileSelector.SetText( aValue);
      end
      else if
        SameText( aType, 'TKnobsInput' ) or
        SameText( aType, 'TKnobsOutput')
      then begin
        aConnector := FindConnector( aName);

        if   Assigned( aConnector)
        then aConnector.WireColor := TColor( StrToInt64( aValue) and $ffffffff);
      end
      else if SameText( aType, 'TKnobsDataMaker')
      then begin
        aDataMaker := FindDataMaker( aName);

        if   Assigned( aDataMaker)
        then aDataMaker.AsString := aValue;
      end
      else if SameText( aType, 'TKnobsGridControl')
      then begin
        aGridControl := FindGridControl( aName);

        if   Assigned( aGridControl)
        then aGridControl.AsString := aValue;
      end
      else if SameText( aType, 'TKnobsData')
      then begin
        aData := FindData( aName);

        if   Assigned( aData)
        then aData.AsString := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SetLockOn( const aType, aName: string);
    var
      aValuedControl : TKnobsValuedControl;
    begin
      if
        SameText( aType, 'TKnobsKnob'     ) or
        SameText( aType, 'TKnobsSmallKnob') or
        SameText( aType, 'TKnobsNoKnob'   ) or
        SameText( aType, 'TKnobsSlider'   ) or
        SameText( aType, 'TKnobsHSlider'  )
      then begin
        aValuedControl := FindValuedControl( aName);

        if   Assigned( aValuedControl)
        then aValuedControl.Locked := True;
      end;
    end;


    procedure   TKnobsCustomModule.SetCCOn( const aType, aName: string; aValue: Byte);
    var
      aControl : IKnobsAutomatable;
    begin
      aControl := FindAutomatable( aName);

      if   Assigned( aControl)
      then begin
        if   aControl.AllowAutomation
        then aControl.AssignedMIDICC := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SetRndOn( const aType, aName: string; aValue: Byte; WasRead: Boolean);
    var
      aControl : IKnobsVariation;
    begin
      aControl := FindVariation( aName);

      if   Assigned( aControl)
      then begin
        if   WasRead
        then aControl.AllowRandomization := ( aValue <> 0) and aControl.CanAllowRandomization
        else begin
          aControl.AllowRandomization :=
            aControl.CanAllowRandomization and
            aControl.CanRandomize          and
            ( aValue <> 0)
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetVarOn( const aType, aName, aValue: string; WasRead: Boolean);
    var
      aControl : IKnobsVariation;
    begin
      if   SameText( aType, 'TKnobsPad')
      then aControl := FindVariation( Copy( aName, 1, Length( aName) - 1))
      else aControl := FindVariation( aName);

      if   Assigned( aControl)
      then begin
        if   WasRead
        then aControl.VariationsAsString := aValue
        else aControl.SetDefaultVariations;
      end;
    end;


    procedure   TKnobsCustomModule.SetLowHighMarksOn( const aType, aName: string; const aLow, aHigh: string);
    var
      aControl : TKnobsValuedControl;
    begin
      aControl := FindValuedControl( aName);

      if   Assigned( aControl)
      then begin
        aControl.FRanges.Clear;
        aControl.FRanges.LowRangeAsStr  := aLow ;
        aControl.FRanges.HighRangeAsStr := aHigh;
      end;
    end;


    procedure   TKnobsCustomModule.SetSignal( const aName: string; const aValue: TSignal);
    var
      i         : Integer;
      p         : Integer;
      aControl  : TKnobsValuedControl;
      aMaker    : TKnobsDataMaker;
      aSlider   : TKnobsSlider;
      aSelector : TKnobsSelector;
    begin
      if   ModuleType = 1211
      then begin
        if   SameText( aName, 'rnd')
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              aSlider := TKnobsSlider( Controls[ i]);
              aSlider.KnobPosition := Random( aSlider.StepCount);
            end;
          end;
        end
        else if SameText( aName, 'rndone')
        then begin
          p := Random( Round( aValue));

          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              if p = 0
              then begin
                aSlider := TKnobsSlider( Controls[ i]);
                aSlider.KnobPosition := Random( aSlider.StepCount);
              end;

              Dec( p);
            end;
          end;
        end;
      end
      else if   ModuleType = 1233
      then begin
        if   SameText( aName, 'clr')
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              aSlider := TKnobsSlider( Controls[ i]);
              aSlider.KnobPosition := 0;
            end;
          end;
        end
        else if   SameText( aName, 'rnd')
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              aSlider := TKnobsSlider( Controls[ i]);
              aSlider.KnobPosition := Random( aSlider.StepCount);
            end;
          end;
        end
        else if SameText( aName, 'rndone')
        then begin
          p := Random( Round( aValue));

          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              if p = 0
              then begin
                aSlider := TKnobsSlider( Controls[ i]);
                aSlider.KnobPosition := Random( aSlider.StepCount);
              end;

              Dec( p);
            end;
          end;
        end;
      end
      else if   ModuleType = 1215
      then begin
        if   SameText( aName, 'rnd')
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              aSlider := TKnobsSlider( Controls[ i]);
              aSlider.KnobPosition := Random( aSlider.StepCount);
            end;
          end;
        end
        else if SameText( aName, 'rndone')
        then begin
          p := Random( Round( aValue));

          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              if p = 0
              then begin
                aSlider := TKnobsSlider( Controls[ i]);
                aSlider.KnobPosition := Random( aSlider.StepCount);
              end;

              Dec( p);
            end;
          end;
        end;
      end
      else if ModuleType = 1230
      then begin
        if   SameText( aName, 'rnd')
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSelector)
            and  ( Controls[ i].Tag = 1)
            then begin
              aSelector := TKnobsSelector( Controls[ i]);
              aSelector.KnobPosition := Random( aSelector.StepCount);
            end;
          end;
        end;
      end
      else if ModuleType = 1224
      then begin
        if   Pos( 'value', aName) = 1
        then begin
          for i := 0 to ControlCount - 1
          do begin
            if   ( Controls[ i] is TKnobsSlider)
            and  ( Controls[ i].Tag = 1)
            then begin
              aSlider := TKnobsSlider( Controls[ i]);

              if   SameText( aSlider.ShortName, aName)
              then aSlider.KnobPosition := Converters.ValueToPos( aSlider.ControlType, aValue, aSlider.StepCount)
            end;
          end;
        end;
      end
      else begin
        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsValuedControl
          then begin
            aControl := TKnobsValuedControl( Controls[ i]);

            if   SameText( aName, aControl.ShortName)
            then begin
              aControl.Locked       := False;
              aControl.KnobPosition := Round( aValue);
              Break;
            end;
          end
          else if Controls[ i] is TKnobsDataMaker
          then begin
            aMaker := TKnobsDataMaker( Controls[ i]);

            if   SameText( aName, aMaker.ShortName)
            then begin
              aMaker.GraphMode := TKnobsDMGraphMode( Round( aValue));
              Break;
            end
            else if SameText( aName, 'rnd')
            then begin
              aMaker.Randomize( aValue);
              Break;
            end
            else if SameText( aName, 'autoscale')
            then begin
              aMaker.AutoScale := SignalToLogic( aValue);
              Break;
            end;
          end
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetLight( const aName: string; const aValue: TSignal);
    var
      i              : Integer;
      anIndicator    : TKnobsIndicator;
      anIndBar       : TKnobsIndicatorBar;
      anIndText      : TKnobsIndicatorText;
      aSelector      : TKnobsSelector;
      aValuedControl : TKnobsValuedControl;
      aMaze          : TKnobsMazeGraphViewer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsIndicator
        then begin
          anIndicator := TKnobsIndicator( Controls[ i]);

          if   SameText( aName, anIndicator.ShortName)
          then begin
            anIndicator.Active := aValue > 0;
            Break;
          end;
        end
        else if Controls[ i] is TKnobsIndicatorBar
        then begin
          anIndBar := TKnobsIndicatorBar( Controls[ i]);

          if   SameText( aName, anIndBar.ShortName)
          then begin
            anIndBar.Value := aValue * anIndBar.IndicatorCount;
            Break;
          end;
        end
        else if Controls[ i] is TKnobsIndicatorText
        then begin
          anIndText := TKnobsIndicatorText( Controls[ i]);

          if   SameText( aName, anIndText.ShortName)
          then begin
            anIndText.Value := aValue;
            Break;
          end;
        end
        else if Controls[ i] is TKnobsSelector
        then begin
          aSelector := TKnobsSelector( Controls[ i]);

          if   SameText( aName, aSelector.ShortName)
          then begin
            aSelector.KnobPosition := Round( aValue);
            Break;
          end;
        end
        else if Controls[ i] is TKnobsMazeGraphViewer
        then begin
          aMaze := TKnobsMazeGraphViewer( Controls[ i]);

          if   SameText( aName, aMaze.ShortName)
          then begin
            aMaze.SelectMazeType( Round( aValue));
            Break;
          end;
        end
        else if ( Controls[ i] is TKnobsValuedControl)
        then begin
          aValuedControl := TKnobsValuedControl( Controls[ i]);

          if   SameText( aName, aValuedControl.ShortName)
          then begin
            if   ( aValuedControl.KnobPosition > 0)
            and  ( aValuedControl.StepCount    > 0)
            then aValuedControl.KnobPosition := 0
            else aValuedControl.KnobPosition := aValuedControl.StepCount - 1;

            aValuedControl.KnobPosition := Converters.ValueToPos( aValuedControl.ControlType, aValue, aValuedControl.StepCount);
            Break;
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetData( const aName: string; const aValue: TSignalArray);
    var
      i           : Integer;
      j           : Integer;
      aDataViewer : TKnobsDataViewer;
      aLocData    : TKnobsData;
      aMaze       : TKnobsMazeGraphViewer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsDataViewer
        then begin
          aDataViewer := TKnobsDataViewer( Controls[ i]);

          if   SameText( aName, aDataViewer.ShortName)
          then aDataViewer.YData := aValue;
        end
        else if Controls[ i] is TKnobsData
        then begin
          aLocData := TKnobsData( Controls[ i]);

          if   SameText( aName, aLocData.ShortName)
          then aLocData.Data := aValue;
        end
        else if Controls[ i] is TKnobsMazeGraphViewer
        then begin
          aMaze := TKnobsMazeGraphViewer( Controls[ i]);

          if   SameText( aName, aMaze.ShortName)
          then begin
            for j := 0 to Min( aMaze.HunterCount, Length( aValue)) - 1
            do aMaze.HunterPosition[ j] := Round( aValue[ j]);
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetXYData( const aName: string; const aValue: TSignalPairFifo);
    var
      i           : Integer;
      aDataViewer : TKnobsDataViewer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsDataViewer
        then begin
          aDataViewer := TKnobsDataViewer( Controls[ i]);

          if   SameText( aName, aDataViewer.ShortName)
          then aDataViewer.XYData := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetInfo( const aName: string; const aValue: string);
    var
      i        : Integer;
      aDisplay : TKnobsDisplay;
      aGrid    : TKnobsGridControl;
      aLabel   : TKnobsTextLabel;
      aPad     : TKnobsPad;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsDisplay
        then begin
          aDisplay := TKnobsDisplay( Controls[ i]);

          if   SameText( aName, aDisplay.ShortName)
          then aDisplay.Caption := aValue;
        end
        else if Controls[ i] is TKnobsGridControl
        then begin
          aGrid := TKnobsGridControl( Controls[ i]);

          if   SameText( aName, aGrid.ShortName)
          then aGrid.AsString := aValue;
        end
        else if Controls[ i] is TKnobsTextLabel
        then begin
          aLabel := TKnobsTextLabel( Controls[ i]);

          if   SameText( aName, aLabel.ShortName)
          then aLabel.Caption := aValue;
        end
        else if Controls[ i] is TKnobsPad
        then begin
          aPad := TKnobsPad( Controls[ i]);

          if   SameText( aName, 'filename')
          then aPad.LoadBackGroundFile( aValue);
        end
      end;
    end;


    procedure   TKnobsCustomModule.SetCursorData( const aName: string; const aValue: TSignalPair);
    var
      i          : Integer;
      aDataMaker : TKnobsDataMaker;
      aGrid      : TKnobsGridControl;
      aMAze      : TKnobsMazeGraphViewer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsDataMaker
        then begin
          aDataMaker := TKnobsDataMaker( Controls[ i]);

          if   SameText( aName, aDataMaker.ShortName) and aDataMaker.FollowCursor
          then aDataMaker.CursorXY := aValue;
        end
        else if Controls[ i] is TKnobsGridControl
        then begin
          aGrid := TKnobsGridControl( Controls[ i]);

          if   SameText( aName, aGrid.ShortName)
          then begin
            aGrid.CursorX := aValue.X;
            aGrid.CursorY := aValue.Y;
          end;
        end
        else if Controls[ i] is TKnobsMazeGraphViewer
        then begin
          aMaze := TKnobsMazeGraphViewer( Controls[ i]);

          if   SameText( aName, aMaze.ShortName)
          then begin
            aMaze.SwanX := aValue.X;
            aMaze.SwanY := aValue.Y;
          end;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetGridData( const aName: string; const aValue: string);
    var
      i     : Integer;
      aGrid : TKnobsGridControl;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsGridControl
        then begin
          aGrid := TKnobsGridControl( Controls[ i]);

          if   SameText( aName, aGrid.ShortName)
          then aGrid.AsString := aValue;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.Dump( var aFile: TextFile; anInd: Integer);
    var
      i : Integer;
    begin
      WriteLnInd( aFile, anInd, Format( 'module : %s', [ Name], AppLocale));

      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsConnector
        then TKnobsConnector( Controls[ i]).Dump( aFile, anInd + 1)
        else if Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).Dump( aFile, anInd + 1);
      end;

      WriteLnInd( aFile, anInd, Format( 'end module : %s', [ Name], AppLocale));
    end;


    procedure   TKnobsCustomModule.Log( aLogClass: TLogClass; const aMsg: string);
    begin
      if   Parent is TKnobsWirePanel
      then TKnobsWirePanel( Parent).Log( aLogClass, Format( 'custom module %s( %s)', [ Name, aMsg], AppLocale))
      else if Assigned( FOnLog)
      then FOnLog( Self, aLogClass, Format( 'custom module %s( %s)', [ Name, aMsg], AppLocale));
    end;


    procedure   TKnobsCustomModule.LogFmt( aLogClass: TLogClass; const aFmt: string; const anArgs: array of const);
    begin
      Log( aLogClass, Format( aFmt, anArgs, AppLocale));
    end;


    function    TKnobsCustomModule.AsBitmap: TBitmap;
    begin
      Result := TBitmap.Create;
      Result.Width  := Width;
      Result.Height := Height;
      PaintTo( Result.Canvas, 0, 0);
    end;


// public

    function    TKnobsCustomModule.FindValuedControl( const aName: string): TKnobsValuedControl;
    begin
      Result := TKnobsValuedControl( FindControl( aName, TKnobsValuedControl));
    end;


    function    TKnobsCustomModule.FindTextControl( const aName: string): TKnobsTextControl;
    begin
      Result := TKnobsTextControl( FindControl( aName, TKnobsTextControl));
    end;


    function    TKnobsCustomModule.FindXYControl( const aName: string): TKnobsXYControl;
    begin
      Result := TKnobsXYControl( FindControl( aName, TKnobsXYControl));
    end;


    function    TKnobsCustomModule.FindDisplay( const aName: string): TKnobsDisplay;
    begin
      Result := TKnobsDisplay( FindControl( aName, TKnobsDisplay));
    end;


    function    TKnobsCustomModule.FindFileSelector( const aName: string): TKnobsFileSelector;
    begin
      Result := TKnobsFileSelector( FindControl( aName, TKnobsFileSelector));
    end;


    function    TKnobsCustomModule.FindConnector( const aName: string): TKnobsConnector;
    begin
      Result := TKnobsConnector( FindControl( aName, TKnobsConnector));
    end;


    function    TKnobsCustomModule.FindDataMaker( const aName: string): TKnobsDataMaker;
    begin
      Result := TKnobsDataMaker( FindControl( aName, TKnobsDataMaker));
    end;


    function    TKnobsCustomModule.FindGridControl( const aName: string): TKnobsGridControl;
    begin
      Result := TKnobsGridControl( FindControl( aName, TKnobsGridControl));
    end;


    function    TKnobsCustomModule.FindData( const aName: string): TKnobsData;
    begin
      Result := TKnobsData( FindControl( aName, TKnobsData));
    end;


    function    TKnobsCustomModule.FindAutomatable( const aName: string): IKnobsAutomatable;
    var
      C : TKnobsComponent;
    begin
      C := FindComponentData( aName, nil);

      if   Assigned( C)
      and  IsAutomatable( C.FComponent)
      then Result := C.FComponent as IKnobsAutomatable
      else Result := nil;
    end;


    function    TKnobsCustomModule.FindVariation( const aName: string): IKnobsVariation;
    var
      C : TKnobsComponent;
    begin
      C := FindComponentData( aName, nil);

      if   Assigned( C)
      and  IsVariation( C.FComponent)
      then Result := C.FComponent as IKnobsVariation
      else Result := nil;
    end;


//  public

    procedure   TKnobsCustomModule.Restack( DoRestack, RedrawWires: Boolean);

      function DoSort( Item1, Item2: Pointer): Integer;

        function CompareModulePositions( F1, F2: TKnobsCustomModule): Integer;
        begin
          if   F1.Left < F2.Left
          then Result := -1
          else if F1.Left > F2.Left
          then Result := 1
          else begin
            if   F1.Top < F2.Top
            then Result := -1
            else if F1.Top > F2.Top
            then Result := 1
            else Result := 0;
          end;
        end;

      begin
        Result := CompareModulePositions( TKnobsCustomModule( Item1), TKnobsCustomModule( Item2));
      end;

    var
      i           : Integer;
      CurrentTop  : Integer;
      CurrentLeft : Integer;
      MinTop      : Integer;
      aWirePanel  : TKnobsWirePanel;
    begin
      if   not DoRestack
      then Exit;

      aWirePanel := FindWirePanel;

      if   Assigned( aWirePanel)
      and  aWirePanel.SnapActive
      then begin
        with aWirePanel
        do begin

          CurrentLeft := OffsetLeft - ScrollOffset.X;
          CurrentTop  := OffsetTop  - ScrollOffset.Y;

          for i := 0 to ModuleCount - 1
          do begin
            with Module[ i]
            do begin
              if   Top < CurrentTop
              then CurrentTop := Top;

              if   Left < CurrentLeft
              then CurrentLeft := Left;
            end;
          end;

          MinTop := CurrentTop;
          FModules.Sort( @ DoSort);

          for i := 0 to ModuleCount - 1
          do begin
            with Module[ i]
            do begin
              TabOrder := i;

              if   Left > CurrentLeft
              then begin
                CurrentLeft := Left;
                CurrentTop  := MinTop;
              end;

              if   Top < CurrentTop
              then Top := CurrentTop;

              CurrentTop := Top + Height;
            end;
          end;

          if   RedrawWires
          then InvalidateWires;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.FixupInsertion;
    begin

      // note: first Restack then FixNames, otherwise naming conflicts will result
      // between the compiled patch and the editor patch.

      Restack( DO_RESTACK, DO_REDRAW_WIRES);
      FixNames;
    end;


    procedure   TKnobsCustomModule.FixDisplays;
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).FixDisplay;
      end;
    end;


    procedure   TKnobsCustomModule.Disconnect;
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsConnector
        then TKnobsConnector( Controls[ i]).DisConnectAll;
      end;
    end;


    procedure   TKnobsCustomModule.SelectUnique;
    var
      aWp : TKnobsWirePanel;
    begin

      // Select this module uniquely in the patch

      aWp := FindWirePanel;

      if   Assigned( aWp)
      and  ( aWp.BlockCount = 0)
      then aWp.SelectUnique( Self);
    end;


    procedure   TKnobsCustomModule.FixSelection;
    var
      aWp : TKnobsWirePanel;
    begin

      // When no modules are selected select this Module uniquely

      aWp := FindWirePanel;

      if   Assigned( aWp)
      and  ( aWp.BlockCount = 0)
      then aWp.FixSelection( Self);
    end;


    procedure   TKnobsCustomModule.CollectInputsWithBaseName( var anInputs: TKnobsConnectors; const aBaseName: string);
    var
      i        : Integer;
      aParts   : TStringList;
      aResults : TStringList;
    begin
      SetLength( anInputs, 0);
      aResults := TStringList.Create;

      try
        aResults.Sorted := True;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsInput
          then begin
            aParts := Explode( Controls[ i].Name, '_');

            try
              if   ( aParts.Count = 2)
              and  ( Pos( aBaseName, aParts[ 1]) = 1)
              then aResults.AddObject( aParts[ 1], Controls[ i]);
            finally
              aParts.DisposeOf;
            end;
          end;
        end;

        for i := 0 to aResults.Count - 1
        do begin
          SetLength( anInputs, Length( anInputs) + 1);
          anInputs[ Length( anInputs) - 1] := TKnobsOutput( aResults.Objects[ i]);
        end;
      finally
        aResults.DisposeOf;
      end;
    end;


    procedure   TKnobsCustomModule.CollectOutputsWithBaseName( var anOutputs: TKnobsConnectors; const aBaseName: string);
    var
      i        : Integer;
      aParts   : TStringList;
      aResults : TStringList;
    begin
      SetLength( anOutputs, 0);
      aResults := TStringList.Create;

      try
        aResults.Sorted := True;

        for i := 0 to ControlCount - 1
        do begin
          if   Controls[ i] is TKnobsOutput
          then begin
            aParts := Explode( Controls[ i].Name, '_');

            try
              if   ( aParts.Count = 2)
              and  ( Pos( aBaseName, aParts[ 1]) = 1)
              then aResults.AddObject( aParts[ 1], Controls[ i]);
            finally
              aParts.DisposeOf;
            end;
          end;
        end;

        for i := 0 to aResults.Count - 1
        do begin
          SetLength( anOutputs, Length( anOutputs) + 1);
          anOutputs[ Length( anOutputs) - 1] := TKnobsOutput( aResults.Objects[ i]);
        end;
      finally
        aResults.DisposeOf;
      end;
    end;


    procedure   TKnobsCustomModule.UnFocus( const aSender: TObject); // virtual;
    var
      aWp : TKnobsWirePanel;
    begin
      if   Assigned( FOnUnFocus)
      then FOnUnFocus( aSender)
      else begin
        aWp := FindWirePanel;

        if   Assigned( aWp)
        then aWp.UnFocus( aSender);
      end;
    end;


    function    TKnobsCustomModule.IsConnected: Boolean;
    var
      i                  : Integer;
      aConnector         : TKnobsConnector;
      HasInputs          : Boolean;
      HasOutputs         : Boolean;
      AnyInputConnected  : Boolean;
      AnyOutputConnected : Boolean;
    begin

      // Implemented as :
      //
      // - when a module has no inputs and no outputs, it is considered connected.
      // - when a module has inputs but no outputs, it is connected when any input  is connected
      // - otherwise, the module does have outputs, it is connected when any output is connected.
      //
      // It should maybe have been implemented based on the side effects a module has, those are inferred
      // in the above, schematically:
      //
      // - ModOutput has the side effect of exporting signals to the outer world, so it is connected
      //   when one of its inputs is connected.
      // - For modules that have no side effects we'd want it to be unconnected when no outputs are connected
      // - modules with no inputs and no outputs are assumed to have a side effect, so they are connected.
      // - When the module is a native OSC module ... it is connected when any input or any output is connected
      //   but only the implementation knows that .. erm .. the synth module that is .. made a HasSideEffects property
      //   to resolve the issue in the designer .. could be used on any inputs only module.

      HasInputs          := False;
      HasOutputs         := False;
      AnyInputConnected  := False;
      AnyOutputConnected := False;

      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsConnector
        then begin
          aConnector := TKnobsConnector( Controls[ i]);

          if   aConnector is TKnobsOutput
          then begin
            HasOutputs := True;

            if   aConnector.Connected
            then AnyOutputConnected := True;
          end;

          if   aConnector is TKnobsInput
          then begin
            HasInputs := True;

            if   aConnector.Connected
            then AnyInputConnected := True;
          end;
        end;
      end;

      if not HasInputs and not HasOutputs  then Result := True                                     // no ins and outs, module is connected
      else if not HasOutputs               then Result := AnyInputConnected or HasSideEffects      // no outputs, connected when any input is connected
      else if HasSideEffects               then Result := AnyInputConnected or AnyOutputConnected  // module has side effects, connected when any in- or out-put is connected
      else                                      Result := AnyOutputConnected;                      // otherwise only connected when an output is connected
    end;


    procedure   TKnobsCustomModule.CopyParamsFrom( aModule: TKnobsCustomModule; CopyStrict: Boolean);
      function PostFixOf( const aName: string): string;
      var
        aParts : TStringList;
      begin
        aParts := Explode( aName, '_');

        try
          if   aParts.Count = 2
          then Result := aParts[ 1]
          else Result := aName;
        finally
          aParts.DisposeOf;
        end;
      end;

    var
      i : Integer;
      j : Integer;
    begin

      // CopyStrict assumes an exact match of source and destination module type
      // Note : Make sure to not copy possible CC settings from source controls, that would get fuzzy.

      if   Assigned( aModule)
      and  ((( aModule.ModuleType = ModuleType) and ( aModule.ControlCount = ControlCount)) or ( not CopyStrict))
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   CopyStrict // Exact match assumed, do a blind copy
          then begin
            if   Controls[ i].ClassType = aModule.Controls[ i].ClassType
            then begin
              if   Controls[ i] is TKnobsValuedControl
              then TKnobsValuedControl( Controls[ i]).KnobPosition := TKnobsValuedControl( aModule.Controls[ i]).KnobPosition
              else if Controls[ i] is TKnobsXYControl
              then begin
                TKnobsXYControl( Controls[ i]).KnobPositionX := TKnobsXYControl( aModule.Controls[ i]).KnobPositionX;
                TKnobsXYControl( Controls[ i]).KnobPositionY := TKnobsXYControl( aModule.Controls[ i]).KnobPositionY;
              end;
              // todo: maybe also copy over stuff like filename for waveplayers etc.
            end;
          end
          else begin // Not an exact match assumed, do a name based copy (names need a match still)
            for j := 0 to aModule.ControlCount - 1
            do begin
              if
                ( Controls[ i].ClassType = aModule.Controls[ j].ClassType) and
                SameText( PostfixOf( Controls[ i].Name), PostFixOf( aModule.Controls[ j].Name))
              then begin
                if   Controls[ i] is TKnobsValuedControl
                then TKnobsValuedControl( Controls[ i]).KnobPosition := TKnobsValuedControl( aModule.Controls[ j]).KnobPosition
                else if Controls[ i] is TKnobsXYControl
                then begin
                  TKnobsXYControl( Controls[ i]).KnobPositionX := TKnobsXYControl( aModule.Controls[ j]).KnobPositionX;
                  TKnobsXYControl( Controls[ i]).KnobPositionY := TKnobsXYControl( aModule.Controls[ j]).KnobPositionY;
                end;

                Break;
                // todo: maybe also copy over stuff like filename for waveplayers etc.
              end
            end;
          end;
        end;
      end;
    end;


    function    TKnobsCustomModule.VisitControls( aControlType: TControlClass; const aName: string; const aHandler: TKnobsOnVisitControl; const aUserData: TObject): Boolean;
    var
      i : Integer;
    begin
      Result := True;

      for i := 0 to ControlCount - 1
      do begin
        if   ( not Assigned( aControlType) and ( aName = ''))
        or  (( Controls[ i] is aControlType) and SameText( aName, Controls[ i].Name))
        then begin
          Result := aHandler( Controls[ i], aControlType, aUserData);

          if   not Result
          then Break;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.CollectCCControls( const aList: TStringList);
    var
      i   : Integer;
      aCC : Byte;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if
          IsAutomatable( Controls[ i])                     and
          (  Controls[ i] as IKnobsAutomatable).AllowAutomation and
          (( Controls[ i] as IKnobsAutomatable).AssignedMIDICC > 0)
        then begin
          aCC := ( Controls[ i] as IKnobsAutomatable).AssignedMIDICC;

          while aList.Count < aCC
          do aList.Add( Format( '%d', [ aList.Count + 1], AppLocale));

          aList.Objects[ aCC - 1] := Controls[ i];
        end;
      end;
    end;


    procedure   TKnobsCustomModule.UnassignMidiCC( aMidiCC: Byte);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if
          IsAutomatable( Controls[ i]) and
          (( Controls[ i] as IKnobsAutomatable).AssignedMIDICC = aMidiCC)
        then ( Controls[ i] as IKnobsAutomatable).AssignedMIDICC := 0;
      end;
    end;


    procedure   TKnobsCustomModule.SetRandomValue( anAmount: TSignal; aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      if   AllowRandomization
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).SetRandomValue( anAmount, aVariation);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.SetRandomValue( anAmount: TSignal); // overload;
    begin
      SetRandomValue( anAmount, ActiveVariation);
    end;


    procedure   TKnobsCustomModule.Randomize( anAmount: TSignal; aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      if   AllowRandomization
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).Randomize( anAmount, aVariation);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.Randomize( anAmount: TSignal); // overload;
    begin
      Randomize( anAmount, ActiveVariation);
    end;


    procedure   TKnobsCustomModule.Mutate( aProb, aRange: TSignal; aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      if   AllowRandomization
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).Mutate( aProb, aRange, aVariation);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.Mutate( aProb, aRange: TSignal); // overload;
    begin
      Mutate( aProb, aRange, ActiveVariation);
    end;


    procedure   TKnobsCustomModule.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom, aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      if   AllowRandomization
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, aVariation);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.MateWith( anXProb, aMutProb, aMutRange: TSignal; aDad, aMom: Integer); // overload;
    begin
      MateWith( anXProb, aMutProb, aMutRange, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsCustomModule.Morph( anAmount: TSignal; aDad, aMom, aVariation: Integer); // overload;
    var
      i : Integer;
    begin
      if   AllowRandomization
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).Morph( anAmount, aDad, aMom, aVariation);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.Morph( anAmount: TSignal; aDad, aMom: Integer); // overload;
    begin
      Morph( anAmount, aDad, aMom, ActiveVariation);
    end;


    procedure   TKnobsCustomModule.LiveMorph( anAmount: TSignal);
    var
      i : Integer;
    begin
      if   AllowRandomization
      then begin
        for i := 0 to ControlCount - 1
        do begin
          if   IsVariation( Controls[ i])
          then ( Controls[ i] as IKnobsVariation).LiveMorph( anAmount);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.AllowAllRandomization( aValue: Boolean);
    var
      i : Integer;
    begin
      AllowRandomization := aValue;

      for i := 0 to ControlCount - 1
      do begin
        if    IsVariation( Controls[ i])
        and (( not aValue) or ( Controls[ i] as IKnobsVariation).CanAllowRandomization)
        then ( Controls[ i] as IKnobsVariation).AllowRandomization := aValue;
      end;
    end;


    procedure   TKnobsCustomModule.SyncControlRandomization( const aControlTypeName: string; aValue: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   IsVariation( Controls[ i])
        and  SameText( ( Controls[ i] as IKnobsVariation).ControlType, aControlTypeName)
        then
          ( Controls[ i] as IKnobsVariation).AllowRandomization :=
            ( Controls[ i] as IKnobsVariation).CanAllowRandomization and aValue;
      end;
    end;


    function    TKnobsCustomModule.HasRandomizableControls: Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to ControlCount - 1
      do begin
        if   IsVariation( Controls[ i])
        and  (( Controls[ i] as IKnobsVariation).CanAllowRandomization)
        then begin
          Result := True;
          Break;
        end;
      end;
    end;


    procedure   TKnobsCustomModule.CountGene( var aCount: Integer);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   IsVariation( Controls[ i])
        then ( Controls[ i] as IKnobsVariation).CountGene( aCount);
      end;
    end;


    procedure   TKnobsCustomModule.FillGene( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   IsVariation( Controls[ i])
        then ( Controls[ i] as IKnobsVariation).FillGene( aGene, aVariation, anIndex);
      end;
    end;


    procedure   TKnobsCustomModule.AcceptGene( const aGene: TKnobsGene; aVariation: Integer; var anIndex: Integer);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   IsVariation( Controls[ i])
        then ( Controls[ i] as IKnobsVariation).AcceptGene( aGene, aVariation, AllowRandomization, anIndex);
      end;
    end;


    procedure   TKnobsCustomModule.CreateValueSnapShot( aTimeStamp: TKnobsTimeStamp; const aHistory: TKnobsEditHistory);
    var
      i        : Integer;
      aControl : TControl;
    begin
      if   Assigned( aHistory)
      then begin
        for i := 0 to ControlCount - 1
        do begin
          aControl := Controls[ i];

          if   IsUndoable( aControl)
          then ( aControl as IKnobsUndoable).AddEditFor( aTimeStamp, Format( '%s    %s', [ FriendlyName, Title], AppLocale), aHistory);
        end;
      end;
    end;


    procedure   TKnobsCustomModule.FixActiveRange( aValue: Integer);
    var
      i : Integer;
    begin
      FActiveRange := aValue;

      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).FixActiveRange( FActiveRange);
      end;
    end;


    procedure   TKnobsCustomModule.SetValueForRange( aRange: Integer; aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to ControlCount - 1
      do begin
        if   Controls[ i] is TKnobsValuedControl
        then TKnobsValuedControl( Controls[ i]).SetValueForRange( aRange, aValue);
      end;
    end;


{ ========
  TKnobsModuleButton = class( TSpeedButton)
  // Button as used in module selector, on click it creates a module
  // of the type that got registered with it.
  private
    FPageName   : string;
    FModuleType : TKnobsModuleType;
    FModuleName : string;
    FShowing    : Boolean;
  public
    property    PageName   : string           read FPageName;
    property    ModuleType : TKnobsModuleType read FModuleType;
    property    ModuleName : string           read FModuleName;
    property    Showing    : Boolean          read FShowing write SetShowing;
  private
}

    procedure   TKnobsModuleButton.SetShowing( aValue: Boolean);
    begin
      if   aValue <> FShowing
      then begin
        FShowing := aValue;
        Invalidate;
      end;
    end;


//  protected

    procedure   TKnobsModuleButton.Paint; // override;
    begin
      if   Showing
      then inherited;
    end;


//  public

    procedure   TKnobsModuleButton.FreeGlyph;
    begin
      Glyph.Assign( nil);
    end;


    procedure   TKnobsModuleButton.LoadGlyph( const aGlyph: TBitmap);
    var
      aFileName : string;
      fGlyph    : TBitmap;
    begin
      aFileName := GLooksPath + '\' + ModuleName + '.bmp';
      fGlyph    := nil;

      if   FileExists( aFileName)
      then begin
        fGlyph := TBitmap.Create;

        try
          with fGlyph
          do begin
            LoadFromFile( aFileName);
            NumGlyphs := Width div Height;
          end;
        except
          FreeAndNil( fGlyph);
        end;
      end;

      if   Assigned( fGlyph)
      then begin
         Glyph.Assign( fGlyph);
         FreeAndNil( fGlyph);
      end
      else if Assigned( aGlyph)
      then Glyph.Assign( aGlyph)
      else Caption := IntToStr( ModuleType);
    end;


    procedure   TKnobsModuleButton.Initialize( const aPageName: string; aModuleType: TKnobsModuleType; const aModuleName: string; const aGlyph: TBitmap);
    begin
      FPageName   := aPageName;
      FModuleType := aModuleType;
      FModuleName := aModuleName;
      AllowAllUp  := True;
      GroupIndex  := 1;
      Flat        := True;
      Margin      := -1;
      LoadGlyph( aGlyph);

      if   Assigned( Glyph)
      then begin
        Glyph.Transparent      := False;
        Glyph.TransparentColor := clNone;
        Glyph.TransparentMode  := tmFixed;
      end;
    end;


{ ========
  TKnobsShifterBtn = class( TBitBtn)
  private
    FRepeatTimer : TTimer;
    FRepeatCount : Integer;
    FMouseDown   : Boolean;
  private
}

    procedure   TKnobsShifterBtn.StartTimer( ams: Cardinal);
    begin
      StopTimer;
      FRepeatTimer.Interval := ams;
      FRepeatTimer.Enabled  := True;
    end;


    procedure   TKnobsShifterBtn.StopTimer;
    begin
      FRepeatTimer.Enabled := False;
    end;


    procedure   TKnobsShifterBtn.TimerFired( aSender: TObject);
    begin
      with FRepeatTimer
      do begin
        if   FRepeatCount > SlowRepeats
        then StartTimer( FastRepeatPause)
        else StartTimer( SlowRepeatPause);
      end;

      if   FMouseDown and MouseCapture
      then begin
        try
          Click;
        except
          StopTimer;
        end;
      end;

      Inc( FRepeatCount);
    end;


//  protected

    procedure   TKnobsShifterBtn.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   not FMouseDown
      and  ( Button = mbLeft)
      then begin
        FMouseDown   := True;
        FRepeatCount := 0;
        StartTimer( InitRepeatPause);
      end;
    end;


    procedure   TKnobsShifterBtn.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;
      StopTimer;
      FMouseDown := False;
    end;


    procedure   TKnobsShifterBtn.SetEnabled( aValue: Boolean); // override;
    begin
      inherited;

      if   not aValue
      then begin
        StopTimer;
        FMouseDown := False;
      end;
    end;


//  public

    constructor TKnobsShifterBtn.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FRepeatTimer := TTimer.Create( nil);
      StopTimer;
      FRepeatTimer.OnTimer := TimerFired;
    end;


    destructor  TKnobsShifterBtn.Destroy; // override;
    begin
      StopTimer;
      FRepeatTimer.OnTimer := nil;
      FreeAndNil( FRepeatTimer);
      inherited;
    end;


{ ========
  TKnobsModuleSelector = class( TPageControl)
  // A tabbed thingy having TModuleButtons on it's pages.
  private
    FRegisteredModules   : TKnobsModuleList;
    FOnModuleButtonClick : TKnobsOnModuleButtonClick;
    FOnGetGlyph          : TKnobsOnGetGlyph;
    FButtonsEnabled      : Boolean;
    FAllowDuplicates     : Boolean;
    FTimer               : TTimer;
    FTabColors           : array of TColor;
    FLeftmostButton      : Integer;
    FShiftersVisible     : Boolean;
  public
    property    LeftmostButton      : Integer                   read FLeftmostButton      write SetLeftmostButton;
    property    ShiftersVisible     : Boolean                   read FShiftersVisible     write SetShiftersVisible;
  published
    property    OnModuleButtonClick : TKnobsOnModuleButtonClick read FonModuleButtonClick write FOnModuleButtonClick;
    property    OnGetGlyph          : TKnobsOnGetGlyph          read FOnGetGlyph          write FonGetGlyph;
    property    ButtonsEnabled      : Boolean                   read FButtonsEnabled      write SetButtonsEnabled;
    property    AllowDuplicates     : Boolean                   read FAllowDuplicates     write FAllowDuplicates     default True;
  strict private
}


    class constructor TKnobsModuleSelector.Create;
    begin
      TStyleManager.Engine.RegisterStyleHook( TKnobsModuleSelector, TKnobsModuleSelectorStyleHook);
    end;


    class destructor  TKnobsModuleSelector.Destroy;
    begin
      // done : sorta ... This will crash .. after the Windows theme was selected by the user and then the program is closed
      try
        TStyleManager.Engine.UnregisterStyleHook( TKnobsModuleSelector, TKnobsModuleSelectorStyleHook);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


//  private

    procedure   TKnobsModuleSelector.DoResize( aSender: TObject);
    begin
      FixPageLayout( ActivePage);
    end;


    procedure   TKnobsModuleSelector.DoTabChanged( aSender: TObject);
    begin
      FixPageLayout( ActivePage);

      if FSearchName <> ''
      then FLastPage := ActivePage;
    end;


    procedure   TKnobsModuleSelector.DoLeftClick( aSender: TObject);
    begin
      if
        Assigned( aSender)                                and
        ( aSender is TKnobsShifterBtn)                    and
        Assigned( TKnobsShifterBtn( aSender).Parent)      and
        ( TKnobsShifterBtn( aSender).Parent is TTabsheet)
      then ShiftButtonsRight( TTabSheet( TKnobsShifterBtn( aSender).Parent));
    end;


    procedure   TKnobsModuleSelector.DoRightClick( aSender: TObject);
    begin
      if
        Assigned( aSender)                                and
        ( aSender is TKnobsShifterBtn)                    and
        Assigned( TKnobsShifterBtn( aSender).Parent)      and
        ( TKnobsShifterBtn( aSender).Parent is TTabsheet)
      then ShiftButtonsLeft( TTabSheet( TKnobsShifterBtn( aSender).Parent));
    end;


    procedure   TKnobsModuleSelector.StartTimeout;
    begin
      if   not Assigned( FTimer)
      then FTimer := TTimer.Create( nil);

      if   Assigned( FTimer)
      then begin
        FTimer.Enabled  := False;
        FTimer.OnTimer  := TimeoutFired;
        FTimer.Interval := 1000;
        FTimer.Enabled  := True;
      end;
    end;


    procedure   TKnobsModuleSelector.StopTimeout;
    begin
      if   Assigned( FTimer)
      then begin
        FTimer.Enabled := False;
        FTimer.OnTimer := nil;
        FreeAndNil( FTimer);
      end;
    end;


    procedure   TKnobsModuleSelector.TimeOutFired( aSender: TObject);
    begin
      AllButtonsUp;
      StopTimeout;
    end;


    procedure   TKnobsModuleSelector.AllButtonsUp;
    var
      i : Integer;
    begin
      with FRegisteredModules
      do begin
        for i := 0 to Count - 1
        do begin
          TKnobsModuleButton( Items[ i]).Down := True;
          TKnobsModuleButton( Items[ i]).Down := False;
        end;
      end;
    end;


//  private

    procedure   TKnobsModuleSelector.SetButtonsEnabled( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FButtonsEnabled
      then begin
        FButtonsEnabled := aValue;

        with FRegisteredModules
        do begin
          for i := 0 to Count - 1
          do TKnobsModuleButton( Items[ i]).Enabled := FButtonsEnabled;
        end;
      end;
    end;


    function    TKnobsModuleSelector.AddPage( const aName: string): TTabSheet;
    begin
      Result := TTabSheet.Create( Self);

      with Result
      do begin
        Visible          := True;
        Caption          := aName;
        PageControl      := Self;
        AlignWithMargins := False;
        OnResize         := DoResize;
      end;

      with TKnobsShifterBtn.Create( Result)
      do begin
        Align   := alLeft;
        Width   := ShifterWidth;
        Caption := '';
        Parent  := Result;
        OnClick := DoLeftClick;
        Visible := False;
      end;

      with TKnobsShifterBtn.Create( Result)
      do begin
        Align   := alRight;
        Width   := ShifterWidth;
        Caption := '';
        Parent  := Result;
        OnClick := DoRightClick;
        Visible := False;
      end;
    end;


    function    TKnobsModuleSelector.PageButtonCount( const aName: string): Integer;
    var
      i : Integer;
    begin
      Result := 0;

      if   PageExists( aName)
      then begin
        with PageByName( aName)
        do begin
          for i := 0 to ControlCount - 1
          do begin
            if   Controls[ i] is TKnobsModuleButton
            then Inc( Result);
          end;
        end;
      end;
    end;


    function    TKnobsModuleSelector.PageExists( const aName: string): Boolean;
    begin
      Result := PageByName( aName) <> nil;
    end;


    function    TKnobsModuleSelector.PageByName( const aName: string): TTabSheet;
    var
      i : Integer;
    begin
      Result := nil;

      for i := 0 to PageCount - 1
      do begin
        if   SameText( Pages[ i].Caption, aName)
        then begin
          Result := Pages[ i];
          Break;
        end;
      end;
    end;


    function    TKnobsModuleSelector.FindModulePage( aModuleType: TKnobsModuleType): TTabSheet;
    var
      i : Integer;
    begin
      Result := nil;

      with FRegisteredModules
      do begin
        for i := 0 to Count - 1
        do begin
          with TKnobsModuleButton( Items[ i])
          do begin
            if   ModuleType = aModuleType
            then begin
              Result := PageByName( PageName);
              Break;
            end;
          end;
        end;
      end;
    end;


    function    TKnobsModuleSelector.FindModuleButton( aModuleType: TKnobsModuleType): TKnobsModuleButton;
    var
      aPage : TTabSheet;
      i     : Integer;
    begin
      aPage  := FindModulePage( aModuleType);
      Result := nil;

      if   Assigned( aPage)
      then begin
        with aPage
        do begin
          for i := 0 to ControlCount - 1
          do begin
            if   Controls[ i] is TKnobsModuleButton
            then begin
              with TKnobsModuleButton( Controls[ i])
              do begin
                if   ModuleType = aModuleType
                then begin
                  Result := TKnobsModuleButton( Controls[ i]);
                  Break;
                end;
              end;
            end;
          end;
        end;
      end;
    end;


    procedure   TKnobsModuleSelector.DoModuleMouseDown( aSender: TObject; aButton: TMouseButton; aShift: TShiftState; X, Y: Integer);
    begin
      if   Assigned( FOnModuleButtonClick)
      and  ( aSender is TKnobsModuleButton)
      and  ( aShift = [ ssLeft])
      then begin
        FOnModuleButtonClick(
          TKnobsModuleButton( aSender),
          TKnobsModuleButton( aSender).ModuleType
        );
        StartTimeout;  // should release the button after a while ... button stays down though ... grrr .. this sort of works ...
      end;
    end;


//  private

    procedure   TKnobsModuleSelector.ShowShifters( aPage: TTabsheet; DoShow: Boolean);
    var
      i : Integer;
    begin
      if   Assigned( aPage)
      then begin
        for i := 0 to aPage.ControlCount - 1
        do begin
          if   aPage.Controls[ i] is TBitBtn
          then begin
            aPage.Controls[ i].Visible := DoShow;
            LeftmostButton[ aPage] := 0;
          end;
        end;
      end;
    end;


    procedure   TKnobsModuleSelector.EnableLeftShifter ( aPage: TTabsheet; DoMakeVisible: Boolean);
    var
      i : Integer;
    begin
      if   Assigned( aPage)
      then begin
        for i := 0 to aPage.ControlCount - 1
        do begin
          if   ( aPage.Controls[ i] is TBitBtn)
          and  ( aPage.Controls[ i].Align = alLeft)
          then aPage.Controls[ i].Enabled := DoMakeVisible;
        end;
      end;
    end;


    procedure   TKnobsModuleSelector.EnableRightShifter( aPage: TTabsheet; DoMakeVisible: Boolean);
    var
      i : Integer;
    begin
      if   Assigned( aPage)
      then begin
        for i := 0 to aPage.ControlCount - 1
        do begin
          if   ( aPage.Controls[ i] is TBitBtn)
          and  ( aPage.Controls[ i].Align = alRight)
          then aPage.Controls[ i].Enabled := DoMakeVisible;
        end;
      end;
    end;


    function    TKnobsModuleSelector.GetShiftersVisible( aPage: TTabsheet): Boolean;
    begin
      if   Assigned( aPage)
      then Result := ( aPage.Tag and $800) <> 0
      else Result := False;
    end;


    procedure   TKnobsModuleSelector.SetShiftersVisible( aPage: TTabSheet; aValue: Boolean);
    begin
      if   Assigned( aPage)
      then begin
        if   aValue
        then aPage.Tag := aPage.Tag or  $800
        else aPage.Tag := aPage.Tag and $7ff;
      end;
    end;


    function    TKnobsModuleSelector.GetLeftmostButton( aPage: TTabsheet): Integer;
    begin
      if   Assigned( aPage)
      then Result := aPage.Tag and $7ff
      else Result := 0;
    end;


    procedure   TKnobsModuleSelector.SetLeftmostButton( aPage: TTabsheet; aValue: Integer);
    begin
      if   Assigned( aPage)
      then begin
        if   aValue <> LeftmostButton[ aPage]
        then begin
          aPage.Tag := ( aPage.Tag and $800) or ( aValue and $7ff);
          ReOrderButtons( aPage);
        end;
      end;
    end;


    procedure   TKnobsModuleSelector.ReOrderButtons( aPage: TTabSheet);
    var
      i         : Integer;
      LastLeft  : Integer;
      MaxRight  : Integer;
    begin
      if   Assigned ( aPage)
      then begin
        if   ShiftersVisible[ aPage]
        then begin
          LastLeft := ShifterWidth + 1;
          MaxRight := aPage.ClientWidth - 1 - ShifterWidth;
        end
        else begin
          LastLeft := 0;
          MaxRight := aPage.ClientWidth;
        end;

        for i := 0 to aPage.ControlCount - 1
        do begin
          if   aPage.Controls[ i] is TKnobsModuleButton
          then begin
            with TKnobsModuleButton( aPage.Controls[ i])
            do begin
              if   ( i >= LeftmostButton[ aPage])
              and  ( LastLeft < MaxRight)
              then begin
                if   Visible
                or  ( aPage.Controls[ i] is TKnobsModuleSeparator)
                then begin
                  Left := LastLeft;
                  Inc( LastLeft, Width + 1);
                end;

                Showing := LastLeft < MaxRight;
              end
              else Showing := False;
            end;
          end;
        end;

        EnableRightShifter( aPage, not CalcFitting( aPage));
        EnableLeftShifter ( aPage, LeftmostButton[ aPage] > 0);
      end;
    end;


    function    TKnobsModuleSelector.CalcFitting( aPage: TTabSheet): Boolean;
    var
      i         : Integer;
      LastRight : Integer;
      MaxRight  : Integer;
    begin
      Result := True;

      if   Assigned( aPage)
      then begin
        if   ShiftersVisible[ aPage]
        then begin
          LastRight := ShifterWidth + 1;
          MaxRight  := aPage.ClientWidth - 1 - ShifterWidth;
        end
        else begin
          LastRight := 0;
          MaxRight  := aPage.ClientWidth;
        end;

        for i := 0 to aPage.ControlCount - 1
        do begin
          if   aPage.Controls[ i] is TKnobsModuleButton
          then begin
            if   i >= LeftmostButton[ aPage]
            then begin
              with TKnobsModuleButton( aPage.Controls[ i])
              do begin
                if   Visible
                or   ( aPage.Controls[ i] is TKnobsModuleSeparator)
                then Inc( LastRight, Width + 1);
              end;
            end;
          end;

          if   LastRight > MaxRight
          then begin
            Result := False;
            Break;
          end;
        end;
      end;
    end;


    function    TKnobsModuleSelector.CalcButtonsWidth( aPage: TTabSheet): Integer;
    var
      i : Integer;
    begin
      Result := 0;

      if   Assigned( aPage)
      then begin
        for i := 0 to aPage.ControlCount - 1
        do begin
          if   aPage.Controls[ i] is TKnobsModuleButton
          then begin
            with TKnobsModuleButton( aPage.Controls[ i])
            do begin
              if   Visible
              or   ( aPage.Controls[ i] is TKnobsModuleSeparator)
              then Inc( Result, Width + 1);
            end;
          end;
        end;
      end;
    end;


    procedure   TKnobsModuleSelector.FixPageLayout( aPage: TTabSheet);
    begin
      if   Assigned( aPage)
      then begin
        ShiftersVisible[ aPage] := CalcButtonsWidth( aPage) >= aPage.ClientWidth;
        ShowShifters( aPage, ShiftersVisible[ aPage]);
        ReOrderButtons ( aPage);
      end;
    end;


    procedure   TKnobsModuleSelector.FormatPage( aPage: TTabSheet);
    begin
      if   Assigned( aPage)
      then begin
        if   PageButtonCount( aPage.Caption) = 0
        then begin
          aPage.DisposeOf;

          if   PageCount > 0
          then ActivePage := Pages[ 0];
        end
        else begin
          LeftmostButton[ aPage] := 0;
          FixPageLayout( aPage);
        end;
      end;
    end;


    procedure   TKnobsModuleSelector.ShiftButtonsLeft( aPage: TTabSheet);
    begin
      if   Assigned( aPage)
      then begin
        if   not CalcFitting( aPage)
        then LeftmostButton[ aPage] := LeftmostButton[ aPage] + 1;
      end;
    end;


    procedure   TKnobsModuleSelector.ShiftButtonsRight( aPage: TTabSheet);
    begin
      if   Assigned( aPage)
      then begin
        if   LeftmostButton[ aPage] > 0
        then LeftmostButton[ aPage] := LeftmostButton[ aPage] - 1;
      end;
    end;


{  protected }

    procedure   TKnobsModuleSelector.Notification( aComponent: TComponent; anOperation: TOperation); // override;
    begin
      inherited;

      if   aComponent is TKnobsModuleButton
      then begin
        if   anOperation = opInsert
        then FRegisteredModules.Add   ( aComponent)
        else FRegisteredModules.Remove( aComponent);
      end;
    end;


    procedure   TKnobsModuleSelector.DrawTab( aTabIndex: Integer; const aRect: TRect; anActive: Boolean); // override;
    var
      p         : Integer;
      i         : Integer;
      S         : string;
      aTextSize : TSize;
      aTextLeft : Integer;
    begin
      if   Assigned( OnDrawTab)
      then OnDrawTab( Self, aTabIndex, aRect, anActive)
      else begin
        p := 0;

        for i := 0 to PageCount - 1
        do begin
          if   Pages[ i].TabVisible
          then begin
            if   p = aTabIndex
            then begin
              S         := Pages[ i].Caption;
              aTextSize := Canvas.TextExtent( S);
              aTextLeft := ( aRect.Width - aTextSize.Width) div 2;
              Canvas.Brush.Color := GetTabColorLo( i);
              Canvas.FillRect( aRect);
              Canvas.TextOut( aRect.left + aTextLeft, aRect.top - 1, S);
              Break;
            end
            else Inc( p);
          end;
        end;
      end;
    end;


{   public }

    constructor TKnobsModuleSelector.Create( anOwner: TComponent); // override;
    begin
      inherited;
      FRegisteredModules := TKnobsModuleList.Create( False);
      HotTrack           := True;
      FAllowDuplicates   := True;
      OnResize           := DoResize;
      OnChange           := DoTabChanged;
    end;


    destructor  TKnobsModuleSelector.Destroy; // override;
    begin
      StopTimeout;
      inherited;
      FRegisteredModules.DisposeOf;
     end;


    procedure   TKnobsModuleSelector.RegisterModuleType( const aPageName: string; aModuleType: TKnobsModuleType; const aBitmapsFolder, aModuleName: string);
    var
      aPage         : TTabSheet;
      aModuleButton : TKnobsModuleButton;
      aGlyph        : TBitmap;
    begin
      if   not FAllowDuplicates
      then begin
        if   Assigned( FindModuleButton( aModuleType))
        and  ( aModuleType >= 0)
        then begin
          raise
            EKnobsNoDuplicates.
              CreateFmt(
                'A module type %d with name ''%s''is already present on page ''%s''',
                [
                  aModuleType,
                  aModuleName,
                  FindModulePage( aModuleType).Caption
                ]
              );
        end;
      end;


      aPage := PageByName( aPageName);

      if   not Assigned( aPage)
      then aPage := AddPage( aPageName);

      ActivePage := aPage;
      ActivePage := Pages[ 0];

      if   aModuleType >= 0
      then aModuleButton := TKnobsModuleButton   .Create( Self)
      else aModuleButton := TKnobsModuleSeparator.Create( Self);

      if   Assigned( FOnGetGlyph)
      then aGlyph := FOnGetGlyph( Self, aBitmapsFolder, aModuleName)
      else aGlyph := nil;

      with aModuleButton
      do begin
        Parent   := aPage;
        Left     := 0;
        Top      := 0;
        Flat     := True;
        Hint     := Format( '%s~%d~', [ aModuleName, aModuleType], AppLocale);
        ShowHint := True;
        Enabled  := FButtonsEnabled;
        Visible  := aModuleType >= 0;
        Initialize( aPageName, aModuleType, aModuleName, aGlyph);
        OnMouseDown := DoModuleMouseDown;

        if   aModuleButton is TKnobsModuleSeparator
        then begin
          Width   := SeparatorWidth;
          Height  := SeparatorHeight;
          Visible := False;
        end
        else begin
          Width  := ButtonWidth;
          Height := ButtonHeight;
        end
      end;

      FormatPage( aPage);
    end;


    procedure   TKnobsModuleSelector.UnregisterModuleType( aModuleType: TKnobsModuleType);
    var
      aModuleButton : TKnobsModuleButton;
      aPage         : TTabSheet;
    begin
      aModuleButton := FindModuleButton( aModuleType);
      aPage         := FindModulePage  ( aModuleType);

      if   Assigned( aModuleButton)
      and  Assigned( aPage        )
      then begin
        aModuleButton.DisposeOf;
        FormatPage( aPage); // This will remove the page when it beccomes empty.
      end;
    end;


    procedure   TKnobsModuleSelector.UnregisterAllTypes;
    var
      i : Integer;
    begin
      for i := PageCount - 1 downto 0
      do Pages[ i].DisposeOf;
    end;


    procedure   TKnobsModuleSelector.GetTabColors( const aGetter: TOnGetTabColor; aMixColor: TColor; aMixHi, aMixLo: Byte);
    var
      i : Integer;
    begin
      SetLength( FTabColorsHi, 0);
      SetLength( FTabColorsLo, 0);

      for i := 0 to PageCount - 1
      do begin
        SetLength( FTabColorsHi, Length( FTabColorsHi) + 1);
        SetLength( FTabColorsLo, Length( FTabColorsLo) + 1);
        FTabColorsHi[ Length( FTabColorsHi) - 1] := aGetter( Self, Pages[ i].Caption, aMixColor, aMixHi);
        FTabColorsLo[ Length( FTabColorsLo) - 1] := aGetter( Self, Pages[ i].Caption, aMixColor, aMixLo);
      end;
    end;


    function    TKnobsModuleSelector.GetTabColorHi( anIndex: Integer): TColor;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < Length( FTabColorsHi))
      then Result := FTabColorsHi[ anIndex]
      else Result := Brush.Color;
    end;


    function    TKnobsModuleSelector.GetTabColorLo( anIndex: Integer): TColor;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < Length( FTabColorsLo))
      then Result := FTabColorsLo[ anIndex]
      else Result := Brush.Color;
    end;


    procedure   TKnobsModuleSelector.Search( const aName: string);
    var
      i           : Integer;
      j           : Integer;
      aPage       : TTabSheet;
      SomeVisible : Boolean;
      AllVisible  : Boolean;
      IsVisible   : Boolean;
    begin
      if   not StrToIntDef( aName, -1) >= 0  // Number search is for modules in the patch only
      then begin
        AllVisible  := True;
        FSearchName := aName;

        for i := PageCount - 1 downto 0
        do begin
          aPage := Pages[ i];

          with aPage
          do begin
            SomeVisible := False;

            for j := 0 to ControlCount - 1
            do begin
              if   Controls[ j] is TKnobsModuleButton
              then begin
                IsVisible := ( Pos( UpperCase( aName), UpperCase( TKnobsModuleButton( Controls[ j]).ModuleName)) <> 0) or ( aName = '');
                TKnobsModuleButton( Controls[ j]).Visible := IsVisible;

                if   IsVisible
                then SomeVisible := True;
              end
            end;

            TabVisible := SomeVisible;

            if   TabVisible
            then begin
              FixPageLayout( aPage);
              ActivePage := aPage;

              if aName <> ''
              then FLastPage := aPage;
            end
            else AllVisible := False;
          end;
        end;

        if AllVisible
        then ActivePage := FLastPage;
      end;
    end;


    procedure   TKnobsModuleSelector.ReloadGlyphs( const aBitmapsFolder: string);
    var
      i             : Integer;
      j             : Integer;
      aPage         : TTabSheet;
      aModuleButton : TKnobsModuleButton;
      aPageName     : string;
      aModuleType   : TKnobsModuleType;
      aModuleName   : string;
      aGlyph        : TBitmap;
    begin
      for i := 0 to PageCount - 1
      do begin
        aPage := Pages[ i];

        for j := 0 to aPage.ControlCount - 1
        do begin
          if   aPage.Controls[ j] is TKnobsModuleButton
          then begin
            aModuleButton := TKnobsModuleButton( aPage.Controls[ j]);

            aPageName   := aModuleButton.PageName;
            aModuleType := aModuleButton.ModuleType;
            aModuleName := aModuleButton.ModuleName;

            if   Assigned( FOnGetGlyph)
            then aGlyph := FOnGetGlyph( Self, aBitmapsFolder, aModuleName)
            else aGlyph := nil;

            aModuleButton.FreeGlyph;
            aModuleButton.Initialize( aPageName, aModuleType, aModuleName, aGlyph);
          end;
        end;
      end;
    end;


{ ========
  TKnobsHintWindow = class( THintWindow)
  private
    FModuleType : string;
    FBitmap     : TBitmap;
    FComment    : string;
  published
    property    Caption;
  protected
}

    function    TKnobsHintWindow.MakeBitmap: TBitmap;
    var
      aModuleTypeInt : TKnobsModuleType;
    begin
      aModuleTypeInt := StrToIntDef( FModuleType, -1);

      if   aModuleTypeInt >= 0
      then begin
        Result   := CreateModuleBitmap( aModuleTypeInt, True);
        FComment := ReadModuleComment ( aModuleTypeInt);
      end
      else begin
        Result   := nil;
        FComment := '';
      end;
    end;


    procedure   TKnobsHintWindow.ParseHint( var aHintStr: string);
    type
      TParseState = ( psNothing, psTilde);
    var
      aHint      : string;
      i          : Integer;
      ParseState : TParseState;
    begin
      FModuleType   := '';
      aHint         := '';
      ParseState    := psNothing;

      for i := 1 to Length( aHintStr)
      do begin

        case ParseState of

          psNothing :
            begin
              if   aHintStr[ i] = '~'
              then ParseState := psTilde
              else aHint := aHint + aHintStr[ i];
            end;

          psTilde :
            begin
              if   aHintStr[ i] = '~'
              then ParseState := psNothing
              else FModuleType := FModuleType + aHintStr[ i];
            end;
        end;

      end;

      if   Assigned( FBitmap)
      then FreeAndNil( FBitmap);

      FBitmap  := MakeBitmap;

      if Assigned( FBitmap)
      then aHintStr := FComment;
    end;


    procedure   TKnobsHintWindow.Paint; // override;
    var
      R          : TRect;
      aTextRect  : TRect;
      aTopMargin : Integer;
    const
      TEXT_MARGIN_LEFT =  2;
      IMG_MARGIN_TOP   =  4;
      TEXT_MARGIN_TOP  =  2;
    begin
      if   Assigned( FBitmap)
      then aTopMargin := IMG_MARGIN_TOP + TEXT_MARGIN_TOP + FBitmap.Height
      else aTopMargin := TEXT_MARGIN_TOP;

      R := ClientRect;
      Inc( R.Left, 2);
      Inc( R.Top , 2);

      with Canvas
      do begin
        Pen.Color   := clInfoBk;
        Brush.Color := clInfoBk;
        Brush.Style := bsSolid;
        Rectangle( 0, 0, R.Width + 2, R.Bottom + 2);

        if   Assigned( FBitmap)
        then Draw( 2, IMG_MARGIN_TOP, FBitmap);

        aTextRect := R;
        Inc( aTextRect.Top , aTopMargin);
        Inc( aTextRect.Left, TEXT_MARGIN_LEFT);
        DrawText( Handle, PChar( Caption), -1, aTextRect, dt_NoClip or dt_WordBreak);
      end;
    end;


//  public

    constructor TKnobsHintWindow.Create( anOwner: TComponent); // override;
    begin
      inherited;

      with Canvas.Font do
      begin
        Name  := 'Arial';
        Color := clBlack;
      end;
    end;


    destructor  TKnobsHintWindow.Destroy; // override;
    begin
      if   Assigned( FBitmap)
      then FreeAndNil( FBitmap);

      inherited;
    end;


    procedure   TKnobsHintWindow.ActivateHint( aRect: TRect; const aHint: string); // override;
    const
      EXTRA_WIDTH  =  8;
      EXTRA_HEIGHT =  8;
    var
      anExtraHeight : Integer;
      aWidth        : Integer;
      aHintStr      : string;
    begin
      aHintStr := aHint;
      ParseHint( aHintStr);
      Caption := aHintStr;

      if   Assigned( FBitmap)
      then begin
        anExtraHeight := EXTRA_HEIGHT + FBitmap.Height;
        aWidth        := EXTRA_WIDTH  + FBitmap.Width;
      end
      else begin
        anExtraHeight := EXTRA_HEIGHT;
        aWidth        := EXTRA_WIDTH + aRect.Width;
      end;

      Inc( aRect.Bottom, anExtraHeight);
      aRect.Right := aRect.Left + aWidth;
      UpdateBoundsRect( aRect);

      if   aRect.Top + Height > Screen.DesktopHeight
      then aRect.Top := Screen.DesktopHeight - Height;

      if   aRect.Left + Width > Screen.DesktopWidth
      then aRect.Left := Screen.DesktopWidth - Width;

      if   aRect.Left < Screen.DesktopLeft
      then aRect.Left := Screen.DesktopLeft;

      if   aRect.Bottom < Screen.DesktopTop
      then aRect.Bottom := Screen.DesktopTop;

      SetWindowPos( Handle, HWND_TOPMOST, aRect.Left, aRect.Top, Width, Height, SWP_SHOWWINDOW or SWP_NOACTIVATE);
      Invalidate;
    end;


{ ========
  TKnobsSimpleHintWindow = class( THintWindow)
  // Hint popup window for controls
  protected
}

    procedure   TKnobsSimpleHintWindow.Paint; // override;
    var
      R : TRect;
    begin
      R := ClientRect;
      R.Inflate( -1, -1);

      with Canvas
      do begin
        Pen.Color   := clInfoBk;
        Brush.Color := clInfoBk;
        Brush.Style := bsSolid;
        Rectangle( 0, 0, R.Width + 2, R.Bottom + 2);

        DrawText( Handle, PChar( Caption), -1, R, dt_NoClip or dt_WordBreak);
      end;
    end;


//  public

    constructor TKnobsSimpleHintWindow.Create( anOwner: TComponent); // override;
    begin
      inherited;

      with Canvas.Font do
      begin
        Name  := 'Arial';
        Color := clBlack;
      end;
    end;


{ ========
  TKnobsTimerSpeedButton = class( TSpeedButton)
  private
    FRepeatTimer    : TTimer;
    FRepeatCount    : Integer;
    FTimeBtnOptions : TKnobsTimeBtnOptions;
  published
    property    TimeBtnOptions: TKnobsTimeBtnOptions read FTimeBtnOptions write FTimeBtnOptions              default [];
    property    StyleElements                                                                                default [];
  private
}

    procedure   TKnobsTimerSpeedButton.TimerExpired( Sender: TObject); // virtual;
    begin
      if   FRepeatCount > SlowRepeats
      then FRepeatTimer.Interval := FastRepeatPause
      else FRepeatTimer.Interval := SlowRepeatPause;

      if   ( FState = bsDown)
      and  MouseCapture
      then begin
        try
          Click;
        except
          FRepeatTimer.Enabled := False;
          raise;
        end;
      end;

      Inc( FRepeatCount);
    end;


    procedure   TKnobsTimerSpeedButton.WMSetFocus( var aMessage: TWMSetFocus); // message WM_SETFOCUS;
    begin
      TimeBtnOptions := TimeBtnOptions + [ tbFocusRect];
      Invalidate;
    end;


    procedure   TKnobsTimerSpeedButton.WMKillFocus( var aMessage: TWMKillFocus); // message WM_KILLFOCUS;
    begin
      TimeBtnOptions := TimeBtnOptions - [ tbFocusRect];
      Invalidate;
    end;


//  protected

    procedure   TKnobsTimerSpeedButton.Paint; // override;
    var
      R: TRect;
    begin
      inherited;

      if   tbFocusRect in FTimeBtnOptions
      then begin
        R := Bounds( 0, 0, Width, Height);
        InflateRect( R, -3, -3);

        if   FState = bsDown
        then OffsetRect( R, 1, 1);

        DrawFocusRect( Canvas.Handle, R);
      end;
    end;


    procedure   TKnobsTimerSpeedButton.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   tbAllowTimer in FTimeBtnOptions
      then begin
        if   not Assigned( FRepeatTimer)
        then FRepeatTimer := TTimer.Create( Self);

        FRepeatCount := 0;

        with FRepeatTimer
        do begin
          Enabled  := False;
          OnTimer  := TimerExpired;
          Interval := InitRepeatPause;
          Enabled  := True;
        end;
      end;
    end;


    procedure   TKnobsTimerSpeedButton.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer); // override;
    begin
      inherited;

      if   Assigned( FRepeatTimer)
      then FRepeatTimer.Enabled  := False;
    end;


//  public

    constructor TKnobsTimerSpeedButton.Create( anOwner: TComponent); // override;
    begin
      inherited;
      TimeBtnOptions  := [ tbAllowTimer];
      StyleElements := [];
    end;


    destructor  TKnobsTimerSpeedButton.Destroy; // override;
    begin
      if   Assigned( FRepeatTimer)
      then FreeAndNil( FRepeatTimer);

      inherited;
    end;


{ ========
  TKnobsDelayedSpeedButton = Class( TKnobsTimerSpeedButton)
  private
    FOnDelayedClick : TNotifyEvent;
  published
    property    OnDelayedClick: TNotifyEvent read FOnDelayedClick write FOnDelayedClick;
  private
}

    procedure   TKnobsDelayedSpeedButton.TimerExpired( Sender: TObject); // override;
    begin
      if   ( FState = bsDown)
      and  MouseCapture
      then begin
        try
          FState := bsUp;
          Repaint;
          DelayedClick;
        finally
          FRepeatTimer.Enabled := False;
        end;
      end;
    end;


    procedure   TKnobsDelayedSpeedButton.DelayedClick;
    begin
      if   Assigned( FOnDelayedClick)
      then FOnDelayedClick( Self);
    end;



initialization

  CreateBitmaps( '');
  SetLength( GRangeSpecs, 5);
  GRangeSpecs[ 0].Name    := 'random limits'; GRangeSpecs[ 0].IsMorph := False;
  GRangeSpecs[ 1].Name    := 'morph 1'      ; GRangeSpecs[ 1].IsMorph := True;
  GRangeSpecs[ 2].Name    := 'morph 2'      ; GRangeSpecs[ 2].IsMorph := True;
  GRangeSpecs[ 3].Name    := 'morph 3'      ; GRangeSpecs[ 3].IsMorph := True;
  GRangeSpecs[ 4].Name    := 'morph 4'      ; GRangeSpecs[ 4].IsMorph := True;

finalization

  FreeBitmaps;

end.

